/*
 ** Copyright 2004-2007 - INL
 ** Written by Eric Leblond <eric.leblond@inl.fr>
 **            Vincent Deffontaines <vincent@inl.fr>
 ** INL http://www.inl.fr/
 **
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, version 2 of the License.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software
 ** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "thread_session.h"
#include "thread_session.moc"
#include <QtGui>

extern nuclient_error_t *err;

ThreadCheckConn::ThreadCheckConn(QObject* _parent) : QThread(_parent)
{
	session = NULL;
	username = "";
	password = "";

	connect(this, SIGNAL(tray_set_connected()), parent(), SLOT(tray_set_connected()));
	connect(this, SIGNAL(tray_set_trying()), parent(), SLOT(tray_set_trying()));
	connect(this, SIGNAL(tray_set_stopped()), parent(), SLOT(tray_set_stopped()));
	connect(this, SIGNAL(tray_report_error()), parent(), SLOT(tray_report_error()));
	connect(this, SIGNAL(tray_report_connected()), parent(), SLOT(tray_report_connected()));
	connect(this, SIGNAL(tray_report_disconnected()), parent(), SLOT(tray_report_disconnected()));
}

void ThreadCheckConn::askStop()
{
	loop_mutex.lock();
	loop_continue = false;
	loop_mutex.unlock();
}

void ThreadCheckConn::setUserInfos(QString _username, QString _password)
{
	username = _username;
	password = _password;

	// We don't know if those username/passwords are valid
	// so if the authentication fails, there is no need to
	// trying reauthenticating to Nuauth in a loop
	loop_continue = false;
	report_errors = true;;
}

void ThreadCheckConn::run()
{
	startAuth();
	while(1)
	{
		loop_mutex.lock();
		if(!loop_continue)
		{
			loop_mutex.unlock();
			break;
		}
		loop_mutex.unlock();

		if(session)
		{
			// A session is opened, so check we are still connected to Nuauth
			if(nu_client_check(session, err) == -1)
			{
				nu_client_delete(session);
				session = NULL;
				if(report_errors)
				{
					emit tray_set_trying();
					emit tray_report_disconnected();
					report_errors = false;
				}
			}
		}
		else
		{
			// We have been disconnected
			// Restart the authentication using the last username and password used
			startAuth();
		}

		sleep(3); // Qt function
	}
	stopAuth();
}

void ThreadCheckConn::startAuth()
{
	// Stop automatic reauthentication after the connection has been lost
	session = nu_client_new(username.toStdString().c_str(),
				password.toStdString().c_str(),
				1,
				err);

	QSettings settings;
	settings.sync();

	// Configure certificate related parameters
	char* chr_key = strdup(settings.value("key").toString().toStdString().c_str());
	char* chr_cert = strdup(settings.value("cert").toString().toStdString().c_str());
	char* chr_ca = strdup(settings.value("ca").toString().toStdString().c_str());

	if(settings.value("use_certificate").toBool())
	{
		if(!nu_client_setup_tls(session, chr_key, chr_cert, chr_ca, "", err))
		{
			free(chr_key);
			free(chr_cert);
			free(chr_ca);
			
			// If the timer is active, this connection is done automatically
			// so we don't display the error message
			if(report_errors)
				emit tray_report_error();
			if(session)
			     	   nu_client_delete(session);
			session = NULL;
			return;
		}
	}
	free(chr_key);
	free(chr_cert);
	free(chr_ca);

	// Connect to NuAuth
	char* chr_hostname;
	char* chr_port;

#ifdef HAVE_DEFAULT_NUAUTH_IP
	if(settings.value("hostname") == "" && session->default_hostname)
		settings.setValue("hostname", session->default_hostname);
	if(settings.value("port") == "" && session->default_port)
		settings.setValue("port", session->default_port);
#endif // #ifdef HAVE_DEFAULT_NUAUTH_IP

	chr_hostname = strdup(settings.value("hostname").toString().toStdString().c_str());
	chr_port = strdup(settings.value("port").toString().toStdString().c_str());

	if(!nu_client_connect(session, chr_hostname, chr_port, err))
	{
		if(report_errors)
			emit tray_report_error();
		free(chr_hostname);
		free(chr_port);

		if(session)
			nu_client_delete(session);
		session = NULL;
		return;
	}
	
	free(chr_hostname);
	free(chr_port);

	emit tray_set_connected();
	emit tray_report_connected();
	loop_continue = true;
	report_errors = true;
}

void ThreadCheckConn::stopAuth()
{
	if(session)
		nu_client_delete(session);
	session = NULL;
	emit tray_set_stopped();
}

