/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1999 David Baum.
 * All Rights Reserved.
 */


#include "parse_util.h"
#include "Expr.h"
#include "parser.h"
#include "AssignStmt.h"
#include "Symbol.h"
#include "GosubStmt.h"
#include "Program.h"
#include "UnaryExpr.h"
#include "BinaryExpr.h"
#include "AtomExpr.h"
#include "ValueExpr.h"
#include "Error.h"
#include "ModExpr.h"
#include "CaseStmt.h"
#include "FunctionDef.h"
#include "AsmStmt.h"
#include "ScopeStmt.h"
#include "AcquireStmt.h"
#include "DeclareStmt.h"
#include "ArrayExpr.h"
#include "TaskIdExpr.h"
#include "IndirectExpr.h"
#include "CatchStmt.h"
#include "ShiftExpr.h"
#include "Resource.h"


// template class to auto-delete an object upon destruction
template <class T> class Deleter
{
public:
	Deleter(T *t) : t_(t) {};
	~Deleter()	{ delete t_; }

private:
	T*			t_;
};



int GetConstantValue(const Expr *e)
{
	Deleter<const Expr> d(e);
	int v = 0;
	bool ok;
	
	ok = e->Evaluate(v);	
	if (!ok)
		Error(kErr_ConstantNeeded).Raise(&e->GetLoc());

	return v;
}


int GetLValue(const Expr *e)
{
	Deleter<const Expr> d(e);

	int lv = e->GetLValue();
	if (lv==kIllegalVar)
		Error(kErr_LValueNeeded).Raise(&e->GetLoc());

	return lv;
}


void CheckLValue(Expr *lval)
{
	if (!lval->PotentialLValue())
		Error(kErr_LValueNeeded).Raise(&lval->GetLoc());
}


Expr *MakeUnaryExpr(int op, Expr *arg)
{
	if (!UnaryExpr::NeedsConstant(op) || arg->PromiseConstant())
		return new UnaryExpr(op, arg);

	if (op == '~') {
		// special case transform ~x  ->  -1 - x
		return new BinaryExpr(
			new AtomExpr(kRCX_ConstantType, -1, arg->GetLoc()),
			'-',
			arg
		);		
	}

	Error(kErr_ConstantNeeded).RaiseLex();
	// return a dummy expr so the parse tree is still valid
	return arg;
}


Expr *MakeBinaryExpr(Expr *lhs, int op, Expr *rhs)
{
	if (!BinaryExpr::NeedsConstant(op) ||
		(rhs->PromiseConstant() &&
		 lhs->PromiseConstant()))
	{
		return new BinaryExpr(lhs, op, rhs);
	}
	
	
	// modulo is a special case
	if (op == '%')
	{
		return new ModExpr(lhs, rhs);
	}
	
	// shifts are also special cases
	if (op == LEFT || op == RIGHT)
	{
		// right argument must be a constant
		if (rhs->PromiseConstant())
		{
			return new ShiftExpr(lhs, rhs, (op==LEFT) ? ShiftExpr::kLeft : ShiftExpr::kRight);
		}

		// if rhs is not constant, this will just fall through to
		// the normal binary expression case, which is an error
	}

	// xor handled by a transform
	if (op == '^')
	{
		// a^b  ->  (-1 - (a&b)) & (a | b)
		Expr *lhc = lhs->Clone(0);
		Expr *rhc = rhs->Clone(0);
		
		Expr *nand = new BinaryExpr(
			new AtomExpr(kRCX_ConstantType, -1, lhs->GetLoc()),
			'-',
			new BinaryExpr(
				lhc,
				'&',
				rhc
			)
		);
		
		return new BinaryExpr(
			nand,
			'&',
			new BinaryExpr(lhs, '|', rhs)
		);
	}
	

	Error(kErr_ConstantNeeded).RaiseLex();
	// return a dummy expr so the parse tree is still valid
	delete rhs;
	return lhs;
}


Expr *MakeValueExpr(Expr *e)
{
	if (!e->PromiseConstant())
	{
		Error(kErr_ConstantNeeded).RaiseLex();
		// return a dummy expr so the parse tree is still valid
		return e;
	}

	// if e can be evaluated, return AtomExpr, otherwise return ValueExpr
	int value;
	if (e->Evaluate(value))
	{
		Expr *newExpr = new AtomExpr(RCX_VALUE_TYPE(value), RCX_VALUE_DATA(value), e->GetLoc());
		delete e;
		return newExpr;
	}
	else
		return new ValueExpr(e);
}


Stmt *MakeAssign2Stmt(Expr *lhs, int op, Expr *rhs)
{
	Expr *dst = lhs->Clone(0);
	Expr *math = MakeBinaryExpr(lhs, op, rhs);
	
	return new AssignStmt(dst, math);
}


CaseStmt* MakeCaseStmt(const Expr *e, const struct LexLocation &loc)
{
	int v = GetConstantValue(e);
	
	if (v < -32768 || v > 32767)
		Error(kErr_CaseRange).RaiseLex();

	
	return new CaseStmt(v, loc);
}


Stmt *MakeAcquireStmt(const Expr *e, Stmt *body, Stmt *handler, LocationNode *ln)
{
	Deleter<LocationNode> d(ln);
	int v = GetConstantValue(e);

	return new AcquireStmt(v, body, handler, ln->GetLoc());
}


Stmt *MakeCatchStmt(const Expr *e, Stmt *body, LocationNode *ln)
{
	Deleter<LocationNode> d(ln);

	int v;
	
	if (e)
		v = GetConstantValue(e);
	else
		v = -1;
	
	return new CatchStmt(v, body, ln->GetLoc());
}



void DefineArg(FunctionDef *f, const Symbol *name, int type)
{
	if (!f->AddArg(name, (FunctionDef::ArgType)type))
		Error(kErr_SymRedef, name->GetKey()).RaiseLex();
}



void BeginScope()
{
	gProgram->PushScope();
}


Stmt *EndScope(Stmt *body)
{
	gProgram->PopScope();
	return new ScopeStmt(body);
}


FunctionDef* BeginFunction(FunctionDef *f, const Symbol *name, bool listing)
{
	f->SetName(name);
	gProgram->AddFunction(f);
	gProgram->PushScope();
	f->CreateArgVars();
	f->SetListingEnabled(listing);
	return f;
}


void EndFunction(FunctionDef *f, Stmt *body, LocationNode *start, LocationNode *end)
{
	f->SetLocations(start, end);
	f->SetBody(body);
	gProgram->PopScope();
}


ConstField *MakeConstField(Expr *e)
{
	ConstField *c = new ConstField(e);
	
	if (!e->PromiseConstant())
	{
		Error(kErr_ConstantNeeded).RaiseLex();
	}
	return c;
}


Expr *MakeVarExpr(const Symbol *name, LocationNode *ln)
{
	Deleter<LocationNode> d(ln);
	bool array;

	int var = gProgram->GetVar(name, array);
	if (var == kIllegalVar)
	{
		// check to see if name is for resource
		const Resource *resource = gProgram->GetResource(name);
		if (resource)
		{
			return new AtomExpr(kRCX_ConstantType, resource->GetNumber() , ln->GetLoc());
		}

		Error(kErr_UndefinedVar, name->GetKey()).Raise(&ln->GetLoc());
	}
	else if (array)
	{
		Error(kErr_VarIsArray, name->GetKey()).Raise(&ln->GetLoc());
	}

	return new AtomExpr(kRCX_VariableType, var, ln->GetLoc());
}


Expr *MakeIndirectExpr(Expr *src, Expr *idx)
{
	if (!src->PromiseConstant())
	{
		Error(kErr_ConstantNeeded).RaiseLex();
		delete src;
		return idx;
	}

	return new IndirectExpr(src, idx);
}


Expr *MakeTaskIdExpr(LocationNode *ln)
{
	Deleter<LocationNode> d(ln);

	return new TaskIdExpr(ln->GetLoc());
}


Expr *MakeArrayExpr(const Symbol *name, LocationNode *ln, Expr *index)
{
	Deleter<LocationNode> d(ln);
	bool array;
	
	int var = gProgram->GetVar(name, array);
	if (var == kIllegalVar)
	{
		Error(kErr_UndefinedVar, name->GetKey()).Raise(&ln->GetLoc());
	}
	else if (!array)
	{
		Error(kErr_VarIsNotArray, name->GetKey()).Raise(&ln->GetLoc());
	}
	
	return new ArrayExpr(var, index);
}



DeclareStmt *MakeDeclareStmt(const Symbol *name, LocationNode *ln, Expr *arraySize)
{
	Deleter<LocationNode> d(ln);
	
	int var = gProgram->CreateVar(name, arraySize);

	int size = 1;
	if (arraySize)
	{
		const RCX_Target *t = gProgram->GetTarget();
		
		if (!t->fArrays)
			Error(kErr_NoTargetArrays, t->fName).Raise(&ln->GetLoc());
		
		if (!arraySize->Evaluate(size))
		{
			Error(kErr_ConstantNeeded).Raise(&arraySize->GetLoc());
		}
		delete arraySize;
	}

	return new DeclareStmt(name, var, ln->GetLoc(), size);
}
