/****************************************************************************************/
/*											*/
/* This program is free software; you can redistribute it and/or modify it under the	*/
/* terms of the GNU General Public License as published by the Free Software		*/
/* Foundation; either version 2 of the License, or (at your option) any later version.	*/
/*											*/
/* This program is distributed in the hope that it will be useful, but WITHOUT ANY	*/
/* WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A	*/
/* PARTICULAR PURPOSE. See the GNU General Public License for more details.		*/
/*											*/
/* You should have received a copy of the GNU General Public License along with this	*/
/* program; (See "LICENSE.GPL"). If not, write to the Free Software Foundation, Inc.,	*/
/* 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.				*/
/*											*/
/*--------------------------------------------------------------------------------------*/
/*											*/
/*		Joerg Anders, TU Chemnitz, Fakultaet fuer Informatik, GERMANY		*/
/*		ja@informatik.tu-chemnitz.de						*/
/*											*/
/*											*/
/****************************************************************************************/

#include "config.h"
#ifdef WITH_TSE3
#include "miditimescale.h"
#include "resource.h"
#include "staff.h"
#include "clef.h"
#include "chord.h"
#include "rest.h"
#include "sign.h"
#include <stdlib.h>
#include <stdio.h>
#include <alloca.h>
#include <math.h>

#define SEGMENT_LEN 1024
#define TSE3TIME2MYMIDITIME(t) ((int) ((double) (t) * (double) QUARTER_LENGTH / (double) (TSE3::Clock::PPQN)))
#define TRI16 (NOTE8_LENGTH / 3)
#define M3 (MULTIPLICATOR / 3)

#define TIM(x) x / M3, x / TRI16, x

NMidiTimeScale::NMidiTimeScale() {
	array_len_ = 0;
	alloc_len_ = 0;
	unrolled_midi_events_ = 0;
	midi_program_ = -1;
}

NMidiTimeScale::~NMidiTimeScale() {
	if (unrolled_midi_events_) {
		free(unrolled_midi_events_);
	}
	unrolled_midi_events_ = 0;
	array_len_ = 0;
	alloc_len_ = 0;
}

int NMidiTimeScale::getMidiProgram() {
	if (midi_program_ < 0) return 0;
	return midi_program_;
}

void NMidiTimeScale::removeEvent(unsigned int idx) {
	if (idx >= array_len_) {
		NResource::abort("NMidiTimeScale::remove");
	}
	for (;idx < array_len_ - 1; idx++) {
		unrolled_midi_events_[idx] = unrolled_midi_events_[idx + 1];
	}
	array_len_--;
}

void NMidiTimeScale::insertEvent(struct unrolled_midi_events_str *ptr) {
	unsigned int i, j, k, l;
	unrolled_midi_events_str *ptr1;
	bool inserted, equal;

	l = array_len_;
	for (i = 0, ptr1 = unrolled_midi_events_; i < l && ptr1->start_time < ptr->start_time; i++, ptr1++);
	inserted = false;
	if (i < l && (ptr1->eventType & EVT_CLASS_REST) == 0 && (ptr1->eventType & EVT_NORMAL_EVENT) && (ptr->eventType & EVT_NORMAL_EVENT)) {
		for (;!inserted && i < l && ptr1->start_time == ptr->start_time; i++, ptr1++) {
			if (ptr1->stop_time == ptr->stop_time) {
				inserted = true;
				equal = false; // Rosegarden produces some MIDI notes n times (?)
				for (k = 0; !equal && k < ptr1->U.norm_evt.num_pitches; k++) {
					if (ptr1->U.norm_evt.pitches[k] == ptr->U.norm_evt.pitches[0]) equal = true;
				}
				if (!equal) {
					if (ptr1->U.norm_evt.num_pitches < MAX_PITCHES) {
						ptr1->U.norm_evt.pitches[ptr1->U.norm_evt.num_pitches] = ptr->U.norm_evt.pitches[0];
						ptr1->U.norm_evt.num_pitches++;
					}
					else {
						fprintf(stderr, "too many pitchs\n");
					}
				}
			}
		}
	}
	if (inserted) return;
	if (array_len_ >= alloc_len_) {
		if (unrolled_midi_events_) {
			alloc_len_ += SEGMENT_LEN;
			if ((unrolled_midi_events_ = (struct unrolled_midi_events_str *)
				realloc(unrolled_midi_events_, alloc_len_ * sizeof(struct unrolled_midi_events_str))) == NULL) {
					NResource::abort("NMidiTimeScale::insertMidiEvent", 1);
			}
		}
		else {
			alloc_len_ = SEGMENT_LEN;
			if ((unrolled_midi_events_ = (struct unrolled_midi_events_str *)
				malloc(alloc_len_ * sizeof(struct unrolled_midi_events_str))) == NULL) {
					NResource::abort("NMidiTimeScale::insertMidiEvent", 2);
			}
		}
	}
	for (j = array_len_; j > i; j--) {
		unrolled_midi_events_[j] = unrolled_midi_events_[j-1];
	}
	unrolled_midi_events_[i] = *ptr;
	array_len_++;
}

int NMidiTimeScale::determine_snap(int length) {
	int compare;

	compare = DOUBLE_WHOLE_LENGTH;

	do {
		if (compare < length) {
			return (compare >> 1);

		}
		compare >>= 1;
	}
	while (compare >= NOTE64_LENGTH);
	return (NOTE64_LENGTH >> 1);
}

#ifdef WITH_TRIPLET_RECOGNITION
int NMidiTimeScale::quantTriplet(int l, struct unrolled_midi_events_str *ptr, bool *addRest, int maxlength) {
	unsigned int testlength;
	unsigned int deltamin1 = (1<<30);
	int i, shifts;
	int ret;

	if (l > maxlength) {
		NResource::abort("NMidiTimeScale::quantTriplet");
	}
	*addRest = (ptr->eventType & EVT_CLASS_REST);

	maxlength /= MULTIPLICATOR / 3;
	l /= MULTIPLICATOR / 3;
	for (shifts = 0; shifts < 9 && (0x3 << shifts) <  maxlength; shifts++);

	testlength = (0x2 << (shifts));
	for (i = shifts; i > 0; i--) {
		if (testlength > (unsigned int) l) {
			testlength >>= 1;
		}
		else {
			deltamin1 = l - testlength;
			break;
		}
	}
	return (MULTIPLICATOR << i);
}

#endif

int NMidiTimeScale::quantNote(int l, int *dotcount, int maxlength) {
	unsigned int testlength;
	unsigned int deltamin3 = (1<<30), deltamin9 = (1<<30);
	int i, j, shifts;
	int ret;
	*dotcount = 0;

	if (l > maxlength) return maxlength;

	maxlength /= MULTIPLICATOR / 3;
	l /= MULTIPLICATOR / 3;
	for (shifts = 0; shifts < 9 && (0x3 << shifts) <  maxlength; shifts++);

	testlength = (0x3 << shifts);
	for (i = shifts; i > 0; i--) {
		if (testlength > (unsigned int) l) {
			testlength >>= 1;
		}
		else {
			deltamin3 = l - testlength;
			break;
		}
	}

	testlength = (0x9 << (shifts - 1));
	for (j = shifts; j > 0; j--) {
		if (testlength > (unsigned int) l) {
			testlength >>= 1;
		}
		else {
			deltamin9 = l - testlength;
			break;
		}
	}

	if (deltamin9 < deltamin3) {
		*dotcount = 1;
		ret = (MULTIPLICATOR << j);
		return ret;
	}
	ret = (MULTIPLICATOR << i);
	return ret;
}



bool NMidiTimeScale::overlapping(unsigned int idx, unrolled_midi_events_str *ptr) {
	unsigned int i, l;
	unsigned int start, stop;
	unrolled_midi_events_str *ptr1;

	if ((ptr->eventType & (EVT_NORMAL_EVENT | EVT_PSEUDO_TRIPLET_NOTE)) == 0) return true;
	l = array_len_;
	for (ptr1 = &(unrolled_midi_events_[0]), i = 0; i < l; i++, ptr1++) {
		if (ptr1->eventType & EVT_PROGRAM_CHANGE) continue;
		if (idx == i) continue;
		if (ptr1->start_time > ptr->stop_time && ptr1->U.norm_evt.triplet_start_time > ptr->stop_time) {
		    	return false;
		}
		start = ((ptr1->eventType & EVT_PSEUDO_TRIPLET_NOTE) == 0 && ptr1->U.norm_evt.triplet_start_time < ptr1->start_time) ? ptr1->U.norm_evt.triplet_start_time : ptr1->start_time;
		stop = ((ptr1->eventType & EVT_PSEUDO_TRIPLET_NOTE) == 0 && ptr1->U.norm_evt.triplet_stop_time > ptr1->stop_time) ? ptr1->U.norm_evt.triplet_stop_time : ptr1->stop_time;
		if (start < ptr->stop_time && stop >= ptr->stop_time) return true;
		if (start < ptr->U.norm_evt.triplet_stop_time && stop >= ptr->U.norm_evt.triplet_stop_time) return true;
	}
	return false;
}


unsigned int NMidiTimeScale::findNextChunkEnd(bool *chukOk, unsigned int *chunkStartIdx) {
	unsigned int chunkEndIdx;
	unsigned int chunkStartTime, chunkEndTime;
	unsigned int i, l;
	unsigned int stop;
	unrolled_midi_events_str *ptr;


	l = array_len_;
	for (i = *chunkStartIdx; i < l && ((unrolled_midi_events_[i].eventType & (EVT_NORMAL_EVENT | EVT_PSEUDO_TRIPLET_NOTE)) == 0); i++);
	if (i >= l) {
		*chukOk = false;
		return 0;
	}
	*chunkStartIdx = i;
	*chukOk = true;
	ptr = &(unrolled_midi_events_[i]);
	chunkEndTime = ptr->stop_time;
	chunkStartTime = ptr->start_time;
	while (chunkEndTime < chunkStartTime + WHOLE_LENGTH) {
		i++; ptr++;
		if (i < l) {
			if ((ptr->eventType & (EVT_NORMAL_EVENT | EVT_PSEUDO_TRIPLET_NOTE)) == 0) continue;
			chunkEndTime = ptr->stop_time;
		}
		else {
			chunkEndIdx = i = *chunkStartIdx;
			ptr = &(unrolled_midi_events_[i]);
			chunkEndTime = ptr->stop_time;

			for (; i < l; i++, ptr++) {
				if (ptr->stop_time > chunkEndTime && (ptr->eventType & (EVT_NORMAL_EVENT | EVT_PSEUDO_TRIPLET_NOTE)) ) {
					chunkEndIdx = i;
					chunkEndTime = ptr->stop_time;
				}
			}
			return chunkEndIdx;
		}
	}
	while(overlapping(i, ptr)) {
		i++; ptr++;
		if (i >= l) {
			chunkEndIdx = i = *chunkStartIdx;
			ptr = &(unrolled_midi_events_[i]);
			chunkEndTime = ptr->stop_time;
			for (; i < l; i++, ptr++) {
				if (ptr->stop_time >= chunkEndTime && (ptr->eventType & (EVT_NORMAL_EVENT | EVT_PSEUDO_TRIPLET_NOTE)) ) {
					chunkEndIdx = i;
					chunkEndTime = ptr->stop_time;
				}
			}
			return chunkEndIdx;
		}
	}
	stop = ptr->stop_time;
	while (ptr->start_time < stop || ((ptr->eventType & (EVT_NORMAL_EVENT | EVT_PSEUDO_TRIPLET_NOTE)) == 0)) { // the next event(s) can have a later start time but a earlier stop time
		i++; ptr++;
		if (i >= l) {
			chunkEndIdx = i = *chunkStartIdx;
			ptr = &(unrolled_midi_events_[i]);
			chunkEndTime = ptr->stop_time;
			for (; i < l; i++, ptr++) {
				if (ptr->stop_time > chunkEndTime && (ptr->eventType & (EVT_NORMAL_EVENT | EVT_PSEUDO_TRIPLET_NOTE)) ) {
					chunkEndIdx = i;
					chunkEndTime = ptr->stop_time;
				}
			}
			return chunkEndIdx;
		}
	}
	return i-1;
}

int NMidiTimeScale::findFirstUclassified(unsigned int chunkStartIdx, unsigned int len) {
	unsigned int i;
	unrolled_midi_events_str *ptr;

	for (i = 0, ptr = &(unrolled_midi_events_[chunkStartIdx]); i < len; i++, ptr++) {
		if ((ptr->eventType & (EVT_NORMAL_EVENT | EVT_PSEUDO_TRIPLET_NOTE)) == 0) continue;
		if (ptr->voice_nr == -1) return (chunkStartIdx + i);
	}
	return -1;
}
	

int NMidiTimeScale::findLastUclassified(unsigned int chunkStartIdx, unsigned int len) {
	unrolled_midi_events_str *ptr;
	unsigned int i, max_stop_time = 0;
	int maxidx = -1;

	for (i = 0, ptr = &(unrolled_midi_events_[chunkStartIdx]); i < len; i++, ptr++) {
		if ((ptr->eventType & (EVT_NORMAL_EVENT | EVT_PSEUDO_TRIPLET_NOTE)) == 0) continue;
		if (ptr->voice_nr >= 0) continue;
		if (ptr->stop_time > max_stop_time) {
			max_stop_time = ptr->stop_time;
			maxidx = chunkStartIdx + i;
		}
		if (ptr->eventType & EVT_NORMAL_EVENT) {
			if (ptr->U.norm_evt.triplet_stop_time > max_stop_time) {
				max_stop_time = ptr->U.norm_evt.triplet_stop_time;
				maxidx = chunkStartIdx + i;
			}
		}
	}
	return maxidx;
}

void NMidiTimeScale::findShortestPath(struct path_elem_str* shortest_path, int fidx, unsigned int chunkStartIdx, unsigned int chunkEndIdx, unsigned int len) {
	unrolled_midi_events_str *ptr, *ptr2;
	unsigned int i, k;
	int minimum_costs, minimum_idx;
	unsigned int idx;
	int idx2;
	int pidx, len2;
	int costs;

	len2 = len;

	if (unrolled_midi_events_[fidx].voice_nr >= 0) {
		NResource::abort("NMidiTimeScale::findShortestPath", 1);
	}
	if (unrolled_midi_events_[fidx].eventType & EVT_PROGRAM_CHANGE) {
		NResource::abort("NMidiTimeScale::findShortestPath", 2);
	}
	pidx = unrolled_midi_events_[fidx].path_idx;
	if (pidx < 0 || pidx >= len2) {
		NResource::abort("NMidiTimeScale::findShortestPath", 3);
	}
	for (i = 0; i <= len; i++) {
		shortest_path[i].costs = -1;
		shortest_path[i].ready = false;
		shortest_path[i].prev_idx = -1;
	}
	shortest_path[pidx].costs = 0;

	while (1) {
		minimum_idx = -1;
		minimum_costs = (1 << 30);
		for (i = 0; i < len; i++) {
			if (!shortest_path[i].ready && shortest_path[i].costs >= 0 && shortest_path[i].costs < minimum_costs) {
				minimum_costs = shortest_path[i].costs;
				minimum_idx = i;
			}
		}
		if (minimum_idx < 0) return;
		if (minimum_idx >= len2) {
			NResource::abort("NMidiTimeScale::findShortestPath", 4);
		}
		idx = shortest_path[minimum_idx].idx;
		if (idx < chunkStartIdx || idx > chunkEndIdx) {
			NResource::abort("NMidiTimeScale::findShortestPath", 5);
		}
		ptr = &(unrolled_midi_events_[idx]);
		if (ptr->voice_nr >= 0) {
			NResource::abort("NMidiTimeScale::findShortestPath", 6);
		}
		if ((ptr->eventType & (EVT_NORMAL_EVENT | EVT_PSEUDO_TRIPLET_NOTE)) == 0) {
			NResource::abort("NMidiTimeScale::findShortestPath", 7);
		}
		for (k = 0; k < len - 1; k++) {
			if (ptr->decision_tree[k].costs >= 0) {
				ptr2 = ptr->decision_tree[k].next_event;
				if (ptr2->voice_nr < 0) {
					costs = minimum_costs + ptr->decision_tree[k].costs;
					idx2 = ptr2->path_idx;
					if (idx2 < 0 || idx2 >= len2) {
						NResource::abort("NMidiTimeScale::findShortestPath", 8);
					}
					if (!shortest_path[idx2].ready && (shortest_path[idx2].costs < 0 || shortest_path[idx2].costs > costs)) {
						shortest_path[idx2].costs = costs;
						shortest_path[idx2].prev_idx = minimum_idx;
					}
				}
			}
		}
		shortest_path[minimum_idx].ready = true;
	}
}

	
void NMidiTimeScale::findPathsInChunk(unsigned int chunkStartIdx, unsigned int chunkEndIdx) {
	unrolled_midi_events_str *ptr, *ptr2;
	unsigned int len = chunkEndIdx - chunkStartIdx;
	unsigned int i, k;
	int pitch_sum;
	bool endof_path;
	int fidx, lidx, pidx, len2;
	struct path_elem_str *shortest_path;
	int voice_nr = 0;

	len2 = len;

	for (i = chunkStartIdx, ptr = &(unrolled_midi_events_[chunkStartIdx]); i <= chunkEndIdx; i++, ptr++) {
		if (ptr->eventType & (EVT_PROGRAM_CHANGE | EVT_PSEUDO_TRIPLET_NOTE)) continue;
		pitch_sum = 0;
		for (k = 0; k < ptr->U.norm_evt.num_pitches; k++) {
			pitch_sum += ptr->U.norm_evt.pitches[k];
		}
		ptr->ave_pitch = (double)  pitch_sum / (double)  ptr->U.norm_evt.num_pitches;
	}

	for (i = chunkStartIdx, ptr = &(unrolled_midi_events_[chunkStartIdx]); i <= chunkEndIdx; i++, ptr++) {
		if ((ptr->eventType & (EVT_NORMAL_EVENT | EVT_PSEUDO_TRIPLET_NOTE)) == 0) continue;
		ptr->decision_tree = (struct decision_tree_str *) alloca(len * sizeof(struct decision_tree_str));
		initialize_desicion_tree(ptr, i, chunkStartIdx, chunkEndIdx, len);
	}

	shortest_path = (struct path_elem_str *) alloca((len + 1) * sizeof(struct path_elem_str));
		
	voice_nr = 0;
	for (i = 0; i <= len; i++) {
		shortest_path[i].idx = chunkStartIdx + i;
		unrolled_midi_events_[chunkStartIdx + i].path_idx = i;
	}
	while ((fidx = findFirstUclassified(chunkStartIdx, len + 1)) >= 0) {
		if (voice_nr >= MAX_VOICES) {
			NResource::abort("NMidiTimeScale::findPathsInChunk", 1);
		}
		if (len == 0) {
			unrolled_midi_events_[fidx].voice_nr = voice_nr;
			voice_nr++;
			continue;
		}
		findShortestPath(shortest_path, fidx, chunkStartIdx, chunkEndIdx, len + 1);
		if ((lidx = findLastUclassified(chunkStartIdx, len+1)) < 0) {
			NResource::abort("NMidiTimeScale::findPathsInChunk", 2);
		}
		endof_path = false;
		pidx = unrolled_midi_events_[lidx].path_idx;
		if (pidx < 0 || pidx > len2) {
			NResource::abort("NMidiTimeScale::findPathsInChunk", 3);
		}
		while (!endof_path) {
			if (unrolled_midi_events_[lidx].voice_nr >= 0) {
				 NResource::abort("NMidiTimeScale::findPathsInChunk", 4);
			}
			unrolled_midi_events_[lidx].voice_nr = voice_nr;
			if (lidx == fidx) {
				endof_path = true;
			}
			else {
				pidx = shortest_path[pidx].prev_idx;
				
				if (pidx == -1) {
					endof_path = true;
				}
				else {
					if (pidx < -1 || pidx > len2) {
						NResource::abort("NMidiTimeScale::findPathsInChunk", 5);
					}
					lidx = shortest_path[pidx].idx;
					if (lidx < chunkStartIdx || lidx > chunkEndIdx) {
						NResource::abort("NMidiTimeScale::findPathsInChunk", 6);
					}
				}
			}
		}
		voice_nr++;
	}
	if (voice_nr > max_voices_) max_voices_ = voice_nr;
}

void NMidiTimeScale::initialize_desicion_tree(unrolled_midi_events_str *ptr, unsigned int idx, unsigned int chunkStartIdx, unsigned int chunkEndIdx, int len) {
#define PITCH_DIST_COST_FAC 2.0
#define START_DIST_FAC 20.0
	unrolled_midi_events_str *ptr1;
	unsigned int i;
	int j;

	for (j = 0, i = chunkStartIdx, ptr1 = &(unrolled_midi_events_[chunkStartIdx]); i <= chunkEndIdx; i++, ptr1++) {
		if (i == idx) continue;
		if (j >= len) {
			NResource::abort("NMidiTimeScale::initialize_desicion_tree");
		}
		ptr->decision_tree[j].next_event = ptr1;
		if (ptr1->start_time < ptr->stop_time || ((ptr1->eventType & ( EVT_NORMAL_EVENT | EVT_PSEUDO_TRIPLET_NOTE)) == 0)) {
			ptr->decision_tree[j].costs = -1;
		}
		else if ((ptr1->eventType & EVT_CLASS_NOTE) && (ptr->eventType & EVT_CLASS_NOTE)){
			ptr->decision_tree[j].costs = (int) (PITCH_DIST_COST_FAC * fabs(ptr->ave_pitch - ptr1->ave_pitch) +
								START_DIST_FAC * fabs(ptr1->start_time -  ptr->stop_time));
		}
		j++;
	}
}
			

void NMidiTimeScale::findVoices() {
	unsigned int chunkStartIdx = 0, chunkEndIdx;
	bool chunkOk, removed, inserted;
	unrolled_midi_events_str *ptr, *ptr2, *templist[TLIST_MAX];
	unsigned int i, l, vnr;
	int j;
	max_voices_ = 0;

#ifdef WITH_TRIPLET_RECOGNITION

	do { // remove all triplet members there are copies of this events in EVT_PSEUDO_TRIPLET_NOTE
		removed = false;
		for (i = 0; i < array_len_; i++) {
			ptr = &(unrolled_midi_events_[i]);
			if ((ptr->eventType & (EVT_PART_OF_TRIPLET | EVT_LAST_IN_TRIPLET | EVT_FIRST_IN_TRIPLET)) == 0) continue;
			removeEvent(i);
			removed = true;
		}
	}
	while (removed);
#endif
	do {
		chunkEndIdx = findNextChunkEnd(&chunkOk, &chunkStartIdx);
		if (chunkOk) {
			l = chunkEndIdx - chunkStartIdx + 1;
#ifdef XXX
			if (l > 1000) {
				printf("s = %d, e = %d l = %d\n", chunkStartIdx, chunkEndIdx, chunkEndIdx - chunkStartIdx + 1); fflush(stdout);
				for (i = chunkStartIdx; i <= chunkEndIdx; i++) {
					printf("%d: t = 0x%x l = %d(%d)\n", i, unrolled_midi_events_[i].eventType,
					(unrolled_midi_events_[i].stop_time - unrolled_midi_events_[i].start_time) / MULTIPLICATOR, 
					unrolled_midi_events_[i].stop_time - unrolled_midi_events_[i].start_time);
				}
				fflush(stdout);
			}
#endif
			findPathsInChunk(chunkStartIdx, chunkEndIdx);
			chunkStartIdx = chunkEndIdx + 1;
		}
	}
	while (chunkOk);

#ifdef WITH_TRIPLET_RECOGNITION

	do { // insert the copies of the triplet members in EVT_PSEUDO_TRIPLET_NOTE again, this is necessary because of
	     // the array shifts, neither the indox nor the pointer to an event can act as
	     // identificator because the array insert and remove copies the array members
		inserted = false;
		for (i = 0; i < array_len_; i++) {
			ptr = &(unrolled_midi_events_[i]);
			if ((ptr->eventType & EVT_PSEUDO_TRIPLET_NOTE) == 0) continue;
			if (ptr->U.pseudo_evt.members_inserted_again) continue;
			l = ptr->U.pseudo_evt.num_triplet_members;
			vnr = ptr->voice_nr;
			ptr->U.pseudo_evt.members_inserted_again = true;
			memcpy(templist, ptr->U.pseudo_evt.triplet_members, l * sizeof(struct unrolled_midi_events_str *));
			for (j = 0; j < l; j++) { // temp vars because "insertEvent" shifts the array
				ptr2 = templist[j];
				ptr2->voice_nr = vnr;
				insertEvent(ptr2); // makes a copy
				free(ptr2);
			}
			inserted = true;
		}
	}
	while (inserted);

#endif
}

#ifdef WITH_TRIPLET_RECOGNITION

int NMidiTimeScale::search_for_event(int start_type, unsigned int start_time, int stop_type, unsigned int stop_time) {
	unrolled_midi_events_str *ptr;
	unsigned int i;
	bool ok;

	for (i = 0, ptr = &(unrolled_midi_events_[0]); i < array_len_; i++, ptr++) {
		if ((ptr->eventType & EVT_NORMAL_EVENT) == 0) continue;
		ok = false;
		switch (start_type) {
			case TRIPLET_SNAP: if (ptr->U.norm_evt.triplet_start_time > start_time + WHOLE_LENGTH) return -1;
					   if (ptr->U.norm_evt.sta2diff <= ptr->U.norm_evt.sta3diff) break;
					   if (ptr->U.norm_evt.triplet_start_time == start_time) ok = true;
					   break;
			case SYSTEM2_SNAP: if (ptr->start_time > start_time) return -1;
					   if (ptr->start_time  == start_time) ok = true;
					   break;
			default: NResource::abort("NMidiTimeScale::search_for_event", 1);
		}
		if (!ok) continue;
		switch (stop_type) {
			case TRIPLET_SNAP: 
					   if (ptr->U.norm_evt.sto2diff <= ptr->U.norm_evt.sto3diff) break;
					   if (ptr->U.norm_evt.triplet_stop_time == stop_time) return i;
					   break;
			case SYSTEM2_SNAP: if (ptr->stop_time  == stop_time) return i;
					   break;
			default: NResource::abort("NMidiTimeScale::search_for_event", 2);
		}
	}
	return -1;
}

int NMidiTimeScale::search_for_event_enlarge(int start_type, unsigned int start_time, int stop_type, unsigned int stop_time) {
	unrolled_midi_events_str *ptr;
	unsigned int i;
	int diff;
	bool ok;

	for (i = 0, ptr = &(unrolled_midi_events_[0]); i < array_len_; i++, ptr++) {
		if ((ptr->eventType & EVT_NORMAL_EVENT) == 0) continue;
		ok = false;
		switch (start_type) {
			case TRIPLET_SNAP: if (ptr->U.norm_evt.triplet_start_time > start_time + WHOLE_LENGTH) return -1;
					   if (ptr->U.norm_evt.sta2diff <= ptr->U.norm_evt.sta3diff) break;
					   if (ptr->U.norm_evt.triplet_start_time == start_time) ok = true;
					   break;
			case SYSTEM2_SNAP: if (ptr->start_time > start_time) return -1;
					   if (ptr->start_time  == start_time) ok = true;
					   break;
			default: NResource::abort("NMidiTimeScale::search_for_event_enlarge", 1);
		}
		if (!ok) continue;
		switch (stop_type) {
			case TRIPLET_SNAP: diff = stop_time - ptr->U.norm_evt.triplet_stop_time;
					if (diff < 0) diff = -diff;
#ifdef XXX
					   if (ok) {
					   	printf("diff = %d(%d/%d), stop_time = %d(%d/%d), raw = %d(%d/%d), ptr->U.norm_evt.triplet_stop_time = %d(%d/%d)\n",
						TIM(diff), TIM(stop_time), TIM(ptr->U.norm_evt.rawlen), TIM(ptr->U.norm_evt.triplet_stop_time)); fflush(stdout);
					   }
#endif
					   if (diff <= (stop_time - start_time) / 4) {
					   	ptr->U.norm_evt.triplet_stop_time = stop_time;
						return i;
					   }
					   break;
			case SYSTEM2_SNAP: 
					   diff = stop_time - ptr->stop_time;
					   if (diff < 0) diff = -diff;
					   if (diff <= (stop_time - start_time) / 4) {
					   	ptr->stop_time = stop_time;
						return i;
					   }
					   break;
			default: NResource::abort("NMidiTimeScale::search_for_event_enlarge", 1);
		}
	}
	return -1;
}

void NMidiTimeScale::addTRest(unsigned int start_time, unsigned int stop_time) {
	int i;

	if (TListPtr_ < 0) return;
	if (TListPtr_ >= TLIST_MAX) {
		NResource::abort("TLIST"); 
	}
	TList_[TListPtr_  ].evt_class = EVT_CLASS_REST;
	TList_[TListPtr_  ].start_time = start_time;
	TList_[TListPtr_++].stop_time = stop_time;
}
	

void NMidiTimeScale::addTidx(int idx) {
	int i;

	if (TListPtr_ < 0) return;
	if (TListPtr_ >= TLIST_MAX) {
		NResource::abort("TLIST"); 
	}
	for (i = 0; i < TListPtr_; i++) {
		if (TList_[i].evt_class == EVT_CLASS_NOTE && TList_[i].idx == idx) {
			TListPtr_ = -1;
			return;
		}
	}
	TList_[TListPtr_  ].evt_class = EVT_CLASS_NOTE;
	TList_[TListPtr_++].idx = idx;
}

#define COST_FOR_REST 500
#define HIGH_COSTS 100000


int NMidiTimeScale::findTripletParts(int T0, int T1, int divis) {
	unrolled_midi_events_str *ptr;
	unsigned int i;
	int idx0, idx1, idx2;

#ifdef XXX
	printf("(qu = %d(%d,%d), Suche Triplet von %d(%d,%d) - %d(%d,%d), divis = %d(%d,%d)\n", QUARTER_LENGTH / M3, QUARTER_LENGTH / MULTIPLICATOR, QUARTER_LENGTH, T0 / M3, T0 / MULTIPLICATOR, T0, T1 / M3, T1 / MULTIPLICATOR, T1, divis / M3, divis / MULTIPLICATOR, divis); fflush(stdout);
#endif
	TListPtr_ = 0;
	if ((idx0 = search_for_event(SYSTEM2_SNAP, T0, TRIPLET_SNAP, T0 + divis)) >= 0) {
		if ((idx1 = search_for_event(TRIPLET_SNAP, T0 + divis, TRIPLET_SNAP, T0 + 2 * divis)) >= 0) {
			if ((idx2 = search_for_event(TRIPLET_SNAP, T0 + 2 * divis, SYSTEM2_SNAP, T1)) >= 0) {
				addTidx(idx0);
				addTidx(idx1);
				addTidx(idx2);
				return 0;
			}
			addTidx(idx0);
			addTidx(idx1);
			addTRest(T0 + 2 * divis, T1);
			return ((divis / TRI16) * COST_FOR_REST);
		}
		if ((idx1 = search_for_event(TRIPLET_SNAP, T0 + divis, SYSTEM2_SNAP, T1)) >= 0) {
			addTidx(idx0);
			addTidx(idx1);
			return 0;
		}
		if ((idx2 = search_for_event(TRIPLET_SNAP, T0 + 2 * divis, SYSTEM2_SNAP, T1)) >= 0) {
			addTidx(idx0);
			addTRest(T0 + divis, T0 + 2 * divis);
			addTidx(idx2);
			return ((divis / TRI16) * COST_FOR_REST);
		}
		addTidx(idx0);
		addTRest(T0 + divis, T1);
		return ((divis / TRI16) * COST_FOR_REST);
	}
	if ((idx2 = search_for_event(TRIPLET_SNAP, T0 + 2 * divis, SYSTEM2_SNAP, T1)) >= 0) {
		if ((idx1 = search_for_event(SYSTEM2_SNAP, T0, TRIPLET_SNAP, T0 + 2 * divis)) >= 0) {
			addTidx(idx1);
			addTidx(idx2);
			return 0;
		}
		if ((idx1 = search_for_event(TRIPLET_SNAP, T0 + divis, TRIPLET_SNAP, T0 + 2 * divis)) >= 0) {
			addTRest(T0, T0 + divis);
			addTidx(idx1);
			addTidx(idx2);
			return ((divis / TRI16) * COST_FOR_REST);
		}
		addTRest(T0, T0 + 2 * divis);
		addTidx(idx2);
		return ((divis / TRI16) * COST_FOR_REST);
	}
	if ((idx2 = search_for_event(TRIPLET_SNAP, T0 + divis, SYSTEM2_SNAP, T1)) >= 0) {
		addTRest(T0, T0 + divis);
		addTidx(idx2);
		return ((divis / TRI16) * COST_FOR_REST);
	}
	if ((idx1 = search_for_event(TRIPLET_SNAP, T0 + divis, TRIPLET_SNAP, T0 + 2 * divis)) >= 0) {
		if ((idx2 = search_for_event(TRIPLET_SNAP, T0 + 2 * divis, SYSTEM2_SNAP, T1)) >= 0) {
			addTRest(T0, T0 + divis);
			addTidx(idx1);
			addTRest(T0 + 2 * divis, T1);
			return (2 * ((divis / TRI16) * COST_FOR_REST));
		}
	}
	return HIGH_COSTS;
}

int NMidiTimeScale::findTripletPartsSloppy(int T0, int T1, int divis) {
	unrolled_midi_events_str *ptr;
	unsigned int i;
	int idx0, idx1, idx2;
	TListPtr_ = 0;
	if ((idx0 = search_for_event_enlarge(SYSTEM2_SNAP, T0, TRIPLET_SNAP, T0 + divis)) >= 0) {
		if ((idx1 = search_for_event_enlarge(TRIPLET_SNAP, T0 + divis, TRIPLET_SNAP, T0 + 2 * divis)) >= 0) {
			if ((idx2 = search_for_event_enlarge(TRIPLET_SNAP, T0 + 2 * divis, SYSTEM2_SNAP, T1)) >= 0) {
				addTidx(idx0);
				addTidx(idx1);
				addTidx(idx2);
				return 0;
			}
			addTidx(idx0);
			addTidx(idx1);
			addTRest(T0 + 2 * divis, T1);
			return ((divis / TRI16) * COST_FOR_REST);
		}
		if ((idx1 = search_for_event_enlarge(TRIPLET_SNAP, T0 + divis, SYSTEM2_SNAP, T1)) >= 0) {
			addTidx(idx0);
			addTidx(idx1);
			return 0;
		}
		if ((idx2 = search_for_event_enlarge(TRIPLET_SNAP, T0 + 2 * divis, SYSTEM2_SNAP, T1)) >= 0) {
			addTidx(idx0);
			addTRest(T0 + divis, T0 + 2 * divis);
			addTidx(idx2);
			return 0;
		}
		addTidx(idx0);
		addTRest(T0 + divis, T1);
		return ((divis / TRI16) * COST_FOR_REST);
	}
	if ((idx2 = search_for_event_enlarge(TRIPLET_SNAP, T0 + 2 * divis, SYSTEM2_SNAP, T1)) >= 0) {
		if ((idx1 = search_for_event_enlarge(SYSTEM2_SNAP, T0, TRIPLET_SNAP, T0 + 2 * divis)) >= 0) {
			addTidx(idx1);
			addTidx(idx2);
			return 0;
		}
		if ((idx1 = search_for_event_enlarge(TRIPLET_SNAP, T0 + divis, TRIPLET_SNAP, T0 + 2 * divis)) >= 0) {
			addTRest(T0, T0 + divis);
			addTidx(idx1);
			addTidx(idx2);
			return ((divis / TRI16) * COST_FOR_REST);
		}
		addTRest(T0, T0 + 2 * divis);
		addTidx(idx2);
		return ((divis / TRI16) * COST_FOR_REST);
	}
	if ((idx2 = search_for_event_enlarge(TRIPLET_SNAP, T0 + divis, SYSTEM2_SNAP, T1)) >= 0) {
		addTRest(T0, T0 + divis);
		addTidx(idx2);
		return ((divis / TRI16) * COST_FOR_REST);
	}
	if ((idx1 = search_for_event_enlarge(TRIPLET_SNAP, T0 + divis, TRIPLET_SNAP, T0 + 2 * divis)) >= 0) {
		if ((idx2 = search_for_event_enlarge(TRIPLET_SNAP, T0 + 2 * divis, SYSTEM2_SNAP, T1)) >= 0) {
			addTRest(T0, T0 + divis);
			addTidx(idx1);
			addTRest(T0 + 2 * divis, T1);
			return (2 * ((divis / TRI16) * COST_FOR_REST));
		}
	}
	return HIGH_COSTS;
}


void NMidiTimeScale::searchForTriplet(int tripletLength, int snapedTripletPosition, int typeFound) {
	int i, j, k, n, divis, costs, minT0, minT1, T0, T1, mincosts;
	struct unrolled_midi_events_str pseudo_note, *ptr2, *ptr3;
	int num_pitches;
	int Tl, tyF;

#ifdef XXX
	printf("1/%d Zerlegung bei (%d/3), n = %d\n", 8 / tripletLength, typeFound + 1, snapedTripletPosition); fflush(stdout);
#endif


	MinTListPtr_ = -1;
	Tl = tripletLength;
	tyF = typeFound;
	mincosts = (1 << 30);
	snapedTripletPosition *= TRI16;
	while (Tl >= 1) {
		divis = Tl * TRI16;
		if (tyF) {
			T0 = snapedTripletPosition - 2 * divis;
			T1 = snapedTripletPosition + divis;
			if (T0 & 0x80000000) continue; // < 0
		}
		else {
			T0 = snapedTripletPosition - divis;
			T1 = snapedTripletPosition + 2 * divis;
			if (T0 & 0x80000000) continue; // < 0
		}
		costs = findTripletParts(T0, T1, divis);
		if (costs < mincosts && TListPtr_ > 0) {
			mincosts = costs;
			minT0 = T0;
			minT1 = T1;
			MinTListPtr_ = TListPtr_;
			memcpy (MinTList_, TList_, TListPtr_ * sizeof(struct tripletMemberStr));
		}
		Tl >>= 1;
		tyF = 1 - tyF;
	}
	if (MinTListPtr_ < 0) {
		Tl = tripletLength;
		mincosts = (1 << 30);
		tyF = typeFound;
		while (Tl >= 1) {
			divis = Tl * TRI16;
			if (tyF) {
				T0 = snapedTripletPosition - 2 * divis;
				T1 = snapedTripletPosition + divis;
				if (T0 & 0x80000000) continue; // < 0
			}
			else {
				T0 = snapedTripletPosition - divis;
				T1 = snapedTripletPosition + 2 * divis;
				if (T0 & 0x80000000) continue; // < 0
			}
			costs = findTripletPartsSloppy(T0, T1, divis);
			if (costs < mincosts && TListPtr_ > 0) {
				mincosts = costs;
				minT0 = T0;
				minT1 = T1;
				MinTListPtr_ = TListPtr_;
				memcpy (MinTList_, TList_, TListPtr_ * sizeof(struct tripletMemberStr));
			}
			Tl >>= 1;
			tyF = 1 - tyF;
		}
	}

	if (MinTListPtr_ >= 0) {
		printf("Triplet\n"); fflush(stdout);
		num_pitches = 0;
		pseudo_note.eventType = EVT_PSEUDO_TRIPLET_NOTE | EVT_CLASS_NOTE;
		pseudo_note.voice_nr = -1;
		pseudo_note.U.pseudo_evt.members_inserted_again = false;
		pseudo_note.ave_pitch = 0.0;
		pseudo_note.start_time = minT0;
		pseudo_note.stop_time = minT1;
		pseudo_note.U.pseudo_evt.num_triplet_members = MinTListPtr_;
		for (i = 0; i < MinTListPtr_; i++) {
			if (MinTList_[i].evt_class & EVT_CLASS_NOTE) {
				ptr2 = &(unrolled_midi_events_[MinTList_[i].idx]);
				if ((ptr2->eventType & EVT_NORMAL_EVENT) == 0) {
					NResource::abort("NMidiTimeScale::checkForTripletMembers", 1);
				}
				if (i == 0) {
					ptr2->eventType = EVT_FIRST_IN_TRIPLET | EVT_CLASS_NOTE;
					ptr2->U.norm_evt.triplet_border = ptr2->U.norm_evt.used_part_time = minT0;
				}
				else if (i == MinTListPtr_ - 1) {
					ptr2->eventType = EVT_LAST_IN_TRIPLET | EVT_CLASS_NOTE;
					ptr2->U.norm_evt.triplet_border = ptr2->U.norm_evt.used_part_time = minT1;
				}
				else {
					ptr2->eventType = EVT_PART_OF_TRIPLET | EVT_CLASS_NOTE;
				}
				if ((ptr3 = (struct unrolled_midi_events_str *) malloc(sizeof(struct unrolled_midi_events_str))) == 0) {
					NResource::abort("NMidiTimeScale::checkForTripletMembers", 2);
				}
				*ptr3 = *ptr2;
				pseudo_note.U.pseudo_evt.triplet_members[i] = ptr3;
				for (j = 0; j < ptr2->U.norm_evt.num_pitches; j++) {
					pseudo_note.ave_pitch += ptr2->U.norm_evt.pitches[j];
					num_pitches++;
				}
			}
			else {
				if ((ptr3 = (struct unrolled_midi_events_str *) malloc(sizeof(struct unrolled_midi_events_str))) == 0) {
					NResource::abort("NMidiTimeScale::checkForTripletMembers", 3);
				}
				if (i == 0) {
					ptr3->eventType = EVT_FIRST_IN_TRIPLET | EVT_CLASS_REST;
					ptr3->U.norm_evt.triplet_border = ptr3->U.norm_evt.used_part_time = minT0;
				}
				else if (i == MinTListPtr_ - 1) {
					ptr3->eventType = EVT_LAST_IN_TRIPLET | EVT_CLASS_REST;
					ptr3->U.norm_evt.triplet_border = ptr3->U.norm_evt.used_part_time = minT1;
				}
				else {
					ptr3->eventType = EVT_PART_OF_TRIPLET | EVT_CLASS_REST;
				}
				ptr3->start_time = MinTList_[i].start_time;
				ptr3->stop_time = MinTList_[i].stop_time;
				pseudo_note.voice_nr = -1;
				pseudo_note.U.pseudo_evt.triplet_members[i] = ptr3;
			}
				
		}
		pseudo_note.ave_pitch /= (double) num_pitches;
		insertEvent(&pseudo_note);
	}
}



void NMidiTimeScale::findTriplets() {
	unrolled_midi_events_str *ptr;
	unsigned int i, snapedTipletPosition;
	int typeFound;
	int k;

#ifdef XXX
	for (i = 0; i < array_len_; i++) {
		ptr = &(unrolled_midi_events_[i]);
		if ((ptr->eventType & EVT_NORMAL_EVENT) == 0) continue;
		if (ptr->U.norm_evt.sta3diff < ptr->U.norm_evt.sta2diff) {
			printf("idx = %d pitch = %d trime = %d(%d/%d)-%d(%d/%d), sta_kind = T\n", i,
			ptr->U.norm_evt.pitches[0],
			ptr->U.norm_evt.triplet_start_time / TRI16, ptr->U.norm_evt.triplet_start_time / MULTIPLICATOR, 
			ptr->U.norm_evt.triplet_start_time , ptr->U.norm_evt.triplet_stop_time  / TRI16,  ptr->U.norm_evt.triplet_stop_time / MULTIPLICATOR,
			ptr->U.norm_evt.triplet_stop_time);
		}
		else {
			printf("idx = %d pitch = %d time = %d(%d/%d) - %d(%d/%d), sta_kind = N\n", i,
			ptr->U.norm_evt.pitches[0],
			ptr->start_time / TRI16, ptr->start_time / MULTIPLICATOR, 
			ptr->start_time , ptr->stop_time  / TRI16,  ptr->stop_time / MULTIPLICATOR,
			ptr->stop_time);
		}
	}
	fflush(stdout);
#endif

	for (k = 1; k <= 8; k <<= 1) {
		for (i = 0; i < array_len_; i++) {
			ptr = &(unrolled_midi_events_[i]);
			if ((ptr->eventType & EVT_NORMAL_EVENT) == 0) continue;
			if (ptr->U.norm_evt.sta3diff >= ptr->U.norm_evt.sta2diff) continue;
			snapedTipletPosition = ptr->U.norm_evt.triplet_start_time / TRI16;
			if ((snapedTipletPosition % 3) && !(snapedTipletPosition % k) && ((snapedTipletPosition % (k << 1)) || (k == 8))) {
				typeFound = ((snapedTipletPosition - k) % 3) ? 1 : 0;
				searchForTriplet(k, snapedTipletPosition, typeFound);
			}
		}
	}
}

#endif


void NMidiTimeScale::insertMidiEvent(TSE3::MidiEvent *midiEvent,  unsigned int min, unsigned int max) {
	struct unrolled_midi_events_str unr_evt;
	int len;
	unsigned int val2, val3, snap2, snap3;
	switch(midiEvent->data.status) {
		case TSE3::MidiCommand_NoteOn:
			if (midiEvent->data.data1 < min || midiEvent->data.data1 > max) break;
			unr_evt.eventType = EVT_NORMAL_EVENT | EVT_CLASS_NOTE;
			unr_evt.U.norm_evt.pitches[0] = midiEvent->data.data1;
			unr_evt.start_time = TSE3TIME2MYMIDITIME(midiEvent->time.pulses);
			unr_evt.stop_time = TSE3TIME2MYMIDITIME(midiEvent->offTime.pulses);
			unr_evt.U.norm_evt.volume = midiEvent->data.data2;
			unr_evt.U.norm_evt.num_pitches = 1;
			unr_evt.voice_nr = -1;
			len = unr_evt.stop_time - unr_evt.start_time;
			unr_evt.U.norm_evt.rawlen = snap2 = determine_snap(len);
			snap3 = snap2 * 2 / 3;
			val2 = ((unr_evt.start_time + snap2 / 2) / snap2) * snap2;
			val3 = ((unr_evt.start_time + snap3 / 2) / snap3) * snap3;
			unr_evt.U.norm_evt.sta2diff = val2 - unr_evt.start_time;
			if (unr_evt.U.norm_evt.sta2diff < 0) unr_evt.U.norm_evt.sta2diff = -unr_evt.U.norm_evt.sta2diff;
			unr_evt.U.norm_evt.sta3diff = val3 - unr_evt.start_time;
			if (unr_evt.U.norm_evt.sta3diff < 0) unr_evt.U.norm_evt.sta3diff = -unr_evt.U.norm_evt.sta3diff;
#ifdef XXX
			printf("Rastung auf %d(%d/%d) 3er Rastung auf %d(%d/%d), qu = %d(%d/%d), sta  = %d(%d/%d) --> 2er = %d(%d/%d), 3er %d(%d/%d)\n",
			snap2 / TRI16, snap2 / MULTIPLICATOR,  snap2,
			snap3 / TRI16, snap3 / MULTIPLICATOR,  snap3,
			QUARTER_LENGTH / TRI16, QUARTER_LENGTH / MULTIPLICATOR,  QUARTER_LENGTH,
			unr_evt.start_time / TRI16, unr_evt.start_time / MULTIPLICATOR, unr_evt.start_time ,
			val2 / TRI16, val2 / MULTIPLICATOR, val2 ,
			val3 / TRI16, val3 / MULTIPLICATOR, val3 ); fflush(stdout);
#endif
			unr_evt.start_time = val2;
			unr_evt.U.norm_evt.triplet_start_time = val3;
			val2 = ((unr_evt.stop_time + snap2 / 2) / snap2) * snap2;
			val3 = ((unr_evt.stop_time + snap3 / 2) / snap3) * snap3;
			unr_evt.U.norm_evt.sto2diff = val2 - unr_evt.stop_time;
			if (unr_evt.U.norm_evt.sto2diff < 0) unr_evt.U.norm_evt.sto2diff = -unr_evt.U.norm_evt.sto2diff;
			unr_evt.U.norm_evt.sto3diff = val3 - unr_evt.stop_time;
			if (unr_evt.U.norm_evt.sto3diff < 0) unr_evt.U.norm_evt.sto3diff = -unr_evt.U.norm_evt.sto3diff;
			unr_evt.stop_time = val2;
			unr_evt.U.norm_evt.triplet_stop_time = val3;
			insertEvent(&unr_evt);
			break;
		case TSE3::MidiCommand_ProgramChange:
			if (midi_program_ < 0) {
				midi_program_ = midiEvent->data.data1;
				break;
			}
			unr_evt.eventType = EVT_PROGRAM_CHANGE;
			unr_evt.U.program_nr = midiEvent->data.data1;
			unr_evt.start_time = TSE3TIME2MYMIDITIME(midiEvent->time.pulses);
			insertEvent(&unr_evt);
			break;
	}
}


void NMidiTimeScale::createVoice(int nr, main_props_str *main_props, staff_props_str *staff_props, NClef *clef, int stem_policy,
			NVoice *voice, bool first, bool drum_channel, int volmindist, bool computeAverageVolume, unsigned int actualVolume, double averageVolume, double dynamic) {
	unsigned int i, j, l;
	int len = 0, len2, len3, newlen;
	unsigned int newVolume;
	unsigned int voiceTime = 0;
	unsigned int nextstart = 0, start = 0, stop = 0;
	unsigned int status, body;
	bool inTriplet = false;
	int dotcount;
	int voldist;
	int following_rest_len;
	bool next_found;
	bool triplet = false;
	bool addRest;
	bool stacatto;
	int line, offs;
	NChord *chord = 0, *chord2;
	NRest *rest;
	NSign *sign;
	QList<NMusElement> *tupletList = 0;
	struct unrolled_midi_events_str *ptr, *nptr;
	l = array_len_;
	voiceTime = 0;
	for (i = 0, ptr = unrolled_midi_events_; i < l; i++,  ptr++) {
		if (ptr->eventType & EVT_PROGRAM_CHANGE) {
			if (nr == 0) {
				sign = new NSign(main_props, staff_props, PROGRAM_CHANGE);
				sign->setProgram(ptr->U.program_nr);
				voice->appendElem(sign);
			}
			continue;
		}
		if (ptr->voice_nr != nr) continue;
		if (ptr->eventType & EVT_PSEUDO_TRIPLET_NOTE) continue;
		start = ((ptr->eventType & EVT_CLASS_REST) || (ptr->eventType & (EVT_FIRST_IN_TRIPLET | EVT_NORMAL_EVENT))) ? ptr->start_time : ptr->U.norm_evt.triplet_start_time;
		len = start - voiceTime;
		while (len > 0) {
			len2 = quantNote(len, &dotcount, WHOLE_LENGTH);
			if (len2 >= NOTE128_LENGTH) {
			/*
				if (triplet) {
					len3 = 2 * len2 / 3;
				}
				else */ if (dotcount) {
					len3 = 3 * len2 / 2;
				}
				else {
					len3 = len2;
				}
				len -= len3;
				voiceTime += len3;
				status = 0;
				if (dotcount) {
					status = STAT_SINGLE_DOT;
				}
				if (!first) {
					status |= STAT_HIDDEN;
				}
				rest = new NRest(main_props, staff_props, &(voice->yRestOffs_), len2, status);
				voice->appendElem(rest);
#ifdef YYY
				if (triplet) {
					if (!inTriplet) {
						tupletList = new QList<NMusElement>();
						inTriplet = true;
					}
					tupletList->append(rest);
				}
				else if (inTriplet) {
					if (tupletList->count() < 2) {
						tupletList->clear();
					}
					else {
						NMusElement::computeTuplet(tupletList, 3, 2);
					}
					tupletList = 0;
					inTriplet = false;
				}
#endif
			}
			else {
				len = 0;
			}
		}
		stop = ((ptr->eventType & EVT_CLASS_REST) || (ptr->eventType & (EVT_NORMAL_EVENT |  EVT_LAST_IN_TRIPLET))) ?  ptr->stop_time : ptr->U.norm_evt.triplet_stop_time;
		status = 0;
		len = stop - start;

#ifdef WITH_TRIPLET_RECOGNITION
		triplet = (ptr->eventType & (EVT_PART_OF_TRIPLET | EVT_LAST_IN_TRIPLET | EVT_FIRST_IN_TRIPLET));
#else
		triplet = false;
#endif
		stacatto = false;
		following_rest_len = 0;
		if (!triplet && !inTriplet && len <= QUARTER_LENGTH) {
			for (next_found = false, nptr = ptr + 1, j = i + 1; j < l; j++,  nptr++) {
				if (nptr->eventType & EVT_PROGRAM_CHANGE) continue;
				if (nptr->voice_nr != nr) continue;
				if (nptr->eventType & EVT_PSEUDO_TRIPLET_NOTE) continue;
				next_found = true;
				break;
			}
			if (next_found) {
				nextstart = ((nptr->eventType & EVT_CLASS_REST) || (nptr->eventType & (EVT_FIRST_IN_TRIPLET | EVT_NORMAL_EVENT))) ? nptr->start_time : nptr->U.norm_evt.triplet_start_time;			     
				following_rest_len = nextstart - stop;
				newlen = following_rest_len + len;
				len2 = quantNote(newlen, &dotcount, DOUBLE_WHOLE_LENGTH);
				len3 = dotcount ? 3 * len2 / 2 : len2;
				if (len3 <= newlen && len3 < 5 * len / 4) {
					stacatto = (len3 >= 2*len);
					stop = start + len;
					len = len3;	
				}
			}
		}

		chord = 0;
		while (len >= NOTE128_LENGTH) {
#ifdef WITH_TRIPLET_RECOGNITION
			if (triplet) {
				len2 = quantTriplet(len, ptr, &addRest, WHOLE_LENGTH);
				dotcount = 0;
			}
			else {
#endif
				addRest = false;
				len2 = quantNote(len, &dotcount, DOUBLE_WHOLE_LENGTH);
#ifdef WITH_TRIPLET_RECOGNITION
			}
#endif
			if (len2 >= NOTE128_LENGTH) {
				if (triplet) {
					len3 = 2 * len2 / 3;
				}
				else if (dotcount) {
					len3 = 3 * len2 / 2;
				}
				else {
					len3 = len2;
				}
				len -= len3;
				voiceTime += len3;
				if (addRest) {
					if (dotcount) {
						status = STAT_SINGLE_DOT;
					}
					if (!first) {
						status |= STAT_HIDDEN;
					}
					rest = new NRest(main_props, staff_props, &(voice->yRestOffs_), len2, status);
					voice->appendElem(rest);
					if (triplet) {
						if (!inTriplet) {
							tupletList = new QList<NMusElement>();
							inTriplet = true;
						}
						tupletList->append(rest);
					}
				}
				else {
					clef->midi2Line(ptr->U.norm_evt.pitches[0], &line, &offs);
					status = dotcount ? STAT_SINGLE_DOT : 0;
					if (computeAverageVolume) {
						newVolume = (int) (averageVolume + dynamic / 127.0 * (double) (ptr->U.norm_evt.volume - averageVolume));
					}
					else {
						newVolume = ptr->U.norm_evt.volume;
					}
					voldist = newVolume - actualVolume;
					if (voldist < 0) voldist = -voldist;
					if (voldist > volmindist) {
						actualVolume = newVolume;
						sign = new NSign(main_props, staff_props, VOLUME_SIG);
						sign->setVolume((actualVolume * 6) / 128, actualVolume);
						voice->appendElem(sign);
					}
					if (chord) {
						chord2 = chord->clone();
						chord2->tieWith(chord);
						chord2->changeLength(len2);
						if (dotcount) {
							chord2->setDotted(1);
						}
						else {
							chord2->setDotted(0);
						}
						voice->appendElem(chord2);
						if (inTriplet) {
							tupletList->append(chord2);
						}
						chord = chord2;
					}
					else {
						body = 0;
						if (drum_channel) {
							body = (STAT_BODY_CROSS << ((line + LINE_OVERFLOW) % 5));
						}
						if (stacatto) {
							status |= STAT_STACC;
						}
						chord = new NChord(main_props, staff_props, line, offs, len2, stem_policy, status | body);
						for (j = 1; j < ptr->U.norm_evt.num_pitches; j++) {
							clef->midi2Line(ptr->U.norm_evt.pitches[j], &line, &offs);
							body = 0;
							if (drum_channel) {
								body = (STAT_BODY_CROSS << ((line + LINE_OVERFLOW) % 5));
							}
							chord->insertNewNote(line, offs, stem_policy, status | body);
						}
						voice->appendElem(chord);
						if (triplet) {
							if (!inTriplet) {
								tupletList = new QList<NMusElement>();
								inTriplet = true;
							}
							tupletList->append(chord);
						}
					}
				}
			}
			else {
				len = 0;
			}
		}
		if (inTriplet && (ptr->eventType & EVT_LAST_IN_TRIPLET)) {
			if (tupletList->count() < 2) {
				tupletList->clear();
			}
			else {
				NMusElement::computeTuplet(tupletList, 3, 2);
			}
			tupletList = 0;
			inTriplet = false;
		}
	}
}

void NMidiTimeScale::sort_voices(int *voicemap) {
#define PITCH_FAC 100.0
#define COUNT_FAC 20.0
	unsigned int i, l, k;
	int idx, j, n, tmp;
	unrolled_midi_events_str *ptr;
	double max_rating, sum;
	double rating[MAX_VOICES];
	int num_pitches;
	int max_ave_idx;

	l = array_len_;
	for (n = 0; n < max_voices_; n++) {
		num_pitches = 1;
		sum = 0.0;
		for (i = 0, ptr = &(unrolled_midi_events_[0]); i < l; i++, ptr++) {
			if ((ptr->eventType & (EVT_NORMAL_EVENT | EVT_PART_OF_TRIPLET | EVT_LAST_IN_TRIPLET)) == 0) continue;
			if (ptr->eventType & EVT_CLASS_REST) continue;
			if (ptr->voice_nr != n) continue;
			for (k = 0; k < ptr->U.norm_evt.num_pitches; k++) {
				sum += ptr->U.norm_evt.pitches[k];
				num_pitches++;
			}
		}
		rating[n] = PITCH_FAC * sum / (double) num_pitches + COUNT_FAC * num_pitches; 
	}

	max_rating = 0.0;
	for (i = 0; i < MAX_VOICES; i++) {
		voicemap[i] = i;
	}

	for (idx = 0; idx < max_voices_ - 1; idx++) {
		max_rating = rating[idx];
		max_ave_idx = idx;
		for (j = idx + 1; j < max_voices_; j++) {
			if (rating[j] > max_rating) {
				max_rating = rating[j];
				max_ave_idx = j;
			}
		}
		if (max_ave_idx != idx) {
			tmp = voicemap[idx];
			voicemap[idx] = voicemap[max_ave_idx];
			voicemap[max_ave_idx] = tmp;
		}
	}
}


void NMidiTimeScale::createStaff(NStaff *staff, bool drum_channel, int volmindist, bool computeAverageVolume, unsigned int actualVolume, double averageVolume, double dynamic) {
	int i;
	NVoice *voice;
	NClef *clef;
	main_props_str *main_props;
	staff_props_str *staff_props;
	int voice_map[MAX_VOICES];

	clef = &(staff->actualClef_);
	main_props = staff->getVoiceNr(0)->getMainPropsAddr();
	staff_props = staff->getStaffPropsAddr();

#ifdef YYY
	divideOverlapping();
#endif
#ifdef VOICE_DEBUG
	outputDistribution();
	printf("%d voices\n", voice_count());
	fflush(stdout);
#endif
#ifdef WITH_TRIPLET_RECOGNITION
	findTriplets();
#endif
	findVoices();
	sort_voices(voice_map);
	if (max_voices_ > 1) {
		staff->addVoices(max_voices_ - 1);
	}
	for (i = 0; i < max_voices_; i++) {
		voice = staff->getVoiceNr(i);
		createVoice(voice_map[i], main_props, staff_props, clef, (max_voices_ == 1) ? STEM_POL_INDIVIDUAL : ((i >= max_voices_ / 2) ? STEM_POL_DOWN : STEM_POL_UP), voice, i == 0, drum_channel, volmindist, computeAverageVolume, actualVolume, averageVolume, dynamic);
	}
}
#endif
