package keys

import (
	"crypto/x509"
	"encoding/pem"
	"fmt"

	"github.com/theupdateframework/notary/tuf/data"
	"github.com/theupdateframework/notary/tuf/signed"
	"github.com/theupdateframework/notary/tuf/utils"
)

// CreateOrAddKey creates a new key in case of ECDSA or ED25519 inside the cryptoservice with the given role and gun
// while in the case of RSA, it takes a key from the map that includes multiple pre-existing RSA keys and adds them
// to the keystore, returning the public key.
func CreateOrAddKey(cs signed.CryptoService, role data.RoleName, gun data.GUN, keyAlgorithm string) (pubKey data.PublicKey, err error) {
	switch keyAlgorithm {
	case data.ECDSAKey, data.ED25519Key:
		pubKey, err = cs.Create(role, gun, keyAlgorithm)
		if err != nil {
			return
		}
	case data.RSAKey:
		raw := map[string][]byte{
			"a74d6c5de798125fb9668fa6dda35039e1bc646ccc6694bcd5dfb4406e32660f": []byte(`-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEA1tQPiItvq67+XPbCxcIOY3JQGU7Yksor5j4rGX0nEeDtHO/d
A6kBYJxCnjKFByJUvtjz8qTNpFQHc4wnCgHQhh6LjVcbWtUdU3ulfPrh95YmP1xn
dvkV44NeN1WE6BXutyTFfooCoBWiTIVOnFHx7xLmhfbRtncjw/O2WOV7pty185dT
HndwQ/9TElwG3256KwLAyWbXYpn+IyJU5qZXrTrhRL6Oapj6g9jyGy+llj4eJDdF
ky3rG2iuwNGF9A5X0Gqajh3l7lIKMDreh+UW37MmL+ZD+aJSltYONT31laG9aPv3
U40yvHSuPUL87CFLu/8GiZ9gsQcumrXVECxEmwIDAQABAoIBAGF+T6ZaEU3K1uj9
m5r/3GPJRwyVObg0RAPzhr/EimyUtUcCnDzDADsF7z24E6MPM4KBFwSxQhongFIx
8q2Z8TZMumxkcaehO/PdE1dKzC3+WFE88D+P/x0HS0S1bOGewn0fFPm+TD62UreD
cA+FKNSksTVus/Go0/RjsAdb7eflQRz0sNyASon9gk3cgi76TVdsQzYRUGAjAdqI
3Snm/T/Wvv4RC4RMdKx5SzpdEENnISyHAeijWONOISGVtO61OfXjY3JoorlyZGFt
YN+gs5+bemrCck+tKlkOczRMVCvK/uoXi+tRbqQr5X5R02wDXNiFX6AEmxQ+C9lA
j2MNrGECgYEA9fEeNQuPKny8Dm3K25edN9RXEdPTWP+8m84H5i4JKXyboGp6JwQ/
ZT9vcIYFRgVw2HXWf/AbSxRZ3Qb501xlYVNc54gLvwIABpcVgfLCpDoOhQHjyrWA
yZnO2kk7tsz3xESwE5Tdf2vqWwqqKPNOZcGZhcqVvYfewK3FjupS7ikCgYEA350z
XXK7UOWLBDsV5OTc8fcmTGKO14dZGzjvDdSEiUh26X+HM1khF94E8L20IyNSTXKt
AyGJ11DMBbvrGBqg0IU8cWcpQvt+f5SEazBuoR/FoUftdmhNuZ44R5vuEMj7Im42
Y2tAkAch1aYdEtvT1nf510FirftFYj4ncRFErSMCgYBitS45ffUsyTk01U3oilY4
NQnpBso5WxKYub0j3fAhbXiMjo+OHz9OZIJWeAOf0ckzpz+6w2RA4t8aPZWBBgFO
aYXNCyYjuCrIelTQfd0uynpPWa3SCrywrQF/TIsh1+vNIwBEK6gmXuqFNaeLnfMm
FeiG8QLeOazmShgTPuqf2QKBgQDKsCc68SAw95/GuffifkB3YtroKf3oBtsR/c1R
7+wis/h1Ng9Kz+NbSveXb5y0rPORF/0S9XeIH8iO+gLi2mAiImLjN+sCuAu1gN12
e3QLwEzXH+dyiKDLO0swPXrQ5Bwe3V+XRkRDSPO2QNfhnDisEhv4rV/Q7A28OzJt
HoXtmQKBgGeCCwIUO5g21BuooVS/sk2d7iqpMAOTtcXUlU1Cddj2jED6bwt8bBPe
hEsmHnN4VECR1Mcl2IWUJzEeTqfXCHU6kwQqMZJEJ/S0+PD+WIefQ89ypMhhFcxc
a+nu6Ky9n0O9DyOVe8FkdhQ1N2mDdvqHvPAm8ZROvxxdWn04oJ+e
-----END RSA PRIVATE KEY-----`),
			"04b7ff29e1072b92afbefa8718adaf1d9244aa2666022bbd7cf8cf0c9d0c5aa5": []byte(`-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEAtdDpJF86XdT4Kt+ce8PLoEfMK+WU8NV3c+J5+VIa50a1Y+Ix
C61QZmOlVWnwBt1mwX5bFGL029eHFcoLbf5inpF1An05f9fTHA/pfp8OmhQZ4UKm
4ECkKvn3r0DsGkNvxFjUvPcSn8LUstg+iBN0T+LFpNrD9gyjYtnfSgphRES+QUU5
Hi0jbj3euPtvyeZdsEJJnpGwYRbPln2pW62VpbvGPHNBZIccmIxleq4XEA/MUwVz
6qNDH1ozrKYRBwYmgBEuZgDZOb5ACGxb0QKG1ekFVZisTW9z0EhAG7OPFoE4awZH
ke3iE3quteVdDUQDgwBWa/K5RBB8KX17dvqGbQIDAQABAoIBACv9xTtgsy2a5wTw
n9suT6mTVapVXUk8TS/D34FcxVeqOx9Uzksl/ysW9x0IieQPU9qDaSQwoVBEANJb
1ZIil9gIF8inZe8x2EmU7AkQESdkr7JaPyhJW69uVeiYi8f7AyL931NtcJspN++h
9NdV5bGii/HMYNe6zRmQ3775bflnQ7zJFzn824s8tph4JIWZHAgSp9AUKSqXq/nE
wz3x6pRCGiCaifmZdcaDqdtEOQwGJEVAwML/E80idEYghWc3EdiYxOA27BN+Y8sJ
JDs55q6eHQNCEWa61AY8kOq25U6efK/eYHgJt06KkpGxfkyJZXhVefeXPlJ4Kg/j
KDvF2IECgYEA7hQe8tdKWsEfUbVXTEkJPvqLs7c5MjFcQMV79En55X4a+eEq/zB7
iTMeKUh0eHtWCqgUT+hWgDDqHZWYFs351OzLOjmFlJAq+QsEjBW/kojPuNUfJLVv
T2BsxL/Uxqa1MO7dzwnUAmm6XBZS5q+//+fyGpQBJQHKlFKCuVJFuWECgYEAw4CW
AkrCV2gIpU71YHwK5R7E6ErTxEn3QFDXZJg2rKYPJ4czOrnsO9BY3slVl6rO1hFY
RShB9tT1QQIiiWUInuEo1oh5laKslkBfgkwrqNm8KxwGDhhYDU0W02fBZmZ4Cq0C
elVGnr4kuOjE8hg6s+fa13UOdnHsVBo/zjhu7I0CgYArsc2sTA79cQxKdrm66nyj
7l/NcAczSxPfX19Mincuw2I4EjdN9pv1pkgW05Fsu1YjRo3TzJRzmoadP2NWbB86
EOuLmBsIFR2N75O6EhPC1REqnxLELfhr3QqZYYuvmdEPVgeng+pBNx5Y4E/+vqrC
UNOCRT+ghx+i8THEyXC6gQKBgQCOAbT07UDFJLhC+1uZPgDb9MGqyc+Q4PlMKmyK
kVYFiomveLFpkutZdzlWeoBFrriXHfo0Ic3ocT/EfLyBeHYOkhaL3o4SdxtN+Gxn
xNRhzea0maJ8UoqC/t/5p3//1aaUep82JQMrA4rBFbMY9yiMzXlUz541wwKbBIYc
UTmhcQKBgQDPPo6p1eMDBlsiCreoYVBdOrCL6ph0QgUZTgjY3xDVaknFpPJDEAS6
yLNj0U/d429yEIahGkR+7/j9rAeSLCBirAYZNOhNHa/vS6d7EIuxxP+64BAqFmSR
ur5VKj9xa4qe51sCCfWYSL8HzrYagb04QEUjcWRDTUqgDwgGBJ4E9A==
-----END RSA PRIVATE KEY-----`),
			"9e402656443f75520f70c015f58aec29c08ecce93835937b6d2faa95fbfaaa4a": []byte(`-----BEGIN RSA PRIVATE KEY-----
MIIEogIBAAKCAQEAv7j5qMh2HS2tUMfB8YZBBuRS6GTdE//QtPYCFrVeJbmmOaf5
1xQQHQXuYGjIGSWAhsbcC7V3zF1Ya8we8fYjzPvN558MQkeJIEx/usgPcvGFywfL
QVizJ+l3FU7EW3IlshiS0lSBYMWmZ7kHsWlElycAnJSwuyR5ei1oFH4nnc/3LvF4
s3Vgxb2ga0t7Id3JSuc/nkLsDNo0ojONy02/LPRXMn7Zl9CYwPTfGEwk90Jpp0Ko
KrCNSbW2y7Cdb8NBuqgD9szqQJC5aXhQM3UZZUFiIQtMcfSGCvb1nuBD0g2/O38N
Iy0i4p7CvP9QPkBnUtARguF2V3TS7j02oh6PfQIDAQABAoIBAGhC98JKAbGsfWxI
tbfbAtH0qQGomHGf1KgUgN5Ik4KAi10fdEb6qiieaRwU33yNYG/rBQ95CEUjHkrB
HrMG4EFBTqB/ZfNFY9UkxPGWipA/uDrJakeHLSavWAtGYpI4aTeJFY88VX+C/RO7
E+vvC1YH59U/Jz0egNFokAlU8cZnuVSsA5dnUfmdtShrN8iTT3TiZ8gUoMYmYqQ3
sO40BTk6IglthbyOWc9lqlmRy80D7hB6/Z7SxuX0mFSdvYIdACcwmVqE//8unkl9
Ea39MRCnvC6TU+niGWkdjAYPyv7kbvZQOyb5pqoM8WTNKsiChfrrzlXZE4axswx7
WpDQqgECgYEA34qtY5Qfj2rx+iTrhm1ul5hZneR6ueaWSOauRdN3fkltXJTu6H0B
5aiWxHPt2Si2ZftTlws7s+IPDpNz+Jke2QlgWiKVZAcqkbqRlxy5dA9qosb/fHl3
rDG3Y4WJZ/8xEF1GxmFxhZVZl95x8Qb4kkSUYgzCHzVWJyZ1hYBbOJECgYEA24+K
an0YQ7NKpxfE6vdbbSu6/pGkC5JNgvUyfyXxz7SVjR80AZTyDSgdyjLZHVPOiPSP
Z1zTGxSXU5/qOdeolmdC8o4uQS40H1moo790Obi6U2PRhsg7jHrJKK9ZYSYED53s
h2iv92DIKNh4sCQqImt7zj/++oD0XYcz2Fl6vi0CgYBBZOw8SUTCS+Ll7o8IEFNd
/a+ZxREx+QbV5MXun2JBi3F2uAr78DXx3VPdvJxD/uj9BHz7nsmkYelNtk0i2Ihx
rZCGTb+uN3p4jz+wGTMXhLsJS3XKEC62LQUdFB+SUNElnGPr/O2cyogfKW/jtspY
378Yv+q5/nxpj5Kpr1cAQQKBgE+zZX2Od3Zr6ReKy1K4BBo3G6Q5XiFNNJHrTC9X
qNXLKdaWkxxX4x3fiCDUIHhC6eBRVpUg3wkfimcyz6W7oVknlufRsPy8NvOCanM2
low1gH2w6GhS2zr9f3QjYwCe+i5pF2l4GRhlq4MYQYc/OYRRHBocpQRuN0aUQ7+v
i5zpAoGANNkC0OiPpQUHRWEl8/6lSiwz4Lx2R0AbYoWWxJDg7LXvyhcA8JsW33Ds
eeO/FNwwDaxBnaG+O9/mGFOo34HXpk62foY+7j4tJDQZtVO3HuaHxqHdG4YISJSu
EL5n+XhfAoS9XHG5IAMygp9WuuKOZ8jBoL+dv+z1wLZrrnr7g4Y=
-----END RSA PRIVATE KEY-----`),
			"543a12911525dfa1ba6a303683f76efb03b9b138aabbf26f7334ec0292119190": []byte(`-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEAq97AwGpTwXutFQKhUVAmiqMFSB2N3vfSIg1p/NYyWquIbkEo
QuIZAEdQFb0CC5T+wS52E4/n7Qs+vS0us3qz98KtYryat0USygcCsoCx+fLhU7M/
C6uYH5u+dexBdW/TKUk+84AvpF/2seYbKK3Z8S0GPYdEBjxreJ3fO5J281b2fOmh
hQPdWvmQ8f3xctd9Ko3UvoIj++YhtekaGrk6VDl/p8uofuERpzfmdAZvimu5vv8z
I1S0rI9MfNU5UqBblIMY9aT7Kwd8+upHnCtmluYwsVK7Yfa5VxdxmLORxbQgCUtb
9nlP05QSumV5Ari0rE9KEJyzdppp7DcZu2AJnwIDAQABAoIBAHthWpAHBH9mUHc/
vxzHZBOsOdNuW/VPhIsrhAWcouNSNounvMZe10HK7+PbHoe/+y4lM2CRsxYJN43I
zLjNgjp/h8bUzGB6dqWPaNFJ7/oV8MoMaFrxYTB3tr+izc8Hatg9m7JkOE9REKDf
JmoKe+VVxCARZbTwVNpuwt1aJIhEnVysVdrR5k8ATajS6KH8ybfMZ9ydXM/nNk74
1ysbAEv0UHNLzE6h/PuzUrh6ncp7S7yvGUYoMEHDI3bNnuI1T1GapgNb9oUdUxgD
X1iUlYxf74HGA5NcenixnBR8AGEv1M/CYpaH9TiBvHzr6U8GI89GtYZUU3uH++vV
9ojbUgkCgYEA0zCUnpr/smAqft0LdgZBvG74DFQLB8vy0qCK0v0ccdsCupiQtlhY
wF5PvdO3EO83Q+4LkIh+51LJH68prrw+20/fukJnwqPp6vHsX3tmVnuRt5eu9l81
KTWuqtJQsv3Ht7IwGaHKKq+RKuem0LuwoKBS/KB0q6YYr0VtttwvKZsCgYEA0FZn
2377AQFsGBEPGIW+Wj3NQQdz86iXQZO156F/1pl6WW5uWkeU7WCmvYOT6bmkua5Z
1zG6KccrsBv+HwosDvq03E2dxekQ//AiFDmgVqNSzDlcLPSFbQnG6GyBWRt4FQCk
X808E+1WM7ilUjIFHn3yXLkC+zUl0mueLJ9U8k0CgYEAwT5yv66UymO0fuFE23pe
LE5eKum8LiYwywqb/PQeUxBULunrgO7WRzAFR22y1YwOxdnMOzSq6oCWCKf/PCSE
X6l2zrh0uWXDZXh5EeWYfwoTDCt/rXDw9ARrD2Pi8+XEGhKycMHae7Sq4NeXjOgx
dsk9JmysVWRmsfqVqf1QKoECgYB1a42jh2DewzdyJQM2FgMgB18MvK+cb7O4UREg
GMTdZakwiG4H0fpdA7FWdLdD7s6pnFukTrGW5Ft6KXzDb88/LgwLcm9u4Roktenv
Lj9xNSHwWo/aAoLE3tLNooIQf8vhQnhiSmEG9ig0hGgg+HIsKl4T5m6IuP61JtSC
X9bXrQKBgBzpN72BHf2KazRoaneAvz849MNTiQ+tebJ5nSchYE+nS2oOjK0w35od
iQjgemdiPxdls4+7mkaPP+IrMOQjkpU7Fn+1KkxAiill27X17ZGx5gSbX9elBX1r
P60ipeh9wyzP45LEiZv+w9Xhq5PjUOYkNxYqZa6M/eYoFc10ojrP
-----END RSA PRIVATE KEY-----`),
			"23a6e01948f38fe16828ae6c5daebba95ef1bd405ce4e5563de1bbb9bfa4874d": []byte(`-----BEGIN RSA PRIVATE KEY-----
MIIEogIBAAKCAQEAxYt6ZCe1dfYDjIvnMgDmr7H+PJUWwnYtNSyXj/ecpS4MoQ2J
RhiH+LNgWryFh9+VweSam0mk2/R+ms0uBUUCcGXJZeRljMkY3qvQLSYRZVA3BrLO
OH6QlV2oDq+Xu0pBT/NqspO7VzPSdNnmM2ZQMugehPxHsrIdtyjlalQcMZa0s2gR
4qnYFaMjfmspw5ioreulrkyAuGkihpznSM7fG89ccyREQdrU6J7ah1N1RXe9WwiJ
pAj5p4slfNYXH17dwdkVLVdu6PHUR/f1LeAmeobe3Y7nXSsSYa927IdoV0rIemxw
1dyJ4chbwCCErbX8ZvcmxAuAxG91WIZAisJLyQIDAQABAoIBAHngCH2cJg8F3cCr
W9L0zV4eMV1NGgKViGSjA0r2GDWsiGwgfs25OsJMYKj6YGOopKtAD8a4k7aETEvT
RI5JuiyAD9YGKETImMhgNgp0zwyzhY9jZ2lhZHBSnOmamcEaGdtqsSxFirJNLY2c
FbMmb1pI7PNc9hsgWsbyuz9zbwWD/+HIYyng2VKFgktWu4N83UmyLwoN4OhQD3kl
JWgRYDztMEalNjDhICCsfoAvo3R/m2F/7ih6hiLbARG0W0gK7I2QTnqDTg9ERIk1
pDtvO5CaWPa7Fn490ux8bhkVtpF1Hor4NaWAf9gtEomVl/ef5tTNNn3N22rU+qZW
7EDjDtECgYEA8sZdeShdNQk5RhOe02iPBamDl8HGIKK6ZkeXBWzzkKr4n+kAj5G+
4ZuF7t2n33/e1AWRfpkTgYA80VWgbkIpJvmCf8LEHb35qAA8KGOfD8s3dpGqAArI
6BOW+MB6q5+W1guM9Toi5ImE7Ql1KslNCyOuOTFQM2y5GwOo/ueNcEUCgYEA0E5a
sJ75AnrzAwopMgsU4GWuGD4hsvF9/APvEQbfvinxVwp+fmHlBmEOd1l5npJc27oT
iKGPn7dovnqv7tMAkF3TLiN+VlmFyKBVvyUwqjCeQX8sHlPU07UrbTNTZ6JjCYk5
4Sqx61dVTTqHDKQDcjryMQQLKnByY00ZKTMEb7UCgYANpK4uF+GdEGailImxccfP
1qXpil87CEoXY2COUoAhmiocunhomRU0EoxlTxmXCtWX7rQtfCbHeVBxpzE74QX1
BcKasXRaA4obMeAvmOGgfpggVrFKDVEbEbbDRfXe3ToZz05dnaXGWCAbPhPSkrt8
q8Wnuyi5qxt/9CW0JYkiTQKBgBSb6/UVhWVtbKWcV78PwZ0B8luLqiHeKqzIZtq0
Z2aVF7+Y1jMC1HnxGFFOlO3PZUiH/G7ELUWMC9zsnoNudNvTrJUgBdvdN5NF6bXj
pBDyhnXMczz85k2b2xQOw+247zt+TC+u3bN/DH17kOj7QSwIEPYwhDpKrcaNVjPx
TDsRAoGAWJ3dC/t2Y0MAYmQX1TPZ2r9aOapM7VYMo71rDQMYQCo7l1+V4qzfmeC2
3gTqwibiPBcBZsezmKpoI7iVEN71++RZJzac4ZsHGDfIOYWoo9T6oekPJL6t1fx2
6HPxab9yp7M29XnFZ5UU7UCai4FEObgrtE52dL41C0m2lF9MXO0=
-----END RSA PRIVATE KEY-----`),
			"314712e2ce66160ce8cb278f0e25e14a7b53fe72368b948d798cafe6d23332a3": []byte(`-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEA3jn7PpMWqSSOo0WWjQ2LABdbymxx5FnSuQiF0g78+dF1T0WM
iQJj2rb9PB+kD3ZWGQ5LYf9pW2BoU8cFN99WxHwPrB0PWGhD7y2BN/BLy7XTNj62
ICzeKZnNjE2DVHiwueECNtJ2505lX4mRhRmQGwL/LCKBg4GOeIRPTmbR/iJ55PaT
dtGva11woL4WyXIgC0s0NVNHHev6lNYD8ZQkSHfgUhhYNQTVIduh2Y+loqftY1LM
/Q07oOmQKwpHOzMraIwr/zk1cfXHzBkc00ma1wo8nzYAdE5oajO27LJSaQigRH6+
j7m7mt0S8aCzFJhdZdI2OT8GoVcVh0ekIYiUHQIDAQABAoIBACqaSwRQBoQesgmz
CY25DvDRiZNg/uScI729WIpQvUbXVPoBh1Os1Z/UfO/N0RH1ycO77c+3u2b9MIL/
IIz5kBTTA94nJA36zytHOgKTWrSPAxLfKuM4Ns8a4UEIiyjiil7YOChyaMEnUl/i
a7dskyfahzja1KF1cO7BzsJlC9UAQ5FvkFPPzc5jobsLD+e9WVxkI+/tKsLUS+AB
/F4DN311OcsQ93GX4WG7a6NaxXe1HiSk5YHD7rGPKjwXcxrMbbtkbMBwjZuHvmiH
iIbiOdKa0hFQ0dS7CXimn4p9/Z5Fw8V+nZR6zULZb9a8x4fIiCXiTl4mkiW9f/hc
X90lhgECgYEA8Hxgc82+MhSRzOdCe9xzOx7skw7P5yM1BshH7w1zC40HJKyVITcy
l5E/B/YT3MOzpMOaA60zftry4NpRpiQynOa8KT7ruvC1pw7IfLd5KuyYy5rXR9wO
XwhM9nofunFnfrkOpwKeOxYgzUagb9SCaCdXwmntCyYnmKUZ8tP9byECgYEA7JAN
KceOMyB8QOOFpsSgVCOZD2wYEAZ4Ynk4qLywwVj4c0QkYELeKSgHBAe0AThkOw7+
GfIC9+nzLzVNTntXbYCLykCPstJRcs6irVgm0M465we7+kEbJw5HFYN+dc/zPn5E
bOco/vogOJ+9tdToj/ftsSwNG0LK9cutPWAEMX0CgYBTr5SI9/HVz56lJVzXaXGY
AzzkN+VVU8UiJRQTNiTwwhTYSE5lDqZGZD8A8DsdOeUL0x2HZgBjhqG6aX68SCBs
xUZ5O/IgTZ+JEPXnoJlDvK23PJzp3sQJggP2Sa083jXbvV5B9AFIIn0rsgYmpFb1
E6vQdbgNhyuETP/opqiswQKBgDpAdp1VIHUCiNZJsH2SaZCOe4BKs9ouXbKR+xvB
6xyRVgy8OODjVmV4D9I0O2UZh8TrOjMVXxnBKDr1uyWaRbsXLlUNdNUp6RiGPZTF
4jkEr4h4ZReItLD3WRW5HYjRPiqjWSIgAf5XhVusKkD+Po5FFOjN2iReXDAOiYhV
fnLZAoGBAOrMdd5HaWQXrldlWfB4YsU3MjPyT9GPBr2WjPo/52a8WEl0qmYeq6Dh
odja/6fSp2br/GPE63dJqr7tG1VlxJ/FTmJYwTxU8jLM9QjKMH+Vxwjp3Ex71Hr0
6u4gPfK2NTchDK9gz61/vKj3Ba9GHVVJOLJ0vk7vZr5Rt/Vfpleg
-----END RSA PRIVATE KEY-----`),
		}

		// its possible that there are multiple keys already in the keystore
		// find out a key ID that's not in the store yet.
		var keyID string
		for id := range raw {
			if cs.GetKey(id) == nil {
				keyID = id
			}
		}

		block, _ := pem.Decode(raw[keyID])
		key, err := x509.ParsePKCS1PrivateKey(block.Bytes)
		if err != nil {
			return nil, err
		}

		privKey, err := utils.RSAToPrivateKey(key)
		if err != nil {
			return nil, err
		}

		err = cs.AddKey(role, gun, privKey)
		if err != nil {
			return nil, err
		}

		pubKey = data.PublicKeyFromPrivate(privKey)
	default:
		return nil, fmt.Errorf("invalid key algorithm type")
	}
	return
}

// GetRSAKey returns a parsed RSA key based on the given size
func GetRSAKey(size int) (data.PrivateKey, error) {
	raw := map[int][]byte{
		1024: []byte(`-----BEGIN RSA PRIVATE KEY-----
MIICXAIBAAKBgQDJ8BO2/HOHLJgrb3srafbNRUD8r0SGNJFi5h7t4vxZ4F5oBW/4
O2/aZmdToinyuCm0eGguK77HAsTfSHqDUoEfuInNg7pPk4F6xa4feQzEeG6P0YaL
+VbApUdCHLBE0tVZg1SCW97+27wqIM4Cl1Tcsbb+aXfgMaOFGxlyga+a6wIDAQAB
AoGBAKDWLH2kGMfjBtghlLKBVWcs75PSbPuPRvTEYIIMNf3HrKmhGwtVG8ORqF5+
XHbLo7vv4tpTUUHkvLUyXxHVVq1oX+QqiRwTRm+ROF0/T6LlrWvTzvowTKtkRbsm
mqIYEbc+fBZ/7gEeW2ycCfE7HWgxNGvbUsK4LNa1ozJbrVEBAkEA8ML0mXyxq+cX
CwWvdXscN9vopLG/y+LKsvlKckoI/Hc0HjPyraq5Docwl2trZEmkvct1EcN8VvcV
vCtVsrAfwQJBANa4EBPfcIH2NKYHxt9cP00n74dVSHpwJYjBnbec5RCzn5UTbqd2
i62AkQREYhHZAryvBVE81JAFW3nqI9ZTpasCQBqEPlBRTXgzYXRTUfnMb1UvoTXS
Zd9cwRppHmvr/4Ve05yn+AhsjyksdouWxyMqgTxuFhy4vQ8O85Pf6fZeM4ECQCPp
Wv8H4thJplqSeGeJFSlBYaVf1SRtN0ndIBTCj+kwMaOMQXiOsiPNmfN9wG09v2Bx
YVFJ/D8uNjN4vo+tI8sCQFbtF+Qkj4uSFDZGMESF6MOgsGt1R1iCpvpMSr9h9V02
LPXyS3ozB7Deq26pEiCrFtHxw2Pb7RJO6GEqH7Dg4oU=
-----END RSA PRIVATE KEY-----`),
		2048: []byte(`-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEAtKGse3BcxXAp5OkLGYq0HfDcCvgag3R/9e8pHUGsJhkSZFrn
ZWAsAVFKSYaYItf1D/g3izqVDMtMpXZ1chNzaRysnbrb/q7JTbiGzXo9FcshyUc9
tcB60wFbvsXE2LaxZcKNxLYXbOvf+tdg/P07oPG24fzYI4+rbZ1wyoORbT1ys33Z
hHyifFvO7rbe69y3HG+xbp7yWYAR4e8Nw9jX8/9sGslAV9vEXOdNL3qlcgsYRGDU
DsUJsnWaMzjstvUxb8mVf9KG2W039ucgaXgBW/jeP3F1VSYFKLd03LvuJ8Ir5E0s
cWjwTd59nm0XbbRI3KiBGnAgrJ4iK07HrUkpDQIDAQABAoIBAHfr1k1lfdH+83Fs
XtgoRAiUviHyMfgQQlwO2eb4kMgCYTmLOJEPVmfRhlZmK18GrUZa7tVaoVYLKum3
SaXg0AB67wcQ5bmiZTdaSPTmMOPlJpsw1wFxtpmcD0MKnfOa5w++KMzub4L63or0
rwmHPi1ODLLgYMbLPW7a1eU9kDFLOnx3RRy9a29hQXxGsRYetrIbKmeDi6c+ndQ8
I5YWObcixxl5GP6CTnEugV7wd2JmXuQRGFdopUwQESCD9VkxDSevQBSPoyZKHxGy
/d3jf0VNlvwsxhD3ybhw8jTN/cmm2LWmP4jylG7iG7YRPVaW/0s39IZ9DnNDwgWB
03Yk2gECgYEA44jcSI5kXOrbBGDdV+wTUoL24Zoc0URX33F15UIOQuQsypaFoRJ6
J23JWEZN99aquuo1+0BBSfiumbpLwSwfXi0nL3oTzS9eOp1HS7AwFGd/OHdpdMsC
w2eInRwCh4GrEf508GXo+tSL2NS8+MFVAG2/SjEf06SroQ/rQ87Qm0ECgYEAyzqr
6YvbAnRBy5GgQlTgxR9r0U8N7lM9g2Tk8uGvYI8zu/Tgia4diHAwK1ymKbl3lf95
3NcHR+ffwOO+nnfFCvmCYXs4ggRCkeopv19bsCLkfnTBNTxPFh6lyLEnn3C+rcCe
ZAkKLrm8BHGviPIgn0aElMQAbhJxTWfClw/VVs0CgYAlDhfZ1R6xJypN9zx04iRv
bpaoPQHubrPk1sR9dpl9+Uz2HTdb+PddznpY3vI5p4Mcd6Ic7eT0GATPUlCeAAKH
wtC74aSx6MHux8hhoiriV8yXNJM/CwTDL+xGsdYTnWFvx8HhmKctmknAIT05QbsH
G9hoS8HEJPAyhbYpz9eXQQKBgQCftPXQTPXJUe86uLBGMEmK34xtKkD6XzPiA/Hf
5PdbXG39cQzbZZcT14YjLWXvOC8AE4qCwACaw1+VR+ROyDRy0W1iieD4W7ysymYQ
XDHDk0gZEEudOE22RlNmCcHnjERsawiN+ISl/5P/sg+OASkdwd8CwZzM43VirP3A
lNLEqQKBgHqj85n8ew23SR0GX0W1PgowaCHHD1p81y2afs1x7H9R1PNuQsTXKnpf
vMiG7Oxakj4WDC5M5KsHWqchqKDycT+J1BfLI58Sf2qo6vtaV+4DARNgzcG/WB4b
VnpsczK/1aUH7iBexuF0YqdPQwzpSvrY0XZcgCFQ52JDn3vjblhX
-----END RSA PRIVATE KEY-----`),
		4096: []byte(`-----BEGIN RSA PRIVATE KEY-----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==
-----END RSA PRIVATE KEY-----`),
	}
	block, _ := pem.Decode(raw[size])
	key, err := x509.ParsePKCS1PrivateKey(block.Bytes)
	if err != nil {
		return nil, err
	}

	privKey, err := utils.RSAToPrivateKey(key)
	if err != nil {
		return nil, err
	}
	return privKey, nil
}
