/*
  seektable.c: extract rows at selected offsets from an indexed
  NoSQL table.

  Original code Copyright (c) 1998 Maurizio Sartori

  Copyright (c) 2004,2006 Carlo Strozzi

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 dated June, 1991.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  $Id: seektable.c,v 1.3 2006/03/10 11:26:13 carlo Exp $

*/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/file.h>
#include <limits.h>

#ifndef HELPDIR
#define HELPDIR "/usr/local/nosql/help"
#endif
#ifndef HELPFILE
#define HELPFILE (HELPDIR "/seektable.txt")
#endif

#ifndef DOCDIR 
#define DOCDIR "/usr/local/nosql/doc"
#endif
#ifndef COPYING
#define COPYING (DOCDIR "/COPYING")
#endif
#ifndef WARRANTY
#define WARRANTY (DOCDIR "/WARRANTY")
#endif

static const char *progname = "seektable";  /* global pointer to argv[0] */

struct {
  unsigned int header	: 1;
  unsigned int verbose	: 1;
} flags;

static int eusage(void) {
    fprintf(stderr, "Usage: %s [options] table\n", progname);
    return 2;
}

static int etoolong2(void) {
    fprintf(stderr, "%s: max. length of table path (%d) exceeded\n",\
	progname, _POSIX_PATH_MAX);
    return 2;
}

int main(int argc, char *argv[]) {

    int i=0, offset, alert=0, min_args=2, rc=1;
    char c;

    char tmpbuf[sizeof(int)+1] = "";	/* work buffer */

    FILE *ifile = NULL, *ofile = NULL, *table = NULL;

    flags.header = 1;
    flags.verbose = 0;

    while (++i < argc && *argv[i] == '-') {

            min_args++;

            if (!strcmp(argv[i], "-N") ||
                     !strcmp(argv[i], "--no-header")) flags.header = 0;

            else if (!strcmp(argv[i], "-v") ||
                     !strcmp(argv[i], "--verbose")) flags.verbose = 1;

            else if (!strcmp(argv[i], "-i") ||
                     !strcmp(argv[i], "--input")) {

              if (++i >= argc || *argv[i] == '-') exit(eusage());

              min_args++;

              if ((ifile = freopen(argv[i], "r", stdin)) < 0) {
                 perror(argv[i]);
                 exit(2);
	      }
            }

            else if (!strcmp(argv[i], "-o") ||
                     !strcmp(argv[i], "--output")) {

              if (++i >= argc || *argv[i] == '-') exit(eusage());

              min_args++;

              if ((ofile = freopen(argv[i], "w", stdout)) < 0) {
                 perror(argv[i]);
                 exit(2);
	      }
            }

            else if (!strcmp(argv[i], "-h") ||
                     !strcmp(argv[i], "--help")) {

                    execlp("grep","grep","-v","^#",HELPFILE,(char *) 0);
                    perror("grep");
                    exit(1);
            }

            else if (!strcmp(argv[i], "--show-copying")) {
                    execlp("cat","cat",COPYING,(char *) 0);
                    perror("cat");
                    exit(1);
            }

            else if (!strcmp(argv[i], "--show-warranty")) {
                    execlp("cat","cat",WARRANTY,(char *) 0);
                    perror("cat");
                    exit(1);
            }
    }

    if (argc < min_args) exit(eusage());

    if ((table = fopen(argv[i], "r")) == 0) {
       fprintf(stderr, "%s: unable to open file %s\n",\
					progname, argv[i]);
       exit(2);
    }

    /* print table header if necessary */
    if (flags.header) {
       while ((c = fgetc(table)) != EOF) {
          putchar(c);
          if (c == '\n') {
             fflush(NULL);   /* Make sure the header is output */
	     break;
	  }
       }
    }

    /* read offsets from stdin */
    i=0;
    while ((c = fgetc(stdin)) != EOF) {
       if (c > 47 && c < 58) {
	  if (alert) alert=i=0;
	  *(tmpbuf+i) = c;
	  i++;
       }
       else if (c == ' ') {
	  alert = 1;
	  continue;			/* ignore blanks */
       }
       else if (i && c == '\n') {
	  *(tmpbuf+i) = '\0';		/* string terminator */
	  offset = atoi(tmpbuf);
	  i=0;				/* rewind buffer */

	  if (offset < 3) continue;	/* skip the minimal table header */

	  if (fseek(table, offset-1, SEEK_SET)) {
	     fprintf(stderr, "%s: cannot seek\n", progname);
	     exit(2);
	  }

	  /*
	  Consider only offsets pointing just after a newline.
	  This will be of use for in-place record deletions.
	  */

	  if (fgetc(table) == '\n') {
	     rc=0;			/* some valid input received */
	     while ((c = fgetc(table)) != EOF) {
		putchar(c);
		if (c == '\n') break;
	     }
	  }
	  else if (flags.verbose)
	     fprintf(stderr, "%s: bogus offset '%d' ignored\n", \
							progname, offset);
       }
       else i=0;			/* rewind buffer */
    }

    exit(rc);
}

/* EOF */
