/* @internal */
namespace ts {
    const ambientModuleSymbolRegex = /^".+"$/;
    const anon = "(anonymous)" as __String & string;

    let nextSymbolId = 1;
    let nextNodeId = 1;
    let nextMergeId = 1;
    let nextFlowId = 1;

    const enum IterationUse {
        AllowsSyncIterablesFlag = 1 << 0,
        AllowsAsyncIterablesFlag = 1 << 1,
        AllowsStringInputFlag = 1 << 2,
        ForOfFlag = 1 << 3,
        YieldStarFlag = 1 << 4,
        SpreadFlag = 1 << 5,
        DestructuringFlag = 1 << 6,

        // Spread, Destructuring, Array element assignment
        Element = AllowsSyncIterablesFlag,
        Spread = AllowsSyncIterablesFlag | SpreadFlag,
        Destructuring = AllowsSyncIterablesFlag | DestructuringFlag,

        ForOf = AllowsSyncIterablesFlag | AllowsStringInputFlag | ForOfFlag,
        ForAwaitOf = AllowsSyncIterablesFlag | AllowsAsyncIterablesFlag | AllowsStringInputFlag | ForOfFlag,

        YieldStar = AllowsSyncIterablesFlag | YieldStarFlag,
        AsyncYieldStar = AllowsSyncIterablesFlag | AllowsAsyncIterablesFlag | YieldStarFlag,

        GeneratorReturnType = AllowsSyncIterablesFlag,
        AsyncGeneratorReturnType = AllowsAsyncIterablesFlag,

    }

    const enum IterationTypeKind {
        Yield,
        Return,
        Next,
    }

    interface IterationTypesResolver {
        iterableCacheKey: "iterationTypesOfAsyncIterable" | "iterationTypesOfIterable";
        iteratorCacheKey: "iterationTypesOfAsyncIterator" | "iterationTypesOfIterator";
        iteratorSymbolName: "asyncIterator" | "iterator";
        getGlobalIteratorType: (reportErrors: boolean) => GenericType;
        getGlobalIterableType: (reportErrors: boolean) => GenericType;
        getGlobalIterableIteratorType: (reportErrors: boolean) => GenericType;
        getGlobalGeneratorType: (reportErrors: boolean) => GenericType;
        resolveIterationType: (type: Type, errorNode: Node | undefined) => Type | undefined;
        mustHaveANextMethodDiagnostic: DiagnosticMessage;
        mustBeAMethodDiagnostic: DiagnosticMessage;
        mustHaveAValueDiagnostic: DiagnosticMessage;
    }

    const enum WideningKind {
        Normal,
        FunctionReturn,
        GeneratorNext,
        GeneratorYield,
    }

    const enum TypeFacts {
        None = 0,
        TypeofEQString = 1 << 0,      // typeof x === "string"
        TypeofEQNumber = 1 << 1,      // typeof x === "number"
        TypeofEQBigInt = 1 << 2,      // typeof x === "bigint"
        TypeofEQBoolean = 1 << 3,     // typeof x === "boolean"
        TypeofEQSymbol = 1 << 4,      // typeof x === "symbol"
        TypeofEQObject = 1 << 5,      // typeof x === "object"
        TypeofEQFunction = 1 << 6,    // typeof x === "function"
        TypeofEQHostObject = 1 << 7,  // typeof x === "xxx"
        TypeofNEString = 1 << 8,      // typeof x !== "string"
        TypeofNENumber = 1 << 9,      // typeof x !== "number"
        TypeofNEBigInt = 1 << 10,     // typeof x !== "bigint"
        TypeofNEBoolean = 1 << 11,     // typeof x !== "boolean"
        TypeofNESymbol = 1 << 12,     // typeof x !== "symbol"
        TypeofNEObject = 1 << 13,     // typeof x !== "object"
        TypeofNEFunction = 1 << 14,   // typeof x !== "function"
        TypeofNEHostObject = 1 << 15, // typeof x !== "xxx"
        EQUndefined = 1 << 16,        // x === undefined
        EQNull = 1 << 17,             // x === null
        EQUndefinedOrNull = 1 << 18,  // x === undefined / x === null
        NEUndefined = 1 << 19,        // x !== undefined
        NENull = 1 << 20,             // x !== null
        NEUndefinedOrNull = 1 << 21,  // x != undefined / x != null
        Truthy = 1 << 22,             // x
        Falsy = 1 << 23,              // !x
        All = (1 << 24) - 1,
        // The following members encode facts about particular kinds of types for use in the getTypeFacts function.
        // The presence of a particular fact means that the given test is true for some (and possibly all) values
        // of that kind of type.
        BaseStringStrictFacts = TypeofEQString | TypeofNENumber | TypeofNEBigInt | TypeofNEBoolean | TypeofNESymbol | TypeofNEObject | TypeofNEFunction | TypeofNEHostObject | NEUndefined | NENull | NEUndefinedOrNull,
        BaseStringFacts = BaseStringStrictFacts | EQUndefined | EQNull | EQUndefinedOrNull | Falsy,
        StringStrictFacts = BaseStringStrictFacts | Truthy | Falsy,
        StringFacts = BaseStringFacts | Truthy,
        EmptyStringStrictFacts = BaseStringStrictFacts | Falsy,
        EmptyStringFacts = BaseStringFacts,
        NonEmptyStringStrictFacts = BaseStringStrictFacts | Truthy,
        NonEmptyStringFacts = BaseStringFacts | Truthy,
        BaseNumberStrictFacts = TypeofEQNumber | TypeofNEString | TypeofNEBigInt | TypeofNEBoolean | TypeofNESymbol | TypeofNEObject | TypeofNEFunction | TypeofNEHostObject | NEUndefined | NENull | NEUndefinedOrNull,
        BaseNumberFacts = BaseNumberStrictFacts | EQUndefined | EQNull | EQUndefinedOrNull | Falsy,
        NumberStrictFacts = BaseNumberStrictFacts | Truthy | Falsy,
        NumberFacts = BaseNumberFacts | Truthy,
        ZeroNumberStrictFacts = BaseNumberStrictFacts | Falsy,
        ZeroNumberFacts = BaseNumberFacts,
        NonZeroNumberStrictFacts = BaseNumberStrictFacts | Truthy,
        NonZeroNumberFacts = BaseNumberFacts | Truthy,
        BaseBigIntStrictFacts = TypeofEQBigInt | TypeofNEString | TypeofNENumber | TypeofNEBoolean | TypeofNESymbol | TypeofNEObject | TypeofNEFunction | TypeofNEHostObject | NEUndefined | NENull | NEUndefinedOrNull,
        BaseBigIntFacts = BaseBigIntStrictFacts | EQUndefined | EQNull | EQUndefinedOrNull | Falsy,
        BigIntStrictFacts = BaseBigIntStrictFacts | Truthy | Falsy,
        BigIntFacts = BaseBigIntFacts | Truthy,
        ZeroBigIntStrictFacts = BaseBigIntStrictFacts | Falsy,
        ZeroBigIntFacts = BaseBigIntFacts,
        NonZeroBigIntStrictFacts = BaseBigIntStrictFacts | Truthy,
        NonZeroBigIntFacts = BaseBigIntFacts | Truthy,
        BaseBooleanStrictFacts = TypeofEQBoolean | TypeofNEString | TypeofNENumber | TypeofNEBigInt | TypeofNESymbol | TypeofNEObject | TypeofNEFunction | TypeofNEHostObject | NEUndefined | NENull | NEUndefinedOrNull,
        BaseBooleanFacts = BaseBooleanStrictFacts | EQUndefined | EQNull | EQUndefinedOrNull | Falsy,
        BooleanStrictFacts = BaseBooleanStrictFacts | Truthy | Falsy,
        BooleanFacts = BaseBooleanFacts | Truthy,
        FalseStrictFacts = BaseBooleanStrictFacts | Falsy,
        FalseFacts = BaseBooleanFacts,
        TrueStrictFacts = BaseBooleanStrictFacts | Truthy,
        TrueFacts = BaseBooleanFacts | Truthy,
        SymbolStrictFacts = TypeofEQSymbol | TypeofNEString | TypeofNENumber | TypeofNEBigInt | TypeofNEBoolean | TypeofNEObject | TypeofNEFunction | TypeofNEHostObject | NEUndefined | NENull | NEUndefinedOrNull | Truthy,
        SymbolFacts = SymbolStrictFacts | EQUndefined | EQNull | EQUndefinedOrNull | Falsy,
        ObjectStrictFacts = TypeofEQObject | TypeofEQHostObject | TypeofNEString | TypeofNENumber | TypeofNEBigInt | TypeofNEBoolean | TypeofNESymbol | TypeofNEFunction | NEUndefined | NENull | NEUndefinedOrNull | Truthy,
        ObjectFacts = ObjectStrictFacts | EQUndefined | EQNull | EQUndefinedOrNull | Falsy,
        FunctionStrictFacts = TypeofEQFunction | TypeofEQHostObject | TypeofNEString | TypeofNENumber | TypeofNEBigInt | TypeofNEBoolean | TypeofNESymbol | TypeofNEObject | NEUndefined | NENull | NEUndefinedOrNull | Truthy,
        FunctionFacts = FunctionStrictFacts | EQUndefined | EQNull | EQUndefinedOrNull | Falsy,
        UndefinedFacts = TypeofNEString | TypeofNENumber | TypeofNEBigInt | TypeofNEBoolean | TypeofNESymbol | TypeofNEObject | TypeofNEFunction | TypeofNEHostObject | EQUndefined | EQUndefinedOrNull | NENull | Falsy,
        NullFacts = TypeofEQObject | TypeofNEString | TypeofNENumber | TypeofNEBigInt | TypeofNEBoolean | TypeofNESymbol | TypeofNEFunction | TypeofNEHostObject | EQNull | EQUndefinedOrNull | NEUndefined | Falsy,
        EmptyObjectStrictFacts = All & ~(EQUndefined | EQNull | EQUndefinedOrNull),
        AllTypeofNE = TypeofNEString | TypeofNENumber | TypeofNEBigInt | TypeofNEBoolean | TypeofNESymbol | TypeofNEObject | TypeofNEFunction | NEUndefined,
        EmptyObjectFacts = All,
    }

    const typeofEQFacts: ReadonlyESMap<string, TypeFacts> = new Map(getEntries({
        string: TypeFacts.TypeofEQString,
        number: TypeFacts.TypeofEQNumber,
        bigint: TypeFacts.TypeofEQBigInt,
        boolean: TypeFacts.TypeofEQBoolean,
        symbol: TypeFacts.TypeofEQSymbol,
        undefined: TypeFacts.EQUndefined,
        object: TypeFacts.TypeofEQObject,
        function: TypeFacts.TypeofEQFunction
    }));

    const typeofNEFacts: ReadonlyESMap<string, TypeFacts> = new Map(getEntries({
        string: TypeFacts.TypeofNEString,
        number: TypeFacts.TypeofNENumber,
        bigint: TypeFacts.TypeofNEBigInt,
        boolean: TypeFacts.TypeofNEBoolean,
        symbol: TypeFacts.TypeofNESymbol,
        undefined: TypeFacts.NEUndefined,
        object: TypeFacts.TypeofNEObject,
        function: TypeFacts.TypeofNEFunction
    }));

    type TypeSystemEntity = Node | Symbol | Type | Signature;

    const enum TypeSystemPropertyName {
        Type,
        ResolvedBaseConstructorType,
        DeclaredType,
        ResolvedReturnType,
        ImmediateBaseConstraint,
        EnumTagType,
        ResolvedTypeArguments,
        ResolvedBaseTypes,
    }

    const enum CheckMode {
        Normal = 0,                     // Normal type checking
        Contextual = 1 << 0,            // Explicitly assigned contextual type, therefore not cacheable
        Inferential = 1 << 1,           // Inferential typing
        SkipContextSensitive = 1 << 2,  // Skip context sensitive function expressions
        SkipGenericFunctions = 1 << 3,  // Skip single signature generic functions
        IsForSignatureHelp = 1 << 4,    // Call resolution for purposes of signature help
    }

    const enum AccessFlags {
        None = 0,
        NoIndexSignatures = 1 << 0,
        Writing = 1 << 1,
        CacheSymbol = 1 << 2,
        NoTupleBoundsCheck = 1 << 3,
    }

    const enum SignatureCheckMode {
        BivariantCallback = 1 << 0,
        StrictCallback    = 1 << 1,
        IgnoreReturnTypes = 1 << 2,
        StrictArity       = 1 << 3,
        Callback          = BivariantCallback | StrictCallback,
    }

    const enum IntersectionState {
        None = 0,
        Source = 1 << 0,
        Target = 1 << 1,
        PropertyCheck = 1 << 2,
        InPropertyCheck = 1 << 3,
    }

    const enum MappedTypeModifiers {
        IncludeReadonly = 1 << 0,
        ExcludeReadonly = 1 << 1,
        IncludeOptional = 1 << 2,
        ExcludeOptional = 1 << 3,
    }

    const enum ExpandingFlags {
        None = 0,
        Source = 1,
        Target = 1 << 1,
        Both = Source | Target,
    }

    const enum MembersOrExportsResolutionKind {
        resolvedExports = "resolvedExports",
        resolvedMembers = "resolvedMembers"
    }

    const enum UnusedKind {
        Local,
        Parameter,
    }

    /** @param containingNode Node to check for parse error */
    type AddUnusedDiagnostic = (containingNode: Node, type: UnusedKind, diagnostic: DiagnosticWithLocation) => void;

    const isNotOverloadAndNotAccessor = and(isNotOverload, isNotAccessor);

    const enum DeclarationMeaning {
        GetAccessor = 1,
        SetAccessor = 2,
        PropertyAssignment = 4,
        Method = 8,
        GetOrSetAccessor = GetAccessor | SetAccessor,
        PropertyAssignmentOrMethod = PropertyAssignment | Method,
    }

    const enum DeclarationSpaces {
        None = 0,
        ExportValue = 1 << 0,
        ExportType = 1 << 1,
        ExportNamespace = 1 << 2,
    }

    function SymbolLinks(this: SymbolLinks) {
    }

    function NodeLinks(this: NodeLinks) {
        this.flags = 0;
    }

    export function getNodeId(node: Node): number {
        if (!node.id) {
            node.id = nextNodeId;
            nextNodeId++;
        }
        return node.id;
    }

    export function getSymbolId(symbol: Symbol): SymbolId {
        if (!symbol.id) {
            symbol.id = nextSymbolId;
            nextSymbolId++;
        }

        return symbol.id;
    }

    export function isInstantiatedModule(node: ModuleDeclaration, preserveConstEnums: boolean) {
        const moduleState = getModuleInstanceState(node);
        return moduleState === ModuleInstanceState.Instantiated ||
            (preserveConstEnums && moduleState === ModuleInstanceState.ConstEnumOnly);
    }

    export function createTypeChecker(host: TypeCheckerHost, produceDiagnostics: boolean): TypeChecker {
        const getPackagesSet = memoize(() => {
            const set = new Set<string>();
            host.getSourceFiles().forEach(sf => {
                if (!sf.resolvedModules) return;

                forEachEntry(sf.resolvedModules, r => {
                    if (r && r.packageId) set.add(r.packageId.name);
                });
            });
            return set;
        });

        // Cancellation that controls whether or not we can cancel in the middle of type checking.
        // In general cancelling is *not* safe for the type checker.  We might be in the middle of
        // computing something, and we will leave our internals in an inconsistent state.  Callers
        // who set the cancellation token should catch if a cancellation exception occurs, and
        // should throw away and create a new TypeChecker.
        //
        // Currently we only support setting the cancellation token when getting diagnostics.  This
        // is because diagnostics can be quite expensive, and we want to allow hosts to bail out if
        // they no longer need the information (for example, if the user started editing again).
        let cancellationToken: CancellationToken | undefined;
        let requestedExternalEmitHelpers: ExternalEmitHelpers;
        let externalHelpersModule: Symbol;

        const Symbol = objectAllocator.getSymbolConstructor();
        const Type = objectAllocator.getTypeConstructor();
        const Signature = objectAllocator.getSignatureConstructor();

        let typeCount = 0;
        let symbolCount = 0;
        let enumCount = 0;
        let totalInstantiationCount = 0;
        let instantiationCount = 0;
        let instantiationDepth = 0;
        let constraintDepth = 0;
        let currentNode: Node | undefined;

        const emptySymbols = createSymbolTable();
        const arrayVariances = [VarianceFlags.Covariant];

        const compilerOptions = host.getCompilerOptions();
        const languageVersion = getEmitScriptTarget(compilerOptions);
        const moduleKind = getEmitModuleKind(compilerOptions);
        const allowSyntheticDefaultImports = getAllowSyntheticDefaultImports(compilerOptions);
        const strictNullChecks = getStrictOptionValue(compilerOptions, "strictNullChecks");
        const strictFunctionTypes = getStrictOptionValue(compilerOptions, "strictFunctionTypes");
        const strictBindCallApply = getStrictOptionValue(compilerOptions, "strictBindCallApply");
        const strictPropertyInitialization = getStrictOptionValue(compilerOptions, "strictPropertyInitialization");
        const noImplicitAny = getStrictOptionValue(compilerOptions, "noImplicitAny");
        const noImplicitThis = getStrictOptionValue(compilerOptions, "noImplicitThis");
        const keyofStringsOnly = !!compilerOptions.keyofStringsOnly;
        const freshObjectLiteralFlag = compilerOptions.suppressExcessPropertyErrors ? 0 : ObjectFlags.FreshLiteral;

        const emitResolver = createResolver();
        const nodeBuilder = createNodeBuilder();

        const globals = createSymbolTable();
        const undefinedSymbol = createSymbol(SymbolFlags.Property, "undefined" as __String);
        undefinedSymbol.declarations = [];

        const globalThisSymbol = createSymbol(SymbolFlags.Module, "globalThis" as __String, CheckFlags.Readonly);
        globalThisSymbol.exports = globals;
        globalThisSymbol.declarations = [];
        globals.set(globalThisSymbol.escapedName, globalThisSymbol);

        const argumentsSymbol = createSymbol(SymbolFlags.Property, "arguments" as __String);
        const requireSymbol = createSymbol(SymbolFlags.Property, "require" as __String);

        /** This will be set during calls to `getResolvedSignature` where services determines an apparent number of arguments greater than what is actually provided. */
        let apparentArgumentCount: number | undefined;

        // for public members that accept a Node or one of its subtypes, we must guard against
        // synthetic nodes created during transformations by calling `getParseTreeNode`.
        // for most of these, we perform the guard only on `checker` to avoid any possible
        // extra cost of calling `getParseTreeNode` when calling these functions from inside the
        // checker.
        const checker: TypeChecker = {
            getNodeCount: () => sum(host.getSourceFiles(), "nodeCount"),
            getIdentifierCount: () => sum(host.getSourceFiles(), "identifierCount"),
            getSymbolCount: () => sum(host.getSourceFiles(), "symbolCount") + symbolCount,
            getTypeCount: () => typeCount,
            getInstantiationCount: () => totalInstantiationCount,
            getRelationCacheSizes: () => ({
                assignable: assignableRelation.size,
                identity: identityRelation.size,
                subtype: subtypeRelation.size,
                strictSubtype: strictSubtypeRelation.size,
            }),
            isUndefinedSymbol: symbol => symbol === undefinedSymbol,
            isArgumentsSymbol: symbol => symbol === argumentsSymbol,
            isUnknownSymbol: symbol => symbol === unknownSymbol,
            getMergedSymbol,
            getDiagnostics,
            getGlobalDiagnostics,
            getTypeOfSymbolAtLocation: (symbol, locationIn) => {
                const location = getParseTreeNode(locationIn);
                return location ? getTypeOfSymbolAtLocation(symbol, location) : errorType;
            },
            getSymbolsOfParameterPropertyDeclaration: (parameterIn, parameterName) => {
                const parameter = getParseTreeNode(parameterIn, isParameter);
                if (parameter === undefined) return Debug.fail("Cannot get symbols of a synthetic parameter that cannot be resolved to a parse-tree node.");
                return getSymbolsOfParameterPropertyDeclaration(parameter, escapeLeadingUnderscores(parameterName));
            },
            getDeclaredTypeOfSymbol,
            getPropertiesOfType,
            getPropertyOfType: (type, name) => getPropertyOfType(type, escapeLeadingUnderscores(name)),
            getPrivateIdentifierPropertyOfType: (leftType: Type, name: string, location: Node) => {
                const node = getParseTreeNode(location);
                if (!node) {
                    return undefined;
                }
                const propName = escapeLeadingUnderscores(name);
                const lexicallyScopedIdentifier = lookupSymbolForPrivateIdentifierDeclaration(propName, node);
                return lexicallyScopedIdentifier ? getPrivateIdentifierPropertyOfType(leftType, lexicallyScopedIdentifier) : undefined;
            },
            getTypeOfPropertyOfType: (type, name) => getTypeOfPropertyOfType(type, escapeLeadingUnderscores(name)),
            getIndexInfoOfType,
            getSignaturesOfType,
            getIndexTypeOfType,
            getBaseTypes,
            getBaseTypeOfLiteralType,
            getWidenedType,
            getTypeFromTypeNode: nodeIn => {
                const node = getParseTreeNode(nodeIn, isTypeNode);
                return node ? getTypeFromTypeNode(node) : errorType;
            },
            getParameterType: getTypeAtPosition,
            getPromisedTypeOfPromise,
            getAwaitedType: type => getAwaitedType(type),
            getReturnTypeOfSignature,
            isNullableType,
            getNullableType,
            getNonNullableType,
            getNonOptionalType: removeOptionalTypeMarker,
            getTypeArguments,
            typeToTypeNode: nodeBuilder.typeToTypeNode,
            indexInfoToIndexSignatureDeclaration: nodeBuilder.indexInfoToIndexSignatureDeclaration,
            signatureToSignatureDeclaration: nodeBuilder.signatureToSignatureDeclaration,
            symbolToEntityName: nodeBuilder.symbolToEntityName,
            symbolToExpression: nodeBuilder.symbolToExpression,
            symbolToTypeParameterDeclarations: nodeBuilder.symbolToTypeParameterDeclarations,
            symbolToParameterDeclaration: nodeBuilder.symbolToParameterDeclaration,
            typeParameterToDeclaration: nodeBuilder.typeParameterToDeclaration,
            getSymbolsInScope: (locationIn, meaning) => {
                const location = getParseTreeNode(locationIn);
                return location ? getSymbolsInScope(location, meaning) : [];
            },
            getSymbolAtLocation: nodeIn => {
                const node = getParseTreeNode(nodeIn);
                // set ignoreErrors: true because any lookups invoked by the API shouldn't cause any new errors
                return node ? getSymbolAtLocation(node, /*ignoreErrors*/ true) : undefined;
            },
            getShorthandAssignmentValueSymbol: nodeIn => {
                const node = getParseTreeNode(nodeIn);
                return node ? getShorthandAssignmentValueSymbol(node) : undefined;
            },
            getExportSpecifierLocalTargetSymbol: nodeIn => {
                const node = getParseTreeNode(nodeIn, isExportSpecifier);
                return node ? getExportSpecifierLocalTargetSymbol(node) : undefined;
            },
            getExportSymbolOfSymbol(symbol) {
                return getMergedSymbol(symbol.exportSymbol || symbol);
            },
            getTypeAtLocation: nodeIn => {
                const node = getParseTreeNode(nodeIn);
                return node ? getTypeOfNode(node) : errorType;
            },
            getTypeOfAssignmentPattern: nodeIn => {
                const node = getParseTreeNode(nodeIn, isAssignmentPattern);
                return node && getTypeOfAssignmentPattern(node) || errorType;
            },
            getPropertySymbolOfDestructuringAssignment: locationIn => {
                const location = getParseTreeNode(locationIn, isIdentifier);
                return location ? getPropertySymbolOfDestructuringAssignment(location) : undefined;
            },
            signatureToString: (signature, enclosingDeclaration, flags, kind) => {
                return signatureToString(signature, getParseTreeNode(enclosingDeclaration), flags, kind);
            },
            typeToString: (type, enclosingDeclaration, flags) => {
                return typeToString(type, getParseTreeNode(enclosingDeclaration), flags);
            },
            symbolToString: (symbol, enclosingDeclaration, meaning, flags) => {
                return symbolToString(symbol, getParseTreeNode(enclosingDeclaration), meaning, flags);
            },
            typePredicateToString: (predicate, enclosingDeclaration, flags) => {
                return typePredicateToString(predicate, getParseTreeNode(enclosingDeclaration), flags);
            },
            writeSignature: (signature, enclosingDeclaration, flags, kind, writer) => {
                return signatureToString(signature, getParseTreeNode(enclosingDeclaration), flags, kind, writer);
            },
            writeType: (type, enclosingDeclaration, flags, writer) => {
                return typeToString(type, getParseTreeNode(enclosingDeclaration), flags, writer);
            },
            writeSymbol: (symbol, enclosingDeclaration, meaning, flags, writer) => {
                return symbolToString(symbol, getParseTreeNode(enclosingDeclaration), meaning, flags, writer);
            },
            writeTypePredicate: (predicate, enclosingDeclaration, flags, writer) => {
                return typePredicateToString(predicate, getParseTreeNode(enclosingDeclaration), flags, writer);
            },
            getAugmentedPropertiesOfType,
            getRootSymbols,
            getContextualType: (nodeIn: Expression, contextFlags?: ContextFlags) => {
                const node = getParseTreeNode(nodeIn, isExpression);
                if (!node) {
                    return undefined;
                }
                const containingCall = findAncestor(node, isCallLikeExpression);
                const containingCallResolvedSignature = containingCall && getNodeLinks(containingCall).resolvedSignature;
                if (contextFlags! & ContextFlags.Completions && containingCall) {
                    let toMarkSkip = node as Node;
                    do {
                        getNodeLinks(toMarkSkip).skipDirectInference = true;
                        toMarkSkip = toMarkSkip.parent;
                    } while (toMarkSkip && toMarkSkip !== containingCall);
                    getNodeLinks(containingCall).resolvedSignature = undefined;
                }
                const result = getContextualType(node, contextFlags);
                if (contextFlags! & ContextFlags.Completions && containingCall) {
                    let toMarkSkip = node as Node;
                    do {
                        getNodeLinks(toMarkSkip).skipDirectInference = undefined;
                        toMarkSkip = toMarkSkip.parent;
                    } while (toMarkSkip && toMarkSkip !== containingCall);
                    getNodeLinks(containingCall).resolvedSignature = containingCallResolvedSignature;
                }
                return result;
            },
            getContextualTypeForObjectLiteralElement: nodeIn => {
                const node = getParseTreeNode(nodeIn, isObjectLiteralElementLike);
                return node ? getContextualTypeForObjectLiteralElement(node) : undefined;
            },
            getContextualTypeForArgumentAtIndex: (nodeIn, argIndex) => {
                const node = getParseTreeNode(nodeIn, isCallLikeExpression);
                return node && getContextualTypeForArgumentAtIndex(node, argIndex);
            },
            getContextualTypeForJsxAttribute: (nodeIn) => {
                const node = getParseTreeNode(nodeIn, isJsxAttributeLike);
                return node && getContextualTypeForJsxAttribute(node);
            },
            isContextSensitive,
            getFullyQualifiedName,
            getResolvedSignature: (node, candidatesOutArray, argumentCount) =>
                getResolvedSignatureWorker(node, candidatesOutArray, argumentCount, CheckMode.Normal),
            getResolvedSignatureForSignatureHelp: (node, candidatesOutArray, argumentCount) =>
                getResolvedSignatureWorker(node, candidatesOutArray, argumentCount, CheckMode.IsForSignatureHelp),
            getExpandedParameters,
            hasEffectiveRestParameter,
            getConstantValue: nodeIn => {
                const node = getParseTreeNode(nodeIn, canHaveConstantValue);
                return node ? getConstantValue(node) : undefined;
            },
            isValidPropertyAccess: (nodeIn, propertyName) => {
                const node = getParseTreeNode(nodeIn, isPropertyAccessOrQualifiedNameOrImportTypeNode);
                return !!node && isValidPropertyAccess(node, escapeLeadingUnderscores(propertyName));
            },
            isValidPropertyAccessForCompletions: (nodeIn, type, property) => {
                const node = getParseTreeNode(nodeIn, isPropertyAccessExpression);
                return !!node && isValidPropertyAccessForCompletions(node, type, property);
            },
            getSignatureFromDeclaration: declarationIn => {
                const declaration = getParseTreeNode(declarationIn, isFunctionLike);
                return declaration ? getSignatureFromDeclaration(declaration) : undefined;
            },
            isImplementationOfOverload: nodeIn => {
                const node = getParseTreeNode(nodeIn, isFunctionLike);
                return node ? isImplementationOfOverload(node) : undefined;
            },
            getImmediateAliasedSymbol,
            getAliasedSymbol: resolveAlias,
            getEmitResolver,
            getExportsOfModule: getExportsOfModuleAsArray,
            getExportsAndPropertiesOfModule,
            getSymbolWalker: createGetSymbolWalker(
                getRestTypeOfSignature,
                getTypePredicateOfSignature,
                getReturnTypeOfSignature,
                getBaseTypes,
                resolveStructuredTypeMembers,
                getTypeOfSymbol,
                getResolvedSymbol,
                getIndexTypeOfStructuredType,
                getConstraintOfTypeParameter,
                getFirstIdentifier,
                getTypeArguments,
            ),
            getAmbientModules,
            getJsxIntrinsicTagNamesAt,
            isOptionalParameter: nodeIn => {
                const node = getParseTreeNode(nodeIn, isParameter);
                return node ? isOptionalParameter(node) : false;
            },
            tryGetMemberInModuleExports: (name, symbol) => tryGetMemberInModuleExports(escapeLeadingUnderscores(name), symbol),
            tryGetMemberInModuleExportsAndProperties: (name, symbol) => tryGetMemberInModuleExportsAndProperties(escapeLeadingUnderscores(name), symbol),
            tryFindAmbientModuleWithoutAugmentations: moduleName => {
                // we deliberately exclude augmentations
                // since we are only interested in declarations of the module itself
                return tryFindAmbientModule(moduleName, /*withAugmentations*/ false);
            },
            getApparentType,
            getUnionType,
            isTypeAssignableTo,
            createAnonymousType,
            createSignature,
            createSymbol,
            createIndexInfo,
            getAnyType: () => anyType,
            getStringType: () => stringType,
            getNumberType: () => numberType,
            createPromiseType,
            createArrayType,
            getElementTypeOfArrayType,
            getBooleanType: () => booleanType,
            getFalseType: (fresh?) => fresh ? falseType : regularFalseType,
            getTrueType: (fresh?) => fresh ? trueType : regularTrueType,
            getVoidType: () => voidType,
            getUndefinedType: () => undefinedType,
            getNullType: () => nullType,
            getESSymbolType: () => esSymbolType,
            getNeverType: () => neverType,
            getOptionalType: () => optionalType,
            isSymbolAccessible,
            isArrayType,
            isTupleType,
            isArrayLikeType,
            isTypeInvalidDueToUnionDiscriminant,
            getAllPossiblePropertiesOfTypes,
            getSuggestedSymbolForNonexistentProperty,
            getSuggestionForNonexistentProperty,
            getSuggestedSymbolForNonexistentJSXAttribute,
            getSuggestedSymbolForNonexistentSymbol: (location, name, meaning) => getSuggestedSymbolForNonexistentSymbol(location, escapeLeadingUnderscores(name), meaning),
            getSuggestionForNonexistentSymbol: (location, name, meaning) => getSuggestionForNonexistentSymbol(location, escapeLeadingUnderscores(name), meaning),
            getSuggestedSymbolForNonexistentModule,
            getSuggestionForNonexistentExport,
            getBaseConstraintOfType,
            getDefaultFromTypeParameter: type => type && type.flags & TypeFlags.TypeParameter ? getDefaultFromTypeParameter(type as TypeParameter) : undefined,
            resolveName(name, location, meaning, excludeGlobals) {
                return resolveName(location, escapeLeadingUnderscores(name), meaning, /*nameNotFoundMessage*/ undefined, /*nameArg*/ undefined, /*isUse*/ false, excludeGlobals);
            },
            getJsxNamespace: n => unescapeLeadingUnderscores(getJsxNamespace(n)),
            getAccessibleSymbolChain,
            getTypePredicateOfSignature,
            resolveExternalModuleName: moduleSpecifierIn => {
                const moduleSpecifier = getParseTreeNode(moduleSpecifierIn, isExpression);
                return moduleSpecifier && resolveExternalModuleName(moduleSpecifier, moduleSpecifier, /*ignoreErrors*/ true);
            },
            resolveExternalModuleSymbol,
            tryGetThisTypeAt: (nodeIn, includeGlobalThis) => {
                const node = getParseTreeNode(nodeIn);
                return node && tryGetThisTypeAt(node, includeGlobalThis);
            },
            getTypeArgumentConstraint: nodeIn => {
                const node = getParseTreeNode(nodeIn, isTypeNode);
                return node && getTypeArgumentConstraint(node);
            },
            getSuggestionDiagnostics: (fileIn, ct) => {
                const file = getParseTreeNode(fileIn, isSourceFile) || Debug.fail("Could not determine parsed source file.");
                if (skipTypeChecking(file, compilerOptions, host)) {
                    return emptyArray;
                }

                let diagnostics: DiagnosticWithLocation[] | undefined;
                try {
                    // Record the cancellation token so it can be checked later on during checkSourceElement.
                    // Do this in a finally block so we can ensure that it gets reset back to nothing after
                    // this call is done.
                    cancellationToken = ct;

                    // Ensure file is type checked
                    checkSourceFile(file);
                    Debug.assert(!!(getNodeLinks(file).flags & NodeCheckFlags.TypeChecked));

                    diagnostics = addRange(diagnostics, suggestionDiagnostics.getDiagnostics(file.fileName));
                    checkUnusedIdentifiers(getPotentiallyUnusedIdentifiers(file), (containingNode, kind, diag) => {
                        if (!containsParseError(containingNode) && !unusedIsError(kind, !!(containingNode.flags & NodeFlags.Ambient))) {
                            (diagnostics || (diagnostics = [])).push({ ...diag, category: DiagnosticCategory.Suggestion });
                        }
                    });

                    return diagnostics || emptyArray;
                }
                finally {
                    cancellationToken = undefined;
                }
            },

            runWithCancellationToken: (token, callback) => {
                try {
                    cancellationToken = token;
                    return callback(checker);
                }
                finally {
                    cancellationToken = undefined;
                }
            },

            getLocalTypeParametersOfClassOrInterfaceOrTypeAlias,
            isDeclarationVisible,
        };

        function getResolvedSignatureWorker(nodeIn: CallLikeExpression, candidatesOutArray: Signature[] | undefined, argumentCount: number | undefined, checkMode: CheckMode): Signature | undefined {
            const node = getParseTreeNode(nodeIn, isCallLikeExpression);
            apparentArgumentCount = argumentCount;
            const res = node ? getResolvedSignature(node, candidatesOutArray, checkMode) : undefined;
            apparentArgumentCount = undefined;
            return res;
        }

        const tupleTypes = new Map<string, GenericType>();
        const unionTypes = new Map<string, UnionType>();
        const intersectionTypes = new Map<string, Type>();
        const literalTypes = new Map<string, LiteralType>();
        const indexedAccessTypes = new Map<string, IndexedAccessType>();
        const substitutionTypes = new Map<string, SubstitutionType>();
        const evolvingArrayTypes: EvolvingArrayType[] = [];
        const undefinedProperties: SymbolTable = new Map();

        const unknownSymbol = createSymbol(SymbolFlags.Property, "unknown" as __String);
        const resolvingSymbol = createSymbol(0, InternalSymbolName.Resolving);

        const anyType = createIntrinsicType(TypeFlags.Any, "any");
        const autoType = createIntrinsicType(TypeFlags.Any, "any");
        const wildcardType = createIntrinsicType(TypeFlags.Any, "any");
        const errorType = createIntrinsicType(TypeFlags.Any, "error");
        const nonInferrableAnyType = createIntrinsicType(TypeFlags.Any, "any", ObjectFlags.ContainsWideningType);
        const unknownType = createIntrinsicType(TypeFlags.Unknown, "unknown");
        const undefinedType = createIntrinsicType(TypeFlags.Undefined, "undefined");
        const undefinedWideningType = strictNullChecks ? undefinedType : createIntrinsicType(TypeFlags.Undefined, "undefined", ObjectFlags.ContainsWideningType);
        const optionalType = createIntrinsicType(TypeFlags.Undefined, "undefined");
        const nullType = createIntrinsicType(TypeFlags.Null, "null");
        const nullWideningType = strictNullChecks ? nullType : createIntrinsicType(TypeFlags.Null, "null", ObjectFlags.ContainsWideningType);
        const stringType = createIntrinsicType(TypeFlags.String, "string");
        const numberType = createIntrinsicType(TypeFlags.Number, "number");
        const bigintType = createIntrinsicType(TypeFlags.BigInt, "bigint");
        const falseType = createIntrinsicType(TypeFlags.BooleanLiteral, "false") as FreshableIntrinsicType;
        const regularFalseType = createIntrinsicType(TypeFlags.BooleanLiteral, "false") as FreshableIntrinsicType;
        const trueType = createIntrinsicType(TypeFlags.BooleanLiteral, "true") as FreshableIntrinsicType;
        const regularTrueType = createIntrinsicType(TypeFlags.BooleanLiteral, "true") as FreshableIntrinsicType;
        trueType.regularType = regularTrueType;
        trueType.freshType = trueType;
        regularTrueType.regularType = regularTrueType;
        regularTrueType.freshType = trueType;
        falseType.regularType = regularFalseType;
        falseType.freshType = falseType;
        regularFalseType.regularType = regularFalseType;
        regularFalseType.freshType = falseType;
        const booleanType = createBooleanType([regularFalseType, regularTrueType]);
        // Also mark all combinations of fresh/regular booleans as "Boolean" so they print as `boolean` instead of `true | false`
        // (The union is cached, so simply doing the marking here is sufficient)
        createBooleanType([regularFalseType, trueType]);
        createBooleanType([falseType, regularTrueType]);
        createBooleanType([falseType, trueType]);
        const esSymbolType = createIntrinsicType(TypeFlags.ESSymbol, "symbol");
        const voidType = createIntrinsicType(TypeFlags.Void, "void");
        const neverType = createIntrinsicType(TypeFlags.Never, "never");
        const silentNeverType = createIntrinsicType(TypeFlags.Never, "never");
        const nonInferrableType = createIntrinsicType(TypeFlags.Never, "never", ObjectFlags.NonInferrableType);
        const implicitNeverType = createIntrinsicType(TypeFlags.Never, "never");
        const unreachableNeverType = createIntrinsicType(TypeFlags.Never, "never");
        const nonPrimitiveType = createIntrinsicType(TypeFlags.NonPrimitive, "object");
        const stringNumberSymbolType = getUnionType([stringType, numberType, esSymbolType]);
        const keyofConstraintType = keyofStringsOnly ? stringType : stringNumberSymbolType;
        const numberOrBigIntType = getUnionType([numberType, bigintType]);

        const restrictiveMapper: TypeMapper = makeFunctionTypeMapper(t => t.flags & TypeFlags.TypeParameter ? getRestrictiveTypeParameter(<TypeParameter>t) : t);
        const permissiveMapper: TypeMapper = makeFunctionTypeMapper(t => t.flags & TypeFlags.TypeParameter ? wildcardType : t);

        const emptyObjectType = createAnonymousType(undefined, emptySymbols, emptyArray, emptyArray, undefined, undefined);
        const emptyJsxObjectType = createAnonymousType(undefined, emptySymbols, emptyArray, emptyArray, undefined, undefined);
        emptyJsxObjectType.objectFlags |= ObjectFlags.JsxAttributes;

        const emptyTypeLiteralSymbol = createSymbol(SymbolFlags.TypeLiteral, InternalSymbolName.Type);
        emptyTypeLiteralSymbol.members = createSymbolTable();
        const emptyTypeLiteralType = createAnonymousType(emptyTypeLiteralSymbol, emptySymbols, emptyArray, emptyArray, undefined, undefined);

        const emptyGenericType = <GenericType><ObjectType>createAnonymousType(undefined, emptySymbols, emptyArray, emptyArray, undefined, undefined);
        emptyGenericType.instantiations = new Map<string, TypeReference>();

        const anyFunctionType = createAnonymousType(undefined, emptySymbols, emptyArray, emptyArray, undefined, undefined);
        // The anyFunctionType contains the anyFunctionType by definition. The flag is further propagated
        // in getPropagatingFlagsOfTypes, and it is checked in inferFromTypes.
        anyFunctionType.objectFlags |= ObjectFlags.NonInferrableType;

        const noConstraintType = createAnonymousType(undefined, emptySymbols, emptyArray, emptyArray, undefined, undefined);
        const circularConstraintType = createAnonymousType(undefined, emptySymbols, emptyArray, emptyArray, undefined, undefined);
        const resolvingDefaultType = createAnonymousType(undefined, emptySymbols, emptyArray, emptyArray, undefined, undefined);

        const markerSuperType = createTypeParameter();
        const markerSubType = createTypeParameter();
        markerSubType.constraint = markerSuperType;
        const markerOtherType = createTypeParameter();

        const noTypePredicate = createTypePredicate(TypePredicateKind.Identifier, "<<unresolved>>", 0, anyType);

        const anySignature = createSignature(undefined, undefined, undefined, emptyArray, anyType, /*resolvedTypePredicate*/ undefined, 0, SignatureFlags.None);
        const unknownSignature = createSignature(undefined, undefined, undefined, emptyArray, errorType, /*resolvedTypePredicate*/ undefined, 0, SignatureFlags.None);
        const resolvingSignature = createSignature(undefined, undefined, undefined, emptyArray, anyType, /*resolvedTypePredicate*/ undefined, 0, SignatureFlags.None);
        const silentNeverSignature = createSignature(undefined, undefined, undefined, emptyArray, silentNeverType, /*resolvedTypePredicate*/ undefined, 0, SignatureFlags.None);

        const enumNumberIndexInfo = createIndexInfo(stringType, /*isReadonly*/ true);

        const iterationTypesCache = new Map<string, IterationTypes>(); // cache for common IterationTypes instances
        const noIterationTypes: IterationTypes = {
            get yieldType(): Type { return Debug.fail("Not supported"); },
            get returnType(): Type { return Debug.fail("Not supported"); },
            get nextType(): Type { return Debug.fail("Not supported"); },
        };

        const anyIterationTypes = createIterationTypes(anyType, anyType, anyType);
        const anyIterationTypesExceptNext = createIterationTypes(anyType, anyType, unknownType);
        const defaultIterationTypes = createIterationTypes(neverType, anyType, undefinedType); // default iteration types for `Iterator`.

        const asyncIterationTypesResolver: IterationTypesResolver = {
            iterableCacheKey: "iterationTypesOfAsyncIterable",
            iteratorCacheKey: "iterationTypesOfAsyncIterator",
            iteratorSymbolName: "asyncIterator",
            getGlobalIteratorType: getGlobalAsyncIteratorType,
            getGlobalIterableType: getGlobalAsyncIterableType,
            getGlobalIterableIteratorType: getGlobalAsyncIterableIteratorType,
            getGlobalGeneratorType: getGlobalAsyncGeneratorType,
            resolveIterationType: getAwaitedType,
            mustHaveANextMethodDiagnostic: Diagnostics.An_async_iterator_must_have_a_next_method,
            mustBeAMethodDiagnostic: Diagnostics.The_0_property_of_an_async_iterator_must_be_a_method,
            mustHaveAValueDiagnostic: Diagnostics.The_type_returned_by_the_0_method_of_an_async_iterator_must_be_a_promise_for_a_type_with_a_value_property,
        };

        const syncIterationTypesResolver: IterationTypesResolver = {
            iterableCacheKey: "iterationTypesOfIterable",
            iteratorCacheKey: "iterationTypesOfIterator",
            iteratorSymbolName: "iterator",
            getGlobalIteratorType,
            getGlobalIterableType,
            getGlobalIterableIteratorType,
            getGlobalGeneratorType,
            resolveIterationType: (type, _errorNode) => type,
            mustHaveANextMethodDiagnostic: Diagnostics.An_iterator_must_have_a_next_method,
            mustBeAMethodDiagnostic: Diagnostics.The_0_property_of_an_iterator_must_be_a_method,
            mustHaveAValueDiagnostic: Diagnostics.The_type_returned_by_the_0_method_of_an_iterator_must_have_a_value_property,
        };

        interface DuplicateInfoForSymbol {
            readonly firstFileLocations: Declaration[];
            readonly secondFileLocations: Declaration[];
            readonly isBlockScoped: boolean;
        }
        interface DuplicateInfoForFiles {
            readonly firstFile: SourceFile;
            readonly secondFile: SourceFile;
            /** Key is symbol name. */
            readonly conflictingSymbols: ESMap<string, DuplicateInfoForSymbol>;
        }
        /** Key is "/path/to/a.ts|/path/to/b.ts". */
        let amalgamatedDuplicates: ESMap<string, DuplicateInfoForFiles> | undefined;
        const reverseMappedCache = new Map<string, Type | undefined>();
        let inInferTypeForHomomorphicMappedType = false;
        let ambientModulesCache: Symbol[] | undefined;
        /**
         * List of every ambient module with a "*" wildcard.
         * Unlike other ambient modules, these can't be stored in `globals` because symbol tables only deal with exact matches.
         * This is only used if there is no exact match.
         */
        let patternAmbientModules: PatternAmbientModule[];
        let patternAmbientModuleAugmentations: ESMap<string, Symbol> | undefined;

        let globalObjectType: ObjectType;
        let globalFunctionType: ObjectType;
        let globalCallableFunctionType: ObjectType;
        let globalNewableFunctionType: ObjectType;
        let globalArrayType: GenericType;
        let globalReadonlyArrayType: GenericType;
        let globalStringType: ObjectType;
        let globalNumberType: ObjectType;
        let globalBooleanType: ObjectType;
        let globalRegExpType: ObjectType;
        let globalThisType: GenericType;
        let anyArrayType: Type;
        let autoArrayType: Type;
        let anyReadonlyArrayType: Type;
        let deferredGlobalNonNullableTypeAlias: Symbol;

        // The library files are only loaded when the feature is used.
        // This allows users to just specify library files they want to used through --lib
        // and they will not get an error from not having unrelated library files
        let deferredGlobalESSymbolConstructorSymbol: Symbol | undefined;
        let deferredGlobalESSymbolType: ObjectType;
        let deferredGlobalTypedPropertyDescriptorType: GenericType;
        let deferredGlobalPromiseType: GenericType;
        let deferredGlobalPromiseLikeType: GenericType;
        let deferredGlobalPromiseConstructorSymbol: Symbol | undefined;
        let deferredGlobalPromiseConstructorLikeType: ObjectType;
        let deferredGlobalIterableType: GenericType;
        let deferredGlobalIteratorType: GenericType;
        let deferredGlobalIterableIteratorType: GenericType;
        let deferredGlobalGeneratorType: GenericType;
        let deferredGlobalIteratorYieldResultType: GenericType;
        let deferredGlobalIteratorReturnResultType: GenericType;
        let deferredGlobalAsyncIterableType: GenericType;
        let deferredGlobalAsyncIteratorType: GenericType;
        let deferredGlobalAsyncIterableIteratorType: GenericType;
        let deferredGlobalAsyncGeneratorType: GenericType;
        let deferredGlobalTemplateStringsArrayType: ObjectType;
        let deferredGlobalImportMetaType: ObjectType;
        let deferredGlobalExtractSymbol: Symbol;
        let deferredGlobalOmitSymbol: Symbol;
        let deferredGlobalBigIntType: ObjectType;

        const allPotentiallyUnusedIdentifiers = new Map<Path, PotentiallyUnusedIdentifier[]>(); // key is file name

        let flowLoopStart = 0;
        let flowLoopCount = 0;
        let sharedFlowCount = 0;
        let flowAnalysisDisabled = false;
        let flowInvocationCount = 0;
        let lastFlowNode: FlowNode | undefined;
        let lastFlowNodeReachable: boolean;
        let flowTypeCache: Type[] | undefined;

        const emptyStringType = getLiteralType("");
        const zeroType = getLiteralType(0);
        const zeroBigIntType = getLiteralType({ negative: false, base10Value: "0" });

        const resolutionTargets: TypeSystemEntity[] = [];
        const resolutionResults: boolean[] = [];
        const resolutionPropertyNames: TypeSystemPropertyName[] = [];

        let suggestionCount = 0;
        const maximumSuggestionCount = 10;
        const mergedSymbols: Symbol[] = [];
        const symbolLinks: SymbolLinks[] = [];
        const nodeLinks: NodeLinks[] = [];
        const flowLoopCaches: ESMap<string, Type>[] = [];
        const flowLoopNodes: FlowNode[] = [];
        const flowLoopKeys: string[] = [];
        const flowLoopTypes: Type[][] = [];
        const sharedFlowNodes: FlowNode[] = [];
        const sharedFlowTypes: FlowType[] = [];
        const flowNodeReachable: (boolean | undefined)[] = [];
        const flowNodePostSuper: (boolean | undefined)[] = [];
        const potentialThisCollisions: Node[] = [];
        const potentialNewTargetCollisions: Node[] = [];
        const potentialWeakMapCollisions: Node[] = [];
        const awaitedTypeStack: number[] = [];

        const diagnostics = createDiagnosticCollection();
        const suggestionDiagnostics = createDiagnosticCollection();

        const typeofTypesByName: ReadonlyESMap<string, Type> = new Map(getEntries({
            string: stringType,
            number: numberType,
            bigint: bigintType,
            boolean: booleanType,
            symbol: esSymbolType,
            undefined: undefinedType
        }));
        const typeofType = createTypeofType();

        let _jsxNamespace: __String;
        let _jsxFactoryEntity: EntityName | undefined;
        let outofbandVarianceMarkerHandler: ((onlyUnreliable: boolean) => void) | undefined;

        const subtypeRelation = new Map<string, RelationComparisonResult>();
        const strictSubtypeRelation = new Map<string, RelationComparisonResult>();
        const assignableRelation = new Map<string, RelationComparisonResult>();
        const comparableRelation = new Map<string, RelationComparisonResult>();
        const identityRelation = new Map<string, RelationComparisonResult>();
        const enumRelation = new Map<string, RelationComparisonResult>();

        const builtinGlobals = createSymbolTable();
        builtinGlobals.set(undefinedSymbol.escapedName, undefinedSymbol);

        initializeTypeChecker();

        return checker;

        function getJsxNamespace(location: Node | undefined): __String {
            if (location) {
                const file = getSourceFileOfNode(location);
                if (file) {
                    if (isJsxOpeningFragment(location)) {
                        if (file.localJsxFragmentNamespace) {
                            return file.localJsxFragmentNamespace;
                        }
                        const jsxFragmentPragma = file.pragmas.get("jsxfrag");
                        if (jsxFragmentPragma) {
                            const chosenPragma = isArray(jsxFragmentPragma) ? jsxFragmentPragma[0] : jsxFragmentPragma;
                            file.localJsxFragmentFactory = parseIsolatedEntityName(chosenPragma.arguments.factory, languageVersion);
                            visitNode(file.localJsxFragmentFactory, markAsSynthetic);
                            if (file.localJsxFragmentFactory) {
                                return file.localJsxFragmentNamespace = getFirstIdentifier(file.localJsxFragmentFactory).escapedText;
                            }
                        }
                        const entity = getJsxFragmentFactoryEntity(location);
                        if (entity) {
                            file.localJsxFragmentFactory = entity;
                            return file.localJsxFragmentNamespace = getFirstIdentifier(entity).escapedText;
                        }
                    }
                    else {
                        if (file.localJsxNamespace) {
                            return file.localJsxNamespace;
                        }
                        const jsxPragma = file.pragmas.get("jsx");
                        if (jsxPragma) {
                            const chosenPragma = isArray(jsxPragma) ? jsxPragma[0] : jsxPragma;
                            file.localJsxFactory = parseIsolatedEntityName(chosenPragma.arguments.factory, languageVersion);
                            visitNode(file.localJsxFactory, markAsSynthetic);
                            if (file.localJsxFactory) {
                                return file.localJsxNamespace = getFirstIdentifier(file.localJsxFactory).escapedText;
                            }
                        }
                    }
                }
            }
            if (!_jsxNamespace) {
                _jsxNamespace = "React" as __String;
                if (compilerOptions.jsxFactory) {
                    _jsxFactoryEntity = parseIsolatedEntityName(compilerOptions.jsxFactory, languageVersion);
                    visitNode(_jsxFactoryEntity, markAsSynthetic);
                    if (_jsxFactoryEntity) {
                        _jsxNamespace = getFirstIdentifier(_jsxFactoryEntity).escapedText;
                    }
                }
                else if (compilerOptions.reactNamespace) {
                    _jsxNamespace = escapeLeadingUnderscores(compilerOptions.reactNamespace);
                }
            }
            if (!_jsxFactoryEntity) {
                _jsxFactoryEntity = factory.createQualifiedName(factory.createIdentifier(unescapeLeadingUnderscores(_jsxNamespace)), "createElement");
            }
            return _jsxNamespace;

            function markAsSynthetic(node: Node): VisitResult<Node> {
                setTextRangePosEnd(node, -1, -1);
                return visitEachChild(node, markAsSynthetic, nullTransformationContext);
            }
        }

        function getEmitResolver(sourceFile: SourceFile, cancellationToken: CancellationToken) {
            // Ensure we have all the type information in place for this file so that all the
            // emitter questions of this resolver will return the right information.
            getDiagnostics(sourceFile, cancellationToken);
            return emitResolver;
        }

        function lookupOrIssueError(location: Node | undefined, message: DiagnosticMessage, arg0?: string | number, arg1?: string | number, arg2?: string | number, arg3?: string | number): Diagnostic {
            const diagnostic = location
                ? createDiagnosticForNode(location, message, arg0, arg1, arg2, arg3)
                : createCompilerDiagnostic(message, arg0, arg1, arg2, arg3);
            const existing = diagnostics.lookup(diagnostic);
            if (existing) {
                return existing;
            }
            else {
                diagnostics.add(diagnostic);
                return diagnostic;
            }
        }

        function errorSkippedOn(key: keyof CompilerOptions, location: Node | undefined, message: DiagnosticMessage, arg0?: string | number, arg1?: string | number, arg2?: string | number, arg3?: string | number): Diagnostic {
            const diagnostic = error(location, message, arg0, arg1, arg2, arg3);
            diagnostic.skippedOn = key;
            return diagnostic;
        }

        function error(location: Node | undefined, message: DiagnosticMessage, arg0?: string | number, arg1?: string | number, arg2?: string | number, arg3?: string | number): Diagnostic {
            const diagnostic = location
                ? createDiagnosticForNode(location, message, arg0, arg1, arg2, arg3)
                : createCompilerDiagnostic(message, arg0, arg1, arg2, arg3);
            diagnostics.add(diagnostic);
            return diagnostic;
        }

        function addErrorOrSuggestion(isError: boolean, diagnostic: DiagnosticWithLocation) {
            if (isError) {
                diagnostics.add(diagnostic);
            }
            else {
                suggestionDiagnostics.add({ ...diagnostic, category: DiagnosticCategory.Suggestion });
            }
        }
        function errorOrSuggestion(isError: boolean, location: Node, message: DiagnosticMessage | DiagnosticMessageChain, arg0?: string | number, arg1?: string | number, arg2?: string | number, arg3?: string | number): void {
            addErrorOrSuggestion(isError, "message" in message ? createDiagnosticForNode(location, message, arg0, arg1, arg2, arg3) : createDiagnosticForNodeFromMessageChain(location, message)); // eslint-disable-line no-in-operator
        }

        function errorAndMaybeSuggestAwait(
            location: Node,
            maybeMissingAwait: boolean,
            message: DiagnosticMessage,
            arg0?: string | number | undefined, arg1?: string | number | undefined, arg2?: string | number | undefined, arg3?: string | number | undefined): Diagnostic {
            const diagnostic = error(location, message, arg0, arg1, arg2, arg3);
            if (maybeMissingAwait) {
                const related = createDiagnosticForNode(location, Diagnostics.Did_you_forget_to_use_await);
                addRelatedInfo(diagnostic, related);
            }
            return diagnostic;
        }

        function createSymbol(flags: SymbolFlags, name: __String, checkFlags?: CheckFlags) {
            symbolCount++;
            const symbol = <TransientSymbol>(new Symbol(flags | SymbolFlags.Transient, name));
            symbol.checkFlags = checkFlags || 0;
            return symbol;
        }

        function getExcludedSymbolFlags(flags: SymbolFlags): SymbolFlags {
            let result: SymbolFlags = 0;
            if (flags & SymbolFlags.BlockScopedVariable) result |= SymbolFlags.BlockScopedVariableExcludes;
            if (flags & SymbolFlags.FunctionScopedVariable) result |= SymbolFlags.FunctionScopedVariableExcludes;
            if (flags & SymbolFlags.Property) result |= SymbolFlags.PropertyExcludes;
            if (flags & SymbolFlags.EnumMember) result |= SymbolFlags.EnumMemberExcludes;
            if (flags & SymbolFlags.Function) result |= SymbolFlags.FunctionExcludes;
            if (flags & SymbolFlags.Class) result |= SymbolFlags.ClassExcludes;
            if (flags & SymbolFlags.Interface) result |= SymbolFlags.InterfaceExcludes;
            if (flags & SymbolFlags.RegularEnum) result |= SymbolFlags.RegularEnumExcludes;
            if (flags & SymbolFlags.ConstEnum) result |= SymbolFlags.ConstEnumExcludes;
            if (flags & SymbolFlags.ValueModule) result |= SymbolFlags.ValueModuleExcludes;
            if (flags & SymbolFlags.Method) result |= SymbolFlags.MethodExcludes;
            if (flags & SymbolFlags.GetAccessor) result |= SymbolFlags.GetAccessorExcludes;
            if (flags & SymbolFlags.SetAccessor) result |= SymbolFlags.SetAccessorExcludes;
            if (flags & SymbolFlags.TypeParameter) result |= SymbolFlags.TypeParameterExcludes;
            if (flags & SymbolFlags.TypeAlias) result |= SymbolFlags.TypeAliasExcludes;
            if (flags & SymbolFlags.Alias) result |= SymbolFlags.AliasExcludes;
            return result;
        }

        function recordMergedSymbol(target: Symbol, source: Symbol) {
            if (!source.mergeId) {
                source.mergeId = nextMergeId;
                nextMergeId++;
            }
            mergedSymbols[source.mergeId] = target;
        }

        function cloneSymbol(symbol: Symbol): Symbol {
            const result = createSymbol(symbol.flags, symbol.escapedName);
            result.declarations = symbol.declarations ? symbol.declarations.slice() : [];
            result.parent = symbol.parent;
            if (symbol.valueDeclaration) result.valueDeclaration = symbol.valueDeclaration;
            if (symbol.constEnumOnlyModule) result.constEnumOnlyModule = true;
            if (symbol.members) result.members = new Map(symbol.members);
            if (symbol.exports) result.exports = new Map(symbol.exports);
            recordMergedSymbol(result, symbol);
            return result;
        }

        /**
         * Note: if target is transient, then it is mutable, and mergeSymbol with both mutate and return it.
         * If target is not transient, mergeSymbol will produce a transient clone, mutate that and return it.
         */
        function mergeSymbol(target: Symbol, source: Symbol, unidirectional = false): Symbol {
            if (!(target.flags & getExcludedSymbolFlags(source.flags)) ||
                (source.flags | target.flags) & SymbolFlags.Assignment) {
                if (source === target) {
                    // This can happen when an export assigned namespace exports something also erroneously exported at the top level
                    // See `declarationFileNoCrashOnExtraExportModifier` for an example
                    return target;
                }
                if (!(target.flags & SymbolFlags.Transient)) {
                    const resolvedTarget = resolveSymbol(target);
                    if (resolvedTarget === unknownSymbol) {
                        return source;
                    }
                    target = cloneSymbol(resolvedTarget);
                }
                // Javascript static-property-assignment declarations always merge, even though they are also values
                if (source.flags & SymbolFlags.ValueModule && target.flags & SymbolFlags.ValueModule && target.constEnumOnlyModule && !source.constEnumOnlyModule) {
                    // reset flag when merging instantiated module into value module that has only const enums
                    target.constEnumOnlyModule = false;
                }
                target.flags |= source.flags;
                if (source.valueDeclaration) {
                    setValueDeclaration(target, source.valueDeclaration);
                }
                addRange(target.declarations, source.declarations);
                if (source.members) {
                    if (!target.members) target.members = createSymbolTable();
                    mergeSymbolTable(target.members, source.members, unidirectional);
                }
                if (source.exports) {
                    if (!target.exports) target.exports = createSymbolTable();
                    mergeSymbolTable(target.exports, source.exports, unidirectional);
                }
                if (!unidirectional) {
                    recordMergedSymbol(target, source);
                }
            }
            else if (target.flags & SymbolFlags.NamespaceModule) {
                // Do not report an error when merging `var globalThis` with the built-in `globalThis`,
                // as we will already report a "Declaration name conflicts..." error, and this error
                // won't make much sense.
                if (target !== globalThisSymbol) {
                    error(getNameOfDeclaration(source.declarations[0]), Diagnostics.Cannot_augment_module_0_with_value_exports_because_it_resolves_to_a_non_module_entity, symbolToString(target));
                }
            }
            else { // error
                const isEitherEnum = !!(target.flags & SymbolFlags.Enum || source.flags & SymbolFlags.Enum);
                const isEitherBlockScoped = !!(target.flags & SymbolFlags.BlockScopedVariable || source.flags & SymbolFlags.BlockScopedVariable);
                const message = isEitherEnum
                    ? Diagnostics.Enum_declarations_can_only_merge_with_namespace_or_other_enum_declarations
                    : isEitherBlockScoped
                        ? Diagnostics.Cannot_redeclare_block_scoped_variable_0
                        : Diagnostics.Duplicate_identifier_0;
                const sourceSymbolFile = source.declarations && getSourceFileOfNode(source.declarations[0]);
                const targetSymbolFile = target.declarations && getSourceFileOfNode(target.declarations[0]);
                const symbolName = symbolToString(source);

                // Collect top-level duplicate identifier errors into one mapping, so we can then merge their diagnostics if there are a bunch
                if (sourceSymbolFile && targetSymbolFile && amalgamatedDuplicates && !isEitherEnum && sourceSymbolFile !== targetSymbolFile) {
                    const firstFile = comparePaths(sourceSymbolFile.path, targetSymbolFile.path) === Comparison.LessThan ? sourceSymbolFile : targetSymbolFile;
                    const secondFile = firstFile === sourceSymbolFile ? targetSymbolFile : sourceSymbolFile;
                    const filesDuplicates = getOrUpdate(amalgamatedDuplicates, `${firstFile.path}|${secondFile.path}`, () =>
                        ({ firstFile, secondFile, conflictingSymbols: new Map() } as DuplicateInfoForFiles));
                    const conflictingSymbolInfo = getOrUpdate(filesDuplicates.conflictingSymbols, symbolName, () =>
                        ({ isBlockScoped: isEitherBlockScoped, firstFileLocations: [], secondFileLocations: [] } as DuplicateInfoForSymbol));
                    addDuplicateLocations(conflictingSymbolInfo.firstFileLocations, source);
                    addDuplicateLocations(conflictingSymbolInfo.secondFileLocations, target);
                }
                else {
                    addDuplicateDeclarationErrorsForSymbols(source, message, symbolName, target);
                    addDuplicateDeclarationErrorsForSymbols(target, message, symbolName, source);
                }
            }
            return target;

            function addDuplicateLocations(locs: Declaration[], symbol: Symbol): void {
                for (const decl of symbol.declarations) {
                    pushIfUnique(locs, decl);
                }
            }
        }

        function addDuplicateDeclarationErrorsForSymbols(target: Symbol, message: DiagnosticMessage, symbolName: string, source: Symbol) {
            forEach(target.declarations, node => {
                addDuplicateDeclarationError(node, message, symbolName, source.declarations);
            });
        }

        function addDuplicateDeclarationError(node: Declaration, message: DiagnosticMessage, symbolName: string, relatedNodes: readonly Declaration[] | undefined) {
            const errorNode = (getExpandoInitializer(node, /*isPrototypeAssignment*/ false) ? getNameOfExpando(node) : getNameOfDeclaration(node)) || node;
            const err = lookupOrIssueError(errorNode, message, symbolName);
            for (const relatedNode of relatedNodes || emptyArray) {
                const adjustedNode = (getExpandoInitializer(relatedNode, /*isPrototypeAssignment*/ false) ? getNameOfExpando(relatedNode) : getNameOfDeclaration(relatedNode)) || relatedNode;
                if (adjustedNode === errorNode) continue;
                err.relatedInformation = err.relatedInformation || [];
                const leadingMessage = createDiagnosticForNode(adjustedNode, Diagnostics._0_was_also_declared_here, symbolName);
                const followOnMessage = createDiagnosticForNode(adjustedNode, Diagnostics.and_here);
                if (length(err.relatedInformation) >= 5 || some(err.relatedInformation, r => compareDiagnostics(r, followOnMessage) === Comparison.EqualTo || compareDiagnostics(r, leadingMessage) === Comparison.EqualTo)) continue;
                addRelatedInfo(err, !length(err.relatedInformation) ? leadingMessage : followOnMessage);
            }
        }

        function combineSymbolTables(first: SymbolTable | undefined, second: SymbolTable | undefined): SymbolTable | undefined {
            if (!first?.size) return second;
            if (!second?.size) return first;
            const combined = createSymbolTable();
            mergeSymbolTable(combined, first);
            mergeSymbolTable(combined, second);
            return combined;
        }

        function mergeSymbolTable(target: SymbolTable, source: SymbolTable, unidirectional = false) {
            source.forEach((sourceSymbol, id) => {
                const targetSymbol = target.get(id);
                target.set(id, targetSymbol ? mergeSymbol(targetSymbol, sourceSymbol, unidirectional) : sourceSymbol);
            });
        }

        function mergeModuleAugmentation(moduleName: StringLiteral | Identifier): void {
            const moduleAugmentation = <ModuleDeclaration>moduleName.parent;
            if (moduleAugmentation.symbol.declarations[0] !== moduleAugmentation) {
                // this is a combined symbol for multiple augmentations within the same file.
                // its symbol already has accumulated information for all declarations
                // so we need to add it just once - do the work only for first declaration
                Debug.assert(moduleAugmentation.symbol.declarations.length > 1);
                return;
            }

            if (isGlobalScopeAugmentation(moduleAugmentation)) {
                mergeSymbolTable(globals, moduleAugmentation.symbol.exports!);
            }
            else {
                // find a module that about to be augmented
                // do not validate names of augmentations that are defined in ambient context
                const moduleNotFoundError = !(moduleName.parent.parent.flags & NodeFlags.Ambient)
                    ? Diagnostics.Invalid_module_name_in_augmentation_module_0_cannot_be_found
                    : undefined;
                let mainModule = resolveExternalModuleNameWorker(moduleName, moduleName, moduleNotFoundError, /*isForAugmentation*/ true);
                if (!mainModule) {
                    return;
                }
                // obtain item referenced by 'export='
                mainModule = resolveExternalModuleSymbol(mainModule);
                if (mainModule.flags & SymbolFlags.Namespace) {
                    // If we're merging an augmentation to a pattern ambient module, we want to
                    // perform the merge unidirectionally from the augmentation ('a.foo') to
                    // the pattern ('*.foo'), so that 'getMergedSymbol()' on a.foo gives you
                    // all the exports both from the pattern and from the augmentation, but
                    // 'getMergedSymbol()' on *.foo only gives you exports from *.foo.
                    if (some(patternAmbientModules, module => mainModule === module.symbol)) {
                        const merged = mergeSymbol(moduleAugmentation.symbol, mainModule, /*unidirectional*/ true);
                        if (!patternAmbientModuleAugmentations) {
                            patternAmbientModuleAugmentations = new Map();
                        }
                        // moduleName will be a StringLiteral since this is not `declare global`.
                        patternAmbientModuleAugmentations.set((moduleName as StringLiteral).text, merged);
                    }
                    else {
                        if (mainModule.exports?.get(InternalSymbolName.ExportStar) && moduleAugmentation.symbol.exports?.size) {
                            // We may need to merge the module augmentation's exports into the target symbols of the resolved exports
                            const resolvedExports = getResolvedMembersOrExportsOfSymbol(mainModule, MembersOrExportsResolutionKind.resolvedExports);
                            for (const [key, value] of arrayFrom(moduleAugmentation.symbol.exports.entries())) {
                                if (resolvedExports.has(key) && !mainModule.exports.has(key)) {
                                    mergeSymbol(resolvedExports.get(key)!, value);
                                }
                            }
                        }
                        mergeSymbol(mainModule, moduleAugmentation.symbol);
                    }
                }
                else {
                    // moduleName will be a StringLiteral since this is not `declare global`.
                    error(moduleName, Diagnostics.Cannot_augment_module_0_because_it_resolves_to_a_non_module_entity, (moduleName as StringLiteral).text);
                }
            }
        }

        function addToSymbolTable(target: SymbolTable, source: SymbolTable, message: DiagnosticMessage) {
            source.forEach((sourceSymbol, id) => {
                const targetSymbol = target.get(id);
                if (targetSymbol) {
                    // Error on redeclarations
                    forEach(targetSymbol.declarations, addDeclarationDiagnostic(unescapeLeadingUnderscores(id), message));
                }
                else {
                    target.set(id, sourceSymbol);
                }
            });

            function addDeclarationDiagnostic(id: string, message: DiagnosticMessage) {
                return (declaration: Declaration) => diagnostics.add(createDiagnosticForNode(declaration, message, id));
            }
        }

        function getSymbolLinks(symbol: Symbol): SymbolLinks {
            if (symbol.flags & SymbolFlags.Transient) return <TransientSymbol>symbol;
            const id = getSymbolId(symbol);
            return symbolLinks[id] || (symbolLinks[id] = new (<any>SymbolLinks)());
        }

        function getNodeLinks(node: Node): NodeLinks {
            const nodeId = getNodeId(node);
            return nodeLinks[nodeId] || (nodeLinks[nodeId] = new (<any>NodeLinks)());
        }

        function isGlobalSourceFile(node: Node) {
            return node.kind === SyntaxKind.SourceFile && !isExternalOrCommonJsModule(<SourceFile>node);
        }

        function getSymbol(symbols: SymbolTable, name: __String, meaning: SymbolFlags): Symbol | undefined {
            if (meaning) {
                const symbol = getMergedSymbol(symbols.get(name));
                if (symbol) {
                    Debug.assert((getCheckFlags(symbol) & CheckFlags.Instantiated) === 0, "Should never get an instantiated symbol here.");
                    if (symbol.flags & meaning) {
                        return symbol;
                    }
                    if (symbol.flags & SymbolFlags.Alias) {
                        const target = resolveAlias(symbol);
                        // Unknown symbol means an error occurred in alias resolution, treat it as positive answer to avoid cascading errors
                        if (target === unknownSymbol || target.flags & meaning) {
                            return symbol;
                        }
                    }
                }
            }
            // return undefined if we can't find a symbol.
        }

        /**
         * Get symbols that represent parameter-property-declaration as parameter and as property declaration
         * @param parameter a parameterDeclaration node
         * @param parameterName a name of the parameter to get the symbols for.
         * @return a tuple of two symbols
         */
        function getSymbolsOfParameterPropertyDeclaration(parameter: ParameterDeclaration, parameterName: __String): [Symbol, Symbol] {
            const constructorDeclaration = parameter.parent;
            const classDeclaration = parameter.parent.parent;

            const parameterSymbol = getSymbol(constructorDeclaration.locals!, parameterName, SymbolFlags.Value);
            const propertySymbol = getSymbol(getMembersOfSymbol(classDeclaration.symbol), parameterName, SymbolFlags.Value);

            if (parameterSymbol && propertySymbol) {
                return [parameterSymbol, propertySymbol];
            }

            return Debug.fail("There should exist two symbols, one as property declaration and one as parameter declaration");
        }

        function isBlockScopedNameDeclaredBeforeUse(declaration: Declaration, usage: Node): boolean {
            const declarationFile = getSourceFileOfNode(declaration);
            const useFile = getSourceFileOfNode(usage);
            const declContainer = getEnclosingBlockScopeContainer(declaration);
            if (declarationFile !== useFile) {
                if ((moduleKind && (declarationFile.externalModuleIndicator || useFile.externalModuleIndicator)) ||
                    (!outFile(compilerOptions)) ||
                    isInTypeQuery(usage) ||
                    declaration.flags & NodeFlags.Ambient) {
                    // nodes are in different files and order cannot be determined
                    return true;
                }
                // declaration is after usage
                // can be legal if usage is deferred (i.e. inside function or in initializer of instance property)
                if (isUsedInFunctionOrInstanceProperty(usage, declaration)) {
                    return true;
                }
                const sourceFiles = host.getSourceFiles();
                return sourceFiles.indexOf(declarationFile) <= sourceFiles.indexOf(useFile);
            }

            if (declaration.pos <= usage.pos && !(isPropertyDeclaration(declaration) && isThisProperty(usage.parent) && !declaration.initializer && !declaration.exclamationToken)) {
                // declaration is before usage
                if (declaration.kind === SyntaxKind.BindingElement) {
                    // still might be illegal if declaration and usage are both binding elements (eg var [a = b, b = b] = [1, 2])
                    const errorBindingElement = getAncestor(usage, SyntaxKind.BindingElement) as BindingElement;
                    if (errorBindingElement) {
                        return findAncestor(errorBindingElement, isBindingElement) !== findAncestor(declaration, isBindingElement) ||
                            declaration.pos < errorBindingElement.pos;
                    }
                    // or it might be illegal if usage happens before parent variable is declared (eg var [a] = a)
                    return isBlockScopedNameDeclaredBeforeUse(getAncestor(declaration, SyntaxKind.VariableDeclaration) as Declaration, usage);
                }
                else if (declaration.kind === SyntaxKind.VariableDeclaration) {
                    // still might be illegal if usage is in the initializer of the variable declaration (eg var a = a)
                    return !isImmediatelyUsedInInitializerOfBlockScopedVariable(declaration as VariableDeclaration, usage);
                }
                else if (isClassDeclaration(declaration)) {
                    // still might be illegal if the usage is within a computed property name in the class (eg class A { static p = "a"; [A.p]() {} })
                    return !findAncestor(usage, n => isComputedPropertyName(n) && n.parent.parent === declaration);
                }
                else if (isPropertyDeclaration(declaration)) {
                    // still might be illegal if a self-referencing property initializer (eg private x = this.x)
                    return !isPropertyImmediatelyReferencedWithinDeclaration(declaration, usage, /*stopAtAnyPropertyDeclaration*/ false);
                }
                else if (isParameterPropertyDeclaration(declaration, declaration.parent)) {
                    // foo = this.bar is illegal in esnext+useDefineForClassFields when bar is a parameter property
                    return !(compilerOptions.target === ScriptTarget.ESNext && !!compilerOptions.useDefineForClassFields
                             && getContainingClass(declaration) === getContainingClass(usage)
                             && isUsedInFunctionOrInstanceProperty(usage, declaration));
                }
                return true;
            }


            // declaration is after usage, but it can still be legal if usage is deferred:
            // 1. inside an export specifier
            // 2. inside a function
            // 3. inside an instance property initializer, a reference to a non-instance property
            //    (except when target: "esnext" and useDefineForClassFields: true and the reference is to a parameter property)
            // 4. inside a static property initializer, a reference to a static method in the same class
            // 5. inside a TS export= declaration (since we will move the export statement during emit to avoid TDZ)
            // or if usage is in a type context:
            // 1. inside a type query (typeof in type position)
            // 2. inside a jsdoc comment
            if (usage.parent.kind === SyntaxKind.ExportSpecifier || (usage.parent.kind === SyntaxKind.ExportAssignment && (usage.parent as ExportAssignment).isExportEquals)) {
                // export specifiers do not use the variable, they only make it available for use
                return true;
            }
            // When resolving symbols for exports, the `usage` location passed in can be the export site directly
            if (usage.kind === SyntaxKind.ExportAssignment && (usage as ExportAssignment).isExportEquals) {
                return true;
            }

            if (!!(usage.flags & NodeFlags.JSDoc) || isInTypeQuery(usage) || usageInTypeDeclaration()) {
                return true;
            }
            if (isUsedInFunctionOrInstanceProperty(usage, declaration)) {
                if (compilerOptions.target === ScriptTarget.ESNext && !!compilerOptions.useDefineForClassFields
                    && getContainingClass(declaration)
                    && (isPropertyDeclaration(declaration) || isParameterPropertyDeclaration(declaration, declaration.parent))) {
                    return !isPropertyImmediatelyReferencedWithinDeclaration(declaration, usage, /*stopAtAnyPropertyDeclaration*/ true);
                }
                else {
                    return true;
                }
            }
            return false;

            function usageInTypeDeclaration() {
                return !!findAncestor(usage, node => isInterfaceDeclaration(node) || isTypeAliasDeclaration(node));
            }

            function isImmediatelyUsedInInitializerOfBlockScopedVariable(declaration: VariableDeclaration, usage: Node): boolean {
                switch (declaration.parent.parent.kind) {
                    case SyntaxKind.VariableStatement:
                    case SyntaxKind.ForStatement:
                    case SyntaxKind.ForOfStatement:
                        // variable statement/for/for-of statement case,
                        // use site should not be inside variable declaration (initializer of declaration or binding element)
                        if (isSameScopeDescendentOf(usage, declaration, declContainer)) {
                            return true;
                        }
                        break;
                }

                // ForIn/ForOf case - use site should not be used in expression part
                const grandparent = declaration.parent.parent;
                return isForInOrOfStatement(grandparent) && isSameScopeDescendentOf(usage, grandparent.expression, declContainer);
            }

            function isUsedInFunctionOrInstanceProperty(usage: Node, declaration: Node): boolean {
                return !!findAncestor(usage, current => {
                    if (current === declContainer) {
                        return "quit";
                    }
                    if (isFunctionLike(current)) {
                        return true;
                    }

                    const initializerOfProperty = current.parent &&
                        current.parent.kind === SyntaxKind.PropertyDeclaration &&
                        (<PropertyDeclaration>current.parent).initializer === current;

                    if (initializerOfProperty) {
                        if (hasSyntacticModifier(current.parent, ModifierFlags.Static)) {
                            if (declaration.kind === SyntaxKind.MethodDeclaration) {
                                return true;
                            }
                        }
                        else {
                            const isDeclarationInstanceProperty = declaration.kind === SyntaxKind.PropertyDeclaration && !hasSyntacticModifier(declaration, ModifierFlags.Static);
                            if (!isDeclarationInstanceProperty || getContainingClass(usage) !== getContainingClass(declaration)) {
                                return true;
                            }
                        }
                    }
                    return false;
                });
            }

            /** stopAtAnyPropertyDeclaration is used for detecting ES-standard class field use-before-def errors */
            function isPropertyImmediatelyReferencedWithinDeclaration(declaration: PropertyDeclaration | ParameterPropertyDeclaration, usage: Node, stopAtAnyPropertyDeclaration: boolean) {
                // always legal if usage is after declaration
                if (usage.end > declaration.end) {
                    return false;
                }

                // still might be legal if usage is deferred (e.g. x: any = () => this.x)
                // otherwise illegal if immediately referenced within the declaration (e.g. x: any = this.x)
                const ancestorChangingReferenceScope = findAncestor(usage, (node: Node) => {
                    if (node === declaration) {
                        return "quit";
                    }

                    switch (node.kind) {
                        case SyntaxKind.ArrowFunction:
                            return true;
                        case SyntaxKind.PropertyDeclaration:
                            // even when stopping at any property declaration, they need to come from the same class
                            return stopAtAnyPropertyDeclaration &&
                                (isPropertyDeclaration(declaration) && node.parent === declaration.parent
                                 || isParameterPropertyDeclaration(declaration, declaration.parent) && node.parent === declaration.parent.parent)
                                ? "quit": true;
                        case SyntaxKind.Block:
                            switch (node.parent.kind) {
                                case SyntaxKind.GetAccessor:
                                case SyntaxKind.MethodDeclaration:
                                case SyntaxKind.SetAccessor:
                                    return true;
                                default:
                                    return false;
                            }
                        default:
                            return false;
                    }
                });

                return ancestorChangingReferenceScope === undefined;
            }
        }

        function useOuterVariableScopeInParameter(result: Symbol, location: Node, lastLocation: Node) {
            const target = getEmitScriptTarget(compilerOptions);
            const functionLocation = <FunctionLikeDeclaration>location;
            if (isParameter(lastLocation) && functionLocation.body && result.valueDeclaration.pos >= functionLocation.body.pos && result.valueDeclaration.end <= functionLocation.body.end) {
                // check for several cases where we introduce temporaries that require moving the name/initializer of the parameter to the body
                // - static field in a class expression
                // - optional chaining pre-es2020
                // - nullish coalesce pre-es2020
                // - spread assignment in binding pattern pre-es2017
                if (target >= ScriptTarget.ES2015) {
                    const links = getNodeLinks(functionLocation);
                    if (links.declarationRequiresScopeChange === undefined) {
                        links.declarationRequiresScopeChange = forEach(functionLocation.parameters, requiresScopeChange) || false;
                    }
                    return !links.declarationRequiresScopeChange;
                }
            }
            return false;

            function requiresScopeChange(node: ParameterDeclaration): boolean {
                return requiresScopeChangeWorker(node.name)
                    || !!node.initializer && requiresScopeChangeWorker(node.initializer);
            }

            function requiresScopeChangeWorker(node: Node): boolean {
                switch (node.kind) {
                    case SyntaxKind.ArrowFunction:
                    case SyntaxKind.FunctionExpression:
                    case SyntaxKind.FunctionDeclaration:
                    case SyntaxKind.Constructor:
                        // do not descend into these
                        return false;
                    case SyntaxKind.MethodDeclaration:
                    case SyntaxKind.GetAccessor:
                    case SyntaxKind.SetAccessor:
                    case SyntaxKind.PropertyAssignment:
                        return requiresScopeChangeWorker((node as MethodDeclaration | AccessorDeclaration | PropertyAssignment).name);
                    case SyntaxKind.PropertyDeclaration:
                        // static properties in classes introduce temporary variables
                        if (hasStaticModifier(node)) {
                            return target < ScriptTarget.ESNext || !compilerOptions.useDefineForClassFields;
                        }
                        return requiresScopeChangeWorker((node as PropertyDeclaration).name);
                    default:
                        // null coalesce and optional chain pre-es2020 produce temporary variables
                        if (isNullishCoalesce(node) || isOptionalChain(node)) {
                            return target < ScriptTarget.ES2020;
                        }
                        if (isBindingElement(node) && node.dotDotDotToken && isObjectBindingPattern(node.parent)) {
                            return target < ScriptTarget.ES2017;
                        }
                        if (isTypeNode(node)) return false;
                        return forEachChild(node, requiresScopeChangeWorker) || false;
                }
            }
        }

        /**
         * Resolve a given name for a given meaning at a given location. An error is reported if the name was not found and
         * the nameNotFoundMessage argument is not undefined. Returns the resolved symbol, or undefined if no symbol with
         * the given name can be found.
         *
         * @param isUse If true, this will count towards --noUnusedLocals / --noUnusedParameters.
         */
        function resolveName(
            location: Node | undefined,
            name: __String,
            meaning: SymbolFlags,
            nameNotFoundMessage: DiagnosticMessage | undefined,
            nameArg: __String | Identifier | undefined,
            isUse: boolean,
            excludeGlobals = false,
            suggestedNameNotFoundMessage?: DiagnosticMessage): Symbol | undefined {
            return resolveNameHelper(location, name, meaning, nameNotFoundMessage, nameArg, isUse, excludeGlobals, getSymbol, suggestedNameNotFoundMessage);
        }

        function resolveNameHelper(
            location: Node | undefined,
            name: __String,
            meaning: SymbolFlags,
            nameNotFoundMessage: DiagnosticMessage | undefined,
            nameArg: __String | Identifier | undefined,
            isUse: boolean,
            excludeGlobals: boolean,
            lookup: typeof getSymbol,
            suggestedNameNotFoundMessage?: DiagnosticMessage): Symbol | undefined {
            const originalLocation = location; // needed for did-you-mean error reporting, which gathers candidates starting from the original location
            let result: Symbol | undefined;
            let lastLocation: Node | undefined;
            let lastSelfReferenceLocation: Node | undefined;
            let propertyWithInvalidInitializer: Node | undefined;
            let associatedDeclarationForContainingInitializerOrBindingName: ParameterDeclaration | BindingElement | undefined;
            let withinDeferredContext = false;
            const errorLocation = location;
            let grandparent: Node;
            let isInExternalModule = false;

            loop: while (location) {
                // Locals of a source file are not in scope (because they get merged into the global symbol table)
                if (location.locals && !isGlobalSourceFile(location)) {
                    if (result = lookup(location.locals, name, meaning)) {
                        let useResult = true;
                        if (isFunctionLike(location) && lastLocation && lastLocation !== (<FunctionLikeDeclaration>location).body) {
                            // symbol lookup restrictions for function-like declarations
                            // - Type parameters of a function are in scope in the entire function declaration, including the parameter
                            //   list and return type. However, local types are only in scope in the function body.
                            // - parameters are only in the scope of function body
                            // This restriction does not apply to JSDoc comment types because they are parented
                            // at a higher level than type parameters would normally be
                            if (meaning & result.flags & SymbolFlags.Type && lastLocation.kind !== SyntaxKind.JSDocComment) {
                                useResult = result.flags & SymbolFlags.TypeParameter
                                    // type parameters are visible in parameter list, return type and type parameter list
                                    ? lastLocation === (<FunctionLikeDeclaration>location).type ||
                                    lastLocation.kind === SyntaxKind.Parameter ||
                                    lastLocation.kind === SyntaxKind.TypeParameter
                                    // local types not visible outside the function body
                                    : false;
                            }
                            if (meaning & result.flags & SymbolFlags.Variable) {
                                // expression inside parameter will lookup as normal variable scope when targeting es2015+
                                if (useOuterVariableScopeInParameter(result, location, lastLocation)) {
                                    useResult = false;
                                }
                                else if (result.flags & SymbolFlags.FunctionScopedVariable) {
                                    // parameters are visible only inside function body, parameter list and return type
                                    // technically for parameter list case here we might mix parameters and variables declared in function,
                                    // however it is detected separately when checking initializers of parameters
                                    // to make sure that they reference no variables declared after them.
                                    useResult =
                                        lastLocation.kind === SyntaxKind.Parameter ||
                                        (
                                            lastLocation === (<FunctionLikeDeclaration>location).type &&
                                            !!findAncestor(result.valueDeclaration, isParameter)
                                        );
                                }
                            }
                        }
                        else if (location.kind === SyntaxKind.ConditionalType) {
                            // A type parameter declared using 'infer T' in a conditional type is visible only in
                            // the true branch of the conditional type.
                            useResult = lastLocation === (<ConditionalTypeNode>location).trueType;
                        }

                        if (useResult) {
                            break loop;
                        }
                        else {
                            result = undefined;
                        }
                    }
                }
                withinDeferredContext = withinDeferredContext || getIsDeferredContext(location, lastLocation);
                switch (location.kind) {
                    case SyntaxKind.SourceFile:
                        if (!isExternalOrCommonJsModule(<SourceFile>location)) break;
                        isInExternalModule = true;
                        // falls through
                    case SyntaxKind.ModuleDeclaration:
                        const moduleExports = getSymbolOfNode(location as SourceFile | ModuleDeclaration).exports || emptySymbols;
                        if (location.kind === SyntaxKind.SourceFile || (isModuleDeclaration(location) && location.flags & NodeFlags.Ambient && !isGlobalScopeAugmentation(location))) {

                            // It's an external module. First see if the module has an export default and if the local
                            // name of that export default matches.
                            if (result = moduleExports.get(InternalSymbolName.Default)) {
                                const localSymbol = getLocalSymbolForExportDefault(result);
                                if (localSymbol && (result.flags & meaning) && localSymbol.escapedName === name) {
                                    break loop;
                                }
                                result = undefined;
                            }

                            // Because of module/namespace merging, a module's exports are in scope,
                            // yet we never want to treat an export specifier as putting a member in scope.
                            // Therefore, if the name we find is purely an export specifier, it is not actually considered in scope.
                            // Two things to note about this:
                            //     1. We have to check this without calling getSymbol. The problem with calling getSymbol
                            //        on an export specifier is that it might find the export specifier itself, and try to
                            //        resolve it as an alias. This will cause the checker to consider the export specifier
                            //        a circular alias reference when it might not be.
                            //     2. We check === SymbolFlags.Alias in order to check that the symbol is *purely*
                            //        an alias. If we used &, we'd be throwing out symbols that have non alias aspects,
                            //        which is not the desired behavior.
                            const moduleExport = moduleExports.get(name);
                            if (moduleExport &&
                                moduleExport.flags === SymbolFlags.Alias &&
                                (getDeclarationOfKind(moduleExport, SyntaxKind.ExportSpecifier) || getDeclarationOfKind(moduleExport, SyntaxKind.NamespaceExport))) {
                                break;
                            }
                        }

                        // ES6 exports are also visible locally (except for 'default'), but commonjs exports are not (except typedefs)
                        if (name !== InternalSymbolName.Default && (result = lookup(moduleExports, name, meaning & SymbolFlags.ModuleMember))) {
                            if (isSourceFile(location) && location.commonJsModuleIndicator && !result.declarations.some(isJSDocTypeAlias)) {
                                result = undefined;
                            }
                            else {
                                break loop;
                            }
                        }
                        break;
                    case SyntaxKind.EnumDeclaration:
                        if (result = lookup(getSymbolOfNode(location)!.exports!, name, meaning & SymbolFlags.EnumMember)) {
                            break loop;
                        }
                        break;
                    case SyntaxKind.PropertyDeclaration:
                        // TypeScript 1.0 spec (April 2014): 8.4.1
                        // Initializer expressions for instance member variables are evaluated in the scope
                        // of the class constructor body but are not permitted to reference parameters or
                        // local variables of the constructor. This effectively means that entities from outer scopes
                        // by the same name as a constructor parameter or local variable are inaccessible
                        // in initializer expressions for instance member variables.
                        if (!hasSyntacticModifier(location, ModifierFlags.Static)) {
                            const ctor = findConstructorDeclaration(location.parent as ClassLikeDeclaration);
                            if (ctor && ctor.locals) {
                                if (lookup(ctor.locals, name, meaning & SymbolFlags.Value)) {
                                    // Remember the property node, it will be used later to report appropriate error
                                    propertyWithInvalidInitializer = location;
                                }
                            }
                        }
                        break;
                    case SyntaxKind.ClassDeclaration:
                    case SyntaxKind.ClassExpression:
                    case SyntaxKind.InterfaceDeclaration:
                        // The below is used to lookup type parameters within a class or interface, as they are added to the class/interface locals
                        // These can never be latebound, so the symbol's raw members are sufficient. `getMembersOfNode` cannot be used, as it would
                        // trigger resolving late-bound names, which we may already be in the process of doing while we're here!
                        if (result = lookup(getSymbolOfNode(location as ClassLikeDeclaration | InterfaceDeclaration).members || emptySymbols, name, meaning & SymbolFlags.Type)) {
                            if (!isTypeParameterSymbolDeclaredInContainer(result, location)) {
                                // ignore type parameters not declared in this container
                                result = undefined;
                                break;
                            }
                            if (lastLocation && hasSyntacticModifier(lastLocation, ModifierFlags.Static)) {
                                // TypeScript 1.0 spec (April 2014): 3.4.1
                                // The scope of a type parameter extends over the entire declaration with which the type
                                // parameter list is associated, with the exception of static member declarations in classes.
                                error(errorLocation, Diagnostics.Static_members_cannot_reference_class_type_parameters);
                                return undefined;
                            }
                            break loop;
                        }
                        if (location.kind === SyntaxKind.ClassExpression && meaning & SymbolFlags.Class) {
                            const className = (<ClassExpression>location).name;
                            if (className && name === className.escapedText) {
                                result = location.symbol;
                                break loop;
                            }
                        }
                        break;
                    case SyntaxKind.ExpressionWithTypeArguments:
                        // The type parameters of a class are not in scope in the base class expression.
                        if (lastLocation === (<ExpressionWithTypeArguments>location).expression && (<HeritageClause>location.parent).token === SyntaxKind.ExtendsKeyword) {
                            const container = location.parent.parent;
                            if (isClassLike(container) && (result = lookup(getSymbolOfNode(container).members!, name, meaning & SymbolFlags.Type))) {
                                if (nameNotFoundMessage) {
                                    error(errorLocation, Diagnostics.Base_class_expressions_cannot_reference_class_type_parameters);
                                }
                                return undefined;
                            }
                        }
                        break;
                    // It is not legal to reference a class's own type parameters from a computed property name that
                    // belongs to the class. For example:
                    //
                    //   function foo<T>() { return '' }
                    //   class C<T> { // <-- Class's own type parameter T
                    //       [foo<T>()]() { } // <-- Reference to T from class's own computed property
                    //   }
                    //
                    case SyntaxKind.ComputedPropertyName:
                        grandparent = location.parent.parent;
                        if (isClassLike(grandparent) || grandparent.kind === SyntaxKind.InterfaceDeclaration) {
                            // A reference to this grandparent's type parameters would be an error
                            if (result = lookup(getSymbolOfNode(grandparent as ClassLikeDeclaration | InterfaceDeclaration).members!, name, meaning & SymbolFlags.Type)) {
                                error(errorLocation, Diagnostics.A_computed_property_name_cannot_reference_a_type_parameter_from_its_containing_type);
                                return undefined;
                            }
                        }
                        break;
                    case SyntaxKind.ArrowFunction:
                        // when targeting ES6 or higher there is no 'arguments' in an arrow function
                        // for lower compile targets the resolved symbol is used to emit an error
                        if (compilerOptions.target! >= ScriptTarget.ES2015) {
                            break;
                        }
                        // falls through
                    case SyntaxKind.MethodDeclaration:
                    case SyntaxKind.Constructor:
                    case SyntaxKind.GetAccessor:
                    case SyntaxKind.SetAccessor:
                    case SyntaxKind.FunctionDeclaration:
                        if (meaning & SymbolFlags.Variable && name === "arguments") {
                            result = argumentsSymbol;
                            break loop;
                        }
                        break;
                    case SyntaxKind.FunctionExpression:
                        if (meaning & SymbolFlags.Variable && name === "arguments") {
                            result = argumentsSymbol;
                            break loop;
                        }

                        if (meaning & SymbolFlags.Function) {
                            const functionName = (<FunctionExpression>location).name;
                            if (functionName && name === functionName.escapedText) {
                                result = location.symbol;
                                break loop;
                            }
                        }
                        break;
                    case SyntaxKind.Decorator:
                        // Decorators are resolved at the class declaration. Resolving at the parameter
                        // or member would result in looking up locals in the method.
                        //
                        //   function y() {}
                        //   class C {
                        //       method(@y x, y) {} // <-- decorator y should be resolved at the class declaration, not the parameter.
                        //   }
                        //
                        if (location.parent && location.parent.kind === SyntaxKind.Parameter) {
                            location = location.parent;
                        }
                        //
                        //   function y() {}
                        //   class C {
                        //       @y method(x, y) {} // <-- decorator y should be resolved at the class declaration, not the method.
                        //   }
                        //

                        // class Decorators are resolved outside of the class to avoid referencing type parameters of that class.
                        //
                        //   type T = number;
                        //   declare function y(x: T): any;
                        //   @param(1 as T) // <-- T should resolve to the type alias outside of class C
                        //   class C<T> {}
                        if (location.parent && (isClassElement(location.parent) || location.parent.kind === SyntaxKind.ClassDeclaration)) {
                            location = location.parent;
                        }
                        break;
                    case SyntaxKind.JSDocTypedefTag:
                    case SyntaxKind.JSDocCallbackTag:
                    case SyntaxKind.JSDocEnumTag:
                        // js type aliases do not resolve names from their host, so skip past it
                        location = getJSDocHost(location);
                        break;
                    case SyntaxKind.Parameter:
                        if (lastLocation && (
                            lastLocation === (location as ParameterDeclaration).initializer ||
                            lastLocation === (location as ParameterDeclaration).name && isBindingPattern(lastLocation))) {
                            if (!associatedDeclarationForContainingInitializerOrBindingName) {
                                associatedDeclarationForContainingInitializerOrBindingName = location as ParameterDeclaration;
                            }
                        }
                        break;
                    case SyntaxKind.BindingElement:
                        if (lastLocation && (
                            lastLocation === (location as BindingElement).initializer ||
                            lastLocation === (location as BindingElement).name && isBindingPattern(lastLocation))) {
                            const root = getRootDeclaration(location);
                            if (root.kind === SyntaxKind.Parameter) {
                                if (!associatedDeclarationForContainingInitializerOrBindingName) {
                                    associatedDeclarationForContainingInitializerOrBindingName = location as BindingElement;
                                }
                            }
                        }
                        break;
                }
                if (isSelfReferenceLocation(location)) {
                    lastSelfReferenceLocation = location;
                }
                lastLocation = location;
                location = location.parent;
            }

            // We just climbed up parents looking for the name, meaning that we started in a descendant node of `lastLocation`.
            // If `result === lastSelfReferenceLocation.symbol`, that means that we are somewhere inside `lastSelfReferenceLocation` looking up a name, and resolving to `lastLocation` itself.
            // That means that this is a self-reference of `lastLocation`, and shouldn't count this when considering whether `lastLocation` is used.
            if (isUse && result && (!lastSelfReferenceLocation || result !== lastSelfReferenceLocation.symbol)) {
                result.isReferenced! |= meaning;
            }

            if (!result) {
                if (lastLocation) {
                    Debug.assert(lastLocation.kind === SyntaxKind.SourceFile);
                    if ((lastLocation as SourceFile).commonJsModuleIndicator && name === "exports" && meaning & lastLocation.symbol.flags) {
                        return lastLocation.symbol;
                    }
                }

                if (!excludeGlobals) {
                    result = lookup(globals, name, meaning);
                }
            }
            if (!result) {
                if (originalLocation && isInJSFile(originalLocation) && originalLocation.parent) {
                    if (isRequireCall(originalLocation.parent, /*checkArgumentIsStringLiteralLike*/ false)) {
                        return requireSymbol;
                    }
                }
            }
            if (!result) {
                if (nameNotFoundMessage) {
                    if (!errorLocation ||
                        !checkAndReportErrorForMissingPrefix(errorLocation, name, nameArg!) && // TODO: GH#18217
                        !checkAndReportErrorForExtendingInterface(errorLocation) &&
                        !checkAndReportErrorForUsingTypeAsNamespace(errorLocation, name, meaning) &&
                        !checkAndReportErrorForExportingPrimitiveType(errorLocation, name) &&
                        !checkAndReportErrorForUsingTypeAsValue(errorLocation, name, meaning) &&
                        !checkAndReportErrorForUsingNamespaceModuleAsValue(errorLocation, name, meaning) &&
                        !checkAndReportErrorForUsingValueAsType(errorLocation, name, meaning)) {
                        let suggestion: Symbol | undefined;
                        if (suggestedNameNotFoundMessage && suggestionCount < maximumSuggestionCount) {
                            suggestion = getSuggestedSymbolForNonexistentSymbol(originalLocation, name, meaning);
                            if (suggestion) {
                                const suggestionName = symbolToString(suggestion);
                                const diagnostic = error(errorLocation, suggestedNameNotFoundMessage, diagnosticName(nameArg!), suggestionName);
                                if (suggestion.valueDeclaration) {
                                    addRelatedInfo(
                                        diagnostic,
                                        createDiagnosticForNode(suggestion.valueDeclaration, Diagnostics._0_is_declared_here, suggestionName)
                                    );
                                }
                            }
                        }
                        if (!suggestion) {
                            error(errorLocation, nameNotFoundMessage, diagnosticName(nameArg!));
                        }
                        suggestionCount++;
                    }
                }
                return undefined;
            }

            // Perform extra checks only if error reporting was requested
            if (nameNotFoundMessage) {
                if (propertyWithInvalidInitializer && !(compilerOptions.target === ScriptTarget.ESNext && compilerOptions.useDefineForClassFields)) {
                    // We have a match, but the reference occurred within a property initializer and the identifier also binds
                    // to a local variable in the constructor where the code will be emitted. Note that this is actually allowed
                    // with ESNext+useDefineForClassFields because the scope semantics are different.
                    const propertyName = (<PropertyDeclaration>propertyWithInvalidInitializer).name;
                    error(errorLocation, Diagnostics.Initializer_of_instance_member_variable_0_cannot_reference_identifier_1_declared_in_the_constructor,
                        declarationNameToString(propertyName), diagnosticName(nameArg!));
                    return undefined;
                }

                // Only check for block-scoped variable if we have an error location and are looking for the
                // name with variable meaning
                //      For example,
                //          declare module foo {
                //              interface bar {}
                //          }
                //      const foo/*1*/: foo/*2*/.bar;
                // The foo at /*1*/ and /*2*/ will share same symbol with two meanings:
                // block-scoped variable and namespace module. However, only when we
                // try to resolve name in /*1*/ which is used in variable position,
                // we want to check for block-scoped
                if (errorLocation &&
                    (meaning & SymbolFlags.BlockScopedVariable ||
                     ((meaning & SymbolFlags.Class || meaning & SymbolFlags.Enum) && (meaning & SymbolFlags.Value) === SymbolFlags.Value))) {
                    const exportOrLocalSymbol = getExportSymbolOfValueSymbolIfExported(result);
                    if (exportOrLocalSymbol.flags & SymbolFlags.BlockScopedVariable || exportOrLocalSymbol.flags & SymbolFlags.Class || exportOrLocalSymbol.flags & SymbolFlags.Enum) {
                        checkResolvedBlockScopedVariable(exportOrLocalSymbol, errorLocation);
                    }
                }

                // If we're in an external module, we can't reference value symbols created from UMD export declarations
                if (result && isInExternalModule && (meaning & SymbolFlags.Value) === SymbolFlags.Value && !(originalLocation!.flags & NodeFlags.JSDoc)) {
                    const merged = getMergedSymbol(result);
                    if (length(merged.declarations) && every(merged.declarations, d => isNamespaceExportDeclaration(d) || isSourceFile(d) && !!d.symbol.globalExports)) {
                        errorOrSuggestion(!compilerOptions.allowUmdGlobalAccess, errorLocation!, Diagnostics._0_refers_to_a_UMD_global_but_the_current_file_is_a_module_Consider_adding_an_import_instead, unescapeLeadingUnderscores(name));
                    }
                }

                // If we're in a parameter initializer or binding name, we can't reference the values of the parameter whose initializer we're within or parameters to the right
                if (result && associatedDeclarationForContainingInitializerOrBindingName && !withinDeferredContext && (meaning & SymbolFlags.Value) === SymbolFlags.Value) {
                    const candidate = getMergedSymbol(getLateBoundSymbol(result));
                    const root = (getRootDeclaration(associatedDeclarationForContainingInitializerOrBindingName) as ParameterDeclaration);
                    // A parameter initializer or binding pattern initializer within a parameter cannot refer to itself
                    if (candidate === getSymbolOfNode(associatedDeclarationForContainingInitializerOrBindingName)) {
                        error(errorLocation, Diagnostics.Parameter_0_cannot_reference_itself, declarationNameToString(associatedDeclarationForContainingInitializerOrBindingName.name));
                    }
                    // And it cannot refer to any declarations which come after it
                    else if (candidate.valueDeclaration && candidate.valueDeclaration.pos > associatedDeclarationForContainingInitializerOrBindingName.pos && root.parent.locals && lookup(root.parent.locals, candidate.escapedName, meaning) === candidate) {
                        error(errorLocation, Diagnostics.Parameter_0_cannot_reference_identifier_1_declared_after_it, declarationNameToString(associatedDeclarationForContainingInitializerOrBindingName.name), declarationNameToString(<Identifier>errorLocation));
                    }
                }
                if (result && errorLocation && meaning & SymbolFlags.Value && result.flags & SymbolFlags.Alias) {
                    checkSymbolUsageInExpressionContext(result, name, errorLocation);
                }
            }
            return result;
        }

        function checkSymbolUsageInExpressionContext(symbol: Symbol, name: __String, useSite: Node) {
            if (!isValidTypeOnlyAliasUseSite(useSite)) {
                const typeOnlyDeclaration = getTypeOnlyAliasDeclaration(symbol);
                if (typeOnlyDeclaration) {
                    const isExport = typeOnlyDeclarationIsExport(typeOnlyDeclaration);
                    const message = isExport
                        ? Diagnostics._0_cannot_be_used_as_a_value_because_it_was_exported_using_export_type
                        : Diagnostics._0_cannot_be_used_as_a_value_because_it_was_imported_using_import_type;
                    const relatedMessage = isExport
                        ? Diagnostics._0_was_exported_here
                        : Diagnostics._0_was_imported_here;
                    const unescapedName = unescapeLeadingUnderscores(name);
                    addRelatedInfo(
                        error(useSite, message, unescapedName),
                        createDiagnosticForNode(typeOnlyDeclaration, relatedMessage, unescapedName));
                }
            }
        }

        function getIsDeferredContext(location: Node, lastLocation: Node | undefined): boolean {
            if (location.kind !== SyntaxKind.ArrowFunction && location.kind !== SyntaxKind.FunctionExpression) {
                // initializers in instance property declaration of class like entities are executed in constructor and thus deferred
                return isTypeQueryNode(location) || ((
                    isFunctionLikeDeclaration(location) ||
                    (location.kind === SyntaxKind.PropertyDeclaration && !hasSyntacticModifier(location, ModifierFlags.Static))
                ) && (!lastLocation || lastLocation !== (location as FunctionLike | PropertyDeclaration).name)); // A name is evaluated within the enclosing scope - so it shouldn't count as deferred
            }
            if (lastLocation && lastLocation === (location as FunctionExpression | ArrowFunction).name) {
                return false;
            }
            // generator functions and async functions are not inlined in control flow when immediately invoked
            if ((location as FunctionExpression | ArrowFunction).asteriskToken || hasSyntacticModifier(location, ModifierFlags.Async)) {
                return true;
            }
            return !getImmediatelyInvokedFunctionExpression(location);
        }

        function isSelfReferenceLocation(node: Node): boolean {
            switch (node.kind) {
                case SyntaxKind.FunctionDeclaration:
                case SyntaxKind.ClassDeclaration:
                case SyntaxKind.InterfaceDeclaration:
                case SyntaxKind.EnumDeclaration:
                case SyntaxKind.TypeAliasDeclaration:
                case SyntaxKind.ModuleDeclaration: // For `namespace N { N; }`
                    return true;
                default:
                    return false;
            }
        }

        function diagnosticName(nameArg: __String | Identifier | PrivateIdentifier) {
            return isString(nameArg) ? unescapeLeadingUnderscores(nameArg as __String) : declarationNameToString(nameArg as Identifier);
        }

        function isTypeParameterSymbolDeclaredInContainer(symbol: Symbol, container: Node) {
            for (const decl of symbol.declarations) {
                if (decl.kind === SyntaxKind.TypeParameter) {
                    const parent = isJSDocTemplateTag(decl.parent) ? getJSDocHost(decl.parent) : decl.parent;
                    if (parent === container) {
                        return !(isJSDocTemplateTag(decl.parent) && find((decl.parent.parent as JSDoc).tags!, isJSDocTypeAlias)); // TODO: GH#18217
                    }
                }
            }

            return false;
        }

        function checkAndReportErrorForMissingPrefix(errorLocation: Node, name: __String, nameArg: __String | Identifier): boolean {
            if (!isIdentifier(errorLocation) || errorLocation.escapedText !== name || isTypeReferenceIdentifier(errorLocation) || isInTypeQuery(errorLocation)) {
                return false;
            }

            const container = getThisContainer(errorLocation, /*includeArrowFunctions*/ false);
            let location = container;
            while (location) {
                if (isClassLike(location.parent)) {
                    const classSymbol = getSymbolOfNode(location.parent);
                    if (!classSymbol) {
                        break;
                    }

                    // Check to see if a static member exists.
                    const constructorType = getTypeOfSymbol(classSymbol);
                    if (getPropertyOfType(constructorType, name)) {
                        error(errorLocation, Diagnostics.Cannot_find_name_0_Did_you_mean_the_static_member_1_0, diagnosticName(nameArg), symbolToString(classSymbol));
                        return true;
                    }

                    // No static member is present.
                    // Check if we're in an instance method and look for a relevant instance member.
                    if (location === container && !hasSyntacticModifier(location, ModifierFlags.Static)) {
                        const instanceType = (<InterfaceType>getDeclaredTypeOfSymbol(classSymbol)).thisType!; // TODO: GH#18217
                        if (getPropertyOfType(instanceType, name)) {
                            error(errorLocation, Diagnostics.Cannot_find_name_0_Did_you_mean_the_instance_member_this_0, diagnosticName(nameArg));
                            return true;
                        }
                    }
                }

                location = location.parent;
            }
            return false;
        }


        function checkAndReportErrorForExtendingInterface(errorLocation: Node): boolean {
            const expression = getEntityNameForExtendingInterface(errorLocation);
            if (expression && resolveEntityName(expression, SymbolFlags.Interface, /*ignoreErrors*/ true)) {
                error(errorLocation, Diagnostics.Cannot_extend_an_interface_0_Did_you_mean_implements, getTextOfNode(expression));
                return true;
            }
            return false;
        }
        /**
         * Climbs up parents to an ExpressionWithTypeArguments, and returns its expression,
         * but returns undefined if that expression is not an EntityNameExpression.
         */
        function getEntityNameForExtendingInterface(node: Node): EntityNameExpression | undefined {
            switch (node.kind) {
                case SyntaxKind.Identifier:
                case SyntaxKind.PropertyAccessExpression:
                    return node.parent ? getEntityNameForExtendingInterface(node.parent) : undefined;
                case SyntaxKind.ExpressionWithTypeArguments:
                    if (isEntityNameExpression((<ExpressionWithTypeArguments>node).expression)) {
                        return <EntityNameExpression>(<ExpressionWithTypeArguments>node).expression;
                    }
                    // falls through
                default:
                    return undefined;
            }
        }

        function checkAndReportErrorForUsingTypeAsNamespace(errorLocation: Node, name: __String, meaning: SymbolFlags): boolean {
            const namespaceMeaning = SymbolFlags.Namespace | (isInJSFile(errorLocation) ? SymbolFlags.Value : 0);
            if (meaning === namespaceMeaning) {
                const symbol = resolveSymbol(resolveName(errorLocation, name, SymbolFlags.Type & ~namespaceMeaning, /*nameNotFoundMessage*/undefined, /*nameArg*/ undefined, /*isUse*/ false));
                const parent = errorLocation.parent;
                if (symbol) {
                    if (isQualifiedName(parent)) {
                        Debug.assert(parent.left === errorLocation, "Should only be resolving left side of qualified name as a namespace");
                        const propName = parent.right.escapedText;
                        const propType = getPropertyOfType(getDeclaredTypeOfSymbol(symbol), propName);
                        if (propType) {
                            error(
                                parent,
                                Diagnostics.Cannot_access_0_1_because_0_is_a_type_but_not_a_namespace_Did_you_mean_to_retrieve_the_type_of_the_property_1_in_0_with_0_1,
                                unescapeLeadingUnderscores(name),
                                unescapeLeadingUnderscores(propName),
                            );
                            return true;
                        }
                    }
                    error(errorLocation, Diagnostics._0_only_refers_to_a_type_but_is_being_used_as_a_namespace_here, unescapeLeadingUnderscores(name));
                    return true;
                }
            }

            return false;
        }

        function checkAndReportErrorForUsingValueAsType(errorLocation: Node, name: __String, meaning: SymbolFlags): boolean {
            if (meaning & (SymbolFlags.Type & ~SymbolFlags.Namespace)) {
                const symbol = resolveSymbol(resolveName(errorLocation, name, ~SymbolFlags.Type & SymbolFlags.Value, /*nameNotFoundMessage*/undefined, /*nameArg*/ undefined, /*isUse*/ false));
                if (symbol && !(symbol.flags & SymbolFlags.Namespace)) {
                    error(errorLocation, Diagnostics._0_refers_to_a_value_but_is_being_used_as_a_type_here_Did_you_mean_typeof_0, unescapeLeadingUnderscores(name));
                    return true;
                }
            }
            return false;
        }

        function isPrimitiveTypeName(name: __String) {
            return name === "any" || name === "string" || name === "number" || name === "boolean" || name === "never" || name === "unknown";
        }

        function checkAndReportErrorForExportingPrimitiveType(errorLocation: Node, name: __String): boolean {
            if (isPrimitiveTypeName(name) && errorLocation.parent.kind === SyntaxKind.ExportSpecifier) {
                error(errorLocation, Diagnostics.Cannot_export_0_Only_local_declarations_can_be_exported_from_a_module, name as string);
                return true;
            }
            return false;
        }

        function checkAndReportErrorForUsingTypeAsValue(errorLocation: Node, name: __String, meaning: SymbolFlags): boolean {
            if (meaning & (SymbolFlags.Value & ~SymbolFlags.NamespaceModule)) {
                if (isPrimitiveTypeName(name)) {
                    error(errorLocation, Diagnostics._0_only_refers_to_a_type_but_is_being_used_as_a_value_here, unescapeLeadingUnderscores(name));
                    return true;
                }
                const symbol = resolveSymbol(resolveName(errorLocation, name, SymbolFlags.Type & ~SymbolFlags.Value, /*nameNotFoundMessage*/undefined, /*nameArg*/ undefined, /*isUse*/ false));
                if (symbol && !(symbol.flags & SymbolFlags.NamespaceModule)) {
                    const message = isES2015OrLaterConstructorName(name)
                        ? Diagnostics._0_only_refers_to_a_type_but_is_being_used_as_a_value_here_Do_you_need_to_change_your_target_library_Try_changing_the_lib_compiler_option_to_es2015_or_later
                        : Diagnostics._0_only_refers_to_a_type_but_is_being_used_as_a_value_here;
                    error(errorLocation, message, unescapeLeadingUnderscores(name));
                    return true;
                }
            }
            return false;
        }

        function isES2015OrLaterConstructorName(n: __String) {
            switch (n) {
                case "Promise":
                case "Symbol":
                case "Map":
                case "WeakMap":
                case "Set":
                case "WeakSet":
                    return true;
            }
            return false;
        }

        function checkAndReportErrorForUsingNamespaceModuleAsValue(errorLocation: Node, name: __String, meaning: SymbolFlags): boolean {
            if (meaning & (SymbolFlags.Value & ~SymbolFlags.NamespaceModule & ~SymbolFlags.Type)) {
                const symbol = resolveSymbol(resolveName(errorLocation, name, SymbolFlags.NamespaceModule & ~SymbolFlags.Value, /*nameNotFoundMessage*/undefined, /*nameArg*/ undefined, /*isUse*/ false));
                if (symbol) {
                    error(
                        errorLocation,
                        Diagnostics.Cannot_use_namespace_0_as_a_value,
                        unescapeLeadingUnderscores(name));
                    return true;
                }
            }
            else if (meaning & (SymbolFlags.Type & ~SymbolFlags.NamespaceModule & ~SymbolFlags.Value)) {
                const symbol = resolveSymbol(resolveName(errorLocation, name, (SymbolFlags.ValueModule | SymbolFlags.NamespaceModule) & ~SymbolFlags.Type, /*nameNotFoundMessage*/undefined, /*nameArg*/ undefined, /*isUse*/ false));
                if (symbol) {
                    error(errorLocation, Diagnostics.Cannot_use_namespace_0_as_a_type, unescapeLeadingUnderscores(name));
                    return true;
                }
            }
            return false;
        }

        function checkResolvedBlockScopedVariable(result: Symbol, errorLocation: Node): void {
            Debug.assert(!!(result.flags & SymbolFlags.BlockScopedVariable || result.flags & SymbolFlags.Class || result.flags & SymbolFlags.Enum));
            if (result.flags & (SymbolFlags.Function | SymbolFlags.FunctionScopedVariable | SymbolFlags.Assignment) && result.flags & SymbolFlags.Class) {
                // constructor functions aren't block scoped
                return;
            }
            // Block-scoped variables cannot be used before their definition
            const declaration = find(
                result.declarations,
                d => isBlockOrCatchScoped(d) || isClassLike(d) || (d.kind === SyntaxKind.EnumDeclaration));

            if (declaration === undefined) return Debug.fail("checkResolvedBlockScopedVariable could not find block-scoped declaration");

            if (!(declaration.flags & NodeFlags.Ambient) && !isBlockScopedNameDeclaredBeforeUse(declaration, errorLocation)) {
                let diagnosticMessage;
                const declarationName = declarationNameToString(getNameOfDeclaration(declaration));
                if (result.flags & SymbolFlags.BlockScopedVariable) {
                    diagnosticMessage = error(errorLocation, Diagnostics.Block_scoped_variable_0_used_before_its_declaration, declarationName);
                }
                else if (result.flags & SymbolFlags.Class) {
                    diagnosticMessage = error(errorLocation, Diagnostics.Class_0_used_before_its_declaration, declarationName);
                }
                else if (result.flags & SymbolFlags.RegularEnum) {
                    diagnosticMessage = error(errorLocation, Diagnostics.Enum_0_used_before_its_declaration, declarationName);
                }
                else {
                    Debug.assert(!!(result.flags & SymbolFlags.ConstEnum));
                    if (compilerOptions.preserveConstEnums) {
                        diagnosticMessage = error(errorLocation, Diagnostics.Enum_0_used_before_its_declaration, declarationName);
                    }
                }

                if (diagnosticMessage) {
                    addRelatedInfo(diagnosticMessage,
                        createDiagnosticForNode(declaration, Diagnostics._0_is_declared_here, declarationName)
                    );
                }
            }
        }

        /* Starting from 'initial' node walk up the parent chain until 'stopAt' node is reached.
         * If at any point current node is equal to 'parent' node - return true.
         * Return false if 'stopAt' node is reached or isFunctionLike(current) === true.
         */
        function isSameScopeDescendentOf(initial: Node, parent: Node | undefined, stopAt: Node): boolean {
            return !!parent && !!findAncestor(initial, n => n === stopAt || isFunctionLike(n) ? "quit" : n === parent);
        }

        function getAnyImportSyntax(node: Node): AnyImportSyntax | undefined {
            switch (node.kind) {
                case SyntaxKind.ImportEqualsDeclaration:
                    return node as ImportEqualsDeclaration;
                case SyntaxKind.ImportClause:
                    return (node as ImportClause).parent;
                case SyntaxKind.NamespaceImport:
                    return (node as NamespaceImport).parent.parent;
                case SyntaxKind.ImportSpecifier:
                    return (node as ImportSpecifier).parent.parent.parent;
                default:
                    return undefined;
            }
        }

        function getDeclarationOfAliasSymbol(symbol: Symbol): Declaration | undefined {
            return find<Declaration>(symbol.declarations, isAliasSymbolDeclaration);
        }

        /**
         * An alias symbol is created by one of the following declarations:
         * import <symbol> = ...
         * import <symbol> from ...
         * import * as <symbol> from ...
         * import { x as <symbol> } from ...
         * export { x as <symbol> } from ...
         * export * as ns <symbol> from ...
         * export = <EntityNameExpression>
         * export default <EntityNameExpression>
         * module.exports = <EntityNameExpression>
         * {<Identifier>}
         * {name: <EntityNameExpression>}
         */
        function isAliasSymbolDeclaration(node: Node): boolean {
            return node.kind === SyntaxKind.ImportEqualsDeclaration ||
                node.kind === SyntaxKind.NamespaceExportDeclaration ||
                node.kind === SyntaxKind.ImportClause && !!(<ImportClause>node).name ||
                node.kind === SyntaxKind.NamespaceImport ||
                node.kind === SyntaxKind.NamespaceExport ||
                node.kind === SyntaxKind.ImportSpecifier ||
                node.kind === SyntaxKind.ExportSpecifier ||
                node.kind === SyntaxKind.ExportAssignment && exportAssignmentIsAlias(<ExportAssignment>node) ||
                isBinaryExpression(node) && getAssignmentDeclarationKind(node) === AssignmentDeclarationKind.ModuleExports && exportAssignmentIsAlias(node) ||
                isPropertyAccessExpression(node)
                    && isBinaryExpression(node.parent)
                    && node.parent.left === node
                    && node.parent.operatorToken.kind === SyntaxKind.EqualsToken
                    && isAliasableOrJsExpression(node.parent.right) ||
                node.kind === SyntaxKind.ShorthandPropertyAssignment ||
                node.kind === SyntaxKind.PropertyAssignment && isAliasableOrJsExpression((node as PropertyAssignment).initializer);
        }

        function isAliasableOrJsExpression(e: Expression) {
            return isAliasableExpression(e) || isFunctionExpression(e) && isJSConstructor(e);
        }

        function getTargetOfImportEqualsDeclaration(node: ImportEqualsDeclaration, dontResolveAlias: boolean): Symbol | undefined {
            if (node.moduleReference.kind === SyntaxKind.ExternalModuleReference) {
                const immediate = resolveExternalModuleName(node, getExternalModuleImportEqualsDeclarationExpression(node));
                const resolved = resolveExternalModuleSymbol(immediate);
                markSymbolOfAliasDeclarationIfTypeOnly(node, immediate, resolved, /*overwriteEmpty*/ false);
                return resolved;
            }
            const resolved = getSymbolOfPartOfRightHandSideOfImportEquals(node.moduleReference, dontResolveAlias);
            checkAndReportErrorForResolvingImportAliasToTypeOnlySymbol(node, resolved);
            return resolved;
        }

        function checkAndReportErrorForResolvingImportAliasToTypeOnlySymbol(node: ImportEqualsDeclaration, resolved: Symbol | undefined) {
            if (markSymbolOfAliasDeclarationIfTypeOnly(node, /*immediateTarget*/ undefined, resolved, /*overwriteEmpty*/ false)) {
                const typeOnlyDeclaration = getTypeOnlyAliasDeclaration(getSymbolOfNode(node))!;
                const isExport = typeOnlyDeclarationIsExport(typeOnlyDeclaration);
                const message = isExport
                    ? Diagnostics.An_import_alias_cannot_reference_a_declaration_that_was_exported_using_export_type
                    : Diagnostics.An_import_alias_cannot_reference_a_declaration_that_was_imported_using_import_type;
                const relatedMessage = isExport
                    ? Diagnostics._0_was_exported_here
                    : Diagnostics._0_was_imported_here;

                // Non-null assertion is safe because the optionality comes from ImportClause,
                // but if an ImportClause was the typeOnlyDeclaration, it had to have a `name`.
                const name = unescapeLeadingUnderscores(typeOnlyDeclaration.name!.escapedText);
                addRelatedInfo(error(node.moduleReference, message), createDiagnosticForNode(typeOnlyDeclaration, relatedMessage, name));
            }
        }

        function resolveExportByName(moduleSymbol: Symbol, name: __String, sourceNode: TypeOnlyCompatibleAliasDeclaration | undefined, dontResolveAlias: boolean) {
            const exportValue = moduleSymbol.exports!.get(InternalSymbolName.ExportEquals);
            if (exportValue) {
                return getPropertyOfType(getTypeOfSymbol(exportValue), name);
            }
            const exportSymbol = moduleSymbol.exports!.get(name);
            const resolved = resolveSymbol(exportSymbol, dontResolveAlias);
            markSymbolOfAliasDeclarationIfTypeOnly(sourceNode, exportSymbol, resolved, /*overwriteEmpty*/ false);
            return resolved;
        }

        function isSyntacticDefault(node: Node) {
            return ((isExportAssignment(node) && !node.isExportEquals) || hasSyntacticModifier(node, ModifierFlags.Default) || isExportSpecifier(node));
        }

        function canHaveSyntheticDefault(file: SourceFile | undefined, moduleSymbol: Symbol, dontResolveAlias: boolean) {
            if (!allowSyntheticDefaultImports) {
                return false;
            }
            // Declaration files (and ambient modules)
            if (!file || file.isDeclarationFile) {
                // Definitely cannot have a synthetic default if they have a syntactic default member specified
                const defaultExportSymbol = resolveExportByName(moduleSymbol, InternalSymbolName.Default, /*sourceNode*/ undefined, /*dontResolveAlias*/ true); // Dont resolve alias because we want the immediately exported symbol's declaration
                if (defaultExportSymbol && some(defaultExportSymbol.declarations, isSyntacticDefault)) {
                    return false;
                }
                // It _might_ still be incorrect to assume there is no __esModule marker on the import at runtime, even if there is no `default` member
                // So we check a bit more,
                if (resolveExportByName(moduleSymbol, escapeLeadingUnderscores("__esModule"), /*sourceNode*/ undefined, dontResolveAlias)) {
                    // If there is an `__esModule` specified in the declaration (meaning someone explicitly added it or wrote it in their code),
                    // it definitely is a module and does not have a synthetic default
                    return false;
                }
                // There are _many_ declaration files not written with esmodules in mind that still get compiled into a format with __esModule set
                // Meaning there may be no default at runtime - however to be on the permissive side, we allow access to a synthetic default member
                // as there is no marker to indicate if the accompanying JS has `__esModule` or not, or is even native esm
                return true;
            }
            // TypeScript files never have a synthetic default (as they are always emitted with an __esModule marker) _unless_ they contain an export= statement
            if (!isSourceFileJS(file)) {
                return hasExportAssignmentSymbol(moduleSymbol);
            }
            // JS files have a synthetic default if they do not contain ES2015+ module syntax (export = is not valid in js) _and_ do not have an __esModule marker
            return !file.externalModuleIndicator && !resolveExportByName(moduleSymbol, escapeLeadingUnderscores("__esModule"), /*sourceNode*/ undefined, dontResolveAlias);
        }

        function getTargetOfImportClause(node: ImportClause, dontResolveAlias: boolean): Symbol | undefined {
            const moduleSymbol = resolveExternalModuleName(node, node.parent.moduleSpecifier);
            if (moduleSymbol) {
                let exportDefaultSymbol: Symbol | undefined;
                if (isShorthandAmbientModuleSymbol(moduleSymbol)) {
                    exportDefaultSymbol = moduleSymbol;
                }
                else {
                    exportDefaultSymbol = resolveExportByName(moduleSymbol, InternalSymbolName.Default, node, dontResolveAlias);
                }

                const file = find(moduleSymbol.declarations, isSourceFile);
                const hasSyntheticDefault = canHaveSyntheticDefault(file, moduleSymbol, dontResolveAlias);
                if (!exportDefaultSymbol && !hasSyntheticDefault) {
                    if (hasExportAssignmentSymbol(moduleSymbol)) {
                        const compilerOptionName = moduleKind >= ModuleKind.ES2015 ? "allowSyntheticDefaultImports" : "esModuleInterop";
                        const exportEqualsSymbol = moduleSymbol.exports!.get(InternalSymbolName.ExportEquals);
                        const exportAssignment = exportEqualsSymbol!.valueDeclaration;
                        const err = error(node.name, Diagnostics.Module_0_can_only_be_default_imported_using_the_1_flag, symbolToString(moduleSymbol), compilerOptionName);

                        addRelatedInfo(err, createDiagnosticForNode(
                            exportAssignment,
                            Diagnostics.This_module_is_declared_with_using_export_and_can_only_be_used_with_a_default_import_when_using_the_0_flag,
                            compilerOptionName
                        ));
                    }
                    else {
                        reportNonDefaultExport(moduleSymbol, node);
                    }
                }
                else if (hasSyntheticDefault) {
                    // per emit behavior, a synthetic default overrides a "real" .default member if `__esModule` is not present
                    const resolved = resolveExternalModuleSymbol(moduleSymbol, dontResolveAlias) || resolveSymbol(moduleSymbol, dontResolveAlias);
                    markSymbolOfAliasDeclarationIfTypeOnly(node, moduleSymbol, resolved, /*overwriteTypeOnly*/ false);
                    return resolved;
                }
                markSymbolOfAliasDeclarationIfTypeOnly(node, exportDefaultSymbol, /*finalTarget*/ undefined, /*overwriteTypeOnly*/ false);
                return exportDefaultSymbol;
            }
        }

        function reportNonDefaultExport(moduleSymbol: Symbol, node: ImportClause) {
            if (moduleSymbol.exports?.has(node.symbol.escapedName)) {
                error(
                    node.name,
                    Diagnostics.Module_0_has_no_default_export_Did_you_mean_to_use_import_1_from_0_instead,
                    symbolToString(moduleSymbol),
                    symbolToString(node.symbol),
                );
            }
            else {
                const diagnostic = error(node.name, Diagnostics.Module_0_has_no_default_export, symbolToString(moduleSymbol));
                const exportStar = moduleSymbol.exports?.get(InternalSymbolName.ExportStar);
                if (exportStar) {
                    const defaultExport = find(exportStar.declarations, decl => !!(
                        isExportDeclaration(decl) && decl.moduleSpecifier &&
                            resolveExternalModuleName(decl, decl.moduleSpecifier)?.exports?.has(InternalSymbolName.Default)
                    ));
                    if (defaultExport) {
                        addRelatedInfo(diagnostic, createDiagnosticForNode(defaultExport, Diagnostics.export_Asterisk_does_not_re_export_a_default));
                    }
                }
            }
        }

        function getTargetOfNamespaceImport(node: NamespaceImport, dontResolveAlias: boolean): Symbol | undefined {
            const moduleSpecifier = node.parent.parent.moduleSpecifier;
            const immediate = resolveExternalModuleName(node, moduleSpecifier);
            const resolved = resolveESModuleSymbol(immediate, moduleSpecifier, dontResolveAlias, /*suppressUsageError*/ false);
            markSymbolOfAliasDeclarationIfTypeOnly(node, immediate, resolved, /*overwriteEmpty*/ false);
            return resolved;
        }

        function getTargetOfNamespaceExport(node: NamespaceExport, dontResolveAlias: boolean): Symbol | undefined {
            const moduleSpecifier = node.parent.moduleSpecifier;
            const immediate = moduleSpecifier && resolveExternalModuleName(node, moduleSpecifier);
            const resolved = moduleSpecifier && resolveESModuleSymbol(immediate, moduleSpecifier, dontResolveAlias, /*suppressUsageError*/ false);
            markSymbolOfAliasDeclarationIfTypeOnly(node, immediate, resolved, /*overwriteEmpty*/ false);
            return resolved;
        }

        // This function creates a synthetic symbol that combines the value side of one symbol with the
        // type/namespace side of another symbol. Consider this example:
        //
        //   declare module graphics {
        //       interface Point {
        //           x: number;
        //           y: number;
        //       }
        //   }
        //   declare var graphics: {
        //       Point: new (x: number, y: number) => graphics.Point;
        //   }
        //   declare module "graphics" {
        //       export = graphics;
        //   }
        //
        // An 'import { Point } from "graphics"' needs to create a symbol that combines the value side 'Point'
        // property with the type/namespace side interface 'Point'.
        function combineValueAndTypeSymbols(valueSymbol: Symbol, typeSymbol: Symbol): Symbol {
            if (valueSymbol === unknownSymbol && typeSymbol === unknownSymbol) {
                return unknownSymbol;
            }
            if (valueSymbol.flags & (SymbolFlags.Type | SymbolFlags.Namespace)) {
                return valueSymbol;
            }
            const result = createSymbol(valueSymbol.flags | typeSymbol.flags, valueSymbol.escapedName);
            result.declarations = deduplicate(concatenate(valueSymbol.declarations, typeSymbol.declarations), equateValues);
            result.parent = valueSymbol.parent || typeSymbol.parent;
            if (valueSymbol.valueDeclaration) result.valueDeclaration = valueSymbol.valueDeclaration;
            if (typeSymbol.members) result.members = new Map(typeSymbol.members);
            if (valueSymbol.exports) result.exports = new Map(valueSymbol.exports);
            return result;
        }

        function getExportOfModule(symbol: Symbol, specifier: ImportOrExportSpecifier, dontResolveAlias: boolean): Symbol | undefined {
            if (symbol.flags & SymbolFlags.Module) {
                const name = (specifier.propertyName ?? specifier.name).escapedText;
                const exportSymbol = getExportsOfSymbol(symbol).get(name);
                const resolved = resolveSymbol(exportSymbol, dontResolveAlias);
                markSymbolOfAliasDeclarationIfTypeOnly(specifier, exportSymbol, resolved, /*overwriteEmpty*/ false);
                return resolved;
            }
        }

        function getPropertyOfVariable(symbol: Symbol, name: __String): Symbol | undefined {
            if (symbol.flags & SymbolFlags.Variable) {
                const typeAnnotation = (<VariableDeclaration>symbol.valueDeclaration).type;
                if (typeAnnotation) {
                    return resolveSymbol(getPropertyOfType(getTypeFromTypeNode(typeAnnotation), name));
                }
            }
        }

        function getExternalModuleMember(node: ImportDeclaration | ExportDeclaration, specifier: ImportOrExportSpecifier, dontResolveAlias = false): Symbol | undefined {
            const moduleSymbol = resolveExternalModuleName(node, node.moduleSpecifier!)!;
            const name = specifier.propertyName || specifier.name;
            const suppressInteropError = name.escapedText === InternalSymbolName.Default && !!(compilerOptions.allowSyntheticDefaultImports || compilerOptions.esModuleInterop);
            const targetSymbol = resolveESModuleSymbol(moduleSymbol, node.moduleSpecifier!, dontResolveAlias, suppressInteropError);
            if (targetSymbol) {
                if (name.escapedText) {
                    if (isShorthandAmbientModuleSymbol(moduleSymbol)) {
                        return moduleSymbol;
                    }

                    let symbolFromVariable: Symbol | undefined;
                    // First check if module was specified with "export=". If so, get the member from the resolved type
                    if (moduleSymbol && moduleSymbol.exports && moduleSymbol.exports.get(InternalSymbolName.ExportEquals)) {
                        symbolFromVariable = getPropertyOfType(getTypeOfSymbol(targetSymbol), name.escapedText);
                    }
                    else {
                        symbolFromVariable = getPropertyOfVariable(targetSymbol, name.escapedText);
                    }

                    // if symbolFromVariable is export - get its final target
                    symbolFromVariable = resolveSymbol(symbolFromVariable, dontResolveAlias);
                    let symbolFromModule = getExportOfModule(targetSymbol, specifier, dontResolveAlias);
                    if (symbolFromModule === undefined && name.escapedText === InternalSymbolName.Default) {
                        const file = find(moduleSymbol.declarations, isSourceFile);
                        if (canHaveSyntheticDefault(file, moduleSymbol, dontResolveAlias)) {
                            symbolFromModule = resolveExternalModuleSymbol(moduleSymbol, dontResolveAlias) || resolveSymbol(moduleSymbol, dontResolveAlias);
                        }
                    }

                    const symbol = symbolFromModule && symbolFromVariable && symbolFromModule !== symbolFromVariable ?
                        combineValueAndTypeSymbols(symbolFromVariable, symbolFromModule) :
                        symbolFromModule || symbolFromVariable;
                    if (!symbol) {
                        const moduleName = getFullyQualifiedName(moduleSymbol, node);
                        const declarationName = declarationNameToString(name);
                        const suggestion = getSuggestedSymbolForNonexistentModule(name, targetSymbol);
                        if (suggestion !== undefined) {
                            const suggestionName = symbolToString(suggestion);
                            const diagnostic = error(name, Diagnostics.Module_0_has_no_exported_member_1_Did_you_mean_2, moduleName, declarationName, suggestionName);
                            if (suggestion.valueDeclaration) {
                                addRelatedInfo(diagnostic,
                                    createDiagnosticForNode(suggestion.valueDeclaration, Diagnostics._0_is_declared_here, suggestionName)
                                );
                            }
                        }
                        else {
                            if (moduleSymbol.exports?.has(InternalSymbolName.Default)) {
                                error(
                                    name,
                                    Diagnostics.Module_0_has_no_exported_member_1_Did_you_mean_to_use_import_1_from_0_instead,
                                    moduleName,
                                    declarationName
                                );
                            }
                            else {
                                reportNonExportedMember(node, name, declarationName, moduleSymbol, moduleName);
                            }
                        }
                    }
                    return symbol;
                }
            }
        }

        function reportNonExportedMember(node: ImportDeclaration | ExportDeclaration, name: Identifier, declarationName: string, moduleSymbol: Symbol, moduleName: string): void {
            const localSymbol = moduleSymbol.valueDeclaration.locals?.get(name.escapedText);
            const exports = moduleSymbol.exports;
            if (localSymbol) {
                const exportedEqualsSymbol = exports?.get(InternalSymbolName.ExportEquals);
                if (exportedEqualsSymbol) {
                    getSymbolIfSameReference(exportedEqualsSymbol, localSymbol) ? reportInvalidImportEqualsExportMember(node, name, declarationName, moduleName) :
                        error(name, Diagnostics.Module_0_has_no_exported_member_1, moduleName, declarationName);
                }
                else {
                    const exportedSymbol = exports ? find(symbolsToArray(exports), symbol => !!getSymbolIfSameReference(symbol, localSymbol)) : undefined;
                    const diagnostic = exportedSymbol ? error(name, Diagnostics.Module_0_declares_1_locally_but_it_is_exported_as_2, moduleName, declarationName, symbolToString(exportedSymbol)) :
                        error(name, Diagnostics.Module_0_declares_1_locally_but_it_is_not_exported, moduleName, declarationName);

                    addRelatedInfo(diagnostic,
                        ...map(localSymbol.declarations, (decl, index) =>
                            createDiagnosticForNode(decl, index === 0 ? Diagnostics._0_is_declared_here : Diagnostics.and_here, declarationName)));
                }
            }
            else {
                error(name, Diagnostics.Module_0_has_no_exported_member_1, moduleName, declarationName);
            }
        }

        function reportInvalidImportEqualsExportMember(node: ImportDeclaration | ExportDeclaration, name: Identifier, declarationName: string, moduleName: string) {
            if (moduleKind >= ModuleKind.ES2015) {
                const message = compilerOptions.esModuleInterop ? Diagnostics._0_can_only_be_imported_by_using_a_default_import :
                    Diagnostics._0_can_only_be_imported_by_turning_on_the_esModuleInterop_flag_and_using_a_default_import;
                error(name, message, declarationName);
            }
            else {
                if (isInJSFile(node)) {
                    const message = compilerOptions.esModuleInterop ? Diagnostics._0_can_only_be_imported_by_using_a_require_call_or_by_using_a_default_import :
                        Diagnostics._0_can_only_be_imported_by_using_a_require_call_or_by_turning_on_the_esModuleInterop_flag_and_using_a_default_import;
                    error(name, message, declarationName);
                }
                else {
                    const message = compilerOptions.esModuleInterop ? Diagnostics._0_can_only_be_imported_by_using_import_1_require_2_or_a_default_import :
                        Diagnostics._0_can_only_be_imported_by_using_import_1_require_2_or_by_turning_on_the_esModuleInterop_flag_and_using_a_default_import;
                    error(name, message, declarationName, declarationName, moduleName);
                }
            }
        }

        function getTargetOfImportSpecifier(node: ImportSpecifier, dontResolveAlias: boolean): Symbol | undefined {
            const resolved = getExternalModuleMember(node.parent.parent.parent, node, dontResolveAlias);
            markSymbolOfAliasDeclarationIfTypeOnly(node, /*immediateTarget*/ undefined, resolved, /*overwriteEmpty*/ false);
            return resolved;
        }

        function getTargetOfNamespaceExportDeclaration(node: NamespaceExportDeclaration, dontResolveAlias: boolean): Symbol {
            const resolved = resolveExternalModuleSymbol(node.parent.symbol, dontResolveAlias);
            markSymbolOfAliasDeclarationIfTypeOnly(node, /*immediateTarget*/ undefined, resolved, /*overwriteEmpty*/ false);
            return resolved;
        }

        function getTargetOfExportSpecifier(node: ExportSpecifier, meaning: SymbolFlags, dontResolveAlias?: boolean) {
            const resolved = node.parent.parent.moduleSpecifier ?
                getExternalModuleMember(node.parent.parent, node, dontResolveAlias) :
                resolveEntityName(node.propertyName || node.name, meaning, /*ignoreErrors*/ false, dontResolveAlias);
            markSymbolOfAliasDeclarationIfTypeOnly(node, /*immediateTarget*/ undefined, resolved, /*overwriteEmpty*/ false);
            return resolved;
        }

        function getTargetOfExportAssignment(node: ExportAssignment | BinaryExpression, dontResolveAlias: boolean): Symbol | undefined {
            const expression = isExportAssignment(node) ? node.expression : node.right;
            const resolved = getTargetOfAliasLikeExpression(expression, dontResolveAlias);
            markSymbolOfAliasDeclarationIfTypeOnly(node, /*immediateTarget*/ undefined, resolved, /*overwriteEmpty*/ false);
            return resolved;
        }

        function getTargetOfAliasLikeExpression(expression: Expression, dontResolveAlias: boolean) {
            if (isClassExpression(expression)) {
                return checkExpressionCached(expression).symbol;
            }
            if (!isEntityName(expression) && !isEntityNameExpression(expression)) {
                return undefined;
            }
            const aliasLike = resolveEntityName(expression, SymbolFlags.Value | SymbolFlags.Type | SymbolFlags.Namespace, /*ignoreErrors*/ true, dontResolveAlias);
            if (aliasLike) {
                return aliasLike;
            }
            checkExpressionCached(expression);
            return getNodeLinks(expression).resolvedSymbol;
        }

        function getTargetOfPropertyAssignment(node: PropertyAssignment, dontRecursivelyResolve: boolean): Symbol | undefined {
            const expression = node.initializer;
            return getTargetOfAliasLikeExpression(expression, dontRecursivelyResolve);
        }

        function getTargetOfPropertyAccessExpression(node: PropertyAccessExpression, dontRecursivelyResolve: boolean): Symbol | undefined {
            if (!(isBinaryExpression(node.parent) && node.parent.left === node && node.parent.operatorToken.kind === SyntaxKind.EqualsToken)) {
                return undefined;
            }

            return getTargetOfAliasLikeExpression(node.parent.right, dontRecursivelyResolve);
        }

        function getTargetOfAliasDeclaration(node: Declaration, dontRecursivelyResolve = false): Symbol | undefined {
            switch (node.kind) {
                case SyntaxKind.ImportEqualsDeclaration:
                    return getTargetOfImportEqualsDeclaration(<ImportEqualsDeclaration>node, dontRecursivelyResolve);
                case SyntaxKind.ImportClause:
                    return getTargetOfImportClause(<ImportClause>node, dontRecursivelyResolve);
                case SyntaxKind.NamespaceImport:
                    return getTargetOfNamespaceImport(<NamespaceImport>node, dontRecursivelyResolve);
                case SyntaxKind.NamespaceExport:
                    return getTargetOfNamespaceExport(<NamespaceExport>node, dontRecursivelyResolve);
                case SyntaxKind.ImportSpecifier:
                    return getTargetOfImportSpecifier(<ImportSpecifier>node, dontRecursivelyResolve);
                case SyntaxKind.ExportSpecifier:
                    return getTargetOfExportSpecifier(<ExportSpecifier>node, SymbolFlags.Value | SymbolFlags.Type | SymbolFlags.Namespace, dontRecursivelyResolve);
                case SyntaxKind.ExportAssignment:
                case SyntaxKind.BinaryExpression:
                    return getTargetOfExportAssignment((<ExportAssignment | BinaryExpression>node), dontRecursivelyResolve);
                case SyntaxKind.NamespaceExportDeclaration:
                    return getTargetOfNamespaceExportDeclaration(<NamespaceExportDeclaration>node, dontRecursivelyResolve);
                case SyntaxKind.ShorthandPropertyAssignment:
                    return resolveEntityName((node as ShorthandPropertyAssignment).name, SymbolFlags.Value | SymbolFlags.Type | SymbolFlags.Namespace, /*ignoreErrors*/ true, dontRecursivelyResolve);
                case SyntaxKind.PropertyAssignment:
                    return getTargetOfPropertyAssignment(node as PropertyAssignment, dontRecursivelyResolve);
                case SyntaxKind.PropertyAccessExpression:
                    return getTargetOfPropertyAccessExpression(node as PropertyAccessExpression, dontRecursivelyResolve);
                default:
                    return Debug.fail();
            }
        }

        /**
         * Indicates that a symbol is an alias that does not merge with a local declaration.
         * OR Is a JSContainer which may merge an alias with a local declaration
         */
        function isNonLocalAlias(symbol: Symbol | undefined, excludes = SymbolFlags.Value | SymbolFlags.Type | SymbolFlags.Namespace): symbol is Symbol {
            if (!symbol) return false;
            return (symbol.flags & (SymbolFlags.Alias | excludes)) === SymbolFlags.Alias || !!(symbol.flags & SymbolFlags.Alias && symbol.flags & SymbolFlags.Assignment);
        }

        function resolveSymbol(symbol: Symbol, dontResolveAlias?: boolean): Symbol;
        function resolveSymbol(symbol: Symbol | undefined, dontResolveAlias?: boolean): Symbol | undefined;
        function resolveSymbol(symbol: Symbol | undefined, dontResolveAlias?: boolean): Symbol | undefined {
            return !dontResolveAlias && isNonLocalAlias(symbol) ? resolveAlias(symbol) : symbol;
        }

        function resolveAlias(symbol: Symbol): Symbol {
            Debug.assert((symbol.flags & SymbolFlags.Alias) !== 0, "Should only get Alias here.");
            const links = getSymbolLinks(symbol);
            if (!links.target) {
                links.target = resolvingSymbol;
                const node = getDeclarationOfAliasSymbol(symbol);
                if (!node) return Debug.fail();
                const target = getTargetOfAliasDeclaration(node);
                if (links.target === resolvingSymbol) {
                    links.target = target || unknownSymbol;
                }
                else {
                    error(node, Diagnostics.Circular_definition_of_import_alias_0, symbolToString(symbol));
                }
            }
            else if (links.target === resolvingSymbol) {
                links.target = unknownSymbol;
            }
            return links.target;
        }

        function tryResolveAlias(symbol: Symbol): Symbol | undefined {
            const links = getSymbolLinks(symbol);
            if (links.target !== resolvingSymbol) {
                return resolveAlias(symbol);
            }

            return undefined;
        }

        /**
         * Marks a symbol as type-only if its declaration is syntactically type-only.
         * If it is not itself marked type-only, but resolves to a type-only alias
         * somewhere in its resolution chain, save a reference to the type-only alias declaration
         * so the alias _not_ marked type-only can be identified as _transitively_ type-only.
         *
         * This function is called on each alias declaration that could be type-only or resolve to
         * another type-only alias during `resolveAlias`, so that later, when an alias is used in a
         * JS-emitting expression, we can quickly determine if that symbol is effectively type-only
         * and issue an error if so.
         *
         * @param aliasDeclaration The alias declaration not marked as type-only
         * has already been marked as not resolving to a type-only alias. Used when recursively resolving qualified
         * names of import aliases, e.g. `import C = a.b.C`. If namespace `a` is not found to be type-only, the
         * import declaration will initially be marked as not resolving to a type-only symbol. But, namespace `b`
         * must still be checked for a type-only marker, overwriting the previous negative result if found.
         * @param immediateTarget The symbol to which the alias declaration immediately resolves
         * @param finalTarget The symbol to which the alias declaration ultimately resolves
         * @param overwriteEmpty Checks `resolvesToSymbol` for type-only declarations even if `aliasDeclaration`
         */
        function markSymbolOfAliasDeclarationIfTypeOnly(
            aliasDeclaration: Declaration | undefined,
            immediateTarget: Symbol | undefined,
            finalTarget: Symbol | undefined,
            overwriteEmpty: boolean,
        ): boolean {
            if (!aliasDeclaration) return false;

            // If the declaration itself is type-only, mark it and return.
            // No need to check what it resolves to.
            const sourceSymbol = getSymbolOfNode(aliasDeclaration);
            if (isTypeOnlyImportOrExportDeclaration(aliasDeclaration)) {
                const links = getSymbolLinks(sourceSymbol);
                links.typeOnlyDeclaration = aliasDeclaration;
                return true;
            }

            const links = getSymbolLinks(sourceSymbol);
            return markSymbolOfAliasDeclarationIfTypeOnlyWorker(links, immediateTarget, overwriteEmpty)
                || markSymbolOfAliasDeclarationIfTypeOnlyWorker(links, finalTarget, overwriteEmpty);
        }

        function markSymbolOfAliasDeclarationIfTypeOnlyWorker(aliasDeclarationLinks: SymbolLinks, target: Symbol | undefined, overwriteEmpty: boolean): boolean {
            if (target && (aliasDeclarationLinks.typeOnlyDeclaration === undefined || overwriteEmpty && aliasDeclarationLinks.typeOnlyDeclaration === false)) {
                const exportSymbol = target.exports?.get(InternalSymbolName.ExportEquals) ?? target;
                const typeOnly = exportSymbol.declarations && find(exportSymbol.declarations, isTypeOnlyImportOrExportDeclaration);
                aliasDeclarationLinks.typeOnlyDeclaration = typeOnly ?? getSymbolLinks(exportSymbol).typeOnlyDeclaration ?? false;
            }
            return !!aliasDeclarationLinks.typeOnlyDeclaration;
        }

        /** Indicates that a symbol directly or indirectly resolves to a type-only import or export. */
        function getTypeOnlyAliasDeclaration(symbol: Symbol): TypeOnlyCompatibleAliasDeclaration | undefined {
            if (!(symbol.flags & SymbolFlags.Alias)) {
                return undefined;
            }
            const links = getSymbolLinks(symbol);
            return links.typeOnlyDeclaration || undefined;
        }

        function markExportAsReferenced(node: ImportEqualsDeclaration | ExportSpecifier) {
            const symbol = getSymbolOfNode(node);
            const target = resolveAlias(symbol);
            if (target) {
                const markAlias = target === unknownSymbol ||
                    ((target.flags & SymbolFlags.Value) && !isConstEnumOrConstEnumOnlyModule(target) && !getTypeOnlyAliasDeclaration(symbol));

                if (markAlias) {
                    markAliasSymbolAsReferenced(symbol);
                }
            }
        }

        // When an alias symbol is referenced, we need to mark the entity it references as referenced and in turn repeat that until
        // we reach a non-alias or an exported entity (which is always considered referenced). We do this by checking the target of
        // the alias as an expression (which recursively takes us back here if the target references another alias).
        function markAliasSymbolAsReferenced(symbol: Symbol) {
            const links = getSymbolLinks(symbol);
            if (!links.referenced) {
                links.referenced = true;
                const node = getDeclarationOfAliasSymbol(symbol);
                if (!node) return Debug.fail();
                // We defer checking of the reference of an `import =` until the import itself is referenced,
                // This way a chain of imports can be elided if ultimately the final input is only used in a type
                // position.
                if (isInternalModuleImportEqualsDeclaration(node)) {
                    const target = resolveSymbol(symbol);
                    if (target === unknownSymbol || target.flags & SymbolFlags.Value) {
                        // import foo = <symbol>
                        checkExpressionCached(<Expression>node.moduleReference);
                    }
                }
            }
        }

        // Aliases that resolve to const enums are not marked as referenced because they are not emitted,
        // but their usage in value positions must be tracked to determine if the import can be type-only.
        function markConstEnumAliasAsReferenced(symbol: Symbol) {
            const links = getSymbolLinks(symbol);
            if (!links.constEnumReferenced) {
                links.constEnumReferenced = true;
            }
        }

        // This function is only for imports with entity names
        function getSymbolOfPartOfRightHandSideOfImportEquals(entityName: EntityName, dontResolveAlias?: boolean): Symbol | undefined {
            // There are three things we might try to look for. In the following examples,
            // the search term is enclosed in |...|:
            //
            //     import a = |b|; // Namespace
            //     import a = |b.c|; // Value, type, namespace
            //     import a = |b.c|.d; // Namespace
            if (entityName.kind === SyntaxKind.Identifier && isRightSideOfQualifiedNameOrPropertyAccess(entityName)) {
                entityName = <QualifiedName>entityName.parent;
            }
            // Check for case 1 and 3 in the above example
            if (entityName.kind === SyntaxKind.Identifier || entityName.parent.kind === SyntaxKind.QualifiedName) {
                return resolveEntityName(entityName, SymbolFlags.Namespace, /*ignoreErrors*/ false, dontResolveAlias);
            }
            else {
                // Case 2 in above example
                // entityName.kind could be a QualifiedName or a Missing identifier
                Debug.assert(entityName.parent.kind === SyntaxKind.ImportEqualsDeclaration);
                return resolveEntityName(entityName, SymbolFlags.Value | SymbolFlags.Type | SymbolFlags.Namespace, /*ignoreErrors*/ false, dontResolveAlias);
            }
        }

        function getFullyQualifiedName(symbol: Symbol, containingLocation?: Node): string {
            return symbol.parent ? getFullyQualifiedName(symbol.parent, containingLocation) + "." + symbolToString(symbol) : symbolToString(symbol, containingLocation, /*meaning*/ undefined, SymbolFormatFlags.DoNotIncludeSymbolChain | SymbolFormatFlags.AllowAnyNodeKind);
        }

        /**
         * Resolves a qualified name and any involved aliases.
         */
        function resolveEntityName(name: EntityNameOrEntityNameExpression, meaning: SymbolFlags, ignoreErrors?: boolean, dontResolveAlias?: boolean, location?: Node): Symbol | undefined {
            if (nodeIsMissing(name)) {
                return undefined;
            }

            const namespaceMeaning = SymbolFlags.Namespace | (isInJSFile(name) ? meaning & SymbolFlags.Value : 0);
            let symbol: Symbol | undefined;
            if (name.kind === SyntaxKind.Identifier) {
                const message = meaning === namespaceMeaning || nodeIsSynthesized(name) ? Diagnostics.Cannot_find_namespace_0 : getCannotFindNameDiagnosticForName(getFirstIdentifier(name));
                const symbolFromJSPrototype = isInJSFile(name) && !nodeIsSynthesized(name) ? resolveEntityNameFromAssignmentDeclaration(name, meaning) : undefined;
                symbol = getMergedSymbol(resolveName(location || name, name.escapedText, meaning, ignoreErrors || symbolFromJSPrototype ? undefined : message, name, /*isUse*/ true));
                if (!symbol) {
                    return getMergedSymbol(symbolFromJSPrototype);
                }
            }
            else if (name.kind === SyntaxKind.QualifiedName || name.kind === SyntaxKind.PropertyAccessExpression) {
                const left = name.kind === SyntaxKind.QualifiedName ? name.left : name.expression;
                const right = name.kind === SyntaxKind.QualifiedName ? name.right : name.name;
                let namespace = resolveEntityName(left, namespaceMeaning, ignoreErrors, /*dontResolveAlias*/ false, location);
                if (!namespace || nodeIsMissing(right)) {
                    return undefined;
                }
                else if (namespace === unknownSymbol) {
                    return namespace;
                }
                if (isInJSFile(name)) {
                    if (namespace.valueDeclaration &&
                        isVariableDeclaration(namespace.valueDeclaration) &&
                        namespace.valueDeclaration.initializer &&
                        isCommonJsRequire(namespace.valueDeclaration.initializer)) {
                        const moduleName = (namespace.valueDeclaration.initializer as CallExpression).arguments[0] as StringLiteral;
                        const moduleSym = resolveExternalModuleName(moduleName, moduleName);
                        if (moduleSym) {
                            const resolvedModuleSymbol = resolveExternalModuleSymbol(moduleSym);
                            if (resolvedModuleSymbol) {
                                namespace = resolvedModuleSymbol;
                            }
                        }
                    }
                }
                symbol = getMergedSymbol(getSymbol(getExportsOfSymbol(namespace), right.escapedText, meaning));
                if (!symbol) {
                    if (!ignoreErrors) {
                        error(right, Diagnostics.Namespace_0_has_no_exported_member_1, getFullyQualifiedName(namespace), declarationNameToString(right));
                    }
                    return undefined;
                }
            }
            else {
                throw Debug.assertNever(name, "Unknown entity name kind.");
            }
            Debug.assert((getCheckFlags(symbol) & CheckFlags.Instantiated) === 0, "Should never get an instantiated symbol here.");
            if (!nodeIsSynthesized(name) && isEntityName(name) && (symbol.flags & SymbolFlags.Alias || name.parent.kind === SyntaxKind.ExportAssignment)) {
                markSymbolOfAliasDeclarationIfTypeOnly(getAliasDeclarationFromName(name), symbol, /*finalTarget*/ undefined, /*overwriteEmpty*/ true);
            }
            return (symbol.flags & meaning) || dontResolveAlias ? symbol : resolveAlias(symbol);
        }

        /**
         * 1. For prototype-property methods like `A.prototype.m = function () ...`, try to resolve names in the scope of `A` too.
         * Note that prototype-property assignment to locations outside the current file (eg globals) doesn't work, so
         * name resolution won't work either.
         * 2. For property assignments like `{ x: function f () { } }`, try to resolve names in the scope of `f` too.
         */
        function resolveEntityNameFromAssignmentDeclaration(name: Identifier, meaning: SymbolFlags) {
            if (isJSDocTypeReference(name.parent)) {
                const secondaryLocation = getAssignmentDeclarationLocation(name.parent);
                if (secondaryLocation) {
                    return resolveName(secondaryLocation, name.escapedText, meaning, /*nameNotFoundMessage*/ undefined, name, /*isUse*/ true);
                }
            }
        }

        function getAssignmentDeclarationLocation(node: TypeReferenceNode): Node | undefined {
            const typeAlias = findAncestor(node, node => !(isJSDocNode(node) || node.flags & NodeFlags.JSDoc) ? "quit" : isJSDocTypeAlias(node));
            if (typeAlias) {
                return;
            }
            const host = getJSDocHost(node);
            if (isExpressionStatement(host) &&
                isBinaryExpression(host.expression) &&
                getAssignmentDeclarationKind(host.expression) === AssignmentDeclarationKind.PrototypeProperty) {
                // X.prototype.m = /** @param {K} p */ function () { } <-- look for K on X's declaration
                const symbol = getSymbolOfNode(host.expression.left);
                if (symbol) {
                    return getDeclarationOfJSPrototypeContainer(symbol);
                }
            }
            if ((isObjectLiteralMethod(host) || isPropertyAssignment(host)) &&
                isBinaryExpression(host.parent.parent) &&
                getAssignmentDeclarationKind(host.parent.parent) === AssignmentDeclarationKind.Prototype) {
                // X.prototype = { /** @param {K} p */m() { } } <-- look for K on X's declaration
                const symbol = getSymbolOfNode(host.parent.parent.left);
                if (symbol) {
                    return getDeclarationOfJSPrototypeContainer(symbol);
                }
            }
            const sig = getEffectiveJSDocHost(node);
            if (sig && isFunctionLike(sig)) {
                const symbol = getSymbolOfNode(sig);
                return symbol && symbol.valueDeclaration;
            }
        }

        function getDeclarationOfJSPrototypeContainer(symbol: Symbol) {
            const decl = symbol.parent!.valueDeclaration;
            if (!decl) {
                return undefined;
            }
            const initializer = isAssignmentDeclaration(decl) ? getAssignedExpandoInitializer(decl) :
                hasOnlyExpressionInitializer(decl) ? getDeclaredExpandoInitializer(decl) :
                undefined;
            return initializer || decl;
        }

        /**
         * Get the real symbol of a declaration with an expando initializer.
         *
         * Normally, declarations have an associated symbol, but when a declaration has an expando
         * initializer, the expando's symbol is the one that has all the members merged into it.
         */
        function getExpandoSymbol(symbol: Symbol): Symbol | undefined {
            const decl = symbol.valueDeclaration;
            if (!decl || !isInJSFile(decl) || symbol.flags & SymbolFlags.TypeAlias || getExpandoInitializer(decl, /*isPrototypeAssignment*/ false)) {
                return undefined;
            }
            const init = isVariableDeclaration(decl) ? getDeclaredExpandoInitializer(decl) : getAssignedExpandoInitializer(decl);
            if (init) {
                const initSymbol = getSymbolOfNode(init);
                if (initSymbol) {
                    return mergeJSSymbols(initSymbol, symbol);
                }
            }
        }

        function resolveExternalModuleName(location: Node, moduleReferenceExpression: Expression, ignoreErrors?: boolean): Symbol | undefined {
            const isClassic = getEmitModuleResolutionKind(compilerOptions) === ModuleResolutionKind.Classic;
            const errorMessage = isClassic?
                                    Diagnostics.Cannot_find_module_0_Did_you_mean_to_set_the_moduleResolution_option_to_node_or_to_add_aliases_to_the_paths_option
                                  : Diagnostics.Cannot_find_module_0_or_its_corresponding_type_declarations;
            return resolveExternalModuleNameWorker(location, moduleReferenceExpression, ignoreErrors ? undefined : errorMessage);
        }

        function resolveExternalModuleNameWorker(location: Node, moduleReferenceExpression: Expression, moduleNotFoundError: DiagnosticMessage | undefined, isForAugmentation = false): Symbol | undefined {
            return isStringLiteralLike(moduleReferenceExpression)
                ? resolveExternalModule(location, moduleReferenceExpression.text, moduleNotFoundError, moduleReferenceExpression, isForAugmentation)
                : undefined;
        }

        function resolveExternalModule(location: Node, moduleReference: string, moduleNotFoundError: DiagnosticMessage | undefined, errorNode: Node, isForAugmentation = false): Symbol | undefined {
            if (startsWith(moduleReference, "@types/")) {
                const diag = Diagnostics.Cannot_import_type_declaration_files_Consider_importing_0_instead_of_1;
                const withoutAtTypePrefix = removePrefix(moduleReference, "@types/");
                error(errorNode, diag, withoutAtTypePrefix, moduleReference);
            }

            const ambientModule = tryFindAmbientModule(moduleReference, /*withAugmentations*/ true);
            if (ambientModule) {
                return ambientModule;
            }
            const currentSourceFile = getSourceFileOfNode(location);
            const resolvedModule = getResolvedModule(currentSourceFile, moduleReference)!; // TODO: GH#18217
            const resolutionDiagnostic = resolvedModule && getResolutionDiagnostic(compilerOptions, resolvedModule);
            const sourceFile = resolvedModule && !resolutionDiagnostic && host.getSourceFile(resolvedModule.resolvedFileName);
            if (sourceFile) {
                if (sourceFile.symbol) {
                    if (resolvedModule.isExternalLibraryImport && !resolutionExtensionIsTSOrJson(resolvedModule.extension)) {
                        errorOnImplicitAnyModule(/*isError*/ false, errorNode, resolvedModule, moduleReference);
                    }
                    // merged symbol is module declaration symbol combined with all augmentations
                    return getMergedSymbol(sourceFile.symbol);
                }
                if (moduleNotFoundError) {
                    // report errors only if it was requested
                    error(errorNode, Diagnostics.File_0_is_not_a_module, sourceFile.fileName);
                }
                return undefined;
            }

            if (patternAmbientModules) {
                const pattern = findBestPatternMatch(patternAmbientModules, _ => _.pattern, moduleReference);
                if (pattern) {
                    // If the module reference matched a pattern ambient module ('*.foo') but there's also a
                    // module augmentation by the specific name requested ('a.foo'), we store the merged symbol
                    // by the augmentation name ('a.foo'), because asking for *.foo should not give you exports
                    // from a.foo.
                    const augmentation = patternAmbientModuleAugmentations && patternAmbientModuleAugmentations.get(moduleReference);
                    if (augmentation) {
                        return getMergedSymbol(augmentation);
                    }
                    return getMergedSymbol(pattern.symbol);
                }
            }

            // May be an untyped module. If so, ignore resolutionDiagnostic.
            if (resolvedModule && !resolutionExtensionIsTSOrJson(resolvedModule.extension) && resolutionDiagnostic === undefined || resolutionDiagnostic === Diagnostics.Could_not_find_a_declaration_file_for_module_0_1_implicitly_has_an_any_type) {
                if (isForAugmentation) {
                    const diag = Diagnostics.Invalid_module_name_in_augmentation_Module_0_resolves_to_an_untyped_module_at_1_which_cannot_be_augmented;
                    error(errorNode, diag, moduleReference, resolvedModule.resolvedFileName);
                }
                else {
                    errorOnImplicitAnyModule(/*isError*/ noImplicitAny && !!moduleNotFoundError, errorNode, resolvedModule, moduleReference);
                }
                // Failed imports and untyped modules are both treated in an untyped manner; only difference is whether we give a diagnostic first.
                return undefined;
            }

            if (moduleNotFoundError) {
                // See if this was possibly a projectReference redirect
                if (resolvedModule) {
                    const redirect = host.getProjectReferenceRedirect(resolvedModule.resolvedFileName);
                    if (redirect) {
                        error(errorNode, Diagnostics.Output_file_0_has_not_been_built_from_source_file_1, redirect, resolvedModule.resolvedFileName);
                        return undefined;
                    }
                }

                if (resolutionDiagnostic) {
                    error(errorNode, resolutionDiagnostic, moduleReference, resolvedModule.resolvedFileName);
                }
                else {
                    const tsExtension = tryExtractTSExtension(moduleReference);
                    if (tsExtension) {
                        const diag = Diagnostics.An_import_path_cannot_end_with_a_0_extension_Consider_importing_1_instead;
                        error(errorNode, diag, tsExtension, removeExtension(moduleReference, tsExtension));
                    }
                    else if (!compilerOptions.resolveJsonModule &&
                        fileExtensionIs(moduleReference, Extension.Json) &&
                        getEmitModuleResolutionKind(compilerOptions) === ModuleResolutionKind.NodeJs &&
                        hasJsonModuleEmitEnabled(compilerOptions)) {
                        error(errorNode, Diagnostics.Cannot_find_module_0_Consider_using_resolveJsonModule_to_import_module_with_json_extension, moduleReference);
                    }
                    else {
                        error(errorNode, moduleNotFoundError, moduleReference);
                    }
                }
            }
            return undefined;
        }

        function errorOnImplicitAnyModule(isError: boolean, errorNode: Node, { packageId, resolvedFileName }: ResolvedModuleFull, moduleReference: string): void {
            const errorInfo = !isExternalModuleNameRelative(moduleReference) && packageId
                ? typesPackageExists(packageId.name)
                    ? chainDiagnosticMessages(
                        /*details*/ undefined,
                        Diagnostics.If_the_0_package_actually_exposes_this_module_consider_sending_a_pull_request_to_amend_https_Colon_Slash_Slashgithub_com_SlashDefinitelyTyped_SlashDefinitelyTyped_Slashtree_Slashmaster_Slashtypes_Slash_1,
                        packageId.name, mangleScopedPackageName(packageId.name))
                    : chainDiagnosticMessages(
                        /*details*/ undefined,
                        Diagnostics.Try_npm_install_types_Slash_1_if_it_exists_or_add_a_new_declaration_d_ts_file_containing_declare_module_0,
                        moduleReference,
                        mangleScopedPackageName(packageId.name))
                : undefined;
            errorOrSuggestion(isError, errorNode, chainDiagnosticMessages(
                errorInfo,
                Diagnostics.Could_not_find_a_declaration_file_for_module_0_1_implicitly_has_an_any_type,
                moduleReference,
                resolvedFileName));
        }
        function typesPackageExists(packageName: string): boolean {
            return getPackagesSet().has(getTypesPackageName(packageName));
        }

        function resolveExternalModuleSymbol(moduleSymbol: Symbol, dontResolveAlias?: boolean): Symbol;
        function resolveExternalModuleSymbol(moduleSymbol: Symbol | undefined, dontResolveAlias?: boolean): Symbol | undefined;
        function resolveExternalModuleSymbol(moduleSymbol: Symbol, dontResolveAlias?: boolean): Symbol {
            if (moduleSymbol?.exports) {
                const exportEquals = resolveSymbol(moduleSymbol.exports.get(InternalSymbolName.ExportEquals), dontResolveAlias);
                const exported = getCommonJsExportEquals(getMergedSymbol(exportEquals), getMergedSymbol(moduleSymbol));
                return getMergedSymbol(exported) || moduleSymbol;
            }
            return undefined!;
        }

        function getCommonJsExportEquals(exported: Symbol | undefined, moduleSymbol: Symbol): Symbol | undefined {
            if (!exported || exported === unknownSymbol || exported === moduleSymbol || moduleSymbol.exports!.size === 1 || exported.flags & SymbolFlags.Alias) {
                return exported;
            }
            const links = getSymbolLinks(exported);
            if (links.cjsExportMerged) {
                return links.cjsExportMerged;
            }
            const merged = exported.flags & SymbolFlags.Transient ? exported : cloneSymbol(exported);
            merged.flags = merged.flags | SymbolFlags.ValueModule;
            if (merged.exports === undefined) {
                merged.exports = createSymbolTable();
            }
            moduleSymbol.exports!.forEach((s, name) => {
                if (name === InternalSymbolName.ExportEquals) return;
                merged.exports!.set(name, merged.exports!.has(name) ? mergeSymbol(merged.exports!.get(name)!, s) : s);
            });
            getSymbolLinks(merged).cjsExportMerged = merged;
            return links.cjsExportMerged = merged;
        }

        // An external module with an 'export =' declaration may be referenced as an ES6 module provided the 'export ='
        // references a symbol that is at least declared as a module or a variable. The target of the 'export =' may
        // combine other declarations with the module or variable (e.g. a class/module, function/module, interface/variable).
        function resolveESModuleSymbol(moduleSymbol: Symbol | undefined, referencingLocation: Node, dontResolveAlias: boolean, suppressInteropError: boolean): Symbol | undefined {
            const symbol = resolveExternalModuleSymbol(moduleSymbol, dontResolveAlias);

            if (!dontResolveAlias && symbol) {
                if (!suppressInteropError && !(symbol.flags & (SymbolFlags.Module | SymbolFlags.Variable)) && !getDeclarationOfKind(symbol, SyntaxKind.SourceFile)) {
                    const compilerOptionName = moduleKind >= ModuleKind.ES2015
                        ? "allowSyntheticDefaultImports"
                        : "esModuleInterop";

                    error(referencingLocation, Diagnostics.This_module_can_only_be_referenced_with_ECMAScript_imports_Slashexports_by_turning_on_the_0_flag_and_referencing_its_default_export, compilerOptionName);

                    return symbol;
                }

                if (compilerOptions.esModuleInterop) {
                    const referenceParent = referencingLocation.parent;
                    if (
                        (isImportDeclaration(referenceParent) && getNamespaceDeclarationNode(referenceParent)) ||
                        isImportCall(referenceParent)
                    ) {
                        const type = getTypeOfSymbol(symbol);
                        let sigs = getSignaturesOfStructuredType(type, SignatureKind.Call);
                        if (!sigs || !sigs.length) {
                            sigs = getSignaturesOfStructuredType(type, SignatureKind.Construct);
                        }
                        if (sigs && sigs.length) {
                            const moduleType = getTypeWithSyntheticDefaultImportType(type, symbol, moduleSymbol!);
                            // Create a new symbol which has the module's type less the call and construct signatures
                            const result = createSymbol(symbol.flags, symbol.escapedName);
                            result.declarations = symbol.declarations ? symbol.declarations.slice() : [];
                            result.parent = symbol.parent;
                            result.target = symbol;
                            result.originatingImport = referenceParent;
                            if (symbol.valueDeclaration) result.valueDeclaration = symbol.valueDeclaration;
                            if (symbol.constEnumOnlyModule) result.constEnumOnlyModule = true;
                            if (symbol.members) result.members = new Map(symbol.members);
                            if (symbol.exports) result.exports = new Map(symbol.exports);
                            const resolvedModuleType = resolveStructuredTypeMembers(moduleType as StructuredType); // Should already be resolved from the signature checks above
                            result.type = createAnonymousType(result, resolvedModuleType.members, emptyArray, emptyArray, resolvedModuleType.stringIndexInfo, resolvedModuleType.numberIndexInfo);
                            return result;
                        }
                    }
                }
            }
            return symbol;
        }

        function hasExportAssignmentSymbol(moduleSymbol: Symbol): boolean {
            return moduleSymbol.exports!.get(InternalSymbolName.ExportEquals) !== undefined;
        }

        function getExportsOfModuleAsArray(moduleSymbol: Symbol): Symbol[] {
            return symbolsToArray(getExportsOfModule(moduleSymbol));
        }

        function getExportsAndPropertiesOfModule(moduleSymbol: Symbol): Symbol[] {
            const exports = getExportsOfModuleAsArray(moduleSymbol);
            const exportEquals = resolveExternalModuleSymbol(moduleSymbol);
            if (exportEquals !== moduleSymbol) {
                addRange(exports, getPropertiesOfType(getTypeOfSymbol(exportEquals)));
            }
            return exports;
        }

        function tryGetMemberInModuleExports(memberName: __String, moduleSymbol: Symbol): Symbol | undefined {
            const symbolTable = getExportsOfModule(moduleSymbol);
            if (symbolTable) {
                return symbolTable.get(memberName);
            }
        }

        function tryGetMemberInModuleExportsAndProperties(memberName: __String, moduleSymbol: Symbol): Symbol | undefined {
            const symbol = tryGetMemberInModuleExports(memberName, moduleSymbol);
            if (symbol) {
                return symbol;
            }

            const exportEquals = resolveExternalModuleSymbol(moduleSymbol);
            if (exportEquals === moduleSymbol) {
                return undefined;
            }

            const type = getTypeOfSymbol(exportEquals);
            return type.flags & TypeFlags.Primitive ||
                getObjectFlags(type) & ObjectFlags.Class ||
                isArrayOrTupleLikeType(type)
                ? undefined
                : getPropertyOfType(type, memberName);
        }

        function getExportsOfSymbol(symbol: Symbol): SymbolTable {
            return symbol.flags & SymbolFlags.LateBindingContainer ? getResolvedMembersOrExportsOfSymbol(symbol, MembersOrExportsResolutionKind.resolvedExports) :
                symbol.flags & SymbolFlags.Module ? getExportsOfModule(symbol) :
                symbol.exports || emptySymbols;
        }

        function getExportsOfModule(moduleSymbol: Symbol): SymbolTable {
            const links = getSymbolLinks(moduleSymbol);
            return links.resolvedExports || (links.resolvedExports = getExportsOfModuleWorker(moduleSymbol));
        }

        interface ExportCollisionTracker {
            specifierText: string;
            exportsWithDuplicate: ExportDeclaration[];
        }

        type ExportCollisionTrackerTable = UnderscoreEscapedMap<ExportCollisionTracker>;

        /**
         * Extends one symbol table with another while collecting information on name collisions for error message generation into the `lookupTable` argument
         * Not passing `lookupTable` and `exportNode` disables this collection, and just extends the tables
         */
        function extendExportSymbols(target: SymbolTable, source: SymbolTable | undefined, lookupTable?: ExportCollisionTrackerTable, exportNode?: ExportDeclaration) {
            if (!source) return;
            source.forEach((sourceSymbol, id) => {
                if (id === InternalSymbolName.Default) return;

                const targetSymbol = target.get(id);
                if (!targetSymbol) {
                    target.set(id, sourceSymbol);
                    if (lookupTable && exportNode) {
                        lookupTable.set(id, {
                            specifierText: getTextOfNode(exportNode.moduleSpecifier!)
                        } as ExportCollisionTracker);
                    }
                }
                else if (lookupTable && exportNode && targetSymbol && resolveSymbol(targetSymbol) !== resolveSymbol(sourceSymbol)) {
                    const collisionTracker = lookupTable.get(id)!;
                    if (!collisionTracker.exportsWithDuplicate) {
                        collisionTracker.exportsWithDuplicate = [exportNode];
                    }
                    else {
                        collisionTracker.exportsWithDuplicate.push(exportNode);
                    }
                }
            });
        }

        function getExportsOfModuleWorker(moduleSymbol: Symbol): SymbolTable {
            const visitedSymbols: Symbol[] = [];

            // A module defined by an 'export=' consists of one export that needs to be resolved
            moduleSymbol = resolveExternalModuleSymbol(moduleSymbol);

            return visit(moduleSymbol) || emptySymbols;

            // The ES6 spec permits export * declarations in a module to circularly reference the module itself. For example,
            // module 'a' can 'export * from "b"' and 'b' can 'export * from "a"' without error.
            function visit(symbol: Symbol | undefined): SymbolTable | undefined {
                if (!(symbol && symbol.exports && pushIfUnique(visitedSymbols, symbol))) {
                    return;
                }
                const symbols = new Map(symbol.exports);
                // All export * declarations are collected in an __export symbol by the binder
                const exportStars = symbol.exports.get(InternalSymbolName.ExportStar);
                if (exportStars) {
                    const nestedSymbols = createSymbolTable();
                    const lookupTable: ExportCollisionTrackerTable = new Map();
                    for (const node of exportStars.declarations) {
                        const resolvedModule = resolveExternalModuleName(node, (node as ExportDeclaration).moduleSpecifier!);
                        const exportedSymbols = visit(resolvedModule);
                        extendExportSymbols(
                            nestedSymbols,
                            exportedSymbols,
                            lookupTable,
                            node as ExportDeclaration
                        );
                    }
                    lookupTable.forEach(({ exportsWithDuplicate }, id) => {
                        // It's not an error if the file with multiple `export *`s with duplicate names exports a member with that name itself
                        if (id === "export=" || !(exportsWithDuplicate && exportsWithDuplicate.length) || symbols.has(id)) {
                            return;
                        }
                        for (const node of exportsWithDuplicate) {
                            diagnostics.add(createDiagnosticForNode(
                                node,
                                Diagnostics.Module_0_has_already_exported_a_member_named_1_Consider_explicitly_re_exporting_to_resolve_the_ambiguity,
                                lookupTable.get(id)!.specifierText,
                                unescapeLeadingUnderscores(id)
                            ));
                        }
                    });
                    extendExportSymbols(symbols, nestedSymbols);
                }
                return symbols;
            }
        }

        function getMergedSymbol(symbol: Symbol): Symbol;
        function getMergedSymbol(symbol: Symbol | undefined): Symbol | undefined;
        function getMergedSymbol(symbol: Symbol | undefined): Symbol | undefined {
            let merged: Symbol;
            return symbol && symbol.mergeId && (merged = mergedSymbols[symbol.mergeId]) ? merged : symbol;
        }

        function getSymbolOfNode(node: Declaration): Symbol;
        function getSymbolOfNode(node: Node): Symbol | undefined;
        function getSymbolOfNode(node: Node): Symbol | undefined {
            return getMergedSymbol(node.symbol && getLateBoundSymbol(node.symbol));
        }

        function getParentOfSymbol(symbol: Symbol): Symbol | undefined {
            return getMergedSymbol(symbol.parent && getLateBoundSymbol(symbol.parent));
        }

        function getAlternativeContainingModules(symbol: Symbol, enclosingDeclaration: Node): Symbol[] {
            const containingFile = getSourceFileOfNode(enclosingDeclaration);
            const id = getNodeId(containingFile);
            const links = getSymbolLinks(symbol);
            let results: Symbol[] | undefined;
            if (links.extendedContainersByFile && (results = links.extendedContainersByFile.get(id))) {
                return results;
            }
            if (containingFile && containingFile.imports) {
                // Try to make an import using an import already in the enclosing file, if possible
                for (const importRef of containingFile.imports) {
                    if (nodeIsSynthesized(importRef)) continue; // Synthetic names can't be resolved by `resolveExternalModuleName` - they'll cause a debug assert if they error
                    const resolvedModule = resolveExternalModuleName(enclosingDeclaration, importRef, /*ignoreErrors*/ true);
                    if (!resolvedModule) continue;
                    const ref = getAliasForSymbolInContainer(resolvedModule, symbol);
                    if (!ref) continue;
                    results = append(results, resolvedModule);
                }
                if (length(results)) {
                    (links.extendedContainersByFile || (links.extendedContainersByFile = new Map())).set(id, results!);
                    return results!;
                }
            }
            if (links.extendedContainers) {
                return links.extendedContainers;
            }
            // No results from files already being imported by this file - expand search (expensive, but not location-specific, so cached)
            const otherFiles = host.getSourceFiles();
            for (const file of otherFiles) {
                if (!isExternalModule(file)) continue;
                const sym = getSymbolOfNode(file);
                const ref = getAliasForSymbolInContainer(sym, symbol);
                if (!ref) continue;
                results = append(results, sym);
            }
            return links.extendedContainers = results || emptyArray;
        }

        /**
         * Attempts to find the symbol corresponding to the container a symbol is in - usually this
         * is just its' `.parent`, but for locals, this value is `undefined`
         */
        function getContainersOfSymbol(symbol: Symbol, enclosingDeclaration: Node | undefined, meaning: SymbolFlags): Symbol[] | undefined {
            const container = getParentOfSymbol(symbol);
            // Type parameters end up in the `members` lists but are not externally visible
            if (container && !(symbol.flags & SymbolFlags.TypeParameter)) {
                const additionalContainers = mapDefined(container.declarations, fileSymbolIfFileSymbolExportEqualsContainer);
                const reexportContainers = enclosingDeclaration && getAlternativeContainingModules(symbol, enclosingDeclaration);
                const objectLiteralContainer = getVariableDeclarationOfObjectLiteral(container, meaning);
                if (enclosingDeclaration && getAccessibleSymbolChain(container, enclosingDeclaration, SymbolFlags.Namespace, /*externalOnly*/ false)) {
                    return append(concatenate(concatenate([container], additionalContainers), reexportContainers), objectLiteralContainer); // This order expresses a preference for the real container if it is in scope
                }
                const res = append(append(additionalContainers, container), objectLiteralContainer);
                return concatenate(res, reexportContainers);
            }
            const candidates = mapDefined(symbol.declarations, d => {
                if (!isAmbientModule(d) && d.parent && hasNonGlobalAugmentationExternalModuleSymbol(d.parent)) {
                    return getSymbolOfNode(d.parent);
                }
                if (isClassExpression(d) && isBinaryExpression(d.parent) && d.parent.operatorToken.kind === SyntaxKind.EqualsToken && isAccessExpression(d.parent.left) && isEntityNameExpression(d.parent.left.expression)) {
                    if (isModuleExportsAccessExpression(d.parent.left) || isExportsIdentifier(d.parent.left.expression)) {
                        return getSymbolOfNode(getSourceFileOfNode(d));
                    }
                    checkExpressionCached(d.parent.left.expression);
                    return getNodeLinks(d.parent.left.expression).resolvedSymbol;
                }
            });
            if (!length(candidates)) {
                return undefined;
            }
            return mapDefined(candidates, candidate => getAliasForSymbolInContainer(candidate, symbol) ? candidate : undefined);

            function fileSymbolIfFileSymbolExportEqualsContainer(d: Declaration) {
                return container && getFileSymbolIfFileSymbolExportEqualsContainer(d, container);
            }
        }

        function getVariableDeclarationOfObjectLiteral(symbol: Symbol, meaning: SymbolFlags) {
            // If we're trying to reference some object literal in, eg `var a = { x: 1 }`, the symbol for the literal, `__object`, is distinct
            // from the symbol of the declaration it is being assigned to. Since we can use the declaration to refer to the literal, however,
            // we'd like to make that connection here - potentially causing us to paint the declaration's visibility, and therefore the literal.
            const firstDecl: Node | false = !!length(symbol.declarations) && first(symbol.declarations);
            if (meaning & SymbolFlags.Value && firstDecl && firstDecl.parent && isVariableDeclaration(firstDecl.parent)) {
                if (isObjectLiteralExpression(firstDecl) && firstDecl === firstDecl.parent.initializer || isTypeLiteralNode(firstDecl) && firstDecl === firstDecl.parent.type) {
                    return getSymbolOfNode(firstDecl.parent);
                }
            }
        }

        function getFileSymbolIfFileSymbolExportEqualsContainer(d: Declaration, container: Symbol) {
            const fileSymbol = getExternalModuleContainer(d);
            const exported = fileSymbol && fileSymbol.exports && fileSymbol.exports.get(InternalSymbolName.ExportEquals);
            return exported && getSymbolIfSameReference(exported, container) ? fileSymbol : undefined;
        }

        function getAliasForSymbolInContainer(container: Symbol, symbol: Symbol) {
            if (container === getParentOfSymbol(symbol)) {
                // fast path, `symbol` is either already the alias or isn't aliased
                return symbol;
            }
            // Check if container is a thing with an `export=` which points directly at `symbol`, and if so, return
            // the container itself as the alias for the symbol
            const exportEquals = container.exports && container.exports.get(InternalSymbolName.ExportEquals);
            if (exportEquals && getSymbolIfSameReference(exportEquals, symbol)) {
                return container;
            }
            const exports = getExportsOfSymbol(container);
            const quick = exports.get(symbol.escapedName);
            if (quick && getSymbolIfSameReference(quick, symbol)) {
                return quick;
            }
            return forEachEntry(exports, exported => {
                if (getSymbolIfSameReference(exported, symbol)) {
                    return exported;
                }
            });
        }

        /**
         * Checks if two symbols, through aliasing and/or merging, refer to the same thing
         */
        function getSymbolIfSameReference(s1: Symbol, s2: Symbol) {
            if (getMergedSymbol(resolveSymbol(getMergedSymbol(s1))) === getMergedSymbol(resolveSymbol(getMergedSymbol(s2)))) {
                return s1;
            }
        }

        function getExportSymbolOfValueSymbolIfExported(symbol: Symbol): Symbol;
        function getExportSymbolOfValueSymbolIfExported(symbol: Symbol | undefined): Symbol | undefined;
        function getExportSymbolOfValueSymbolIfExported(symbol: Symbol | undefined): Symbol | undefined {
            return getMergedSymbol(symbol && (symbol.flags & SymbolFlags.ExportValue) !== 0 ? symbol.exportSymbol : symbol);
        }

        function symbolIsValue(symbol: Symbol): boolean {
            return !!(symbol.flags & SymbolFlags.Value || symbol.flags & SymbolFlags.Alias && resolveAlias(symbol).flags & SymbolFlags.Value && !getTypeOnlyAliasDeclaration(symbol));
        }

        function findConstructorDeclaration(node: ClassLikeDeclaration): ConstructorDeclaration | undefined {
            const members = node.members;
            for (const member of members) {
                if (member.kind === SyntaxKind.Constructor && nodeIsPresent((<ConstructorDeclaration>member).body)) {
                    return <ConstructorDeclaration>member;
                }
            }
        }

        function createType(flags: TypeFlags): Type {
            const result = new Type(checker, flags);
            typeCount++;
            result.id = typeCount;
            return result;
        }

        function createIntrinsicType(kind: TypeFlags, intrinsicName: string, objectFlags: ObjectFlags = 0): IntrinsicType {
            const type = <IntrinsicType>createType(kind);
            type.intrinsicName = intrinsicName;
            type.objectFlags = objectFlags;
            return type;
        }

        function createBooleanType(trueFalseTypes: readonly Type[]): IntrinsicType & UnionType {
            const type = <IntrinsicType & UnionType>getUnionType(trueFalseTypes);
            type.flags |= TypeFlags.Boolean;
            type.intrinsicName = "boolean";
            return type;
        }

        function createObjectType(objectFlags: ObjectFlags, symbol?: Symbol): ObjectType {
            const type = <ObjectType>createType(TypeFlags.Object);
            type.objectFlags = objectFlags;
            type.symbol = symbol!;
            type.members = undefined;
            type.properties = undefined;
            type.callSignatures = undefined;
            type.constructSignatures = undefined;
            type.stringIndexInfo = undefined;
            type.numberIndexInfo = undefined;
            return type;
        }

        function createTypeofType() {
            return getUnionType(arrayFrom(typeofEQFacts.keys(), getLiteralType));
        }

        function createTypeParameter(symbol?: Symbol) {
            const type = <TypeParameter>createType(TypeFlags.TypeParameter);
            if (symbol) type.symbol = symbol;
            return type;
        }

        // A reserved member name starts with two underscores, but the third character cannot be an underscore,
        // @, or #. A third underscore indicates an escaped form of an identifier that started
        // with at least two underscores. The @ character indicates that the name is denoted by a well known ES
        // Symbol instance and the # character indicates that the name is a PrivateIdentifier.
        function isReservedMemberName(name: __String) {
            return (name as string).charCodeAt(0) === CharacterCodes._ &&
                (name as string).charCodeAt(1) === CharacterCodes._ &&
                (name as string).charCodeAt(2) !== CharacterCodes._ &&
                (name as string).charCodeAt(2) !== CharacterCodes.at &&
                (name as string).charCodeAt(2) !== CharacterCodes.hash;
        }

        function getNamedMembers(members: SymbolTable): Symbol[] {
            let result: Symbol[] | undefined;
            members.forEach((symbol, id) => {
                if (!isReservedMemberName(id) && symbolIsValue(symbol)) {
                    (result || (result = [])).push(symbol);
                }
            });
            return result || emptyArray;
        }

        function setStructuredTypeMembers(type: StructuredType, members: SymbolTable, callSignatures: readonly Signature[], constructSignatures: readonly Signature[], stringIndexInfo: IndexInfo | undefined, numberIndexInfo: IndexInfo | undefined): ResolvedType {
            (<ResolvedType>type).members = members;
            (<ResolvedType>type).properties = members === emptySymbols ? emptyArray : getNamedMembers(members);
            (<ResolvedType>type).callSignatures = callSignatures;
            (<ResolvedType>type).constructSignatures = constructSignatures;
            (<ResolvedType>type).stringIndexInfo = stringIndexInfo;
            (<ResolvedType>type).numberIndexInfo = numberIndexInfo;
            return <ResolvedType>type;
        }

        function createAnonymousType(symbol: Symbol | undefined, members: SymbolTable, callSignatures: readonly Signature[], constructSignatures: readonly Signature[], stringIndexInfo: IndexInfo | undefined, numberIndexInfo: IndexInfo | undefined): ResolvedType {
            return setStructuredTypeMembers(createObjectType(ObjectFlags.Anonymous, symbol),
                members, callSignatures, constructSignatures, stringIndexInfo, numberIndexInfo);
        }

        function forEachSymbolTableInScope<T>(enclosingDeclaration: Node | undefined, callback: (symbolTable: SymbolTable) => T): T {
            let result: T;
            for (let location = enclosingDeclaration; location; location = location.parent) {
                // Locals of a source file are not in scope (because they get merged into the global symbol table)
                if (location.locals && !isGlobalSourceFile(location)) {
                    if (result = callback(location.locals)) {
                        return result;
                    }
                }
                switch (location.kind) {
                    case SyntaxKind.SourceFile:
                        if (!isExternalOrCommonJsModule(<SourceFile>location)) {
                            break;
                        }
                        // falls through
                    case SyntaxKind.ModuleDeclaration:
                        const sym = getSymbolOfNode(location as ModuleDeclaration);
                        // `sym` may not have exports if this module declaration is backed by the symbol for a `const` that's being rewritten
                        // into a namespace - in such cases, it's best to just let the namespace appear empty (the const members couldn't have referred
                        // to one another anyway)
                        if (result = callback(sym?.exports || emptySymbols)) {
                            return result;
                        }
                        break;
                    case SyntaxKind.ClassDeclaration:
                    case SyntaxKind.ClassExpression:
                    case SyntaxKind.InterfaceDeclaration:
                        // Type parameters are bound into `members` lists so they can merge across declarations
                        // This is troublesome, since in all other respects, they behave like locals :cries:
                        // TODO: the below is shared with similar code in `resolveName` - in fact, rephrasing all this symbol
                        // lookup logic in terms of `resolveName` would be nice
                        // The below is used to lookup type parameters within a class or interface, as they are added to the class/interface locals
                        // These can never be latebound, so the symbol's raw members are sufficient. `getMembersOfNode` cannot be used, as it would
                        // trigger resolving late-bound names, which we may already be in the process of doing while we're here!
                        let table: UnderscoreEscapedMap<Symbol> | undefined;
                        // TODO: Should this filtered table be cached in some way?
                        (getSymbolOfNode(location as ClassLikeDeclaration | InterfaceDeclaration).members || emptySymbols).forEach((memberSymbol, key) => {
                            if (memberSymbol.flags & (SymbolFlags.Type & ~SymbolFlags.Assignment)) {
                                (table || (table = createSymbolTable())).set(key, memberSymbol);
                            }
                        });
                        if (table && (result = callback(table))) {
                            return result;
                        }
                        break;
                }
            }

            return callback(globals);
        }

        function getQualifiedLeftMeaning(rightMeaning: SymbolFlags) {
            // If we are looking in value space, the parent meaning is value, other wise it is namespace
            return rightMeaning === SymbolFlags.Value ? SymbolFlags.Value : SymbolFlags.Namespace;
        }

        function getAccessibleSymbolChain(symbol: Symbol | undefined, enclosingDeclaration: Node | undefined, meaning: SymbolFlags, useOnlyExternalAliasing: boolean, visitedSymbolTablesMap: ESMap<SymbolId, SymbolTable[]> = new Map()): Symbol[] | undefined {
            if (!(symbol && !isPropertyOrMethodDeclarationSymbol(symbol))) {
                return undefined;
            }

            const id = getSymbolId(symbol);
            let visitedSymbolTables = visitedSymbolTablesMap.get(id);
            if (!visitedSymbolTables) {
                visitedSymbolTablesMap.set(id, visitedSymbolTables = []);
            }
            return forEachSymbolTableInScope(enclosingDeclaration, getAccessibleSymbolChainFromSymbolTable);

            /**
             * @param {ignoreQualification} boolean Set when a symbol is being looked for through the exports of another symbol (meaning we have a route to qualify it already)
             */
            function getAccessibleSymbolChainFromSymbolTable(symbols: SymbolTable, ignoreQualification?: boolean): Symbol[] | undefined {
                if (!pushIfUnique(visitedSymbolTables!, symbols)) {
                    return undefined;
                }

                const result = trySymbolTable(symbols, ignoreQualification);
                visitedSymbolTables!.pop();
                return result;
            }

            function canQualifySymbol(symbolFromSymbolTable: Symbol, meaning: SymbolFlags) {
                // If the symbol is equivalent and doesn't need further qualification, this symbol is accessible
                return !needsQualification(symbolFromSymbolTable, enclosingDeclaration, meaning) ||
                    // If symbol needs qualification, make sure that parent is accessible, if it is then this symbol is accessible too
                    !!getAccessibleSymbolChain(symbolFromSymbolTable.parent, enclosingDeclaration, getQualifiedLeftMeaning(meaning), useOnlyExternalAliasing, visitedSymbolTablesMap);
            }

            function isAccessible(symbolFromSymbolTable: Symbol, resolvedAliasSymbol?: Symbol, ignoreQualification?: boolean) {
                return (symbol === (resolvedAliasSymbol || symbolFromSymbolTable) || getMergedSymbol(symbol) === getMergedSymbol(resolvedAliasSymbol || symbolFromSymbolTable)) &&
                    // if the symbolFromSymbolTable is not external module (it could be if it was determined as ambient external module and would be in globals table)
                    // and if symbolFromSymbolTable or alias resolution matches the symbol,
                    // check the symbol can be qualified, it is only then this symbol is accessible
                    !some(symbolFromSymbolTable.declarations, hasNonGlobalAugmentationExternalModuleSymbol) &&
                    (ignoreQualification || canQualifySymbol(getMergedSymbol(symbolFromSymbolTable), meaning));
            }

            function trySymbolTable(symbols: SymbolTable, ignoreQualification: boolean | undefined): Symbol[] | undefined {
                // If symbol is directly available by its name in the symbol table
                if (isAccessible(symbols.get(symbol!.escapedName)!, /*resolvedAliasSymbol*/ undefined, ignoreQualification)) {
                    return [symbol!];
                }

                // Check if symbol is any of the aliases in scope
                const result = forEachEntry(symbols, symbolFromSymbolTable => {
                    if (symbolFromSymbolTable.flags & SymbolFlags.Alias
                        && symbolFromSymbolTable.escapedName !== InternalSymbolName.ExportEquals
                        && symbolFromSymbolTable.escapedName !== InternalSymbolName.Default
                        && !(isUMDExportSymbol(symbolFromSymbolTable) && enclosingDeclaration && isExternalModule(getSourceFileOfNode(enclosingDeclaration)))
                        // If `!useOnlyExternalAliasing`, we can use any type of alias to get the name
                        && (!useOnlyExternalAliasing || some(symbolFromSymbolTable.declarations, isExternalModuleImportEqualsDeclaration))
                        // While exports are generally considered to be in scope, export-specifier declared symbols are _not_
                        // See similar comment in `resolveName` for details
                        && (ignoreQualification || !getDeclarationOfKind(symbolFromSymbolTable, SyntaxKind.ExportSpecifier))
                    ) {

                        const resolvedImportedSymbol = resolveAlias(symbolFromSymbolTable);
                        const candidate = getCandidateListForSymbol(symbolFromSymbolTable, resolvedImportedSymbol, ignoreQualification);
                        if (candidate) {
                            return candidate;
                        }
                    }
                    if (symbolFromSymbolTable.escapedName === symbol!.escapedName && symbolFromSymbolTable.exportSymbol) {
                        if (isAccessible(getMergedSymbol(symbolFromSymbolTable.exportSymbol), /*aliasSymbol*/ undefined, ignoreQualification)) {
                            return [symbol!];
                        }
                    }
                });

                // If there's no result and we're looking at the global symbol table, treat `globalThis` like an alias and try to lookup thru that
                return result || (symbols === globals ? getCandidateListForSymbol(globalThisSymbol, globalThisSymbol, ignoreQualification) : undefined);
            }

            function getCandidateListForSymbol(symbolFromSymbolTable: Symbol, resolvedImportedSymbol: Symbol, ignoreQualification: boolean | undefined) {
                if (isAccessible(symbolFromSymbolTable, resolvedImportedSymbol, ignoreQualification)) {
                    return [symbolFromSymbolTable];
                }

                // Look in the exported members, if we can find accessibleSymbolChain, symbol is accessible using this chain
                // but only if the symbolFromSymbolTable can be qualified
                const candidateTable = getExportsOfSymbol(resolvedImportedSymbol);
                const accessibleSymbolsFromExports = candidateTable && getAccessibleSymbolChainFromSymbolTable(candidateTable, /*ignoreQualification*/ true);
                if (accessibleSymbolsFromExports && canQualifySymbol(symbolFromSymbolTable, getQualifiedLeftMeaning(meaning))) {
                    return [symbolFromSymbolTable].concat(accessibleSymbolsFromExports);
                }
            }
        }

        function needsQualification(symbol: Symbol, enclosingDeclaration: Node | undefined, meaning: SymbolFlags) {
            let qualify = false;
            forEachSymbolTableInScope(enclosingDeclaration, symbolTable => {
                // If symbol of this name is not available in the symbol table we are ok
                let symbolFromSymbolTable = getMergedSymbol(symbolTable.get(symbol.escapedName));
                if (!symbolFromSymbolTable) {
                    // Continue to the next symbol table
                    return false;
                }
                // If the symbol with this name is present it should refer to the symbol
                if (symbolFromSymbolTable === symbol) {
                    // No need to qualify
                    return true;
                }

                // Qualify if the symbol from symbol table has same meaning as expected
                symbolFromSymbolTable = (symbolFromSymbolTable.flags & SymbolFlags.Alias && !getDeclarationOfKind(symbolFromSymbolTable, SyntaxKind.ExportSpecifier)) ? resolveAlias(symbolFromSymbolTable) : symbolFromSymbolTable;
                if (symbolFromSymbolTable.flags & meaning) {
                    qualify = true;
                    return true;
                }

                // Continue to the next symbol table
                return false;
            });

            return qualify;
        }

        function isPropertyOrMethodDeclarationSymbol(symbol: Symbol) {
            if (symbol.declarations && symbol.declarations.length) {
                for (const declaration of symbol.declarations) {
                    switch (declaration.kind) {
                        case SyntaxKind.PropertyDeclaration:
                        case SyntaxKind.MethodDeclaration:
                        case SyntaxKind.GetAccessor:
                        case SyntaxKind.SetAccessor:
                            continue;
                        default:
                            return false;
                    }
                }
                return true;
            }
            return false;
        }

        function isTypeSymbolAccessible(typeSymbol: Symbol, enclosingDeclaration: Node | undefined): boolean {
            const access = isSymbolAccessibleWorker(typeSymbol, enclosingDeclaration, SymbolFlags.Type, /*shouldComputeAliasesToMakeVisible*/ false, /*allowModules*/ true);
            return access.accessibility === SymbolAccessibility.Accessible;
        }

        function isValueSymbolAccessible(typeSymbol: Symbol, enclosingDeclaration: Node | undefined): boolean {
            const access = isSymbolAccessibleWorker(typeSymbol, enclosingDeclaration, SymbolFlags.Value, /*shouldComputeAliasesToMakeVisible*/ false, /*allowModules*/ true);
            return access.accessibility === SymbolAccessibility.Accessible;
        }

        function isSymbolAccessibleByFlags(typeSymbol: Symbol, enclosingDeclaration: Node | undefined, flags: SymbolFlags): boolean {
            const access = isSymbolAccessibleWorker(typeSymbol, enclosingDeclaration, flags, /*shouldComputeAliasesToMakeVisible*/ false, /*allowModules*/ false);
            return access.accessibility === SymbolAccessibility.Accessible;
        }

        function isAnySymbolAccessible(symbols: Symbol[] | undefined, enclosingDeclaration: Node | undefined, initialSymbol: Symbol, meaning: SymbolFlags, shouldComputeAliasesToMakeVisible: boolean, allowModules: boolean): SymbolAccessibilityResult | undefined {
            if (!length(symbols)) return;

            let hadAccessibleChain: Symbol | undefined;
            let earlyModuleBail = false;
            for (const symbol of symbols!) {
                // Symbol is accessible if it by itself is accessible
                const accessibleSymbolChain = getAccessibleSymbolChain(symbol, enclosingDeclaration, meaning, /*useOnlyExternalAliasing*/ false);
                if (accessibleSymbolChain) {
                    hadAccessibleChain = symbol;
                    const hasAccessibleDeclarations = hasVisibleDeclarations(accessibleSymbolChain[0], shouldComputeAliasesToMakeVisible);
                    if (hasAccessibleDeclarations) {
                        return hasAccessibleDeclarations;
                    }
                }
                else if (allowModules) {
                    if (some(symbol.declarations, hasNonGlobalAugmentationExternalModuleSymbol)) {
                        if (shouldComputeAliasesToMakeVisible) {
                            earlyModuleBail = true;
                            // Generally speaking, we want to use the aliases that already exist to refer to a module, if present
                            // In order to do so, we need to find those aliases in order to retain them in declaration emit; so
                            // if we are in declaration emit, we cannot use the fast path for module visibility until we've exhausted
                            // all other visibility options (in order to capture the possible aliases used to reference the module)
                            continue;
                        }
                        // Any meaning of a module symbol is always accessible via an `import` type
                        return {
                            accessibility: SymbolAccessibility.Accessible
                        };
                    }
                }

                // If we haven't got the accessible symbol, it doesn't mean the symbol is actually inaccessible.
                // It could be a qualified symbol and hence verify the path
                // e.g.:
                // module m {
                //     export class c {
                //     }
                // }
                // const x: typeof m.c
                // In the above example when we start with checking if typeof m.c symbol is accessible,
                // we are going to see if c can be accessed in scope directly.
                // But it can't, hence the accessible is going to be undefined, but that doesn't mean m.c is inaccessible
                // It is accessible if the parent m is accessible because then m.c can be accessed through qualification

                const containers = getContainersOfSymbol(symbol, enclosingDeclaration, meaning);
                const parentResult = isAnySymbolAccessible(containers, enclosingDeclaration, initialSymbol, initialSymbol === symbol ? getQualifiedLeftMeaning(meaning) : meaning, shouldComputeAliasesToMakeVisible, allowModules);
                if (parentResult) {
                    return parentResult;
                }
            }

            if (earlyModuleBail) {
                return {
                    accessibility: SymbolAccessibility.Accessible
                };
            }

            if (hadAccessibleChain) {
                return {
                    accessibility: SymbolAccessibility.NotAccessible,
                    errorSymbolName: symbolToString(initialSymbol, enclosingDeclaration, meaning),
                    errorModuleName: hadAccessibleChain !== initialSymbol ? symbolToString(hadAccessibleChain, enclosingDeclaration, SymbolFlags.Namespace) : undefined,
                };
            }
        }

        /**
         * Check if the given symbol in given enclosing declaration is accessible and mark all associated alias to be visible if requested
         *
         * @param symbol a Symbol to check if accessible
         * @param enclosingDeclaration a Node containing reference to the symbol
         * @param meaning a SymbolFlags to check if such meaning of the symbol is accessible
         * @param shouldComputeAliasToMakeVisible a boolean value to indicate whether to return aliases to be mark visible in case the symbol is accessible
         */
        function isSymbolAccessible(symbol: Symbol | undefined, enclosingDeclaration: Node | undefined, meaning: SymbolFlags, shouldComputeAliasesToMakeVisible: boolean): SymbolAccessibilityResult {
            return isSymbolAccessibleWorker(symbol, enclosingDeclaration, meaning, shouldComputeAliasesToMakeVisible, /*allowModules*/ true);
        }

        function isSymbolAccessibleWorker(symbol: Symbol | undefined, enclosingDeclaration: Node | undefined, meaning: SymbolFlags, shouldComputeAliasesToMakeVisible: boolean, allowModules: boolean): SymbolAccessibilityResult {
            if (symbol && enclosingDeclaration) {
                const result = isAnySymbolAccessible([symbol], enclosingDeclaration, symbol, meaning, shouldComputeAliasesToMakeVisible, allowModules);
                if (result) {
                    return result;
                }

                // This could be a symbol that is not exported in the external module
                // or it could be a symbol from different external module that is not aliased and hence cannot be named
                const symbolExternalModule = forEach(symbol.declarations, getExternalModuleContainer);
                if (symbolExternalModule) {
                    const enclosingExternalModule = getExternalModuleContainer(enclosingDeclaration);
                    if (symbolExternalModule !== enclosingExternalModule) {
                        // name from different external module that is not visible
                        return {
                            accessibility: SymbolAccessibility.CannotBeNamed,
                            errorSymbolName: symbolToString(symbol, enclosingDeclaration, meaning),
                            errorModuleName: symbolToString(symbolExternalModule)
                        };
                    }
                }

                // Just a local name that is not accessible
                return {
                    accessibility: SymbolAccessibility.NotAccessible,
                    errorSymbolName: symbolToString(symbol, enclosingDeclaration, meaning),
                };
            }

            return { accessibility: SymbolAccessibility.Accessible };
        }

        function getExternalModuleContainer(declaration: Node) {
            const node = findAncestor(declaration, hasExternalModuleSymbol);
            return node && getSymbolOfNode(node);
        }

        function hasExternalModuleSymbol(declaration: Node) {
            return isAmbientModule(declaration) || (declaration.kind === SyntaxKind.SourceFile && isExternalOrCommonJsModule(<SourceFile>declaration));
        }

        function hasNonGlobalAugmentationExternalModuleSymbol(declaration: Node) {
            return isModuleWithStringLiteralName(declaration) || (declaration.kind === SyntaxKind.SourceFile && isExternalOrCommonJsModule(<SourceFile>declaration));
        }

        function hasVisibleDeclarations(symbol: Symbol, shouldComputeAliasToMakeVisible: boolean): SymbolVisibilityResult | undefined {
            let aliasesToMakeVisible: LateVisibilityPaintedStatement[] | undefined;
            if (!every(filter(symbol.declarations, d => d.kind !== SyntaxKind.Identifier), getIsDeclarationVisible)) {
                return undefined;
            }
            return { accessibility: SymbolAccessibility.Accessible, aliasesToMakeVisible };

            function getIsDeclarationVisible(declaration: Declaration) {
                if (!isDeclarationVisible(declaration)) {
                    // Mark the unexported alias as visible if its parent is visible
                    // because these kind of aliases can be used to name types in declaration file

                    const anyImportSyntax = getAnyImportSyntax(declaration);
                    if (anyImportSyntax &&
                        !hasSyntacticModifier(anyImportSyntax, ModifierFlags.Export) && // import clause without export
                        isDeclarationVisible(anyImportSyntax.parent)) {
                        return addVisibleAlias(declaration, anyImportSyntax);
                    }
                    else if (isVariableDeclaration(declaration) && isVariableStatement(declaration.parent.parent) &&
                        !hasSyntacticModifier(declaration.parent.parent, ModifierFlags.Export) && // unexported variable statement
                        isDeclarationVisible(declaration.parent.parent.parent)) {
                        return addVisibleAlias(declaration, declaration.parent.parent);
                    }
                    else if (isLateVisibilityPaintedStatement(declaration) // unexported top-level statement
                        && !hasSyntacticModifier(declaration, ModifierFlags.Export)
                        && isDeclarationVisible(declaration.parent)) {
                        return addVisibleAlias(declaration, declaration);
                    }

                    // Declaration is not visible
                    return false;
                }

                return true;
            }

            function addVisibleAlias(declaration: Declaration, aliasingStatement: LateVisibilityPaintedStatement) {
                // In function "buildTypeDisplay" where we decide whether to write type-alias or serialize types,
                // we want to just check if type- alias is accessible or not but we don't care about emitting those alias at that time
                // since we will do the emitting later in trackSymbol.
                if (shouldComputeAliasToMakeVisible) {
                    getNodeLinks(declaration).isVisible = true;
                    aliasesToMakeVisible = appendIfUnique(aliasesToMakeVisible, aliasingStatement);
                }
                return true;
            }
        }

        function isEntityNameVisible(entityName: EntityNameOrEntityNameExpression, enclosingDeclaration: Node): SymbolVisibilityResult {
            // get symbol of the first identifier of the entityName
            let meaning: SymbolFlags;
            if (entityName.parent.kind === SyntaxKind.TypeQuery ||
                isExpressionWithTypeArgumentsInClassExtendsClause(entityName.parent) ||
                entityName.parent.kind === SyntaxKind.ComputedPropertyName) {
                // Typeof value
                meaning = SymbolFlags.Value | SymbolFlags.ExportValue;
            }
            else if (entityName.kind === SyntaxKind.QualifiedName || entityName.kind === SyntaxKind.PropertyAccessExpression ||
                entityName.parent.kind === SyntaxKind.ImportEqualsDeclaration) {
                // Left identifier from type reference or TypeAlias
                // Entity name of the import declaration
                meaning = SymbolFlags.Namespace;
            }
            else {
                // Type Reference or TypeAlias entity = Identifier
                meaning = SymbolFlags.Type;
            }

            const firstIdentifier = getFirstIdentifier(entityName);
            const symbol = resolveName(enclosingDeclaration, firstIdentifier.escapedText, meaning, /*nodeNotFoundErrorMessage*/ undefined, /*nameArg*/ undefined, /*isUse*/ false);
            if (symbol && symbol.flags & SymbolFlags.TypeParameter && meaning & SymbolFlags.Type) {
                return { accessibility: SymbolAccessibility.Accessible };
            }

            // Verify if the symbol is accessible
            return (symbol && hasVisibleDeclarations(symbol, /*shouldComputeAliasToMakeVisible*/ true)) || {
                accessibility: SymbolAccessibility.NotAccessible,
                errorSymbolName: getTextOfNode(firstIdentifier),
                errorNode: firstIdentifier
            };
        }

        function symbolToString(symbol: Symbol, enclosingDeclaration?: Node, meaning?: SymbolFlags, flags: SymbolFormatFlags = SymbolFormatFlags.AllowAnyNodeKind, writer?: EmitTextWriter): string {
            let nodeFlags = NodeBuilderFlags.IgnoreErrors;
            if (flags & SymbolFormatFlags.UseOnlyExternalAliasing) {
                nodeFlags |= NodeBuilderFlags.UseOnlyExternalAliasing;
            }
            if (flags & SymbolFormatFlags.WriteTypeParametersOrArguments) {
                nodeFlags |= NodeBuilderFlags.WriteTypeParametersInQualifiedName;
            }
            if (flags & SymbolFormatFlags.UseAliasDefinedOutsideCurrentScope) {
                nodeFlags |= NodeBuilderFlags.UseAliasDefinedOutsideCurrentScope;
            }
            if (flags & SymbolFormatFlags.DoNotIncludeSymbolChain) {
                nodeFlags |= NodeBuilderFlags.DoNotIncludeSymbolChain;
            }
            const builder = flags & SymbolFormatFlags.AllowAnyNodeKind ? nodeBuilder.symbolToExpression : nodeBuilder.symbolToEntityName;
            return writer ? symbolToStringWorker(writer).getText() : usingSingleLineStringWriter(symbolToStringWorker);

            function symbolToStringWorker(writer: EmitTextWriter) {
                const entity = builder(symbol, meaning!, enclosingDeclaration, nodeFlags)!; // TODO: GH#18217
                // add neverAsciiEscape for GH#39027
                const printer = enclosingDeclaration?.kind === SyntaxKind.SourceFile ? createPrinter({ removeComments: true, neverAsciiEscape: true }) : createPrinter({ removeComments: true });
                const sourceFile = enclosingDeclaration && getSourceFileOfNode(enclosingDeclaration);
                printer.writeNode(EmitHint.Unspecified, entity, /*sourceFile*/ sourceFile, writer);
                return writer;
            }
        }

        function signatureToString(signature: Signature, enclosingDeclaration?: Node, flags = TypeFormatFlags.None, kind?: SignatureKind, writer?: EmitTextWriter): string {
            return writer ? signatureToStringWorker(writer).getText() : usingSingleLineStringWriter(signatureToStringWorker);

            function signatureToStringWorker(writer: EmitTextWriter) {
                let sigOutput: SyntaxKind;
                if (flags & TypeFormatFlags.WriteArrowStyleSignature) {
                    sigOutput = kind === SignatureKind.Construct ? SyntaxKind.ConstructorType : SyntaxKind.FunctionType;
                }
                else {
                    sigOutput = kind === SignatureKind.Construct ? SyntaxKind.ConstructSignature : SyntaxKind.CallSignature;
                }
                const sig = nodeBuilder.signatureToSignatureDeclaration(signature, sigOutput, enclosingDeclaration, toNodeBuilderFlags(flags) | NodeBuilderFlags.IgnoreErrors | NodeBuilderFlags.WriteTypeParametersInQualifiedName);
                const printer = createPrinter({ removeComments: true, omitTrailingSemicolon: true });
                const sourceFile = enclosingDeclaration && getSourceFileOfNode(enclosingDeclaration);
                printer.writeNode(EmitHint.Unspecified, sig!, /*sourceFile*/ sourceFile, getTrailingSemicolonDeferringWriter(writer)); // TODO: GH#18217
                return writer;
            }
        }

        function typeToString(type: Type, enclosingDeclaration?: Node, flags: TypeFormatFlags = TypeFormatFlags.AllowUniqueESSymbolType | TypeFormatFlags.UseAliasDefinedOutsideCurrentScope, writer: EmitTextWriter = createTextWriter("")): string {
            const noTruncation = compilerOptions.noErrorTruncation || flags & TypeFormatFlags.NoTruncation;
            const typeNode = nodeBuilder.typeToTypeNode(type, enclosingDeclaration, toNodeBuilderFlags(flags) | NodeBuilderFlags.IgnoreErrors | (noTruncation ? NodeBuilderFlags.NoTruncation : 0), writer);
            if (typeNode === undefined) return Debug.fail("should always get typenode");
            const options = { removeComments: true };
            const printer = createPrinter(options);
            const sourceFile = enclosingDeclaration && getSourceFileOfNode(enclosingDeclaration);
            printer.writeNode(EmitHint.Unspecified, typeNode, /*sourceFile*/ sourceFile, writer);
            const result = writer.getText();

            const maxLength = noTruncation ? noTruncationMaximumTruncationLength * 2 : defaultMaximumTruncationLength * 2;
            if (maxLength && result && result.length >= maxLength) {
                return result.substr(0, maxLength - "...".length) + "...";
            }
            return result;
        }

        function getTypeNamesForErrorDisplay(left: Type, right: Type): [string, string] {
            let leftStr = symbolValueDeclarationIsContextSensitive(left.symbol) ? typeToString(left, left.symbol.valueDeclaration) : typeToString(left);
            let rightStr = symbolValueDeclarationIsContextSensitive(right.symbol) ? typeToString(right, right.symbol.valueDeclaration) : typeToString(right);
            if (leftStr === rightStr) {
                leftStr = getTypeNameForErrorDisplay(left);
                rightStr = getTypeNameForErrorDisplay(right);
            }
            return [leftStr, rightStr];
        }

        function getTypeNameForErrorDisplay(type: Type) {
            return typeToString(type, /*enclosingDeclaration*/ undefined, TypeFormatFlags.UseFullyQualifiedType);
        }

        function symbolValueDeclarationIsContextSensitive(symbol: Symbol): boolean {
            return symbol && symbol.valueDeclaration && isExpression(symbol.valueDeclaration) && !isContextSensitive(symbol.valueDeclaration);
        }

        function toNodeBuilderFlags(flags = TypeFormatFlags.None): NodeBuilderFlags {
            return flags & TypeFormatFlags.NodeBuilderFlagsMask;
        }

        function createNodeBuilder() {
            return {
                typeToTypeNode: (type: Type, enclosingDeclaration?: Node, flags?: NodeBuilderFlags, tracker?: SymbolTracker) =>
                    withContext(enclosingDeclaration, flags, tracker, context => typeToTypeNodeHelper(type, context)),
                indexInfoToIndexSignatureDeclaration: (indexInfo: IndexInfo, kind: IndexKind, enclosingDeclaration?: Node, flags?: NodeBuilderFlags, tracker?: SymbolTracker) =>
                    withContext(enclosingDeclaration, flags, tracker, context => indexInfoToIndexSignatureDeclarationHelper(indexInfo, kind, context, /*typeNode*/ undefined)),
                signatureToSignatureDeclaration: (signature: Signature, kind: SignatureDeclaration["kind"], enclosingDeclaration?: Node, flags?: NodeBuilderFlags, tracker?: SymbolTracker) =>
                    withContext(enclosingDeclaration, flags, tracker, context => signatureToSignatureDeclarationHelper(signature, kind, context)),
                symbolToEntityName: (symbol: Symbol, meaning: SymbolFlags, enclosingDeclaration?: Node, flags?: NodeBuilderFlags, tracker?: SymbolTracker) =>
                    withContext(enclosingDeclaration, flags, tracker, context => symbolToName(symbol, context, meaning, /*expectsIdentifier*/ false)),
                symbolToExpression: (symbol: Symbol, meaning: SymbolFlags, enclosingDeclaration?: Node, flags?: NodeBuilderFlags, tracker?: SymbolTracker) =>
                    withContext(enclosingDeclaration, flags, tracker, context => symbolToExpression(symbol, context, meaning)),
                symbolToTypeParameterDeclarations: (symbol: Symbol, enclosingDeclaration?: Node, flags?: NodeBuilderFlags, tracker?: SymbolTracker) =>
                    withContext(enclosingDeclaration, flags, tracker, context => typeParametersToTypeParameterDeclarations(symbol, context)),
                symbolToParameterDeclaration: (symbol: Symbol, enclosingDeclaration?: Node, flags?: NodeBuilderFlags, tracker?: SymbolTracker) =>
                    withContext(enclosingDeclaration, flags, tracker, context => symbolToParameterDeclaration(symbol, context)),
                typeParameterToDeclaration: (parameter: TypeParameter, enclosingDeclaration?: Node, flags?: NodeBuilderFlags, tracker?: SymbolTracker) =>
                    withContext(enclosingDeclaration, flags, tracker, context => typeParameterToDeclaration(parameter, context)),
                symbolTableToDeclarationStatements: (symbolTable: SymbolTable, enclosingDeclaration?: Node, flags?: NodeBuilderFlags, tracker?: SymbolTracker, bundled?: boolean) =>
                    withContext(enclosingDeclaration, flags, tracker, context => symbolTableToDeclarationStatements(symbolTable, context, bundled)),
            };

            function withContext<T>(enclosingDeclaration: Node | undefined, flags: NodeBuilderFlags | undefined, tracker: SymbolTracker | undefined, cb: (context: NodeBuilderContext) => T): T | undefined {
                Debug.assert(enclosingDeclaration === undefined || (enclosingDeclaration.flags & NodeFlags.Synthesized) === 0);
                const context: NodeBuilderContext = {
                    enclosingDeclaration,
                    flags: flags || NodeBuilderFlags.None,
                    // If no full tracker is provided, fake up a dummy one with a basic limited-functionality moduleResolverHost
                    tracker: tracker && tracker.trackSymbol ? tracker : { trackSymbol: noop, moduleResolverHost: flags! & NodeBuilderFlags.DoNotIncludeSymbolChain ? {
                        getCommonSourceDirectory: !!(host as Program).getCommonSourceDirectory ? () => (host as Program).getCommonSourceDirectory() : () => "",
                        getSourceFiles: () => host.getSourceFiles(),
                        getCurrentDirectory: () => host.getCurrentDirectory(),
                        getSymlinkCache: maybeBind(host, host.getSymlinkCache),
                        useCaseSensitiveFileNames: maybeBind(host, host.useCaseSensitiveFileNames),
                        redirectTargetsMap: host.redirectTargetsMap,
                        getProjectReferenceRedirect: fileName => host.getProjectReferenceRedirect(fileName),
                        isSourceOfProjectReferenceRedirect: fileName => host.isSourceOfProjectReferenceRedirect(fileName),
                        fileExists: fileName => host.fileExists(fileName),
                    } : undefined },
                    encounteredError: false,
                    visitedTypes: undefined,
                    symbolDepth: undefined,
                    inferTypeParameters: undefined,
                    approximateLength: 0
                };
                const resultingNode = cb(context);
                return context.encounteredError ? undefined : resultingNode;
            }

            function checkTruncationLength(context: NodeBuilderContext): boolean {
                if (context.truncating) return context.truncating;
                return context.truncating = context.approximateLength > ((context.flags & NodeBuilderFlags.NoTruncation) ? noTruncationMaximumTruncationLength : defaultMaximumTruncationLength);
            }

            function typeToTypeNodeHelper(type: Type, context: NodeBuilderContext): TypeNode {
                if (cancellationToken && cancellationToken.throwIfCancellationRequested) {
                    cancellationToken.throwIfCancellationRequested();
                }
                const inTypeAlias = context.flags & NodeBuilderFlags.InTypeAlias;
                context.flags &= ~NodeBuilderFlags.InTypeAlias;

                if (!type) {
                    if (!(context.flags & NodeBuilderFlags.AllowEmptyUnionOrIntersection)) {
                        context.encounteredError = true;
                        return undefined!; // TODO: GH#18217
                    }
                    context.approximateLength += 3;
                    return factory.createKeywordTypeNode(SyntaxKind.AnyKeyword);
                }

                if (!(context.flags & NodeBuilderFlags.NoTypeReduction)) {
                    type = getReducedType(type);
                }

                if (type.flags & TypeFlags.Any) {
                    context.approximateLength += 3;
                    return factory.createKeywordTypeNode(SyntaxKind.AnyKeyword);
                }
                if (type.flags & TypeFlags.Unknown) {
                    return factory.createKeywordTypeNode(SyntaxKind.UnknownKeyword);
                }
                if (type.flags & TypeFlags.String) {
                    context.approximateLength += 6;
                    return factory.createKeywordTypeNode(SyntaxKind.StringKeyword);
                }
                if (type.flags & TypeFlags.Number) {
                    context.approximateLength += 6;
                    return factory.createKeywordTypeNode(SyntaxKind.NumberKeyword);
                }
                if (type.flags & TypeFlags.BigInt) {
                    context.approximateLength += 6;
                    return factory.createKeywordTypeNode(SyntaxKind.BigIntKeyword);
                }
                if (type.flags & TypeFlags.Boolean) {
                    context.approximateLength += 7;
                    return factory.createKeywordTypeNode(SyntaxKind.BooleanKeyword);
                }
                if (type.flags & TypeFlags.EnumLiteral && !(type.flags & TypeFlags.Union)) {
                    const parentSymbol = getParentOfSymbol(type.symbol)!;
                    const parentName = symbolToTypeNode(parentSymbol, context, SymbolFlags.Type);
                    const enumLiteralName = getDeclaredTypeOfSymbol(parentSymbol) === type
                        ? parentName
                        : appendReferenceToType(
                            parentName as TypeReferenceNode | ImportTypeNode,
                            factory.createTypeReferenceNode(symbolName(type.symbol), /*typeArguments*/ undefined)
                        );
                    return enumLiteralName;
                }
                if (type.flags & TypeFlags.EnumLike) {
                    return symbolToTypeNode(type.symbol, context, SymbolFlags.Type);
                }
                if (type.flags & TypeFlags.StringLiteral) {
                    context.approximateLength += ((<StringLiteralType>type).value.length + 2);
                    return factory.createLiteralTypeNode(setEmitFlags(factory.createStringLiteral((<StringLiteralType>type).value, !!(context.flags & NodeBuilderFlags.UseSingleQuotesForStringLiteralType)), EmitFlags.NoAsciiEscaping));
                }
                if (type.flags & TypeFlags.NumberLiteral) {
                    const value = (<NumberLiteralType>type).value;
                    context.approximateLength += ("" + value).length;
                    return factory.createLiteralTypeNode(value < 0 ? factory.createPrefixUnaryExpression(SyntaxKind.MinusToken, factory.createNumericLiteral(-value)) : factory.createNumericLiteral(value));
                }
                if (type.flags & TypeFlags.BigIntLiteral) {
                    context.approximateLength += (pseudoBigIntToString((<BigIntLiteralType>type).value).length) + 1;
                    return factory.createLiteralTypeNode((factory.createBigIntLiteral((<BigIntLiteralType>type).value)));
                }
                if (type.flags & TypeFlags.BooleanLiteral) {
                    context.approximateLength += (<IntrinsicType>type).intrinsicName.length;
                    return factory.createLiteralTypeNode((<IntrinsicType>type).intrinsicName === "true" ? factory.createTrue() : factory.createFalse());
                }
                if (type.flags & TypeFlags.UniqueESSymbol) {
                    if (!(context.flags & NodeBuilderFlags.AllowUniqueESSymbolType)) {
                        if (isValueSymbolAccessible(type.symbol, context.enclosingDeclaration)) {
                            context.approximateLength += 6;
                            return symbolToTypeNode(type.symbol, context, SymbolFlags.Value);
                        }
                        if (context.tracker.reportInaccessibleUniqueSymbolError) {
                            context.tracker.reportInaccessibleUniqueSymbolError();
                        }
                    }
                    context.approximateLength += 13;
                    return factory.createTypeOperatorNode(SyntaxKind.UniqueKeyword, factory.createKeywordTypeNode(SyntaxKind.SymbolKeyword));
                }
                if (type.flags & TypeFlags.Void) {
                    context.approximateLength += 4;
                    return factory.createKeywordTypeNode(SyntaxKind.VoidKeyword);
                }
                if (type.flags & TypeFlags.Undefined) {
                    context.approximateLength += 9;
                    return factory.createKeywordTypeNode(SyntaxKind.UndefinedKeyword);
                }
                if (type.flags & TypeFlags.Null) {
                    context.approximateLength += 4;
                    return factory.createLiteralTypeNode(factory.createNull());
                }
                if (type.flags & TypeFlags.Never) {
                    context.approximateLength += 5;
                    return factory.createKeywordTypeNode(SyntaxKind.NeverKeyword);
                }
                if (type.flags & TypeFlags.ESSymbol) {
                    context.approximateLength += 6;
                    return factory.createKeywordTypeNode(SyntaxKind.SymbolKeyword);
                }
                if (type.flags & TypeFlags.NonPrimitive) {
                    context.approximateLength += 6;
                    return factory.createKeywordTypeNode(SyntaxKind.ObjectKeyword);
                }
                if (isThisTypeParameter(type)) {
                    if (context.flags & NodeBuilderFlags.InObjectTypeLiteral) {
                        if (!context.encounteredError && !(context.flags & NodeBuilderFlags.AllowThisInObjectLiteral)) {
                            context.encounteredError = true;
                        }
                        if (context.tracker.reportInaccessibleThisError) {
                            context.tracker.reportInaccessibleThisError();
                        }
                    }
                    context.approximateLength += 4;
                    return factory.createThisTypeNode();
                }

                if (!inTypeAlias && type.aliasSymbol && (context.flags & NodeBuilderFlags.UseAliasDefinedOutsideCurrentScope || isTypeSymbolAccessible(type.aliasSymbol, context.enclosingDeclaration))) {
                    const typeArgumentNodes = mapToTypeNodes(type.aliasTypeArguments, context);
                    if (isReservedMemberName(type.aliasSymbol.escapedName) && !(type.aliasSymbol.flags & SymbolFlags.Class)) return factory.createTypeReferenceNode(factory.createIdentifier(""), typeArgumentNodes);
                    return symbolToTypeNode(type.aliasSymbol, context, SymbolFlags.Type, typeArgumentNodes);
                }

                const objectFlags = getObjectFlags(type);

                if (objectFlags & ObjectFlags.Reference) {
                    Debug.assert(!!(type.flags & TypeFlags.Object));
                    return (<TypeReference>type).node ? visitAndTransformType(type, typeReferenceToTypeNode) : typeReferenceToTypeNode(<TypeReference>type);
                }
                if (type.flags & TypeFlags.TypeParameter || objectFlags & ObjectFlags.ClassOrInterface) {
                    if (type.flags & TypeFlags.TypeParameter && contains(context.inferTypeParameters, type)) {
                        context.approximateLength += (symbolName(type.symbol).length + 6);
                        return factory.createInferTypeNode(typeParameterToDeclarationWithConstraint(type as TypeParameter, context, /*constraintNode*/ undefined));
                    }
                    if (context.flags & NodeBuilderFlags.GenerateNamesForShadowedTypeParams &&
                        type.flags & TypeFlags.TypeParameter &&
                        !isTypeSymbolAccessible(type.symbol, context.enclosingDeclaration)) {
                        const name = typeParameterToName(type, context);
                        context.approximateLength += idText(name).length;
                        return factory.createTypeReferenceNode(factory.createIdentifier(idText(name)), /*typeArguments*/ undefined);
                    }
                    // Ignore constraint/default when creating a usage (as opposed to declaration) of a type parameter.
                    return type.symbol
                        ? symbolToTypeNode(type.symbol, context, SymbolFlags.Type)
                        : factory.createTypeReferenceNode(factory.createIdentifier("?"), /*typeArguments*/ undefined);
                }
                if (type.flags & (TypeFlags.Union | TypeFlags.Intersection)) {
                    const types = type.flags & TypeFlags.Union ? formatUnionTypes((<UnionType>type).types) : (<IntersectionType>type).types;
                    if (length(types) === 1) {
                        return typeToTypeNodeHelper(types[0], context);
                    }
                    const typeNodes = mapToTypeNodes(types, context, /*isBareList*/ true);
                    if (typeNodes && typeNodes.length > 0) {
                        const unionOrIntersectionTypeNode = type.flags & TypeFlags.Union ? factory.createUnionTypeNode(typeNodes) : factory.createIntersectionTypeNode(typeNodes);
                        return unionOrIntersectionTypeNode;
                    }
                    else {
                        if (!context.encounteredError && !(context.flags & NodeBuilderFlags.AllowEmptyUnionOrIntersection)) {
                            context.encounteredError = true;
                        }
                        return undefined!; // TODO: GH#18217
                    }
                }
                if (objectFlags & (ObjectFlags.Anonymous | ObjectFlags.Mapped)) {
                    Debug.assert(!!(type.flags & TypeFlags.Object));
                    // The type is an object literal type.
                    return createAnonymousTypeNode(<ObjectType>type);
                }
                if (type.flags & TypeFlags.Index) {
                    const indexedType = (<IndexType>type).type;
                    context.approximateLength += 6;
                    const indexTypeNode = typeToTypeNodeHelper(indexedType, context);
                    return factory.createTypeOperatorNode(SyntaxKind.KeyOfKeyword, indexTypeNode);
                }
                if (type.flags & TypeFlags.IndexedAccess) {
                    const objectTypeNode = typeToTypeNodeHelper((<IndexedAccessType>type).objectType, context);
                    const indexTypeNode = typeToTypeNodeHelper((<IndexedAccessType>type).indexType, context);
                    context.approximateLength += 2;
                    return factory.createIndexedAccessTypeNode(objectTypeNode, indexTypeNode);
                }
                if (type.flags & TypeFlags.Conditional) {
                    const checkTypeNode = typeToTypeNodeHelper((<ConditionalType>type).checkType, context);
                    const saveInferTypeParameters = context.inferTypeParameters;
                    context.inferTypeParameters = (<ConditionalType>type).root.inferTypeParameters;
                    const extendsTypeNode = typeToTypeNodeHelper((<ConditionalType>type).extendsType, context);
                    context.inferTypeParameters = saveInferTypeParameters;
                    const trueTypeNode = typeToTypeNodeOrCircularityElision(getTrueTypeFromConditionalType(<ConditionalType>type));
                    const falseTypeNode = typeToTypeNodeOrCircularityElision(getFalseTypeFromConditionalType(<ConditionalType>type));
                    context.approximateLength += 15;
                    return factory.createConditionalTypeNode(checkTypeNode, extendsTypeNode, trueTypeNode, falseTypeNode);
                }
                if (type.flags & TypeFlags.Substitution) {
                    return typeToTypeNodeHelper((<SubstitutionType>type).baseType, context);
                }

                return Debug.fail("Should be unreachable.");


                function typeToTypeNodeOrCircularityElision(type: Type) {
                    if (type.flags & TypeFlags.Union) {
                        if (context.visitedTypes?.has(getTypeId(type))) {
                            if (!(context.flags & NodeBuilderFlags.AllowAnonymousIdentifier)) {
                                context.encounteredError = true;
                                context.tracker?.reportCyclicStructureError?.();
                            }
                            return createElidedInformationPlaceholder(context);
                        }
                        return visitAndTransformType(type, type => typeToTypeNodeHelper(type, context));
                    }
                    return typeToTypeNodeHelper(type, context);
                }

                function createMappedTypeNodeFromType(type: MappedType) {
                    Debug.assert(!!(type.flags & TypeFlags.Object));
                    const readonlyToken = type.declaration.readonlyToken ? <ReadonlyToken | PlusToken | MinusToken>factory.createToken(type.declaration.readonlyToken.kind) : undefined;
                    const questionToken = type.declaration.questionToken ? <QuestionToken | PlusToken | MinusToken>factory.createToken(type.declaration.questionToken.kind) : undefined;
                    let appropriateConstraintTypeNode: TypeNode;
                    if (isMappedTypeWithKeyofConstraintDeclaration(type)) {
                        // We have a { [P in keyof T]: X }
                        // We do this to ensure we retain the toplevel keyof-ness of the type which may be lost due to keyof distribution during `getConstraintTypeFromMappedType`
                        appropriateConstraintTypeNode = factory.createTypeOperatorNode(SyntaxKind.KeyOfKeyword, typeToTypeNodeHelper(getModifiersTypeFromMappedType(type), context));
                    }
                    else {
                        appropriateConstraintTypeNode = typeToTypeNodeHelper(getConstraintTypeFromMappedType(type), context);
                    }
                    const typeParameterNode = typeParameterToDeclarationWithConstraint(getTypeParameterFromMappedType(type), context, appropriateConstraintTypeNode);
                    const templateTypeNode = typeToTypeNodeHelper(getTemplateTypeFromMappedType(type), context);
                    const mappedTypeNode = factory.createMappedTypeNode(readonlyToken, typeParameterNode, questionToken, templateTypeNode);
                    context.approximateLength += 10;
                    return setEmitFlags(mappedTypeNode, EmitFlags.SingleLine);
                }

                function createAnonymousTypeNode(type: ObjectType): TypeNode {
                    const typeId = type.id;
                    const symbol = type.symbol;
                    if (symbol) {
                        if (isJSConstructor(symbol.valueDeclaration)) {
                            // Instance and static types share the same symbol; only add 'typeof' for the static side.
                            const isInstanceType = type === getDeclaredTypeOfClassOrInterface(symbol) ? SymbolFlags.Type : SymbolFlags.Value;
                            return symbolToTypeNode(symbol, context, isInstanceType);
                        }
                        // Always use 'typeof T' for type of class, enum, and module objects
                        else if (symbol.flags & SymbolFlags.Class && !getBaseTypeVariableOfClass(symbol) && !(symbol.valueDeclaration.kind === SyntaxKind.ClassExpression && context.flags & NodeBuilderFlags.WriteClassExpressionAsTypeLiteral) ||
                            symbol.flags & (SymbolFlags.Enum | SymbolFlags.ValueModule) ||
                            shouldWriteTypeOfFunctionSymbol()) {
                            return symbolToTypeNode(symbol, context, SymbolFlags.Value);
                        }
                        else if (context.visitedTypes?.has(typeId)) {
                            // If type is an anonymous type literal in a type alias declaration, use type alias name
                            const typeAlias = getTypeAliasForTypeLiteral(type);
                            if (typeAlias) {
                                // The specified symbol flags need to be reinterpreted as type flags
                                return symbolToTypeNode(typeAlias, context, SymbolFlags.Type);
                            }
                            else {
                                return createElidedInformationPlaceholder(context);
                            }
                        }
                        else {
                            return visitAndTransformType(type, createTypeNodeFromObjectType);
                        }
                    }
                    else {
                        // Anonymous types without a symbol are never circular.
                        return createTypeNodeFromObjectType(type);
                    }
                    function shouldWriteTypeOfFunctionSymbol() {
                        const isStaticMethodSymbol = !!(symbol.flags & SymbolFlags.Method) &&  // typeof static method
                            some(symbol.declarations, declaration => hasSyntacticModifier(declaration, ModifierFlags.Static));
                        const isNonLocalFunctionSymbol = !!(symbol.flags & SymbolFlags.Function) &&
                            (symbol.parent || // is exported function symbol
                                forEach(symbol.declarations, declaration =>
                                    declaration.parent.kind === SyntaxKind.SourceFile || declaration.parent.kind === SyntaxKind.ModuleBlock));
                        if (isStaticMethodSymbol || isNonLocalFunctionSymbol) {
                            // typeof is allowed only for static/non local functions
                            return (!!(context.flags & NodeBuilderFlags.UseTypeOfFunction) || (context.visitedTypes?.has(typeId))) && // it is type of the symbol uses itself recursively
                                (!(context.flags & NodeBuilderFlags.UseStructuralFallback) || isValueSymbolAccessible(symbol, context.enclosingDeclaration)); // And the build is going to succeed without visibility error or there is no structural fallback allowed
                        }
                    }
                }

                function visitAndTransformType<T>(type: Type, transform: (type: Type) => T) {
                    const typeId = type.id;
                    const isConstructorObject = getObjectFlags(type) & ObjectFlags.Anonymous && type.symbol && type.symbol.flags & SymbolFlags.Class;
                    const id = getObjectFlags(type) & ObjectFlags.Reference && (<TypeReference>type).node ? "N" + getNodeId((<TypeReference>type).node!) :
                        type.symbol ? (isConstructorObject ? "+" : "") + getSymbolId(type.symbol) :
                        undefined;
                    // Since instantiations of the same anonymous type have the same symbol, tracking symbols instead
                    // of types allows us to catch circular references to instantiations of the same anonymous type
                    if (!context.visitedTypes) {
                        context.visitedTypes = new Set();
                    }
                    if (id && !context.symbolDepth) {
                        context.symbolDepth = new Map();
                    }

                    let depth: number | undefined;
                    if (id) {
                        depth = context.symbolDepth!.get(id) || 0;
                        if (depth > 10) {
                            return createElidedInformationPlaceholder(context);
                        }
                        context.symbolDepth!.set(id, depth + 1);
                    }
                    context.visitedTypes.add(typeId);
                    const result = transform(type);
                    context.visitedTypes.delete(typeId);
                    if (id) {
                        context.symbolDepth!.set(id, depth!);
                    }
                    return result;
                }

                function createTypeNodeFromObjectType(type: ObjectType): TypeNode {
                    if (isGenericMappedType(type) || (type as MappedType).containsError) {
                        return createMappedTypeNodeFromType(type as MappedType);
                    }

                    const resolved = resolveStructuredTypeMembers(type);
                    if (!resolved.properties.length && !resolved.stringIndexInfo && !resolved.numberIndexInfo) {
                        if (!resolved.callSignatures.length && !resolved.constructSignatures.length) {
                            context.approximateLength += 2;
                            return setEmitFlags(factory.createTypeLiteralNode(/*members*/ undefined), EmitFlags.SingleLine);
                        }

                        if (resolved.callSignatures.length === 1 && !resolved.constructSignatures.length) {
                            const signature = resolved.callSignatures[0];
                            const signatureNode = <FunctionTypeNode>signatureToSignatureDeclarationHelper(signature, SyntaxKind.FunctionType, context);
                            return signatureNode;

                        }

                        if (resolved.constructSignatures.length === 1 && !resolved.callSignatures.length) {
                            const signature = resolved.constructSignatures[0];
                            const signatureNode = <ConstructorTypeNode>signatureToSignatureDeclarationHelper(signature, SyntaxKind.ConstructorType, context);
                            return signatureNode;
                        }
                    }

                    const savedFlags = context.flags;
                    context.flags |= NodeBuilderFlags.InObjectTypeLiteral;
                    const members = createTypeNodesFromResolvedType(resolved);
                    context.flags = savedFlags;
                    const typeLiteralNode = factory.createTypeLiteralNode(members);
                    context.approximateLength += 2;
                    return setEmitFlags(typeLiteralNode, (context.flags & NodeBuilderFlags.MultilineObjectLiterals) ? 0 : EmitFlags.SingleLine);
                }

                function typeReferenceToTypeNode(type: TypeReference) {
                    const typeArguments: readonly Type[] = getTypeArguments(type);
                    if (type.target === globalArrayType || type.target === globalReadonlyArrayType) {
                        if (context.flags & NodeBuilderFlags.WriteArrayAsGenericType) {
                            const typeArgumentNode = typeToTypeNodeHelper(typeArguments[0], context);
                            return factory.createTypeReferenceNode(type.target === globalArrayType ? "Array" : "ReadonlyArray", [typeArgumentNode]);
                        }
                        const elementType = typeToTypeNodeHelper(typeArguments[0], context);
                        const arrayType = factory.createArrayTypeNode(elementType);
                        return type.target === globalArrayType ? arrayType : factory.createTypeOperatorNode(SyntaxKind.ReadonlyKeyword, arrayType);
                    }
                    else if (type.target.objectFlags & ObjectFlags.Tuple) {
                        if (typeArguments.length > 0) {
                            const arity = getTypeReferenceArity(type);
                            const tupleConstituentNodes = mapToTypeNodes(typeArguments.slice(0, arity), context);
                            if (tupleConstituentNodes) {
                                if ((type.target as TupleType).labeledElementDeclarations) {
                                    for (let i = 0; i < tupleConstituentNodes.length; i++) {
                                        const flags = (type.target as TupleType).elementFlags[i];
                                        tupleConstituentNodes[i] = factory.createNamedTupleMember(
                                            flags & ElementFlags.Variable ? factory.createToken(SyntaxKind.DotDotDotToken) : undefined,
                                            factory.createIdentifier(unescapeLeadingUnderscores(getTupleElementLabel((type.target as TupleType).labeledElementDeclarations![i]))),
                                            flags & ElementFlags.Optional ? factory.createToken(SyntaxKind.QuestionToken) : undefined,
                                            flags & ElementFlags.Rest ? factory.createArrayTypeNode(tupleConstituentNodes[i]) :
                                            tupleConstituentNodes[i]
                                        );
                                    }
                                }
                                else {
                                    for (let i = 0; i < Math.min(arity, tupleConstituentNodes.length); i++) {
                                        const flags = (type.target as TupleType).elementFlags[i];
                                        tupleConstituentNodes[i] =
                                            flags & ElementFlags.Variable ? factory.createRestTypeNode(flags & ElementFlags.Rest ? factory.createArrayTypeNode(tupleConstituentNodes[i]) : tupleConstituentNodes[i]) :
                                            flags & ElementFlags.Optional ? factory.createOptionalTypeNode(tupleConstituentNodes[i]) :
                                            tupleConstituentNodes[i];
                                    }
                                }
                                const tupleTypeNode = setEmitFlags(factory.createTupleTypeNode(tupleConstituentNodes), EmitFlags.SingleLine);
                                return (<TupleType>type.target).readonly ? factory.createTypeOperatorNode(SyntaxKind.ReadonlyKeyword, tupleTypeNode) : tupleTypeNode;
                            }
                        }
                        if (context.encounteredError || (context.flags & NodeBuilderFlags.AllowEmptyTuple)) {
                            const tupleTypeNode = setEmitFlags(factory.createTupleTypeNode([]), EmitFlags.SingleLine);
                            return (<TupleType>type.target).readonly ? factory.createTypeOperatorNode(SyntaxKind.ReadonlyKeyword, tupleTypeNode) : tupleTypeNode;
                        }
                        context.encounteredError = true;
                        return undefined!; // TODO: GH#18217
                    }
                    else if (context.flags & NodeBuilderFlags.WriteClassExpressionAsTypeLiteral &&
                        type.symbol.valueDeclaration &&
                        isClassLike(type.symbol.valueDeclaration) &&
                        !isValueSymbolAccessible(type.symbol, context.enclosingDeclaration)
                    ) {
                        return createAnonymousTypeNode(type);
                    }
                    else {
                        const outerTypeParameters = type.target.outerTypeParameters;
                        let i = 0;
                        let resultType: TypeReferenceNode | ImportTypeNode | undefined;
                        if (outerTypeParameters) {
                            const length = outerTypeParameters.length;
                            while (i < length) {
                                // Find group of type arguments for type parameters with the same declaring container.
                                const start = i;
                                const parent = getParentSymbolOfTypeParameter(outerTypeParameters[i])!;
                                do {
                                    i++;
                                } while (i < length && getParentSymbolOfTypeParameter(outerTypeParameters[i]) === parent);
                                // When type parameters are their own type arguments for the whole group (i.e. we have
                                // the default outer type arguments), we don't show the group.
                                if (!rangeEquals(outerTypeParameters, typeArguments, start, i)) {
                                    const typeArgumentSlice = mapToTypeNodes(typeArguments.slice(start, i), context);
                                    const flags = context.flags;
                                    context.flags |= NodeBuilderFlags.ForbidIndexedAccessSymbolReferences;
                                    const ref = symbolToTypeNode(parent, context, SymbolFlags.Type, typeArgumentSlice) as TypeReferenceNode | ImportTypeNode;
                                    context.flags = flags;
                                    resultType = !resultType ? ref : appendReferenceToType(resultType, ref as TypeReferenceNode);
                                }
                            }
                        }
                        let typeArgumentNodes: readonly TypeNode[] | undefined;
                        if (typeArguments.length > 0) {
                            const typeParameterCount = (type.target.typeParameters || emptyArray).length;
                            typeArgumentNodes = mapToTypeNodes(typeArguments.slice(i, typeParameterCount), context);
                        }
                        const flags = context.flags;
                        context.flags |= NodeBuilderFlags.ForbidIndexedAccessSymbolReferences;
                        const finalRef = symbolToTypeNode(type.symbol, context, SymbolFlags.Type, typeArgumentNodes);
                        context.flags = flags;
                        return !resultType ? finalRef : appendReferenceToType(resultType, finalRef as TypeReferenceNode);
                    }
                }


                function appendReferenceToType(root: TypeReferenceNode | ImportTypeNode, ref: TypeReferenceNode): TypeReferenceNode | ImportTypeNode {
                    if (isImportTypeNode(root)) {
                        // first shift type arguments
                        let typeArguments = root.typeArguments;
                        let qualifier = root.qualifier;
                        if (qualifier) {
                            if (isIdentifier(qualifier)) {
                                qualifier = factory.updateIdentifier(qualifier, typeArguments);
                            }
                            else {
                                qualifier = factory.updateQualifiedName(qualifier,
                                    qualifier.left,
                                    factory.updateIdentifier(qualifier.right, typeArguments));
                            }
                        }
                        typeArguments = ref.typeArguments;
                        // then move qualifiers
                        const ids = getAccessStack(ref);
                        for (const id of ids) {
                            qualifier = qualifier ? factory.createQualifiedName(qualifier, id) : id;
                        }
                        return factory.updateImportTypeNode(
                            root,
                            root.argument,
                            qualifier,
                            typeArguments,
                            root.isTypeOf);
                    }
                    else {
                        // first shift type arguments
                        let typeArguments = root.typeArguments;
                        let typeName = root.typeName;
                        if (isIdentifier(typeName)) {
                            typeName = factory.updateIdentifier(typeName, typeArguments);
                        }
                        else {
                            typeName = factory.updateQualifiedName(typeName,
                                typeName.left,
                                factory.updateIdentifier(typeName.right, typeArguments));
                        }
                        typeArguments = ref.typeArguments;
                        // then move qualifiers
                        const ids = getAccessStack(ref);
                        for (const id of ids) {
                            typeName = factory.createQualifiedName(typeName, id);
                        }
                        return factory.updateTypeReferenceNode(
                            root,
                            typeName,
                            typeArguments);
                    }
                }

                function getAccessStack(ref: TypeReferenceNode): Identifier[] {
                    let state = ref.typeName;
                    const ids = [];
                    while (!isIdentifier(state)) {
                        ids.unshift(state.right);
                        state = state.left;
                    }
                    ids.unshift(state);
                    return ids;
                }

                function createTypeNodesFromResolvedType(resolvedType: ResolvedType): TypeElement[] | undefined {
                    if (checkTruncationLength(context)) {
                        return [factory.createPropertySignature(/*modifiers*/ undefined, "...", /*questionToken*/ undefined, /*type*/ undefined)];
                    }
                    const typeElements: TypeElement[] = [];
                    for (const signature of resolvedType.callSignatures) {
                        typeElements.push(<CallSignatureDeclaration>signatureToSignatureDeclarationHelper(signature, SyntaxKind.CallSignature, context));
                    }
                    for (const signature of resolvedType.constructSignatures) {
                        typeElements.push(<ConstructSignatureDeclaration>signatureToSignatureDeclarationHelper(signature, SyntaxKind.ConstructSignature, context));
                    }
                    if (resolvedType.stringIndexInfo) {
                        let indexSignature: IndexSignatureDeclaration;
                        if (resolvedType.objectFlags & ObjectFlags.ReverseMapped) {
                            indexSignature = indexInfoToIndexSignatureDeclarationHelper(
                                createIndexInfo(anyType, resolvedType.stringIndexInfo.isReadonly, resolvedType.stringIndexInfo.declaration),
                                IndexKind.String,
                                context,
                                createElidedInformationPlaceholder(context));
                        }
                        else {
                            indexSignature = indexInfoToIndexSignatureDeclarationHelper(resolvedType.stringIndexInfo, IndexKind.String, context, /*typeNode*/ undefined);
                        }
                        typeElements.push(indexSignature);
                    }
                    if (resolvedType.numberIndexInfo) {
                        typeElements.push(indexInfoToIndexSignatureDeclarationHelper(resolvedType.numberIndexInfo, IndexKind.Number, context, /*typeNode*/ undefined));
                    }

                    const properties = resolvedType.properties;
                    if (!properties) {
                        return typeElements;
                    }

                    let i = 0;
                    for (const propertySymbol of properties) {
                        i++;
                        if (context.flags & NodeBuilderFlags.WriteClassExpressionAsTypeLiteral) {
                            if (propertySymbol.flags & SymbolFlags.Prototype) {
                                continue;
                            }
                            if (getDeclarationModifierFlagsFromSymbol(propertySymbol) & (ModifierFlags.Private | ModifierFlags.Protected) && context.tracker.reportPrivateInBaseOfClassExpression) {
                                context.tracker.reportPrivateInBaseOfClassExpression(unescapeLeadingUnderscores(propertySymbol.escapedName));
                            }
                        }
                        if (checkTruncationLength(context) && (i + 2 < properties.length - 1)) {
                            typeElements.push(factory.createPropertySignature(/*modifiers*/ undefined, `... ${properties.length - i} more ...`, /*questionToken*/ undefined, /*type*/ undefined));
                            addPropertyToElementList(properties[properties.length - 1], context, typeElements);
                            break;
                        }
                        addPropertyToElementList(propertySymbol, context, typeElements);

                    }
                    return typeElements.length ? typeElements : undefined;
                }
            }

            function createElidedInformationPlaceholder(context: NodeBuilderContext) {
                context.approximateLength += 3;
                if (!(context.flags & NodeBuilderFlags.NoTruncation)) {
                    return factory.createTypeReferenceNode(factory.createIdentifier("..."), /*typeArguments*/ undefined);
                }
                return factory.createKeywordTypeNode(SyntaxKind.AnyKeyword);
            }

            function addPropertyToElementList(propertySymbol: Symbol, context: NodeBuilderContext, typeElements: TypeElement[]) {
                const propertyIsReverseMapped = !!(getCheckFlags(propertySymbol) & CheckFlags.ReverseMapped);
                const propertyType = propertyIsReverseMapped && context.flags & NodeBuilderFlags.InReverseMappedType ?
                    anyType : getTypeOfSymbol(propertySymbol);
                const saveEnclosingDeclaration = context.enclosingDeclaration;
                context.enclosingDeclaration = undefined;
                if (context.tracker.trackSymbol && getCheckFlags(propertySymbol) & CheckFlags.Late) {
                    const decl = first(propertySymbol.declarations);
                    if (hasLateBindableName(decl)) {
                        if (isBinaryExpression(decl)) {
                            const name = getNameOfDeclaration(decl);
                            if (name && isElementAccessExpression(name) && isPropertyAccessEntityNameExpression(name.argumentExpression)) {
                                trackComputedName(name.argumentExpression, saveEnclosingDeclaration, context);
                            }
                        }
                        else {
                            trackComputedName(decl.name.expression, saveEnclosingDeclaration, context);
                        }
                    }
                }
                context.enclosingDeclaration = saveEnclosingDeclaration;
                const propertyName = getPropertyNameNodeForSymbol(propertySymbol, context);
                context.approximateLength += (symbolName(propertySymbol).length + 1);
                const optionalToken = propertySymbol.flags & SymbolFlags.Optional ? factory.createToken(SyntaxKind.QuestionToken) : undefined;
                if (propertySymbol.flags & (SymbolFlags.Function | SymbolFlags.Method) && !getPropertiesOfObjectType(propertyType).length && !isReadonlySymbol(propertySymbol)) {
                    const signatures = getSignaturesOfType(filterType(propertyType, t => !(t.flags & TypeFlags.Undefined)), SignatureKind.Call);
                    for (const signature of signatures) {
                        const methodDeclaration = <MethodSignature>signatureToSignatureDeclarationHelper(signature, SyntaxKind.MethodSignature, context, { name: propertyName, questionToken: optionalToken });
                        typeElements.push(preserveCommentsOn(methodDeclaration));
                    }
                }
                else {
                    const savedFlags = context.flags;
                    context.flags |= propertyIsReverseMapped ? NodeBuilderFlags.InReverseMappedType : 0;
                    let propertyTypeNode: TypeNode;
                    if (propertyIsReverseMapped && !!(savedFlags & NodeBuilderFlags.InReverseMappedType)) {
                        propertyTypeNode = createElidedInformationPlaceholder(context);
                    }
                    else {
                        propertyTypeNode = propertyType ? serializeTypeForDeclaration(context, propertyType, propertySymbol, saveEnclosingDeclaration) : factory.createKeywordTypeNode(SyntaxKind.AnyKeyword);
                    }
                    context.flags = savedFlags;

                    const modifiers = isReadonlySymbol(propertySymbol) ? [factory.createToken(SyntaxKind.ReadonlyKeyword)] : undefined;
                    if (modifiers) {
                        context.approximateLength += 9;
                    }
                    const propertySignature = factory.createPropertySignature(
                        modifiers,
                        propertyName,
                        optionalToken,
                        propertyTypeNode);

                    typeElements.push(preserveCommentsOn(propertySignature));
                }

                function preserveCommentsOn<T extends Node>(node: T) {
                    if (some(propertySymbol.declarations, d => d.kind === SyntaxKind.JSDocPropertyTag)) {
                        const d = find(propertySymbol.declarations, d => d.kind === SyntaxKind.JSDocPropertyTag)! as JSDocPropertyTag;
                        const commentText = d.comment;
                        if (commentText) {
                            setSyntheticLeadingComments(node, [{ kind: SyntaxKind.MultiLineCommentTrivia, text: "*\n * " + commentText.replace(/\n/g, "\n * ") + "\n ", pos: -1, end: -1, hasTrailingNewLine: true }]);
                        }
                    }
                    else if (propertySymbol.valueDeclaration) {
                        // Copy comments to node for declaration emit
                        setCommentRange(node, propertySymbol.valueDeclaration);
                    }
                    return node;
                }
            }

            function mapToTypeNodes(types: readonly Type[] | undefined, context: NodeBuilderContext, isBareList?: boolean): TypeNode[] | undefined {
                if (some(types)) {
                    if (checkTruncationLength(context)) {
                        if (!isBareList) {
                            return [factory.createTypeReferenceNode("...", /*typeArguments*/ undefined)];
                        }
                        else if (types.length > 2) {
                            return [
                                typeToTypeNodeHelper(types[0], context),
                                factory.createTypeReferenceNode(`... ${types.length - 2} more ...`, /*typeArguments*/ undefined),
                                typeToTypeNodeHelper(types[types.length - 1], context)
                            ];
                        }
                    }
                    const mayHaveNameCollisions = !(context.flags & NodeBuilderFlags.UseFullyQualifiedType);
                    /** Map from type reference identifier text to [type, index in `result` where the type node is] */
                    const seenNames = mayHaveNameCollisions ? createUnderscoreEscapedMultiMap<[Type, number]>() : undefined;
                    const result: TypeNode[] = [];
                    let i = 0;
                    for (const type of types) {
                        i++;
                        if (checkTruncationLength(context) && (i + 2 < types.length - 1)) {
                            result.push(factory.createTypeReferenceNode(`... ${types.length - i} more ...`, /*typeArguments*/ undefined));
                            const typeNode = typeToTypeNodeHelper(types[types.length - 1], context);
                            if (typeNode) {
                                result.push(typeNode);
                            }
                            break;
                        }
                        context.approximateLength += 2; // Account for whitespace + separator
                        const typeNode = typeToTypeNodeHelper(type, context);
                        if (typeNode) {
                            result.push(typeNode);
                            if (seenNames && isIdentifierTypeReference(typeNode)) {
                                seenNames.add(typeNode.typeName.escapedText, [type, result.length - 1]);
                            }
                        }
                    }

                    if (seenNames) {
                        // To avoid printing types like `[Foo, Foo]` or `Bar & Bar` where
                        // occurrences of the same name actually come from different
                        // namespaces, go through the single-identifier type reference nodes
                        // we just generated, and see if any names were generated more than
                        // once while referring to different types. If so, regenerate the
                        // type node for each entry by that name with the
                        // `UseFullyQualifiedType` flag enabled.
                        const saveContextFlags = context.flags;
                        context.flags |= NodeBuilderFlags.UseFullyQualifiedType;
                        seenNames.forEach(types => {
                            if (!arrayIsHomogeneous(types, ([a], [b]) => typesAreSameReference(a, b))) {
                                for (const [type, resultIndex] of types) {
                                    result[resultIndex] = typeToTypeNodeHelper(type, context);
                                }
                            }
                        });
                        context.flags = saveContextFlags;
                    }

                    return result;
                }
            }

            function typesAreSameReference(a: Type, b: Type): boolean {
                return a === b
                    || !!a.symbol && a.symbol === b.symbol
                    || !!a.aliasSymbol && a.aliasSymbol === b.aliasSymbol;
            }

            function indexInfoToIndexSignatureDeclarationHelper(indexInfo: IndexInfo, kind: IndexKind, context: NodeBuilderContext, typeNode: TypeNode | undefined): IndexSignatureDeclaration {
                const name = getNameFromIndexInfo(indexInfo) || "x";
                const indexerTypeNode = factory.createKeywordTypeNode(kind === IndexKind.String ? SyntaxKind.StringKeyword : SyntaxKind.NumberKeyword);

                const indexingParameter = factory.createParameterDeclaration(
                    /*decorators*/ undefined,
                    /*modifiers*/ undefined,
                    /*dotDotDotToken*/ undefined,
                    name,
                    /*questionToken*/ undefined,
                    indexerTypeNode,
                    /*initializer*/ undefined);
                if (!typeNode) {
                    typeNode = typeToTypeNodeHelper(indexInfo.type || anyType, context);
                }
                if (!indexInfo.type && !(context.flags & NodeBuilderFlags.AllowEmptyIndexInfoType)) {
                    context.encounteredError = true;
                }
                context.approximateLength += (name.length + 4);
                return factory.createIndexSignature(
                    /*decorators*/ undefined,
                    indexInfo.isReadonly ? [factory.createToken(SyntaxKind.ReadonlyKeyword)] : undefined,
                    [indexingParameter],
                    typeNode);
            }

            interface SignatureToSignatureDeclarationOptions {
                modifiers?: readonly Modifier[];
                name?: PropertyName;
                questionToken?: QuestionToken;
                privateSymbolVisitor?: (s: Symbol) => void;
                bundledImports?: boolean;
            }

            function signatureToSignatureDeclarationHelper(signature: Signature, kind: SignatureDeclaration["kind"], context: NodeBuilderContext, options?: SignatureToSignatureDeclarationOptions): SignatureDeclaration {
                const suppressAny = context.flags & NodeBuilderFlags.SuppressAnyReturnType;
                if (suppressAny) context.flags &= ~NodeBuilderFlags.SuppressAnyReturnType; // suppress only toplevel `any`s
                let typeParameters: TypeParameterDeclaration[] | undefined;
                let typeArguments: TypeNode[] | undefined;
                if (context.flags & NodeBuilderFlags.WriteTypeArgumentsOfSignature && signature.target && signature.mapper && signature.target.typeParameters) {
                    typeArguments = signature.target.typeParameters.map(parameter => typeToTypeNodeHelper(instantiateType(parameter, signature.mapper), context));
                }
                else {
                    typeParameters = signature.typeParameters && signature.typeParameters.map(parameter => typeParameterToDeclaration(parameter, context));
                }

                const parameters = getExpandedParameters(signature, /*skipUnionExpanding*/ true)[0].map(parameter => symbolToParameterDeclaration(parameter, context, kind === SyntaxKind.Constructor, options?.privateSymbolVisitor, options?.bundledImports));
                if (signature.thisParameter) {
                    const thisParameter = symbolToParameterDeclaration(signature.thisParameter, context);
                    parameters.unshift(thisParameter);
                }

                let returnTypeNode: TypeNode | undefined;
                const typePredicate = getTypePredicateOfSignature(signature);
                if (typePredicate) {
                    const assertsModifier = typePredicate.kind === TypePredicateKind.AssertsThis || typePredicate.kind === TypePredicateKind.AssertsIdentifier ?
                        factory.createToken(SyntaxKind.AssertsKeyword) :
                        undefined;
                    const parameterName = typePredicate.kind === TypePredicateKind.Identifier || typePredicate.kind === TypePredicateKind.AssertsIdentifier ?
                        setEmitFlags(factory.createIdentifier(typePredicate.parameterName), EmitFlags.NoAsciiEscaping) :
                        factory.createThisTypeNode();
                    const typeNode = typePredicate.type && typeToTypeNodeHelper(typePredicate.type, context);
                    returnTypeNode = factory.createTypePredicateNode(assertsModifier, parameterName, typeNode);
                }
                else {
                    const returnType = getReturnTypeOfSignature(signature);
                    if (returnType && !(suppressAny && isTypeAny(returnType))) {
                        returnTypeNode = serializeReturnTypeForSignature(context, returnType, signature, options?.privateSymbolVisitor, options?.bundledImports);
                    }
                    else if (!suppressAny) {
                        returnTypeNode = factory.createKeywordTypeNode(SyntaxKind.AnyKeyword);
                    }
                }
                context.approximateLength += 3; // Usually a signature contributes a few more characters than this, but 3 is the minimum

                const node =
                    kind === SyntaxKind.CallSignature ? factory.createCallSignature(typeParameters, parameters, returnTypeNode) :
                    kind === SyntaxKind.ConstructSignature ? factory.createConstructSignature(typeParameters, parameters, returnTypeNode) :
                    kind === SyntaxKind.MethodSignature ? factory.createMethodSignature(options?.modifiers, options?.name ?? factory.createIdentifier(""), options?.questionToken, typeParameters, parameters, returnTypeNode) :
                    kind === SyntaxKind.MethodDeclaration ? factory.createMethodDeclaration(/*decorators*/ undefined, options?.modifiers, /*asteriskToken*/ undefined, options?.name ?? factory.createIdentifier(""), /*questionToken*/ undefined, typeParameters, parameters, returnTypeNode, /*body*/ undefined) :
                    kind === SyntaxKind.Constructor ? factory.createConstructorDeclaration(/*decorators*/ undefined, options?.modifiers, parameters, /*body*/ undefined) :
                    kind === SyntaxKind.GetAccessor ? factory.createGetAccessorDeclaration(/*decorators*/ undefined, options?.modifiers, options?.name ?? factory.createIdentifier(""), parameters, returnTypeNode, /*body*/ undefined) :
                    kind === SyntaxKind.SetAccessor ? factory.createSetAccessorDeclaration(/*decorators*/ undefined, options?.modifiers, options?.name ?? factory.createIdentifier(""), parameters, /*body*/ undefined) :
                    kind === SyntaxKind.IndexSignature ? factory.createIndexSignature(/*decorators*/ undefined, options?.modifiers, parameters, returnTypeNode) :
                    kind === SyntaxKind.JSDocFunctionType ? factory.createJSDocFunctionType(parameters, returnTypeNode) :
                    kind === SyntaxKind.FunctionType ? factory.createFunctionTypeNode(typeParameters, parameters, returnTypeNode ?? factory.createTypeReferenceNode(factory.createIdentifier(""))) :
                    kind === SyntaxKind.ConstructorType ? factory.createConstructorTypeNode(typeParameters, parameters, returnTypeNode ?? factory.createTypeReferenceNode(factory.createIdentifier(""))) :
                    kind === SyntaxKind.FunctionDeclaration ? factory.createFunctionDeclaration(/*decorators*/ undefined, options?.modifiers, /*asteriskToken*/ undefined, options?.name ? cast(options.name, isIdentifier) : factory.createIdentifier(""), typeParameters, parameters, returnTypeNode, /*body*/ undefined) :
                    kind === SyntaxKind.FunctionExpression ? factory.createFunctionExpression(options?.modifiers, /*asteriskToken*/ undefined, options?.name ? cast(options.name, isIdentifier) : factory.createIdentifier(""), typeParameters, parameters, returnTypeNode, factory.createBlock([])) :
                    kind === SyntaxKind.ArrowFunction ? factory.createArrowFunction(options?.modifiers, typeParameters, parameters, returnTypeNode, /*equalsGreaterThanToken*/ undefined, factory.createBlock([])) :
                    Debug.assertNever(kind);

                if (typeArguments) {
                    node.typeArguments = factory.createNodeArray(typeArguments);
                }

                return node;
            }

            function typeParameterToDeclarationWithConstraint(type: TypeParameter, context: NodeBuilderContext, constraintNode: TypeNode | undefined): TypeParameterDeclaration {
                const savedContextFlags = context.flags;
                context.flags &= ~NodeBuilderFlags.WriteTypeParametersInQualifiedName; // Avoids potential infinite loop when building for a claimspace with a generic
                const name = typeParameterToName(type, context);
                const defaultParameter = getDefaultFromTypeParameter(type);
                const defaultParameterNode = defaultParameter && typeToTypeNodeHelper(defaultParameter, context);
                context.flags = savedContextFlags;
                return factory.createTypeParameterDeclaration(name, constraintNode, defaultParameterNode);
            }

            function typeParameterToDeclaration(type: TypeParameter, context: NodeBuilderContext, constraint = getConstraintOfTypeParameter(type)): TypeParameterDeclaration {
                const constraintNode = constraint && typeToTypeNodeHelper(constraint, context);
                return typeParameterToDeclarationWithConstraint(type, context, constraintNode);
            }

            function symbolToParameterDeclaration(parameterSymbol: Symbol, context: NodeBuilderContext, preserveModifierFlags?: boolean, privateSymbolVisitor?: (s: Symbol) => void, bundledImports?: boolean): ParameterDeclaration {
                let parameterDeclaration: ParameterDeclaration | JSDocParameterTag | undefined = getDeclarationOfKind<ParameterDeclaration>(parameterSymbol, SyntaxKind.Parameter);
                if (!parameterDeclaration && !isTransientSymbol(parameterSymbol)) {
                    parameterDeclaration = getDeclarationOfKind<JSDocParameterTag>(parameterSymbol, SyntaxKind.JSDocParameterTag);
                }

                let parameterType = getTypeOfSymbol(parameterSymbol);
                if (parameterDeclaration && isRequiredInitializedParameter(parameterDeclaration)) {
                    parameterType = getOptionalType(parameterType);
                }
                if ((context.flags & NodeBuilderFlags.NoUndefinedOptionalParameterType) && parameterDeclaration && !isJSDocParameterTag(parameterDeclaration) && isOptionalUninitializedParameter(parameterDeclaration)) {
                    parameterType = getTypeWithFacts(parameterType, TypeFacts.NEUndefined);
                }
                const parameterTypeNode = serializeTypeForDeclaration(context, parameterType, parameterSymbol, context.enclosingDeclaration, privateSymbolVisitor, bundledImports);

                const modifiers = !(context.flags & NodeBuilderFlags.OmitParameterModifiers) && preserveModifierFlags && parameterDeclaration && parameterDeclaration.modifiers ? parameterDeclaration.modifiers.map(factory.cloneNode) : undefined;
                const isRest = parameterDeclaration && isRestParameter(parameterDeclaration) || getCheckFlags(parameterSymbol) & CheckFlags.RestParameter;
                const dotDotDotToken = isRest ? factory.createToken(SyntaxKind.DotDotDotToken) : undefined;
                const name = parameterDeclaration ? parameterDeclaration.name ?
                    parameterDeclaration.name.kind === SyntaxKind.Identifier ? setEmitFlags(factory.cloneNode(parameterDeclaration.name), EmitFlags.NoAsciiEscaping) :
                    parameterDeclaration.name.kind === SyntaxKind.QualifiedName ? setEmitFlags(factory.cloneNode(parameterDeclaration.name.right), EmitFlags.NoAsciiEscaping) :
                    cloneBindingName(parameterDeclaration.name) :
                    symbolName(parameterSymbol) :
                    symbolName(parameterSymbol);
                const isOptional = parameterDeclaration && isOptionalParameter(parameterDeclaration) || getCheckFlags(parameterSymbol) & CheckFlags.OptionalParameter;
                const questionToken = isOptional ? factory.createToken(SyntaxKind.QuestionToken) : undefined;
                const parameterNode = factory.createParameterDeclaration(
                    /*decorators*/ undefined,
                    modifiers,
                    dotDotDotToken,
                    name,
                    questionToken,
                    parameterTypeNode,
                    /*initializer*/ undefined);
                context.approximateLength += symbolName(parameterSymbol).length + 3;
                return parameterNode;

                function cloneBindingName(node: BindingName): BindingName {
                    return <BindingName>elideInitializerAndSetEmitFlags(node);
                    function elideInitializerAndSetEmitFlags(node: Node): Node {
                        if (context.tracker.trackSymbol && isComputedPropertyName(node) && isLateBindableName(node)) {
                            trackComputedName(node.expression, context.enclosingDeclaration, context);
                        }
                        let visited = visitEachChild(node, elideInitializerAndSetEmitFlags, nullTransformationContext, /*nodesVisitor*/ undefined, elideInitializerAndSetEmitFlags)!;
                        if (isBindingElement(visited)) {
                            visited = factory.updateBindingElement(
                                visited,
                                visited.dotDotDotToken,
                                visited.propertyName,
                                visited.name,
                                /*initializer*/ undefined);
                        }
                        if (!nodeIsSynthesized(visited)) {
                            visited = factory.cloneNode(visited);
                        }
                        return setEmitFlags(visited, EmitFlags.SingleLine | EmitFlags.NoAsciiEscaping);
                    }
                }
            }

            function trackComputedName(accessExpression: EntityNameOrEntityNameExpression, enclosingDeclaration: Node | undefined, context: NodeBuilderContext) {
                if (!context.tracker.trackSymbol) return;
                // get symbol of the first identifier of the entityName
                const firstIdentifier = getFirstIdentifier(accessExpression);
                const name = resolveName(firstIdentifier, firstIdentifier.escapedText, SymbolFlags.Value | SymbolFlags.ExportValue, /*nodeNotFoundErrorMessage*/ undefined, /*nameArg*/ undefined, /*isUse*/ true);
                if (name) {
                    context.tracker.trackSymbol(name, enclosingDeclaration, SymbolFlags.Value);
                }
            }

            function lookupSymbolChain(symbol: Symbol, context: NodeBuilderContext, meaning: SymbolFlags, yieldModuleSymbol?: boolean) {
                context.tracker.trackSymbol!(symbol, context.enclosingDeclaration, meaning); // TODO: GH#18217
                return lookupSymbolChainWorker(symbol, context, meaning, yieldModuleSymbol);
            }

            function lookupSymbolChainWorker(symbol: Symbol, context: NodeBuilderContext, meaning: SymbolFlags, yieldModuleSymbol?: boolean) {
                // Try to get qualified name if the symbol is not a type parameter and there is an enclosing declaration.
                let chain: Symbol[];
                const isTypeParameter = symbol.flags & SymbolFlags.TypeParameter;
                if (!isTypeParameter && (context.enclosingDeclaration || context.flags & NodeBuilderFlags.UseFullyQualifiedType) && !(context.flags & NodeBuilderFlags.DoNotIncludeSymbolChain)) {
                    chain = Debug.checkDefined(getSymbolChain(symbol, meaning, /*endOfChain*/ true));
                    Debug.assert(chain && chain.length > 0);
                }
                else {
                    chain = [symbol];
                }
                return chain;

                /** @param endOfChain Set to false for recursive calls; non-recursive calls should always output something. */
                function getSymbolChain(symbol: Symbol, meaning: SymbolFlags, endOfChain: boolean): Symbol[] | undefined {
                    let accessibleSymbolChain = getAccessibleSymbolChain(symbol, context.enclosingDeclaration, meaning, !!(context.flags & NodeBuilderFlags.UseOnlyExternalAliasing));
                    let parentSpecifiers: (string | undefined)[];
                    if (!accessibleSymbolChain ||
                        needsQualification(accessibleSymbolChain[0], context.enclosingDeclaration, accessibleSymbolChain.length === 1 ? meaning : getQualifiedLeftMeaning(meaning))) {

                        // Go up and add our parent.
                        const parents = getContainersOfSymbol(accessibleSymbolChain ? accessibleSymbolChain[0] : symbol, context.enclosingDeclaration, meaning);
                        if (length(parents)) {
                            parentSpecifiers = parents!.map(symbol =>
                                some(symbol.declarations, hasNonGlobalAugmentationExternalModuleSymbol)
                                    ? getSpecifierForModuleSymbol(symbol, context)
                                    : undefined);
                            const indices = parents!.map((_, i) => i);
                            indices.sort(sortByBestName);
                            const sortedParents = indices.map(i => parents![i]);
                            for (const parent of sortedParents) {
                                const parentChain = getSymbolChain(parent, getQualifiedLeftMeaning(meaning), /*endOfChain*/ false);
                                if (parentChain) {
                                    if (parent.exports && parent.exports.get(InternalSymbolName.ExportEquals) &&
                                        getSymbolIfSameReference(parent.exports.get(InternalSymbolName.ExportEquals)!, symbol)) {
                                        // parentChain root _is_ symbol - symbol is a module export=, so it kinda looks like it's own parent
                                        // No need to lookup an alias for the symbol in itself
                                        accessibleSymbolChain = parentChain;
                                        break;
                                    }
                                    accessibleSymbolChain = parentChain.concat(accessibleSymbolChain || [getAliasForSymbolInContainer(parent, symbol) || symbol]);
                                    break;
                                }
                            }
                        }
                    }

                    if (accessibleSymbolChain) {
                        return accessibleSymbolChain;
                    }
                    if (
                        // If this is the last part of outputting the symbol, always output. The cases apply only to parent symbols.
                        endOfChain ||
                        // If a parent symbol is an anonymous type, don't write it.
                        !(symbol.flags & (SymbolFlags.TypeLiteral | SymbolFlags.ObjectLiteral))) {
                        // If a parent symbol is an external module, don't write it. (We prefer just `x` vs `"foo/bar".x`.)
                        if (!endOfChain && !yieldModuleSymbol && !!forEach(symbol.declarations, hasNonGlobalAugmentationExternalModuleSymbol)) {
                            return;
                        }
                        return [symbol];
                    }

                    function sortByBestName(a: number, b: number) {
                        const specifierA = parentSpecifiers[a];
                        const specifierB = parentSpecifiers[b];
                        if (specifierA && specifierB) {
                            const isBRelative = pathIsRelative(specifierB);
                            if (pathIsRelative(specifierA) === isBRelative) {
                                // Both relative or both non-relative, sort by number of parts
                                return moduleSpecifiers.countPathComponents(specifierA) - moduleSpecifiers.countPathComponents(specifierB);
                            }
                            if (isBRelative) {
                                // A is non-relative, B is relative: prefer A
                                return -1;
                            }
                            // A is relative, B is non-relative: prefer B
                            return 1;
                        }
                        return 0;
                    }
                }
            }

            function typeParametersToTypeParameterDeclarations(symbol: Symbol, context: NodeBuilderContext) {
                let typeParameterNodes: NodeArray<TypeParameterDeclaration> | undefined;
                const targetSymbol = getTargetSymbol(symbol);
                if (targetSymbol.flags & (SymbolFlags.Class | SymbolFlags.Interface | SymbolFlags.TypeAlias)) {
                    typeParameterNodes = factory.createNodeArray(map(getLocalTypeParametersOfClassOrInterfaceOrTypeAlias(symbol), tp => typeParameterToDeclaration(tp, context)));
                }
                return typeParameterNodes;
            }

            function lookupTypeParameterNodes(chain: Symbol[], index: number, context: NodeBuilderContext) {
                Debug.assert(chain && 0 <= index && index < chain.length);
                const symbol = chain[index];
                const symbolId = getSymbolId(symbol);
                if (context.typeParameterSymbolList?.has(symbolId)) {
                    return undefined;
                }
                (context.typeParameterSymbolList || (context.typeParameterSymbolList = new Set())).add(symbolId);
                let typeParameterNodes: readonly TypeNode[] | readonly TypeParameterDeclaration[] | undefined;
                if (context.flags & NodeBuilderFlags.WriteTypeParametersInQualifiedName && index < (chain.length - 1)) {
                    const parentSymbol = symbol;
                    const nextSymbol = chain[index + 1];
                    if (getCheckFlags(nextSymbol) & CheckFlags.Instantiated) {
                        const params = getTypeParametersOfClassOrInterface(
                            parentSymbol.flags & SymbolFlags.Alias ? resolveAlias(parentSymbol) : parentSymbol
                        );
                        typeParameterNodes = mapToTypeNodes(map(params, t => getMappedType(t, (nextSymbol as TransientSymbol).mapper!)), context);
                    }
                    else {
                        typeParameterNodes = typeParametersToTypeParameterDeclarations(symbol, context);
                    }
                }
                return typeParameterNodes;
            }

            /**
             * Given A[B][C][D], finds A[B]
             */
            function getTopmostIndexedAccessType(top: IndexedAccessTypeNode): IndexedAccessTypeNode {
                if (isIndexedAccessTypeNode(top.objectType)) {
                    return getTopmostIndexedAccessType(top.objectType);
                }
                return top;
            }

            function getSpecifierForModuleSymbol(symbol: Symbol, context: NodeBuilderContext) {
                let file = getDeclarationOfKind<SourceFile>(symbol, SyntaxKind.SourceFile);
                if (!file) {
                    const equivalentFileSymbol = firstDefined(symbol.declarations, d => getFileSymbolIfFileSymbolExportEqualsContainer(d, symbol));
                    if (equivalentFileSymbol) {
                        file = getDeclarationOfKind<SourceFile>(equivalentFileSymbol, SyntaxKind.SourceFile);
                    }
                }
                if (file && file.moduleName !== undefined) {
                    // Use the amd name if it is available
                    return file.moduleName;
                }
                if (!file) {
                    if (context.tracker.trackReferencedAmbientModule) {
                        const ambientDecls = filter(symbol.declarations, isAmbientModule);
                        if (length(ambientDecls)) {
                            for (const decl of ambientDecls) {
                                context.tracker.trackReferencedAmbientModule(decl, symbol);
                            }
                        }
                    }
                    if (ambientModuleSymbolRegex.test(symbol.escapedName as string)) {
                        return (symbol.escapedName as string).substring(1, (symbol.escapedName as string).length - 1);
                    }
                }
                if (!context.enclosingDeclaration || !context.tracker.moduleResolverHost) {
                    // If there's no context declaration, we can't lookup a non-ambient specifier, so we just use the symbol name
                    if (ambientModuleSymbolRegex.test(symbol.escapedName as string)) {
                        return (symbol.escapedName as string).substring(1, (symbol.escapedName as string).length - 1);
                    }
                    return getSourceFileOfNode(getNonAugmentationDeclaration(symbol)!).fileName; // A resolver may not be provided for baselines and errors - in those cases we use the fileName in full
                }
                const contextFile = getSourceFileOfNode(getOriginalNode(context.enclosingDeclaration));
                const links = getSymbolLinks(symbol);
                let specifier = links.specifierCache && links.specifierCache.get(contextFile.path);
                if (!specifier) {
                    const isBundle = !!outFile(compilerOptions);
                    // For declaration bundles, we need to generate absolute paths relative to the common source dir for imports,
                    // just like how the declaration emitter does for the ambient module declarations - we can easily accomplish this
                    // using the `baseUrl` compiler option (which we would otherwise never use in declaration emit) and a non-relative
                    // specifier preference
                    const { moduleResolverHost } = context.tracker;
                    const specifierCompilerOptions = isBundle ? { ...compilerOptions, baseUrl: moduleResolverHost.getCommonSourceDirectory() } : compilerOptions;
                    specifier = first(moduleSpecifiers.getModuleSpecifiers(
                        symbol,
                        specifierCompilerOptions,
                        contextFile,
                        moduleResolverHost,
                        { importModuleSpecifierPreference: isBundle ? "non-relative" : "relative" },
                    ));
                    links.specifierCache ??= new Map();
                    links.specifierCache.set(contextFile.path, specifier);
                }
                return specifier;
            }

            function symbolToTypeNode(symbol: Symbol, context: NodeBuilderContext, meaning: SymbolFlags, overrideTypeArguments?: readonly TypeNode[]): TypeNode {
                const chain = lookupSymbolChain(symbol, context, meaning, !(context.flags & NodeBuilderFlags.UseAliasDefinedOutsideCurrentScope)); // If we're using aliases outside the current scope, dont bother with the module

                const isTypeOf = meaning === SymbolFlags.Value;
                if (some(chain[0].declarations, hasNonGlobalAugmentationExternalModuleSymbol)) {
                    // module is root, must use `ImportTypeNode`
                    const nonRootParts = chain.length > 1 ? createAccessFromSymbolChain(chain, chain.length - 1, 1) : undefined;
                    const typeParameterNodes = overrideTypeArguments || lookupTypeParameterNodes(chain, 0, context);
                    const specifier = getSpecifierForModuleSymbol(chain[0], context);
                    if (!(context.flags & NodeBuilderFlags.AllowNodeModulesRelativePaths) && getEmitModuleResolutionKind(compilerOptions) === ModuleResolutionKind.NodeJs && specifier.indexOf("/node_modules/") >= 0) {
                        // If ultimately we can only name the symbol with a reference that dives into a `node_modules` folder, we should error
                        // since declaration files with these kinds of references are liable to fail when published :(
                        context.encounteredError = true;
                        if (context.tracker.reportLikelyUnsafeImportRequiredError) {
                            context.tracker.reportLikelyUnsafeImportRequiredError(specifier);
                        }
                    }
                    const lit = factory.createLiteralTypeNode(factory.createStringLiteral(specifier));
                    if (context.tracker.trackExternalModuleSymbolOfImportTypeNode) context.tracker.trackExternalModuleSymbolOfImportTypeNode(chain[0]);
                    context.approximateLength += specifier.length + 10; // specifier + import("")
                    if (!nonRootParts || isEntityName(nonRootParts)) {
                        if (nonRootParts) {
                            const lastId = isIdentifier(nonRootParts) ? nonRootParts : nonRootParts.right;
                            lastId.typeArguments = undefined;
                        }
                        return factory.createImportTypeNode(lit, nonRootParts as EntityName, typeParameterNodes as readonly TypeNode[], isTypeOf);
                    }
                    else {
                        const splitNode = getTopmostIndexedAccessType(nonRootParts);
                        const qualifier = (splitNode.objectType as TypeReferenceNode).typeName;
                        return factory.createIndexedAccessTypeNode(factory.createImportTypeNode(lit, qualifier, typeParameterNodes as readonly TypeNode[], isTypeOf), splitNode.indexType);
                    }
                }

                const entityName = createAccessFromSymbolChain(chain, chain.length - 1, 0);
                if (isIndexedAccessTypeNode(entityName)) {
                    return entityName; // Indexed accesses can never be `typeof`
                }
                if (isTypeOf) {
                    return factory.createTypeQueryNode(entityName);
                }
                else {
                    const lastId = isIdentifier(entityName) ? entityName : entityName.right;
                    const lastTypeArgs = lastId.typeArguments;
                    lastId.typeArguments = undefined;
                    return factory.createTypeReferenceNode(entityName, lastTypeArgs as NodeArray<TypeNode>);
                }

                function createAccessFromSymbolChain(chain: Symbol[], index: number, stopper: number): EntityName | IndexedAccessTypeNode {
                    const typeParameterNodes = index === (chain.length - 1) ? overrideTypeArguments : lookupTypeParameterNodes(chain, index, context);
                    const symbol = chain[index];

                    const parent = chain[index - 1];
                    let symbolName: string | undefined;
                    if (index === 0) {
                        context.flags |= NodeBuilderFlags.InInitialEntityName;
                        symbolName = getNameOfSymbolAsWritten(symbol, context);
                        context.approximateLength += (symbolName ? symbolName.length : 0) + 1;
                        context.flags ^= NodeBuilderFlags.InInitialEntityName;
                    }
                    else {
                        if (parent && getExportsOfSymbol(parent)) {
                            const exports = getExportsOfSymbol(parent);
                            forEachEntry(exports, (ex, name) => {
                                if (getSymbolIfSameReference(ex, symbol) && !isLateBoundName(name) && name !== InternalSymbolName.ExportEquals) {
                                    symbolName = unescapeLeadingUnderscores(name);
                                    return true;
                                }
                            });
                        }
                    }
                    if (!symbolName) {
                        symbolName = getNameOfSymbolAsWritten(symbol, context);
                    }
                    context.approximateLength += symbolName.length + 1;

                    if (!(context.flags & NodeBuilderFlags.ForbidIndexedAccessSymbolReferences) && parent &&
                        getMembersOfSymbol(parent) && getMembersOfSymbol(parent).get(symbol.escapedName) &&
                        getSymbolIfSameReference(getMembersOfSymbol(parent).get(symbol.escapedName)!, symbol)) {
                        // Should use an indexed access
                        const LHS = createAccessFromSymbolChain(chain, index - 1, stopper);
                        if (isIndexedAccessTypeNode(LHS)) {
                            return factory.createIndexedAccessTypeNode(LHS, factory.createLiteralTypeNode(factory.createStringLiteral(symbolName)));
                        }
                        else {
                            return factory.createIndexedAccessTypeNode(factory.createTypeReferenceNode(LHS, typeParameterNodes as readonly TypeNode[]), factory.createLiteralTypeNode(factory.createStringLiteral(symbolName)));
                        }
                    }

                    const identifier = setEmitFlags(factory.createIdentifier(symbolName, typeParameterNodes), EmitFlags.NoAsciiEscaping);
                    identifier.symbol = symbol;

                    if (index > stopper) {
                        const LHS = createAccessFromSymbolChain(chain, index - 1, stopper);
                        if (!isEntityName(LHS)) {
                            return Debug.fail("Impossible construct - an export of an indexed access cannot be reachable");
                        }
                        return factory.createQualifiedName(LHS, identifier);
                    }
                    return identifier;
                }
            }

            function typeParameterShadowsNameInScope(escapedName: __String, context: NodeBuilderContext, type: TypeParameter) {
                const result = resolveName(context.enclosingDeclaration, escapedName, SymbolFlags.Type, /*nameNotFoundArg*/ undefined, escapedName, /*isUse*/ false);
                if (result) {
                    if (result.flags & SymbolFlags.TypeParameter && result === type.symbol) {
                        return false;
                    }
                    return true;
                }
                return false;
            }

            function typeParameterToName(type: TypeParameter, context: NodeBuilderContext) {
                if (context.flags & NodeBuilderFlags.GenerateNamesForShadowedTypeParams && context.typeParameterNames) {
                    const cached = context.typeParameterNames.get(getTypeId(type));
                    if (cached) {
                        return cached;
                    }
                }
                let result = symbolToName(type.symbol, context, SymbolFlags.Type, /*expectsIdentifier*/ true);
                if (!(result.kind & SyntaxKind.Identifier)) {
                    return factory.createIdentifier("(Missing type parameter)");
                }
                if (context.flags & NodeBuilderFlags.GenerateNamesForShadowedTypeParams) {
                    const rawtext = result.escapedText as string;
                    let i = 0;
                    let text = rawtext;
                    while (context.typeParameterNamesByText?.has(text) || typeParameterShadowsNameInScope(text as __String, context, type)) {
                        i++;
                        text = `${rawtext}_${i}`;
                    }
                    if (text !== rawtext) {
                        result = factory.createIdentifier(text, result.typeArguments);
                    }
                    (context.typeParameterNames || (context.typeParameterNames = new Map())).set(getTypeId(type), result);
                    (context.typeParameterNamesByText || (context.typeParameterNamesByText = new Set())).add(result.escapedText as string);
                }
                return result;
            }

            function symbolToName(symbol: Symbol, context: NodeBuilderContext, meaning: SymbolFlags, expectsIdentifier: true): Identifier;
            function symbolToName(symbol: Symbol, context: NodeBuilderContext, meaning: SymbolFlags, expectsIdentifier: false): EntityName;
            function symbolToName(symbol: Symbol, context: NodeBuilderContext, meaning: SymbolFlags, expectsIdentifier: boolean): EntityName {
                const chain = lookupSymbolChain(symbol, context, meaning);

                if (expectsIdentifier && chain.length !== 1
                    && !context.encounteredError
                    && !(context.flags & NodeBuilderFlags.AllowQualifedNameInPlaceOfIdentifier)) {
                    context.encounteredError = true;
                }
                return createEntityNameFromSymbolChain(chain, chain.length - 1);

                function createEntityNameFromSymbolChain(chain: Symbol[], index: number): EntityName {
                    const typeParameterNodes = lookupTypeParameterNodes(chain, index, context);
                    const symbol = chain[index];

                    if (index === 0) {
                        context.flags |= NodeBuilderFlags.InInitialEntityName;
                    }
                    const symbolName = getNameOfSymbolAsWritten(symbol, context);
                    if (index === 0) {
                        context.flags ^= NodeBuilderFlags.InInitialEntityName;
                    }

                    const identifier = setEmitFlags(factory.createIdentifier(symbolName, typeParameterNodes), EmitFlags.NoAsciiEscaping);
                    identifier.symbol = symbol;

                    return index > 0 ? factory.createQualifiedName(createEntityNameFromSymbolChain(chain, index - 1), identifier) : identifier;
                }
            }

            function symbolToExpression(symbol: Symbol, context: NodeBuilderContext, meaning: SymbolFlags) {
                const chain = lookupSymbolChain(symbol, context, meaning);

                return createExpressionFromSymbolChain(chain, chain.length - 1);

                function createExpressionFromSymbolChain(chain: Symbol[], index: number): Expression {
                    const typeParameterNodes = lookupTypeParameterNodes(chain, index, context);
                    const symbol = chain[index];

                    if (index === 0) {
                        context.flags |= NodeBuilderFlags.InInitialEntityName;
                    }
                    let symbolName = getNameOfSymbolAsWritten(symbol, context);
                    if (index === 0) {
                        context.flags ^= NodeBuilderFlags.InInitialEntityName;
                    }
                    let firstChar = symbolName.charCodeAt(0);

                    if (isSingleOrDoubleQuote(firstChar) && some(symbol.declarations, hasNonGlobalAugmentationExternalModuleSymbol)) {
                        return factory.createStringLiteral(getSpecifierForModuleSymbol(symbol, context));
                    }
                    const canUsePropertyAccess = firstChar === CharacterCodes.hash ?
                        symbolName.length > 1 && isIdentifierStart(symbolName.charCodeAt(1), languageVersion) :
                        isIdentifierStart(firstChar, languageVersion);
                    if (index === 0 || canUsePropertyAccess) {
                        const identifier = setEmitFlags(factory.createIdentifier(symbolName, typeParameterNodes), EmitFlags.NoAsciiEscaping);
                        identifier.symbol = symbol;

                        return index > 0 ? factory.createPropertyAccessExpression(createExpressionFromSymbolChain(chain, index - 1), identifier) : identifier;
                    }
                    else {
                        if (firstChar === CharacterCodes.openBracket) {
                            symbolName = symbolName.substring(1, symbolName.length - 1);
                            firstChar = symbolName.charCodeAt(0);
                        }
                        let expression: Expression | undefined;
                        if (isSingleOrDoubleQuote(firstChar)) {
                            expression = factory.createStringLiteral(
                                symbolName
                                    .substring(1, symbolName.length - 1)
                                    .replace(/\\./g, s => s.substring(1)),
                                firstChar === CharacterCodes.singleQuote);
                        }
                        else if (("" + +symbolName) === symbolName) {
                            expression = factory.createNumericLiteral(+symbolName);
                        }
                        if (!expression) {
                            expression = setEmitFlags(factory.createIdentifier(symbolName, typeParameterNodes), EmitFlags.NoAsciiEscaping);
                            expression.symbol = symbol;
                        }
                        return factory.createElementAccessExpression(createExpressionFromSymbolChain(chain, index - 1), expression);
                    }
                }
            }

            function isStringNamed(d: Declaration) {
                const name = getNameOfDeclaration(d);
                return !!name && isStringLiteral(name);
            }

            function isSingleQuotedStringNamed(d: Declaration) {
                const name = getNameOfDeclaration(d);
                return !!(name && isStringLiteral(name) && (name.singleQuote || !nodeIsSynthesized(name) && startsWith(getTextOfNode(name, /*includeTrivia*/ false), "'")));
            }

            function getPropertyNameNodeForSymbol(symbol: Symbol, context: NodeBuilderContext) {
                const singleQuote = !!length(symbol.declarations) && every(symbol.declarations, isSingleQuotedStringNamed);
                const fromNameType = getPropertyNameNodeForSymbolFromNameType(symbol, context, singleQuote);
                if (fromNameType) {
                    return fromNameType;
                }
                if (isKnownSymbol(symbol)) {
                    return factory.createComputedPropertyName(factory.createPropertyAccessExpression(factory.createIdentifier("Symbol"), (symbol.escapedName as string).substr(3)));
                }
                const rawName = unescapeLeadingUnderscores(symbol.escapedName);
                const stringNamed = !!length(symbol.declarations) && every(symbol.declarations, isStringNamed);
                return createPropertyNameNodeForIdentifierOrLiteral(rawName, stringNamed, singleQuote);
            }

            // See getNameForSymbolFromNameType for a stringy equivalent
            function getPropertyNameNodeForSymbolFromNameType(symbol: Symbol, context: NodeBuilderContext, singleQuote?: boolean) {
                const nameType = getSymbolLinks(symbol).nameType;
                if (nameType) {
                    if (nameType.flags & TypeFlags.StringOrNumberLiteral) {
                        const name = "" + (<StringLiteralType | NumberLiteralType>nameType).value;
                        if (!isIdentifierText(name, compilerOptions.target) && !isNumericLiteralName(name)) {
                            return factory.createStringLiteral(name, !!singleQuote);
                        }
                        if (isNumericLiteralName(name) && startsWith(name, "-")) {
                            return factory.createComputedPropertyName(factory.createNumericLiteral(+name));
                        }
                        return createPropertyNameNodeForIdentifierOrLiteral(name);
                    }
                    if (nameType.flags & TypeFlags.UniqueESSymbol) {
                        return factory.createComputedPropertyName(symbolToExpression((<UniqueESSymbolType>nameType).symbol, context, SymbolFlags.Value));
                    }
                }
            }

            function createPropertyNameNodeForIdentifierOrLiteral(name: string, stringNamed?: boolean, singleQuote?: boolean) {
                return isIdentifierText(name, compilerOptions.target) ? factory.createIdentifier(name) :
                    !stringNamed && isNumericLiteralName(name) && +name >= 0 ? factory.createNumericLiteral(+name) :
                    factory.createStringLiteral(name, !!singleQuote);
            }

            function cloneNodeBuilderContext(context: NodeBuilderContext): NodeBuilderContext {
                const initial: NodeBuilderContext = { ...context };
                // Make type parameters created within this context not consume the name outside this context
                // The symbol serializer ends up creating many sibling scopes that all need "separate" contexts when
                // it comes to naming things - within a normal `typeToTypeNode` call, the node builder only ever descends
                // through the type tree, so the only cases where we could have used distinct sibling scopes was when there
                // were multiple generic overloads with similar generated type parameter names
                // The effect:
                // When we write out
                // export const x: <T>(x: T) => T
                // export const y: <T>(x: T) => T
                // we write it out like that, rather than as
                // export const x: <T>(x: T) => T
                // export const y: <T_1>(x: T_1) => T_1
                if (initial.typeParameterNames) {
                    initial.typeParameterNames = new Map(initial.typeParameterNames);
                }
                if (initial.typeParameterNamesByText) {
                    initial.typeParameterNamesByText = new Set(initial.typeParameterNamesByText);
                }
                if (initial.typeParameterSymbolList) {
                    initial.typeParameterSymbolList = new Set(initial.typeParameterSymbolList);
                }
                return initial;
            }


            function getDeclarationWithTypeAnnotation(symbol: Symbol, enclosingDeclaration: Node | undefined) {
                return symbol.declarations && find(symbol.declarations, s => !!getEffectiveTypeAnnotationNode(s) && (!enclosingDeclaration || !!findAncestor(s, n => n === enclosingDeclaration)));
            }

            function existingTypeNodeIsNotReferenceOrIsReferenceWithCompatibleTypeArgumentCount(existing: TypeNode, type: Type) {
                return !(getObjectFlags(type) & ObjectFlags.Reference) || !isTypeReferenceNode(existing) || length(existing.typeArguments) >= getMinTypeArgumentCount((type as TypeReference).target.typeParameters);
            }

            /**
             * Unlike `typeToTypeNodeHelper`, this handles setting up the `AllowUniqueESSymbolType` flag
             * so a `unique symbol` is returned when appropriate for the input symbol, rather than `typeof sym`
             */
            function serializeTypeForDeclaration(context: NodeBuilderContext, type: Type, symbol: Symbol, enclosingDeclaration: Node | undefined, includePrivateSymbol?: (s: Symbol) => void, bundled?: boolean) {
                if (type !== errorType && enclosingDeclaration) {
                    const declWithExistingAnnotation = getDeclarationWithTypeAnnotation(symbol, enclosingDeclaration);
                    if (declWithExistingAnnotation && !isFunctionLikeDeclaration(declWithExistingAnnotation)) {
                        // try to reuse the existing annotation
                        const existing = getEffectiveTypeAnnotationNode(declWithExistingAnnotation)!;
                        if (getTypeFromTypeNode(existing) === type && existingTypeNodeIsNotReferenceOrIsReferenceWithCompatibleTypeArgumentCount(existing, type)) {
                            const result = serializeExistingTypeNode(context, existing, includePrivateSymbol, bundled);
                            if (result) {
                                return result;
                            }
                        }
                    }
                }
                const oldFlags = context.flags;
                if (type.flags & TypeFlags.UniqueESSymbol &&
                    type.symbol === symbol) {
                    context.flags |= NodeBuilderFlags.AllowUniqueESSymbolType;
                }
                const result = typeToTypeNodeHelper(type, context);
                context.flags = oldFlags;
                return result;
            }

            function serializeReturnTypeForSignature(context: NodeBuilderContext, type: Type, signature: Signature, includePrivateSymbol?: (s: Symbol) => void, bundled?: boolean) {
                if (type !== errorType && context.enclosingDeclaration) {
                    const annotation = signature.declaration && getEffectiveReturnTypeNode(signature.declaration);
                    if (!!findAncestor(annotation, n => n === context.enclosingDeclaration) && annotation && instantiateType(getTypeFromTypeNode(annotation), signature.mapper) === type && existingTypeNodeIsNotReferenceOrIsReferenceWithCompatibleTypeArgumentCount(annotation, type)) {
                        const result = serializeExistingTypeNode(context, annotation, includePrivateSymbol, bundled);
                        if (result) {
                            return result;
                        }
                    }
                }
                return typeToTypeNodeHelper(type, context);
            }

            function serializeExistingTypeNode(context: NodeBuilderContext, existing: TypeNode, includePrivateSymbol?: (s: Symbol) => void, bundled?: boolean) {
                if (cancellationToken && cancellationToken.throwIfCancellationRequested) {
                    cancellationToken.throwIfCancellationRequested();
                }
                let hadError = false;
                const file = getSourceFileOfNode(existing);
                const transformed = visitNode(existing, visitExistingNodeTreeSymbols);
                if (hadError) {
                    return undefined;
                }
                return transformed === existing ? setTextRange(factory.cloneNode(existing), existing) : transformed;

                function visitExistingNodeTreeSymbols<T extends Node>(node: T): Node {
                    // We don't _actually_ support jsdoc namepath types, emit `any` instead
                    if (isJSDocAllType(node) || node.kind === SyntaxKind.JSDocNamepathType) {
                        return factory.createKeywordTypeNode(SyntaxKind.AnyKeyword);
                    }
                    if (isJSDocUnknownType(node)) {
                        return factory.createKeywordTypeNode(SyntaxKind.UnknownKeyword);
                    }
                    if (isJSDocNullableType(node)) {
                        return factory.createUnionTypeNode([visitNode(node.type, visitExistingNodeTreeSymbols), factory.createLiteralTypeNode(factory.createNull())]);
                    }
                    if (isJSDocOptionalType(node)) {
                        return factory.createUnionTypeNode([visitNode(node.type, visitExistingNodeTreeSymbols), factory.createKeywordTypeNode(SyntaxKind.UndefinedKeyword)]);
                    }
                    if (isJSDocNonNullableType(node)) {
                        return visitNode(node.type, visitExistingNodeTreeSymbols);
                    }
                    if (isJSDocVariadicType(node)) {
                        return factory.createArrayTypeNode(visitNode((node as JSDocVariadicType).type, visitExistingNodeTreeSymbols));
                    }
                    if (isJSDocTypeLiteral(node)) {
                        return factory.createTypeLiteralNode(map(node.jsDocPropertyTags, t => {
                            const name = isIdentifier(t.name) ? t.name : t.name.right;
                            const typeViaParent = getTypeOfPropertyOfType(getTypeFromTypeNode(node), name.escapedText);
                            const overrideTypeNode = typeViaParent && t.typeExpression && getTypeFromTypeNode(t.typeExpression.type) !== typeViaParent ? typeToTypeNodeHelper(typeViaParent, context) : undefined;

                            return factory.createPropertySignature(
                                /*modifiers*/ undefined,
                                name,
                                t.typeExpression && isJSDocOptionalType(t.typeExpression.type) ? factory.createToken(SyntaxKind.QuestionToken) : undefined,
                                overrideTypeNode || (t.typeExpression && visitNode(t.typeExpression.type, visitExistingNodeTreeSymbols)) || factory.createKeywordTypeNode(SyntaxKind.AnyKeyword)
                            );
                        }));
                    }
                    if (isTypeReferenceNode(node) && isIdentifier(node.typeName) && node.typeName.escapedText === "") {
                        return setOriginalNode(factory.createKeywordTypeNode(SyntaxKind.AnyKeyword), node);
                    }
                    if ((isExpressionWithTypeArguments(node) || isTypeReferenceNode(node)) && isJSDocIndexSignature(node)) {
                        return factory.createTypeLiteralNode([factory.createIndexSignature(
                            /*decorators*/ undefined,
                            /*modifiers*/ undefined,
                            [factory.createParameterDeclaration(
                                /*decorators*/ undefined,
                                /*modifiers*/ undefined,
                                /*dotdotdotToken*/ undefined,
                                "x",
                                /*questionToken*/ undefined,
                                visitNode(node.typeArguments![0], visitExistingNodeTreeSymbols)
                            )],
                            visitNode(node.typeArguments![1], visitExistingNodeTreeSymbols)
                        )]);
                    }
                    if (isJSDocFunctionType(node)) {
                        if (isJSDocConstructSignature(node)) {
                            let newTypeNode: TypeNode | undefined;
                            return factory.createConstructorTypeNode(
                                visitNodes(node.typeParameters, visitExistingNodeTreeSymbols),
                                mapDefined(node.parameters, (p, i) => p.name && isIdentifier(p.name) && p.name.escapedText === "new" ? (newTypeNode = p.type, undefined) : factory.createParameterDeclaration(
                                    /*decorators*/ undefined,
                                    /*modifiers*/ undefined,
                                    getEffectiveDotDotDotForParameter(p),
                                    getNameForJSDocFunctionParameter(p, i),
                                    p.questionToken,
                                    visitNode(p.type, visitExistingNodeTreeSymbols),
                                    /*initializer*/ undefined
                                )),
                                visitNode(newTypeNode || node.type, visitExistingNodeTreeSymbols) || factory.createKeywordTypeNode(SyntaxKind.AnyKeyword)
                            );
                        }
                        else {
                            return factory.createFunctionTypeNode(
                                visitNodes(node.typeParameters, visitExistingNodeTreeSymbols),
                                map(node.parameters, (p, i) => factory.createParameterDeclaration(
                                    /*decorators*/ undefined,
                                    /*modifiers*/ undefined,
                                    getEffectiveDotDotDotForParameter(p),
                                    getNameForJSDocFunctionParameter(p, i),
                                    p.questionToken,
                                    visitNode(p.type, visitExistingNodeTreeSymbols),
                                    /*initializer*/ undefined
                                )),
                                visitNode(node.type, visitExistingNodeTreeSymbols) || factory.createKeywordTypeNode(SyntaxKind.AnyKeyword)
                            );
                        }
                    }
                    if (isTypeReferenceNode(node) && isInJSDoc(node) && (!existingTypeNodeIsNotReferenceOrIsReferenceWithCompatibleTypeArgumentCount(node, getTypeFromTypeNode(node)) || getIntendedTypeFromJSDocTypeReference(node) || unknownSymbol === resolveTypeReferenceName(getTypeReferenceName(node), SymbolFlags.Type, /*ignoreErrors*/ true))) {
                        return setOriginalNode(typeToTypeNodeHelper(getTypeFromTypeNode(node), context), node);
                    }
                    if (isLiteralImportTypeNode(node)) {
                        return factory.updateImportTypeNode(
                            node,
                            factory.updateLiteralTypeNode(node.argument, rewriteModuleSpecifier(node, node.argument.literal)),
                            node.qualifier,
                            visitNodes(node.typeArguments, visitExistingNodeTreeSymbols, isTypeNode),
                            node.isTypeOf
                        );
                    }

                    if (isEntityName(node) || isEntityNameExpression(node)) {
                        const leftmost = getFirstIdentifier(node);
                        if (isInJSFile(node) && (isExportsIdentifier(leftmost) || isModuleExportsAccessExpression(leftmost.parent) || (isQualifiedName(leftmost.parent) && isModuleIdentifier(leftmost.parent.left) && isExportsIdentifier(leftmost.parent.right)))) {
                            hadError = true;
                            return node;
                        }
                        const sym = resolveEntityName(leftmost, SymbolFlags.All, /*ignoreErrors*/ true, /*dontResolveALias*/ true);
                        if (sym) {
                            if (isSymbolAccessible(sym, context.enclosingDeclaration, SymbolFlags.All, /*shouldComputeAliasesToMakeVisible*/ false).accessibility !== SymbolAccessibility.Accessible) {
                                hadError = true;
                            }
                            else {
                                context.tracker?.trackSymbol?.(sym, context.enclosingDeclaration, SymbolFlags.All);
                                includePrivateSymbol?.(sym);
                            }
                            if (isIdentifier(node)) {
                                const name = sym.flags & SymbolFlags.TypeParameter ? typeParameterToName(getDeclaredTypeOfSymbol(sym), context) : factory.cloneNode(node);
                                name.symbol = sym; // for quickinfo, which uses identifier symbol information
                                return setEmitFlags(setOriginalNode(name, node), EmitFlags.NoAsciiEscaping);
                            }
                        }
                    }

                    if (file && isTupleTypeNode(node) && (getLineAndCharacterOfPosition(file, node.pos).line === getLineAndCharacterOfPosition(file, node.end).line)) {
                        setEmitFlags(node, EmitFlags.SingleLine);
                    }

                    return visitEachChild(node, visitExistingNodeTreeSymbols, nullTransformationContext);

                    function getEffectiveDotDotDotForParameter(p: ParameterDeclaration) {
                        return p.dotDotDotToken || (p.type && isJSDocVariadicType(p.type) ? factory.createToken(SyntaxKind.DotDotDotToken) : undefined);
                    }

                    /** Note that `new:T` parameters are not handled, but should be before calling this function. */
                    function getNameForJSDocFunctionParameter(p: ParameterDeclaration, index: number) {
                        return p.name && isIdentifier(p.name) && p.name.escapedText === "this" ? "this"
                            : getEffectiveDotDotDotForParameter(p) ? `args`
                            : `arg${index}`;
                    }

                    function rewriteModuleSpecifier(parent: ImportTypeNode, lit: StringLiteral) {
                        if (bundled) {
                            if (context.tracker && context.tracker.moduleResolverHost) {
                                const targetFile = getExternalModuleFileFromDeclaration(parent);
                                if (targetFile) {
                                    const getCanonicalFileName = createGetCanonicalFileName(!!host.useCaseSensitiveFileNames);
                                    const resolverHost = {
                                        getCanonicalFileName,
                                        getCurrentDirectory: () => context.tracker.moduleResolverHost!.getCurrentDirectory(),
                                        getCommonSourceDirectory: () => context.tracker.moduleResolverHost!.getCommonSourceDirectory()
                                    };
                                    const newName = getResolvedExternalModuleName(resolverHost, targetFile);
                                    return factory.createStringLiteral(newName);
                                }
                            }
                        }
                        else {
                            if (context.tracker && context.tracker.trackExternalModuleSymbolOfImportTypeNode) {
                                const moduleSym = resolveExternalModuleNameWorker(lit, lit, /*moduleNotFoundError*/ undefined);
                                if (moduleSym) {
                                    context.tracker.trackExternalModuleSymbolOfImportTypeNode(moduleSym);
                                }
                            }
                        }
                        return lit;
                    }
                }
            }

            function symbolTableToDeclarationStatements(symbolTable: SymbolTable, context: NodeBuilderContext, bundled?: boolean): Statement[] {
                const serializePropertySymbolForClass = makeSerializePropertySymbol<ClassElement>(factory.createPropertyDeclaration, SyntaxKind.MethodDeclaration, /*useAcessors*/ true);
                const serializePropertySymbolForInterfaceWorker = makeSerializePropertySymbol<TypeElement>((_decorators, mods, name, question, type) => factory.createPropertySignature(mods, name, question, type), SyntaxKind.MethodSignature, /*useAcessors*/ false);

                // TODO: Use `setOriginalNode` on original declaration names where possible so these declarations see some kind of
                // declaration mapping

                // We save the enclosing declaration off here so it's not adjusted by well-meaning declaration
                // emit codepaths which want to apply more specific contexts (so we can still refer to the root real declaration
                // we're trying to emit from later on)
                const enclosingDeclaration = context.enclosingDeclaration!;
                let results: Statement[] = [];
                const visitedSymbols = new Set<number>();
                const deferredPrivatesStack: ESMap<SymbolId, Symbol>[] = [];
                const oldcontext = context;
                context = {
                    ...oldcontext,
                    usedSymbolNames: new Set(oldcontext.usedSymbolNames),
                    remappedSymbolNames: new Map(),
                    tracker: {
                        ...oldcontext.tracker,
                        trackSymbol: (sym, decl, meaning) => {
                            const accessibleResult = isSymbolAccessible(sym, decl, meaning, /*computeALiases*/ false);
                            if (accessibleResult.accessibility === SymbolAccessibility.Accessible) {
                                // Lookup the root symbol of the chain of refs we'll use to access it and serialize it
                                const chain = lookupSymbolChainWorker(sym, context, meaning);
                                if (!(sym.flags & SymbolFlags.Property)) {
                                    includePrivateSymbol(chain[0]);
                                }
                            }
                            else if (oldcontext.tracker && oldcontext.tracker.trackSymbol) {
                                oldcontext.tracker.trackSymbol(sym, decl, meaning);
                            }
                        }
                    }
                };
                forEachEntry(symbolTable, (symbol, name) => {
                    const baseName = unescapeLeadingUnderscores(name);
                    void getInternalSymbolName(symbol, baseName); // Called to cache values into `usedSymbolNames` and `remappedSymbolNames`
                });
                let addingDeclare = !bundled;
                const exportEquals = symbolTable.get(InternalSymbolName.ExportEquals);
                if (exportEquals && symbolTable.size > 1 && exportEquals.flags & SymbolFlags.Alias) {
                    symbolTable = createSymbolTable();
                    // Remove extraneous elements from root symbol table (they'll be mixed back in when the target of the `export=` is looked up)
                    symbolTable.set(InternalSymbolName.ExportEquals, exportEquals);
                }

                visitSymbolTable(symbolTable);
                return mergeRedundantStatements(results);

                function isIdentifierAndNotUndefined(node: Node | undefined): node is Identifier {
                    return !!node && node.kind === SyntaxKind.Identifier;
                }

                function getNamesOfDeclaration(statement: Statement): Identifier[] {
                    if (isVariableStatement(statement)) {
                        return filter(map(statement.declarationList.declarations, getNameOfDeclaration), isIdentifierAndNotUndefined);
                    }
                    return filter([getNameOfDeclaration(statement as DeclarationStatement)], isIdentifierAndNotUndefined);
                }

                function flattenExportAssignedNamespace(statements: Statement[]) {
                    const exportAssignment = find(statements, isExportAssignment);
                    const nsIndex = findIndex(statements, isModuleDeclaration);
                    let ns = nsIndex !== -1 ? statements[nsIndex] as ModuleDeclaration : undefined;
                    if (ns && exportAssignment && exportAssignment.isExportEquals &&
                        isIdentifier(exportAssignment.expression) && isIdentifier(ns.name) && idText(ns.name) === idText(exportAssignment.expression) &&
                        ns.body && isModuleBlock(ns.body)) {
                        // Pass 0: Correct situations where a module has both an `export = ns` and multiple top-level exports by stripping the export modifiers from
                        //  the top-level exports and exporting them in the targeted ns, as can occur when a js file has both typedefs and `module.export` assignments
                        const excessExports = filter(statements, s => !!(getEffectiveModifierFlags(s) & ModifierFlags.Export));
                        const name = ns.name;
                        let body = ns.body;
                        if (length(excessExports)) {
                            ns = factory.updateModuleDeclaration(
                                ns,
                                ns.decorators,
                                ns.modifiers,
                                ns.name,
                                body = factory.updateModuleBlock(
                                    body,
                                    factory.createNodeArray([...ns.body.statements, factory.createExportDeclaration(
                                        /*decorators*/ undefined,
                                        /*modifiers*/ undefined,
                                        /*isTypeOnly*/ false,
                                        factory.createNamedExports(map(flatMap(excessExports, e => getNamesOfDeclaration(e)), id => factory.createExportSpecifier(/*alias*/ undefined, id))),
                                        /*moduleSpecifier*/ undefined
                                    )])
                                )
                            );
                            statements = [...statements.slice(0, nsIndex), ns, ...statements.slice(nsIndex + 1)];
                        }

                        // Pass 1: Flatten `export namespace _exports {} export = _exports;` so long as the `export=` only points at a single namespace declaration
                        if (!find(statements, s => s !== ns && nodeHasName(s, name))) {
                            results = [];
                            // If the namespace contains no export assignments or declarations, and no declarations flagged with `export`, then _everything_ is exported -
                            // to respect this as the top level, we need to add an `export` modifier to everything
                            const mixinExportFlag = !some(body.statements, s => hasSyntacticModifier(s, ModifierFlags.Export) || isExportAssignment(s) || isExportDeclaration(s));
                            forEach(body.statements, s => {
                                addResult(s, mixinExportFlag ? ModifierFlags.Export : ModifierFlags.None); // Recalculates the ambient (and export, if applicable from above) flag
                            });
                            statements = [...filter(statements, s => s !== ns && s !== exportAssignment), ...results];
                        }
                    }
                    return statements;
                }

                function mergeExportDeclarations(statements: Statement[]) {
                    // Pass 2: Combine all `export {}` declarations
                    const exports = filter(statements, d => isExportDeclaration(d) && !d.moduleSpecifier && !!d.exportClause && isNamedExports(d.exportClause)) as ExportDeclaration[];
                    if (length(exports) > 1) {
                        const nonExports = filter(statements, d => !isExportDeclaration(d) || !!d.moduleSpecifier || !d.exportClause);
                        statements = [...nonExports, factory.createExportDeclaration(
                            /*decorators*/ undefined,
                            /*modifiers*/ undefined,
                            /*isTypeOnly*/ false,
                            factory.createNamedExports(flatMap(exports, e => cast(e.exportClause, isNamedExports).elements)),
                            /*moduleSpecifier*/ undefined
                        )];
                    }
                    // Pass 2b: Also combine all `export {} from "..."` declarations as needed
                    const reexports = filter(statements, d => isExportDeclaration(d) && !!d.moduleSpecifier && !!d.exportClause && isNamedExports(d.exportClause)) as ExportDeclaration[];
                    if (length(reexports) > 1) {
                        const groups = group(reexports, decl => isStringLiteral(decl.moduleSpecifier!) ? ">" + decl.moduleSpecifier.text : ">");
                        if (groups.length !== reexports.length) {
                            for (const group of groups) {
                                if (group.length > 1) {
                                    // remove group members from statements and then merge group members and add back to statements
                                    statements = [
                                        ...filter(statements, s => group.indexOf(s as ExportDeclaration) === -1),
                                        factory.createExportDeclaration(
                                            /*decorators*/ undefined,
                                            /*modifiers*/ undefined,
                                            /*isTypeOnly*/ false,
                                            factory.createNamedExports(flatMap(group, e => cast(e.exportClause, isNamedExports).elements)),
                                            group[0].moduleSpecifier
                                        )
                                    ];
                                }
                            }
                        }
                    }
                    return statements;
                }

                function inlineExportModifiers(statements: Statement[]) {
                    // Pass 3: Move all `export {}`'s to `export` modifiers where possible
                    const index = findIndex(statements, d => isExportDeclaration(d) && !d.moduleSpecifier && !!d.exportClause && isNamedExports(d.exportClause));
                    if (index >= 0) {
                        const exportDecl = statements[index] as ExportDeclaration & { readonly exportClause: NamedExports };
                        const replacements = mapDefined(exportDecl.exportClause.elements, e => {
                            if (!e.propertyName) {
                                // export {name} - look thru `statements` for `name`, and if all results can take an `export` modifier, do so and filter it
                                const indices = indicesOf(statements);
                                const associatedIndices = filter(indices, i => nodeHasName(statements[i], e.name));
                                if (length(associatedIndices) && every(associatedIndices, i => canHaveExportModifier(statements[i]))) {
                                    for (const index of associatedIndices) {
                                        statements[index] = addExportModifier(statements[index] as Extract<HasModifiers, Statement>);
                                    }
                                    return undefined;
                                }
                            }
                            return e;
                        });
                        if (!length(replacements)) {
                            // all clauses removed, remove the export declaration
                            orderedRemoveItemAt(statements, index);
                        }
                        else {
                            // some items filtered, others not - update the export declaration
                            statements[index] = factory.updateExportDeclaration(
                                exportDecl,
                                exportDecl.decorators,
                                exportDecl.modifiers,
                                exportDecl.isTypeOnly,
                                factory.updateNamedExports(
                                    exportDecl.exportClause,
                                    replacements
                                ),
                                exportDecl.moduleSpecifier
                            );
                        }
                    }
                    return statements;
                }

                function mergeRedundantStatements(statements: Statement[]) {
                    statements = flattenExportAssignedNamespace(statements);
                    statements = mergeExportDeclarations(statements);
                    statements = inlineExportModifiers(statements);

                    // Not a cleanup, but as a final step: If there is a mix of `export` and non-`export` declarations, but no `export =` or `export {}` add a `export {};` so
                    // declaration privacy is respected.
                    if (enclosingDeclaration &&
                        ((isSourceFile(enclosingDeclaration) && isExternalOrCommonJsModule(enclosingDeclaration)) || isModuleDeclaration(enclosingDeclaration)) &&
                        (!some(statements, isExternalModuleIndicator) || (!hasScopeMarker(statements) && some(statements, needsScopeMarker)))) {
                        statements.push(createEmptyExports(factory));
                    }
                    return statements;
                }

                function canHaveExportModifier(node: Statement): node is Extract<HasModifiers, Statement> {
                    return isEnumDeclaration(node) ||
                            isVariableStatement(node) ||
                            isFunctionDeclaration(node) ||
                            isClassDeclaration(node) ||
                            (isModuleDeclaration(node) && !isExternalModuleAugmentation(node) && !isGlobalScopeAugmentation(node)) ||
                            isInterfaceDeclaration(node) ||
                            isTypeDeclaration(node);
                }

                function addExportModifier(node: Extract<HasModifiers, Statement>) {
                    const flags = (getEffectiveModifierFlags(node) | ModifierFlags.Export) & ~ModifierFlags.Ambient;
                    return factory.updateModifiers(node, flags);
                }

                function removeExportModifier(node: Extract<HasModifiers, Statement>) {
                    const flags = getEffectiveModifierFlags(node) & ~ModifierFlags.Export;
                    return factory.updateModifiers(node, flags);
                }

                function visitSymbolTable(symbolTable: SymbolTable, suppressNewPrivateContext?: boolean, propertyAsAlias?: boolean) {
                    if (!suppressNewPrivateContext) {
                        deferredPrivatesStack.push(new Map());
                    }
                    symbolTable.forEach((symbol: Symbol) => {
                        serializeSymbol(symbol, /*isPrivate*/ false, !!propertyAsAlias);
                    });
                    if (!suppressNewPrivateContext) {
                        // deferredPrivates will be filled up by visiting the symbol table
                        // And will continue to iterate as elements are added while visited `deferredPrivates`
                        // (As that's how a map iterator is defined to work)
                        deferredPrivatesStack[deferredPrivatesStack.length - 1].forEach((symbol: Symbol) => {
                            serializeSymbol(symbol, /*isPrivate*/ true, !!propertyAsAlias);
                        });
                        deferredPrivatesStack.pop();
                    }
                }

                function serializeSymbol(symbol: Symbol, isPrivate: boolean, propertyAsAlias: boolean) {
                    // cache visited list based on merged symbol, since we want to use the unmerged top-level symbol, but
                    // still skip reserializing it if we encounter the merged product later on
                    const visitedSym = getMergedSymbol(symbol);
                    if (visitedSymbols.has(getSymbolId(visitedSym))) {
                        return; // Already printed
                    }
                    visitedSymbols.add(getSymbolId(visitedSym));
                    // Only actually serialize symbols within the correct enclosing declaration, otherwise do nothing with the out-of-context symbol
                    const skipMembershipCheck = !isPrivate; // We only call this on exported symbols when we know they're in the correct scope
                    if (skipMembershipCheck || (!!length(symbol.declarations) && some(symbol.declarations, d => !!findAncestor(d, n => n === enclosingDeclaration)))) {
                        const oldContext = context;
                        context = cloneNodeBuilderContext(context);
                        const result = serializeSymbolWorker(symbol, isPrivate, propertyAsAlias);
                        context = oldContext;
                        return result;
                    }
                }


                // Synthesize declarations for a symbol - might be an Interface, a Class, a Namespace, a Type, a Variable (const, let, or var), an Alias
                // or a merge of some number of those.
                // An interesting challenge is ensuring that when classes merge with namespaces and interfaces, is keeping
                // each symbol in only one of the representations
                // Also, synthesizing a default export of some kind
                // If it's an alias: emit `export default ref`
                // If it's a property: emit `export default _default` with a `_default` prop
                // If it's a class/interface/function: emit a class/interface/function with a `default` modifier
                // These forms can merge, eg (`export default 12; export default interface A {}`)
                function serializeSymbolWorker(symbol: Symbol, isPrivate: boolean, propertyAsAlias: boolean) {
                    const symbolName = unescapeLeadingUnderscores(symbol.escapedName);
                    const isDefault = symbol.escapedName === InternalSymbolName.Default;
                    if (isPrivate && !(context.flags & NodeBuilderFlags.AllowAnonymousIdentifier) && isStringANonContextualKeyword(symbolName) && !isDefault) {
                        // Oh no. We cannot use this symbol's name as it's name... It's likely some jsdoc had an invalid name like `export` or `default` :(
                        context.encounteredError = true;
                        // TODO: Issue error via symbol tracker?
                        return; // If we need to emit a private with a keyword name, we're done for, since something else will try to refer to it by that name
                    }
                    let needsPostExportDefault = isDefault && !!(
                           symbol.flags & SymbolFlags.ExportDoesNotSupportDefaultModifier
                        || (symbol.flags & SymbolFlags.Function && length(getPropertiesOfType(getTypeOfSymbol(symbol))))
                    ) && !(symbol.flags & SymbolFlags.Alias); // An alias symbol should preclude needing to make an alias ourselves
                    let needsExportDeclaration = !needsPostExportDefault && !isPrivate && isStringANonContextualKeyword(symbolName) && !isDefault;
                    // `serializeVariableOrProperty` will handle adding the export declaration if it is run (since `getInternalSymbolName` will create the name mapping), so we need to ensuer we unset `needsExportDeclaration` if it is
                    if (needsPostExportDefault || needsExportDeclaration) {
                        isPrivate = true;
                    }
                    const modifierFlags = (!isPrivate ? ModifierFlags.Export : 0) | (isDefault && !needsPostExportDefault ? ModifierFlags.Default : 0);
                    const isConstMergedWithNS = symbol.flags & SymbolFlags.Module &&
                        symbol.flags & (SymbolFlags.BlockScopedVariable | SymbolFlags.FunctionScopedVariable | SymbolFlags.Property) &&
                        symbol.escapedName !== InternalSymbolName.ExportEquals;
                    const isConstMergedWithNSPrintableAsSignatureMerge = isConstMergedWithNS && isTypeRepresentableAsFunctionNamespaceMerge(getTypeOfSymbol(symbol), symbol);
                    if (symbol.flags & (SymbolFlags.Function | SymbolFlags.Method) || isConstMergedWithNSPrintableAsSignatureMerge) {
                        serializeAsFunctionNamespaceMerge(getTypeOfSymbol(symbol), symbol, getInternalSymbolName(symbol, symbolName), modifierFlags);
                    }
                    if (symbol.flags & SymbolFlags.TypeAlias) {
                        serializeTypeAlias(symbol, symbolName, modifierFlags);
                    }
                    // Need to skip over export= symbols below - json source files get a single `Property` flagged
                    // symbol of name `export=` which needs to be handled like an alias. It's not great, but it is what it is.
                    if (symbol.flags & (SymbolFlags.BlockScopedVariable | SymbolFlags.FunctionScopedVariable | SymbolFlags.Property)
                        && symbol.escapedName !== InternalSymbolName.ExportEquals
                        && !(symbol.flags & SymbolFlags.Prototype)
                        && !(symbol.flags & SymbolFlags.Class)
                        && !isConstMergedWithNSPrintableAsSignatureMerge) {
                        if (propertyAsAlias) {
                            const createdExport = serializeMaybeAliasAssignment(symbol);
                            if (createdExport) {
                                needsExportDeclaration = false;
                                needsPostExportDefault = false;
                            }
                        }
                        else {
                            const type = getTypeOfSymbol(symbol);
                            const localName = getInternalSymbolName(symbol, symbolName);
                            if (!(symbol.flags & SymbolFlags.Function) && isTypeRepresentableAsFunctionNamespaceMerge(type, symbol)) {
                                // If the type looks like a function declaration + ns could represent it, and it's type is sourced locally, rewrite it into a function declaration + ns
                                serializeAsFunctionNamespaceMerge(type, symbol, localName, modifierFlags);
                            }
                            else {
                                // A Class + Property merge is made for a `module.exports.Member = class {}`, and it doesn't serialize well as either a class _or_ a property symbol - in fact, _it behaves like an alias!_
                                // `var` is `FunctionScopedVariable`, `const` and `let` are `BlockScopedVariable`, and `module.exports.thing =` is `Property`
                                const flags = !(symbol.flags & SymbolFlags.BlockScopedVariable) ? undefined
                                    : isConstVariable(symbol) ? NodeFlags.Const
                                    : NodeFlags.Let;
                                const name = (needsPostExportDefault || !(symbol.flags & SymbolFlags.Property)) ? localName : getUnusedName(localName, symbol);
                                let textRange: Node | undefined = symbol.declarations && find(symbol.declarations, d => isVariableDeclaration(d));
                                if (textRange && isVariableDeclarationList(textRange.parent) && textRange.parent.declarations.length === 1) {
                                    textRange = textRange.parent.parent;
                                }
                                const statement = setTextRange(factory.createVariableStatement(/*modifiers*/ undefined, factory.createVariableDeclarationList([
                                    factory.createVariableDeclaration(name, /*exclamationToken*/ undefined, serializeTypeForDeclaration(context, type, symbol, enclosingDeclaration, includePrivateSymbol, bundled))
                                ], flags)), textRange);
                                addResult(statement, name !== localName ? modifierFlags & ~ModifierFlags.Export : modifierFlags);
                                if (name !== localName && !isPrivate) {
                                    // We rename the variable declaration we generate for Property symbols since they may have a name which
                                    // conflicts with a local declaration. For example, given input:
                                    // ```
                                    // function g() {}
                                    // module.exports.g = g
                                    // ```
                                    // In such a situation, we have a local variable named `g`, and a separate exported variable named `g`.
                                    // Naively, we would emit
                                    // ```
                                    // function g() {}
                                    // export const g: typeof g;
                                    // ```
                                    // That's obviously incorrect - the `g` in the type annotation needs to refer to the local `g`, but
                                    // the export declaration shadows it.
                                    // To work around that, we instead write
                                    // ```
                                    // function g() {}
                                    // const g_1: typeof g;
                                    // export { g_1 as g };
                                    // ```
                                    // To create an export named `g` that does _not_ shadow the local `g`
                                    addResult(
                                        factory.createExportDeclaration(
                                            /*decorators*/ undefined,
                                            /*modifiers*/ undefined,
                                            /*isTypeOnly*/ false,
                                            factory.createNamedExports([factory.createExportSpecifier(name, localName)])
                                        ),
                                        ModifierFlags.None
                                    );
                                    needsExportDeclaration = false;
                                    needsPostExportDefault = false;
                                }
                            }
                        }
                    }
                    if (symbol.flags & SymbolFlags.Enum) {
                        serializeEnum(symbol, symbolName, modifierFlags);
                    }
                    if (symbol.flags & SymbolFlags.Class) {
                        if (symbol.flags & SymbolFlags.Property && isBinaryExpression(symbol.valueDeclaration.parent) && isClassExpression(symbol.valueDeclaration.parent.right)) {
                            // Looks like a `module.exports.Sub = class {}` - if we serialize `symbol` as a class, the result will have no members,
                            // since the classiness is actually from the target of the effective alias the symbol is. yes. A BlockScopedVariable|Class|Property
                            // _really_ acts like an Alias, and none of a BlockScopedVariable, Class, or Property. This is the travesty of JS binding today.
                            serializeAsAlias(symbol, getInternalSymbolName(symbol, symbolName), modifierFlags);
                        }
                        else {
                            serializeAsClass(symbol, getInternalSymbolName(symbol, symbolName), modifierFlags);
                        }
                    }
                    if ((symbol.flags & (SymbolFlags.ValueModule | SymbolFlags.NamespaceModule) && (!isConstMergedWithNS || isTypeOnlyNamespace(symbol))) || isConstMergedWithNSPrintableAsSignatureMerge) {
                        serializeModule(symbol, symbolName, modifierFlags);
                    }
                    if (symbol.flags & SymbolFlags.Interface) {
                        serializeInterface(symbol, symbolName, modifierFlags);
                    }
                    if (symbol.flags & SymbolFlags.Alias) {
                        serializeAsAlias(symbol, getInternalSymbolName(symbol, symbolName), modifierFlags);
                    }
                    if (symbol.flags & SymbolFlags.Property && symbol.escapedName === InternalSymbolName.ExportEquals) {
                        serializeMaybeAliasAssignment(symbol);
                    }
                    if (symbol.flags & SymbolFlags.ExportStar) {
                        // synthesize export * from "moduleReference"
                        // Straightforward - only one thing to do - make an export declaration
                        for (const node of symbol.declarations) {
                            const resolvedModule = resolveExternalModuleName(node, (node as ExportDeclaration).moduleSpecifier!);
                            if (!resolvedModule) continue;
                            addResult(factory.createExportDeclaration(/*decorators*/ undefined, /*modifiers*/ undefined, /*isTypeOnly*/ false, /*exportClause*/ undefined, factory.createStringLiteral(getSpecifierForModuleSymbol(resolvedModule, context))), ModifierFlags.None);
                        }
                    }
                    if (needsPostExportDefault) {
                        addResult(factory.createExportAssignment(/*decorators*/ undefined, /*modifiers*/ undefined, /*isExportAssignment*/ false, factory.createIdentifier(getInternalSymbolName(symbol, symbolName))), ModifierFlags.None);
                    }
                    else if (needsExportDeclaration) {
                        addResult(factory.createExportDeclaration(
                            /*decorators*/ undefined,
                            /*modifiers*/ undefined,
                            /*isTypeOnly*/ false,
                            factory.createNamedExports([factory.createExportSpecifier(getInternalSymbolName(symbol, symbolName), symbolName)])
                        ), ModifierFlags.None);
                    }
                }

                function includePrivateSymbol(symbol: Symbol) {
                    if (some(symbol.declarations, isParameterDeclaration)) return;
                    Debug.assertIsDefined(deferredPrivatesStack[deferredPrivatesStack.length - 1]);
                    getUnusedName(unescapeLeadingUnderscores(symbol.escapedName), symbol); // Call to cache unique name for symbol
                    // Blanket moving (import) aliases into the root private context should work, since imports are not valid within namespaces
                    // (so they must have been in the root to begin with if they were real imports) cjs `require` aliases (an upcoming feature)
                    // will throw a wrench in this, since those may have been nested, but we'll need to synthesize them in the outer scope
                    // anyway, as that's the only place the import they translate to is valid. In such a case, we might need to use a unique name
                    // for the moved import; which hopefully the above `getUnusedName` call should produce.
                    const isExternalImportAlias = !!(symbol.flags & SymbolFlags.Alias) && !some(symbol.declarations, d =>
                        !!findAncestor(d, isExportDeclaration) ||
                        isNamespaceExport(d) ||
                        (isImportEqualsDeclaration(d) && !isExternalModuleReference(d.moduleReference))
                    );
                    deferredPrivatesStack[isExternalImportAlias ? 0 : (deferredPrivatesStack.length - 1)].set(getSymbolId(symbol), symbol);
                }

                function isExportingScope(enclosingDeclaration: Node) {
                    return ((isSourceFile(enclosingDeclaration) && (isExternalOrCommonJsModule(enclosingDeclaration) || isJsonSourceFile(enclosingDeclaration))) ||
                        (isAmbientModule(enclosingDeclaration) && !isGlobalScopeAugmentation(enclosingDeclaration)));
                }

                // Prepends a `declare` and/or `export` modifier if the context requires it, and then adds `node` to `result` and returns `node`
                function addResult(node: Statement, additionalModifierFlags: ModifierFlags) {
                    if (canHaveModifiers(node)) {
                        let newModifierFlags: ModifierFlags = ModifierFlags.None;
                        if (additionalModifierFlags & ModifierFlags.Export &&
                            context.enclosingDeclaration &&
                            (isExportingScope(context.enclosingDeclaration) || isModuleDeclaration(context.enclosingDeclaration)) &&
                            canHaveExportModifier(node)
                        ) {
                            // Classes, namespaces, variables, functions, interfaces, and types should all be `export`ed in a module context if not private
                            newModifierFlags |= ModifierFlags.Export;
                        }
                        if (addingDeclare && !(newModifierFlags & ModifierFlags.Export) &&
                            (!context.enclosingDeclaration || !(context.enclosingDeclaration.flags & NodeFlags.Ambient)) &&
                            (isEnumDeclaration(node) || isVariableStatement(node) || isFunctionDeclaration(node) || isClassDeclaration(node) || isModuleDeclaration(node))) {
                            // Classes, namespaces, variables, enums, and functions all need `declare` modifiers to be valid in a declaration file top-level scope
                            newModifierFlags |= ModifierFlags.Ambient;
                        }
                        if ((additionalModifierFlags & ModifierFlags.Default) && (isClassDeclaration(node) || isInterfaceDeclaration(node) || isFunctionDeclaration(node))) {
                            newModifierFlags |= ModifierFlags.Default;
                        }
                        if (newModifierFlags) {
                            node = factory.updateModifiers(node, newModifierFlags | getEffectiveModifierFlags(node));
                        }
                    }
                    results.push(node);
                }

                function serializeTypeAlias(symbol: Symbol, symbolName: string, modifierFlags: ModifierFlags) {
                    const aliasType = getDeclaredTypeOfTypeAlias(symbol);
                    const typeParams = getSymbolLinks(symbol).typeParameters;
                    const typeParamDecls = map(typeParams, p => typeParameterToDeclaration(p, context));
                    const jsdocAliasDecl = find(symbol.declarations, isJSDocTypeAlias);
                    const commentText = jsdocAliasDecl ? jsdocAliasDecl.comment || jsdocAliasDecl.parent.comment : undefined;
                    const oldFlags = context.flags;
                    context.flags |= NodeBuilderFlags.InTypeAlias;
                    addResult(setSyntheticLeadingComments(
                        factory.createTypeAliasDeclaration(/*decorators*/ undefined, /*modifiers*/ undefined, getInternalSymbolName(symbol, symbolName), typeParamDecls, typeToTypeNodeHelper(aliasType, context)),
                        !commentText ? [] : [{ kind: SyntaxKind.MultiLineCommentTrivia, text: "*\n * " + commentText.replace(/\n/g, "\n * ") + "\n ", pos: -1, end: -1, hasTrailingNewLine: true }]
                    ), modifierFlags);
                    context.flags = oldFlags;
                }

                function serializeInterface(symbol: Symbol, symbolName: string, modifierFlags: ModifierFlags) {
                    const interfaceType = getDeclaredTypeOfClassOrInterface(symbol);
                    const localParams = getLocalTypeParametersOfClassOrInterfaceOrTypeAlias(symbol);
                    const typeParamDecls = map(localParams, p => typeParameterToDeclaration(p, context));
                    const baseTypes = getBaseTypes(interfaceType);
                    const baseType = length(baseTypes) ? getIntersectionType(baseTypes) : undefined;
                    const members = flatMap<Symbol, TypeElement>(getPropertiesOfType(interfaceType), p => serializePropertySymbolForInterface(p, baseType));
                    const callSignatures = serializeSignatures(SignatureKind.Call, interfaceType, baseType, SyntaxKind.CallSignature) as CallSignatureDeclaration[];
                    const constructSignatures = serializeSignatures(SignatureKind.Construct, interfaceType, baseType, SyntaxKind.ConstructSignature) as ConstructSignatureDeclaration[];
                    const indexSignatures = serializeIndexSignatures(interfaceType, baseType);

                    const heritageClauses = !length(baseTypes) ? undefined : [factory.createHeritageClause(SyntaxKind.ExtendsKeyword, mapDefined(baseTypes, b => trySerializeAsTypeReference(b, SymbolFlags.Value)))];
                    addResult(factory.createInterfaceDeclaration(
                        /*decorators*/ undefined,
                        /*modifiers*/ undefined,
                        getInternalSymbolName(symbol, symbolName),
                        typeParamDecls,
                        heritageClauses,
                        [...indexSignatures, ...constructSignatures, ...callSignatures, ...members]
                    ), modifierFlags);
                }

                function getNamespaceMembersForSerialization(symbol: Symbol) {
                    return !symbol.exports ? [] : filter(arrayFrom(symbol.exports.values()), isNamespaceMember);
                }

                function isTypeOnlyNamespace(symbol: Symbol) {
                    return every(getNamespaceMembersForSerialization(symbol), m => !(resolveSymbol(m).flags & SymbolFlags.Value));
                }

                function serializeModule(symbol: Symbol, symbolName: string, modifierFlags: ModifierFlags) {
                    const members = getNamespaceMembersForSerialization(symbol);
                    // Split NS members up by declaration - members whose parent symbol is the ns symbol vs those whose is not (but were added in later via merging)
                    const locationMap = arrayToMultiMap(members, m => m.parent && m.parent === symbol ? "real" : "merged");
                    const realMembers = locationMap.get("real") || emptyArray;
                    const mergedMembers = locationMap.get("merged") || emptyArray;
                    // TODO: `suppressNewPrivateContext` is questionable -we need to simply be emitting privates in whatever scope they were declared in, rather
                    // than whatever scope we traverse to them in. That's a bit of a complex rewrite, since we're not _actually_ tracking privates at all in advance,
                    // so we don't even have placeholders to fill in.
                    if (length(realMembers)) {
                        const localName = getInternalSymbolName(symbol, symbolName);
                        serializeAsNamespaceDeclaration(realMembers, localName, modifierFlags, !!(symbol.flags & (SymbolFlags.Function | SymbolFlags.Assignment)));
                    }
                    if (length(mergedMembers)) {
                        const containingFile = getSourceFileOfNode(context.enclosingDeclaration);
                        const localName = getInternalSymbolName(symbol, symbolName);
                        const nsBody = factory.createModuleBlock([factory.createExportDeclaration(
                            /*decorators*/ undefined,
                            /*modifiers*/ undefined,
                            /*isTypeOnly*/ false,
                            factory.createNamedExports(mapDefined(filter(mergedMembers, n => n.escapedName !== InternalSymbolName.ExportEquals), s => {
                                const name = unescapeLeadingUnderscores(s.escapedName);
                                const localName = getInternalSymbolName(s, name);
                                const aliasDecl = s.declarations && getDeclarationOfAliasSymbol(s);
                                if (containingFile && (aliasDecl ? containingFile !== getSourceFileOfNode(aliasDecl) : !some(s.declarations, d => getSourceFileOfNode(d) === containingFile))) {
                                    context.tracker?.reportNonlocalAugmentation?.(containingFile, symbol, s);
                                    return undefined;
                                }
                                const target = aliasDecl && getTargetOfAliasDeclaration(aliasDecl, /*dontRecursivelyResolve*/ true);
                                includePrivateSymbol(target || s);
                                const targetName = target ? getInternalSymbolName(target, unescapeLeadingUnderscores(target.escapedName)) : localName;
                                return factory.createExportSpecifier(name === targetName ? undefined : targetName, name);
                            }))
                        )]);
                        addResult(factory.createModuleDeclaration(
                            /*decorators*/ undefined,
                            /*modifiers*/ undefined,
                            factory.createIdentifier(localName),
                            nsBody,
                            NodeFlags.Namespace
                        ), ModifierFlags.None);
                    }
                }

                function serializeEnum(symbol: Symbol, symbolName: string, modifierFlags: ModifierFlags) {
                    addResult(factory.createEnumDeclaration(
                        /*decorators*/ undefined,
                        factory.createModifiersFromModifierFlags(isConstEnumSymbol(symbol) ? ModifierFlags.Const : 0),
                        getInternalSymbolName(symbol, symbolName),
                        map(filter(getPropertiesOfType(getTypeOfSymbol(symbol)), p => !!(p.flags & SymbolFlags.EnumMember)), p => {
                            // TODO: Handle computed names
                            // I hate that to get the initialized value we need to walk back to the declarations here; but there's no
                            // other way to get the possible const value of an enum member that I'm aware of, as the value is cached
                            // _on the declaration_, not on the declaration's symbol...
                            const initializedValue = p.declarations && p.declarations[0] && isEnumMember(p.declarations[0]) ? getConstantValue(p.declarations[0] as EnumMember) : undefined;
                            return factory.createEnumMember(unescapeLeadingUnderscores(p.escapedName), initializedValue === undefined ? undefined :
                                typeof initializedValue === "string" ? factory.createStringLiteral(initializedValue) :
                                factory.createNumericLiteral(initializedValue));
                        })
                    ), modifierFlags);
                }

                function serializeAsFunctionNamespaceMerge(type: Type, symbol: Symbol, localName: string, modifierFlags: ModifierFlags) {
                    const signatures = getSignaturesOfType(type, SignatureKind.Call);
                    for (const sig of signatures) {
                        // Each overload becomes a separate function declaration, in order
                        const decl = signatureToSignatureDeclarationHelper(sig, SyntaxKind.FunctionDeclaration, context, { name: factory.createIdentifier(localName), privateSymbolVisitor: includePrivateSymbol, bundledImports: bundled }) as FunctionDeclaration;
                        // for expressions assigned to `var`s, use the `var` as the text range
                        addResult(setTextRange(decl, sig.declaration && isVariableDeclaration(sig.declaration.parent) && sig.declaration.parent.parent || sig.declaration), modifierFlags);
                    }
                    // Module symbol emit will take care of module-y members, provided it has exports
                    if (!(symbol.flags & (SymbolFlags.ValueModule | SymbolFlags.NamespaceModule) && !!symbol.exports && !!symbol.exports.size)) {
                        const props = filter(getPropertiesOfType(type), isNamespaceMember);
                        serializeAsNamespaceDeclaration(props, localName, modifierFlags, /*suppressNewPrivateContext*/ true);
                    }
                }

                function serializeAsNamespaceDeclaration(props: readonly Symbol[], localName: string, modifierFlags: ModifierFlags, suppressNewPrivateContext: boolean) {
                    if (length(props)) {
                        const localVsRemoteMap = arrayToMultiMap(props, p =>
                            !length(p.declarations) || some(p.declarations, d =>
                                getSourceFileOfNode(d) === getSourceFileOfNode(context.enclosingDeclaration!)
                            ) ? "local" : "remote"
                        );
                        const localProps = localVsRemoteMap.get("local") || emptyArray;
                        // handle remote props first - we need to make an `import` declaration that points at the module containing each remote
                        // prop in the outermost scope (TODO: a namespace within a namespace would need to be appropriately handled by this)
                        // Example:
                        // import Foo_1 = require("./exporter");
                        // export namespace ns {
                        //     import Foo = Foo_1.Foo;
                        //     export { Foo };
                        //     export const c: number;
                        // }
                        // This is needed because in JS, statements like `const x = require("./f")` support both type and value lookup, even if they're
                        // normally just value lookup (so it functions kinda like an alias even when it's not an alias)
                        // _Usually_, we'll simply print the top-level as an alias instead of a `var` in such situations, however is is theoretically
                        // possible to encounter a situation where a type has members from both the current file and other files - in those situations,
                        // emit akin to the above would be needed.

                        // Add a namespace
                        // Create namespace as non-synthetic so it is usable as an enclosing declaration
                        let fakespace = parseNodeFactory.createModuleDeclaration(/*decorators*/ undefined, /*modifiers*/ undefined, factory.createIdentifier(localName), factory.createModuleBlock([]), NodeFlags.Namespace);
                        setParent(fakespace, enclosingDeclaration as SourceFile | NamespaceDeclaration);
                        fakespace.locals = createSymbolTable(props);
                        fakespace.symbol = props[0].parent!;

                        const oldResults = results;
                        results = [];
                        const oldAddingDeclare = addingDeclare;
                        addingDeclare = false;
                        const subcontext = { ...context, enclosingDeclaration: fakespace };
                        const oldContext = context;
                        context = subcontext;
                        // TODO: implement handling for the localVsRemoteMap.get("remote") - should be difficult to trigger (see comment above), as only interesting cross-file js merges should make this possible
                        visitSymbolTable(createSymbolTable(localProps), suppressNewPrivateContext, /*propertyAsAlias*/ true);
                        context = oldContext;
                        addingDeclare = oldAddingDeclare;
                        const declarations = results;
                        results = oldResults;
                        // replace namespace with synthetic version
                        const defaultReplaced = map(declarations, d => isExportAssignment(d) && !d.isExportEquals && isIdentifier(d.expression) ? factory.createExportDeclaration(
                            /*decorators*/ undefined,
                            /*modifiers*/ undefined,
                            /*isTypeOnly*/ false,
                            factory.createNamedExports([factory.createExportSpecifier(d.expression, factory.createIdentifier(InternalSymbolName.Default))])
                        ) : d);
                        const exportModifierStripped = every(defaultReplaced, d => hasSyntacticModifier(d, ModifierFlags.Export)) ? map(defaultReplaced, removeExportModifier) : defaultReplaced;
                        fakespace = factory.updateModuleDeclaration(
                            fakespace,
                            fakespace.decorators,
                            fakespace.modifiers,
                            fakespace.name,
                            factory.createModuleBlock(exportModifierStripped));
                        addResult(fakespace, modifierFlags); // namespaces can never be default exported
                    }
                }

                function isNamespaceMember(p: Symbol) {
                    return !!(p.flags & (SymbolFlags.Type | SymbolFlags.Namespace | SymbolFlags.Alias)) ||
                        !(p.flags & SymbolFlags.Prototype || p.escapedName === "prototype" || p.valueDeclaration && isClassLike(p.valueDeclaration.parent));
                }

                function serializeAsClass(symbol: Symbol, localName: string, modifierFlags: ModifierFlags) {
                    const localParams = getLocalTypeParametersOfClassOrInterfaceOrTypeAlias(symbol);
                    const typeParamDecls = map(localParams, p => typeParameterToDeclaration(p, context));
                    const classType = getDeclaredTypeOfClassOrInterface(symbol);
                    const baseTypes = getBaseTypes(classType);
                    const implementsExpressions = mapDefined(getImplementsTypes(classType), serializeImplementedType);
                    const staticType = getTypeOfSymbol(symbol);
                    const isClass = !!staticType.symbol?.valueDeclaration && isClassLike(staticType.symbol.valueDeclaration);
                    const staticBaseType = isClass
                        ? getBaseConstructorTypeOfClass(staticType as InterfaceType)
                        : anyType;
                    const heritageClauses = [
                        ...!length(baseTypes) ? [] : [factory.createHeritageClause(SyntaxKind.ExtendsKeyword, map(baseTypes, b => serializeBaseType(b, staticBaseType, localName)))],
                        ...!length(implementsExpressions) ? [] : [factory.createHeritageClause(SyntaxKind.ImplementsKeyword, implementsExpressions)]
                    ];
                    const symbolProps = getNonInterhitedProperties(classType, baseTypes, getPropertiesOfType(classType));
                    const publicSymbolProps = filter(symbolProps, s => {
                        // `valueDeclaration` could be undefined if inherited from
                        // a union/intersection base type, but inherited properties
                        // don't matter here.
                        const valueDecl = s.valueDeclaration;
                        return valueDecl && !(isNamedDeclaration(valueDecl) && isPrivateIdentifier(valueDecl.name));
                    });
                    const hasPrivateIdentifier = some(symbolProps, s => {
                        // `valueDeclaration` could be undefined if inherited from
                        // a union/intersection base type, but inherited properties
                        // don't matter here.
                        const valueDecl = s.valueDeclaration;
                        return valueDecl && isNamedDeclaration(valueDecl) && isPrivateIdentifier(valueDecl.name);
                    });
                    // Boil down all private properties into a single one.
                    const privateProperties = hasPrivateIdentifier ?
                        [factory.createPropertyDeclaration(
                            /*decorators*/ undefined,
                            /*modifiers*/ undefined,
                            factory.createPrivateIdentifier("#private"),
                            /*questionOrExclamationToken*/ undefined,
                            /*type*/ undefined,
                            /*initializer*/ undefined,
                        )] :
                        emptyArray;
                    const publicProperties = flatMap<Symbol, ClassElement>(publicSymbolProps, p => serializePropertySymbolForClass(p, /*isStatic*/ false, baseTypes[0]));
                    // Consider static members empty if symbol also has function or module meaning - function namespacey emit will handle statics
                    const staticMembers = flatMap(
                        filter(getPropertiesOfType(staticType), p => !(p.flags & SymbolFlags.Prototype) && p.escapedName !== "prototype" && !isNamespaceMember(p)),
                        p => serializePropertySymbolForClass(p, /*isStatic*/ true, staticBaseType));
                    // When we encounter an `X.prototype.y` assignment in a JS file, we bind `X` as a class regardless as to whether
                    // the value is ever initialized with a class or function-like value. For cases where `X` could never be
                    // created via `new`, we will inject a `private constructor()` declaration to indicate it is not createable.
                    const isNonConstructableClassLikeInJsFile =
                        !isClass &&
                        !!symbol.valueDeclaration &&
                        isInJSFile(symbol.valueDeclaration) &&
                        !some(getSignaturesOfType(staticType, SignatureKind.Construct));
                    const constructors = isNonConstructableClassLikeInJsFile ?
                        [factory.createConstructorDeclaration(/*decorators*/ undefined, factory.createModifiersFromModifierFlags(ModifierFlags.Private), [], /*body*/ undefined)] :
                        serializeSignatures(SignatureKind.Construct, staticType, baseTypes[0], SyntaxKind.Constructor) as ConstructorDeclaration[];
                    const indexSignatures = serializeIndexSignatures(classType, baseTypes[0]);
                    addResult(setTextRange(factory.createClassDeclaration(
                        /*decorators*/ undefined,
                        /*modifiers*/ undefined,
                        localName,
                        typeParamDecls,
                        heritageClauses,
                        [...indexSignatures, ...staticMembers, ...constructors, ...publicProperties, ...privateProperties]
                    ), symbol.declarations && filter(symbol.declarations, d => isClassDeclaration(d) || isClassExpression(d))[0]), modifierFlags);
                }

                function serializeAsAlias(symbol: Symbol, localName: string, modifierFlags: ModifierFlags) {
                    // synthesize an alias, eg `export { symbolName as Name }`
                    // need to mark the alias `symbol` points at
                    // as something we need to serialize as a private declaration as well
                    const node = getDeclarationOfAliasSymbol(symbol);
                    if (!node) return Debug.fail();
                    const target = getMergedSymbol(getTargetOfAliasDeclaration(node, /*dontRecursivelyResolve*/ true));
                    if (!target) {
                        return;
                    }
                    let verbatimTargetName = unescapeLeadingUnderscores(target.escapedName);
                    if (verbatimTargetName === InternalSymbolName.ExportEquals && (compilerOptions.esModuleInterop || compilerOptions.allowSyntheticDefaultImports)) {
                        // target refers to an `export=` symbol that was hoisted into a synthetic default - rename here to match
                        verbatimTargetName = InternalSymbolName.Default;
                    }
                    const targetName = getInternalSymbolName(target, verbatimTargetName);
                    includePrivateSymbol(target); // the target may be within the same scope - attempt to serialize it first
                    switch (node.kind) {
                        case SyntaxKind.ImportEqualsDeclaration:
                            // Could be a local `import localName = ns.member` or
                            // an external `import localName = require("whatever")`
                            const isLocalImport = !(target.flags & SymbolFlags.ValueModule);
                            addResult(factory.createImportEqualsDeclaration(
                                /*decorators*/ undefined,
                                /*modifiers*/ undefined,
                                factory.createIdentifier(localName),
                                isLocalImport
                                    ? symbolToName(target, context, SymbolFlags.All, /*expectsIdentifier*/ false)
                                    : factory.createExternalModuleReference(factory.createStringLiteral(getSpecifierForModuleSymbol(symbol, context)))
                            ), isLocalImport ? modifierFlags : ModifierFlags.None);
                            break;
                        case SyntaxKind.NamespaceExportDeclaration:
                            // export as namespace foo
                            // TODO: Not part of a file's local or export symbol tables
                            // Is bound into file.symbol.globalExports instead, which we don't currently traverse
                            addResult(factory.createNamespaceExportDeclaration(idText((node as NamespaceExportDeclaration).name)), ModifierFlags.None);
                            break;
                        case SyntaxKind.ImportClause:
                            addResult(factory.createImportDeclaration(
                                /*decorators*/ undefined,
                                /*modifiers*/ undefined,
                                factory.createImportClause(/*isTypeOnly*/ false, factory.createIdentifier(localName), /*namedBindings*/ undefined),
                                // We use `target.parent || target` below as `target.parent` is unset when the target is a module which has been export assigned
                                // And then made into a default by the `esModuleInterop` or `allowSyntheticDefaultImports` flag
                                // In such cases, the `target` refers to the module itself already
                                factory.createStringLiteral(getSpecifierForModuleSymbol(target.parent || target, context))
                            ), ModifierFlags.None);
                            break;
                        case SyntaxKind.NamespaceImport:
                            addResult(factory.createImportDeclaration(
                                /*decorators*/ undefined,
                                /*modifiers*/ undefined,
                                factory.createImportClause(/*isTypeOnly*/ false, /*importClause*/ undefined, factory.createNamespaceImport(factory.createIdentifier(localName))),
                                factory.createStringLiteral(getSpecifierForModuleSymbol(target, context))
                            ), ModifierFlags.None);
                            break;
                        case SyntaxKind.NamespaceExport:
                            addResult(factory.createExportDeclaration(
                                /*decorators*/ undefined,
                                /*modifiers*/ undefined,
                                /*isTypeOnly*/ false,
                                factory.createNamespaceExport(factory.createIdentifier(localName)),
                                factory.createStringLiteral(getSpecifierForModuleSymbol(target, context))
                            ), ModifierFlags.None);
                            break;
                        case SyntaxKind.ImportSpecifier:
                            addResult(factory.createImportDeclaration(
                                /*decorators*/ undefined,
                                /*modifiers*/ undefined,
                                factory.createImportClause(
                                    /*isTypeOnly*/ false,
                                    /*importClause*/ undefined,
                                    factory.createNamedImports([
                                        factory.createImportSpecifier(
                                            localName !== verbatimTargetName ? factory.createIdentifier(verbatimTargetName) : undefined,
                                            factory.createIdentifier(localName)
                                        )
                                    ])),
                                factory.createStringLiteral(getSpecifierForModuleSymbol(target.parent || target, context))
                            ), ModifierFlags.None);
                            break;
                        case SyntaxKind.ExportSpecifier:
                            // does not use localName because the symbol name in this case refers to the name in the exports table,
                            // which we must exactly preserve
                            const specifier = (node.parent.parent as ExportDeclaration).moduleSpecifier;
                            // targetName is only used when the target is local, as otherwise the target is an alias that points at
                            // another file
                            serializeExportSpecifier(
                                unescapeLeadingUnderscores(symbol.escapedName),
                                specifier ? verbatimTargetName : targetName,
                                specifier && isStringLiteralLike(specifier) ? factory.createStringLiteral(specifier.text) : undefined
                            );
                            break;
                        case SyntaxKind.ExportAssignment:
                            serializeMaybeAliasAssignment(symbol);
                            break;
                        case SyntaxKind.BinaryExpression:
                        case SyntaxKind.PropertyAccessExpression:
                            // Could be best encoded as though an export specifier or as though an export assignment
                            // If name is default or export=, do an export assignment
                            // Otherwise do an export specifier
                            if (symbol.escapedName === InternalSymbolName.Default || symbol.escapedName === InternalSymbolName.ExportEquals) {
                                serializeMaybeAliasAssignment(symbol);
                            }
                            else {
                                serializeExportSpecifier(localName, targetName);
                            }
                            break;
                        default:
                            return Debug.failBadSyntaxKind(node, "Unhandled alias declaration kind in symbol serializer!");
                    }
                }

                function serializeExportSpecifier(localName: string, targetName: string, specifier?: Expression) {
                    addResult(factory.createExportDeclaration(
                        /*decorators*/ undefined,
                        /*modifiers*/ undefined,
                        /*isTypeOnly*/ false,
                        factory.createNamedExports([factory.createExportSpecifier(localName !== targetName ? targetName : undefined, localName)]),
                        specifier
                    ), ModifierFlags.None);
                }

                /**
                 * Returns `true` if an export assignment or declaration was produced for the symbol
                 */
                function serializeMaybeAliasAssignment(symbol: Symbol): boolean {
                    if (symbol.flags & SymbolFlags.Prototype) {
                        return false;
                    }
                    const name = unescapeLeadingUnderscores(symbol.escapedName);
                    const isExportEquals = name === InternalSymbolName.ExportEquals;
                    const isDefault = name === InternalSymbolName.Default;
                    const isExportAssignment = isExportEquals || isDefault;
                    // synthesize export = ref
                    // ref should refer to either be a locally scoped symbol which we need to emit, or
                    // a reference to another namespace/module which we may need to emit an `import` statement for
                    const aliasDecl = symbol.declarations && getDeclarationOfAliasSymbol(symbol);
                    // serialize what the alias points to, preserve the declaration's initializer
                    const target = aliasDecl && getTargetOfAliasDeclaration(aliasDecl, /*dontRecursivelyResolve*/ true);
                    // If the target resolves and resolves to a thing defined in this file, emit as an alias, otherwise emit as a const
                    if (target && length(target.declarations) && some(target.declarations, d => getSourceFileOfNode(d) === getSourceFileOfNode(enclosingDeclaration))) {
                        // In case `target` refers to a namespace member, look at the declaration and serialize the leftmost symbol in it
                        // eg, `namespace A { export class B {} }; exports = A.B;`
                        // Technically, this is all that's required in the case where the assignment is an entity name expression
                        const expr = isExportAssignment ? getExportAssignmentExpression(aliasDecl as ExportAssignment | BinaryExpression) : getPropertyAssignmentAliasLikeExpression(aliasDecl as ShorthandPropertyAssignment | PropertyAssignment | PropertyAccessExpression);
                        const first = isEntityNameExpression(expr) ? getFirstNonModuleExportsIdentifier(expr) : undefined;
                        const referenced = first && resolveEntityName(first, SymbolFlags.All, /*ignoreErrors*/ true, /*dontResolveAlias*/ true, enclosingDeclaration);
                        if (referenced || target) {
                            includePrivateSymbol(referenced || target);
                        }

                        // We disable the context's symbol tracker for the duration of this name serialization
                        // as, by virtue of being here, the name is required to print something, and we don't want to
                        // issue a visibility error on it. Only anonymous classes that an alias points at _would_ issue
                        // a visibility error here (as they're not visible within any scope), but we want to hoist them
                        // into the containing scope anyway, so we want to skip the visibility checks.
                        const oldTrack = context.tracker.trackSymbol;
                        context.tracker.trackSymbol = noop;
                        if (isExportAssignment) {
                            results.push(factory.createExportAssignment(
                                /*decorators*/ undefined,
                                /*modifiers*/ undefined,
                                isExportEquals,
                                symbolToExpression(target, context, SymbolFlags.All)
                            ));
                        }
                        else {
                            if (first === expr) {
                                // serialize as `export {target as name}`
                                serializeExportSpecifier(name, idText(first));
                            }
                            else if (isClassExpression(expr)) {
                                serializeExportSpecifier(name, getInternalSymbolName(target, symbolName(target)));
                            }
                            else {
                                // serialize as `import _Ref = t.arg.et; export { _Ref as name }`
                                const varName = getUnusedName(name, symbol);
                                addResult(factory.createImportEqualsDeclaration(
                                    /*decorators*/ undefined,
                                    /*modifiers*/ undefined,
                                    factory.createIdentifier(varName),
                                    symbolToName(target, context, SymbolFlags.All, /*expectsIdentifier*/ false)
                                ), ModifierFlags.None);
                                serializeExportSpecifier(name, varName);
                            }
                        }
                        context.tracker.trackSymbol = oldTrack;
                        return true;
                    }
                    else {
                        // serialize as an anonymous property declaration
                        const varName = getUnusedName(name, symbol);
                        // We have to use `getWidenedType` here since the object within a json file is unwidened within the file
                        // (Unwidened types can only exist in expression contexts and should never be serialized)
                        const typeToSerialize = getWidenedType(getTypeOfSymbol(getMergedSymbol(symbol)));
                        if (isTypeRepresentableAsFunctionNamespaceMerge(typeToSerialize, symbol)) {
                            // If there are no index signatures and `typeToSerialize` is an object type, emit as a namespace instead of a const
                            serializeAsFunctionNamespaceMerge(typeToSerialize, symbol, varName, isExportAssignment ? ModifierFlags.None : ModifierFlags.Export);
                        }
                        else {
                            const statement = factory.createVariableStatement(/*modifiers*/ undefined, factory.createVariableDeclarationList([
                                factory.createVariableDeclaration(varName, /*exclamationToken*/ undefined, serializeTypeForDeclaration(context, typeToSerialize, symbol, enclosingDeclaration, includePrivateSymbol, bundled))
                            ], NodeFlags.Const));
                            addResult(statement, name === varName ? ModifierFlags.Export : ModifierFlags.None);
                        }
                        if (isExportAssignment) {
                            results.push(factory.createExportAssignment(
                                /*decorators*/ undefined,
                                /*modifiers*/ undefined,
                                isExportEquals,
                                factory.createIdentifier(varName)
                            ));
                            return true;
                        }
                        else if (name !== varName) {
                            serializeExportSpecifier(name, varName);
                            return true;
                        }
                        return false;
                    }
                }

                function isTypeRepresentableAsFunctionNamespaceMerge(typeToSerialize: Type, hostSymbol: Symbol) {
                    // Only object types which are not constructable, or indexable, whose members all come from the
                    // context source file, and whose property names are all valid identifiers and not late-bound, _and_
                    // whose input is not type annotated (if the input symbol has an annotation we can reuse, we should prefer it)
                    const ctxSrc = getSourceFileOfNode(context.enclosingDeclaration);
                    return getObjectFlags(typeToSerialize) & (ObjectFlags.Anonymous | ObjectFlags.Mapped) &&
                    !getIndexInfoOfType(typeToSerialize, IndexKind.String) &&
                    !getIndexInfoOfType(typeToSerialize, IndexKind.Number) &&
                    !!(length(getPropertiesOfType(typeToSerialize)) || length(getSignaturesOfType(typeToSerialize, SignatureKind.Call))) &&
                    !length(getSignaturesOfType(typeToSerialize, SignatureKind.Construct)) && // TODO: could probably serialize as function + ns + class, now that that's OK
                    !getDeclarationWithTypeAnnotation(hostSymbol, enclosingDeclaration) &&
                    !(typeToSerialize.symbol && some(typeToSerialize.symbol.declarations, d => getSourceFileOfNode(d) !== ctxSrc)) &&
                    !some(getPropertiesOfType(typeToSerialize), p => isLateBoundName(p.escapedName)) &&
                    !some(getPropertiesOfType(typeToSerialize), p => some(p.declarations, d => getSourceFileOfNode(d) !== ctxSrc)) &&
                    every(getPropertiesOfType(typeToSerialize), p => isIdentifierText(symbolName(p), languageVersion));
                }

                function makeSerializePropertySymbol<T extends Node>(createProperty: (
                    decorators: readonly Decorator[] | undefined,
                    modifiers: readonly Modifier[] | undefined,
                    name: string | PropertyName,
                    questionOrExclamationToken: QuestionToken | undefined,
                    type: TypeNode | undefined,
                    initializer: Expression | undefined
                ) => T, methodKind: SignatureDeclaration["kind"], useAccessors: true): (p: Symbol, isStatic: boolean, baseType: Type | undefined) => (T | AccessorDeclaration | (T | AccessorDeclaration)[]);
                function makeSerializePropertySymbol<T extends Node>(createProperty: (
                    decorators: readonly Decorator[] | undefined,
                    modifiers: readonly Modifier[] | undefined,
                    name: string | PropertyName,
                    questionOrExclamationToken: QuestionToken | undefined,
                    type: TypeNode | undefined,
                    initializer: Expression | undefined
                ) => T, methodKind: SignatureDeclaration["kind"], useAccessors: false): (p: Symbol, isStatic: boolean, baseType: Type | undefined) => (T | T[]);
                function makeSerializePropertySymbol<T extends Node>(createProperty: (
                    decorators: readonly Decorator[] | undefined,
                    modifiers: readonly Modifier[] | undefined,
                    name: string | PropertyName,
                    questionOrExclamationToken: QuestionToken | undefined,
                    type: TypeNode | undefined,
                    initializer: Expression | undefined
                ) => T, methodKind: SignatureDeclaration["kind"], useAccessors: boolean): (p: Symbol, isStatic: boolean, baseType: Type | undefined) => (T | AccessorDeclaration | (T | AccessorDeclaration)[]) {
                    return function serializePropertySymbol(p: Symbol, isStatic: boolean, baseType: Type | undefined): (T | AccessorDeclaration | (T | AccessorDeclaration)[]) {
                        const modifierFlags = getDeclarationModifierFlagsFromSymbol(p);
                        const isPrivate = !!(modifierFlags & ModifierFlags.Private);
                        if (isStatic && (p.flags & (SymbolFlags.Type | SymbolFlags.Namespace | SymbolFlags.Alias))) {
                            // Only value-only-meaning symbols can be correctly encoded as class statics, type/namespace/alias meaning symbols
                            // need to be merged namespace members
                            return [];
                        }
                        if (p.flags & SymbolFlags.Prototype ||
                            (baseType && getPropertyOfType(baseType, p.escapedName)
                             && isReadonlySymbol(getPropertyOfType(baseType, p.escapedName)!) === isReadonlySymbol(p)
                             && (p.flags & SymbolFlags.Optional) === (getPropertyOfType(baseType, p.escapedName)!.flags & SymbolFlags.Optional)
                             && isTypeIdenticalTo(getTypeOfSymbol(p), getTypeOfPropertyOfType(baseType, p.escapedName)!))) {
                            return [];
                        }
                        const flag = (modifierFlags & ~ModifierFlags.Async) | (isStatic ? ModifierFlags.Static : 0);
                        const name = getPropertyNameNodeForSymbol(p, context);
                        const firstPropertyLikeDecl = find(p.declarations, or(isPropertyDeclaration, isAccessor, isVariableDeclaration, isPropertySignature, isBinaryExpression, isPropertyAccessExpression));
                        if (p.flags & SymbolFlags.Accessor && useAccessors) {
                            const result: AccessorDeclaration[] = [];
                            if (p.flags & SymbolFlags.SetAccessor) {
                                result.push(setTextRange(factory.createSetAccessorDeclaration(
                                    /*decorators*/ undefined,
                                    factory.createModifiersFromModifierFlags(flag),
                                    name,
                                    [factory.createParameterDeclaration(
                                        /*decorators*/ undefined,
                                        /*modifiers*/ undefined,
                                        /*dotDotDotToken*/ undefined,
                                        "arg",
                                        /*questionToken*/ undefined,
                                        isPrivate ? undefined : serializeTypeForDeclaration(context, getTypeOfSymbol(p), p, enclosingDeclaration, includePrivateSymbol, bundled)
                                    )],
                                    /*body*/ undefined
                                ), find(p.declarations, isSetAccessor) || firstPropertyLikeDecl));
                            }
                            if (p.flags & SymbolFlags.GetAccessor) {
                                const isPrivate = modifierFlags & ModifierFlags.Private;
                                result.push(setTextRange(factory.createGetAccessorDeclaration(
                                    /*decorators*/ undefined,
                                    factory.createModifiersFromModifierFlags(flag),
                                    name,
                                    [],
                                    isPrivate ? undefined : serializeTypeForDeclaration(context, getTypeOfSymbol(p), p, enclosingDeclaration, includePrivateSymbol, bundled),
                                    /*body*/ undefined
                                ), find(p.declarations, isGetAccessor) || firstPropertyLikeDecl));
                            }
                            return result;
                        }
                        // This is an else/if as accessors and properties can't merge in TS, but might in JS
                        // If this happens, we assume the accessor takes priority, as it imposes more constraints
                        else if (p.flags & (SymbolFlags.Property | SymbolFlags.Variable)) {
                            return setTextRange(createProperty(
                                /*decorators*/ undefined,
                                factory.createModifiersFromModifierFlags((isReadonlySymbol(p) ? ModifierFlags.Readonly : 0) | flag),
                                name,
                                p.flags & SymbolFlags.Optional ? factory.createToken(SyntaxKind.QuestionToken) : undefined,
                                isPrivate ? undefined : serializeTypeForDeclaration(context, getTypeOfSymbol(p), p, enclosingDeclaration, includePrivateSymbol, bundled),
                                // TODO: https://github.com/microsoft/TypeScript/pull/32372#discussion_r328386357
                                // interface members can't have initializers, however class members _can_
                                /*initializer*/ undefined
                            ), find(p.declarations, or(isPropertyDeclaration, isVariableDeclaration)) || firstPropertyLikeDecl);
                        }
                        if (p.flags & (SymbolFlags.Method | SymbolFlags.Function)) {
                            const type = getTypeOfSymbol(p);
                            const signatures = getSignaturesOfType(type, SignatureKind.Call);
                            if (flag & ModifierFlags.Private) {
                                return setTextRange(createProperty(
                                    /*decorators*/ undefined,
                                    factory.createModifiersFromModifierFlags((isReadonlySymbol(p) ? ModifierFlags.Readonly : 0) | flag),
                                    name,
                                    p.flags & SymbolFlags.Optional ? factory.createToken(SyntaxKind.QuestionToken) : undefined,
                                    /*type*/ undefined,
                                    /*initializer*/ undefined
                                ), find(p.declarations, isFunctionLikeDeclaration) || signatures[0] && signatures[0].declaration || p.declarations[0]);
                            }

                            const results = [];
                            for (const sig of signatures) {
                                // Each overload becomes a separate method declaration, in order
                                const decl = signatureToSignatureDeclarationHelper(
                                    sig,
                                    methodKind,
                                    context,
                                    {
                                        name,
                                        questionToken: p.flags & SymbolFlags.Optional ? factory.createToken(SyntaxKind.QuestionToken) : undefined,
                                        modifiers: flag ? factory.createModifiersFromModifierFlags(flag) : undefined
                                    }
                                );
                                results.push(setTextRange(decl, sig.declaration));
                            }
                            return results as unknown as T[];
                        }
                        // The `Constructor`'s symbol isn't in the class's properties lists, obviously, since it's a signature on the static
                        return Debug.fail(`Unhandled class member kind! ${(p as any).__debugFlags || p.flags}`);
                    };
                }

                function serializePropertySymbolForInterface(p: Symbol, baseType: Type | undefined) {
                    return serializePropertySymbolForInterfaceWorker(p, /*isStatic*/ false, baseType);
                }

                function serializeSignatures(kind: SignatureKind, input: Type, baseType: Type | undefined, outputKind: SignatureDeclaration["kind"]) {
                    const signatures = getSignaturesOfType(input, kind);
                    if (kind === SignatureKind.Construct) {
                        if (!baseType && every(signatures, s => length(s.parameters) === 0)) {
                            return []; // No base type, every constructor is empty - elide the extraneous `constructor()`
                        }
                        if (baseType) {
                            // If there is a base type, if every signature in the class is identical to a signature in the baseType, elide all the declarations
                            const baseSigs = getSignaturesOfType(baseType, SignatureKind.Construct);
                            if (!length(baseSigs) && every(signatures, s => length(s.parameters) === 0)) {
                                return []; // Base had no explicit signatures, if all our signatures are also implicit, return an empty list
                            }
                            if (baseSigs.length === signatures.length) {
                                let failed = false;
                                for (let i = 0; i < baseSigs.length; i++) {
                                    if (!compareSignaturesIdentical(signatures[i], baseSigs[i], /*partialMatch*/ false, /*ignoreThisTypes*/ false, /*ignoreReturnTypes*/ true, compareTypesIdentical)) {
                                        failed = true;
                                        break;
                                    }
                                }
                                if (!failed) {
                                    return []; // Every signature was identical - elide constructor list as it is inherited
                                }
                            }
                        }
                        let privateProtected: ModifierFlags = 0;
                        for (const s of signatures) {
                            if (s.declaration) {
                                privateProtected |= getSelectedEffectiveModifierFlags(s.declaration, ModifierFlags.Private | ModifierFlags.Protected);
                            }
                        }
                        if (privateProtected) {
                            return [setTextRange(factory.createConstructorDeclaration(
                                /*decorators*/ undefined,
                                factory.createModifiersFromModifierFlags(privateProtected),
                                /*parameters*/ [],
                                /*body*/ undefined,
                            ), signatures[0].declaration)];
                        }
                    }

                    const results = [];
                    for (const sig of signatures) {
                        // Each overload becomes a separate constructor declaration, in order
                        const decl = signatureToSignatureDeclarationHelper(sig, outputKind, context);
                        results.push(setTextRange(decl, sig.declaration));
                    }
                    return results;
                }

                function serializeIndexSignatures(input: Type, baseType: Type | undefined) {
                    const results: IndexSignatureDeclaration[] = [];
                    for (const type of [IndexKind.String, IndexKind.Number]) {
                        const info = getIndexInfoOfType(input, type);
                        if (info) {
                            if (baseType) {
                                const baseInfo = getIndexInfoOfType(baseType, type);
                                if (baseInfo) {
                                    if (isTypeIdenticalTo(info.type, baseInfo.type)) {
                                        continue; // elide identical index signatures
                                    }
                                }
                            }
                            results.push(indexInfoToIndexSignatureDeclarationHelper(info, type, context, /*typeNode*/ undefined));
                        }
                    }
                    return results;
                }

                function serializeBaseType(t: Type, staticType: Type, rootName: string) {
                    const ref = trySerializeAsTypeReference(t, SymbolFlags.Value);
                    if (ref) {
                        return ref;
                    }
                    const tempName = getUnusedName(`${rootName}_base`);
                    const statement = factory.createVariableStatement(/*modifiers*/ undefined, factory.createVariableDeclarationList([
                        factory.createVariableDeclaration(tempName, /*exclamationToken*/ undefined, typeToTypeNodeHelper(staticType, context))
                    ], NodeFlags.Const));
                    addResult(statement, ModifierFlags.None);
                    return factory.createExpressionWithTypeArguments(factory.createIdentifier(tempName), /*typeArgs*/ undefined);
                }

                function trySerializeAsTypeReference(t: Type, flags: SymbolFlags) {
                    let typeArgs: TypeNode[] | undefined;
                    let reference: Expression | undefined;

                    // We don't use `isValueSymbolAccessible` below. since that considers alternative containers (like modules)
                    // which we can't write out in a syntactically valid way as an expression
                    if ((t as TypeReference).target && isSymbolAccessibleByFlags((t as TypeReference).target.symbol, enclosingDeclaration, flags)) {
                        typeArgs = map(getTypeArguments(t as TypeReference), t => typeToTypeNodeHelper(t, context));
                        reference = symbolToExpression((t as TypeReference).target.symbol, context, SymbolFlags.Type);
                    }
                    else if (t.symbol && isSymbolAccessibleByFlags(t.symbol, enclosingDeclaration, flags)) {
                        reference = symbolToExpression(t.symbol, context, SymbolFlags.Type);
                    }
                    if (reference) {
                        return factory.createExpressionWithTypeArguments(reference, typeArgs);
                    }
                }

                function serializeImplementedType(t: Type) {
                    const ref = trySerializeAsTypeReference(t, SymbolFlags.Type);
                    if (ref) {
                        return ref;
                    }
                    if (t.symbol) {
                        return factory.createExpressionWithTypeArguments(symbolToExpression(t.symbol, context, SymbolFlags.Type), /*typeArgs*/ undefined);
                    }
                }

                function getUnusedName(input: string, symbol?: Symbol): string {
                    const id = symbol ? getSymbolId(symbol) : undefined;
                    if (id) {
                        if (context.remappedSymbolNames!.has(id)) {
                            return context.remappedSymbolNames!.get(id)!;
                        }
                    }
                    if (symbol) {
                        input = getNameCandidateWorker(symbol, input);
                    }
                    let i = 0;
                    const original = input;
                    while (context.usedSymbolNames?.has(input)) {
                        i++;
                        input = `${original}_${i}`;
                    }
                    context.usedSymbolNames?.add(input);
                    if (id) {
                        context.remappedSymbolNames!.set(id, input);
                    }
                    return input;
                }

                function getNameCandidateWorker(symbol: Symbol, localName: string) {
                    if (localName === InternalSymbolName.Default || localName === InternalSymbolName.Class || localName === InternalSymbolName.Function) {
                        const flags = context.flags;
                        context.flags |= NodeBuilderFlags.InInitialEntityName;
                        const nameCandidate = getNameOfSymbolAsWritten(symbol, context);
                        context.flags = flags;
                        localName = nameCandidate.length > 0 && isSingleOrDoubleQuote(nameCandidate.charCodeAt(0)) ? stripQuotes(nameCandidate) : nameCandidate;
                    }
                    if (localName === InternalSymbolName.Default) {
                        localName = "_default";
                    }
                    else if (localName === InternalSymbolName.ExportEquals) {
                        localName = "_exports";
                    }
                    localName = isIdentifierText(localName, languageVersion) && !isStringANonContextualKeyword(localName) ? localName : "_" + localName.replace(/[^a-zA-Z0-9]/g, "_");
                    return localName;
                }

                function getInternalSymbolName(symbol: Symbol, localName: string) {
                    const id = getSymbolId(symbol);
                    if (context.remappedSymbolNames!.has(id)) {
                        return context.remappedSymbolNames!.get(id)!;
                    }
                    localName = getNameCandidateWorker(symbol, localName);
                    // The result of this is going to be used as the symbol's name - lock it in, so `getUnusedName` will also pick it up
                    context.remappedSymbolNames!.set(id, localName);
                    return localName;
                }
            }
        }

        function typePredicateToString(typePredicate: TypePredicate, enclosingDeclaration?: Node, flags: TypeFormatFlags = TypeFormatFlags.UseAliasDefinedOutsideCurrentScope, writer?: EmitTextWriter): string {
            return writer ? typePredicateToStringWorker(writer).getText() : usingSingleLineStringWriter(typePredicateToStringWorker);

            function typePredicateToStringWorker(writer: EmitTextWriter) {
                const predicate = factory.createTypePredicateNode(
                    typePredicate.kind === TypePredicateKind.AssertsThis || typePredicate.kind === TypePredicateKind.AssertsIdentifier ? factory.createToken(SyntaxKind.AssertsKeyword) : undefined,
                    typePredicate.kind === TypePredicateKind.Identifier || typePredicate.kind === TypePredicateKind.AssertsIdentifier ? factory.createIdentifier(typePredicate.parameterName) : factory.createThisTypeNode(),
                    typePredicate.type && nodeBuilder.typeToTypeNode(typePredicate.type, enclosingDeclaration, toNodeBuilderFlags(flags) | NodeBuilderFlags.IgnoreErrors | NodeBuilderFlags.WriteTypeParametersInQualifiedName)! // TODO: GH#18217
                );
                const printer = createPrinter({ removeComments: true });
                const sourceFile = enclosingDeclaration && getSourceFileOfNode(enclosingDeclaration);
                printer.writeNode(EmitHint.Unspecified, predicate, /*sourceFile*/ sourceFile, writer);
                return writer;
            }
        }

        function formatUnionTypes(types: readonly Type[]): Type[] {
            const result: Type[] = [];
            let flags: TypeFlags = 0;
            for (let i = 0; i < types.length; i++) {
                const t = types[i];
                flags |= t.flags;
                if (!(t.flags & TypeFlags.Nullable)) {
                    if (t.flags & (TypeFlags.BooleanLiteral | TypeFlags.EnumLiteral)) {
                        const baseType = t.flags & TypeFlags.BooleanLiteral ? booleanType : getBaseTypeOfEnumLiteralType(<LiteralType>t);
                        if (baseType.flags & TypeFlags.Union) {
                            const count = (<UnionType>baseType).types.length;
                            if (i + count <= types.length && getRegularTypeOfLiteralType(types[i + count - 1]) === getRegularTypeOfLiteralType((<UnionType>baseType).types[count - 1])) {
                                result.push(baseType);
                                i += count - 1;
                                continue;
                            }
                        }
                    }
                    result.push(t);
                }
            }
            if (flags & TypeFlags.Null) result.push(nullType);
            if (flags & TypeFlags.Undefined) result.push(undefinedType);
            return result || types;
        }

        function visibilityToString(flags: ModifierFlags): string | undefined {
            if (flags === ModifierFlags.Private) {
                return "private";
            }
            if (flags === ModifierFlags.Protected) {
                return "protected";
            }
            return "public";
        }

        function getTypeAliasForTypeLiteral(type: Type): Symbol | undefined {
            if (type.symbol && type.symbol.flags & SymbolFlags.TypeLiteral) {
                const node = walkUpParenthesizedTypes(type.symbol.declarations[0].parent);
                if (node.kind === SyntaxKind.TypeAliasDeclaration) {
                    return getSymbolOfNode(node);
                }
            }
            return undefined;
        }

        function isTopLevelInExternalModuleAugmentation(node: Node): boolean {
            return node && node.parent &&
                node.parent.kind === SyntaxKind.ModuleBlock &&
                isExternalModuleAugmentation(node.parent.parent);
        }

        interface NodeBuilderContext {
            enclosingDeclaration: Node | undefined;
            flags: NodeBuilderFlags;
            tracker: SymbolTracker;

            // State
            encounteredError: boolean;
            visitedTypes: Set<number> | undefined;
            symbolDepth: ESMap<string, number> | undefined;
            inferTypeParameters: TypeParameter[] | undefined;
            approximateLength: number;
            truncating?: boolean;
            typeParameterSymbolList?: Set<number>;
            typeParameterNames?: ESMap<TypeId, Identifier>;
            typeParameterNamesByText?: Set<string>;
            usedSymbolNames?: Set<string>;
            remappedSymbolNames?: ESMap<SymbolId, string>;
        }

        function isDefaultBindingContext(location: Node) {
            return location.kind === SyntaxKind.SourceFile || isAmbientModule(location);
        }

        function getNameOfSymbolFromNameType(symbol: Symbol, context?: NodeBuilderContext) {
            const nameType = getSymbolLinks(symbol).nameType;
            if (nameType) {
                if (nameType.flags & TypeFlags.StringOrNumberLiteral) {
                    const name = "" + (<StringLiteralType | NumberLiteralType>nameType).value;
                    if (!isIdentifierText(name, compilerOptions.target) && !isNumericLiteralName(name)) {
                        return `"${escapeString(name, CharacterCodes.doubleQuote)}"`;
                    }
                    if (isNumericLiteralName(name) && startsWith(name, "-")) {
                        return `[${name}]`;
                    }
                    return name;
                }
                if (nameType.flags & TypeFlags.UniqueESSymbol) {
                    return `[${getNameOfSymbolAsWritten((<UniqueESSymbolType>nameType).symbol, context)}]`;
                }
            }
        }

        /**
         * Gets a human-readable name for a symbol.
         * Should *not* be used for the right-hand side of a `.` -- use `symbolName(symbol)` for that instead.
         *
         * Unlike `symbolName(symbol)`, this will include quotes if the name is from a string literal.
         * It will also use a representation of a number as written instead of a decimal form, e.g. `0o11` instead of `9`.
         */
        function getNameOfSymbolAsWritten(symbol: Symbol, context?: NodeBuilderContext): string {
            if (context && symbol.escapedName === InternalSymbolName.Default && !(context.flags & NodeBuilderFlags.UseAliasDefinedOutsideCurrentScope) &&
                // If it's not the first part of an entity name, it must print as `default`
                (!(context.flags & NodeBuilderFlags.InInitialEntityName) ||
                // if the symbol is synthesized, it will only be referenced externally it must print as `default`
                !symbol.declarations ||
                // if not in the same binding context (source file, module declaration), it must print as `default`
                (context.enclosingDeclaration && findAncestor(symbol.declarations[0], isDefaultBindingContext) !== findAncestor(context.enclosingDeclaration, isDefaultBindingContext)))) {
                return "default";
            }
            if (symbol.declarations && symbol.declarations.length) {
                let declaration = firstDefined(symbol.declarations, d => getNameOfDeclaration(d) ? d : undefined); // Try using a declaration with a name, first
                const name = declaration && getNameOfDeclaration(declaration);
                if (declaration && name) {
                    if (isCallExpression(declaration) && isBindableObjectDefinePropertyCall(declaration)) {
                        return symbolName(symbol);
                    }
                    if (isComputedPropertyName(name) && !(getCheckFlags(symbol) & CheckFlags.Late)) {
                        const nameType = getSymbolLinks(symbol).nameType;
                        if (nameType && nameType.flags & TypeFlags.StringOrNumberLiteral) {
                            // Computed property name isn't late bound, but has a well-known name type - use name type to generate a symbol name
                            const result = getNameOfSymbolFromNameType(symbol, context);
                            if (result !== undefined) {
                                return result;
                            }
                        }
                    }
                    return declarationNameToString(name);
                }
                if (!declaration) {
                    declaration = symbol.declarations[0]; // Declaration may be nameless, but we'll try anyway
                }
                if (declaration.parent && declaration.parent.kind === SyntaxKind.VariableDeclaration) {
                    return declarationNameToString((<VariableDeclaration>declaration.parent).name);
                }
                switch (declaration.kind) {
                    case SyntaxKind.ClassExpression:
                    case SyntaxKind.FunctionExpression:
                    case SyntaxKind.ArrowFunction:
                        if (context && !context.encounteredError && !(context.flags & NodeBuilderFlags.AllowAnonymousIdentifier)) {
                            context.encounteredError = true;
                        }
                        return declaration.kind === SyntaxKind.ClassExpression ? "(Anonymous class)" : "(Anonymous function)";
                }
            }
            const name = getNameOfSymbolFromNameType(symbol, context);
            return name !== undefined ? name : symbolName(symbol);
        }

        function isDeclarationVisible(node: Node): boolean {
            if (node) {
                const links = getNodeLinks(node);
                if (links.isVisible === undefined) {
                    links.isVisible = !!determineIfDeclarationIsVisible();
                }
                return links.isVisible;
            }

            return false;

            function determineIfDeclarationIsVisible() {
                switch (node.kind) {
                    case SyntaxKind.JSDocCallbackTag:
                    case SyntaxKind.JSDocTypedefTag:
                    case SyntaxKind.JSDocEnumTag:
                        // Top-level jsdoc type aliases are considered exported
                        // First parent is comment node, second is hosting declaration or token; we only care about those tokens or declarations whose parent is a source file
                        return !!(node.parent && node.parent.parent && node.parent.parent.parent && isSourceFile(node.parent.parent.parent));
                    case SyntaxKind.BindingElement:
                        return isDeclarationVisible(node.parent.parent);
                    case SyntaxKind.VariableDeclaration:
                        if (isBindingPattern((node as VariableDeclaration).name) &&
                            !((node as VariableDeclaration).name as BindingPattern).elements.length) {
                            // If the binding pattern is empty, this variable declaration is not visible
                            return false;
                        }
                        // falls through
                    case SyntaxKind.ModuleDeclaration:
                    case SyntaxKind.ClassDeclaration:
                    case SyntaxKind.InterfaceDeclaration:
                    case SyntaxKind.TypeAliasDeclaration:
                    case SyntaxKind.FunctionDeclaration:
                    case SyntaxKind.EnumDeclaration:
                    case SyntaxKind.ImportEqualsDeclaration:
                        // external module augmentation is always visible
                        if (isExternalModuleAugmentation(node)) {
                            return true;
                        }
                        const parent = getDeclarationContainer(node);
                        // If the node is not exported or it is not ambient module element (except import declaration)
                        if (!(getCombinedModifierFlags(node as Declaration) & ModifierFlags.Export) &&
                            !(node.kind !== SyntaxKind.ImportEqualsDeclaration && parent.kind !== SyntaxKind.SourceFile && parent.flags & NodeFlags.Ambient)) {
                            return isGlobalSourceFile(parent);
                        }
                        // Exported members/ambient module elements (exception import declaration) are visible if parent is visible
                        return isDeclarationVisible(parent);

                    case SyntaxKind.PropertyDeclaration:
                    case SyntaxKind.PropertySignature:
                    case SyntaxKind.GetAccessor:
                    case SyntaxKind.SetAccessor:
                    case SyntaxKind.MethodDeclaration:
                    case SyntaxKind.MethodSignature:
                        if (hasEffectiveModifier(node, ModifierFlags.Private | ModifierFlags.Protected)) {
                            // Private/protected properties/methods are not visible
                            return false;
                        }
                        // Public properties/methods are visible if its parents are visible, so:
                        // falls through

                    case SyntaxKind.Constructor:
                    case SyntaxKind.ConstructSignature:
                    case SyntaxKind.CallSignature:
                    case SyntaxKind.IndexSignature:
                    case SyntaxKind.Parameter:
                    case SyntaxKind.ModuleBlock:
                    case SyntaxKind.FunctionType:
                    case SyntaxKind.ConstructorType:
                    case SyntaxKind.TypeLiteral:
                    case SyntaxKind.TypeReference:
                    case SyntaxKind.ArrayType:
                    case SyntaxKind.TupleType:
                    case SyntaxKind.UnionType:
                    case SyntaxKind.IntersectionType:
                    case SyntaxKind.ParenthesizedType:
                    case SyntaxKind.NamedTupleMember:
                        return isDeclarationVisible(node.parent);

                    // Default binding, import specifier and namespace import is visible
                    // only on demand so by default it is not visible
                    case SyntaxKind.ImportClause:
                    case SyntaxKind.NamespaceImport:
                    case SyntaxKind.ImportSpecifier:
                        return false;

                    // Type parameters are always visible
                    case SyntaxKind.TypeParameter:

                    // Source file and namespace export are always visible
                    // falls through
                    case SyntaxKind.SourceFile:
                    case SyntaxKind.NamespaceExportDeclaration:
                        return true;

                    // Export assignments do not create name bindings outside the module
                    case SyntaxKind.ExportAssignment:
                        return false;

                    default:
                        return false;
                }
            }
        }

        function collectLinkedAliases(node: Identifier, setVisibility?: boolean): Node[] | undefined {
            let exportSymbol: Symbol | undefined;
            if (node.parent && node.parent.kind === SyntaxKind.ExportAssignment) {
                exportSymbol = resolveName(node, node.escapedText, SymbolFlags.Value | SymbolFlags.Type | SymbolFlags.Namespace | SymbolFlags.Alias, /*nameNotFoundMessage*/ undefined, node, /*isUse*/ false);
            }
            else if (node.parent.kind === SyntaxKind.ExportSpecifier) {
                exportSymbol = getTargetOfExportSpecifier(<ExportSpecifier>node.parent, SymbolFlags.Value | SymbolFlags.Type | SymbolFlags.Namespace | SymbolFlags.Alias);
            }
            let result: Node[] | undefined;
            let visited: Set<number> | undefined;
            if (exportSymbol) {
                visited = new Set();
                visited.add(getSymbolId(exportSymbol));
                buildVisibleNodeList(exportSymbol.declarations);
            }
            return result;

            function buildVisibleNodeList(declarations: Declaration[]) {
                forEach(declarations, declaration => {
                    const resultNode = getAnyImportSyntax(declaration) || declaration;
                    if (setVisibility) {
                        getNodeLinks(declaration).isVisible = true;
                    }
                    else {
                        result = result || [];
                        pushIfUnique(result, resultNode);
                    }

                    if (isInternalModuleImportEqualsDeclaration(declaration)) {
                        // Add the referenced top container visible
                        const internalModuleReference = <Identifier | QualifiedName>declaration.moduleReference;
                        const firstIdentifier = getFirstIdentifier(internalModuleReference);
                        const importSymbol = resolveName(declaration, firstIdentifier.escapedText, SymbolFlags.Value | SymbolFlags.Type | SymbolFlags.Namespace,
                            undefined, undefined, /*isUse*/ false);
                        if (importSymbol && visited) {
                            if (tryAddToSet(visited, getSymbolId(importSymbol))) {
                                buildVisibleNodeList(importSymbol.declarations);
                            }
                        }
                    }
                });
            }
        }

        /**
         * Push an entry on the type resolution stack. If an entry with the given target and the given property name
         * is already on the stack, and no entries in between already have a type, then a circularity has occurred.
         * In this case, the result values of the existing entry and all entries pushed after it are changed to false,
         * and the value false is returned. Otherwise, the new entry is just pushed onto the stack, and true is returned.
         * In order to see if the same query has already been done before, the target object and the propertyName both
         * must match the one passed in.
         *
         * @param target The symbol, type, or signature whose type is being queried
         * @param propertyName The property name that should be used to query the target for its type
         */
        function pushTypeResolution(target: TypeSystemEntity, propertyName: TypeSystemPropertyName): boolean {
            const resolutionCycleStartIndex = findResolutionCycleStartIndex(target, propertyName);
            if (resolutionCycleStartIndex >= 0) {
                // A cycle was found
                const { length } = resolutionTargets;
                for (let i = resolutionCycleStartIndex; i < length; i++) {
                    resolutionResults[i] = false;
                }
                return false;
            }
            resolutionTargets.push(target);
            resolutionResults.push(/*items*/ true);
            resolutionPropertyNames.push(propertyName);
            return true;
        }

        function findResolutionCycleStartIndex(target: TypeSystemEntity, propertyName: TypeSystemPropertyName): number {
            for (let i = resolutionTargets.length - 1; i >= 0; i--) {
                if (hasType(resolutionTargets[i], resolutionPropertyNames[i])) {
                    return -1;
                }
                if (resolutionTargets[i] === target && resolutionPropertyNames[i] === propertyName) {
                    return i;
                }
            }
            return -1;
        }

        function hasType(target: TypeSystemEntity, propertyName: TypeSystemPropertyName): boolean {
            switch (propertyName) {
                case TypeSystemPropertyName.Type:
                    return !!getSymbolLinks(<Symbol>target).type;
                case TypeSystemPropertyName.EnumTagType:
                    return !!(getNodeLinks(target as JSDocEnumTag).resolvedEnumType);
                case TypeSystemPropertyName.DeclaredType:
                    return !!getSymbolLinks(<Symbol>target).declaredType;
                case TypeSystemPropertyName.ResolvedBaseConstructorType:
                    return !!(<InterfaceType>target).resolvedBaseConstructorType;
                case TypeSystemPropertyName.ResolvedReturnType:
                    return !!(<Signature>target).resolvedReturnType;
                case TypeSystemPropertyName.ImmediateBaseConstraint:
                    return !!(<Type>target).immediateBaseConstraint;
                case TypeSystemPropertyName.ResolvedTypeArguments:
                    return !!(target as TypeReference).resolvedTypeArguments;
                case TypeSystemPropertyName.ResolvedBaseTypes:
                    return !!(target as InterfaceType).baseTypesResolved;
            }
            return Debug.assertNever(propertyName);
        }

        /**
         * Pop an entry from the type resolution stack and return its associated result value. The result value will
         * be true if no circularities were detected, or false if a circularity was found.
         */
        function popTypeResolution(): boolean {
            resolutionTargets.pop();
            resolutionPropertyNames.pop();
            return resolutionResults.pop()!;
        }

        function getDeclarationContainer(node: Node): Node {
            return findAncestor(getRootDeclaration(node), node => {
                switch (node.kind) {
                    case SyntaxKind.VariableDeclaration:
                    case SyntaxKind.VariableDeclarationList:
                    case SyntaxKind.ImportSpecifier:
                    case SyntaxKind.NamedImports:
                    case SyntaxKind.NamespaceImport:
                    case SyntaxKind.ImportClause:
                        return false;
                    default:
                        return true;
                }
            })!.parent;
        }

        function getTypeOfPrototypeProperty(prototype: Symbol): Type {
            // TypeScript 1.0 spec (April 2014): 8.4
            // Every class automatically contains a static property member named 'prototype',
            // the type of which is an instantiation of the class type with type Any supplied as a type argument for each type parameter.
            // It is an error to explicitly declare a static property member with the name 'prototype'.
            const classType = <InterfaceType>getDeclaredTypeOfSymbol(getParentOfSymbol(prototype)!);
            return classType.typeParameters ? createTypeReference(<GenericType>classType, map(classType.typeParameters, _ => anyType)) : classType;
        }

        // Return the type of the given property in the given type, or undefined if no such property exists
        function getTypeOfPropertyOfType(type: Type, name: __String): Type | undefined {
            const prop = getPropertyOfType(type, name);
            return prop ? getTypeOfSymbol(prop) : undefined;
        }

        function getTypeOfPropertyOrIndexSignature(type: Type, name: __String): Type {
            return getTypeOfPropertyOfType(type, name) || isNumericLiteralName(name) && getIndexTypeOfType(type, IndexKind.Number) || getIndexTypeOfType(type, IndexKind.String) || unknownType;
        }

        function isTypeAny(type: Type | undefined) {
            return type && (type.flags & TypeFlags.Any) !== 0;
        }

        // Return the type of a binding element parent. We check SymbolLinks first to see if a type has been
        // assigned by contextual typing.
        function getTypeForBindingElementParent(node: BindingElementGrandparent) {
            const symbol = getSymbolOfNode(node);
            return symbol && getSymbolLinks(symbol).type || getTypeForVariableLikeDeclaration(node, /*includeOptionality*/ false);
        }

        function getRestType(source: Type, properties: PropertyName[], symbol: Symbol | undefined): Type {
            source = filterType(source, t => !(t.flags & TypeFlags.Nullable));
            if (source.flags & TypeFlags.Never) {
                return emptyObjectType;
            }
            if (source.flags & TypeFlags.Union) {
                return mapType(source, t => getRestType(t, properties, symbol));
            }
            const omitKeyType = getUnionType(map(properties, getLiteralTypeFromPropertyName));
            if (isGenericObjectType(source) || isGenericIndexType(omitKeyType)) {
                if (omitKeyType.flags & TypeFlags.Never) {
                    return source;
                }

                const omitTypeAlias = getGlobalOmitSymbol();
                if (!omitTypeAlias) {
                    return errorType;
                }
                return getTypeAliasInstantiation(omitTypeAlias, [source, omitKeyType]);
            }
            const members = createSymbolTable();
            for (const prop of getPropertiesOfType(source)) {
                if (!isTypeAssignableTo(getLiteralTypeFromProperty(prop, TypeFlags.StringOrNumberLiteralOrUnique), omitKeyType)
                    && !(getDeclarationModifierFlagsFromSymbol(prop) & (ModifierFlags.Private | ModifierFlags.Protected))
                    && isSpreadableProperty(prop)) {
                    members.set(prop.escapedName, getSpreadSymbol(prop, /*readonly*/ false));
                }
            }
            const stringIndexInfo = getIndexInfoOfType(source, IndexKind.String);
            const numberIndexInfo = getIndexInfoOfType(source, IndexKind.Number);
            const result = createAnonymousType(symbol, members, emptyArray, emptyArray, stringIndexInfo, numberIndexInfo);
            result.objectFlags |= ObjectFlags.ObjectRestType;
            return result;
        }

        // Determine the control flow type associated with a destructuring declaration or assignment. The following
        // forms of destructuring are possible:
        //   let { x } = obj;  // BindingElement
        //   let [ x ] = obj;  // BindingElement
        //   { x } = obj;      // ShorthandPropertyAssignment
        //   { x: v } = obj;   // PropertyAssignment
        //   [ x ] = obj;      // Expression
        // We construct a synthetic element access expression corresponding to 'obj.x' such that the control
        // flow analyzer doesn't have to handle all the different syntactic forms.
        function getFlowTypeOfDestructuring(node: BindingElement | PropertyAssignment | ShorthandPropertyAssignment | Expression, declaredType: Type) {
            const reference = getSyntheticElementAccess(node);
            return reference ? getFlowTypeOfReference(reference, declaredType) : declaredType;
        }

        function getSyntheticElementAccess(node: BindingElement | PropertyAssignment | ShorthandPropertyAssignment | Expression): ElementAccessExpression | undefined {
            const parentAccess = getParentElementAccess(node);
            if (parentAccess && parentAccess.flowNode) {
                const propName = getDestructuringPropertyName(node);
                if (propName) {
                    const literal = setTextRange(parseNodeFactory.createStringLiteral(propName), node);
                    const result = setTextRange(parseNodeFactory.createElementAccessExpression(parentAccess, literal), node);
                    setParent(literal, result);
                    setParent(result, node);
                    result.flowNode = parentAccess.flowNode;
                    return result;
                }
            }
        }

        function getParentElementAccess(node: BindingElement | PropertyAssignment | ShorthandPropertyAssignment | Expression) {
            const ancestor = node.parent.parent;
            switch (ancestor.kind) {
                case SyntaxKind.BindingElement:
                case SyntaxKind.PropertyAssignment:
                    return getSyntheticElementAccess(<BindingElement | PropertyAssignment>ancestor);
                case SyntaxKind.ArrayLiteralExpression:
                    return getSyntheticElementAccess(<Expression>node.parent);
                case SyntaxKind.VariableDeclaration:
                    return (<VariableDeclaration>ancestor).initializer;
                case SyntaxKind.BinaryExpression:
                    return (<BinaryExpression>ancestor).right;
            }
        }

        function getDestructuringPropertyName(node: BindingElement | PropertyAssignment | ShorthandPropertyAssignment | Expression) {
            const parent = node.parent;
            if (node.kind === SyntaxKind.BindingElement && parent.kind === SyntaxKind.ObjectBindingPattern) {
                return getLiteralPropertyNameText((<BindingElement>node).propertyName || <Identifier>(<BindingElement>node).name);
            }
            if (node.kind === SyntaxKind.PropertyAssignment || node.kind === SyntaxKind.ShorthandPropertyAssignment) {
                return getLiteralPropertyNameText((<PropertyAssignment | ShorthandPropertyAssignment>node).name);
            }
            return "" + (<NodeArray<Node>>(<BindingPattern | ArrayLiteralExpression>parent).elements).indexOf(node);
        }

        function getLiteralPropertyNameText(name: PropertyName) {
            const type = getLiteralTypeFromPropertyName(name);
            return type.flags & (TypeFlags.StringLiteral | TypeFlags.NumberLiteral) ? "" + (<StringLiteralType | NumberLiteralType>type).value : undefined;
        }

        /** Return the inferred type for a binding element */
        function getTypeForBindingElement(declaration: BindingElement): Type | undefined {
            const pattern = declaration.parent;
            let parentType = getTypeForBindingElementParent(pattern.parent);
            // If no type or an any type was inferred for parent, infer that for the binding element
            if (!parentType || isTypeAny(parentType)) {
                return parentType;
            }
            // Relax null check on ambient destructuring parameters, since the parameters have no implementation and are just documentation
            if (strictNullChecks && declaration.flags & NodeFlags.Ambient && isParameterDeclaration(declaration)) {
                parentType = getNonNullableType(parentType);
            }
            // Filter `undefined` from the type we check against if the parent has an initializer and that initializer is not possibly `undefined`
            else if (strictNullChecks && pattern.parent.initializer && !(getTypeFacts(getTypeOfInitializer(pattern.parent.initializer)) & TypeFacts.EQUndefined)) {
                parentType = getTypeWithFacts(parentType, TypeFacts.NEUndefined);
            }

            let type: Type | undefined;
            if (pattern.kind === SyntaxKind.ObjectBindingPattern) {
                if (declaration.dotDotDotToken) {
                    parentType = getReducedType(parentType);
                    if (parentType.flags & TypeFlags.Unknown || !isValidSpreadType(parentType)) {
                        error(declaration, Diagnostics.Rest_types_may_only_be_created_from_object_types);
                        return errorType;
                    }
                    const literalMembers: PropertyName[] = [];
                    for (const element of pattern.elements) {
                        if (!element.dotDotDotToken) {
                            literalMembers.push(element.propertyName || element.name as Identifier);
                        }
                    }
                    type = getRestType(parentType, literalMembers, declaration.symbol);
                }
                else {
                    // Use explicitly specified property name ({ p: xxx } form), or otherwise the implied name ({ p } form)
                    const name = declaration.propertyName || <Identifier>declaration.name;
                    const indexType = getLiteralTypeFromPropertyName(name);
                    const declaredType = getConstraintForLocation(getIndexedAccessType(parentType, indexType, name), declaration.name);
                    type = getFlowTypeOfDestructuring(declaration, declaredType);
                }
            }
            else {
                // This elementType will be used if the specific property corresponding to this index is not
                // present (aka the tuple element property). This call also checks that the parentType is in
                // fact an iterable or array (depending on target language).
                const elementType = checkIteratedTypeOrElementType(IterationUse.Destructuring, parentType, undefinedType, pattern);
                const index = pattern.elements.indexOf(declaration);
                if (declaration.dotDotDotToken) {
                    // If the parent is a tuple type, the rest element has a tuple type of the
                    // remaining tuple element types. Otherwise, the rest element has an array type with same
                    // element type as the parent type.
                    type = everyType(parentType, isTupleType) ?
                        mapType(parentType, t => sliceTupleType(<TupleTypeReference>t, index)) :
                        createArrayType(elementType);
                }
                else if (isArrayLikeType(parentType)) {
                    const indexType = getLiteralType(index);
                    const accessFlags = hasDefaultValue(declaration) ? AccessFlags.NoTupleBoundsCheck : 0;
                    const declaredType = getConstraintForLocation(getIndexedAccessTypeOrUndefined(parentType, indexType, declaration.name, accessFlags) || errorType, declaration.name);
                    type = getFlowTypeOfDestructuring(declaration, declaredType);
                }
                else {
                    type = elementType;
                }
            }
            if (!declaration.initializer) {
                return type;
            }
            if (getEffectiveTypeAnnotationNode(walkUpBindingElementsAndPatterns(declaration))) {
                // In strict null checking mode, if a default value of a non-undefined type is specified, remove
                // undefined from the final type.
                return strictNullChecks && !(getFalsyFlags(checkDeclarationInitializer(declaration)) & TypeFlags.Undefined) ?
                    getTypeWithFacts(type, TypeFacts.NEUndefined) :
                    type;
            }
            return widenTypeInferredFromInitializer(declaration, getUnionType([getTypeWithFacts(type, TypeFacts.NEUndefined), checkDeclarationInitializer(declaration)], UnionReduction.Subtype));
        }

        function getTypeForDeclarationFromJSDocComment(declaration: Node) {
            const jsdocType = getJSDocType(declaration);
            if (jsdocType) {
                return getTypeFromTypeNode(jsdocType);
            }
            return undefined;
        }

        function isNullOrUndefined(node: Expression) {
            const expr = skipParentheses(node);
            return expr.kind === SyntaxKind.NullKeyword || expr.kind === SyntaxKind.Identifier && getResolvedSymbol(<Identifier>expr) === undefinedSymbol;
        }

        function isEmptyArrayLiteral(node: Expression) {
            const expr = skipParentheses(node);
            return expr.kind === SyntaxKind.ArrayLiteralExpression && (<ArrayLiteralExpression>expr).elements.length === 0;
        }

        function addOptionality(type: Type, optional = true): Type {
            return strictNullChecks && optional ? getOptionalType(type) : type;
        }

        // Return the inferred type for a variable, parameter, or property declaration
        function getTypeForVariableLikeDeclaration(declaration: ParameterDeclaration | PropertyDeclaration | PropertySignature | VariableDeclaration | BindingElement | JSDocPropertyLikeTag, includeOptionality: boolean): Type | undefined {
            // A variable declared in a for..in statement is of type string, or of type keyof T when the
            // right hand expression is of a type parameter type.
            if (isVariableDeclaration(declaration) && declaration.parent.parent.kind === SyntaxKind.ForInStatement) {
                const indexType = getIndexType(getNonNullableTypeIfNeeded(checkExpression(declaration.parent.parent.expression)));
                return indexType.flags & (TypeFlags.TypeParameter | TypeFlags.Index) ? getExtractStringType(indexType) : stringType;
            }

            if (isVariableDeclaration(declaration) && declaration.parent.parent.kind === SyntaxKind.ForOfStatement) {
                // checkRightHandSideOfForOf will return undefined if the for-of expression type was
                // missing properties/signatures required to get its iteratedType (like
                // [Symbol.iterator] or next). This may be because we accessed properties from anyType,
                // or it may have led to an error inside getElementTypeOfIterable.
                const forOfStatement = declaration.parent.parent;
                return checkRightHandSideOfForOf(forOfStatement) || anyType;
            }

            if (isBindingPattern(declaration.parent)) {
                return getTypeForBindingElement(<BindingElement>declaration);
            }

            const isOptional = includeOptionality && (
                isParameter(declaration) && isJSDocOptionalParameter(declaration)
                || isOptionalJSDocPropertyLikeTag(declaration)
                || !isBindingElement(declaration) && !isVariableDeclaration(declaration) && !!declaration.questionToken);

            // Use type from type annotation if one is present
            const declaredType = tryGetTypeFromEffectiveTypeNode(declaration);
            if (declaredType) {
                return addOptionality(declaredType, isOptional);
            }

            if ((noImplicitAny || isInJSFile(declaration)) &&
                isVariableDeclaration(declaration) && !isBindingPattern(declaration.name) &&
                !(getCombinedModifierFlags(declaration) & ModifierFlags.Export) && !(declaration.flags & NodeFlags.Ambient)) {
                // If --noImplicitAny is on or the declaration is in a Javascript file,
                // use control flow tracked 'any' type for non-ambient, non-exported var or let variables with no
                // initializer or a 'null' or 'undefined' initializer.
                if (!(getCombinedNodeFlags(declaration) & NodeFlags.Const) && (!declaration.initializer || isNullOrUndefined(declaration.initializer))) {
                    return autoType;
                }
                // Use control flow tracked 'any[]' type for non-ambient, non-exported variables with an empty array
                // literal initializer.
                if (declaration.initializer && isEmptyArrayLiteral(declaration.initializer)) {
                    return autoArrayType;
                }
            }

            if (isParameter(declaration)) {
                const func = <FunctionLikeDeclaration>declaration.parent;
                // For a parameter of a set accessor, use the type of the get accessor if one is present
                if (func.kind === SyntaxKind.SetAccessor && !hasNonBindableDynamicName(func)) {
                    const getter = getDeclarationOfKind<AccessorDeclaration>(getSymbolOfNode(declaration.parent), SyntaxKind.GetAccessor);
                    if (getter) {
                        const getterSignature = getSignatureFromDeclaration(getter);
                        const thisParameter = getAccessorThisParameter(func as AccessorDeclaration);
                        if (thisParameter && declaration === thisParameter) {
                            // Use the type from the *getter*
                            Debug.assert(!thisParameter.type);
                            return getTypeOfSymbol(getterSignature.thisParameter!);
                        }
                        return getReturnTypeOfSignature(getterSignature);
                    }
                }
                if (isInJSFile(declaration)) {
                    const typeTag = getJSDocType(func);
                    if (typeTag && isFunctionTypeNode(typeTag)) {
                        const signature = getSignatureFromDeclaration(typeTag);
                        const pos = func.parameters.indexOf(declaration);
                        return declaration.dotDotDotToken ? getRestTypeAtPosition(signature, pos) : getTypeAtPosition(signature, pos);
                    }
                }
                // Use contextual parameter type if one is available
                const type = declaration.symbol.escapedName === InternalSymbolName.This ? getContextualThisParameterType(func) : getContextuallyTypedParameterType(declaration);
                if (type) {
                    return addOptionality(type, isOptional);
                }
            }

            // Use the type of the initializer expression if one is present and the declaration is
            // not a parameter of a contextually typed function
            if (hasOnlyExpressionInitializer(declaration) && !!declaration.initializer) {
                if (isInJSFile(declaration) && !isParameter(declaration)) {
                    const containerObjectType = getJSContainerObjectType(declaration, getSymbolOfNode(declaration), getDeclaredExpandoInitializer(declaration));
                    if (containerObjectType) {
                        return containerObjectType;
                    }
                }
                const type = widenTypeInferredFromInitializer(declaration, checkDeclarationInitializer(declaration));
                return addOptionality(type, isOptional);
            }

            if (isPropertyDeclaration(declaration) && !hasStaticModifier(declaration) && (noImplicitAny || isInJSFile(declaration))) {
                // We have a property declaration with no type annotation or initializer, in noImplicitAny mode or a .js file.
                // Use control flow analysis of this.xxx assignments in the constructor to determine the type of the property.
                const constructor = findConstructorDeclaration(declaration.parent);
                const type = constructor ? getFlowTypeInConstructor(declaration.symbol, constructor) :
                    getEffectiveModifierFlags(declaration) & ModifierFlags.Ambient ? getTypeOfPropertyInBaseClass(declaration.symbol) :
                    undefined;
                return type && addOptionality(type, isOptional);
            }

            if (isJsxAttribute(declaration)) {
                // if JSX attribute doesn't have initializer, by default the attribute will have boolean value of true.
                // I.e <Elem attr /> is sugar for <Elem attr={true} />
                return trueType;
            }

            // If the declaration specifies a binding pattern and is not a parameter of a contextually
            // typed function, use the type implied by the binding pattern
            if (isBindingPattern(declaration.name)) {
                return getTypeFromBindingPattern(declaration.name, /*includePatternInType*/ false, /*reportErrors*/ true);
            }

            // No type specified and nothing can be inferred
            return undefined;
        }

        function isConstructorDeclaredProperty(symbol: Symbol) {
            // A property is considered a constructor declared property when all declaration sites are this.xxx assignments,
            // when no declaration sites have JSDoc type annotations, and when at least one declaration site is in the body of
            // a class constructor.
            if (symbol.valueDeclaration && isBinaryExpression(symbol.valueDeclaration)) {
                const links = getSymbolLinks(symbol);
                if (links.isConstructorDeclaredProperty === undefined) {
                    links.isConstructorDeclaredProperty = !!getDeclaringConstructor(symbol) && every(symbol.declarations, declaration =>
                        isBinaryExpression(declaration) &&
                        getAssignmentDeclarationKind(declaration) === AssignmentDeclarationKind.ThisProperty &&
                        (declaration.left.kind !== SyntaxKind.ElementAccessExpression || isStringOrNumericLiteralLike((<ElementAccessExpression>declaration.left).argumentExpression)) &&
                        !getAnnotatedTypeForAssignmentDeclaration(/*declaredType*/ undefined, declaration, symbol, declaration));
                }
                return links.isConstructorDeclaredProperty;
            }
            return false;
        }

        function isAutoTypedProperty(symbol: Symbol) {
            // A property is auto-typed when its declaration has no type annotation or initializer and we're in
            // noImplicitAny mode or a .js file.
            const declaration = symbol.valueDeclaration;
            return declaration && isPropertyDeclaration(declaration) && !getEffectiveTypeAnnotationNode(declaration) &&
                !declaration.initializer && (noImplicitAny || isInJSFile(declaration));
        }

        function getDeclaringConstructor(symbol: Symbol) {
            for (const declaration of symbol.declarations) {
                const container = getThisContainer(declaration, /*includeArrowFunctions*/ false);
                if (container && (container.kind === SyntaxKind.Constructor || isJSConstructor(container))) {
                    return <ConstructorDeclaration>container;
                }
            }
        }

        function getFlowTypeInConstructor(symbol: Symbol, constructor: ConstructorDeclaration) {
            const reference = factory.createPropertyAccessExpression(factory.createThis(), unescapeLeadingUnderscores(symbol.escapedName));
            setParent(reference.expression, reference);
            setParent(reference, constructor);
            reference.flowNode = constructor.returnFlowNode;
            const flowType = getFlowTypeOfProperty(reference, symbol);
            if (noImplicitAny && (flowType === autoType || flowType === autoArrayType)) {
                error(symbol.valueDeclaration, Diagnostics.Member_0_implicitly_has_an_1_type, symbolToString(symbol), typeToString(flowType));
            }
            // We don't infer a type if assignments are only null or undefined.
            return everyType(flowType, isNullableType) ? undefined : convertAutoToAny(flowType);
        }

        function getFlowTypeOfProperty(reference: Node, prop: Symbol | undefined) {
            const initialType = prop && (!isAutoTypedProperty(prop) || getEffectiveModifierFlags(prop.valueDeclaration) & ModifierFlags.Ambient) && getTypeOfPropertyInBaseClass(prop) || undefinedType;
            return getFlowTypeOfReference(reference, autoType, initialType);
        }

        function getWidenedTypeForAssignmentDeclaration(symbol: Symbol, resolvedSymbol?: Symbol) {
            // function/class/{} initializers are themselves containers, so they won't merge in the same way as other initializers
            const container = getAssignedExpandoInitializer(symbol.valueDeclaration);
            if (container) {
                const tag = getJSDocTypeTag(container);
                if (tag && tag.typeExpression) {
                    return getTypeFromTypeNode(tag.typeExpression);
                }
                const containerObjectType = getJSContainerObjectType(symbol.valueDeclaration, symbol, container);
                return containerObjectType || getWidenedLiteralType(checkExpressionCached(container));
            }
            let type;
            let definedInConstructor = false;
            let definedInMethod = false;
            // We use control flow analysis to determine the type of the property if the property qualifies as a constructor
            // declared property and the resulting control flow type isn't just undefined or null.
            if (isConstructorDeclaredProperty(symbol)) {
                type = getFlowTypeInConstructor(symbol, getDeclaringConstructor(symbol)!);
            }
            if (!type) {
                let jsdocType: Type | undefined;
                let types: Type[] | undefined;
                for (const declaration of symbol.declarations) {
                    const expression = (isBinaryExpression(declaration) || isCallExpression(declaration)) ? declaration :
                        isAccessExpression(declaration) ? isBinaryExpression(declaration.parent) ? declaration.parent : declaration :
                        undefined;
                    if (!expression) {
                        continue; // Non-assignment declaration merged in (eg, an Identifier to mark the thing as a namespace) - skip over it and pull type info from elsewhere
                    }

                    const kind = isAccessExpression(expression)
                        ? getAssignmentDeclarationPropertyAccessKind(expression)
                        : getAssignmentDeclarationKind(expression);
                    if (kind === AssignmentDeclarationKind.ThisProperty) {
                        if (isDeclarationInConstructor(expression)) {
                            definedInConstructor = true;
                        }
                        else {
                            definedInMethod = true;
                        }
                    }
                    if (!isCallExpression(expression)) {
                        jsdocType = getAnnotatedTypeForAssignmentDeclaration(jsdocType, expression, symbol, declaration);
                    }
                    if (!jsdocType) {
                        (types || (types = [])).push((isBinaryExpression(expression) || isCallExpression(expression)) ? getInitializerTypeFromAssignmentDeclaration(symbol, resolvedSymbol, expression, kind) : neverType);
                    }
                }
                type = jsdocType;
                if (!type) {
                    if (!length(types)) {
                        return errorType; // No types from any declarations :(
                    }
                    let constructorTypes = definedInConstructor ? getConstructorDefinedThisAssignmentTypes(types!, symbol.declarations) : undefined;
                    // use only the constructor types unless they were only assigned null | undefined (including widening variants)
                    if (definedInMethod) {
                        const propType = getTypeOfPropertyInBaseClass(symbol);
                        if (propType) {
                            (constructorTypes || (constructorTypes = [])).push(propType);
                            definedInConstructor = true;
                        }
                    }
                    const sourceTypes = some(constructorTypes, t => !!(t.flags & ~TypeFlags.Nullable)) ? constructorTypes : types; // TODO: GH#18217
                    type = getUnionType(sourceTypes!, UnionReduction.Subtype);
                }
            }
            const widened = getWidenedType(addOptionality(type, definedInMethod && !definedInConstructor));
            if (filterType(widened, t => !!(t.flags & ~TypeFlags.Nullable)) === neverType) {
                reportImplicitAny(symbol.valueDeclaration, anyType);
                return anyType;
            }
            return widened;
        }

        function getJSContainerObjectType(decl: Node, symbol: Symbol, init: Expression | undefined): Type | undefined {
            if (!isInJSFile(decl) || !init || !isObjectLiteralExpression(init) || init.properties.length) {
                return undefined;
            }
            const exports = createSymbolTable();
            while (isBinaryExpression(decl) || isPropertyAccessExpression(decl)) {
                const s = getSymbolOfNode(decl);
                if (s?.exports?.size) {
                    mergeSymbolTable(exports, s.exports);
                }
                decl = isBinaryExpression(decl) ? decl.parent : decl.parent.parent;
            }
            const s = getSymbolOfNode(decl);
            if (s?.exports?.size) {
                mergeSymbolTable(exports, s.exports);
            }
            const type = createAnonymousType(symbol, exports, emptyArray, emptyArray, undefined, undefined);
            type.objectFlags |= ObjectFlags.JSLiteral;
            return type;
        }

        function getAnnotatedTypeForAssignmentDeclaration(declaredType: Type | undefined, expression: Expression, symbol: Symbol, declaration: Declaration) {
            const typeNode = getEffectiveTypeAnnotationNode(expression.parent);
            if (typeNode) {
                const type = getWidenedType(getTypeFromTypeNode(typeNode));
                if (!declaredType) {
                    return type;
                }
                else if (declaredType !== errorType && type !== errorType && !isTypeIdenticalTo(declaredType, type)) {
                    errorNextVariableOrPropertyDeclarationMustHaveSameType(/*firstDeclaration*/ undefined, declaredType, declaration, type);
                }
            }
            if (symbol.parent) {
                const typeNode = getEffectiveTypeAnnotationNode(symbol.parent.valueDeclaration);
                if (typeNode) {
                    return getTypeOfPropertyOfType(getTypeFromTypeNode(typeNode), symbol.escapedName);
                }
            }

            return declaredType;
        }

        /** If we don't have an explicit JSDoc type, get the type from the initializer. */
        function getInitializerTypeFromAssignmentDeclaration(symbol: Symbol, resolvedSymbol: Symbol | undefined, expression: BinaryExpression | CallExpression, kind: AssignmentDeclarationKind) {
            if (isCallExpression(expression)) {
                if (resolvedSymbol) {
                    return getTypeOfSymbol(resolvedSymbol); // This shouldn't happen except under some hopefully forbidden merges of export assignments and object define assignments
                }
                const objectLitType = checkExpressionCached((expression as BindableObjectDefinePropertyCall).arguments[2]);
                const valueType = getTypeOfPropertyOfType(objectLitType, "value" as __String);
                if (valueType) {
                    return valueType;
                }
                const getFunc = getTypeOfPropertyOfType(objectLitType, "get" as __String);
                if (getFunc) {
                    const getSig = getSingleCallSignature(getFunc);
                    if (getSig) {
                        return getReturnTypeOfSignature(getSig);
                    }
                }
                const setFunc = getTypeOfPropertyOfType(objectLitType, "set" as __String);
                if (setFunc) {
                    const setSig = getSingleCallSignature(setFunc);
                    if (setSig) {
                        return getTypeOfFirstParameterOfSignature(setSig);
                    }
                }
                return anyType;
            }
            if (containsSameNamedThisProperty(expression.left, expression.right)) {
                return anyType;
            }
            const type = resolvedSymbol ? getTypeOfSymbol(resolvedSymbol) : getWidenedLiteralType(checkExpressionCached(expression.right));
            if (type.flags & TypeFlags.Object &&
                kind === AssignmentDeclarationKind.ModuleExports &&
                symbol.escapedName === InternalSymbolName.ExportEquals) {
                const exportedType = resolveStructuredTypeMembers(type as ObjectType);
                const members = createSymbolTable();
                copyEntries(exportedType.members, members);
                if (resolvedSymbol && !resolvedSymbol.exports) {
                    resolvedSymbol.exports = createSymbolTable();
                }
                (resolvedSymbol || symbol).exports!.forEach((s, name) => {
                    const exportedMember = members.get(name)!;
                    if (exportedMember && exportedMember !== s) {
                        if (s.flags & SymbolFlags.Value && exportedMember.flags & SymbolFlags.Value) {
                            // If the member has an additional value-like declaration, union the types from the two declarations,
                            // but issue an error if they occurred in two different files. The purpose is to support a JS file with
                            // a pattern like:
                            //
                            // module.exports = { a: true };
                            // module.exports.a = 3;
                            //
                            // but we may have a JS file with `module.exports = { a: true }` along with a TypeScript module augmentation
                            // declaring an `export const a: number`. In that case, we issue a duplicate identifier error, because
                            // it's unclear what that's supposed to mean, so it's probably a mistake.
                            if (getSourceFileOfNode(s.valueDeclaration) !== getSourceFileOfNode(exportedMember.valueDeclaration)) {
                                const unescapedName = unescapeLeadingUnderscores(s.escapedName);
                                const exportedMemberName = tryCast(exportedMember.valueDeclaration, isNamedDeclaration)?.name || exportedMember.valueDeclaration;
                                addRelatedInfo(
                                    error(s.valueDeclaration, Diagnostics.Duplicate_identifier_0, unescapedName),
                                    createDiagnosticForNode(exportedMemberName, Diagnostics._0_was_also_declared_here, unescapedName));
                                addRelatedInfo(
                                    error(exportedMemberName, Diagnostics.Duplicate_identifier_0, unescapedName),
                                    createDiagnosticForNode(s.valueDeclaration, Diagnostics._0_was_also_declared_here, unescapedName));
                            }
                            const union = createSymbol(s.flags | exportedMember.flags, name);
                            union.type = getUnionType([getTypeOfSymbol(s), getTypeOfSymbol(exportedMember)]);
                            union.valueDeclaration = exportedMember.valueDeclaration;
                            union.declarations = concatenate(exportedMember.declarations, s.declarations);
                            members.set(name, union);
                        }
                        else {
                            members.set(name, mergeSymbol(s, exportedMember));
                        }
                    }
                    else {
                        members.set(name, s);
                    }
                });
                const result = createAnonymousType(
                    exportedType.symbol,
                    members,
                    exportedType.callSignatures,
                    exportedType.constructSignatures,
                    exportedType.stringIndexInfo,
                    exportedType.numberIndexInfo);
                result.objectFlags |= (getObjectFlags(type) & ObjectFlags.JSLiteral); // Propagate JSLiteral flag
                return result;
            }
            if (isEmptyArrayLiteralType(type)) {
                reportImplicitAny(expression, anyArrayType);
                return anyArrayType;
            }
            return type;
        }

        function containsSameNamedThisProperty(thisProperty: Expression, expression: Expression) {
            return isPropertyAccessExpression(thisProperty)
                && thisProperty.expression.kind === SyntaxKind.ThisKeyword
                && forEachChildRecursively(expression, n => isMatchingReference(thisProperty, n));
        }

        function isDeclarationInConstructor(expression: Expression) {
            const thisContainer = getThisContainer(expression, /*includeArrowFunctions*/ false);
            // Properties defined in a constructor (or base constructor, or javascript constructor function) don't get undefined added.
            // Function expressions that are assigned to the prototype count as methods.
            return thisContainer.kind === SyntaxKind.Constructor ||
                thisContainer.kind === SyntaxKind.FunctionDeclaration ||
                (thisContainer.kind === SyntaxKind.FunctionExpression && !isPrototypePropertyAssignment(thisContainer.parent));
        }

        function getConstructorDefinedThisAssignmentTypes(types: Type[], declarations: Declaration[]): Type[] | undefined {
            Debug.assert(types.length === declarations.length);
            return types.filter((_, i) => {
                const declaration = declarations[i];
                const expression = isBinaryExpression(declaration) ? declaration :
                    isBinaryExpression(declaration.parent) ? declaration.parent : undefined;
                return expression && isDeclarationInConstructor(expression);
            });
        }

        // Return the type implied by a binding pattern element. This is the type of the initializer of the element if
        // one is present. Otherwise, if the element is itself a binding pattern, it is the type implied by the binding
        // pattern. Otherwise, it is the type any.
        function getTypeFromBindingElement(element: BindingElement, includePatternInType?: boolean, reportErrors?: boolean): Type {
            if (element.initializer) {
                // The type implied by a binding pattern is independent of context, so we check the initializer with no
                // contextual type or, if the element itself is a binding pattern, with the type implied by that binding
                // pattern.
                const contextualType = isBindingPattern(element.name) ? getTypeFromBindingPattern(element.name, /*includePatternInType*/ true, /*reportErrors*/ false) : unknownType;
                return addOptionality(widenTypeInferredFromInitializer(element, checkDeclarationInitializer(element, contextualType)));
            }
            if (isBindingPattern(element.name)) {
                return getTypeFromBindingPattern(element.name, includePatternInType, reportErrors);
            }
            if (reportErrors && !declarationBelongsToPrivateAmbientMember(element)) {
                reportImplicitAny(element, anyType);
            }
            // When we're including the pattern in the type (an indication we're obtaining a contextual type), we
            // use the non-inferrable any type. Inference will never directly infer this type, but it is possible
            // to infer a type that contains it, e.g. for a binding pattern like [foo] or { foo }. In such cases,
            // widening of the binding pattern type substitutes a regular any for the non-inferrable any.
            return includePatternInType ? nonInferrableAnyType : anyType;
        }

        // Return the type implied by an object binding pattern
        function getTypeFromObjectBindingPattern(pattern: ObjectBindingPattern, includePatternInType: boolean, reportErrors: boolean): Type {
            const members = createSymbolTable();
            let stringIndexInfo: IndexInfo | undefined;
            let objectFlags = ObjectFlags.ObjectLiteral | ObjectFlags.ContainsObjectOrArrayLiteral;
            forEach(pattern.elements, e => {
                const name = e.propertyName || <Identifier>e.name;
                if (e.dotDotDotToken) {
                    stringIndexInfo = createIndexInfo(anyType, /*isReadonly*/ false);
                    return;
                }

                const exprType = getLiteralTypeFromPropertyName(name);
                if (!isTypeUsableAsPropertyName(exprType)) {
                    // do not include computed properties in the implied type
                    objectFlags |= ObjectFlags.ObjectLiteralPatternWithComputedProperties;
                    return;
                }
                const text = getPropertyNameFromType(exprType);
                const flags = SymbolFlags.Property | (e.initializer ? SymbolFlags.Optional : 0);
                const symbol = createSymbol(flags, text);
                symbol.type = getTypeFromBindingElement(e, includePatternInType, reportErrors);
                symbol.bindingElement = e;
                members.set(symbol.escapedName, symbol);
            });
            const result = createAnonymousType(undefined, members, emptyArray, emptyArray, stringIndexInfo, undefined);
            result.objectFlags |= objectFlags;
            if (includePatternInType) {
                result.pattern = pattern;
                result.objectFlags |= ObjectFlags.ContainsObjectOrArrayLiteral;
            }
            return result;
        }

        // Return the type implied by an array binding pattern
        function getTypeFromArrayBindingPattern(pattern: BindingPattern, includePatternInType: boolean, reportErrors: boolean): Type {
            const elements = pattern.elements;
            const lastElement = lastOrUndefined(elements);
            const restElement = lastElement && lastElement.kind === SyntaxKind.BindingElement && lastElement.dotDotDotToken ? lastElement : undefined;
            if (elements.length === 0 || elements.length === 1 && restElement) {
                return languageVersion >= ScriptTarget.ES2015 ? createIterableType(anyType) : anyArrayType;
            }
            const elementTypes = map(elements, e => isOmittedExpression(e) ? anyType : getTypeFromBindingElement(e, includePatternInType, reportErrors));
            const minLength = findLastIndex(elements, e => !(e === restElement || isOmittedExpression(e) || hasDefaultValue(e)), elements.length - 1) + 1;
            const elementFlags = map(elements, (e, i) => e === restElement ? ElementFlags.Rest : i >= minLength ? ElementFlags.Optional : ElementFlags.Required);
            let result = <TypeReference>createTupleType(elementTypes, elementFlags);
            if (includePatternInType) {
                result = cloneTypeReference(result);
                result.pattern = pattern;
                result.objectFlags |= ObjectFlags.ContainsObjectOrArrayLiteral;
            }
            return result;
        }

        // Return the type implied by a binding pattern. This is the type implied purely by the binding pattern itself
        // and without regard to its context (i.e. without regard any type annotation or initializer associated with the
        // declaration in which the binding pattern is contained). For example, the implied type of [x, y] is [any, any]
        // and the implied type of { x, y: z = 1 } is { x: any; y: number; }. The type implied by a binding pattern is
        // used as the contextual type of an initializer associated with the binding pattern. Also, for a destructuring
        // parameter with no type annotation or initializer, the type implied by the binding pattern becomes the type of
        // the parameter.
        function getTypeFromBindingPattern(pattern: BindingPattern, includePatternInType = false, reportErrors = false): Type {
            return pattern.kind === SyntaxKind.ObjectBindingPattern
                ? getTypeFromObjectBindingPattern(pattern, includePatternInType, reportErrors)
                : getTypeFromArrayBindingPattern(pattern, includePatternInType, reportErrors);
        }

        // Return the type associated with a variable, parameter, or property declaration. In the simple case this is the type
        // specified in a type annotation or inferred from an initializer. However, in the case of a destructuring declaration it
        // is a bit more involved. For example:
        //
        //   var [x, s = ""] = [1, "one"];
        //
        // Here, the array literal [1, "one"] is contextually typed by the type [any, string], which is the implied type of the
        // binding pattern [x, s = ""]. Because the contextual type is a tuple type, the resulting type of [1, "one"] is the
        // tuple type [number, string]. Thus, the type inferred for 'x' is number and the type inferred for 's' is string.
        function getWidenedTypeForVariableLikeDeclaration(declaration: ParameterDeclaration | PropertyDeclaration | PropertySignature | VariableDeclaration | BindingElement | JSDocPropertyLikeTag, reportErrors?: boolean): Type {
            return widenTypeForVariableLikeDeclaration(getTypeForVariableLikeDeclaration(declaration, /*includeOptionality*/ true), declaration, reportErrors);
        }

        function widenTypeForVariableLikeDeclaration(type: Type | undefined, declaration: any, reportErrors?: boolean) {
            if (type) {
                if (reportErrors) {
                    reportErrorsFromWidening(declaration, type);
                }

                // always widen a 'unique symbol' type if the type was created for a different declaration.
                if (type.flags & TypeFlags.UniqueESSymbol && (isBindingElement(declaration) || !declaration.type) && type.symbol !== getSymbolOfNode(declaration)) {
                    type = esSymbolType;
                }

                return getWidenedType(type);
            }

            // Rest parameters default to type any[], other parameters default to type any
            type = isParameter(declaration) && declaration.dotDotDotToken ? anyArrayType : anyType;

            // Report implicit any errors unless this is a private property within an ambient declaration
            if (reportErrors) {
                if (!declarationBelongsToPrivateAmbientMember(declaration)) {
                    reportImplicitAny(declaration, type);
                }
            }
            return type;
        }

        function declarationBelongsToPrivateAmbientMember(declaration: VariableLikeDeclaration) {
            const root = getRootDeclaration(declaration);
            const memberDeclaration = root.kind === SyntaxKind.Parameter ? root.parent : root;
            return isPrivateWithinAmbient(memberDeclaration);
        }

        function tryGetTypeFromEffectiveTypeNode(declaration: Declaration) {
            const typeNode = getEffectiveTypeAnnotationNode(declaration);
            if (typeNode) {
                return getTypeFromTypeNode(typeNode);
            }
        }

        function getTypeOfVariableOrParameterOrProperty(symbol: Symbol): Type {
            const links = getSymbolLinks(symbol);
            if (!links.type) {
                const type = getTypeOfVariableOrParameterOrPropertyWorker(symbol);
                // For a contextually typed parameter it is possible that a type has already
                // been assigned (in assignTypeToParameterAndFixTypeParameters), and we want
                // to preserve this type.
                if (!links.type) {
                    links.type = type;
                }
            }
            return links.type;
        }

        function getTypeOfVariableOrParameterOrPropertyWorker(symbol: Symbol) {
            // Handle prototype property
            if (symbol.flags & SymbolFlags.Prototype) {
                return getTypeOfPrototypeProperty(symbol);
            }
            // CommonsJS require and module both have type any.
            if (symbol === requireSymbol) {
                return anyType;
            }
            if (symbol.flags & SymbolFlags.ModuleExports) {
                const fileSymbol = getSymbolOfNode(getSourceFileOfNode(symbol.valueDeclaration));
                const members = createSymbolTable();
                members.set("exports" as __String, fileSymbol);
                return createAnonymousType(symbol, members, emptyArray, emptyArray, undefined, undefined);
            }
            // Handle catch clause variables
            const declaration = symbol.valueDeclaration;
            if (isCatchClauseVariableDeclarationOrBindingElement(declaration)) {
                const decl = declaration as VariableDeclaration;
                if (!decl.type) return anyType;
                const type = getTypeOfNode(decl.type);
                // an errorType will make `checkTryStatement` issue an error
                return isTypeAny(type) || type === unknownType ? type : errorType;
            }
            // Handle export default expressions
            if (isSourceFile(declaration) && isJsonSourceFile(declaration)) {
                if (!declaration.statements.length) {
                    return emptyObjectType;
                }
                return getWidenedType(getWidenedLiteralType(checkExpression(declaration.statements[0].expression)));
            }

            // Handle variable, parameter or property
            if (!pushTypeResolution(symbol, TypeSystemPropertyName.Type)) {
                // Symbol is property of some kind that is merged with something - should use `getTypeOfFuncClassEnumModule` and not `getTypeOfVariableOrParameterOrProperty`
                if (symbol.flags & SymbolFlags.ValueModule && !(symbol.flags & SymbolFlags.Assignment)) {
                    return getTypeOfFuncClassEnumModule(symbol);
                }
                return reportCircularityError(symbol);
            }
            let type: Type | undefined;
            if (declaration.kind === SyntaxKind.ExportAssignment) {
                type = widenTypeForVariableLikeDeclaration(checkExpressionCached((<ExportAssignment>declaration).expression), declaration);
            }
            else if (
                isBinaryExpression(declaration) ||
                (isInJSFile(declaration) &&
                (isCallExpression(declaration) || (isPropertyAccessExpression(declaration) || isBindableStaticElementAccessExpression(declaration)) && isBinaryExpression(declaration.parent)))) {
                type = getWidenedTypeForAssignmentDeclaration(symbol);
            }
            else if (isPropertyAccessExpression(declaration)
                || isElementAccessExpression(declaration)
                || isIdentifier(declaration)
                || isStringLiteralLike(declaration)
                || isNumericLiteral(declaration)
                || isClassDeclaration(declaration)
                || isFunctionDeclaration(declaration)
                || (isMethodDeclaration(declaration) && !isObjectLiteralMethod(declaration))
                || isMethodSignature(declaration)
                || isSourceFile(declaration)) {
                // Symbol is property of some kind that is merged with something - should use `getTypeOfFuncClassEnumModule` and not `getTypeOfVariableOrParameterOrProperty`
                if (symbol.flags & (SymbolFlags.Function | SymbolFlags.Method | SymbolFlags.Class | SymbolFlags.Enum | SymbolFlags.ValueModule)) {
                    return getTypeOfFuncClassEnumModule(symbol);
                }
                type = isBinaryExpression(declaration.parent) ?
                    getWidenedTypeForAssignmentDeclaration(symbol) :
                    tryGetTypeFromEffectiveTypeNode(declaration) || anyType;
            }
            else if (isPropertyAssignment(declaration)) {
                type = tryGetTypeFromEffectiveTypeNode(declaration) || checkPropertyAssignment(declaration);
            }
            else if (isJsxAttribute(declaration)) {
                type = tryGetTypeFromEffectiveTypeNode(declaration) || checkJsxAttribute(declaration);
            }
            else if (isShorthandPropertyAssignment(declaration)) {
                type = tryGetTypeFromEffectiveTypeNode(declaration) || checkExpressionForMutableLocation(declaration.name, CheckMode.Normal);
            }
            else if (isObjectLiteralMethod(declaration)) {
                type = tryGetTypeFromEffectiveTypeNode(declaration) || checkObjectLiteralMethod(declaration, CheckMode.Normal);
            }
            else if (isParameter(declaration)
                     || isPropertyDeclaration(declaration)
                     || isPropertySignature(declaration)
                     || isVariableDeclaration(declaration)
                     || isBindingElement(declaration)
                     || isJSDocPropertyLikeTag(declaration)) {
                type = getWidenedTypeForVariableLikeDeclaration(declaration, /*includeOptionality*/ true);
            }
            // getTypeOfSymbol dispatches some JS merges incorrectly because their symbol flags are not mutually exclusive.
            // Re-dispatch based on valueDeclaration.kind instead.
            else if (isEnumDeclaration(declaration)) {
                type = getTypeOfFuncClassEnumModule(symbol);
            }
            else if (isEnumMember(declaration)) {
                type = getTypeOfEnumMember(symbol);
            }
            else if (isAccessor(declaration)) {
                type = resolveTypeOfAccessors(symbol);
            }
            else {
                return Debug.fail("Unhandled declaration kind! " + Debug.formatSyntaxKind(declaration.kind) + " for " + Debug.formatSymbol(symbol));
            }

            if (!popTypeResolution()) {
                // Symbol is property of some kind that is merged with something - should use `getTypeOfFuncClassEnumModule` and not `getTypeOfVariableOrParameterOrProperty`
                if (symbol.flags & SymbolFlags.ValueModule && !(symbol.flags & SymbolFlags.Assignment)) {
                    return getTypeOfFuncClassEnumModule(symbol);
                }
                return reportCircularityError(symbol);
            }
            return type;
        }

        function getAnnotatedAccessorTypeNode(accessor: AccessorDeclaration | undefined): TypeNode | undefined {
            if (accessor) {
                if (accessor.kind === SyntaxKind.GetAccessor) {
                    const getterTypeAnnotation = getEffectiveReturnTypeNode(accessor);
                    return getterTypeAnnotation;
                }
                else {
                    const setterTypeAnnotation = getEffectiveSetAccessorTypeAnnotationNode(accessor);
                    return setterTypeAnnotation;
                }
            }
            return undefined;
        }

        function getAnnotatedAccessorType(accessor: AccessorDeclaration | undefined): Type | undefined {
            const node = getAnnotatedAccessorTypeNode(accessor);
            return node && getTypeFromTypeNode(node);
        }

        function getAnnotatedAccessorThisParameter(accessor: AccessorDeclaration): Symbol | undefined {
            const parameter = getAccessorThisParameter(accessor);
            return parameter && parameter.symbol;
        }

        function getThisTypeOfDeclaration(declaration: SignatureDeclaration): Type | undefined {
            return getThisTypeOfSignature(getSignatureFromDeclaration(declaration));
        }

        function getTypeOfAccessors(symbol: Symbol): Type {
            const links = getSymbolLinks(symbol);
            return links.type || (links.type = getTypeOfAccessorsWorker(symbol));
        }

        function getTypeOfAccessorsWorker(symbol: Symbol): Type {
            if (!pushTypeResolution(symbol, TypeSystemPropertyName.Type)) {
                return errorType;
            }

            let type = resolveTypeOfAccessors(symbol);

            if (!popTypeResolution()) {
                type = anyType;
                if (noImplicitAny) {
                    const getter = getDeclarationOfKind<AccessorDeclaration>(symbol, SyntaxKind.GetAccessor);
                    error(getter, Diagnostics._0_implicitly_has_return_type_any_because_it_does_not_have_a_return_type_annotation_and_is_referenced_directly_or_indirectly_in_one_of_its_return_expressions, symbolToString(symbol));
                }
            }
            return type;
        }

        function resolveTypeOfAccessors(symbol: Symbol) {
            const getter = getDeclarationOfKind<AccessorDeclaration>(symbol, SyntaxKind.GetAccessor);
            const setter = getDeclarationOfKind<AccessorDeclaration>(symbol, SyntaxKind.SetAccessor);

            if (getter && isInJSFile(getter)) {
                const jsDocType = getTypeForDeclarationFromJSDocComment(getter);
                if (jsDocType) {
                    return jsDocType;
                }
            }
            // First try to see if the user specified a return type on the get-accessor.
            const getterReturnType = getAnnotatedAccessorType(getter);
            if (getterReturnType) {
                return getterReturnType;
            }
            else {
                // If the user didn't specify a return type, try to use the set-accessor's parameter type.
                const setterParameterType = getAnnotatedAccessorType(setter);
                if (setterParameterType) {
                    return setterParameterType;
                }
                else {
                    // If there are no specified types, try to infer it from the body of the get accessor if it exists.
                    if (getter && getter.body) {
                        return getReturnTypeFromBody(getter);
                    }
                    // Otherwise, fall back to 'any'.
                    else {
                        if (setter) {
                            if (!isPrivateWithinAmbient(setter)) {
                                errorOrSuggestion(noImplicitAny, setter, Diagnostics.Property_0_implicitly_has_type_any_because_its_set_accessor_lacks_a_parameter_type_annotation, symbolToString(symbol));
                            }
                        }
                        else {
                            Debug.assert(!!getter, "there must exist a getter as we are current checking either setter or getter in this function");
                            if (!isPrivateWithinAmbient(getter)) {
                                errorOrSuggestion(noImplicitAny, getter, Diagnostics.Property_0_implicitly_has_type_any_because_its_get_accessor_lacks_a_return_type_annotation, symbolToString(symbol));
                            }
                        }
                        return anyType;
                    }
                }
            }
        }

        function getBaseTypeVariableOfClass(symbol: Symbol) {
            const baseConstructorType = getBaseConstructorTypeOfClass(getDeclaredTypeOfClassOrInterface(symbol));
            return baseConstructorType.flags & TypeFlags.TypeVariable ? baseConstructorType :
                baseConstructorType.flags & TypeFlags.Intersection ? find((baseConstructorType as IntersectionType).types, t => !!(t.flags & TypeFlags.TypeVariable)) :
                undefined;
        }

        function getTypeOfFuncClassEnumModule(symbol: Symbol): Type {
            let links = getSymbolLinks(symbol);
            const originalLinks = links;
            if (!links.type) {
                const jsDeclaration = symbol.valueDeclaration && getDeclarationOfExpando(symbol.valueDeclaration);
                if (jsDeclaration) {
                    const merged = mergeJSSymbols(symbol, getSymbolOfNode(jsDeclaration));
                    if (merged) {
                        // note:we overwrite links because we just cloned the symbol
                        symbol = links = merged;
                    }
                }
                originalLinks.type = links.type = getTypeOfFuncClassEnumModuleWorker(symbol);
            }
            return links.type;
        }

        function getTypeOfFuncClassEnumModuleWorker(symbol: Symbol): Type {
            const declaration = symbol.valueDeclaration;
            if (symbol.flags & SymbolFlags.Module && isShorthandAmbientModuleSymbol(symbol)) {
                return anyType;
            }
            else if (declaration && (declaration.kind === SyntaxKind.BinaryExpression ||
                     isAccessExpression(declaration) &&
                     declaration.parent.kind === SyntaxKind.BinaryExpression)) {
                return getWidenedTypeForAssignmentDeclaration(symbol);
            }
            else if (symbol.flags & SymbolFlags.ValueModule && declaration && isSourceFile(declaration) && declaration.commonJsModuleIndicator) {
                const resolvedModule = resolveExternalModuleSymbol(symbol);
                if (resolvedModule !== symbol) {
                    if (!pushTypeResolution(symbol, TypeSystemPropertyName.Type)) {
                        return errorType;
                    }
                    const exportEquals = getMergedSymbol(symbol.exports!.get(InternalSymbolName.ExportEquals)!);
                    const type = getWidenedTypeForAssignmentDeclaration(exportEquals, exportEquals === resolvedModule ? undefined : resolvedModule);
                    if (!popTypeResolution()) {
                        return reportCircularityError(symbol);
                    }
                    return type;
                }
            }
            const type = createObjectType(ObjectFlags.Anonymous, symbol);
            if (symbol.flags & SymbolFlags.Class) {
                const baseTypeVariable = getBaseTypeVariableOfClass(symbol);
                return baseTypeVariable ? getIntersectionType([type, baseTypeVariable]) : type;
            }
            else {
                return strictNullChecks && symbol.flags & SymbolFlags.Optional ? getOptionalType(type) : type;
            }
        }

        function getTypeOfEnumMember(symbol: Symbol): Type {
            const links = getSymbolLinks(symbol);
            return links.type || (links.type = getDeclaredTypeOfEnumMember(symbol));
        }

        function getTypeOfAlias(symbol: Symbol): Type {
            const links = getSymbolLinks(symbol);
            if (!links.type) {
                const targetSymbol = resolveAlias(symbol);

                // It only makes sense to get the type of a value symbol. If the result of resolving
                // the alias is not a value, then it has no type. To get the type associated with a
                // type symbol, call getDeclaredTypeOfSymbol.
                // This check is important because without it, a call to getTypeOfSymbol could end
                // up recursively calling getTypeOfAlias, causing a stack overflow.
                links.type = targetSymbol.flags & SymbolFlags.Value
                    ? getTypeOfSymbol(targetSymbol)
                    : errorType;
            }
            return links.type;
        }

        function getTypeOfInstantiatedSymbol(symbol: Symbol): Type {
            const links = getSymbolLinks(symbol);
            if (!links.type) {
                if (!pushTypeResolution(symbol, TypeSystemPropertyName.Type)) {
                    return links.type = errorType;
                }
                let type = instantiateType(getTypeOfSymbol(links.target!), links.mapper);
                if (!popTypeResolution()) {
                    type = reportCircularityError(symbol);
                }
                links.type = type;
            }
            return links.type;
        }

        function reportCircularityError(symbol: Symbol) {
            const declaration = <VariableLikeDeclaration>symbol.valueDeclaration;
            // Check if variable has type annotation that circularly references the variable itself
            if (getEffectiveTypeAnnotationNode(declaration)) {
                error(symbol.valueDeclaration, Diagnostics._0_is_referenced_directly_or_indirectly_in_its_own_type_annotation,
                    symbolToString(symbol));
                return errorType;
            }
            // Check if variable has initializer that circularly references the variable itself
            if (noImplicitAny && (declaration.kind !== SyntaxKind.Parameter || (<HasInitializer>declaration).initializer)) {
                error(symbol.valueDeclaration, Diagnostics._0_implicitly_has_type_any_because_it_does_not_have_a_type_annotation_and_is_referenced_directly_or_indirectly_in_its_own_initializer,
                    symbolToString(symbol));
            }
            // Circularities could also result from parameters in function expressions that end up
            // having themselves as contextual types following type argument inference. In those cases
            // we have already reported an implicit any error so we don't report anything here.
            return anyType;
        }

        function getTypeOfSymbolWithDeferredType(symbol: Symbol) {
            const links = getSymbolLinks(symbol);
            if (!links.type) {
                Debug.assertIsDefined(links.deferralParent);
                Debug.assertIsDefined(links.deferralConstituents);
                links.type = links.deferralParent.flags & TypeFlags.Union ? getUnionType(links.deferralConstituents) : getIntersectionType(links.deferralConstituents);
            }
            return links.type;
        }

        function getTypeOfSymbol(symbol: Symbol): Type {
            const checkFlags = getCheckFlags(symbol);
            if (checkFlags & CheckFlags.DeferredType) {
                return getTypeOfSymbolWithDeferredType(symbol);
            }
            if (checkFlags & CheckFlags.Instantiated) {
                return getTypeOfInstantiatedSymbol(symbol);
            }
            if (checkFlags & CheckFlags.Mapped) {
                return getTypeOfMappedSymbol(symbol as MappedSymbol);
            }
            if (checkFlags & CheckFlags.ReverseMapped) {
                return getTypeOfReverseMappedSymbol(symbol as ReverseMappedSymbol);
            }
            if (symbol.flags & (SymbolFlags.Variable | SymbolFlags.Property)) {
                return getTypeOfVariableOrParameterOrProperty(symbol);
            }
            if (symbol.flags & (SymbolFlags.Function | SymbolFlags.Method | SymbolFlags.Class | SymbolFlags.Enum | SymbolFlags.ValueModule)) {
                return getTypeOfFuncClassEnumModule(symbol);
            }
            if (symbol.flags & SymbolFlags.EnumMember) {
                return getTypeOfEnumMember(symbol);
            }
            if (symbol.flags & SymbolFlags.Accessor) {
                return getTypeOfAccessors(symbol);
            }
            if (symbol.flags & SymbolFlags.Alias) {
                return getTypeOfAlias(symbol);
            }
            return errorType;
        }

        function isReferenceToType(type: Type, target: Type) {
            return type !== undefined
                && target !== undefined
                && (getObjectFlags(type) & ObjectFlags.Reference) !== 0
                && (<TypeReference>type).target === target;
        }

        function getTargetType(type: Type): Type {
            return getObjectFlags(type) & ObjectFlags.Reference ? (<TypeReference>type).target : type;
        }

        // TODO: GH#18217 If `checkBase` is undefined, we should not call this because this will always return false.
        function hasBaseType(type: Type, checkBase: Type | undefined) {
            return check(type);
            function check(type: Type): boolean {
                if (getObjectFlags(type) & (ObjectFlags.ClassOrInterface | ObjectFlags.Reference)) {
                    const target = <InterfaceType>getTargetType(type);
                    return target === checkBase || some(getBaseTypes(target), check);
                }
                else if (type.flags & TypeFlags.Intersection) {
                    return some((<IntersectionType>type).types, check);
                }
                return false;
            }
        }

        // Appends the type parameters given by a list of declarations to a set of type parameters and returns the resulting set.
        // The function allocates a new array if the input type parameter set is undefined, but otherwise it modifies the set
        // in-place and returns the same array.
        function appendTypeParameters(typeParameters: TypeParameter[] | undefined, declarations: readonly TypeParameterDeclaration[]): TypeParameter[] | undefined {
            for (const declaration of declarations) {
                typeParameters = appendIfUnique(typeParameters, getDeclaredTypeOfTypeParameter(getSymbolOfNode(declaration)));
            }
            return typeParameters;
        }

        // Return the outer type parameters of a node or undefined if the node has no outer type parameters.
        function getOuterTypeParameters(node: Node, includeThisTypes?: boolean): TypeParameter[] | undefined {
            while (true) {
                node = node.parent; // TODO: GH#18217 Use SourceFile kind check instead
                if (node && isBinaryExpression(node)) {
                    // prototype assignments get the outer type parameters of their constructor function
                    const assignmentKind = getAssignmentDeclarationKind(node);
                    if (assignmentKind === AssignmentDeclarationKind.Prototype || assignmentKind === AssignmentDeclarationKind.PrototypeProperty) {
                        const symbol = getSymbolOfNode(node.left);
                        if (symbol && symbol.parent && !findAncestor(symbol.parent.valueDeclaration, d => node === d)) {
                            node = symbol.parent.valueDeclaration;
                        }
                    }
                }
                if (!node) {
                    return undefined;
                }
                switch (node.kind) {
                    case SyntaxKind.VariableStatement:
                    case SyntaxKind.ClassDeclaration:
                    case SyntaxKind.ClassExpression:
                    case SyntaxKind.InterfaceDeclaration:
                    case SyntaxKind.CallSignature:
                    case SyntaxKind.ConstructSignature:
                    case SyntaxKind.MethodSignature:
                    case SyntaxKind.FunctionType:
                    case SyntaxKind.ConstructorType:
                    case SyntaxKind.JSDocFunctionType:
                    case SyntaxKind.FunctionDeclaration:
                    case SyntaxKind.MethodDeclaration:
                    case SyntaxKind.FunctionExpression:
                    case SyntaxKind.ArrowFunction:
                    case SyntaxKind.TypeAliasDeclaration:
                    case SyntaxKind.JSDocTemplateTag:
                    case SyntaxKind.JSDocTypedefTag:
                    case SyntaxKind.JSDocEnumTag:
                    case SyntaxKind.JSDocCallbackTag:
                    case SyntaxKind.MappedType:
                    case SyntaxKind.ConditionalType:
                        const outerTypeParameters = getOuterTypeParameters(node, includeThisTypes);
                        if (node.kind === SyntaxKind.MappedType) {
                            return append(outerTypeParameters, getDeclaredTypeOfTypeParameter(getSymbolOfNode((<MappedTypeNode>node).typeParameter)));
                        }
                        else if (node.kind === SyntaxKind.ConditionalType) {
                            return concatenate(outerTypeParameters, getInferTypeParameters(<ConditionalTypeNode>node));
                        }
                        else if (node.kind === SyntaxKind.VariableStatement && !isInJSFile(node)) {
                            break;
                        }
                        const outerAndOwnTypeParameters = appendTypeParameters(outerTypeParameters, getEffectiveTypeParameterDeclarations(<DeclarationWithTypeParameters>node));
                        const thisType = includeThisTypes &&
                            (node.kind === SyntaxKind.ClassDeclaration || node.kind === SyntaxKind.ClassExpression || node.kind === SyntaxKind.InterfaceDeclaration || isJSConstructor(node)) &&
                            getDeclaredTypeOfClassOrInterface(getSymbolOfNode(node as ClassLikeDeclaration | InterfaceDeclaration)).thisType;
                        return thisType ? append(outerAndOwnTypeParameters, thisType) : outerAndOwnTypeParameters;
                    case SyntaxKind.JSDocParameterTag:
                        const paramSymbol = getParameterSymbolFromJSDoc(node as JSDocParameterTag);
                        if (paramSymbol) {
                            node = paramSymbol.valueDeclaration;
                        }
                        break;
                }
            }
        }

        // The outer type parameters are those defined by enclosing generic classes, methods, or functions.
        function getOuterTypeParametersOfClassOrInterface(symbol: Symbol): TypeParameter[] | undefined {
            const declaration = symbol.flags & SymbolFlags.Class ? symbol.valueDeclaration : getDeclarationOfKind(symbol, SyntaxKind.InterfaceDeclaration)!;
            Debug.assert(!!declaration, "Class was missing valueDeclaration -OR- non-class had no interface declarations");
            return getOuterTypeParameters(declaration);
        }

        // The local type parameters are the combined set of type parameters from all declarations of the class,
        // interface, or type alias.
        function getLocalTypeParametersOfClassOrInterfaceOrTypeAlias(symbol: Symbol): TypeParameter[] | undefined {
            let result: TypeParameter[] | undefined;
            for (const node of symbol.declarations) {
                if (node.kind === SyntaxKind.InterfaceDeclaration ||
                    node.kind === SyntaxKind.ClassDeclaration ||
                    node.kind === SyntaxKind.ClassExpression ||
                    isJSConstructor(node) ||
                    isTypeAlias(node)) {
                    const declaration = <InterfaceDeclaration | TypeAliasDeclaration | JSDocTypedefTag | JSDocCallbackTag>node;
                    result = appendTypeParameters(result, getEffectiveTypeParameterDeclarations(declaration));
                }
            }
            return result;
        }

        // The full set of type parameters for a generic class or interface type consists of its outer type parameters plus
        // its locally declared type parameters.
        function getTypeParametersOfClassOrInterface(symbol: Symbol): TypeParameter[] | undefined {
            return concatenate(getOuterTypeParametersOfClassOrInterface(symbol), getLocalTypeParametersOfClassOrInterfaceOrTypeAlias(symbol));
        }

        // A type is a mixin constructor if it has a single construct signature taking no type parameters and a single
        // rest parameter of type any[].
        function isMixinConstructorType(type: Type) {
            const signatures = getSignaturesOfType(type, SignatureKind.Construct);
            if (signatures.length === 1) {
                const s = signatures[0];
                return !s.typeParameters && s.parameters.length === 1 && signatureHasRestParameter(s) && getElementTypeOfArrayType(getTypeOfParameter(s.parameters[0])) === anyType;
            }
            return false;
        }

        function isConstructorType(type: Type): boolean {
            if (getSignaturesOfType(type, SignatureKind.Construct).length > 0) {
                return true;
            }
            if (type.flags & TypeFlags.TypeVariable) {
                const constraint = getBaseConstraintOfType(type);
                return !!constraint && isMixinConstructorType(constraint);
            }
            return false;
        }

        function getBaseTypeNodeOfClass(type: InterfaceType): ExpressionWithTypeArguments | undefined {
            return getEffectiveBaseTypeNode(type.symbol.valueDeclaration as ClassLikeDeclaration);
        }

        function getConstructorsForTypeArguments(type: Type, typeArgumentNodes: readonly TypeNode[] | undefined, location: Node): readonly Signature[] {
            const typeArgCount = length(typeArgumentNodes);
            const isJavascript = isInJSFile(location);
            return filter(getSignaturesOfType(type, SignatureKind.Construct),
                sig => (isJavascript || typeArgCount >= getMinTypeArgumentCount(sig.typeParameters)) && typeArgCount <= length(sig.typeParameters));
        }

        function getInstantiatedConstructorsForTypeArguments(type: Type, typeArgumentNodes: readonly TypeNode[] | undefined, location: Node): readonly Signature[] {
            const signatures = getConstructorsForTypeArguments(type, typeArgumentNodes, location);
            const typeArguments = map(typeArgumentNodes, getTypeFromTypeNode);
            return sameMap<Signature>(signatures, sig => some(sig.typeParameters) ? getSignatureInstantiation(sig, typeArguments, isInJSFile(location)) : sig);
        }

        /**
         * The base constructor of a class can resolve to
         * * undefinedType if the class has no extends clause,
         * * unknownType if an error occurred during resolution of the extends expression,
         * * nullType if the extends expression is the null value,
         * * anyType if the extends expression has type any, or
         * * an object type with at least one construct signature.
         */
        function getBaseConstructorTypeOfClass(type: InterfaceType): Type {
            if (!type.resolvedBaseConstructorType) {
                const decl = <ClassLikeDeclaration>type.symbol.valueDeclaration;
                const extended = getEffectiveBaseTypeNode(decl);
                const baseTypeNode = getBaseTypeNodeOfClass(type);
                if (!baseTypeNode) {
                    return type.resolvedBaseConstructorType = undefinedType;
                }
                if (!pushTypeResolution(type, TypeSystemPropertyName.ResolvedBaseConstructorType)) {
                    return errorType;
                }
                const baseConstructorType = checkExpression(baseTypeNode.expression);
                if (extended && baseTypeNode !== extended) {
                    Debug.assert(!extended.typeArguments); // Because this is in a JS file, and baseTypeNode is in an @extends tag
                    checkExpression(extended.expression);
                }
                if (baseConstructorType.flags & (TypeFlags.Object | TypeFlags.Intersection)) {
                    // Resolving the members of a class requires us to resolve the base class of that class.
                    // We force resolution here such that we catch circularities now.
                    resolveStructuredTypeMembers(<ObjectType>baseConstructorType);
                }
                if (!popTypeResolution()) {
                    error(type.symbol.valueDeclaration, Diagnostics._0_is_referenced_directly_or_indirectly_in_its_own_base_expression, symbolToString(type.symbol));
                    return type.resolvedBaseConstructorType = errorType;
                }
                if (!(baseConstructorType.flags & TypeFlags.Any) && baseConstructorType !== nullWideningType && !isConstructorType(baseConstructorType)) {
                    const err = error(baseTypeNode.expression, Diagnostics.Type_0_is_not_a_constructor_function_type, typeToString(baseConstructorType));
                    if (baseConstructorType.flags & TypeFlags.TypeParameter) {
                        const constraint = getConstraintFromTypeParameter(baseConstructorType);
                        let ctorReturn: Type = unknownType;
                        if (constraint) {
                            const ctorSig = getSignaturesOfType(constraint, SignatureKind.Construct);
                            if (ctorSig[0]) {
                                ctorReturn = getReturnTypeOfSignature(ctorSig[0]);
                            }
                        }
                        addRelatedInfo(err, createDiagnosticForNode(baseConstructorType.symbol.declarations[0], Diagnostics.Did_you_mean_for_0_to_be_constrained_to_type_new_args_Colon_any_1, symbolToString(baseConstructorType.symbol), typeToString(ctorReturn)));
                    }
                    return type.resolvedBaseConstructorType = errorType;
                }
                type.resolvedBaseConstructorType = baseConstructorType;
            }
            return type.resolvedBaseConstructorType;
        }

        function getImplementsTypes(type: InterfaceType): BaseType[] {
            let resolvedImplementsTypes: BaseType[] = emptyArray;
            for (const declaration of type.symbol.declarations) {
                const implementsTypeNodes = getEffectiveImplementsTypeNodes(declaration as ClassLikeDeclaration);
                if (!implementsTypeNodes) continue;
                for (const node of implementsTypeNodes) {
                    const implementsType = getTypeFromTypeNode(node);
                    if (implementsType !== errorType) {
                        if (resolvedImplementsTypes === emptyArray) {
                            resolvedImplementsTypes = [<ObjectType>implementsType];
                        }
                        else {
                            resolvedImplementsTypes.push(implementsType);
                        }
                    }
                }
            }
            return resolvedImplementsTypes;
        }

        function reportCircularBaseType(node: Node, type: Type) {
            error(node, Diagnostics.Type_0_recursively_references_itself_as_a_base_type, typeToString(type, /*enclosingDeclaration*/ undefined, TypeFormatFlags.WriteArrayAsGenericType));
        }

        function getBaseTypes(type: InterfaceType): BaseType[] {
            if (!type.baseTypesResolved) {
                if (pushTypeResolution(type, TypeSystemPropertyName.ResolvedBaseTypes)) {
                    if (type.objectFlags & ObjectFlags.Tuple) {
                        type.resolvedBaseTypes = [getTupleBaseType(<TupleType>type)];
                    }
                    else if (type.symbol.flags & (SymbolFlags.Class | SymbolFlags.Interface)) {
                        if (type.symbol.flags & SymbolFlags.Class) {
                            resolveBaseTypesOfClass(type);
                        }
                        if (type.symbol.flags & SymbolFlags.Interface) {
                            resolveBaseTypesOfInterface(type);
                        }
                    }
                    else {
                        Debug.fail("type must be class or interface");
                    }
                    if (!popTypeResolution()) {
                        for (const declaration of type.symbol.declarations) {
                            if (declaration.kind === SyntaxKind.ClassDeclaration || declaration.kind === SyntaxKind.InterfaceDeclaration) {
                                reportCircularBaseType(declaration, type);
                            }
                        }
                    }
                }
                type.baseTypesResolved = true;
            }
            return type.resolvedBaseTypes;
        }

        function getTupleBaseType(type: TupleType) {
            const elementTypes = sameMap(type.typeParameters, (t, i) => type.elementFlags[i] & ElementFlags.Variadic ? getIndexedAccessType(t, numberType) : t);
            return createArrayType(getUnionType(elementTypes || emptyArray), type.readonly);
        }

        function resolveBaseTypesOfClass(type: InterfaceType) {
            type.resolvedBaseTypes = resolvingEmptyArray;
            const baseConstructorType = getApparentType(getBaseConstructorTypeOfClass(type));
            if (!(baseConstructorType.flags & (TypeFlags.Object | TypeFlags.Intersection | TypeFlags.Any))) {
                return type.resolvedBaseTypes = emptyArray;
            }
            const baseTypeNode = getBaseTypeNodeOfClass(type)!;
            let baseType: Type;
            const originalBaseType = baseConstructorType.symbol ? getDeclaredTypeOfSymbol(baseConstructorType.symbol) : undefined;
            if (baseConstructorType.symbol && baseConstructorType.symbol.flags & SymbolFlags.Class &&
                areAllOuterTypeParametersApplied(originalBaseType!)) {
                // When base constructor type is a class with no captured type arguments we know that the constructors all have the same type parameters as the
                // class and all return the instance type of the class. There is no need for further checks and we can apply the
                // type arguments in the same manner as a type reference to get the same error reporting experience.
                baseType = getTypeFromClassOrInterfaceReference(baseTypeNode, baseConstructorType.symbol);
            }
            else if (baseConstructorType.flags & TypeFlags.Any) {
                baseType = baseConstructorType;
            }
            else {
                // The class derives from a "class-like" constructor function, check that we have at least one construct signature
                // with a matching number of type parameters and use the return type of the first instantiated signature. Elsewhere
                // we check that all instantiated signatures return the same type.
                const constructors = getInstantiatedConstructorsForTypeArguments(baseConstructorType, baseTypeNode.typeArguments, baseTypeNode);
                if (!constructors.length) {
                    error(baseTypeNode.expression, Diagnostics.No_base_constructor_has_the_specified_number_of_type_arguments);
                    return type.resolvedBaseTypes = emptyArray;
                }
                baseType = getReturnTypeOfSignature(constructors[0]);
            }

            if (baseType === errorType) {
                return type.resolvedBaseTypes = emptyArray;
            }
            const reducedBaseType = getReducedType(baseType);
            if (!isValidBaseType(reducedBaseType)) {
                const elaboration = elaborateNeverIntersection(/*errorInfo*/ undefined, baseType);
                const diagnostic = chainDiagnosticMessages(elaboration, Diagnostics.Base_constructor_return_type_0_is_not_an_object_type_or_intersection_of_object_types_with_statically_known_members, typeToString(reducedBaseType));
                diagnostics.add(createDiagnosticForNodeFromMessageChain(baseTypeNode.expression, diagnostic));
                return type.resolvedBaseTypes = emptyArray;
            }
            if (type === reducedBaseType || hasBaseType(reducedBaseType, type)) {
                error(type.symbol.valueDeclaration, Diagnostics.Type_0_recursively_references_itself_as_a_base_type,
                    typeToString(type, /*enclosingDeclaration*/ undefined, TypeFormatFlags.WriteArrayAsGenericType));
                return type.resolvedBaseTypes = emptyArray;
            }
            if (type.resolvedBaseTypes === resolvingEmptyArray) {
                // Circular reference, likely through instantiation of default parameters
                // (otherwise there'd be an error from hasBaseType) - this is fine, but `.members` should be reset
                // as `getIndexedAccessType` via `instantiateType` via `getTypeFromClassOrInterfaceReference` forces a
                // partial instantiation of the members without the base types fully resolved
                type.members = undefined;
            }
            return type.resolvedBaseTypes = [reducedBaseType];
        }

        function areAllOuterTypeParametersApplied(type: Type): boolean { // TODO: GH#18217 Shouldn't this take an InterfaceType?
            // An unapplied type parameter has its symbol still the same as the matching argument symbol.
            // Since parameters are applied outer-to-inner, only the last outer parameter needs to be checked.
            const outerTypeParameters = (<InterfaceType>type).outerTypeParameters;
            if (outerTypeParameters) {
                const last = outerTypeParameters.length - 1;
                const typeArguments = getTypeArguments(<TypeReference>type);
                return outerTypeParameters[last].symbol !== typeArguments[last].symbol;
            }
            return true;
        }

        // A valid base type is `any`, an object type or intersection of object types.
        function isValidBaseType(type: Type): type is BaseType {
            if (type.flags & TypeFlags.TypeParameter) {
                const constraint = getBaseConstraintOfType(type);
                if (constraint) {
                    return isValidBaseType(constraint);
                }
            }
            // TODO: Given that we allow type parmeters here now, is this `!isGenericMappedType(type)` check really needed?
            // There's no reason a `T` should be allowed while a `Readonly<T>` should not.
            return !!(type.flags & (TypeFlags.Object | TypeFlags.NonPrimitive | TypeFlags.Any) && !isGenericMappedType(type) ||
                type.flags & TypeFlags.Intersection && every((<IntersectionType>type).types, isValidBaseType));
        }

        function resolveBaseTypesOfInterface(type: InterfaceType): void {
            type.resolvedBaseTypes = type.resolvedBaseTypes || emptyArray;
            for (const declaration of type.symbol.declarations) {
                if (declaration.kind === SyntaxKind.InterfaceDeclaration && getInterfaceBaseTypeNodes(<InterfaceDeclaration>declaration)) {
                    for (const node of getInterfaceBaseTypeNodes(<InterfaceDeclaration>declaration)!) {
                        const baseType = getReducedType(getTypeFromTypeNode(node));
                        if (baseType !== errorType) {
                            if (isValidBaseType(baseType)) {
                                if (type !== baseType && !hasBaseType(baseType, type)) {
                                    if (type.resolvedBaseTypes === emptyArray) {
                                        type.resolvedBaseTypes = [<ObjectType>baseType];
                                    }
                                    else {
                                        type.resolvedBaseTypes.push(baseType);
                                    }
                                }
                                else {
                                    reportCircularBaseType(declaration, type);
                                }
                            }
                            else {
                                error(node, Diagnostics.An_interface_can_only_extend_an_object_type_or_intersection_of_object_types_with_statically_known_members);
                            }
                        }
                    }
                }
            }
        }

        /**
         * Returns true if the interface given by the symbol is free of "this" references.
         *
         * Specifically, the result is true if the interface itself contains no references
         * to "this" in its body, if all base types are interfaces,
         * and if none of the base interfaces have a "this" type.
         */
        function isThislessInterface(symbol: Symbol): boolean {
            for (const declaration of symbol.declarations) {
                if (declaration.kind === SyntaxKind.InterfaceDeclaration) {
                    if (declaration.flags & NodeFlags.ContainsThis) {
                        return false;
                    }
                    const baseTypeNodes = getInterfaceBaseTypeNodes(<InterfaceDeclaration>declaration);
                    if (baseTypeNodes) {
                        for (const node of baseTypeNodes) {
                            if (isEntityNameExpression(node.expression)) {
                                const baseSymbol = resolveEntityName(node.expression, SymbolFlags.Type, /*ignoreErrors*/ true);
                                if (!baseSymbol || !(baseSymbol.flags & SymbolFlags.Interface) || getDeclaredTypeOfClassOrInterface(baseSymbol).thisType) {
                                    return false;
                                }
                            }
                        }
                    }
                }
            }
            return true;
        }

        function getDeclaredTypeOfClassOrInterface(symbol: Symbol): InterfaceType {
            let links = getSymbolLinks(symbol);
            const originalLinks = links;
            if (!links.declaredType) {
                const kind = symbol.flags & SymbolFlags.Class ? ObjectFlags.Class : ObjectFlags.Interface;
                const merged = mergeJSSymbols(symbol, getAssignedClassSymbol(symbol.valueDeclaration));
                if (merged) {
                    // note:we overwrite links because we just cloned the symbol
                    symbol = links = merged;
                }

                const type = originalLinks.declaredType = links.declaredType = <InterfaceType>createObjectType(kind, symbol);
                const outerTypeParameters = getOuterTypeParametersOfClassOrInterface(symbol);
                const localTypeParameters = getLocalTypeParametersOfClassOrInterfaceOrTypeAlias(symbol);
                // A class or interface is generic if it has type parameters or a "this" type. We always give classes a "this" type
                // because it is not feasible to analyze all members to determine if the "this" type escapes the class (in particular,
                // property types inferred from initializers and method return types inferred from return statements are very hard
                // to exhaustively analyze). We give interfaces a "this" type if we can't definitely determine that they are free of
                // "this" references.
                if (outerTypeParameters || localTypeParameters || kind === ObjectFlags.Class || !isThislessInterface(symbol)) {
                    type.objectFlags |= ObjectFlags.Reference;
                    type.typeParameters = concatenate(outerTypeParameters, localTypeParameters);
                    type.outerTypeParameters = outerTypeParameters;
                    type.localTypeParameters = localTypeParameters;
                    (<GenericType>type).instantiations = new Map<string, TypeReference>();
                    (<GenericType>type).instantiations.set(getTypeListId(type.typeParameters), <GenericType>type);
                    (<GenericType>type).target = <GenericType>type;
                    (<GenericType>type).resolvedTypeArguments = type.typeParameters;
                    type.thisType = createTypeParameter(symbol);
                    type.thisType.isThisType = true;
                    type.thisType.constraint = type;
                }
            }
            return <InterfaceType>links.declaredType;
        }

        function getDeclaredTypeOfTypeAlias(symbol: Symbol): Type {
            const links = getSymbolLinks(symbol);
            if (!links.declaredType) {
                // Note that we use the links object as the target here because the symbol object is used as the unique
                // identity for resolution of the 'type' property in SymbolLinks.
                if (!pushTypeResolution(symbol, TypeSystemPropertyName.DeclaredType)) {
                    return errorType;
                }

                const declaration = Debug.checkDefined(find(symbol.declarations, isTypeAlias), "Type alias symbol with no valid declaration found");
                const typeNode = isJSDocTypeAlias(declaration) ? declaration.typeExpression : declaration.type;
                // If typeNode is missing, we will error in checkJSDocTypedefTag.
                let type = typeNode ? getTypeFromTypeNode(typeNode) : errorType;

                if (popTypeResolution()) {
                    const typeParameters = getLocalTypeParametersOfClassOrInterfaceOrTypeAlias(symbol);
                    if (typeParameters) {
                        // Initialize the instantiation cache for generic type aliases. The declared type corresponds to
                        // an instantiation of the type alias with the type parameters supplied as type arguments.
                        links.typeParameters = typeParameters;
                        links.instantiations = new Map<string, Type>();
                        links.instantiations.set(getTypeListId(typeParameters), type);
                    }
                }
                else {
                    type = errorType;
                    error(isNamedDeclaration(declaration) ? declaration.name : declaration || declaration, Diagnostics.Type_alias_0_circularly_references_itself, symbolToString(symbol));
                }
                links.declaredType = type;
            }
            return links.declaredType;
        }

        function isStringConcatExpression(expr: Node): boolean {
            if (isStringLiteralLike(expr)) {
                return true;
            }
            else if (expr.kind === SyntaxKind.BinaryExpression) {
                return isStringConcatExpression((<BinaryExpression>expr).left) && isStringConcatExpression((<BinaryExpression>expr).right);
            }
            return false;
        }

        function isLiteralEnumMember(member: EnumMember) {
            const expr = member.initializer;
            if (!expr) {
                return !(member.flags & NodeFlags.Ambient);
            }
            switch (expr.kind) {
                case SyntaxKind.StringLiteral:
                case SyntaxKind.NumericLiteral:
                case SyntaxKind.NoSubstitutionTemplateLiteral:
                    return true;
                case SyntaxKind.PrefixUnaryExpression:
                    return (<PrefixUnaryExpression>expr).operator === SyntaxKind.MinusToken &&
                        (<PrefixUnaryExpression>expr).operand.kind === SyntaxKind.NumericLiteral;
                case SyntaxKind.Identifier:
                    return nodeIsMissing(expr) || !!getSymbolOfNode(member.parent).exports!.get((<Identifier>expr).escapedText);
                case SyntaxKind.BinaryExpression:
                    return isStringConcatExpression(expr);
                default:
                    return false;
            }
        }

        function getEnumKind(symbol: Symbol): EnumKind {
            const links = getSymbolLinks(symbol);
            if (links.enumKind !== undefined) {
                return links.enumKind;
            }
            let hasNonLiteralMember = false;
            for (const declaration of symbol.declarations) {
                if (declaration.kind === SyntaxKind.EnumDeclaration) {
                    for (const member of (<EnumDeclaration>declaration).members) {
                        if (member.initializer && isStringLiteralLike(member.initializer)) {
                            return links.enumKind = EnumKind.Literal;
                        }
                        if (!isLiteralEnumMember(member)) {
                            hasNonLiteralMember = true;
                        }
                    }
                }
            }
            return links.enumKind = hasNonLiteralMember ? EnumKind.Numeric : EnumKind.Literal;
        }

        function getBaseTypeOfEnumLiteralType(type: Type) {
            return type.flags & TypeFlags.EnumLiteral && !(type.flags & TypeFlags.Union) ? getDeclaredTypeOfSymbol(getParentOfSymbol(type.symbol)!) : type;
        }

        function getDeclaredTypeOfEnum(symbol: Symbol): Type {
            const links = getSymbolLinks(symbol);
            if (links.declaredType) {
                return links.declaredType;
            }
            if (getEnumKind(symbol) === EnumKind.Literal) {
                enumCount++;
                const memberTypeList: Type[] = [];
                for (const declaration of symbol.declarations) {
                    if (declaration.kind === SyntaxKind.EnumDeclaration) {
                        for (const member of (<EnumDeclaration>declaration).members) {
                            const value = getEnumMemberValue(member);
                            const memberType = getFreshTypeOfLiteralType(getLiteralType(value !== undefined ? value : 0, enumCount, getSymbolOfNode(member)));
                            getSymbolLinks(getSymbolOfNode(member)).declaredType = memberType;
                            memberTypeList.push(getRegularTypeOfLiteralType(memberType));
                        }
                    }
                }
                if (memberTypeList.length) {
                    const enumType = getUnionType(memberTypeList, UnionReduction.Literal, symbol, /*aliasTypeArguments*/ undefined);
                    if (enumType.flags & TypeFlags.Union) {
                        enumType.flags |= TypeFlags.EnumLiteral;
                        enumType.symbol = symbol;
                    }
                    return links.declaredType = enumType;
                }
            }
            const enumType = createType(TypeFlags.Enum);
            enumType.symbol = symbol;
            return links.declaredType = enumType;
        }

        function getDeclaredTypeOfEnumMember(symbol: Symbol): Type {
            const links = getSymbolLinks(symbol);
            if (!links.declaredType) {
                const enumType = getDeclaredTypeOfEnum(getParentOfSymbol(symbol)!);
                if (!links.declaredType) {
                    links.declaredType = enumType;
                }
            }
            return links.declaredType;
        }

        function getDeclaredTypeOfTypeParameter(symbol: Symbol): TypeParameter {
            const links = getSymbolLinks(symbol);
            return links.declaredType || (links.declaredType = createTypeParameter(symbol));
        }

        function getDeclaredTypeOfAlias(symbol: Symbol): Type {
            const links = getSymbolLinks(symbol);
            return links.declaredType || (links.declaredType = getDeclaredTypeOfSymbol(resolveAlias(symbol)));
        }

        function getDeclaredTypeOfSymbol(symbol: Symbol): Type {
            return tryGetDeclaredTypeOfSymbol(symbol) || errorType;
        }

        function tryGetDeclaredTypeOfSymbol(symbol: Symbol): Type | undefined {
            if (symbol.flags & (SymbolFlags.Class | SymbolFlags.Interface)) {
                return getDeclaredTypeOfClassOrInterface(symbol);
            }
            if (symbol.flags & SymbolFlags.TypeAlias) {
                return getDeclaredTypeOfTypeAlias(symbol);
            }
            if (symbol.flags & SymbolFlags.TypeParameter) {
                return getDeclaredTypeOfTypeParameter(symbol);
            }
            if (symbol.flags & SymbolFlags.Enum) {
                return getDeclaredTypeOfEnum(symbol);
            }
            if (symbol.flags & SymbolFlags.EnumMember) {
                return getDeclaredTypeOfEnumMember(symbol);
            }
            if (symbol.flags & SymbolFlags.Alias) {
                return getDeclaredTypeOfAlias(symbol);
            }
            return undefined;
        }

        /**
         * A type is free of this references if it's the any, string, number, boolean, symbol, or void keyword, a string
         * literal type, an array with an element type that is free of this references, or a type reference that is
         * free of this references.
         */
        function isThislessType(node: TypeNode): boolean {
            switch (node.kind) {
                case SyntaxKind.AnyKeyword:
                case SyntaxKind.UnknownKeyword:
                case SyntaxKind.StringKeyword:
                case SyntaxKind.NumberKeyword:
                case SyntaxKind.BigIntKeyword:
                case SyntaxKind.BooleanKeyword:
                case SyntaxKind.SymbolKeyword:
                case SyntaxKind.ObjectKeyword:
                case SyntaxKind.VoidKeyword:
                case SyntaxKind.UndefinedKeyword:
                case SyntaxKind.NeverKeyword:
                case SyntaxKind.LiteralType:
                    return true;
                case SyntaxKind.ArrayType:
                    return isThislessType((<ArrayTypeNode>node).elementType);
                case SyntaxKind.TypeReference:
                    return !(node as TypeReferenceNode).typeArguments || (node as TypeReferenceNode).typeArguments!.every(isThislessType);
            }
            return false;
        }

        /** A type parameter is thisless if its constraint is thisless, or if it has no constraint. */
        function isThislessTypeParameter(node: TypeParameterDeclaration) {
            const constraint = getEffectiveConstraintOfTypeParameter(node);
            return !constraint || isThislessType(constraint);
        }

        /**
         * A variable-like declaration is free of this references if it has a type annotation
         * that is thisless, or if it has no type annotation and no initializer (and is thus of type any).
         */
        function isThislessVariableLikeDeclaration(node: VariableLikeDeclaration): boolean {
            const typeNode = getEffectiveTypeAnnotationNode(node);
            return typeNode ? isThislessType(typeNode) : !hasInitializer(node);
        }

        /**
         * A function-like declaration is considered free of `this` references if it has a return type
         * annotation that is free of this references and if each parameter is thisless and if
         * each type parameter (if present) is thisless.
         */
        function isThislessFunctionLikeDeclaration(node: FunctionLikeDeclaration): boolean {
            const returnType = getEffectiveReturnTypeNode(node);
            const typeParameters = getEffectiveTypeParameterDeclarations(node);
            return (node.kind === SyntaxKind.Constructor || (!!returnType && isThislessType(returnType))) &&
                node.parameters.every(isThislessVariableLikeDeclaration) &&
                typeParameters.every(isThislessTypeParameter);
        }

        /**
         * Returns true if the class or interface member given by the symbol is free of "this" references. The
         * function may return false for symbols that are actually free of "this" references because it is not
         * feasible to perform a complete analysis in all cases. In particular, property members with types
         * inferred from their initializers and function members with inferred return types are conservatively
         * assumed not to be free of "this" references.
         */
        function isThisless(symbol: Symbol): boolean {
            if (symbol.declarations && symbol.declarations.length === 1) {
                const declaration = symbol.declarations[0];
                if (declaration) {
                    switch (declaration.kind) {
                        case SyntaxKind.PropertyDeclaration:
                        case SyntaxKind.PropertySignature:
                            return isThislessVariableLikeDeclaration(<VariableLikeDeclaration>declaration);
                        case SyntaxKind.MethodDeclaration:
                        case SyntaxKind.MethodSignature:
                        case SyntaxKind.Constructor:
                        case SyntaxKind.GetAccessor:
                        case SyntaxKind.SetAccessor:
                            return isThislessFunctionLikeDeclaration(<FunctionLikeDeclaration | AccessorDeclaration>declaration);
                    }
                }
            }
            return false;
        }

        // The mappingThisOnly flag indicates that the only type parameter being mapped is "this". When the flag is true,
        // we check symbols to see if we can quickly conclude they are free of "this" references, thus needing no instantiation.
        function createInstantiatedSymbolTable(symbols: Symbol[], mapper: TypeMapper, mappingThisOnly: boolean): SymbolTable {
            const result = createSymbolTable();
            for (const symbol of symbols) {
                result.set(symbol.escapedName, mappingThisOnly && isThisless(symbol) ? symbol : instantiateSymbol(symbol, mapper));
            }
            return result;
        }

        function addInheritedMembers(symbols: SymbolTable, baseSymbols: Symbol[]) {
            for (const s of baseSymbols) {
                if (!symbols.has(s.escapedName) && !isStaticPrivateIdentifierProperty(s)) {
                    symbols.set(s.escapedName, s);
                }
            }
        }

        function isStaticPrivateIdentifierProperty(s: Symbol): boolean {
            return !!s.valueDeclaration && isPrivateIdentifierPropertyDeclaration(s.valueDeclaration) && hasSyntacticModifier(s.valueDeclaration, ModifierFlags.Static);
        }

        function resolveDeclaredMembers(type: InterfaceType): InterfaceTypeWithDeclaredMembers {
            if (!(<InterfaceTypeWithDeclaredMembers>type).declaredProperties) {
                const symbol = type.symbol;
                const members = getMembersOfSymbol(symbol);
                (<InterfaceTypeWithDeclaredMembers>type).declaredProperties = getNamedMembers(members);
                // Start with signatures at empty array in case of recursive types
                (<InterfaceTypeWithDeclaredMembers>type).declaredCallSignatures = emptyArray;
                (<InterfaceTypeWithDeclaredMembers>type).declaredConstructSignatures = emptyArray;

                (<InterfaceTypeWithDeclaredMembers>type).declaredCallSignatures = getSignaturesOfSymbol(members.get(InternalSymbolName.Call));
                (<InterfaceTypeWithDeclaredMembers>type).declaredConstructSignatures = getSignaturesOfSymbol(members.get(InternalSymbolName.New));
                (<InterfaceTypeWithDeclaredMembers>type).declaredStringIndexInfo = getIndexInfoOfSymbol(symbol, IndexKind.String);
                (<InterfaceTypeWithDeclaredMembers>type).declaredNumberIndexInfo = getIndexInfoOfSymbol(symbol, IndexKind.Number);
            }
            return <InterfaceTypeWithDeclaredMembers>type;
        }

        /**
         * Indicates whether a type can be used as a property name.
         */
        function isTypeUsableAsPropertyName(type: Type): type is StringLiteralType | NumberLiteralType | UniqueESSymbolType {
            return !!(type.flags & TypeFlags.StringOrNumberLiteralOrUnique);
        }

        /**
         * Indicates whether a declaration name is definitely late-bindable.
         * A declaration name is only late-bindable if:
         * - It is a `ComputedPropertyName`.
         * - Its expression is an `Identifier` or either a `PropertyAccessExpression` an
         * `ElementAccessExpression` consisting only of these same three types of nodes.
         * - The type of its expression is a string or numeric literal type, or is a `unique symbol` type.
         */
        function isLateBindableName(node: DeclarationName): node is LateBoundName {
            if (!isComputedPropertyName(node) && !isElementAccessExpression(node)) {
                return false;
            }
            const expr = isComputedPropertyName(node) ? node.expression : node.argumentExpression;
            return isEntityNameExpression(expr)
                && isTypeUsableAsPropertyName(isComputedPropertyName(node) ? checkComputedPropertyName(node) : checkExpressionCached(expr));
        }

        function isLateBoundName(name: __String): boolean {
            return (name as string).charCodeAt(0) === CharacterCodes._ &&
                (name as string).charCodeAt(1) === CharacterCodes._ &&
                (name as string).charCodeAt(2) === CharacterCodes.at;
        }

        /**
         * Indicates whether a declaration has a late-bindable dynamic name.
         */
        function hasLateBindableName(node: Declaration): node is LateBoundDeclaration | LateBoundBinaryExpressionDeclaration {
            const name = getNameOfDeclaration(node);
            return !!name && isLateBindableName(name);
        }

        /**
         * Indicates whether a declaration has a dynamic name that cannot be late-bound.
         */
        function hasNonBindableDynamicName(node: Declaration) {
            return hasDynamicName(node) && !hasLateBindableName(node);
        }

        /**
         * Indicates whether a declaration name is a dynamic name that cannot be late-bound.
         */
        function isNonBindableDynamicName(node: DeclarationName) {
            return isDynamicName(node) && !isLateBindableName(node);
        }

        /**
         * Gets the symbolic name for a member from its type.
         */
        function getPropertyNameFromType(type: StringLiteralType | NumberLiteralType | UniqueESSymbolType): __String {
            if (type.flags & TypeFlags.UniqueESSymbol) {
                return (<UniqueESSymbolType>type).escapedName;
            }
            if (type.flags & (TypeFlags.StringLiteral | TypeFlags.NumberLiteral)) {
                return escapeLeadingUnderscores("" + (<StringLiteralType | NumberLiteralType>type).value);
            }
            return Debug.fail();
        }

        /**
         * Adds a declaration to a late-bound dynamic member. This performs the same function for
         * late-bound members that `addDeclarationToSymbol` in binder.ts performs for early-bound
         * members.
         */
        function addDeclarationToLateBoundSymbol(symbol: Symbol, member: LateBoundDeclaration | BinaryExpression, symbolFlags: SymbolFlags) {
            Debug.assert(!!(getCheckFlags(symbol) & CheckFlags.Late), "Expected a late-bound symbol.");
            symbol.flags |= symbolFlags;
            getSymbolLinks(member.symbol).lateSymbol = symbol;
            if (!symbol.declarations) {
                symbol.declarations = [member];
            }
            else {
                symbol.declarations.push(member);
            }
            if (symbolFlags & SymbolFlags.Value) {
                if (!symbol.valueDeclaration || symbol.valueDeclaration.kind !== member.kind) {
                    symbol.valueDeclaration = member;
                }
            }
        }

        /**
         * Performs late-binding of a dynamic member. This performs the same function for
         * late-bound members that `declareSymbol` in binder.ts performs for early-bound
         * members.
         *
         * If a symbol is a dynamic name from a computed property, we perform an additional "late"
         * binding phase to attempt to resolve the name for the symbol from the type of the computed
         * property's expression. If the type of the expression is a string-literal, numeric-literal,
         * or unique symbol type, we can use that type as the name of the symbol.
         *
         * For example, given:
         *
         *   const x = Symbol();
         *
         *   interface I {
         *     [x]: number;
         *   }
         *
         * The binder gives the property `[x]: number` a special symbol with the name "__computed".
         * In the late-binding phase we can type-check the expression `x` and see that it has a
         * unique symbol type which we can then use as the name of the member. This allows users
         * to define custom symbols that can be used in the members of an object type.
         *
         * @param parent The containing symbol for the member.
         * @param earlySymbols The early-bound symbols of the parent.
         * @param lateSymbols The late-bound symbols of the parent.
         * @param decl The member to bind.
         */
        function lateBindMember(parent: Symbol, earlySymbols: SymbolTable | undefined, lateSymbols: UnderscoreEscapedMap<TransientSymbol>, decl: LateBoundDeclaration | LateBoundBinaryExpressionDeclaration) {
            Debug.assert(!!decl.symbol, "The member is expected to have a symbol.");
            const links = getNodeLinks(decl);
            if (!links.resolvedSymbol) {
                // In the event we attempt to resolve the late-bound name of this member recursively,
                // fall back to the early-bound name of this member.
                links.resolvedSymbol = decl.symbol;
                const declName = isBinaryExpression(decl) ? decl.left : decl.name;
                const type = isElementAccessExpression(declName) ? checkExpressionCached(declName.argumentExpression) : checkComputedPropertyName(declName);
                if (isTypeUsableAsPropertyName(type)) {
                    const memberName = getPropertyNameFromType(type);
                    const symbolFlags = decl.symbol.flags;

                    // Get or add a late-bound symbol for the member. This allows us to merge late-bound accessor declarations.
                    let lateSymbol = lateSymbols.get(memberName);
                    if (!lateSymbol) lateSymbols.set(memberName, lateSymbol = createSymbol(SymbolFlags.None, memberName, CheckFlags.Late));

                    // Report an error if a late-bound member has the same name as an early-bound member,
                    // or if we have another early-bound symbol declaration with the same name and
                    // conflicting flags.
                    const earlySymbol = earlySymbols && earlySymbols.get(memberName);
                    if (lateSymbol.flags & getExcludedSymbolFlags(symbolFlags) || earlySymbol) {
                        // If we have an existing early-bound member, combine its declarations so that we can
                        // report an error at each declaration.
                        const declarations = earlySymbol ? concatenate(earlySymbol.declarations, lateSymbol.declarations) : lateSymbol.declarations;
                        const name = !(type.flags & TypeFlags.UniqueESSymbol) && unescapeLeadingUnderscores(memberName) || declarationNameToString(declName);
                        forEach(declarations, declaration => error(getNameOfDeclaration(declaration) || declaration, Diagnostics.Property_0_was_also_declared_here, name));
                        error(declName || decl, Diagnostics.Duplicate_property_0, name);
                        lateSymbol = createSymbol(SymbolFlags.None, memberName, CheckFlags.Late);
                    }
                    lateSymbol.nameType = type;
                    addDeclarationToLateBoundSymbol(lateSymbol, decl, symbolFlags);
                    if (lateSymbol.parent) {
                        Debug.assert(lateSymbol.parent === parent, "Existing symbol parent should match new one");
                    }
                    else {
                        lateSymbol.parent = parent;
                    }
                    return links.resolvedSymbol = lateSymbol;
                }
            }
            return links.resolvedSymbol;
        }

        function getResolvedMembersOrExportsOfSymbol(symbol: Symbol, resolutionKind: MembersOrExportsResolutionKind): UnderscoreEscapedMap<Symbol> {
            const links = getSymbolLinks(symbol);
            if (!links[resolutionKind]) {
                const isStatic = resolutionKind === MembersOrExportsResolutionKind.resolvedExports;
                const earlySymbols = !isStatic ? symbol.members :
                    symbol.flags & SymbolFlags.Module ? getExportsOfModuleWorker(symbol) :
                    symbol.exports;

                // In the event we recursively resolve the members/exports of the symbol, we
                // set the initial value of resolvedMembers/resolvedExports to the early-bound
                // members/exports of the symbol.
                links[resolutionKind] = earlySymbols || emptySymbols;

                // fill in any as-yet-unresolved late-bound members.
                const lateSymbols = createSymbolTable() as UnderscoreEscapedMap<TransientSymbol>;
                for (const decl of symbol.declarations) {
                    const members = getMembersOfDeclaration(decl);
                    if (members) {
                        for (const member of members) {
                            if (isStatic === hasStaticModifier(member) && hasLateBindableName(member)) {
                                lateBindMember(symbol, earlySymbols, lateSymbols, member);
                            }
                        }
                    }
                }
                const assignments = symbol.assignmentDeclarationMembers;
                if (assignments) {
                    const decls = arrayFrom(assignments.values());
                    for (const member of decls) {
                        const assignmentKind = getAssignmentDeclarationKind(member as BinaryExpression | CallExpression);
                        const isInstanceMember = assignmentKind === AssignmentDeclarationKind.PrototypeProperty
                            || assignmentKind === AssignmentDeclarationKind.ThisProperty
                            || assignmentKind === AssignmentDeclarationKind.ObjectDefinePrototypeProperty
                            || assignmentKind === AssignmentDeclarationKind.Prototype; // A straight `Prototype` assignment probably can never have a computed name
                        if (isStatic === !isInstanceMember && hasLateBindableName(member)) {
                            lateBindMember(symbol, earlySymbols, lateSymbols, member);
                        }
                    }
                }

                links[resolutionKind] = combineSymbolTables(earlySymbols, lateSymbols) || emptySymbols;
            }

            return links[resolutionKind]!;
        }

        /**
         * Gets a SymbolTable containing both the early- and late-bound members of a symbol.
         *
         * For a description of late-binding, see `lateBindMember`.
         */
        function getMembersOfSymbol(symbol: Symbol) {
            return symbol.flags & SymbolFlags.LateBindingContainer
                ? getResolvedMembersOrExportsOfSymbol(symbol, MembersOrExportsResolutionKind.resolvedMembers)
                : symbol.members || emptySymbols;
        }

        /**
         * If a symbol is the dynamic name of the member of an object type, get the late-bound
         * symbol of the member.
         *
         * For a description of late-binding, see `lateBindMember`.
         */
        function getLateBoundSymbol(symbol: Symbol): Symbol {
            if (symbol.flags & SymbolFlags.ClassMember && symbol.escapedName === InternalSymbolName.Computed) {
                const links = getSymbolLinks(symbol);
                if (!links.lateSymbol && some(symbol.declarations, hasLateBindableName)) {
                    // force late binding of members/exports. This will set the late-bound symbol
                    const parent = getMergedSymbol(symbol.parent)!;
                    if (some(symbol.declarations, hasStaticModifier)) {
                        getExportsOfSymbol(parent);
                    }
                    else {
                        getMembersOfSymbol(parent);
                    }
                }
                return links.lateSymbol || (links.lateSymbol = symbol);
            }
            return symbol;
        }

        function getTypeWithThisArgument(type: Type, thisArgument?: Type, needApparentType?: boolean): Type {
            if (getObjectFlags(type) & ObjectFlags.Reference) {
                const target = (<TypeReference>type).target;
                const typeArguments = getTypeArguments(<TypeReference>type);
                if (length(target.typeParameters) === length(typeArguments)) {
                    const ref = createTypeReference(target, concatenate(typeArguments, [thisArgument || target.thisType!]));
                    return needApparentType ? getApparentType(ref) : ref;
                }
            }
            else if (type.flags & TypeFlags.Intersection) {
                return getIntersectionType(map((<IntersectionType>type).types, t => getTypeWithThisArgument(t, thisArgument, needApparentType)));
            }
            return needApparentType ? getApparentType(type) : type;
        }

        function resolveObjectTypeMembers(type: ObjectType, source: InterfaceTypeWithDeclaredMembers, typeParameters: readonly TypeParameter[], typeArguments: readonly Type[]) {
            let mapper: TypeMapper | undefined;
            let members: SymbolTable;
            let callSignatures: readonly Signature[];
            let constructSignatures: readonly Signature[] | undefined;
            let stringIndexInfo: IndexInfo | undefined;
            let numberIndexInfo: IndexInfo | undefined;
            if (rangeEquals(typeParameters, typeArguments, 0, typeParameters.length)) {
                members = source.symbol ? getMembersOfSymbol(source.symbol) : createSymbolTable(source.declaredProperties);
                callSignatures = source.declaredCallSignatures;
                constructSignatures = source.declaredConstructSignatures;
                stringIndexInfo = source.declaredStringIndexInfo;
                numberIndexInfo = source.declaredNumberIndexInfo;
            }
            else {
                mapper = createTypeMapper(typeParameters, typeArguments);
                members = createInstantiatedSymbolTable(source.declaredProperties, mapper, /*mappingThisOnly*/ typeParameters.length === 1);
                callSignatures = instantiateSignatures(source.declaredCallSignatures, mapper);
                constructSignatures = instantiateSignatures(source.declaredConstructSignatures, mapper);
                stringIndexInfo = instantiateIndexInfo(source.declaredStringIndexInfo, mapper);
                numberIndexInfo = instantiateIndexInfo(source.declaredNumberIndexInfo, mapper);
            }
            const baseTypes = getBaseTypes(source);
            if (baseTypes.length) {
                if (source.symbol && members === getMembersOfSymbol(source.symbol)) {
                    members = createSymbolTable(source.declaredProperties);
                }
                setStructuredTypeMembers(type, members, callSignatures, constructSignatures, stringIndexInfo, numberIndexInfo);
                const thisArgument = lastOrUndefined(typeArguments);
                for (const baseType of baseTypes) {
                    const instantiatedBaseType = thisArgument ? getTypeWithThisArgument(instantiateType(baseType, mapper), thisArgument) : baseType;
                    addInheritedMembers(members, getPropertiesOfType(instantiatedBaseType));
                    callSignatures = concatenate(callSignatures, getSignaturesOfType(instantiatedBaseType, SignatureKind.Call));
                    constructSignatures = concatenate(constructSignatures, getSignaturesOfType(instantiatedBaseType, SignatureKind.Construct));
                    if (!stringIndexInfo) {
                        stringIndexInfo = instantiatedBaseType === anyType ?
                            createIndexInfo(anyType, /*isReadonly*/ false) :
                            getIndexInfoOfType(instantiatedBaseType, IndexKind.String);
                    }
                    numberIndexInfo = numberIndexInfo || getIndexInfoOfType(instantiatedBaseType, IndexKind.Number);
                }
            }
            setStructuredTypeMembers(type, members, callSignatures, constructSignatures, stringIndexInfo, numberIndexInfo);
        }

        function resolveClassOrInterfaceMembers(type: InterfaceType): void {
            resolveObjectTypeMembers(type, resolveDeclaredMembers(type), emptyArray, emptyArray);
        }

        function resolveTypeReferenceMembers(type: TypeReference): void {
            const source = resolveDeclaredMembers(type.target);
            const typeParameters = concatenate(source.typeParameters!, [source.thisType!]);
            const typeArguments = getTypeArguments(type);
            const paddedTypeArguments = typeArguments.length === typeParameters.length ? typeArguments : concatenate(typeArguments, [type]);
            resolveObjectTypeMembers(type, source, typeParameters, paddedTypeArguments);
        }

        function createSignature(
            declaration: SignatureDeclaration | JSDocSignature | undefined,
            typeParameters: readonly TypeParameter[] | undefined,
            thisParameter: Symbol | undefined,
            parameters: readonly Symbol[],
            resolvedReturnType: Type | undefined,
            resolvedTypePredicate: TypePredicate | undefined,
            minArgumentCount: number,
            flags: SignatureFlags
        ): Signature {
            const sig = new Signature(checker, flags);
            sig.declaration = declaration;
            sig.typeParameters = typeParameters;
            sig.parameters = parameters;
            sig.thisParameter = thisParameter;
            sig.resolvedReturnType = resolvedReturnType;
            sig.resolvedTypePredicate = resolvedTypePredicate;
            sig.minArgumentCount = minArgumentCount;
            sig.target = undefined;
            sig.mapper = undefined;
            sig.unionSignatures = undefined;
            return sig;
        }

        function cloneSignature(sig: Signature): Signature {
            const result = createSignature(sig.declaration, sig.typeParameters, sig.thisParameter, sig.parameters, /*resolvedReturnType*/ undefined,
                /*resolvedTypePredicate*/ undefined, sig.minArgumentCount, sig.flags & SignatureFlags.PropagatingFlags);
            result.target = sig.target;
            result.mapper = sig.mapper;
            result.unionSignatures = sig.unionSignatures;
            return result;
        }

        function createUnionSignature(signature: Signature, unionSignatures: Signature[]) {
            const result = cloneSignature(signature);
            result.unionSignatures = unionSignatures;
            result.target = undefined;
            result.mapper = undefined;
            return result;
        }

        function getOptionalCallSignature(signature: Signature, callChainFlags: SignatureFlags): Signature {
            if ((signature.flags & SignatureFlags.CallChainFlags) === callChainFlags) {
                return signature;
            }
            if (!signature.optionalCallSignatureCache) {
                signature.optionalCallSignatureCache = {};
            }
            const key = callChainFlags === SignatureFlags.IsInnerCallChain ? "inner" : "outer";
            return signature.optionalCallSignatureCache[key]
                || (signature.optionalCallSignatureCache[key] = createOptionalCallSignature(signature, callChainFlags));
        }

        function createOptionalCallSignature(signature: Signature, callChainFlags: SignatureFlags) {
            Debug.assert(callChainFlags === SignatureFlags.IsInnerCallChain || callChainFlags === SignatureFlags.IsOuterCallChain,
                "An optional call signature can either be for an inner call chain or an outer call chain, but not both.");
            const result = cloneSignature(signature);
            result.flags |= callChainFlags;
            return result;
        }

        function getExpandedParameters(sig: Signature, skipUnionExpanding?: boolean): readonly (readonly Symbol[])[] {
            if (signatureHasRestParameter(sig)) {
                const restIndex = sig.parameters.length - 1;
                const restType = getTypeOfSymbol(sig.parameters[restIndex]);
                if (isTupleType(restType)) {
                    return [expandSignatureParametersWithTupleMembers(restType, restIndex)];
                }
                else if (!skipUnionExpanding && restType.flags & TypeFlags.Union && every((restType as UnionType).types, isTupleType)) {
                    return map((restType as UnionType).types, t => expandSignatureParametersWithTupleMembers(t as TupleTypeReference, restIndex));
                }
            }
            return [sig.parameters];

            function expandSignatureParametersWithTupleMembers(restType: TupleTypeReference, restIndex: number) {
                const elementTypes = getTypeArguments(restType);
                const associatedNames = restType.target.labeledElementDeclarations;
                const restParams = map(elementTypes, (t, i) => {
                    // Lookup the label from the individual tuple passed in before falling back to the signature `rest` parameter name
                    const tupleLabelName = !!associatedNames && getTupleElementLabel(associatedNames[i]);
                    const name = tupleLabelName || getParameterNameAtPosition(sig, restIndex + i, restType);
                    const flags = restType.target.elementFlags[i];
                    const checkFlags = flags & ElementFlags.Variable ? CheckFlags.RestParameter :
                        flags & ElementFlags.Optional ? CheckFlags.OptionalParameter : 0;
                    const symbol = createSymbol(SymbolFlags.FunctionScopedVariable, name, checkFlags);
                    symbol.type = flags & ElementFlags.Rest ? createArrayType(t) : t;
                    return symbol;
                });
                return concatenate(sig.parameters.slice(0, restIndex), restParams);
            }
        }

        function getDefaultConstructSignatures(classType: InterfaceType): Signature[] {
            const baseConstructorType = getBaseConstructorTypeOfClass(classType);
            const baseSignatures = getSignaturesOfType(baseConstructorType, SignatureKind.Construct);
            if (baseSignatures.length === 0) {
                return [createSignature(undefined, classType.localTypeParameters, undefined, emptyArray, classType, /*resolvedTypePredicate*/ undefined, 0, SignatureFlags.None)];
            }
            const baseTypeNode = getBaseTypeNodeOfClass(classType)!;
            const isJavaScript = isInJSFile(baseTypeNode);
            const typeArguments = typeArgumentsFromTypeReferenceNode(baseTypeNode);
            const typeArgCount = length(typeArguments);
            const result: Signature[] = [];
            for (const baseSig of baseSignatures) {
                const minTypeArgumentCount = getMinTypeArgumentCount(baseSig.typeParameters);
                const typeParamCount = length(baseSig.typeParameters);
                if (isJavaScript || typeArgCount >= minTypeArgumentCount && typeArgCount <= typeParamCount) {
                    const sig = typeParamCount ? createSignatureInstantiation(baseSig, fillMissingTypeArguments(typeArguments, baseSig.typeParameters, minTypeArgumentCount, isJavaScript)) : cloneSignature(baseSig);
                    sig.typeParameters = classType.localTypeParameters;
                    sig.resolvedReturnType = classType;
                    result.push(sig);
                }
            }
            return result;
        }

        function findMatchingSignature(signatureList: readonly Signature[], signature: Signature, partialMatch: boolean, ignoreThisTypes: boolean, ignoreReturnTypes: boolean): Signature | undefined {
            for (const s of signatureList) {
                if (compareSignaturesIdentical(s, signature, partialMatch, ignoreThisTypes, ignoreReturnTypes, partialMatch ? compareTypesSubtypeOf : compareTypesIdentical)) {
                    return s;
                }
            }
        }

        function findMatchingSignatures(signatureLists: readonly (readonly Signature[])[], signature: Signature, listIndex: number): Signature[] | undefined {
            if (signature.typeParameters) {
                // We require an exact match for generic signatures, so we only return signatures from the first
                // signature list and only if they have exact matches in the other signature lists.
                if (listIndex > 0) {
                    return undefined;
                }
                for (let i = 1; i < signatureLists.length; i++) {
                    if (!findMatchingSignature(signatureLists[i], signature, /*partialMatch*/ false, /*ignoreThisTypes*/ false, /*ignoreReturnTypes*/ false)) {
                        return undefined;
                    }
                }
                return [signature];
            }
            let result: Signature[] | undefined;
            for (let i = 0; i < signatureLists.length; i++) {
                // Allow matching non-generic signatures to have excess parameters and different return types.
                // Prefer matching this types if possible.
                const match = i === listIndex ? signature : findMatchingSignature(signatureLists[i], signature, /*partialMatch*/ true, /*ignoreThisTypes*/ false, /*ignoreReturnTypes*/ true);
                if (!match) {
                    return undefined;
                }
                result = appendIfUnique(result, match);
            }
            return result;
        }

        // The signatures of a union type are those signatures that are present in each of the constituent types.
        // Generic signatures must match exactly, but non-generic signatures are allowed to have extra optional
        // parameters and may differ in return types. When signatures differ in return types, the resulting return
        // type is the union of the constituent return types.
        function getUnionSignatures(signatureLists: readonly (readonly Signature[])[]): Signature[] {
            let result: Signature[] | undefined;
            let indexWithLengthOverOne: number | undefined;
            for (let i = 0; i < signatureLists.length; i++) {
                if (signatureLists[i].length === 0) return emptyArray;
                if (signatureLists[i].length > 1) {
                    indexWithLengthOverOne = indexWithLengthOverOne === undefined ? i : -1; // -1 is a signal there are multiple overload sets
                }
                for (const signature of signatureLists[i]) {
                    // Only process signatures with parameter lists that aren't already in the result list
                    if (!result || !findMatchingSignature(result, signature, /*partialMatch*/ false, /*ignoreThisTypes*/ false, /*ignoreReturnTypes*/ true)) {
                        const unionSignatures = findMatchingSignatures(signatureLists, signature, i);
                        if (unionSignatures) {
                            let s = signature;
                            // Union the result types when more than one signature matches
                            if (unionSignatures.length > 1) {
                                let thisParameter = signature.thisParameter;
                                const firstThisParameterOfUnionSignatures = forEach(unionSignatures, sig => sig.thisParameter);
                                if (firstThisParameterOfUnionSignatures) {
                                    const thisType = getIntersectionType(mapDefined(unionSignatures, sig => sig.thisParameter && getTypeOfSymbol(sig.thisParameter)));
                                    thisParameter = createSymbolWithType(firstThisParameterOfUnionSignatures, thisType);
                                }
                                s = createUnionSignature(signature, unionSignatures);
                                s.thisParameter = thisParameter;
                            }
                            (result || (result = [])).push(s);
                        }
                    }
                }
            }
            if (!length(result) && indexWithLengthOverOne !== -1) {
                // No sufficiently similar signature existed to subsume all the other signatures in the union - time to see if we can make a single
                // signature that handles all over them. We only do this when there are overloads in only one constituent.
                // (Overloads are conditional in nature and having overloads in multiple constituents would necessitate making a power set of
                // signatures from the type, whose ordering would be non-obvious)
                const masterList = signatureLists[indexWithLengthOverOne !== undefined ? indexWithLengthOverOne : 0];
                let results: Signature[] | undefined = masterList.slice();
                for (const signatures of signatureLists) {
                    if (signatures !== masterList) {
                        const signature = signatures[0];
                        Debug.assert(!!signature, "getUnionSignatures bails early on empty signature lists and should not have empty lists on second pass");
                        results = signature.typeParameters && some(results, s => !!s.typeParameters) ? undefined : map(results, sig => combineSignaturesOfUnionMembers(sig, signature));
                        if (!results) {
                            break;
                        }
                    }
                }
                result = results;
            }
            return result || emptyArray;
        }

        function combineUnionThisParam(left: Symbol | undefined, right: Symbol | undefined): Symbol | undefined {
            if (!left || !right) {
                return left || right;
            }
            // A signature `this` type might be a read or a write position... It's very possible that it should be invariant
            // and we should refuse to merge signatures if there are `this` types and they do not match. However, so as to be
            // permissive when calling, for now, we'll intersect the `this` types just like we do for param types in union signatures.
            const thisType = getIntersectionType([getTypeOfSymbol(left), getTypeOfSymbol(right)]);
            return createSymbolWithType(left, thisType);
        }

        function combineUnionParameters(left: Signature, right: Signature) {
            const leftCount = getParameterCount(left);
            const rightCount = getParameterCount(right);
            const longest = leftCount >= rightCount ? left : right;
            const shorter = longest === left ? right : left;
            const longestCount = longest === left ? leftCount : rightCount;
            const eitherHasEffectiveRest = (hasEffectiveRestParameter(left) || hasEffectiveRestParameter(right));
            const needsExtraRestElement = eitherHasEffectiveRest && !hasEffectiveRestParameter(longest);
            const params = new Array<Symbol>(longestCount + (needsExtraRestElement ? 1 : 0));
            for (let i = 0; i < longestCount; i++) {
                const longestParamType = tryGetTypeAtPosition(longest, i)!;
                const shorterParamType = tryGetTypeAtPosition(shorter, i) || unknownType;
                const unionParamType = getIntersectionType([longestParamType, shorterParamType]);
                const isRestParam = eitherHasEffectiveRest && !needsExtraRestElement && i === (longestCount - 1);
                const isOptional = i >= getMinArgumentCount(longest) && i >= getMinArgumentCount(shorter);
                const leftName = i >= leftCount ? undefined : getParameterNameAtPosition(left, i);
                const rightName = i >= rightCount ? undefined : getParameterNameAtPosition(right, i);

                const paramName = leftName === rightName ? leftName :
                    !leftName ? rightName :
                    !rightName ? leftName :
                    undefined;
                const paramSymbol = createSymbol(
                    SymbolFlags.FunctionScopedVariable | (isOptional && !isRestParam ? SymbolFlags.Optional : 0),
                    paramName || `arg${i}` as __String
                );
                paramSymbol.type = isRestParam ? createArrayType(unionParamType) : unionParamType;
                params[i] = paramSymbol;
            }
            if (needsExtraRestElement) {
                const restParamSymbol = createSymbol(SymbolFlags.FunctionScopedVariable, "args" as __String);
                restParamSymbol.type = createArrayType(getTypeAtPosition(shorter, longestCount));
                params[longestCount] = restParamSymbol;
            }
            return params;
        }

        function combineSignaturesOfUnionMembers(left: Signature, right: Signature): Signature {
            const declaration = left.declaration;
            const params = combineUnionParameters(left, right);
            const thisParam = combineUnionThisParam(left.thisParameter, right.thisParameter);
            const minArgCount = Math.max(left.minArgumentCount, right.minArgumentCount);
            const result = createSignature(
                declaration,
                left.typeParameters || right.typeParameters,
                thisParam,
                params,
                /*resolvedReturnType*/ undefined,
                /*resolvedTypePredicate*/ undefined,
                minArgCount,
                (left.flags | right.flags) & SignatureFlags.PropagatingFlags
            );
            result.unionSignatures = concatenate(left.unionSignatures || [left], [right]);
            return result;
        }

        function getUnionIndexInfo(types: readonly Type[], kind: IndexKind): IndexInfo | undefined {
            const indexTypes: Type[] = [];
            let isAnyReadonly = false;
            for (const type of types) {
                const indexInfo = getIndexInfoOfType(getApparentType(type), kind);
                if (!indexInfo) {
                    return undefined;
                }
                indexTypes.push(indexInfo.type);
                isAnyReadonly = isAnyReadonly || indexInfo.isReadonly;
            }
            return createIndexInfo(getUnionType(indexTypes, UnionReduction.Subtype), isAnyReadonly);
        }

        function resolveUnionTypeMembers(type: UnionType) {
            // The members and properties collections are empty for union types. To get all properties of a union
            // type use getPropertiesOfType (only the language service uses this).
            const callSignatures = getUnionSignatures(map(type.types, t => t === globalFunctionType ? [unknownSignature] : getSignaturesOfType(t, SignatureKind.Call)));
            const constructSignatures = getUnionSignatures(map(type.types, t => getSignaturesOfType(t, SignatureKind.Construct)));
            const stringIndexInfo = getUnionIndexInfo(type.types, IndexKind.String);
            const numberIndexInfo = getUnionIndexInfo(type.types, IndexKind.Number);
            setStructuredTypeMembers(type, emptySymbols, callSignatures, constructSignatures, stringIndexInfo, numberIndexInfo);
        }

        function intersectTypes(type1: Type, type2: Type): Type;
        function intersectTypes(type1: Type | undefined, type2: Type | undefined): Type | undefined;
        function intersectTypes(type1: Type | undefined, type2: Type | undefined): Type | undefined {
            return !type1 ? type2 : !type2 ? type1 : getIntersectionType([type1, type2]);
        }

        function intersectIndexInfos(info1: IndexInfo | undefined, info2: IndexInfo | undefined): IndexInfo | undefined {
            return !info1 ? info2 : !info2 ? info1 : createIndexInfo(
                getIntersectionType([info1.type, info2.type]), info1.isReadonly && info2.isReadonly);
        }

        function unionSpreadIndexInfos(info1: IndexInfo | undefined, info2: IndexInfo | undefined): IndexInfo | undefined {
            return info1 && info2 && createIndexInfo(
                getUnionType([info1.type, info2.type]), info1.isReadonly || info2.isReadonly);
        }

        function findMixins(types: readonly Type[]): readonly boolean[] {
            const constructorTypeCount = countWhere(types, (t) => getSignaturesOfType(t, SignatureKind.Construct).length > 0);
            const mixinFlags = map(types, isMixinConstructorType);
            if (constructorTypeCount > 0 && constructorTypeCount === countWhere(mixinFlags, (b) => b)) {
                const firstMixinIndex = mixinFlags.indexOf(/*searchElement*/ true);
                mixinFlags[firstMixinIndex] = false;
            }
            return mixinFlags;
        }

        function includeMixinType(type: Type, types: readonly Type[], mixinFlags: readonly boolean[], index: number): Type {
            const mixedTypes: Type[] = [];
            for (let i = 0; i < types.length; i++) {
                if (i === index) {
                    mixedTypes.push(type);
                }
                else if (mixinFlags[i]) {
                    mixedTypes.push(getReturnTypeOfSignature(getSignaturesOfType(types[i], SignatureKind.Construct)[0]));
                }
            }
            return getIntersectionType(mixedTypes);
        }

        function resolveIntersectionTypeMembers(type: IntersectionType) {
            // The members and properties collections are empty for intersection types. To get all properties of an
            // intersection type use getPropertiesOfType (only the language service uses this).
            let callSignatures: Signature[] | undefined;
            let constructSignatures: Signature[] | undefined;
            let stringIndexInfo: IndexInfo | undefined;
            let numberIndexInfo: IndexInfo | undefined;
            const types = type.types;
            const mixinFlags = findMixins(types);
            const mixinCount = countWhere(mixinFlags, (b) => b);
            for (let i = 0; i < types.length; i++) {
                const t = type.types[i];
                // When an intersection type contains mixin constructor types, the construct signatures from
                // those types are discarded and their return types are mixed into the return types of all
                // other construct signatures in the intersection type. For example, the intersection type
                // '{ new(...args: any[]) => A } & { new(s: string) => B }' has a single construct signature
                // 'new(s: string) => A & B'.
                if (!mixinFlags[i]) {
                    let signatures = getSignaturesOfType(t, SignatureKind.Construct);
                    if (signatures.length && mixinCount > 0) {
                        signatures = map(signatures, s => {
                            const clone = cloneSignature(s);
                            clone.resolvedReturnType = includeMixinType(getReturnTypeOfSignature(s), types, mixinFlags, i);
                            return clone;
                        });
                    }
                    constructSignatures = appendSignatures(constructSignatures, signatures);
                }
                callSignatures = appendSignatures(callSignatures, getSignaturesOfType(t, SignatureKind.Call));
                stringIndexInfo = intersectIndexInfos(stringIndexInfo, getIndexInfoOfType(t, IndexKind.String));
                numberIndexInfo = intersectIndexInfos(numberIndexInfo, getIndexInfoOfType(t, IndexKind.Number));
            }
            setStructuredTypeMembers(type, emptySymbols, callSignatures || emptyArray, constructSignatures || emptyArray, stringIndexInfo, numberIndexInfo);
        }

        function appendSignatures(signatures: Signature[] | undefined, newSignatures: readonly Signature[]) {
            for (const sig of newSignatures) {
                if (!signatures || every(signatures, s => !compareSignaturesIdentical(s, sig, /*partialMatch*/ false, /*ignoreThisTypes*/ false, /*ignoreReturnTypes*/ false, compareTypesIdentical))) {
                    signatures = append(signatures, sig);
                }
            }
            return signatures;
        }

        /**
         * Converts an AnonymousType to a ResolvedType.
         */
        function resolveAnonymousTypeMembers(type: AnonymousType) {
            const symbol = getMergedSymbol(type.symbol);
            if (type.target) {
                setStructuredTypeMembers(type, emptySymbols, emptyArray, emptyArray, undefined, undefined);
                const members = createInstantiatedSymbolTable(getPropertiesOfObjectType(type.target), type.mapper!, /*mappingThisOnly*/ false);
                const callSignatures = instantiateSignatures(getSignaturesOfType(type.target, SignatureKind.Call), type.mapper!);
                const constructSignatures = instantiateSignatures(getSignaturesOfType(type.target, SignatureKind.Construct), type.mapper!);
                const stringIndexInfo = instantiateIndexInfo(getIndexInfoOfType(type.target, IndexKind.String), type.mapper!);
                const numberIndexInfo = instantiateIndexInfo(getIndexInfoOfType(type.target, IndexKind.Number), type.mapper!);
                setStructuredTypeMembers(type, members, callSignatures, constructSignatures, stringIndexInfo, numberIndexInfo);
            }
            else if (symbol.flags & SymbolFlags.TypeLiteral) {
                setStructuredTypeMembers(type, emptySymbols, emptyArray, emptyArray, undefined, undefined);
                const members = getMembersOfSymbol(symbol);
                const callSignatures = getSignaturesOfSymbol(members.get(InternalSymbolName.Call));
                const constructSignatures = getSignaturesOfSymbol(members.get(InternalSymbolName.New));
                const stringIndexInfo = getIndexInfoOfSymbol(symbol, IndexKind.String);
                const numberIndexInfo = getIndexInfoOfSymbol(symbol, IndexKind.Number);
                setStructuredTypeMembers(type, members, callSignatures, constructSignatures, stringIndexInfo, numberIndexInfo);
            }
            else {
                // Combinations of function, class, enum and module
                let members = emptySymbols;
                let stringIndexInfo: IndexInfo | undefined;
                if (symbol.exports) {
                    members = getExportsOfSymbol(symbol);
                    if (symbol === globalThisSymbol) {
                        const varsOnly = new Map<string, Symbol>() as SymbolTable;
                        members.forEach(p => {
                            if (!(p.flags & SymbolFlags.BlockScoped)) {
                                varsOnly.set(p.escapedName, p);
                            }
                        });
                        members = varsOnly;
                    }
                }
                setStructuredTypeMembers(type, members, emptyArray, emptyArray, undefined, undefined);
                if (symbol.flags & SymbolFlags.Class) {
                    const classType = getDeclaredTypeOfClassOrInterface(symbol);
                    const baseConstructorType = getBaseConstructorTypeOfClass(classType);
                    if (baseConstructorType.flags & (TypeFlags.Object | TypeFlags.Intersection | TypeFlags.TypeVariable)) {
                        members = createSymbolTable(getNamedMembers(members));
                        addInheritedMembers(members, getPropertiesOfType(baseConstructorType));
                    }
                    else if (baseConstructorType === anyType) {
                        stringIndexInfo = createIndexInfo(anyType, /*isReadonly*/ false);
                    }
                }
                const numberIndexInfo = symbol.flags & SymbolFlags.Enum && (getDeclaredTypeOfSymbol(symbol).flags & TypeFlags.Enum ||
                    some(type.properties, prop => !!(getTypeOfSymbol(prop).flags & TypeFlags.NumberLike))) ? enumNumberIndexInfo : undefined;
                setStructuredTypeMembers(type, members, emptyArray, emptyArray, stringIndexInfo, numberIndexInfo);
                // We resolve the members before computing the signatures because a signature may use
                // typeof with a qualified name expression that circularly references the type we are
                // in the process of resolving (see issue #6072). The temporarily empty signature list
                // will never be observed because a qualified name can't reference signatures.
                if (symbol.flags & (SymbolFlags.Function | SymbolFlags.Method)) {
                    type.callSignatures = getSignaturesOfSymbol(symbol);
                }
                // And likewise for construct signatures for classes
                if (symbol.flags & SymbolFlags.Class) {
                    const classType = getDeclaredTypeOfClassOrInterface(symbol);
                    let constructSignatures = symbol.members ? getSignaturesOfSymbol(symbol.members.get(InternalSymbolName.Constructor)) : emptyArray;
                    if (symbol.flags & SymbolFlags.Function) {
                        constructSignatures = addRange(constructSignatures.slice(), mapDefined(
                            type.callSignatures,
                            sig => isJSConstructor(sig.declaration) ?
                                createSignature(sig.declaration, sig.typeParameters, sig.thisParameter, sig.parameters, classType, /*resolvedTypePredicate*/ undefined, sig.minArgumentCount, sig.flags & SignatureFlags.PropagatingFlags) :
                                undefined));
                    }
                    if (!constructSignatures.length) {
                        constructSignatures = getDefaultConstructSignatures(classType);
                    }
                    type.constructSignatures = constructSignatures;
                }
            }
        }

        function resolveReverseMappedTypeMembers(type: ReverseMappedType) {
            const indexInfo = getIndexInfoOfType(type.source, IndexKind.String);
            const modifiers = getMappedTypeModifiers(type.mappedType);
            const readonlyMask = modifiers & MappedTypeModifiers.IncludeReadonly ? false : true;
            const optionalMask = modifiers & MappedTypeModifiers.IncludeOptional ? 0 : SymbolFlags.Optional;
            const stringIndexInfo = indexInfo && createIndexInfo(inferReverseMappedType(indexInfo.type, type.mappedType, type.constraintType), readonlyMask && indexInfo.isReadonly);
            const members = createSymbolTable();
            for (const prop of getPropertiesOfType(type.source)) {
                const checkFlags = CheckFlags.ReverseMapped | (readonlyMask && isReadonlySymbol(prop) ? CheckFlags.Readonly : 0);
                const inferredProp = createSymbol(SymbolFlags.Property | prop.flags & optionalMask, prop.escapedName, checkFlags) as ReverseMappedSymbol;
                inferredProp.declarations = prop.declarations;
                inferredProp.nameType = getSymbolLinks(prop).nameType;
                inferredProp.propertyType = getTypeOfSymbol(prop);
                inferredProp.mappedType = type.mappedType;
                inferredProp.constraintType = type.constraintType;
                members.set(prop.escapedName, inferredProp);
            }
            setStructuredTypeMembers(type, members, emptyArray, emptyArray, stringIndexInfo, undefined);
        }

        // Return the lower bound of the key type in a mapped type. Intuitively, the lower
        // bound includes those keys that are known to always be present, for example because
        // because of constraints on type parameters (e.g. 'keyof T' for a constrained T).
        function getLowerBoundOfKeyType(type: Type): Type {
            if (type.flags & (TypeFlags.Any | TypeFlags.Primitive)) {
                return type;
            }
            if (type.flags & TypeFlags.Index) {
                const t = getApparentType((<IndexType>type).type);
                return isGenericTupleType(t) ? getKnownKeysOfTupleType(t) : getIndexType(t);
            }
            if (type.flags & TypeFlags.Conditional) {
                if ((<ConditionalType>type).root.isDistributive) {
                    const checkType = (<ConditionalType>type).checkType;
                    const constraint = getLowerBoundOfKeyType(checkType);
                    if (constraint !== checkType) {
                        return getConditionalTypeInstantiation(<ConditionalType>type, prependTypeMapping((<ConditionalType>type).root.checkType, constraint, (<ConditionalType>type).mapper));
                    }
                }
                return type;
            }
            if (type.flags & TypeFlags.Union) {
                return getUnionType(sameMap((<UnionType>type).types, getLowerBoundOfKeyType));
            }
            if (type.flags & TypeFlags.Intersection) {
                return getIntersectionType(sameMap((<UnionType>type).types, getLowerBoundOfKeyType));
            }
            return neverType;
        }

        /** Resolve the members of a mapped type { [P in K]: T } */
        function resolveMappedTypeMembers(type: MappedType) {
            const members: SymbolTable = createSymbolTable();
            let stringIndexInfo: IndexInfo | undefined;
            let numberIndexInfo: IndexInfo | undefined;
            // Resolve upfront such that recursive references see an empty object type.
            setStructuredTypeMembers(type, emptySymbols, emptyArray, emptyArray, undefined, undefined);
            // In { [P in K]: T }, we refer to P as the type parameter type, K as the constraint type,
            // and T as the template type.
            const typeParameter = getTypeParameterFromMappedType(type);
            const constraintType = getConstraintTypeFromMappedType(type);
            const templateType = getTemplateTypeFromMappedType(<MappedType>type.target || type);
            const modifiersType = getApparentType(getModifiersTypeFromMappedType(type)); // The 'T' in 'keyof T'
            const templateModifiers = getMappedTypeModifiers(type);
            const include = keyofStringsOnly ? TypeFlags.StringLiteral : TypeFlags.StringOrNumberLiteralOrUnique;
            if (isMappedTypeWithKeyofConstraintDeclaration(type)) {
                // We have a { [P in keyof T]: X }
                for (const prop of getPropertiesOfType(modifiersType)) {
                    addMemberForKeyType(getLiteralTypeFromProperty(prop, include));
                }
                if (modifiersType.flags & TypeFlags.Any || getIndexInfoOfType(modifiersType, IndexKind.String)) {
                    addMemberForKeyType(stringType);
                }
                if (!keyofStringsOnly && getIndexInfoOfType(modifiersType, IndexKind.Number)) {
                    addMemberForKeyType(numberType);
                }
            }
            else {
                forEachType(getLowerBoundOfKeyType(constraintType), addMemberForKeyType);
            }
            setStructuredTypeMembers(type, members, emptyArray, emptyArray, stringIndexInfo, numberIndexInfo);

            function addMemberForKeyType(t: Type) {
                // Create a mapper from T to the current iteration type constituent. Then, if the
                // mapped type is itself an instantiated type, combine the iteration mapper with the
                // instantiation mapper.
                const templateMapper = appendTypeMapping(type.mapper, typeParameter, t);
                // If the current iteration type constituent is a string literal type, create a property.
                // Otherwise, for type string create a string index signature.
                if (isTypeUsableAsPropertyName(t)) {
                    const propName = getPropertyNameFromType(t);
                    // String enum members from separate enums with identical values
                    // are distinct types with the same property name. Make the resulting
                    // property symbol's name type be the union of those enum member types.
                    const existingProp = members.get(propName) as MappedSymbol | undefined;
                    if (existingProp) {
                        existingProp.nameType = getUnionType([existingProp.nameType!, t]);
                        existingProp.mapper = appendTypeMapping(type.mapper, typeParameter, existingProp.nameType);
                    }
                    else {
                        const modifiersProp = getPropertyOfType(modifiersType, propName);
                        const isOptional = !!(templateModifiers & MappedTypeModifiers.IncludeOptional ||
                            !(templateModifiers & MappedTypeModifiers.ExcludeOptional) && modifiersProp && modifiersProp.flags & SymbolFlags.Optional);
                        const isReadonly = !!(templateModifiers & MappedTypeModifiers.IncludeReadonly ||
                            !(templateModifiers & MappedTypeModifiers.ExcludeReadonly) && modifiersProp && isReadonlySymbol(modifiersProp));
                        const stripOptional = strictNullChecks && !isOptional && modifiersProp && modifiersProp.flags & SymbolFlags.Optional;
                        const prop = <MappedSymbol>createSymbol(SymbolFlags.Property | (isOptional ? SymbolFlags.Optional : 0), propName,
                            CheckFlags.Mapped | (isReadonly ? CheckFlags.Readonly : 0) | (stripOptional ? CheckFlags.StripOptional : 0));
                        prop.mappedType = type;
                        if (modifiersProp) {
                            prop.syntheticOrigin = modifiersProp;
                            prop.declarations = modifiersProp.declarations;
                        }
                        prop.nameType = t;
                        prop.mapper = templateMapper;
                        members.set(propName, prop);
                    }
                }
                else if (t.flags & (TypeFlags.Any | TypeFlags.String | TypeFlags.Number | TypeFlags.Enum)) {
                    const propType = instantiateType(templateType, templateMapper);
                    if (t.flags & (TypeFlags.Any | TypeFlags.String)) {
                        stringIndexInfo = createIndexInfo(propType, !!(templateModifiers & MappedTypeModifiers.IncludeReadonly));
                    }
                    else {
                        numberIndexInfo = createIndexInfo(numberIndexInfo ? getUnionType([numberIndexInfo.type, propType]) : propType,
                            !!(templateModifiers & MappedTypeModifiers.IncludeReadonly));
                    }
                }
            }
        }

        function getTypeOfMappedSymbol(symbol: MappedSymbol) {
            if (!symbol.type) {
                if (!pushTypeResolution(symbol, TypeSystemPropertyName.Type)) {
                    symbol.mappedType.containsError = true;
                    return errorType;
                }
                const templateType = getTemplateTypeFromMappedType(<MappedType>symbol.mappedType.target || symbol.mappedType);
                const propType = instantiateType(templateType, symbol.mapper);
                // When creating an optional property in strictNullChecks mode, if 'undefined' isn't assignable to the
                // type, we include 'undefined' in the type. Similarly, when creating a non-optional property in strictNullChecks
                // mode, if the underlying property is optional we remove 'undefined' from the type.
                let type = strictNullChecks && symbol.flags & SymbolFlags.Optional && !maybeTypeOfKind(propType, TypeFlags.Undefined | TypeFlags.Void) ? getOptionalType(propType) :
                    symbol.checkFlags & CheckFlags.StripOptional ? getTypeWithFacts(propType, TypeFacts.NEUndefined) :
                    propType;
                if (!popTypeResolution()) {
                    error(currentNode, Diagnostics.Type_of_property_0_circularly_references_itself_in_mapped_type_1, symbolToString(symbol), typeToString(symbol.mappedType));
                    type = errorType;
                }
                symbol.type = type;
                symbol.mapper = undefined!;
            }
            return symbol.type;
        }

        function getTypeParameterFromMappedType(type: MappedType) {
            return type.typeParameter ||
                (type.typeParameter = getDeclaredTypeOfTypeParameter(getSymbolOfNode(type.declaration.typeParameter)));
        }

        function getConstraintTypeFromMappedType(type: MappedType) {
            return type.constraintType ||
                (type.constraintType = getConstraintOfTypeParameter(getTypeParameterFromMappedType(type)) || errorType);
        }

        function getTemplateTypeFromMappedType(type: MappedType) {
            return type.templateType ||
                (type.templateType = type.declaration.type ?
                    instantiateType(addOptionality(getTypeFromTypeNode(type.declaration.type), !!(getMappedTypeModifiers(type) & MappedTypeModifiers.IncludeOptional)), type.mapper) :
                    errorType);
        }

        function getConstraintDeclarationForMappedType(type: MappedType) {
            return getEffectiveConstraintOfTypeParameter(type.declaration.typeParameter);
        }

        function isMappedTypeWithKeyofConstraintDeclaration(type: MappedType) {
            const constraintDeclaration = getConstraintDeclarationForMappedType(type)!; // TODO: GH#18217
            return constraintDeclaration.kind === SyntaxKind.TypeOperator &&
                (<TypeOperatorNode>constraintDeclaration).operator === SyntaxKind.KeyOfKeyword;
        }

        function getModifiersTypeFromMappedType(type: MappedType) {
            if (!type.modifiersType) {
                if (isMappedTypeWithKeyofConstraintDeclaration(type)) {
                    // If the constraint declaration is a 'keyof T' node, the modifiers type is T. We check
                    // AST nodes here because, when T is a non-generic type, the logic below eagerly resolves
                    // 'keyof T' to a literal union type and we can't recover T from that type.
                    type.modifiersType = instantiateType(getTypeFromTypeNode((<TypeOperatorNode>getConstraintDeclarationForMappedType(type)).type), type.mapper);
                }
                else {
                    // Otherwise, get the declared constraint type, and if the constraint type is a type parameter,
                    // get the constraint of that type parameter. If the resulting type is an indexed type 'keyof T',
                    // the modifiers type is T. Otherwise, the modifiers type is unknown.
                    const declaredType = <MappedType>getTypeFromMappedTypeNode(type.declaration);
                    const constraint = getConstraintTypeFromMappedType(declaredType);
                    const extendedConstraint = constraint && constraint.flags & TypeFlags.TypeParameter ? getConstraintOfTypeParameter(<TypeParameter>constraint) : constraint;
                    type.modifiersType = extendedConstraint && extendedConstraint.flags & TypeFlags.Index ? instantiateType((<IndexType>extendedConstraint).type, type.mapper) : unknownType;
                }
            }
            return type.modifiersType;
        }

        function getMappedTypeModifiers(type: MappedType): MappedTypeModifiers {
            const declaration = type.declaration;
            return (declaration.readonlyToken ? declaration.readonlyToken.kind === SyntaxKind.MinusToken ? MappedTypeModifiers.ExcludeReadonly : MappedTypeModifiers.IncludeReadonly : 0) |
                (declaration.questionToken ? declaration.questionToken.kind === SyntaxKind.MinusToken ? MappedTypeModifiers.ExcludeOptional : MappedTypeModifiers.IncludeOptional : 0);
        }

        function getMappedTypeOptionality(type: MappedType): number {
            const modifiers = getMappedTypeModifiers(type);
            return modifiers & MappedTypeModifiers.ExcludeOptional ? -1 : modifiers & MappedTypeModifiers.IncludeOptional ? 1 : 0;
        }

        function getCombinedMappedTypeOptionality(type: MappedType): number {
            const optionality = getMappedTypeOptionality(type);
            const modifiersType = getModifiersTypeFromMappedType(type);
            return optionality || (isGenericMappedType(modifiersType) ? getMappedTypeOptionality(modifiersType) : 0);
        }

        function isPartialMappedType(type: Type) {
            return !!(getObjectFlags(type) & ObjectFlags.Mapped && getMappedTypeModifiers(<MappedType>type) & MappedTypeModifiers.IncludeOptional);
        }

        function isGenericMappedType(type: Type): type is MappedType {
            return !!(getObjectFlags(type) & ObjectFlags.Mapped) && isGenericIndexType(getConstraintTypeFromMappedType(<MappedType>type));
        }

        function resolveStructuredTypeMembers(type: StructuredType): ResolvedType {
            if (!(<ResolvedType>type).members) {
                if (type.flags & TypeFlags.Object) {
                    if ((<ObjectType>type).objectFlags & ObjectFlags.Reference) {
                        resolveTypeReferenceMembers(<TypeReference>type);
                    }
                    else if ((<ObjectType>type).objectFlags & ObjectFlags.ClassOrInterface) {
                        resolveClassOrInterfaceMembers(<InterfaceType>type);
                    }
                    else if ((<ReverseMappedType>type).objectFlags & ObjectFlags.ReverseMapped) {
                        resolveReverseMappedTypeMembers(type as ReverseMappedType);
                    }
                    else if ((<ObjectType>type).objectFlags & ObjectFlags.Anonymous) {
                        resolveAnonymousTypeMembers(<AnonymousType>type);
                    }
                    else if ((<MappedType>type).objectFlags & ObjectFlags.Mapped) {
                        resolveMappedTypeMembers(<MappedType>type);
                    }
                }
                else if (type.flags & TypeFlags.Union) {
                    resolveUnionTypeMembers(<UnionType>type);
                }
                else if (type.flags & TypeFlags.Intersection) {
                    resolveIntersectionTypeMembers(<IntersectionType>type);
                }
            }
            return <ResolvedType>type;
        }

        /** Return properties of an object type or an empty array for other types */
        function getPropertiesOfObjectType(type: Type): Symbol[] {
            if (type.flags & TypeFlags.Object) {
                return resolveStructuredTypeMembers(<ObjectType>type).properties;
            }
            return emptyArray;
        }

        /** If the given type is an object type and that type has a property by the given name,
         * return the symbol for that property. Otherwise return undefined.
         */
        function getPropertyOfObjectType(type: Type, name: __String): Symbol | undefined {
            if (type.flags & TypeFlags.Object) {
                const resolved = resolveStructuredTypeMembers(<ObjectType>type);
                const symbol = resolved.members.get(name);
                if (symbol && symbolIsValue(symbol)) {
                    return symbol;
                }
            }
        }

        function getPropertiesOfUnionOrIntersectionType(type: UnionOrIntersectionType): Symbol[] {
            if (!type.resolvedProperties) {
                const members = createSymbolTable();
                for (const current of type.types) {
                    for (const prop of getPropertiesOfType(current)) {
                        if (!members.has(prop.escapedName)) {
                            const combinedProp = getPropertyOfUnionOrIntersectionType(type, prop.escapedName);
                            if (combinedProp) {
                                members.set(prop.escapedName, combinedProp);
                            }
                        }
                    }
                    // The properties of a union type are those that are present in all constituent types, so
                    // we only need to check the properties of the first type without index signature
                    if (type.flags & TypeFlags.Union && !getIndexInfoOfType(current, IndexKind.String) && !getIndexInfoOfType(current, IndexKind.Number)) {
                        break;
                    }
                }
                type.resolvedProperties = getNamedMembers(members);
            }
            return type.resolvedProperties;
        }

        function getPropertiesOfType(type: Type): Symbol[] {
            type = getReducedApparentType(type);
            return type.flags & TypeFlags.UnionOrIntersection ?
                getPropertiesOfUnionOrIntersectionType(<UnionType>type) :
                getPropertiesOfObjectType(type);
        }

        function isTypeInvalidDueToUnionDiscriminant(contextualType: Type, obj: ObjectLiteralExpression | JsxAttributes): boolean {
            const list = obj.properties as NodeArray<ObjectLiteralElementLike | JsxAttributeLike>;
            return list.some(property => {
                const nameType = property.name && getLiteralTypeFromPropertyName(property.name);
                const name = nameType && isTypeUsableAsPropertyName(nameType) ? getPropertyNameFromType(nameType) : undefined;
                const expected = name === undefined ? undefined : getTypeOfPropertyOfType(contextualType, name);
                return !!expected && isLiteralType(expected) && !isTypeAssignableTo(getTypeOfNode(property), expected);
            });
        }

        function getAllPossiblePropertiesOfTypes(types: readonly Type[]): Symbol[] {
            const unionType = getUnionType(types);
            if (!(unionType.flags & TypeFlags.Union)) {
                return getAugmentedPropertiesOfType(unionType);
            }

            const props = createSymbolTable();
            for (const memberType of types) {
                for (const { escapedName } of getAugmentedPropertiesOfType(memberType)) {
                    if (!props.has(escapedName)) {
                        const prop = createUnionOrIntersectionProperty(unionType as UnionType, escapedName);
                        // May be undefined if the property is private
                        if (prop) props.set(escapedName, prop);
                    }
                }
            }
            return arrayFrom(props.values());
        }

        function getConstraintOfType(type: InstantiableType | UnionOrIntersectionType): Type | undefined {
            return type.flags & TypeFlags.TypeParameter ? getConstraintOfTypeParameter(<TypeParameter>type) :
                type.flags & TypeFlags.IndexedAccess ? getConstraintOfIndexedAccess(<IndexedAccessType>type) :
                type.flags & TypeFlags.Conditional ? getConstraintOfConditionalType(<ConditionalType>type) :
                getBaseConstraintOfType(type);
        }

        function getConstraintOfTypeParameter(typeParameter: TypeParameter): Type | undefined {
            return hasNonCircularBaseConstraint(typeParameter) ? getConstraintFromTypeParameter(typeParameter) : undefined;
        }

        function getConstraintOfIndexedAccess(type: IndexedAccessType) {
            return hasNonCircularBaseConstraint(type) ? getConstraintFromIndexedAccess(type) : undefined;
        }

        function getSimplifiedTypeOrConstraint(type: Type) {
            const simplified = getSimplifiedType(type, /*writing*/ false);
            return simplified !== type ? simplified : getConstraintOfType(type);
        }

        function getConstraintFromIndexedAccess(type: IndexedAccessType) {
            const indexConstraint = getSimplifiedTypeOrConstraint(type.indexType);
            if (indexConstraint && indexConstraint !== type.indexType) {
                const indexedAccess = getIndexedAccessTypeOrUndefined(type.objectType, indexConstraint);
                if (indexedAccess) {
                    return indexedAccess;
                }
            }
            const objectConstraint = getSimplifiedTypeOrConstraint(type.objectType);
            if (objectConstraint && objectConstraint !== type.objectType) {
                return getIndexedAccessTypeOrUndefined(objectConstraint, type.indexType);
            }
            return undefined;
        }

        function getDefaultConstraintOfConditionalType(type: ConditionalType) {
            if (!type.resolvedDefaultConstraint) {
                // An `any` branch of a conditional type would normally be viral - specifically, without special handling here,
                // a conditional type with a single branch of type `any` would be assignable to anything, since it's constraint would simplify to
                // just `any`. This result is _usually_ unwanted - so instead here we elide an `any` branch from the constraint type,
                // in effect treating `any` like `never` rather than `unknown` in this location.
                const trueConstraint = getInferredTrueTypeFromConditionalType(type);
                const falseConstraint = getFalseTypeFromConditionalType(type);
                type.resolvedDefaultConstraint = isTypeAny(trueConstraint) ? falseConstraint : isTypeAny(falseConstraint) ? trueConstraint : getUnionType([trueConstraint, falseConstraint]);
            }
            return type.resolvedDefaultConstraint;
        }

        function getConstraintOfDistributiveConditionalType(type: ConditionalType): Type | undefined {
            // Check if we have a conditional type of the form 'T extends U ? X : Y', where T is a constrained
            // type parameter. If so, create an instantiation of the conditional type where T is replaced
            // with its constraint. We do this because if the constraint is a union type it will be distributed
            // over the conditional type and possibly reduced. For example, 'T extends undefined ? never : T'
            // removes 'undefined' from T.
            // We skip returning a distributive constraint for a restrictive instantiation of a conditional type
            // as the constraint for all type params (check type included) have been replace with `unknown`, which
            // is going to produce even more false positive/negative results than the distribute constraint already does.
            // Please note: the distributive constraint is a kludge for emulating what a negated type could to do filter
            // a union - once negated types exist and are applied to the conditional false branch, this "constraint"
            // likely doesn't need to exist.
            if (type.root.isDistributive && type.restrictiveInstantiation !== type) {
                const simplified = getSimplifiedType(type.checkType, /*writing*/ false);
                const constraint = simplified === type.checkType ? getConstraintOfType(simplified) : simplified;
                if (constraint && constraint !== type.checkType) {
                    const instantiated = getConditionalTypeInstantiation(type, prependTypeMapping(type.root.checkType, constraint, type.mapper));
                    if (!(instantiated.flags & TypeFlags.Never)) {
                        return instantiated;
                    }
                }
            }
            return undefined;
        }

        function getConstraintFromConditionalType(type: ConditionalType) {
            return getConstraintOfDistributiveConditionalType(type) || getDefaultConstraintOfConditionalType(type);
        }

        function getConstraintOfConditionalType(type: ConditionalType) {
            return hasNonCircularBaseConstraint(type) ? getConstraintFromConditionalType(type) : undefined;
        }

        function getEffectiveConstraintOfIntersection(types: readonly Type[], targetIsUnion: boolean) {
            let constraints: Type[] | undefined;
            let hasDisjointDomainType = false;
            for (const t of types) {
                if (t.flags & TypeFlags.Instantiable) {
                    // We keep following constraints as long as we have an instantiable type that is known
                    // not to be circular or infinite (hence we stop on index access types).
                    let constraint = getConstraintOfType(t);
                    while (constraint && constraint.flags & (TypeFlags.TypeParameter | TypeFlags.Index | TypeFlags.Conditional)) {
                        constraint = getConstraintOfType(constraint);
                    }
                    if (constraint) {
                        constraints = append(constraints, constraint);
                        if (targetIsUnion) {
                            constraints = append(constraints, t);
                        }
                    }
                }
                else if (t.flags & TypeFlags.DisjointDomains) {
                    hasDisjointDomainType = true;
                }
            }
            // If the target is a union type or if we are intersecting with types belonging to one of the
            // disjoint domains, we may end up producing a constraint that hasn't been examined before.
            if (constraints && (targetIsUnion || hasDisjointDomainType)) {
                if (hasDisjointDomainType) {
                    // We add any types belong to one of the disjoint domains because they might cause the final
                    // intersection operation to reduce the union constraints.
                    for (const t of types) {
                        if (t.flags & TypeFlags.DisjointDomains) {
                            constraints = append(constraints, t);
                        }
                    }
                }
                return getIntersectionType(constraints);
            }
            return undefined;
        }

        function getBaseConstraintOfType(type: Type): Type | undefined {
            if (type.flags & (TypeFlags.InstantiableNonPrimitive | TypeFlags.UnionOrIntersection)) {
                const constraint = getResolvedBaseConstraint(<InstantiableType | UnionOrIntersectionType>type);
                return constraint !== noConstraintType && constraint !== circularConstraintType ? constraint : undefined;
            }
            return type.flags & TypeFlags.Index ? keyofConstraintType : undefined;
        }

        /**
         * This is similar to `getBaseConstraintOfType` except it returns the input type if there's no base constraint, instead of `undefined`
         * It also doesn't map indexes to `string`, as where this is used this would be unneeded (and likely undesirable)
         */
        function getBaseConstraintOrType(type: Type) {
            return getBaseConstraintOfType(type) || type;
        }

        function hasNonCircularBaseConstraint(type: InstantiableType): boolean {
            return getResolvedBaseConstraint(type) !== circularConstraintType;
        }

        /**
         * Return the resolved base constraint of a type variable. The noConstraintType singleton is returned if the
         * type variable has no constraint, and the circularConstraintType singleton is returned if the constraint
         * circularly references the type variable.
         */
        function getResolvedBaseConstraint(type: InstantiableType | UnionOrIntersectionType): Type {
            let nonTerminating = false;
            return type.resolvedBaseConstraint ||
                (type.resolvedBaseConstraint = getTypeWithThisArgument(getImmediateBaseConstraint(type), type));

            function getImmediateBaseConstraint(t: Type): Type {
                if (!t.immediateBaseConstraint) {
                    if (!pushTypeResolution(t, TypeSystemPropertyName.ImmediateBaseConstraint)) {
                        return circularConstraintType;
                    }
                    if (constraintDepth >= 50) {
                        // We have reached 50 recursive invocations of getImmediateBaseConstraint and there is a
                        // very high likelihood we're dealing with an infinite generic type that perpetually generates
                        // new type identities as we descend into it. We stop the recursion here and mark this type
                        // and the outer types as having circular constraints.
                        error(currentNode, Diagnostics.Type_instantiation_is_excessively_deep_and_possibly_infinite);
                        nonTerminating = true;
                        return t.immediateBaseConstraint = noConstraintType;
                    }
                    constraintDepth++;
                    let result = computeBaseConstraint(getSimplifiedType(t, /*writing*/ false));
                    constraintDepth--;
                    if (!popTypeResolution()) {
                        if (t.flags & TypeFlags.TypeParameter) {
                            const errorNode = getConstraintDeclaration(<TypeParameter>t);
                            if (errorNode) {
                                const diagnostic = error(errorNode, Diagnostics.Type_parameter_0_has_a_circular_constraint, typeToString(t));
                                if (currentNode && !isNodeDescendantOf(errorNode, currentNode) && !isNodeDescendantOf(currentNode, errorNode)) {
                                    addRelatedInfo(diagnostic, createDiagnosticForNode(currentNode, Diagnostics.Circularity_originates_in_type_at_this_location));
                                }
                            }
                        }
                        result = circularConstraintType;
                    }
                    if (nonTerminating) {
                        result = circularConstraintType;
                    }
                    t.immediateBaseConstraint = result || noConstraintType;
                }
                return t.immediateBaseConstraint;
            }

            function getBaseConstraint(t: Type): Type | undefined {
                const c = getImmediateBaseConstraint(t);
                return c !== noConstraintType && c !== circularConstraintType ? c : undefined;
            }

            function computeBaseConstraint(t: Type): Type | undefined {
                if (t.flags & TypeFlags.TypeParameter) {
                    const constraint = getConstraintFromTypeParameter(<TypeParameter>t);
                    return (t as TypeParameter).isThisType || !constraint ?
                        constraint :
                        getBaseConstraint(constraint);
                }
                if (t.flags & TypeFlags.UnionOrIntersection) {
                    const types = (<UnionOrIntersectionType>t).types;
                    const baseTypes: Type[] = [];
                    for (const type of types) {
                        const baseType = getBaseConstraint(type);
                        if (baseType) {
                            baseTypes.push(baseType);
                        }
                    }
                    return t.flags & TypeFlags.Union && baseTypes.length === types.length ? getUnionType(baseTypes) :
                        t.flags & TypeFlags.Intersection && baseTypes.length ? getIntersectionType(baseTypes) :
                        undefined;
                }
                if (t.flags & TypeFlags.Index) {
                    return keyofConstraintType;
                }
                if (t.flags & TypeFlags.IndexedAccess) {
                    const baseObjectType = getBaseConstraint((<IndexedAccessType>t).objectType);
                    const baseIndexType = getBaseConstraint((<IndexedAccessType>t).indexType);
                    const baseIndexedAccess = baseObjectType && baseIndexType && getIndexedAccessTypeOrUndefined(baseObjectType, baseIndexType);
                    return baseIndexedAccess && getBaseConstraint(baseIndexedAccess);
                }
                if (t.flags & TypeFlags.Conditional) {
                    const constraint = getConstraintFromConditionalType(<ConditionalType>t);
                    constraintDepth++; // Penalize repeating conditional types (this captures the recursion within getConstraintFromConditionalType and carries it forward)
                    const result = constraint && getBaseConstraint(constraint);
                    constraintDepth--;
                    return result;
                }
                if (t.flags & TypeFlags.Substitution) {
                    return getBaseConstraint((<SubstitutionType>t).substitute);
                }
                return t;
            }
        }

        function getApparentTypeOfIntersectionType(type: IntersectionType) {
            return type.resolvedApparentType || (type.resolvedApparentType = getTypeWithThisArgument(type, type, /*apparentType*/ true));
        }

        function getResolvedTypeParameterDefault(typeParameter: TypeParameter): Type | undefined {
            if (!typeParameter.default) {
                if (typeParameter.target) {
                    const targetDefault = getResolvedTypeParameterDefault(typeParameter.target);
                    typeParameter.default = targetDefault ? instantiateType(targetDefault, typeParameter.mapper) : noConstraintType;
                }
                else {
                    // To block recursion, set the initial value to the resolvingDefaultType.
                    typeParameter.default = resolvingDefaultType;
                    const defaultDeclaration = typeParameter.symbol && forEach(typeParameter.symbol.declarations, decl => isTypeParameterDeclaration(decl) && decl.default);
                    const defaultType = defaultDeclaration ? getTypeFromTypeNode(defaultDeclaration) : noConstraintType;
                    if (typeParameter.default === resolvingDefaultType) {
                        // If we have not been called recursively, set the correct default type.
                        typeParameter.default = defaultType;
                    }
                }
            }
            else if (typeParameter.default === resolvingDefaultType) {
                // If we are called recursively for this type parameter, mark the default as circular.
                typeParameter.default = circularConstraintType;
            }
            return typeParameter.default;
        }

        /**
         * Gets the default type for a type parameter.
         *
         * If the type parameter is the result of an instantiation, this gets the instantiated
         * default type of its target. If the type parameter has no default type or the default is
         * circular, `undefined` is returned.
         */
        function getDefaultFromTypeParameter(typeParameter: TypeParameter): Type | undefined {
            const defaultType = getResolvedTypeParameterDefault(typeParameter);
            return defaultType !== noConstraintType && defaultType !== circularConstraintType ? defaultType : undefined;
        }

        function hasNonCircularTypeParameterDefault(typeParameter: TypeParameter) {
            return getResolvedTypeParameterDefault(typeParameter) !== circularConstraintType;
        }

        /**
         * Indicates whether the declaration of a typeParameter has a default type.
         */
        function hasTypeParameterDefault(typeParameter: TypeParameter): boolean {
            return !!(typeParameter.symbol && forEach(typeParameter.symbol.declarations, decl => isTypeParameterDeclaration(decl) && decl.default));
        }

        function getApparentTypeOfMappedType(type: MappedType) {
            return type.resolvedApparentType || (type.resolvedApparentType = getResolvedApparentTypeOfMappedType(type));
        }

        function getResolvedApparentTypeOfMappedType(type: MappedType) {
            const typeVariable = getHomomorphicTypeVariable(type);
            if (typeVariable) {
                const constraint = getConstraintOfTypeParameter(typeVariable);
                if (constraint && (isArrayType(constraint) || isTupleType(constraint))) {
                    return instantiateType(type, prependTypeMapping(typeVariable, constraint, type.mapper));
                }
            }
            return type;
        }

        /**
         * For a type parameter, return the base constraint of the type parameter. For the string, number,
         * boolean, and symbol primitive types, return the corresponding object types. Otherwise return the
         * type itself.
         */
        function getApparentType(type: Type): Type {
            const t = type.flags & TypeFlags.Instantiable ? getBaseConstraintOfType(type) || unknownType : type;
            return getObjectFlags(t) & ObjectFlags.Mapped ? getApparentTypeOfMappedType(<MappedType>t) :
                t.flags & TypeFlags.Intersection ? getApparentTypeOfIntersectionType(<IntersectionType>t) :
                t.flags & TypeFlags.StringLike ? globalStringType :
                t.flags & TypeFlags.NumberLike ? globalNumberType :
                t.flags & TypeFlags.BigIntLike ? getGlobalBigIntType(/*reportErrors*/ languageVersion >= ScriptTarget.ES2020) :
                t.flags & TypeFlags.BooleanLike ? globalBooleanType :
                t.flags & TypeFlags.ESSymbolLike ? getGlobalESSymbolType(/*reportErrors*/ languageVersion >= ScriptTarget.ES2015) :
                t.flags & TypeFlags.NonPrimitive ? emptyObjectType :
                t.flags & TypeFlags.Index ? keyofConstraintType :
                t.flags & TypeFlags.Unknown && !strictNullChecks ? emptyObjectType :
                t;
        }

        function getReducedApparentType(type: Type): Type {
            // Since getApparentType may return a non-reduced union or intersection type, we need to perform
            // type reduction both before and after obtaining the apparent type. For example, given a type parameter
            // 'T extends A | B', the type 'T & X' becomes 'A & X | B & X' after obtaining the apparent type, and
            // that type may need further reduction to remove empty intersections.
            return getReducedType(getApparentType(getReducedType(type)));
        }

        function createUnionOrIntersectionProperty(containingType: UnionOrIntersectionType, name: __String): Symbol | undefined {
            let singleProp: Symbol | undefined;
            let propSet: ESMap<SymbolId, Symbol> | undefined;
            let indexTypes: Type[] | undefined;
            const isUnion = containingType.flags & TypeFlags.Union;
            // Flags we want to propagate to the result if they exist in all source symbols
            let optionalFlag = isUnion ? SymbolFlags.None : SymbolFlags.Optional;
            let syntheticFlag = CheckFlags.SyntheticMethod;
            let checkFlags = 0;
            for (const current of containingType.types) {
                const type = getApparentType(current);
                if (!(type === errorType || type.flags & TypeFlags.Never)) {
                    const prop = getPropertyOfType(type, name);
                    const modifiers = prop ? getDeclarationModifierFlagsFromSymbol(prop) : 0;
                    if (prop) {
                        if (isUnion) {
                            optionalFlag |= (prop.flags & SymbolFlags.Optional);
                        }
                        else {
                            optionalFlag &= prop.flags;
                        }
                        if (!singleProp) {
                            singleProp = prop;
                        }
                        else if (prop !== singleProp) {
                            if (!propSet) {
                                propSet = new Map<SymbolId, Symbol>();
                                propSet.set(getSymbolId(singleProp), singleProp);
                            }
                            const id = getSymbolId(prop);
                            if (!propSet.has(id)) {
                                propSet.set(id, prop);
                            }
                        }
                        checkFlags |= (isReadonlySymbol(prop) ? CheckFlags.Readonly : 0) |
                            (!(modifiers & ModifierFlags.NonPublicAccessibilityModifier) ? CheckFlags.ContainsPublic : 0) |
                            (modifiers & ModifierFlags.Protected ? CheckFlags.ContainsProtected : 0) |
                            (modifiers & ModifierFlags.Private ? CheckFlags.ContainsPrivate : 0) |
                            (modifiers & ModifierFlags.Static ? CheckFlags.ContainsStatic : 0);
                        if (!isPrototypeProperty(prop)) {
                            syntheticFlag = CheckFlags.SyntheticProperty;
                        }
                    }
                    else if (isUnion) {
                        const indexInfo = !isLateBoundName(name) && (isNumericLiteralName(name) && getIndexInfoOfType(type, IndexKind.Number) || getIndexInfoOfType(type, IndexKind.String));
                        if (indexInfo) {
                            checkFlags |= CheckFlags.WritePartial | (indexInfo.isReadonly ? CheckFlags.Readonly : 0);
                            indexTypes = append(indexTypes, isTupleType(type) ? getRestTypeOfTupleType(type) || undefinedType : indexInfo.type);
                        }
                        else if (isObjectLiteralType(type)) {
                            checkFlags |= CheckFlags.WritePartial;
                            indexTypes = append(indexTypes, undefinedType);
                        }
                        else {
                            checkFlags |= CheckFlags.ReadPartial;
                        }
                    }
                }
            }
            if (!singleProp || isUnion && (propSet || checkFlags & CheckFlags.Partial) && checkFlags & (CheckFlags.ContainsPrivate | CheckFlags.ContainsProtected)) {
                // No property was found, or, in a union, a property has a private or protected declaration in one
                // constituent, but is missing or has a different declaration in another constituent.
                return undefined;
            }
            if (!propSet && !(checkFlags & CheckFlags.ReadPartial) && !indexTypes) {
                return singleProp;
            }
            const props = propSet ? arrayFrom(propSet.values()) : [singleProp];
            let declarations: Declaration[] | undefined;
            let firstType: Type | undefined;
            let nameType: Type | undefined;
            const propTypes: Type[] = [];
            let firstValueDeclaration: Declaration | undefined;
            let hasNonUniformValueDeclaration = false;
            for (const prop of props) {
                if (!firstValueDeclaration) {
                    firstValueDeclaration = prop.valueDeclaration;
                }
                else if (prop.valueDeclaration && prop.valueDeclaration !== firstValueDeclaration) {
                    hasNonUniformValueDeclaration = true;
                }
                declarations = addRange(declarations, prop.declarations);
                const type = getTypeOfSymbol(prop);
                if (!firstType) {
                    firstType = type;
                    nameType = getSymbolLinks(prop).nameType;
                }
                else if (type !== firstType) {
                    checkFlags |= CheckFlags.HasNonUniformType;
                }
                if (isLiteralType(type)) {
                    checkFlags |= CheckFlags.HasLiteralType;
                }
                if (type.flags & TypeFlags.Never) {
                    checkFlags |= CheckFlags.HasNeverType;
                }
                propTypes.push(type);
            }
            addRange(propTypes, indexTypes);
            const result = createSymbol(SymbolFlags.Property | optionalFlag, name, syntheticFlag | checkFlags);
            result.containingType = containingType;
            if (!hasNonUniformValueDeclaration && firstValueDeclaration) {
                result.valueDeclaration = firstValueDeclaration;

                // Inherit information about parent type.
                if (firstValueDeclaration.symbol.parent) {
                    result.parent = firstValueDeclaration.symbol.parent;
                }
            }

            result.declarations = declarations!;
            result.nameType = nameType;
            if (propTypes.length > 2) {
                // When `propTypes` has the potential to explode in size when normalized, defer normalization until absolutely needed
                result.checkFlags |= CheckFlags.DeferredType;
                result.deferralParent = containingType;
                result.deferralConstituents = propTypes;
            }
            else {
                result.type = isUnion ? getUnionType(propTypes) : getIntersectionType(propTypes);
            }
            return result;
        }

        // Return the symbol for a given property in a union or intersection type, or undefined if the property
        // does not exist in any constituent type. Note that the returned property may only be present in some
        // constituents, in which case the isPartial flag is set when the containing type is union type. We need
        // these partial properties when identifying discriminant properties, but otherwise they are filtered out
        // and do not appear to be present in the union type.
        function getUnionOrIntersectionProperty(type: UnionOrIntersectionType, name: __String): Symbol | undefined {
            const properties = type.propertyCache || (type.propertyCache = createSymbolTable());
            let property = properties.get(name);
            if (!property) {
                property = createUnionOrIntersectionProperty(type, name);
                if (property) {
                    properties.set(name, property);
                }
            }
            return property;
        }

        function getPropertyOfUnionOrIntersectionType(type: UnionOrIntersectionType, name: __String): Symbol | undefined {
            const property = getUnionOrIntersectionProperty(type, name);
            // We need to filter out partial properties in union types
            return property && !(getCheckFlags(property) & CheckFlags.ReadPartial) ? property : undefined;
        }

        /**
         * Return the reduced form of the given type. For a union type, it is a union of the normalized constituent types.
         * For an intersection of types containing one or more mututally exclusive discriminant properties, it is 'never'.
         * For all other types, it is simply the type itself. Discriminant properties are considered mutually exclusive when
         * no constituent property has type 'never', but the intersection of the constituent property types is 'never'.
         */
        function getReducedType(type: Type): Type {
            if (type.flags & TypeFlags.Union && (<UnionType>type).objectFlags & ObjectFlags.ContainsIntersections) {
                return (<UnionType>type).resolvedReducedType || ((<UnionType>type).resolvedReducedType = getReducedUnionType(<UnionType>type));
            }
            else if (type.flags & TypeFlags.Intersection) {
                if (!((<IntersectionType>type).objectFlags & ObjectFlags.IsNeverIntersectionComputed)) {
                    (<IntersectionType>type).objectFlags |= ObjectFlags.IsNeverIntersectionComputed |
                        (some(getPropertiesOfUnionOrIntersectionType(<IntersectionType>type), isNeverReducedProperty) ? ObjectFlags.IsNeverIntersection : 0);
                }
                return (<IntersectionType>type).objectFlags & ObjectFlags.IsNeverIntersection ? neverType : type;
            }
            return type;
        }

        function getReducedUnionType(unionType: UnionType) {
            const reducedTypes = sameMap(unionType.types, getReducedType);
            if (reducedTypes === unionType.types) {
                return unionType;
            }
            const reduced = getUnionType(reducedTypes);
            if (reduced.flags & TypeFlags.Union) {
                (<UnionType>reduced).resolvedReducedType = reduced;
            }
            return reduced;
        }

        function isNeverReducedProperty(prop: Symbol) {
            return isDiscriminantWithNeverType(prop) || isConflictingPrivateProperty(prop);
        }

        function isDiscriminantWithNeverType(prop: Symbol) {
            // Return true for a synthetic non-optional property with non-uniform types, where at least one is
            // a literal type and none is never, that reduces to never.
            return !(prop.flags & SymbolFlags.Optional) &&
                (getCheckFlags(prop) & (CheckFlags.Discriminant | CheckFlags.HasNeverType)) === CheckFlags.Discriminant &&
                !!(getTypeOfSymbol(prop).flags & TypeFlags.Never);
        }

        function isConflictingPrivateProperty(prop: Symbol) {
            // Return true for a synthetic property with multiple declarations, at least one of which is private.
            return !prop.valueDeclaration && !!(getCheckFlags(prop) & CheckFlags.ContainsPrivate);
        }

        function elaborateNeverIntersection(errorInfo: DiagnosticMessageChain | undefined, type: Type) {
            if (getObjectFlags(type) & ObjectFlags.IsNeverIntersection) {
                const neverProp = find(getPropertiesOfUnionOrIntersectionType(<IntersectionType>type), isDiscriminantWithNeverType);
                if (neverProp) {
                    return chainDiagnosticMessages(errorInfo, Diagnostics.The_intersection_0_was_reduced_to_never_because_property_1_has_conflicting_types_in_some_constituents,
                        typeToString(type, /*enclosingDeclaration*/ undefined, TypeFormatFlags.NoTypeReduction), symbolToString(neverProp));
                }
                const privateProp = find(getPropertiesOfUnionOrIntersectionType(<IntersectionType>type), isConflictingPrivateProperty);
                if (privateProp) {
                    return chainDiagnosticMessages(errorInfo, Diagnostics.The_intersection_0_was_reduced_to_never_because_property_1_exists_in_multiple_constituents_and_is_private_in_some,
                        typeToString(type, /*enclosingDeclaration*/ undefined, TypeFormatFlags.NoTypeReduction), symbolToString(privateProp));
                }
            }
            return errorInfo;
        }

        /**
         * Return the symbol for the property with the given name in the given type. Creates synthetic union properties when
         * necessary, maps primitive types and type parameters are to their apparent types, and augments with properties from
         * Object and Function as appropriate.
         *
         * @param type a type to look up property from
         * @param name a name of property to look up in a given type
         */
        function getPropertyOfType(type: Type, name: __String): Symbol | undefined {
            type = getReducedApparentType(type);
            if (type.flags & TypeFlags.Object) {
                const resolved = resolveStructuredTypeMembers(<ObjectType>type);
                const symbol = resolved.members.get(name);
                if (symbol && symbolIsValue(symbol)) {
                    return symbol;
                }
                const functionType = resolved === anyFunctionType ? globalFunctionType :
                    resolved.callSignatures.length ? globalCallableFunctionType :
                    resolved.constructSignatures.length ? globalNewableFunctionType :
                    undefined;
                if (functionType) {
                    const symbol = getPropertyOfObjectType(functionType, name);
                    if (symbol) {
                        return symbol;
                    }
                }
                return getPropertyOfObjectType(globalObjectType, name);
            }
            if (type.flags & TypeFlags.UnionOrIntersection) {
                return getPropertyOfUnionOrIntersectionType(<UnionOrIntersectionType>type, name);
            }
            return undefined;
        }

        function getSignaturesOfStructuredType(type: Type, kind: SignatureKind): readonly Signature[] {
            if (type.flags & TypeFlags.StructuredType) {
                const resolved = resolveStructuredTypeMembers(<ObjectType>type);
                return kind === SignatureKind.Call ? resolved.callSignatures : resolved.constructSignatures;
            }
            return emptyArray;
        }

        /**
         * Return the signatures of the given kind in the given type. Creates synthetic union signatures when necessary and
         * maps primitive types and type parameters are to their apparent types.
         */
        function getSignaturesOfType(type: Type, kind: SignatureKind): readonly Signature[] {
            return getSignaturesOfStructuredType(getReducedApparentType(type), kind);
        }

        function getIndexInfoOfStructuredType(type: Type, kind: IndexKind): IndexInfo | undefined {
            if (type.flags & TypeFlags.StructuredType) {
                const resolved = resolveStructuredTypeMembers(<ObjectType>type);
                return kind === IndexKind.String ? resolved.stringIndexInfo : resolved.numberIndexInfo;
            }
        }

        function getIndexTypeOfStructuredType(type: Type, kind: IndexKind): Type | undefined {
            const info = getIndexInfoOfStructuredType(type, kind);
            return info && info.type;
        }

        // Return the indexing info of the given kind in the given type. Creates synthetic union index types when necessary and
        // maps primitive types and type parameters are to their apparent types.
        function getIndexInfoOfType(type: Type, kind: IndexKind): IndexInfo | undefined {
            return getIndexInfoOfStructuredType(getReducedApparentType(type), kind);
        }

        // Return the index type of the given kind in the given type. Creates synthetic union index types when necessary and
        // maps primitive types and type parameters are to their apparent types.
        function getIndexTypeOfType(type: Type, kind: IndexKind): Type | undefined {
            return getIndexTypeOfStructuredType(getReducedApparentType(type), kind);
        }

        function getImplicitIndexTypeOfType(type: Type, kind: IndexKind): Type | undefined {
            if (isObjectTypeWithInferableIndex(type)) {
                const propTypes: Type[] = [];
                for (const prop of getPropertiesOfType(type)) {
                    if (kind === IndexKind.String || isNumericLiteralName(prop.escapedName)) {
                        propTypes.push(getTypeOfSymbol(prop));
                    }
                }
                if (kind === IndexKind.String) {
                    append(propTypes, getIndexTypeOfType(type, IndexKind.Number));
                }
                if (propTypes.length) {
                    return getUnionType(propTypes);
                }
            }
            return undefined;
        }

        // Return list of type parameters with duplicates removed (duplicate identifier errors are generated in the actual
        // type checking functions).
        function getTypeParametersFromDeclaration(declaration: DeclarationWithTypeParameters): TypeParameter[] | undefined {
            let result: TypeParameter[] | undefined;
            for (const node of getEffectiveTypeParameterDeclarations(declaration)) {
                result = appendIfUnique(result, getDeclaredTypeOfTypeParameter(node.symbol));
            }
            return result;
        }

        function symbolsToArray(symbols: SymbolTable): Symbol[] {
            const result: Symbol[] = [];
            symbols.forEach((symbol, id) => {
                if (!isReservedMemberName(id)) {
                    result.push(symbol);
                }
            });
            return result;
        }

        function isJSDocOptionalParameter(node: ParameterDeclaration) {
            return isInJSFile(node) && (
                // node.type should only be a JSDocOptionalType when node is a parameter of a JSDocFunctionType
                node.type && node.type.kind === SyntaxKind.JSDocOptionalType
                || getJSDocParameterTags(node).some(({ isBracketed, typeExpression }) =>
                    isBracketed || !!typeExpression && typeExpression.type.kind === SyntaxKind.JSDocOptionalType));
        }

        function tryFindAmbientModule(moduleName: string, withAugmentations: boolean) {
            if (isExternalModuleNameRelative(moduleName)) {
                return undefined;
            }
            const symbol = getSymbol(globals, '"' + moduleName + '"' as __String, SymbolFlags.ValueModule);
            // merged symbol is module declaration symbol combined with all augmentations
            return symbol && withAugmentations ? getMergedSymbol(symbol) : symbol;
        }

        function isOptionalParameter(node: ParameterDeclaration | JSDocParameterTag | JSDocPropertyTag) {
            if (hasQuestionToken(node) || isOptionalJSDocPropertyLikeTag(node) || isJSDocOptionalParameter(node)) {
                return true;
            }

            if (node.initializer) {
                const signature = getSignatureFromDeclaration(node.parent);
                const parameterIndex = node.parent.parameters.indexOf(node);
                Debug.assert(parameterIndex >= 0);
                return parameterIndex >= getMinArgumentCount(signature, /*strongArityForUntypedJS*/ true);
            }
            const iife = getImmediatelyInvokedFunctionExpression(node.parent);
            if (iife) {
                return !node.type &&
                    !node.dotDotDotToken &&
                    node.parent.parameters.indexOf(node) >= iife.arguments.length;
            }

            return false;
        }

        function isOptionalJSDocPropertyLikeTag(node: Node): node is JSDocPropertyLikeTag {
            if (!isJSDocPropertyLikeTag(node)) {
                return false;
            }
            const { isBracketed, typeExpression } = node;
            return isBracketed || !!typeExpression && typeExpression.type.kind === SyntaxKind.JSDocOptionalType;
        }

        function createTypePredicate(kind: TypePredicateKind, parameterName: string | undefined, parameterIndex: number | undefined, type: Type | undefined): TypePredicate {
            return { kind, parameterName, parameterIndex, type } as TypePredicate;
        }

        /**
         * Gets the minimum number of type arguments needed to satisfy all non-optional type
         * parameters.
         */
        function getMinTypeArgumentCount(typeParameters: readonly TypeParameter[] | undefined): number {
            let minTypeArgumentCount = 0;
            if (typeParameters) {
                for (let i = 0; i < typeParameters.length; i++) {
                    if (!hasTypeParameterDefault(typeParameters[i])) {
                        minTypeArgumentCount = i + 1;
                    }
                }
            }
            return minTypeArgumentCount;
        }

        /**
         * Fill in default types for unsupplied type arguments. If `typeArguments` is undefined
         * when a default type is supplied, a new array will be created and returned.
         *
         * @param typeArguments The supplied type arguments.
         * @param typeParameters The requested type parameters.
         * @param minTypeArgumentCount The minimum number of required type arguments.
         */
        function fillMissingTypeArguments(typeArguments: readonly Type[], typeParameters: readonly TypeParameter[] | undefined, minTypeArgumentCount: number, isJavaScriptImplicitAny: boolean): Type[];
        function fillMissingTypeArguments(typeArguments: readonly Type[] | undefined, typeParameters: readonly TypeParameter[] | undefined, minTypeArgumentCount: number, isJavaScriptImplicitAny: boolean): Type[] | undefined;
        function fillMissingTypeArguments(typeArguments: readonly Type[] | undefined, typeParameters: readonly TypeParameter[] | undefined, minTypeArgumentCount: number, isJavaScriptImplicitAny: boolean) {
            const numTypeParameters = length(typeParameters);
            if (!numTypeParameters) {
                return [];
            }
            const numTypeArguments = length(typeArguments);
            if (isJavaScriptImplicitAny || (numTypeArguments >= minTypeArgumentCount && numTypeArguments <= numTypeParameters)) {
                const result = typeArguments ? typeArguments.slice() : [];
                // Map invalid forward references in default types to the error type
                for (let i = numTypeArguments; i < numTypeParameters; i++) {
                    result[i] = errorType;
                }
                const baseDefaultType = getDefaultTypeArgumentType(isJavaScriptImplicitAny);
                for (let i = numTypeArguments; i < numTypeParameters; i++) {
                    let defaultType = getDefaultFromTypeParameter(typeParameters![i]);
                    if (isJavaScriptImplicitAny && defaultType && (isTypeIdenticalTo(defaultType, unknownType) || isTypeIdenticalTo(defaultType, emptyObjectType))) {
                        defaultType = anyType;
                    }
                    result[i] = defaultType ? instantiateType(defaultType, createTypeMapper(typeParameters!, result)) : baseDefaultType;
                }
                result.length = typeParameters!.length;
                return result;
            }
            return typeArguments && typeArguments.slice();
        }

        function getSignatureFromDeclaration(declaration: SignatureDeclaration | JSDocSignature): Signature {
            const links = getNodeLinks(declaration);
            if (!links.resolvedSignature) {
                const parameters: Symbol[] = [];
                let flags = SignatureFlags.None;
                let minArgumentCount = 0;
                let thisParameter: Symbol | undefined;
                let hasThisParameter = false;
                const iife = getImmediatelyInvokedFunctionExpression(declaration);
                const isJSConstructSignature = isJSDocConstructSignature(declaration);
                const isUntypedSignatureInJSFile = !iife &&
                    isInJSFile(declaration) &&
                    isValueSignatureDeclaration(declaration) &&
                    !hasJSDocParameterTags(declaration) &&
                    !getJSDocType(declaration);
                if (isUntypedSignatureInJSFile) {
                    flags |= SignatureFlags.IsUntypedSignatureInJSFile;
                }

                // If this is a JSDoc construct signature, then skip the first parameter in the
                // parameter list.  The first parameter represents the return type of the construct
                // signature.
                for (let i = isJSConstructSignature ? 1 : 0; i < declaration.parameters.length; i++) {
                    const param = declaration.parameters[i];

                    let paramSymbol = param.symbol;
                    const type = isJSDocParameterTag(param) ? (param.typeExpression && param.typeExpression.type) : param.type;
                    // Include parameter symbol instead of property symbol in the signature
                    if (paramSymbol && !!(paramSymbol.flags & SymbolFlags.Property) && !isBindingPattern(param.name)) {
                        const resolvedSymbol = resolveName(param, paramSymbol.escapedName, SymbolFlags.Value, undefined, undefined, /*isUse*/ false);
                        paramSymbol = resolvedSymbol!;
                    }
                    if (i === 0 && paramSymbol.escapedName === InternalSymbolName.This) {
                        hasThisParameter = true;
                        thisParameter = param.symbol;
                    }
                    else {
                        parameters.push(paramSymbol);
                    }

                    if (type && type.kind === SyntaxKind.LiteralType) {
                        flags |= SignatureFlags.HasLiteralTypes;
                    }

                    // Record a new minimum argument count if this is not an optional parameter
                    const isOptionalParameter = isOptionalJSDocPropertyLikeTag(param) ||
                        param.initializer || param.questionToken || param.dotDotDotToken ||
                        iife && parameters.length > iife.arguments.length && !type ||
                        isJSDocOptionalParameter(param);
                    if (!isOptionalParameter) {
                        minArgumentCount = parameters.length;
                    }
                }

                // If only one accessor includes a this-type annotation, the other behaves as if it had the same type annotation
                if ((declaration.kind === SyntaxKind.GetAccessor || declaration.kind === SyntaxKind.SetAccessor) &&
                    !hasNonBindableDynamicName(declaration) &&
                    (!hasThisParameter || !thisParameter)) {
                    const otherKind = declaration.kind === SyntaxKind.GetAccessor ? SyntaxKind.SetAccessor : SyntaxKind.GetAccessor;
                    const other = getDeclarationOfKind<AccessorDeclaration>(getSymbolOfNode(declaration), otherKind);
                    if (other) {
                        thisParameter = getAnnotatedAccessorThisParameter(other);
                    }
                }

                const classType = declaration.kind === SyntaxKind.Constructor ?
                    getDeclaredTypeOfClassOrInterface(getMergedSymbol((<ClassDeclaration>declaration.parent).symbol))
                    : undefined;
                const typeParameters = classType ? classType.localTypeParameters : getTypeParametersFromDeclaration(declaration);
                if (hasRestParameter(declaration) || isInJSFile(declaration) && maybeAddJsSyntheticRestParameter(declaration, parameters)) {
                    flags |= SignatureFlags.HasRestParameter;
                }
                links.resolvedSignature = createSignature(declaration, typeParameters, thisParameter, parameters,
                    /*resolvedReturnType*/ undefined, /*resolvedTypePredicate*/ undefined,
                    minArgumentCount, flags);
            }
            return links.resolvedSignature;
        }

        /**
         * A JS function gets a synthetic rest parameter if it references `arguments` AND:
         * 1. It has no parameters but at least one `@param` with a type that starts with `...`
         * OR
         * 2. It has at least one parameter, and the last parameter has a matching `@param` with a type that starts with `...`
         */
        function maybeAddJsSyntheticRestParameter(declaration: SignatureDeclaration | JSDocSignature, parameters: Symbol[]): boolean {
            if (isJSDocSignature(declaration) || !containsArgumentsReference(declaration)) {
                return false;
            }
            const lastParam = lastOrUndefined(declaration.parameters);
            const lastParamTags = lastParam ? getJSDocParameterTags(lastParam) : getJSDocTags(declaration).filter(isJSDocParameterTag);
            const lastParamVariadicType = firstDefined(lastParamTags, p =>
                p.typeExpression && isJSDocVariadicType(p.typeExpression.type) ? p.typeExpression.type : undefined);

            const syntheticArgsSymbol = createSymbol(SymbolFlags.Variable, "args" as __String, CheckFlags.RestParameter);
            syntheticArgsSymbol.type = lastParamVariadicType ? createArrayType(getTypeFromTypeNode(lastParamVariadicType.type)) : anyArrayType;
            if (lastParamVariadicType) {
                // Replace the last parameter with a rest parameter.
                parameters.pop();
            }
            parameters.push(syntheticArgsSymbol);
            return true;
        }

        function getSignatureOfTypeTag(node: SignatureDeclaration | JSDocSignature) {
            // should be attached to a function declaration or expression
            if (!(isInJSFile(node) && isFunctionLikeDeclaration(node))) return undefined;
            const typeTag = getJSDocTypeTag(node);
            const signature = typeTag && typeTag.typeExpression && getSingleCallSignature(getTypeFromTypeNode(typeTag.typeExpression));
            return signature && getErasedSignature(signature);
        }

        function getReturnTypeOfTypeTag(node: SignatureDeclaration | JSDocSignature) {
            const signature = getSignatureOfTypeTag(node);
            return signature && getReturnTypeOfSignature(signature);
        }

        function containsArgumentsReference(declaration: SignatureDeclaration): boolean {
            const links = getNodeLinks(declaration);
            if (links.containsArgumentsReference === undefined) {
                if (links.flags & NodeCheckFlags.CaptureArguments) {
                    links.containsArgumentsReference = true;
                }
                else {
                    links.containsArgumentsReference = traverse((declaration as FunctionLikeDeclaration).body!);
                }
            }
            return links.containsArgumentsReference;

            function traverse(node: Node): boolean {
                if (!node) return false;
                switch (node.kind) {
                    case SyntaxKind.Identifier:
                        return (<Identifier>node).escapedText === "arguments" && isExpressionNode(node);

                    case SyntaxKind.PropertyDeclaration:
                    case SyntaxKind.MethodDeclaration:
                    case SyntaxKind.GetAccessor:
                    case SyntaxKind.SetAccessor:
                        return (<NamedDeclaration>node).name!.kind === SyntaxKind.ComputedPropertyName
                            && traverse((<NamedDeclaration>node).name!);

                    default:
                        return !nodeStartsNewLexicalEnvironment(node) && !isPartOfTypeNode(node) && !!forEachChild(node, traverse);
                }
            }
        }

        function getSignaturesOfSymbol(symbol: Symbol | undefined): Signature[] {
            if (!symbol) return emptyArray;
            const result: Signature[] = [];
            for (let i = 0; i < symbol.declarations.length; i++) {
                const decl = symbol.declarations[i];
                if (!isFunctionLike(decl)) continue;
                // Don't include signature if node is the implementation of an overloaded function. A node is considered
                // an implementation node if it has a body and the previous node is of the same kind and immediately
                // precedes the implementation node (i.e. has the same parent and ends where the implementation starts).
                if (i > 0 && (decl as FunctionLikeDeclaration).body) {
                    const previous = symbol.declarations[i - 1];
                    if (decl.parent === previous.parent && decl.kind === previous.kind && decl.pos === previous.end) {
                        continue;
                    }
                }
                result.push(getSignatureFromDeclaration(decl));
            }
            return result;
        }

        function resolveExternalModuleTypeByLiteral(name: StringLiteral) {
            const moduleSym = resolveExternalModuleName(name, name);
            if (moduleSym) {
                const resolvedModuleSymbol = resolveExternalModuleSymbol(moduleSym);
                if (resolvedModuleSymbol) {
                    return getTypeOfSymbol(resolvedModuleSymbol);
                }
            }

            return anyType;
        }

        function getThisTypeOfSignature(signature: Signature): Type | undefined {
            if (signature.thisParameter) {
                return getTypeOfSymbol(signature.thisParameter);
            }
        }

        function getTypePredicateOfSignature(signature: Signature): TypePredicate | undefined {
            if (!signature.resolvedTypePredicate) {
                if (signature.target) {
                    const targetTypePredicate = getTypePredicateOfSignature(signature.target);
                    signature.resolvedTypePredicate = targetTypePredicate ? instantiateTypePredicate(targetTypePredicate, signature.mapper!) : noTypePredicate;
                }
                else if (signature.unionSignatures) {
                    signature.resolvedTypePredicate = getUnionTypePredicate(signature.unionSignatures) || noTypePredicate;
                }
                else {
                    const type = signature.declaration && getEffectiveReturnTypeNode(signature.declaration);
                    let jsdocPredicate: TypePredicate | undefined;
                    if (!type && isInJSFile(signature.declaration)) {
                        const jsdocSignature = getSignatureOfTypeTag(signature.declaration!);
                        if (jsdocSignature && signature !== jsdocSignature) {
                            jsdocPredicate = getTypePredicateOfSignature(jsdocSignature);
                        }
                    }
                    signature.resolvedTypePredicate = type && isTypePredicateNode(type) ?
                        createTypePredicateFromTypePredicateNode(type, signature) :
                        jsdocPredicate || noTypePredicate;
                }
                Debug.assert(!!signature.resolvedTypePredicate);
            }
            return signature.resolvedTypePredicate === noTypePredicate ? undefined : signature.resolvedTypePredicate;
        }

        function createTypePredicateFromTypePredicateNode(node: TypePredicateNode, signature: Signature): TypePredicate {
            const parameterName = node.parameterName;
            const type = node.type && getTypeFromTypeNode(node.type);
            return parameterName.kind === SyntaxKind.ThisType ?
                createTypePredicate(node.assertsModifier ? TypePredicateKind.AssertsThis : TypePredicateKind.This, /*parameterName*/ undefined, /*parameterIndex*/ undefined, type) :
                createTypePredicate(node.assertsModifier ? TypePredicateKind.AssertsIdentifier : TypePredicateKind.Identifier, parameterName.escapedText as string,
                    findIndex(signature.parameters, p => p.escapedName === parameterName.escapedText), type);
        }

        function getReturnTypeOfSignature(signature: Signature): Type {
            if (!signature.resolvedReturnType) {
                if (!pushTypeResolution(signature, TypeSystemPropertyName.ResolvedReturnType)) {
                    return errorType;
                }
                let type = signature.target ? instantiateType(getReturnTypeOfSignature(signature.target), signature.mapper) :
                    signature.unionSignatures ? getUnionType(map(signature.unionSignatures, getReturnTypeOfSignature), UnionReduction.Subtype) :
                    getReturnTypeFromAnnotation(signature.declaration!) ||
                    (nodeIsMissing((<FunctionLikeDeclaration>signature.declaration).body) ? anyType : getReturnTypeFromBody(<FunctionLikeDeclaration>signature.declaration));
                if (signature.flags & SignatureFlags.IsInnerCallChain) {
                    type = addOptionalTypeMarker(type);
                }
                else if (signature.flags & SignatureFlags.IsOuterCallChain) {
                    type = getOptionalType(type);
                }
                if (!popTypeResolution()) {
                    if (signature.declaration) {
                        const typeNode = getEffectiveReturnTypeNode(signature.declaration);
                        if (typeNode) {
                            error(typeNode, Diagnostics.Return_type_annotation_circularly_references_itself);
                        }
                        else if (noImplicitAny) {
                            const declaration = <Declaration>signature.declaration;
                            const name = getNameOfDeclaration(declaration);
                            if (name) {
                                error(name, Diagnostics._0_implicitly_has_return_type_any_because_it_does_not_have_a_return_type_annotation_and_is_referenced_directly_or_indirectly_in_one_of_its_return_expressions, declarationNameToString(name));
                            }
                            else {
                                error(declaration, Diagnostics.Function_implicitly_has_return_type_any_because_it_does_not_have_a_return_type_annotation_and_is_referenced_directly_or_indirectly_in_one_of_its_return_expressions);
                            }
                        }
                    }
                    type = anyType;
                }
                signature.resolvedReturnType = type;
            }
            return signature.resolvedReturnType;
        }

        function getReturnTypeFromAnnotation(declaration: SignatureDeclaration | JSDocSignature) {
            if (declaration.kind === SyntaxKind.Constructor) {
                return getDeclaredTypeOfClassOrInterface(getMergedSymbol((<ClassDeclaration>declaration.parent).symbol));
            }
            if (isJSDocConstructSignature(declaration)) {
                return getTypeFromTypeNode((declaration.parameters[0] as ParameterDeclaration).type!); // TODO: GH#18217
            }
            const typeNode = getEffectiveReturnTypeNode(declaration);
            if (typeNode) {
                return getTypeFromTypeNode(typeNode);
            }
            if (declaration.kind === SyntaxKind.GetAccessor && !hasNonBindableDynamicName(declaration)) {
                const jsDocType = isInJSFile(declaration) && getTypeForDeclarationFromJSDocComment(declaration);
                if (jsDocType) {
                    return jsDocType;
                }
                const setter = getDeclarationOfKind<AccessorDeclaration>(getSymbolOfNode(declaration), SyntaxKind.SetAccessor);
                const setterType = getAnnotatedAccessorType(setter);
                if (setterType) {
                    return setterType;
                }
            }
            return getReturnTypeOfTypeTag(declaration);
        }

        function isResolvingReturnTypeOfSignature(signature: Signature) {
            return !signature.resolvedReturnType && findResolutionCycleStartIndex(signature, TypeSystemPropertyName.ResolvedReturnType) >= 0;
        }

        function getRestTypeOfSignature(signature: Signature): Type {
            return tryGetRestTypeOfSignature(signature) || anyType;
        }

        function tryGetRestTypeOfSignature(signature: Signature): Type | undefined {
            if (signatureHasRestParameter(signature)) {
                const sigRestType = getTypeOfSymbol(signature.parameters[signature.parameters.length - 1]);
                const restType = isTupleType(sigRestType) ? getRestTypeOfTupleType(sigRestType) : sigRestType;
                return restType && getIndexTypeOfType(restType, IndexKind.Number);
            }
            return undefined;
        }

        function getSignatureInstantiation(signature: Signature, typeArguments: Type[] | undefined, isJavascript: boolean, inferredTypeParameters?: readonly TypeParameter[]): Signature {
            const instantiatedSignature = getSignatureInstantiationWithoutFillingInTypeArguments(signature, fillMissingTypeArguments(typeArguments, signature.typeParameters, getMinTypeArgumentCount(signature.typeParameters), isJavascript));
            if (inferredTypeParameters) {
                const returnSignature = getSingleCallOrConstructSignature(getReturnTypeOfSignature(instantiatedSignature));
                if (returnSignature) {
                    const newReturnSignature = cloneSignature(returnSignature);
                    newReturnSignature.typeParameters = inferredTypeParameters;
                    const newInstantiatedSignature = cloneSignature(instantiatedSignature);
                    newInstantiatedSignature.resolvedReturnType = getOrCreateTypeFromSignature(newReturnSignature);
                    return newInstantiatedSignature;
                }
            }
            return instantiatedSignature;
        }

        function getSignatureInstantiationWithoutFillingInTypeArguments(signature: Signature, typeArguments: readonly Type[] | undefined): Signature {
            const instantiations = signature.instantiations || (signature.instantiations = new Map<string, Signature>());
            const id = getTypeListId(typeArguments);
            let instantiation = instantiations.get(id);
            if (!instantiation) {
                instantiations.set(id, instantiation = createSignatureInstantiation(signature, typeArguments));
            }
            return instantiation;
        }

        function createSignatureInstantiation(signature: Signature, typeArguments: readonly Type[] | undefined): Signature {
            return instantiateSignature(signature, createSignatureTypeMapper(signature, typeArguments), /*eraseTypeParameters*/ true);
        }

        function createSignatureTypeMapper(signature: Signature, typeArguments: readonly Type[] | undefined): TypeMapper {
            return createTypeMapper(signature.typeParameters!, typeArguments);
        }

        function getErasedSignature(signature: Signature): Signature {
            return signature.typeParameters ?
                signature.erasedSignatureCache || (signature.erasedSignatureCache = createErasedSignature(signature)) :
                signature;
        }

        function createErasedSignature(signature: Signature) {
            // Create an instantiation of the signature where all type arguments are the any type.
            return instantiateSignature(signature, createTypeEraser(signature.typeParameters!), /*eraseTypeParameters*/ true);
        }

        function getCanonicalSignature(signature: Signature): Signature {
            return signature.typeParameters ?
                signature.canonicalSignatureCache || (signature.canonicalSignatureCache = createCanonicalSignature(signature)) :
                signature;
        }

        function createCanonicalSignature(signature: Signature) {
            // Create an instantiation of the signature where each unconstrained type parameter is replaced with
            // its original. When a generic class or interface is instantiated, each generic method in the class or
            // interface is instantiated with a fresh set of cloned type parameters (which we need to handle scenarios
            // where different generations of the same type parameter are in scope). This leads to a lot of new type
            // identities, and potentially a lot of work comparing those identities, so here we create an instantiation
            // that uses the original type identities for all unconstrained type parameters.
            return getSignatureInstantiation(
                signature,
                map(signature.typeParameters, tp => tp.target && !getConstraintOfTypeParameter(tp.target) ? tp.target : tp),
                isInJSFile(signature.declaration));
        }

        function getBaseSignature(signature: Signature) {
            const typeParameters = signature.typeParameters;
            if (typeParameters) {
                const typeEraser = createTypeEraser(typeParameters);
                const baseConstraints = map(typeParameters, tp => instantiateType(getBaseConstraintOfType(tp), typeEraser) || unknownType);
                return instantiateSignature(signature, createTypeMapper(typeParameters, baseConstraints), /*eraseTypeParameters*/ true);
            }
            return signature;
        }

        function getOrCreateTypeFromSignature(signature: Signature): ObjectType {
            // There are two ways to declare a construct signature, one is by declaring a class constructor
            // using the constructor keyword, and the other is declaring a bare construct signature in an
            // object type literal or interface (using the new keyword). Each way of declaring a constructor
            // will result in a different declaration kind.
            if (!signature.isolatedSignatureType) {
                const kind = signature.declaration ? signature.declaration.kind : SyntaxKind.Unknown;
                const isConstructor = kind === SyntaxKind.Constructor || kind === SyntaxKind.ConstructSignature || kind === SyntaxKind.ConstructorType;
                const type = createObjectType(ObjectFlags.Anonymous);
                type.members = emptySymbols;
                type.properties = emptyArray;
                type.callSignatures = !isConstructor ? [signature] : emptyArray;
                type.constructSignatures = isConstructor ? [signature] : emptyArray;
                signature.isolatedSignatureType = type;
            }

            return signature.isolatedSignatureType;
        }

        function getIndexSymbol(symbol: Symbol): Symbol | undefined {
            return symbol.members!.get(InternalSymbolName.Index);
        }

        function getIndexDeclarationOfSymbol(symbol: Symbol, kind: IndexKind): IndexSignatureDeclaration | undefined {
            const syntaxKind = kind === IndexKind.Number ? SyntaxKind.NumberKeyword : SyntaxKind.StringKeyword;
            const indexSymbol = getIndexSymbol(symbol);
            if (indexSymbol) {
                for (const decl of indexSymbol.declarations) {
                    const node = cast(decl, isIndexSignatureDeclaration);
                    if (node.parameters.length === 1) {
                        const parameter = node.parameters[0];
                        if (parameter.type && parameter.type.kind === syntaxKind) {
                            return node;
                        }
                    }
                }
            }

            return undefined;
        }

        function createIndexInfo(type: Type, isReadonly: boolean, declaration?: IndexSignatureDeclaration): IndexInfo {
            return { type, isReadonly, declaration };
        }

        function getIndexInfoOfSymbol(symbol: Symbol, kind: IndexKind): IndexInfo | undefined {
            const declaration = getIndexDeclarationOfSymbol(symbol, kind);
            if (declaration) {
                return createIndexInfo(declaration.type ? getTypeFromTypeNode(declaration.type) : anyType,
                    hasEffectiveModifier(declaration, ModifierFlags.Readonly), declaration);
            }
            return undefined;
        }

        function getConstraintDeclaration(type: TypeParameter): TypeNode | undefined {
            return mapDefined(filter(type.symbol && type.symbol.declarations, isTypeParameterDeclaration), getEffectiveConstraintOfTypeParameter)[0];
        }

        function getInferredTypeParameterConstraint(typeParameter: TypeParameter) {
            let inferences: Type[] | undefined;
            if (typeParameter.symbol) {
                for (const declaration of typeParameter.symbol.declarations) {
                    if (declaration.parent.kind === SyntaxKind.InferType) {
                        // When an 'infer T' declaration is immediately contained in a type reference node
                        // (such as 'Foo<infer T>'), T's constraint is inferred from the constraint of the
                        // corresponding type parameter in 'Foo'. When multiple 'infer T' declarations are
                        // present, we form an intersection of the inferred constraint types.
                        const grandParent = declaration.parent.parent;
                        if (grandParent.kind === SyntaxKind.TypeReference) {
                            const typeReference = <TypeReferenceNode>grandParent;
                            const typeParameters = getTypeParametersForTypeReference(typeReference);
                            if (typeParameters) {
                                const index = typeReference.typeArguments!.indexOf(<TypeNode>declaration.parent);
                                if (index < typeParameters.length) {
                                    const declaredConstraint = getConstraintOfTypeParameter(typeParameters[index]);
                                    if (declaredConstraint) {
                                        // Type parameter constraints can reference other type parameters so
                                        // constraints need to be instantiated. If instantiation produces the
                                        // type parameter itself, we discard that inference. For example, in
                                        //   type Foo<T extends string, U extends T> = [T, U];
                                        //   type Bar<T> = T extends Foo<infer X, infer X> ? Foo<X, X> : T;
                                        // the instantiated constraint for U is X, so we discard that inference.
                                        const mapper = createTypeMapper(typeParameters, getEffectiveTypeArguments(typeReference, typeParameters));
                                        const constraint = instantiateType(declaredConstraint, mapper);
                                        if (constraint !== typeParameter) {
                                            inferences = append(inferences, constraint);
                                        }
                                    }
                                }
                            }
                        }
                        // When an 'infer T' declaration is immediately contained in a rest parameter declaration, a rest type
                        // or a named rest tuple element, we infer an 'unknown[]' constraint.
                        else if (grandParent.kind === SyntaxKind.Parameter && (<ParameterDeclaration>grandParent).dotDotDotToken ||
                            grandParent.kind === SyntaxKind.RestType ||
                            grandParent.kind === SyntaxKind.NamedTupleMember && (<NamedTupleMember>grandParent).dotDotDotToken) {
                            inferences = append(inferences, createArrayType(unknownType));
                        }
                    }
                }
            }
            return inferences && getIntersectionType(inferences);
        }

        /** This is a worker function. Use getConstraintOfTypeParameter which guards against circular constraints. */
        function getConstraintFromTypeParameter(typeParameter: TypeParameter): Type | undefined {
            if (!typeParameter.constraint) {
                if (typeParameter.target) {
                    const targetConstraint = getConstraintOfTypeParameter(typeParameter.target);
                    typeParameter.constraint = targetConstraint ? instantiateType(targetConstraint, typeParameter.mapper) : noConstraintType;
                }
                else {
                    const constraintDeclaration = getConstraintDeclaration(typeParameter);
                    if (!constraintDeclaration) {
                        typeParameter.constraint = getInferredTypeParameterConstraint(typeParameter) || noConstraintType;
                    }
                    else {
                        let type = getTypeFromTypeNode(constraintDeclaration);
                        if (type.flags & TypeFlags.Any && type !== errorType) { // Allow errorType to propegate to keep downstream errors suppressed
                            // use keyofConstraintType as the base constraint for mapped type key constraints (unknown isn;t assignable to that, but `any` was),
                            // use unknown otherwise
                            type = constraintDeclaration.parent.parent.kind === SyntaxKind.MappedType ? keyofConstraintType : unknownType;
                        }
                        typeParameter.constraint = type;
                    }
                }
            }
            return typeParameter.constraint === noConstraintType ? undefined : typeParameter.constraint;
        }

        function getParentSymbolOfTypeParameter(typeParameter: TypeParameter): Symbol | undefined {
            const tp = getDeclarationOfKind<TypeParameterDeclaration>(typeParameter.symbol, SyntaxKind.TypeParameter)!;
            const host = isJSDocTemplateTag(tp.parent) ? getHostSignatureFromJSDoc(tp.parent) : tp.parent;
            return host && getSymbolOfNode(host);
        }

        function getTypeListId(types: readonly Type[] | undefined) {
            let result = "";
            if (types) {
                const length = types.length;
                let i = 0;
                while (i < length) {
                    const startId = types[i].id;
                    let count = 1;
                    while (i + count < length && types[i + count].id === startId + count) {
                        count++;
                    }
                    if (result.length) {
                        result += ",";
                    }
                    result += startId;
                    if (count > 1) {
                        result += ":" + count;
                    }
                    i += count;
                }
            }
            return result;
        }

        // This function is used to propagate certain flags when creating new object type references and union types.
        // It is only necessary to do so if a constituent type might be the undefined type, the null type, the type
        // of an object literal or the anyFunctionType. This is because there are operations in the type checker
        // that care about the presence of such types at arbitrary depth in a containing type.
        function getPropagatingFlagsOfTypes(types: readonly Type[], excludeKinds: TypeFlags): ObjectFlags {
            let result: ObjectFlags = 0;
            for (const type of types) {
                if (!(type.flags & excludeKinds)) {
                    result |= getObjectFlags(type);
                }
            }
            return result & ObjectFlags.PropagatingFlags;
        }

        function createTypeReference(target: GenericType, typeArguments: readonly Type[] | undefined): TypeReference {
            const id = getTypeListId(typeArguments);
            let type = target.instantiations.get(id);
            if (!type) {
                type = <TypeReference>createObjectType(ObjectFlags.Reference, target.symbol);
                target.instantiations.set(id, type);
                type.objectFlags |= typeArguments ? getPropagatingFlagsOfTypes(typeArguments, /*excludeKinds*/ 0) : 0;
                type.target = target;
                type.resolvedTypeArguments = typeArguments;
            }
            return type;
        }

        function cloneTypeReference(source: TypeReference): TypeReference {
            const type = <TypeReference>createType(source.flags);
            type.symbol = source.symbol;
            type.objectFlags = source.objectFlags;
            type.target = source.target;
            type.resolvedTypeArguments = source.resolvedTypeArguments;
            return type;
        }

        function createDeferredTypeReference(target: GenericType, node: TypeReferenceNode | ArrayTypeNode | TupleTypeNode, mapper?: TypeMapper): DeferredTypeReference {
            const aliasSymbol = getAliasSymbolForTypeNode(node);
            const aliasTypeArguments = getTypeArgumentsForAliasSymbol(aliasSymbol);
            const type = <DeferredTypeReference>createObjectType(ObjectFlags.Reference, target.symbol);
            type.target = target;
            type.node = node;
            type.mapper = mapper;
            type.aliasSymbol = aliasSymbol;
            type.aliasTypeArguments = mapper ? instantiateTypes(aliasTypeArguments, mapper) : aliasTypeArguments;
            return type;
        }

        function getTypeArguments(type: TypeReference): readonly Type[] {
            if (!type.resolvedTypeArguments) {
                if (!pushTypeResolution(type, TypeSystemPropertyName.ResolvedTypeArguments)) {
                    return type.target.localTypeParameters?.map(() => errorType) || emptyArray;
                }
                const node = type.node;
                const typeArguments = !node ? emptyArray :
                    node.kind === SyntaxKind.TypeReference ? concatenate(type.target.outerTypeParameters, getEffectiveTypeArguments(node, type.target.localTypeParameters!)) :
                    node.kind === SyntaxKind.ArrayType ? [getTypeFromTypeNode(node.elementType)] :
                    map(node.elements, getTypeFromTypeNode);
                if (popTypeResolution()) {
                    type.resolvedTypeArguments = type.mapper ? instantiateTypes(typeArguments, type.mapper) : typeArguments;
                }
                else {
                    type.resolvedTypeArguments = type.target.localTypeParameters?.map(() => errorType) || emptyArray;
                    error(
                        type.node || currentNode,
                        type.target.symbol ? Diagnostics.Type_arguments_for_0_circularly_reference_themselves : Diagnostics.Tuple_type_arguments_circularly_reference_themselves,
                        type.target.symbol && symbolToString(type.target.symbol)
                    );
                }
            }
            return type.resolvedTypeArguments;
        }

        function getTypeReferenceArity(type: TypeReference): number {
            return length(type.target.typeParameters);
        }


        /**
         * Get type from type-reference that reference to class or interface
         */
        function getTypeFromClassOrInterfaceReference(node: NodeWithTypeArguments, symbol: Symbol): Type {
            const type = <InterfaceType>getDeclaredTypeOfSymbol(getMergedSymbol(symbol));
            const typeParameters = type.localTypeParameters;
            if (typeParameters) {
                const numTypeArguments = length(node.typeArguments);
                const minTypeArgumentCount = getMinTypeArgumentCount(typeParameters);
                const isJs = isInJSFile(node);
                const isJsImplicitAny = !noImplicitAny && isJs;
                if (!isJsImplicitAny && (numTypeArguments < minTypeArgumentCount || numTypeArguments > typeParameters.length)) {
                    const missingAugmentsTag = isJs && isExpressionWithTypeArguments(node) && !isJSDocAugmentsTag(node.parent);
                    const diag = minTypeArgumentCount === typeParameters.length ?
                        missingAugmentsTag ?
                            Diagnostics.Expected_0_type_arguments_provide_these_with_an_extends_tag :
                            Diagnostics.Generic_type_0_requires_1_type_argument_s :
                        missingAugmentsTag ?
                            Diagnostics.Expected_0_1_type_arguments_provide_these_with_an_extends_tag :
                            Diagnostics.Generic_type_0_requires_between_1_and_2_type_arguments;

                    const typeStr = typeToString(type, /*enclosingDeclaration*/ undefined, TypeFormatFlags.WriteArrayAsGenericType);
                    error(node, diag, typeStr, minTypeArgumentCount, typeParameters.length);
                    if (!isJs) {
                        // TODO: Adopt same permissive behavior in TS as in JS to reduce follow-on editing experience failures (requires editing fillMissingTypeArguments)
                        return errorType;
                    }
                }
                if (node.kind === SyntaxKind.TypeReference && isDeferredTypeReferenceNode(<TypeReferenceNode>node, length(node.typeArguments) !== typeParameters.length)) {
                    return createDeferredTypeReference(<GenericType>type, <TypeReferenceNode>node, /*mapper*/ undefined);
                }
                // In a type reference, the outer type parameters of the referenced class or interface are automatically
                // supplied as type arguments and the type reference only specifies arguments for the local type parameters
                // of the class or interface.
                const typeArguments = concatenate(type.outerTypeParameters, fillMissingTypeArguments(typeArgumentsFromTypeReferenceNode(node), typeParameters, minTypeArgumentCount, isJs));
                return createTypeReference(<GenericType>type, typeArguments);
            }
            return checkNoTypeArguments(node, symbol) ? type : errorType;
        }

        function getTypeAliasInstantiation(symbol: Symbol, typeArguments: readonly Type[] | undefined): Type {
            const type = getDeclaredTypeOfSymbol(symbol);
            const links = getSymbolLinks(symbol);
            const typeParameters = links.typeParameters!;
            const id = getTypeListId(typeArguments);
            let instantiation = links.instantiations!.get(id);
            if (!instantiation) {
                links.instantiations!.set(id, instantiation = instantiateType(type, createTypeMapper(typeParameters, fillMissingTypeArguments(typeArguments, typeParameters, getMinTypeArgumentCount(typeParameters), isInJSFile(symbol.valueDeclaration)))));
            }
            return instantiation;
        }

        /**
         * Get type from reference to type alias. When a type alias is generic, the declared type of the type alias may include
         * references to the type parameters of the alias. We replace those with the actual type arguments by instantiating the
         * declared type. Instantiations are cached using the type identities of the type arguments as the key.
         */
        function getTypeFromTypeAliasReference(node: NodeWithTypeArguments, symbol: Symbol): Type {
            const type = getDeclaredTypeOfSymbol(symbol);
            const typeParameters = getSymbolLinks(symbol).typeParameters;
            if (typeParameters) {
                const numTypeArguments = length(node.typeArguments);
                const minTypeArgumentCount = getMinTypeArgumentCount(typeParameters);
                if (numTypeArguments < minTypeArgumentCount || numTypeArguments > typeParameters.length) {
                    error(node,
                        minTypeArgumentCount === typeParameters.length ?
                            Diagnostics.Generic_type_0_requires_1_type_argument_s :
                            Diagnostics.Generic_type_0_requires_between_1_and_2_type_arguments,
                        symbolToString(symbol),
                        minTypeArgumentCount,
                        typeParameters.length);
                    return errorType;
                }
                return getTypeAliasInstantiation(symbol, typeArgumentsFromTypeReferenceNode(node));
            }
            return checkNoTypeArguments(node, symbol) ? type : errorType;
        }

        function getTypeReferenceName(node: TypeReferenceType): EntityNameOrEntityNameExpression | undefined {
            switch (node.kind) {
                case SyntaxKind.TypeReference:
                    return node.typeName;
                case SyntaxKind.ExpressionWithTypeArguments:
                    // We only support expressions that are simple qualified names. For other
                    // expressions this produces undefined.
                    const expr = node.expression;
                    if (isEntityNameExpression(expr)) {
                        return expr;
                    }
                // fall through;
            }

            return undefined;
        }

        function resolveTypeReferenceName(typeReferenceName: EntityNameExpression | EntityName | undefined, meaning: SymbolFlags, ignoreErrors?: boolean) {
            if (!typeReferenceName) {
                return unknownSymbol;
            }

            return resolveEntityName(typeReferenceName, meaning, ignoreErrors) || unknownSymbol;
        }

        function getTypeReferenceType(node: NodeWithTypeArguments, symbol: Symbol): Type {
            if (symbol === unknownSymbol) {
                return errorType;
            }
            symbol = getExpandoSymbol(symbol) || symbol;
            if (symbol.flags & (SymbolFlags.Class | SymbolFlags.Interface)) {
                return getTypeFromClassOrInterfaceReference(node, symbol);
            }
            if (symbol.flags & SymbolFlags.TypeAlias) {
                return getTypeFromTypeAliasReference(node, symbol);
            }
            // Get type from reference to named type that cannot be generic (enum or type parameter)
            const res = tryGetDeclaredTypeOfSymbol(symbol);
            if (res) {
                return checkNoTypeArguments(node, symbol) ? getRegularTypeOfLiteralType(res) : errorType;
            }
            if (symbol.flags & SymbolFlags.Value && isJSDocTypeReference(node)) {
                const jsdocType = getTypeFromJSDocValueReference(node, symbol);
                if (jsdocType) {
                    return jsdocType;
                }
                else {
                    // Resolve the type reference as a Type for the purpose of reporting errors.
                    resolveTypeReferenceName(getTypeReferenceName(node), SymbolFlags.Type);
                    return getTypeOfSymbol(symbol);
                }
            }
            return errorType;
        }

        /**
         * A JSdoc TypeReference may be to a value, but resolve it as a type anyway.
         * Note: If the value is imported from commonjs, it should really be an alias,
         * but this function's special-case code fakes alias resolution as well.
         */
        function getTypeFromJSDocValueReference(node: NodeWithTypeArguments, symbol: Symbol): Type | undefined {
            const links = getNodeLinks(node);
            if (!links.resolvedJSDocType) {
                const valueType = getTypeOfSymbol(symbol);
                let typeType = valueType;
                if (symbol.valueDeclaration) {
                    const decl = getRootDeclaration(symbol.valueDeclaration);
                    let isRequireAlias = false;
                    if (isVariableDeclaration(decl) && decl.initializer) {
                        let expr = decl.initializer;
                        // skip past entity names, eg `require("x").a.b.c`
                        while (isPropertyAccessExpression(expr)) {
                            expr = expr.expression;
                        }
                        isRequireAlias = isCallExpression(expr) && isRequireCall(expr, /*requireStringLiteralLikeArgument*/ true) && !!valueType.symbol;
                    }
                    const isImportTypeWithQualifier = node.kind === SyntaxKind.ImportType && (node as ImportTypeNode).qualifier;
                    // valueType might not have a symbol, eg, {import('./b').STRING_LITERAL}
                    if (valueType.symbol && (isRequireAlias || isImportTypeWithQualifier)) {
                        typeType = getTypeReferenceType(node, valueType.symbol);
                    }
                }
                links.resolvedJSDocType = typeType;
            }
            return links.resolvedJSDocType;
        }

        function getSubstitutionType(baseType: Type, substitute: Type) {
            if (substitute.flags & TypeFlags.AnyOrUnknown || substitute === baseType) {
                return baseType;
            }
            const id = `${getTypeId(baseType)}>${getTypeId(substitute)}`;
            const cached = substitutionTypes.get(id);
            if (cached) {
                return cached;
            }
            const result = <SubstitutionType>createType(TypeFlags.Substitution);
            result.baseType = baseType;
            result.substitute = substitute;
            substitutionTypes.set(id, result);
            return result;
        }

        function isUnaryTupleTypeNode(node: TypeNode) {
            return node.kind === SyntaxKind.TupleType && (<TupleTypeNode>node).elements.length === 1;
        }

        function getImpliedConstraint(type: Type, checkNode: TypeNode, extendsNode: TypeNode): Type | undefined {
            return isUnaryTupleTypeNode(checkNode) && isUnaryTupleTypeNode(extendsNode) ? getImpliedConstraint(type, (<TupleTypeNode>checkNode).elements[0], (<TupleTypeNode>extendsNode).elements[0]) :
                getActualTypeVariable(getTypeFromTypeNode(checkNode)) === type ? getTypeFromTypeNode(extendsNode) :
                undefined;
        }

        function getConditionalFlowTypeOfType(type: Type, node: Node) {
            let constraints: Type[] | undefined;
            while (node && !isStatement(node) && node.kind !== SyntaxKind.JSDocComment) {
                const parent = node.parent;
                if (parent.kind === SyntaxKind.ConditionalType && node === (<ConditionalTypeNode>parent).trueType) {
                    const constraint = getImpliedConstraint(type, (<ConditionalTypeNode>parent).checkType, (<ConditionalTypeNode>parent).extendsType);
                    if (constraint) {
                        constraints = append(constraints, constraint);
                    }
                }
                node = parent;
            }
            return constraints ? getSubstitutionType(type, getIntersectionType(append(constraints, type))) : type;
        }

        function isJSDocTypeReference(node: Node): node is TypeReferenceNode {
            return !!(node.flags & NodeFlags.JSDoc) && (node.kind === SyntaxKind.TypeReference || node.kind === SyntaxKind.ImportType);
        }

        function checkNoTypeArguments(node: NodeWithTypeArguments, symbol?: Symbol) {
            if (node.typeArguments) {
                error(node, Diagnostics.Type_0_is_not_generic, symbol ? symbolToString(symbol) : (<TypeReferenceNode>node).typeName ? declarationNameToString((<TypeReferenceNode>node).typeName) : anon);
                return false;
            }
            return true;
        }

        function getIntendedTypeFromJSDocTypeReference(node: TypeReferenceNode): Type | undefined {
            if (isIdentifier(node.typeName)) {
                const typeArgs = node.typeArguments;
                switch (node.typeName.escapedText) {
                    case "String":
                        checkNoTypeArguments(node);
                        return stringType;
                    case "Number":
                        checkNoTypeArguments(node);
                        return numberType;
                    case "Boolean":
                        checkNoTypeArguments(node);
                        return booleanType;
                    case "Void":
                        checkNoTypeArguments(node);
                        return voidType;
                    case "Undefined":
                        checkNoTypeArguments(node);
                        return undefinedType;
                    case "Null":
                        checkNoTypeArguments(node);
                        return nullType;
                    case "Function":
                    case "function":
                        checkNoTypeArguments(node);
                        return globalFunctionType;
                    case "array":
                        return (!typeArgs || !typeArgs.length) && !noImplicitAny ? anyArrayType : undefined;
                    case "promise":
                        return (!typeArgs || !typeArgs.length) && !noImplicitAny ? createPromiseType(anyType) : undefined;
                    case "Object":
                        if (typeArgs && typeArgs.length === 2) {
                            if (isJSDocIndexSignature(node)) {
                                const indexed = getTypeFromTypeNode(typeArgs[0]);
                                const target = getTypeFromTypeNode(typeArgs[1]);
                                const index = createIndexInfo(target, /*isReadonly*/ false);
                                return createAnonymousType(undefined, emptySymbols, emptyArray, emptyArray, indexed === stringType ? index : undefined, indexed === numberType ? index : undefined);
                            }
                            return anyType;
                        }
                        checkNoTypeArguments(node);
                        return !noImplicitAny ? anyType : undefined;
                }
            }
        }

        function getTypeFromJSDocNullableTypeNode(node: JSDocNullableType) {
            const type = getTypeFromTypeNode(node.type);
            return strictNullChecks ? getNullableType(type, TypeFlags.Null) : type;
        }

        function getTypeFromTypeReference(node: TypeReferenceType): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                // handle LS queries on the `const` in `x as const` by resolving to the type of `x`
                if (isConstTypeReference(node) && isAssertionExpression(node.parent)) {
                    links.resolvedSymbol = unknownSymbol;
                    return links.resolvedType = checkExpressionCached(node.parent.expression);
                }
                let symbol: Symbol | undefined;
                let type: Type | undefined;
                const meaning = SymbolFlags.Type;
                if (isJSDocTypeReference(node)) {
                    type = getIntendedTypeFromJSDocTypeReference(node);
                    if (!type) {
                        symbol = resolveTypeReferenceName(getTypeReferenceName(node), meaning, /*ignoreErrors*/ true);
                        if (symbol === unknownSymbol) {
                            symbol = resolveTypeReferenceName(getTypeReferenceName(node), meaning | SymbolFlags.Value);
                        }
                        else {
                            resolveTypeReferenceName(getTypeReferenceName(node), meaning); // Resolve again to mark errors, if any
                        }
                        type = getTypeReferenceType(node, symbol);
                    }
                }
                if (!type) {
                    symbol = resolveTypeReferenceName(getTypeReferenceName(node), meaning);
                    type = getTypeReferenceType(node, symbol);
                }
                // Cache both the resolved symbol and the resolved type. The resolved symbol is needed when we check the
                // type reference in checkTypeReferenceNode.
                links.resolvedSymbol = symbol;
                links.resolvedType = type;
            }
            return links.resolvedType;
        }

        function typeArgumentsFromTypeReferenceNode(node: NodeWithTypeArguments): Type[] | undefined {
            return map(node.typeArguments, getTypeFromTypeNode);
        }

        function getTypeFromTypeQueryNode(node: TypeQueryNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                // TypeScript 1.0 spec (April 2014): 3.6.3
                // The expression is processed as an identifier expression (section 4.3)
                // or property access expression(section 4.10),
                // the widened type(section 3.9) of which becomes the result.
                links.resolvedType = getRegularTypeOfLiteralType(getWidenedType(checkExpression(node.exprName)));
            }
            return links.resolvedType;
        }

        function getTypeOfGlobalSymbol(symbol: Symbol | undefined, arity: number): ObjectType {

            function getTypeDeclaration(symbol: Symbol): Declaration | undefined {
                const declarations = symbol.declarations;
                for (const declaration of declarations) {
                    switch (declaration.kind) {
                        case SyntaxKind.ClassDeclaration:
                        case SyntaxKind.InterfaceDeclaration:
                        case SyntaxKind.EnumDeclaration:
                            return declaration;
                    }
                }
            }

            if (!symbol) {
                return arity ? emptyGenericType : emptyObjectType;
            }
            const type = getDeclaredTypeOfSymbol(symbol);
            if (!(type.flags & TypeFlags.Object)) {
                error(getTypeDeclaration(symbol), Diagnostics.Global_type_0_must_be_a_class_or_interface_type, symbolName(symbol));
                return arity ? emptyGenericType : emptyObjectType;
            }
            if (length((<InterfaceType>type).typeParameters) !== arity) {
                error(getTypeDeclaration(symbol), Diagnostics.Global_type_0_must_have_1_type_parameter_s, symbolName(symbol), arity);
                return arity ? emptyGenericType : emptyObjectType;
            }
            return <ObjectType>type;
        }

        function getGlobalValueSymbol(name: __String, reportErrors: boolean): Symbol | undefined {
            return getGlobalSymbol(name, SymbolFlags.Value, reportErrors ? Diagnostics.Cannot_find_global_value_0 : undefined);
        }

        function getGlobalTypeSymbol(name: __String, reportErrors: boolean): Symbol | undefined {
            return getGlobalSymbol(name, SymbolFlags.Type, reportErrors ? Diagnostics.Cannot_find_global_type_0 : undefined);
        }

        function getGlobalSymbol(name: __String, meaning: SymbolFlags, diagnostic: DiagnosticMessage | undefined): Symbol | undefined {
            // Don't track references for global symbols anyway, so value if `isReference` is arbitrary
            return resolveName(undefined, name, meaning, diagnostic, name, /*isUse*/ false);
        }

        function getGlobalType(name: __String, arity: 0, reportErrors: boolean): ObjectType;
        function getGlobalType(name: __String, arity: number, reportErrors: boolean): GenericType;
        function getGlobalType(name: __String, arity: number, reportErrors: boolean): ObjectType | undefined {
            const symbol = getGlobalTypeSymbol(name, reportErrors);
            return symbol || reportErrors ? getTypeOfGlobalSymbol(symbol, arity) : undefined;
        }

        function getGlobalTypedPropertyDescriptorType() {
            return deferredGlobalTypedPropertyDescriptorType || (deferredGlobalTypedPropertyDescriptorType = getGlobalType("TypedPropertyDescriptor" as __String, /*arity*/ 1, /*reportErrors*/ true)) || emptyGenericType;
        }

        function getGlobalTemplateStringsArrayType() {
            return deferredGlobalTemplateStringsArrayType || (deferredGlobalTemplateStringsArrayType = getGlobalType("TemplateStringsArray" as __String, /*arity*/ 0, /*reportErrors*/ true)) || emptyObjectType;
        }

        function getGlobalImportMetaType() {
            return deferredGlobalImportMetaType || (deferredGlobalImportMetaType = getGlobalType("ImportMeta" as __String, /*arity*/ 0, /*reportErrors*/ true)) || emptyObjectType;
        }

        function getGlobalESSymbolConstructorSymbol(reportErrors: boolean) {
            return deferredGlobalESSymbolConstructorSymbol || (deferredGlobalESSymbolConstructorSymbol = getGlobalValueSymbol("Symbol" as __String, reportErrors));
        }

        function getGlobalESSymbolType(reportErrors: boolean) {
            return deferredGlobalESSymbolType || (deferredGlobalESSymbolType = getGlobalType("Symbol" as __String, /*arity*/ 0, reportErrors)) || emptyObjectType;
        }

        function getGlobalPromiseType(reportErrors: boolean) {
            return deferredGlobalPromiseType || (deferredGlobalPromiseType = getGlobalType("Promise" as __String, /*arity*/ 1, reportErrors)) || emptyGenericType;
        }

        function getGlobalPromiseLikeType(reportErrors: boolean) {
            return deferredGlobalPromiseLikeType || (deferredGlobalPromiseLikeType = getGlobalType("PromiseLike" as __String, /*arity*/ 1, reportErrors)) || emptyGenericType;
        }

        function getGlobalPromiseConstructorSymbol(reportErrors: boolean): Symbol | undefined {
            return deferredGlobalPromiseConstructorSymbol || (deferredGlobalPromiseConstructorSymbol = getGlobalValueSymbol("Promise" as __String, reportErrors));
        }

        function getGlobalPromiseConstructorLikeType(reportErrors: boolean) {
            return deferredGlobalPromiseConstructorLikeType || (deferredGlobalPromiseConstructorLikeType = getGlobalType("PromiseConstructorLike" as __String, /*arity*/ 0, reportErrors)) || emptyObjectType;
        }

        function getGlobalAsyncIterableType(reportErrors: boolean) {
            return deferredGlobalAsyncIterableType || (deferredGlobalAsyncIterableType = getGlobalType("AsyncIterable" as __String, /*arity*/ 1, reportErrors)) || emptyGenericType;
        }

        function getGlobalAsyncIteratorType(reportErrors: boolean) {
            return deferredGlobalAsyncIteratorType || (deferredGlobalAsyncIteratorType = getGlobalType("AsyncIterator" as __String, /*arity*/ 3, reportErrors)) || emptyGenericType;
        }

        function getGlobalAsyncIterableIteratorType(reportErrors: boolean) {
            return deferredGlobalAsyncIterableIteratorType || (deferredGlobalAsyncIterableIteratorType = getGlobalType("AsyncIterableIterator" as __String, /*arity*/ 1, reportErrors)) || emptyGenericType;
        }

        function getGlobalAsyncGeneratorType(reportErrors: boolean) {
            return deferredGlobalAsyncGeneratorType || (deferredGlobalAsyncGeneratorType = getGlobalType("AsyncGenerator" as __String, /*arity*/ 3, reportErrors)) || emptyGenericType;
        }

        function getGlobalIterableType(reportErrors: boolean) {
            return deferredGlobalIterableType || (deferredGlobalIterableType = getGlobalType("Iterable" as __String, /*arity*/ 1, reportErrors)) || emptyGenericType;
        }

        function getGlobalIteratorType(reportErrors: boolean) {
            return deferredGlobalIteratorType || (deferredGlobalIteratorType = getGlobalType("Iterator" as __String, /*arity*/ 3, reportErrors)) || emptyGenericType;
        }

        function getGlobalIterableIteratorType(reportErrors: boolean) {
            return deferredGlobalIterableIteratorType || (deferredGlobalIterableIteratorType = getGlobalType("IterableIterator" as __String, /*arity*/ 1, reportErrors)) || emptyGenericType;
        }

        function getGlobalGeneratorType(reportErrors: boolean) {
            return deferredGlobalGeneratorType || (deferredGlobalGeneratorType = getGlobalType("Generator" as __String, /*arity*/ 3, reportErrors)) || emptyGenericType;
        }

        function getGlobalIteratorYieldResultType(reportErrors: boolean) {
            return deferredGlobalIteratorYieldResultType || (deferredGlobalIteratorYieldResultType = getGlobalType("IteratorYieldResult" as __String, /*arity*/ 1, reportErrors)) || emptyGenericType;
        }

        function getGlobalIteratorReturnResultType(reportErrors: boolean) {
            return deferredGlobalIteratorReturnResultType || (deferredGlobalIteratorReturnResultType = getGlobalType("IteratorReturnResult" as __String, /*arity*/ 1, reportErrors)) || emptyGenericType;
        }

        function getGlobalTypeOrUndefined(name: __String, arity = 0): ObjectType | undefined {
            const symbol = getGlobalSymbol(name, SymbolFlags.Type, /*diagnostic*/ undefined);
            return symbol && <GenericType>getTypeOfGlobalSymbol(symbol, arity);
        }

        function getGlobalExtractSymbol(): Symbol {
            return deferredGlobalExtractSymbol || (deferredGlobalExtractSymbol = getGlobalSymbol("Extract" as __String, SymbolFlags.TypeAlias, Diagnostics.Cannot_find_global_type_0)!); // TODO: GH#18217
        }

        function getGlobalOmitSymbol(): Symbol {
            return deferredGlobalOmitSymbol || (deferredGlobalOmitSymbol = getGlobalSymbol("Omit" as __String, SymbolFlags.TypeAlias, Diagnostics.Cannot_find_global_type_0)!); // TODO: GH#18217
        }

        function getGlobalBigIntType(reportErrors: boolean) {
            return deferredGlobalBigIntType || (deferredGlobalBigIntType = getGlobalType("BigInt" as __String, /*arity*/ 0, reportErrors)) || emptyObjectType;
        }

        /**
         * Instantiates a global type that is generic with some element type, and returns that instantiation.
         */
        function createTypeFromGenericGlobalType(genericGlobalType: GenericType, typeArguments: readonly Type[]): ObjectType {
            return genericGlobalType !== emptyGenericType ? createTypeReference(genericGlobalType, typeArguments) : emptyObjectType;
        }

        function createTypedPropertyDescriptorType(propertyType: Type): Type {
            return createTypeFromGenericGlobalType(getGlobalTypedPropertyDescriptorType(), [propertyType]);
        }

        function createIterableType(iteratedType: Type): Type {
            return createTypeFromGenericGlobalType(getGlobalIterableType(/*reportErrors*/ true), [iteratedType]);
        }

        function createArrayType(elementType: Type, readonly?: boolean): ObjectType {
            return createTypeFromGenericGlobalType(readonly ? globalReadonlyArrayType : globalArrayType, [elementType]);
        }

        function getTupleElementFlags(node: TypeNode) {
            switch (node.kind) {
                case SyntaxKind.OptionalType:
                    return ElementFlags.Optional;
                case SyntaxKind.RestType:
                    return getRestTypeElementFlags(node as RestTypeNode);
                case SyntaxKind.NamedTupleMember:
                    return (node as NamedTupleMember).questionToken ? ElementFlags.Optional :
                        (node as NamedTupleMember).dotDotDotToken ? getRestTypeElementFlags(node as NamedTupleMember) :
                        ElementFlags.Required;
                default:
                    return ElementFlags.Required;
            }
        }

        function getRestTypeElementFlags(node: RestTypeNode | NamedTupleMember) {
            return getArrayElementTypeNode(node.type) ? ElementFlags.Rest : ElementFlags.Variadic;
        }

        function getArrayOrTupleTargetType(node: ArrayTypeNode | TupleTypeNode): GenericType {
            const readonly = isReadonlyTypeOperator(node.parent);
            const elementType = getArrayElementTypeNode(node);
            if (elementType) {
                return readonly ? globalReadonlyArrayType : globalArrayType;
            }
            const elementFlags = map((node as TupleTypeNode).elements, getTupleElementFlags);
            const missingName = some((node as TupleTypeNode).elements, e => e.kind !== SyntaxKind.NamedTupleMember);
            return getTupleTargetType(elementFlags, readonly, /*associatedNames*/ missingName ? undefined : (node as TupleTypeNode).elements as readonly NamedTupleMember[]);
        }

        // Return true if the given type reference node is directly aliased or if it needs to be deferred
        // because it is possibly contained in a circular chain of eagerly resolved types.
        function isDeferredTypeReferenceNode(node: TypeReferenceNode | ArrayTypeNode | TupleTypeNode, hasDefaultTypeArguments?: boolean) {
            return !!getAliasSymbolForTypeNode(node) || isResolvedByTypeAlias(node) && (
                node.kind === SyntaxKind.ArrayType ? mayResolveTypeAlias(node.elementType) :
                node.kind === SyntaxKind.TupleType ? some(node.elements, mayResolveTypeAlias) :
                hasDefaultTypeArguments || some(node.typeArguments, mayResolveTypeAlias));
        }

        // Return true when the given node is transitively contained in type constructs that eagerly
        // resolve their constituent types. We include SyntaxKind.TypeReference because type arguments
        // of type aliases are eagerly resolved.
        function isResolvedByTypeAlias(node: Node): boolean {
            const parent = node.parent;
            switch (parent.kind) {
                case SyntaxKind.ParenthesizedType:
                case SyntaxKind.NamedTupleMember:
                case SyntaxKind.TypeReference:
                case SyntaxKind.UnionType:
                case SyntaxKind.IntersectionType:
                case SyntaxKind.IndexedAccessType:
                case SyntaxKind.ConditionalType:
                case SyntaxKind.TypeOperator:
                case SyntaxKind.ArrayType:
                case SyntaxKind.TupleType:
                    return isResolvedByTypeAlias(parent);
                case SyntaxKind.TypeAliasDeclaration:
                    return true;
            }
            return false;
        }

        // Return true if resolving the given node (i.e. getTypeFromTypeNode) possibly causes resolution
        // of a type alias.
        function mayResolveTypeAlias(node: Node): boolean {
            switch (node.kind) {
                case SyntaxKind.TypeReference:
                    return isJSDocTypeReference(node) || !!(resolveTypeReferenceName((<TypeReferenceNode>node).typeName, SymbolFlags.Type).flags & SymbolFlags.TypeAlias);
                case SyntaxKind.TypeQuery:
                    return true;
                case SyntaxKind.TypeOperator:
                    return (<TypeOperatorNode>node).operator !== SyntaxKind.UniqueKeyword && mayResolveTypeAlias((<TypeOperatorNode>node).type);
                case SyntaxKind.ParenthesizedType:
                case SyntaxKind.OptionalType:
                case SyntaxKind.NamedTupleMember:
                case SyntaxKind.JSDocOptionalType:
                case SyntaxKind.JSDocNullableType:
                case SyntaxKind.JSDocNonNullableType:
                case SyntaxKind.JSDocTypeExpression:
                    return mayResolveTypeAlias((<ParenthesizedTypeNode | OptionalTypeNode | JSDocTypeReferencingNode | NamedTupleMember>node).type);
                case SyntaxKind.RestType:
                    return (<RestTypeNode>node).type.kind !== SyntaxKind.ArrayType || mayResolveTypeAlias((<ArrayTypeNode>(<RestTypeNode>node).type).elementType);
                case SyntaxKind.UnionType:
                case SyntaxKind.IntersectionType:
                    return some((<UnionOrIntersectionTypeNode>node).types, mayResolveTypeAlias);
                case SyntaxKind.IndexedAccessType:
                    return mayResolveTypeAlias((<IndexedAccessTypeNode>node).objectType) || mayResolveTypeAlias((<IndexedAccessTypeNode>node).indexType);
                case SyntaxKind.ConditionalType:
                    return mayResolveTypeAlias((<ConditionalTypeNode>node).checkType) || mayResolveTypeAlias((<ConditionalTypeNode>node).extendsType) ||
                        mayResolveTypeAlias((<ConditionalTypeNode>node).trueType) || mayResolveTypeAlias((<ConditionalTypeNode>node).falseType);
            }
            return false;
        }

        function getTypeFromArrayOrTupleTypeNode(node: ArrayTypeNode | TupleTypeNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                const target = getArrayOrTupleTargetType(node);
                if (target === emptyGenericType) {
                    links.resolvedType = emptyObjectType;
                }
                else if (!(node.kind === SyntaxKind.TupleType && some(node.elements, e => !!(getTupleElementFlags(e) & ElementFlags.Variadic))) && isDeferredTypeReferenceNode(node)) {
                    links.resolvedType = node.kind === SyntaxKind.TupleType && node.elements.length === 0 ? target :
                        createDeferredTypeReference(target, node, /*mapper*/ undefined);
                }
                else {
                    const elementTypes = node.kind === SyntaxKind.ArrayType ? [getTypeFromTypeNode(node.elementType)] : map(node.elements, getTypeFromTypeNode);
                    links.resolvedType = createNormalizedTypeReference(target, elementTypes);
                }
            }
            return links.resolvedType;
        }

        function isReadonlyTypeOperator(node: Node) {
            return isTypeOperatorNode(node) && node.operator === SyntaxKind.ReadonlyKeyword;
        }

        function createTupleType(elementTypes: readonly Type[], elementFlags?: readonly ElementFlags[], readonly = false, namedMemberDeclarations?: readonly (NamedTupleMember | ParameterDeclaration)[]) {
            const tupleTarget = getTupleTargetType(elementFlags || map(elementTypes, _ => ElementFlags.Required), readonly, namedMemberDeclarations);
            return tupleTarget === emptyGenericType ? emptyObjectType :
                elementTypes.length ? createNormalizedTypeReference(tupleTarget, elementTypes) :
                tupleTarget;
        }

        function getTupleTargetType(elementFlags: readonly ElementFlags[], readonly: boolean, namedMemberDeclarations?: readonly (NamedTupleMember | ParameterDeclaration)[]): GenericType {
            if (elementFlags.length === 1 && elementFlags[0] & ElementFlags.Rest) {
                // [...X[]] is equivalent to just X[]
                return readonly ? globalReadonlyArrayType : globalArrayType;
            }
            const key = map(elementFlags, f => f & ElementFlags.Required ? "#" : f & ElementFlags.Optional ? "?" : f & ElementFlags.Rest ? "." : "*").join() +
                (readonly ? "R" : "") +
                (namedMemberDeclarations && namedMemberDeclarations.length ? "," + map(namedMemberDeclarations, getNodeId).join(",") : "");
            let type = tupleTypes.get(key);
            if (!type) {
                tupleTypes.set(key, type = createTupleTargetType(elementFlags, readonly, namedMemberDeclarations));
            }
            return type;
        }

        // We represent tuple types as type references to synthesized generic interface types created by
        // this function. The types are of the form:
        //
        //   interface Tuple<T0, T1, T2, ...> extends Array<T0 | T1 | T2 | ...> { 0: T0, 1: T1, 2: T2, ... }
        //
        // Note that the generic type created by this function has no symbol associated with it. The same
        // is true for each of the synthesized type parameters.
        function createTupleTargetType(elementFlags: readonly ElementFlags[], readonly: boolean, namedMemberDeclarations: readonly (NamedTupleMember | ParameterDeclaration)[] | undefined): TupleType {
            const arity = elementFlags.length;
            const minLength = findLastIndex(elementFlags, f => !!(f & (ElementFlags.Required | ElementFlags.Variadic))) + 1;
            let typeParameters: TypeParameter[] | undefined;
            const properties: Symbol[] = [];
            let combinedFlags: ElementFlags = 0;
            if (arity) {
                typeParameters = new Array(arity);
                for (let i = 0; i < arity; i++) {
                    const typeParameter = typeParameters[i] = createTypeParameter();
                    const flags = elementFlags[i];
                    combinedFlags |= flags;
                    if (!(combinedFlags & ElementFlags.Variable)) {
                        const property = createSymbol(SymbolFlags.Property | (flags & ElementFlags.Optional ? SymbolFlags.Optional : 0),
                            "" + i as __String, readonly ? CheckFlags.Readonly : 0);
                        property.tupleLabelDeclaration = namedMemberDeclarations?.[i];
                        property.type = typeParameter;
                        properties.push(property);
                    }
                }
            }
            const fixedLength = properties.length;
            const lengthSymbol = createSymbol(SymbolFlags.Property, "length" as __String);
            if (combinedFlags & ElementFlags.Variable) {
                lengthSymbol.type = numberType;
            }
            else {
                const literalTypes = [];
                for (let i = minLength; i <= arity; i++) literalTypes.push(getLiteralType(i));
                lengthSymbol.type = getUnionType(literalTypes);
            }
            properties.push(lengthSymbol);
            const type = <TupleType & InterfaceTypeWithDeclaredMembers>createObjectType(ObjectFlags.Tuple | ObjectFlags.Reference);
            type.typeParameters = typeParameters;
            type.outerTypeParameters = undefined;
            type.localTypeParameters = typeParameters;
            type.instantiations = new Map<string, TypeReference>();
            type.instantiations.set(getTypeListId(type.typeParameters), <GenericType>type);
            type.target = <GenericType>type;
            type.resolvedTypeArguments = type.typeParameters;
            type.thisType = createTypeParameter();
            type.thisType.isThisType = true;
            type.thisType.constraint = type;
            type.declaredProperties = properties;
            type.declaredCallSignatures = emptyArray;
            type.declaredConstructSignatures = emptyArray;
            type.declaredStringIndexInfo = undefined;
            type.declaredNumberIndexInfo = undefined;
            type.elementFlags = elementFlags;
            type.minLength = minLength;
            type.fixedLength = fixedLength;
            type.hasRestElement = !!(combinedFlags & ElementFlags.Variable);
            type.combinedFlags = combinedFlags;
            type.readonly = readonly;
            type.labeledElementDeclarations = namedMemberDeclarations;
            return type;
        }

        function createNormalizedTypeReference(target: GenericType, typeArguments: readonly Type[] | undefined) {
            return target.objectFlags & ObjectFlags.Tuple && (<TupleType>target).combinedFlags & ElementFlags.Variadic ?
                createNormalizedTupleType(target as TupleType, typeArguments!) :
                createTypeReference(target, typeArguments);
        }

        function createNormalizedTupleType(target: TupleType, elementTypes: readonly Type[]): Type {
            // Transform [A, ...(X | Y | Z)] into [A, ...X] | [A, ...Y] | [A, ...Z]
            const unionIndex = findIndex(elementTypes, (t, i) => !!(target.elementFlags[i] & ElementFlags.Variadic && t.flags & (TypeFlags.Never | TypeFlags.Union)));
            if (unionIndex >= 0) {
                return mapType(elementTypes[unionIndex], t => createNormalizedTupleType(target, replaceElement(elementTypes, unionIndex, t)));
            }
            // If there are no variadic elements with non-generic types, just create a type reference with the same target type.
            const spreadIndex = findIndex(elementTypes, (t, i) => !!(target.elementFlags[i] & ElementFlags.Variadic) && !(t.flags & TypeFlags.InstantiableNonPrimitive) && !isGenericMappedType(t));
            if (spreadIndex < 0) {
                return createTypeReference(target, elementTypes);
            }
            // We have non-generic variadic elements that need normalization.
            const expandedTypes: Type[] = [];
            const expandedFlags: ElementFlags[] = [];
            let expandedDeclarations: (NamedTupleMember | ParameterDeclaration)[] | undefined = [];
            let optionalIndex = -1;
            let restTypes: Type[] | undefined;
            for (let i = 0; i < elementTypes.length; i++) {
                const type = elementTypes[i];
                const flags = target.elementFlags[i];
                if (flags & ElementFlags.Variadic) {
                    if (type.flags & TypeFlags.InstantiableNonPrimitive || isGenericMappedType(type)) {
                        // Generic variadic elements stay as they are (except following a rest element).
                        addElementOrRest(type, ElementFlags.Variadic, target.labeledElementDeclarations?.[i]);
                    }
                    else if (isTupleType(type)) {
                        // Spread variadic elements with tuple types into the resulting tuple.
                        forEach(getTypeArguments(type), (t, n) => addElementOrRest(t, type.target.elementFlags[n], type.target.labeledElementDeclarations?.[n]));
                    }
                    else {
                        // Treat everything else as an array type and create a rest element.
                        addElementOrRest(isArrayLikeType(type) && getIndexTypeOfType(type, IndexKind.Number) || errorType, ElementFlags.Rest, target.labeledElementDeclarations?.[i]);
                    }
                }
                else {
                    // Copy other element kinds with no change.
                    addElementOrRest(type, flags, target.labeledElementDeclarations?.[i]);
                }
            }
            if (restTypes) {
                // Create a union of the collected rest element types.
                expandedTypes[expandedTypes.length - 1] = getUnionType(restTypes);
            }
            const tupleTarget = getTupleTargetType(expandedFlags, target.readonly, expandedDeclarations);
            return tupleTarget === emptyGenericType ? emptyObjectType :
                expandedFlags.length ? createTypeReference(tupleTarget, expandedTypes) :
                tupleTarget;

            function addElementOrRest(type: Type, flags: ElementFlags, declaration: NamedTupleMember | ParameterDeclaration | undefined) {
                if (restTypes) {
                    // A rest element was previously added, so simply collect the type of this element.
                    restTypes.push(flags & ElementFlags.Variadic ? getIndexedAccessType(type, numberType) : type);
                }
                else {
                    if (flags & ElementFlags.Required && optionalIndex >= 0) {
                        // Turn preceding optional elements into required elements
                        for (let i = optionalIndex; i < expandedFlags.length; i++) {
                            if (expandedFlags[i] & ElementFlags.Optional) expandedFlags[i] = ElementFlags.Required;
                        }
                        optionalIndex = -1;
                    }
                    else if (flags & ElementFlags.Optional && optionalIndex < 0) {
                        optionalIndex = expandedFlags.length;
                    }
                    else if (flags & ElementFlags.Rest) {
                        // Start collecting element types when a rest element is added.
                        restTypes = [type];
                    }
                    expandedTypes.push(type);
                    expandedFlags.push(flags);
                    if (expandedDeclarations && declaration) {
                        expandedDeclarations.push(declaration);
                    }
                    else {
                        expandedDeclarations = undefined;
                    }
                }
            }
        }

        function sliceTupleType(type: TupleTypeReference, index: number, endSkipCount = 0) {
            const target = type.target;
            const endIndex = getTypeReferenceArity(type) - endSkipCount;
            return index > target.fixedLength ? getRestArrayTypeOfTupleType(type) || createTupleType(emptyArray) :
                createTupleType(getTypeArguments(type).slice(index, endIndex), target.elementFlags.slice(index, endIndex),
                    /*readonly*/ false, target.labeledElementDeclarations && target.labeledElementDeclarations.slice(index, endIndex));
        }

        function getKnownKeysOfTupleType(type: TupleTypeReference) {
            return getUnionType(append(arrayOf(type.target.fixedLength, i => getLiteralType("" + i)),
                getIndexType(type.target.readonly ? globalReadonlyArrayType : globalArrayType)));
        }

        function getTypeFromOptionalTypeNode(node: OptionalTypeNode): Type {
            const type = getTypeFromTypeNode(node.type);
            return strictNullChecks ? getOptionalType(type) : type;
        }

        function getTypeId(type: Type): TypeId {
            return type.id;
        }

        function containsType(types: readonly Type[], type: Type): boolean {
            return binarySearch(types, type, getTypeId, compareValues) >= 0;
        }

        function insertType(types: Type[], type: Type): boolean {
            const index = binarySearch(types, type, getTypeId, compareValues);
            if (index < 0) {
                types.splice(~index, 0, type);
                return true;
            }
            return false;
        }

        function addTypeToUnion(typeSet: Type[], includes: TypeFlags, type: Type) {
            const flags = type.flags;
            if (flags & TypeFlags.Union) {
                return addTypesToUnion(typeSet, includes, (<UnionType>type).types);
            }
            // We ignore 'never' types in unions
            if (!(flags & TypeFlags.Never)) {
                includes |= flags & TypeFlags.IncludesMask;
                if (flags & TypeFlags.StructuredOrInstantiable) includes |= TypeFlags.IncludesStructuredOrInstantiable;
                if (type === wildcardType) includes |= TypeFlags.IncludesWildcard;
                if (!strictNullChecks && flags & TypeFlags.Nullable) {
                    if (!(getObjectFlags(type) & ObjectFlags.ContainsWideningType)) includes |= TypeFlags.IncludesNonWideningType;
                }
                else {
                    const len = typeSet.length;
                    const index = len && type.id > typeSet[len - 1].id ? ~len : binarySearch(typeSet, type, getTypeId, compareValues);
                    if (index < 0) {
                        typeSet.splice(~index, 0, type);
                    }
                }
            }
            return includes;
        }

        // Add the given types to the given type set. Order is preserved, duplicates are removed,
        // and nested types of the given kind are flattened into the set.
        function addTypesToUnion(typeSet: Type[], includes: TypeFlags, types: readonly Type[]): TypeFlags {
            for (const type of types) {
                includes = addTypeToUnion(typeSet, includes, type);
            }
            return includes;
        }

        function isSetOfLiteralsFromSameEnum(types: readonly Type[]): boolean {
            const first = types[0];
            if (first.flags & TypeFlags.EnumLiteral) {
                const firstEnum = getParentOfSymbol(first.symbol);
                for (let i = 1; i < types.length; i++) {
                    const other = types[i];
                    if (!(other.flags & TypeFlags.EnumLiteral) || (firstEnum !== getParentOfSymbol(other.symbol))) {
                        return false;
                    }
                }
                return true;
            }

            return false;
        }

        function removeSubtypes(types: Type[], primitivesOnly: boolean): boolean {
            const len = types.length;
            if (len === 0 || isSetOfLiteralsFromSameEnum(types)) {
                return true;
            }
            let i = len;
            let count = 0;
            while (i > 0) {
                i--;
                const source = types[i];
                for (const target of types) {
                    if (source !== target) {
                        if (count === 100000) {
                            // After 100000 subtype checks we estimate the remaining amount of work by assuming the
                            // same ratio of checks per element. If the estimated number of remaining type checks is
                            // greater than an upper limit we deem the union type too complex to represent. The
                            // upper limit is 25M for unions of primitives only, and 1M otherwise. This for example
                            // caps union types at 5000 unique literal types and 1000 unique object types.
                            const estimatedCount = (count / (len - i)) * len;
                            if (estimatedCount > (primitivesOnly ? 25000000 : 1000000)) {
                                error(currentNode, Diagnostics.Expression_produces_a_union_type_that_is_too_complex_to_represent);
                                return false;
                            }
                        }
                        count++;
                        if (isTypeRelatedTo(source, target, strictSubtypeRelation) && (
                            !(getObjectFlags(getTargetType(source)) & ObjectFlags.Class) ||
                            !(getObjectFlags(getTargetType(target)) & ObjectFlags.Class) ||
                            isTypeDerivedFrom(source, target))) {
                            orderedRemoveItemAt(types, i);
                            break;
                        }
                    }
                }
            }
            return true;
        }

        function removeRedundantLiteralTypes(types: Type[], includes: TypeFlags) {
            let i = types.length;
            while (i > 0) {
                i--;
                const t = types[i];
                const remove =
                    t.flags & TypeFlags.StringLiteral && includes & TypeFlags.String ||
                    t.flags & TypeFlags.NumberLiteral && includes & TypeFlags.Number ||
                    t.flags & TypeFlags.BigIntLiteral && includes & TypeFlags.BigInt ||
                    t.flags & TypeFlags.UniqueESSymbol && includes & TypeFlags.ESSymbol ||
                    isFreshLiteralType(t) && containsType(types, (<LiteralType>t).regularType);
                if (remove) {
                    orderedRemoveItemAt(types, i);
                }
            }
        }

        // We sort and deduplicate the constituent types based on object identity. If the subtypeReduction
        // flag is specified we also reduce the constituent type set to only include types that aren't subtypes
        // of other types. Subtype reduction is expensive for large union types and is possible only when union
        // types are known not to circularly reference themselves (as is the case with union types created by
        // expression constructs such as array literals and the || and ?: operators). Named types can
        // circularly reference themselves and therefore cannot be subtype reduced during their declaration.
        // For example, "type Item = string | (() => Item" is a named type that circularly references itself.
        function getUnionType(types: readonly Type[], unionReduction: UnionReduction = UnionReduction.Literal, aliasSymbol?: Symbol, aliasTypeArguments?: readonly Type[]): Type {
            if (types.length === 0) {
                return neverType;
            }
            if (types.length === 1) {
                return types[0];
            }
            const typeSet: Type[] = [];
            const includes = addTypesToUnion(typeSet, 0, types);
            if (unionReduction !== UnionReduction.None) {
                if (includes & TypeFlags.AnyOrUnknown) {
                    return includes & TypeFlags.Any ? includes & TypeFlags.IncludesWildcard ? wildcardType : anyType : unknownType;
                }
                switch (unionReduction) {
                    case UnionReduction.Literal:
                        if (includes & (TypeFlags.Literal | TypeFlags.UniqueESSymbol)) {
                            removeRedundantLiteralTypes(typeSet, includes);
                        }
                        break;
                    case UnionReduction.Subtype:
                        if (!removeSubtypes(typeSet, !(includes & TypeFlags.IncludesStructuredOrInstantiable))) {
                            return errorType;
                        }
                        break;
                }
                if (typeSet.length === 0) {
                    return includes & TypeFlags.Null ? includes & TypeFlags.IncludesNonWideningType ? nullType : nullWideningType :
                        includes & TypeFlags.Undefined ? includes & TypeFlags.IncludesNonWideningType ? undefinedType : undefinedWideningType :
                        neverType;
                }
            }
            const objectFlags = (includes & TypeFlags.NotPrimitiveUnion ? 0 : ObjectFlags.PrimitiveUnion) |
                (includes & TypeFlags.Intersection ? ObjectFlags.ContainsIntersections : 0);
            return getUnionTypeFromSortedList(typeSet, objectFlags, aliasSymbol, aliasTypeArguments);
        }

        function getUnionTypePredicate(signatures: readonly Signature[]): TypePredicate | undefined {
            let first: TypePredicate | undefined;
            const types: Type[] = [];
            for (const sig of signatures) {
                const pred = getTypePredicateOfSignature(sig);
                if (!pred || pred.kind === TypePredicateKind.AssertsThis || pred.kind === TypePredicateKind.AssertsIdentifier) {
                    continue;
                }

                if (first) {
                    if (!typePredicateKindsMatch(first, pred)) {
                        // No common type predicate.
                        return undefined;
                    }
                }
                else {
                    first = pred;
                }
                types.push(pred.type);
            }
            if (!first) {
                // No union signatures had a type predicate.
                return undefined;
            }
            const unionType = getUnionType(types);
            return createTypePredicate(first.kind, first.parameterName, first.parameterIndex, unionType);
        }

        function typePredicateKindsMatch(a: TypePredicate, b: TypePredicate): boolean {
            return a.kind === b.kind && a.parameterIndex === b.parameterIndex;
        }

        // This function assumes the constituent type list is sorted and deduplicated.
        function getUnionTypeFromSortedList(types: Type[], objectFlags: ObjectFlags, aliasSymbol?: Symbol, aliasTypeArguments?: readonly Type[]): Type {
            if (types.length === 0) {
                return neverType;
            }
            if (types.length === 1) {
                return types[0];
            }
            const id = getTypeListId(types);
            let type = unionTypes.get(id);
            if (!type) {
                type = <UnionType>createType(TypeFlags.Union);
                unionTypes.set(id, type);
                type.objectFlags = objectFlags | getPropagatingFlagsOfTypes(types, /*excludeKinds*/ TypeFlags.Nullable);
                type.types = types;
                /*
                Note: This is the alias symbol (or lack thereof) that we see when we first encounter this union type.
                For aliases of identical unions, eg `type T = A | B; type U = A | B`, the symbol of the first alias encountered is the aliasSymbol.
                (In the language service, the order may depend on the order in which a user takes actions, such as hovering over symbols.)
                It's important that we create equivalent union types only once, so that's an unfortunate side effect.
                */
                type.aliasSymbol = aliasSymbol;
                type.aliasTypeArguments = aliasTypeArguments;
            }
            return type;
        }

        function getTypeFromUnionTypeNode(node: UnionTypeNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                const aliasSymbol = getAliasSymbolForTypeNode(node);
                links.resolvedType = getUnionType(map(node.types, getTypeFromTypeNode), UnionReduction.Literal,
                    aliasSymbol, getTypeArgumentsForAliasSymbol(aliasSymbol));
            }
            return links.resolvedType;
        }

        function addTypeToIntersection(typeSet: ESMap<string, Type>, includes: TypeFlags, type: Type) {
            const flags = type.flags;
            if (flags & TypeFlags.Intersection) {
                return addTypesToIntersection(typeSet, includes, (<IntersectionType>type).types);
            }
            if (isEmptyAnonymousObjectType(type)) {
                if (!(includes & TypeFlags.IncludesEmptyObject)) {
                    includes |= TypeFlags.IncludesEmptyObject;
                    typeSet.set(type.id.toString(), type);
                }
            }
            else {
                if (flags & TypeFlags.AnyOrUnknown) {
                    if (type === wildcardType) includes |= TypeFlags.IncludesWildcard;
                }
                else if ((strictNullChecks || !(flags & TypeFlags.Nullable)) && !typeSet.has(type.id.toString())) {
                    if (type.flags & TypeFlags.Unit && includes & TypeFlags.Unit) {
                        // We have seen two distinct unit types which means we should reduce to an
                        // empty intersection. Adding TypeFlags.NonPrimitive causes that to happen.
                        includes |= TypeFlags.NonPrimitive;
                    }
                    typeSet.set(type.id.toString(), type);
                }
                includes |= flags & TypeFlags.IncludesMask;
            }
            return includes;
        }

        // Add the given types to the given type set. Order is preserved, freshness is removed from literal
        // types, duplicates are removed, and nested types of the given kind are flattened into the set.
        function addTypesToIntersection(typeSet: ESMap<string, Type>, includes: TypeFlags, types: readonly Type[]) {
            for (const type of types) {
                includes = addTypeToIntersection(typeSet, includes, getRegularTypeOfLiteralType(type));
            }
            return includes;
        }

        function removeRedundantPrimitiveTypes(types: Type[], includes: TypeFlags) {
            let i = types.length;
            while (i > 0) {
                i--;
                const t = types[i];
                const remove =
                    t.flags & TypeFlags.String && includes & TypeFlags.StringLiteral ||
                    t.flags & TypeFlags.Number && includes & TypeFlags.NumberLiteral ||
                    t.flags & TypeFlags.BigInt && includes & TypeFlags.BigIntLiteral ||
                    t.flags & TypeFlags.ESSymbol && includes & TypeFlags.UniqueESSymbol;
                if (remove) {
                    orderedRemoveItemAt(types, i);
                }
            }
        }

        // Check that the given type has a match in every union. A given type is matched by
        // an identical type, and a literal type is additionally matched by its corresponding
        // primitive type.
        function eachUnionContains(unionTypes: UnionType[], type: Type) {
            for (const u of unionTypes) {
                if (!containsType(u.types, type)) {
                    const primitive = type.flags & TypeFlags.StringLiteral ? stringType :
                        type.flags & TypeFlags.NumberLiteral ? numberType :
                        type.flags & TypeFlags.BigIntLiteral ? bigintType :
                        type.flags & TypeFlags.UniqueESSymbol ? esSymbolType :
                        undefined;
                    if (!primitive || !containsType(u.types, primitive)) {
                        return false;
                    }
                }
            }
            return true;
        }

        function extractIrreducible(types: Type[], flag: TypeFlags) {
            if (every(types, t => !!(t.flags & TypeFlags.Union) && some((t as UnionType).types, tt => !!(tt.flags & flag)))) {
                for (let i = 0; i < types.length; i++) {
                    types[i] = filterType(types[i], t => !(t.flags & flag));
                }
                return true;
            }
            return false;
        }

        // If the given list of types contains more than one union of primitive types, replace the
        // first with a union containing an intersection of those primitive types, then remove the
        // other unions and return true. Otherwise, do nothing and return false.
        function intersectUnionsOfPrimitiveTypes(types: Type[]) {
            let unionTypes: UnionType[] | undefined;
            const index = findIndex(types, t => !!(getObjectFlags(t) & ObjectFlags.PrimitiveUnion));
            if (index < 0) {
                return false;
            }
            let i = index + 1;
            // Remove all but the first union of primitive types and collect them in
            // the unionTypes array.
            while (i < types.length) {
                const t = types[i];
                if (getObjectFlags(t) & ObjectFlags.PrimitiveUnion) {
                    (unionTypes || (unionTypes = [<UnionType>types[index]])).push(<UnionType>t);
                    orderedRemoveItemAt(types, i);
                }
                else {
                    i++;
                }
            }
            // Return false if there was only one union of primitive types
            if (!unionTypes) {
                return false;
            }
            // We have more than one union of primitive types, now intersect them. For each
            // type in each union we check if the type is matched in every union and if so
            // we include it in the result.
            const checked: Type[] = [];
            const result: Type[] = [];
            for (const u of unionTypes) {
                for (const t of u.types) {
                    if (insertType(checked, t)) {
                        if (eachUnionContains(unionTypes, t)) {
                            insertType(result, t);
                        }
                    }
                }
            }
            // Finally replace the first union with the result
            types[index] = getUnionTypeFromSortedList(result, ObjectFlags.PrimitiveUnion);
            return true;
        }

        function createIntersectionType(types: Type[], aliasSymbol?: Symbol, aliasTypeArguments?: readonly Type[]) {
            const result = <IntersectionType>createType(TypeFlags.Intersection);
            result.objectFlags = getPropagatingFlagsOfTypes(types, /*excludeKinds*/ TypeFlags.Nullable);
            result.types = types;
            result.aliasSymbol = aliasSymbol; // See comment in `getUnionTypeFromSortedList`.
            result.aliasTypeArguments = aliasTypeArguments;
            return result;
        }

        // We normalize combinations of intersection and union types based on the distributive property of the '&'
        // operator. Specifically, because X & (A | B) is equivalent to X & A | X & B, we can transform intersection
        // types with union type constituents into equivalent union types with intersection type constituents and
        // effectively ensure that union types are always at the top level in type representations.
        //
        // We do not perform structural deduplication on intersection types. Intersection types are created only by the &
        // type operator and we can't reduce those because we want to support recursive intersection types. For example,
        // a type alias of the form "type List<T> = T & { next: List<T> }" cannot be reduced during its declaration.
        // Also, unlike union types, the order of the constituent types is preserved in order that overload resolution
        // for intersections of types with signatures can be deterministic.
        function getIntersectionType(types: readonly Type[], aliasSymbol?: Symbol, aliasTypeArguments?: readonly Type[]): Type {
            const typeMembershipMap: ESMap<string, Type> = new Map();
            const includes = addTypesToIntersection(typeMembershipMap, 0, types);
            const typeSet: Type[] = arrayFrom(typeMembershipMap.values());
            // An intersection type is considered empty if it contains
            // the type never, or
            // more than one unit type or,
            // an object type and a nullable type (null or undefined), or
            // a string-like type and a type known to be non-string-like, or
            // a number-like type and a type known to be non-number-like, or
            // a symbol-like type and a type known to be non-symbol-like, or
            // a void-like type and a type known to be non-void-like, or
            // a non-primitive type and a type known to be primitive.
            if (includes & TypeFlags.Never ||
                strictNullChecks && includes & TypeFlags.Nullable && includes & (TypeFlags.Object | TypeFlags.NonPrimitive | TypeFlags.IncludesEmptyObject) ||
                includes & TypeFlags.NonPrimitive && includes & (TypeFlags.DisjointDomains & ~TypeFlags.NonPrimitive) ||
                includes & TypeFlags.StringLike && includes & (TypeFlags.DisjointDomains & ~TypeFlags.StringLike) ||
                includes & TypeFlags.NumberLike && includes & (TypeFlags.DisjointDomains & ~TypeFlags.NumberLike) ||
                includes & TypeFlags.BigIntLike && includes & (TypeFlags.DisjointDomains & ~TypeFlags.BigIntLike) ||
                includes & TypeFlags.ESSymbolLike && includes & (TypeFlags.DisjointDomains & ~TypeFlags.ESSymbolLike) ||
                includes & TypeFlags.VoidLike && includes & (TypeFlags.DisjointDomains & ~TypeFlags.VoidLike)) {
                return neverType;
            }
            if (includes & TypeFlags.Any) {
                return includes & TypeFlags.IncludesWildcard ? wildcardType : anyType;
            }
            if (!strictNullChecks && includes & TypeFlags.Nullable) {
                return includes & TypeFlags.Undefined ? undefinedType : nullType;
            }
            if (includes & TypeFlags.String && includes & TypeFlags.StringLiteral ||
                includes & TypeFlags.Number && includes & TypeFlags.NumberLiteral ||
                includes & TypeFlags.BigInt && includes & TypeFlags.BigIntLiteral ||
                includes & TypeFlags.ESSymbol && includes & TypeFlags.UniqueESSymbol) {
                removeRedundantPrimitiveTypes(typeSet, includes);
            }
            if (includes & TypeFlags.IncludesEmptyObject && includes & TypeFlags.Object) {
                orderedRemoveItemAt(typeSet, findIndex(typeSet, isEmptyAnonymousObjectType));
            }
            if (typeSet.length === 0) {
                return unknownType;
            }
            if (typeSet.length === 1) {
                return typeSet[0];
            }
            const id = getTypeListId(typeSet);
            let result = intersectionTypes.get(id);
            if (!result) {
                if (includes & TypeFlags.Union) {
                    if (intersectUnionsOfPrimitiveTypes(typeSet)) {
                        // When the intersection creates a reduced set (which might mean that *all* union types have
                        // disappeared), we restart the operation to get a new set of combined flags. Once we have
                        // reduced we'll never reduce again, so this occurs at most once.
                        result = getIntersectionType(typeSet, aliasSymbol, aliasTypeArguments);
                    }
                    else if (extractIrreducible(typeSet, TypeFlags.Undefined)) {
                        result = getUnionType([getIntersectionType(typeSet), undefinedType], UnionReduction.Literal, aliasSymbol, aliasTypeArguments);
                    }
                    else if (extractIrreducible(typeSet, TypeFlags.Null)) {
                        result = getUnionType([getIntersectionType(typeSet), nullType], UnionReduction.Literal, aliasSymbol, aliasTypeArguments);
                    }
                    else {
                        // We are attempting to construct a type of the form X & (A | B) & Y. Transform this into a type of
                        // the form X & A & Y | X & B & Y and recursively reduce until no union type constituents remain.
                        // If the estimated size of the resulting union type exceeds 100000 constituents, report an error.
                        const size = reduceLeft(typeSet, (n, t) => n * (t.flags & TypeFlags.Union ? (<UnionType>t).types.length : 1), 1);
                        if (size >= 100000) {
                            error(currentNode, Diagnostics.Expression_produces_a_union_type_that_is_too_complex_to_represent);
                            return errorType;
                        }
                        const unionIndex = findIndex(typeSet, t => (t.flags & TypeFlags.Union) !== 0);
                        const unionType = <UnionType>typeSet[unionIndex];
                        result = getUnionType(map(unionType.types, t => getIntersectionType(replaceElement(typeSet, unionIndex, t))),
                            UnionReduction.Literal, aliasSymbol, aliasTypeArguments);
                    }
                }
                else {
                    result = createIntersectionType(typeSet, aliasSymbol, aliasTypeArguments);
                }
                intersectionTypes.set(id, result);
            }
            return result;
        }

        function getTypeFromIntersectionTypeNode(node: IntersectionTypeNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                const aliasSymbol = getAliasSymbolForTypeNode(node);
                links.resolvedType = getIntersectionType(map(node.types, getTypeFromTypeNode),
                    aliasSymbol, getTypeArgumentsForAliasSymbol(aliasSymbol));
            }
            return links.resolvedType;
        }

        function createIndexType(type: InstantiableType | UnionOrIntersectionType, stringsOnly: boolean) {
            const result = <IndexType>createType(TypeFlags.Index);
            result.type = type;
            result.stringsOnly = stringsOnly;
            return result;
        }

        function getIndexTypeForGenericType(type: InstantiableType | UnionOrIntersectionType, stringsOnly: boolean) {
            return stringsOnly ?
                type.resolvedStringIndexType || (type.resolvedStringIndexType = createIndexType(type, /*stringsOnly*/ true)) :
                type.resolvedIndexType || (type.resolvedIndexType = createIndexType(type, /*stringsOnly*/ false));
        }

        function getLiteralTypeFromPropertyName(name: PropertyName) {
            if (isPrivateIdentifier(name)) {
                return neverType;
            }
            return isIdentifier(name) ? getLiteralType(unescapeLeadingUnderscores(name.escapedText)) :
                getRegularTypeOfLiteralType(isComputedPropertyName(name) ? checkComputedPropertyName(name) : checkExpression(name));
        }

        function getBigIntLiteralType(node: BigIntLiteral): LiteralType {
            return getLiteralType({
                negative: false,
                base10Value: parsePseudoBigInt(node.text)
            });
        }

        function getLiteralTypeFromProperty(prop: Symbol, include: TypeFlags) {
            if (!(getDeclarationModifierFlagsFromSymbol(prop) & ModifierFlags.NonPublicAccessibilityModifier)) {
                let type = getSymbolLinks(getLateBoundSymbol(prop)).nameType;
                if (!type && !isKnownSymbol(prop)) {
                    if (prop.escapedName === InternalSymbolName.Default) {
                        type = getLiteralType("default");
                    }
                    else {
                        const name = prop.valueDeclaration && getNameOfDeclaration(prop.valueDeclaration) as PropertyName;
                        type = name && getLiteralTypeFromPropertyName(name) || getLiteralType(symbolName(prop));
                    }
                }
                if (type && type.flags & include) {
                    return type;
                }
            }
            return neverType;
        }

        function getLiteralTypeFromProperties(type: Type, include: TypeFlags) {
            return getUnionType(map(getPropertiesOfType(type), p => getLiteralTypeFromProperty(p, include)));
        }

        function getNonEnumNumberIndexInfo(type: Type) {
            const numberIndexInfo = getIndexInfoOfType(type, IndexKind.Number);
            return numberIndexInfo !== enumNumberIndexInfo ? numberIndexInfo : undefined;
        }

        function getIndexType(type: Type, stringsOnly = keyofStringsOnly, noIndexSignatures?: boolean): Type {
            type = getReducedType(type);
            return type.flags & TypeFlags.Union ? getIntersectionType(map((<IntersectionType>type).types, t => getIndexType(t, stringsOnly, noIndexSignatures))) :
                type.flags & TypeFlags.Intersection ? getUnionType(map((<IntersectionType>type).types, t => getIndexType(t, stringsOnly, noIndexSignatures))) :
                type.flags & TypeFlags.InstantiableNonPrimitive || isGenericTupleType(type) ? getIndexTypeForGenericType(<InstantiableType | UnionOrIntersectionType>type, stringsOnly) :
                getObjectFlags(type) & ObjectFlags.Mapped ? filterType(getConstraintTypeFromMappedType(<MappedType>type), t => !(noIndexSignatures && t.flags & (TypeFlags.Any | TypeFlags.String))) :
                type === wildcardType ? wildcardType :
                type.flags & TypeFlags.Unknown ? neverType :
                type.flags & (TypeFlags.Any | TypeFlags.Never) ? keyofConstraintType :
                stringsOnly ? !noIndexSignatures && getIndexInfoOfType(type, IndexKind.String) ? stringType : getLiteralTypeFromProperties(type, TypeFlags.StringLiteral) :
                !noIndexSignatures && getIndexInfoOfType(type, IndexKind.String) ? getUnionType([stringType, numberType, getLiteralTypeFromProperties(type, TypeFlags.UniqueESSymbol)]) :
                getNonEnumNumberIndexInfo(type) ? getUnionType([numberType, getLiteralTypeFromProperties(type, TypeFlags.StringLiteral | TypeFlags.UniqueESSymbol)]) :
                getLiteralTypeFromProperties(type, TypeFlags.StringOrNumberLiteralOrUnique);
        }

        function getExtractStringType(type: Type) {
            if (keyofStringsOnly) {
                return type;
            }
            const extractTypeAlias = getGlobalExtractSymbol();
            return extractTypeAlias ? getTypeAliasInstantiation(extractTypeAlias, [type, stringType]) : stringType;
        }

        function getIndexTypeOrString(type: Type): Type {
            const indexType = getExtractStringType(getIndexType(type));
            return indexType.flags & TypeFlags.Never ? stringType : indexType;
        }

        function getTypeFromTypeOperatorNode(node: TypeOperatorNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                switch (node.operator) {
                    case SyntaxKind.KeyOfKeyword:
                        links.resolvedType = getIndexType(getTypeFromTypeNode(node.type));
                        break;
                    case SyntaxKind.UniqueKeyword:
                        links.resolvedType = node.type.kind === SyntaxKind.SymbolKeyword
                            ? getESSymbolLikeTypeForNode(walkUpParenthesizedTypes(node.parent))
                            : errorType;
                        break;
                    case SyntaxKind.ReadonlyKeyword:
                        links.resolvedType = getTypeFromTypeNode(node.type);
                        break;
                    default:
                        throw Debug.assertNever(node.operator);
                }
            }
            return links.resolvedType;
        }

        function createIndexedAccessType(objectType: Type, indexType: Type, aliasSymbol: Symbol | undefined, aliasTypeArguments: readonly Type[] | undefined) {
            const type = <IndexedAccessType>createType(TypeFlags.IndexedAccess);
            type.objectType = objectType;
            type.indexType = indexType;
            type.aliasSymbol = aliasSymbol;
            type.aliasTypeArguments = aliasTypeArguments;
            return type;
        }

        /**
         * Returns if a type is or consists of a JSLiteral object type
         * In addition to objects which are directly literals,
         * * unions where every element is a jsliteral
         * * intersections where at least one element is a jsliteral
         * * and instantiable types constrained to a jsliteral
         * Should all count as literals and not print errors on access or assignment of possibly existing properties.
         * This mirrors the behavior of the index signature propagation, to which this behaves similarly (but doesn't affect assignability or inference).
         */
        function isJSLiteralType(type: Type): boolean {
            if (noImplicitAny) {
                return false; // Flag is meaningless under `noImplicitAny` mode
            }
            if (getObjectFlags(type) & ObjectFlags.JSLiteral) {
                return true;
            }
            if (type.flags & TypeFlags.Union) {
                return every((type as UnionType).types, isJSLiteralType);
            }
            if (type.flags & TypeFlags.Intersection) {
                return some((type as IntersectionType).types, isJSLiteralType);
            }
            if (type.flags & TypeFlags.Instantiable) {
                return isJSLiteralType(getResolvedBaseConstraint(type));
            }
            return false;
        }

        function getPropertyNameFromIndex(indexType: Type, accessNode: StringLiteral | Identifier | PrivateIdentifier | ObjectBindingPattern | ArrayBindingPattern | ComputedPropertyName | NumericLiteral | IndexedAccessTypeNode | ElementAccessExpression | SyntheticExpression | undefined) {
            const accessExpression = accessNode && accessNode.kind === SyntaxKind.ElementAccessExpression ? accessNode : undefined;
            return isTypeUsableAsPropertyName(indexType) ?
                getPropertyNameFromType(indexType) :
                accessExpression && checkThatExpressionIsProperSymbolReference(accessExpression.argumentExpression, indexType, /*reportError*/ false) ?
                    getPropertyNameForKnownSymbolName(idText((<PropertyAccessExpression>accessExpression.argumentExpression).name)) :
                    accessNode && isPropertyName(accessNode) ?
                        // late bound names are handled in the first branch, so here we only need to handle normal names
                        getPropertyNameForPropertyNameNode(accessNode) :
                        undefined;
        }

        function isUncalledFunctionReference(node: Node, symbol: Symbol) {
            return !(symbol.flags & (SymbolFlags.Function | SymbolFlags.Method))
                || !isCallLikeExpression(findAncestor(node, n => !isAccessExpression(n)) || node.parent)
                && every(symbol.declarations, d => !isFunctionLike(d) || !!(getCombinedNodeFlags(d) & NodeFlags.Deprecated));
        }

        function getPropertyTypeForIndexType(originalObjectType: Type, objectType: Type, indexType: Type, fullIndexType: Type, suppressNoImplicitAnyError: boolean, accessNode: ElementAccessExpression | IndexedAccessTypeNode | PropertyName | BindingName | SyntheticExpression | undefined, accessFlags: AccessFlags, reportDeprecated?: boolean) {
            const accessExpression = accessNode && accessNode.kind === SyntaxKind.ElementAccessExpression ? accessNode : undefined;
            const propName = accessNode && isPrivateIdentifier(accessNode) ? undefined : getPropertyNameFromIndex(indexType, accessNode);
            if (propName !== undefined) {
                const prop = getPropertyOfType(objectType, propName);
                if (prop) {
                    if (reportDeprecated && accessNode && prop.valueDeclaration?.flags & NodeFlags.Deprecated && isUncalledFunctionReference(accessNode, prop)) {
                        const deprecatedNode = accessExpression?.argumentExpression ?? (isIndexedAccessTypeNode(accessNode) ? accessNode.indexType : accessNode);
                        errorOrSuggestion(/* isError */ false, deprecatedNode, Diagnostics._0_is_deprecated, propName as string);
                    }
                    if (accessExpression) {
                        markPropertyAsReferenced(prop, accessExpression, /*isThisAccess*/ accessExpression.expression.kind === SyntaxKind.ThisKeyword);
                        if (isAssignmentToReadonlyEntity(accessExpression, prop, getAssignmentTargetKind(accessExpression))) {
                            error(accessExpression.argumentExpression, Diagnostics.Cannot_assign_to_0_because_it_is_a_read_only_property, symbolToString(prop));
                            return undefined;
                        }
                        if (accessFlags & AccessFlags.CacheSymbol) {
                            getNodeLinks(accessNode!).resolvedSymbol = prop;
                        }
                        if (isThisPropertyAccessInConstructor(accessExpression, prop)) {
                            return autoType;
                        }
                    }
                    const propType = getTypeOfSymbol(prop);
                    return accessExpression && getAssignmentTargetKind(accessExpression) !== AssignmentKind.Definite ?
                        getFlowTypeOfReference(accessExpression, propType) :
                        propType;
                }
                if (everyType(objectType, isTupleType) && isNumericLiteralName(propName) && +propName >= 0) {
                    if (accessNode && everyType(objectType, t => !(<TupleTypeReference>t).target.hasRestElement) && !(accessFlags & AccessFlags.NoTupleBoundsCheck)) {
                        const indexNode = getIndexNodeForAccessExpression(accessNode);
                        if (isTupleType(objectType)) {
                            error(indexNode, Diagnostics.Tuple_type_0_of_length_1_has_no_element_at_index_2,
                                typeToString(objectType), getTypeReferenceArity(objectType), unescapeLeadingUnderscores(propName));
                        }
                        else {
                            error(indexNode, Diagnostics.Property_0_does_not_exist_on_type_1, unescapeLeadingUnderscores(propName), typeToString(objectType));
                        }
                    }
                    errorIfWritingToReadonlyIndex(getIndexInfoOfType(objectType, IndexKind.Number));
                    return mapType(objectType, t => getRestTypeOfTupleType(<TupleTypeReference>t) || undefinedType);
                }
            }
            if (!(indexType.flags & TypeFlags.Nullable) && isTypeAssignableToKind(indexType, TypeFlags.StringLike | TypeFlags.NumberLike | TypeFlags.ESSymbolLike)) {
                if (objectType.flags & (TypeFlags.Any | TypeFlags.Never)) {
                    return objectType;
                }
                const stringIndexInfo = getIndexInfoOfType(objectType, IndexKind.String);
                const indexInfo = isTypeAssignableToKind(indexType, TypeFlags.NumberLike) && getIndexInfoOfType(objectType, IndexKind.Number) || stringIndexInfo;
                if (indexInfo) {
                    if (accessFlags & AccessFlags.NoIndexSignatures && indexInfo === stringIndexInfo) {
                        if (accessExpression) {
                            error(accessExpression, Diagnostics.Type_0_cannot_be_used_to_index_type_1, typeToString(indexType), typeToString(originalObjectType));
                        }
                        return undefined;
                    }
                    if (accessNode && !isTypeAssignableToKind(indexType, TypeFlags.String | TypeFlags.Number)) {
                        const indexNode = getIndexNodeForAccessExpression(accessNode);
                        error(indexNode, Diagnostics.Type_0_cannot_be_used_as_an_index_type, typeToString(indexType));
                        return indexInfo.type;
                    }
                    errorIfWritingToReadonlyIndex(indexInfo);
                    return indexInfo.type;
                }
                if (indexType.flags & TypeFlags.Never) {
                    return neverType;
                }
                if (isJSLiteralType(objectType)) {
                    return anyType;
                }
                if (accessExpression && !isConstEnumObjectType(objectType)) {
                    if (objectType.symbol === globalThisSymbol && propName !== undefined && globalThisSymbol.exports!.has(propName) && (globalThisSymbol.exports!.get(propName)!.flags & SymbolFlags.BlockScoped)) {
                        error(accessExpression, Diagnostics.Property_0_does_not_exist_on_type_1, unescapeLeadingUnderscores(propName), typeToString(objectType));
                    }
                    else if (noImplicitAny && !compilerOptions.suppressImplicitAnyIndexErrors && !suppressNoImplicitAnyError) {
                        if (propName !== undefined && typeHasStaticProperty(propName, objectType)) {
                            error(accessExpression, Diagnostics.Property_0_is_a_static_member_of_type_1, propName as string, typeToString(objectType));
                        }
                        else if (getIndexTypeOfType(objectType, IndexKind.Number)) {
                            error(accessExpression.argumentExpression, Diagnostics.Element_implicitly_has_an_any_type_because_index_expression_is_not_of_type_number);
                        }
                        else {
                            let suggestion: string | undefined;
                            if (propName !== undefined && (suggestion = getSuggestionForNonexistentProperty(propName as string, objectType))) {
                                if (suggestion !== undefined) {
                                    error(accessExpression.argumentExpression, Diagnostics.Property_0_does_not_exist_on_type_1_Did_you_mean_2, propName as string, typeToString(objectType), suggestion);
                                }
                            }
                            else {
                                const suggestion = getSuggestionForNonexistentIndexSignature(objectType, accessExpression, indexType);
                                if (suggestion !== undefined) {
                                    error(accessExpression, Diagnostics.Element_implicitly_has_an_any_type_because_type_0_has_no_index_signature_Did_you_mean_to_call_1, typeToString(objectType), suggestion);
                                }
                                else {
                                    let errorInfo: DiagnosticMessageChain | undefined;
                                    if (indexType.flags & TypeFlags.EnumLiteral) {
                                        errorInfo = chainDiagnosticMessages(/* details */ undefined, Diagnostics.Property_0_does_not_exist_on_type_1, "[" + typeToString(indexType) + "]", typeToString(objectType));
                                    }
                                    else if (indexType.flags & TypeFlags.UniqueESSymbol) {
                                        const symbolName = getFullyQualifiedName((indexType as UniqueESSymbolType).symbol, accessExpression);
                                        errorInfo = chainDiagnosticMessages(/* details */ undefined, Diagnostics.Property_0_does_not_exist_on_type_1, "[" + symbolName + "]", typeToString(objectType));
                                    }
                                    else if (indexType.flags & TypeFlags.StringLiteral) {
                                        errorInfo = chainDiagnosticMessages(/* details */ undefined, Diagnostics.Property_0_does_not_exist_on_type_1, (indexType as StringLiteralType).value, typeToString(objectType));
                                    }
                                    else if (indexType.flags & TypeFlags.NumberLiteral) {
                                        errorInfo = chainDiagnosticMessages(/* details */ undefined, Diagnostics.Property_0_does_not_exist_on_type_1, (indexType as NumberLiteralType).value, typeToString(objectType));
                                    }
                                    else if (indexType.flags & (TypeFlags.Number | TypeFlags.String)) {
                                        errorInfo = chainDiagnosticMessages(/* details */ undefined, Diagnostics.No_index_signature_with_a_parameter_of_type_0_was_found_on_type_1, typeToString(indexType), typeToString(objectType));
                                    }

                                    errorInfo = chainDiagnosticMessages(
                                        errorInfo,
                                        Diagnostics.Element_implicitly_has_an_any_type_because_expression_of_type_0_can_t_be_used_to_index_type_1, typeToString(fullIndexType), typeToString(objectType)
                                    );
                                    diagnostics.add(createDiagnosticForNodeFromMessageChain(accessExpression, errorInfo));
                                }
                            }
                        }
                    }
                    return undefined;
                }
            }
            if (isJSLiteralType(objectType)) {
                return anyType;
            }
            if (accessNode) {
                const indexNode = getIndexNodeForAccessExpression(accessNode);
                if (indexType.flags & (TypeFlags.StringLiteral | TypeFlags.NumberLiteral)) {
                    error(indexNode, Diagnostics.Property_0_does_not_exist_on_type_1, "" + (<StringLiteralType | NumberLiteralType>indexType).value, typeToString(objectType));
                }
                else if (indexType.flags & (TypeFlags.String | TypeFlags.Number)) {
                    error(indexNode, Diagnostics.Type_0_has_no_matching_index_signature_for_type_1, typeToString(objectType), typeToString(indexType));
                }
                else {
                    error(indexNode, Diagnostics.Type_0_cannot_be_used_as_an_index_type, typeToString(indexType));
                }
            }
            if (isTypeAny(indexType)) {
                return indexType;
            }
            return undefined;

            function errorIfWritingToReadonlyIndex(indexInfo: IndexInfo | undefined): void {
                if (indexInfo && indexInfo.isReadonly && accessExpression && (isAssignmentTarget(accessExpression) || isDeleteTarget(accessExpression))) {
                    error(accessExpression, Diagnostics.Index_signature_in_type_0_only_permits_reading, typeToString(objectType));
                }
            }
        }

        function getIndexNodeForAccessExpression(accessNode: ElementAccessExpression | IndexedAccessTypeNode | PropertyName | BindingName | SyntheticExpression) {
            return accessNode.kind === SyntaxKind.ElementAccessExpression ? accessNode.argumentExpression :
                accessNode.kind === SyntaxKind.IndexedAccessType ? accessNode.indexType :
                accessNode.kind === SyntaxKind.ComputedPropertyName ? accessNode.expression :
                accessNode;
        }

        function isGenericObjectType(type: Type): boolean {
            if (type.flags & TypeFlags.UnionOrIntersection) {
                if (!((<UnionOrIntersectionType>type).objectFlags & ObjectFlags.IsGenericObjectTypeComputed)) {
                    (<UnionOrIntersectionType>type).objectFlags |= ObjectFlags.IsGenericObjectTypeComputed |
                        (some((<UnionOrIntersectionType>type).types, isGenericObjectType) ? ObjectFlags.IsGenericObjectType : 0);
                }
                return !!((<UnionOrIntersectionType>type).objectFlags & ObjectFlags.IsGenericObjectType);
            }
            return !!(type.flags & TypeFlags.InstantiableNonPrimitive) || isGenericMappedType(type) || isGenericTupleType(type);
        }

        function isGenericIndexType(type: Type): boolean {
            if (type.flags & TypeFlags.UnionOrIntersection) {
                if (!((<UnionOrIntersectionType>type).objectFlags & ObjectFlags.IsGenericIndexTypeComputed)) {
                    (<UnionOrIntersectionType>type).objectFlags |= ObjectFlags.IsGenericIndexTypeComputed |
                        (some((<UnionOrIntersectionType>type).types, isGenericIndexType) ? ObjectFlags.IsGenericIndexType : 0);
                }
                return !!((<UnionOrIntersectionType>type).objectFlags & ObjectFlags.IsGenericIndexType);
            }
            return !!(type.flags & (TypeFlags.InstantiableNonPrimitive | TypeFlags.Index));
        }

        function isThisTypeParameter(type: Type): boolean {
            return !!(type.flags & TypeFlags.TypeParameter && (<TypeParameter>type).isThisType);
        }

        function getSimplifiedType(type: Type, writing: boolean): Type {
            return type.flags & TypeFlags.IndexedAccess ? getSimplifiedIndexedAccessType(<IndexedAccessType>type, writing) :
                type.flags & TypeFlags.Conditional ? getSimplifiedConditionalType(<ConditionalType>type, writing) :
                type;
        }

        function distributeIndexOverObjectType(objectType: Type, indexType: Type, writing: boolean) {
            // (T | U)[K] -> T[K] | U[K] (reading)
            // (T | U)[K] -> T[K] & U[K] (writing)
            // (T & U)[K] -> T[K] & U[K]
            if (objectType.flags & TypeFlags.UnionOrIntersection) {
                const types = map((objectType as UnionOrIntersectionType).types, t => getSimplifiedType(getIndexedAccessType(t, indexType), writing));
                return objectType.flags & TypeFlags.Intersection || writing ? getIntersectionType(types) : getUnionType(types);
            }
        }

        function distributeObjectOverIndexType(objectType: Type, indexType: Type, writing: boolean) {
            // T[A | B] -> T[A] | T[B] (reading)
            // T[A | B] -> T[A] & T[B] (writing)
            if (indexType.flags & TypeFlags.Union) {
                const types = map((indexType as UnionType).types, t => getSimplifiedType(getIndexedAccessType(objectType, t), writing));
                return writing ? getIntersectionType(types) : getUnionType(types);
            }
        }

        function unwrapSubstitution(type: Type): Type {
            if (type.flags & TypeFlags.Substitution) {
                return (type as SubstitutionType).substitute;
            }
            return type;
        }

        // Transform an indexed access to a simpler form, if possible. Return the simpler form, or return
        // the type itself if no transformation is possible. The writing flag indicates that the type is
        // the target of an assignment.
        function getSimplifiedIndexedAccessType(type: IndexedAccessType, writing: boolean): Type {
            const cache = writing ? "simplifiedForWriting" : "simplifiedForReading";
            if (type[cache]) {
                return type[cache] === circularConstraintType ? type : type[cache]!;
            }
            type[cache] = circularConstraintType;
            // We recursively simplify the object type as it may in turn be an indexed access type. For example, with
            // '{ [P in T]: { [Q in U]: number } }[T][U]' we want to first simplify the inner indexed access type.
            const objectType = unwrapSubstitution(getSimplifiedType(type.objectType, writing));
            const indexType = getSimplifiedType(type.indexType, writing);
            // T[A | B] -> T[A] | T[B] (reading)
            // T[A | B] -> T[A] & T[B] (writing)
            const distributedOverIndex = distributeObjectOverIndexType(objectType, indexType, writing);
            if (distributedOverIndex) {
                return type[cache] = distributedOverIndex;
            }
            // Only do the inner distributions if the index can no longer be instantiated to cause index distribution again
            if (!(indexType.flags & TypeFlags.Instantiable)) {
                // (T | U)[K] -> T[K] | U[K] (reading)
                // (T | U)[K] -> T[K] & U[K] (writing)
                // (T & U)[K] -> T[K] & U[K]
                const distributedOverObject = distributeIndexOverObjectType(objectType, indexType, writing);
                if (distributedOverObject) {
                    return type[cache] = distributedOverObject;
                }
            }
            // So ultimately (reading):
            // ((A & B) | C)[K1 | K2] -> ((A & B) | C)[K1] | ((A & B) | C)[K2] -> (A & B)[K1] | C[K1] | (A & B)[K2] | C[K2] -> (A[K1] & B[K1]) | C[K1] | (A[K2] & B[K2]) | C[K2]

            // A generic tuple type indexed by a number exists only when the index type doesn't select a
            // fixed element. We simplify to either the combined type of all elements (when the index type
            // the actual number type) or to the combined type of all non-fixed elements.
            if (isGenericTupleType(objectType) && indexType.flags & TypeFlags.NumberLike) {
                const elementType = getElementTypeOfSliceOfTupleType(objectType, indexType.flags & TypeFlags.Number ? 0 : objectType.target.fixedLength, /*endSkipCount*/ 0, writing);
                if (elementType) {
                    return type[cache] = elementType;
                }
            }
            // If the object type is a mapped type { [P in K]: E }, where K is generic, instantiate E using a mapper
            // that substitutes the index type for P. For example, for an index access { [P in K]: Box<T[P]> }[X], we
            // construct the type Box<T[X]>.
            if (isGenericMappedType(objectType)) {
                return type[cache] = mapType(substituteIndexedMappedType(objectType, type.indexType), t => getSimplifiedType(t, writing));
            }
            return type[cache] = type;
        }

        function getSimplifiedConditionalType(type: ConditionalType, writing: boolean) {
            const checkType = type.checkType;
            const extendsType = type.extendsType;
            const trueType = getTrueTypeFromConditionalType(type);
            const falseType = getFalseTypeFromConditionalType(type);
            // Simplifications for types of the form `T extends U ? T : never` and `T extends U ? never : T`.
            if (falseType.flags & TypeFlags.Never && getActualTypeVariable(trueType) === getActualTypeVariable(checkType)) {
                if (checkType.flags & TypeFlags.Any || isTypeAssignableTo(getRestrictiveInstantiation(checkType), getRestrictiveInstantiation(extendsType))) { // Always true
                    return getSimplifiedType(trueType, writing);
                }
                else if (isIntersectionEmpty(checkType, extendsType)) { // Always false
                    return neverType;
                }
            }
            else if (trueType.flags & TypeFlags.Never && getActualTypeVariable(falseType) === getActualTypeVariable(checkType)) {
                if (!(checkType.flags & TypeFlags.Any) && isTypeAssignableTo(getRestrictiveInstantiation(checkType), getRestrictiveInstantiation(extendsType))) { // Always true
                    return neverType;
                }
                else if (checkType.flags & TypeFlags.Any || isIntersectionEmpty(checkType, extendsType)) { // Always false
                    return getSimplifiedType(falseType, writing);
                }
            }
            return type;
        }

        /**
         * Invokes union simplification logic to determine if an intersection is considered empty as a union constituent
         */
        function isIntersectionEmpty(type1: Type, type2: Type) {
            return !!(getUnionType([intersectTypes(type1, type2), neverType]).flags & TypeFlags.Never);
        }

        function substituteIndexedMappedType(objectType: MappedType, index: Type) {
            const mapper = createTypeMapper([getTypeParameterFromMappedType(objectType)], [index]);
            const templateMapper = combineTypeMappers(objectType.mapper, mapper);
            return instantiateType(getTemplateTypeFromMappedType(objectType), templateMapper);
        }

        function getIndexedAccessType(objectType: Type, indexType: Type, accessNode?: ElementAccessExpression | IndexedAccessTypeNode | PropertyName | BindingName | SyntheticExpression, aliasSymbol?: Symbol, aliasTypeArguments?: readonly Type[]): Type {
            return getIndexedAccessTypeOrUndefined(objectType, indexType, accessNode, AccessFlags.None, aliasSymbol, aliasTypeArguments) || (accessNode ? errorType : unknownType);
        }

        function indexTypeLessThan(indexType: Type, limit: number) {
            return everyType(indexType, t => {
                if (t.flags & TypeFlags.StringOrNumberLiteral) {
                    const propName = getPropertyNameFromType(<StringLiteralType | NumberLiteralType>t);
                    if (isNumericLiteralName(propName)) {
                        const index = +propName;
                        return index >= 0 && index < limit;
                    }
                }
                return false;
            });
        }

        function getIndexedAccessTypeOrUndefined(objectType: Type, indexType: Type, accessNode?: ElementAccessExpression | IndexedAccessTypeNode | PropertyName | BindingName | SyntheticExpression, accessFlags = AccessFlags.None, aliasSymbol?: Symbol, aliasTypeArguments?: readonly Type[]): Type | undefined {
            if (objectType === wildcardType || indexType === wildcardType) {
                return wildcardType;
            }
            // If the object type has a string index signature and no other members we know that the result will
            // always be the type of that index signature and we can simplify accordingly.
            if (isStringIndexSignatureOnlyType(objectType) && !(indexType.flags & TypeFlags.Nullable) && isTypeAssignableToKind(indexType, TypeFlags.String | TypeFlags.Number)) {
                indexType = stringType;
            }
            // If the index type is generic, or if the object type is generic and doesn't originate in an expression and
            // the operation isn't exclusively indexing the fixed (non-variadic) portion of a tuple type, we are performing
            // a higher-order index access where we cannot meaningfully access the properties of the object type. Note that
            // for a generic T and a non-generic K, we eagerly resolve T[K] if it originates in an expression. This is to
            // preserve backwards compatibility. For example, an element access 'this["foo"]' has always been resolved
            // eagerly using the constraint type of 'this' at the given location.
            if (isGenericIndexType(indexType) || (accessNode && accessNode.kind !== SyntaxKind.IndexedAccessType ?
                isGenericTupleType(objectType) && !indexTypeLessThan(indexType, objectType.target.fixedLength) :
                isGenericObjectType(objectType) && !(isTupleType(objectType) && indexTypeLessThan(indexType, objectType.target.fixedLength)))) {
                if (objectType.flags & TypeFlags.AnyOrUnknown) {
                    return objectType;
                }
                // Defer the operation by creating an indexed access type.
                const id = objectType.id + "," + indexType.id;
                let type = indexedAccessTypes.get(id);
                if (!type) {
                    indexedAccessTypes.set(id, type = createIndexedAccessType(objectType, indexType, aliasSymbol, aliasTypeArguments));
                }
                return type;
            }
            // In the following we resolve T[K] to the type of the property in T selected by K.
            // We treat boolean as different from other unions to improve errors;
            // skipping straight to getPropertyTypeForIndexType gives errors with 'boolean' instead of 'true'.
            const apparentObjectType = getReducedApparentType(objectType);
            if (indexType.flags & TypeFlags.Union && !(indexType.flags & TypeFlags.Boolean)) {
                const propTypes: Type[] = [];
                let wasMissingProp = false;
                for (const t of (<UnionType>indexType).types) {
                    const propType = getPropertyTypeForIndexType(objectType, apparentObjectType, t, indexType, wasMissingProp, accessNode, accessFlags);
                    if (propType) {
                        propTypes.push(propType);
                    }
                    else if (!accessNode) {
                        // If there's no error node, we can immeditely stop, since error reporting is off
                        return undefined;
                    }
                    else {
                        // Otherwise we set a flag and return at the end of the loop so we still mark all errors
                        wasMissingProp = true;
                    }
                }
                if (wasMissingProp) {
                    return undefined;
                }
                return accessFlags & AccessFlags.Writing ? getIntersectionType(propTypes, aliasSymbol, aliasTypeArguments) : getUnionType(propTypes, UnionReduction.Literal, aliasSymbol, aliasTypeArguments);
            }
            return getPropertyTypeForIndexType(objectType, apparentObjectType, indexType, indexType, /* supressNoImplicitAnyError */ false, accessNode, accessFlags | AccessFlags.CacheSymbol, /* reportDeprecated */ true);
        }

        function getTypeFromIndexedAccessTypeNode(node: IndexedAccessTypeNode) {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                const objectType = getTypeFromTypeNode(node.objectType);
                const indexType = getTypeFromTypeNode(node.indexType);
                const potentialAlias = getAliasSymbolForTypeNode(node);
                const resolved = getIndexedAccessType(objectType, indexType, node, potentialAlias, getTypeArgumentsForAliasSymbol(potentialAlias));
                links.resolvedType = resolved.flags & TypeFlags.IndexedAccess &&
                    (<IndexedAccessType>resolved).objectType === objectType &&
                    (<IndexedAccessType>resolved).indexType === indexType ?
                    getConditionalFlowTypeOfType(resolved, node) : resolved;
            }
            return links.resolvedType;
        }

        function getTypeFromMappedTypeNode(node: MappedTypeNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                const type = <MappedType>createObjectType(ObjectFlags.Mapped, node.symbol);
                type.declaration = node;
                type.aliasSymbol = getAliasSymbolForTypeNode(node);
                type.aliasTypeArguments = getTypeArgumentsForAliasSymbol(type.aliasSymbol);
                links.resolvedType = type;
                // Eagerly resolve the constraint type which forces an error if the constraint type circularly
                // references itself through one or more type aliases.
                getConstraintTypeFromMappedType(type);
            }
            return links.resolvedType;
        }

        function getActualTypeVariable(type: Type): Type {
            if (type.flags & TypeFlags.Substitution) {
                return (<SubstitutionType>type).baseType;
            }
            if (type.flags & TypeFlags.IndexedAccess && (
                (<IndexedAccessType>type).objectType.flags & TypeFlags.Substitution ||
                (<IndexedAccessType>type).indexType.flags & TypeFlags.Substitution)) {
                return getIndexedAccessType(getActualTypeVariable((<IndexedAccessType>type).objectType), getActualTypeVariable((<IndexedAccessType>type).indexType));
            }
            return type;
        }

        function getConditionalType(root: ConditionalRoot, mapper: TypeMapper | undefined): Type {
            let result;
            let extraTypes: Type[] | undefined;
            // We loop here for an immediately nested conditional type in the false position, effectively treating
            // types of the form 'A extends B ? X : C extends D ? Y : E extends F ? Z : ...' as a single construct for
            // purposes of resolution. This means such types aren't subject to the instatiation depth limiter.
            while (true) {
                const checkType = instantiateType(root.checkType, mapper);
                const checkTypeInstantiable = isGenericObjectType(checkType) || isGenericIndexType(checkType);
                const extendsType = instantiateType(root.extendsType, mapper);
                if (checkType === wildcardType || extendsType === wildcardType) {
                    return wildcardType;
                }
                let combinedMapper: TypeMapper | undefined;
                if (root.inferTypeParameters) {
                    const context = createInferenceContext(root.inferTypeParameters, /*signature*/ undefined, InferenceFlags.None);
                    // We skip inference of the possible `infer` types unles the `extendsType` _is_ an infer type
                    // if it was, it's trivial to say that extendsType = checkType, however such a pattern is used to
                    // "reset" the type being build up during constraint calculation and avoid making an apparently "infinite" constraint
                    // so in those cases we refain from performing inference and retain the uninfered type parameter
                    if (!checkTypeInstantiable || !some(root.inferTypeParameters, t => t === extendsType)) {
                        // We don't want inferences from constraints as they may cause us to eagerly resolve the
                        // conditional type instead of deferring resolution. Also, we always want strict function
                        // types rules (i.e. proper contravariance) for inferences.
                        inferTypes(context.inferences, checkType, extendsType, InferencePriority.NoConstraints | InferencePriority.AlwaysStrict);
                    }
                    combinedMapper = mergeTypeMappers(mapper, context.mapper);
                }
                // Instantiate the extends type including inferences for 'infer T' type parameters
                const inferredExtendsType = combinedMapper ? instantiateType(root.extendsType, combinedMapper) : extendsType;
                // We attempt to resolve the conditional type only when the check and extends types are non-generic
                if (!checkTypeInstantiable && !isGenericObjectType(inferredExtendsType) && !isGenericIndexType(inferredExtendsType)) {
                    // Return falseType for a definitely false extends check. We check an instantiations of the two
                    // types with type parameters mapped to the wildcard type, the most permissive instantiations
                    // possible (the wildcard type is assignable to and from all types). If those are not related,
                    // then no instantiations will be and we can just return the false branch type.
                    if (!(inferredExtendsType.flags & TypeFlags.AnyOrUnknown) && (checkType.flags & TypeFlags.Any || !isTypeAssignableTo(getPermissiveInstantiation(checkType), getPermissiveInstantiation(inferredExtendsType)))) {
                        // Return union of trueType and falseType for 'any' since it matches anything
                        if (checkType.flags & TypeFlags.Any) {
                            (extraTypes || (extraTypes = [])).push(instantiateTypeWithoutDepthIncrease(root.trueType, combinedMapper || mapper));
                        }
                        // If falseType is an immediately nested conditional type that isn't distributive or has an
                        // identical checkType, switch to that type and loop.
                        const falseType = root.falseType;
                        if (falseType.flags & TypeFlags.Conditional) {
                            const newRoot = (<ConditionalType>falseType).root;
                            if (newRoot.node.parent === root.node && (!newRoot.isDistributive || newRoot.checkType === root.checkType)) {
                                root = newRoot;
                                continue;
                            }
                        }
                        result = instantiateTypeWithoutDepthIncrease(falseType, mapper);
                        break;
                    }
                    // Return trueType for a definitely true extends check. We check instantiations of the two
                    // types with type parameters mapped to their restrictive form, i.e. a form of the type parameter
                    // that has no constraint. This ensures that, for example, the type
                    //   type Foo<T extends { x: any }> = T extends { x: string } ? string : number
                    // doesn't immediately resolve to 'string' instead of being deferred.
                    if (inferredExtendsType.flags & TypeFlags.AnyOrUnknown || isTypeAssignableTo(getRestrictiveInstantiation(checkType), getRestrictiveInstantiation(inferredExtendsType))) {
                        result = instantiateTypeWithoutDepthIncrease(root.trueType, combinedMapper || mapper);
                        break;
                    }
                }
                // Return a deferred type for a check that is neither definitely true nor definitely false
                const erasedCheckType = getActualTypeVariable(checkType);
                result = <ConditionalType>createType(TypeFlags.Conditional);
                result.root = root;
                result.checkType = erasedCheckType;
                result.extendsType = extendsType;
                result.mapper = mapper;
                result.combinedMapper = combinedMapper;
                result.aliasSymbol = root.aliasSymbol;
                result.aliasTypeArguments = instantiateTypes(root.aliasTypeArguments, mapper!); // TODO: GH#18217
                break;
            }
            return extraTypes ? getUnionType(append(extraTypes, result)) : result;
        }

        function getTrueTypeFromConditionalType(type: ConditionalType) {
            return type.resolvedTrueType || (type.resolvedTrueType = instantiateType(type.root.trueType, type.mapper));
        }

        function getFalseTypeFromConditionalType(type: ConditionalType) {
            return type.resolvedFalseType || (type.resolvedFalseType = instantiateType(type.root.falseType, type.mapper));
        }

        function getInferredTrueTypeFromConditionalType(type: ConditionalType) {
            return type.resolvedInferredTrueType || (type.resolvedInferredTrueType = type.combinedMapper ? instantiateType(type.root.trueType, type.combinedMapper) : getTrueTypeFromConditionalType(type));
        }

        function getInferTypeParameters(node: ConditionalTypeNode): TypeParameter[] | undefined {
            let result: TypeParameter[] | undefined;
            if (node.locals) {
                node.locals.forEach(symbol => {
                    if (symbol.flags & SymbolFlags.TypeParameter) {
                        result = append(result, getDeclaredTypeOfSymbol(symbol));
                    }
                });
            }
            return result;
        }

        function getTypeFromConditionalTypeNode(node: ConditionalTypeNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                const checkType = getTypeFromTypeNode(node.checkType);
                const aliasSymbol = getAliasSymbolForTypeNode(node);
                const aliasTypeArguments = getTypeArgumentsForAliasSymbol(aliasSymbol);
                const allOuterTypeParameters = getOuterTypeParameters(node, /*includeThisTypes*/ true);
                const outerTypeParameters = aliasTypeArguments ? allOuterTypeParameters : filter(allOuterTypeParameters, tp => isTypeParameterPossiblyReferenced(tp, node));
                const root: ConditionalRoot = {
                    node,
                    checkType,
                    extendsType: getTypeFromTypeNode(node.extendsType),
                    trueType: getTypeFromTypeNode(node.trueType),
                    falseType: getTypeFromTypeNode(node.falseType),
                    isDistributive: !!(checkType.flags & TypeFlags.TypeParameter),
                    inferTypeParameters: getInferTypeParameters(node),
                    outerTypeParameters,
                    instantiations: undefined,
                    aliasSymbol,
                    aliasTypeArguments
                };
                links.resolvedType = getConditionalType(root, /*mapper*/ undefined);
                if (outerTypeParameters) {
                    root.instantiations = new Map<string, Type>();
                    root.instantiations.set(getTypeListId(outerTypeParameters), links.resolvedType);
                }
            }
            return links.resolvedType;
        }

        function getTypeFromInferTypeNode(node: InferTypeNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                links.resolvedType = getDeclaredTypeOfTypeParameter(getSymbolOfNode(node.typeParameter));
            }
            return links.resolvedType;
        }

        function getIdentifierChain(node: EntityName): Identifier[] {
            if (isIdentifier(node)) {
                return [node];
            }
            else {
                return append(getIdentifierChain(node.left), node.right);
            }
        }

        function getTypeFromImportTypeNode(node: ImportTypeNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                if (node.isTypeOf && node.typeArguments) { // Only the non-typeof form can make use of type arguments
                    error(node, Diagnostics.Type_arguments_cannot_be_used_here);
                    links.resolvedSymbol = unknownSymbol;
                    return links.resolvedType = errorType;
                }
                if (!isLiteralImportTypeNode(node)) {
                    error(node.argument, Diagnostics.String_literal_expected);
                    links.resolvedSymbol = unknownSymbol;
                    return links.resolvedType = errorType;
                }
                const targetMeaning = node.isTypeOf ? SymbolFlags.Value : node.flags & NodeFlags.JSDoc ? SymbolFlags.Value | SymbolFlags.Type : SymbolFlags.Type;
                // TODO: Future work: support unions/generics/whatever via a deferred import-type
                const innerModuleSymbol = resolveExternalModuleName(node, node.argument.literal);
                if (!innerModuleSymbol) {
                    links.resolvedSymbol = unknownSymbol;
                    return links.resolvedType = errorType;
                }
                const moduleSymbol = resolveExternalModuleSymbol(innerModuleSymbol, /*dontResolveAlias*/ false);
                if (!nodeIsMissing(node.qualifier)) {
                    const nameStack: Identifier[] = getIdentifierChain(node.qualifier!);
                    let currentNamespace = moduleSymbol;
                    let current: Identifier | undefined;
                    while (current = nameStack.shift()) {
                        const meaning = nameStack.length ? SymbolFlags.Namespace : targetMeaning;
                        const next = getSymbol(getExportsOfSymbol(getMergedSymbol(resolveSymbol(currentNamespace))), current.escapedText, meaning);
                        if (!next) {
                            error(current, Diagnostics.Namespace_0_has_no_exported_member_1, getFullyQualifiedName(currentNamespace), declarationNameToString(current));
                            return links.resolvedType = errorType;
                        }
                        getNodeLinks(current).resolvedSymbol = next;
                        getNodeLinks(current.parent).resolvedSymbol = next;
                        currentNamespace = next;
                    }
                    links.resolvedType = resolveImportSymbolType(node, links, currentNamespace, targetMeaning);
                }
                else {
                    if (moduleSymbol.flags & targetMeaning) {
                        links.resolvedType = resolveImportSymbolType(node, links, moduleSymbol, targetMeaning);
                    }
                    else {
                        const errorMessage = targetMeaning === SymbolFlags.Value
                            ? Diagnostics.Module_0_does_not_refer_to_a_value_but_is_used_as_a_value_here
                            : Diagnostics.Module_0_does_not_refer_to_a_type_but_is_used_as_a_type_here_Did_you_mean_typeof_import_0;

                        error(node, errorMessage, node.argument.literal.text);

                        links.resolvedSymbol = unknownSymbol;
                        links.resolvedType = errorType;
                    }
                }
            }
            return links.resolvedType;
        }

        function resolveImportSymbolType(node: ImportTypeNode, links: NodeLinks, symbol: Symbol, meaning: SymbolFlags) {
            const resolvedSymbol = resolveSymbol(symbol);
            links.resolvedSymbol = resolvedSymbol;
            if (meaning === SymbolFlags.Value) {
                return getTypeOfSymbol(symbol); // intentionally doesn't use resolved symbol so type is cached as expected on the alias
            }
            else {
                return getTypeReferenceType(node, resolvedSymbol); // getTypeReferenceType doesn't handle aliases - it must get the resolved symbol
            }
        }

        function getTypeFromTypeLiteralOrFunctionOrConstructorTypeNode(node: TypeNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                // Deferred resolution of members is handled by resolveObjectTypeMembers
                const aliasSymbol = getAliasSymbolForTypeNode(node);
                if (getMembersOfSymbol(node.symbol).size === 0 && !aliasSymbol) {
                    links.resolvedType = emptyTypeLiteralType;
                }
                else {
                    let type = createObjectType(ObjectFlags.Anonymous, node.symbol);
                    type.aliasSymbol = aliasSymbol;
                    type.aliasTypeArguments = getTypeArgumentsForAliasSymbol(aliasSymbol);
                    if (isJSDocTypeLiteral(node) && node.isArrayType) {
                        type = createArrayType(type);
                    }
                    links.resolvedType = type;
                }
            }
            return links.resolvedType;
        }

        function getAliasSymbolForTypeNode(node: Node) {
            let host = node.parent;
            while (isParenthesizedTypeNode(host) || isTypeOperatorNode(host) && host.operator === SyntaxKind.ReadonlyKeyword) {
                host = host.parent;
            }
            return isTypeAlias(host) ? getSymbolOfNode(host) : undefined;
        }

        function getTypeArgumentsForAliasSymbol(symbol: Symbol | undefined) {
            return symbol ? getLocalTypeParametersOfClassOrInterfaceOrTypeAlias(symbol) : undefined;
        }

        function isNonGenericObjectType(type: Type) {
            return !!(type.flags & TypeFlags.Object) && !isGenericMappedType(type);
        }

        function isEmptyObjectTypeOrSpreadsIntoEmptyObject(type: Type) {
            return isEmptyObjectType(type) || !!(type.flags & (TypeFlags.Null | TypeFlags.Undefined | TypeFlags.BooleanLike | TypeFlags.NumberLike | TypeFlags.BigIntLike | TypeFlags.StringLike | TypeFlags.EnumLike | TypeFlags.NonPrimitive | TypeFlags.Index));
        }

        function isSinglePropertyAnonymousObjectType(type: Type) {
            return !!(type.flags & TypeFlags.Object) &&
                !!(getObjectFlags(type) & ObjectFlags.Anonymous) &&
                (length(getPropertiesOfType(type)) === 1 || every(getPropertiesOfType(type), p => !!(p.flags & SymbolFlags.Optional)));
        }

        function tryMergeUnionOfObjectTypeAndEmptyObject(type: UnionType, readonly: boolean): Type | undefined {
            if (type.types.length === 2) {
                const firstType = type.types[0];
                const secondType = type.types[1];
                if (every(type.types, isEmptyObjectTypeOrSpreadsIntoEmptyObject)) {
                    return isEmptyObjectType(firstType) ? firstType : isEmptyObjectType(secondType) ? secondType : emptyObjectType;
                }
                if (isEmptyObjectTypeOrSpreadsIntoEmptyObject(firstType) && isSinglePropertyAnonymousObjectType(secondType)) {
                    return getAnonymousPartialType(secondType);
                }
                if (isEmptyObjectTypeOrSpreadsIntoEmptyObject(secondType) && isSinglePropertyAnonymousObjectType(firstType)) {
                    return getAnonymousPartialType(firstType);
                }
            }

            function getAnonymousPartialType(type: Type) {
                // gets the type as if it had been spread, but where everything in the spread is made optional
                const members = createSymbolTable();
                for (const prop of getPropertiesOfType(type)) {
                    if (getDeclarationModifierFlagsFromSymbol(prop) & (ModifierFlags.Private | ModifierFlags.Protected)) {
                        // do nothing, skip privates
                    }
                    else if (isSpreadableProperty(prop)) {
                        const isSetonlyAccessor = prop.flags & SymbolFlags.SetAccessor && !(prop.flags & SymbolFlags.GetAccessor);
                        const flags = SymbolFlags.Property | SymbolFlags.Optional;
                        const result = createSymbol(flags, prop.escapedName, readonly ? CheckFlags.Readonly : 0);
                        result.type = isSetonlyAccessor ? undefinedType : getTypeOfSymbol(prop);
                        result.declarations = prop.declarations;
                        result.nameType = getSymbolLinks(prop).nameType;
                        result.syntheticOrigin = prop;
                        members.set(prop.escapedName, result);
                    }
                }
                const spread = createAnonymousType(
                    type.symbol,
                    members,
                    emptyArray,
                    emptyArray,
                    getIndexInfoOfType(type, IndexKind.String),
                    getIndexInfoOfType(type, IndexKind.Number));
                spread.objectFlags |= ObjectFlags.ObjectLiteral | ObjectFlags.ContainsObjectOrArrayLiteral;
                return spread;
            }
        }

        /**
         * Since the source of spread types are object literals, which are not binary,
         * this function should be called in a left folding style, with left = previous result of getSpreadType
         * and right = the new element to be spread.
         */
        function getSpreadType(left: Type, right: Type, symbol: Symbol | undefined, objectFlags: ObjectFlags, readonly: boolean): Type {
            if (left.flags & TypeFlags.Any || right.flags & TypeFlags.Any) {
                return anyType;
            }
            if (left.flags & TypeFlags.Unknown || right.flags & TypeFlags.Unknown) {
                return unknownType;
            }
            if (left.flags & TypeFlags.Never) {
                return right;
            }
            if (right.flags & TypeFlags.Never) {
                return left;
            }
            if (left.flags & TypeFlags.Union) {
                const merged = tryMergeUnionOfObjectTypeAndEmptyObject(left as UnionType, readonly);
                if (merged) {
                    return getSpreadType(merged, right, symbol, objectFlags, readonly);
                }
                return mapType(left, t => getSpreadType(t, right, symbol, objectFlags, readonly));
            }
            if (right.flags & TypeFlags.Union) {
                const merged = tryMergeUnionOfObjectTypeAndEmptyObject(right as UnionType, readonly);
                if (merged) {
                    return getSpreadType(left, merged, symbol, objectFlags, readonly);
                }
                return mapType(right, t => getSpreadType(left, t, symbol, objectFlags, readonly));
            }
            if (right.flags & (TypeFlags.BooleanLike | TypeFlags.NumberLike | TypeFlags.BigIntLike | TypeFlags.StringLike | TypeFlags.EnumLike | TypeFlags.NonPrimitive | TypeFlags.Index)) {
                return left;
            }

            if (isGenericObjectType(left) || isGenericObjectType(right)) {
                if (isEmptyObjectType(left)) {
                    return right;
                }
                // When the left type is an intersection, we may need to merge the last constituent of the
                // intersection with the right type. For example when the left type is 'T & { a: string }'
                // and the right type is '{ b: string }' we produce 'T & { a: string, b: string }'.
                if (left.flags & TypeFlags.Intersection) {
                    const types = (<IntersectionType>left).types;
                    const lastLeft = types[types.length - 1];
                    if (isNonGenericObjectType(lastLeft) && isNonGenericObjectType(right)) {
                        return getIntersectionType(concatenate(types.slice(0, types.length - 1), [getSpreadType(lastLeft, right, symbol, objectFlags, readonly)]));
                    }
                }
                return getIntersectionType([left, right]);
            }

            const members = createSymbolTable();
            const skippedPrivateMembers = new Set<__String>();
            let stringIndexInfo: IndexInfo | undefined;
            let numberIndexInfo: IndexInfo | undefined;
            if (left === emptyObjectType) {
                // for the first spread element, left === emptyObjectType, so take the right's string indexer
                stringIndexInfo = getIndexInfoOfType(right, IndexKind.String);
                numberIndexInfo = getIndexInfoOfType(right, IndexKind.Number);
            }
            else {
                stringIndexInfo = unionSpreadIndexInfos(getIndexInfoOfType(left, IndexKind.String), getIndexInfoOfType(right, IndexKind.String));
                numberIndexInfo = unionSpreadIndexInfos(getIndexInfoOfType(left, IndexKind.Number), getIndexInfoOfType(right, IndexKind.Number));
            }

            for (const rightProp of getPropertiesOfType(right)) {
                if (getDeclarationModifierFlagsFromSymbol(rightProp) & (ModifierFlags.Private | ModifierFlags.Protected)) {
                    skippedPrivateMembers.add(rightProp.escapedName);
                }
                else if (isSpreadableProperty(rightProp)) {
                    members.set(rightProp.escapedName, getSpreadSymbol(rightProp, readonly));
                }
            }

            for (const leftProp of getPropertiesOfType(left)) {
                if (skippedPrivateMembers.has(leftProp.escapedName) || !isSpreadableProperty(leftProp)) {
                    continue;
                }
                if (members.has(leftProp.escapedName)) {
                    const rightProp = members.get(leftProp.escapedName)!;
                    const rightType = getTypeOfSymbol(rightProp);
                    if (rightProp.flags & SymbolFlags.Optional) {
                        const declarations = concatenate(leftProp.declarations, rightProp.declarations);
                        const flags = SymbolFlags.Property | (leftProp.flags & SymbolFlags.Optional);
                        const result = createSymbol(flags, leftProp.escapedName);
                        result.type = getUnionType([getTypeOfSymbol(leftProp), getTypeWithFacts(rightType, TypeFacts.NEUndefined)]);
                        result.leftSpread = leftProp;
                        result.rightSpread = rightProp;
                        result.declarations = declarations;
                        result.nameType = getSymbolLinks(leftProp).nameType;
                        members.set(leftProp.escapedName, result);
                    }
                }
                else {
                    members.set(leftProp.escapedName, getSpreadSymbol(leftProp, readonly));
                }
            }

            const spread = createAnonymousType(
                symbol,
                members,
                emptyArray,
                emptyArray,
                getIndexInfoWithReadonly(stringIndexInfo, readonly),
                getIndexInfoWithReadonly(numberIndexInfo, readonly));
            spread.objectFlags |= ObjectFlags.ObjectLiteral | ObjectFlags.ContainsObjectOrArrayLiteral | ObjectFlags.ContainsSpread | objectFlags;
            return spread;
        }

        /** We approximate own properties as non-methods plus methods that are inside the object literal */
        function isSpreadableProperty(prop: Symbol): boolean {
            return !some(prop.declarations, isPrivateIdentifierPropertyDeclaration) &&
                (!(prop.flags & (SymbolFlags.Method | SymbolFlags.GetAccessor | SymbolFlags.SetAccessor)) ||
                    !prop.declarations.some(decl => isClassLike(decl.parent)));
        }

        function getSpreadSymbol(prop: Symbol, readonly: boolean) {
            const isSetonlyAccessor = prop.flags & SymbolFlags.SetAccessor && !(prop.flags & SymbolFlags.GetAccessor);
            if (!isSetonlyAccessor && readonly === isReadonlySymbol(prop)) {
                return prop;
            }
            const flags = SymbolFlags.Property | (prop.flags & SymbolFlags.Optional);
            const result = createSymbol(flags, prop.escapedName, readonly ? CheckFlags.Readonly : 0);
            result.type = isSetonlyAccessor ? undefinedType : getTypeOfSymbol(prop);
            result.declarations = prop.declarations;
            result.nameType = getSymbolLinks(prop).nameType;
            result.syntheticOrigin = prop;
            return result;
        }

        function getIndexInfoWithReadonly(info: IndexInfo | undefined, readonly: boolean) {
            return info && info.isReadonly !== readonly ? createIndexInfo(info.type, readonly, info.declaration) : info;
        }

        function createLiteralType(flags: TypeFlags, value: string | number | PseudoBigInt, symbol: Symbol | undefined) {
            const type = <LiteralType>createType(flags);
            type.symbol = symbol!;
            type.value = value;
            return type;
        }

        function getFreshTypeOfLiteralType(type: Type): Type {
            if (type.flags & TypeFlags.Literal) {
                if (!(<LiteralType>type).freshType) {
                    const freshType = createLiteralType(type.flags, (<LiteralType>type).value, (<LiteralType>type).symbol);
                    freshType.regularType = <LiteralType>type;
                    freshType.freshType = freshType;
                    (<LiteralType>type).freshType = freshType;
                }
                return (<LiteralType>type).freshType;
            }
            return type;
        }

        function getRegularTypeOfLiteralType(type: Type): Type {
            return type.flags & TypeFlags.Literal ? (<LiteralType>type).regularType :
                type.flags & TypeFlags.Union ? ((<UnionType>type).regularType || ((<UnionType>type).regularType = getUnionType(sameMap((<UnionType>type).types, getRegularTypeOfLiteralType)) as UnionType)) :
                type;
        }

        function isFreshLiteralType(type: Type) {
            return !!(type.flags & TypeFlags.Literal) && (<LiteralType>type).freshType === type;
        }

        function getLiteralType(value: string | number | PseudoBigInt, enumId?: number, symbol?: Symbol) {
            // We store all literal types in a single map with keys of the form '#NNN' and '@SSS',
            // where NNN is the text representation of a numeric literal and SSS are the characters
            // of a string literal. For literal enum members we use 'EEE#NNN' and 'EEE@SSS', where
            // EEE is a unique id for the containing enum type.
            const qualifier = typeof value === "number" ? "#" : typeof value === "string" ? "@" : "n";
            const key = (enumId ? enumId : "") + qualifier + (typeof value === "object" ? pseudoBigIntToString(value) : value);
            let type = literalTypes.get(key);
            if (!type) {
                const flags = (typeof value === "number" ? TypeFlags.NumberLiteral :
                    typeof value === "string" ? TypeFlags.StringLiteral : TypeFlags.BigIntLiteral) |
                    (enumId ? TypeFlags.EnumLiteral : 0);
                literalTypes.set(key, type = createLiteralType(flags, value, symbol));
                type.regularType = type;
            }
            return type;
        }

        function getTypeFromLiteralTypeNode(node: LiteralTypeNode): Type {
            if (node.literal.kind === SyntaxKind.NullKeyword) {
                return nullType;
            }
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                links.resolvedType = getRegularTypeOfLiteralType(checkExpression(node.literal));
            }
            return links.resolvedType;
        }

        function createUniqueESSymbolType(symbol: Symbol) {
            const type = <UniqueESSymbolType>createType(TypeFlags.UniqueESSymbol);
            type.symbol = symbol;
            type.escapedName = `__@${type.symbol.escapedName}@${getSymbolId(type.symbol)}` as __String;
            return type;
        }

        function getESSymbolLikeTypeForNode(node: Node) {
            if (isValidESSymbolDeclaration(node)) {
                const symbol = getSymbolOfNode(node);
                const links = getSymbolLinks(symbol);
                return links.uniqueESSymbolType || (links.uniqueESSymbolType = createUniqueESSymbolType(symbol));
            }
            return esSymbolType;
        }

        function getThisType(node: Node): Type {
            const container = getThisContainer(node, /*includeArrowFunctions*/ false);
            const parent = container && container.parent;
            if (parent && (isClassLike(parent) || parent.kind === SyntaxKind.InterfaceDeclaration)) {
                if (!hasSyntacticModifier(container, ModifierFlags.Static) &&
                    (!isConstructorDeclaration(container) || isNodeDescendantOf(node, container.body))) {
                    return getDeclaredTypeOfClassOrInterface(getSymbolOfNode(parent as ClassLikeDeclaration | InterfaceDeclaration)).thisType!;
                }
            }

            // inside x.prototype = { ... }
            if (parent && isObjectLiteralExpression(parent) && isBinaryExpression(parent.parent) && getAssignmentDeclarationKind(parent.parent) === AssignmentDeclarationKind.Prototype) {
                return getDeclaredTypeOfClassOrInterface(getSymbolOfNode(parent.parent.left)!.parent!).thisType!;
            }
            // /** @return {this} */
            // x.prototype.m = function() { ... }
            const host = node.flags & NodeFlags.JSDoc ? getHostSignatureFromJSDoc(node) : undefined;
            if (host && isFunctionExpression(host) && isBinaryExpression(host.parent) && getAssignmentDeclarationKind(host.parent) === AssignmentDeclarationKind.PrototypeProperty) {
                return getDeclaredTypeOfClassOrInterface(getSymbolOfNode(host.parent.left)!.parent!).thisType!;
            }
            // inside constructor function C() { ... }
            if (isJSConstructor(container) && isNodeDescendantOf(node, container.body)) {
                return getDeclaredTypeOfClassOrInterface(getSymbolOfNode(container)).thisType!;
            }
            error(node, Diagnostics.A_this_type_is_available_only_in_a_non_static_member_of_a_class_or_interface);
            return errorType;
        }

        function getTypeFromThisTypeNode(node: ThisExpression | ThisTypeNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                links.resolvedType = getThisType(node);
            }
            return links.resolvedType;
        }

        function getTypeFromRestTypeNode(node: RestTypeNode | NamedTupleMember) {
            return getTypeFromTypeNode(getArrayElementTypeNode(node.type) || node.type);
        }

        function getArrayElementTypeNode(node: TypeNode): TypeNode | undefined {
            switch (node.kind) {
                case SyntaxKind.ParenthesizedType:
                    return getArrayElementTypeNode((node as ParenthesizedTypeNode).type);
                case SyntaxKind.TupleType:
                    if ((node as TupleTypeNode).elements.length === 1) {
                        node = (node as TupleTypeNode).elements[0];
                        if (node.kind === SyntaxKind.RestType || node.kind === SyntaxKind.NamedTupleMember && (node as NamedTupleMember).dotDotDotToken) {
                            return getArrayElementTypeNode((node as RestTypeNode | NamedTupleMember).type);
                        }
                    }
                    break;
                case SyntaxKind.ArrayType:
                    return (node as ArrayTypeNode).elementType;
            }
            return undefined;
        }

        function getTypeFromNamedTupleTypeNode(node: NamedTupleMember): Type {
            const links = getNodeLinks(node);
            return links.resolvedType || (links.resolvedType =
                    node.dotDotDotToken ? getTypeFromRestTypeNode(node) :
                    node.questionToken && strictNullChecks ? getOptionalType(getTypeFromTypeNode(node.type)) :
                    getTypeFromTypeNode(node.type));
        }

        function getTypeFromTypeNode(node: TypeNode): Type {
            return getConditionalFlowTypeOfType(getTypeFromTypeNodeWorker(node), node);
        }

        function getTypeFromTypeNodeWorker(node: TypeNode): Type {
            switch (node.kind) {
                case SyntaxKind.AnyKeyword:
                case SyntaxKind.JSDocAllType:
                case SyntaxKind.JSDocUnknownType:
                    return anyType;
                case SyntaxKind.UnknownKeyword:
                    return unknownType;
                case SyntaxKind.StringKeyword:
                    return stringType;
                case SyntaxKind.NumberKeyword:
                    return numberType;
                case SyntaxKind.BigIntKeyword:
                    return bigintType;
                case SyntaxKind.BooleanKeyword:
                    return booleanType;
                case SyntaxKind.SymbolKeyword:
                    return esSymbolType;
                case SyntaxKind.VoidKeyword:
                    return voidType;
                case SyntaxKind.UndefinedKeyword:
                    return undefinedType;
                case SyntaxKind.NullKeyword as TypeNodeSyntaxKind:
                    // TODO(rbuckton): `NullKeyword` is no longer a `TypeNode`, but we defensively allow it here because of incorrect casts in the Language Service.
                    return nullType;
                case SyntaxKind.NeverKeyword:
                    return neverType;
                case SyntaxKind.ObjectKeyword:
                    return node.flags & NodeFlags.JavaScriptFile && !noImplicitAny ? anyType : nonPrimitiveType;
                case SyntaxKind.ThisType:
                case SyntaxKind.ThisKeyword as TypeNodeSyntaxKind:
                    // TODO(rbuckton): `ThisKeyword` is no longer a `TypeNode`, but we defensively allow it here because of incorrect casts in the Language Service and because of `isPartOfTypeNode`.
                    return getTypeFromThisTypeNode(node as ThisExpression | ThisTypeNode);
                case SyntaxKind.LiteralType:
                    return getTypeFromLiteralTypeNode(<LiteralTypeNode>node);
                case SyntaxKind.TypeReference:
                    return getTypeFromTypeReference(<TypeReferenceNode>node);
                case SyntaxKind.TypePredicate:
                    return (<TypePredicateNode>node).assertsModifier ? voidType : booleanType;
                case SyntaxKind.ExpressionWithTypeArguments:
                    return getTypeFromTypeReference(<ExpressionWithTypeArguments>node);
                case SyntaxKind.TypeQuery:
                    return getTypeFromTypeQueryNode(<TypeQueryNode>node);
                case SyntaxKind.ArrayType:
                case SyntaxKind.TupleType:
                    return getTypeFromArrayOrTupleTypeNode(<ArrayTypeNode | TupleTypeNode>node);
                case SyntaxKind.OptionalType:
                    return getTypeFromOptionalTypeNode(<OptionalTypeNode>node);
                case SyntaxKind.UnionType:
                    return getTypeFromUnionTypeNode(<UnionTypeNode>node);
                case SyntaxKind.IntersectionType:
                    return getTypeFromIntersectionTypeNode(<IntersectionTypeNode>node);
                case SyntaxKind.JSDocNullableType:
                    return getTypeFromJSDocNullableTypeNode(<JSDocNullableType>node);
                case SyntaxKind.JSDocOptionalType:
                    return addOptionality(getTypeFromTypeNode((node as JSDocOptionalType).type));
                case SyntaxKind.NamedTupleMember:
                    return getTypeFromNamedTupleTypeNode(node as NamedTupleMember);
                case SyntaxKind.ParenthesizedType:
                case SyntaxKind.JSDocNonNullableType:
                case SyntaxKind.JSDocTypeExpression:
                    return getTypeFromTypeNode((<ParenthesizedTypeNode | JSDocTypeReferencingNode | JSDocTypeExpression | NamedTupleMember>node).type);
                case SyntaxKind.RestType:
                    return getTypeFromRestTypeNode(<RestTypeNode>node);
                case SyntaxKind.JSDocVariadicType:
                    return getTypeFromJSDocVariadicType(node as JSDocVariadicType);
                case SyntaxKind.FunctionType:
                case SyntaxKind.ConstructorType:
                case SyntaxKind.TypeLiteral:
                case SyntaxKind.JSDocTypeLiteral:
                case SyntaxKind.JSDocFunctionType:
                case SyntaxKind.JSDocSignature:
                    return getTypeFromTypeLiteralOrFunctionOrConstructorTypeNode(node);
                case SyntaxKind.TypeOperator:
                    return getTypeFromTypeOperatorNode(<TypeOperatorNode>node);
                case SyntaxKind.IndexedAccessType:
                    return getTypeFromIndexedAccessTypeNode(<IndexedAccessTypeNode>node);
                case SyntaxKind.MappedType:
                    return getTypeFromMappedTypeNode(<MappedTypeNode>node);
                case SyntaxKind.ConditionalType:
                    return getTypeFromConditionalTypeNode(<ConditionalTypeNode>node);
                case SyntaxKind.InferType:
                    return getTypeFromInferTypeNode(<InferTypeNode>node);
                case SyntaxKind.ImportType:
                    return getTypeFromImportTypeNode(<ImportTypeNode>node);
                // This function assumes that an identifier, qualified name, or property access expression is a type expression
                // Callers should first ensure this by calling `isPartOfTypeNode`
                // TODO(rbuckton): These aren't valid TypeNodes, but we treat them as such because of `isPartOfTypeNode`, which returns `true` for things that aren't `TypeNode`s.
                case SyntaxKind.Identifier as TypeNodeSyntaxKind:
                case SyntaxKind.QualifiedName as TypeNodeSyntaxKind:
                case SyntaxKind.PropertyAccessExpression as TypeNodeSyntaxKind:
                    const symbol = getSymbolAtLocation(node);
                    return symbol ? getDeclaredTypeOfSymbol(symbol) : errorType;
                default:
                    return errorType;
            }
        }

        function instantiateList<T>(items: readonly T[], mapper: TypeMapper, instantiator: (item: T, mapper: TypeMapper) => T): readonly T[];
        function instantiateList<T>(items: readonly T[] | undefined, mapper: TypeMapper, instantiator: (item: T, mapper: TypeMapper) => T): readonly T[] | undefined;
        function instantiateList<T>(items: readonly T[] | undefined, mapper: TypeMapper, instantiator: (item: T, mapper: TypeMapper) => T): readonly T[] | undefined {
            if (items && items.length) {
                for (let i = 0; i < items.length; i++) {
                    const item = items[i];
                    const mapped = instantiator(item, mapper);
                    if (item !== mapped) {
                        const result = i === 0 ? [] : items.slice(0, i);
                        result.push(mapped);
                        for (i++; i < items.length; i++) {
                            result.push(instantiator(items[i], mapper));
                        }
                        return result;
                    }
                }
            }
            return items;
        }

        function instantiateTypes(types: readonly Type[], mapper: TypeMapper): readonly Type[];
        function instantiateTypes(types: readonly Type[] | undefined, mapper: TypeMapper): readonly Type[] | undefined;
        function instantiateTypes(types: readonly Type[] | undefined, mapper: TypeMapper): readonly Type[] | undefined {
            return instantiateList<Type>(types, mapper, instantiateType);
        }

        function instantiateSignatures(signatures: readonly Signature[], mapper: TypeMapper): readonly Signature[] {
            return instantiateList<Signature>(signatures, mapper, instantiateSignature);
        }

        function createTypeMapper(sources: readonly TypeParameter[], targets: readonly Type[] | undefined): TypeMapper {
            return sources.length === 1 ? makeUnaryTypeMapper(sources[0], targets ? targets[0] : anyType) : makeArrayTypeMapper(sources, targets);
        }

        function getMappedType(type: Type, mapper: TypeMapper): Type {
            switch (mapper.kind) {
                case TypeMapKind.Simple:
                    return type === mapper.source ? mapper.target : type;
                case TypeMapKind.Array:
                    const sources = mapper.sources;
                    const targets = mapper.targets;
                    for (let i = 0; i < sources.length; i++) {
                        if (type === sources[i]) {
                            return targets ? targets[i] : anyType;
                        }
                    }
                    return type;
                case TypeMapKind.Function:
                    return mapper.func(type);
                case TypeMapKind.Composite:
                case TypeMapKind.Merged:
                    const t1 = getMappedType(type, mapper.mapper1);
                    return t1 !== type && mapper.kind === TypeMapKind.Composite ? instantiateType(t1, mapper.mapper2) : getMappedType(t1, mapper.mapper2);
            }
        }

        function makeUnaryTypeMapper(source: Type, target: Type): TypeMapper {
            return { kind: TypeMapKind.Simple, source, target };
        }

        function makeArrayTypeMapper(sources: readonly TypeParameter[], targets: readonly Type[] | undefined): TypeMapper {
            return { kind: TypeMapKind.Array, sources, targets };
        }

        function makeFunctionTypeMapper(func: (t: Type) => Type): TypeMapper {
            return { kind: TypeMapKind.Function, func };
        }

        function makeCompositeTypeMapper(kind: TypeMapKind.Composite | TypeMapKind.Merged, mapper1: TypeMapper, mapper2: TypeMapper): TypeMapper {
            return { kind, mapper1, mapper2 };
        }

        function createTypeEraser(sources: readonly TypeParameter[]): TypeMapper {
            return createTypeMapper(sources, /*targets*/ undefined);
        }

        /**
         * Maps forward-references to later types parameters to the empty object type.
         * This is used during inference when instantiating type parameter defaults.
         */
        function createBackreferenceMapper(context: InferenceContext, index: number): TypeMapper {
            return makeFunctionTypeMapper(t => findIndex(context.inferences, info => info.typeParameter === t) >= index ? unknownType : t);
        }

        function combineTypeMappers(mapper1: TypeMapper | undefined, mapper2: TypeMapper): TypeMapper {
            return mapper1 ? makeCompositeTypeMapper(TypeMapKind.Composite, mapper1, mapper2) : mapper2;
        }

        function mergeTypeMappers(mapper1: TypeMapper | undefined, mapper2: TypeMapper): TypeMapper {
            return mapper1 ? makeCompositeTypeMapper(TypeMapKind.Merged, mapper1, mapper2) : mapper2;
        }

        function prependTypeMapping(source: Type, target: Type, mapper: TypeMapper | undefined) {
            return !mapper ? makeUnaryTypeMapper(source, target) : makeCompositeTypeMapper(TypeMapKind.Merged, makeUnaryTypeMapper(source, target), mapper);
        }

        function appendTypeMapping(mapper: TypeMapper | undefined, source: Type, target: Type) {
            return !mapper ? makeUnaryTypeMapper(source, target) : makeCompositeTypeMapper(TypeMapKind.Merged, mapper, makeUnaryTypeMapper(source, target));
        }

        function getRestrictiveTypeParameter(tp: TypeParameter) {
            return tp.constraint === unknownType ? tp : tp.restrictiveInstantiation || (
                tp.restrictiveInstantiation = createTypeParameter(tp.symbol),
                (tp.restrictiveInstantiation as TypeParameter).constraint = unknownType,
                tp.restrictiveInstantiation
            );
        }

        function cloneTypeParameter(typeParameter: TypeParameter): TypeParameter {
            const result = createTypeParameter(typeParameter.symbol);
            result.target = typeParameter;
            return result;
        }

        function instantiateTypePredicate(predicate: TypePredicate, mapper: TypeMapper): TypePredicate {
            return createTypePredicate(predicate.kind, predicate.parameterName, predicate.parameterIndex, instantiateType(predicate.type, mapper));
        }

        function instantiateSignature(signature: Signature, mapper: TypeMapper, eraseTypeParameters?: boolean): Signature {
            let freshTypeParameters: TypeParameter[] | undefined;
            if (signature.typeParameters && !eraseTypeParameters) {
                // First create a fresh set of type parameters, then include a mapping from the old to the
                // new type parameters in the mapper function. Finally store this mapper in the new type
                // parameters such that we can use it when instantiating constraints.
                freshTypeParameters = map(signature.typeParameters, cloneTypeParameter);
                mapper = combineTypeMappers(createTypeMapper(signature.typeParameters, freshTypeParameters), mapper);
                for (const tp of freshTypeParameters) {
                    tp.mapper = mapper;
                }
            }
            // Don't compute resolvedReturnType and resolvedTypePredicate now,
            // because using `mapper` now could trigger inferences to become fixed. (See `createInferenceContext`.)
            // See GH#17600.
            const result = createSignature(signature.declaration, freshTypeParameters,
                signature.thisParameter && instantiateSymbol(signature.thisParameter, mapper),
                instantiateList(signature.parameters, mapper, instantiateSymbol),
                /*resolvedReturnType*/ undefined,
                /*resolvedTypePredicate*/ undefined,
                signature.minArgumentCount,
                signature.flags & SignatureFlags.PropagatingFlags);
            result.target = signature;
            result.mapper = mapper;
            return result;
        }

        function instantiateSymbol(symbol: Symbol, mapper: TypeMapper): Symbol {
            const links = getSymbolLinks(symbol);
            if (links.type && !couldContainTypeVariables(links.type)) {
                // If the type of the symbol is already resolved, and if that type could not possibly
                // be affected by instantiation, simply return the symbol itself.
                return symbol;
            }
            if (getCheckFlags(symbol) & CheckFlags.Instantiated) {
                // If symbol being instantiated is itself a instantiation, fetch the original target and combine the
                // type mappers. This ensures that original type identities are properly preserved and that aliases
                // always reference a non-aliases.
                symbol = links.target!;
                mapper = combineTypeMappers(links.mapper, mapper);
            }
            // Keep the flags from the symbol we're instantiating.  Mark that is instantiated, and
            // also transient so that we can just store data on it directly.
            const result = createSymbol(symbol.flags, symbol.escapedName, CheckFlags.Instantiated | getCheckFlags(symbol) & (CheckFlags.Readonly | CheckFlags.Late | CheckFlags.OptionalParameter | CheckFlags.RestParameter));
            result.declarations = symbol.declarations;
            result.parent = symbol.parent;
            result.target = symbol;
            result.mapper = mapper;
            if (symbol.valueDeclaration) {
                result.valueDeclaration = symbol.valueDeclaration;
            }
            if (links.nameType) {
                result.nameType = links.nameType;
            }
            return result;
        }

        function getObjectTypeInstantiation(type: AnonymousType | DeferredTypeReference, mapper: TypeMapper) {
            const target = type.objectFlags & ObjectFlags.Instantiated ? type.target! : type;
            const declaration = type.objectFlags & ObjectFlags.Reference ? (<TypeReference>type).node! : type.symbol.declarations[0];
            const links = getNodeLinks(declaration);
            let typeParameters = links.outerTypeParameters;
            if (!typeParameters) {
                // The first time an anonymous type is instantiated we compute and store a list of the type
                // parameters that are in scope (and therefore potentially referenced). For type literals that
                // aren't the right hand side of a generic type alias declaration we optimize by reducing the
                // set of type parameters to those that are possibly referenced in the literal.
                let outerTypeParameters = getOuterTypeParameters(declaration, /*includeThisTypes*/ true);
                if (isJSConstructor(declaration)) {
                    const templateTagParameters = getTypeParametersFromDeclaration(declaration as DeclarationWithTypeParameters);
                    outerTypeParameters = addRange(outerTypeParameters, templateTagParameters);
                }
                typeParameters = outerTypeParameters || emptyArray;
                typeParameters = (target.objectFlags & ObjectFlags.Reference || target.symbol.flags & SymbolFlags.TypeLiteral) && !target.aliasTypeArguments ?
                    filter(typeParameters, tp => isTypeParameterPossiblyReferenced(tp, declaration)) :
                    typeParameters;
                links.outerTypeParameters = typeParameters;
                if (typeParameters.length) {
                    links.instantiations = new Map<string, Type>();
                    links.instantiations.set(getTypeListId(typeParameters), target);
                }
            }
            if (typeParameters.length) {
                // We are instantiating an anonymous type that has one or more type parameters in scope. Apply the
                // mapper to the type parameters to produce the effective list of type arguments, and compute the
                // instantiation cache key from the type IDs of the type arguments.
                const combinedMapper = combineTypeMappers(type.mapper, mapper);
                const typeArguments = map(typeParameters, t => getMappedType(t, combinedMapper));
                const id = getTypeListId(typeArguments);
                let result = links.instantiations!.get(id);
                if (!result) {
                    const newMapper = createTypeMapper(typeParameters, typeArguments);
                    result = target.objectFlags & ObjectFlags.Reference ? createDeferredTypeReference((<DeferredTypeReference>type).target, (<DeferredTypeReference>type).node, newMapper) :
                        target.objectFlags & ObjectFlags.Mapped ? instantiateMappedType(<MappedType>target, newMapper) :
                        instantiateAnonymousType(target, newMapper);
                    links.instantiations!.set(id, result);
                }
                return result;
            }
            return type;
        }

        function maybeTypeParameterReference(node: Node) {
            return !(node.kind === SyntaxKind.QualifiedName ||
                node.parent.kind === SyntaxKind.TypeReference && (<TypeReferenceNode>node.parent).typeArguments && node === (<TypeReferenceNode>node.parent).typeName ||
                node.parent.kind === SyntaxKind.ImportType && (node.parent as ImportTypeNode).typeArguments && node === (node.parent as ImportTypeNode).qualifier);
        }

        function isTypeParameterPossiblyReferenced(tp: TypeParameter, node: Node) {
            // If the type parameter doesn't have exactly one declaration, if there are invening statement blocks
            // between the node and the type parameter declaration, if the node contains actual references to the
            // type parameter, or if the node contains type queries, we consider the type parameter possibly referenced.
            if (tp.symbol && tp.symbol.declarations && tp.symbol.declarations.length === 1) {
                const container = tp.symbol.declarations[0].parent;
                for (let n = node; n !== container; n = n.parent) {
                    if (!n || n.kind === SyntaxKind.Block || n.kind === SyntaxKind.ConditionalType && forEachChild((<ConditionalTypeNode>n).extendsType, containsReference)) {
                        return true;
                    }
                }
                return !!forEachChild(node, containsReference);
            }
            return true;
            function containsReference(node: Node): boolean {
                switch (node.kind) {
                    case SyntaxKind.ThisType:
                        return !!tp.isThisType;
                    case SyntaxKind.Identifier:
                        return !tp.isThisType && isPartOfTypeNode(node) && maybeTypeParameterReference(node) &&
                            getTypeFromTypeNodeWorker(<TypeNode>node) === tp; // use worker because we're looking for === equality
                    case SyntaxKind.TypeQuery:
                        return true;
                }
                return !!forEachChild(node, containsReference);
            }
        }

        function getHomomorphicTypeVariable(type: MappedType) {
            const constraintType = getConstraintTypeFromMappedType(type);
            if (constraintType.flags & TypeFlags.Index) {
                const typeVariable = getActualTypeVariable((<IndexType>constraintType).type);
                if (typeVariable.flags & TypeFlags.TypeParameter) {
                    return <TypeParameter>typeVariable;
                }
            }
            return undefined;
        }

        function instantiateMappedType(type: MappedType, mapper: TypeMapper): Type {
            // For a homomorphic mapped type { [P in keyof T]: X }, where T is some type variable, the mapping
            // operation depends on T as follows:
            // * If T is a primitive type no mapping is performed and the result is simply T.
            // * If T is a union type we distribute the mapped type over the union.
            // * If T is an array we map to an array where the element type has been transformed.
            // * If T is a tuple we map to a tuple where the element types have been transformed.
            // * Otherwise we map to an object type where the type of each property has been transformed.
            // For example, when T is instantiated to a union type A | B, we produce { [P in keyof A]: X } |
            // { [P in keyof B]: X }, and when when T is instantiated to a union type A | undefined, we produce
            // { [P in keyof A]: X } | undefined.
            const typeVariable = getHomomorphicTypeVariable(type);
            if (typeVariable) {
                const mappedTypeVariable = instantiateType(typeVariable, mapper);
                if (typeVariable !== mappedTypeVariable) {
                    return mapType(getReducedType(mappedTypeVariable), t => {
                        if (t.flags & (TypeFlags.AnyOrUnknown | TypeFlags.InstantiableNonPrimitive | TypeFlags.Object | TypeFlags.Intersection) && t !== wildcardType && t !== errorType) {
                            if (isGenericTupleType(t)) {
                                return instantiateMappedGenericTupleType(t, type, typeVariable, mapper);
                            }
                            const replacementMapper = prependTypeMapping(typeVariable, t, mapper);
                            return isArrayType(t) ? instantiateMappedArrayType(t, type, replacementMapper) :
                                isTupleType(t) ? instantiateMappedTupleType(t, type, replacementMapper) :
                                instantiateAnonymousType(type, replacementMapper);
                        }
                        return t;
                    });
                }
            }
            return instantiateAnonymousType(type, mapper);
        }

        function getModifiedReadonlyState(state: boolean, modifiers: MappedTypeModifiers) {
            return modifiers & MappedTypeModifiers.IncludeReadonly ? true : modifiers & MappedTypeModifiers.ExcludeReadonly ? false : state;
        }

        function instantiateMappedGenericTupleType(tupleType: TupleTypeReference, mappedType: MappedType, typeVariable: TypeVariable, mapper: TypeMapper) {
            // When a tuple type is generic (i.e. when it contains variadic elements), we want to eagerly map the
            // non-generic elements and defer mapping the generic elements. In order to facilitate this, we transform
            // M<[A, B?, ...T, ...C[]] into [...M<[A]>, ...M<[B?]>, ...M<T>, ...M<C[]>] and then rely on tuple type
            // normalization to resolve the non-generic parts of the resulting tuple.
            const elementFlags = tupleType.target.elementFlags;
            const elementTypes = map(getTypeArguments(tupleType), (t, i) => {
                const singleton = elementFlags[i] & ElementFlags.Variadic ? t :
                    elementFlags[i] & ElementFlags.Rest ? createArrayType(t) :
                    createTupleType([t], [elementFlags[i]]);
                // The singleton is never a generic tuple type, so it is safe to recurse here.
                return instantiateMappedType(mappedType, prependTypeMapping(typeVariable, singleton, mapper));
            });
            const newReadonly = getModifiedReadonlyState(tupleType.target.readonly, getMappedTypeModifiers(mappedType));
            return createTupleType(elementTypes, map(elementTypes, _ => ElementFlags.Variadic), newReadonly);
        }

        function instantiateMappedArrayType(arrayType: Type, mappedType: MappedType, mapper: TypeMapper) {
            const elementType = instantiateMappedTypeTemplate(mappedType, numberType, /*isOptional*/ true, mapper);
            return elementType === errorType ? errorType :
                createArrayType(elementType, getModifiedReadonlyState(isReadonlyArrayType(arrayType), getMappedTypeModifiers(mappedType)));
        }

        function instantiateMappedTupleType(tupleType: TupleTypeReference, mappedType: MappedType, mapper: TypeMapper) {
            const elementFlags = tupleType.target.elementFlags;
            const elementTypes = map(getTypeArguments(tupleType), (_, i) =>
                instantiateMappedTypeTemplate(mappedType, getLiteralType("" + i), !!(elementFlags[i] & ElementFlags.Optional), mapper));
            const modifiers = getMappedTypeModifiers(mappedType);
            const newTupleModifiers = modifiers & MappedTypeModifiers.IncludeOptional ? map(elementFlags, f => f & ElementFlags.Required ? ElementFlags.Optional : f) :
                modifiers & MappedTypeModifiers.ExcludeOptional ? map(elementFlags, f => f & ElementFlags.Optional ? ElementFlags.Required : f) :
                elementFlags;
            const newReadonly = getModifiedReadonlyState(tupleType.target.readonly, modifiers);
            return contains(elementTypes, errorType) ? errorType :
                createTupleType(elementTypes, newTupleModifiers, newReadonly, tupleType.target.labeledElementDeclarations);
        }

        function instantiateMappedTypeTemplate(type: MappedType, key: Type, isOptional: boolean, mapper: TypeMapper) {
            const templateMapper = appendTypeMapping(mapper, getTypeParameterFromMappedType(type), key);
            const propType = instantiateType(getTemplateTypeFromMappedType(<MappedType>type.target || type), templateMapper);
            const modifiers = getMappedTypeModifiers(type);
            return strictNullChecks && modifiers & MappedTypeModifiers.IncludeOptional && !maybeTypeOfKind(propType, TypeFlags.Undefined | TypeFlags.Void) ? getOptionalType(propType) :
                strictNullChecks && modifiers & MappedTypeModifiers.ExcludeOptional && isOptional ? getTypeWithFacts(propType, TypeFacts.NEUndefined) :
                propType;
        }

        function instantiateAnonymousType(type: AnonymousType, mapper: TypeMapper): AnonymousType {
            const result = <AnonymousType>createObjectType(type.objectFlags | ObjectFlags.Instantiated, type.symbol);
            if (type.objectFlags & ObjectFlags.Mapped) {
                (<MappedType>result).declaration = (<MappedType>type).declaration;
                // C.f. instantiateSignature
                const origTypeParameter = getTypeParameterFromMappedType(<MappedType>type);
                const freshTypeParameter = cloneTypeParameter(origTypeParameter);
                (<MappedType>result).typeParameter = freshTypeParameter;
                mapper = combineTypeMappers(makeUnaryTypeMapper(origTypeParameter, freshTypeParameter), mapper);
                freshTypeParameter.mapper = mapper;
            }
            result.target = type;
            result.mapper = mapper;
            result.aliasSymbol = type.aliasSymbol;
            result.aliasTypeArguments = instantiateTypes(type.aliasTypeArguments, mapper);
            return result;
        }

        function getConditionalTypeInstantiation(type: ConditionalType, mapper: TypeMapper): Type {
            const root = type.root;
            if (root.outerTypeParameters) {
                // We are instantiating a conditional type that has one or more type parameters in scope. Apply the
                // mapper to the type parameters to produce the effective list of type arguments, and compute the
                // instantiation cache key from the type IDs of the type arguments.
                const typeArguments = map(root.outerTypeParameters, t => getMappedType(t, mapper));
                const id = getTypeListId(typeArguments);
                let result = root.instantiations!.get(id);
                if (!result) {
                    const newMapper = createTypeMapper(root.outerTypeParameters, typeArguments);
                    result = instantiateConditionalType(root, newMapper);
                    root.instantiations!.set(id, result);
                }
                return result;
            }
            return type;
        }

        function instantiateConditionalType(root: ConditionalRoot, mapper: TypeMapper): Type {
            // Check if we have a conditional type where the check type is a naked type parameter. If so,
            // the conditional type is distributive over union types and when T is instantiated to a union
            // type A | B, we produce (A extends U ? X : Y) | (B extends U ? X : Y).
            if (root.isDistributive) {
                const checkType = <TypeParameter>root.checkType;
                const instantiatedType = getMappedType(checkType, mapper);
                if (checkType !== instantiatedType && instantiatedType.flags & (TypeFlags.Union | TypeFlags.Never)) {
                    return mapType(instantiatedType, t => getConditionalType(root, prependTypeMapping(checkType, t, mapper)));
                }
            }
            return getConditionalType(root, mapper);
        }

        function instantiateType(type: Type, mapper: TypeMapper | undefined): Type;
        function instantiateType(type: Type | undefined, mapper: TypeMapper | undefined): Type | undefined;
        function instantiateType(type: Type | undefined, mapper: TypeMapper | undefined): Type | undefined {
            if (!(type && mapper && couldContainTypeVariables(type))) {
                return type;
            }
            if (instantiationDepth === 50 || instantiationCount >= 5000000) {
                // We have reached 50 recursive type instantiations and there is a very high likelyhood we're dealing
                // with a combination of infinite generic types that perpetually generate new type identities. We stop
                // the recursion here by yielding the error type.
                error(currentNode, Diagnostics.Type_instantiation_is_excessively_deep_and_possibly_infinite);
                return errorType;
            }
            totalInstantiationCount++;
            instantiationCount++;
            instantiationDepth++;
            const result = instantiateTypeWorker(type, mapper);
            instantiationDepth--;
            return result;
        }

        /**
         * This can be used to avoid the penalty on instantiation depth for types which result from immediate
         * simplification. It essentially removes the depth increase done in `instantiateType`.
         */
        function instantiateTypeWithoutDepthIncrease(type: Type, mapper: TypeMapper | undefined) {
            instantiationDepth--;
            const result = instantiateType(type, mapper);
            instantiationDepth++;
            return result;
        }

        function instantiateTypeWorker(type: Type, mapper: TypeMapper): Type {
            const flags = type.flags;
            if (flags & TypeFlags.TypeParameter) {
                return getMappedType(type, mapper);
            }
            if (flags & TypeFlags.Object) {
                const objectFlags = (<ObjectType>type).objectFlags;
                if (objectFlags & (ObjectFlags.Reference | ObjectFlags.Anonymous | ObjectFlags.Mapped)) {
                    if (objectFlags & ObjectFlags.Reference && !((<TypeReference>type).node)) {
                        const resolvedTypeArguments = (<TypeReference>type).resolvedTypeArguments;
                        const newTypeArguments = instantiateTypes(resolvedTypeArguments, mapper);
                        return newTypeArguments !== resolvedTypeArguments ? createNormalizedTypeReference((<TypeReference>type).target, newTypeArguments) : type;
                    }
                    return getObjectTypeInstantiation(<TypeReference | AnonymousType | MappedType>type, mapper);
                }
                return type;
            }
            if (flags & TypeFlags.UnionOrIntersection) {
                const types = (<UnionOrIntersectionType>type).types;
                const newTypes = instantiateTypes(types, mapper);
                return newTypes === types ? type :
                    flags & TypeFlags.Intersection ?
                        getIntersectionType(newTypes, type.aliasSymbol, instantiateTypes(type.aliasTypeArguments, mapper)) :
                        getUnionType(newTypes, UnionReduction.Literal, type.aliasSymbol, instantiateTypes(type.aliasTypeArguments, mapper));
            }
            if (flags & TypeFlags.Index) {
                return getIndexType(instantiateType((<IndexType>type).type, mapper));
            }
            if (flags & TypeFlags.IndexedAccess) {
                return getIndexedAccessType(instantiateType((<IndexedAccessType>type).objectType, mapper), instantiateType((<IndexedAccessType>type).indexType, mapper), /*accessNode*/ undefined, type.aliasSymbol, instantiateTypes(type.aliasTypeArguments, mapper));
            }
            if (flags & TypeFlags.Conditional) {
                return getConditionalTypeInstantiation(<ConditionalType>type, combineTypeMappers((<ConditionalType>type).mapper, mapper));
            }
            if (flags & TypeFlags.Substitution) {
                const maybeVariable = instantiateType((<SubstitutionType>type).baseType, mapper);
                if (maybeVariable.flags & TypeFlags.TypeVariable) {
                    return getSubstitutionType(maybeVariable as TypeVariable, instantiateType((<SubstitutionType>type).substitute, mapper));
                }
                else {
                    const sub = instantiateType((<SubstitutionType>type).substitute, mapper);
                    if (sub.flags & TypeFlags.AnyOrUnknown || isTypeAssignableTo(getRestrictiveInstantiation(maybeVariable), getRestrictiveInstantiation(sub))) {
                        return maybeVariable;
                    }
                    return sub;
                }
            }
            return type;
        }

        function getPermissiveInstantiation(type: Type) {
            return type.flags & (TypeFlags.Primitive | TypeFlags.AnyOrUnknown | TypeFlags.Never) ? type :
                type.permissiveInstantiation || (type.permissiveInstantiation = instantiateType(type, permissiveMapper));
        }

        function getRestrictiveInstantiation(type: Type) {
            if (type.flags & (TypeFlags.Primitive | TypeFlags.AnyOrUnknown | TypeFlags.Never)) {
                return type;
            }
            if (type.restrictiveInstantiation) {
                return type.restrictiveInstantiation;
            }
            type.restrictiveInstantiation = instantiateType(type, restrictiveMapper);
            // We set the following so we don't attempt to set the restrictive instance of a restrictive instance
            // which is redundant - we'll produce new type identities, but all type params have already been mapped.
            // This also gives us a way to detect restrictive instances upon comparisons and _disable_ the "distributeive constraint"
            // assignability check for them, which is distinctly unsafe, as once you have a restrctive instance, all the type parameters
            // are constrained to `unknown` and produce tons of false positives/negatives!
            type.restrictiveInstantiation.restrictiveInstantiation = type.restrictiveInstantiation;
            return type.restrictiveInstantiation;
        }

        function instantiateIndexInfo(info: IndexInfo | undefined, mapper: TypeMapper): IndexInfo | undefined {
            return info && createIndexInfo(instantiateType(info.type, mapper), info.isReadonly, info.declaration);
        }

        // Returns true if the given expression contains (at any level of nesting) a function or arrow expression
        // that is subject to contextual typing.
        function isContextSensitive(node: Expression | MethodDeclaration | ObjectLiteralElementLike | JsxAttributeLike | JsxChild): boolean {
            Debug.assert(node.kind !== SyntaxKind.MethodDeclaration || isObjectLiteralMethod(node));
            switch (node.kind) {
                case SyntaxKind.FunctionExpression:
                case SyntaxKind.ArrowFunction:
                case SyntaxKind.MethodDeclaration:
                case SyntaxKind.FunctionDeclaration: // Function declarations can have context when annotated with a jsdoc @type
                    return isContextSensitiveFunctionLikeDeclaration(<FunctionExpression | ArrowFunction | MethodDeclaration>node);
                case SyntaxKind.ObjectLiteralExpression:
                    return some((<ObjectLiteralExpression>node).properties, isContextSensitive);
                case SyntaxKind.ArrayLiteralExpression:
                    return some((<ArrayLiteralExpression>node).elements, isContextSensitive);
                case SyntaxKind.ConditionalExpression:
                    return isContextSensitive((<ConditionalExpression>node).whenTrue) ||
                        isContextSensitive((<ConditionalExpression>node).whenFalse);
                case SyntaxKind.BinaryExpression:
                    return ((<BinaryExpression>node).operatorToken.kind === SyntaxKind.BarBarToken || (<BinaryExpression>node).operatorToken.kind === SyntaxKind.QuestionQuestionToken) &&
                        (isContextSensitive((<BinaryExpression>node).left) || isContextSensitive((<BinaryExpression>node).right));
                case SyntaxKind.PropertyAssignment:
                    return isContextSensitive((<PropertyAssignment>node).initializer);
                case SyntaxKind.ParenthesizedExpression:
                    return isContextSensitive((<ParenthesizedExpression>node).expression);
                case SyntaxKind.JsxAttributes:
                    return some((<JsxAttributes>node).properties, isContextSensitive) || isJsxOpeningElement(node.parent) && some(node.parent.parent.children, isContextSensitive);
                case SyntaxKind.JsxAttribute: {
                    // If there is no initializer, JSX attribute has a boolean value of true which is not context sensitive.
                    const { initializer } = node as JsxAttribute;
                    return !!initializer && isContextSensitive(initializer);
                }
                case SyntaxKind.JsxExpression: {
                    // It is possible to that node.expression is undefined (e.g <div x={} />)
                    const { expression } = node as JsxExpression;
                    return !!expression && isContextSensitive(expression);
                }
            }

            return false;
        }

        function isContextSensitiveFunctionLikeDeclaration(node: FunctionLikeDeclaration): boolean {
            return (!isFunctionDeclaration(node) || isInJSFile(node) && !!getTypeForDeclarationFromJSDocComment(node)) &&
                (hasContextSensitiveParameters(node) || hasContextSensitiveReturnExpression(node));
        }

        function hasContextSensitiveParameters(node: FunctionLikeDeclaration) {
            // Functions with type parameters are not context sensitive.
            if (!node.typeParameters) {
                // Functions with any parameters that lack type annotations are context sensitive.
                if (some(node.parameters, p => !getEffectiveTypeAnnotationNode(p))) {
                    return true;
                }
                if (node.kind !== SyntaxKind.ArrowFunction) {
                    // If the first parameter is not an explicit 'this' parameter, then the function has
                    // an implicit 'this' parameter which is subject to contextual typing.
                    const parameter = firstOrUndefined(node.parameters);
                    if (!(parameter && parameterIsThisKeyword(parameter))) {
                        return true;
                    }
                }
            }
            return false;
        }

        function hasContextSensitiveReturnExpression(node: FunctionLikeDeclaration) {
            // TODO(anhans): A block should be context-sensitive if it has a context-sensitive return value.
            return !node.typeParameters && !getEffectiveReturnTypeNode(node) && !!node.body && node.body.kind !== SyntaxKind.Block && isContextSensitive(node.body);
        }

        function isContextSensitiveFunctionOrObjectLiteralMethod(func: Node): func is FunctionExpression | ArrowFunction | MethodDeclaration {
            return (isInJSFile(func) && isFunctionDeclaration(func) || isFunctionExpressionOrArrowFunction(func) || isObjectLiteralMethod(func)) &&
                isContextSensitiveFunctionLikeDeclaration(func);
        }

        function getTypeWithoutSignatures(type: Type): Type {
            if (type.flags & TypeFlags.Object) {
                const resolved = resolveStructuredTypeMembers(<ObjectType>type);
                if (resolved.constructSignatures.length || resolved.callSignatures.length) {
                    const result = createObjectType(ObjectFlags.Anonymous, type.symbol);
                    result.members = resolved.members;
                    result.properties = resolved.properties;
                    result.callSignatures = emptyArray;
                    result.constructSignatures = emptyArray;
                    return result;
                }
            }
            else if (type.flags & TypeFlags.Intersection) {
                return getIntersectionType(map((<IntersectionType>type).types, getTypeWithoutSignatures));
            }
            return type;
        }

        // TYPE CHECKING

        function isTypeIdenticalTo(source: Type, target: Type): boolean {
            return isTypeRelatedTo(source, target, identityRelation);
        }

        function compareTypesIdentical(source: Type, target: Type): Ternary {
            return isTypeRelatedTo(source, target, identityRelation) ? Ternary.True : Ternary.False;
        }

        function compareTypesAssignable(source: Type, target: Type): Ternary {
            return isTypeRelatedTo(source, target, assignableRelation) ? Ternary.True : Ternary.False;
        }

        function compareTypesSubtypeOf(source: Type, target: Type): Ternary {
            return isTypeRelatedTo(source, target, subtypeRelation) ? Ternary.True : Ternary.False;
        }

        function isTypeSubtypeOf(source: Type, target: Type): boolean {
            return isTypeRelatedTo(source, target, subtypeRelation);
        }

        function isTypeAssignableTo(source: Type, target: Type): boolean {
            return isTypeRelatedTo(source, target, assignableRelation);
        }

        // An object type S is considered to be derived from an object type T if
        // S is a union type and every constituent of S is derived from T,
        // T is a union type and S is derived from at least one constituent of T, or
        // S is a type variable with a base constraint that is derived from T,
        // T is one of the global types Object and Function and S is a subtype of T, or
        // T occurs directly or indirectly in an 'extends' clause of S.
        // Note that this check ignores type parameters and only considers the
        // inheritance hierarchy.
        function isTypeDerivedFrom(source: Type, target: Type): boolean {
            return source.flags & TypeFlags.Union ? every((<UnionType>source).types, t => isTypeDerivedFrom(t, target)) :
                target.flags & TypeFlags.Union ? some((<UnionType>target).types, t => isTypeDerivedFrom(source, t)) :
                source.flags & TypeFlags.InstantiableNonPrimitive ? isTypeDerivedFrom(getBaseConstraintOfType(source) || unknownType, target) :
                target === globalObjectType ? !!(source.flags & (TypeFlags.Object | TypeFlags.NonPrimitive)) :
                target === globalFunctionType ? !!(source.flags & TypeFlags.Object) && isFunctionObjectType(source as ObjectType) :
                hasBaseType(source, getTargetType(target));
        }

        /**
         * This is *not* a bi-directional relationship.
         * If one needs to check both directions for comparability, use a second call to this function or 'checkTypeComparableTo'.
         *
         * A type S is comparable to a type T if some (but not necessarily all) of the possible values of S are also possible values of T.
         * It is used to check following cases:
         *   - the types of the left and right sides of equality/inequality operators (`===`, `!==`, `==`, `!=`).
         *   - the types of `case` clause expressions and their respective `switch` expressions.
         *   - the type of an expression in a type assertion with the type being asserted.
         */
        function isTypeComparableTo(source: Type, target: Type): boolean {
            return isTypeRelatedTo(source, target, comparableRelation);
        }

        function areTypesComparable(type1: Type, type2: Type): boolean {
            return isTypeComparableTo(type1, type2) || isTypeComparableTo(type2, type1);
        }

        function checkTypeAssignableTo(source: Type, target: Type, errorNode: Node | undefined, headMessage?: DiagnosticMessage, containingMessageChain?: () => DiagnosticMessageChain | undefined, errorOutputObject?: { errors?: Diagnostic[] }): boolean {
            return checkTypeRelatedTo(source, target, assignableRelation, errorNode, headMessage, containingMessageChain, errorOutputObject);
        }

        /**
         * Like `checkTypeAssignableTo`, but if it would issue an error, instead performs structural comparisons of the types using the given expression node to
         * attempt to issue more specific errors on, for example, specific object literal properties or tuple members.
         */
        function checkTypeAssignableToAndOptionallyElaborate(source: Type, target: Type, errorNode: Node | undefined, expr: Expression | undefined, headMessage?: DiagnosticMessage, containingMessageChain?: () => DiagnosticMessageChain | undefined): boolean {
            return checkTypeRelatedToAndOptionallyElaborate(source, target, assignableRelation, errorNode, expr, headMessage, containingMessageChain, /*errorOutputContainer*/ undefined);
        }

        function checkTypeRelatedToAndOptionallyElaborate(
            source: Type,
            target: Type,
            relation: ESMap<string, RelationComparisonResult>,
            errorNode: Node | undefined,
            expr: Expression | undefined,
            headMessage: DiagnosticMessage | undefined,
            containingMessageChain: (() => DiagnosticMessageChain | undefined) | undefined,
            errorOutputContainer: { errors?: Diagnostic[], skipLogging?: boolean } | undefined
        ): boolean {
            if (isTypeRelatedTo(source, target, relation)) return true;
            if (!errorNode || !elaborateError(expr, source, target, relation, headMessage, containingMessageChain, errorOutputContainer)) {
                return checkTypeRelatedTo(source, target, relation, errorNode, headMessage, containingMessageChain, errorOutputContainer);
            }
            return false;
        }

        function isOrHasGenericConditional(type: Type): boolean {
            return !!(type.flags & TypeFlags.Conditional || (type.flags & TypeFlags.Intersection && some((type as IntersectionType).types, isOrHasGenericConditional)));
        }

        function elaborateError(
            node: Expression | undefined,
            source: Type,
            target: Type,
            relation: ESMap<string, RelationComparisonResult>,
            headMessage: DiagnosticMessage | undefined,
            containingMessageChain: (() => DiagnosticMessageChain | undefined) | undefined,
            errorOutputContainer: { errors?: Diagnostic[], skipLogging?: boolean } | undefined
        ): boolean {
            if (!node || isOrHasGenericConditional(target)) return false;
            if (!checkTypeRelatedTo(source, target, relation, /*errorNode*/ undefined)
                && elaborateDidYouMeanToCallOrConstruct(node, source, target, relation, headMessage, containingMessageChain, errorOutputContainer)) {
                return true;
            }
            switch (node.kind) {
                case SyntaxKind.JsxExpression:
                case SyntaxKind.ParenthesizedExpression:
                    return elaborateError((node as ParenthesizedExpression | JsxExpression).expression, source, target, relation, headMessage, containingMessageChain, errorOutputContainer);
                case SyntaxKind.BinaryExpression:
                    switch ((node as BinaryExpression).operatorToken.kind) {
                        case SyntaxKind.EqualsToken:
                        case SyntaxKind.CommaToken:
                            return elaborateError((node as BinaryExpression).right, source, target, relation, headMessage, containingMessageChain, errorOutputContainer);
                    }
                    break;
                case SyntaxKind.ObjectLiteralExpression:
                    return elaborateObjectLiteral(node as ObjectLiteralExpression, source, target, relation, containingMessageChain, errorOutputContainer);
                case SyntaxKind.ArrayLiteralExpression:
                    return elaborateArrayLiteral(node as ArrayLiteralExpression, source, target, relation, containingMessageChain, errorOutputContainer);
                case SyntaxKind.JsxAttributes:
                    return elaborateJsxComponents(node as JsxAttributes, source, target, relation, containingMessageChain, errorOutputContainer);
                case SyntaxKind.ArrowFunction:
                    return elaborateArrowFunction(node as ArrowFunction, source, target, relation, containingMessageChain, errorOutputContainer);
            }
            return false;
        }

        function elaborateDidYouMeanToCallOrConstruct(
            node: Expression,
            source: Type,
            target: Type,
            relation: ESMap<string, RelationComparisonResult>,
            headMessage: DiagnosticMessage | undefined,
            containingMessageChain: (() => DiagnosticMessageChain | undefined) | undefined,
            errorOutputContainer: { errors?: Diagnostic[], skipLogging?: boolean } | undefined
        ): boolean {
            const callSignatures = getSignaturesOfType(source, SignatureKind.Call);
            const constructSignatures = getSignaturesOfType(source, SignatureKind.Construct);
            for (const signatures of [constructSignatures, callSignatures]) {
                if (some(signatures, s => {
                    const returnType = getReturnTypeOfSignature(s);
                    return !(returnType.flags & (TypeFlags.Any | TypeFlags.Never)) && checkTypeRelatedTo(returnType, target, relation, /*errorNode*/ undefined);
                })) {
                    const resultObj: { errors?: Diagnostic[] } = errorOutputContainer || {};
                    checkTypeAssignableTo(source, target, node, headMessage, containingMessageChain, resultObj);
                    const diagnostic = resultObj.errors![resultObj.errors!.length - 1];
                    addRelatedInfo(diagnostic, createDiagnosticForNode(
                        node,
                        signatures === constructSignatures ? Diagnostics.Did_you_mean_to_use_new_with_this_expression : Diagnostics.Did_you_mean_to_call_this_expression
                    ));
                    return true;
                }
            }
            return false;
        }

        function elaborateArrowFunction(
            node: ArrowFunction,
            source: Type,
            target: Type,
            relation: ESMap<string, RelationComparisonResult>,
            containingMessageChain: (() => DiagnosticMessageChain | undefined) | undefined,
            errorOutputContainer: { errors?: Diagnostic[], skipLogging?: boolean } | undefined
        ): boolean {
            // Don't elaborate blocks
            if (isBlock(node.body)) {
                return false;
            }
            // Or functions with annotated parameter types
            if (some(node.parameters, ts.hasType)) {
                return false;
            }
            const sourceSig = getSingleCallSignature(source);
            if (!sourceSig) {
                return false;
            }
            const targetSignatures = getSignaturesOfType(target, SignatureKind.Call);
            if (!length(targetSignatures)) {
                return false;
            }
            const returnExpression = node.body;
            const sourceReturn = getReturnTypeOfSignature(sourceSig);
            const targetReturn = getUnionType(map(targetSignatures, getReturnTypeOfSignature));
            if (!checkTypeRelatedTo(sourceReturn, targetReturn, relation, /*errorNode*/ undefined)) {
                const elaborated = returnExpression && elaborateError(returnExpression, sourceReturn, targetReturn, relation, /*headMessage*/ undefined, containingMessageChain, errorOutputContainer);
                if (elaborated) {
                    return elaborated;
                }
                const resultObj: { errors?: Diagnostic[] } = errorOutputContainer || {};
                checkTypeRelatedTo(sourceReturn, targetReturn, relation, returnExpression, /*message*/ undefined, containingMessageChain, resultObj);
                if (resultObj.errors) {
                    if (target.symbol && length(target.symbol.declarations)) {
                        addRelatedInfo(resultObj.errors[resultObj.errors.length - 1], createDiagnosticForNode(
                            target.symbol.declarations[0],
                            Diagnostics.The_expected_type_comes_from_the_return_type_of_this_signature,
                        ));
                    }
                    if ((getFunctionFlags(node) & FunctionFlags.Async) === 0
                        // exclude cases where source itself is promisy - this way we don't make a suggestion when relating
                        // an IPromise and a Promise that are slightly different
                        && !getTypeOfPropertyOfType(sourceReturn, "then" as __String)
                        && checkTypeRelatedTo(createPromiseType(sourceReturn), targetReturn, relation, /*errorNode*/ undefined)
                    ) {
                        addRelatedInfo(resultObj.errors[resultObj.errors.length - 1], createDiagnosticForNode(
                            node,
                            Diagnostics.Did_you_mean_to_mark_this_function_as_async
                        ));
                    }
                    return true;
                }
            }
            return false;
        }

        function getBestMatchIndexedAccessTypeOrUndefined(source: Type, target: Type, nameType: Type) {
            const idx = getIndexedAccessTypeOrUndefined(target, nameType);
            if (idx) {
                return idx;
            }
            if (target.flags & TypeFlags.Union) {
                const best = getBestMatchingType(source, target as UnionType);
                if (best) {
                    return getIndexedAccessTypeOrUndefined(best, nameType);
                }
            }
        }

        function checkExpressionForMutableLocationWithContextualType(next: Expression, sourcePropType: Type) {
            next.contextualType = sourcePropType;
            try {
                return checkExpressionForMutableLocation(next, CheckMode.Contextual, sourcePropType);
            }
            finally {
                next.contextualType = undefined;
            }
        }

        type ElaborationIterator = IterableIterator<{ errorNode: Node, innerExpression: Expression | undefined, nameType: Type, errorMessage?: DiagnosticMessage | undefined }>;
        /**
         * For every element returned from the iterator, checks that element to issue an error on a property of that element's type
         * If that element would issue an error, we first attempt to dive into that element's inner expression and issue a more specific error by recuring into `elaborateError`
         * Otherwise, we issue an error on _every_ element which fail the assignability check
         */
        function elaborateElementwise(
            iterator: ElaborationIterator,
            source: Type,
            target: Type,
            relation: ESMap<string, RelationComparisonResult>,
            containingMessageChain: (() => DiagnosticMessageChain | undefined) | undefined,
            errorOutputContainer: { errors?: Diagnostic[], skipLogging?: boolean } | undefined
        ) {
            // Assignability failure - check each prop individually, and if that fails, fall back on the bad error span
            let reportedError = false;
            for (let status = iterator.next(); !status.done; status = iterator.next()) {
                const { errorNode: prop, innerExpression: next, nameType, errorMessage } = status.value;
                const targetPropType = getBestMatchIndexedAccessTypeOrUndefined(source, target, nameType);
                if (!targetPropType || targetPropType.flags & TypeFlags.IndexedAccess) continue; // Don't elaborate on indexes on generic variables
                const sourcePropType = getIndexedAccessTypeOrUndefined(source, nameType);
                if (sourcePropType && !checkTypeRelatedTo(sourcePropType, targetPropType, relation, /*errorNode*/ undefined)) {
                    const elaborated = next && elaborateError(next, sourcePropType, targetPropType, relation, /*headMessage*/ undefined, containingMessageChain, errorOutputContainer);
                    if (elaborated) {
                        reportedError = true;
                    }
                    else {
                        // Issue error on the prop itself, since the prop couldn't elaborate the error
                        const resultObj: { errors?: Diagnostic[] } = errorOutputContainer || {};
                        // Use the expression type, if available
                        const specificSource = next ? checkExpressionForMutableLocationWithContextualType(next, sourcePropType) : sourcePropType;
                        const result = checkTypeRelatedTo(specificSource, targetPropType, relation, prop, errorMessage, containingMessageChain, resultObj);
                        if (result && specificSource !== sourcePropType) {
                            // If for whatever reason the expression type doesn't yield an error, make sure we still issue an error on the sourcePropType
                            checkTypeRelatedTo(sourcePropType, targetPropType, relation, prop, errorMessage, containingMessageChain, resultObj);
                        }
                        if (resultObj.errors) {
                            const reportedDiag = resultObj.errors[resultObj.errors.length - 1];
                            const propertyName = isTypeUsableAsPropertyName(nameType) ? getPropertyNameFromType(nameType) : undefined;
                            const targetProp = propertyName !== undefined ? getPropertyOfType(target, propertyName) : undefined;

                            let issuedElaboration = false;
                            if (!targetProp) {
                                const indexInfo = isTypeAssignableToKind(nameType, TypeFlags.NumberLike) && getIndexInfoOfType(target, IndexKind.Number) ||
                                    getIndexInfoOfType(target, IndexKind.String) ||
                                    undefined;
                                if (indexInfo && indexInfo.declaration && !getSourceFileOfNode(indexInfo.declaration).hasNoDefaultLib) {
                                    issuedElaboration = true;
                                    addRelatedInfo(reportedDiag, createDiagnosticForNode(indexInfo.declaration, Diagnostics.The_expected_type_comes_from_this_index_signature));
                                }
                            }

                            if (!issuedElaboration && (targetProp && length(targetProp.declarations) || target.symbol && length(target.symbol.declarations))) {
                                const targetNode = targetProp && length(targetProp.declarations) ? targetProp.declarations[0] : target.symbol.declarations[0];
                                if (!getSourceFileOfNode(targetNode).hasNoDefaultLib) {
                                    addRelatedInfo(reportedDiag, createDiagnosticForNode(
                                        targetNode,
                                        Diagnostics.The_expected_type_comes_from_property_0_which_is_declared_here_on_type_1,
                                        propertyName && !(nameType.flags & TypeFlags.UniqueESSymbol) ? unescapeLeadingUnderscores(propertyName) : typeToString(nameType),
                                        typeToString(target)
                                    ));
                                }
                            }
                        }
                        reportedError = true;
                    }
                }
            }
            return reportedError;
        }

        function *generateJsxAttributes(node: JsxAttributes): ElaborationIterator {
            if (!length(node.properties)) return;
            for (const prop of node.properties) {
                if (isJsxSpreadAttribute(prop)) continue;
                yield { errorNode: prop.name, innerExpression: prop.initializer, nameType: getLiteralType(idText(prop.name)) };
            }
        }

        function *generateJsxChildren(node: JsxElement, getInvalidTextDiagnostic: () => DiagnosticMessage): ElaborationIterator {
            if (!length(node.children)) return;
            let memberOffset = 0;
            for (let i = 0; i < node.children.length; i++) {
                const child = node.children[i];
                const nameType = getLiteralType(i - memberOffset);
                const elem = getElaborationElementForJsxChild(child, nameType, getInvalidTextDiagnostic);
                if (elem) {
                    yield elem;
                }
                else {
                    memberOffset++;
                }
            }
        }

        function getElaborationElementForJsxChild(child: JsxChild, nameType: LiteralType, getInvalidTextDiagnostic: () => DiagnosticMessage) {
            switch (child.kind) {
                case SyntaxKind.JsxExpression:
                    // child is of the type of the expression
                    return { errorNode: child, innerExpression: child.expression, nameType };
                case SyntaxKind.JsxText:
                    if (child.containsOnlyTriviaWhiteSpaces) {
                        break; // Whitespace only jsx text isn't real jsx text
                    }
                    // child is a string
                    return { errorNode: child, innerExpression: undefined, nameType, errorMessage: getInvalidTextDiagnostic() };
                case SyntaxKind.JsxElement:
                case SyntaxKind.JsxSelfClosingElement:
                case SyntaxKind.JsxFragment:
                    // child is of type JSX.Element
                    return { errorNode: child, innerExpression: child, nameType };
                default:
                    return Debug.assertNever(child, "Found invalid jsx child");
            }
        }

        function getSemanticJsxChildren(children: NodeArray<JsxChild>) {
            return filter(children, i => !isJsxText(i) || !i.containsOnlyTriviaWhiteSpaces);
        }

        function elaborateJsxComponents(
            node: JsxAttributes,
            source: Type,
            target: Type,
            relation: ESMap<string, RelationComparisonResult>,
            containingMessageChain: (() => DiagnosticMessageChain | undefined) | undefined,
            errorOutputContainer: { errors?: Diagnostic[], skipLogging?: boolean } | undefined
        ) {
            let result = elaborateElementwise(generateJsxAttributes(node), source, target, relation, containingMessageChain, errorOutputContainer);
            let invalidTextDiagnostic: DiagnosticMessage | undefined;
            if (isJsxOpeningElement(node.parent) && isJsxElement(node.parent.parent)) {
                const containingElement = node.parent.parent;
                const childPropName = getJsxElementChildrenPropertyName(getJsxNamespaceAt(node));
                const childrenPropName = childPropName === undefined ? "children" : unescapeLeadingUnderscores(childPropName);
                const childrenNameType = getLiteralType(childrenPropName);
                const childrenTargetType = getIndexedAccessType(target, childrenNameType);
                const validChildren = getSemanticJsxChildren(containingElement.children);
                if (!length(validChildren)) {
                    return result;
                }
                const moreThanOneRealChildren = length(validChildren) > 1;
                const arrayLikeTargetParts = filterType(childrenTargetType, isArrayOrTupleLikeType);
                const nonArrayLikeTargetParts = filterType(childrenTargetType, t => !isArrayOrTupleLikeType(t));
                if (moreThanOneRealChildren) {
                    if (arrayLikeTargetParts !== neverType) {
                        const realSource = createTupleType(checkJsxChildren(containingElement, CheckMode.Normal));
                        const children = generateJsxChildren(containingElement, getInvalidTextualChildDiagnostic);
                        result = elaborateElementwise(children, realSource, arrayLikeTargetParts, relation, containingMessageChain, errorOutputContainer) || result;
                    }
                    else if (!isTypeRelatedTo(getIndexedAccessType(source, childrenNameType), childrenTargetType, relation)) {
                        // arity mismatch
                        result = true;
                        const diag = error(
                            containingElement.openingElement.tagName,
                            Diagnostics.This_JSX_tag_s_0_prop_expects_a_single_child_of_type_1_but_multiple_children_were_provided,
                            childrenPropName,
                            typeToString(childrenTargetType)
                        );
                        if (errorOutputContainer && errorOutputContainer.skipLogging) {
                            (errorOutputContainer.errors || (errorOutputContainer.errors = [])).push(diag);
                        }
                    }
                }
                else {
                    if (nonArrayLikeTargetParts !== neverType) {
                        const child = validChildren[0];
                        const elem = getElaborationElementForJsxChild(child, childrenNameType, getInvalidTextualChildDiagnostic);
                        if (elem) {
                            result = elaborateElementwise(
                                (function*() { yield elem; })(),
                                source,
                                target,
                                relation,
                                containingMessageChain,
                                errorOutputContainer
                            ) || result;
                        }
                    }
                    else if (!isTypeRelatedTo(getIndexedAccessType(source, childrenNameType), childrenTargetType, relation)) {
                        // arity mismatch
                        result = true;
                        const diag = error(
                            containingElement.openingElement.tagName,
                            Diagnostics.This_JSX_tag_s_0_prop_expects_type_1_which_requires_multiple_children_but_only_a_single_child_was_provided,
                            childrenPropName,
                            typeToString(childrenTargetType)
                        );
                        if (errorOutputContainer && errorOutputContainer.skipLogging) {
                            (errorOutputContainer.errors || (errorOutputContainer.errors = [])).push(diag);
                        }
                    }
                }
            }
            return result;

            function getInvalidTextualChildDiagnostic() {
                if (!invalidTextDiagnostic) {
                    const tagNameText = getTextOfNode(node.parent.tagName);
                    const childPropName = getJsxElementChildrenPropertyName(getJsxNamespaceAt(node));
                    const childrenPropName = childPropName === undefined ? "children" : unescapeLeadingUnderscores(childPropName);
                    const childrenTargetType = getIndexedAccessType(target, getLiteralType(childrenPropName));
                    const diagnostic = Diagnostics._0_components_don_t_accept_text_as_child_elements_Text_in_JSX_has_the_type_string_but_the_expected_type_of_1_is_2;
                    invalidTextDiagnostic = { ...diagnostic, key: "!!ALREADY FORMATTED!!", message: formatMessage(/*_dummy*/ undefined, diagnostic, tagNameText, childrenPropName, typeToString(childrenTargetType)) };
                }
                return invalidTextDiagnostic;
            }
        }

        function *generateLimitedTupleElements(node: ArrayLiteralExpression, target: Type): ElaborationIterator {
            const len = length(node.elements);
            if (!len) return;
            for (let i = 0; i < len; i++) {
                // Skip elements which do not exist in the target - a length error on the tuple overall is likely better than an error on a mismatched index signature
                if (isTupleLikeType(target) && !getPropertyOfType(target, ("" + i) as __String)) continue;
                const elem = node.elements[i];
                if (isOmittedExpression(elem)) continue;
                const nameType = getLiteralType(i);
                yield { errorNode: elem, innerExpression: elem, nameType };
            }
        }

        function elaborateArrayLiteral(
            node: ArrayLiteralExpression,
            source: Type,
            target: Type,
            relation: ESMap<string, RelationComparisonResult>,
            containingMessageChain: (() => DiagnosticMessageChain | undefined) | undefined,
            errorOutputContainer: { errors?: Diagnostic[], skipLogging?: boolean } | undefined
        ) {
            if (target.flags & TypeFlags.Primitive) return false;
            if (isTupleLikeType(source)) {
                return elaborateElementwise(generateLimitedTupleElements(node, target), source, target, relation, containingMessageChain, errorOutputContainer);
            }
            // recreate a tuple from the elements, if possible
            // Since we're re-doing the expression type, we need to reapply the contextual type
            const oldContext = node.contextualType;
            node.contextualType = target;
            try {
                const tupleizedType = checkArrayLiteral(node, CheckMode.Contextual, /*forceTuple*/ true);
                node.contextualType = oldContext;
                if (isTupleLikeType(tupleizedType)) {
                    return elaborateElementwise(generateLimitedTupleElements(node, target), tupleizedType, target, relation, containingMessageChain, errorOutputContainer);
                }
                return false;
            }
            finally {
                node.contextualType = oldContext;
            }
        }

        function *generateObjectLiteralElements(node: ObjectLiteralExpression): ElaborationIterator {
            if (!length(node.properties)) return;
            for (const prop of node.properties) {
                if (isSpreadAssignment(prop)) continue;
                const type = getLiteralTypeFromProperty(getSymbolOfNode(prop), TypeFlags.StringOrNumberLiteralOrUnique);
                if (!type || (type.flags & TypeFlags.Never)) {
                    continue;
                }
                switch (prop.kind) {
                    case SyntaxKind.SetAccessor:
                    case SyntaxKind.GetAccessor:
                    case SyntaxKind.MethodDeclaration:
                    case SyntaxKind.ShorthandPropertyAssignment:
                        yield { errorNode: prop.name, innerExpression: undefined, nameType: type };
                        break;
                    case SyntaxKind.PropertyAssignment:
                        yield { errorNode: prop.name, innerExpression: prop.initializer, nameType: type, errorMessage: isComputedNonLiteralName(prop.name) ? Diagnostics.Type_of_computed_property_s_value_is_0_which_is_not_assignable_to_type_1 : undefined };
                        break;
                    default:
                        Debug.assertNever(prop);
                }
            }
        }

        function elaborateObjectLiteral(
            node: ObjectLiteralExpression,
            source: Type,
            target: Type,
            relation: ESMap<string, RelationComparisonResult>,
            containingMessageChain: (() => DiagnosticMessageChain | undefined) | undefined,
            errorOutputContainer: { errors?: Diagnostic[], skipLogging?: boolean } | undefined
        ) {
            if (target.flags & TypeFlags.Primitive) return false;
            return elaborateElementwise(generateObjectLiteralElements(node), source, target, relation, containingMessageChain, errorOutputContainer);
        }

        /**
         * This is *not* a bi-directional relationship.
         * If one needs to check both directions for comparability, use a second call to this function or 'isTypeComparableTo'.
         */
        function checkTypeComparableTo(source: Type, target: Type, errorNode: Node, headMessage?: DiagnosticMessage, containingMessageChain?: () => DiagnosticMessageChain | undefined): boolean {
            return checkTypeRelatedTo(source, target, comparableRelation, errorNode, headMessage, containingMessageChain);
        }

        function isSignatureAssignableTo(source: Signature,
            target: Signature,
            ignoreReturnTypes: boolean): boolean {
            return compareSignaturesRelated(source, target, ignoreReturnTypes ? SignatureCheckMode.IgnoreReturnTypes : 0, /*reportErrors*/ false,
                /*errorReporter*/ undefined, /*errorReporter*/ undefined, compareTypesAssignable, /*reportUnreliableMarkers*/ undefined) !== Ternary.False;
        }

        type ErrorReporter = (message: DiagnosticMessage, arg0?: string, arg1?: string) => void;

        /**
         * Returns true if `s` is `(...args: any[]) => any` or `(this: any, ...args: any[]) => any`
         */
        function isAnySignature(s: Signature) {
            return !s.typeParameters && (!s.thisParameter || isTypeAny(getTypeOfParameter(s.thisParameter))) && s.parameters.length === 1 &&
                signatureHasRestParameter(s) && (getTypeOfParameter(s.parameters[0]) === anyArrayType || isTypeAny(getTypeOfParameter(s.parameters[0]))) &&
                isTypeAny(getReturnTypeOfSignature(s));
        }

        /**
         * See signatureRelatedTo, compareSignaturesIdentical
         */
        function compareSignaturesRelated(source: Signature,
            target: Signature,
            checkMode: SignatureCheckMode,
            reportErrors: boolean,
            errorReporter: ErrorReporter | undefined,
            incompatibleErrorReporter: ((source: Type, target: Type) => void) | undefined,
            compareTypes: TypeComparer,
            reportUnreliableMarkers: TypeMapper | undefined): Ternary {
            // TODO (drosen): De-duplicate code between related functions.
            if (source === target) {
                return Ternary.True;
            }

            if (isAnySignature(target)) {
                return Ternary.True;
            }

            const targetCount = getParameterCount(target);
            const sourceHasMoreParameters = !hasEffectiveRestParameter(target) &&
                (checkMode & SignatureCheckMode.StrictArity ? hasEffectiveRestParameter(source) || getParameterCount(source) > targetCount : getMinArgumentCount(source) > targetCount);
            if (sourceHasMoreParameters) {
                return Ternary.False;
            }

            if (source.typeParameters && source.typeParameters !== target.typeParameters) {
                target = getCanonicalSignature(target);
                source = instantiateSignatureInContextOf(source, target, /*inferenceContext*/ undefined, compareTypes);
            }

            const sourceCount = getParameterCount(source);
            const sourceRestType = getNonArrayRestType(source);
            const targetRestType = getNonArrayRestType(target);
            if (sourceRestType || targetRestType) {
                void instantiateType(sourceRestType || targetRestType, reportUnreliableMarkers);
            }
            if (sourceRestType && targetRestType && sourceCount !== targetCount) {
                // We're not able to relate misaligned complex rest parameters
                return Ternary.False;
            }

            const kind = target.declaration ? target.declaration.kind : SyntaxKind.Unknown;
            const strictVariance = !(checkMode & SignatureCheckMode.Callback) && strictFunctionTypes && kind !== SyntaxKind.MethodDeclaration &&
                kind !== SyntaxKind.MethodSignature && kind !== SyntaxKind.Constructor;
            let result = Ternary.True;

            const sourceThisType = getThisTypeOfSignature(source);
            if (sourceThisType && sourceThisType !== voidType) {
                const targetThisType = getThisTypeOfSignature(target);
                if (targetThisType) {
                    // void sources are assignable to anything.
                    const related = !strictVariance && compareTypes(sourceThisType, targetThisType, /*reportErrors*/ false)
                        || compareTypes(targetThisType, sourceThisType, reportErrors);
                    if (!related) {
                        if (reportErrors) {
                            errorReporter!(Diagnostics.The_this_types_of_each_signature_are_incompatible);
                        }
                        return Ternary.False;
                    }
                    result &= related;
                }
            }

            const paramCount = sourceRestType || targetRestType ? Math.min(sourceCount, targetCount) : Math.max(sourceCount, targetCount);
            const restIndex = sourceRestType || targetRestType ? paramCount - 1 : -1;

            for (let i = 0; i < paramCount; i++) {
                const sourceType = i === restIndex ? getRestTypeAtPosition(source, i) : getTypeAtPosition(source, i);
                const targetType = i === restIndex ? getRestTypeAtPosition(target, i) : getTypeAtPosition(target, i);
                // In order to ensure that any generic type Foo<T> is at least co-variant with respect to T no matter
                // how Foo uses T, we need to relate parameters bi-variantly (given that parameters are input positions,
                // they naturally relate only contra-variantly). However, if the source and target parameters both have
                // function types with a single call signature, we know we are relating two callback parameters. In
                // that case it is sufficient to only relate the parameters of the signatures co-variantly because,
                // similar to return values, callback parameters are output positions. This means that a Promise<T>,
                // where T is used only in callback parameter positions, will be co-variant (as opposed to bi-variant)
                // with respect to T.
                const sourceSig = checkMode & SignatureCheckMode.Callback ? undefined : getSingleCallSignature(getNonNullableType(sourceType));
                const targetSig = checkMode & SignatureCheckMode.Callback ? undefined : getSingleCallSignature(getNonNullableType(targetType));
                const callbacks = sourceSig && targetSig && !getTypePredicateOfSignature(sourceSig) && !getTypePredicateOfSignature(targetSig) &&
                    (getFalsyFlags(sourceType) & TypeFlags.Nullable) === (getFalsyFlags(targetType) & TypeFlags.Nullable);
                let related = callbacks ?
                    compareSignaturesRelated(targetSig!, sourceSig!, (checkMode & SignatureCheckMode.StrictArity) | (strictVariance ? SignatureCheckMode.StrictCallback : SignatureCheckMode.BivariantCallback), reportErrors, errorReporter, incompatibleErrorReporter, compareTypes, reportUnreliableMarkers) :
                    !(checkMode & SignatureCheckMode.Callback) && !strictVariance && compareTypes(sourceType, targetType, /*reportErrors*/ false) || compareTypes(targetType, sourceType, reportErrors);
                // With strict arity, (x: number | undefined) => void is a subtype of (x?: number | undefined) => void
                if (related && checkMode & SignatureCheckMode.StrictArity && i >= getMinArgumentCount(source) && i < getMinArgumentCount(target) && compareTypes(sourceType, targetType, /*reportErrors*/ false)) {
                    related = Ternary.False;
                }
                if (!related) {
                    if (reportErrors) {
                        errorReporter!(Diagnostics.Types_of_parameters_0_and_1_are_incompatible,
                            unescapeLeadingUnderscores(getParameterNameAtPosition(source, i)),
                            unescapeLeadingUnderscores(getParameterNameAtPosition(target, i)));
                    }
                    return Ternary.False;
                }
                result &= related;
            }

            if (!(checkMode & SignatureCheckMode.IgnoreReturnTypes)) {
                // If a signature resolution is already in-flight, skip issuing a circularity error
                // here and just use the `any` type directly
                const targetReturnType = isResolvingReturnTypeOfSignature(target) ? anyType
                    : target.declaration && isJSConstructor(target.declaration) ? getDeclaredTypeOfClassOrInterface(getMergedSymbol(target.declaration.symbol))
                    : getReturnTypeOfSignature(target);
                if (targetReturnType === voidType) {
                    return result;
                }
                const sourceReturnType = isResolvingReturnTypeOfSignature(source) ? anyType
                    : source.declaration && isJSConstructor(source.declaration) ? getDeclaredTypeOfClassOrInterface(getMergedSymbol(source.declaration.symbol))
                    : getReturnTypeOfSignature(source);

                // The following block preserves behavior forbidding boolean returning functions from being assignable to type guard returning functions
                const targetTypePredicate = getTypePredicateOfSignature(target);
                if (targetTypePredicate) {
                    const sourceTypePredicate = getTypePredicateOfSignature(source);
                    if (sourceTypePredicate) {
                        result &= compareTypePredicateRelatedTo(sourceTypePredicate, targetTypePredicate, reportErrors, errorReporter, compareTypes);
                    }
                    else if (isIdentifierTypePredicate(targetTypePredicate)) {
                        if (reportErrors) {
                            errorReporter!(Diagnostics.Signature_0_must_be_a_type_predicate, signatureToString(source));
                        }
                        return Ternary.False;
                    }
                }
                else {
                    // When relating callback signatures, we still need to relate return types bi-variantly as otherwise
                    // the containing type wouldn't be co-variant. For example, interface Foo<T> { add(cb: () => T): void }
                    // wouldn't be co-variant for T without this rule.
                    result &= checkMode & SignatureCheckMode.BivariantCallback && compareTypes(targetReturnType, sourceReturnType, /*reportErrors*/ false) ||
                        compareTypes(sourceReturnType, targetReturnType, reportErrors);
                    if (!result && reportErrors && incompatibleErrorReporter) {
                        incompatibleErrorReporter(sourceReturnType, targetReturnType);
                    }
                }

            }

            return result;
        }

        function compareTypePredicateRelatedTo(
            source: TypePredicate,
            target: TypePredicate,
            reportErrors: boolean,
            errorReporter: ErrorReporter | undefined,
            compareTypes: (s: Type, t: Type, reportErrors?: boolean) => Ternary): Ternary {
            if (source.kind !== target.kind) {
                if (reportErrors) {
                    errorReporter!(Diagnostics.A_this_based_type_guard_is_not_compatible_with_a_parameter_based_type_guard);
                    errorReporter!(Diagnostics.Type_predicate_0_is_not_assignable_to_1, typePredicateToString(source), typePredicateToString(target));
                }
                return Ternary.False;
            }

            if (source.kind === TypePredicateKind.Identifier || source.kind === TypePredicateKind.AssertsIdentifier) {
                if (source.parameterIndex !== (target as IdentifierTypePredicate).parameterIndex) {
                    if (reportErrors) {
                        errorReporter!(Diagnostics.Parameter_0_is_not_in_the_same_position_as_parameter_1, source.parameterName, (target as IdentifierTypePredicate).parameterName);
                        errorReporter!(Diagnostics.Type_predicate_0_is_not_assignable_to_1, typePredicateToString(source), typePredicateToString(target));
                    }
                    return Ternary.False;
                }
            }

            const related = source.type === target.type ? Ternary.True :
                source.type && target.type ? compareTypes(source.type, target.type, reportErrors) :
                Ternary.False;
            if (related === Ternary.False && reportErrors) {
                errorReporter!(Diagnostics.Type_predicate_0_is_not_assignable_to_1, typePredicateToString(source), typePredicateToString(target));
            }
            return related;
        }

        function isImplementationCompatibleWithOverload(implementation: Signature, overload: Signature): boolean {
            const erasedSource = getErasedSignature(implementation);
            const erasedTarget = getErasedSignature(overload);

            // First see if the return types are compatible in either direction.
            const sourceReturnType = getReturnTypeOfSignature(erasedSource);
            const targetReturnType = getReturnTypeOfSignature(erasedTarget);
            if (targetReturnType === voidType
                || isTypeRelatedTo(targetReturnType, sourceReturnType, assignableRelation)
                || isTypeRelatedTo(sourceReturnType, targetReturnType, assignableRelation)) {

                return isSignatureAssignableTo(erasedSource, erasedTarget, /*ignoreReturnTypes*/ true);
            }

            return false;
        }

        function isEmptyResolvedType(t: ResolvedType) {
            return t !== anyFunctionType &&
                t.properties.length === 0 &&
                t.callSignatures.length === 0 &&
                t.constructSignatures.length === 0 &&
                !t.stringIndexInfo &&
                !t.numberIndexInfo;
        }

        function isEmptyObjectType(type: Type): boolean {
            return type.flags & TypeFlags.Object ? !isGenericMappedType(type) && isEmptyResolvedType(resolveStructuredTypeMembers(<ObjectType>type)) :
                type.flags & TypeFlags.NonPrimitive ? true :
                type.flags & TypeFlags.Union ? some((<UnionType>type).types, isEmptyObjectType) :
                type.flags & TypeFlags.Intersection ? every((<UnionType>type).types, isEmptyObjectType) :
                false;
        }

        function isEmptyAnonymousObjectType(type: Type) {
            return !!(getObjectFlags(type) & ObjectFlags.Anonymous && (
                (<ResolvedType>type).members && isEmptyResolvedType(<ResolvedType>type) ||
                type.symbol && type.symbol.flags & SymbolFlags.TypeLiteral && getMembersOfSymbol(type.symbol).size === 0));
        }

        function isStringIndexSignatureOnlyType(type: Type): boolean {
            return type.flags & TypeFlags.Object && !isGenericMappedType(type) && getPropertiesOfType(type).length === 0 && getIndexInfoOfType(type, IndexKind.String) && !getIndexInfoOfType(type, IndexKind.Number) ||
                type.flags & TypeFlags.UnionOrIntersection && every((<UnionOrIntersectionType>type).types, isStringIndexSignatureOnlyType) ||
                false;
        }

        function isEnumTypeRelatedTo(sourceSymbol: Symbol, targetSymbol: Symbol, errorReporter?: ErrorReporter) {
            if (sourceSymbol === targetSymbol) {
                return true;
            }
            const id = getSymbolId(sourceSymbol) + "," + getSymbolId(targetSymbol);
            const entry = enumRelation.get(id);
            if (entry !== undefined && !(!(entry & RelationComparisonResult.Reported) && entry & RelationComparisonResult.Failed && errorReporter)) {
                return !!(entry & RelationComparisonResult.Succeeded);
            }
            if (sourceSymbol.escapedName !== targetSymbol.escapedName || !(sourceSymbol.flags & SymbolFlags.RegularEnum) || !(targetSymbol.flags & SymbolFlags.RegularEnum)) {
                enumRelation.set(id, RelationComparisonResult.Failed | RelationComparisonResult.Reported);
                return false;
            }
            const targetEnumType = getTypeOfSymbol(targetSymbol);
            for (const property of getPropertiesOfType(getTypeOfSymbol(sourceSymbol))) {
                if (property.flags & SymbolFlags.EnumMember) {
                    const targetProperty = getPropertyOfType(targetEnumType, property.escapedName);
                    if (!targetProperty || !(targetProperty.flags & SymbolFlags.EnumMember)) {
                        if (errorReporter) {
                            errorReporter(Diagnostics.Property_0_is_missing_in_type_1, symbolName(property),
                                typeToString(getDeclaredTypeOfSymbol(targetSymbol), /*enclosingDeclaration*/ undefined, TypeFormatFlags.UseFullyQualifiedType));
                            enumRelation.set(id, RelationComparisonResult.Failed | RelationComparisonResult.Reported);
                        }
                        else {
                            enumRelation.set(id, RelationComparisonResult.Failed);
                        }
                        return false;
                    }
                }
            }
            enumRelation.set(id, RelationComparisonResult.Succeeded);
            return true;
        }

        function isSimpleTypeRelatedTo(source: Type, target: Type, relation: ESMap<string, RelationComparisonResult>, errorReporter?: ErrorReporter) {
            const s = source.flags;
            const t = target.flags;
            if (t & TypeFlags.AnyOrUnknown || s & TypeFlags.Never || source === wildcardType) return true;
            if (t & TypeFlags.Never) return false;
            if (s & TypeFlags.StringLike && t & TypeFlags.String) return true;
            if (s & TypeFlags.StringLiteral && s & TypeFlags.EnumLiteral &&
                t & TypeFlags.StringLiteral && !(t & TypeFlags.EnumLiteral) &&
                (<StringLiteralType>source).value === (<StringLiteralType>target).value) return true;
            if (s & TypeFlags.NumberLike && t & TypeFlags.Number) return true;
            if (s & TypeFlags.NumberLiteral && s & TypeFlags.EnumLiteral &&
                t & TypeFlags.NumberLiteral && !(t & TypeFlags.EnumLiteral) &&
                (<NumberLiteralType>source).value === (<NumberLiteralType>target).value) return true;
            if (s & TypeFlags.BigIntLike && t & TypeFlags.BigInt) return true;
            if (s & TypeFlags.BooleanLike && t & TypeFlags.Boolean) return true;
            if (s & TypeFlags.ESSymbolLike && t & TypeFlags.ESSymbol) return true;
            if (s & TypeFlags.Enum && t & TypeFlags.Enum && isEnumTypeRelatedTo(source.symbol, target.symbol, errorReporter)) return true;
            if (s & TypeFlags.EnumLiteral && t & TypeFlags.EnumLiteral) {
                if (s & TypeFlags.Union && t & TypeFlags.Union && isEnumTypeRelatedTo(source.symbol, target.symbol, errorReporter)) return true;
                if (s & TypeFlags.Literal && t & TypeFlags.Literal &&
                    (<LiteralType>source).value === (<LiteralType>target).value &&
                    isEnumTypeRelatedTo(getParentOfSymbol(source.symbol)!, getParentOfSymbol(target.symbol)!, errorReporter)) return true;
            }
            if (s & TypeFlags.Undefined && (!strictNullChecks || t & (TypeFlags.Undefined | TypeFlags.Void))) return true;
            if (s & TypeFlags.Null && (!strictNullChecks || t & TypeFlags.Null)) return true;
            if (s & TypeFlags.Object && t & TypeFlags.NonPrimitive) return true;
            if (relation === assignableRelation || relation === comparableRelation) {
                if (s & TypeFlags.Any) return true;
                // Type number or any numeric literal type is assignable to any numeric enum type or any
                // numeric enum literal type. This rule exists for backwards compatibility reasons because
                // bit-flag enum types sometimes look like literal enum types with numeric literal values.
                if (s & (TypeFlags.Number | TypeFlags.NumberLiteral) && !(s & TypeFlags.EnumLiteral) && (
                    t & TypeFlags.Enum || t & TypeFlags.NumberLiteral && t & TypeFlags.EnumLiteral)) return true;
            }
            return false;
        }

        function isTypeRelatedTo(source: Type, target: Type, relation: ESMap<string, RelationComparisonResult>) {
            if (isFreshLiteralType(source)) {
                source = (<FreshableType>source).regularType;
            }
            if (isFreshLiteralType(target)) {
                target = (<FreshableType>target).regularType;
            }
            if (source === target) {
                return true;
            }
            if (relation !== identityRelation) {
                if (relation === comparableRelation && !(target.flags & TypeFlags.Never) && isSimpleTypeRelatedTo(target, source, relation) || isSimpleTypeRelatedTo(source, target, relation)) {
                    return true;
                }
            }
            else {
                if (!(source.flags & TypeFlags.UnionOrIntersection) && !(target.flags & TypeFlags.UnionOrIntersection) &&
                    source.flags !== target.flags && !(source.flags & TypeFlags.Substructure)) return false;
            }
            if (source.flags & TypeFlags.Object && target.flags & TypeFlags.Object) {
                const related = relation.get(getRelationKey(source, target, IntersectionState.None, relation));
                if (related !== undefined) {
                    return !!(related & RelationComparisonResult.Succeeded);
                }
            }
            if (source.flags & TypeFlags.StructuredOrInstantiable || target.flags & TypeFlags.StructuredOrInstantiable) {
                return checkTypeRelatedTo(source, target, relation, /*errorNode*/ undefined);
            }
            return false;
        }

        function isIgnoredJsxProperty(source: Type, sourceProp: Symbol) {
            return getObjectFlags(source) & ObjectFlags.JsxAttributes && !isUnhyphenatedJsxName(sourceProp.escapedName);
        }

        function getNormalizedType(type: Type, writing: boolean): Type {
            while (true) {
                const t = isFreshLiteralType(type) ? (<FreshableType>type).regularType :
                    getObjectFlags(type) & ObjectFlags.Reference && (<TypeReference>type).node ? createTypeReference((<TypeReference>type).target, getTypeArguments(<TypeReference>type)) :
                    type.flags & TypeFlags.UnionOrIntersection ? getReducedType(type) :
                    type.flags & TypeFlags.Substitution ? writing ? (<SubstitutionType>type).baseType : (<SubstitutionType>type).substitute :
                    type.flags & TypeFlags.Simplifiable ? getSimplifiedType(type, writing) :
                    type;
                if (t === type) break;
                type = t;
            }
            return type;
        }

        /**
         * Checks if 'source' is related to 'target' (e.g.: is a assignable to).
         * @param source The left-hand-side of the relation.
         * @param target The right-hand-side of the relation.
         * @param relation The relation considered. One of 'identityRelation', 'subtypeRelation', 'assignableRelation', or 'comparableRelation'.
         * Used as both to determine which checks are performed and as a cache of previously computed results.
         * @param errorNode The suggested node upon which all errors will be reported, if defined. This may or may not be the actual node used.
         * @param headMessage If the error chain should be prepended by a head message, then headMessage will be used.
         * @param containingMessageChain A chain of errors to prepend any new errors found.
         * @param errorOutputContainer Return the diagnostic. Do not log if 'skipLogging' is truthy.
         */
        function checkTypeRelatedTo(
            source: Type,
            target: Type,
            relation: ESMap<string, RelationComparisonResult>,
            errorNode: Node | undefined,
            headMessage?: DiagnosticMessage,
            containingMessageChain?: () => DiagnosticMessageChain | undefined,
            errorOutputContainer?: { errors?: Diagnostic[], skipLogging?: boolean },
        ): boolean {
            let errorInfo: DiagnosticMessageChain | undefined;
            let relatedInfo: [DiagnosticRelatedInformation, ...DiagnosticRelatedInformation[]] | undefined;
            let maybeKeys: string[];
            let sourceStack: Type[];
            let targetStack: Type[];
            let maybeCount = 0;
            let depth = 0;
            let expandingFlags = ExpandingFlags.None;
            let overflow = false;
            let overrideNextErrorInfo = 0; // How many `reportRelationError` calls should be skipped in the elaboration pyramid
            let lastSkippedInfo: [Type, Type] | undefined;
            let incompatibleStack: [DiagnosticMessage, (string | number)?, (string | number)?, (string | number)?, (string | number)?][] = [];
            let inPropertyCheck = false;

            Debug.assert(relation !== identityRelation || !errorNode, "no error reporting in identity checking");

            const result = isRelatedTo(source, target, /*reportErrors*/ !!errorNode, headMessage);
            if (incompatibleStack.length) {
                reportIncompatibleStack();
            }
            if (overflow) {
                const diag = error(errorNode || currentNode, Diagnostics.Excessive_stack_depth_comparing_types_0_and_1, typeToString(source), typeToString(target));
                if (errorOutputContainer) {
                    (errorOutputContainer.errors || (errorOutputContainer.errors = [])).push(diag);
                }
            }
            else if (errorInfo) {
                if (containingMessageChain) {
                    const chain = containingMessageChain();
                    if (chain) {
                        concatenateDiagnosticMessageChains(chain, errorInfo);
                        errorInfo = chain;
                    }
                }

                let relatedInformation: DiagnosticRelatedInformation[] | undefined;
                // Check if we should issue an extra diagnostic to produce a quickfix for a slightly incorrect import statement
                if (headMessage && errorNode && !result && source.symbol) {
                    const links = getSymbolLinks(source.symbol);
                    if (links.originatingImport && !isImportCall(links.originatingImport)) {
                        const helpfulRetry = checkTypeRelatedTo(getTypeOfSymbol(links.target!), target, relation, /*errorNode*/ undefined);
                        if (helpfulRetry) {
                            // Likely an incorrect import. Issue a helpful diagnostic to produce a quickfix to change the import
                            const diag = createDiagnosticForNode(links.originatingImport, Diagnostics.Type_originates_at_this_import_A_namespace_style_import_cannot_be_called_or_constructed_and_will_cause_a_failure_at_runtime_Consider_using_a_default_import_or_import_require_here_instead);
                            relatedInformation = append(relatedInformation, diag); // Cause the error to appear with the error that triggered it
                        }
                    }
                }
                const diag = createDiagnosticForNodeFromMessageChain(errorNode!, errorInfo, relatedInformation);
                if (relatedInfo) {
                    addRelatedInfo(diag, ...relatedInfo);
                }
                if (errorOutputContainer) {
                    (errorOutputContainer.errors || (errorOutputContainer.errors = [])).push(diag);
                }
                if (!errorOutputContainer || !errorOutputContainer.skipLogging) {
                    diagnostics.add(diag);
                }
            }
            if (errorNode && errorOutputContainer && errorOutputContainer.skipLogging && result === Ternary.False) {
                Debug.assert(!!errorOutputContainer.errors, "missed opportunity to interact with error.");
            }
            return result !== Ternary.False;

            function resetErrorInfo(saved: ReturnType<typeof captureErrorCalculationState>) {
                errorInfo = saved.errorInfo;
                lastSkippedInfo = saved.lastSkippedInfo;
                incompatibleStack = saved.incompatibleStack;
                overrideNextErrorInfo = saved.overrideNextErrorInfo;
                relatedInfo = saved.relatedInfo;
            }

            function captureErrorCalculationState() {
                return {
                    errorInfo,
                    lastSkippedInfo,
                    incompatibleStack: incompatibleStack.slice(),
                    overrideNextErrorInfo,
                    relatedInfo: !relatedInfo ? undefined : relatedInfo.slice() as ([DiagnosticRelatedInformation, ...DiagnosticRelatedInformation[]] | undefined)
                };
            }

            function reportIncompatibleError(message: DiagnosticMessage, arg0?: string | number, arg1?: string | number, arg2?: string | number, arg3?: string | number) {
                overrideNextErrorInfo++; // Suppress the next relation error
                lastSkippedInfo = undefined; // Reset skipped info cache
                incompatibleStack.push([message, arg0, arg1, arg2, arg3]);
            }

            function reportIncompatibleStack() {
                const stack = incompatibleStack;
                incompatibleStack = [];
                const info = lastSkippedInfo;
                lastSkippedInfo = undefined;
                if (stack.length === 1) {
                    reportError(...stack[0]);
                    if (info) {
                        // Actually do the last relation error
                        reportRelationError(/*headMessage*/ undefined, ...info);
                    }
                    return;
                }
                // The first error will be the innermost, while the last will be the outermost - so by popping off the end,
                // we can build from left to right
                let path = "";
                const secondaryRootErrors: typeof incompatibleStack = [];
                while (stack.length) {
                    const [msg, ...args] = stack.pop()!;
                    switch (msg.code) {
                        case Diagnostics.Types_of_property_0_are_incompatible.code: {
                            // Parenthesize a `new` if there is one
                            if (path.indexOf("new ") === 0) {
                                path = `(${path})`;
                            }
                            const str = "" + args[0];
                            // If leading, just print back the arg (irrespective of if it's a valid identifier)
                            if (path.length === 0) {
                                path = `${str}`;
                            }
                            // Otherwise write a dotted name if possible
                            else if (isIdentifierText(str, compilerOptions.target)) {
                                path = `${path}.${str}`;
                            }
                            // Failing that, check if the name is already a computed name
                            else if (str[0] === "[" && str[str.length - 1] === "]") {
                                path = `${path}${str}`;
                            }
                            // And finally write out a computed name as a last resort
                            else {
                                path = `${path}[${str}]`;
                            }
                            break;
                        }
                        case Diagnostics.Call_signature_return_types_0_and_1_are_incompatible.code:
                        case Diagnostics.Construct_signature_return_types_0_and_1_are_incompatible.code:
                        case Diagnostics.Call_signatures_with_no_arguments_have_incompatible_return_types_0_and_1.code:
                        case Diagnostics.Construct_signatures_with_no_arguments_have_incompatible_return_types_0_and_1.code: {
                            if (path.length === 0) {
                                // Don't flatten signature compatability errors at the start of a chain - instead prefer
                                // to unify (the with no arguments bit is excessive for printback) and print them back
                                let mappedMsg = msg;
                                if (msg.code === Diagnostics.Call_signatures_with_no_arguments_have_incompatible_return_types_0_and_1.code) {
                                    mappedMsg = Diagnostics.Call_signature_return_types_0_and_1_are_incompatible;
                                }
                                else if (msg.code === Diagnostics.Construct_signatures_with_no_arguments_have_incompatible_return_types_0_and_1.code) {
                                    mappedMsg = Diagnostics.Construct_signature_return_types_0_and_1_are_incompatible;
                                }
                                secondaryRootErrors.unshift([mappedMsg, args[0], args[1]]);
                            }
                            else {
                                const prefix = (msg.code === Diagnostics.Construct_signature_return_types_0_and_1_are_incompatible.code ||
                                    msg.code === Diagnostics.Construct_signatures_with_no_arguments_have_incompatible_return_types_0_and_1.code)
                                        ? "new "
                                        : "";
                                const params = (msg.code === Diagnostics.Call_signatures_with_no_arguments_have_incompatible_return_types_0_and_1.code ||
                                    msg.code === Diagnostics.Construct_signatures_with_no_arguments_have_incompatible_return_types_0_and_1.code)
                                        ? ""
                                        : "...";
                                path = `${prefix}${path}(${params})`;
                            }
                            break;
                        }
                        default:
                            return Debug.fail(`Unhandled Diagnostic: ${msg.code}`);
                    }
                }
                if (path) {
                    reportError(path[path.length - 1] === ")"
                        ? Diagnostics.The_types_returned_by_0_are_incompatible_between_these_types
                        : Diagnostics.The_types_of_0_are_incompatible_between_these_types,
                        path
                    );
                }
                else {
                    // Remove the innermost secondary error as it will duplicate the error already reported by `reportRelationError` on entry
                    secondaryRootErrors.shift();
                }
                for (const [msg, ...args] of secondaryRootErrors) {
                    const originalValue = msg.elidedInCompatabilityPyramid;
                    msg.elidedInCompatabilityPyramid = false; // Teporarily override elision to ensure error is reported
                    reportError(msg, ...args);
                    msg.elidedInCompatabilityPyramid = originalValue;
                }
                if (info) {
                    // Actually do the last relation error
                    reportRelationError(/*headMessage*/ undefined, ...info);
                }
            }

            function reportError(message: DiagnosticMessage, arg0?: string | number, arg1?: string | number, arg2?: string | number, arg3?: string | number): void {
                Debug.assert(!!errorNode);
                if (incompatibleStack.length) reportIncompatibleStack();
                if (message.elidedInCompatabilityPyramid) return;
                errorInfo = chainDiagnosticMessages(errorInfo, message, arg0, arg1, arg2, arg3);
            }

            function associateRelatedInfo(info: DiagnosticRelatedInformation) {
                Debug.assert(!!errorInfo);
                if (!relatedInfo) {
                    relatedInfo = [info];
                }
                else {
                    relatedInfo.push(info);
                }
            }

            function reportRelationError(message: DiagnosticMessage | undefined, source: Type, target: Type) {
                if (incompatibleStack.length) reportIncompatibleStack();
                const [sourceType, targetType] = getTypeNamesForErrorDisplay(source, target);
                let generalizedSource = source;
                let generalizedSourceType = sourceType;

                if (isLiteralType(source) && !typeCouldHaveTopLevelSingletonTypes(target)) {
                    generalizedSource = getBaseTypeOfLiteralType(source);
                    Debug.assert(!isTypeAssignableTo(generalizedSource, target), "generalized source shouldn't be assignable");
                    generalizedSourceType = getTypeNameForErrorDisplay(generalizedSource);
                }

                if (target.flags & TypeFlags.TypeParameter) {
                    const constraint = getBaseConstraintOfType(target);
                    let needsOriginalSource;
                    if (constraint && (isTypeAssignableTo(generalizedSource, constraint) || (needsOriginalSource = isTypeAssignableTo(source, constraint)))) {
                        reportError(
                            Diagnostics._0_is_assignable_to_the_constraint_of_type_1_but_1_could_be_instantiated_with_a_different_subtype_of_constraint_2,
                            needsOriginalSource ? sourceType : generalizedSourceType,
                            targetType,
                            typeToString(constraint),
                        );
                    }
                    else {
                        reportError(
                            Diagnostics._0_could_be_instantiated_with_an_arbitrary_type_which_could_be_unrelated_to_1,
                            targetType,
                            generalizedSourceType
                        );
                    }
                }

                if (!message) {
                    if (relation === comparableRelation) {
                        message = Diagnostics.Type_0_is_not_comparable_to_type_1;
                    }
                    else if (sourceType === targetType) {
                        message = Diagnostics.Type_0_is_not_assignable_to_type_1_Two_different_types_with_this_name_exist_but_they_are_unrelated;
                    }
                    else {
                        message = Diagnostics.Type_0_is_not_assignable_to_type_1;
                    }
                }

                reportError(message, generalizedSourceType, targetType);
            }

            function tryElaborateErrorsForPrimitivesAndObjects(source: Type, target: Type) {
                const sourceType = symbolValueDeclarationIsContextSensitive(source.symbol) ? typeToString(source, source.symbol.valueDeclaration) : typeToString(source);
                const targetType = symbolValueDeclarationIsContextSensitive(target.symbol) ? typeToString(target, target.symbol.valueDeclaration) : typeToString(target);

                if ((globalStringType === source && stringType === target) ||
                    (globalNumberType === source && numberType === target) ||
                    (globalBooleanType === source && booleanType === target) ||
                    (getGlobalESSymbolType(/*reportErrors*/ false) === source && esSymbolType === target)) {
                    reportError(Diagnostics._0_is_a_primitive_but_1_is_a_wrapper_object_Prefer_using_0_when_possible, targetType, sourceType);
                }
            }

            /**
             * Try and elaborate array and tuple errors. Returns false
             * if we have found an elaboration, or we should ignore
             * any other elaborations when relating the `source` and
             * `target` types.
             */
            function tryElaborateArrayLikeErrors(source: Type, target: Type, reportErrors: boolean): boolean {
                /**
                 * The spec for elaboration is:
                 * - If the source is a readonly tuple and the target is a mutable array or tuple, elaborate on mutability and skip property elaborations.
                 * - If the source is a tuple then skip property elaborations if the target is an array or tuple.
                 * - If the source is a readonly array and the target is a mutable array or tuple, elaborate on mutability and skip property elaborations.
                 * - If the source an array then skip property elaborations if the target is a tuple.
                 */
                if (isTupleType(source)) {
                    if (source.target.readonly && isMutableArrayOrTuple(target)) {
                        if (reportErrors) {
                            reportError(Diagnostics.The_type_0_is_readonly_and_cannot_be_assigned_to_the_mutable_type_1, ty