/* program window
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

/* Trace tree rebuilds too.
#define DEBUG_TREE
 */

#include "ip.h"

#include "BITMAPS/toolbox_open.xpm"
#include "BITMAPS/toolbox_closed.xpm"
#include "BITMAPS/tools.xpm"
#include "BITMAPS/separator.xpm"
#include "BITMAPS/floppy.xpm"

static iWindowClass *parent_class = NULL;

static GSList *program_all = NULL;

static GdkPixmap *program_open = NULL;
static GdkBitmap *program_open_mask = NULL;
static GdkPixmap *program_closed = NULL;
static GdkBitmap *program_closed_mask = NULL;
static GdkPixmap *program_item = NULL;
static GdkBitmap *program_item_mask = NULL;
static GdkPixmap *program_sep = NULL;
static GdkBitmap *program_sep_mask = NULL;
static GdkPixmap *program_col = NULL;
static GdkBitmap *program_col_mask = NULL;

static GtkWidget *program_menu = NULL;

static void *
program_freeze_sub( Program *program )
{
	gtk_clist_freeze( GTK_CLIST( program->ctree ) );

	return( NULL );
}

/* Freeze all program windows. This is used (eg.) during toolkit load to stop
 * the tree views going mad.
 */
void
program_freeze( void )
{
	slist_map( program_all, (SListMapFn) program_freeze_sub, NULL );
}

static void *
program_thaw_sub( Program *program )
{
	gtk_clist_thaw( GTK_CLIST( program->ctree ) );

	return( NULL );
}

void
program_thaw( void )
{
	slist_map( program_all, (SListMapFn) program_thaw_sub, NULL );
}

/* Update the title.
 */
static void
program_title( Program *program )
{
	BufInfo buf;
	char txt[512];

#ifdef DEBUG
	printf( "program_title\n" );
#endif /*DEBUG*/

	buf_init_static( &buf, txt, 512 );
	buf_appendf( &buf, _( "Program" ) );
	if( program->kit ) {
		buf_appendf( &buf, " - %s", IOBJECT( program->kit )->name );

		if( FILEMODEL( program->kit )->modified ) {
			buf_appendf( &buf, " [" );
			buf_appendf( &buf, _( "modified" ) );
			buf_appendf( &buf, "]" );
		}
	}
	if( program->tool )
		buf_appendf( &buf, " - %s", IOBJECT( program->tool )->name );

	iwindow_set_title( IWINDOW( program ), buf_all( &buf ) );
}

static void
program_info( Program *program, BufInfo *buf )
{
	buf_appendf( buf, _( "Edit window" ) );
	buf_appendf( buf, "\n" );
	buf_appendf( buf, "dirty = \"%s\"\n", bool_to_char( program->dirty ) );
	buf_appendf( buf, "\n" );

	if( program->kit ) {
		iobject_info( IOBJECT( program->kit ), buf );
		buf_appendf( buf, "\n" );
	}
	if( program->tool ) {
		iobject_info( IOBJECT( program->tool ), buf );
		buf_appendf( buf, "\n" );
	}
}

/* Deselect the current symbol and kit.
 */
static void
program_deselect( Program *program )
{
	if( program->sym ) {
		FREESID( program->sym_changed_sid, program->sym );
		if( GTK_WIDGET_REALIZED( program->text ) )
			gtk_editable_delete_text( 
				GTK_EDITABLE( program->text ), 0, -1 );
		program->dirty = FALSE;
		program->sym = NULL;
	}

	if( program->tool ) {
		program->pos = -1;
		FREESID( program->tool_destroy_sid, program->tool );
		program->tool = NULL;
	}

	if( program->kit ) {
		FREESID( program->kit_destroy_sid, program->kit );
		program->kit = NULL;
	}

	program_title( program );
}

static void
program_find_reset( Program *program )
{
	FREESID( program->find_sym_destroy_sid, program->find_sym );
	program->find_sym = NULL;
	program->find_start = 0;
	program->find_end = 0;
}

static void
program_find_destroy_cb( Symbol *sym, Program *program )
{
	program_find_reset( program );
}

static void
program_find_note( Program *program, Symbol *sym, int start, int end )
{
	program_find_reset( program );

	program->find_sym = sym;
	program->find_sym_destroy_sid = 
		g_signal_connect( G_OBJECT( sym ), "destroy",
			G_CALLBACK( program_find_destroy_cb ), program );
	program->find_start = start;
	program->find_end = end;
}

static gboolean
program_find_pos( Program *program, const char *text, int *start, int *end )
{
#ifdef HAVE_REGEXEC
	if( program->regexp ) {
		regmatch_t matches[1];

		if( !regexec( program->comp, text, 1, matches, 0 ) ) {
			*start = matches[0].rm_so;
			*end = matches[0].rm_eo;

			return( TRUE );
		}
	}
	else 
#endif /*HAVE_REGEXEC*/
	if( program->csens ) {
		char *p;

		if( (p = strstr( text, program->search )) ) {
			*start = p - text;
			*end = *start + strlen( program->search );

			return( TRUE );
		}
	}
	else {
		char *p;

		if( (p = strcasestr( text, program->search )) ) {
			*start = p - text;
			*end = *start + strlen( program->search );

			return( TRUE );
		}
	}

	return( FALSE );
}

static void *
program_find_tool( Tool *tool, Program *program, gboolean *skipping )
{
	Symbol *sym;

	if( tool->type != TOOL_SYM )
		return( NULL );
	sym = tool->sym;

	/* In search mode? Check if we've found the start point.
	 */
	if( *skipping ) {
		if( sym == program->find_sym || !program->find_sym )
			*skipping = FALSE;
	}

	/* Reached start point? Check from start onwards.
	 */
	if( !*skipping ) {
		if( sym->expr && sym->expr->compile && 
			program->find_start < 
				strlen( sym->expr->compile->text ) ) {
			int start, end;

			if( program_find_pos( program, 
				sym->expr->compile->text + program->find_start, 
				&start, &end ) ) {
				program_find_note( program, sym, 
					start + program->find_start, 
					end + program->find_start );
				return( tool );
			}
		}

		program_find_reset( program );
	}

	return( NULL );
}

static void *
program_find_toolkit( Toolkit *kit, Program *program, gboolean *skipping )
{
	return( icontainer_map( ICONTAINER( kit ), 
		(icontainer_map_fn) program_find_tool, program, &skipping ) );
}

static gboolean
program_find( Program *program )
{
	gboolean skipping = TRUE;

	if( toolkitgroup_map( program->kitg,
		(toolkit_map_fn) program_find_toolkit, program, &skipping ) )
		return( TRUE );

	return( FALSE );
}

static void
program_destroy( GtkObject *object )
{
	Program *program;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_PROGRAM( object ) );

	program = PROGRAM( object );

#ifdef DEBUG
	printf( "program_destroy\n" );
#endif /*DEBUG*/

	/* My instance destroy stuff.
	 */
	program_deselect( program );
	FREESID( program->kitgroup_changed_sid, program->kitg );
	FREESID( program->kitgroup_destroy_sid, program->kitg );
	UNREF( program->ifac );

	FREEF( g_free, program->search );
#ifdef HAVE_REGEXEC
	FREEF( regfree, program->comp );
#endif /*HAVE_REGEXEC*/

	program_find_reset( program );

	program_all = g_slist_remove( program_all, program );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

static void
program_edit_dia_done_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	Tool *tool = TOOL( client );
	Stringset *ss = STRINGSET( iwnd );
	StringsetChild *name = stringset_child_get( ss, _( "Name" ) );
	StringsetChild *file = stringset_child_get( ss, _( "Filename" ) );

	char name_text[1024];
	char file_text[1024];

	if( !get_geditable_string( name->entry, name_text, 1024 ) ||
		!get_geditable_filename( file->entry, file_text, 1024 ) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	if( !tool_new_dia( tool->kit, 
		ICONTAINER( tool )->pos, name_text, file_text ) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	nfn( sys, IWINDOW_TRUE );
}

static void
program_edit_dia( Program *program, Tool *tool )
{
	GtkWidget *ss = stringset_new();

	assert( tool->type == TOOL_DIA );

	stringset_child_new( STRINGSET( ss ), 
		_( "Name" ), IOBJECT( tool )->name, _( "Menu item text" ) );
	stringset_child_new( STRINGSET( ss ), 
		_( "Filename" ), FILEMODEL( tool )->filename, 
		_( "Load column from this file" ) );

	iwindow_set_title( IWINDOW( ss ), _( "Edit Column Item \"%s\"" ), 
		IOBJECT( tool )->name );
	idialog_set_callbacks( IDIALOG( ss ), 
		iwindow_true_cb, NULL, NULL, tool );
	idialog_set_help_tag( IDIALOG( ss ), "sec:diaref" );
	idialog_add_ok( IDIALOG( ss ), 
		program_edit_dia_done_cb, _( "Set column item" ) );
	idialog_set_parent( IDIALOG( ss ), GTK_WIDGET( program ) );
	idialog_set_iobject( IDIALOG( ss ), IOBJECT( tool ) );
	iwindow_build( IWINDOW( ss ) );

	gtk_widget_show( ss );
}

static void
program_edit_object( GtkWidget *menu, Program *program, GtkCTreeNode *node )
{
	GtkCTree *ctree = GTK_CTREE( program->ctree );
	Model *model = MODEL( gtk_ctree_node_get_row_data( ctree, node ) );

	program_select( program, model );

	if( IS_TOOL( model ) ) {
		Tool *tool = TOOL( model );

		/* Edit these peeps with a dialog.
		 */
		if( tool->type == TOOL_DIA ) {
			program_edit_dia( program, tool );
		}
	}
}

static gboolean
program_is_saveable( Model *model )
{
	if( !IS_TOOLKIT( model ) ) {
		error_top( _( "Unable to save." ) );
		error_sub( _( "You can only save toolkits, not tools." ) );
		return( FALSE );
	}

	if( IS_TOOLKIT( model ) && TOOLKIT( model )->pseudo ) {
		error_top( _( "Unable to save." ) );
		error_sub( _( "You can't save auto-generated toolkits." ) );
		return( FALSE );
	}

	return( TRUE );
}

static void
program_save_object( GtkWidget *menu, Program *program, GtkCTreeNode *node )
{
	GtkCTree *ctree = GTK_CTREE( program->ctree );
	Model *model = MODEL( gtk_ctree_node_get_row_data( ctree, node ) );

	if( !program_is_saveable( model ) ) {
		box_alert( GTK_WIDGET( program ) );
		return;
	}

	filemodel_inter_save( IWINDOW( program ), FILEMODEL( model ) );
}

static void
program_saveas_object( GtkWidget *menu, Program *program, GtkCTreeNode *node )
{
	GtkCTree *ctree = GTK_CTREE( program->ctree );
	Model *model = MODEL( gtk_ctree_node_get_row_data( ctree, node ) );

	if( !program_is_saveable( model ) ) {
		box_alert( GTK_WIDGET( program ) );
		return;
	}

	filemodel_inter_saveas( IWINDOW( program ), FILEMODEL( model ) );
}

static void
program_remove_object( GtkWidget *menu, Program *program, GtkCTreeNode *node )
{
	GtkCTree *ctree = GTK_CTREE( program->ctree );
	Model *model = MODEL( gtk_ctree_node_get_row_data( ctree, node ) );

	model_check_destroy( GTK_WIDGET( program ), model );
}

static void
program_class_init( ProgramClass *class )
{
	GtkObjectClass *object_class = (GtkObjectClass *) class;

	GtkWidget *pane;

	parent_class = g_type_class_peek_parent( class );

	object_class->destroy = program_destroy;

	/* Create signals.
	 */

	/* Init methods.
	 */

	/* Static init.
	 */
	program_open = gdk_pixmap_create_from_xpm_d( main_window_gdk,
		&program_open_mask, NULL, toolbox_open );
	program_closed = gdk_pixmap_create_from_xpm_d( main_window_gdk,
		&program_closed_mask, NULL, toolbox_closed );
	program_item = gdk_pixmap_create_from_xpm_d( main_window_gdk,
		&program_item_mask, NULL, tools );
	program_sep = gdk_pixmap_create_from_xpm_d( main_window_gdk,
		&program_sep_mask, NULL, separator );
	program_col = gdk_pixmap_create_from_xpm_d( main_window_gdk,
		&program_col_mask, NULL, floppy );

	pane = program_menu = popup_build( _( "Toolkit tree menu" ) );
	popup_add_but( pane, _( "Edit ..." ), 
		POPUP_FUNC( program_edit_object ) );
	popup_add_but( pane, GTK_STOCK_SAVE,
		POPUP_FUNC( program_save_object ) );
	popup_add_but( pane, GTK_STOCK_SAVE_AS,
		POPUP_FUNC( program_saveas_object ) );
	menu_add_sep( pane );
	popup_add_but( pane, GTK_STOCK_CLOSE,
		POPUP_FUNC( program_remove_object ) );
}

/* We store Symbol pointers in the tree for TOOL_SYM tools, and Tool pointers
 * for everything else. Get and set these guys.
 *
 * We store Symbol pointers so that tools don't vanish on parse.
 */
static Tool *
program_get_tool_data( Program *program, GtkCTreeNode *node )
{
	gpointer data = gtk_ctree_node_get_row_data( 
		GTK_CTREE( program->ctree ), node );

	if( !data )
		return( NULL );
	if( IS_TOOL( data ) )
		return( TOOL( data ) );
	if( IS_SYMBOL( data ) )
		return( SYMBOL( data )->tool );

	/* If the symbol/whatever has been deleted, we can get here.
	 */
	return( NULL );
}

static void
program_set_tool_data( Program *program, GtkCTreeNode *node, Tool *tool )
{
	gpointer data;

	switch( tool->type ) {
	case TOOL_SYM:
		data = tool->sym;
		break;

	case TOOL_DIA:
	case TOOL_SEP:
		data = tool;
		break;

	default:
		assert( FALSE );
	}

	gtk_ctree_node_set_row_data( GTK_CTREE( program->ctree ), node, data );
}

static void *
program_tree_tool_test( GtkCTreeNode *node, Tool *tool, Program *program )
{
	if( program_get_tool_data( program, node ) == tool )
		return( node );

	return( NULL );
}

static void *
program_tree_kit_test( GtkCTreeNode *node, Toolkit *kit, Program *program )
{
	if( gtk_ctree_node_get_row_data( GTK_CTREE( program->ctree ), node ) ==
		kit )
		return( node );

	return( NULL );
}

static void
program_tree_sym_list( GtkCTree *ctree, GtkCTreeNode *node, 
	GSList **tool_notused )
{
	gboolean is_leaf;

	gtk_ctree_get_node_info( ctree, node,
		NULL, NULL, NULL, NULL, NULL, NULL, &is_leaf, NULL );

	/* Called once for enclosing kit too. Can't check data, it may be
 	 * junk.
	 */
	if( is_leaf )
		*tool_notused = g_slist_prepend( *tool_notused, node );
}

static void
program_node_new( Program *program, Tool *tool )
{
	GtkCTreeNode *node;
	char *text[1];

	switch( tool->type ) {
	case TOOL_SYM:
		text[0] = IOBJECT( tool )->name;
		node = gtk_ctree_insert_node( GTK_CTREE( program->ctree ),
			program->parent, program->sibling, 
			text, 5, 
			program_item, program_item_mask, 
			NULL, NULL,
			TRUE, TRUE );
		break;

	case TOOL_SEP:
		text[0] = "";
		node = gtk_ctree_insert_node( GTK_CTREE( program->ctree ),
			program->parent, program->sibling, 
			text, 5, 
			program_sep, program_sep_mask, 
			NULL, NULL,
			TRUE, TRUE );
		break;

	case TOOL_DIA:
		text[0] = IOBJECT( tool )->name;
		node = gtk_ctree_insert_node( GTK_CTREE( program->ctree ),
			program->parent, program->sibling, 
			text, 5, 
			program_col, program_col_mask, 
			NULL, NULL,
			TRUE, TRUE );
		break;

	default:
		assert( FALSE );
	}

	program->sibling = node;
	program_set_tool_data( program, node, tool );

#ifdef DEBUG
	printf( "program_node_new: %s\n", IOBJECT( tool )->name );
#endif /*DEBUG*/
}

static void *
program_tree_tool_update( Tool *tool, Program *program, GSList **tool_notused )
{
	GtkCTreeNode *node;

	/* Do we have a node for this tool already?
	 */
	if( (node = slist_map2( *tool_notused, 
		(SListMap2Fn) program_tree_tool_test, tool, program )) ) {
		*tool_notused = g_slist_remove( *tool_notused, node );
		program->sibling = node;
	}
	else 
		program_node_new( program, tool );

	return( NULL );
}

static void *
program_tree_junk_node( GtkCTreeNode *node, Program *program )
{
	gtk_ctree_remove_node( GTK_CTREE( program->ctree ), node );

	return( NULL );
}

#ifdef DEBUG
static void *
program_tree_node_print( GtkCTreeNode *node, Program *program )
{
	char *name;

	gtk_ctree_get_node_info( GTK_CTREE( program->ctree ), node, 
		&name, NULL, NULL, NULL, NULL, NULL, NULL, NULL );
	printf( "  node %s\n", name );

	return( NULL );
}
#endif /*DEBUG*/

static void *
program_tree_kit_update( Toolkit *kit, Program *program, GSList **kit_notused )
{
	GSList *tool_notused = NULL;
	GtkCTreeNode *node;

	/* Do we have a node for this kit already?
	 */
	if( (node = slist_map2( *kit_notused, 
		(SListMap2Fn) program_tree_kit_test, kit, program )) ) 
		*kit_notused = g_slist_remove( *kit_notused, node );
	else {
		char *text[1];

		/* Nope, new kit node.
		 */
		text[0] = IOBJECT( kit )->name;
		node = gtk_ctree_insert_node( GTK_CTREE( program->ctree ),
			NULL, NULL, 
			text, 5, 
			program_closed, program_closed_mask, 
			program_open, program_open_mask,
			FALSE, FALSE );
		gtk_ctree_node_set_row_data( GTK_CTREE( program->ctree ),
			node, kit );

#ifdef DEBUG
		printf( "program_tree_kit_update: new node for %s\n",
			IOBJECT( kit )->name );
#endif /*DEBUG*/
	}

	/* Build a list of all nodes we have now for this kit.
	 */
	gtk_ctree_pre_recursive( GTK_CTREE( program->ctree ), node,
		(GtkCTreeFunc) program_tree_sym_list, &tool_notused );

#ifdef DEBUG_TREE
	printf( "program_tree_kit_update: current tool nodes:\n" );
	slist_map( tool_notused, 
		(SListMapFn) program_tree_node_print, program );
#endif /*DEBUG_TREE*/

	/* Update these tools.
	 */
	program->sibling = NULL;
	program->parent = node;
	icontainer_map_rev( ICONTAINER( kit ), 
		(icontainer_map_fn) program_tree_tool_update, 
		program, &tool_notused );

#ifdef DEBUG
	if( tool_notused ) {
		printf( "program_tree_kit_update: removing tool nodes:\n" );
		slist_map( tool_notused, 
			(SListMapFn) program_tree_node_print, program );
	}
#endif /*DEBUG*/

	/* ... and remove unused.
	 */
	slist_map( tool_notused, 
		(SListMapFn) program_tree_junk_node, program );
	FREEF( g_slist_free, tool_notused );

	return( NULL );
}

static void
program_tree_kit_list( GtkCTree *ctree, GtkCTreeNode *node, 
	GSList **kit_notused )
{
	gboolean is_leaf;

	gtk_ctree_get_node_info( ctree, node,
		NULL, NULL, NULL, NULL, NULL, NULL, &is_leaf, NULL );

	/* Called once for enclosing kit too. Can't check data, it may be
 	 * junk.
	 */
	if( !is_leaf )
		*kit_notused = g_slist_prepend( *kit_notused, node );
}

/* Make sure the tree is up to date.
 */
static void
program_tree_update( Program *program )
{
	GSList *kit_notused = NULL;

#ifdef DEBUG
	printf( "program_tree_update: starting for tree 0x%x\n",
		(unsigned int) program->ctree );
#endif /*DEBUG*/

	gtk_clist_freeze( GTK_CLIST( program->ctree ) );

	/* Build a list of all kit nodes we have now.
	 */
	gtk_ctree_pre_recursive_to_depth( GTK_CTREE( program->ctree ), NULL,
		1, (GtkCTreeFunc) program_tree_kit_list, &kit_notused );

#ifdef DEBUG_TREE
	printf( "program_tree_update: current kit nodes:\n" );
	slist_map( kit_notused, 
		(SListMapFn) program_tree_node_print, program );
#endif /*DEBUG_TREE*/

	/* Do add/update.
	 */
	toolkitgroup_map( program->kitg, 
		(toolkit_map_fn) program_tree_kit_update, 
		program, &kit_notused );

#ifdef DEBUG
	if( kit_notused ) {
		printf( "program_tree_update: removing unused kit nodes:\n" );
		slist_map( kit_notused, 
			(SListMapFn) program_tree_node_print, program );
	}
#endif /*DEBUG*/

	/* Remove unused.
	 */
	slist_map( kit_notused, 
		(SListMapFn) program_tree_junk_node, program );

	FREEF( g_slist_free, kit_notused );

	/* And resort.
	 */
	gtk_ctree_sort_recursive( GTK_CTREE( program->ctree ), NULL );

	gtk_clist_thaw( GTK_CLIST( program->ctree ) );
}

/* Some kit/tool has changed ... update.
 */
static void
program_kitgroup_changed( Model *model, Program *program )
{
	program_tree_update( program );
	program_title( program );
}

static void
program_kitgroup_destroy( Model *model, Program *program )
{
	/* Our toolkitgroup has gone! Give up on the world.
	 */
	program->kitgroup_changed_sid = 0;
	program->kitgroup_destroy_sid = 0;

	iwindow_kill( IWINDOW( program ) );
}

static void
program_init( Program *program )
{
	program->kitg = NULL;

	program->text = NULL;
	program->ctree = NULL;
	program->ifac = NULL;

	program->kitgroup_changed_sid = 0;
	program->kitgroup_destroy_sid = 0;

	program->kit = NULL;
	program->kit_destroy_sid = 0;

	program->sym = NULL;
	program->dirty = FALSE;
	program->sym_changed_sid = 0;

	program->tool = NULL;
	program->pos = -1;
	program->tool_destroy_sid = 0;

	program->search = NULL;
	program->csens = FALSE;
	program->regexp = FALSE;
	program->fromtop = TRUE;
#ifdef HAVE_REGEXEC
	program->comp = NULL;
#endif /*HAVE_REGEXEC*/
}

GtkType
program_get_type( void )
{
	static GtkType program_type = 0;

	if( !program_type ) {
		static const GtkTypeInfo info = {
			"Program",
			sizeof( Program ),
			sizeof( ProgramClass ),
			(GtkClassInitFunc) program_class_init,
			(GtkObjectInitFunc) program_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		program_type = gtk_type_unique( TYPE_IWINDOW, &info );
	}

	return( program_type );
}

/* The kit we have selected has been destroyed.
 */
static void
program_kit_destroy( Toolkit *kit, Program *program )
{
	assert( program->kit == kit );

	program_deselect( program );
}

static void
program_set_text( Program *program, const char *fmt, ... )
{
	va_list ap;
	char str[MAX_STRSIZE];
	BufInfo buf;

	buf_init_static( &buf, str, MAX_STRSIZE );
	va_start( ap, fmt );
	(void) buf_vappendf( &buf, fmt, ap );
	va_end( ap );

	gtk_text_freeze( GTK_TEXT( program->text ) );

	gtk_editable_delete_text( GTK_EDITABLE( program->text ), 0, -1 );
	gtk_text_insert( GTK_TEXT( program->text ),
		NULL, NULL, NULL, buf_all( &buf ), buf_len( &buf ) );

	gtk_text_thaw( GTK_TEXT( program->text ) );

	gtk_widget_set_sensitive( GTK_WIDGET( program->text ), 
		!program->kit->pseudo );

	program->dirty = FALSE;
}

/* Swap text for sym's text.
 */
static void
program_sym_text( Program *program, Symbol *sym )
{
	char str[MAX_STRSIZE];
	BufInfo buf;

	switch( sym->type ) {
	case SYM_EXTERNAL:
		buf_init_static( &buf, str, MAX_STRSIZE );
		vips_usage( &buf, sym->function );
		program_set_text( program, "%s", buf_all( &buf ) );
		break;

	case SYM_BUILTIN:
		buf_init_static( &buf, str, MAX_STRSIZE );
		builtin_usage( &buf, sym->builtin );
		program_set_text( program, "%s", buf_all( &buf ) );
		break;

	case SYM_VALUE:
		program_set_text( program, "%s", sym->expr->compile->text );
		break;

	default:
		break;
	}

	program_title( program );
}

/* The sym we are editing has changed.
 */
static void
program_sym_changed( Symbol *sym, Program *program )
{
	assert( program->sym == sym );

	if( program->dirty ) 
		box_info( GTK_WIDGET( program ),
			_( "Tool changed." ),
			_( "Tool changed by someone else!\n"
			"You're going to loose edits unless you're "
			"careful." ) );
	else 
		program_sym_text( program, sym );
}

/* The sym we are editing has been destroyed.
 */
static void
program_tool_destroy( Tool *tool, Program *program )
{
	assert( program->tool == tool );

	program_deselect( program );
}

/* Search for the node for this data.
 */
static GtkCTreeNode *
program_node_find( Program *program, gpointer data )
{
	return( gtk_ctree_find_by_row_data( 
		GTK_CTREE( program->ctree ), NULL, data ) );
}

static void
program_tree_select( Program *program, gpointer data )
{
	GtkCTreeNode *node = program_node_find( program, data );

	if( node )
		gtk_ctree_select( GTK_CTREE( program->ctree ), node );
}

static void
program_tree_expand( Program *program, gpointer data )
{
	GtkCTreeNode *node = program_node_find( program, data );

	if( node )
		gtk_ctree_expand( GTK_CTREE( program->ctree ), node );
}

static void
program_tree_scroll( Program *program, gpointer data )
{
	GtkCTreeNode *node = program_node_find( program, data );

	if( node && gtk_ctree_node_is_visible( 
		GTK_CTREE( program->ctree ), node ) == GTK_VISIBILITY_NONE )
		gtk_ctree_node_moveto( 
			GTK_CTREE( program->ctree ), node, 0, 0.5, 0.5 );
}

/* Compare function for tree sort.
 */
static int
program_tree_compare( GtkCTree *ctree, GtkCTreeRow *a, GtkCTreeRow *b )
{
	const void *data1 = a->row.data;
	const void *data2 = b->row.data;

	if( IS_SYMBOL( data1 ) && IS_SYMBOL( data2 ) )
		return( ICONTAINER( SYMBOL( data1 )->tool )->pos - 
			ICONTAINER( SYMBOL( data2 )->tool )->pos );
	else if( IS_TOOLKIT( data1 ) && IS_TOOLKIT( data2 ) )
		return( ICONTAINER( data1 )->pos - ICONTAINER( data2 )->pos );

	return( 0 );
}

static void
program_tree_move_before( GtkCTree *ctree, 
	GtkCTreeNode *child, GtkCTreeNode *parent, GtkCTreeNode *sibling, 
	Program *program )
{
	gpointer cdata = gtk_ctree_node_get_row_data( ctree, child );
	gpointer pdata = gtk_ctree_node_get_row_data( ctree, parent );

	/* Only allow dragging a tool between kits.
	 */
	if( IS_TOOLKIT( cdata ) || !IS_TOOLKIT( pdata ) ) {
		gtk_signal_emit_stop_by_name( 
			GTK_OBJECT( ctree ), "tree_move" );
		box_info( GTK_WIDGET( program ), 
			_( "Bad drag." ),
			_( "Sorry, you can only drag tools between toolkits\n"
			"You can't reorder toolkits, you can't nest toolkits\n"
			"and you can't drag tools to the top level." ) );
	}

	/* Disallow for pseudo kits.
	 */
	else if( (pdata && TOOLKIT( pdata )->pseudo) || 
		(IS_TOOL( cdata ) && TOOL( cdata )->kit->pseudo) ||
		(IS_SYMBOL( cdata ) && SYMBOL( cdata )->tool->kit->pseudo) ) {
		gtk_signal_emit_stop_by_name( 
			GTK_OBJECT( ctree ), "tree_move" );
		box_info( GTK_WIDGET( program ), 
			_( "Bad drag." ),
			_( "Sorry, you can't drag to or from from pseudo "
			"toolkits." ) );
	}
}

static void
program_tree_move_after( GtkCTree *ctree, 
	GtkCTreeNode *child, GtkCTreeNode *parent, GtkCTreeNode *sibling, 
	Program *program )
{
	Tool *tool = program_get_tool_data( program, child );
	Toolkit *kit = TOOLKIT( gtk_ctree_node_get_row_data( ctree, parent ) );
	Tool *btool = program_get_tool_data( program, sibling );
	Model *bmodel = !btool ? NULL : MODEL( btool );

	/* Move child to toolkit parent, place before sibling;
	 * slibling == NULL means place at end.
	 */
	g_object_ref( G_OBJECT( tool ) );
	icontainer_child_remove( ICONTAINER( tool ) );
	icontainer_child_add_before( ICONTAINER( kit ), 
		ICONTAINER( tool ), ICONTAINER( bmodel ) );
	g_object_unref( G_OBJECT( tool ) );
	filemodel_set_modified( FILEMODEL( kit ), TRUE );
	program_deselect( program );

	/* Select the row we dragged ... does not seem to be an easy way to 
	 * move the focus here too. Can't just do gtk_ctree_select( child );
	 * since child may no longer be valid.
	 */
	if( tool->sym )
		program_tree_select( program, tool->sym );
	else
		program_tree_select( program, tool );

	iobject_changed( IOBJECT( tool ) );
}

static gboolean
program_tree_event( GtkCTree *ctree, GdkEvent *ev, Program *program )
{
	int row, column;

        if( ev->type == GDK_BUTTON_PRESS && ev->button.button == 3 &&
		gtk_clist_get_selection_info( GTK_CLIST( ctree ), 
			ev->button.x, ev->button.y, &row, &column ) ) {
		GtkCTreeNode *node = gtk_ctree_node_nth( ctree, row );

		popup_link( GTK_WIDGET( program ), program_menu, node );
		popup_show( GTK_WIDGET( program ), ev );
	}

	return( FALSE );
}

/* Select a new kit in the tree. 
 */
static void
program_select_kit( Program *program, Toolkit *kit, gboolean *changed )
{
	/* None? Pick "untitled".
	 */
	if( !kit )
		kit = toolkit_by_name( program->kitg, "untitled" );

	if( program->kit == kit )
		return;

	program_deselect( program );

	program->kit = kit;
	program->kit_destroy_sid = g_signal_connect( G_OBJECT( kit ), 
		"destroy", G_CALLBACK( program_kit_destroy ), program );
	if( changed )
		*changed = TRUE;

	program_tree_select( program, kit );
	program_title( program );
}

/* Select a tool in the tree. 
 */
static void
program_select_tool( Program *program, Tool *tool, gboolean *changed )
{
	if( program->tool == tool )
		return;

	program_deselect( program );

	program_select_kit( program, tool->kit, NULL );

	program->tool = tool;
	program->pos = ICONTAINER( tool )->pos;
	program->tool_destroy_sid = g_signal_connect( G_OBJECT( tool ), 
		"destroy", G_CALLBACK( program_tool_destroy ), program );
	if( changed )
		*changed = TRUE;

	program_tree_expand( program, tool->kit );
	program_tree_select( program, tool );
	program_title( program );
}

/* Select a new symbol in the tree. 
 */
static void
program_select_sym( Program *program, Symbol *sym, gboolean *changed )
{
	assert( sym->tool );

	if( program->sym == sym )
		return;

	program_deselect( program );

	program_select_tool( program, sym->tool, NULL );

	program->sym = sym;
	program->dirty = FALSE;
	program->sym_changed_sid = g_signal_connect( G_OBJECT( sym ), 
		"changed", G_CALLBACK( program_sym_changed ), program );
	program_sym_text( program, sym );
	if( changed )
		*changed = TRUE;

	program_tree_expand( program, sym->tool->kit );
	program_tree_select( program, sym );
	program_title( program );
}

/* Read and parse the text.
 */
static gboolean
program_parse( Program *program )
{
	char *txt;
	char buffer[MAX_STRSIZE];

	if( !program->dirty )
		return( TRUE );

	/* Irritatingly, we need to append a ';'.
	 */
	txt = gtk_editable_get_chars( GTK_EDITABLE( program->text ), 0, -1 );
	im_snprintf( buffer, MAX_STRSIZE, "%s;", txt );
	FREEF( g_free, txt );

	if( strspn( buffer, WHITESPACE ";" ) == strlen( buffer ) ) 
		return( TRUE );

	/* Make sure we've got a kit.
	 */
	program_select_kit( program, program->kit, NULL );

	/* ... and parse the new text into it.
	 */
	attach_input_string( buffer );
	if( !parse_onedef( program->kit, program->pos ) ) {
		gtk_editable_select_region( GTK_EDITABLE( program->text ),
			input_state.charpos,
			input_state.charpos - yyleng );
		return( FALSE );
	}

	program->dirty = FALSE;
	filemodel_set_modified( FILEMODEL( program->kit ), TRUE );

	/* Reselect last_top_sym, the last thing the parser saw. 
	 */
	if( last_top_sym && last_top_sym->tool ) {
		if( program->sym && last_top_sym != program->sym )
			IDESTROY( program->sym );

		program_select_sym( program, last_top_sym, NULL );
		program_tree_scroll( program, program->sym );

		/* Signal to other program windows that we've changed this sym.
		 */
		iobject_changed( IOBJECT( program->sym ) );
	}

	symbol_recalculate_all();

	return( TRUE );
}

static void
program_tool_new_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );
	Toolkit *kit;

	/* Existing text changed? Parse it.
	 */
	if( program->dirty && !program_parse( program ) ) {
		box_alert( GTK_WIDGET( program ) );
		return;
	}

	kit = program->kit;
	program_deselect( program );
	program_select_kit( program, kit, NULL );

	program_tree_scroll( program, program->kit );
}

static void
program_toolkit_new_done_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	Stringset *ss = STRINGSET( iwnd );
	StringsetChild *name = stringset_child_get( ss, _( "Name" ) );
	StringsetChild *caption = stringset_child_get( ss, _( "Caption" ) );
	Program *program = PROGRAM( client );

	Toolkit *kit;
	BufInfo buf;
	char str[1024];
	char name_text[1024];
	char caption_text[1024];

	if( !get_geditable_name( name->entry, name_text, 1024 ) ||
		!get_geditable_string( caption->entry, caption_text, 1024 ) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	/* Make a filename from the name ... user start directory.
	 */
	buf_init_static( &buf, str, 1024 );
	buf_appendf( &buf, "$HOME/.$PACKAGE-$VERSION/start/%s.def", name_text );
	kit = toolkit_new_filename( main_toolkitgroup, buf_all( &buf ) );

	/* Set caption.
	 */
	if( strspn( caption_text, WHITESPACE ) != strlen( caption_text ) )
		iobject_set( IOBJECT( kit ), NULL, caption_text );
	else
		iobject_set( IOBJECT( kit ), NULL, "untitled" );

	program_select_kit( program, kit, NULL );
	program_tree_scroll( program, kit );

	nfn( sys, IWINDOW_TRUE );
}

static void
program_toolkit_new_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );
	GtkWidget *ss = stringset_new();

	stringset_child_new( STRINGSET( ss ), 
		_( "Name" ), "", _( "Set toolkit name here" ) );
	stringset_child_new( STRINGSET( ss ), 
		_( "Caption" ), "", _( "Set toolkit caption here" ) );
	iwindow_set_title( IWINDOW( ss ), _( "New Toolkit" ) );
	idialog_set_callbacks( IDIALOG( ss ), 
		iwindow_true_cb, NULL, NULL, program );
	idialog_set_help_tag( IDIALOG( ss ), "workspace" );
	idialog_add_ok( IDIALOG( ss ), 
		program_toolkit_new_done_cb, _( "Create" ) );
	idialog_set_parent( IDIALOG( ss ), GTK_WIDGET( program ) );
	iwindow_build( IWINDOW( ss ) );

	gtk_widget_show( ss );
}

static gboolean
program_check_kit( Program *program )
{
	if( !program->kit ) {
		box_info( GTK_WIDGET( program ), 
			_( "Nothing selected." ),
			_( "No toolkit selected." ) );
		return( FALSE );
	}

	return( TRUE );
}

static void
program_separator_new_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );
	Tool *tool;
	int pos;

	if( !program_check_kit( program ) )
		return;

	pos = icontainer_pos_last( ICONTAINER( program->kit ) );
	tool = tool_new_sep( program->kit, pos + 1 );
	program_select_tool( program, tool, NULL );
}

static void
program_column_item_new_done_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	Stringset *ss = STRINGSET( iwnd );
	StringsetChild *name = stringset_child_get( ss, _( "Name" ) );
	StringsetChild *file = stringset_child_get( ss, _( "Filename" ) );
	Program *program = PROGRAM( client );
	Tool *tool;

	int pos;
	char name_text[1024];
	char file_text[1024];

	if( !get_geditable_name( name->entry, name_text, 1024 ) ||
		!get_geditable_filename( file->entry, file_text, 1024 ) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	pos = icontainer_pos_last( ICONTAINER( program->kit ) );
	tool = tool_new_dia( program->kit, pos + 1, name_text, file_text );
	program_select_tool( program, tool, NULL );

	nfn( sys, IWINDOW_TRUE );
}

static void
program_column_item_new_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );
	GtkWidget *ss;

	if( !program_check_kit( program ) )
		return;

	ss = stringset_new();
	stringset_child_new( STRINGSET( ss ), 
		_( "Name" ), "", _( "Display this name" ) );
	stringset_child_new( STRINGSET( ss ), 
		_( "Filename" ), "", _( "Load this file" ) );
	iwindow_set_title( IWINDOW( ss ), "New Column Item" );
	idialog_set_callbacks( IDIALOG( ss ), 
		iwindow_true_cb, NULL, NULL, program );
	idialog_add_ok( IDIALOG( ss ), 
		program_column_item_new_done_cb, _( "Create" ) );
	idialog_set_parent( IDIALOG( ss ), GTK_WIDGET( program ) );
	iwindow_build( IWINDOW( ss ) );

	gtk_widget_show( ss );
}

static void
program_program_new_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );

	(void) program_new( program->kitg );
}

static void
program_workspace_new_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	workspacegroup_workspace_new( main_workspacegroup, widget );
}

static void *
program_load_file_fn( Filesel *filesel, 
	const char *filename, Program *program, void *b )
{
	Toolkit *kit;

	if( !(kit = toolkit_new_from_file( main_toolkitgroup, filename )) ) 
		return( filesel );

	program_select_kit( program, kit, NULL );
	program_tree_scroll( program, kit );

	return( NULL );
}

/* Callback from load browser.
 */
static void
program_load_file_cb( iWindow *iwnd, 
	void *client, iWindowNotifyFn nfn, void *sys )
{
	Filesel *filesel = FILESEL( iwnd );
	Program *program = PROGRAM( client );

	program_freeze();

	if( filesel_map_filename_multi( filesel,
		(FileselMapFn) program_load_file_fn, program, NULL ) ) {
		program_thaw();
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	program_thaw();
	symbol_recalculate_all();

	nfn( sys, IWINDOW_TRUE );
}

static void
program_open_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );
	GtkWidget *filesel = filesel_new();

	iwindow_set_title( IWINDOW( filesel ), _( "Load Definition" ) );
	filesel_set_flags( FILESEL( filesel ), FALSE, FALSE );
	filesel_set_filetype( FILESEL( filesel ), filesel_type_definition, 0 ); 
	idialog_set_parent( IDIALOG( filesel ), GTK_WIDGET( program ) );
	filesel_set_done( FILESEL( filesel ), program_load_file_cb, program );
	filesel_set_multi( FILESEL( filesel ), TRUE );
	iwindow_build( IWINDOW( filesel ) );

	gtk_widget_show( GTK_WIDGET( filesel ) );
}

static void
program_save_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );

	if( !program_check_kit( program ) )
		return;

	filemodel_inter_save( IWINDOW( program ), FILEMODEL( program->kit ) );
}

static void
program_save_as_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );

	if( !program_check_kit( program ) )
		return;

	filemodel_inter_saveas( IWINDOW( program ), FILEMODEL( program->kit ) );
}

static void
program_process_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );

	if( !program_parse( program ) )
		box_alert( GTK_WIDGET( program ) );
}

static void
program_reload_menus_cb( iWindow *iwnd, 
	void *client, iWindowNotifyFn nfn, void *sys )
{
	main_reload();

	nfn( sys, IWINDOW_TRUE );
}

/* Reload all menus.
 */
static void
program_reload_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );

	box_yesno( GTK_WIDGET( program ),
		program_reload_menus_cb, iwindow_true_cb, NULL,
		iwindow_notify_null, NULL,
		_( "Reload" ), 
		_( "Reload startup objects?" ),
		_( "Would you like to reload all startup menus, workspaces\n"
		"and plugins now? This may take a few seconds." ) );
}

static void
program_quit_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );

	iwindow_kill( IWINDOW( program ) );
}

static void
program_cut_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );

	gtk_editable_cut_clipboard( GTK_EDITABLE( program->text ) );
}

static void
program_copy_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );

	gtk_editable_copy_clipboard( GTK_EDITABLE( program->text ) );
}

static void
program_paste_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );

	gtk_editable_paste_clipboard( GTK_EDITABLE( program->text ) );
}

static void
program_select_all_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );

	gtk_editable_select_region( GTK_EDITABLE( program->text ), 0, -1 );
}

static void
program_remove_tool_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );

	if( program->tool ) 
		model_check_destroy( GTK_WIDGET( program ), 
			MODEL( program->tool ) );
	else 
		box_info( GTK_WIDGET( program ), _( "No tool selected" ), "" );
}

static void
program_remove_toolkit_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );

	if( !program_check_kit( program ) )
		return;

	model_check_destroy( GTK_WIDGET( program ), MODEL( program->kit ) );
}

static void
program_find_done_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	Find *find = FIND( iwnd );
	Program *program = PROGRAM( client );

	FREEF( g_free, program->search );

	program->search = 
		gtk_editable_get_chars( GTK_EDITABLE( find->search ), 0, -1 );
	program->csens = GTK_TOGGLE_BUTTON( find->csens )->active;
#ifdef HAVE_REGEXEC
	program->regexp = GTK_TOGGLE_BUTTON( find->regexp )->active;
#endif /*HAVE_REGEXEC*/
	program->fromtop = GTK_TOGGLE_BUTTON( find->fromtop )->active;

#ifdef HAVE_REGEXEC
	if( program->regexp ) {
		int flags = 0;

		if( !program->comp )
			program->comp = INEW( NULL, regex_t );

		if( !program->csens )
			flags |= REG_ICASE;

		if( regcomp( program->comp, program->search, flags ) != 0 ) {
			error_top( _( "Parse error." ) );
			error_sub( _( "Bad regular expression." ) );
			nfn( sys, IWINDOW_ERROR );
			return;
		}
	}
#endif /*HAVE_REGEXEC*/

	if( program->fromtop )
		program_find_reset( program );
	else
		program->find_start += 1;

	if( program_find( program ) ) {
		program_select_sym( program, program->find_sym, NULL );
		gtk_editable_select_region( GTK_EDITABLE( program->text ), 
			program->find_start, program->find_end );
	}
	else {
		error_top( _( "Not found." ) );
		error_sub( _( "No match found for \"%s\"." ), program->search );
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	nfn( sys, IWINDOW_TRUE );
}

static void
program_find_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );
	GtkWidget *find = find_new();

	iwindow_set_title( IWINDOW( find ), _( "Find in all Toolkits" ) );
	idialog_set_callbacks( IDIALOG( find ), 
		iwindow_true_cb, NULL, NULL, program );
	idialog_set_help_tag( IDIALOG( find ), "toolkit_find" );
	idialog_add_ok( IDIALOG( find ), program_find_done_cb, GTK_STOCK_FIND );
	idialog_set_parent( IDIALOG( find ), GTK_WIDGET( program ) );
	idialog_set_cancel_text( IDIALOG( find ), GTK_STOCK_CLOSE );
	iwindow_build( IWINDOW( find ) );

	if( program->search )
		set_gentry( FIND( find )->search, "%s", program->search );
	set_tooltip( FIND( find )->search, _( "Enter search string here" ) );
        gtk_toggle_button_set_active( 
		GTK_TOGGLE_BUTTON( FIND( find )->csens ), program->csens );
#ifdef HAVE_REGEXEC
        gtk_toggle_button_set_active( 
		GTK_TOGGLE_BUTTON( FIND( find )->regexp ), program->regexp );
#endif /*HAVE_REGEXEC*/
        gtk_toggle_button_set_active( 
		GTK_TOGGLE_BUTTON( FIND( find )->fromtop ), program->fromtop );

	gtk_widget_show( find );
}

static void
program_find_again_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );

	if( program->find_sym ) 
		program->find_start += 1;

	if( program_find( program ) ) {
		program_select_sym( program, program->find_sym, NULL );
		gtk_editable_select_region( GTK_EDITABLE( program->text ), 
			program->find_start, program->find_end );
	}
	else
		box_info( GTK_WIDGET( program ), 
			_( "Not found." ),
			_( "No match found." ) );
}

static void
program_goto_done_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	Program *program = PROGRAM( client );
	Stringset *ss = STRINGSET( iwnd );
	StringsetChild *name = stringset_child_get( ss, _( "Name" ) );
	Symbol *sym;
	char name_text[1024];

	if( !get_geditable_string( name->entry, name_text, 1024 ) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	if( !(sym = compile_lookup( program->kitg->root->expr->compile, 
		name_text )) ) {
		error_top( _( "Not found." ) );
		error_sub( _( "No top-level symbol called \"%s\"." ), 
			name_text );
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	if( !sym->tool ) {
		error_top( _( "Not found." ) );
		error_sub( _( "Symbol \"%s\" has no tool inforation." ), 
			name_text );
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	if( !program_select( program, MODEL( sym ) ) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	nfn( sys, IWINDOW_TRUE );
}

static void
program_goto_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );
	GtkWidget *ss = stringset_new();
	StringsetChild *name;

	name = stringset_child_new( STRINGSET( ss ), 
		_( "Name" ), "", _( "Go to definition of this symbol" ) );

	iwindow_set_title( IWINDOW( ss ), _( "Go to Definition" ) );
	idialog_set_callbacks( IDIALOG( ss ), 
		iwindow_true_cb, NULL, NULL, program );
	idialog_add_ok( IDIALOG( ss ), program_goto_done_cb, "Go" );
	idialog_set_parent( IDIALOG( ss ), GTK_WIDGET( program ) );
	iwindow_build( IWINDOW( ss ) );

	gtk_widget_show( ss );

	/* Now try to paste the selection into the name widget.
	 */
	gtk_editable_copy_clipboard( GTK_EDITABLE( program->text ) );
	gtk_editable_paste_clipboard( GTK_EDITABLE( name->entry ) );
}

static void
program_info_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );
	BufInfo buf;
	char txt[MAX_STRSIZE];

	buf_init_static( &buf, txt, MAX_STRSIZE );
	program_info( program, &buf );
	box_info( GTK_WIDGET( program ), _( "Object information." ), 
		"%s", buf_all( &buf ) );
}

static void
program_trace_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	(void) trace_new();
}

static void
program_linkreport_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );
	BufInfo buf;
	char txt[MAX_STRSIZE];
	gboolean bad_links;

	buf_init_static( &buf, txt, MAX_STRSIZE );
	bad_links = FALSE;
	(void) toolkitgroup_map( program->kitg,
		(toolkit_map_fn) toolkit_linkreport, &buf, &bad_links );
	if( !bad_links )
		buf_appendf( &buf, _( "No unresolved symbols found." ) );

	box_info( GTK_WIDGET( program ), _( "Link report." ), 
		"%s", buf_all( &buf ) );
}

static void
program_errorreport_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );
	BufInfo buf;
	char txt[MAX_STRSIZE];

	buf_init_static( &buf, txt, MAX_STRSIZE );
	expr_error_print_all( &buf );
	if( buf_isempty( &buf ) )
		buf_appendf( &buf, _( "No errors found." ) );

	box_info( GTK_WIDGET( program ), _( "Error report." ), 
		"%s", buf_all( &buf ) );
}

static void
program_tool_help_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Program *program = PROGRAM( callback_data );
	BufInfo buf;
	char txt[512];

	buf_init_static( &buf, txt, 512 );

	if( program->tool && program->tool->type == TOOL_SYM && 
		program->kit && program->kit->pseudo ) {
		switch( program->tool->sym->type ) {
		case SYM_EXTERNAL:
			buf_appendf( &buf, "file://"
				VIPS_DOCPATH "/man/%s.3.html", 
				IOBJECT( program->tool->sym )->name );
			box_url( GTK_WIDGET( program ), buf_all( &buf ) );
			break;

		case SYM_BUILTIN:
			box_help( GTK_WIDGET( program ), "tb:builtin" );
			break;

		default:
			break;
		}
	}
	else
		box_info( GTK_WIDGET( program ), 
			_( "No documentation available." ),
			_( "On-line documentation is only currently\n"
			"available for VIPS functions and nip builtins." ) );

}

/* Menu bar.
 */
static GtkItemFactoryEntry program_menu_items[] = {
	{ N_( "/_File" ),            			NULL,         
		NULL,			0, "<Branch>" },
	{ N_( "/File/_New" ), 			NULL, 
		NULL, 			0, "<Branch>" },
	{ N_( "/File/New/_Tool" ), 			NULL,
		program_tool_new_ifcb, 	0, 
			"<StockItem>", GTK_STOCK_NEW },
	{ N_( "/File/New/Tool_kit ..." ), 		NULL,
		program_toolkit_new_ifcb,	0,
			"<StockItem>", GTK_STOCK_NEW },
	{ N_( "/File/New/Separator ..." ), 		NULL,
		program_separator_new_ifcb,	0, 
			"<StockItem>", GTK_STOCK_NEW },
	{ N_( "/File/New/Column Item ..." ), 		NULL,
		program_column_item_new_ifcb,	0,
			"<StockItem>", GTK_STOCK_NEW },
	{ N_( "/File/New/Program Window" ),		NULL,
		program_program_new_ifcb, 	0, 
			"<StockItem>", GTK_STOCK_NEW },
	{ N_( "/File/New/Workspace ..." ),		NULL,
		program_workspace_new_ifcb, 	0, 
			"<StockItem>", GTK_STOCK_NEW },
	{ N_( "/File/_Open Toolkit ..." ), 		"<control>O", 
		program_open_ifcb, 	0, 
			"<StockItem>", GTK_STOCK_OPEN },
	{ N_( "/File/sep2" ),        			NULL,         	
		NULL,    		0, "<Separator>" },
	{ N_( "/File/_Save Toolkit" ), 			"<control>S", 
		program_save_ifcb, 	0, 
			"<StockItem>", GTK_STOCK_SAVE },
	{ N_( "/File/Save Toolkit _As ..." ), 		NULL,
		program_save_as_ifcb,	0, 
			"<StockItem>", GTK_STOCK_SAVE_AS },
	{ N_( "/File/sep1" ),        			NULL,         	
		NULL,    		0, "<Separator>" },
	{ N_( "/File/Pr_ocess Text ..." ), 		NULL,
		program_process_ifcb, 	0, NULL },
	{ N_( "/File/_Reload Start Stuff ..." ), 	NULL, 
		program_reload_ifcb,   	0, NULL },
	{ N_( "/File/sep3" ),        			NULL,         	
		NULL,    		0, "<Separator>" },
	{ N_( "/File/_Close" ),      			NULL,
		program_quit_ifcb,     	0,
			"<StockItem>", GTK_STOCK_CLOSE },

	{ N_( "/_Edit" ),            			NULL,         
		NULL,			0, "<Branch>" },
	{ N_( "/Edit/C_ut" ),            		NULL,
		program_cut_ifcb,	0, 
			"<StockItem>", GTK_STOCK_CUT },
	{ N_( "/Edit/_Copy" ),            		NULL,
		program_copy_ifcb,	0, 
			"<StockItem>", GTK_STOCK_COPY },
	{ N_( "/Edit/_Paste" ),            		NULL,
		program_paste_ifcb,	0, 
			"<StockItem>", GTK_STOCK_PASTE },
	{ N_( "/Edit/_Select All" ),            	"<Ctrl>A",         
		program_select_all_ifcb,0, NULL },
	{ N_( "/Edit/sep2" ),        			NULL,         	
		NULL,    		0, "<Separator>" },
	{ N_( "/Edit/Delete This _Tool ..." ),    	NULL,
		program_remove_tool_ifcb,0, NULL },
	{ N_( "/Edit/Delete This Tool_kit ..." ),    	NULL,
		program_remove_toolkit_ifcb,0, NULL },
	{ N_( "/Edit/sep4" ),        			NULL,         	
		NULL,    		0, "<Separator>" },
	{ N_( "/Edit/_Find ..." ),   			"<control>F", 
		program_find_ifcb,     	0, 
			"<StockItem>", GTK_STOCK_FIND },
	{ N_( "/Edit/Find _Next" ),       		"<control>G", 
		program_find_again_ifcb, 0, NULL },
	{ N_( "/Edit/_Jump to Definition of ..." ),     NULL, 
		program_goto_ifcb, 	0, 
			"<StockItem>", GTK_STOCK_JUMP_TO },
	{ N_( "/Edit/sep4" ),        			NULL,         	
		NULL,    		0, "<Separator>" },
	{ N_( "/Edit/_Info ..." ),    		NULL,
		program_info_ifcb,	0, NULL },

	{ N_( "/_Debug" ),           			NULL,         
		NULL,			0, "<Branch>" },
	{ N_( "/Debug/_Trace ..." ),         		NULL,
		program_trace_ifcb,	0, NULL },
	{ N_( "/Debug/_Link ..." ),          		NULL,
		program_linkreport_ifcb,0, NULL },
	{ N_( "/Debug/_Errors ..." ),          		NULL,
		program_errorreport_ifcb,0, NULL },

	{ N_( "/_Help" ),            			NULL,         
		NULL,			0, "<LastBranch>" },
	{ N_( "/Help/_Program ..." ),  			NULL,         
		box_help_ifcb,		GPOINTER_TO_UINT( "sec:progwin" ) },
	{ N_( "/Help/_Documentation For This Tool ..." ),NULL,         
		program_tool_help_ifcb,	GPOINTER_TO_UINT( "program" ) }
};

gboolean
program_select( Program *program, Model *model )
{
	/* Existing text changed? Parse it.
	 */
	if( program->dirty && !program_parse( program ) ) 
		return( FALSE );

	if( model ) {
		gboolean changed = FALSE;

		if( IS_SYMBOL( model ) ) 
			program_select_sym( program, 
				SYMBOL( model ), &changed );
		else if( IS_TOOL( model ) ) 
			program_select_tool( program, 
				TOOL( model ), &changed );
		else if( IS_TOOLKIT( model ) )
			program_select_kit( program, 
				TOOLKIT( model ), &changed );

		if( changed )
			program_tree_scroll( program, model );
	}

	return( TRUE );
}

static void        
program_select_row( GtkCTree *ctree, 
	GtkCTreeNode *node, gint column, Program *program )
{
	gpointer data = gtk_ctree_node_get_row_data( ctree, node );

	if( !program_select( program, data ) ) {
		/*
			
			FIXME .. see note below

		GtkCTreeNode *node;
		 */

		/* Deselect failed ... bounce back to the previous sym again.
		 */
		box_alert( GTK_WIDGET( program ) );
		gtk_signal_emit_stop_by_name( GTK_OBJECT( ctree ), 
			"tree_select_row" );

		/*

			FIXME ... this mysteriously causes a loop (and often
			an X server crash) on gtk2

		if( (node = program_node_find( program, data )) )
			gtk_ctree_select( ctree, node );
		 */
	}
}

static void
program_text_changed( GtkText *text, Program *program )
{
	if( !program->dirty ) {
		program->dirty = TRUE;
		program_title( program );
	}
}

static void
program_build( Program *program, GtkWidget *vbox )
{
	iWindow *iwnd = IWINDOW( program );

	GtkWidget *mbar;
	GtkWidget *swin;
	GtkWidget *pane;
	GtkAdjustment *hadj;
	GtkAdjustment *vadj;

	/* Make menu bar
	 */
	program->ifac = gtk_item_factory_new( GTK_TYPE_MENU_BAR, 
		"<program>", iwnd->accel_group );
	g_object_ref( program->ifac );
	gtk_object_sink( GTK_OBJECT( program->ifac ) );
#ifdef ENABLE_NLS
	gtk_item_factory_set_translate_func( program->ifac,
		(GtkTranslateFunc) gettext, NULL, NULL );
#endif /* ENABLE_NLS */
	gtk_item_factory_create_items( program->ifac, 
		IM_NUMBER( program_menu_items ), program_menu_items, program );
	mbar = gtk_item_factory_get_widget( program->ifac, "<program>" );
	gtk_box_pack_start( GTK_BOX( vbox ), mbar, FALSE, FALSE, 0 );
	gtk_widget_show( mbar );

	pane = gtk_hpaned_new();
	gtk_box_pack_start( GTK_BOX( vbox ), pane, TRUE, TRUE, 0 );
	gtk_widget_show( pane );

	swin = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( swin ),
		GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC );
	gtk_paned_pack1( GTK_PANED( pane ), swin, FALSE, FALSE );
	gtk_widget_show( swin );

	program->ctree = gtk_ctree_new( 1, 0 );
        gtk_signal_connect( GTK_OBJECT( program->ctree ), "tree-select-row",
                GTK_SIGNAL_FUNC( program_select_row ), program );
	gtk_clist_set_column_auto_resize( GTK_CLIST( program->ctree ), 
		0, TRUE );
	gtk_clist_set_compare_func( GTK_CLIST( program->ctree ),
		(GtkCListCompareFunc) program_tree_compare );
	gtk_clist_set_reorderable( GTK_CLIST( program->ctree ), TRUE );
	gtk_signal_connect( GTK_OBJECT( program->ctree ), "tree_move",
		GTK_SIGNAL_FUNC( program_tree_move_before ), program );
	gtk_signal_connect_after( GTK_OBJECT( program->ctree ), "tree_move",
		GTK_SIGNAL_FUNC( program_tree_move_after ), program );
	gtk_signal_connect( GTK_OBJECT( program->ctree ), "event",
		GTK_SIGNAL_FUNC( program_tree_event ), program );
	gtk_container_add( GTK_CONTAINER( swin ), program->ctree );
	gtk_widget_show( program->ctree );

	swin = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( swin ),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC );
	gtk_paned_pack2( GTK_PANED( pane ), swin, TRUE, TRUE );
	hadj = gtk_scrolled_window_get_hadjustment( 
		GTK_SCROLLED_WINDOW( swin ) );
	vadj = gtk_scrolled_window_get_vadjustment( 
		GTK_SCROLLED_WINDOW( swin ) );
	gtk_widget_show( swin );

	program->text = gtk_text_new( hadj, vadj );
	gtk_text_set_editable( GTK_TEXT( program->text ), TRUE );
        gtk_signal_connect( GTK_OBJECT( program->text ), "changed",
                GTK_SIGNAL_FUNC( program_text_changed ), program );
	gtk_container_add( GTK_CONTAINER( swin ), program->text );
	gtk_widget_show( program->text );

	gtk_widget_grab_focus( program->text );
}

static void
program_popdown( iWindow *iwnd, void *client, iWindowNotifyFn nfn, void *sys )
{
	Program *program = PROGRAM( client );

        if( program->dirty && !program_parse( program ) )
                nfn( sys, IWINDOW_ERROR );
        else
                nfn( sys, IWINDOW_TRUE );
}

static void
program_link( Program *program, Toolkitgroup *kitg )
{
	program->kitg = kitg;
	program_title( program );
	gtk_window_set_default_size( GTK_WINDOW( program ), 750, 300 );
	iwindow_set_build( IWINDOW( program ), 
		(iWindowBuildFn) program_build, NULL, NULL, NULL );
	iwindow_set_popdown( IWINDOW( program ),
		(iWindowFn) program_popdown, program );
	iwindow_build( IWINDOW( program ) );
	program_all = g_slist_prepend( program_all, program );
	program_tree_update( program );

	program->kitgroup_changed_sid = 
		g_signal_connect( G_OBJECT( program->kitg ), "changed",
			G_CALLBACK( program_kitgroup_changed ), program );
	program->kitgroup_destroy_sid = 
		g_signal_connect( G_OBJECT( program->kitg ), "destroy",
			G_CALLBACK( program_kitgroup_destroy ), program );

	gtk_widget_show( GTK_WIDGET( program ) ); 
}

Program *
program_new( Toolkitgroup *kitg )
{
	Program *program = gtk_type_new( TYPE_PROGRAM );

	program_link( program, kitg );

	return( program );
}
