/* run the display for an input matrixview in a workspace 
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

#include "ip.h"

#ifdef HAVE_GTK_EXTRA
#include <gtkextra/gtksheet.h>
#endif /*HAVE_GTK_EXTRA*/

static GraphicviewClass *parent_class = NULL;

static void
matrixview_destroy( GtkObject *object )
{
    	Matrixview *matrixview;

    	g_return_if_fail( object != NULL );
    	g_return_if_fail( IS_MATRIXVIEW( object ) );

#ifdef DEBUG
    	printf( "matrixview_destroy\n" );
#endif /*DEBUG*/

    	matrixview = MATRIXVIEW( object );

    	/* My instance destroy stuff.
    	 */
    	FREEF( g_slist_free, matrixview->items );
	FREE( matrixview->cell_text );

    	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

#ifdef HAVE_GTK_EXTRA
static gboolean
matrixview_scan_string( char *txt, double *out, gboolean *changed )
{
    	double v;

    	if( sscanf( txt, "%lg", &v ) != 1 ) {
		error_top( _( "Bad floating point number." ) );
		error_sub( _( "\"%s\" is not a floating point number." ), txt );
    		return( FALSE );
    	}

    	if( *out != v ) {
    		*out = v;
    		*changed = TRUE;
    	}

    	return( TRUE );
}
#endif /*HAVE_GTK_EXTRA*/

static gboolean
matrixview_scan_text( Matrixview *matrixview, GtkWidget *txt, 
    	double *out, gboolean *changed )
{
    	double v;

    	if( !get_geditable_double( txt, &v ) ) 
    		return( FALSE );

    	if( *out != v ) {
    		*out = v;
    		*changed = TRUE;
    	}

    	return( TRUE );
}

/* Search and read all text widgets and refill matrix. set_dirty this symbol
 * if there was a change. Return non-NULL if we found an error.
 */
static void *
matrixview_scan( View *view )
{
    	Matrixview *matrixview = MATRIXVIEW( view );
    	Matrix *matrix = MATRIX( VOBJECT( matrixview )->iobject );
    	Expr *expr = HEAPMODEL( matrix )->row->expr;

    	gboolean changed;
    	int x, y, i;
    	GSList *p;

#ifdef DEBUG
    	printf( "matrixview_scan\n" );
#endif /*DEBUG*/

    	/* Should be text widgets there ... either text or tslider.
    	 */
    	if( matrixview->display != MATRIX_DISPLAY_TEXT && 
    		matrixview->display != MATRIX_DISPLAY_TEXT_SCALE_OFFSET && 
    		matrixview->display != MATRIX_DISPLAY_SLIDER )
    		return( NULL );

    	expr_error_clear( expr );
    	changed = FALSE;

    	/* Check for scale and offset, if present.
    	 */
    	if( matrixview->scale && 
		!matrixview_scan_text( matrixview,
			matrixview->scale, &matrix->scale, &changed ) ) {
		expr_error_set( expr );
    		return( view );
	}
    	if( matrixview->offset && 
		!matrixview_scan_text( matrixview, 
			matrixview->offset, &matrix->offset, &changed ) ) {
		expr_error_set( expr );
    		return( view );
    	}

    	/* Loop thro' all matrix widgets.
    	 */
    	if( matrixview->items ) 
    		for( i = 0, p = matrixview->items, y = 0; 
    			y < matrixview->height; y++ )
    			for( x = 0; x < matrixview->width; 
    				x++, p = p->next, i++ ) {
    				GtkWidget *item = GTK_WIDGET( p->data );

    				GtkWidget *entry;

    				/* It's either an entry, or a tslider.
    				 */
    				if( IS_TSLIDER( item ) ) 
    					entry = TSLIDER( item )->entry;
    				else
    					entry = item;

    				if( !matrixview_scan_text( matrixview, entry,
    					&matrix->value[i], &changed ) ) {
					error_top( _( "Bad value." ) );
					error_sub( _( "Cell (%d, %d):\n%s" ), 
    						x, y, error_get_sub() );
					expr_error_set( expr );

    					return( view );
    				}
    			}

#ifdef HAVE_GTK_EXTRA
    	if( matrixview->sheet ) {
    		GtkSheet *sheet = GTK_SHEET( matrixview->sheet );

    		for( i = 0, y = 0; y < matrixview->height; y++ )
    			for( x = 0; x < matrixview->width; x++, i++ ) {
    				char *txt = gtk_sheet_cell_get_text(
    					sheet, y, x );

    				if( !matrixview_scan_string( txt,
    					&matrix->value[i], &changed ) ) {
					error_top( _( "Bad value." ) );
					error_sub( _( "Cell (%d, %d):\n%s" ), 
    						x, y, error_get_sub() );
					expr_error_set( expr );

    					return( view );
    				}
    			}
    	}
#endif /*HAVE_GTK_EXTRA*/

    	if( changed ) 
    		classmodel_update( CLASSMODEL( matrix ) ) ;

    	return( VIEW_CLASS( parent_class )->scan( view ) );
}

/* Change to a toggle widget. 
 */
/*ARGSUSED*/
static void
matrixview_toggle_change_cb( GtkWidget *widget, Matrixview *matrixview )
{
    	Matrix *matrix = MATRIX( VOBJECT( matrixview )->iobject );

    	int pos = g_slist_index( matrixview->items, widget );

#ifdef DEBUG
    	printf( "matrixview_toggle_change_cb\n" );
#endif /*DEBUG*/

    	/* Cycle value.
    	 */
    	switch( (int) matrix->value[pos] ) {
    	case 0:
    		matrix->value[pos] = 128.0;
    		break;

    	case 255:
    		matrix->value[pos] = 0.0;
    		break;

    	default:
    		matrix->value[pos] = 255.0;
    		break;
    	}

    	classmodel_update( CLASSMODEL( matrix ) );
    	symbol_recalculate_all();
}

/* Build a set of toggle items for a matrix. 
 */
static void
matrixview_toggle_build( Matrixview *matrixview )
{
    	Matrix *matrix = MATRIX( VOBJECT( matrixview )->iobject );

    	int x, y;
    	int cx, cy;

    	matrixview->table = gtk_table_new( matrix->height, matrix->width, 
    		TRUE );
    	gtk_box_pack_start( GTK_BOX( matrixview->box ), 
    		matrixview->table, FALSE, FALSE, 0 );

    	/* Find the centre position, if there is one. We give this a special
    	 * name; it is highlighted by our resource file.
    	 */
    	cx = -1; cy = -1;
    	if( matrix->height & 0x1 )
    		cy = matrix->height >> 1;
    	if( matrix->width & 0x1 )
    		cx = matrix->width >> 1;

    	/* Build contents.
    	 */
    	for( y = 0; y < matrix->height; y++ )
    		for( x = 0; x < matrix->width; x++ ) {
    			GtkWidget *but;

    			but = gtk_button_new_with_label( "0" );
    			gtk_signal_connect( GTK_OBJECT( but ), "clicked", 
    				GTK_SIGNAL_FUNC( matrixview_toggle_change_cb ),
    				matrixview );
    			if( x == cx && y == cy )
    				set_name( but, "centre_widget" );
			/*

				FIXME ... this b0rks thanks to pangolayout
				confusion

    			set_fixed( GTK_BIN( but )->child, 1 );
			 */

    			gtk_table_attach( GTK_TABLE( matrixview->table ), but,
    				x, x + 1, y, y + 1, GTK_FILL, GTK_FILL, 2, 2 );
    			matrixview->items = 
    				g_slist_append( matrixview->items, but );
    		}
}

/* Change to a scale in a Tslider. 
 */
/*ARGSUSED*/
static void
matrixview_slider_change_cb( Tslider *tslider, Matrixview *matrixview )
{
    	Matrix *matrix = MATRIX( VOBJECT( matrixview )->iobject );
    	int pos = g_slist_index( matrixview->items, tslider );

    	assert( pos >= 0 );

    	/* Install value.
    	 */
    	if( matrix->value[pos] != tslider->svalue ) {
    		matrix->value[pos] = tslider->svalue;

    		classmodel_update( CLASSMODEL( matrix ) );
    		symbol_recalculate_all();
    	}
}

/* Build a set of slider items for a matrix. 
 */
static void
matrixview_slider_build( Matrixview *matrixview )
{
    	Matrix *matrix = MATRIX( VOBJECT( matrixview )->iobject );

    	int x, y;

    	matrixview->table = gtk_table_new( matrix->height, matrix->width, 
    		TRUE );
    	gtk_box_pack_start( GTK_BOX( matrixview->box ), 
    		matrixview->table, TRUE, TRUE, 0 );

    	for( y = 0; y < matrix->height; y++ )
    		for( x = 0; x < matrix->width; x++ ) {
    			Tslider *tslider = tslider_new();

    			tslider_set_conversions( tslider, NULL, NULL );
    			tslider->from = -2;
    			tslider->to = 2;
    			tslider->digits = 3;

    			gtk_signal_connect_object( GTK_OBJECT( tslider ), 
    				"text_changed",
    				GTK_SIGNAL_FUNC( view_changed_cb ), 
    				GTK_OBJECT( matrixview ) );
    			gtk_signal_connect_object( GTK_OBJECT( tslider ), 
    				"activate", 
    				GTK_SIGNAL_FUNC( view_activate_cb ), 
    				GTK_OBJECT( matrixview ) );
    			gtk_signal_connect( GTK_OBJECT( tslider ), 
    				"slider_changed", 
    				GTK_SIGNAL_FUNC( matrixview_slider_change_cb ),
    				matrixview );

    			gtk_container_set_border_width( 
    				GTK_CONTAINER( tslider ), 2 );
    			gtk_table_attach_defaults( 
    				GTK_TABLE( matrixview->table ), 
    				GTK_WIDGET( tslider ),
    				x, x + 1, y, y + 1 );
    			matrixview->items = g_slist_append( matrixview->items, 
    				tslider );
    		}
}

static void
matrixview_text_focus_in( GtkWidget *entry )
{
    	gtk_editable_select_region( GTK_EDITABLE( entry ), 0, -1 );
}

static void
matrixview_text_focus_out( GtkWidget *entry )
{
    	gtk_editable_select_region( GTK_EDITABLE( entry ), 0, 0 );
}

static void
matrixview_text_connect( Matrixview *matrixview, GtkWidget *txt )
{
    	gtk_signal_connect_object( GTK_OBJECT( txt ), "changed",
    		GTK_SIGNAL_FUNC( view_changed_cb ), 
    		GTK_OBJECT( matrixview ) );
    	gtk_signal_connect_object( GTK_OBJECT( txt ), "activate",
    		GTK_SIGNAL_FUNC( view_activate_cb ), 
    		GTK_OBJECT( matrixview ) );

    	/* Select text on focus-in, deselect on focus out.
    	 */
    	gtk_signal_connect( GTK_OBJECT( txt ), "focus_in_event",
    		GTK_SIGNAL_FUNC( matrixview_text_focus_in ), NULL );
    	gtk_signal_connect( GTK_OBJECT( txt ), "focus_out_event",
    		GTK_SIGNAL_FUNC( matrixview_text_focus_out ), NULL );
}

static void
matrixview_text_build_scale_offset( Matrixview *matrixview )
{
	GtkSizeGroup *group;

    	matrixview->cbox = gtk_vbox_new( FALSE, 2 );
        gtk_box_pack_end( GTK_BOX( matrixview->box ), 
    		GTK_WIDGET( matrixview->cbox ), FALSE, FALSE, 0 );

	group = gtk_size_group_new( GTK_SIZE_GROUP_HORIZONTAL );

	matrixview->scale = 
		build_glabeltext4( matrixview->cbox, group, _( "Scale" ) );
	gtk_entry_set_width_chars( GTK_ENTRY( matrixview->scale ), 6 );
    	matrixview_text_connect( matrixview, matrixview->scale );

	matrixview->offset = 
		build_glabeltext4( matrixview->cbox, group, _( "Offset" ) );
	gtk_entry_set_width_chars( GTK_ENTRY( matrixview->offset ), 6 );
    	matrixview_text_connect( matrixview, matrixview->offset );

	UNREF( group );
}

#ifndef HAVE_GTK_EXTRA
/* Build a set of text items for a matrix. 
 */
static void
matrixview_text_build( Matrixview *matrixview )
{
    	Matrix *matrix = MATRIX( VOBJECT( matrixview )->iobject );

    	int x, y;

    	matrixview->table = gtk_table_new( matrix->height, matrix->width, 
    		TRUE );
    	gtk_box_pack_start( GTK_BOX( matrixview->box ), 
    		matrixview->table, FALSE, FALSE, 0 );

    	for( y = 0; y < matrix->height; y++ )
    		for( x = 0; x < matrix->width; x++ ) {
    			GtkWidget *txt;

    			txt = build_entry( 8 );
    			gtk_table_attach( GTK_TABLE( matrixview->table ), txt,
    				x, x + 1, y, y + 1, GTK_FILL, GTK_FILL, 2, 2 );
    			matrixview_text_connect( matrixview, txt );

    			matrixview->items = 
    				g_slist_append( matrixview->items, txt );
    		}

    	if( matrix->display == MATRIX_DISPLAY_TEXT_SCALE_OFFSET )
    		/* Make the scale/offset widgets too.
    		 */
    		matrixview_text_build_scale_offset( matrixview );
}
#endif /*!HAVE_GTK_EXTRA*/

#ifdef HAVE_GTK_EXTRA
static int 
matrixview_text_traverse( GtkSheet *sheet,
    	int old_row, int old_col, int *new_row, int *new_col,
    	Matrixview *matrixview )
{
    	Matrix *matrix = MATRIX( VOBJECT( matrixview )->iobject );
    	Row *row = HEAPMODEL( matrix )->row;
	Mainw *mainw = MAINW( idialog_get_root( widget ) );

    	char *new_text = gtk_sheet_cell_get_text( sheet, *new_row, *new_col );

    	BufInfo buf;
    	char str[MAX_LINELENGTH];

	/* Make a note of what's in this cell before any editing ... "changed"
	 * does a strcmp() on this to spot edit actions.
	 */
	SETSTR( matrixview->cell_text, new_text );
	matrixview->cell_row = *new_row;
	matrixview->cell_col = *new_col;

    	buf_init_static( &buf, str, MAX_LINELENGTH );
    	row_qualified_name( row, &buf );
	/* Expands to (eg) "A2: cell (1,2): 45" ... status line display during
	 * matrix traverse.
	 */
    	buf_appendf( &buf, _( ": cell (%d, %d): %s" ), 
		*new_col, *new_row, new_text );
	mainw_set_status( mainw, "%s", buf_all( &buf ) );

    	return( TRUE );
}

static void 
matrixview_text_changed( GtkSheet *sheet,
    	int row, int col, Matrixview *matrixview )
{
	/* "changed" is emitted on many changes :-( compare text with start
	 * text to see if we have an edit.
	 */
	if( matrixview->cell_text && 
		!VIEW( matrixview )->scannable &&
		row == matrixview->cell_row && col == matrixview->cell_col ) {
		char *text = gtk_sheet_cell_get_text( sheet, row, col );

		if( strcmp( matrixview->cell_text, text ) != 0 ) {
#ifdef DEBUG
			printf( "matrixview_text_changed: "
				"old = \"%s\", new = \"%s\"\n",
				matrixview->cell_text, text );
#endif /*DEBUG*/

			view_changed_cb( VIEW( matrixview ) );
		}
	}
}

static gboolean
matrixview_text_event( GtkWidget *widget, 
	GdkEvent *ev, Matrixview *matrixview )
{
        if( ev->type != GDK_KEY_PRESS || ev->key.keyval != GDK_Return )
                return( FALSE );

	view_activate_cb( VIEW( matrixview ) );

        return( FALSE );
}

static void
matrixview_text_sheet_build( Matrixview *matrixview )
{
    	Matrix *matrix = MATRIX( VOBJECT( matrixview )->iobject );
	GtkWidget *entry;
	GtkAdjustment *hadj, *vadj;

    	int cell_width;
    	int cell_height;

    	matrixview->sheet = gtk_sheet_new( matrix->height, matrix->width, "" );
    	gtk_sheet_hide_column_titles( GTK_SHEET( matrixview->sheet ) );
    	gtk_sheet_hide_row_titles( GTK_SHEET( matrixview->sheet ) );
    	gtk_sheet_set_selection_mode( GTK_SHEET( matrixview->sheet ), 
    		GTK_SELECTION_SINGLE );
    	cell_width = GTK_SHEET( matrixview->sheet )->row_title_area.width;
    	cell_height = GTK_SHEET( matrixview->sheet )->column_title_area.height;
    	gtk_signal_connect( GTK_OBJECT( matrixview->sheet ), "traverse",
    		GTK_SIGNAL_FUNC( matrixview_text_traverse ), matrixview );
    	gtk_signal_connect( GTK_OBJECT( matrixview->sheet ), "changed",
    		GTK_SIGNAL_FUNC( matrixview_text_changed ), matrixview );

	/* We can't connect to "activate" on sheet's entry :-( most
	 * gtk_sheet's fail to emit it. Have to parse events ourselves.
	 */
	entry = gtk_sheet_get_entry( GTK_SHEET( matrixview->sheet ) );
	gtk_signal_connect( GTK_OBJECT( entry ), "event",
    		GTK_SIGNAL_FUNC( matrixview_text_event ), 
		GTK_OBJECT( matrixview ) );

	/* For large matricies, display in a scrolled window.
	 */
	if( matrix->width * cell_width > 500 || 
		matrix->height * cell_height > 500 ) {
		matrixview->swin = gtk_scrolled_window_new( NULL, NULL );
		gtk_scrolled_window_set_policy( 
			GTK_SCROLLED_WINDOW( matrixview->swin ),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC );
		hadj = gtk_scrolled_window_get_hadjustment( 
			GTK_SCROLLED_WINDOW( matrixview->swin ) );
		vadj = gtk_scrolled_window_get_vadjustment( 
			GTK_SCROLLED_WINDOW( matrixview->swin ) );

		gtk_widget_set_size_request( GTK_WIDGET( matrixview->swin ), 
			IM_MIN( 500, matrix->width * cell_width + 40 ), 
			IM_MIN( 500, matrix->height * cell_height + 40 ) );

		gtk_container_add( GTK_CONTAINER( matrixview->swin ), 
			matrixview->sheet );
		gtk_box_pack_start( GTK_BOX( matrixview->box ), 
			matrixview->swin, FALSE, FALSE, 0 );
	}	
	else {
		hadj = GTK_ADJUSTMENT( 
			gtk_adjustment_new( 0, 0, 100, 10, 10, 10 ) );
		vadj = GTK_ADJUSTMENT( 
			gtk_adjustment_new( 0, 0, 100, 10, 10, 10 ) );

		gtk_widget_set_size_request( GTK_WIDGET( matrixview->sheet ), 
			matrix->width * cell_width, 
			matrix->height * cell_height );

		gtk_box_pack_start( GTK_BOX( matrixview->box ), 
			matrixview->sheet, FALSE, FALSE, 0 );
	}

    	gtk_sheet_set_hadjustment( GTK_SHEET( matrixview->sheet ), hadj );
    	gtk_sheet_set_vadjustment( GTK_SHEET( matrixview->sheet ), vadj );
}

static void
matrixview_text_build( Matrixview *matrixview )
{
    	Matrix *matrix = MATRIX( VOBJECT( matrixview )->iobject );

	/* Sheet hates zero width/height. Can happen during class construct.
	 */
	if( matrix->width > 0 && matrix->height > 0 )
		matrixview_text_sheet_build( matrixview );

    	if( matrix->display == MATRIX_DISPLAY_TEXT_SCALE_OFFSET )
    		matrixview_text_build_scale_offset( matrixview );
}

#endif /*HAVE_GTK_EXTRA*/

/* Set the label on a toggle button to reflect its value.
 */
static void
matrixview_toggle_set_label( GtkWidget *button, double v )
{
    	GtkWidget *label = GTK_BIN( button )->child;

    	g_return_if_fail( GTK_IS_LABEL( label ) );

    	switch( (int) v ) {
    	case 0:
    		set_glabel( label, "0" );
    		break;

    	case 255:
    		set_glabel( label, "1" );
    		break;

    	default:
    		set_glabel( label, "*" );
    		break;
    	}
}

/* Refresh a set of toggle items for a matrix. 
 */
static void
matrixview_toggle_refresh( Matrixview *matrixview )
{
    	Matrix *matrix = MATRIX( VOBJECT( matrixview )->iobject );

    	int x, y;
    	GSList *p;

    	for( p = matrixview->items, y = 0; y < matrixview->height; y++ )
    		for( x = 0; x < matrixview->width; x++, p = p->next ) {
    			GtkWidget *wid = GTK_WIDGET( p->data );
    			int pos = x + y * matrixview->width;

    			matrixview_toggle_set_label( wid, matrix->value[pos] );
    		}
}

/* Refresh a set of slider items for a matrix. 
 */
static void
matrixview_slider_refresh( Matrixview *matrixview )
{
    	Matrix *matrix = MATRIX( VOBJECT( matrixview )->iobject );

    	int x, y;
    	GSList *p;

    	for( p = matrixview->items, y = 0; y < matrixview->height; y++ )
    		for( x = 0; x < matrixview->width; x++, p = p->next ) {
    			Tslider *tslider = TSLIDER( p->data );
    			int pos = x + y * matrixview->width;

    			tslider->value = matrix->value[pos];
    			tslider->svalue = matrix->value[pos];

    			tslider_changed( tslider );
    		}
}

static void
matrixview_text_set( Matrixview *matrixview, GtkWidget *txt, double val )
{
    	if( txt ) {
    		gtk_signal_handler_block_by_data( 
    			GTK_OBJECT( txt ), matrixview );
    		set_gentry( txt, "%g", val ); 
    		gtk_signal_handler_unblock_by_data( 
    			GTK_OBJECT( txt ), matrixview );
    	}
}

#ifndef HAVE_GTK_EXTRA
/* Fill the widgets!
 */
static void
matrixview_text_refresh( Matrixview *matrixview )
{
    	Matrix *matrix = MATRIX( VOBJECT( matrixview )->iobject );

    	int x, y;
    	GSList *p;

    	for( p = matrixview->items, y = 0; y < matrixview->height; y++ )
    		for( x = 0; x < matrixview->width; x++, p = p->next ) {
    			GtkWidget *item = GTK_WIDGET( p->data );
    			int i = x + y * matrixview->width;
    			double val = matrix->value[i];

    			matrixview_text_set( matrixview, item, val );
    		}

    	matrixview_text_set( matrixview, matrixview->scale, matrix->scale );
    	matrixview_text_set( matrixview, matrixview->offset, matrix->offset );
}
#endif /*!HAVE_GTK_EXTRA*/

#ifdef HAVE_GTK_EXTRA
static void
matrixview_text_refresh( Matrixview *matrixview )
{
    	Matrix *matrix = MATRIX( VOBJECT( matrixview )->iobject );

    	int x, y;

	/* width and height can be zero (during class construct), in which
	 * case they'll be no gtk_sheet.
	 */
	if( matrixview->sheet )
		gtk_sheet_freeze( GTK_SHEET( matrixview->sheet ) );

    	for( y = 0; y < matrixview->height; y++ )
    		for( x = 0; x < matrixview->width; x++ ) {
    			int i = x + y * matrixview->width;
    			double val = matrix->value[i];
    			char buf[256];

    			snprintf( buf, 256, "%g", val );
			gtk_signal_handler_block_by_data( 
				GTK_OBJECT( matrixview->sheet ), matrixview );
    			gtk_sheet_set_cell_text( GTK_SHEET( matrixview->sheet ),
                                        y, x, buf );
			gtk_signal_handler_unblock_by_data( 
				GTK_OBJECT( matrixview->sheet ), matrixview );

			if( x == matrixview->cell_col && 
				y == matrixview->cell_row ) 
				SETSTR( matrixview->cell_text, buf );
    		}

	if( matrixview->sheet )
		gtk_sheet_thaw( GTK_SHEET( matrixview->sheet ) );

    	matrixview_text_set( matrixview, matrixview->scale, matrix->scale );
    	matrixview_text_set( matrixview, matrixview->offset, matrix->offset );
}
#endif /*HAVE_GTK_EXTRA*/

static void
matrixview_refresh( vObject *vobject )
{
    	Matrixview *matrixview = MATRIXVIEW( vobject );
    	Matrix *matrix = MATRIX( VOBJECT( matrixview )->iobject );
    	gboolean built = FALSE;

#ifdef DEBUG
    	printf( "matrixview_refresh\n" );
#endif /*DEBUG*/

    	/* Is there a UI already there we can reuse? Has to be same size and
    	 * type.
    	 */
    	if( matrixview->display != matrix->display || 
    		matrixview->width != matrix->width || 
    		matrixview->height != matrix->height ) {
    		/* Kill old UI stuff.
    		 */
    		FREEF( gtk_widget_destroy, matrixview->table );
    		FREEF( gtk_widget_destroy, matrixview->sheet );
    		FREEF( gtk_widget_destroy, matrixview->swin );
    		FREEF( g_slist_free, matrixview->items );
    		FREEF( gtk_widget_destroy, matrixview->cbox );
		FREE( matrixview->cell_text );
		matrixview->cell_row = -1;
		matrixview->cell_col = -1;
    		matrixview->scale = NULL;
    		matrixview->offset = NULL;

    		/* Make new contents. 
    		 */
		switch( matrix->display ) {
		case MATRIX_DISPLAY_TOGGLE:
			matrixview_toggle_build( matrixview );
			break;

		case MATRIX_DISPLAY_SLIDER:
			matrixview_slider_build( matrixview );
			break;

		case MATRIX_DISPLAY_TEXT:
		case MATRIX_DISPLAY_TEXT_SCALE_OFFSET:
			matrixview_text_build( matrixview );
			break;

		default:
			assert( FALSE );
		}

    		matrixview->width = matrix->width;
    		matrixview->height = matrix->height;
    		matrixview->display = matrix->display;

    		built = TRUE;
    	}

    	switch( matrixview->display ) {
    	case MATRIX_DISPLAY_TOGGLE:
    		matrixview_toggle_refresh( matrixview );
    		break;

    	case MATRIX_DISPLAY_SLIDER:
    		matrixview_slider_refresh( matrixview );
    		break;

    	case MATRIX_DISPLAY_TEXT:
    	case MATRIX_DISPLAY_TEXT_SCALE_OFFSET:
    		matrixview_text_refresh( matrixview );
    		break;

    	default:
    		assert( FALSE );
    	}

    	/* If we've built a new display, need to show after _refresh.
    	 */
    	if( built ) {
    		gtk_widget_show_all( GTK_WIDGET( matrixview ) );
    		view_resize( VIEW( matrixview ) );
    	}

    	VOBJECT_CLASS( parent_class )->refresh( vobject );
}

static void
matrixview_class_init( MatrixviewClass *class )
{
    	GtkObjectClass *object_class = (GtkObjectClass *) class;
    	vObjectClass *vobject_class = (vObjectClass *) class;
    	ViewClass *view_class = (ViewClass *) class;

	parent_class = g_type_class_peek_parent( class );

    	object_class->destroy = matrixview_destroy;

    	/* Create signals.
    	 */

    	/* Init methods.
    	 */
    	vobject_class->refresh = matrixview_refresh;

    	view_class->scan = matrixview_scan;
}

static void
matrixview_init( Matrixview *matrixview )
{
#ifdef DEBUG
    	printf( "matrixview_init\n" );
#endif /*DEBUG*/

    	matrixview->box = gtk_hbox_new( FALSE, 12 );
        gtk_box_pack_start( GTK_BOX( matrixview ), 
    		GTK_WIDGET( matrixview->box ), FALSE, FALSE, 0 );

    	/* Build on 1st refresh.
    	 */
    	matrixview->sheet = NULL;
    	matrixview->swin = NULL;
    	matrixview->cell_text = NULL;
    	matrixview->cell_row = -1;
    	matrixview->cell_col = -1;
    	matrixview->table = NULL;
    	matrixview->items = NULL;
    	matrixview->width = -1;
    	matrixview->height = -1;
    	matrixview->cbox = NULL;
    	matrixview->scale = NULL;
    	matrixview->offset = NULL;
}

GtkType
matrixview_get_type( void )
{
    	static GtkType matrixview_type = 0;

    	if( !matrixview_type ) {
    		static const GtkTypeInfo info = {
    			"Matrixview",
    			sizeof( Matrixview ),
    			sizeof( MatrixviewClass ),
    			(GtkClassInitFunc) matrixview_class_init,
    			(GtkObjectInitFunc) matrixview_init,
    			/* reserved_1 */ NULL,
    			/* reserved_2 */ NULL,
    			(GtkClassInitFunc) NULL,
    		};

    		matrixview_type = gtk_type_unique( TYPE_GRAPHICVIEW, &info );
    	}

    	return( matrixview_type );
}

View *
matrixview_new( void )
{
    	Matrixview *matrixview = gtk_type_new( TYPE_MATRIXVIEW );

    	return( VIEW( matrixview ) );
}

