/* make and manage base windows ... dialog (messagebox, file box), top
 * level windows
 */

/*

    Copyright (C) 1991-2001, The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/* 
  
    build interface:

	iwnd = iwindow_new( type );
	iwindow_set_*( iwnd, ... );
	iwindow_build( iwnd );

    destroy interface:

    	iwindow_kill()

		'cancellable' kill ... user popdown can return IWINDOW_ERROR
		or IWINDOW_FALSE to prevent popdown

	gtk_widget_destroy()

		non-cancellable ... popdown is not called

	so ... don't free() in popdown, subclass iwnd and free() in _destroy()

 */

#include "ip.h"

/* Cursor bitmaps.
 */
#include "BITMAPS/dropper_src.xbm"
#include "BITMAPS/dropper_msk.xbm"
#include "BITMAPS/magin_src.xbm"
#include "BITMAPS/magout_src.xbm"
#include "BITMAPS/mag_msk.xbm"
#include "BITMAPS/watch_1.xbm"
#include "BITMAPS/watch_2.xbm"
#include "BITMAPS/watch_3.xbm"
#include "BITMAPS/watch_4.xbm"
#include "BITMAPS/watch_5.xbm"
#include "BITMAPS/watch_6.xbm"
#include "BITMAPS/watch_7.xbm"
#include "BITMAPS/watch_8.xbm"
#include "BITMAPS/watch_msk.xbm"

/*
#define DEBUG
 */

static GtkWindowClass *parent_class = NULL;

/* List of all iwindows.
 */
static GSList *iwindow_all = NULL;

/* All our cursors.
 */
static GdkCursor *iwindow_cursor[IWINDOW_SHAPE_LAST] = { NULL };

int
iwindow_number( void )
{
	return( g_slist_length( iwindow_all ) );
}

/* Over all windows.
 */
void *
iwindow_map_all( iWindowMapFn fn, void *a )
{
	return( slist_map( iwindow_all, (SListMapFn) fn, a ) );
}

/* Make a custom cursor ... source, mask, width, height and hot spot position.
 */
static GdkCursor *
iwindow_make_cursor_data( guchar *src_bits, guchar *msk_bits, 
	int w, int h, int x, int y )
{
	GdkPixmap *src;
	GdkPixmap *msk;
	GdkCursor *cursor;
	GdkColor fg = { 0, 255 << 8, 255 << 8, 255 << 8 };
	GdkColor bg = { 0, 0, 0, 0 };

	src = gdk_bitmap_create_from_data( NULL, 
		(const char *) src_bits, w, h );
	msk = gdk_bitmap_create_from_data( NULL, 
		(const char *) msk_bits, w, h );
	cursor = gdk_cursor_new_from_pixmap( src, msk, &fg, &bg, x, y );
	gdk_pixmap_unref( src );
	gdk_pixmap_unref( msk );

	return( cursor );
}

/* Build all the cursors.
 */
static void
iwindow_make_cursors( void )
{
	/* Init standard cursors with this table.
	 */
	static GdkCursorType standards[] = {
		GDK_CURSOR_IS_PIXMAP,	/* IWINDOW_SHAPE_DROPPER */
		GDK_PENCIL,		/* IWINDOW_SHAPE_PEN */
		GDK_HAND2,		/* IWINDOW_SHAPE_SMUDGE */
		GDK_SPIDER,		/* IWINDOW_SHAPE_SMEAR */
		GDK_GOBBLER,		/* IWINDOW_SHAPE_TEXT */
		GDK_SIZING,		/* IWINDOW_SHAPE_RECT */
		GDK_TREK,		/* IWINDOW_SHAPE_FLOOD */
		GDK_FLEUR,		/* IWINDOW_SHAPE_MOVE */
		GDK_CROSSHAIR,		/* IWINDOW_SHAPE_EDIT */
		GDK_CURSOR_IS_PIXMAP, 	/* IWINDOW_SHAPE_MAGIN */
		GDK_CURSOR_IS_PIXMAP, 	/* IWINDOW_SHAPE_MAGOUT */
		GDK_TOP_SIDE,		/* IWINDOW_SHAPE_TOP */
		GDK_BOTTOM_SIDE,	/* IWINDOW_SHAPE_BOTTOM */
		GDK_LEFT_SIDE,		/* IWINDOW_SHAPE_LEFT */
		GDK_RIGHT_SIDE,		/* IWINDOW_SHAPE_RIGHT */
		GDK_TOP_RIGHT_CORNER,	/* IWINDOW_SHAPE_TOPRIGHT */
		GDK_TOP_LEFT_CORNER,	/* IWINDOW_SHAPE_TOPLEFT */
		GDK_BOTTOM_RIGHT_CORNER,/* IWINDOW_SHAPE_BOTTOMRIGHT, */
		GDK_BOTTOM_LEFT_CORNER,	/* IWINDOW_SHAPE_BOTTOMLEFT */
	};

	/* All the bits for the rotating cursor.
	 */
	static guchar *watch_bits[] = {
		watch_1_bits,
		watch_2_bits,
		watch_3_bits,
		watch_4_bits,
		watch_5_bits,
		watch_6_bits,
		watch_7_bits,
		watch_8_bits,
	};

	int i;

	if( iwindow_cursor[0] )
		return;

	/* Easy ones first.
	 */
	for( i = 0; i < IM_NUMBER( standards ); i++ )
		if( standards[i] != GDK_CURSOR_IS_PIXMAP )
			iwindow_cursor[i] = gdk_cursor_new( standards[i] );

	/* Custom cursors.
	 */
	iwindow_cursor[IWINDOW_SHAPE_DROPPER] = iwindow_make_cursor_data( 
		dropper_src_bits, dropper_msk_bits,
		dropper_src_width, dropper_src_height, 0, 15 );
	iwindow_cursor[IWINDOW_SHAPE_MAGIN] = iwindow_make_cursor_data( 
		magin_src_bits, mag_msk_bits, 
		mag_msk_width, mag_msk_height, 6, 6 );
	iwindow_cursor[IWINDOW_SHAPE_MAGOUT] = iwindow_make_cursor_data( 
		magout_src_bits, mag_msk_bits,
		mag_msk_width, mag_msk_height, 6, 6 );

	/* The hglasses.
	 */
	for( i = 0; i < IM_NUMBER( watch_bits ); i++ )
		iwindow_cursor[IWINDOW_SHAPE_HGLASS1 + i] = 
			iwindow_make_cursor_data( 
				watch_bits[i], watch_msk_bits,
				watch_1_width, watch_1_height, 7, 7 );
}

/* Get the work window.
 */
static GdkWindow *
iwindow_get_work_window( iWindow *iwnd )
{
	if( iwnd->work_window )
		return( iwnd->work_window );
	else
		return( GTK_WIDGET( iwnd )->window );
}

/* Update the cursor for a window.
 */
static void *
iwindow_cursor_update( iWindow *iwnd )
{
	if( GTK_WIDGET_REALIZED( GTK_WIDGET( iwnd ) ) ) {
		GSList *p;

		/* Global shape set? Use that for the whole window.
		 */
		if( iwnd->shape != IWINDOW_SHAPE_NONE ) {
			gdk_window_set_cursor( GTK_WIDGET( iwnd )->window, 
				iwindow_cursor[iwnd->shape] );
			gdk_window_set_cursor( iwindow_get_work_window( iwnd ),
				iwindow_cursor[iwnd->shape] );
			gdk_flush();

			return( NULL );
		}

		/* No global shape ... make sure there's no global cursor on
		 * this window.
		 */
		gdk_window_set_cursor( GTK_WIDGET( iwnd )->window, NULL );
		gdk_window_set_cursor( iwindow_get_work_window( iwnd ), NULL );

		/* And set the work area to the highest non-NONE shape on 
		 * the stack.
		 */
		for( p = iwnd->contexts; p; p = p->next ) {
			iWindowCursorContext *cntxt = 
				(iWindowCursorContext *) p->data;

			if( cntxt->shape != IWINDOW_SHAPE_NONE ) {
				gdk_window_set_cursor( 
					iwindow_get_work_window( iwnd ),
					iwindow_cursor[cntxt->shape] );
				return( NULL );
			}
		}

		gdk_flush();
	}

	return( NULL );
}

/* Set a global cursor for a window.
 */
static void *
iwindow_cursor_set( iWindow *iwnd, iWindowShape shape )
{
	iwnd->shape = shape;
	iwindow_cursor_update( iwnd );

	return( NULL );
}

static int iwindow_hglass_count = 0;

void
animate_hourglass( void )
{
	static iWindowShape shape = IWINDOW_SHAPE_HGLASS1;

	if( iwindow_hglass_count ) {
		iwindow_map_all( 
			(iWindowMapFn) iwindow_cursor_set, (void *) shape );

		shape += 1;
		if( shape > IWINDOW_SHAPE_HGLASS8 )
			shape = IWINDOW_SHAPE_HGLASS1;
	}
}

void
set_hourglass( void )
{
	assert( iwindow_hglass_count >= 0 );

	iwindow_hglass_count += 1;

	if( iwindow_hglass_count == 1 ) 
		animate_hourglass();
}

void
set_pointer( void )
{
	assert( iwindow_hglass_count > 0 );

	iwindow_hglass_count -= 1;

	if( !iwindow_hglass_count )
		iwindow_map_all( (iWindowMapFn) iwindow_cursor_set, 
			(void *) IWINDOW_SHAPE_NONE );
}

iWindowCursorContext *
iwindow_cursor_context_new( iWindow *iwnd )
{
	iWindowCursorContext *cntxt = INEW( NULL, iWindowCursorContext );

	cntxt->iwnd = iwnd;
	cntxt->shape = IWINDOW_SHAPE_NONE;
	iwnd->contexts = g_slist_append( iwnd->contexts, cntxt );

	return( cntxt );
}

void
iwindow_cursor_context_destroy( iWindowCursorContext *cntxt )
{
	iWindow *iwnd = cntxt->iwnd;

	iwnd->contexts = g_slist_remove( iwnd->contexts, cntxt );
	FREE( cntxt );
	iwindow_cursor_update( iwnd );
}

void
iwindow_cursor_context_set_cursor( iWindowCursorContext *cntxt, 
	iWindowShape shape )
{
	if( cntxt->shape != shape ) {
		cntxt->shape = shape;
		iwindow_cursor_update( cntxt->iwnd );
	}
}

iWindowSusp *
iwindow_susp_new( iWindowFn fn, 
	iWindow *iwnd, void *client, iWindowNotifyFn nfn, void *sys )
{
        iWindowSusp *susp;

        if( !(susp = INEW( NULL, iWindowSusp )) )
		return( NULL );

        susp->fn = fn; 
        susp->iwnd = iwnd; 
        susp->client = client; 
        susp->nfn = nfn;
        susp->sys = sys;

        return( susp );
}

/* Trigger a suspension's reply, and free it.
 */
void 
iwindow_susp_return( void *sys, iWindowResult result )
{
	iWindowSusp *susp = IWINDOW_SUSP( sys );

        susp->nfn( susp->sys, result );

        im_free( susp );
}

/* Compose two iWindowFns ... if this one succeeded, trigger the next in turn.
 * Otherwise bail out.
 */
void
iwindow_susp_comp( void *sys, iWindowResult result )
{
	iWindowSusp *susp = IWINDOW_SUSP( sys );

	if( result == IWINDOW_TRUE ) {
		susp->fn( susp->iwnd, susp->client, susp->nfn, susp->sys );
		im_free( susp );
	}
	else 
		iwindow_susp_return( sys, result );
}

/* Null window callback.
 */
void 
iwindow_true_cb( iWindow *iwnd, void *client, iWindowNotifyFn nfn, void *sys ) 
{ 
	nfn( sys, IWINDOW_TRUE );
}

void 
iwindow_false_cb( iWindow *iwnd, void *client, iWindowNotifyFn nfn, void *sys ) 
{ 
	nfn( sys, IWINDOW_FALSE );
}

/* Null notify callback.
 */
void iwindow_notify_null( void *client, iWindowResult result ) { }

/* Final end of a window. Destroy!
 */
static void
iwindow_final_death( iWindow *iwnd )
{
#ifdef DEBUG
	printf( "iwindow_final_death: %s\n", iwnd->title );
#endif /*DEBUG*/

	assert( iwnd->pending == 0 && iwnd->destroy );

	/* Clean up.
	 */
	gtk_widget_destroy( GTK_WIDGET( iwnd ) );
}

/* A notify comes back ... adjust the pending count. If this is a zombie and
 * this is the final pending, it's final death. 
 */
void
iwindow_notify_return( iWindow *iwnd )
{
#ifdef DEBUG
	printf( "iwindow_notify_return: %s (pending = %d)\n", 
		iwnd->title, iwnd->pending );
#endif /*DEBUG*/

	assert( iwnd->pending > 0 );

	iwnd->pending--;
	if( iwnd->destroy && iwnd->pending == 0 ) {
#ifdef DEBUG
		printf( "iwindow_notify_return: zombie death %s\n",
			iwnd->title );
#endif /*DEBUG*/
		iwindow_final_death( iwnd );
	}
}

/* Send a notify off, tell the client to come back to back.
 */
void
iwindow_notify_send( iWindow *iwnd, 
	iWindowFn fn, void *client, iWindowNotifyFn back, void *sys )
{
#ifdef DEBUG
	printf( "iwindow_notify_send: %s (pending = %d)\n", 
		iwnd->title, iwnd->pending );
#endif /*DEBUG*/

	iwnd->pending++;
	if( fn )
		fn( iwnd, client, back, sys );
	else
		back( sys, IWINDOW_TRUE );
}

static void
iwindow_finalize( GObject *gobject )
{
	iWindow *iwnd = IWINDOW( gobject );

#ifdef DEBUG
	printf( "iwindow_finalize: %s\n", iwnd->title );
#endif /*DEBUG*/

	/* My instance destroy stuff.
	 */
	iwindow_all = g_slist_remove( iwindow_all, iwnd );
	FREE( iwnd->title );

	G_OBJECT_CLASS( parent_class )->finalize( gobject );

	/* Last window and we've got through startup? Quit the application.
	 * Test for 1, since main() makes an invisible iwindow as the root.
	 */
	if( iwindow_number() == 1 && !main_starting )
		main_quit_test();
}

static void
iwindow_popdown_notify( iWindow *iwnd, iWindowResult result )
{
#ifdef DEBUG
	printf( "iwindow_popdown_notify: %s\n", iwnd->title );
#endif /*DEBUG*/

	if( result == IWINDOW_ERROR )
		box_alert( GTK_WIDGET( iwnd ) );
	else if( result == IWINDOW_TRUE )
		iwindow_kill( iwnd );

	if( result != IWINDOW_TRUE ) {
#ifdef DEBUG
		printf( "iwindow_popdown_notify: %s: kill cancelled!\n", 
			iwnd->title );
#endif /*DEBUG*/

		/* Cancel popdown.
		 */
		iwnd->destroy = FALSE;
	}
	else
		/* Popdown confirmed ... unmap.
		 */
		gtk_widget_unmap( GTK_WIDGET( iwnd ) );

	call_string_filename( (call_string_fn) gtk_accel_map_save,
		IP_SAVEDIR IM_DIR_SEP_STR "accel_map", NULL, NULL, NULL );

	iwindow_notify_return( iwnd );
}

static gboolean
iwindow_delete_event( GtkWidget *widget, GdkEventAny *event )
{
	iWindow *iwnd = IWINDOW( widget );

#ifdef DEBUG
	printf( "iwindow_delete_event: %s\n", iwnd->title );
#endif /*DEBUG*/

	if( !iwnd->destroy ) {
#ifdef DEBUG
		printf( "iwindow_delete_event: starting destroy\n" );
#endif /*DEBUG*/

		iwindow_kill( iwnd );
	}

	/* Never delete here ... wait for iwindow_popdown_notify to
	 * confirm the kill.
	 */
	return( TRUE );
}

static void 
iwindow_real_build( GtkWidget *widget )
{
	iWindow *iwnd = IWINDOW( widget );

#ifdef DEBUG
	printf( "iwindow_real_build: %s\n", iwnd->title );
#endif /*DEBUG*/

        gtk_container_set_border_width( GTK_CONTAINER( iwnd ), 0 );

        iwnd->work = gtk_vbox_new( FALSE, 0 );
        gtk_container_add( GTK_CONTAINER( iwnd ), iwnd->work );

	/* Call per-instance build.
	 */
        if( iwnd->build )
		iwnd->build( iwnd, iwnd->work, 
			iwnd->build_a, iwnd->build_b, iwnd->build_c );

	if( iwnd->title )
		gtk_window_set_title( GTK_WINDOW( iwnd ), iwnd->title );

        gtk_widget_show( iwnd->work );
}

static void
iwindow_class_init( iWindowClass *class )
{
	GObjectClass *gobject_class;
	GtkWidgetClass *widget_class;

	gobject_class = (GObjectClass *) class;
	widget_class = (GtkWidgetClass *) class;

	parent_class = g_type_class_peek_parent( class );

	/* Init methods.
	 */
	gobject_class->finalize = iwindow_finalize;

	widget_class->delete_event = iwindow_delete_event;

	class->build = iwindow_real_build;

	/* Create signals.
	 */

	/* Static class data init.
	 */
	iwindow_make_cursors();
}

static void
iwindow_init( iWindow *iwnd )
{
#ifdef DEBUG
	printf( "iwindow_init: %s\n", iwnd->title );
#endif /*DEBUG*/

	/* Init our instance fields.
	 */
	iwnd->work = NULL;

	/* Might as well.
	 */
	iwnd->accel_group = gtk_accel_group_new();
	gtk_window_add_accel_group( GTK_WINDOW( iwnd ), iwnd->accel_group );
	g_object_unref( iwnd->accel_group );

	iwnd->title = NULL;

	iwnd->build = NULL;
	iwnd->popdown = iwindow_true_cb;

	iwnd->destroy = FALSE;
	iwnd->pending = 0;

	iwnd->shape = IWINDOW_SHAPE_NONE;
	iwnd->contexts = NULL;
	iwnd->work_window = NULL;

	iwindow_all = g_slist_prepend( iwindow_all, iwnd );
}

GtkType
iwindow_get_type( void )
{
	static GtkType iwindow_type = 0;

	if (!iwindow_type) {
		static const GtkTypeInfo iwnd_info = {
			"iWindow",
			sizeof( iWindow ),
			sizeof( iWindowClass ),
			(GtkClassInitFunc) iwindow_class_init,
			(GtkObjectInitFunc) iwindow_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		iwindow_type = gtk_type_unique( GTK_TYPE_WINDOW, &iwnd_info );
	}

	return( iwindow_type );
}

GtkWidget *
iwindow_new( GtkWindowType type )
{
	iWindow *iwnd = gtk_type_new( TYPE_IWINDOW );
	GtkWindow *gwnd = GTK_WINDOW( iwnd );

	/* Init superclass.
	 */
	gwnd->type = type;

	return( GTK_WIDGET( iwnd ) );
}

void 
iwindow_set_title( iWindow *iwnd, const char *title, ... )
{
	va_list ap;
	char buf[1024];

        va_start( ap, title );
        (void) im_vsnprintf( buf, 1024, title, ap );
        va_end( ap );

	SETSTR( iwnd->title, buf );

	/* If the window has been realised, update title now.
	 */
	if( GTK_WIDGET_REALIZED( GTK_WIDGET( iwnd ) ) )
		gtk_window_set_title( GTK_WINDOW( iwnd ), iwnd->title );
}

void 
iwindow_set_build( iWindow *iwnd, 
	iWindowBuildFn build, void *build_a, void *build_b, void *build_c )
{
	iwnd->build = build;
	iwnd->build_a = build_a; 
	iwnd->build_b = build_b; 
	iwnd->build_c = build_c;
}

void 
iwindow_set_popdown( iWindow *iwnd, iWindowFn popdown, void *popdown_a )
{
	iwnd->popdown = popdown;
	iwnd->popdown_a = popdown_a;
}

void 
iwindow_set_work_window( iWindow *iwnd, GdkWindow *work_window )
{
	iwnd->work_window = work_window;
	iwindow_cursor_update( iwnd );
}

void *
iwindow_kill( iWindow *iwnd )
{
#ifdef DEBUG
	printf( "iwindow_kill: %s\n", iwnd->title );
#endif /*DEBUG*/

	if( !iwnd->destroy ) {
#ifdef DEBUG
		printf( "... starting destroy for %s\n", iwnd->title );
#endif /*DEBUG*/

		iwnd->destroy = TRUE;

		/* Don't kill directly, wait for popdown_notify to do it.
		 */
		iwindow_notify_send( iwnd, iwnd->popdown, iwnd->popdown_a,
			(iWindowNotifyFn) iwindow_popdown_notify, iwnd );
	}

	return( NULL );
}

void 
iwindow_build( iWindow *iwnd )
{
#ifdef DEBUG
	printf( "iwindow_build: %s\n", iwnd->title );
#endif /*DEBUG*/

	IWINDOW_GET_CLASS( iwnd )->build( GTK_WIDGET( iwnd ) );
}

/* Get the root window for a widget.
 */
GtkWidget *
iwindow_get_root( GtkWidget *widget )
{
	GtkWidget *toplevel = gtk_widget_get_toplevel( widget );
	GtkWidget *child = gtk_bin_get_child( GTK_BIN( toplevel ) );

	/* If this is a menu pane, get the widget that popped this menu up.
	 */
	if( GTK_IS_MENU( child ) ) {
		GtkWidget *parent = 
			gtk_menu_get_attach_widget( GTK_MENU( child ) );

		return( iwindow_get_root( parent ) );
	}

	return( toplevel );
}
