/* like a heapmodel, but we represent a class in the heap
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#include "ip.h"

/*
#define DEBUG
 */

static HeapmodelClass *parent_class = NULL;

void *
classmodel_get_instance( Classmodel *classmodel )
{
	ClassmodelClass *class = CLASSMODEL_GET_CLASS( classmodel );

	if( class && class->get_instance )
		return( class->get_instance( classmodel ) );

	return( NULL );
}

static void
classmodel_graphic_save_cb( iWindow *iwnd, 
	void *client, iWindowNotifyFn nfn, void *sys )
{
	Filesel *filesel = FILESEL( iwnd );
	Classmodel *classmodel = CLASSMODEL( client );
	ClassmodelClass *class = CLASSMODEL_GET_CLASS( classmodel );
	char *filename;

	if( (filename = filesel_get_filename( filesel )) ) {
		if( class->graphic_save( classmodel, 
			GTK_WIDGET( iwnd ), filename ) ) {
			path_add_file( filename );
			SETSTR( classmodel->filename, filename );
			iobject_changed( IOBJECT( classmodel ) );

			nfn( sys, IWINDOW_TRUE );
		}
		else
			nfn( sys, IWINDOW_ERROR );

		g_free( filename );
	}
	else
		nfn( sys, IWINDOW_ERROR );
}

void
classmodel_graphic_save( Classmodel *classmodel, GtkWidget *parent )
{
	ClassmodelClass *class = CLASSMODEL_GET_CLASS( classmodel );
	GtkWidget *filesel;
	BufInfo buf;
	char txt[100];

	if( !class->graphic_save ) {
		error_top( _( "Not implemented." ) );
		error_sub( _( "_%s() method not implemented for %s." ), 
			"graphic_save", G_OBJECT_TYPE_NAME( classmodel ) );
		box_alert( parent );
		return;
	}

	filesel = filesel_new();
	buf_init_static( &buf, txt, 100 );
	row_qualified_name( HEAPMODEL( classmodel )->row, &buf );
	iwindow_set_title( IWINDOW( filesel ), _( "Save %s \"%s\"" ), 
		G_OBJECT_TYPE_NAME( classmodel ), buf_all( &buf ) );
	filesel_set_flags( FILESEL( filesel ), TRUE, TRUE );
	filesel_set_filetype( FILESEL( filesel ), 
		class->filetype, class->default_filetype );
	idialog_set_parent( IDIALOG( filesel ), parent );
	idialog_set_iobject( IDIALOG( filesel ), IOBJECT( classmodel ) );
	filesel_set_done( FILESEL( filesel ), 
		classmodel_graphic_save_cb, classmodel );
	iwindow_build( IWINDOW( filesel ) );

	if( classmodel->filename )
		filesel_set_filename( FILESEL( filesel ), 
			classmodel->filename );

	gtk_widget_show( GTK_WIDGET( filesel ) );
}

static void
classmodel_graphic_replace_cb( iWindow *iwnd, 
	void *client, iWindowNotifyFn nfn, void *sys )
{
	Filesel *filesel = FILESEL( iwnd );
	Classmodel *classmodel = CLASSMODEL( client );
	ClassmodelClass *class = CLASSMODEL_GET_CLASS( classmodel );
	char *filename;

	if( (filename = filesel_get_filename( filesel )) ) {
		if( class->graphic_replace( classmodel, 
			GTK_WIDGET( iwnd ), filename ) ) {
			symbol_recalculate_all();
			path_add_file( filename );
			SETSTR( classmodel->filename, filename );
			iobject_changed( IOBJECT( classmodel ) );

			nfn( sys, IWINDOW_TRUE );
		}
		else
			nfn( sys, IWINDOW_ERROR );

		g_free( filename );
	}
	else
		nfn( sys, IWINDOW_ERROR );
}

void
classmodel_graphic_replace( Classmodel *classmodel, GtkWidget *parent )
{
	ClassmodelClass *class = CLASSMODEL_GET_CLASS( classmodel );
	GtkWidget *filesel;
	BufInfo buf;
	char txt[100];

	if( !class->graphic_replace ) {
		error_top( _( "Not implemented." ) );
		error_sub( _( "_%s() method not implemented for %s." ), 
			"graphic_replace",
			G_OBJECT_TYPE_NAME( classmodel ) );
		box_alert( parent );
		return;
	}

	buf_init_static( &buf, txt, 100 );
	row_qualified_name( HEAPMODEL( classmodel )->row, &buf );
	filesel = filesel_new();
	iwindow_set_title( IWINDOW( filesel ), _( "Replace %s \"%s\"" ), 
		G_OBJECT_TYPE_NAME( classmodel ), buf_all( &buf ) );
	filesel_set_flags( FILESEL( filesel ), TRUE, FALSE );
	filesel_set_filetype( FILESEL( filesel ), 
		class->filetype, class->default_filetype );
	idialog_set_parent( IDIALOG( filesel ), parent );
	idialog_set_iobject( IDIALOG( filesel ), IOBJECT( classmodel ) );
	filesel_set_done( FILESEL( filesel ), 
		classmodel_graphic_replace_cb, classmodel );
	iwindow_build( IWINDOW( filesel ) );

	if( classmodel->filename )
		filesel_set_filename( FILESEL( filesel ), 
			classmodel->filename );

	gtk_widget_show( GTK_WIDGET( filesel ) );
}

/* Make and break links between classmodels and the iimages displaying them.
 */
static void 
classmodel_iimage_link( Classmodel *classmodel, iImage *iimage )
{
	if( !g_slist_find( classmodel->iimages, iimage ) ) {
#ifdef DEBUG
		printf( "classmodel_iimage_link: linking " );
		row_name_print( HEAPMODEL( classmodel )->row );
		printf( " to " );
		row_name_print( HEAPMODEL( iimage )->row );
		printf( "\n" );
#endif /*DEBUG*/

		iimage->classmodels = 
			g_slist_prepend( iimage->classmodels, classmodel );
		classmodel->iimages = 
			g_slist_prepend( classmodel->iimages, iimage );
	}
}

void *
classmodel_iimage_unlink( Classmodel *classmodel, iImage *iimage )
{
	if( g_slist_find( classmodel->iimages, iimage ) ) {
#ifdef DEBUG
		printf( "classmodel_iimage_unlink: unlinking " );
		row_name_print( HEAPMODEL( classmodel )->row );
		printf( " from " );
		row_name_print( HEAPMODEL( iimage )->row );
		printf( "\n" );
#endif /*DEBUG*/

		iimage->classmodels = 
			g_slist_remove( iimage->classmodels, classmodel );
		classmodel->iimages = 
			g_slist_remove( classmodel->iimages, iimage );
	}

	return( NULL );
}

static void *
classmodel_iimage_unlink_rev( iImage *iimage, Classmodel *classmodel )
{
	return( classmodel_iimage_unlink( classmodel, iimage ) );
}

typedef struct {
	Classmodel *classmodel;
	Imageinfo *ii;
} ClassmodelSearch;

static void *
classmodel_iimage_expr_model( Model *model, ClassmodelSearch *parms )
{
	/* Look for iimages which aren't super ... ie. if this is a class
	 * derived from Image, display on the derived class, not on the
	 * superclass.
	 */
	if( IS_IIMAGE( model ) && !is_super( HEAPMODEL( model )->row->sym ) &&
		!is_this( HEAPMODEL( model )->row->sym ) ) {
		iImage *iimage = IIMAGE( model );

		if( iimage->instance.ii == parms->ii ) 
			classmodel_iimage_link( parms->classmodel, iimage );
	}

	return( NULL );
}

/* This classmodel is defined on an Imageinfo recorded as having been the value
 * of expr ... find an associated iImage, and link to that. 
 */
static void *
classmodel_iimage_expr( Expr *expr, ClassmodelSearch *parms )
{
	if( expr->row ) {
#ifdef DEBUG
		printf( "classmodel_iimage_expr: starting for " );
		row_name_print( expr->row );
		printf( "\n" );
#endif /*DEBUG*/

		/* Search this part of the tally for an iImage with ii as its
		 * derived value, and link to us. 
		 */
		(void) icontainer_map_all( ICONTAINER( expr->row->top_row ), 
			(icontainer_map_fn) classmodel_iimage_expr_model, 
			parms );
	}

	return( NULL );
}

/* classmodel is defined on ii ... update all the classmodel->iimage links.
 */
void
classmodel_iimage_update( Classmodel *classmodel, Imageinfo *ii )
{
	ClassmodelSearch parms;

	parms.classmodel = classmodel;
	parms.ii = ii;
	slist_map( classmodel->iimages, 
		(SListMapFn) classmodel_iimage_unlink_rev, classmodel );

	/* Don't make links for supers/this.
	 */
	if( HEAPMODEL( classmodel )->row->sym &&
		!is_super( HEAPMODEL( classmodel )->row->sym ) &&
		!is_this( HEAPMODEL( classmodel )->row->sym ) ) {
#ifdef DEBUG
		printf( "classmodel_iimage_update: " );
		row_name_print( HEAPMODEL( classmodel )->row );
		printf( " is defined on ii \"%s\" ... searching for client "
			"displays\n", ii->im->filename );
#endif /*DEBUG*/
		slist_map( imageinfo_expr_which( ii ), 
			(SListMapFn) classmodel_iimage_expr, &parms );
	}
}

/* Trigger the class_new method for a classmodel ... look for a constructor:
 * try CLASS_edit, then if that's not defined, try CLASS. Eg.  
 *	"A1.Slider_edit" from to value
 * if Slider_edit is not defined, try
 *	"A1.Slider" from to value
 */
static gboolean
classmodel_class_instance_new( Classmodel *classmodel )
{
	ClassmodelClass *class = CLASSMODEL_GET_CLASS( classmodel );
	Row *row = HEAPMODEL( classmodel )->row;
	PElement *root = &row->expr->root;
	const char *cname = IOBJECT( classmodel )->name;
	Reduce *rc = reduce_context;

	char cname_new[256];
	PElement fn;

	/* Find and build.
	 */
	im_snprintf( cname_new, 256, "%s_edit", cname );
	if( !class_get_member( root, cname_new, NULL, &fn ) ) {
		if( !class_get_member( root, cname, NULL, &fn ) ) 
			return( FALSE );
	}
	if( !class->class_new( classmodel, &fn, root ) )
		return( FALSE );

	/* Reduce to base type.
	 */
	if( !reduce_pelement( rc, reduce_spine, root ) ) 
		return( FALSE );

	/* We have a new heap struct ... tell everyone to get new pointers.
	 */
	if( heapmodel_new_heap( HEAPMODEL( row ), root ) )
		return( FALSE );

	return( TRUE );
}

static void
classmodel_dispose( GObject *gobject )
{
	Classmodel *classmodel;

	g_return_if_fail( gobject != NULL );
	g_return_if_fail( IS_CLASSMODEL( gobject ) );

	classmodel = CLASSMODEL( gobject );

	/* My instance destroy stuff.
	 */
	slist_map( classmodel->iimages, 
		(SListMapFn) classmodel_iimage_unlink_rev, classmodel );
	FREE( classmodel->filename );

	G_OBJECT_CLASS( parent_class )->dispose( gobject );
}

static void
classmodel_parent_add( iContainer *child )
{
	assert( IS_CLASSMODEL( child ) );

	ICONTAINER_CLASS( parent_class )->parent_add( child );
}

static void *
classmodel_update_model( Heapmodel *heapmodel )
{
	ClassmodelClass *class = CLASSMODEL_GET_CLASS( heapmodel );

#ifdef DEBUG
	printf( "classmodel_update_model: " );
	row_name_print( heapmodel->row );
	printf( "\n" );
#endif /*DEBUG*/

	if( heapmodel->row && heapmodel->row->expr ) {
		Classmodel *classmodel = CLASSMODEL( heapmodel );
		Expr *expr = heapmodel->row->expr;

		if( !heapmodel->modified ) {
			if( !class->class_get( classmodel, &expr->root ) )
				return( heapmodel );
		}
	}

	return( HEAPMODEL_CLASS( parent_class )->update_model( heapmodel ) );
}

static void *
classmodel_update_heap( Heapmodel *heapmodel )
{
	Classmodel *classmodel = CLASSMODEL( heapmodel );

	/* Nasty: classmodel_class_instance_new() can (indirectly) destroy us.
	 * Wrap a _ref()/_unref() pair around it to make sure we stay alive.
	 */
	g_object_ref( G_OBJECT( heapmodel ) );

	/* Build a new instance from the model.
	 */
	if( !classmodel_class_instance_new( classmodel ) ) {
		g_object_unref( G_OBJECT( heapmodel ) );
		return( heapmodel );
	}

	if( HEAPMODEL_CLASS( parent_class )->update_heap( heapmodel ) ) {
		g_object_unref( G_OBJECT( heapmodel ) );
		return( heapmodel );
	}

	g_object_unref( G_OBJECT( heapmodel ) );

	return( NULL );
}

static void *
classmodel_clear_edited( Heapmodel *heapmodel )
{
	Classmodel *classmodel = CLASSMODEL( heapmodel );

	classmodel_set_edited( classmodel, FALSE );

	return( HEAPMODEL_CLASS( parent_class )->clear_edited( heapmodel ) );
}

static gboolean
classmodel_real_class_get( Classmodel *classmodel, PElement *root )
{
	return( TRUE );
}

static void
classmodel_class_init( ClassmodelClass *class )
{
	GObjectClass *gobject_class = (GObjectClass *) class;
	iContainerClass *icontainer_class = (iContainerClass *) class;
	HeapmodelClass *heapmodel_class = (HeapmodelClass *) class;
	ClassmodelClass *classmodel_class = (ClassmodelClass *) class;

	parent_class = g_type_class_peek_parent( class );

	/* Init methods.
	 */
	gobject_class->dispose = classmodel_dispose;

	icontainer_class->parent_add = classmodel_parent_add;

	heapmodel_class->update_model = classmodel_update_model;
	heapmodel_class->update_heap = classmodel_update_heap;
	heapmodel_class->clear_edited = classmodel_clear_edited;

	classmodel_class->get_instance = NULL;

	classmodel_class->class_get = classmodel_real_class_get;
	classmodel_class->class_new = NULL;

	classmodel_class->graphic_save = NULL;
	classmodel_class->graphic_replace = NULL;

	classmodel_class->filetype = filesel_type_any;
	classmodel_class->default_filetype = 0;
}

static void
classmodel_init( Classmodel *classmodel )
{
	Model *model = MODEL( classmodel );

	model->display = FALSE;

        classmodel->edited = FALSE;

        classmodel->iimages = NULL;
        classmodel->views = NULL;

        classmodel->filename = NULL;
}

GType
classmodel_get_type( void )
{
	static GType type = 0;

	if( !type ) {
		static const GTypeInfo info = {
			sizeof( ClassmodelClass ),
			NULL,           /* base_init */
			NULL,           /* base_finalize */
			(GClassInitFunc) classmodel_class_init,
			NULL,           /* class_finalize */
			NULL,           /* class_data */
			sizeof( Classmodel ),
			32,             /* n_preallocs */
			(GInstanceInitFunc) classmodel_init,
		};

		type = g_type_register_static( TYPE_HEAPMODEL, 
			"Classmodel", &info, 0 );
	}

	return( type );
}

void
classmodel_set_edited( Classmodel *classmodel, gboolean edited )
{
	if( classmodel->edited != edited ) {
#ifdef DEBUG
		printf( "classmodel_set_edited: " );
		row_name_print( HEAPMODEL( classmodel )->row );
		printf( " %s\n", bool_to_char( edited ) );
#endif /*DEBUG*/

		classmodel->edited = edited;
		iobject_changed( IOBJECT( classmodel ) );

		if( HEAPMODEL( classmodel )->row && 
			HEAPMODEL( classmodel )->row->expr )
			expr_dirty( HEAPMODEL( classmodel )->row->expr, 
				link_serial_new() );
	}

	/* Mark eds for application.
	 */
	if( edited )
		heapmodel_set_modified( HEAPMODEL( classmodel ), TRUE );
}

/* The model has changed: mark for recomp.
 */
void
classmodel_update( Classmodel *classmodel )
{
	Row *row = HEAPMODEL( classmodel )->row;

	/* Eg. for no symol on load.
	 */
	if( !row->expr )
		return;

#ifdef DEBUG
	printf( "classmodel_update: " );
	row_name_print( HEAPMODEL( classmodel )->row );
	printf( "\n" );
#endif /*DEBUG*/

	/* classmodel_update_heap() will rebuild us on recomp.
	 */
	classmodel_set_edited( classmodel, TRUE );
	expr_dirty( row->expr, link_serial_new() );
	filemodel_set_modified( FILEMODEL( row->ws ), TRUE );
}
