/* abstract base class for items which can form a row in a tally
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
#define DEBUG_VIEWCHILD
 */

/* Time each refresh
#define DEBUG_TIME
 */

#include "ip.h"

static GtkVBoxClass *parent_class = NULL;

static Queue *view_dirty = NULL;

static GSList *view_scannable = NULL;

static GSList *view_resettable = NULL;

static gint view_idle_running = 0;

static View *view_base = NULL;

/* Remove from refresh queue.
 */
static void 
view_refresh_dequeue( View *view )
{
	if( view->dirty ) {
#ifdef DEBUG
		printf( "view_refresh_dequeue: \"%s\"\n", 
			OBJECT_CLASS_NAME( view ) );
#endif /*DEBUG*/

		view->dirty = FALSE;
		queue_remove( view_dirty, view );
	}
}

#ifdef DEBUG_TIME
/* Refresh all views at once and time them.
 */
static gint
view_idle_refresh( gpointer user_data )
{
	static GTimer *refresh_timer = NULL;
	double last_elapsed;
	double worst_time = 0.0;
	int worst_index = -1;
	void *data;
	int n;

	if( !refresh_timer )
		refresh_timer = g_timer_new();

	g_timer_reset( refresh_timer );

	printf( "view_idle_refresh: starting ...\n" );

	for( n = 0; (data = queue_head( view_dirty )); n++ ) {
		View *view = VIEW( data );
		double elapsed;

		view_refresh( view );
		view->dirty = FALSE;
		elapsed = g_timer_elapsed( refresh_timer, NULL );

		if( elapsed - last_elapsed > worst_time ) {
			worst_time = elapsed - last_elapsed;
			worst_index = n;
		}

		last_elapsed = elapsed;
	}

	printf( "view_idle_refresh: done after %gs (%d refreshes)\n",
		g_timer_elapsed( refresh_timer, NULL ), n );

	printf( "view_idle_refresh: worst %gs (refresh %d)\n", 
		worst_time, worst_index );

	view_idle_running = 0;

	return( FALSE );
}
#else /*DEBUG_TIME*/
/* Refresh stuff off the dirty list. Runs as an idle task.
 */
static gint
view_idle_refresh( gpointer user_data )
{
	void *data;

	while( (data = queue_head( view_dirty ) ) ) {
		View *view = VIEW( data );

#ifdef DEBUG
		printf( "view_idle_refresh: starting \"%s\" (0x%x)\n", 
			OBJECT_CLASS_NAME( view ), (unsigned int) view );
#endif /*DEBUG*/

		view_refresh( view );
		view->dirty = FALSE;
	}

	view_idle_running = 0;


	return( FALSE );
}
#endif /*DEBUG_TIME*/

/* Mark something for refresh. Seldom call this directly ... just change the 
 * model and all views will have a refresh queued.
 */
void 
view_refresh_queue( View *view )
{
	if( !view->dirty ) {
#ifdef DEBUG
		printf( "view_refresh_queue: %s (0x%x)", 
			OBJECT_CLASS_NAME( view ), (unsigned int) view );
		if( view->model )
			printf( ", model %s \"%s\"", 
				OBJECT_CLASS_NAME( view->model ), 
				NN( view->model->name ) );
		printf( "\n" );
#endif /*DEBUG*/

		view->dirty = TRUE;
		queue_add( view_dirty, view );

		if( !view_idle_running )
			view_idle_running = 
				gtk_idle_add( view_idle_refresh, NULL );
	}
}

void
view_scannable_register( View *view )
{
	/* Must have a scan method.
	 */
	assert( VIEW_CLASS( GTK_OBJECT( view )->klass )->scan );

	if( !view->scannable ) {
		view_scannable = g_slist_prepend( view_scannable, view );
		view->scannable = TRUE;
	}
}

void
view_scannable_unregister( View *view )
{
	if( view->scannable ) {
		view_scannable = g_slist_remove( view_scannable, view );
		view->scannable = FALSE;
	}
}

gboolean
view_scan_all( void )
{
	if( slist_map( view_scannable, (SListMapFn) view_scan, NULL ) )
		return( FALSE );

	view_reset_all();

	return( TRUE );
}

void
view_resettable_register( View *view )
{
	/* Must have a reset method.
	 */
	assert( VIEW_CLASS( GTK_OBJECT( view )->klass )->reset );

	if( !view->resettable ) {
		view_resettable = g_slist_prepend( view_resettable, view );
		view->resettable = TRUE;
	}
}

void
view_resettable_unregister( View *view )
{
	if( view->resettable ) {
		view_resettable = g_slist_remove( view_resettable, view );
		view->resettable = FALSE;
	}
}

void
view_reset_all( void )
{
	(void) slist_map( view_resettable, (SListMapFn) view_reset, NULL );
}

/* One of the children of the model we watch has changed ... create or destroy
 * the child view as required.
 */
static void
view_viewchild_changed( Model *model, ViewChild *viewchild )
{
	/* Don't write to the viewchild->view_child pointer ... let
	 * view_real_child_add() and view_real_child_remove() do this.
	 */
	if( !model->display && viewchild->child_view ) {
#ifdef DEBUG_VIEWCHILD
		printf( "view_viewchild_changed: %s \"%s\", removing view\n", 
			OBJECT_CLASS_NAME( model ), NN( model->name ) );
#endif /*DEBUG_VIEWCHILD*/

		gtk_widget_destroy( GTK_WIDGET( viewchild->child_view ) );
	}
	else if( model->display && !viewchild->child_view ) {
		View *child_view;

#ifdef DEBUG_VIEWCHILD
		printf( "view_viewchild_changed: %s \"%s\", adding view\n", 
			OBJECT_CLASS_NAME( model ), NN( model->name ) );
#endif /*DEBUG_VIEWCHILD*/

		child_view = model_view_new( model );
		view_link( child_view, model, viewchild->view );
	}
}

static ViewChild *
view_viewchild_new( View *view, Model *child_model )
{
	ViewChild *viewchild;

#ifdef DEBUG_VIEWCHILD
	printf( "view_viewchild_new: view \"%s\" watching %s \"%s\"\n", 
		OBJECT_CLASS_NAME( view ), 
		OBJECT_CLASS_NAME( child_model ), NN( child_model->name ) );
#endif /*DEBUG_VIEWCHILD*/

	if( !(viewchild = IM_NEW( NULL, ViewChild )) )
		return( NULL );

	viewchild->view = view;
	viewchild->child_model = child_model;
	viewchild->child_model_changed_sid = 
		gtk_signal_connect( GTK_OBJECT( child_model ), "changed", 
			GTK_SIGNAL_FUNC( view_viewchild_changed ), viewchild );
	viewchild->child_view = NULL;

	view->managed = g_slist_append( view->managed, viewchild );

	/* Make a view for this child, if we need one.
	 */
	view_viewchild_changed( child_model, viewchild );

	return( viewchild );
}

static void *
view_viewchild_destroy( ViewChild *viewchild )
{
	View *view = viewchild->view;

#ifdef DEBUG_VIEWCHILD
	printf( "view_viewchild_destroy: view %s watching model %s\n",
		OBJECT_CLASS_NAME( viewchild->view ), 
		OBJECT_CLASS_NAME( viewchild->child_model ) );
#endif /*DEBUG_VIEWCHILD*/

	FREEO( viewchild->child_view );
	FREESID( viewchild->child_model_changed_sid, viewchild->child_model );
	view->managed = g_slist_remove( view->managed, viewchild );

	im_free( viewchild );

	return( NULL );
}

static void *
view_viewchild_test_child_model( ViewChild *viewchild, Model *child_model )
{
	if( viewchild->child_model == child_model )
		return( viewchild );

	return( NULL );
}

/* Do we have a model?
 */
gboolean view_hasmodel( View *view ) { return( view->model != 0 ); }

void *
view_model_test( View *view, Model *model )
{
	if( view->model == model )
		return( view );

	return( NULL );
}

/* Link to enclosing model and view. 
 */
void 
view_link( View *view, Model *model, View *parent )
{
	VIEW_CLASS( GTK_OBJECT( view )->klass )->link( view, model, parent );
}

/* Add a child.
 */
void
view_child_add( View *parent, View *child )
{
	VIEW_CLASS( GTK_OBJECT( parent )->klass )->child_add( parent, child );
}

/* Remove a child.
 */
void
view_child_remove( View *parent, View *child )
{
	VIEW_CLASS( GTK_OBJECT( parent )->klass )->
		child_remove( parent, child );
}

/* Break link to model. 
 */
void 
view_unlink( View *view )
{
	assert( view != NULL );
	assert( view->model != NULL );
	assert( IS_VIEW( view ) && IS_MODEL( view->model ) );

	FREESID( view->changed_sid, view->model );
	FREESID( view->scrollto_sid, view->model );
	FREESID( view->reset_sid, view->model );
	FREESID( view->destroy_sid, view->model );
	FREESID( view->child_add_sid, view->model );
	FREESID( view->child_remove_sid, view->model );

	view->model = NULL;
}

static void
view_destroy( GtkObject *object )
{
	View *view;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_VIEW( object ) );

	view = VIEW( object );

#ifdef DEBUG
	printf( "view_destroy: \"%s\"\n", OBJECT_CLASS_NAME( object ) );
#endif /*DEBUG*/

	/* We're probably changing the size of our enclosing column.
	 */
	view_resize( view );

	if( view->scannable )
		view_scannable_unregister( view );
	if( view->resettable )
		view_resettable_unregister( view );

	slist_map( view->managed,
		(SListMapFn) view_viewchild_destroy, NULL );

	if( view->parent )
		view_child_remove( view->parent, view );

	if( view->model )
		view_unlink( view );

	view_refresh_dequeue( view );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

static void 
view_finalize( GtkObject *object )
{
#ifdef DEBUG
	printf( "view_finalize: \"%s\"\n", OBJECT_CLASS_NAME( object ) );
#endif /*DEBUG*/

	GTK_OBJECT_CLASS( parent_class )->finalize( object );
}

/* Called for model changed signal ... queue a refresh. 
 */
static void
view_model_changed( Model *model, View *view )
{
	assert( IS_MODEL( model ) );
	assert( IS_VIEW( view ) );

#ifdef DEBUG
	printf( "view_model_changed: %s %s \"%s\"\n", 
		OBJECT_CLASS_NAME( view ), 
		OBJECT_CLASS_NAME( model ), NN( model->name ) );
#endif /*DEBUG*/

	view_refresh_queue( view );
}

/* Called for model scrollto signal ... try scrolling.
 */
static void
view_model_scrollto( Model *model, View *view )
{
	assert( IS_MODEL( model ) );
	assert( IS_VIEW( view ) );

#ifdef DEBUG
	printf( "view_model_scrollto: %s\n", model->name );
#endif /*DEBUG*/

	view_scrollto( view );
}

/* Called for model reset signal ... try resetting.
 */
static void
view_model_reset( Model *model, View *view )
{
	assert( IS_MODEL( model ) );
	assert( IS_VIEW( view ) );

#ifdef DEBUG
	printf( "view_model_reset: %s\n", model->name );
#endif /*DEBUG*/

	view_reset( view );
}

/* Called for model destroy signal ... kill the view too.
 */
static void
view_model_destroy( Model *model, View *view )
{
#ifdef DEBUG
	printf( "view_model_destroy: model %s \"%s\"\n", 
		OBJECT_CLASS_NAME( model ), NN( model->name ) );
#endif /*DEBUG*/

	assert( IS_MODEL( model ) );
	assert( IS_VIEW( view ) );
	assert( view->model == model );

	gtk_widget_destroy( GTK_WIDGET( view ) );
}

/* Called for model child_add signal ... start watching that child.
 */
static void
view_model_child_add( Model *parent, Model *child, int pos, View *parent_view )
{
#ifdef DEBUG
	ViewChild *viewchild;

	printf( "view_model_child_add: parent %s \"%s\"\n", 
		OBJECT_CLASS_NAME( parent ), NN( parent->name ) );
#endif /*DEBUG*/

	assert( IS_MODEL( parent ) );
	assert( IS_MODEL( child ) );
	assert( IS_VIEW( parent_view ) );

#ifdef DEBUG
	viewchild = slist_map( parent_view->managed,
		(SListMapFn) view_viewchild_test_child_model, child );
	assert( !viewchild );
#endif /*DEBUG*/

	(void) view_viewchild_new( parent_view, child ); 
}

/* Called for model child_remove signal ... stop watching that child. child
 * may have been finalized already.
 */
static void
view_model_child_remove( Model *parent, Model *child, View *parent_view )
{
	ViewChild *viewchild;

#ifdef DEBUG
	printf( "view_model_child_remove: parent %s \"%s\"\n", 
		OBJECT_CLASS_NAME( parent ), NN( parent->name ) );
#endif /*DEBUG*/

	assert( IS_MODEL( parent ) );

	viewchild = slist_map( parent_view->managed,
		(SListMapFn) view_viewchild_test_child_model, child );

	assert( viewchild );

	(void) view_viewchild_destroy( viewchild ); 
}

static void *
view_real_link_sub( Model *child_model, View *parent_view )
{
	(void) view_viewchild_new( parent_view, child_model ); 

	return( NULL );
}

/* Link to model and to enclosing view. 
 */
static void 
view_real_link( View *view, Model *model, View *parent_view )
{
	assert( view != NULL );
	assert( IS_VIEW( view ) && IS_MODEL( model ) );
	assert( !view->model );

#ifdef DEBUG
	printf( "view_real_link: linking %s to model %s \"%s\"\n", 
		OBJECT_CLASS_NAME( view ), 
		OBJECT_CLASS_NAME( model ), NN( model->name ) );
#endif /*DEBUG*/

	view->model = model;
	if( parent_view )
		view_child_add( parent_view, view );

	view->changed_sid = 
		gtk_signal_connect( GTK_OBJECT( model ), "changed", 
			GTK_SIGNAL_FUNC( view_model_changed ), view );
	view->scrollto_sid = 
		gtk_signal_connect( GTK_OBJECT( model ), "scrollto", 
			GTK_SIGNAL_FUNC( view_model_scrollto ), view );
	view->reset_sid = 
		gtk_signal_connect( GTK_OBJECT( model ), "reset", 
			GTK_SIGNAL_FUNC( view_model_reset ), view );
	view->destroy_sid = 
		gtk_signal_connect( GTK_OBJECT( model ), "destroy", 
			GTK_SIGNAL_FUNC( view_model_destroy ), view );
	view->child_add_sid = 
		gtk_signal_connect( GTK_OBJECT( model ), "child_add", 
			GTK_SIGNAL_FUNC( view_model_child_add ), view );
	view->child_remove_sid = 
		gtk_signal_connect( GTK_OBJECT( model ), "child_remove", 
			GTK_SIGNAL_FUNC( view_model_child_remove ), view );

	model_map( model,
		(model_map_fn) view_real_link_sub, view, NULL );

	gtk_widget_show( GTK_WIDGET( view ) );
}

static void
view_real_child_add( View *parent, View *child )
{
	ViewChild *viewchild;

	assert( IS_VIEW( parent ) && IS_VIEW( child ) );
	assert( child->parent == NULL );

#ifdef DEBUG
	printf( "view_real_child_add: parent %s, child %s\n", 
		OBJECT_CLASS_NAME( parent ), OBJECT_CLASS_NAME( child ) );
#endif /*DEBUG*/

	viewchild = slist_map( parent->managed,
		(SListMapFn) view_viewchild_test_child_model, child->model );

	assert( viewchild );
	assert( viewchild->child_view == NULL );

	child->parent = parent;
	viewchild->child_view = child;
}

static void
view_real_child_remove( View *parent, View *child )
{
	ViewChild *viewchild;

	assert( child->parent == parent );

#ifdef DEBUG
	printf( "view_real_child_remove: parent %s, child %s\n", 
		OBJECT_CLASS_NAME( parent ), OBJECT_CLASS_NAME( child ) );
#endif /*DEBUG*/

	viewchild = slist_map( parent->managed,
		(SListMapFn) view_viewchild_test_child_model, child->model );

	if( viewchild ) {
		assert( viewchild->child_view == child );

		viewchild->child_view = NULL;
	}

	child->parent = NULL;
}

static void 
view_real_reset( View *view )
{
	view_resettable_unregister( view );
}

static void *
view_real_scan( View *view )
{
	Model *model = view->model;
	Heapmodel *heapmodel;

	view_scannable_unregister( view );

	/* If we've changed something in this model, mark it for recomp.
	 */
	if( model && IS_HEAPMODEL( model ) && 
		(heapmodel = HEAPMODEL( model ))->modified && 
		heapmodel->row ) {
		Expr *expr = heapmodel->row->expr;

		if( expr )
			(void) expr_dirty( expr, link_serial_new() );
	}

	return( NULL );
}

static void
view_real_refresh( View *view )
{
}

static void
view_class_init( ViewClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass*) klass;

	parent_class = gtk_type_class( GTK_TYPE_VBOX );

	object_class->destroy = view_destroy;
	object_class->finalize = view_finalize;

	/* Create signals.
	 */

	/* Init default methods.
	 */
	klass->link = view_real_link;
	klass->child_add = view_real_child_add;
	klass->child_remove = view_real_child_remove;

	klass->reset = view_real_reset;
	klass->scan = view_real_scan;
	klass->refresh = view_real_refresh;
	klass->event = NULL;
	klass->scrollto = NULL;

	/* Static init.
	 */
	view_dirty = queue_new();
}

static void
view_init( View *view )
{
	/* Init our instance fields.
	 */
	view->model = NULL;
	view->changed_sid = 0;
	view->scrollto_sid = 0;
	view->reset_sid = 0;
	view->destroy_sid = 0;
	view->child_add_sid = 0;
	view->child_remove_sid = 0;

	view->parent = NULL;

	view->dirty = FALSE;
	view->scannable = FALSE;
	view->resettable = FALSE;

	/* All new views will need refreshing.
	 */
	view_refresh_queue( view );
}

GtkType
view_get_type( void )
{
	static GtkType view_type = 0;

	if( !view_type ) {
		static const GtkTypeInfo view_info = {
			"View",
			sizeof( View ),
			sizeof( ViewClass ),
			(GtkClassInitFunc) view_class_init,
			(GtkObjectInitFunc) view_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		view_type = gtk_type_unique( GTK_TYPE_VBOX, &view_info );
	}

	return( view_type );
}

void
view_base_init( void )
{
	/* Make an empty view to init us.
	 */
	view_base = gtk_type_new( TYPE_VIEW );
}

/* Trigger the reset method for a view.
 */
void *
view_reset( View *view )
{
	ViewClass *klass = VIEW_CLASS( GTK_OBJECT( view )->klass );

	if( klass->reset )
		klass->reset( view );

	return( NULL );
}

/* Trigger the scan method for a view.
 */
void *
view_scan( View *view )
{
	ViewClass *klass = VIEW_CLASS( GTK_OBJECT( view )->klass );

	if( klass->scan )
		return( klass->scan( view ) );

	return( NULL );
}

/* Trigger the refresh method for a view.
 */
void *
view_refresh( View *view )
{
	ViewClass *klass = VIEW_CLASS( GTK_OBJECT( view )->klass );

	if( klass->refresh ) 
		klass->refresh( view );

	return( NULL );
}

/* Trigger the event method for a view.
 */
gboolean
view_event( View *view, GdkEvent *ev )
{
	ViewClass *klass = VIEW_CLASS( GTK_OBJECT( view )->klass );

	if( klass->event )
		return( klass->event( view, ev ) );

	return( FALSE );
}

/* Trigger the scrollto method for a view.
 */
void *
view_scrollto( View *view )
{
	ViewClass *klass = VIEW_CLASS( GTK_OBJECT( view )->klass );

	if( klass->scrollto )
		klass->scrollto( view );

	return( NULL );
}

static void *
view_map_sub( ViewChild *viewchild, view_map_fn fn, void *a, void *b )
{
	if( viewchild->child_view )
		return( fn( viewchild->child_view, a, b ) );
	
	return( NULL );
}

/* Map over a view's children.
 */
void *
view_map( View *view, view_map_fn fn, void *a, void *b )
{
	return( slist_map3( view->managed, 
		(SListMap3Fn) view_map_sub, (void *) fn, a, b ) );
}

/* Apply a function to view, and to all it's children.
 */
void *
view_map_all( View *view, view_map_fn fn, void *a )
{
	View *result;

	if( (result = fn( view, a, NULL )) )
		return( result );

	return( view_map( view, 
		(view_map_fn) view_map_all, (void *) fn, a ) );
}

void
view_save_as_cb( GtkWidget *menu, GtkWidget *host, View *view )
{
	Model *model = VIEW( view )->model;

	if( IS_FILEMODEL( model ) ) {
		iWindow *iwnd = IWINDOW( view_get_toplevel( view ) );

		filemodel_inter_saveas( iwnd, FILEMODEL( model ) );
	}
}

void
view_save_cb( GtkWidget *menu, GtkWidget *host, View *view )
{
	Model *model = VIEW( view )->model;

	if( IS_FILEMODEL( model ) ) {
		iWindow *iwnd = IWINDOW( view_get_toplevel( view ) );

		filemodel_inter_save( iwnd, FILEMODEL( model ) );
	}
}

void
view_close_cb( GtkWidget *menu, GtkWidget *host, View *view )
{
	Model *model = VIEW( view )->model;

	if( IS_FILEMODEL( model ) ) {
		iWindow *iwnd = IWINDOW( view_get_toplevel( view ) );

		filemodel_inter_savenclose( iwnd, FILEMODEL( model ) );
	}
}

/* Callback for "activate" on a view.
 */
void
view_activate_cb( View *view )
{
	view_scannable_register( view );
	symbol_recalculate_all();
}

/* Callback for "changed" on a view.
 */
void 
view_changed_cb( View *view )
{
	/* Make sure it's on the scannable list.
	 */
	view_scannable_register( view );
}

void
view_not_implemented_cb( GtkWidget *menu, GtkWidget *host, View *view )
{
	ierrors( "Not implemented" );
	box_alert( view_get_toplevel( view ) );
}

GtkWidget *
view_get_toplevel( View *view )
{
	while( IS_VIEW( view ) && view->parent )
		view = view->parent;

	return( gtk_widget_get_toplevel( GTK_WIDGET( view ) ) );
}

Columnview *
view_get_columnview( View *child )
{
	View *view;

	for( view = child; view && !IS_COLUMNVIEW( view ); view = view->parent )
		;

	if( !view )
		return( NULL );

	return( COLUMNVIEW( view ) );
}

/* A view has changed size ... rethink the enclosing column geo. Helps table
 * to work good.
 */
void *
view_resize( View *view )
{
	Columnview *cview = view_get_columnview( view );

	if( cview )
		gtk_widget_queue_resize( GTK_WIDGET( cview ) );

	return( NULL );
}
