/* Some basic util functions.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

/* Prettyprint xml save files.

 	FIXME ... slight mem leak, and save files are much larger ... but
	leave it on for convenience

 */
#define DEBUG_SAVEFILE

#include "ip.h"

/* Thing we point to helpful error messages. Replace with something more
 * sensible.
 */
static char error_buf[MAX_STRSIZE];	/* Store here */
const char *error_string = &error_buf[0];

/* Useful: Error message and quit. Emergencies only ... we don't tidy up
 * properly.
 */
/*VARARGS1*/
void
error( const char *fmt, ... )
{
	va_list args;

	fprintf( stderr, IP_NAME ": " );

        va_start( args, fmt );
        (void) vfprintf( stderr, fmt, args );
        va_end( args );

	fprintf( stderr, "\n" );

#ifndef NDEBUG
	/* Make a coredump.
	 */
	abort();
#endif /*NDEBUG*/

	exit( 1 );
}

/* Set this to block error messages. Useful if we've found an error, we want
 * to clean up, but we don't want any of the clean-up code to touch the error
 * buffer.
 */
static int error_level = 0;

void
error_block( void )
{
	error_level++;
}

void
error_unblock( void )
{
	assert( error_level );

	error_level--;
}

/* Set error_string.
 */
void
ierrors( const char *fmt, ... )
{
	if( !error_level ) {
		va_list ap;
		char str[MAX_STRSIZE];
		BufInfo buf;

		buf_init_static( &buf, str, MAX_STRSIZE );
		va_start( ap, fmt );
		(void) buf_vappendf( &buf, fmt, ap );
		va_end( ap );

		strcpy( error_buf, buf_all( &buf ) );

#ifdef DEBUG
		printf( "ierrors: %s\n", buf_all( &buf ) );
#endif /*DEBUG*/
	}
}

/* As above, but append any VIPS error messages too.
 */
void
verrors( const char *fmt, ... )
{	
	if( !error_level ) {
		va_list ap;
		BufInfo buf;

		buf_init_static( &buf, error_buf, MAX_STRSIZE );

		va_start( ap, fmt );
		(void) buf_vappendf( &buf, fmt, ap );
		va_end( ap );

		(void) buf_appendf( &buf, "\nvips error: %s", 
			im_errorstring() );
		im_clear_error_string();

		strcpy( error_buf, buf_all( &buf ) );
	}
}

/* Set an xml property printf() style.
 */
gboolean
set_prop( xmlNode *xnode, const char *name, const char *fmt, ... )
{
	va_list ap;
	char value[MAX_STRSIZE];

        va_start( ap, fmt );
	(void) im_vsnprintf( value, MAX_STRSIZE, fmt, ap );
        va_end( ap );

	if( !xmlSetProp( xnode, name, value ) ) {
		ierrors( "unable to set property \"%s\" to value \"%s\"",
			name, value );
		return( FALSE );
	}

	return( TRUE );
}

/* Set an xml property from an optionally NULL string.
 */
gboolean
set_sprop( xmlNode *xnode, const char *name, const char *value )
{
	if( value && !set_prop( xnode, name, "%s", value ) )
		return( FALSE );

	return( TRUE );
}

/* Save a list of strings. For name=="fred" and n strings in list, save as
 * "fredn" == n, "fred0" == list[0], etc.
 */
gboolean
set_slprop( xmlNode *xnode, const char *name, GSList *labels )
{
	if( labels ) {
		char buf[256];
		int i;

		(void) im_snprintf( buf, 256, "%sn", name );
		if( !set_prop( xnode, buf, "%d", g_slist_length( labels ) ) )
			return( FALSE );

		for( i = 0; labels; i++, labels = labels->next ) {
			const char *label = (const char *) labels->data;

			(void) im_snprintf( buf, 256, "%s%d", name, i );
			if( !set_sprop( xnode, buf, label ) )
				return( FALSE );
		}
	}

	return( TRUE );
}

/* Save an array of double. For name=="fred" and n doubles in array, save as
 * "fredn" == n, "fred0" == array[0], etc.
 */
gboolean
set_dlprop( xmlNode *xnode, const char *name, double *values, int n )
{
	char buf[256];
	int i;

	(void) im_snprintf( buf, 256, "%sn", name );
	if( !set_prop( xnode, buf, "%d", n ) )
		return( FALSE );

	for( i = 0; i < n; i++ ) {
		(void) im_snprintf( buf, 256, "%s%d", name, i );
		if( !set_prop( xnode, buf, "%g", values[i] ) )
			return( FALSE );
	}

	return( TRUE );
}

gboolean
get_sprop( xmlNode *xnode, const char *name, char *buf, int sz )
{
	char *value = xmlGetProp( xnode, name );

	if( !value ) 
		return( FALSE );

	im_strncpy( buf, value, sz );
	FREEF( xmlFree, value );

	return( TRUE );
}

gboolean
get_spropb( xmlNode *xnode, const char *name, BufInfo *buf )
{
	char *value = xmlGetProp( xnode, name );

	if( !value ) 
		return( FALSE );

	buf_appends( buf, value );
	FREEF( xmlFree, value );

	return( TRUE );
}

gboolean
get_iprop( xmlNode *xnode, const char *name, int *out )
{
	char buf[256];

	if( !get_sprop( xnode, name, buf, 256 ) ) 
		return( FALSE );

	*out = atoi( buf );

	return( TRUE );
}

gboolean
get_dprop( xmlNode *xnode, const char *name, double *out )
{
	char buf[256];

	if( !get_sprop( xnode, name, buf, 256 ) ) 
		return( FALSE );

	*out = atof( buf );

	return( TRUE );
}

gboolean
get_bprop( xmlNode *xnode, const char *name, gboolean *out )
{
	char buf[256];

	if( !get_sprop( xnode, name, buf, 256 ) ) 
		return( FALSE );

	*out = strcasecmp( buf, "true" ) == 0;

	return( TRUE );
}

/* Load a list of strings. For name=="fred", look for "fredn" == number of
 * strings to load, "fred0" == list[0], etc.
 */
gboolean
get_slprop( xmlNode *xnode, const char *name, GSList **out )
{
	char buf[256];
	int n, i;

	(void) im_snprintf( buf, 256, "%sn", name );
	if( !get_iprop( xnode, buf, &n ) )
		return( FALSE );

	for( i = 0; i < n; i++ ) {
		(void) im_snprintf( buf, 256, "%s%d", name, i );
		if( !get_sprop( xnode, buf, buf, 256 ) )
			return( FALSE );

		*out = g_slist_append( *out, g_strdup( buf ) );
	}

	return( TRUE );
}

/* Load an array of double. For name=="fred", look for "fredn" == length of
 * array, "fred0" == array[0], etc.
 */
gboolean
get_dlprop( xmlNode *xnode, const char *name, double **out )
{
	char buf[256];
	int n, i;

	(void) im_snprintf( buf, 256, "%sn", name );
	if( !get_iprop( xnode, buf, &n ) )
		return( FALSE );

	if( !(*out = IM_ARRAY( NULL, n, double )) )
		return( FALSE );

	for( i = 0; i < n; i++ ) {
		(void) im_snprintf( buf, 256, "%s%d", name, i );
		if( !get_dprop( xnode, buf, *out + i ) )
			return( FALSE );
	}

	return( TRUE );
}

/* Find the first child node with a name.
 */
xmlNode *
get_node( xmlNode *base, const char *name )
{
	xmlNode *i;

	for( i = base->children; i; i = i->next )
		if( strcmp( i->name, name ) == 0 )
			return( i );

	return( NULL );
}

static void
prettify_tree_sub( xmlNode *xnode, int indent )
{
	xmlNode *txt;
	xmlNode *next;

	for(;;) {
		next = xnode->next;

		/* According to memprof, this leaks :-( If you cut it out into
		 * a separate prog though, it's OK

		 	FIXME ... how odd

		 */
		txt = xmlNewText( "\n" );
		xmlAddPrevSibling( xnode, txt );
		txt = xmlNewText( spc( indent ) );
		xmlAddPrevSibling( xnode, txt );

		if( xnode->children )
			prettify_tree_sub( xnode->children, indent + 2 );

		if( !next )
			break;
		
		xnode = next;
	}

	txt = xmlNewText( spc( indent - 2 ) );
	xmlAddNextSibling( xnode, txt );
	txt = xmlNewText( "\n" );
	xmlAddNextSibling( xnode, txt );
}

/* Walk an XML document, adding extra blank text elements so that it's easier
 * to read. Don't call me twice!
 */
void
prettify_tree( xmlDoc *xdoc )
{
#ifdef DEBUG_SAVEFILE
	xmlNode *xnode = xmlDocGetRootElement( xdoc );

	prettify_tree_sub( xnode, 0 );
#endif /*DEBUG_SAVEFILE*/
}

static int rect_n_rects = 0;

/* Allocate and free rects.
 */
Rect *
rect_dup( Rect *init )
{
	Rect *new_rect;

	if( !(new_rect = IM_NEW( NULL, Rect )) )
		return( NULL );

	*new_rect = *init;

	rect_n_rects += 1;

	return( new_rect );
}

void *
rect_free( Rect *rect )
{
	im_free( rect );
	rect_n_rects -= 1;

	return( NULL );
}

/* Test two lists for eqality.
 */
gboolean
slist_equal( GSList *l1, GSList *l2 )
{
	while( l1 && l2 ) {
		if( l1->data != l2->data )
			return( FALSE );

		l1 = l1->next;
		l2 = l2->next;
	}

	if( l1 || l2 )
		return( FALSE );
	
	return( TRUE );
}

/* Map over an slist.
 */
void *
slist_map( GSList *list, SListMapFn fn, gpointer user_data )
{
	while( list ) {
		GSList *next = list->next;
		void *t = fn( list->data, user_data );

		if( t )
			return( t );
		list = next;
	}

	return( NULL );
}

/* Map a 2-ary function over an slist.
 */
void *
slist_map2( GSList *list, SListMap2Fn fn, gpointer a, gpointer b )
{
	while( list ) {
		GSList *next = list->next;
		void *t = fn( list->data, a, b );

		if( t )
			return( t );
		list = next;
	}

	return( NULL );
}

/* Map a 3-ary function over an slist.
 */
void *
slist_map3( GSList *list, SListMap3Fn fn, gpointer a, gpointer b, gpointer c )
{
	while( list ) {
		GSList *next = list->next;
		void *t = fn( list->data, a, b, c );

		if( t )
			return( t );
		list = next;
	}

	return( NULL );
}

/* Map a 4-ary function over an slist.
 */
void *
slist_map4( GSList *list, SListMap4Fn fn, 
	gpointer a, gpointer b, gpointer c, gpointer d )
{
	while( list ) {
		GSList *next = list->next;
		void *t = fn( list->data, a, b, c, d );

		if( t )
			return( t );
		list = next;
	}

	return( NULL );
}

/* Map a 5-ary function over an slist.
 */
void *
slist_map5( GSList *list, SListMap5Fn fn, 
	gpointer a, gpointer b, gpointer c, gpointer d, gpointer e )
{
	while( list ) {
		GSList *next = list->next;
		void *t = fn( list->data, a, b, c, d, e );

		if( t )
			return( t );
		list = next;
	}

	return( NULL );
}

/* Map backwards.
 */
void *
slist_map_rev( GSList *list, SListMapFn fn, gpointer a )
{
        void *res, *data;

        if( !list )
                return( NULL );

        data = list->data;
        if( list->next )
                if( (res = slist_map_rev( list->next, fn, a )) )
                        return( res );

        if( (res = fn( data, a )) )
                return( res );

        return( NULL );
}

/* Map 2ary backwards.
 */
void *
slist_map2_rev( GSList *list, SListMap2Fn fn, gpointer a, gpointer b )
{
        void *res, *data;

        if( !list )
                return( NULL );

        data = list->data;
        if( list->next )
                if( (res = slist_map2_rev( list->next, fn, a, b )) )
                        return( res );

        if( (res = fn( data, a, b )) )
                return( res );

        return( NULL );
}

/* Map 3ary backwards.
 */
void *
slist_map3_rev( GSList *list, SListMap3Fn fn, void *a, void *b, void *c )
{
        void *res, *data;

        if( !list )
                return( NULL );

        data = list->data;
        if( list->next )
                if( (res = slist_map3_rev( list->next, fn, a, b, c )) )
                        return( res );

        if( (res = fn( data, a, b, c )) )
                return( res );

        return( NULL );
}

void *
map_equal( void *a, void *b )
{
	if( a == b )
		return( a );

	return( NULL );
}

void *
slist_fold( GSList *list, void *start, SListFoldFn fn, void *a )
{
        void *c;
        GSList *ths, *next;

        for( c = start, ths = list; ths; ths = next ) {
                next = ths->next;

                if( !(c = fn( ths->data, c, a )) )
			return( NULL );
        }

        return( c );
}

void *
slist_fold2( GSList *list, void *start, SListFold2Fn fn, void *a, void *b )
{
        void *c;
        GSList *ths, *next;

        for( c = start, ths = list; ths; ths = next ) {
                next = ths->next;

                if( !(c = fn( ths->data, c, a, b )) )
			return( NULL );
        }

        return( c );
}

static void
slist_free_all_cb( gpointer thing, gpointer dummy )
{
	g_free( thing );
}

/* Free a g_slist of things which need g_free()ing.
 */
void
slist_free_all( GSList *list )
{
	g_slist_foreach( list, slist_free_all_cb, NULL );
	g_slist_free( list );
}

/* Remove all occurences of an item from a list.
 */
GSList *
slist_remove_all( GSList *list, gpointer data )
{
	GSList *tmp;
	GSList *prev;

	prev = NULL;
	tmp = list;

	while( tmp ) {
		if( tmp->data == data ) {
			GSList *next = tmp->next;

			if( prev )
				prev->next = next;
			if( list == tmp )
				list = next;

			tmp->next = NULL;
			g_slist_free( tmp );
			tmp = next;
		}
		else {
			prev = tmp;
			tmp = tmp->next;
		}
	}

	return( list );
}

Queue *
queue_new( void )
{
	Queue *q = IM_NEW( NULL, Queue );

	q->list = NULL;
	q->tail = NULL;

	return( q );
}

void *
queue_head( Queue *q )
{
	void *data;

	assert( q );

	if( !q->list )	
		return( NULL );

	data = q->list->data;

	q->list = g_slist_remove( q->list, data );
	if( !q->list )
		q->tail = NULL;

	return( data );
}

void
queue_add( Queue *q, void *data )
{
	if( !q->tail ) {
		assert( !q->list );

		q->list = q->tail = g_slist_append( q->list, data );
	}
	else {
		assert( q->list );

		(void) g_slist_append( q->tail, data );
		q->tail = q->tail->next;
	}
}

/* Not very fast! But used infrequently.
 */
void 
queue_remove( Queue *q, void *data )
{
	GSList *ele;

	assert( q && q->list && q->tail );

	ele = g_slist_find( q->list, data );
	assert( ele );

	q->list = g_slist_remove( q->list, data );

	if( ele == q->tail ) 
		q->tail = g_slist_last( q->list );
}

void
buf_rewind( BufInfo *buf )
{
	buf->i = 0;
	buf->lasti = 0;
	buf->full = FALSE;

	if( buf->base )
		strcpy( buf->base, "" );
}

/* Power on init.
 */
void
buf_init( BufInfo *buf )
{
	buf->base = NULL;
	buf->mx = 0;
	buf->dynamic = FALSE;
	buf_rewind( buf );
}

/* Reset to power on state ... only needed for dynamic bufs.
 */
void
buf_destroy( BufInfo *buf )
{
	if( buf->dynamic ) {
		FREE( buf->base );
	}

	buf_init( buf );
}

/* Set to a static string.
 */
void
buf_set_static( BufInfo *buf, char *base, int mx )
{
	assert( mx >= 4 );

	buf_destroy( buf );

	buf->base = base;
	buf->mx = mx;
	buf->dynamic = FALSE;
	buf_rewind( buf );
}

void
buf_init_static( BufInfo *buf, char *base, int mx )
{
	buf_init( buf );
	buf_set_static( buf, base, mx );
}

/* Set to a dynamic string.
 */
void
buf_set_dynamic( BufInfo *buf, int mx )
{
	assert( mx >= 4 );

	if( buf->mx == mx && buf->dynamic ) 
		/* No change?
		 */
		buf_rewind( buf );
	else {
		buf_destroy( buf );

		if( !(buf->base = IM_ARRAY( NULL, mx, char )) )
			/* No error return, so just block writes.
			 */
			buf->full = TRUE;
		else {
			buf->mx = mx;
			buf->dynamic = TRUE;
			buf_rewind( buf );
		}
	}
}

void
buf_init_dynamic( BufInfo *buf, int mx )
{
	buf_init( buf );
	buf_set_dynamic( buf, mx );
}

/* Append at most sz chars from string to buf. sz < 0 means unlimited.
 * Error on overflow.
 */
gboolean
buf_appendns( BufInfo *buf, const char *str, int sz )
{
	int len;
	int n;
	int avail;
	int cpy;

	if( buf->full )
		return( FALSE );

	/* Amount we want to copy.
	 */
	len = strlen( str );
	if( sz >= 0 )
		n = IM_MIN( sz, len );
	else
		n = len;

	/* Space available.
	 */
	avail = buf->mx - buf->i - 4;

	/* Amount we actually copy.
	 */
	cpy = IM_MIN( n, avail );

	strncpy( buf->base + buf->i, str, cpy );
	buf->i += cpy;

	if( buf->i >= buf->mx - 4 ) {
		buf->full = TRUE;
		strcpy( buf->base + buf->mx - 4, "..." );
		buf->i = buf->mx - 1;
		return( FALSE );
	}

	return( TRUE );
}

/* Append a string to a buf. Error on overflow.
 */
gboolean
buf_appends( BufInfo *buf, const char *str )
{
	return( buf_appendns( buf, str, -1 ) );
}

/* Append the first line of a string to buf.
 */
gboolean
buf_appendline( BufInfo *buf, const char *str )
{
	char *p;
	int sz;

	if( !str )
		return( TRUE );

	/* Find next '\n' or '\0'.
	 */
	if( (p = strchr( str, '\n' )) )
		sz = p - str;
	else
		sz = strlen( str );

	return( buf_appendns( buf, str, sz ) );
}

/* Append a character to a buf. Error on overflow.
 */
gboolean
buf_appendc( BufInfo *buf, char ch )
{
	return( buf_appendns( buf, &ch, 1 ) );
}

/* Swap the rightmost occurence of old for new.
 */
gboolean
buf_change( BufInfo *buf, const char *old, const char *new )
{
	int olen = strlen( old );
	int nlen = strlen( new );
	int i;

	if( buf->full )
		return( FALSE );
	if( buf->i - olen + nlen > buf->mx - 4 ) {
		buf->full = TRUE;
		return( FALSE );
	}

	/* Find pos of old.
	 */
	for( i = buf->i - olen; i > 0; i-- )
		if( isprefix( old, buf->base + i ) )
			break;
	assert( i >= 0 );

	/* Move tail of buffer to make right-size space for new.
	 */
	memmove( buf->base + i + nlen, buf->base + i + olen,
		buf->i - i - olen );

	/* Copy new in.
	 */
	memcpy( buf->base + i, new, nlen );
	buf->i = i + nlen + (buf->i - i - olen);

	return( TRUE );
}

/* Remove the last character.
 */
gboolean
buf_removec( BufInfo *buf, char ch )
{
	if( buf->full )
		return( FALSE );

	if( buf->i <= 0 ) 
		return( FALSE );
	buf->i -= 1;
	assert( buf->base[buf->i] == ch );

	return( TRUE );
}

/* Append to a buf, args as printf. Error on overflow.
 */
gboolean
buf_appendf( BufInfo *buf, const char *fmt, ... )
{
	char str[MAX_STRSIZE];
	va_list ap;

        va_start( ap, fmt );
        (void) im_vsnprintf( str, MAX_STRSIZE, fmt, ap );
        va_end( ap );

	return( buf_appends( buf, str ) );
}

/* Append to a buf, args as vprintf. Error on overflow.
 */
gboolean
buf_vappendf( BufInfo *buf, const char *fmt, va_list ap )
{
	char str[MAX_STRSIZE];

        (void) im_vsnprintf( str, MAX_STRSIZE, fmt, ap );

	return( buf_appends( buf, str ) );
}

/* Append a number ... if the number is -ve, add brackets.
 */
gboolean
buf_appendd( BufInfo *buf, int d )
{
	if( d < 0 )
		return( buf_appendf( buf, " (%d)", d ) );
	else
		return( buf_appendf( buf, " %d", d ) );
}

/* Append a string, escaping C stuff.
 */
gboolean
buf_appendsc( BufInfo *buf, const char *str )
{
	char buffer[FILENAME_MAX];
	char buffer2[FILENAME_MAX];

	/* /2 to leave a bit of space.
	 */
	im_strncpy( buffer, str, FILENAME_MAX / 2 );

	/* 

		FIXME ... possible buffer overflow :-(

	 */
	strecpy( buffer2, buffer );

	return( buf_appends( buf, buffer2 ) );
}

/* Read all text from buffer.
 */
const char *
buf_all( BufInfo *buf )
{
	buf->base[buf->i] = '\0';

	return( buf->base );
}

/* Test for buffer empty.
 */
gboolean
buf_isempty( BufInfo *buf )
{
	return( buf->i == 0 );
}

/* Test for buffer full.
 */
gboolean
buf_isfull( BufInfo *buf )
{
	return( buf->full );
}

/* Buffer length ... still need to do buf_all().
 */
int
buf_len( BufInfo *buf )
{
	return( buf->i );
}

/* Test for string a ends string b. 
 */
gboolean
ispostfix( const char *a, const char *b )
{	
	int n = strlen( a );
	int m = strlen( b );

	if( m < n )
		return( FALSE );

	return( !strcmp( a, b + m - n ) );
}

/* Test for string a starts string b. 
 */
gboolean
isprefix( const char *a, const char *b )
{
	int n = strlen( a );
	int m = strlen( b );
	int i;

	if( m < n )
		return( FALSE );
	for( i = 0; i < n; i++ )
		if( a[i] != b[i] )
			return( FALSE );
	
	return( TRUE );
}

/* Test for string a starts string b ... case insensitive. 
 */
gboolean
iscaseprefix( const char *a, const char *b )
{
	int n = strlen( a );
	int m = strlen( b );
	int i;

	if( m < n )
		return( FALSE );
	for( i = 0; i < n; i++ )
		if( toupper( a[i] ) != toupper( b[i] ) )
			return( FALSE );
	
	return( TRUE );
}

/* Like strstr(), but case-insensitive.
 */
char *
strcasestr( const char *haystack, const char *needle )
{
	int i;
	int hlen = strlen( haystack );
	int nlen = strlen( needle );

	for( i = 0; i <= hlen - nlen; i++ )
		if( iscaseprefix( needle, haystack + i ) )
			return( (char *) (haystack + i) );

	return( NULL );
}

#ifndef HAVE_STRCCPY
/* Fix this! copy a string, interpreting (a few) C-language escape codes.
 */
char *
strccpy( char *output, const char *input )
{
	const char *p;
	char *q;

	for( p = input, q = output; *p; p++, q++ )
		if( p[0] == '\\' ) {
			switch( p[1] ) {
			case 'n':
				q[0] = '\n';
				break;

			case 't':
				q[0] = '\t';
				break;

			case '"':
				q[0] = '\"';
				break;

			case '\'':
				q[0] = '\'';
				break;

			case '\\':
				q[0] = '\\';
				break;

			default:
				/* Leave uninterpreted.
				 */
				q[0] = p[0];
				q[1] = p[1];
				q++;
				break;
			}

			p++;
		}
		else
			q[0] = p[0];
	q[0] = '\0';

	return( output );
}

/* Copy a string, expanding escape characters into C-language escape codes.
 */
char *
strecpy( char *output, const char *input )
{
	const char *p;
	char *q;

	for( p = input, q = output; *p; p++, q++ )
		switch( p[0] ) {
		case '\n':
			q[0] = '\\';
			q[1] = 'n';
			q++;
			break;

		case '\t':
			q[0] = '\\';
			q[1] = 't';
			q++;
			break;

		case '\"':
			q[0] = '\\';
			q[1] = '\"';
			q++;
			break;

		case '\'':
			q[0] = '\\';
			q[1] = '\'';
			q++;
			break;

		case '\\':
			q[0] = '\\';
			q[1] = '\\';
			q++;
			break;

		default:
			q[0] = p[0];
			break;
		}
	q[0] = '\0';

	return( output );
}
#endif /*HAVE_STRCCPY*/

/* Is a character in a string?
 */
static int
instr( char c, const char *spn )
{
	const char *p;

	for( p = spn; *p; p++ )
		if( *p == c )
			return( 1 );
	
	return( 0 );
}

/* Doh ... not everyone has strrspn(), define one.
 */
const char *
my_strrspn( const char *p, const char *spn )
{
	const char *p1;

	for( p1 = p + strlen( p ) - 1; p1 >= p && instr( *p1, spn ); p1-- )
		;

	p1++;
	if( strlen( p1 ) == 0 )
		return( NULL );
	else
		return( p1 );
}

/* Find the rightmost occurence of string a in string b.
 */
const char *
findrightmost( const char *a, const char *b )
{	
	int la = strlen( a );
	int lb = strlen( b );
	int i;

	if( lb < la )
		return( NULL );

	for( i = lb - la; i > 0; i-- )
		if( strncmp( a, &b[i], la ) == 0 )
			return( &b[i] );

	return( NULL );
}

/* Useful transformation: strip off a set of suffixes (eg. ".v", ".icon",
 * ".hr"), add a single new suffix (eg. ".hr.v"). 
 */
void
change_suffix( const char *name, char *out, 
	const char *new, const char **olds, int nolds )
{	
	char *p;
	int i;

	/* Copy start string.
	 */
	strcpy( out, name );

	/* Drop all matching suffixes.
	 */
	while( (p = strrchr( out, '.' )) ) {
		/* Found suffix - test against list of alternatives. Ignore
		 * case.
		 */
		for( i = 0; i < nolds; i++ )
			if( strcasecmp( p, olds[i] ) == 0 ) {
				*p = '\0';
				break;
			}

		/* Found match? If not, break from loop.
		 */
		if( *p )
			break;
	}

	/* Add new suffix.
	 */
	strcat( out, new );
}

/* Drop leading and trim trailing non-alphanum characters. NULL if nothing 
 * left. The result can be a variable name.
 */
char *
trim_nonalpha( char *text )
{
	char *p, *q;

	/* Skip any initial non-alpha characters.
	 */
	for( q = text; *q && !isalpha( (int)(*q) ); q++ )
		;

	/* Find next non-alphanumeric character. 
	 */
	for( p = q; *p && isalnum( (int)(*p) ); p++ )
		;
	*p = '\0';

	if( strlen( q ) == 0 )
		return( NULL );
	else
		return( q );
}

/* Drop leading and trim trailing whitespace characters. NULL if nothing 
 * left. 
 */
char *
trim_white( char *text )
{
	char *p, *q;

	/* Skip any initial whitespace characters.
	 */
	for( q = text; *q && isspace( (int)(*q) ); q++ )
		;

	/* Find rightmost non-space char.
	 */
	for( p = q + strlen( q ) - 1; p > q && isspace( (int)(*p) ); p-- )
		;
	p[1] = '\0';

	if( strlen( q ) == 0 )
		return( NULL );
	else
		return( q );
}

/* Get a pointer to a band element in a region.
 */
void *
get_element( REGION *ireg, int x, int y, int b )
{
	IMAGE *im = ireg->im;

	PEL *data;
	int es = IM_IMAGE_SIZEOF_ELEMENT( im );
	Rect iarea;

	/* Make sure we can read from this descriptor.
	 */
	if( im_pincheck( im ) ) {
		verrors( "unable to read from image" );
		return( NULL );
	}

	/* Ask for the area we need.
	 */
	iarea.left = x;
	iarea.top = y;
	iarea.width = 1;
	iarea.height = 1;
	if( im_prepare( ireg, &iarea ) ) {
		verrors( "unable to read pixel" );
		return( NULL );
	}

	/* Find a pointer to the start of the data.
	 */
	data = (PEL *) IM_REGION_ADDR( ireg, x, y ) + b * es;

	return( (void *) data );
}

/* Decode band formats in a friendly way.
 */
const char *bandfmt_names[] = {
	"8-bit unsigned integer",	/* IM_BANDFMT_UCHAR */ 
	"8-bit signed integer",		/* IM_BANDFMT_CHAR */
	"16-bit unsigned integer", 	/* IM_BANDFMT_USHORT */
	"16-bit signed integer", 	/* IM_BANDFMT_SHORT */
	"32-bit unsigned integer", 	/* IM_BANDFMT_UINT */
	"32-bit signed integer", 	/* IM_BANDFMT_INT */
	"32-bit float", 		/* IM_BANDFMT_FLOAT */
	"64-bit complex", 		/* IM_BANDFMT_COMPLEX */
	"64-bit float", 		/* IM_BANDFMT_DOUBLE */
	"128-bit complex" 		/* IM_BANDFMT_DPCOMPLEX */
};
const int nbandfmt_names = IM_NUMBER( bandfmt_names );

const char *
decode_bandfmt( int f )
{
	if( f > nbandfmt_names || f < 0 )
		return( "<unknown format>" );
	else
		return( bandfmt_names[f] );
}

/* Decode type names in a way consistent with the menus.
 */
const char *type_names[] = {
	"multiband",			/* IM_TYPE_MULTIBAND	0 */
	"mono",				/* IM_TYPE_B_W		1 */
	"luminance",			/* LUMINACE		2 */
	"xray",				/* XRAY			3 */
	"infrared",			/* IR			4 */
	"Yuv",				/* YUV			5 */
	"red_only",			/* RED_ONLY		6 */
	"green_only",			/* GREEN_ONLY		7 */
	"blue_only",			/* BLUE_ONLY		8 */
	"power_spectrum",		/* POWER_SPECTRUM	9 */
	"histogram",			/* IM_TYPE_HISTOGRAM	10 */
	"lookup_table",			/* LUT			11 */
	"XYZ",				/* IM_TYPE_XYZ		12 */
	"Lab",				/* IM_TYPE_LAB		13 */
	"CMC",				/* CMC			14 */
	"CMYK",				/* IM_TYPE_CMYK		15 */
	"LabQ",				/* IM_TYPE_LABQ		16 */
	"RGB",				/* IM_TYPE_RGB		17 */
	"UCS",				/* IM_TYPE_UCS		18 */
	"LCh",				/* IM_TYPE_LCH		19 */
	"<undefined>",			/* ??			20 */
	"LabS",				/* IM_TYPE_LABS		21 */
	"sRGB",				/* IM_TYPE_sRGB		22 */
	"Yxy",				/* IM_TYPE_YXY		23 */
	"Fourier",			/* IM_TYPE_FOURIER	24 */
};
const int ntype_names = IM_NUMBER( type_names );

const char *
decode_type( int t )
{
	if( t > ntype_names || t < 0 )
		return( "<unknown type>" );
	else
		return( type_names[t] );
}

/* Make an info string about a file.
 */
void
get_image_info( BufInfo *buf, const char *name )
{
	char name2[FILENAME_MAX];
	char *type;

	type = NULL;
	expand_variables( name, name2 );
	if( im_istiff( name2 ) )
		type = "TIFF image";
	else if( im_isjpeg( name2 ) )
		type = "JPEG image";
	else if( im_ispng( name2 ) )
		type = "PNG image";
	else if( im_isppm( name2 ) )
		type = "PPM/PGM/PBM image";
	else if( im_ismagick( name2 ) )
		type = "libMagick-supported image";
	else if( im_isvips( name2 ) ) 
		type = "VIPS image";

	if( type ) {
		IMAGE *im;

		if( !(im = im_open_header( name2 )) ) {
			buf_appends( buf, im_errorstring() );
			return;
		}

		buf_appendf( buf, "%s, %s, %s, %dx%d, %d bands",
			im__skip_dir( name ), type, 
			decode_bandfmt( im->BandFmt ),
			im->Xsize, im->Ysize, im->Bands );

		im_close( im );
	}
	else {
		struct stat st;

		/* Just read size.
		 */
		if( stat( name2, &st ) != -1 ) {
			buf_appendf( buf, "%s, ", im__skip_dir( name ) );
			to_size( buf, st.st_size );
		}
	}
}

/* Expand environment variables from in to out. Return true if we performed an
 * expansion, false for no variables there.
 */
static gboolean
expand_once( char *in, char *out )
{
	char *p, *q;
	gboolean have_substituted = FALSE;

	/* Scan and copy.
	 */
	for( p = in, q = out; (*q = *p) && (q - out) < FILENAME_MAX; p++, q++ )
		/* Did we just copy a '$'?
		 */
		if( *p == '$' ) {
			char vname[FILENAME_MAX];
			char *r;
			char *subst;

			/* Extract the variable name.
			 */
			p++;
			for( r = vname; 
				isalnum( (int)(*r = *p) ) && 
					(r - vname) < FILENAME_MAX;
				p++, r++ )
				;
			*r = '\0';
			p--;

			/* Look up variable.
			 */
			subst = getenv( vname );

			/* Copy variable contents.
			 */
			if( subst ) {
				for( r = subst; 
					(*q = *r) && (q - out) < FILENAME_MAX; 
					r++, q++ )
					;
			}
			q--;

			/* Remember we have performed a substitution.
			 */
			have_substituted = TRUE;
		}
		/* Or a '~' at the start of the string?
		 */
		else if( *p == '~' && p == in ) {
			char *r;
			char *subst = getenv( "HOME" );

			/* Copy variable contents.
			 */
			if( subst ) {
				for( r = subst; 
					(*q = *r) && (q - out) < FILENAME_MAX;
					r++, q++ )
					;
			}
			q--;

			/* Remember we have performed a substitution.
			 */
			have_substituted = TRUE;
		}

	return( have_substituted );
}

/* Expand all variables! Don't touch in, assume out[] is at least
 * FILENAME_MAX bytes.
 */
void
expand_variables( const char *in, char *out )
{
	char buf[FILENAME_MAX];
	char *p1 = (char *) in;		/* Discard const, but safe */
	char *p2 = out;

	/* Expand any environment variables in component.
	 */
	while( expand_once( p1, p2 ) ) 
		/* We have expanded --- swap the buffers and try
		 * again.
		 */
		if( p2 == out ) {
			p1 = out;
			p2 = buf;
		}
		else {
			p1 = buf;
			p2 = out;
		}
}

static void
swap_chars( char *buf, char from, char to )
{
	int i;

	for( i = 0; buf[i]; i++ )
		if( buf[i] == from )
			buf[i] = to;
}

/* If we use '/' seps, swap all '\' for '/' ... likewise vice versa.
 */
void
nativize_path( char *buf )
{
	if( IM_DIR_SEP == '/' )
		swap_chars( buf, '\\', '/' );
	else
		swap_chars( buf, '/', '\\' );
}

/* Call a function, building and expanding a string arg.
 */
void *
call_string( call_string_fn fn, const char *name, void *a, void *b, void *c )
{
	char buf[FILENAME_MAX];

	expand_variables( name, buf );
        nativize_path( buf );

	return( fn( buf, a, b, c ) );
}

void *
call_stringva( call_string_fn fn, 
	const char *name, va_list ap, void *a, void *b, void *c )
{
	char buf[FILENAME_MAX];

        (void) im_vsnprintf( buf, FILENAME_MAX, name, ap );

	return( call_string( fn, buf, a, b, c ) );
}

void *
call_stringf( call_string_fn fn, const char *name, ... )
{
	va_list ap;
	void *res;

        va_start( ap, name );
        res = call_stringva( fn, name, ap, NULL, NULL, NULL );
        va_end( ap );

	return( res );
}

void
putenvf( const char *name, ... )
{
	va_list ap;
	char buf[FILENAME_MAX];

        va_start( ap, name );
        (void) im_vsnprintf( buf, FILENAME_MAX, name, ap );
        va_end( ap );

	putenv( g_strdup( buf ) );
}

/* File exists? 
 */
gboolean
existsf( const char *name, ... )
{
	va_list ap;
	gboolean res;

        va_start( ap, name );
        res = (int) call_stringva( (call_string_fn) im_existsf, 
		name, ap, NULL, NULL, NULL );
        va_end( ap );

	return( res );
}

gboolean
mkdirf( const char *name, ... )
{
	va_list ap;
	gboolean res;

        va_start( ap, name );
        res = call_stringva( (call_string_fn) mkdir, 
		name, ap, (void *) 0755, NULL, NULL ) == NULL;
        va_end( ap );

	return( res );
}

/* system(), with printf() args.
 */
int
systemf( const char *fmt, ... )
{
	va_list ap;
	int res;

        va_start( ap, fmt );
        res = (int) call_stringva( (call_string_fn) system, 
		fmt, ap, NULL, NULL, NULL );
        va_end( ap );

	return( res );
}

/* popen(), with printf-style args.
 */
FILE *
popenf( const char *fmt, const char *mode, ... )
{
        va_list ap;
	FILE *res;

        va_start( ap, mode );
        res = (FILE *) call_stringva( (call_string_fn) popen, 
		fmt, ap, (char *) mode, NULL, NULL );
        va_end( ap );

        return( res );
}

/* Relative or absolute dir path? Have to expand env vars to see.
 */
gboolean
is_absolute( const char *fname )
{
	char buf[FILENAME_MAX];

	expand_variables( fname, buf );

	if( im_path_is_absolute( buf ) )
		return( TRUE );
	else
		return( FALSE );
}

/* OK filename? Ban ':' characters, they may confuse im_open(). Except on
 * winders :-(
 */
gboolean
is_valid_filename( const char *name )
{
	const char *p;

#ifndef HAVE_WINDOWS_H
	if( strchr( name, ':' ) ) {
		ierrors( "filenames may not contain ':' characters" );
		return( FALSE );
	}
#endif

	if( strlen( name ) > PATHLEN ) {
		ierrors( "filename is too long" );
		return( FALSE );
	}
	if( (p = strrchr( name, IM_DIR_SEP )) &&
		strspn( p + 1, WHITESPACE ) == strlen( p + 1 ) ) {
		ierrors( "filename contains only blank characters" );
		return( FALSE );
	}

	return( TRUE );
}

/* im_strdup(), with NULL supplied.
 */
char *im_strdupn( const char *txt ) { return( im_strdup( NULL, txt ) ); }

/* Free an iOpenFile.
 */
void
file_close( iOpenFile *of )
{
	FREEF( fclose, of->fp );
	FREE( of->fname );
	FREE( of->fname_real );
	FREE( of );
}

/* Make an iOpenFile*.
 */
static iOpenFile *
file_build( const char *fname )
{
	iOpenFile *of;

	if( !(of = IM_NEW( NULL, iOpenFile )) )
		return( NULL );

	of->fp = NULL;
	of->fname = NULL;
	of->fname_real = NULL;
	of->last_errno = 0;

	SETSTR( of->fname, fname );
	if( !of->fname ) {
		file_close( of );
		return( NULL );
	}

	return( of );
}

/* Find and open for read.
 */
iOpenFile *
file_open_read( const char *name, ... )
{
	va_list ap;
	char buf[FILENAME_MAX];
	iOpenFile *of;

        va_start( ap, name );
        (void) im_vsnprintf( buf, FILENAME_MAX, name, ap );
        va_end( ap );
        of = file_build( buf );

	if( !of )
		return( NULL );
	if( !(of->fname_real = path_find_file( PATH_SEARCH, of->fname )) ) {
		ierrors( "unable to open file \"%s\"\n%s",
			of->fname, g_strerror( errno ) );
		file_close( of );
		return( NULL );
	}

	if( !(of->fp = (FILE *) call_string( (call_string_fn) fopen, 
		of->fname_real, "r", NULL, NULL )) ) {
		ierrors( "unable to open file \"%s\"\n%s",
			of->fname_real, g_strerror( errno ) );
		file_close( of );
		return( NULL );
	}

	of->read = TRUE;

	return( of );
}

/* Open stdin for read.
 */
iOpenFile *
file_open_read_stdin()
{
	iOpenFile *of;

	if( !(of = file_build( "stdin" )) )
		return( NULL );
	SETSTR( of->fname_real, of->fname );
	if( !of->fname_real ) {
		file_close( of );
		return( NULL );
	}
	of->fp = stdin;
	of->read = TRUE;

	return( of );
}

/* Find and open for write.
 */
iOpenFile *
file_open_write( const char *name, ... )
{
	va_list ap;
	char buf[FILENAME_MAX];
	iOpenFile *of;

        va_start( ap, name );
        (void) im_vsnprintf( buf, FILENAME_MAX, name, ap );
        va_end( ap );
        of = file_build( name );

	if( !of )
		return( NULL );
	SETSTR( of->fname_real, of->fname );
	if( !of->fname_real ) {
		file_close( of );
		return( NULL );
	}
	if( !(of->fp = (FILE *) call_string( (call_string_fn) fopen,
		of->fname_real, "w", NULL, NULL )) ) {
		ierrors( "unable to write file \"%s\"\n%s",
			of->fname_real, g_strerror( errno ) );
		file_close( of );
		return( NULL );
	}

	of->read = FALSE;

	return( of );
}

/* fprintf() to a file, checking result.
 */
gboolean
file_write( iOpenFile *of, const char *fmt, ... )
{
	va_list ap;

        va_start( ap, fmt );
	if( vfprintf( of->fp, fmt, ap ) == EOF ) {
		of->last_errno = errno;
		ierrors( "unable to write file \"%s\"\n%s",
			of->fname_real, g_strerror( of->last_errno ) );
		return( FALSE );
	}
        va_end( ap );

	return( TRUE );
}

/* Save a string ... if non-NULL. Eg. 
 *	fred="boink!"
 */
gboolean
file_write_var( iOpenFile *of, const char *name, const char *value )
{
	if( value )
		return( file_write( of, " %s=\"%s\"", name, value ) );
	
	return( TRUE );
}

/* Load up a file as a string.
 */
char *
file_read( iOpenFile *of )
{
	long len;
	char *str;

	/* Find length.
	 */
	fseek( of->fp, 0L, 2 );
	len = ftell( of->fp );
	rewind( of->fp );
	if( len < 0 || len > 1024*1024 ) {
		ierrors( "bad length for file \"%s\"", of->fname_real );
		return( NULL );
	}

	/* Allocate memory and fill.
	 */
	if( !(str = im_malloc( NULL, len + 1 )) ) 
		return( NULL );
	if( (long) fread( str, sizeof( char ), (size_t) len, of->fp ) != len ) {
		of->last_errno = errno;
		FREE( str );
		ierrors( "unable to read from file \"%s\"\n%s",
			of->fname_real, g_strerror( of->last_errno ) );
		return( NULL );
	}
	str[ len ] = '\0';

	return( str );
}

/* Escape "%" characters in a string.
 */
char *
escape_percent( const char *in, char *out )
{
	const char *p;
	char *q;

	for( p = in, q = out; *p; p++, q++ )
		if( *p == '%' ) {
			q[0] = '%';
			q[1] = '%';
			q++;
		}
		else
			*q = *p;
	*q = '\0';

	return( out );
}

/* Return a string of n spaces. Buffer is zapped each time!
 */
const char *
spc( int n )
{
	int i;
	static char buf[ 200 ];

	n = IM_MIN( 190, n );

	for( i = 0; i < n; i++ )
		buf[i] = ' ';
	buf[i] = '\0';

	return( buf );
}

/* Like strtok(), but better. Give a string and a list of break characters;
 * write a '\0' into the string over the first break character and return a
 * pointer to the next non-break character. If there are no break characters,
 * then return a pointer to the end of the string. If passed a pointer to an
 * empty string or NULL, then return NULL.
 */
char *
break_token( char *str, const char *brk )
{
	char *p;

	/* Is the string empty? If yes, return NULL immediately.
	 */
	if( !str || !*str )
		return( NULL );

	/* Skip initial break characters.
	 */
	p = str + strspn( str, brk );

	/* Search for the first break character after the token.
	 */
	p += strcspn( p, brk );

	/* Is there string left?
	 */
	if( *p ) {
		/* Write in an end-of-string mark and return the start of the
		 * next token.
		 */
		*p++ = '\0';
		p += strspn( p, brk );
	}
	
	return( p );
}

/* Turn a number to a string. 0 is "A", 1 is "B", 25 is "Z", 26 is "AA", 27 is
 * "AB", etc.
 */
void
number_to_string( int n, char *buf )
{
	do {
		int rem = n % 26;

		*buf++ = (char) (rem + (int) 'A');
		n /= 26;
	} while( n > 0 );

	*buf ='\0';
}

/* Hash from a pointer into an index.
 */
unsigned int
hash_pointer( void *p )
{
	return( 0xffffff & (unsigned int) p );
}

/* Hash from a character string into an index.
 */
unsigned int
hash_name( const char *n )
{
	unsigned int h;

	for( h = 0; *n != '\0'; n += 1 ) 
		h = (h << 5) - h + *n;

	return( h ); 
}

/* Find the space remaining in a directory, in bytes. A double for >32bit
 * problem avoidance. <0 for error.
 */
#ifdef HAVE_SYS_STATVFS_H
double
find_space( const char *name )
{
	struct statvfs st;
	double sz;

	if( call_string( (call_string_fn) statvfs, 
		(gpointer) name, &st, NULL, NULL ) )
		/* Set to error value. 
		 */
		sz = -1;
	else
		sz = IM_MAX( 0, (double) st.f_frsize * st.f_bavail );

	return( sz );
}
#elif (HAVE_SYS_VFS_H || HAVE_SYS_MOUNT_H)
double
find_space( const char *name )
{
	struct statfs st;
	double sz;

	if( call_string( (call_string_fn) statfs, 
		(gpointer) name, &st, NULL, NULL ) )
		sz = -1;
	else
		sz = IM_MAX( 0, (double) st.f_bsize * st.f_bavail );

	return( sz );
}
#elif defined HAVE_WINDOWS_H
double
find_space( const char *name )
{
	ULARGE_INTEGER avail;
	ULARGE_INTEGER total;
	ULARGE_INTEGER free;
	double sz;
	char name2[FILENAME_MAX];

       expand_variables( name, name2 );
       if( name2[1] == ':' )
               name2[3] = 0;

	if( !GetDiskFreeSpaceEx( name2, &avail, &total, &free ) )
		sz = -1;
	else
		sz = IM_MAX( 0, (double) (__int64) avail.QuadPart );

	return( sz );
}
#else
double 
find_space( const char *name )
{
	return( -1 );
}
#endif /*HAVE_SYS_STATVFS_H*/

/* Turn a number of bytes into a sensible string ... eg "12", "12KB", "12MB",
 * "12GB" etc.
 */
void
to_size( BufInfo *buf, double sz )
{
	const static char *size_names[] = { "bytes", "KB", "MB", "GB", "TB" };
	int i;

	assert( sz >= 0 );

	for( i = 0; sz > 1024 && i < IM_NUMBER( size_names ); sz /= 1024, i++ )
		;

	if( i == 0 )
		/* No decimal places for bytes.
		 */
		buf_appendf( buf, "%g %s", sz, size_names[i] );
	else
		buf_appendf( buf, "%.2f %s", sz, size_names[i] );
}


/* Make a name for a temp file. Add the specified extension.
 */
gboolean
temp_name( char *name, const char *type )
{
	/* Some mkstemp() require files to actually exist before they don't
	 * reuse the filename :-( add an extra field.
	 */
	static int n = 0;

	const char *dir;
#ifdef HAVE_MKSTEMP
	int fd;
#endif /*HAVE_MKSTEMP*/
	char buf[FILENAME_MAX];

	dir = PATH_TMP;
	if( !existsf( "%s", dir ) )
		dir = IM_DIR_SEP_STR "tmp";

	im_snprintf( name, FILENAME_MAX, "%s" IM_DIR_SEP_STR 
		"untitled-" PACKAGE "-%d-XXXXXXX", 
		dir, n++ );
	expand_variables( name, buf );

#ifdef HAVE_MKSTEMP
	fd = mkstemp( buf );
	if( fd == -1 ) {
		error( "unable to make temporary file \"%s\"\n%s",
			buf, g_strerror( errno ) );
		return( FALSE );
	}
	close( fd );
#else /*HAVE_MKSTEMP*/
#ifdef HAVE_MKTEMP
	mktemp( buf );
#else /*HAVE_MKTEMP*/
#error "no temporary file name maker found"
#endif /*HAVE_MKTEMP*/
#endif /*HAVE_MKSTEMP*/
	unlink( buf );

	im_snprintf( name, FILENAME_MAX, "%s.%s", buf, type );

	return( TRUE );
}

/* Max/min of an area.
 */
int
findmaxmin( IMAGE *in, 
	int left, int top, int width, int height, double *min, double *max )
{
	DOUBLEMASK *msk;
	IMAGE *t1;

	if( !(t1 = im_open( "temp", "p" )) )
		return( -1 );
	if( im_extract_area( in, t1, left, top, width, height ) ||
		!(msk = im_stats( t1 )) )
		return( -1 );
	im_close( t1 );
	
	*min = msk->coeff[0];
	*max = msk->coeff[1];

	im_free_dmask( msk );

	return( 0 );
}

gboolean
char_to_bool( char *str, void *out )
{	
	gboolean *t = (gboolean *) out;
	
	if( strcasecmp( "TRUE", str ) == 0 )
		*t = TRUE;
	else
		*t = FALSE;
	
	return( TRUE );
}

char *
bool_to_char( gboolean b )
{	
	if( b )
		return( "true" );
	else
		return( "false" );
}

/* Increment a name ... strip any trailing numbers, add one, put numbers back.
 * Start at 1 if no number there. buf should be at least namelen chars.
 */
void
increment_name( char *buf )
{
	char *p;
	int n;

        if( (p = (char *) my_strrspn( buf, NUMERIC )) ) 
                n = atoi( p );
	else {
		p = buf + strlen( buf );
		n = 0;
	}

        im_snprintf( p, NAMELEN - (p - buf), "%d", n + 1 ); 
}

/* Extract the first line of a string in to buf, extract no more than len
 * chars.
 */
int
extract_first_line( char *buf, char *str, int len )
{
        char *p;
        int n;

        /* Find next '\n' or '\0'.
         */
        if( (p = strchr( str, '\n' )) )
                n = p - str;
        else
                n = strlen( str );
        n = IM_MIN( len - 1, n );
        
        /* Copy those characters and make sure we have a '\0'.
         */
        strncpy( buf, str, n );
        buf[n] = '\0';

        return( n );
}

/* Make a valid ip name from a filename. Skip leading stuff, from trailing
 * stuff, filter remaining chars for allowed in names set.
 */
void
name_from_filename( const char *in, char *out )
{
	const char *p, *q;
	int len;

	/* Make up a name ... skip leading path prefix, and any non-alpha.
	 */
	p = im__skip_dir( in ); 
	for( q = p; *q && (isalnum( *q ) || *q == '\'' || *q == '_'); q++ )
		;

	len = IM_MIN( NAMELEN - 1, q - p );

	if( len == 0 )
		strcpy( out, "untitled" );
	else {
		im_strncpy( out, p, len + 1 );
		out[len] = '\0';
	}
}

/* Do any leak checking we can.
 */
void
util_check_all_destroyed( void )
{
	if( rect_n_rects )
		printf( "rect_n_rects == %d\n", rect_n_rects );
}

