/* abstract base class for things which form the model half of a model/view
 * pair
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

#include "ip.h"

/* Stuff from bison ... needed as we call the lexer directly to rewrite
 * expressions.
 */
#include "parse.h"

/* Our signals. 
 */
enum {
	CHANGED,	/* Model has changed ... all views should update */
	SCROLLTO,	/* Views should try to make themselves visible */
	RESET,		/* Reset edit mode in views */
	CHILD_ADD,	/* Model is about to gain a child */
	CHILD_REMOVE,	/* Model is about to loose a child */
	LAST_SIGNAL
};

static GtkObjectClass *parent_class = NULL;

static guint model_signals[LAST_SIGNAL] = { 0 };

/* Base model ... built at startup.
 */
Model *model_base = NULL;

/* All the model classes which can be built from XML.
 */
static GSList *model_registered_loadable = NULL;

/* The loadstate the lexer gets its rename stuff from.
 */
ModelLoadState *model_loadstate = NULL;

/* Rename list functions.
 */
static void *
model_loadstate_rename_destroy( ModelRename *rename )
{
	FREE( rename->old_name );
	FREE( rename->new_name );
	FREE( rename );

	return( NULL );
}

ModelRename *
model_loadstate_rename_new( ModelLoadState *state, 
	const char *old_name, const char *new_name )
{
	ModelRename *rename;

	if( !(rename = IM_NEW( NULL, ModelRename )) )
		return( NULL );
	rename->old_name = im_strdup( NULL, old_name );
	rename->new_name = im_strdup( NULL, new_name );
	if( !rename->old_name || !rename->new_name ) {
		model_loadstate_rename_destroy( rename );
		return( NULL );
	}

	state->renames = g_slist_prepend( state->renames, rename );

	return( rename );
}

void
model_loadstate_destroy( ModelLoadState *state )
{
	/* We are probably registered as the xml error handler ... unregister!
	 */
	xmlSetGenericErrorFunc( NULL, NULL );

	FREE( state->filename );
	FREEF( xmlFreeDoc, state->xdoc );
	slist_map( state->renames, 
		(SListMapFn) model_loadstate_rename_destroy, NULL );
	g_slist_free( state->renames );
	FREE( state );
}

static void
model_loadstate_error( ModelLoadState *state, const char *fmt, ... )
{
	va_list ap;

	va_start( ap, fmt );
	(void) buf_vappendf( &state->error_log, fmt, ap );
	va_end( ap );
}

ModelLoadState *
model_loadstate_new( const char *filename )
{
	ModelLoadState *state;

	if( !(state = IM_NEW( NULL, ModelLoadState )) )
		return( NULL );
	state->renames = NULL;
	state->xdoc = NULL;
	if( !(state->filename = im_strdup( NULL, filename )) ) {
		model_loadstate_destroy( state );
		return( NULL );
	}
	buf_init_static( &state->error_log, 
		state->error_log_buffer, MAX_STRSIZE );

	xmlSetGenericErrorFunc( state, 
		(xmlGenericErrorFunc) model_loadstate_error );
	if( !(state->xdoc = (xmlDoc *) call_string( 
		(call_string_fn) xmlParseFile, 
		state->filename, NULL, NULL, NULL )) ) {
		im_errormsg( "xml_load:\n%s", buf_all( &state->error_log ) );
		verrors( "xmlParseFile( \"%s\" ) failed\n", state->filename );
		model_loadstate_destroy( state );
		return( NULL );
	}

	return( state );
}

/* If old_name is on the global rewrite list, rewrite it! Called from the
 * lexer.
 */
char *
model_loadstate_rewrite_name( char *name )
{
	ModelLoadState *state = model_loadstate;
	GSList *i;

	if( !state || !state->renames )
		return( NULL );

	for( i = state->renames; i; i = i->next ) {
		ModelRename *rename = (ModelRename *) (i->data);

		if( strcmp( name, rename->old_name ) == 0 )
			return( rename->new_name );
	}

	return( NULL );
}

/* Need to set this! From parse.y.
 */
extern jmp_buf parse_error_point;

/* Use the lexer to rewrite an expression, swapping all symbols on the rewrite 
 * list.
 */
void
model_loadstate_rewrite( ModelLoadState *state, char *old_rhs, char *new_rhs )
{
	int yychar;
	extern int yylex( void );

	model_loadstate = state;
	attach_input_string( old_rhs );
	if( setjmp( parse_error_point ) ) {
		/* Here for yyerror in lex. Just ignore errors --- the parser
		 * will spot them later anyway.
		 */
		model_loadstate = NULL;
		return; 
	}

	/* Lex and rewrite.
	 */
	while( (yychar = yylex()) > 0 ) 
		/* Fee any stuff the lexer might have allocated. 
		 */
		switch( yychar ) {
		case TK_CONST:
			tree_const_destroy( &yylval.yy_const );
			break;

		case TK_IDENT:
			FREE( yylval.yy_name );
			break;

		default:
			break;
		}

	model_loadstate = NULL;

	/* Take copy of lexed and rewritten stuff.
	 */
	im_strncpy( new_rhs, buf_all( &lex_text ), MAX_STRSIZE );
}

void *
model_changed( Model *model )
{
	assert( IS_MODEL( model ) );

	if( !GTK_OBJECT_DESTROYED( model ) ) 
		gtk_signal_emit( GTK_OBJECT( model ), model_signals[CHANGED] );

	return( NULL );
}

/* Register a model subclass as loadable ... what we allow when we load an
 * XML node's children.
 */
void 
model_register_loadable( ModelClass *model_class )
{
	model_registered_loadable = g_slist_prepend( model_registered_loadable, 
		model_class );
}

/* Trigger the view_new() method.
 */
View *
model_view_new( Model *model )
{
	ModelClass *klass = MODEL_CLASS( GTK_OBJECT( model )->klass );

	if( klass->view_new )
		return( klass->view_new() );

	return( NULL );
}

void
model_scrollto( Model *model )
{
	assert( IS_MODEL( model ) );

	gtk_signal_emit( GTK_OBJECT( model ), model_signals[SCROLLTO] );
}

void
model_reset( Model *model )
{
	assert( IS_MODEL( model ) );

	gtk_signal_emit( GTK_OBJECT( model ), model_signals[RESET] );
}

/* Trigger the edit method for a model.
 */
void *
model_edit( GtkWidget *parent, Model *model )
{
	ModelClass *klass = MODEL_CLASS( GTK_OBJECT( model )->klass );

	if( klass->edit )
		klass->edit( parent, model );

	return( NULL );
}

/* Trigger the info method for a model.
 */
void *
model_info( Model *model, BufInfo *buf )
{
	ModelClass *klass = MODEL_CLASS( GTK_OBJECT( model )->klass );

	if( klass->info )
		klass->info( model, buf );

	return( NULL );
}

/* Trigger the save method.
 */
void *
model_save( Model *model, xmlNode *xnode )
{
	ModelClass *model_class = MODEL_CLASS( GTK_OBJECT( model )->klass );

	if( model_save_test( model ) ) {
		if( model_class->save && !model_class->save( model, xnode ) )
			return( model );
	}

	return( NULL );
}

/* Trigger the save_test method.
 */
gboolean
model_save_test( Model *model )
{
	ModelClass *model_class = MODEL_CLASS( GTK_OBJECT( model )->klass );

	if( model_class->save_test )
		return( model_class->save_test( model ) );

	return( TRUE );
}

void *
model_save_text( Model *model, iOpenFile *of )
{
	ModelClass *model_class = MODEL_CLASS( GTK_OBJECT( model )->klass );

	if( model_class->save_text && !model_class->save_text( model, of ) )
		return( model );

	return( NULL );
}

/* Trigger the load methods for a model.
 */
void *
model_load( Model *model, 
	ModelLoadState *state, Model *parent, xmlNode *xnode )
{
	ModelClass *model_class = MODEL_CLASS( GTK_OBJECT( model )->klass );

	if( model_class->load ) {
		if( !model_class->load( model, state, parent, xnode ) )
			return( model );
	}
	else 
		ierrors( "no _load() method for class \"%s\"", 
			CLASS_NAME( model_class ) );

	return( NULL );
}

void *
model_load_text( Model *model, Model *parent, iOpenFile *of )
{
	ModelClass *model_class = MODEL_CLASS( GTK_OBJECT( model )->klass );

	if( model_class->load_text ) {
		if( !model_class->load_text( model, parent, of ) )
			return( model );
	}
	else 
		ierrors( "no _load_text() method for class \"%s\"", 
			CLASS_NAME( model_class ) );

	return( NULL );
}

/* Trigger the empty method for a model.
 */
void *
model_empty( Model *model )
{
	ModelClass *klass = MODEL_CLASS( GTK_OBJECT( model )->klass );

	if( klass->empty )
		klass->empty( model );

	return( NULL );
}

/* Number of children.
 */
int
model_get_n_children( Model *model )
{
	return( g_slist_length( model->children ) );
}

/* Test the name field ... handy with map.
 */
void *
model_test_name( Model *model, const char *name )
{
	if( strcmp( model->name, name ) == 0 )
		return( model );
	
	return( NULL );
}

/* Map over a model's children.
 */
void *
model_map( Model *model, model_map_fn fn, void *a, void *b )
{
	return( slist_map2( model->children, (SListMap2Fn) fn, a, b ) );
}

void *
model_map3( Model *model, model_map3_fn fn, void *a, void *b, void *c )
{
	return( slist_map3( model->children, (SListMap3Fn) fn, a, b, c ) );
}

void *
model_map4( Model *model, model_map4_fn fn, void *a, void *b, void *c, void *d )
{
	return( slist_map4( model->children, (SListMap4Fn) fn, a, b, c, d ) );
}

void *
model_map5( Model *model, 
	model_map5_fn fn, void *a, void *b, void *c, void *d, void *e )
{
	return( slist_map5( model->children, 
		(SListMap5Fn) fn, a, b, c, d, e ) );
}

/* Map in reverse order.
 */
void *
model_map_rev( Model *model, model_map_fn fn, void *a, void *b )
{
	return( slist_map2_rev( model->children, (SListMap2Fn) fn, a, b ) );
}

/* Apply a function to a tree of models, bottom up.
 */
void *
model_map_all( Model *model, model_map_fn fn, void *a )
{
	Model *result;

	if( (result = model_map( model, 
		(model_map_fn) model_map_all, (void *) fn, a )) )
		return( result );

	return( fn( model, a, NULL ) );
}

void *
model_map2_all( Model *model, model_map_fn fn, void *a, void *b )
{
	Model *result;

	if( (result = model_map3( model, 
		(model_map3_fn) model_map2_all, (void *) fn, a, b )) )
		return( result );

	return( fn( model, a, b ) );
}

void *
model_map3_all( Model *model, model_map3_fn fn, void *a, void *b, void *c )
{
	Model *result;

	if( (result = model_map4( model, 
		(model_map4_fn) model_map3_all, (void *) fn, a, b, c )) )
		return( result );

	return( fn( model, a, b, c ) );
}

void *
model_map4_all( Model *model, 
	model_map4_fn fn, void *a, void *b, void *c, void *d )
{
	Model *result;

	if( (result = model_map5( model, 
		(model_map5_fn) model_map4_all, (void *) fn, a, b, c, d )) )
		return( result );

	return( fn( model, a, b, c, d ) );
}

/* Apply a function to the children of a model.
 */
void *
model_map_all_intrans( Model *model, model_map_fn fn, void *a )
{
	return( model_map( model, 
		(model_map_fn) model_map_all, (void *) fn, a ) );
}

static gint
model_pos_compare( Model *a, Model *b )
{
        return( a->pos - b->pos );
}

void
model_pos_sort( Model *model )
{
        model->children = g_slist_sort( model->children, 
		(GCompareFunc) model_pos_compare );
	model_changed( model );
}

static void *
model_pos_last_sub( Model *model, int *max )
{
	if( model->pos > *max )
		*max = model->pos;
	
	return( NULL );
}

int
model_pos_last( Model *model )
{
	int max = -1;

	model_map( model,
		(model_map_fn) model_pos_last_sub, &max, NULL );

	return( max );
}

static void *
model_pos_renumber_sub( Model *model, int *n )
{
	model->pos = *n;
	model_changed( model );
	*n += 1;

	return( NULL );
}

void
model_pos_renumber( Model *model )
{
	int n = 0;

	model_map( model,
		(model_map_fn) model_pos_renumber_sub, &n, NULL );
	model_changed( model );
}

static gint
model_name_compare( Model *a, Model *b )
{
        return( strcasecmp( a->name, b->name ) );
}

void
model_name_sort( Model *model )
{
        model->children = g_slist_sort( model->children, 
		(GCompareFunc) model_name_compare );
	model_pos_renumber( model );
	model_changed( model );
}

/* Add a child.
 */
void
model_child_add( Model *parent, Model *child, int pos )
{
	assert( IS_MODEL( parent ) );
	assert( IS_MODEL( child ) );

	gtk_signal_emit( GTK_OBJECT( parent ), 
		model_signals[CHILD_ADD], child, pos );
}

/* Add a child before another child. after == NULL means append.
 */
void
model_child_add_before( Model *parent, Model *child, Model *before )
{
	int pos;

	assert( !before || IS_MODEL( before ) );
	assert( !before || before->parent == parent );

	pos = g_slist_index( parent->children, before );
	model_child_add( parent, child, pos );
}

/* pos == 0 ... move to start
 * pos == -1 ... move to end
 * pos == n ... move before sibling at position n
 */
void
model_child_move( Model *child, int pos )
{
	Model *parent = child->parent;

	parent->children = g_slist_remove( parent->children, child );

        if( pos >= 0 )
                parent->children = g_slist_insert( parent->children,
                        child, pos );
        else
                parent->children = g_slist_append( parent->children, child );

        model_pos_renumber( parent );
}

/* Remove a child.
 */
void 
model_child_remove( Model *parent, Model *child )
{
	gtk_signal_emit( GTK_OBJECT( parent ), 
		model_signals[CHILD_REMOVE], child );
}

void
model_set( Model *model, const char *name, const char *caption )
{
	if( name && name != model->name ) {
		SETSTR( model->name, name );
		model_changed( model );
	}
	if( caption && caption != model->caption ) {
		SETSTR( model->caption, caption );
		model_changed( model );
	}
}

/* Destroy all children.
 */
void 
model_destroy_children( Model *parent )
{
	if( parent->children ) {
		while( parent->children ) {
			Model *child = MODEL( parent->children->data );

			gtk_object_destroy( GTK_OBJECT( child ) );
		}
	}
}

static void
model_destroy( GtkObject *object )
{
	Model *model;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_MODEL( object ) );

	model = MODEL( object );

#ifdef DEBUG
	printf( "model_destroy: %s \"%s\"\n", 
		OBJECT_CLASS_NAME( object ), NN( model->name ) );
#endif /*DEBUG*/
	
	/* Explicitly destroy children ... we don't use the gtk container
	 * model, so we have to do this ourselves.
	 */
	model_destroy_children( model );

	/* Unlink from parent.
	 */
	if( model->parent ) 
		model_child_remove( model->parent, model );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

/* Separate out finalize so we junk model names as late as possible ... handy
 * for debug.
 */
static void
model_finalize( GtkObject *object )
{
	Model *model;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_MODEL( object ) );

	model = MODEL( object );

#ifdef DEBUG
	printf( "model_finalize: \"%s\", name \"%s\"\n", 
		OBJECT_CLASS_NAME( object ), NN( model->name ) );
#endif /*DEBUG*/

	FREE( model->name );
	FREE( model->caption );

	GTK_OBJECT_CLASS( parent_class )->finalize( object );
}

static void
model_real_changed( Model *model )
{
}

static void
model_real_info( Model *model, BufInfo *buf )
{
	buf_appendf( buf, "Object :: \"%s\"\n", OBJECT_CLASS_NAME( model ) );
	buf_appendf( buf, "name = \"%s\"\n", NN( model->name ) );
	buf_appendf( buf, "caption = \"%s\"\n", NN( model->caption ) );
	buf_appendf( buf, "pos = \"%d\"\n", model->pos );
	buf_appendf( buf, "display = \"%s\"\n", 
		bool_to_char( model->display ) );
}

#ifdef DEBUG
static void *
model_name_print( Model *model )
{
	printf( "%s \"%s\" (0x%x), pos %d; ", OBJECT_CLASS_NAME( model ),
		NN( model->name ), (unsigned int) model, 
		model->pos );

	return( NULL );
}
#endif /*DEBUG*/

static void
model_real_child_add( Model *parent, Model *child, int pos )
{
        ModelClass *cklass = MODEL_CLASS( GTK_OBJECT( child )->klass );

        assert( IS_MODEL( parent ) && IS_MODEL( child ) );
        assert( child->parent == NULL );

        if( pos >= 0 )
                parent->children = g_slist_insert( parent->children,
                        child, pos );
        else
                parent->children = g_slist_append( parent->children, child );
        child->parent = parent;
        child->pos = pos;

        model_pos_renumber( parent );

        gtk_object_ref( GTK_OBJECT( child ) );
        gtk_object_sink( GTK_OBJECT( child ) );

        /* We've made the link ... trigger the parent_add() on the child.
         */
        cklass->parent_add( child, parent );

#ifdef DEBUG
        printf( "model_real_child_add: %s \"%s\" (0x%x): now has children: ",
                OBJECT_CLASS_NAME( parent ),
                NN( parent->name ), (unsigned int) parent );
        model_map( parent, (model_map_fn) model_name_print, NULL, NULL );
        printf( "\n" );
#endif /*DEBUG*/
}

static void 
model_real_child_remove( Model *parent, Model *child )
{
	ModelClass *cklass = MODEL_CLASS( GTK_OBJECT( child )->klass );

	assert( IS_MODEL( parent ) && IS_MODEL( child ) );
	assert( child->parent == parent );

#ifdef DEBUG
	printf( "model_real_child_remove: parent \"%s\", name \"%s\"; "
		"child \"%s\", name \"%s\"\n", 
		OBJECT_CLASS_NAME( parent ), NN( parent->name ),
		OBJECT_CLASS_NAME( child ), NN( child->name ) );
#endif /*DEBUG*/

	/* We're about to break the link ... trigger the parent_remove() on 
	 * the child.
	 */
	cklass->parent_remove( child, parent );

	parent->children = g_slist_remove( parent->children, child );
	child->parent = NULL;

	gtk_object_unref( GTK_OBJECT( child ) );

	model_changed( parent );
}

static void
model_real_parent_add( Model *child, Model *parent )
{
#ifdef DEBUG
	printf( "model_real_parent_add: child \"%s\", name \"%s\"; " 
		"parent \"%s\", name \"%s\"\n", 
		OBJECT_CLASS_NAME( parent ), NN( parent->name ),
		OBJECT_CLASS_NAME( child ), NN( child->name ) );
#endif /*DEBUG*/
}

static void
model_real_parent_remove( Model *child, Model *parent )
{
#ifdef DEBUG
	printf( "model_real_parent_remove: child \"%s\", name \"%s\"; "
		"parent \"%s\", name \"%s\"\n", 
		OBJECT_CLASS_NAME( parent ), NN( parent->name ),
		OBJECT_CLASS_NAME( child ), NN( child->name ) );
#endif /*DEBUG*/
}

static xmlNode *
model_real_save( Model *model, xmlNode *xnode )
{
	const char *tname = OBJECT_CLASS_NAME( model );
	xmlNode *xthis;

	if( !(xthis = xmlNewChild( xnode, NULL, tname, NULL )) ) {
		ierrors( "model_save: xmlNewChild() failed" );
		return( NULL );
	}

	if( !set_sprop( xthis, "name", model->name ) ||
		!set_sprop( xthis, "caption", model->caption ) )
		return( NULL );

	if( model_map( model, 
		(model_map_fn) model_save, xthis, NULL ) )
		return( NULL );

	return( xthis );
}

static void *
model_new_xml_sub( ModelClass *model_class, 
	ModelLoadState *state, Model *parent, xmlNode *xnode )
{
	GtkType type = GTK_OBJECT_CLASS( model_class )->type;
	char *tname = gtk_type_name( type );

	if( strcmp( xnode->name, tname ) == 0 ) {
		Model *model = MODEL( gtk_type_new( type ) );

		if( model_load( model, state, parent, xnode ) ) {
			FREEO( model );
			return( model_class );
		}

		return( NULL );
	}

	return( NULL );
}

gboolean
model_new_xml( ModelLoadState *state, Model *parent, xmlNode *xnode )
{
	/* 

		FIXME ... slow! some sort of hash? time this at some point

	 */
	if( slist_map3( model_registered_loadable,
		(SListMap3Fn) model_new_xml_sub, state, parent, xnode ) )
		return( FALSE );

	return( TRUE );
}

static gboolean 
model_real_load( Model *model,
	ModelLoadState *state, Model *parent, xmlNode *xnode )
{
	const char *tname = OBJECT_CLASS_NAME( model );
	xmlNode *i;

	char name[NAMELEN];
	char caption[NAMELEN];

	/* Should just be a sanity check.
	 */
	if( strcmp( xnode->name, tname ) != 0 ) {
		ierrors( "XML load: can't load node of type \"%s\" into "
			"object of type \"%s\"", xnode->name, tname );
		return( FALSE );
	}

	if( get_sprop( xnode, "name", name, NAMELEN ) )
		model_set( MODEL( model ), name, NULL );
	if( get_sprop( xnode, "caption", caption, NAMELEN ) )
		model_set( MODEL( model ), NULL, caption );

	if( !MODEL( model )->parent )
		model_child_add( parent, MODEL( model ), -1 );

	for( i = xnode->children; i; i = i->next ) 
		if( !model_new_xml( state, MODEL( model ), i ) )
			return( FALSE );

#ifdef DEBUG
	printf( "model_real_load: finished loading %s (name = %s)\n", 
		tname, NN( model->name ) );
#endif /*DEBUG*/

	return( TRUE );
}

static void
model_class_init( ModelClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass*) klass;

	parent_class = gtk_type_class( GTK_TYPE_OBJECT );

	object_class->destroy = model_destroy;
	object_class->finalize = model_finalize;

	klass->view_new = NULL;
	klass->changed = model_real_changed;
	klass->edit = NULL;
	klass->scrollto = NULL;
	klass->reset = NULL;
	klass->info = model_real_info;
	klass->child_add = model_real_child_add;
	klass->child_remove = model_real_child_remove;
	klass->parent_add = model_real_parent_add;
	klass->parent_remove = model_real_parent_remove;
	klass->save = model_real_save;
	klass->save_test = NULL;
	klass->save_text = NULL;
	klass->load = model_real_load;
	klass->load_text = NULL;
	klass->empty = NULL;

	/* Create signals.
	 */
	model_signals[CHANGED] = gtk_signal_new( "changed",
		GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET( ModelClass, changed ),
		gtk_marshal_NONE__NONE,
		GTK_TYPE_NONE, 0 );

	model_signals[SCROLLTO] = gtk_signal_new( "scrollto",
		GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET( ModelClass, scrollto ),
		gtk_marshal_NONE__NONE,
		GTK_TYPE_NONE, 0 );

	model_signals[RESET] = gtk_signal_new( "reset",
		GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET( ModelClass, reset ),
		gtk_marshal_NONE__NONE,
		GTK_TYPE_NONE, 0 );

	model_signals[CHILD_ADD] = gtk_signal_new( "child_add",
		GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET( ModelClass, child_add ),
		gtk_marshal_NONE__POINTER_INT,
		GTK_TYPE_NONE, 2, TYPE_MODEL, GTK_TYPE_INT );

	model_signals[CHILD_REMOVE] = gtk_signal_new( "child_remove",
		GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET( ModelClass, child_remove ),
		gtk_marshal_NONE__POINTER,
		GTK_TYPE_NONE, 1, TYPE_MODEL );

	gtk_object_class_add_signals( object_class, 
		model_signals, LAST_SIGNAL );
}

static void
model_init( Model *model )
{
	/* Init our instance fields.
	 */
	model->children = NULL;
	model->parent = NULL;
	model->pos = -1;

	model->name = NULL;
	model->caption = NULL;

	model->display = TRUE;
}

GtkType
model_get_type( void )
{
	static GtkType model_type = 0;

	if( !model_type ) {
		static const GtkTypeInfo info = {
			"Model",
			sizeof( Model ),
			sizeof( ModelClass ),
			(GtkClassInitFunc) model_class_init,
			(GtkObjectInitFunc) model_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		model_type = gtk_type_unique( GTK_TYPE_OBJECT, &info );
	}

	return( model_type );
}

void
model_base_init( void )
{
	model_base = gtk_type_new( TYPE_MODEL );

	/* We have to init some of our other classes to get them registered 
	 * with the class loader.
	 */
	(void) gtk_type_class( TYPE_WORKSPACE );
	(void) gtk_type_class( TYPE_COLUMN );
	(void) gtk_type_class( TYPE_SUBCOLUMN );
	(void) gtk_type_class( TYPE_ROW );
	(void) gtk_type_class( TYPE_RHS );
	(void) gtk_type_class( TYPE_ITEXT );
	(void) gtk_type_class( TYPE_TOGGLE );
	(void) gtk_type_class( TYPE_SLIDER );
	(void) gtk_type_class( TYPE_OPTION );
	(void) gtk_type_class( TYPE_MATRIX );
	(void) gtk_type_class( TYPE_IIMAGE );
	(void) gtk_type_class( TYPE_IREGION );
	(void) gtk_type_class( TYPE_IARROW );
	(void) gtk_type_class( TYPE_COLOUR );
}

typedef struct {
	iDialog *idlg;		/* The yesno we run */
	Model *model;		/* The model we watch */
	guint dsid;		/* sid for the destroy */
} ModelCheckDestroy;

/* OK to destroy.
 */
static void
model_check_destroy_sub( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	ModelCheckDestroy *mcd = (ModelCheckDestroy *) client;

	mcd->idlg = NULL;
	gtk_object_destroy( GTK_OBJECT( mcd->model ) );

	symbol_recalculate_all();

	nfn( sys, IWINDOW_TRUE );
}

/* Our dialog is done.
 */
static void 
model_check_destroy_finished( void *client, iWindowResult result ) 
{ 
	ModelCheckDestroy *mcd = (ModelCheckDestroy *) client;

	FREESID( mcd->dsid, mcd->model );
	FREE( mcd );
}

/* The model we are watching has been killed, maybe by us.
 */
static void
model_check_destroy_destroy( Model *model, ModelCheckDestroy *mcd )
{
	mcd->dsid = 0;
	mcd->model = NULL;

	if( mcd->idlg ) {
		iWindow *iwnd = IWINDOW( mcd->idlg );

		mcd->idlg = NULL;
		iwindow_kill( iwnd );
	}
}

void
model_check_destroy( GtkWidget *parent, Model *model )
{
	BufInfo buf;
	char str[30];
	const char *name;

	ModelCheckDestroy *mcd = IM_NEW( NULL, ModelCheckDestroy );

	mcd->idlg = NULL;
	mcd->model = model;
	mcd->dsid = 0;

	if( IS_SYMBOL( model ) ) {
		buf_init_static( &buf, str, 30 );
		symbol_qualified_name( SYMBOL( model ), &buf );
		name = buf_all( &buf );
	}
	else
		name = model->name;

	mcd->idlg = box_yesno( parent,
		model_check_destroy_sub, iwindow_true_cb, mcd, 
		model_check_destroy_finished, mcd,
		"Remove",
		"are you sure you want to remove %s \"%s\"?", 
		OBJECT_CLASS_NAME( model ), name );

	mcd->dsid = gtk_signal_connect( GTK_OBJECT( model ), "destroy",
		GTK_SIGNAL_FUNC( model_check_destroy_destroy ), mcd );
}

/* Set the ->display var ... return TRUE if we change something.
 */
gboolean
model_set_display( Model *model, gboolean display )
{
	gboolean changed = FALSE;

	/* Do as two ifs to in case we're not using 0/1 for bool.
	 */
	if( model && display && !model->display ) {
		model->display = TRUE;
		changed = TRUE;
	} 
	else if( model && !display && model->display ) {
		model->display = FALSE;
		changed = TRUE;
	}

	if( changed )
		model_changed( model );

	return( changed );
}

/* Useful for model_map_all() ... trigger all heapmodel_clear_edited()
 * methods.
 */
void *
model_clear_edited( Model *model )
{
	void *result;

	if( IS_HEAPMODEL( model ) && 
		(result = heapmodel_clear_edited( HEAPMODEL( model ) )) )
		return( result );

	return( NULL );
}
