/* a text item in a workspace
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/* 
#define DEBUG
 */

#include "ip.h"

static HeapmodelClass *parent_class = NULL;

/* Max chars we print. Limit to MAX_LINELENGTH to stop out-of-control stuff.
 */
#define LINELENGTH ( \
	IM_MIN( MAX_LINELENGTH, watch_int_get( "CALC_LINELENGTH", 80 ) ) \
)

static void
itext_destroy( GtkObject *object )
{
	iText *itext;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_ITEXT( object ) );

	itext = ITEXT( object );

#ifdef DEBUG
	printf( "itext_destroy\n" );
#endif /*DEBUG*/

	/* My instance destroy stuff.
	 */
	FREE( itext->formula );
	FREE( itext->formula_default );
	buf_destroy( &itext->value );
	buf_destroy( &itext->decompile );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

/* Fwd ref this.
 */
static gboolean itext_add_element( BufInfo *buf, 
	PElement *base, gboolean top );

/* Sub-fn of below, callback for list print. Eval and print the item into 
 * the buffer, separating with commas as required. 
 */
static void *
itext_add_list( PElement *base, BufInfo *buf, gboolean *first )
{
	Reduce *rc = reduce_context;

	if( *first )
		*first = FALSE;
	else 
		buf_appends( buf, ", " );

	/* Reduce the head, and print.
	 */
	if( !reduce_pelement( rc, reduce_spine, base ) ) 
		return( base );
	if( !itext_add_element( buf, base, FALSE ) )
		return( base );

	/* Buffer full? Abort list print.
	 */
	if( buf->full )
		return( base );

	return( NULL );
}

/* Sub-fn of below, callback for string print. Print the chars into the 
 * buffer.
 */
static void *
itext_add_string( PElement *base, BufInfo *buf )
{
	Reduce *rc = reduce_context;

	/* Reduce the head, and add the char.
	 */
	if( !reduce_pelement( rc, reduce_spine, base ) ) 
		return( base );
	if( PEISCHAR( base ) )
		buf_appendf( buf, "%c", PEGETCHAR( base ) );
	else {
		/* Help! Fall back to ordinary item print.
		 */
		buf_appends( buf, ", " );
		if( !itext_add_element( buf, base, FALSE ) )
			return( base );
	}

	/* Buffer full? Abort string print.
	 */
	if( buf->full )
		return( base );

	return( NULL );
}

/* Try to decompile.
 */
static gboolean
itext_decompile_element( BufInfo *buf, PElement *base, gboolean top )
{
	gboolean result;

	/* Set the value label for a tally entry.
	 */
	if( PEISNOVAL( base ) ) 
		buf_appends( buf, "No value" );
	else if( PEISREAL( base ) ) 
		buf_appendf( buf, "%g", PEGETREAL( base ) );
	else if( PEISBOOL( base ) ) 
		buf_appends( buf, bool_to_char( PEGETBOOL( base ) ) );
	else if( PEISCHAR( base ) ) 
		buf_appendf( buf, "'%c'", (int) PEGETCHAR( base ) );
	else if( PEISCOMPLEX( base ) ) 
		buf_appendf( buf, "(%g,%g)", 
			PEGETREALPART( base ), PEGETIMAGPART( base ) );
	else if( PEISELIST( base ) ) {
		buf_appends( buf, "[ ]" );
	}
	else if( !heap_isstring( base, &result ) ) 
		/* Eval error.
		 */
		return( FALSE );
	else if( result ) {
		buf_appends( buf, "\"" );
		if( heap_map_list( base,
			(heap_map_list_fn) itext_add_string, buf, NULL ) )
			return( FALSE );
		buf_appends( buf, "\"" );
	}
	else if( PEISLIST( base ) ) {
		gboolean first = TRUE;

		buf_appends( buf, "[" );
		if( heap_map_list( base,
			(heap_map_list_fn) itext_add_list, buf, &first ) )
			return( FALSE );
		buf_appends( buf, "]" );
	}
	else if( PEISIMAGE( base ) ) {
		Imageinfo *ii = PEGETII( base );

		if( !top )
			buf_appends( buf, "(" );

		if( ii && ii->name ) 
			buf_appendf( buf, "vips_image \"%s\"", ii->name );
		else 
			buf_appendf( buf, "vips_image \"unknown\"" );

		if( !top )
			buf_appends( buf, ")" );
	}
	else if( PEISCLASS( base ) ) {
		Compile *compile = PEGETCLASSCOMPILE( base );
		PElement params;
		int i;

		if( !top )
			buf_appends( buf, "(" );

		symbol_qualified_name( compile->sym, buf );

		/* Skip over the secrets, then decompile all the args.
		 */
		PEGETCLASSSECRET( &params, base );
		for( i = 0; i < compile->nsecret; i++ ) {
			HeapNode *hn = PEGETVAL( &params );

			PEPOINTRIGHT( hn, &params );
		}

		for( i = 0; i < compile->nparam; i++ ) {
			HeapNode *hn = PEGETVAL( &params );
			HeapNode *sv = GETLEFT( hn );
			PElement value;

			PEPOINTRIGHT( sv, &value );
			buf_appends( buf, " " );
			if( !itext_decompile_element( buf, &value, FALSE ) )
				return( FALSE );

			PEPOINTRIGHT( hn, &params );
		}

		if( !top )
			buf_appends( buf, ")" );

	}
	else if( PEISSYMREF( base ) ) 
		buf_appends( buf, MODEL( PEGETSYMREF( base ) )->name );
	else if( PEISTAG( base ) ) 
		buf_appends( buf, PEGETTAG( base ) );
	else
		buf_appends( buf, "<function>" );

	return( TRUE );
}

/* Little wrapper ... used for formatting error messages, etc. FALSE for eval
 * error.
 */
gboolean 
itext_decompile( BufInfo *buf, PElement *root )
{
	/* Evaluate and print off values.
	 */
	if( !reduce_pelement( reduce_context, reduce_spine, root ) ) 
		return( FALSE );

	if( !itext_decompile_element( buf, root, TRUE ) && !buf->full ) 
		/* Tally eval failed, and buffer is not full ... must
		 * have been an eval error.
		 */
		return( FALSE );

	return( TRUE );
}

/* Same, but everror on eval fail.
 */
void
itext_decompile_ev( Reduce *rc, BufInfo *buf, PElement *root )
{
	if( !itext_decompile( buf, root ) )
		everror( rc, "%s", error_string );
}

/* Print function for computed values. 
 */
static gboolean
itext_add_element( BufInfo *buf, PElement *base, gboolean top )
{
	gboolean result;

	/* Set the value label for a tally entry.
	 */
	if( PEISNOVAL( base ) ) 
		buf_appends( buf, "No value" );
	else if( PEISREAL( base ) ) 
		buf_appendf( buf, "%.7g", PEGETREAL( base ) );
	else if( PEISBOOL( base ) ) 
		buf_appends( buf, bool_to_char( PEGETBOOL( base ) ) );
	else if( PEISCHAR( base ) ) 
		buf_appendf( buf, "'%c'", (int) PEGETCHAR( base ) );
	else if( PEISCOMPLEX( base ) ) 
		buf_appendf( buf, "(%.12g,%.12g)", 
			PEGETREALPART( base ), PEGETIMAGPART( base ) );
	else if( PEISELIST( base ) ) {
		buf_appends( buf, "[ ]" );
	}
	else if( !heap_isstring( base, &result ) ) 
		/* Eval error.
		 */
		return( FALSE );
	else if( result ) {
		/* Only generate quotes for non-top-level string objects.
		 */
		if( !top ) 
			buf_appends( buf, "\"" );

		/* Print string contents.
		 */
		if( heap_map_list( base,
			(heap_map_list_fn) itext_add_string, buf, NULL ) )
			return( FALSE );

		if( !top ) 
			buf_appends( buf, "\"" );
	}
	else if( PEISLIST( base ) ) {
		gboolean first = TRUE;

		buf_appends( buf, "[" );
		if( heap_map_list( base,
			(heap_map_list_fn) itext_add_list, buf, &first ) )
			return( FALSE );
		buf_appends( buf, "]" );
	}
	else if( PEISIMAGE( base ) ) {
		Imageinfo *ii = PEGETII( base );
		IMAGE *im = imageinfo_get( FALSE, ii );

		if( !im ) {
			buf_appends( buf, "No image value" );
			return( TRUE );
		}

		/* Coded? Special warning.
		 */
		if( im->Coding != IM_CODING_NONE ) 
			buf_appendf( buf, "%s, ", 
				im_Coding2char( im->Coding ) );

		/* Main stuff.
		 */
		buf_appendf( buf, "%dx%d %s pixels, %d band%s",
			im->Xsize, im->Ysize,
			decode_bandfmt( im->BandFmt ),
			im->Bands, im->Bands==1?"":"s" );
	}
	else if( PEISCLASS( base ) ) {
		Compile *compile = PEGETCLASSCOMPILE( base );
		PElement params;
		int i;

		/* Name.
		 */
		symbol_qualified_name( compile->sym, buf );

		/* Skip over the secrets, then decompile all the args.
		 */
		PEGETCLASSSECRET( &params, base );
		for( i = 0; i < compile->nsecret; i++ ) {
			HeapNode *hn = PEGETVAL( &params );

			PEPOINTRIGHT( hn, &params );
		}

		for( i = 0; i < compile->nparam; i++ ) {
			HeapNode *hn = PEGETVAL( &params );
			HeapNode *sv = GETLEFT( hn );
			PElement value;

			PEPOINTRIGHT( sv, &value );
			buf_appends( buf, " " );
			if( !itext_decompile_element( buf, &value, FALSE ) )
				return( FALSE );

			PEPOINTRIGHT( hn, &params );
		}
	}
	else if( PEISSYMREF( base ) ) {
		Symbol *sym = PEGETSYMREF( base );

		if( is_scope( sym ) )
			buf_appendf( buf, "<scope \"%s\">", 
				MODEL( sym )->name );
		else
			buf_appendf( buf, "<reference to symbol \"%s\">", 
				MODEL( sym )->name );
	}
	else
		buf_appends( buf, "<function>" );

	return( TRUE );
}

/* Make a decompiled expr from a value.
 */
static gboolean
itext_make_decompiled_string( Expr *expr, BufInfo *buf )
{
	/* Old error on this expression?
	 */
	if( expr->err ) {
		expr_error_get( expr );
		return( FALSE );
	}

	/* Evaluate and print off values.
	 */
	if( !itext_decompile( buf, &expr->root ) )
		return( FALSE );

	return( TRUE );
}

/* Make a value string from an expression.
 */
static gboolean
itext_make_value_string( Expr *expr, BufInfo *buf )
{
	Reduce *rc = reduce_context;

	/* Old error on this expression?
	 */
	if( expr->err ) {
		expr_error_get( expr );
		return( FALSE );
	}

	/* Evaluate and print off values.
	 */
	if( !reduce_pelement( rc, reduce_spine, &expr->root ) ) 
		return( FALSE );

	if( !itext_add_element( buf, &expr->root, TRUE ) && !buf->full ) 
		/* Tally eval failed, and buffer is not full ... must
		 * have been an eval error.
		 */
		return( FALSE );

	return( TRUE );
}

static void *
itext_update_model( Heapmodel *heapmodel )
{
	iText *itext = ITEXT( heapmodel );
        Row *row = HEAPMODEL( itext )->row;
	Expr *expr = row->expr;

#ifdef DEBUG
	printf( "itext_update_model: " );
	row_name_print( row );
	if( row->sym && row->sym->dirty )
		printf( " (dirty)" );
	printf( "\n" );
#endif /*DEBUG*/

	/* Update value display. We can't update the value display if this
	 * sym is dirty, since we might have pointers to deleted symbols in
	 * the heap (if we are dirty because one of our parents has been 
	 * deleted).

	 	FIXME ... this seem a bit restrictive :-( ... could just
		block reads of symbol pointers in
		itext_make_value_string() instead?

	 */
	buf_set_dynamic( &itext->value, LINELENGTH );
	if( expr && !expr->err && !row->sym->dirty ) {
		if( !itext_make_value_string( expr, &itext->value ) ) {
			buf_appendline( &itext->value, error_string );
			expr_error_set( expr );
		}
	}

	buf_set_dynamic( &itext->decompile, LINELENGTH );
	if( expr && !expr->err && !row->sym->dirty ) {
		if( !itext_make_decompiled_string( expr, &itext->decompile ) ) {
			buf_appendline( &itext->decompile, error_string );
			expr_error_set( expr );
		}
	}

#ifdef DEBUG
	printf( "itext_update_model: " );
	row_name_print( row );
	printf( " has value: %s\n", buf_all( &itext->value ) );
#endif /*DEBUG*/

	/* If this is a non-edited row, update the source.
	 */
	if( !itext->edited || row == row->top_row ) {
		const char *new_formula;

		if( expr && expr->compile && expr->compile->rhstext ) 
			new_formula = expr->compile->rhstext;
		else 
			new_formula = buf_all( &itext->decompile );

		SETSTR( itext->formula_default, new_formula );

		/* Don't use itext_set_formula(), as we don't want to set
		 * _modified or recomp.
		 */
		SETSTR( itext->formula, itext->formula_default );
	}

	return( HEAPMODEL_CLASS( parent_class )->update_model( heapmodel ) );
}

/* Build param lists.
 */
static void *
itext_update_heap_sub( Symbol *sym, BufInfo *buf )
{
	buf_appendf( buf, "%s ", MODEL( sym )->name );

	return( NULL );
}

/* model->modified is set ... parse, compile, and mark for recomp.
 */
static void *
itext_update_heap( Heapmodel *heapmodel )
{
	iText *itext = ITEXT( heapmodel );
        Row *row = heapmodel->row;
	Expr *expr = row->expr;

	char txt[MAX_STRSIZE];
	BufInfo buf;
	ParseRhsSyntax syntax;

	buf_init_static( &buf, txt, MAX_STRSIZE );
	if( is_super( row->sym ) ) {
		/* A super member ... special syntax.
		 */
		buf_appendf( &buf, "%s", itext->formula );

		syntax = PARSE_SUPER;
	}
	else {
		/* Build a new params + '=' + rhs string.
		 */
		if( expr->compile ) 
			(void) slist_map( expr->compile->param, 
				(SListMapFn) itext_update_heap_sub, &buf );
		buf_appendf( &buf, "= %s;", itext->formula );

		syntax = PARSE_PARAMS;
	}

	/* Parse and compile.
	 */
	expr_error_clear( expr );
	attach_input_string( buf_all( &buf ) );
	if( !parse_rhs( expr, syntax, FALSE ) ) {
		expr_error_set( expr );
		return( heapmodel );
	}

	/* Mark for recomp.
	 */
	(void) expr_dirty( expr, link_serial_new() );

	return( HEAPMODEL_CLASS( parent_class )->update_heap( heapmodel ) );
}

static void *
itext_clear_edited( Heapmodel *heapmodel )
{
	iText *itext = ITEXT( heapmodel );

#ifdef DEBUG
	printf( "itext_clear_edited: " );
	row_name_print( HEAPMODEL( itext )->row );
	printf( "\n" );
#endif /*DEBUG*/

	if( itext->edited ) {
		itext_set_edited( itext, FALSE );

		/*

			FIXME ... formula_default is not always set for cloned
			edited rows! fix this properly

		 */
		if( itext->formula_default )
			itext_set_formula( itext, itext->formula_default );
		else 
			printf( "itext_clear_edited: FIXME!\n" );

		(void) expr_dirty( heapmodel->row->expr, link_serial_new() );

		/* Don't clear HEAPMODEL( itext )->modified, we want to make
		 * sure we re-parse and compile the default value to break any
		 * old links we might have.
		 */
	}

	return( HEAPMODEL_CLASS( parent_class )->clear_edited( heapmodel ) );
}

static void
itext_parent_add( Model *child, Model *parent )
{
	iText *itext = ITEXT( child );
	Row *row;

	assert( IS_RHS( parent ) );

	MODEL_CLASS( parent_class )->parent_add( child, parent );

	row = HEAPMODEL( itext )->row;

#ifdef DEBUG
	printf( "itext_new: " );
	row_name_print( row );
	printf( "\n" );
#endif /*DEBUG*/

	/* Top rows default to edited.
	 */
	if( row == row->top_row )
		itext->edited = TRUE;
}

static gboolean
itext_load( Model *model, 
	ModelLoadState *state, Model *parent, xmlNode *xnode )
{
	iText *itext = ITEXT( model );

	char formula[MAX_STRSIZE];
	char formula2[MAX_STRSIZE];

	if( !IS_RHS( parent ) ) {
		ierrors( "itext_load: can only add a itext to a rhs" );
		return( FALSE );
	}

	if( get_sprop( xnode, "formula", formula, MAX_STRSIZE ) ) {
		model_loadstate_rewrite( state, formula, formula2 );
		itext_set_formula( itext, formula2 ); 
		itext_set_edited( itext, TRUE );
	}

	return( MODEL_CLASS( parent_class )->load( model, 
		state, parent, xnode ) );
}

static xmlNode *
itext_save( Model *model, xmlNode *xnode )
{
	iText *itext = ITEXT( model );
	Row *row = HEAPMODEL( model )->row;

	xmlNode *xthis;

	if( !(xthis = MODEL_CLASS( parent_class )->save( model, xnode )) )
		return( NULL );

	if( itext->edited || row->top_row == row )
		if( !set_sprop( xthis, "formula", itext->formula ) )
			return( NULL );

	return( xthis );
}

static void
itext_class_init( iTextClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	ModelClass *model_class = (ModelClass *) klass;
	HeapmodelClass *heapmodel_class = (HeapmodelClass *) klass;

	parent_class = gtk_type_class( TYPE_HEAPMODEL );

	object_class->destroy = itext_destroy;

	/* Create signals.
	 */

	/* Init methods.
	 */
	model_class->view_new = itextview_new;
	model_class->parent_add = itext_parent_add;
	model_class->save = itext_save;
	model_class->load = itext_load;

	heapmodel_class->update_model = itext_update_model;
	heapmodel_class->update_heap = itext_update_heap;
	heapmodel_class->clear_edited = itext_clear_edited;

	/* Static init.
	 */
	model_register_loadable( MODEL_CLASS( klass ) );
}

static void
itext_init( iText *itext )
{
	Model *model = MODEL( itext );

	model->display = FALSE;

	itext->formula = NULL;
	itext->formula_default = NULL;
	buf_init( &itext->value );
	buf_init( &itext->decompile );
	itext->edited = FALSE;

	/* Some defaults changed in _parent_add() above.
	 */
}

GtkType
itext_get_type( void )
{
	static GtkType itext_type = 0;

	if( !itext_type ) {
		static const GtkTypeInfo itext_info = {
			"iText",
			sizeof( iText ),
			sizeof( iTextClass ),
			(GtkClassInitFunc) itext_class_init,
			(GtkObjectInitFunc) itext_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		itext_type = gtk_type_unique( TYPE_HEAPMODEL, &itext_info );
	}

	return( itext_type );
}

iText *
itext_new( Rhs *rhs )
{
	iText *itext = gtk_type_new( TYPE_ITEXT );

	model_child_add( MODEL( rhs ), MODEL( itext ), -1 );

	return( itext );
}

void
itext_set_edited( iText *itext, gboolean edited )
{
	Heapmodel *heapmodel = HEAPMODEL( itext );

	if( itext->edited != edited ) {
#ifdef DEBUG
		printf( "itext_set_edited: " );
		row_name_print( heapmodel->row );
		printf( " %s\n", bool_to_char( edited ) );
#endif /*DEBUG*/

		itext->edited = edited;
		model_changed( MODEL( itext ) );
	}

	if( edited ) 
		heapmodel_set_modified( heapmodel, TRUE );
}

void
itext_set_formula( iText *itext, const char *formula )
{
	if( !itext->formula || strcmp( itext->formula, formula ) != 0 ) {
		SETSTR( itext->formula, formula )

		heapmodel_set_modified( HEAPMODEL( itext ), TRUE );

		model_changed( MODEL( itext ) );
	}
}
