/* Imagepresent widget code.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#include "ip.h"

/* Define to trace button press events.
#define EVENT
 */

/* Our signals. 
 */
enum {
	CHANGED,			/* Display state of child has changed */
	LAST_SIGNAL
};

static guint imagepresent_signals[LAST_SIGNAL] = { 0 };

/* Snap if closer than this.
 */
const int imagepresent_snap_threshold = 10;

/* Default show states.
 */
#define DISPLAY_RULERS (watch_bool_get( "DISPLAY_RULERS", FALSE ))

/* The ruler popup menu.
 */
static GtkWidget *imagepresent_ruler_menu = NULL;

/* XPMs for tool menu.
 */
#include "BITMAPS/select.xpm"		/* Defines select_xpm */
#include "BITMAPS/pan.xpm"		/* pan_xpm */
#include "BITMAPS/magin.xpm"		/* magin_xpm */
#include "BITMAPS/magout.xpm"		/* magout_xpm */
#include "BITMAPS/paint.xpm"		/* paint_xpm */

/* The above as GdkPixmaps.
 */
static GdkPixmap *tool_pixmap[IMAGEPRESENT_STATE_LAST] = { NULL };
static GdkBitmap *tool_mask_bitmap[IMAGEPRESENT_STATE_LAST] = { NULL };

/* Accelerators for tools.
 */
static const gchar *tool_accelerator[IMAGEPRESENT_STATE_LAST] = {
	"<ctrl>s",			/* IMAGEPRESENT_STATE_SELECT */
	"<ctrl>p",			/* IMAGEPRESENT_STATE_PAN */
	"equal",			/* IMAGEPRESENT_STATE_MAGIN */
	"minus",			/* IMAGEPRESENT_STATE_MAGOUT */
	"<ctrl>e"                       /* IMAGEPRESENT_STATE_PAINT */
};

/* Tips for tools.
 */
static const gchar *tool_tips[IMAGEPRESENT_STATE_LAST] = {
	"Edit regions (+CTRL to create)",/* IMAGEPRESENT_STATE_SELECT */
	"Pan image (also use middle mouse button)", /* IMAGEPRESENT_STATE_PAN */
	"Zoom in (also 'i' key)",	/* IMAGEPRESENT_STATE_MAGIN */
	"Zoom out (also 'o' key)",	/* IMAGEPRESENT_STATE_MAGOUT */
	"Paint"				/* IMAGEPRESENT_STATE_PAINT */
};

/* Cursor shape for each tool.
 */
iWindowShape tool_shape[IMAGEPRESENT_STATE_LAST] = {
	IWINDOW_SHAPE_NONE,		/* IMAGEPRESENT_STATE_SELECT */
	IWINDOW_SHAPE_MOVE, 		/* IMAGEPRESENT_STATE_PAN */
	IWINDOW_SHAPE_MAGIN,		/* IMAGEPRESENT_STATE_MAGIN */
	IWINDOW_SHAPE_MAGOUT,		/* IMAGEPRESENT_STATE_MAGOUT */
	IWINDOW_SHAPE_PEN		/* IMAGEPRESENT_STATE_PAINT */
};

/* Extract a method by event name.
 */
#define EVENT_METHOD(i,x) \
	GTK_SIGNAL_FUNC( GTK_WIDGET_CLASS( GTK_OBJECT( i )->klass )->x )

/* Parent class.
 */
static GtkWidgetClass *parent_class = NULL;

/* To save Imagepresent pointers.
 */
static const gchar *key_imagepresent = "Imagepresent";
static GQuark quark_imagepresent = 0;

static void
imagepresent_destroy( GtkObject *object )
{
	Imagepresent *ip;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_IMAGEPRESENT( object ) );

	ip = IMAGEPRESENT( object );

	FREEFO( gtk_object_destroy, ip->tooltips );
	FREEFI( gtk_timeout_remove, ip->scroll_timeout );
	FREEF( iwindow_cursor_context_destroy, ip->cntxt );

	/* Don't free, when our id dies it'll kill id->conv and this signal
	 * too.
	 */
	ip->conv_changed_sid = 0;
	ip->last_paint_ii = NULL;

	/*
		Why don't we dest this?
	FREEFO( gtk_object_destroy, ip->group );
	 */

	GTK_OBJECT_CLASS( parent_class )->destroy( object );

	/* Child views should all have removed themselves.
	 */
	assert( ip->regionviews == NULL );
}

static void
imagepresent_ruler_hide_cb( GtkWidget *wid, GtkWidget *host, Imagepresent *ip )
{
	imagepresent_rulers( ip, FALSE );
}

static void
imagepresent_class_init( ImagepresentClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;

	GtkWidget *pane;

	/* Init parent class.
	 */
        parent_class = gtk_type_class( GTK_TYPE_VBOX );

        object_class->destroy = imagepresent_destroy;

	/* Create quarks.
	 */
	quark_imagepresent = g_quark_from_static_string( key_imagepresent );

	/* Init default methods.
	 */
	klass->changed = NULL;

	/* Static class init.
	 */
	pane = imagepresent_ruler_menu = popup_build( "Ruler menu" );
	popup_add_but( pane, "Hide", imagepresent_ruler_hide_cb );

	/* Create signals.
	 */
	imagepresent_signals[CHANGED] = gtk_signal_new( "changed",
		GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET( ImagepresentClass, changed ),
		gtk_marshal_NONE__NONE,
		GTK_TYPE_NONE, 0 );

	gtk_object_class_add_signals( object_class, 
		imagepresent_signals, LAST_SIGNAL );
}

/* Make sure the menu has the right item selected. Add 1 to
 * allow for the tearoff at the top.
 */
static void
imagepresent_refresh_menu( Imagepresent *ip )
{
	GList *children = GTK_MENU_SHELL( ip->menu )->children;
	GtkWidget *item = g_list_nth_data( children, ip->state + 1 );
	GtkCheckMenuItem *check = GTK_CHECK_MENU_ITEM( item );

	if( !check->active )
		gtk_check_menu_item_set_active( check, TRUE );
}

/* Callback for check_paintable() in imagepresent_set_state. 
 */
static void
imagepresent_set_paintbox_cb( void *client, iWindowResult result )
{
	Imagepresent *ip = IMAGEPRESENT( client );
	Conversion *conv = ip->id->conv;

	if( result == IWINDOW_TRUE ) {
		ip->state = IMAGEPRESENT_STATE_PAINT;
		ip->last_paint_ii = conv->ii;
		conv->paintbox = TRUE;
		model_changed( MODEL( conv ) );
		imagepresent_refresh_menu( ip );
	}
}

static void
imagepresent_refresh_cursor( Imagepresent *ip )
{
	iWindowShape shape;

	if( ip->state == IMAGEPRESENT_STATE_PAINT )
		shape = paintboxview_shape[ip->id->conv->paintbox_tool];
	else
		shape = tool_shape[ip->state];

	if( ip->cntxt )
		iwindow_cursor_context_set_cursor( ip->cntxt, shape );
}

/* Set the viewer state ... set the cursor too.
 */
void
imagepresent_set_state( Imagepresent *ip, ImagepresentState state )
{
	if( state != ip->state && state == IMAGEPRESENT_STATE_PAINT ) {
		Conversion *conv = ip->id->conv;

		/* Special ... check and warn on this image first.
		 */
		imageinfo_check_paintable( GTK_WIDGET( ip ), conv->ii,
			imagepresent_set_paintbox_cb, ip );

		/* We've not set the state yet ... signal "changed" on the
		 * model to flick whatever asked for this change (e3g. View
		 * menu) back to the model state.
		 */
		model_changed( MODEL( conv ) );
	}
	else if( state != ip->state ) {
		ip->state = state;
		imagepresent_refresh_cursor( ip );
		imagepresent_refresh_menu( ip );
	}
}

/* adj has changed, rethink a ruler.
 */
static void
imagepresent_hruler_rethink_cb( GtkAdjustment *adj, Imagepresent *ip )
{
	Conversion *conv = ip->id->conv;
	IMAGE *im = imageinfo_get( FALSE, conv->ii );

	/* Display position.
	 */
	int dx_from = adj->value;
	int dx_to = adj->value + ip->id->screen.width;

	/* Mag as a scale factor.
	 */
	double scale = conversion_dmag( conv->mag );

	/* Position in image.
	 */
	double ix_from = dx_from / scale;
	double ix_to = dx_to / scale;

	if( im ) {
		ix_from -= im->Xoffset;
		ix_to -= im->Xoffset;
	}

	gtk_ruler_set_range( GTK_RULER( ip->hrule ), 
		ix_from, ix_to, 0, ix_to - ix_from );
}

static void
imagepresent_vruler_rethink_cb( GtkAdjustment *adj, Imagepresent *ip )
{
	Conversion *conv = ip->id->conv;
	IMAGE *im = imageinfo_get( FALSE, conv->ii );

	/* Display position.
	 */
	int dy_from = adj->value;
	int dy_to = adj->value + ip->id->screen.height;

	/* Mag as a scale factor.
	 */
	double scale = conversion_dmag( conv->mag );

	/* Position in image.
	 */
	double iy_from = dy_from / scale;
	double iy_to = dy_to / scale;

	if( im ) {
		iy_from -= im->Yoffset;
		iy_to -= im->Yoffset;
	}

	gtk_ruler_set_range( GTK_RULER( ip->vrule ), 
		iy_from, iy_to, 0, iy_to - iy_from );
}

static gint
imagepresent_hruler_event( GtkWidget *widget, GdkEvent *ev, Imagepresent *ip )
{
	Conversion *conv = ip->id->conv;
	IMAGE *im = imageinfo_get( FALSE, conv->ii );

	int dx, dy, ix, iy;

	switch( ev->type ) {
	case GDK_BUTTON_PRESS:
		switch( ev->button.button ) {
		case 1:
			imagepresent_set_state( ip, IMAGEPRESENT_STATE_SELECT );

			imagedisplay_xev_to_disp( ip->id, 
				ev->button.x, ev->button.y, &dx, &dy );
			conversion_disp_to_im( conv, dx, dy, &ix, &iy );
			ip->floating.left = -im->Xoffset;
			ip->floating.top = -im->Xoffset;
			ip->floating.width = im->Xsize;
			ip->floating.height = 0;
			ip->regionview = regionview_new( NULL, 
				&ip->floating, ip );
			ip->regionview->frozen = TRUE;
			ip->regionview->type = REGIONVIEW_HGUIDE;
			ip->regionview->shape = IWINDOW_SHAPE_BOTTOM;
			regionview_attach( ip->regionview );

			break;

		default:
			break;
		}
		break;

	case GDK_BUTTON_RELEASE:
		switch( ev->button.button ) {
		case 1:
			break;

		default:
			break;
		}
		break;

	default:
		break;
	}

	return( FALSE );
}

static gint
imagepresent_vruler_event( GtkWidget *widget, GdkEvent *ev, Imagepresent *ip )
{
	Conversion *conv = ip->id->conv;
	IMAGE *im = imageinfo_get( FALSE, conv->ii );

	int dx, dy, ix, iy;

	switch( ev->type ) {
	case GDK_BUTTON_PRESS:
		switch( ev->button.button ) {
		case 1:
			imagepresent_set_state( ip, IMAGEPRESENT_STATE_SELECT );

			imagedisplay_xev_to_disp( ip->id, 
				ev->button.x, ev->button.y, &dx, &dy );
			conversion_disp_to_im( conv, dx, dy, &ix, &iy );
			ip->floating.left = -im->Xoffset;
			ip->floating.top = -im->Yoffset;
			ip->floating.width = 0;
			ip->floating.height = im->Ysize;
			ip->regionview = regionview_new( NULL, 
				&ip->floating, ip );
			ip->regionview->frozen = TRUE;
			ip->regionview->type = REGIONVIEW_VGUIDE;
			ip->regionview->shape = IWINDOW_SHAPE_RIGHT;
			regionview_attach( ip->regionview );

			return( TRUE );

		default:
			break;
		}
		break;

	case GDK_BUTTON_RELEASE:
		switch( ev->button.button ) {
		case 1:
			break;

		default:
			break;
		}
		break;

	default:
		break;
	}

	return( FALSE );
}

/* "activate" on the tool menu.
 */
static void
imagepresent_menu_select_cb( GtkWidget *widget, ImagepresentState state )
{
	Imagepresent *ip = IMAGEPRESENT( gtk_object_get_data_by_id( 
		GTK_OBJECT( widget ), quark_imagepresent ) );

	if( GTK_CHECK_MENU_ITEM( widget )->active )
		imagepresent_set_state( ip, state );
}

/* Build the tools menu.
 */
static GtkWidget *
imagepresent_build_tool_menu( Imagepresent *ip ) 
{
	static char **maps[IMAGEPRESENT_STATE_LAST] = {
		select_xpm, pan_xpm, magin_xpm, magout_xpm, 
		paint_xpm
	};

	GtkWidget *item;
	GtkWidget *menu;
	GtkWidget *pixmapw;
	GSList *group;
	guint key;
	GdkModifierType mods;
	int i;

	if( !tool_pixmap[0] ) {
		for( i = 0; i < IMAGEPRESENT_STATE_LAST; i++ )
			tool_pixmap[i] = gdk_pixmap_create_from_xpm_d( 
				main_window_gdk,
				&tool_mask_bitmap[i], NULL, maps[i] );
	}

	menu = gtk_menu_new();

	item = gtk_tearoff_menu_item_new();
	gtk_menu_append( GTK_MENU( menu ), item );
	gtk_widget_show( item );

	for( group = NULL, i = 0; i < IMAGEPRESENT_STATE_LAST; i++ ) {
		ip->tool[i] = item = gtk_radio_menu_item_new( group );
		group = gtk_radio_menu_item_group( 
			GTK_RADIO_MENU_ITEM( item ) );
		gtk_check_menu_item_set_show_toggle( 
			GTK_CHECK_MENU_ITEM( item ), FALSE );
		gtk_tooltips_set_tip( ip->tooltips, item, 
			tool_tips[i], "ContextHelp/buttons/1" );

		pixmapw = gtk_pixmap_new( tool_pixmap[i], tool_mask_bitmap[i] );
		gtk_container_add( GTK_CONTAINER( item ), pixmapw );
		gtk_object_set_data_by_id( GTK_OBJECT( item ), 
			quark_imagepresent, ip );
		gtk_signal_connect( GTK_OBJECT( item ), "activate",
			GTK_SIGNAL_FUNC( imagepresent_menu_select_cb ), 
			GINT_TO_POINTER( i ) );
		gtk_accelerator_parse( tool_accelerator[i], &key, &mods );
		gtk_accel_group_add( ip->group, 
			key, mods, GTK_ACCEL_VISIBLE,
			GTK_OBJECT( item ), "activate" );

		gtk_menu_append( GTK_MENU( menu ), item );
		gtk_widget_show( pixmapw );
		gtk_widget_show( item );
	}

	return( menu );
}

/* Detach the popup menu ... this should never get called, as the parent is not
 * a menu. Implement it anyway.
 */
static void
imagepresent_menu_detacher( GtkWidget *widget, GtkMenu *menu )
{
	Imagepresent *ip;

	g_return_if_fail( widget != NULL );
	g_return_if_fail( IS_IMAGEPRESENT( widget ) );

	ip = IMAGEPRESENT( widget );
	g_return_if_fail( ip->menu == (GtkWidget *) menu );

	printf( "imagepresent_menu_detacher: detaching tool menu\n" );
	ip->menu = NULL;
}

/* Track this during a snap.
 */
typedef struct {
	Imagepresent *ip;

	int x;			/* Start point */
	int y;
	int off_x;		/* Current snap offset */
	int off_y;
	int best_x;		/* 'Closeness' of best snap so far */
	int best_y;
} ImagepresentSnap;

static void *
imagepresent_snap_sub( Regionview *regionview, 
	ImagepresentSnap *snap, gboolean *snapped )
{
	Conversion *conv = snap->ip->id->conv;
	Rect area;

	/* Only static h/v guides.
	 */
	if( regionview->type != REGIONVIEW_HGUIDE && 
		regionview->type != REGIONVIEW_VGUIDE )
		return( NULL );
	if( regionview->state != REGIONVIEW_WAIT )
		return( NULL );

	/* Work in display cods.
	 */
	conversion_im_to_disp_rect( conv, &regionview->area, &area );

	if( regionview->type == REGIONVIEW_HGUIDE ) {
		int score = abs( area.top - snap->y );

		if( score < snap->best_y ) {
			snap->off_y = area.top - snap->y;
			snap->best_y = score;
			*snapped = TRUE;
		}
	}
	else {
		int score = abs( area.left - snap->x );

		if( score < snap->best_x ) {
			snap->off_x = area.left - snap->x;
			snap->best_x = score;
			*snapped = TRUE;
		}
	}

	return( NULL );
}

static gboolean
imagepresent_snap( Imagepresent *ip, ImagepresentSnap *snap )
{
	gboolean snapped;

	snap->ip = ip;
	snap->off_x = 0;
	snap->off_y = 0;
	snap->best_x = imagepresent_snap_threshold;
	snap->best_y = imagepresent_snap_threshold;

	snapped = FALSE;
	slist_map2( ip->regionviews,
		(SListMap2Fn) imagepresent_snap_sub, snap, &snapped );

	return( snapped );
}

gboolean
imagepresent_snap_point( Imagepresent *ip, int x, int y, int *sx, int *sy )
{
	ImagepresentSnap snap;
	gboolean snapped;

	snap.x = x;
	snap.y = y;

	snapped = imagepresent_snap( ip, &snap );

	*sx = x + snap.off_x;
	*sy = y + snap.off_y;

	return( snapped );
}

gboolean
imagepresent_snap_rect( Imagepresent *ip, Rect *in, Rect *out )
{
	ImagepresentSnap snap[8];
	int i, best, best_score;
	gboolean snapped;

	/* Snap the corners plus the edge centres, take the best score.
	 */
	snap[0].x = in->left;
	snap[0].y = in->top;
	snap[1].x = in->left + in->width;
	snap[1].y = in->top;
	snap[2].x = in->left + in->width;
	snap[2].y = in->top + in->height;
	snap[3].x = in->left;
	snap[3].y = in->top + in->height;
	snap[4].x = in->left + in->width / 2;
	snap[4].y = in->top;
	snap[5].x = in->left + in->width;
	snap[5].y = in->top + in->height / 2;
	snap[6].x = in->left + in->width / 2;
	snap[6].y = in->top + in->height;
	snap[7].x = in->left;
	snap[7].y = in->top + in->height / 2;

	for( snapped = FALSE, i = 0; i < 8; i++ )
		snapped |= imagepresent_snap( ip, &snap[i] );

	for( best_score = 999, i = 0; i < 7; i++ )
		if( snap[i].best_x < best_score ) {
			best = i;
			best_score = snap[i].best_x;
		}
	out->left = in->left + snap[best].off_x;

	for( best_score = 999, i = 0; i < 7; i++ )
		if( snap[i].best_y < best_score ) {
			best = i;
			best_score = snap[i].best_y;
		}
	out->top = in->top + snap[best].off_y;

	out->width = in->width;
	out->height = in->height;

	return( snapped );
}

/* Zoom in ... try to get the bit of the image pointed to by x/y in the 
 * centre of the screen.
 */
static void
imagepresent_zoom_in( Imagepresent *ip, int x, int y )
{
	int ix, iy;

	/* Translate into IMAGE coordinates.
	 */
	conversion_disp_to_im( ip->id->conv, x, y, &ix, &iy );

	/* Zoom.
	 */
	imagedisplay_set_mag_position( ip->id, 
		conversion_double( ip->id->conv->mag ), ix, iy ); 
}

static void
imagepresent_zoom_out( Imagepresent *ip )
{
	imagedisplay_set_mag( ip->id, 
		conversion_halve( ip->id->conv->mag ) );
}

void
imagepresent_paint_start( Imagepresent *ip, int x, int y )
{
	Imagedisplay *id = ip->id;
	Conversion *conv = id->conv;

	IMAGE *im, *im2;
	int ix, iy;

	imagepresent_snap_point( ip, x, y, &x, &y );
	conversion_disp_to_im( conv, x, y, &ix, &iy );

	switch( conv->paintbox_tool ) {
	case PAINTBOX_DROPPER:
	case PAINTBOX_FLOOD:
	case PAINTBOX_BLOB:
		break;

	case PAINTBOX_PEN:
	case PAINTBOX_SMUDGE:
		ip->paint_last_x = ix;
		ip->paint_last_y = iy;
		break;

	case PAINTBOX_LINE:
		ip->paint_last_x = ix;
		ip->paint_last_y = iy;

		im = imageinfo_get( FALSE, conv->ii );
		ip->floating.left = ix - im->Xoffset;
		ip->floating.top = iy - im->Yoffset;
		ip->floating.width = 0;
		ip->floating.height = 0;
		ip->regionview = regionview_new( NULL, &ip->floating, ip );
		ip->regionview->frozen = TRUE;
		ip->regionview->type = REGIONVIEW_LINE;
		ip->regionview->shape = IWINDOW_SHAPE_BOTTOMRIGHT;
		regionview_attach( ip->regionview );
		break;

	case PAINTBOX_RECT:
		im = imageinfo_get( FALSE, conv->ii );
		ip->floating.left = ix - im->Xoffset;
		ip->floating.top = iy - im->Yoffset;
		ip->floating.width = 0;
		ip->floating.height = 0;
		ip->regionview = regionview_new( NULL, &ip->floating, ip );
		ip->regionview->frozen = TRUE;
		ip->regionview->type = REGIONVIEW_BOX;
		ip->regionview->shape = IWINDOW_SHAPE_BOTTOMRIGHT;
		regionview_attach( ip->regionview );
		break;

	case PAINTBOX_TEXT:
		ip->paint_last_x = ix;
		ip->paint_last_y = iy;

		if( !conversion_refresh_text( conv ) ) {
			box_alert( GTK_WIDGET( ip ) );
			break;
		}

		im = imageinfo_get( FALSE, conv->ii );
		im2 = imageinfo_get( FALSE, conv->paintbox_text_mask );
		ip->floating.left = ix - im->Xoffset;
		ip->floating.top = iy - im->Yoffset +
			conv->paintbox_text_area.top;
		ip->floating.width = im2->Xsize;
		ip->floating.height = im2->Ysize;
		ip->regionview = regionview_new( NULL, &ip->floating, ip );
		ip->regionview->frozen = TRUE;
		ip->regionview->type = REGIONVIEW_BOX;
		ip->regionview->shape = IWINDOW_SHAPE_EDIT;
		regionview_attach( ip->regionview );
		break;

	default:
		assert( FALSE );
	}
}

/* Left button press event.
 */
static void
imagepresent_left_press( Imagepresent *ip, GdkEvent *ev, int x, int y )
{
	switch( ip->state ) {
	case IMAGEPRESENT_STATE_SELECT:
		if( !ip->regionview && ev->button.state & GDK_CONTROL_MASK ) {
			Conversion *conv = ip->id->conv;
			IMAGE *im = imageinfo_get( FALSE, conv->ii );
			int ix, iy;

			imagepresent_snap_point( ip, x, y, &x, &y );
			conversion_disp_to_im( ip->id->conv, x, y, &ix, &iy );
			ip->floating.left = ix - im->Xoffset;
			ip->floating.top = iy - im->Yoffset;
			ip->floating.width = 0;
			ip->floating.height = 0;
			ip->regionview = 
				regionview_new( NULL, &ip->floating, ip );
			ip->regionview->frozen = FALSE;
			ip->regionview->type = REGIONVIEW_POINT;
			ip->regionview->shape = IWINDOW_SHAPE_BOTTOMRIGHT;
			regionview_attach( ip->regionview );
		}

		break;

	case IMAGEPRESENT_STATE_PAN:
		/* Save drag start as canvas cods.
		 */
		ip->dx = x;
		ip->dy = y;

		break;

	case IMAGEPRESENT_STATE_MAGIN:
		imagepresent_zoom_in( ip, x, y );
		break;

	case IMAGEPRESENT_STATE_MAGOUT:
		imagepresent_zoom_out( ip );
		break;

        case IMAGEPRESENT_STATE_PAINT:
                imagepresent_paint_start( ip, x, y );
                break;

	default:
		break;
	}
}

/* After a paint action: mark all subsequent things dirty, recalc if prefs say
 * so.
 */
void
imagepresent_paint_recalc( Imagepresent *ip )
{
	Classmodel *classmodel = CLASSMODEL( ip->iimage );
	Row *row = HEAPMODEL( classmodel )->row;

	expr_dirty_intrans( row->expr, link_serial_new() );

	if( PAINTBOX_RECOMP )
		symbol_recalculate_all();
}

static void
imagepresent_paint_stop( Imagepresent *ip, int x, int y )
{
	Conversion *conv = ip->id->conv;
	Imageinfo *imageinfo = ip->iimage->instance.image;
	Rect oper;
	int ix, iy;

	imagepresent_snap_point( ip, x, y, &x, &y );
	conversion_disp_to_im( ip->id->conv, x, y, &ix, &iy );

	switch( conv->paintbox_tool ) {
	case PAINTBOX_DROPPER:
		if( im_rect_includespoint( &conv->underlay, ix, iy ) ) 
			if( !imageinfo_paint_dropper( imageinfo, 
				conv->paintbox_ink, 
				ix, iy ) )
				box_alert( GTK_WIDGET( ip ) );
		break;

	case PAINTBOX_PEN:
		if( !imageinfo_paint_line( imageinfo, conv->paintbox_ink,
			ip->id->conv->paintbox_nib, 
			ip->paint_last_x, ip->paint_last_y, ix, iy ) )
			box_alert( GTK_WIDGET( ip ) );
		break;

	case PAINTBOX_LINE:
		if( ip->regionview ) { 
			IMAGE *im = imageinfo_get( FALSE, conv->ii );

			FREEO( ip->regionview );

			ip->floating.left += im->Xoffset;
			ip->floating.top += im->Yoffset;

			if( !imageinfo_paint_line( imageinfo, 
				conv->paintbox_ink,
				ip->id->conv->paintbox_nib, 
				ip->floating.left, ip->floating.top,
				IM_RECT_RIGHT( &ip->floating ),
				IM_RECT_BOTTOM( &ip->floating ) ) )
				box_alert( GTK_WIDGET( ip ) );
		}
		break;

	case PAINTBOX_RECT:
		if( ip->regionview ) { 
			IMAGE *im = imageinfo_get( FALSE, conv->ii );

			FREEO( ip->regionview );

			ip->floating.left += im->Xoffset;
			ip->floating.top += im->Yoffset;
			im_rect_normalise( &ip->floating );

			if( !imageinfo_paint_rect( imageinfo, 
				conv->paintbox_ink, &ip->floating ) )
				box_alert( GTK_WIDGET( ip ) );
		}
		break;

	case PAINTBOX_FLOOD:
		if( !imageinfo_paint_flood( imageinfo, conv->paintbox_ink, 
			ix, iy, FALSE ) )
			box_alert( GTK_WIDGET( ip ) );
		break;

	case PAINTBOX_BLOB:
		if( !imageinfo_paint_flood( imageinfo, conv->paintbox_ink, 
			ix, iy, TRUE ) )
			box_alert( GTK_WIDGET( ip ) );
		break;

	case PAINTBOX_TEXT:
		if( ip->regionview ) { 
			FREEO( ip->regionview );

			if( !imageinfo_paint_mask( imageinfo, 
				conv->paintbox_ink, conv->paintbox_text_mask, 
				ip->floating.left, ip->floating.top ) )
				box_alert( GTK_WIDGET( ip ) );
		}
		break;

	case PAINTBOX_SMUDGE:
		/* Area to smudge in display cods.
		 */
		oper.left = -10;
		oper.top = -10;
		oper.width = 20;
		oper.height = 20;

		/* Translate to IMAGE cods.
		 */
		conversion_disp_to_im_rect( conv, &oper, &oper );

		if( !imageinfo_paint_smudge( imageinfo,
			&oper, ip->paint_last_x, ip->paint_last_y, ix, iy ) )
			box_alert( GTK_WIDGET( ip ) );
		break;

	default:
		assert( FALSE );
	}

	imagepresent_paint_recalc( ip );
	imageinfo_undo_mark( imageinfo );
}

/* Left button release event.
 */
static void
imagepresent_left_release( Imagepresent *ip, GdkEvent *ev )
{
	Row *row = HEAPMODEL( ip->iimage )->row;
	int x, y;

	switch( ip->state ) {
	case IMAGEPRESENT_STATE_SELECT:
		if( ip->regionview && row ) {
			/* Make a new region.
			 */
			char txt[NAMELEN];
			BufInfo buf;
			Symbol *sym;

			buf_init_static( &buf, txt, NAMELEN );

			switch( ip->regionview->type ) {
			case REGIONVIEW_POINT:
				buf_appendf( &buf, "%s ", CLASS_POINT );
				row_qualified_name( row, &buf );
				buf_appendd( &buf, ip->floating.left );
				buf_appendd( &buf, ip->floating.top );
				break;

			case REGIONVIEW_REGION:
				buf_appendf( &buf, "%s ", CLASS_REGION );
				row_qualified_name( row, &buf );
				buf_appendd( &buf, ip->floating.left );
				buf_appendd( &buf, ip->floating.top );
				buf_appendd( &buf, ip->floating.width );
				buf_appendd( &buf, ip->floating.height );
				break;

			case REGIONVIEW_ARROW:
				buf_appendf( &buf, "%s ", CLASS_ARROW );
				row_qualified_name( row, &buf );
				buf_appendd( &buf, ip->floating.left );
				buf_appendd( &buf, ip->floating.top );
				buf_appendd( &buf, ip->floating.width );
				buf_appendd( &buf, ip->floating.height );
				break;

			case REGIONVIEW_HGUIDE:
				buf_appendf( &buf, "%s ", CLASS_HGUIDE );
				row_qualified_name( row, &buf );
				buf_appendd( &buf, ip->floating.top );
				break;

			case REGIONVIEW_VGUIDE:
				buf_appendf( &buf, "%s ", CLASS_VGUIDE );
				row_qualified_name( row, &buf );
				buf_appendd( &buf, ip->floating.left );
				break;

			default:
				assert( FALSE );
			}

			FREEO( ip->regionview );
			if( !(sym = mainw_def_new( buf_all( &buf ) )) ) 
				box_alert( NULL );
		}
		break;

	case IMAGEPRESENT_STATE_PAINT:
		imagedisplay_xev_to_disp( ip->id, 
			ev->button.x, ev->button.y, &x, &y );
                imagepresent_paint_stop( ip, x, y );
		break;
 
	default:
		break;
	}
}

static void
imagepresent_paint_motion( Imagepresent *ip, int x, int y )
{
	Conversion *conv = ip->id->conv;
	Imageinfo *imageinfo = ip->iimage->instance.image;
	Rect oper;
	int ix, iy;

	imagepresent_snap_point( ip, x, y, &x, &y );
	conversion_disp_to_im( conv, x, y, &ix, &iy );

	switch( conv->paintbox_tool ) {
	case PAINTBOX_DROPPER:
		if( im_rect_includespoint( &conv->underlay, ix, iy ) ) 
			if( !imageinfo_paint_dropper( imageinfo, 
				conv->paintbox_ink, 
				ix, iy ) )
				box_alert( GTK_WIDGET( ip ) );
		break;

	case PAINTBOX_PEN:
		if( !imageinfo_paint_line( imageinfo, conv->paintbox_ink,
			conv->paintbox_nib, 
			ip->paint_last_x, ip->paint_last_y, ix, iy ) )
			box_alert( GTK_WIDGET( ip ) );
		ip->paint_last_x = ix;
		ip->paint_last_y = iy;
		break;

	case PAINTBOX_LINE:
		/* rubberband
		 */
		break;

	case PAINTBOX_SMUDGE:
		/* Area to smudge in display cods.
		 */
		oper.left = -10;
		oper.top = -10;
		oper.width = 20;
		oper.height = 20;

		/* Translate to IMAGE cods.
		 */
		conversion_disp_to_im_rect( conv, &oper, &oper );

		if( !imageinfo_paint_smudge( imageinfo, 
			&oper, ip->paint_last_x, ip->paint_last_y, ix, iy ) )
			box_alert( GTK_WIDGET( ip ) );
		ip->paint_last_x = ix;
		ip->paint_last_y = iy;
		break;

	default:
		break;
	}
}

/* Left button motion event.
 */
static void
imagepresent_left_motion( Imagepresent *ip, int x, int y )
{
	switch( ip->state ) {
	case IMAGEPRESENT_STATE_SELECT:
		break;

	case IMAGEPRESENT_STATE_PAN:
		imagedisplay_set_position( ip->id, 
			ip->dx - (x - ip->id->visible.left),
			ip->dy - (y - ip->id->visible.top) );
		break;

	case IMAGEPRESENT_STATE_MAGIN:
		break;

	case IMAGEPRESENT_STATE_MAGOUT:
		break;

	case IMAGEPRESENT_STATE_PAINT:
                imagepresent_paint_motion( ip, x, y );
		break;

	default:
		break;
	}
}

/* Main event loop.
 */
static gint
imagepresent_event( GtkWidget *widget, GdkEvent *ev, Imagepresent *ip )
{
	GtkAdjustment *vadj = imagedisplay_get_vadj( ip->id );
	int x, y;
	int incr;

#ifdef EVENT
	if( ev->type == GDK_BUTTON_PRESS )
		printf( "imagepresent_event: GDK_BUTTON_PRESS\n" );
#endif /*EVENT*/

	switch( ev->type ) {
	case GDK_BUTTON_PRESS:
		imagedisplay_xev_to_disp( ip->id, 
			ev->button.x, ev->button.y, &x, &y );

		switch( ev->button.button ) {
		case 1:
			imagepresent_left_press( ip, ev, x, y );

			break;

		case 2:
#ifdef DEBUG
			printf( "button2 press: at %gx%g\n",
				ev->button.x, ev->button.y );
#endif /*DEBUG*/

			/* Switch to pan, for this drag.
			 */
			ip->save_state = ip->state;
			imagepresent_set_state( ip, IMAGEPRESENT_STATE_PAN );
			imagepresent_left_press( ip, ev, x, y );

			break;

		case 3:
#ifdef DEBUG
			printf( "button3 press: at %gx%g\n",
				ev->button.x, ev->button.y );
#endif /*DEBUG*/

			if( ip->menu )
				gtk_menu_popup( GTK_MENU( ip->menu ), 
					NULL, NULL, NULL, NULL, 
					3, ev->button.time );

			break;
		
		case 4:
		case 5:
			/* Scroll wheel.
			 */
			incr = vadj->step_increment;
			if( ev->button.state & GDK_SHIFT_MASK )
				incr *= 4;
			if( ev->button.state & GDK_CONTROL_MASK )
				incr *= 8;
			if( ev->button.button == 4 )
				incr *= -1;

			imagedisplay_set_position( ip->id, 
				ip->id->visible.left,
				ip->id->visible.top + incr );

			break;

		default:
			break;
		}

		break;

	case GDK_BUTTON_RELEASE:
		switch( ev->button.button ) {
		case 1:
			imagepresent_left_release( ip, ev );

			break;

		case 2:
#ifdef DEBUG
			printf( "button2 release: at %gx%g\n",
				ev->button.x, ev->button.y );
#endif /*DEBUG*/

			imagepresent_set_state( ip, ip->save_state );

			break;

		default:
			break;
		}

		break;

	case GDK_MOTION_NOTIFY:
		ip->last_x = ev->motion.x;
		ip->last_y = ev->motion.y;

		imagedisplay_xev_to_disp( ip->id, 
			ev->motion.x, ev->motion.y, &x, &y );

		if( ev->motion.state & GDK_BUTTON1_MASK ||
			ev->motion.state & GDK_BUTTON2_MASK ) 
			imagepresent_left_motion( ip, x, y );

		break;

	case GDK_ENTER_NOTIFY:
		ip->inside = TRUE;

		break;

	case GDK_LEAVE_NOTIFY:
		ip->inside = FALSE;

		break;

	default:
		break;
	}

	return( FALSE );
}

static gint
imagepresent_key_press_event( GtkWidget *widget, GdkEventKey *event, 
	Imagepresent *ip )
{
	Imagedisplay *id = ip->id;
	GtkAdjustment *hadj = imagedisplay_get_hadj( id );
	GtkAdjustment *vadj = imagedisplay_get_vadj( id );

	gboolean handled = FALSE;

	switch( event->keyval ) {
	case GDK_Left:
		if( !(event->state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK)) )
			imagedisplay_set_position( id, 
				id->visible.left - hadj->step_increment, 
				id->visible.top );
		else if( event->state & GDK_SHIFT_MASK )
			imagedisplay_set_position( id, 
				id->visible.left - hadj->page_increment, 
				id->visible.top );
		else if( event->state & GDK_CONTROL_MASK )
			imagedisplay_set_position( id, 0, id->visible.top );

		handled = TRUE;
		break;

	case GDK_Right:
		if( !(event->state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK)) )
			imagedisplay_set_position( id, 
				id->visible.left + hadj->step_increment, 
				id->visible.top );
		else if( event->state & GDK_SHIFT_MASK )
			imagedisplay_set_position( id, 
				id->visible.left + hadj->page_increment, 
				id->visible.top );
		else if( event->state & GDK_CONTROL_MASK )
			imagedisplay_set_position( id, 
				id->conv->canvas.width, id->visible.top );

		handled = TRUE;
		break;

	case GDK_Up:
		if( !(event->state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK)) )
			imagedisplay_set_position( id, 
				id->visible.left,
				id->visible.top - vadj->step_increment );
		else if( event->state & GDK_SHIFT_MASK )
			imagedisplay_set_position( id, 
				id->visible.left,
				id->visible.top - vadj->page_increment );
		else if( event->state & GDK_CONTROL_MASK )
			imagedisplay_set_position( id, id->visible.left, 0 );

		handled = TRUE;
		break;

	case GDK_Down:
		if( !(event->state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK)) )
			imagedisplay_set_position( id, 
				id->visible.left,
				id->visible.top + vadj->step_increment );
		else if( event->state & GDK_SHIFT_MASK )
			imagedisplay_set_position( id, 
				id->visible.left,
				id->visible.top + vadj->page_increment );
		else if( event->state & GDK_CONTROL_MASK )
			imagedisplay_set_position( id, 
				id->visible.left, id->conv->canvas.height );

		handled = TRUE;
		break;

	case GDK_i:
		if( ip->inside ) {
			/* Zoom on mouse posn.
			 */
			int dx, dy;
			int ix, iy;

			imagedisplay_xev_to_disp( id, 
				ip->last_x, ip->last_y, &dx, &dy );
			conversion_disp_to_im( id->conv, dx, dy, &ix, &iy );
			imagedisplay_set_mag_position( id, 
				conversion_double( id->conv->mag ), ix, iy ); 
		}
		else {
			/* Zoom on centre.
			 */
			int dx = id->visible.left + id->visible.width / 2;
			int dy = id->visible.top + id->visible.height / 2;
			int ix, iy;

			conversion_disp_to_im( id->conv, dx, dy, &ix, &iy );
			imagedisplay_set_mag_position( id, 
				conversion_double( id->conv->mag ), ix, iy ); 
		}

		handled = TRUE;
		break;

	case GDK_o: {
		int dx = id->visible.left + id->visible.width / 2;
		int dy = id->visible.top + id->visible.height / 2;
		int ix, iy;

		conversion_disp_to_im( id->conv, dx, dy, &ix, &iy );
		imagedisplay_set_mag_position( id, 
			conversion_halve( id->conv->mag ), ix, iy ); 

		handled = TRUE;
		break;
	}

	default:
		break;
	}

	if( handled ) {
		gtk_signal_emit_stop_by_name( GTK_OBJECT( widget ), 
			"key_press_event" );

		return( TRUE );
	}

	return( FALSE );
}

/* Connect to our enclosing iwnd on realize.
 */
static void
imagepresent_realize( Imagepresent *ip )
{
	if( !ip->cntxt ) {
		iWindow *iwnd = 
			IWINDOW( gtk_widget_get_toplevel( GTK_WIDGET( ip ) ) );	

		ip->cntxt = iwindow_cursor_context_new( iwnd );
		imagepresent_refresh_cursor( ip );
	}
}

static void
imagepresent_init( Imagepresent *ip )
{
	GtkAdjustment *hadj;
	GtkAdjustment *vadj;
	GtkWidget *frame;
	GtkWidget *table;

	int i;

	/* Basic init.
	 */
	ip->conv_changed_sid = 0;
	ip->last_paint_ii = NULL;
	ip->state = IMAGEPRESENT_STATE_SELECT;
	ip->cntxt = NULL;
	for( i = 0; i < IMAGEPRESENT_STATE_LAST; i++ ) 
		ip->tool[i] = NULL;
	ip->dx = ip->dy = 0;
	ip->show_rulers = DISPLAY_RULERS;
	ip->group = gtk_accel_group_new();
	ip->menu = NULL;
	ip->tooltips = gtk_tooltips_new();
	ip->last_x = 0;
	ip->last_y = 0;
	ip->inside = FALSE;
	ip->scroll_timeout = 0;
	ip->u = 0;
	ip->v = 0;
	ip->regionview = NULL;
	ip->paint_last_x = 0;
	ip->paint_last_y = 0;
	ip->regionviews = NULL;

	/* Frame table.
	 */
	frame = gtk_frame_new( NULL );
	gtk_frame_set_shadow_type( GTK_FRAME( frame ), GTK_SHADOW_OUT );
	gtk_box_pack_end( GTK_BOX( ip ), frame, TRUE, TRUE, 0 );
	gtk_widget_show( frame );

	/* Make main imagedisplay table.
	 */
	table = GTK_WIDGET( gtk_table_new( 3, 3, FALSE ) );
	gtk_container_add( GTK_CONTAINER( frame ), table );
	gtk_widget_show( table );

	/* Make canvas.
	 */
	ip->id = imagedisplay_new( NULL, NULL, NULL );
	gtk_signal_connect( GTK_OBJECT( ip ), "realize",
		GTK_SIGNAL_FUNC( imagepresent_realize ), NULL );

	/* Press/release/motion-notify stuff.
	 */
	gtk_widget_set_events( GTK_WIDGET( ip->id ), 
		GDK_KEY_PRESS_MASK | GDK_POINTER_MOTION_MASK |
		GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK ); 
	gtk_signal_connect_after( GTK_OBJECT( ip->id ), "event",
		GTK_SIGNAL_FUNC( imagepresent_event ), ip );
	gtk_signal_connect( GTK_OBJECT( ip->id ), "key_press_event",
		GTK_SIGNAL_FUNC( imagepresent_key_press_event ), ip );

	/* Make h/v sbars.
	 */
	hadj = imagedisplay_get_hadj( ip->id );
	vadj = imagedisplay_get_vadj( ip->id );
	ip->hbar = GTK_HSCROLLBAR( gtk_hscrollbar_new( hadj ) );
	ip->vbar = GTK_VSCROLLBAR( gtk_vscrollbar_new( vadj ) );
	GTK_WIDGET_UNSET_FLAGS( GTK_WIDGET( ip->hbar ), GTK_CAN_FOCUS );
	GTK_WIDGET_UNSET_FLAGS( GTK_WIDGET( ip->vbar ), GTK_CAN_FOCUS );

	/* Make rulers.
	 */
	ip->hrule = GTK_HRULER( gtk_hruler_new() );
	gtk_ruler_set_metric( GTK_RULER( ip->hrule ), GTK_PIXELS );
	gtk_signal_connect_object( GTK_OBJECT( ip->id ), "motion_notify_event",
		EVENT_METHOD( ip->hrule, motion_notify_event ),
		GTK_OBJECT( ip->hrule ) );
        gtk_signal_connect( GTK_OBJECT( hadj ), "changed",
		GTK_SIGNAL_FUNC( imagepresent_hruler_rethink_cb ), ip );
        gtk_signal_connect( GTK_OBJECT( hadj ), "value_changed",
		GTK_SIGNAL_FUNC( imagepresent_hruler_rethink_cb ), ip );
	GTK_WIDGET_UNSET_FLAGS( GTK_WIDGET( ip->hrule ), GTK_CAN_FOCUS );
	gtk_widget_show( GTK_WIDGET( ip->hrule ) );

	ip->vrule = GTK_VRULER( gtk_vruler_new() );
	gtk_ruler_set_metric( GTK_RULER( ip->vrule ), GTK_PIXELS );
	gtk_signal_connect_object( GTK_OBJECT( ip->id ), "motion_notify_event",
		EVENT_METHOD( ip->vrule, motion_notify_event ),
		GTK_OBJECT( ip->vrule ) );
        gtk_signal_connect( GTK_OBJECT( vadj ), "changed",
		GTK_SIGNAL_FUNC( imagepresent_vruler_rethink_cb ), ip );
        gtk_signal_connect( GTK_OBJECT( vadj ), "value_changed",
		GTK_SIGNAL_FUNC( imagepresent_vruler_rethink_cb ), ip );
	GTK_WIDGET_UNSET_FLAGS( GTK_WIDGET( ip->vrule ), GTK_CAN_FOCUS );
	gtk_widget_show( GTK_WIDGET( ip->vrule ) );

	ip->heb = GTK_EVENT_BOX( gtk_event_box_new() );
        gtk_container_add( GTK_CONTAINER( ip->heb ), GTK_WIDGET( ip->hrule ) );
        gtk_signal_connect( GTK_OBJECT( ip->heb ), "event",
		GTK_SIGNAL_FUNC( imagepresent_hruler_event ), ip );
        popup_attach( GTK_WIDGET( ip->heb ), imagepresent_ruler_menu, ip );

	ip->veb = GTK_EVENT_BOX( gtk_event_box_new() );
        gtk_container_add( GTK_CONTAINER( ip->veb ), GTK_WIDGET( ip->vrule ) );
        gtk_signal_connect( GTK_OBJECT( ip->veb ), "event",
		GTK_SIGNAL_FUNC( imagepresent_vruler_event ), ip );
        popup_attach( GTK_WIDGET( ip->veb ), imagepresent_ruler_menu, ip );

	/* Attach all widgets to table.
	 */
	gtk_table_attach( GTK_TABLE( table ), GTK_WIDGET( ip->hbar ), 
		1, 2, 2, 3,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		GTK_FILL, 
		2, 2 );
	gtk_table_attach( GTK_TABLE( table ), GTK_WIDGET( ip->vbar ), 
		2, 3, 1, 2,
		GTK_FILL, 
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		2, 2 );
	gtk_table_attach( GTK_TABLE( table ), GTK_WIDGET( ip->heb ), 
		1, 2, 0, 1,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		GTK_FILL, 
		2, 2 );
	gtk_table_attach( GTK_TABLE( table ), GTK_WIDGET( ip->veb ), 
		0, 1, 1, 2,
		GTK_FILL, 
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		2, 2 );
	gtk_table_attach( GTK_TABLE( table ), GTK_WIDGET( ip->id ), 
		1, 2, 1, 2,
		GTK_FILL | GTK_EXPAND | GTK_SHRINK, 
		GTK_FILL | GTK_EXPAND | GTK_SHRINK,
		2, 2 );
	gtk_widget_show( GTK_WIDGET( ip->id ) );
	gtk_widget_show( GTK_WIDGET( ip->hbar ) );
	gtk_widget_show( GTK_WIDGET( ip->vbar ) );

	widget_visible( GTK_WIDGET( ip->heb ), DISPLAY_RULERS );
	widget_visible( GTK_WIDGET( ip->veb ), DISPLAY_RULERS );

	ip->menu = imagepresent_build_tool_menu( ip );
	gtk_menu_attach_to_widget( GTK_MENU( ip->menu ), 
		GTK_WIDGET( ip ), imagepresent_menu_detacher );
}

/* Make class id.
 */
guint
imagepresent_get_type( void )
{
	static guint ip_type = 0;

	if( !ip_type ) {
		GtkTypeInfo info = {
			"Imagepresent",
			sizeof( Imagepresent ),
			sizeof( ImagepresentClass ),
			(GtkClassInitFunc) imagepresent_class_init,
			(GtkObjectInitFunc) imagepresent_init,
			/* reserved1 */ NULL,
			/* reserved2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		/* Create our type.
		 */
		ip_type = gtk_type_unique( GTK_TYPE_VBOX, &info );
	}

	return( ip_type );
}

/* The conversion has changed ... update!
 */
static void
imagepresent_conv_changed_cb( Conversion *conv, Imagepresent *ip )
{
	imagepresent_refresh_cursor( ip );

	/* Has the image changed since we last checked it was OK to paint?
	 * Bounce out of paint mode if it has.
	 */
	if( ip->state == IMAGEPRESENT_STATE_PAINT &&
		ip->last_paint_ii &&
		conv->ii != ip->last_paint_ii ) {
		ip->last_paint_ii = NULL;
		imagepresent_set_state( ip, IMAGEPRESENT_STATE_SELECT );
	}
}

/* Make a new Imagepresent. Pass in the image we should show, or NULL.
 */
Imagepresent *
imagepresent_new( Iimage *iimage, Conversion *conv )
{
	Imagepresent *ip = gtk_type_new( TYPE_IMAGEPRESENT );

	ip->iimage = iimage;
	imagedisplay_set_conversion( ip->id, conv );
	ip->conv_changed_sid = gtk_signal_connect( 
		GTK_OBJECT( ip->id->conv ), "changed", 
		GTK_SIGNAL_FUNC( imagepresent_conv_changed_cb ), ip );

	return( ip );
}

void
imagepresent_set_position( Imagepresent *ip, int x, int y )
{
	imagedisplay_set_position( ip->id, x, y );
}

/* Show/hide rulers.
 */
void
imagepresent_rulers( Imagepresent *ip, gboolean show_rulers )
{
	if( ip->show_rulers != show_rulers ) {
		widget_visible( GTK_WIDGET( ip->heb ), show_rulers );
		widget_visible( GTK_WIDGET( ip->veb ), show_rulers );
		ip->show_rulers = show_rulers;

		gtk_signal_emit( GTK_OBJECT( ip ),
			imagepresent_signals[CHANGED] );
	}
}

/* Background scroller.
 */
static gint
imagepresent_scroll( Imagepresent *ip )
{
	imagepresent_set_position( ip, 
		ip->id->visible.left + ip->u, ip->id->visible.top + ip->v );

	return( TRUE );
}

void
imagepresent_scroll_start( Imagepresent *ip, int u, int v )
{
	if( !ip->scroll_timeout )
		ip->scroll_timeout = gtk_timeout_add( 100, 
			(GtkFunction) imagepresent_scroll, ip );

	ip->u = u;
	ip->v = v;
}

void
imagepresent_scroll_stop( Imagepresent *ip )
{
	FREEFI( gtk_timeout_remove, ip->scroll_timeout );
	ip->u = 0;
	ip->v = 0;
}
