/* ip's file selectors.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

*/

#include "ip.h"

/* Define for debugging output.
#define DEBUG
 */

#define PIN_FILESEL (watch_bool_get( "CALC_PIN_FILESEL", FALSE ))

#define IP_JPEG_Q (watch_int_get( "JPEG_Q", 75 ))
#define IP_JPEG_ICC_PROFILE \
	(watch_string_get( "JPEG_ICC_PROFILE", \
		"$VIPSHOME/share/nip/data/sRGB.icm" ))
#define IP_PPM_MODE (watch_int_get( "PPM_MODE", 0 ))
#define IP_PNG_COMPRESSION (watch_int_get( "PNG_COMPRESSION", 6 ))
#define IP_PNG_INTERLACE (watch_int_get( "PNG_INTERLACE", 0 ))
#define IP_TIFF_COMPRESSION (watch_int_get( "TIFF_COMPRESSION", 0 ))
#define IP_TIFF_JPEG_Q (watch_int_get( "TIFF_JPEG_Q", 75 ))
#define IP_TIFF_LAYOUT (watch_int_get( "TIFF_LAYOUT", 0 ))
#define IP_TIFF_TILE_WIDTH (watch_int_get( "TIFF_TILE_WIDTH", 128 ))
#define IP_TIFF_TILE_HEIGHT (watch_int_get( "TIFF_TILE_HEIGHT", 128 ))
#define IP_TIFF_MULTI_RES (watch_int_get( "TIFF_MULTI_RES", 0 ))

/* TIFF save possibilities.
 */
typedef enum {
	TIFF_COMPRESSION_NONE = 0,	/* No compression */
	TIFF_COMPRESSION_LZW,		/* Lempel-Ziv compression */
	TIFF_COMPRESSION_DEFLATE,	/* Zip (deflate) compression */
	TIFF_COMPRESSION_PACKBITS,	/* Packbits compression */
	TIFF_COMPRESSION_JPEG		/* JPEG compression */
} TiffCompressionType;

typedef enum {
	TIFF_LAYOUT_STRIP = 0,		/* Strip TIFF */
	TIFF_LAYOUT_TILE		/* Tiled TIFF */
} TiffLayoutType;

typedef enum {
	TIFF_MULTIRES_FLAT = 0,		/* Flat file */
	TIFF_MULTIRES_PYRAMID		/* Pyramidal TIFF */
} TiffMultiresType;

/* Keep a list of all filesels currently active ... we use this for refresh on
 * new file.
 */
static GSList *filesel_all = NULL;

static GtkFileSelection2Class *parent_class = NULL;

static const char *tiff_suffs[] = { ".tif", ".tiff", NULL };
static const char *jpeg_suffs[] = { ".jpg", ".jpeg", ".jpe", NULL };
static const char *png_suffs[] = { ".png", NULL };
static const char *ppm_suffs[] = { ".ppm", ".pgm", ".pbm", NULL };
static const char *icc_suffs[] = { ".icc", ".icm", NULL };
static const char *vips_suffs[] = { ".v", NULL };
static const char *workspace_suffs[] = { ".ws", NULL };
static const char *rec_suffs[] = { ".rec", NULL };
static const char *mor_suffs[] = { ".mor", NULL };
static const char *con_suffs[] = { ".con", NULL };
static const char *mat_suffs[] = { ".mat", NULL };
static const char *def_suffs[] = { ".def", NULL };
static const char *all_suffs[] = { "", NULL };

static GtkFileSelection2FileType
        filesel_tfile_type = 
		{ "TIFF image files (*.tif, *.tiff)", tiff_suffs },
        filesel_jfile_type = 
		{ "JPEG image files (*.jpg, *.jpeg, *.jpe)", jpeg_suffs },
        filesel_nfile_type = 
		{ "PNG image files (*.png)", png_suffs },
        filesel_vfile_type = 
		{ "VIPS image files (*.v)", vips_suffs },
        filesel_pfile_type = 
		{ "PPM image files (*.ppm, *.pgm, *.pbm)", ppm_suffs },
        filesel_wfile_type = 
		{ "Workspace files (*.ws)", workspace_suffs },
        filesel_rfile_type = 
		{ "Recombination matrix files (*.rec)", rec_suffs },
        filesel_mfile_type = 
		{ "Morphology matrix files (*.mor)", mor_suffs },
        filesel_cfile_type = 
		{ "Convolution matrix files (*.con)", con_suffs },
        filesel_xfile_type = 
		{ "Matrix files (*.mat)", mat_suffs },
        filesel_dfile_type = 
		{ "Definition files (*.def)", def_suffs },
        filesel_ifile_type = 
		{ "ICC profiles (*.icc, *.icm)", icc_suffs },
        filesel_allfile_type = 
		{ "All files (*)", all_suffs };

GtkFileSelection2FileType
        *filesel_type_definition[] = { 
		&filesel_dfile_type, &filesel_allfile_type, NULL 
	},
        *filesel_type_workspace[] = { 
		&filesel_wfile_type, &filesel_allfile_type, NULL 
	},
        *filesel_type_image[] = { 
		&filesel_vfile_type, &filesel_tfile_type, &filesel_jfile_type, 
		&filesel_pfile_type, &filesel_nfile_type, 
		&filesel_allfile_type, NULL 
	},
        *filesel_type_matrix[] = { 
		&filesel_xfile_type, &filesel_cfile_type, &filesel_rfile_type, 
		&filesel_mfile_type, &filesel_allfile_type, NULL 
	},
        *filesel_type_any[] = { 
		&filesel_vfile_type, &filesel_tfile_type, &filesel_jfile_type, 
		&filesel_pfile_type, &filesel_nfile_type,
		&filesel_xfile_type, &filesel_cfile_type, &filesel_rfile_type, 
		&filesel_mfile_type, 
		&filesel_ifile_type, 
		&filesel_allfile_type, NULL 
	};

/* Is a file of type ... just look at the suffix.
 */
static gboolean
is_file_type( GtkFileSelection2FileType *type, const char *filename )
{
	const char **p;
	const char *suf;

	if( (suf = strrchr( filename, '.' )) ) {
		for( p = type->suffixes; *p; p++ )
			if( strcasecmp( suf, *p ) == 0 )
				return( TRUE );
	}

	return( FALSE );
}

/* Map TIFF formats to char* for VIPS.
 */
static char *
decode_tiff_compression( TiffCompressionType tc )
{
	switch( tc ) {
	case TIFF_COMPRESSION_LZW:	return( "lzw" );
	case TIFF_COMPRESSION_DEFLATE:	return( "deflate" );
	case TIFF_COMPRESSION_PACKBITS:	return( "packbits" );
	case TIFF_COMPRESSION_JPEG:	return( "jpeg" );

	case TIFF_COMPRESSION_NONE:
	default:
		return( "none" );
	}
}

static char *
decode_tiff_layout( TiffLayoutType tf )
{
	switch( tf ) {
	case TIFF_LAYOUT_TILE:		return( "tile" );

	case TIFF_LAYOUT_STRIP:
	default:
		return( "strip" );
	}
}

static char *
decode_tiff_multires( TiffMultiresType tm )
{
	switch( tm ) {
	case TIFF_MULTIRES_PYRAMID:	return( "pyramid" );

	case TIFF_MULTIRES_FLAT:
	default:
		return( "flat" );
	}
}

/* Make a TIFF save format string.
 */
static void
filesel_tiff_mode( char *out )
{
	char ctype[ 256 ];
	char ltype[ 256 ];
	char buf[ 256 ];

	strcpy( ctype, decode_tiff_compression( IP_TIFF_COMPRESSION ) );
	if( IP_TIFF_COMPRESSION == TIFF_COMPRESSION_JPEG ) {
		im_snprintf( buf, 256, ":%d", IP_TIFF_JPEG_Q );
		strcat( ctype, buf );
	}

	strcpy( ltype, decode_tiff_layout( IP_TIFF_LAYOUT ) );
	if( IP_TIFF_LAYOUT == TIFF_LAYOUT_TILE ) {
		im_snprintf( buf, 256, ":%dx%d", 
			IP_TIFF_TILE_WIDTH,
			IP_TIFF_TILE_WIDTH );
		strcat( ltype, buf );
	}

	im_snprintf( out, 256, "%s,%s,%s", 
		ctype, ltype, decode_tiff_multires( IP_TIFF_MULTI_RES ) );
}

/* Make a JPEG save format string.
 */
static void
filesel_jpeg_mode( char *out )
{
	char buf[FILENAME_MAX];
	char buf2[FILENAME_MAX];

	im_strncpy( buf, IP_JPEG_ICC_PROFILE, FILENAME_MAX );
        expand_variables( buf, buf2 );
        nativize_path( buf2 );
	im_snprintf( out, 256, "%d,%s", IP_JPEG_Q, buf2 );
}

/* Make a PNG save format string.
 */
static void
filesel_png_mode( char *out )
{
	im_snprintf( out, 256, "%d,%d", IP_PNG_COMPRESSION, IP_PNG_INTERLACE );
}

/* Make a PPM save format string.
 */
static void
filesel_ppm_mode( char *out )
{
	switch( IP_PPM_MODE ) {
	case 0:
		im_snprintf( out, 256, "binary" );
		break;

	default:
		im_snprintf( out, 256, "ascii" );
		break;
	}
}

typedef void (*make_mode_fn)( char *buf );

typedef struct {
	GtkFileSelection2FileType *type;
	make_mode_fn mode_fn;
} FileselMode;

static FileselMode filesel_mode_table[] = {
	{ &filesel_jfile_type, filesel_jpeg_mode },
	{ &filesel_nfile_type, filesel_png_mode },
	{ &filesel_tfile_type, filesel_tiff_mode },
	{ &filesel_pfile_type, filesel_ppm_mode }
};

/* Add our image save settings to the end of a filename.
 */
void
filesel_add_mode( char *filename )
{
	int i;

	for( i = 0; i < IM_NUMBER( filesel_mode_table ); i++ )
		if( is_file_type( filesel_mode_table[i].type, filename ) ) {
			char mode[256];
			int l = strlen( filename );

			filesel_mode_table[i].mode_fn( mode );
			im_snprintf( filename + l, NAMELEN - l, ":%s", mode );
			return;
		}
}

static void
filesel_destroy( GtkObject *object )
{
	Filesel *filesel;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_FILESEL( object ) );

	filesel = FILESEL( object );

	/* My instance destroy stuff.
	 */
	filesel_all = g_slist_remove( filesel_all, filesel );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

/* Update `space free' label.
 */
static void
filesel_space_update( Filesel *filesel, const char *dirname )
{
	double sz = find_space( dirname );

	if( sz < 0 )
		set_glabel( filesel->space, 
			"unable to determine space free in \"%s\"", dirname );
	else {
		char txt[NAMELEN];
		BufInfo buf;

		buf_init_static( &buf, txt, NAMELEN );
		to_size( &buf, sz );
		buf_appendf( &buf, " free in \"%s\"", dirname );
		set_glabel( filesel->space, "%s", buf_all( &buf ) );
	}
}

/* New dir entered signal.
 */
static void
filesel_dir_enter_cb( GtkWidget *widget, const char *dirname, gpointer data )
{
	Filesel *filesel = FILESEL( widget );

#ifdef DEBUG
	printf( "filesel_dir_enter_cb: %s - \"%s\"\n", 
		NN( IWINDOW( filesel )->title ), dirname );
#endif /*DEBUG*/

	if( filesel->browse )
		browse_refresh( filesel->browse, dirname );
        filesel_space_update( filesel, dirname );

        /* Is this directory on session path? If not, add it.
         */
        path_add( dirname );
}

/* Update file info display.
 */
static void
filesel_info_update( Filesel *filesel, const char *name )
{
	char txt[MAX_STRSIZE];
	BufInfo buf;

	buf_init_static( &buf, txt, MAX_STRSIZE );
	get_image_info( &buf, name );
	set_glabel( filesel->info, "%s", buf_all( &buf ) );
}

gboolean 
filesel_set_filename( Filesel *filesel, const char *name )
{
	char buf[FILENAME_MAX];

	if( !is_valid_filename( name ) ) 
		return( FALSE );

        expand_variables( name, buf );
        nativize_path( buf );
        gtk_file_selection2_set_filename( GTK_FILE_SELECTION2( filesel ), buf );

	return( TRUE );
}

/* Read the filename out ... test for sanity.
 */
const char *
filesel_get_filename( Filesel *filesel )
{
	const char *name = gtk_file_selection2_get_filename(
		GTK_FILE_SELECTION2( filesel ) );
	
	if( !is_valid_filename( name ) ) 
		return( NULL );

	return( name );
}

/* Get filename multi ... map over the selected filenames.
 */
void *
filesel_map_filename_multi( Filesel *filesel,
	FileselMapFn fn, void *a, void *b )
{
	GSList *names = gtk_file_selection2_get_filename_multi( 
		GTK_FILE_SELECTION2( filesel ) );
	GSList *p;

	for( p = names; p; p = p->next ) {
		char *filename = (char *) p->data;
		void *res;

		if( (res = fn( filesel, filename, a, b )) ) {
			FREEF( slist_free_all, names );
			return( res );
		}
	}
	FREEF( slist_free_all, names );

	return( NULL );
}

/* New file selected signal.
 */
static void
filesel_file_select_cb( GtkWidget *widget, const char *filename, gpointer data )
{
	Filesel *filesel = FILESEL( widget );
        const char *name = filesel_get_filename( filesel );

#ifdef DEBUG
	printf( "filesel_file_select_cb: %s - \"%s\"\n", 
		NN( IWINDOW( filesel )->title ), NN( name ) );
#endif /*DEBUG*/

	if( name )
		filesel_info_update( filesel, name );
}

/* Refresh signal.
 */
static void
filesel_refresh_cb( GtkWidget *widget, gpointer data )
{
	Filesel *filesel = FILESEL( widget );
	char dir[PATHLEN];

#ifdef DEBUG
	printf( "filesel_refresh_cb: %s\n", NN( IWINDOW( filesel )->title ) );
#endif /*DEBUG*/

	if( filesel_get_dir( filesel, dir ) ) {
		filesel_space_update( filesel, dir  );

		if( filesel->browse )
			browse_refresh( filesel->browse, dir );
	}
}

/* Add to the history list.
 */
static void *
filesel_history_add( const char *dir, Filesel *filesel )
{
        GtkFileSelection2 *fs = GTK_FILE_SELECTION2( filesel );
	char buf[FILENAME_MAX];

        expand_variables( dir, buf );
        nativize_path( buf );
        gtk_file_selection2_update_history_menu( fs, buf );

        return( NULL );
}

/* Increment filename on OK.
 */
static void
filesel_auto_incr_cb( GtkWidget *tog, Filesel *filesel )
{
        filesel->incr = GTK_TOGGLE_BUTTON( tog )->active;
}

/* Popup a new browse window.
 */
/*ARGSUSED*/
static void
browse_cb( GtkWidget *wid, Filesel *filesel )
{
        if( !filesel->browse ) {
		char dir[PATHLEN];

                filesel->browse = BROWSE( browse_new() );
		browse_set_filesel( filesel->browse, filesel );
		iwindow_build( IWINDOW( filesel->browse ) );
		gtk_widget_show( GTK_WIDGET( filesel->browse ) );

		if( !filesel_get_dir( filesel, dir ) ) {
			box_alert( GTK_WIDGET( filesel ) );
			return;
		}

		browse_refresh( filesel->browse, dir );
	}
}

static void
filesel_build( GtkWidget *widget )
{
	Filesel *filesel = FILESEL( widget );
	GtkFileSelection2 *fs = GTK_FILE_SELECTION2( widget );
	iDialog *idlg = IDIALOG( widget );

	GtkWidget *hb;
	GtkWidget *but;
	GtkWidget *tog;

#ifdef DEBUG
	printf( "filesel_build: %s\n", NN( IWINDOW( filesel )->title ) );
#endif /*DEBUG*/

	/* Call all builds in superclasses.
	 */
	if( IWINDOW_CLASS( parent_class )->build )
		(*IWINDOW_CLASS( parent_class )->build)( widget );

        /* Turn on options.
         */
        gtk_file_selection2_set_file_types( fs, filesel->type );
        gtk_file_selection2_set_file_type( fs, filesel->default_type );
        gtk_file_selection2_show_file_types( fs );
        gtk_signal_connect( GTK_OBJECT( fs ), "dir_enter",
                GTK_SIGNAL_FUNC( filesel_dir_enter_cb ), NULL );
        gtk_signal_connect( GTK_OBJECT( fs ), "file_select",
                GTK_SIGNAL_FUNC( filesel_file_select_cb ), NULL );
        gtk_signal_connect( GTK_OBJECT( fs ), "refresh",
                GTK_SIGNAL_FUNC( filesel_refresh_cb ), NULL );

        /* Init history.
         */
        slist_map( path_session,
                (SListMapFn) filesel_history_add, filesel );
        slist_map( PATH_SEARCH,
                (SListMapFn) filesel_history_add, filesel );

        /* Layout our extra stuff.
         */
        gtk_container_set_border_width( GTK_CONTAINER( fs->work ), 5 );

        /* Space free label. 
         */
	filesel->space = gtk_label_new( "" );
	gtk_misc_set_alignment( GTK_MISC( filesel->space ), 0, 0.5 );
	gtk_box_pack_start( GTK_BOX( fs->work ), 
		filesel->space, FALSE, FALSE, 0 );
	gtk_widget_show( filesel->space );

        /* File info label.
         */
        filesel->info = gtk_label_new( "" );
        gtk_misc_set_alignment( GTK_MISC( filesel->info ), 0, 0.5 );
        gtk_box_pack_start( GTK_BOX( fs->work ), filesel->info, TRUE, TRUE, 0 );
        gtk_widget_show( filesel->info );

        hb = gtk_hbox_new( FALSE, 3 );
        gtk_box_pack_start( GTK_BOX( fs->work ), hb, TRUE, TRUE, 0 );
        gtk_widget_show( hb );

        /* Auto-increment toggle.
         */
	if( filesel->save ) {
		tog = gtk_check_button_new_with_label( "Increment filename" );
		gtk_signal_connect( GTK_OBJECT( tog ), "toggled",
			GTK_SIGNAL_FUNC( filesel_auto_incr_cb ), filesel );
		gtk_box_pack_start( GTK_BOX( hb ), tog, FALSE, FALSE, 0 );
		gtk_widget_show( tog );
		set_tooltip( tog, "After OK, add 1 to the last number in the "
			"file name" );
	}

        /* Browse thumbnails.
         */
        if( filesel->imls ) {
                but = gtk_button_new_with_label( "Show thumbnails" );
                gtk_box_pack_end( GTK_BOX( hb ), but, FALSE, FALSE, 0 );
                gtk_widget_show( but );
                gtk_signal_connect( GTK_OBJECT( but ), "clicked",
                        GTK_SIGNAL_FUNC( browse_cb ), filesel );
		set_tooltip( but, "Show thumbnails for files in this "
			"directory" );
        }

        hb = gtk_hbox_new( FALSE, 3 );
        gtk_box_pack_start( GTK_BOX( fs->work ), hb, TRUE, TRUE, 0 );
        gtk_widget_show( hb );

	/* If this is a load and there's no filename set, select the first one
	 * which matches the pattern.
	 */
	if( !filesel->save && !filesel_get_filename( filesel ) ) {
		GtkCList *file_list = GTK_CLIST( 
			GTK_FILE_SELECTION2( filesel )->file_list );

		if( file_list->row_list )
			gtk_clist_select_row( file_list, 0, 0 );
	}

	gtk_widget_show( idlg->work );

        gtk_file_selection2_refresh( fs );
}

static void
filesel_class_init( FileselClass *klass )
{
	GtkObjectClass *object_class;
	iWindowClass *iwindow_class;

	object_class = (GtkObjectClass *) klass;
	iwindow_class = (iWindowClass *) klass;

	object_class->destroy = filesel_destroy;
	iwindow_class->build = filesel_build;

	parent_class = gtk_type_class( GTK_TYPE_FILE_SELECTION2 );
}

/* Get the current name, no dir.
 */
const char *
filesel_get_name( Filesel *filesel )
{
        const char *name = filesel_get_filename( filesel );
        char *p;

        if( name && (p = strrchr( name, IM_DIR_SEP )) )
                return( p + 1 );
        else
                return( name );
}

/* Get the current directory. buf should be at least PATHLEN in size.
 */
char *
filesel_get_dir( Filesel *filesel, char *buf )
{
	const char *name = gtk_file_selection2_get_filename(
		GTK_FILE_SELECTION2( filesel ) );
        char *p;

	if( !name )
		return( NULL );

	/* If a dir is selected, we'll have a trailing IM_DIR_SEP, so this is
	 * safe.
	 */
        im_strncpy( buf, name, PATHLEN );
        if( (p = strrchr( buf, IM_DIR_SEP )) )
                *p = '\0';

        return( buf );
}

/* Increment filename. If there's no number there now, assume zero.
 */
static void
filesel_increment_filename( Filesel *filesel )
{
        const char *name = filesel_get_name( filesel );
        char buf[ NAMELEN ];
        char buf2[ NAMELEN ];
        const char *suflist[100];     /* Max no of suffs ... should be OK */
        int nsuffs;
        char *p;
        int i, j, n;

        /* Get the current filename.
         */
	if( !name )
		return;
        im_snprintf( buf2, NAMELEN, "%s", name );

        /* Make list of all suffixes in set of file types.
         */
        for( nsuffs = 0, i = 0; filesel->type[i]; i++ )
                for( j = 0; filesel->type[i]->suffixes[j]; j++, nsuffs++ )
                        suflist[nsuffs] = filesel->type[i]->suffixes[j];

        /* Drop any existing suffixes.
         */
        change_suffix( buf2, buf, "", (const char **) suflist, nsuffs );

        /* Look for any number at the end of the filename, read them, and
         * zap them.
         */
        if( (p = (char *) my_strrspn( buf, NUMERIC )) ) {
                /* There are numbers ... read into n.
                 */
                n = atoi( p );
                *p = '\0';
        }
        else
                /* No numbers .. assume 0.
                 */
                n = 0;

        /* Increment the number, and append. Put the suffix back as well.
         */
        n++;
        im_snprintf( buf2, NAMELEN, "%s%d%s", buf, n, 
		gtk_file_selection2_get_file_type( 
			GTK_FILE_SELECTION2( filesel ) ) );

        (void) filesel_set_filename( filesel, buf2 );
}

static void *
filesel_refresh( Filesel *filesel )
{
        gtk_file_selection2_refresh( GTK_FILE_SELECTION2( filesel ) );

        return( NULL );
}

/* There may be a new file ... ask all fsb's to refresh.
 */
void
filesel_refresh_all( void )
{
        (void) slist_map( filesel_all, (SListMapFn) filesel_refresh, NULL );
}

/* Back from the user function ... unset the hourglass, and update.
 */
static void
filesel_trigger2( void *sys, iWindowResult result )
{
	iWindowSusp *susp = (iWindowSusp *) sys;
	Filesel *filesel = FILESEL( susp->client );

	/* Back from the user function ... restore the pointer.
	 */
	set_pointer();

	/* If this is a save, assume that there is now a new file, 
	 * and ask all fsb's to update.
	 */
	if( filesel->save && result != IWINDOW_ERROR )
		filesel_refresh_all();

	if( result != IWINDOW_TRUE ) {
		/* Failure ... bomb out.
		 */
		iwindow_susp_return( susp, result );
		return;
	}

	/* Increment the filename, if required.
	 */
	if( filesel->incr ) {
		filesel_increment_filename( filesel );
		filesel_refresh( filesel );
	}

	/* Success!
	 */
	iwindow_susp_return( susp, result );
}

/* Start of user done ... shut down our suspension, and set the hglass.
 */
static void
filesel_trigger( iWindow *iwnd, Filesel *filesel, 
	iWindowNotifyFn nfn, void *sys )
{
	/* Suspend the callback for a bit.
	 */
	iWindowSusp *susp = iwindow_susp_new( NULL, iwnd, filesel, nfn, sys );

	set_hourglass();
	filesel->done_cb( IWINDOW( filesel ), 
		filesel->client, filesel_trigger2, susp );
}

/* OK in "file exists" yesno ... trigger the user callback.
 */
static void
filesel_done_yesno( iWindow *iwnd, 
	void *client, iWindowNotifyFn nfn, void *sys )
{
	Filesel *filesel = FILESEL( client );

	filesel_trigger( iwnd, filesel, nfn, sys );
}

/*ARGSUSED*/
static void
filesel_help_cb( iWindow *iwnd, void *client, iWindowNotifyFn nfn, void *sys )
{
	box_help( GTK_WIDGET( iwnd ), "sec:loadsave" );

	nfn( sys, IWINDOW_TRUE );
}

static void
filesel_init( Filesel *filesel )
{
#ifdef DEBUG
	printf( "filesel_init: %s\n", NN( IWINDOW( filesel )->title ) );
#endif /*DEBUG*/

	filesel->incr = FALSE;
	filesel->imls = FALSE;
	filesel->save = FALSE;
	filesel->browse = NULL;
	filesel->type = NULL;
	filesel->default_type = 0;
	filesel->done_cb = NULL;
	filesel->client = NULL;

	idialog_set_callbacks( IDIALOG( filesel ), 
		iwindow_true_cb, filesel_help_cb, NULL, NULL, NULL );
	idialog_set_nosep( IDIALOG( filesel ), TRUE );
	idialog_set_pinup( IDIALOG( filesel ), TRUE );

	filesel_all = g_slist_prepend( filesel_all, filesel );
}

GtkType
filesel_get_type( void )
{
	static GtkType type = 0;

	if( !type ) {
		static const GtkTypeInfo info = {
			"Filesel",
			sizeof( Filesel ),
			sizeof( FileselClass ),
			(GtkClassInitFunc) filesel_class_init,
			(GtkObjectInitFunc) filesel_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		type = gtk_type_unique( GTK_TYPE_FILE_SELECTION2, &info );
	}

	return( type );
}

GtkWidget *
filesel_new( void )
{
	Filesel *filesel = (Filesel *) gtk_type_new( TYPE_FILESEL );

	return( GTK_WIDGET( filesel ) );
}

void 
filesel_set_done( Filesel *filesel, iWindowFn done_cb, void *client )
{
	filesel->done_cb = done_cb;
	filesel->client = client;
}

static void
filesel_done_cb( iWindow *iwnd, void *client, iWindowNotifyFn nfn, void *sys )
{
	Filesel *filesel = FILESEL( iwnd );
        const char *name = filesel_get_filename( filesel );

#ifdef DEBUG
	printf( "filesel_done\n" );
#endif /*DEBUG*/

	if( !name ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	/* File exists and we are saving? Do a yesno before we carry on.
	 */
	if( filesel->save && existsf( "%s", name ) ) {
		box_yesno( GTK_WIDGET( filesel ), 
			filesel_done_yesno, iwindow_true_cb, filesel, 
			nfn, sys, 
			"Overwrite",
			"file \"%s\" exists.\n"
			"OK to overwrite?", name );
	}
	else
		/* Just call the user function directly.
		 */
		filesel_trigger( iwnd, filesel, nfn, sys );
}

void 
filesel_set_flags( Filesel *filesel, gboolean imls, gboolean save )
{
	filesel->imls = imls;
	filesel->save = save;

	idialog_add_ok( IDIALOG( filesel ), filesel_done_cb, 
		save ? "Save" : "Load" );
}

void 
filesel_set_type( Filesel *filesel, 
	GtkFileSelection2FileType **type, int default_type )
{
	filesel->type = type;
	filesel->default_type = default_type;
}

