/* an image class object in a workspace
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

#include "ip.h"

static ClassmodelClass *parent_class = NULL;

static void
colour_destroy( GtkObject *object )
{
	Colour *colour;

#ifdef DEBUG
	printf( "colour_destroy\n" );
#endif /*DEBUG*/

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_COLOUR( object ) );

	colour = COLOUR( object );

	/* My instance destroy stuff.
	 */
	FREE( colour->colour_space );
	buf_destroy( &colour->caption );
	FREESID( colour->ii_area_changed_sid, colour->ii );
	FREESID( colour->ii_destroy_sid, colour->ii );
	FREEF( imageinfo_destroy_nonheap, colour->ii );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

static void 
colour_edit( GtkWidget *parent, Model *model )
{
        Colour *colour = COLOUR( model );

	imageinfo_colour_edit( parent, colour->ii );
}

static xmlNode *
colour_save( Model *model, xmlNode *xnode )
{
        Colour *colour = COLOUR( model );

	xmlNode *xthis;

	if( !(xthis = MODEL_CLASS( parent_class )->save( model, xnode )) )
		return( NULL );

	if( CLASSMODEL( model )->edited ) {
		if( !set_prop( xthis, "colour_space", "%s",
			colour->colour_space ) ||
			!set_prop( xthis, "value0", "%g", colour->value[0] ) ||
			!set_prop( xthis, "value1", "%g", colour->value[1] ) ||
			!set_prop( xthis, "value2", "%g", colour->value[2] ) )
			return( NULL );
	}

	return( xthis );
}

/* Set of allowed colour_space strings. Do a case-insensitive match.
 */
static const char *colour_colour_space[] = {
	"xyz", 		/* index 0 */
	"yxy", 		/* index 1 */
	"lab", 		/* index 2 */
	"lch", 		/* index 3 */
	"ucs", 		/* index 4 */
	"rgb", 		/* index 5 */
	"srgb"		/* index 6 */
};

/* For each allowed colourspace, the corresponding VIPS Type value.
 */
static const int colour_type[] = {
	12, 		/* IM_TYPE_XYZ */
	23, 		/* IM_TYPE_YXY */
	13, 		/* IM_TYPE_LAB */
	19, 		/* IM_TYPE_LCH */
	18, 		/* IM_TYPE_UCS */
	17, 		/* IM_TYPE_RGB */
	22 		/* IM_TYPE_sRGB */
};

/* Imageinfo area has changed signal.
 */
static void
colour_ii_area_changed_cb( Imageinfo *imageinfo, Rect *dirty, Colour *colour )
{
	Classmodel *classmodel = CLASSMODEL( colour );
	int i;

	for( i = 0; i < 3; i++ )
		colour->value[i] = ((float *) colour->ii->im->data)[i];

	/* Rebuild object.
	 */
	classmodel_update( classmodel );
	symbol_recalculate_all();
}

/* Imageinfo has been destroyed signal.
 */
static void
colour_ii_destroy_cb( Imageinfo *imageinfo, Colour *colour )
{
	assert( colour->ii_area_changed_sid );

	colour->ii_area_changed_sid = 0;
	colour->ii_destroy_sid = 0;
	colour->ii = NULL;
}

/* Refresh the 1 pixel image and the caption.
 */
static void
colour_refresh( Colour *colour )
{
	int i;

#ifdef DEBUG
	printf( "colour_refresh: " );
	row_name_print( HEAPMODEL( colour )->row );
	printf( " space=\"%s\", value0=%g, value1=%g, value2=%g\n",
		NN( colour->colour_space ),
		colour->value[0], colour->value[1], colour->value[2] );
#endif /*DEBUG*/

	/* Look up the colour_space string and try to set the type 
	 * appropriately.
	 */
	if( colour->colour_space ) {
		for( i = 0; i < IM_NUMBER( colour_colour_space ); i++ )
			if( strcasecmp( colour->colour_space, 
				colour_colour_space[i] ) == 0 ) 
				break;
	}
	else
		i = -1;

	if( i != IM_NUMBER( colour_colour_space ) )
		colour->type = colour_type[i];
	else
		colour->type = -1;

	FREESID( colour->ii_area_changed_sid, colour->ii );
	FREESID( colour->ii_destroy_sid, colour->ii );
	FREEF( imageinfo_destroy_nonheap, colour->ii );

	if( (colour->ii = imageinfo_new_temp( reduce_context->hi, "t" )) ) {
		/* Make a 3 band 32-bit FLOAT memory image.
		 */
		imageinfo_dup_nonheap( colour->ii );

		im_initdesc( colour->ii->im, 1, 1, 3, 
			IM_BBITS_FLOAT, IM_BANDFMT_FLOAT, 
			IM_CODING_NONE, colour->type, 
			1.0, 1.0, 0, 0 );
		if( im_setupout( colour->ii->im ) )
			FREEF( imageinfo_destroy_nonheap, colour->ii );
	}

	if( colour->ii ) {
		colour->ii->im->Type = colour->type;
		for( i = 0; i < 3; i++ )
			((float *) colour->ii->im->data)[i] = colour->value[i];
	}

	buf_rewind( &colour->caption );
	buf_appendf( &colour->caption, CLASS_COLOUR " %s [%g, %g, %g]",
		NN( colour->colour_space ),
		colour->value[0], colour->value[1], colour->value[2] );

	if( colour->ii ) {
		colour->ii_area_changed_sid = 
			gtk_signal_connect( GTK_OBJECT( colour->ii ), 
			"area_changed", 
			GTK_SIGNAL_FUNC( colour_ii_area_changed_cb ), colour );
		colour->ii_destroy_sid = 
			gtk_signal_connect( GTK_OBJECT( colour->ii ), 
			"destroy", 
			GTK_SIGNAL_FUNC( colour_ii_destroy_cb ), colour );
	}
}

static gboolean
colour_load( Model *model, 
	ModelLoadState *state, Model *parent, xmlNode *xnode )
{
        Colour *colour = COLOUR( model );
	char colour_space[NAMELEN];

#ifdef DEBUG
	printf( "colour_load: " );
	row_name_print( HEAPMODEL( colour )->row );
	printf( "\n" );
#endif /*DEBUG*/

	if( !IS_RHS( parent ) ) {
		ierrors( "colour_load: can only add an colour to a rhs" );
		return( FALSE );
	}

	if( get_sprop( xnode, "colour_space", colour_space, NAMELEN ) &&
		get_dprop( xnode, "value0", &colour->value[0] ) &&
		get_dprop( xnode, "value1", &colour->value[1] ) &&
		get_dprop( xnode, "value2", &colour->value[2] ) ) {
		SETSTR( colour->colour_space, colour_space );
		colour_refresh( colour );
		classmodel_set_edited( CLASSMODEL( model ), TRUE );
	}

	return( MODEL_CLASS( parent_class )->load( model, 
		state, parent, xnode ) );
}

/* Update Colour from heap.
 */
static gboolean
colour_class_get( Classmodel *classmodel, PElement *root )
{
        Colour *colour = COLOUR( classmodel );
	char colour_space[NAMELEN];
	double value[3];
	int l;
	int i;

#ifdef DEBUG
	printf( "colour_class_get: " );
	row_name_print( HEAPMODEL( colour )->row );
	printf( "\n" );
#endif /*DEBUG*/

	if( !class_get_member_string( root, MEMBER_COLOUR_SPACE, 
		colour_space, NAMELEN ) ||
		!class_get_member_realvec( root, MEMBER_VALUE, value, 3, &l ) )
		return( FALSE );
	if( l != 3 ) {
		ierrors( "colour_class_get: 3 band value only" );
		return( FALSE );
	}
	SETSTR( colour->colour_space, colour_space );
	for( i = 0; i < 3; i++ )
		colour->value[i] = value[i];

	colour_refresh( colour );

	return( CLASSMODEL_CLASS( parent_class )->class_get( 
		classmodel, root ) );
}

/* Make a new "fn value" application.
 */
static gboolean
colour_class_new( Classmodel *classmodel, PElement *fn, PElement *out )
{
	Heap *hi = reduce_context->hi;
	Colour *colour = COLOUR( classmodel );
	PElement rhs;

#ifdef DEBUG
	printf( "colour_class_new: " );
	row_name_print( HEAPMODEL( colour )->row );
	printf( "\n" );
#endif /*DEBUG*/

	/* Make application nodes.
	 */
	heap_appl_init( out, fn );
	if( !heap_appl_add( hi, out, &rhs ) ||
		!heap_string_new( hi, colour->colour_space, &rhs ) ||
		!heap_appl_add( hi, out, &rhs ) ||
		!heap_realvec_new( hi, 3, colour->value, &rhs ) )
		return( FALSE );

	return( TRUE );
}

static void
colour_class_init( ColourClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	ModelClass *model_class = (ModelClass *) klass;
	ClassmodelClass *classmodel_class = (ClassmodelClass *) klass;

	parent_class = gtk_type_class( TYPE_CLASSMODEL );

	object_class->destroy = colour_destroy;

	/* Create signals.
	 */

	/* Init methods.
	 */
	model_class->view_new = colourview_new;
	model_class->edit = colour_edit;
	model_class->save = colour_save;
	model_class->load = colour_load;

	classmodel_class->class_get = colour_class_get;
	classmodel_class->class_new = colour_class_new;

	/* Static init.
	 */
	model_register_loadable( MODEL_CLASS( klass ) );
}

static void
colour_init( Colour *colour )
{
	colour->value[0] = 0.0;
	colour->value[1] = 0.0;
	colour->value[2] = 0.0;
	colour->colour_space = NULL;
	colour->type = -1;
	buf_init_dynamic( &colour->caption, MAX_LINELENGTH );
	colour->ii = NULL;
	colour->ii_area_changed_sid = 0;
	colour->ii_destroy_sid = 0;

	model_set( MODEL( colour ), CLASS_COLOUR, NULL );
}

GtkType
colour_get_type( void )
{
	static GtkType colour_type = 0;

	if( !colour_type ) {
		static const GtkTypeInfo info = {
			"Colour",
			sizeof( Colour ),
			sizeof( ColourClass ),
			(GtkClassInitFunc) colour_class_init,
			(GtkObjectInitFunc) colour_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		colour_type = gtk_type_unique( TYPE_CLASSMODEL, &info );
	}

	return( colour_type );
}

Classmodel *
colour_new( Rhs *rhs )
{
	Colour *colour = gtk_type_new( TYPE_COLOUR );

	model_child_add( MODEL( rhs ), MODEL( colour ), -1 );

	return( CLASSMODEL( colour ) );
}
