/**************************************************************************/
/*                                N I C E                                 */
/*             A high-level object-oriented research language             */
/*                        (c) Daniel Bonniot 2002                         */
/*                                                                        */
/*  This program is free software; you can redistribute it and/or modify  */
/*  it under the terms of the GNU General Public License as published by  */
/*  the Free Software Foundation; either version 2 of the License, or     */
/*  (at your option) any later version.                                   */
/*                                                                        */
/**************************************************************************/

package bossa.syntax;

import java.util.*;
import bossa.util.*;

import mlsub.typing.Polytype;
import mlsub.typing.Monotype;
import mlsub.typing.FunType;
import mlsub.typing.Constraint;

/**
   A functional abstraction expression.

   @version $Date: 2004/02/25 11:23:29 $
   @author Daniel Bonniot (d.bonniot@mail.dotcom.fr)
*/
public class FunExp extends Expression implements Function
{
  public FunExp(bossa.syntax.Constraint cst, List formals, Statement body)
  {
    this.formals = (MonoSymbol[]) 
      formals.toArray(new MonoSymbol[formals.size()]);
    
    this.constraint = cst;
    this.body = body;
  }

  FunExp(bossa.syntax.Constraint cst, MonoSymbol[] formals, Statement body)
  {
    this.formals = formals;
    this.constraint = cst;
    this.body = body;
    setLocation(body.location());
  }

  public mlsub.typing.Monotype getExpectedType()
  {
    return null;
  }

  public void checkReturnedType(mlsub.typing.Polytype returned)
    throws Function.ReturnTypeError
  {
    if (inferredReturnType == null)
      inferredReturnType = returned;
    else
      {
	Polytype old = inferredReturnType;
	inferredReturnType = Polytype.union(inferredReturnType, returned);

	if (! inferredReturnType.trySimplify())
	  throw new FunExp.IncompatibleReturnType(old);
      }

    /* This is disabled, since currently default values of class fields are
       typechecked twice.
    if (type != null)
      Internal.error("Return statement discovered after computation of the type");
    */
  }

  private boolean alwaysReturns;
  void setAlwaysReturns(boolean value) { alwaysReturns = value; }

  void computeType()
  {
    if (inferredReturnType == null)
      // There is not return statement in the function.
      {
        if (alwaysReturns)
          /* This function never returns normally since there was no 
             inferredReturnType, so it always throws an exception.
             Therefore, it can be considered to return any type.
          */
          inferredReturnType = Polytype.bottom();
        else
          inferredReturnType = PrimitiveType.voidPolytype;
      }
    else
      if (! alwaysReturns && 
	  ! nice.tools.typing.Types.isVoid(inferredReturnType))
	throw User.error(this, "Missing return statement");

    Monotype t = new FunType(MonoSymbol.getMonotype(formals), 
			     inferredReturnType.getMonotype());
    type = new Polytype
      (Constraint.and(cst, inferredReturnType.getConstraint()),
       bossa.syntax.Monotype.sure(t));
  }

  private Polytype inferredReturnType;
  Polytype inferredReturnType()
  {
    getType();
    return inferredReturnType;
  }

  /****************************************************************
   * Code generation
   ****************************************************************/

  public gnu.expr.Expression compile()
  {
    gnu.expr.LambdaExp res = nice.tools.code.Gen.createMethod
      (null, 
       nice.tools.code.Types.javaType(MonoSymbol.getMonotype(formals)), 
       nice.tools.code.Types.javaType(inferredReturnType()), 
       formals, false);
    nice.tools.code.Gen.setMethodBody(res, body.generateCode());
    return res;
  }
  
  /****************************************************************
   * Printing
   ****************************************************************/
  
  public String toString()
  {
    return 
      (constraint == null ? 
       mlsub.typing.Constraint.toString(cst) : constraint.toString())
      + "("
      + Util.map("",", ","",formals)
      + ") => "
      + (body instanceof ReturnStmt ? 
	 ((ReturnStmt) body).value.toString() : body.toString())
      ;
  }
  
  MonoSymbol[] formals;
  bossa.syntax.Constraint constraint;
  Constraint cst;
  Statement body;
  boolean mightEscape = true;
}
