/* $Id: newsflash.c,v 1.6 1996/08/07 00:41:01 chris Exp $ */

/* $Log: newsflash.c,v $
 * Revision 1.6  1996/08/07  00:41:01  chris
 * fixed a silly bug with long options
 *
 * Revision 1.5  1995/11/12  00:05:52  chris
 * Introduced 'servernick' in serverlist file.  Now used to prepend any
 * output instead of the FQDN.
 *
 * Revision 1.4  1995/10/23  22:47:49  chris
 * updated help message to show settings from config.h
 *
 * Revision 1.3  1995/10/02  01:37:04  chris
 * Modified some verbosity checks.
 *
 * Revision 1.2  1995/10/01  20:53:59  chris
 * Added help function.  Made line distinction in serverlist
 * file independent from host FQDN, so several lines containing
 * the same hostname should now work nicely.
 *
 * Revision 1.1  1995/10/01  18:53:17  chris
 * Initial revision
 * */

#include <getopt.h>
#include <sys/wait.h>
#include <waitstatus.h>

#include "newsflash.h"
#include "spool.h"

static char rcsid[]="$Id: newsflash.c,v 1.6 1996/08/07 00:41:01 chris Exp $";

static char serverlist[PATHSTRING];
static time_t startuptime;

#ifndef DEBUG
int verbosity=DEFAULT_VERBOSITY;
#else
int verbosity=99;
#endif

void say_hello(void)
{
  printf("Newsflash version %s, %s, by Chris Blum <chris@phil.uni-sb.de>\n", VERSION, DATE);
}

void show_help(void)
{
  say_hello();
  printf("       --- Fastest news retriever in the west ---\n\n"
	 "  newsflash [-v[#]] [-V] [-s file] [-l host] [-p#]\n\n"
	 "  -v[#]    verbosity (0-3, default is %d, default for -v only: %d)\n"
	 "  -V       show version\n"
	 "  -s file  serverlist file (default: " SERVERLIST ")\n"
	 "  -l host  name of host to spool articles to (default: %s)\n"
	 "  -p#      port on which to offer articles (default: %d)\n",
	 DEFAULT_VERBOSITY, DEFAULT_HIGH_VERBOSITY,LOCALHOST,LOCALPORT);
}


int update_serverlist(char *filename, struct serverS *srv)
{
  FILE *f,*g;
  struct serverS *s;
  char line[VERYLONGSTRING];
  int linecount;

  f=fopen(filename,"r+");
  if (!f) {
    fprintf(stderr, "Can't open %s for read & write\n",filename);
    return 1;
  }
  g=tmpfile();
  if (!g) {
    fprintf(stderr, "Oops, can't create temporary file!\n");
    return 1;
  }
  linecount=-1;
  while (fgets(line,VERYLONGSTRING,f)) {
    linecount++;
    if (*line=='#' || *line=='\n') fputs(line,g);
    else {
      for (s=srv; s; s=s->next) {
	if (s->lineinfile==linecount) break;
      }
      if (!s) {
	fprintf(stderr,"Oops, couldn't find entry for %s in serverlist\n",s->nickname);
      }
      else if (s->error) {
	if (verbosity>=3) if (s->error) printf("%s exited with error %d\n",s->nickname,s->error);
	fputs(line,g);
      }
      else {
	fprintf(g,"%s\t%s\t%d\t%s\t%ld\n", s->nickname, s->servername, s->port,
		s->grouplist, startuptime-TIMEGAUGE);
	if (verbosity>=3) printf("%s: updating entry\n",s->nickname);
      }
    }
  }
  rewind(f);
  rewind(g);
  ftruncate(fileno(f),0);
  while (fgets(line,VERYLONGSTRING,g)) fputs(line,f);
  fclose(f);
  fclose(g);
  return 0;
}

struct serverS *read_serverlist(char *filename)
{
  FILE *f;
  char line[VERYLONGSTRING];
  int linecount;
  struct serverS *anchor = NULL, *o = NULL, *n;
  char *c, *d;

  f=fopen(filename, "r");
  if (!f) {
    fprintf(stderr, "Can't open %s for read access\n", filename);
    return NULL;
  }
  linecount=-1;
  while (fgets(line,VERYLONGSTRING,f)) {
    linecount++;
    if (*line=='#' || isspace(*line)) continue;
    n=malloc(sizeof(struct serverS));
    if (!n) {
      fprintf(stderr,"Oops, not enough memory!\n");
      return NULL;
    }
    bzero(n,sizeof(struct serverS));
    for (c=line,d=n->nickname; *c&&!isspace(*c); c++,d++) *d=*c;
    *d=0;
    while (isspace(*c)) c++;
    for (d=n->servername; *c&&!isspace(*c); c++,d++) *d=*c;
    *d=0;
    while (isspace(*c)) c++;
    n->port=atoi(c);
    while (*c&&!isspace(*c)) c++;
    while (isspace(*c)) c++;
    for (d=n->grouplist; *c&&!isspace(*c); c++,d++) *d=*c;
    *d=0;
    while (isspace(*c)) c++;
    n->lasttime=atoi(c);
    n->next=NULL;
    n->lineinfile=linecount;
    if (!anchor) anchor=n;
    else o->next=n;
    o=n;
  }
  fclose(f);
  return anchor;
}

void start_spool(struct serverS *s)
{
  pid_t pid;

#ifdef DEBUG
  printf("starting spool for %s\n",s->nickname);
#endif
  pid=fork();
  if (pid<0) {
    fprintf(stderr,"Can't fork control process for %s\n",s->nickname);
    s->error=1;
    return;
  }
  if (!pid) {
    exit(spool(s));
  }
  else s->control_pid=pid;
}


int process_serverlist(char *filename)
{
  struct serverS *slist, *s;
  int x;

  time(&startuptime);
  slist=read_serverlist(filename);
  if (!slist) {
    printf("No servers found.\n");
    return 0;
  }
  for (s=slist; s; s=s->next) {
    s->error=0;
    start_spool(s);
#ifdef WELLBEHAVEDKIDS
    if (s->next) sleep(2);
#endif
  }
  for (s=slist; s; s=s->next) {
    waitpid(s->control_pid,&x,0);
    if (WIFEXITED(x)) {
      s->error=WEXITSTATUS(x);
    }
    else if (WIFSIGNALED(x)) {
      x=WTERMSIG(x);
      s->error=100+x;
    }
    else {
#ifdef DEBUG
      printf("Bummer!  Unknown termination of child process, code %d\n",x);
#endif
      s->error=0;
    }
#ifdef DEBUG
    if (s->error) printf("%s: returned with error code %d\n",s->nickname, s->error);
#endif
  }
  update_serverlist(filename, slist);
  return 0;
}


void process_arguments(int argc, char **argv)
{
  static struct option long_options[] = {
    {"serverlist",1,NULL,'s'},
    {"localhost",1,NULL,'l'},
    {"localport",1,NULL,'p'},
    {"help",0,NULL,'h'},
    {"version",0,NULL,'V'},
    {NULL,0,NULL,0}
  };
  int c;
  int option_index;

#ifndef DEBUG
  opterr = 0;
#endif

  /* defaults */
  strcpy(serverlist,SERVERLIST);
  strcpy(localhost,LOCALHOST);
  localport=LOCALPORT;

  while ((c = getopt_long(argc,argv,"s:l:p::hv::V",long_options,&option_index)) != -1) {
    switch (c) {
    case 'v':
      if (optarg) verbosity=atoi(optarg);
      else verbosity=DEFAULT_VERBOSITY;
      break;
    case 'V':
      say_hello();
      exit(0);
      break;
    case 'h':
      show_help();
      exit(0);
    case 's':
      if (optarg) strcpy(serverlist,optarg);
      else fprintf(stderr,"Ignoring option -s without argument\n");
      break;
    case 'l':
      if (optarg) strcpy(localhost,optarg);
      else fprintf(stderr,"Ignoring option -l without argument\n");
      break;
    case 'p':
      if (optarg) localport=atoi(optarg);
      else fprintf(stderr,"Ignoring option -p without argument\n");
      break;
    case '?':
      fprintf(stderr,"Ignoring unknown option '%c'; --help for help\n",optopt);
      break;
    default:
      fprintf(stderr,"Oops, got unknown option code 0x%X\n",c);
      break;
    } /* switch (c) */
  } /* while (getopt_long() != -1) */
}




int main(int argc, char **argv)
{
  process_arguments(argc, argv);
  return process_serverlist(serverlist);
}





