/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#ifndef ASN1CERT_H
#define ASN1CERT_H

#include <mVector.h>
#include <ASN1/NewPKIObject.h>
#include <openssl/asn1.h>
#include <openssl/x509.h>
#include <openssl/asn1.h>
#include <openssl/asn1t.h>
#include <openssl/x509v3.h>
#include <openssl/evp.h>
#include <openssl/pem.h>
#include <openssl/ossl_typ.h>
#include "Asn1Helper.h"
#include "Asn1Err.h"
#include <PKI_CERT.h>
#include <PKI_CSR.h>
#include <PKI_CRL.h>
#include <PKI_PKCS12.h>
#include <PKI_P7B.h>
#include <mVector.h>
#include <mString.h>

/*!
  This enumeration defines the possible states of a certificate
*/
typedef enum en_CERT_STATE
{
	CERT_STATE_VALID = 1,		//!< Certificate is valid
	CERT_STATE_REVOKED = 2,		//!< Certificate is revoked
	CERT_STATE_SUSPENDED = 3,	//!< Certificate is suspended
	CERT_STATE_UNKNOWN = 4		//!< Unknown state
}CERT_STATE;



/*!
	This structure represents a certificat request for a PKCS#12 generation.
*/
typedef struct st_NEWPKI_REQUEST_CERT_BODY_P12
{
	ASN1_INTEGER * keylen;			//!< The length of the key to generate
	ASN1_UTF8STRING * passwd; 		//!< The PKCS#12 password
}NEWPKI_REQUEST_CERT_BODY_P12;
DECLARE_ASN1_ITEM(NEWPKI_REQUEST_CERT_BODY_P12)

#define REQUEST_CERT_BODY_PKCS10	0	//!< Requesting a certificate from a PKCS#10
#define REQUEST_CERT_BODY_PKCS12	1	//!< Requesting a certificate for a PKCS#12

/*!
	This structure represents the body of a certificat request.
*/
typedef struct st_NEWPKI_REQUEST_CERT_BODY
{
	int type;
	union
	{
		X509_REQ * pkcs10;								//!< The PKCS#10 certificate request
		NEWPKI_REQUEST_CERT_BODY_P12 * p12_privdatas;	//!< The info to generate the PKCS#12
		
	}d;
}NEWPKI_REQUEST_CERT_BODY;
DECLARE_ASN1_ITEM(NEWPKI_REQUEST_CERT_BODY)

#define REQUEST_CERT_FLAG_SEND_ADMIN_NOTIFICATION		1	//!< This flag is used to request that the request creator admin should be notified when the RA gets the response back from the CA
#define REQUEST_CERT_FLAG_SEND_CERT_TO_USER				2	//!< This flag is used to request that the certificate/PKCS#12 should be sent to the end-user
#define REQUEST_CERT_FLAG_SEND_P12_PASSWD_TO_USER		4	//!< This flag is used to request that the RA sends the PKCS#12 password to the end-user
#define REQUEST_CERT_FLAG_RA_GENERATE_P12_PASSWD		8	//!< This flag is used to request that the RA generates the PKCS#12 password

/*!
	[RA] This structure represents a certificat request.
*/
typedef struct st_NEWPKI_REQUEST_CERT
{
	ASN1_INTEGER * profile_id;			//!< The owner profile
	ASN1_UTF8STRING * ca_name;			//!< The name of the CA
	NEWPKI_REQUEST_CERT_BODY * request;	//!< The request body
	ASN1_INTEGER * validity;			//!< The number of days the certificate should be valid for
	ASN1_INTEGER * type;				//!< The certificate type, see NEWPKI_PROFILE_CERT_TYPE_PKCS10
	ASN1_BIT_STRING * flags;			//!< The flags, see REQUEST_CERT_FLAG_SEND_ADMIN_NOTIFICATION
}NEWPKI_REQUEST_CERT;
DECLARE_ASN1_ITEM(NEWPKI_REQUEST_CERT)

/*!
	[EE] This structure represents a certificat request.
*/
typedef struct st_NEWPKI_EE_REQUEST_CERT
{
	ASN1_INTEGER * profile_id;			//!< The owner profile
	ASN1_UTF8STRING * ca_name;			//!< The name of the CA
	NEWPKI_REQUEST_CERT_BODY * request;	//!< The request body
	ASN1_INTEGER * type;				//!< The certificate type, see NEWPKI_PROFILE_CERT_TYPE_PKCS10
}NEWPKI_EE_REQUEST_CERT;
DECLARE_ASN1_ITEM(NEWPKI_EE_REQUEST_CERT)


/*!
	[EE] This structure represents a certificat revocation.
*/
typedef struct st_NEWPKI_EE_REQUEST_REV
{
	ASN1_INTEGER * cert_id;			//!< The owner profile
}NEWPKI_EE_REQUEST_REV;
DECLARE_ASN1_ITEM(NEWPKI_EE_REQUEST_REV)



/*!
  This structure represents an certification request
*/
typedef struct st_NEWPKI_CERT_REQUEST
{
	ASN1_INTEGER * id;					//!< An internal ID from the requesting entity, this id is sent back with the response
	X509_REQ * request;					//!< The certificate request
	ASN1_INTEGER * validity;			//!< The number of days the certificate should be valid for
	ASN1_UTF8STRING * ldap_uid;			//!< The LDAP UID of the user this request id for, the LDAP UID is latter used for publication
}NEWPKI_CERT_REQUEST;
DECLARE_ASN1_ITEM(NEWPKI_CERT_REQUEST)


/*!
  This structure represents an revocation request
*/
typedef struct st_NEWPKI_REV_REQUEST
{
	ASN1_INTEGER * id;				//!< An internal ID from the requesting entity, this id is sent back with the response
	ASN1_INTEGER * serial;			//!< The serial of the certificate to revoke
	ASN1_INTEGER * reason;			//!< The reason for this revocation
	ASN1_UTF8STRING * ldap_uid;		//!< The LDAP UID of the user this request id for, the LDAP UID is latter used for publication
}NEWPKI_REV_REQUEST;
DECLARE_ASN1_ITEM(NEWPKI_REV_REQUEST)


/*!
  This structure represents a suspension request
*/
typedef struct st_NEWPKI_SUSP_REQUEST
{
	ASN1_INTEGER * id;				//!< An internal ID from the requesting entity, this id is sent back with the response
	ASN1_INTEGER * serial;			//!< The serial of the certificate to suspend
	ASN1_UTF8STRING * ldap_uid;		//!< The LDAP UID of the user this request id for, the LDAP UID is latter used for publication
}NEWPKI_SUSP_REQUEST;
DECLARE_ASN1_ITEM(NEWPKI_SUSP_REQUEST)


/*!
  This structure represents an unsuspension request
*/
typedef struct st_NEWPKI_UNSUSP_REQUEST
{
	ASN1_INTEGER * id;				//!< An internal ID from the requesting entity, this id is sent back with the response
	ASN1_INTEGER * serial;			//!< The serial of the certificate to unsuspend
	ASN1_UTF8STRING * ldap_uid;		//!< The LDAP UID of the user this request id for, the LDAP UID is latter used for publication
}NEWPKI_UNSUSP_REQUEST;
DECLARE_ASN1_ITEM(NEWPKI_UNSUSP_REQUEST)


/*!
  This structure represents the body of a revocation publication request
*/
typedef struct st_NEWPKI_PUB_REQUEST_BODY_REV
{
	X509 * cert;					//!< The certificate that has been revoked
	ASN1_UTCTIME * revdate;			//!< The revocation date
}NEWPKI_PUB_REQUEST_BODY_REV;
DECLARE_ASN1_ITEM(NEWPKI_PUB_REQUEST_BODY_REV)

#define NEWPKI_PUB_REQUEST_CERT 0	//!< The publication request is for a certification
#define NEWPKI_PUB_REQUEST_REV 1	//!< The publication request is for a revocation
#define NEWPKI_PUB_REQUEST_CRL 2	//!< The publication request is for a CRL

/*!
  This structure represents the body of a publication request
*/
typedef struct st_NEWPKI_PUB_REQUEST_BODY
{
	int type;
	union
	{
		X509 * cert;							//!< A certificate when publishing a new certificate, type = NEWPKI_PUB_REQUEST_CERT
		NEWPKI_PUB_REQUEST_BODY_REV * rev;		//!< Revocation info when publishing a revocation, type = NEWPKI_PUB_REQUEST_REV
		X509_CRL * crl;							//!< A CRL when publishing a CRL, type = NEWPKI_PUB_REQUEST_CRL
	}d;
}NEWPKI_PUB_REQUEST_BODY;
DECLARE_ASN1_ITEM(NEWPKI_PUB_REQUEST_BODY)

/*!
  This structure represents a publication request
*/
typedef struct st_NEWPKI_PUB_REQUEST
{
	ASN1_UTF8STRING * ldap_uid;			//!< The UID for a cert or a rev, Empty for a CRL 
	STACK_OF(X509) * ParentCerts;		//!< The list of parents certs for a cert or a rev
	ASN1_UTF8STRING * object;			//!< A string name for the published object
	NEWPKI_PUB_REQUEST_BODY * body;		//!< The body
}NEWPKI_PUB_REQUEST;
DECLARE_ASN1_ITEM(NEWPKI_PUB_REQUEST)


/*!
  This structure represents a publication request
*/
typedef struct st_NEWPKI_BACKUP_REQUEST
{
	ASN1_OBJECT * sym_algo;			//!< The symetrical encryption algo
	ASN1_OCTET_STRING * backup;		//!< The encrypted representation of a NEWPKI_BACKUP
}NEWPKI_BACKUP_REQUEST;
DECLARE_ASN1_ITEM(NEWPKI_BACKUP_REQUEST)


/*!
  This structure represents an EE validation request
*/
typedef struct st_NEWPKI_EE_VAL_REQUEST
{
	ASN1_INTEGER * ra_id;	//!< The RA id
	X509_NAME * dn;			//!< The DN
}NEWPKI_EE_VAL_REQUEST;


typedef struct st_NEWPKI_EE_DN_PUBLISH
{
	ASN1_INTEGER * ee_id;	//!< The EE id
	X509_NAME * dn;			//!< The DN
}NEWPKI_EE_DN_PUBLISH;

typedef struct st_NEWPKI_EE_DN_REMOVE
{
	ASN1_INTEGER * ee_id;	//!< The EE id
}NEWPKI_EE_DN_REMOVE;

typedef struct st_NEWPKI_EE_UPDATE_CERT_STATUS
{
	ASN1_INTEGER * ra_id;	//!< The RA id
	ASN1_INTEGER * status;	//!< The new status
	X509_CRL * last_crl;	//!< The optional last CRL
}NEWPKI_EE_UPDATE_CERT_STATUS;

/*!
  This structure represents a certificate response for an EE,
  it might either have the certificate or an error
*/
typedef struct st_NEWPKI_EE_CERT_PUBLISH
{
	ASN1_INTEGER * profile_id;		//!< The owner profile ID
	ASN1_INTEGER * ra_id;			//!< The RA Id of the certificate
	ASN1_INTEGER * type;			//!< The certificate type
	ASN1_UTF8STRING * ca_name;		//!< The CA name
	X509 * certificate;				//!< The generated certificate
	PKCS7 * p7b;					//!< The optional generated PKCS7
	PKCS12 * p12;					//!< The optional generated PKCS12
	X509_CRL * last_crl;			//!< The optional last CRL
}NEWPKI_EE_CERT_PUBLISH;
DECLARE_ASN1_ITEM(NEWPKI_EE_CERT_PUBLISH)


#define NEWPKI_REQUEST_TYPE_CERT					0	//!< The request is for a certification
#define NEWPKI_REQUEST_TYPE_REV						1	//!< The request is for a revocation
#define NEWPKI_REQUEST_TYPE_PUB						2	//!< The request is for a publication
#define NEWPKI_REQUEST_TYPE_BACKUP					3	//!< The request is for a backup
#define NEWPKI_REQUEST_TYPE_EE_VAL					4	//!< The request is for a EE validation request
#define NEWPKI_REQUEST_TYPE_EE_DN_PUBLISH			5	//!< The request is for the publication of a DN by an EE
#define NEWPKI_REQUEST_TYPE_EE_DN_REMOVE			6	//!< The request is for the removal of a DN on a EE
#define NEWPKI_REQUEST_TYPE_EE_CERT					7	//!< The request is for a certification from an EE
#define NEWPKI_REQUEST_TYPE_SUSP					8	//!< The request is for a suspension
#define NEWPKI_REQUEST_TYPE_UNSUSP					9	//!< The request is for an unsuspension
#define NEWPKI_REQUEST_TYPE_EE_REV					10	//!< The request is for a revocation from an EE
#define NEWPKI_REQUEST_TYPE_EE_UPDATE_CERT_STATUS	11	//!< The request if for updating the status of a certificate on the EE
#define NEWPKI_REQUEST_TYPE_EE_CERT_PUBLISH			12	//!< The request if for publishing a certificate on the EE


/*!
  This structure represents an inter-entity request
*/
typedef struct st_NEWPKI_REQUEST
{
	int type;
	union
	{
		NEWPKI_CERT_REQUEST * cert_request;					//!< A certification request
		NEWPKI_REV_REQUEST * rev_request;					//!< A revocation request
		NEWPKI_PUB_REQUEST * pub_request;					//!< A publication request
		NEWPKI_BACKUP_REQUEST * backup_request;				//!< A backup request
		NEWPKI_EE_VAL_REQUEST * ee_val_request;				//!< An EE request
		NEWPKI_EE_DN_PUBLISH * ee_dn_publish;				//!< The publication of a DN by an EE
		NEWPKI_EE_DN_REMOVE * ee_dn_remove;					//!< The removal of a DN on an EE
		NEWPKI_EE_REQUEST_CERT * ee_cert_request;			//!< A certification request from an EE
		NEWPKI_SUSP_REQUEST * susp_request;					//!< A suspension request
		NEWPKI_UNSUSP_REQUEST * unsusp_request;				//!< An unsuspension request
		NEWPKI_EE_REQUEST_REV * ee_cert_revoke;				//!< A revocation request from an EE
		NEWPKI_EE_UPDATE_CERT_STATUS * ee_upd_cert_status;	//!< The update of a certificate status on the EE side
		NEWPKI_EE_CERT_PUBLISH * ee_cert_publish;			//!< The publication of a certificate on the EE
	}d;
}NEWPKI_REQUEST;
DECLARE_ASN1_ITEM(NEWPKI_REQUEST)



/*!
  This structure represents an inter-entity request signed and crypted. Requests are signed by the sender and crypted for the recipient.
*/
typedef struct st_CRYPTED_NEWPKI_REQUEST
{
	X509_PUBKEY * sender;				//!< The public key of the sender
	X509_PUBKEY * recipient;			//!< The public key of the recipient
	ASN1_ENCRYPT_SIGN * c_request;		//!< The Crypted representation of a NEWPKI_REQUEST
	ASN1_OCTET_STRING * transactionID;	//!< The internal ID of the request
}CRYPTED_NEWPKI_REQUEST;
DECLARE_ASN1_ITEM(CRYPTED_NEWPKI_REQUEST)

/*!
	This structure  represents a list of requests.
*/
typedef struct st_CRYPTED_NEWPKI_REQUESTS
{
	STACK_OF(CRYPTED_NEWPKI_REQUEST) * requests;	//!< The list of requests
}CRYPTED_NEWPKI_REQUESTS;
DECLARE_ASN1_ITEM(CRYPTED_NEWPKI_REQUESTS)


#define CERT_RESPONSE_OK			1	//!< The requested certificate has been successfully generated
#define CERT_RESPONSE_ERROR			2	//!< There was an error generation the requested certificate

/*!
  This structure represents a certificate response,
  it might either have the certificate or an error
*/
typedef struct st_NEWPKI_CERT_RESPONSE
{
	ASN1_INTEGER * id;				//!< The internal ID from the requesting entity
	ASN1_INTEGER * status;			//!< The status, CERT_RESPONSE_OK or CERT_RESPONSE_ERROR
	X509_CRL * last_crl;			//!< The last CRL
	X509 * certificate;				//!< The optional generated certificate
	STACK_OF(X509) * ParentCerts;	//!< The parents certs
	STACK_OF(ERROR_ENTRY) * errors;	//!< The optional errors
}NEWPKI_CERT_RESPONSE;
DECLARE_ASN1_ITEM(NEWPKI_CERT_RESPONSE)

/*!
  This structure represents a certificate response for an EE,
  it might either have the certificate or an error
*/
typedef struct st_NEWPKI_EE_CERT_RESPONSE
{
	ASN1_INTEGER * status;			//!< The status, CERT_RESPONSE_OK or CERT_RESPONSE_ERROR
	ASN1_INTEGER * ra_id;			//!< The RA Id of the certificate
	X509 * certificate;				//!< The optional generated certificate
	PKCS7 * p7b;					//!< The optional generated PKCS7
	PKCS12 * p12;					//!< The optional generated PKCS12
	STACK_OF(ERROR_ENTRY) * errors;	//!< The optional errors
	X509_CRL * last_crl;			//!< The optional last CRL
}NEWPKI_EE_CERT_RESPONSE;
DECLARE_ASN1_ITEM(NEWPKI_EE_CERT_RESPONSE)



#define REV_RESPONSE_OK				1	//!< The requested revocation has been successfully processed
#define REV_RESPONSE_ERROR			2	//!< There was an error processing the requested revocation

/*!
  This structure represents a revocation response
*/
typedef struct st_NEWPKI_REV_RESPONSE
{
	ASN1_INTEGER * id;				//!< The internal ID from the requesting entity
	ASN1_INTEGER * status;			//!< The status, REV_RESPONSE_OK or REV_RESPONSE_ERROR
	ASN1_INTEGER * cert_status;		//!< The status of the certificate
	X509_CRL * last_crl;			//!< The last CRL
	STACK_OF(ERROR_ENTRY) * errors;	//!< The optional errors
}NEWPKI_REV_RESPONSE;
DECLARE_ASN1_ITEM(NEWPKI_REV_RESPONSE)


/*!
  This structure represents a revocation response for an EE
*/
typedef struct st_NEWPKI_EE_REV_RESPONSE
{
	ASN1_INTEGER * ra_id;			//!< The internal RA ID
	ASN1_INTEGER * status;			//!< The status, REV_RESPONSE_OK or REV_RESPONSE_ERROR
	ASN1_INTEGER * cert_status;		//!< The status of the certificate
	X509_CRL * last_crl;			//!< The last CRL
	STACK_OF(ERROR_ENTRY) * errors;	//!< The optional errors
}NEWPKI_EE_REV_RESPONSE;
DECLARE_ASN1_ITEM(NEWPKI_EE_REV_RESPONSE)


#define SUSP_RESPONSE_OK			1	//!< The requested suspension has been successfully processed
#define SUSP_RESPONSE_ERROR			2	//!< There was an error processing the requested suspension

/*!
  This structure represents a suspension response
*/
typedef struct st_NEWPKI_SUSP_RESPONSE
{
	ASN1_INTEGER * id;				//!< The internal ID from the requesting entity
	ASN1_INTEGER * status;			//!< The status, SUSP_RESPONSE_OK or SUSP_RESPONSE_ERROR
	ASN1_INTEGER * cert_status;		//!< The status of the certificate
	X509_CRL * last_crl;			//!< The last CRL
	STACK_OF(ERROR_ENTRY) * errors;	//!< The optional errors
}NEWPKI_SUSP_RESPONSE;
DECLARE_ASN1_ITEM(NEWPKI_SUSP_RESPONSE)


#define UNSUSP_RESPONSE_OK			1	//!< The requested unsuspension has been successfully processed
#define UNSUSP_RESPONSE_ERROR		2	//!< There was an error processing the requested unsuspension

/*!
  This structure represents a unsuspension response
*/
typedef struct st_NEWPKI_UNSUSP_RESPONSE
{
	ASN1_INTEGER * id;				//!< The internal ID from the requesting entity
	ASN1_INTEGER * status;			//!< The status, UNSUSP_RESPONSE_OK or UNSUSP_RESPONSE_ERROR
	ASN1_INTEGER * cert_status;		//!< The status of the certificate
	X509_CRL * last_crl;			//!< The last CRL
	STACK_OF(ERROR_ENTRY) * errors;	//!< The optional errors
}NEWPKI_UNSUSP_RESPONSE;
DECLARE_ASN1_ITEM(NEWPKI_UNSUSP_RESPONSE)


#define PUB_RESPONSE_PUBLISHED		1	//!< The requested publication has been successfully processed
#define PUB_RESPONSE_ERROR			2	//!< There was an error processing the requested publication

/*!
  This structure represents a publication response
*/
typedef struct st_NEWPKI_PUB_RESPONSE
{
	ASN1_INTEGER * status;			//!< The status, PUB_RESPONSE_PUBLISHED or PUB_RESPONSE_ERROR
	ASN1_INTEGER * type;			//!< The publication type, NEWPKI_PUB_REQUEST_CERT or NEWPKI_PUB_REQUEST_REV or NEWPKI_PUB_REQUEST_CRL
	ASN1_UTF8STRING * object;		//!< A string name for the published object
	STACK_OF(ERROR_ENTRY) * errors;	//!< The optional errors
}NEWPKI_PUB_RESPONSE;
DECLARE_ASN1_ITEM(NEWPKI_PUB_RESPONSE)

/*!
  This structure represents an EE validation request
*/
typedef struct st_NEWPKI_EE_VAL_RESPONSE
{
	ASN1_INTEGER * ra_id;	//!< The RA id
	X509_NAME * dn;			//!< The validated DN
}NEWPKI_EE_VAL_RESPONSE;

/*!
  This structure represents an DN publication response,
  it might either have an error
*/
typedef struct st_NEWPKI_EE_DN_PUBLISH_RESPONSE
{
	STACK_OF(ERROR_ENTRY) * errors;	//!< The optional errors
}NEWPKI_EE_DN_PUBLISH_RESPONSE;
DECLARE_ASN1_ITEM(NEWPKI_EE_DN_PUBLISH_RESPONSE)

/*!
  This structure represents an DN removal response,
  it might either have an error
*/
typedef struct st_NEWPKI_EE_DN_REMOVE_RESPONSE
{
	STACK_OF(ERROR_ENTRY) * errors;	//!< The optional errors
}NEWPKI_EE_DN_REMOVE_RESPONSE;
DECLARE_ASN1_ITEM(NEWPKI_EE_DN_PUBLISH_RESPONSE)


/*!
  This structure represents a certificate status update response
  it might either have an error
*/
typedef struct st_NEWPKI_EE_UPDATE_CERT_STATUS_RESPONSE
{
	STACK_OF(ERROR_ENTRY) * errors;	//!< The optional errors
}NEWPKI_EE_UPDATE_CERT_STATUS_RESPONSE;
DECLARE_ASN1_ITEM(NEWPKI_EE_UPDATE_CERT_STATUS_RESPONSE)

/*!
  This structure represents a certificate publish response
  it might either have an error
*/
typedef struct st_NEWPKI_EE_CERT_PUBLISH_RESPONSE
{
	STACK_OF(ERROR_ENTRY) * errors;	//!< The optional errors
}NEWPKI_EE_CERT_PUBLISH_RESPONSE;
DECLARE_ASN1_ITEM(NEWPKI_EE_UPDATE_CERT_STATUS_RESPONSE)




#define NEWPKI_RESPONSE_TYPE_CERT						0		//!< The response is for a certification
#define NEWPKI_RESPONSE_TYPE_REV						1		//!< The response is for a revocation
#define NEWPKI_RESPONSE_TYPE_ERR						2		//!< There was an error processing the request
#define NEWPKI_RESPONSE_TYPE_PUB						3		//!< The response is for a publication
#define NEWPKI_RESPONSE_TYPE_EE_VAL						4		//!< The response is an user validated DN
#define NEWPKI_RESPONSE_TYPE_EE_DN_PUBLISH				5		//!< The response is an EE DN publication response
#define NEWPKI_RESPONSE_TYPE_EE_DN_REMOVE				6		//!< The response is an EE DN removal response
#define NEWPKI_RESPONSE_TYPE_EE_CERT					7		//!< The response is for a certification from an EE
#define NEWPKI_RESPONSE_TYPE_SUSP						8		//!< The response is for a suspension
#define NEWPKI_RESPONSE_TYPE_UNSUSP						9		//!< The response is for an unsuspension
#define NEWPKI_RESPONSE_TYPE_EE_REV						10		//!< The response is for a revocation for an EE
#define NEWPKI_RESPONSE_TYPE_EE_UPDATE_CERT_STATUS		11		//!< The response if for updating the status of a certificate on the EE
#define NEWPKI_RESPONSE_TYPE_EE_CERT_PUBLISH			12		//!< The response if an EE certificate publication


/*!
  This structure represents an inter-entity response.
*/
typedef struct st_NEWPKI_RESPONSE
{
	int type;
	union
	{
		NEWPKI_CERT_RESPONSE * cert_response;									//!< A certification response
		NEWPKI_REV_RESPONSE * rev_response;										//!< A revocation response
		NEWPKI_PUB_RESPONSE * pub_response;										//!< A publication response
		NEWPKI_EE_VAL_RESPONSE * ee_val_response;								//!< An EE validation response
		NEWPKI_EE_DN_PUBLISH_RESPONSE * ee_dn_publish_response;					//!< An EE DN publication response
		NEWPKI_EE_DN_REMOVE_RESPONSE * ee_dn_remove_response;					//!< An EE DN removal response
		NEWPKI_EE_CERT_RESPONSE * ee_cert_response;								//!< A certification response for an EE
		NEWPKI_SUSP_RESPONSE * susp_response;									//!< A suspension response
		NEWPKI_UNSUSP_RESPONSE * unsusp_response;								//!< A unsuspension response
		NEWPKI_EE_REV_RESPONSE * ee_rev_response;								//!< A revocation response for an EE
		NEWPKI_EE_UPDATE_CERT_STATUS_RESPONSE * ee_upd_cert_status_response;	//!< An EE cert status update response
		NEWPKI_EE_CERT_PUBLISH_RESPONSE * ee_cert_publish_response;				//!< An EE cet publication response
		STACK_OF(ERROR_ENTRY) * errors;											//!< The was an error before even processing the requests (ACCESS DENIED, etc)
	}d;
}NEWPKI_RESPONSE;

/*!
  This structure represents an inter-entity response signed and crypted. Responses are signed by the sender and crypted for the recipient.
*/
typedef struct st_CRYPTED_NEWPKI_RESPONSE
{
	X509_PUBKEY * sender;				//!< The public key of the sender
	X509_PUBKEY * recipient;			//!< The public key of the recipient
	ASN1_ENCRYPT_SIGN * c_response;		//!< The Crypted representation of NEWPKI_RESPONSE
	ASN1_OCTET_STRING * transactionID;	//!< The internal ID of the response
}CRYPTED_NEWPKI_RESPONSE;
DECLARE_ASN1_ITEM(CRYPTED_NEWPKI_RESPONSE)



/*!
	This structure represents a list of responses.
*/
typedef struct st_CRYPTED_NEWPKI_RESPONSES
{
	STACK_OF(CRYPTED_NEWPKI_RESPONSE) * responses;	//!< The list of responses
}CRYPTED_NEWPKI_RESPONSES;
DECLARE_ASN1_ITEM(CRYPTED_NEWPKI_RESPONSES)



#define NEWPKI_OBJECT_TYPE_REQ				0		//!< The newpki object is a request
#define NEWPKI_OBJECT_TYPE_RESP				1		//!< The newpki object is a response
#define NEWPKI_OBJECT_TYPE_DEL				2		//!< The newpki object is a response deletion


#define NEWPKI_OBJECT_STATUS_IMPORTED		0		//!< The newpki object was imported
#define NEWPKI_OBJECT_STATUS_KNOWN			1		//!< The newpki object is already known by the repository
#define NEWPKI_OBJECT_STATUS_DELETED		2		//!< The newpki object was deleted, and you should do the same thing


/*!
	This structure represents the body of a newpi object.
*/
typedef struct st_WAITING_NEWPKI_OBJECT_BODY
{
	int type;
	union
	{
		CRYPTED_NEWPKI_REQUEST * request;		//!< The object is a request
		CRYPTED_NEWPKI_RESPONSE * response;		//!< The object is a response
		ASN1_OCTET_STRING * transactionID;		//!< A transaction ID related object
	}d;
}WAITING_NEWPKI_OBJECT_BODY;
DECLARE_ASN1_ITEM(WAITING_NEWPKI_OBJECT_BODY)


/*!
	This structure represents a list of transaction IDs
*/
typedef struct st_TRANSACTION_IDS
{
	STACK_OF(ASN1_OCTET_STRING) * transactionIds;	//!< The list
}TRANSACTION_IDS;
DECLARE_ASN1_ITEM(TRANSACTION_IDS)


/*!
	This structure is used by repositories for synchronization
	purposes.
*/
typedef struct st_OBJECTS_LIST_SYNCHRO
{
	TRANSACTION_IDS * known_requests;			//!< The list of known requets
	TRANSACTION_IDS * known_responses;			//!< The list of known responses
	TRANSACTION_IDS * want_requests;			//!< The list of requets requested by repository
	TRANSACTION_IDS * want_responses;			//!< The list of responses requested by repository
	TRANSACTION_IDS * delete_responses;			//!< The responses the other peer should delete
	STACK_OF(WAITING_NEWPKI_OBJECT) * objects;	//!< The list of objects the requesting repository doesn't know
}OBJECTS_LIST_SYNCHRO;
DECLARE_ASN1_ITEM(OBJECTS_LIST_SYNCHRO)


/*!
	This structure is used by repositories to propagate a newpki object.
*/
typedef struct st_WAITING_NEWPKI_OBJECT
{
	WAITING_NEWPKI_OBJECT_BODY * object;		//!< The body of the object
	STACK_OF(X509_PUBKEY) * rep_path;			//!< The list of all the repositories that have already processed this object
}WAITING_NEWPKI_OBJECT;
DECLARE_ASN1_ITEM(WAITING_NEWPKI_OBJECT)

#define INTERNAL_CA_KEY_TYPE_KEY 0		//!< The private key is a software key
#define INTERNAL_CA_KEY_TYPE_ENGINE 1	//!< The private key is a key id stored in an engine

/*!
	This structure is the representation of a private key
*/
typedef struct st_INTERNAL_CA_KEY
{
	int type;	//!< The type of key
	union
	{
		RSA * privkey;				//!< A software key
		ASN1_UTF8STRING * keyid;	//!< A key Id
	}d;
}INTERNAL_CA_KEY;
DECLARE_ASN1_ITEM(INTERNAL_CA_KEY)


/*!
	This structure is the representation of CA.
*/
typedef struct st_INTERNAL_CA
{
	X509 * ca_cert;							//!< The certificate.
	INTERNAL_CA_KEY * ca_key;				//!< The private key.
	ASN1_ENCRYPT_SIGN * crypted_body;		//!< The crypted body.
}INTERNAL_CA;
DECLARE_ASN1_ITEM(INTERNAL_CA)


/*!
  This class is the representation of NEWPKI_REQUEST_CERT_BODY_P12
*/
class NewpkiRequestCertBodyP12 : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiRequestCertBodyP12();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiRequestCertBodyP12(const NewpkiRequestCertBodyP12 & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiRequestCertBodyP12();


	/*! \brief This function converts a NEWPKI_REQUEST_CERT_BODY_P12 to a NewpkiRequestCertBodyP12.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_REQUEST_CERT_BODY_P12 * Datas);

	/*! \brief This function converts a NewpkiRequestCertBodyP12 to a NEWPKI_REQUEST_CERT_BODY_P12.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_REQUEST_CERT_BODY_P12 ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_REQUEST_CERT_BODY_P12.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::keylen.
	 *  \param c_keylen [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_keylen(unsigned long c_keylen);

	/*! \brief This function returns ASN1_INTEGER::keylen.
	 *  \return The value of ASN1_INTEGER::keylen.
	 */
	unsigned long get_keylen() const;


	/*! \brief This function sets ASN1_UTF8STRING::passwd.
	 *  \param c_passwd [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_passwd(const mString & c_passwd);

	/*! \brief This function returns ASN1_UTF8STRING::passwd.
	 *  \return The value of ASN1_UTF8STRING::passwd.
	 */
	const mString & get_passwd() const;

	/*! \brief This function returns ASN1_UTF8STRING::passwd.
	 *  \return The value of ASN1_UTF8STRING::passwd.
	 */
	mString & get_passwd();


	/*! \brief This function copies a NewpkiRequestCertBodyP12 to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiRequestCertBodyP12 & other);
	/*! \brief This member is an empty instance of NewpkiRequestCertBodyP12.
	 */
	static NewpkiRequestCertBodyP12 EmptyInstance;

private:
	unsigned long m_keylen;
	mString m_passwd;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_REQUEST_CERT_BODY
*/
class NewpkiRequestCertBody : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiRequestCertBody();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiRequestCertBody(const NewpkiRequestCertBody & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiRequestCertBody();


	/*! \brief This function converts a NEWPKI_REQUEST_CERT_BODY to a NewpkiRequestCertBody.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_REQUEST_CERT_BODY * Datas);

	/*! \brief This function converts a NewpkiRequestCertBody to a NEWPKI_REQUEST_CERT_BODY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_REQUEST_CERT_BODY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_REQUEST_CERT_BODY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets NEWPKI_REQUEST_CERT_BODY_P12::d.p12_privdatas.
	 *  \param c_p12Privdatas [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_p12Privdatas(const NewpkiRequestCertBodyP12 & c_p12Privdatas);

	/*! \brief This function returns NEWPKI_REQUEST_CERT_BODY_P12::d.p12_privdatas.
	 *  \return The value of NEWPKI_REQUEST_CERT_BODY_P12::d.p12_privdatas.
	 */
	const NewpkiRequestCertBodyP12 & get_p12Privdatas() const;

	/*! \brief This function returns NEWPKI_REQUEST_CERT_BODY_P12::d.p12_privdatas.
	 *  \return The value of NEWPKI_REQUEST_CERT_BODY_P12::d.p12_privdatas.
	 */
	NewpkiRequestCertBodyP12 & get_p12Privdatas();


	/*! \brief This function sets X509_REQ::d.pkcs10.
	 *  \param c_pkcs10 [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_pkcs10(const PKI_CSR & c_pkcs10);

	/*! \brief This function returns X509_REQ::d.pkcs10.
	 *  \return The value of X509_REQ::d.pkcs10.
	 */
	const PKI_CSR & get_pkcs10() const;

	/*! \brief This function returns X509_REQ::d.pkcs10.
	 *  \return The value of X509_REQ::d.pkcs10.
	 */
	PKI_CSR & get_pkcs10();


	/*! \brief This function copies a NewpkiRequestCertBody to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiRequestCertBody & other);
	/*! \brief This member is an empty instance of NewpkiRequestCertBody.
	 */
	static NewpkiRequestCertBody EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	NewpkiRequestCertBodyP12 * m_p12Privdatas;
	PKI_CSR * m_pkcs10;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_REQUEST_CERT
*/
class NewpkiRequestCert : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiRequestCert();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiRequestCert(const NewpkiRequestCert & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiRequestCert();


	/*! \brief This function converts a NEWPKI_REQUEST_CERT to a NewpkiRequestCert.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_REQUEST_CERT * Datas);

	/*! \brief This function converts a NewpkiRequestCert to a NEWPKI_REQUEST_CERT.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_REQUEST_CERT ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_REQUEST_CERT.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::ca_name.
	 *  \param c_caName [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_caName(const mString & c_caName);

	/*! \brief This function returns ASN1_UTF8STRING::ca_name.
	 *  \return The value of ASN1_UTF8STRING::ca_name.
	 */
	const mString & get_caName() const;

	/*! \brief This function returns ASN1_UTF8STRING::ca_name.
	 *  \return The value of ASN1_UTF8STRING::ca_name.
	 */
	mString & get_caName();


	/*! \brief This function sets ASN1_BIT_STRING::flags.
	 *  \param c_flags [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_flags(const ASN1_BIT_STRING * c_flags);

	/*! \brief This function returns ASN1_BIT_STRING::flags.
	 *  \return The value of ASN1_BIT_STRING::flags.
	 */
	const ASN1_BIT_STRING * get_flags() const;

	/*! \brief This function returns ASN1_BIT_STRING::flags.
	 *  \return The value of ASN1_BIT_STRING::flags.
	 */
	ASN1_BIT_STRING * get_flags();


	/*! \brief This function sets ASN1_INTEGER::profile_id.
	 *  \param c_profileId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_profileId(unsigned long c_profileId);

	/*! \brief This function returns ASN1_INTEGER::profile_id.
	 *  \return The value of ASN1_INTEGER::profile_id.
	 */
	unsigned long get_profileId() const;


	/*! \brief This function sets NEWPKI_REQUEST_CERT_BODY::request.
	 *  \param c_request [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_request(const NewpkiRequestCertBody & c_request);

	/*! \brief This function returns NEWPKI_REQUEST_CERT_BODY::request.
	 *  \return The value of NEWPKI_REQUEST_CERT_BODY::request.
	 */
	const NewpkiRequestCertBody & get_request() const;

	/*! \brief This function returns NEWPKI_REQUEST_CERT_BODY::request.
	 *  \return The value of NEWPKI_REQUEST_CERT_BODY::request.
	 */
	NewpkiRequestCertBody & get_request();


	/*! \brief This function sets ASN1_INTEGER::type.
	 *  \param c_type [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_type(unsigned long c_type);

	/*! \brief This function returns ASN1_INTEGER::type.
	 *  \return The value of ASN1_INTEGER::type.
	 */
	unsigned long get_type() const;


	/*! \brief This function sets ASN1_INTEGER::validity.
	 *  \param c_validity [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_validity(unsigned long c_validity);

	/*! \brief This function returns ASN1_INTEGER::validity.
	 *  \return The value of ASN1_INTEGER::validity.
	 */
	unsigned long get_validity() const;


	/*! \brief This function copies a NewpkiRequestCert to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiRequestCert & other);
	/*! \brief This member is an empty instance of NewpkiRequestCert.
	 */
	static NewpkiRequestCert EmptyInstance;

private:
	mString m_caName;
	ASN1_BIT_STRING * m_flags;
	unsigned long m_profileId;
	NewpkiRequestCertBody m_request;
	unsigned long m_type;
	unsigned long m_validity;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_EE_REQUEST_CERT
*/
class NewpkiEeRequestCert : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiEeRequestCert();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiEeRequestCert(const NewpkiEeRequestCert & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiEeRequestCert();


	/*! \brief This function converts a NEWPKI_EE_REQUEST_CERT to a NewpkiEeRequestCert.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_EE_REQUEST_CERT * Datas);

	/*! \brief This function converts a NewpkiEeRequestCert to a NEWPKI_EE_REQUEST_CERT.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_EE_REQUEST_CERT ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_EE_REQUEST_CERT.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::ca_name.
	 *  \param c_caName [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_caName(const mString & c_caName);

	/*! \brief This function returns ASN1_UTF8STRING::ca_name.
	 *  \return The value of ASN1_UTF8STRING::ca_name.
	 */
	const mString & get_caName() const;

	/*! \brief This function returns ASN1_UTF8STRING::ca_name.
	 *  \return The value of ASN1_UTF8STRING::ca_name.
	 */
	mString & get_caName();


	/*! \brief This function sets ASN1_INTEGER::profile_id.
	 *  \param c_profileId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_profileId(unsigned long c_profileId);

	/*! \brief This function returns ASN1_INTEGER::profile_id.
	 *  \return The value of ASN1_INTEGER::profile_id.
	 */
	unsigned long get_profileId() const;


	/*! \brief This function sets NEWPKI_REQUEST_CERT_BODY::request.
	 *  \param c_request [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_request(const NewpkiRequestCertBody & c_request);

	/*! \brief This function returns NEWPKI_REQUEST_CERT_BODY::request.
	 *  \return The value of NEWPKI_REQUEST_CERT_BODY::request.
	 */
	const NewpkiRequestCertBody & get_request() const;

	/*! \brief This function returns NEWPKI_REQUEST_CERT_BODY::request.
	 *  \return The value of NEWPKI_REQUEST_CERT_BODY::request.
	 */
	NewpkiRequestCertBody & get_request();


	/*! \brief This function sets ASN1_INTEGER::type.
	 *  \param c_type [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_type(unsigned long c_type);

	/*! \brief This function returns ASN1_INTEGER::type.
	 *  \return The value of ASN1_INTEGER::type.
	 */
	unsigned long get_type() const;


	/*! \brief This function copies a NewpkiEeRequestCert to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiEeRequestCert & other);
	/*! \brief This member is an empty instance of NewpkiEeRequestCert.
	 */
	static NewpkiEeRequestCert EmptyInstance;

private:
	mString m_caName;
	unsigned long m_profileId;
	NewpkiRequestCertBody m_request;
	unsigned long m_type;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_EE_REQUEST_REV
*/
class NewpkiEeRequestRev : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiEeRequestRev();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiEeRequestRev(const NewpkiEeRequestRev & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiEeRequestRev();


	/*! \brief This function converts a NEWPKI_EE_REQUEST_REV to a NewpkiEeRequestRev.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_EE_REQUEST_REV * Datas);

	/*! \brief This function converts a NewpkiEeRequestRev to a NEWPKI_EE_REQUEST_REV.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_EE_REQUEST_REV ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_EE_REQUEST_REV.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::cert_id.
	 *  \param c_certId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certId(unsigned long c_certId);

	/*! \brief This function returns ASN1_INTEGER::cert_id.
	 *  \return The value of ASN1_INTEGER::cert_id.
	 */
	unsigned long get_certId() const;


	/*! \brief This function copies a NewpkiEeRequestRev to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiEeRequestRev & other);
	/*! \brief This member is an empty instance of NewpkiEeRequestRev.
	 */
	static NewpkiEeRequestRev EmptyInstance;

private:
	unsigned long m_certId;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_CERT_REQUEST
*/
class NewpkiCertRequest : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiCertRequest();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiCertRequest(const NewpkiCertRequest & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiCertRequest();


	/*! \brief This function converts a NEWPKI_CERT_REQUEST to a NewpkiCertRequest.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_CERT_REQUEST * Datas);

	/*! \brief This function converts a NewpkiCertRequest to a NEWPKI_CERT_REQUEST.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_CERT_REQUEST ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_CERT_REQUEST.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::id.
	 *  \param c_id [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_id(unsigned long c_id);

	/*! \brief This function returns ASN1_INTEGER::id.
	 *  \return The value of ASN1_INTEGER::id.
	 */
	unsigned long get_id() const;


	/*! \brief This function sets ASN1_UTF8STRING::ldap_uid.
	 *  \param c_ldapUid [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapUid(const mString & c_ldapUid);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_uid.
	 *  \return The value of ASN1_UTF8STRING::ldap_uid.
	 */
	const mString & get_ldapUid() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_uid.
	 *  \return The value of ASN1_UTF8STRING::ldap_uid.
	 */
	mString & get_ldapUid();


	/*! \brief This function sets X509_REQ::request.
	 *  \param c_request [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_request(const PKI_CSR & c_request);

	/*! \brief This function returns X509_REQ::request.
	 *  \return The value of X509_REQ::request.
	 */
	const PKI_CSR & get_request() const;

	/*! \brief This function returns X509_REQ::request.
	 *  \return The value of X509_REQ::request.
	 */
	PKI_CSR & get_request();


	/*! \brief This function sets ASN1_INTEGER::validity.
	 *  \param c_validity [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_validity(unsigned long c_validity);

	/*! \brief This function returns ASN1_INTEGER::validity.
	 *  \return The value of ASN1_INTEGER::validity.
	 */
	unsigned long get_validity() const;


	/*! \brief This function copies a NewpkiCertRequest to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiCertRequest & other);
	/*! \brief This member is an empty instance of NewpkiCertRequest.
	 */
	static NewpkiCertRequest EmptyInstance;

private:
	unsigned long m_id;
	mString m_ldapUid;
	PKI_CSR m_request;
	unsigned long m_validity;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_REV_REQUEST
*/
class NewpkiRevRequest : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiRevRequest();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiRevRequest(const NewpkiRevRequest & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiRevRequest();


	/*! \brief This function converts a NEWPKI_REV_REQUEST to a NewpkiRevRequest.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_REV_REQUEST * Datas);

	/*! \brief This function converts a NewpkiRevRequest to a NEWPKI_REV_REQUEST.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_REV_REQUEST ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_REV_REQUEST.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::id.
	 *  \param c_id [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_id(unsigned long c_id);

	/*! \brief This function returns ASN1_INTEGER::id.
	 *  \return The value of ASN1_INTEGER::id.
	 */
	unsigned long get_id() const;


	/*! \brief This function sets ASN1_UTF8STRING::ldap_uid.
	 *  \param c_ldapUid [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapUid(const mString & c_ldapUid);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_uid.
	 *  \return The value of ASN1_UTF8STRING::ldap_uid.
	 */
	const mString & get_ldapUid() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_uid.
	 *  \return The value of ASN1_UTF8STRING::ldap_uid.
	 */
	mString & get_ldapUid();


	/*! \brief This function sets ASN1_INTEGER::reason.
	 *  \param c_reason [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_reason(unsigned long c_reason);

	/*! \brief This function returns ASN1_INTEGER::reason.
	 *  \return The value of ASN1_INTEGER::reason.
	 */
	unsigned long get_reason() const;


	/*! \brief This function sets ASN1_INTEGER::serial.
	 *  \param c_serial [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_serial(unsigned long c_serial);

	/*! \brief This function returns ASN1_INTEGER::serial.
	 *  \return The value of ASN1_INTEGER::serial.
	 */
	unsigned long get_serial() const;


	/*! \brief This function copies a NewpkiRevRequest to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiRevRequest & other);
	/*! \brief This member is an empty instance of NewpkiRevRequest.
	 */
	static NewpkiRevRequest EmptyInstance;

private:
	unsigned long m_id;
	mString m_ldapUid;
	unsigned long m_reason;
	unsigned long m_serial;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_SUSP_REQUEST
*/
class NewpkiSuspRequest : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiSuspRequest();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiSuspRequest(const NewpkiSuspRequest & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiSuspRequest();


	/*! \brief This function converts a NEWPKI_SUSP_REQUEST to a NewpkiSuspRequest.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_SUSP_REQUEST * Datas);

	/*! \brief This function converts a NewpkiSuspRequest to a NEWPKI_SUSP_REQUEST.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_SUSP_REQUEST ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_SUSP_REQUEST.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::id.
	 *  \param c_id [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_id(unsigned long c_id);

	/*! \brief This function returns ASN1_INTEGER::id.
	 *  \return The value of ASN1_INTEGER::id.
	 */
	unsigned long get_id() const;


	/*! \brief This function sets ASN1_UTF8STRING::ldap_uid.
	 *  \param c_ldapUid [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapUid(const mString & c_ldapUid);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_uid.
	 *  \return The value of ASN1_UTF8STRING::ldap_uid.
	 */
	const mString & get_ldapUid() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_uid.
	 *  \return The value of ASN1_UTF8STRING::ldap_uid.
	 */
	mString & get_ldapUid();


	/*! \brief This function sets ASN1_INTEGER::serial.
	 *  \param c_serial [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_serial(unsigned long c_serial);

	/*! \brief This function returns ASN1_INTEGER::serial.
	 *  \return The value of ASN1_INTEGER::serial.
	 */
	unsigned long get_serial() const;


	/*! \brief This function copies a NewpkiSuspRequest to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiSuspRequest & other);
	/*! \brief This member is an empty instance of NewpkiSuspRequest.
	 */
	static NewpkiSuspRequest EmptyInstance;

private:
	unsigned long m_id;
	mString m_ldapUid;
	unsigned long m_serial;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_UNSUSP_REQUEST
*/
class NewpkiUnsuspRequest : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiUnsuspRequest();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiUnsuspRequest(const NewpkiUnsuspRequest & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiUnsuspRequest();


	/*! \brief This function converts a NEWPKI_UNSUSP_REQUEST to a NewpkiUnsuspRequest.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_UNSUSP_REQUEST * Datas);

	/*! \brief This function converts a NewpkiUnsuspRequest to a NEWPKI_UNSUSP_REQUEST.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_UNSUSP_REQUEST ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_UNSUSP_REQUEST.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::id.
	 *  \param c_id [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_id(unsigned long c_id);

	/*! \brief This function returns ASN1_INTEGER::id.
	 *  \return The value of ASN1_INTEGER::id.
	 */
	unsigned long get_id() const;


	/*! \brief This function sets ASN1_UTF8STRING::ldap_uid.
	 *  \param c_ldapUid [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapUid(const mString & c_ldapUid);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_uid.
	 *  \return The value of ASN1_UTF8STRING::ldap_uid.
	 */
	const mString & get_ldapUid() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_uid.
	 *  \return The value of ASN1_UTF8STRING::ldap_uid.
	 */
	mString & get_ldapUid();


	/*! \brief This function sets ASN1_INTEGER::serial.
	 *  \param c_serial [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_serial(unsigned long c_serial);

	/*! \brief This function returns ASN1_INTEGER::serial.
	 *  \return The value of ASN1_INTEGER::serial.
	 */
	unsigned long get_serial() const;


	/*! \brief This function copies a NewpkiUnsuspRequest to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiUnsuspRequest & other);
	/*! \brief This member is an empty instance of NewpkiUnsuspRequest.
	 */
	static NewpkiUnsuspRequest EmptyInstance;

private:
	unsigned long m_id;
	mString m_ldapUid;
	unsigned long m_serial;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_PUB_REQUEST_BODY_REV
*/
class NewpkiPubRequestBodyRev : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiPubRequestBodyRev();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiPubRequestBodyRev(const NewpkiPubRequestBodyRev & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiPubRequestBodyRev();


	/*! \brief This function converts a NEWPKI_PUB_REQUEST_BODY_REV to a NewpkiPubRequestBodyRev.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_PUB_REQUEST_BODY_REV * Datas);

	/*! \brief This function converts a NewpkiPubRequestBodyRev to a NEWPKI_PUB_REQUEST_BODY_REV.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_PUB_REQUEST_BODY_REV ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_PUB_REQUEST_BODY_REV.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509::cert.
	 *  \param c_cert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_cert(const PKI_CERT & c_cert);

	/*! \brief This function returns X509::cert.
	 *  \return The value of X509::cert.
	 */
	const PKI_CERT & get_cert() const;

	/*! \brief This function returns X509::cert.
	 *  \return The value of X509::cert.
	 */
	PKI_CERT & get_cert();


	/*! \brief This function sets ASN1_UTCTIME::revdate.
	 *  \param c_revdate [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_revdate(const ASN1_UTCTIME * c_revdate);

	/*! \brief This function returns ASN1_UTCTIME::revdate.
	 *  \return The value of ASN1_UTCTIME::revdate.
	 */
	const ASN1_UTCTIME * get_revdate() const;

	/*! \brief This function returns ASN1_UTCTIME::revdate.
	 *  \return The value of ASN1_UTCTIME::revdate.
	 */
	ASN1_UTCTIME * get_revdate();


	/*! \brief This function copies a NewpkiPubRequestBodyRev to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiPubRequestBodyRev & other);
	/*! \brief This member is an empty instance of NewpkiPubRequestBodyRev.
	 */
	static NewpkiPubRequestBodyRev EmptyInstance;

private:
	PKI_CERT m_cert;
	ASN1_UTCTIME * m_revdate;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_PUB_REQUEST_BODY
*/
class NewpkiPubRequestBody : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiPubRequestBody();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiPubRequestBody(const NewpkiPubRequestBody & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiPubRequestBody();


	/*! \brief This function converts a NEWPKI_PUB_REQUEST_BODY to a NewpkiPubRequestBody.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_PUB_REQUEST_BODY * Datas);

	/*! \brief This function converts a NewpkiPubRequestBody to a NEWPKI_PUB_REQUEST_BODY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_PUB_REQUEST_BODY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_PUB_REQUEST_BODY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets X509::d.cert.
	 *  \param c_cert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_cert(const PKI_CERT & c_cert);

	/*! \brief This function returns X509::d.cert.
	 *  \return The value of X509::d.cert.
	 */
	const PKI_CERT & get_cert() const;

	/*! \brief This function returns X509::d.cert.
	 *  \return The value of X509::d.cert.
	 */
	PKI_CERT & get_cert();


	/*! \brief This function sets X509_CRL::d.crl.
	 *  \param c_crl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_crl(const PKI_CRL & c_crl);

	/*! \brief This function returns X509_CRL::d.crl.
	 *  \return The value of X509_CRL::d.crl.
	 */
	const PKI_CRL & get_crl() const;

	/*! \brief This function returns X509_CRL::d.crl.
	 *  \return The value of X509_CRL::d.crl.
	 */
	PKI_CRL & get_crl();


	/*! \brief This function sets NEWPKI_PUB_REQUEST_BODY_REV::d.rev.
	 *  \param c_rev [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_rev(const NewpkiPubRequestBodyRev & c_rev);

	/*! \brief This function returns NEWPKI_PUB_REQUEST_BODY_REV::d.rev.
	 *  \return The value of NEWPKI_PUB_REQUEST_BODY_REV::d.rev.
	 */
	const NewpkiPubRequestBodyRev & get_rev() const;

	/*! \brief This function returns NEWPKI_PUB_REQUEST_BODY_REV::d.rev.
	 *  \return The value of NEWPKI_PUB_REQUEST_BODY_REV::d.rev.
	 */
	NewpkiPubRequestBodyRev & get_rev();


	/*! \brief This function copies a NewpkiPubRequestBody to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiPubRequestBody & other);
	/*! \brief This member is an empty instance of NewpkiPubRequestBody.
	 */
	static NewpkiPubRequestBody EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	PKI_CERT * m_cert;
	PKI_CRL * m_crl;
	NewpkiPubRequestBodyRev * m_rev;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_PUB_REQUEST
*/
class NewpkiPubRequest : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiPubRequest();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiPubRequest(const NewpkiPubRequest & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiPubRequest();


	/*! \brief This function converts a NEWPKI_PUB_REQUEST to a NewpkiPubRequest.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_PUB_REQUEST * Datas);

	/*! \brief This function converts a NewpkiPubRequest to a NEWPKI_PUB_REQUEST.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_PUB_REQUEST ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_PUB_REQUEST.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509::ParentCerts.
	 *  \param c_parentcerts [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_parentcerts(const mVector< PKI_CERT > & c_parentcerts);

	/*! \brief This function returns X509::ParentCerts.
	 *  \return The value of X509::ParentCerts.
	 */
	const mVector< PKI_CERT > & get_parentcerts() const;

	/*! \brief This function returns X509::ParentCerts.
	 *  \return The value of X509::ParentCerts.
	 */
	mVector< PKI_CERT > & get_parentcerts();


	/*! \brief This function sets NEWPKI_PUB_REQUEST_BODY::body.
	 *  \param c_body [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_body(const NewpkiPubRequestBody & c_body);

	/*! \brief This function returns NEWPKI_PUB_REQUEST_BODY::body.
	 *  \return The value of NEWPKI_PUB_REQUEST_BODY::body.
	 */
	const NewpkiPubRequestBody & get_body() const;

	/*! \brief This function returns NEWPKI_PUB_REQUEST_BODY::body.
	 *  \return The value of NEWPKI_PUB_REQUEST_BODY::body.
	 */
	NewpkiPubRequestBody & get_body();


	/*! \brief This function sets ASN1_UTF8STRING::ldap_uid.
	 *  \param c_ldapUid [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapUid(const mString & c_ldapUid);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_uid.
	 *  \return The value of ASN1_UTF8STRING::ldap_uid.
	 */
	const mString & get_ldapUid() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_uid.
	 *  \return The value of ASN1_UTF8STRING::ldap_uid.
	 */
	mString & get_ldapUid();


	/*! \brief This function sets ASN1_UTF8STRING::object.
	 *  \param c_object [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_object(const mString & c_object);

	/*! \brief This function returns ASN1_UTF8STRING::object.
	 *  \return The value of ASN1_UTF8STRING::object.
	 */
	const mString & get_object() const;

	/*! \brief This function returns ASN1_UTF8STRING::object.
	 *  \return The value of ASN1_UTF8STRING::object.
	 */
	mString & get_object();


	/*! \brief This function copies a NewpkiPubRequest to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiPubRequest & other);
	/*! \brief This member is an empty instance of NewpkiPubRequest.
	 */
	static NewpkiPubRequest EmptyInstance;

private:
	mVector< PKI_CERT > m_parentcerts;
	NewpkiPubRequestBody m_body;
	mString m_ldapUid;
	mString m_object;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_BACKUP_REQUEST
*/
class NewpkiBackupRequest : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiBackupRequest();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiBackupRequest(const NewpkiBackupRequest & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiBackupRequest();


	/*! \brief This function converts a NEWPKI_BACKUP_REQUEST to a NewpkiBackupRequest.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_BACKUP_REQUEST * Datas);

	/*! \brief This function converts a NewpkiBackupRequest to a NEWPKI_BACKUP_REQUEST.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_BACKUP_REQUEST ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_BACKUP_REQUEST.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_OCTET_STRING::backup.
	 *  \param c_backup [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_backup(const Asn1OctetString & c_backup);

	/*! \brief This function returns ASN1_OCTET_STRING::backup.
	 *  \return The value of ASN1_OCTET_STRING::backup.
	 */
	const Asn1OctetString & get_backup() const;

	/*! \brief This function returns ASN1_OCTET_STRING::backup.
	 *  \return The value of ASN1_OCTET_STRING::backup.
	 */
	Asn1OctetString & get_backup();


	/*! \brief This function sets ASN1_OBJECT::sym_algo.
	 *  \param c_symAlgo [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_symAlgo(const ASN1_OBJECT * c_symAlgo);

	/*! \brief This function returns ASN1_OBJECT::sym_algo.
	 *  \return The value of ASN1_OBJECT::sym_algo.
	 */
	const ASN1_OBJECT * get_symAlgo() const;

	/*! \brief This function returns ASN1_OBJECT::sym_algo.
	 *  \return The value of ASN1_OBJECT::sym_algo.
	 */
	ASN1_OBJECT * get_symAlgo();


	/*! \brief This function copies a NewpkiBackupRequest to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiBackupRequest & other);
	/*! \brief This member is an empty instance of NewpkiBackupRequest.
	 */
	static NewpkiBackupRequest EmptyInstance;

private:
	Asn1OctetString m_backup;
	ASN1_OBJECT * m_symAlgo;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_EE_VAL_REQUEST
*/
class NewpkiEeValRequest : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiEeValRequest();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiEeValRequest(const NewpkiEeValRequest & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiEeValRequest();


	/*! \brief This function converts a NEWPKI_EE_VAL_REQUEST to a NewpkiEeValRequest.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_EE_VAL_REQUEST * Datas);

	/*! \brief This function converts a NewpkiEeValRequest to a NEWPKI_EE_VAL_REQUEST.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_EE_VAL_REQUEST ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_EE_VAL_REQUEST.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509_NAME::dn.
	 *  \param c_dn [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dn(const X509_NAME * c_dn);

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	const X509_NAME * get_dn() const;

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	X509_NAME * get_dn();


	/*! \brief This function sets ASN1_INTEGER::ra_id.
	 *  \param c_raId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_raId(unsigned long c_raId);

	/*! \brief This function returns ASN1_INTEGER::ra_id.
	 *  \return The value of ASN1_INTEGER::ra_id.
	 */
	unsigned long get_raId() const;


	/*! \brief This function copies a NewpkiEeValRequest to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiEeValRequest & other);
	/*! \brief This member is an empty instance of NewpkiEeValRequest.
	 */
	static NewpkiEeValRequest EmptyInstance;

private:
	X509_NAME * m_dn;
	unsigned long m_raId;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_EE_DN_PUBLISH
*/
class NewpkiEeDnPublish : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiEeDnPublish();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiEeDnPublish(const NewpkiEeDnPublish & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiEeDnPublish();


	/*! \brief This function converts a NEWPKI_EE_DN_PUBLISH to a NewpkiEeDnPublish.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_EE_DN_PUBLISH * Datas);

	/*! \brief This function converts a NewpkiEeDnPublish to a NEWPKI_EE_DN_PUBLISH.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_EE_DN_PUBLISH ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_EE_DN_PUBLISH.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509_NAME::dn.
	 *  \param c_dn [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dn(const X509_NAME * c_dn);

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	const X509_NAME * get_dn() const;

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	X509_NAME * get_dn();


	/*! \brief This function sets ASN1_INTEGER::ee_id.
	 *  \param c_eeId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeId(unsigned long c_eeId);

	/*! \brief This function returns ASN1_INTEGER::ee_id.
	 *  \return The value of ASN1_INTEGER::ee_id.
	 */
	unsigned long get_eeId() const;


	/*! \brief This function copies a NewpkiEeDnPublish to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiEeDnPublish & other);
	/*! \brief This member is an empty instance of NewpkiEeDnPublish.
	 */
	static NewpkiEeDnPublish EmptyInstance;

private:
	X509_NAME * m_dn;
	unsigned long m_eeId;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_EE_DN_REMOVE
*/
class NewpkiEeDnRemove : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiEeDnRemove();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiEeDnRemove(const NewpkiEeDnRemove & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiEeDnRemove();


	/*! \brief This function converts a NEWPKI_EE_DN_REMOVE to a NewpkiEeDnRemove.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_EE_DN_REMOVE * Datas);

	/*! \brief This function converts a NewpkiEeDnRemove to a NEWPKI_EE_DN_REMOVE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_EE_DN_REMOVE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_EE_DN_REMOVE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::ee_id.
	 *  \param c_eeId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeId(unsigned long c_eeId);

	/*! \brief This function returns ASN1_INTEGER::ee_id.
	 *  \return The value of ASN1_INTEGER::ee_id.
	 */
	unsigned long get_eeId() const;


	/*! \brief This function copies a NewpkiEeDnRemove to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiEeDnRemove & other);
	/*! \brief This member is an empty instance of NewpkiEeDnRemove.
	 */
	static NewpkiEeDnRemove EmptyInstance;

private:
	unsigned long m_eeId;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_EE_UPDATE_CERT_STATUS
*/
class NewpkiEeUpdateCertStatus : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiEeUpdateCertStatus();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiEeUpdateCertStatus(const NewpkiEeUpdateCertStatus & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiEeUpdateCertStatus();


	/*! \brief This function converts a NEWPKI_EE_UPDATE_CERT_STATUS to a NewpkiEeUpdateCertStatus.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_EE_UPDATE_CERT_STATUS * Datas);

	/*! \brief This function converts a NewpkiEeUpdateCertStatus to a NEWPKI_EE_UPDATE_CERT_STATUS.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_EE_UPDATE_CERT_STATUS ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_EE_UPDATE_CERT_STATUS.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509_CRL::last_crl.
	 *  \param c_lastCrl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_lastCrl(const PKI_CRL & c_lastCrl);

	/*! \brief This function returns X509_CRL::last_crl.
	 *  \return The value of X509_CRL::last_crl.
	 */
	const PKI_CRL & get_lastCrl() const;

	/*! \brief This function returns X509_CRL::last_crl.
	 *  \return The value of X509_CRL::last_crl.
	 */
	PKI_CRL & get_lastCrl();


	/*! \brief This function sets ASN1_INTEGER::ra_id.
	 *  \param c_raId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_raId(unsigned long c_raId);

	/*! \brief This function returns ASN1_INTEGER::ra_id.
	 *  \return The value of ASN1_INTEGER::ra_id.
	 */
	unsigned long get_raId() const;


	/*! \brief This function sets ASN1_INTEGER::status.
	 *  \param c_status [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_status(unsigned long c_status);

	/*! \brief This function returns ASN1_INTEGER::status.
	 *  \return The value of ASN1_INTEGER::status.
	 */
	unsigned long get_status() const;


	/*! \brief This function copies a NewpkiEeUpdateCertStatus to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiEeUpdateCertStatus & other);
	/*! \brief This member is an empty instance of NewpkiEeUpdateCertStatus.
	 */
	static NewpkiEeUpdateCertStatus EmptyInstance;

private:
	PKI_CRL m_lastCrl;
	unsigned long m_raId;
	unsigned long m_status;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_EE_CERT_PUBLISH
*/
class NewpkiEeCertPublish : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiEeCertPublish();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiEeCertPublish(const NewpkiEeCertPublish & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiEeCertPublish();


	/*! \brief This function converts a NEWPKI_EE_CERT_PUBLISH to a NewpkiEeCertPublish.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_EE_CERT_PUBLISH * Datas);

	/*! \brief This function converts a NewpkiEeCertPublish to a NEWPKI_EE_CERT_PUBLISH.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_EE_CERT_PUBLISH ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_EE_CERT_PUBLISH.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::ca_name.
	 *  \param c_caName [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_caName(const mString & c_caName);

	/*! \brief This function returns ASN1_UTF8STRING::ca_name.
	 *  \return The value of ASN1_UTF8STRING::ca_name.
	 */
	const mString & get_caName() const;

	/*! \brief This function returns ASN1_UTF8STRING::ca_name.
	 *  \return The value of ASN1_UTF8STRING::ca_name.
	 */
	mString & get_caName();


	/*! \brief This function sets X509::certificate.
	 *  \param c_certificate [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certificate(const PKI_CERT & c_certificate);

	/*! \brief This function returns X509::certificate.
	 *  \return The value of X509::certificate.
	 */
	const PKI_CERT & get_certificate() const;

	/*! \brief This function returns X509::certificate.
	 *  \return The value of X509::certificate.
	 */
	PKI_CERT & get_certificate();


	/*! \brief This function sets X509_CRL::last_crl.
	 *  \param c_lastCrl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_lastCrl(const PKI_CRL & c_lastCrl);

	/*! \brief This function returns X509_CRL::last_crl.
	 *  \return The value of X509_CRL::last_crl.
	 */
	const PKI_CRL & get_lastCrl() const;

	/*! \brief This function returns X509_CRL::last_crl.
	 *  \return The value of X509_CRL::last_crl.
	 */
	PKI_CRL & get_lastCrl();


	/*! \brief This function sets PKCS12::p12.
	 *  \param c_p12 [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_p12(const PKI_PKCS12 & c_p12);

	/*! \brief This function returns PKCS12::p12.
	 *  \return The value of PKCS12::p12.
	 */
	const PKI_PKCS12 & get_p12() const;

	/*! \brief This function returns PKCS12::p12.
	 *  \return The value of PKCS12::p12.
	 */
	PKI_PKCS12 & get_p12();


	/*! \brief This function sets PKCS7::p7b.
	 *  \param c_p7b [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_p7b(const PKI_P7B & c_p7b);

	/*! \brief This function returns PKCS7::p7b.
	 *  \return The value of PKCS7::p7b.
	 */
	const PKI_P7B & get_p7b() const;

	/*! \brief This function returns PKCS7::p7b.
	 *  \return The value of PKCS7::p7b.
	 */
	PKI_P7B & get_p7b();


	/*! \brief This function sets ASN1_INTEGER::profile_id.
	 *  \param c_profileId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_profileId(unsigned long c_profileId);

	/*! \brief This function returns ASN1_INTEGER::profile_id.
	 *  \return The value of ASN1_INTEGER::profile_id.
	 */
	unsigned long get_profileId() const;


	/*! \brief This function sets ASN1_INTEGER::ra_id.
	 *  \param c_raId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_raId(unsigned long c_raId);

	/*! \brief This function returns ASN1_INTEGER::ra_id.
	 *  \return The value of ASN1_INTEGER::ra_id.
	 */
	unsigned long get_raId() const;


	/*! \brief This function sets ASN1_INTEGER::type.
	 *  \param c_type [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_type(unsigned long c_type);

	/*! \brief This function returns ASN1_INTEGER::type.
	 *  \return The value of ASN1_INTEGER::type.
	 */
	unsigned long get_type() const;


	/*! \brief This function copies a NewpkiEeCertPublish to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiEeCertPublish & other);
	/*! \brief This member is an empty instance of NewpkiEeCertPublish.
	 */
	static NewpkiEeCertPublish EmptyInstance;

private:
	mString m_caName;
	PKI_CERT m_certificate;
	PKI_CRL m_lastCrl;
	PKI_PKCS12 m_p12;
	PKI_P7B m_p7b;
	unsigned long m_profileId;
	unsigned long m_raId;
	unsigned long m_type;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_REQUEST
*/
class NewpkiRequest : public NewPKIObject, public NewPKISignCryptObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiRequest();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiRequest(const NewpkiRequest & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiRequest();


	/*! \brief This function converts a NEWPKI_REQUEST to a NewpkiRequest.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_REQUEST * Datas);

	/*! \brief This function converts a NewpkiRequest to a NEWPKI_REQUEST.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_REQUEST ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_REQUEST.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets NEWPKI_BACKUP_REQUEST::d.backup_request.
	 *  \param c_backupRequest [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_backupRequest(const NewpkiBackupRequest & c_backupRequest);

	/*! \brief This function returns NEWPKI_BACKUP_REQUEST::d.backup_request.
	 *  \return The value of NEWPKI_BACKUP_REQUEST::d.backup_request.
	 */
	const NewpkiBackupRequest & get_backupRequest() const;

	/*! \brief This function returns NEWPKI_BACKUP_REQUEST::d.backup_request.
	 *  \return The value of NEWPKI_BACKUP_REQUEST::d.backup_request.
	 */
	NewpkiBackupRequest & get_backupRequest();


	/*! \brief This function sets NEWPKI_CERT_REQUEST::d.cert_request.
	 *  \param c_certRequest [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certRequest(const NewpkiCertRequest & c_certRequest);

	/*! \brief This function returns NEWPKI_CERT_REQUEST::d.cert_request.
	 *  \return The value of NEWPKI_CERT_REQUEST::d.cert_request.
	 */
	const NewpkiCertRequest & get_certRequest() const;

	/*! \brief This function returns NEWPKI_CERT_REQUEST::d.cert_request.
	 *  \return The value of NEWPKI_CERT_REQUEST::d.cert_request.
	 */
	NewpkiCertRequest & get_certRequest();


	/*! \brief This function sets NEWPKI_EE_CERT_PUBLISH::d.ee_cert_publish.
	 *  \param c_eeCertPublish [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeCertPublish(const NewpkiEeCertPublish & c_eeCertPublish);

	/*! \brief This function returns NEWPKI_EE_CERT_PUBLISH::d.ee_cert_publish.
	 *  \return The value of NEWPKI_EE_CERT_PUBLISH::d.ee_cert_publish.
	 */
	const NewpkiEeCertPublish & get_eeCertPublish() const;

	/*! \brief This function returns NEWPKI_EE_CERT_PUBLISH::d.ee_cert_publish.
	 *  \return The value of NEWPKI_EE_CERT_PUBLISH::d.ee_cert_publish.
	 */
	NewpkiEeCertPublish & get_eeCertPublish();


	/*! \brief This function sets NEWPKI_EE_REQUEST_CERT::d.ee_cert_request.
	 *  \param c_eeCertRequest [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeCertRequest(const NewpkiEeRequestCert & c_eeCertRequest);

	/*! \brief This function returns NEWPKI_EE_REQUEST_CERT::d.ee_cert_request.
	 *  \return The value of NEWPKI_EE_REQUEST_CERT::d.ee_cert_request.
	 */
	const NewpkiEeRequestCert & get_eeCertRequest() const;

	/*! \brief This function returns NEWPKI_EE_REQUEST_CERT::d.ee_cert_request.
	 *  \return The value of NEWPKI_EE_REQUEST_CERT::d.ee_cert_request.
	 */
	NewpkiEeRequestCert & get_eeCertRequest();


	/*! \brief This function sets NEWPKI_EE_REQUEST_REV::d.ee_cert_revoke.
	 *  \param c_eeCertRevoke [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeCertRevoke(const NewpkiEeRequestRev & c_eeCertRevoke);

	/*! \brief This function returns NEWPKI_EE_REQUEST_REV::d.ee_cert_revoke.
	 *  \return The value of NEWPKI_EE_REQUEST_REV::d.ee_cert_revoke.
	 */
	const NewpkiEeRequestRev & get_eeCertRevoke() const;

	/*! \brief This function returns NEWPKI_EE_REQUEST_REV::d.ee_cert_revoke.
	 *  \return The value of NEWPKI_EE_REQUEST_REV::d.ee_cert_revoke.
	 */
	NewpkiEeRequestRev & get_eeCertRevoke();


	/*! \brief This function sets NEWPKI_EE_DN_PUBLISH::d.ee_dn_publish.
	 *  \param c_eeDnPublish [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeDnPublish(const NewpkiEeDnPublish & c_eeDnPublish);

	/*! \brief This function returns NEWPKI_EE_DN_PUBLISH::d.ee_dn_publish.
	 *  \return The value of NEWPKI_EE_DN_PUBLISH::d.ee_dn_publish.
	 */
	const NewpkiEeDnPublish & get_eeDnPublish() const;

	/*! \brief This function returns NEWPKI_EE_DN_PUBLISH::d.ee_dn_publish.
	 *  \return The value of NEWPKI_EE_DN_PUBLISH::d.ee_dn_publish.
	 */
	NewpkiEeDnPublish & get_eeDnPublish();


	/*! \brief This function sets NEWPKI_EE_DN_REMOVE::d.ee_dn_remove.
	 *  \param c_eeDnRemove [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeDnRemove(const NewpkiEeDnRemove & c_eeDnRemove);

	/*! \brief This function returns NEWPKI_EE_DN_REMOVE::d.ee_dn_remove.
	 *  \return The value of NEWPKI_EE_DN_REMOVE::d.ee_dn_remove.
	 */
	const NewpkiEeDnRemove & get_eeDnRemove() const;

	/*! \brief This function returns NEWPKI_EE_DN_REMOVE::d.ee_dn_remove.
	 *  \return The value of NEWPKI_EE_DN_REMOVE::d.ee_dn_remove.
	 */
	NewpkiEeDnRemove & get_eeDnRemove();


	/*! \brief This function sets NEWPKI_EE_UPDATE_CERT_STATUS::d.ee_upd_cert_status.
	 *  \param c_eeUpdCertStatus [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeUpdCertStatus(const NewpkiEeUpdateCertStatus & c_eeUpdCertStatus);

	/*! \brief This function returns NEWPKI_EE_UPDATE_CERT_STATUS::d.ee_upd_cert_status.
	 *  \return The value of NEWPKI_EE_UPDATE_CERT_STATUS::d.ee_upd_cert_status.
	 */
	const NewpkiEeUpdateCertStatus & get_eeUpdCertStatus() const;

	/*! \brief This function returns NEWPKI_EE_UPDATE_CERT_STATUS::d.ee_upd_cert_status.
	 *  \return The value of NEWPKI_EE_UPDATE_CERT_STATUS::d.ee_upd_cert_status.
	 */
	NewpkiEeUpdateCertStatus & get_eeUpdCertStatus();


	/*! \brief This function sets NEWPKI_EE_VAL_REQUEST::d.ee_val_request.
	 *  \param c_eeValRequest [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeValRequest(const NewpkiEeValRequest & c_eeValRequest);

	/*! \brief This function returns NEWPKI_EE_VAL_REQUEST::d.ee_val_request.
	 *  \return The value of NEWPKI_EE_VAL_REQUEST::d.ee_val_request.
	 */
	const NewpkiEeValRequest & get_eeValRequest() const;

	/*! \brief This function returns NEWPKI_EE_VAL_REQUEST::d.ee_val_request.
	 *  \return The value of NEWPKI_EE_VAL_REQUEST::d.ee_val_request.
	 */
	NewpkiEeValRequest & get_eeValRequest();


	/*! \brief This function sets NEWPKI_PUB_REQUEST::d.pub_request.
	 *  \param c_pubRequest [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_pubRequest(const NewpkiPubRequest & c_pubRequest);

	/*! \brief This function returns NEWPKI_PUB_REQUEST::d.pub_request.
	 *  \return The value of NEWPKI_PUB_REQUEST::d.pub_request.
	 */
	const NewpkiPubRequest & get_pubRequest() const;

	/*! \brief This function returns NEWPKI_PUB_REQUEST::d.pub_request.
	 *  \return The value of NEWPKI_PUB_REQUEST::d.pub_request.
	 */
	NewpkiPubRequest & get_pubRequest();


	/*! \brief This function sets NEWPKI_REV_REQUEST::d.rev_request.
	 *  \param c_revRequest [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_revRequest(const NewpkiRevRequest & c_revRequest);

	/*! \brief This function returns NEWPKI_REV_REQUEST::d.rev_request.
	 *  \return The value of NEWPKI_REV_REQUEST::d.rev_request.
	 */
	const NewpkiRevRequest & get_revRequest() const;

	/*! \brief This function returns NEWPKI_REV_REQUEST::d.rev_request.
	 *  \return The value of NEWPKI_REV_REQUEST::d.rev_request.
	 */
	NewpkiRevRequest & get_revRequest();


	/*! \brief This function sets NEWPKI_SUSP_REQUEST::d.susp_request.
	 *  \param c_suspRequest [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_suspRequest(const NewpkiSuspRequest & c_suspRequest);

	/*! \brief This function returns NEWPKI_SUSP_REQUEST::d.susp_request.
	 *  \return The value of NEWPKI_SUSP_REQUEST::d.susp_request.
	 */
	const NewpkiSuspRequest & get_suspRequest() const;

	/*! \brief This function returns NEWPKI_SUSP_REQUEST::d.susp_request.
	 *  \return The value of NEWPKI_SUSP_REQUEST::d.susp_request.
	 */
	NewpkiSuspRequest & get_suspRequest();


	/*! \brief This function sets NEWPKI_UNSUSP_REQUEST::d.unsusp_request.
	 *  \param c_unsuspRequest [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_unsuspRequest(const NewpkiUnsuspRequest & c_unsuspRequest);

	/*! \brief This function returns NEWPKI_UNSUSP_REQUEST::d.unsusp_request.
	 *  \return The value of NEWPKI_UNSUSP_REQUEST::d.unsusp_request.
	 */
	const NewpkiUnsuspRequest & get_unsuspRequest() const;

	/*! \brief This function returns NEWPKI_UNSUSP_REQUEST::d.unsusp_request.
	 *  \return The value of NEWPKI_UNSUSP_REQUEST::d.unsusp_request.
	 */
	NewpkiUnsuspRequest & get_unsuspRequest();


	/*! \brief This function converts the local instance to a Asn1EncryptSign.
	 *  \param cryptinfo [OUT] The result.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \param sig_md [IN] The hash algorythm.
	 *  \param crypt_cypher [IN] The symetrical encipherment hash.
	 *  \return true on success, false on failure.
	 */
	bool to_SignEncrypt(Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey, const EVP_MD * sig_md, const EVP_CIPHER * crypt_cypher) const;

	/*! \brief This function loads the local instance from a Asn1EncryptSign.
	 *  \param cryptinfo [IN] The signed and crypted datas.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \return true on success, false on failure.
	 */
	bool from_SignEncrypt(const Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey);


	/*! \brief This function copies a NewpkiRequest to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiRequest & other);
	/*! \brief This member is an empty instance of NewpkiRequest.
	 */
	static NewpkiRequest EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	NewpkiBackupRequest * m_backupRequest;
	NewpkiCertRequest * m_certRequest;
	NewpkiEeCertPublish * m_eeCertPublish;
	NewpkiEeRequestCert * m_eeCertRequest;
	NewpkiEeRequestRev * m_eeCertRevoke;
	NewpkiEeDnPublish * m_eeDnPublish;
	NewpkiEeDnRemove * m_eeDnRemove;
	NewpkiEeUpdateCertStatus * m_eeUpdCertStatus;
	NewpkiEeValRequest * m_eeValRequest;
	NewpkiPubRequest * m_pubRequest;
	NewpkiRevRequest * m_revRequest;
	NewpkiSuspRequest * m_suspRequest;
	NewpkiUnsuspRequest * m_unsuspRequest;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of CRYPTED_NEWPKI_REQUEST
*/
class CryptedNewpkiRequest : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	CryptedNewpkiRequest();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	CryptedNewpkiRequest(const CryptedNewpkiRequest & other);

	/*! \brief This is the destructor.
	 */
	~CryptedNewpkiRequest();


	/*! \brief This function converts a CRYPTED_NEWPKI_REQUEST to a CryptedNewpkiRequest.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const CRYPTED_NEWPKI_REQUEST * Datas);

	/*! \brief This function converts a CryptedNewpkiRequest to a CRYPTED_NEWPKI_REQUEST.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(CRYPTED_NEWPKI_REQUEST ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of CRYPTED_NEWPKI_REQUEST.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_ENCRYPT_SIGN::c_request.
	 *  \param c_request [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_request(const Asn1EncryptSign & c_request);

	/*! \brief This function returns ASN1_ENCRYPT_SIGN::c_request.
	 *  \return The value of ASN1_ENCRYPT_SIGN::c_request.
	 */
	const Asn1EncryptSign & get_request() const;

	/*! \brief This function returns ASN1_ENCRYPT_SIGN::c_request.
	 *  \return The value of ASN1_ENCRYPT_SIGN::c_request.
	 */
	Asn1EncryptSign & get_request();


	/*! \brief This function sets X509_PUBKEY::recipient.
	 *  \param c_recipient [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_recipient(const X509_PUBKEY * c_recipient);

	/*! \brief This function returns X509_PUBKEY::recipient.
	 *  \return The value of X509_PUBKEY::recipient.
	 */
	const X509_PUBKEY * get_recipient() const;

	/*! \brief This function returns X509_PUBKEY::recipient.
	 *  \return The value of X509_PUBKEY::recipient.
	 */
	X509_PUBKEY * get_recipient();


	/*! \brief This function sets X509_PUBKEY::sender.
	 *  \param c_sender [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_sender(const X509_PUBKEY * c_sender);

	/*! \brief This function returns X509_PUBKEY::sender.
	 *  \return The value of X509_PUBKEY::sender.
	 */
	const X509_PUBKEY * get_sender() const;

	/*! \brief This function returns X509_PUBKEY::sender.
	 *  \return The value of X509_PUBKEY::sender.
	 */
	X509_PUBKEY * get_sender();


	/*! \brief This function sets ASN1_OCTET_STRING::transactionID.
	 *  \param c_transactionid [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_transactionid(const Asn1OctetString & c_transactionid);

	/*! \brief This function returns ASN1_OCTET_STRING::transactionID.
	 *  \return The value of ASN1_OCTET_STRING::transactionID.
	 */
	const Asn1OctetString & get_transactionid() const;

	/*! \brief This function returns ASN1_OCTET_STRING::transactionID.
	 *  \return The value of ASN1_OCTET_STRING::transactionID.
	 */
	Asn1OctetString & get_transactionid();


	/*! \brief This funciton generates the PEM representation of the local instance.
	 *  \param PemDatas [OUT] the PEM output.
	 *  \return true on success, false on failure.
	 */
	bool to_PEM(mString & PemDatas) const;

	/*! \brief This function loads the local instance from a PEM representation.
	 *  \param PemDatas [IN] the PEM input.
	 *  \return true on success, false on failure.
	 */
	bool from_PEM(const mString & PemDatas);


	/*! \brief This function copies a CryptedNewpkiRequest to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const CryptedNewpkiRequest & other);
	/*! \brief This member is an empty instance of CryptedNewpkiRequest.
	 */
	static CryptedNewpkiRequest EmptyInstance;

private:
	Asn1EncryptSign m_request;
	X509_PUBKEY * m_recipient;
	X509_PUBKEY * m_sender;
	Asn1OctetString m_transactionid;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of CRYPTED_NEWPKI_REQUESTS
*/
class CryptedNewpkiRequests : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	CryptedNewpkiRequests();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	CryptedNewpkiRequests(const CryptedNewpkiRequests & other);

	/*! \brief This is the destructor.
	 */
	~CryptedNewpkiRequests();


	/*! \brief This function converts a CRYPTED_NEWPKI_REQUESTS to a CryptedNewpkiRequests.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const CRYPTED_NEWPKI_REQUESTS * Datas);

	/*! \brief This function converts a CryptedNewpkiRequests to a CRYPTED_NEWPKI_REQUESTS.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(CRYPTED_NEWPKI_REQUESTS ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of CRYPTED_NEWPKI_REQUESTS.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets CRYPTED_NEWPKI_REQUEST::requests.
	 *  \param c_requests [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_requests(const mVector< CryptedNewpkiRequest > & c_requests);

	/*! \brief This function returns CRYPTED_NEWPKI_REQUEST::requests.
	 *  \return The value of CRYPTED_NEWPKI_REQUEST::requests.
	 */
	const mVector< CryptedNewpkiRequest > & get_requests() const;

	/*! \brief This function returns CRYPTED_NEWPKI_REQUEST::requests.
	 *  \return The value of CRYPTED_NEWPKI_REQUEST::requests.
	 */
	mVector< CryptedNewpkiRequest > & get_requests();


	/*! \brief This function copies a CryptedNewpkiRequests to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const CryptedNewpkiRequests & other);
	/*! \brief This member is an empty instance of CryptedNewpkiRequests.
	 */
	static CryptedNewpkiRequests EmptyInstance;

private:
	mVector< CryptedNewpkiRequest > m_requests;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_CERT_RESPONSE
*/
class NewpkiCertResponse : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiCertResponse();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiCertResponse(const NewpkiCertResponse & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiCertResponse();


	/*! \brief This function converts a NEWPKI_CERT_RESPONSE to a NewpkiCertResponse.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_CERT_RESPONSE * Datas);

	/*! \brief This function converts a NewpkiCertResponse to a NEWPKI_CERT_RESPONSE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_CERT_RESPONSE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_CERT_RESPONSE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509::ParentCerts.
	 *  \param c_parentcerts [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_parentcerts(const mVector< PKI_CERT > & c_parentcerts);

	/*! \brief This function returns X509::ParentCerts.
	 *  \return The value of X509::ParentCerts.
	 */
	const mVector< PKI_CERT > & get_parentcerts() const;

	/*! \brief This function returns X509::ParentCerts.
	 *  \return The value of X509::ParentCerts.
	 */
	mVector< PKI_CERT > & get_parentcerts();


	/*! \brief This function sets X509::certificate.
	 *  \param c_certificate [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certificate(const PKI_CERT & c_certificate);

	/*! \brief This function returns X509::certificate.
	 *  \return The value of X509::certificate.
	 */
	const PKI_CERT & get_certificate() const;

	/*! \brief This function returns X509::certificate.
	 *  \return The value of X509::certificate.
	 */
	PKI_CERT & get_certificate();


	/*! \brief This function sets ERROR_ENTRY::errors.
	 *  \param c_errors [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_errors(const mVector< ErrorEntry > & c_errors);

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	const mVector< ErrorEntry > & get_errors() const;

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	mVector< ErrorEntry > & get_errors();


	/*! \brief This function sets ASN1_INTEGER::id.
	 *  \param c_id [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_id(unsigned long c_id);

	/*! \brief This function returns ASN1_INTEGER::id.
	 *  \return The value of ASN1_INTEGER::id.
	 */
	unsigned long get_id() const;


	/*! \brief This function sets X509_CRL::last_crl.
	 *  \param c_lastCrl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_lastCrl(const PKI_CRL & c_lastCrl);

	/*! \brief This function returns X509_CRL::last_crl.
	 *  \return The value of X509_CRL::last_crl.
	 */
	const PKI_CRL & get_lastCrl() const;

	/*! \brief This function returns X509_CRL::last_crl.
	 *  \return The value of X509_CRL::last_crl.
	 */
	PKI_CRL & get_lastCrl();


	/*! \brief This function sets ASN1_INTEGER::status.
	 *  \param c_status [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_status(unsigned long c_status);

	/*! \brief This function returns ASN1_INTEGER::status.
	 *  \return The value of ASN1_INTEGER::status.
	 */
	unsigned long get_status() const;


	/*! \brief This function copies a NewpkiCertResponse to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiCertResponse & other);
	/*! \brief This member is an empty instance of NewpkiCertResponse.
	 */
	static NewpkiCertResponse EmptyInstance;

private:
	mVector< PKI_CERT > m_parentcerts;
	PKI_CERT m_certificate;
	mVector< ErrorEntry > m_errors;
	unsigned long m_id;
	PKI_CRL m_lastCrl;
	unsigned long m_status;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_EE_CERT_RESPONSE
*/
class NewpkiEeCertResponse : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiEeCertResponse();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiEeCertResponse(const NewpkiEeCertResponse & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiEeCertResponse();


	/*! \brief This function converts a NEWPKI_EE_CERT_RESPONSE to a NewpkiEeCertResponse.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_EE_CERT_RESPONSE * Datas);

	/*! \brief This function converts a NewpkiEeCertResponse to a NEWPKI_EE_CERT_RESPONSE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_EE_CERT_RESPONSE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_EE_CERT_RESPONSE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509::certificate.
	 *  \param c_certificate [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certificate(const PKI_CERT & c_certificate);

	/*! \brief This function returns X509::certificate.
	 *  \return The value of X509::certificate.
	 */
	const PKI_CERT & get_certificate() const;

	/*! \brief This function returns X509::certificate.
	 *  \return The value of X509::certificate.
	 */
	PKI_CERT & get_certificate();


	/*! \brief This function sets ERROR_ENTRY::errors.
	 *  \param c_errors [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_errors(const mVector< ErrorEntry > & c_errors);

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	const mVector< ErrorEntry > & get_errors() const;

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	mVector< ErrorEntry > & get_errors();


	/*! \brief This function sets X509_CRL::last_crl.
	 *  \param c_lastCrl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_lastCrl(const PKI_CRL & c_lastCrl);

	/*! \brief This function returns X509_CRL::last_crl.
	 *  \return The value of X509_CRL::last_crl.
	 */
	const PKI_CRL & get_lastCrl() const;

	/*! \brief This function returns X509_CRL::last_crl.
	 *  \return The value of X509_CRL::last_crl.
	 */
	PKI_CRL & get_lastCrl();


	/*! \brief This function sets PKCS12::p12.
	 *  \param c_p12 [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_p12(const PKI_PKCS12 & c_p12);

	/*! \brief This function returns PKCS12::p12.
	 *  \return The value of PKCS12::p12.
	 */
	const PKI_PKCS12 & get_p12() const;

	/*! \brief This function returns PKCS12::p12.
	 *  \return The value of PKCS12::p12.
	 */
	PKI_PKCS12 & get_p12();


	/*! \brief This function sets PKCS7::p7b.
	 *  \param c_p7b [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_p7b(const PKI_P7B & c_p7b);

	/*! \brief This function returns PKCS7::p7b.
	 *  \return The value of PKCS7::p7b.
	 */
	const PKI_P7B & get_p7b() const;

	/*! \brief This function returns PKCS7::p7b.
	 *  \return The value of PKCS7::p7b.
	 */
	PKI_P7B & get_p7b();


	/*! \brief This function sets ASN1_INTEGER::ra_id.
	 *  \param c_raId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_raId(unsigned long c_raId);

	/*! \brief This function returns ASN1_INTEGER::ra_id.
	 *  \return The value of ASN1_INTEGER::ra_id.
	 */
	unsigned long get_raId() const;


	/*! \brief This function sets ASN1_INTEGER::status.
	 *  \param c_status [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_status(unsigned long c_status);

	/*! \brief This function returns ASN1_INTEGER::status.
	 *  \return The value of ASN1_INTEGER::status.
	 */
	unsigned long get_status() const;


	/*! \brief This function copies a NewpkiEeCertResponse to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiEeCertResponse & other);
	/*! \brief This member is an empty instance of NewpkiEeCertResponse.
	 */
	static NewpkiEeCertResponse EmptyInstance;

private:
	PKI_CERT m_certificate;
	mVector< ErrorEntry > m_errors;
	PKI_CRL m_lastCrl;
	PKI_PKCS12 m_p12;
	PKI_P7B m_p7b;
	unsigned long m_raId;
	unsigned long m_status;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_REV_RESPONSE
*/
class NewpkiRevResponse : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiRevResponse();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiRevResponse(const NewpkiRevResponse & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiRevResponse();


	/*! \brief This function converts a NEWPKI_REV_RESPONSE to a NewpkiRevResponse.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_REV_RESPONSE * Datas);

	/*! \brief This function converts a NewpkiRevResponse to a NEWPKI_REV_RESPONSE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_REV_RESPONSE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_REV_RESPONSE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::cert_status.
	 *  \param c_certStatus [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certStatus(unsigned long c_certStatus);

	/*! \brief This function returns ASN1_INTEGER::cert_status.
	 *  \return The value of ASN1_INTEGER::cert_status.
	 */
	unsigned long get_certStatus() const;


	/*! \brief This function sets ERROR_ENTRY::errors.
	 *  \param c_errors [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_errors(const mVector< ErrorEntry > & c_errors);

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	const mVector< ErrorEntry > & get_errors() const;

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	mVector< ErrorEntry > & get_errors();


	/*! \brief This function sets ASN1_INTEGER::id.
	 *  \param c_id [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_id(unsigned long c_id);

	/*! \brief This function returns ASN1_INTEGER::id.
	 *  \return The value of ASN1_INTEGER::id.
	 */
	unsigned long get_id() const;


	/*! \brief This function sets X509_CRL::last_crl.
	 *  \param c_lastCrl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_lastCrl(const PKI_CRL & c_lastCrl);

	/*! \brief This function returns X509_CRL::last_crl.
	 *  \return The value of X509_CRL::last_crl.
	 */
	const PKI_CRL & get_lastCrl() const;

	/*! \brief This function returns X509_CRL::last_crl.
	 *  \return The value of X509_CRL::last_crl.
	 */
	PKI_CRL & get_lastCrl();


	/*! \brief This function sets ASN1_INTEGER::status.
	 *  \param c_status [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_status(unsigned long c_status);

	/*! \brief This function returns ASN1_INTEGER::status.
	 *  \return The value of ASN1_INTEGER::status.
	 */
	unsigned long get_status() const;


	/*! \brief This function copies a NewpkiRevResponse to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiRevResponse & other);
	/*! \brief This member is an empty instance of NewpkiRevResponse.
	 */
	static NewpkiRevResponse EmptyInstance;

private:
	unsigned long m_certStatus;
	mVector< ErrorEntry > m_errors;
	unsigned long m_id;
	PKI_CRL m_lastCrl;
	unsigned long m_status;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_EE_REV_RESPONSE
*/
class NewpkiEeRevResponse : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiEeRevResponse();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiEeRevResponse(const NewpkiEeRevResponse & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiEeRevResponse();


	/*! \brief This function converts a NEWPKI_EE_REV_RESPONSE to a NewpkiEeRevResponse.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_EE_REV_RESPONSE * Datas);

	/*! \brief This function converts a NewpkiEeRevResponse to a NEWPKI_EE_REV_RESPONSE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_EE_REV_RESPONSE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_EE_REV_RESPONSE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::cert_status.
	 *  \param c_certStatus [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certStatus(unsigned long c_certStatus);

	/*! \brief This function returns ASN1_INTEGER::cert_status.
	 *  \return The value of ASN1_INTEGER::cert_status.
	 */
	unsigned long get_certStatus() const;


	/*! \brief This function sets ERROR_ENTRY::errors.
	 *  \param c_errors [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_errors(const mVector< ErrorEntry > & c_errors);

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	const mVector< ErrorEntry > & get_errors() const;

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	mVector< ErrorEntry > & get_errors();


	/*! \brief This function sets X509_CRL::last_crl.
	 *  \param c_lastCrl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_lastCrl(const PKI_CRL & c_lastCrl);

	/*! \brief This function returns X509_CRL::last_crl.
	 *  \return The value of X509_CRL::last_crl.
	 */
	const PKI_CRL & get_lastCrl() const;

	/*! \brief This function returns X509_CRL::last_crl.
	 *  \return The value of X509_CRL::last_crl.
	 */
	PKI_CRL & get_lastCrl();


	/*! \brief This function sets ASN1_INTEGER::ra_id.
	 *  \param c_raId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_raId(unsigned long c_raId);

	/*! \brief This function returns ASN1_INTEGER::ra_id.
	 *  \return The value of ASN1_INTEGER::ra_id.
	 */
	unsigned long get_raId() const;


	/*! \brief This function sets ASN1_INTEGER::status.
	 *  \param c_status [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_status(unsigned long c_status);

	/*! \brief This function returns ASN1_INTEGER::status.
	 *  \return The value of ASN1_INTEGER::status.
	 */
	unsigned long get_status() const;


	/*! \brief This function copies a NewpkiEeRevResponse to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiEeRevResponse & other);
	/*! \brief This member is an empty instance of NewpkiEeRevResponse.
	 */
	static NewpkiEeRevResponse EmptyInstance;

private:
	unsigned long m_certStatus;
	mVector< ErrorEntry > m_errors;
	PKI_CRL m_lastCrl;
	unsigned long m_raId;
	unsigned long m_status;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_SUSP_RESPONSE
*/
class NewpkiSuspResponse : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiSuspResponse();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiSuspResponse(const NewpkiSuspResponse & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiSuspResponse();


	/*! \brief This function converts a NEWPKI_SUSP_RESPONSE to a NewpkiSuspResponse.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_SUSP_RESPONSE * Datas);

	/*! \brief This function converts a NewpkiSuspResponse to a NEWPKI_SUSP_RESPONSE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_SUSP_RESPONSE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_SUSP_RESPONSE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::cert_status.
	 *  \param c_certStatus [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certStatus(unsigned long c_certStatus);

	/*! \brief This function returns ASN1_INTEGER::cert_status.
	 *  \return The value of ASN1_INTEGER::cert_status.
	 */
	unsigned long get_certStatus() const;


	/*! \brief This function sets ERROR_ENTRY::errors.
	 *  \param c_errors [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_errors(const mVector< ErrorEntry > & c_errors);

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	const mVector< ErrorEntry > & get_errors() const;

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	mVector< ErrorEntry > & get_errors();


	/*! \brief This function sets ASN1_INTEGER::id.
	 *  \param c_id [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_id(unsigned long c_id);

	/*! \brief This function returns ASN1_INTEGER::id.
	 *  \return The value of ASN1_INTEGER::id.
	 */
	unsigned long get_id() const;


	/*! \brief This function sets X509_CRL::last_crl.
	 *  \param c_lastCrl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_lastCrl(const PKI_CRL & c_lastCrl);

	/*! \brief This function returns X509_CRL::last_crl.
	 *  \return The value of X509_CRL::last_crl.
	 */
	const PKI_CRL & get_lastCrl() const;

	/*! \brief This function returns X509_CRL::last_crl.
	 *  \return The value of X509_CRL::last_crl.
	 */
	PKI_CRL & get_lastCrl();


	/*! \brief This function sets ASN1_INTEGER::status.
	 *  \param c_status [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_status(unsigned long c_status);

	/*! \brief This function returns ASN1_INTEGER::status.
	 *  \return The value of ASN1_INTEGER::status.
	 */
	unsigned long get_status() const;


	/*! \brief This function copies a NewpkiSuspResponse to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiSuspResponse & other);
	/*! \brief This member is an empty instance of NewpkiSuspResponse.
	 */
	static NewpkiSuspResponse EmptyInstance;

private:
	unsigned long m_certStatus;
	mVector< ErrorEntry > m_errors;
	unsigned long m_id;
	PKI_CRL m_lastCrl;
	unsigned long m_status;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_UNSUSP_RESPONSE
*/
class NewpkiUnsuspResponse : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiUnsuspResponse();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiUnsuspResponse(const NewpkiUnsuspResponse & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiUnsuspResponse();


	/*! \brief This function converts a NEWPKI_UNSUSP_RESPONSE to a NewpkiUnsuspResponse.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_UNSUSP_RESPONSE * Datas);

	/*! \brief This function converts a NewpkiUnsuspResponse to a NEWPKI_UNSUSP_RESPONSE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_UNSUSP_RESPONSE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_UNSUSP_RESPONSE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::cert_status.
	 *  \param c_certStatus [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certStatus(unsigned long c_certStatus);

	/*! \brief This function returns ASN1_INTEGER::cert_status.
	 *  \return The value of ASN1_INTEGER::cert_status.
	 */
	unsigned long get_certStatus() const;


	/*! \brief This function sets ERROR_ENTRY::errors.
	 *  \param c_errors [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_errors(const mVector< ErrorEntry > & c_errors);

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	const mVector< ErrorEntry > & get_errors() const;

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	mVector< ErrorEntry > & get_errors();


	/*! \brief This function sets ASN1_INTEGER::id.
	 *  \param c_id [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_id(unsigned long c_id);

	/*! \brief This function returns ASN1_INTEGER::id.
	 *  \return The value of ASN1_INTEGER::id.
	 */
	unsigned long get_id() const;


	/*! \brief This function sets X509_CRL::last_crl.
	 *  \param c_lastCrl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_lastCrl(const PKI_CRL & c_lastCrl);

	/*! \brief This function returns X509_CRL::last_crl.
	 *  \return The value of X509_CRL::last_crl.
	 */
	const PKI_CRL & get_lastCrl() const;

	/*! \brief This function returns X509_CRL::last_crl.
	 *  \return The value of X509_CRL::last_crl.
	 */
	PKI_CRL & get_lastCrl();


	/*! \brief This function sets ASN1_INTEGER::status.
	 *  \param c_status [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_status(unsigned long c_status);

	/*! \brief This function returns ASN1_INTEGER::status.
	 *  \return The value of ASN1_INTEGER::status.
	 */
	unsigned long get_status() const;


	/*! \brief This function copies a NewpkiUnsuspResponse to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiUnsuspResponse & other);
	/*! \brief This member is an empty instance of NewpkiUnsuspResponse.
	 */
	static NewpkiUnsuspResponse EmptyInstance;

private:
	unsigned long m_certStatus;
	mVector< ErrorEntry > m_errors;
	unsigned long m_id;
	PKI_CRL m_lastCrl;
	unsigned long m_status;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_PUB_RESPONSE
*/
class NewpkiPubResponse : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiPubResponse();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiPubResponse(const NewpkiPubResponse & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiPubResponse();


	/*! \brief This function converts a NEWPKI_PUB_RESPONSE to a NewpkiPubResponse.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_PUB_RESPONSE * Datas);

	/*! \brief This function converts a NewpkiPubResponse to a NEWPKI_PUB_RESPONSE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_PUB_RESPONSE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_PUB_RESPONSE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ERROR_ENTRY::errors.
	 *  \param c_errors [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_errors(const mVector< ErrorEntry > & c_errors);

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	const mVector< ErrorEntry > & get_errors() const;

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	mVector< ErrorEntry > & get_errors();


	/*! \brief This function sets ASN1_UTF8STRING::object.
	 *  \param c_object [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_object(const mString & c_object);

	/*! \brief This function returns ASN1_UTF8STRING::object.
	 *  \return The value of ASN1_UTF8STRING::object.
	 */
	const mString & get_object() const;

	/*! \brief This function returns ASN1_UTF8STRING::object.
	 *  \return The value of ASN1_UTF8STRING::object.
	 */
	mString & get_object();


	/*! \brief This function sets ASN1_INTEGER::status.
	 *  \param c_status [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_status(unsigned long c_status);

	/*! \brief This function returns ASN1_INTEGER::status.
	 *  \return The value of ASN1_INTEGER::status.
	 */
	unsigned long get_status() const;


	/*! \brief This function sets ASN1_INTEGER::type.
	 *  \param c_type [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_type(unsigned long c_type);

	/*! \brief This function returns ASN1_INTEGER::type.
	 *  \return The value of ASN1_INTEGER::type.
	 */
	unsigned long get_type() const;


	/*! \brief This function copies a NewpkiPubResponse to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiPubResponse & other);
	/*! \brief This member is an empty instance of NewpkiPubResponse.
	 */
	static NewpkiPubResponse EmptyInstance;

private:
	mVector< ErrorEntry > m_errors;
	mString m_object;
	unsigned long m_status;
	unsigned long m_type;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_EE_VAL_RESPONSE
*/
class NewpkiEeValResponse : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiEeValResponse();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiEeValResponse(const NewpkiEeValResponse & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiEeValResponse();


	/*! \brief This function converts a NEWPKI_EE_VAL_RESPONSE to a NewpkiEeValResponse.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_EE_VAL_RESPONSE * Datas);

	/*! \brief This function converts a NewpkiEeValResponse to a NEWPKI_EE_VAL_RESPONSE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_EE_VAL_RESPONSE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_EE_VAL_RESPONSE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509_NAME::dn.
	 *  \param c_dn [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dn(const X509_NAME * c_dn);

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	const X509_NAME * get_dn() const;

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	X509_NAME * get_dn();


	/*! \brief This function sets ASN1_INTEGER::ra_id.
	 *  \param c_raId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_raId(unsigned long c_raId);

	/*! \brief This function returns ASN1_INTEGER::ra_id.
	 *  \return The value of ASN1_INTEGER::ra_id.
	 */
	unsigned long get_raId() const;


	/*! \brief This function copies a NewpkiEeValResponse to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiEeValResponse & other);
	/*! \brief This member is an empty instance of NewpkiEeValResponse.
	 */
	static NewpkiEeValResponse EmptyInstance;

private:
	X509_NAME * m_dn;
	unsigned long m_raId;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_EE_DN_PUBLISH_RESPONSE
*/
class NewpkiEeDnPublishResponse : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiEeDnPublishResponse();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiEeDnPublishResponse(const NewpkiEeDnPublishResponse & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiEeDnPublishResponse();


	/*! \brief This function converts a NEWPKI_EE_DN_PUBLISH_RESPONSE to a NewpkiEeDnPublishResponse.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_EE_DN_PUBLISH_RESPONSE * Datas);

	/*! \brief This function converts a NewpkiEeDnPublishResponse to a NEWPKI_EE_DN_PUBLISH_RESPONSE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_EE_DN_PUBLISH_RESPONSE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_EE_DN_PUBLISH_RESPONSE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ERROR_ENTRY::errors.
	 *  \param c_errors [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_errors(const mVector< ErrorEntry > & c_errors);

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	const mVector< ErrorEntry > & get_errors() const;

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	mVector< ErrorEntry > & get_errors();


	/*! \brief This function copies a NewpkiEeDnPublishResponse to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiEeDnPublishResponse & other);
	/*! \brief This member is an empty instance of NewpkiEeDnPublishResponse.
	 */
	static NewpkiEeDnPublishResponse EmptyInstance;

private:
	mVector< ErrorEntry > m_errors;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_EE_DN_REMOVE_RESPONSE
*/
class NewpkiEeDnRemoveResponse : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiEeDnRemoveResponse();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiEeDnRemoveResponse(const NewpkiEeDnRemoveResponse & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiEeDnRemoveResponse();


	/*! \brief This function converts a NEWPKI_EE_DN_REMOVE_RESPONSE to a NewpkiEeDnRemoveResponse.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_EE_DN_REMOVE_RESPONSE * Datas);

	/*! \brief This function converts a NewpkiEeDnRemoveResponse to a NEWPKI_EE_DN_REMOVE_RESPONSE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_EE_DN_REMOVE_RESPONSE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_EE_DN_REMOVE_RESPONSE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ERROR_ENTRY::errors.
	 *  \param c_errors [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_errors(const mVector< ErrorEntry > & c_errors);

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	const mVector< ErrorEntry > & get_errors() const;

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	mVector< ErrorEntry > & get_errors();


	/*! \brief This function copies a NewpkiEeDnRemoveResponse to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiEeDnRemoveResponse & other);
	/*! \brief This member is an empty instance of NewpkiEeDnRemoveResponse.
	 */
	static NewpkiEeDnRemoveResponse EmptyInstance;

private:
	mVector< ErrorEntry > m_errors;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_EE_UPDATE_CERT_STATUS_RESPONSE
*/
class NewpkiEeUpdateCertStatusResponse : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiEeUpdateCertStatusResponse();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiEeUpdateCertStatusResponse(const NewpkiEeUpdateCertStatusResponse & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiEeUpdateCertStatusResponse();


	/*! \brief This function converts a NEWPKI_EE_UPDATE_CERT_STATUS_RESPONSE to a NewpkiEeUpdateCertStatusResponse.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_EE_UPDATE_CERT_STATUS_RESPONSE * Datas);

	/*! \brief This function converts a NewpkiEeUpdateCertStatusResponse to a NEWPKI_EE_UPDATE_CERT_STATUS_RESPONSE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_EE_UPDATE_CERT_STATUS_RESPONSE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_EE_UPDATE_CERT_STATUS_RESPONSE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ERROR_ENTRY::errors.
	 *  \param c_errors [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_errors(const mVector< ErrorEntry > & c_errors);

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	const mVector< ErrorEntry > & get_errors() const;

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	mVector< ErrorEntry > & get_errors();


	/*! \brief This function copies a NewpkiEeUpdateCertStatusResponse to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiEeUpdateCertStatusResponse & other);
	/*! \brief This member is an empty instance of NewpkiEeUpdateCertStatusResponse.
	 */
	static NewpkiEeUpdateCertStatusResponse EmptyInstance;

private:
	mVector< ErrorEntry > m_errors;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_EE_CERT_PUBLISH_RESPONSE
*/
class NewpkiEeCertPublishResponse : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiEeCertPublishResponse();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiEeCertPublishResponse(const NewpkiEeCertPublishResponse & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiEeCertPublishResponse();


	/*! \brief This function converts a NEWPKI_EE_CERT_PUBLISH_RESPONSE to a NewpkiEeCertPublishResponse.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_EE_CERT_PUBLISH_RESPONSE * Datas);

	/*! \brief This function converts a NewpkiEeCertPublishResponse to a NEWPKI_EE_CERT_PUBLISH_RESPONSE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_EE_CERT_PUBLISH_RESPONSE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_EE_CERT_PUBLISH_RESPONSE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ERROR_ENTRY::errors.
	 *  \param c_errors [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_errors(const mVector< ErrorEntry > & c_errors);

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	const mVector< ErrorEntry > & get_errors() const;

	/*! \brief This function returns ERROR_ENTRY::errors.
	 *  \return The value of ERROR_ENTRY::errors.
	 */
	mVector< ErrorEntry > & get_errors();


	/*! \brief This function copies a NewpkiEeCertPublishResponse to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiEeCertPublishResponse & other);
	/*! \brief This member is an empty instance of NewpkiEeCertPublishResponse.
	 */
	static NewpkiEeCertPublishResponse EmptyInstance;

private:
	mVector< ErrorEntry > m_errors;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_RESPONSE
*/
class NewpkiResponse : public NewPKIObject, public NewPKISignCryptObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiResponse();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiResponse(const NewpkiResponse & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiResponse();


	/*! \brief This function converts a NEWPKI_RESPONSE to a NewpkiResponse.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_RESPONSE * Datas);

	/*! \brief This function converts a NewpkiResponse to a NEWPKI_RESPONSE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_RESPONSE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_RESPONSE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets NEWPKI_CERT_RESPONSE::d.cert_response.
	 *  \param c_certResponse [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certResponse(const NewpkiCertResponse & c_certResponse);

	/*! \brief This function returns NEWPKI_CERT_RESPONSE::d.cert_response.
	 *  \return The value of NEWPKI_CERT_RESPONSE::d.cert_response.
	 */
	const NewpkiCertResponse & get_certResponse() const;

	/*! \brief This function returns NEWPKI_CERT_RESPONSE::d.cert_response.
	 *  \return The value of NEWPKI_CERT_RESPONSE::d.cert_response.
	 */
	NewpkiCertResponse & get_certResponse();


	/*! \brief This function sets NEWPKI_EE_CERT_PUBLISH_RESPONSE::d.ee_cert_publish_response.
	 *  \param c_eeCertPublishResponse [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeCertPublishResponse(const NewpkiEeCertPublishResponse & c_eeCertPublishResponse);

	/*! \brief This function returns NEWPKI_EE_CERT_PUBLISH_RESPONSE::d.ee_cert_publish_response.
	 *  \return The value of NEWPKI_EE_CERT_PUBLISH_RESPONSE::d.ee_cert_publish_response.
	 */
	const NewpkiEeCertPublishResponse & get_eeCertPublishResponse() const;

	/*! \brief This function returns NEWPKI_EE_CERT_PUBLISH_RESPONSE::d.ee_cert_publish_response.
	 *  \return The value of NEWPKI_EE_CERT_PUBLISH_RESPONSE::d.ee_cert_publish_response.
	 */
	NewpkiEeCertPublishResponse & get_eeCertPublishResponse();


	/*! \brief This function sets NEWPKI_EE_CERT_RESPONSE::d.ee_cert_response.
	 *  \param c_eeCertResponse [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeCertResponse(const NewpkiEeCertResponse & c_eeCertResponse);

	/*! \brief This function returns NEWPKI_EE_CERT_RESPONSE::d.ee_cert_response.
	 *  \return The value of NEWPKI_EE_CERT_RESPONSE::d.ee_cert_response.
	 */
	const NewpkiEeCertResponse & get_eeCertResponse() const;

	/*! \brief This function returns NEWPKI_EE_CERT_RESPONSE::d.ee_cert_response.
	 *  \return The value of NEWPKI_EE_CERT_RESPONSE::d.ee_cert_response.
	 */
	NewpkiEeCertResponse & get_eeCertResponse();


	/*! \brief This function sets NEWPKI_EE_DN_PUBLISH_RESPONSE::d.ee_dn_publish_response.
	 *  \param c_eeDnPublishResponse [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeDnPublishResponse(const NewpkiEeDnPublishResponse & c_eeDnPublishResponse);

	/*! \brief This function returns NEWPKI_EE_DN_PUBLISH_RESPONSE::d.ee_dn_publish_response.
	 *  \return The value of NEWPKI_EE_DN_PUBLISH_RESPONSE::d.ee_dn_publish_response.
	 */
	const NewpkiEeDnPublishResponse & get_eeDnPublishResponse() const;

	/*! \brief This function returns NEWPKI_EE_DN_PUBLISH_RESPONSE::d.ee_dn_publish_response.
	 *  \return The value of NEWPKI_EE_DN_PUBLISH_RESPONSE::d.ee_dn_publish_response.
	 */
	NewpkiEeDnPublishResponse & get_eeDnPublishResponse();


	/*! \brief This function sets NEWPKI_EE_DN_REMOVE_RESPONSE::d.ee_dn_remove_response.
	 *  \param c_eeDnRemoveResponse [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeDnRemoveResponse(const NewpkiEeDnRemoveResponse & c_eeDnRemoveResponse);

	/*! \brief This function returns NEWPKI_EE_DN_REMOVE_RESPONSE::d.ee_dn_remove_response.
	 *  \return The value of NEWPKI_EE_DN_REMOVE_RESPONSE::d.ee_dn_remove_response.
	 */
	const NewpkiEeDnRemoveResponse & get_eeDnRemoveResponse() const;

	/*! \brief This function returns NEWPKI_EE_DN_REMOVE_RESPONSE::d.ee_dn_remove_response.
	 *  \return The value of NEWPKI_EE_DN_REMOVE_RESPONSE::d.ee_dn_remove_response.
	 */
	NewpkiEeDnRemoveResponse & get_eeDnRemoveResponse();


	/*! \brief This function sets NEWPKI_EE_REV_RESPONSE::d.ee_rev_response.
	 *  \param c_eeRevResponse [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeRevResponse(const NewpkiEeRevResponse & c_eeRevResponse);

	/*! \brief This function returns NEWPKI_EE_REV_RESPONSE::d.ee_rev_response.
	 *  \return The value of NEWPKI_EE_REV_RESPONSE::d.ee_rev_response.
	 */
	const NewpkiEeRevResponse & get_eeRevResponse() const;

	/*! \brief This function returns NEWPKI_EE_REV_RESPONSE::d.ee_rev_response.
	 *  \return The value of NEWPKI_EE_REV_RESPONSE::d.ee_rev_response.
	 */
	NewpkiEeRevResponse & get_eeRevResponse();


	/*! \brief This function sets NEWPKI_EE_UPDATE_CERT_STATUS_RESPONSE::d.ee_upd_cert_status_response.
	 *  \param c_eeUpdCertStatusResponse [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeUpdCertStatusResponse(const NewpkiEeUpdateCertStatusResponse & c_eeUpdCertStatusResponse);

	/*! \brief This function returns NEWPKI_EE_UPDATE_CERT_STATUS_RESPONSE::d.ee_upd_cert_status_response.
	 *  \return The value of NEWPKI_EE_UPDATE_CERT_STATUS_RESPONSE::d.ee_upd_cert_status_response.
	 */
	const NewpkiEeUpdateCertStatusResponse & get_eeUpdCertStatusResponse() const;

	/*! \brief This function returns NEWPKI_EE_UPDATE_CERT_STATUS_RESPONSE::d.ee_upd_cert_status_response.
	 *  \return The value of NEWPKI_EE_UPDATE_CERT_STATUS_RESPONSE::d.ee_upd_cert_status_response.
	 */
	NewpkiEeUpdateCertStatusResponse & get_eeUpdCertStatusResponse();


	/*! \brief This function sets NEWPKI_EE_VAL_RESPONSE::d.ee_val_response.
	 *  \param c_eeValResponse [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeValResponse(const NewpkiEeValResponse & c_eeValResponse);

	/*! \brief This function returns NEWPKI_EE_VAL_RESPONSE::d.ee_val_response.
	 *  \return The value of NEWPKI_EE_VAL_RESPONSE::d.ee_val_response.
	 */
	const NewpkiEeValResponse & get_eeValResponse() const;

	/*! \brief This function returns NEWPKI_EE_VAL_RESPONSE::d.ee_val_response.
	 *  \return The value of NEWPKI_EE_VAL_RESPONSE::d.ee_val_response.
	 */
	NewpkiEeValResponse & get_eeValResponse();


	/*! \brief This function sets ERROR_ENTRY::d.errors.
	 *  \param c_errors [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_errors(const mVector< ErrorEntry > & c_errors);

	/*! \brief This function returns ERROR_ENTRY::d.errors.
	 *  \return The value of ERROR_ENTRY::d.errors.
	 */
	const mVector< ErrorEntry > & get_errors() const;

	/*! \brief This function returns ERROR_ENTRY::d.errors.
	 *  \return The value of ERROR_ENTRY::d.errors.
	 */
	mVector< ErrorEntry > & get_errors();


	/*! \brief This function sets NEWPKI_PUB_RESPONSE::d.pub_response.
	 *  \param c_pubResponse [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_pubResponse(const NewpkiPubResponse & c_pubResponse);

	/*! \brief This function returns NEWPKI_PUB_RESPONSE::d.pub_response.
	 *  \return The value of NEWPKI_PUB_RESPONSE::d.pub_response.
	 */
	const NewpkiPubResponse & get_pubResponse() const;

	/*! \brief This function returns NEWPKI_PUB_RESPONSE::d.pub_response.
	 *  \return The value of NEWPKI_PUB_RESPONSE::d.pub_response.
	 */
	NewpkiPubResponse & get_pubResponse();


	/*! \brief This function sets NEWPKI_REV_RESPONSE::d.rev_response.
	 *  \param c_revResponse [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_revResponse(const NewpkiRevResponse & c_revResponse);

	/*! \brief This function returns NEWPKI_REV_RESPONSE::d.rev_response.
	 *  \return The value of NEWPKI_REV_RESPONSE::d.rev_response.
	 */
	const NewpkiRevResponse & get_revResponse() const;

	/*! \brief This function returns NEWPKI_REV_RESPONSE::d.rev_response.
	 *  \return The value of NEWPKI_REV_RESPONSE::d.rev_response.
	 */
	NewpkiRevResponse & get_revResponse();


	/*! \brief This function sets NEWPKI_SUSP_RESPONSE::d.susp_response.
	 *  \param c_suspResponse [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_suspResponse(const NewpkiSuspResponse & c_suspResponse);

	/*! \brief This function returns NEWPKI_SUSP_RESPONSE::d.susp_response.
	 *  \return The value of NEWPKI_SUSP_RESPONSE::d.susp_response.
	 */
	const NewpkiSuspResponse & get_suspResponse() const;

	/*! \brief This function returns NEWPKI_SUSP_RESPONSE::d.susp_response.
	 *  \return The value of NEWPKI_SUSP_RESPONSE::d.susp_response.
	 */
	NewpkiSuspResponse & get_suspResponse();


	/*! \brief This function sets NEWPKI_UNSUSP_RESPONSE::d.unsusp_response.
	 *  \param c_unsuspResponse [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_unsuspResponse(const NewpkiUnsuspResponse & c_unsuspResponse);

	/*! \brief This function returns NEWPKI_UNSUSP_RESPONSE::d.unsusp_response.
	 *  \return The value of NEWPKI_UNSUSP_RESPONSE::d.unsusp_response.
	 */
	const NewpkiUnsuspResponse & get_unsuspResponse() const;

	/*! \brief This function returns NEWPKI_UNSUSP_RESPONSE::d.unsusp_response.
	 *  \return The value of NEWPKI_UNSUSP_RESPONSE::d.unsusp_response.
	 */
	NewpkiUnsuspResponse & get_unsuspResponse();


	/*! \brief This function converts the local instance to a Asn1EncryptSign.
	 *  \param cryptinfo [OUT] The result.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \param sig_md [IN] The hash algorythm.
	 *  \param crypt_cypher [IN] The symetrical encipherment hash.
	 *  \return true on success, false on failure.
	 */
	bool to_SignEncrypt(Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey, const EVP_MD * sig_md, const EVP_CIPHER * crypt_cypher) const;

	/*! \brief This function loads the local instance from a Asn1EncryptSign.
	 *  \param cryptinfo [IN] The signed and crypted datas.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \return true on success, false on failure.
	 */
	bool from_SignEncrypt(const Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey);


	/*! \brief This function copies a NewpkiResponse to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiResponse & other);
	/*! \brief This member is an empty instance of NewpkiResponse.
	 */
	static NewpkiResponse EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	NewpkiCertResponse * m_certResponse;
	NewpkiEeCertPublishResponse * m_eeCertPublishResponse;
	NewpkiEeCertResponse * m_eeCertResponse;
	NewpkiEeDnPublishResponse * m_eeDnPublishResponse;
	NewpkiEeDnRemoveResponse * m_eeDnRemoveResponse;
	NewpkiEeRevResponse * m_eeRevResponse;
	NewpkiEeUpdateCertStatusResponse * m_eeUpdCertStatusResponse;
	NewpkiEeValResponse * m_eeValResponse;
	mVector< ErrorEntry > * m_errors;
	NewpkiPubResponse * m_pubResponse;
	NewpkiRevResponse * m_revResponse;
	NewpkiSuspResponse * m_suspResponse;
	NewpkiUnsuspResponse * m_unsuspResponse;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of CRYPTED_NEWPKI_RESPONSE
*/
class CryptedNewpkiResponse : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	CryptedNewpkiResponse();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	CryptedNewpkiResponse(const CryptedNewpkiResponse & other);

	/*! \brief This is the destructor.
	 */
	~CryptedNewpkiResponse();


	/*! \brief This function converts a CRYPTED_NEWPKI_RESPONSE to a CryptedNewpkiResponse.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const CRYPTED_NEWPKI_RESPONSE * Datas);

	/*! \brief This function converts a CryptedNewpkiResponse to a CRYPTED_NEWPKI_RESPONSE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(CRYPTED_NEWPKI_RESPONSE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of CRYPTED_NEWPKI_RESPONSE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_ENCRYPT_SIGN::c_response.
	 *  \param c_response [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_response(const Asn1EncryptSign & c_response);

	/*! \brief This function returns ASN1_ENCRYPT_SIGN::c_response.
	 *  \return The value of ASN1_ENCRYPT_SIGN::c_response.
	 */
	const Asn1EncryptSign & get_response() const;

	/*! \brief This function returns ASN1_ENCRYPT_SIGN::c_response.
	 *  \return The value of ASN1_ENCRYPT_SIGN::c_response.
	 */
	Asn1EncryptSign & get_response();


	/*! \brief This function sets X509_PUBKEY::recipient.
	 *  \param c_recipient [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_recipient(const X509_PUBKEY * c_recipient);

	/*! \brief This function returns X509_PUBKEY::recipient.
	 *  \return The value of X509_PUBKEY::recipient.
	 */
	const X509_PUBKEY * get_recipient() const;

	/*! \brief This function returns X509_PUBKEY::recipient.
	 *  \return The value of X509_PUBKEY::recipient.
	 */
	X509_PUBKEY * get_recipient();


	/*! \brief This function sets X509_PUBKEY::sender.
	 *  \param c_sender [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_sender(const X509_PUBKEY * c_sender);

	/*! \brief This function returns X509_PUBKEY::sender.
	 *  \return The value of X509_PUBKEY::sender.
	 */
	const X509_PUBKEY * get_sender() const;

	/*! \brief This function returns X509_PUBKEY::sender.
	 *  \return The value of X509_PUBKEY::sender.
	 */
	X509_PUBKEY * get_sender();


	/*! \brief This function sets ASN1_OCTET_STRING::transactionID.
	 *  \param c_transactionid [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_transactionid(const Asn1OctetString & c_transactionid);

	/*! \brief This function returns ASN1_OCTET_STRING::transactionID.
	 *  \return The value of ASN1_OCTET_STRING::transactionID.
	 */
	const Asn1OctetString & get_transactionid() const;

	/*! \brief This function returns ASN1_OCTET_STRING::transactionID.
	 *  \return The value of ASN1_OCTET_STRING::transactionID.
	 */
	Asn1OctetString & get_transactionid();


	/*! \brief This funciton generates the PEM representation of the local instance.
	 *  \param PemDatas [OUT] the PEM output.
	 *  \return true on success, false on failure.
	 */
	bool to_PEM(mString & PemDatas) const;

	/*! \brief This function loads the local instance from a PEM representation.
	 *  \param PemDatas [IN] the PEM input.
	 *  \return true on success, false on failure.
	 */
	bool from_PEM(const mString & PemDatas);


	/*! \brief This function copies a CryptedNewpkiResponse to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const CryptedNewpkiResponse & other);
	/*! \brief This member is an empty instance of CryptedNewpkiResponse.
	 */
	static CryptedNewpkiResponse EmptyInstance;

private:
	Asn1EncryptSign m_response;
	X509_PUBKEY * m_recipient;
	X509_PUBKEY * m_sender;
	Asn1OctetString m_transactionid;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of CRYPTED_NEWPKI_RESPONSES
*/
class CryptedNewpkiResponses : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	CryptedNewpkiResponses();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	CryptedNewpkiResponses(const CryptedNewpkiResponses & other);

	/*! \brief This is the destructor.
	 */
	~CryptedNewpkiResponses();


	/*! \brief This function converts a CRYPTED_NEWPKI_RESPONSES to a CryptedNewpkiResponses.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const CRYPTED_NEWPKI_RESPONSES * Datas);

	/*! \brief This function converts a CryptedNewpkiResponses to a CRYPTED_NEWPKI_RESPONSES.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(CRYPTED_NEWPKI_RESPONSES ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of CRYPTED_NEWPKI_RESPONSES.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets CRYPTED_NEWPKI_RESPONSE::responses.
	 *  \param c_responses [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_responses(const mVector< CryptedNewpkiResponse > & c_responses);

	/*! \brief This function returns CRYPTED_NEWPKI_RESPONSE::responses.
	 *  \return The value of CRYPTED_NEWPKI_RESPONSE::responses.
	 */
	const mVector< CryptedNewpkiResponse > & get_responses() const;

	/*! \brief This function returns CRYPTED_NEWPKI_RESPONSE::responses.
	 *  \return The value of CRYPTED_NEWPKI_RESPONSE::responses.
	 */
	mVector< CryptedNewpkiResponse > & get_responses();


	/*! \brief This function copies a CryptedNewpkiResponses to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const CryptedNewpkiResponses & other);
	/*! \brief This member is an empty instance of CryptedNewpkiResponses.
	 */
	static CryptedNewpkiResponses EmptyInstance;

private:
	mVector< CryptedNewpkiResponse > m_responses;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of WAITING_NEWPKI_OBJECT_BODY
*/
class WaitingNewpkiObjectBody : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	WaitingNewpkiObjectBody();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	WaitingNewpkiObjectBody(const WaitingNewpkiObjectBody & other);

	/*! \brief This is the destructor.
	 */
	~WaitingNewpkiObjectBody();


	/*! \brief This function converts a WAITING_NEWPKI_OBJECT_BODY to a WaitingNewpkiObjectBody.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const WAITING_NEWPKI_OBJECT_BODY * Datas);

	/*! \brief This function converts a WaitingNewpkiObjectBody to a WAITING_NEWPKI_OBJECT_BODY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(WAITING_NEWPKI_OBJECT_BODY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of WAITING_NEWPKI_OBJECT_BODY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets CRYPTED_NEWPKI_REQUEST::d.request.
	 *  \param c_request [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_request(const CryptedNewpkiRequest & c_request);

	/*! \brief This function returns CRYPTED_NEWPKI_REQUEST::d.request.
	 *  \return The value of CRYPTED_NEWPKI_REQUEST::d.request.
	 */
	const CryptedNewpkiRequest & get_request() const;

	/*! \brief This function returns CRYPTED_NEWPKI_REQUEST::d.request.
	 *  \return The value of CRYPTED_NEWPKI_REQUEST::d.request.
	 */
	CryptedNewpkiRequest & get_request();


	/*! \brief This function sets CRYPTED_NEWPKI_RESPONSE::d.response.
	 *  \param c_response [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_response(const CryptedNewpkiResponse & c_response);

	/*! \brief This function returns CRYPTED_NEWPKI_RESPONSE::d.response.
	 *  \return The value of CRYPTED_NEWPKI_RESPONSE::d.response.
	 */
	const CryptedNewpkiResponse & get_response() const;

	/*! \brief This function returns CRYPTED_NEWPKI_RESPONSE::d.response.
	 *  \return The value of CRYPTED_NEWPKI_RESPONSE::d.response.
	 */
	CryptedNewpkiResponse & get_response();


	/*! \brief This function sets ASN1_OCTET_STRING::d.transactionID.
	 *  \param c_transactionid [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_transactionid(const Asn1OctetString & c_transactionid);

	/*! \brief This function returns ASN1_OCTET_STRING::d.transactionID.
	 *  \return The value of ASN1_OCTET_STRING::d.transactionID.
	 */
	const Asn1OctetString & get_transactionid() const;

	/*! \brief This function returns ASN1_OCTET_STRING::d.transactionID.
	 *  \return The value of ASN1_OCTET_STRING::d.transactionID.
	 */
	Asn1OctetString & get_transactionid();


	/*! \brief This function copies a WaitingNewpkiObjectBody to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const WaitingNewpkiObjectBody & other);
	/*! \brief This member is an empty instance of WaitingNewpkiObjectBody.
	 */
	static WaitingNewpkiObjectBody EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	CryptedNewpkiRequest * m_request;
	CryptedNewpkiResponse * m_response;
	Asn1OctetString * m_transactionid;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of TRANSACTION_IDS
*/
class TransactionIds : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	TransactionIds();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	TransactionIds(const TransactionIds & other);

	/*! \brief This is the destructor.
	 */
	~TransactionIds();


	/*! \brief This function converts a TRANSACTION_IDS to a TransactionIds.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const TRANSACTION_IDS * Datas);

	/*! \brief This function converts a TransactionIds to a TRANSACTION_IDS.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(TRANSACTION_IDS ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of TRANSACTION_IDS.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_OCTET_STRING::transactionIds.
	 *  \param c_transactionids [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_transactionids(const mVector< Asn1OctetString > & c_transactionids);

	/*! \brief This function returns ASN1_OCTET_STRING::transactionIds.
	 *  \return The value of ASN1_OCTET_STRING::transactionIds.
	 */
	const mVector< Asn1OctetString > & get_transactionids() const;

	/*! \brief This function returns ASN1_OCTET_STRING::transactionIds.
	 *  \return The value of ASN1_OCTET_STRING::transactionIds.
	 */
	mVector< Asn1OctetString > & get_transactionids();


	/*! \brief This function copies a TransactionIds to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const TransactionIds & other);
	/*! \brief This member is an empty instance of TransactionIds.
	 */
	static TransactionIds EmptyInstance;

private:
	mVector< Asn1OctetString > m_transactionids;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of WAITING_NEWPKI_OBJECT
*/
class WaitingNewpkiObject : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	WaitingNewpkiObject();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	WaitingNewpkiObject(const WaitingNewpkiObject & other);

	/*! \brief This is the destructor.
	 */
	~WaitingNewpkiObject();


	/*! \brief This function converts a WAITING_NEWPKI_OBJECT to a WaitingNewpkiObject.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const WAITING_NEWPKI_OBJECT * Datas);

	/*! \brief This function converts a WaitingNewpkiObject to a WAITING_NEWPKI_OBJECT.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(WAITING_NEWPKI_OBJECT ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of WAITING_NEWPKI_OBJECT.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets WAITING_NEWPKI_OBJECT_BODY::object.
	 *  \param c_object [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_object(const WaitingNewpkiObjectBody & c_object);

	/*! \brief This function returns WAITING_NEWPKI_OBJECT_BODY::object.
	 *  \return The value of WAITING_NEWPKI_OBJECT_BODY::object.
	 */
	const WaitingNewpkiObjectBody & get_object() const;

	/*! \brief This function returns WAITING_NEWPKI_OBJECT_BODY::object.
	 *  \return The value of WAITING_NEWPKI_OBJECT_BODY::object.
	 */
	WaitingNewpkiObjectBody & get_object();


	/*! \brief This function sets X509_PUBKEY::rep_path.
	 *  \param c_repPath [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_repPath(const STACK_OF(X509_PUBKEY) * c_repPath);

	/*! \brief This function returns X509_PUBKEY::rep_path.
	 *  \return The value of X509_PUBKEY::rep_path.
	 */
	const STACK_OF(X509_PUBKEY) * get_repPath() const;

	/*! \brief This function returns X509_PUBKEY::rep_path.
	 *  \return The value of X509_PUBKEY::rep_path.
	 */
	STACK_OF(X509_PUBKEY) * get_repPath();


	/*! \brief This funciton generates the PEM representation of the local instance.
	 *  \param PemDatas [OUT] the PEM output.
	 *  \return true on success, false on failure.
	 */
	bool to_PEM(mString & PemDatas) const;

	/*! \brief This function loads the local instance from a PEM representation.
	 *  \param PemDatas [IN] the PEM input.
	 *  \return true on success, false on failure.
	 */
	bool from_PEM(const mString & PemDatas);


	/*! \brief This function copies a WaitingNewpkiObject to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const WaitingNewpkiObject & other);
	/*! \brief This member is an empty instance of WaitingNewpkiObject.
	 */
	static WaitingNewpkiObject EmptyInstance;

private:
	WaitingNewpkiObjectBody m_object;
	STACK_OF(X509_PUBKEY) * m_repPath;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of OBJECTS_LIST_SYNCHRO
*/
class ObjectsListSynchro : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	ObjectsListSynchro();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	ObjectsListSynchro(const ObjectsListSynchro & other);

	/*! \brief This is the destructor.
	 */
	~ObjectsListSynchro();


	/*! \brief This function converts a OBJECTS_LIST_SYNCHRO to a ObjectsListSynchro.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const OBJECTS_LIST_SYNCHRO * Datas);

	/*! \brief This function converts a ObjectsListSynchro to a OBJECTS_LIST_SYNCHRO.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(OBJECTS_LIST_SYNCHRO ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of OBJECTS_LIST_SYNCHRO.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets TRANSACTION_IDS::delete_responses.
	 *  \param c_deleteResponses [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_deleteResponses(const TransactionIds & c_deleteResponses);

	/*! \brief This function returns TRANSACTION_IDS::delete_responses.
	 *  \return The value of TRANSACTION_IDS::delete_responses.
	 */
	const TransactionIds & get_deleteResponses() const;

	/*! \brief This function returns TRANSACTION_IDS::delete_responses.
	 *  \return The value of TRANSACTION_IDS::delete_responses.
	 */
	TransactionIds & get_deleteResponses();


	/*! \brief This function sets TRANSACTION_IDS::known_requests.
	 *  \param c_knownRequests [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_knownRequests(const TransactionIds & c_knownRequests);

	/*! \brief This function returns TRANSACTION_IDS::known_requests.
	 *  \return The value of TRANSACTION_IDS::known_requests.
	 */
	const TransactionIds & get_knownRequests() const;

	/*! \brief This function returns TRANSACTION_IDS::known_requests.
	 *  \return The value of TRANSACTION_IDS::known_requests.
	 */
	TransactionIds & get_knownRequests();


	/*! \brief This function sets TRANSACTION_IDS::known_responses.
	 *  \param c_knownResponses [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_knownResponses(const TransactionIds & c_knownResponses);

	/*! \brief This function returns TRANSACTION_IDS::known_responses.
	 *  \return The value of TRANSACTION_IDS::known_responses.
	 */
	const TransactionIds & get_knownResponses() const;

	/*! \brief This function returns TRANSACTION_IDS::known_responses.
	 *  \return The value of TRANSACTION_IDS::known_responses.
	 */
	TransactionIds & get_knownResponses();


	/*! \brief This function sets WAITING_NEWPKI_OBJECT::objects.
	 *  \param c_objects [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_objects(const mVector< WaitingNewpkiObject > & c_objects);

	/*! \brief This function returns WAITING_NEWPKI_OBJECT::objects.
	 *  \return The value of WAITING_NEWPKI_OBJECT::objects.
	 */
	const mVector< WaitingNewpkiObject > & get_objects() const;

	/*! \brief This function returns WAITING_NEWPKI_OBJECT::objects.
	 *  \return The value of WAITING_NEWPKI_OBJECT::objects.
	 */
	mVector< WaitingNewpkiObject > & get_objects();


	/*! \brief This function sets TRANSACTION_IDS::want_requests.
	 *  \param c_wantRequests [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_wantRequests(const TransactionIds & c_wantRequests);

	/*! \brief This function returns TRANSACTION_IDS::want_requests.
	 *  \return The value of TRANSACTION_IDS::want_requests.
	 */
	const TransactionIds & get_wantRequests() const;

	/*! \brief This function returns TRANSACTION_IDS::want_requests.
	 *  \return The value of TRANSACTION_IDS::want_requests.
	 */
	TransactionIds & get_wantRequests();


	/*! \brief This function sets TRANSACTION_IDS::want_responses.
	 *  \param c_wantResponses [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_wantResponses(const TransactionIds & c_wantResponses);

	/*! \brief This function returns TRANSACTION_IDS::want_responses.
	 *  \return The value of TRANSACTION_IDS::want_responses.
	 */
	const TransactionIds & get_wantResponses() const;

	/*! \brief This function returns TRANSACTION_IDS::want_responses.
	 *  \return The value of TRANSACTION_IDS::want_responses.
	 */
	TransactionIds & get_wantResponses();


	/*! \brief This function copies a ObjectsListSynchro to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const ObjectsListSynchro & other);
	/*! \brief This member is an empty instance of ObjectsListSynchro.
	 */
	static ObjectsListSynchro EmptyInstance;

private:
	TransactionIds m_deleteResponses;
	TransactionIds m_knownRequests;
	TransactionIds m_knownResponses;
	mVector< WaitingNewpkiObject > m_objects;
	TransactionIds m_wantRequests;
	TransactionIds m_wantResponses;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of INTERNAL_CA_KEY
*/
class InternalCaKey : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	InternalCaKey();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	InternalCaKey(const InternalCaKey & other);

	/*! \brief This is the destructor.
	 */
	~InternalCaKey();


	/*! \brief This function converts a INTERNAL_CA_KEY to a InternalCaKey.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const INTERNAL_CA_KEY * Datas);

	/*! \brief This function converts a InternalCaKey to a INTERNAL_CA_KEY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(INTERNAL_CA_KEY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of INTERNAL_CA_KEY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets ASN1_UTF8STRING::d.keyid.
	 *  \param c_keyid [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_keyid(const mString & c_keyid);

	/*! \brief This function returns ASN1_UTF8STRING::d.keyid.
	 *  \return The value of ASN1_UTF8STRING::d.keyid.
	 */
	const mString & get_keyid() const;

	/*! \brief This function returns ASN1_UTF8STRING::d.keyid.
	 *  \return The value of ASN1_UTF8STRING::d.keyid.
	 */
	mString & get_keyid();


	/*! \brief This function sets RSA::d.privkey.
	 *  \param c_privkey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_privkey(const RSA * c_privkey);

	/*! \brief This function returns RSA::d.privkey.
	 *  \return The value of RSA::d.privkey.
	 */
	const RSA * get_privkey() const;

	/*! \brief This function returns RSA::d.privkey.
	 *  \return The value of RSA::d.privkey.
	 */
	RSA * get_privkey();


	/*! \brief This function copies a InternalCaKey to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const InternalCaKey & other);
	/*! \brief This member is an empty instance of InternalCaKey.
	 */
	static InternalCaKey EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	mString * m_keyid;
	RSA * m_privkey;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of INTERNAL_CA
*/
class InternalCa : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	InternalCa();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	InternalCa(const InternalCa & other);

	/*! \brief This is the destructor.
	 */
	~InternalCa();


	/*! \brief This function converts a INTERNAL_CA to a InternalCa.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const INTERNAL_CA * Datas);

	/*! \brief This function converts a InternalCa to a INTERNAL_CA.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(INTERNAL_CA ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of INTERNAL_CA.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509::ca_cert.
	 *  \param c_caCert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_caCert(const PKI_CERT & c_caCert);

	/*! \brief This function returns X509::ca_cert.
	 *  \return The value of X509::ca_cert.
	 */
	const PKI_CERT & get_caCert() const;

	/*! \brief This function returns X509::ca_cert.
	 *  \return The value of X509::ca_cert.
	 */
	PKI_CERT & get_caCert();


	/*! \brief This function sets INTERNAL_CA_KEY::ca_key.
	 *  \param c_caKey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_caKey(const InternalCaKey & c_caKey);

	/*! \brief This function returns INTERNAL_CA_KEY::ca_key.
	 *  \return The value of INTERNAL_CA_KEY::ca_key.
	 */
	const InternalCaKey & get_caKey() const;

	/*! \brief This function returns INTERNAL_CA_KEY::ca_key.
	 *  \return The value of INTERNAL_CA_KEY::ca_key.
	 */
	InternalCaKey & get_caKey();


	/*! \brief This function sets ASN1_ENCRYPT_SIGN::crypted_body.
	 *  \param c_cryptedBody [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_cryptedBody(const Asn1EncryptSign & c_cryptedBody);

	/*! \brief This function returns ASN1_ENCRYPT_SIGN::crypted_body.
	 *  \return The value of ASN1_ENCRYPT_SIGN::crypted_body.
	 */
	const Asn1EncryptSign & get_cryptedBody() const;

	/*! \brief This function returns ASN1_ENCRYPT_SIGN::crypted_body.
	 *  \return The value of ASN1_ENCRYPT_SIGN::crypted_body.
	 */
	Asn1EncryptSign & get_cryptedBody();


	/*! \brief This funciton generates the PEM representation of the local instance.
	 *  \param PemDatas [OUT] the PEM output.
	 *  \return true on success, false on failure.
	 */
	bool to_PEM(mString & PemDatas) const;

	/*! \brief This function loads the local instance from a PEM representation.
	 *  \param PemDatas [IN] the PEM input.
	 *  \return true on success, false on failure.
	 */
	bool from_PEM(const mString & PemDatas);


	/*! \brief This function copies a InternalCa to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const InternalCa & other);
	/*! \brief This member is an empty instance of InternalCa.
	 */
	static InternalCa EmptyInstance;

private:
	PKI_CERT m_caCert;
	InternalCaKey m_caKey;
	Asn1EncryptSign m_cryptedBody;
	void resetAll();
	void freeAll();
protected:
};


/*! \brief This function validates that a cert request respects the constraints
 *  \param reqType [IN] The request type.
 *  \param entityType [IN] The entity type.
 *  \param request [IN] The cert request.
 *  \param minpwdlen [IN] The minimum password length.
 *  \param minkeylen [IN] The minimum key length.
 *  \param confFlags [IN] The configuration flags.
 *  \return true on success, false on failure.
 */
bool ValidateRequestConstraints(int entityType, int reqType, const NewpkiRequestCertBody & request, int minpwdlen, int minkeylen, const ASN1_BIT_STRING * confFlags);

#endif // #ifndef ASN1CERT_H
