/*
 *  Copyright (C) 2003  David A Knight
 *  
 *  based on ephy-main.c,
 *  Copyright (C) 2000-2002 Marco Pesenti Gritti
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "neutrino-shell.h"
#include "neutrino.h"
#include "neutrino-application.h"

#include <libbonoboui.h>
#include <libgnome/gnome-program.h>
#include <libgnomeui/gnome-ui-init.h>

#ifdef HAVE_GNOME_VFS
#include <libgnomevfs/gnome-vfs-init.h>
#endif

#include <glade/glade-init.h>

#define NEUTRINO_FACTORY_OAFIID "OAFIID:Neutrino_Factory"

static gboolean neutrino_main_application_init( void );
static gboolean neutrino_main_start( gpointer data );

GnomeProgram *program;
CORBA_Environment corba_env;

static BonoboObject *application_object;
static gboolean first_instance;

static gboolean quit_option = FALSE; /* --quit */

static struct poptOption popt_options[] = {
	 { NULL, '\0', POPT_ARG_INCLUDE_TABLE, &bonobo_activation_popt_options, 0, NULL, NULL },
	 
	{ "quit", 'q', POPT_ARG_NONE, &quit_option, 0,
	  N_( "Close the Neutrino server down" ),
	  NULL },
	 
	{ NULL, 0, 0, NULL, 0, NULL, NULL }
};

int main( int argc, char *argv[] )
{
	poptContext context;
	GValue context_as_value = { 0 };
	GnomeProgram *program;
	
	/* init i18n */
#ifdef ENABLE_NLS
	bindtextdomain( GETTEXT_PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset( GETTEXT_PACKAGE, "UTF-8" );
	textdomain( GETTEXT_PACKAGE );
#endif

	g_set_application_name( _( "Neutrino" ) );

	program = gnome_program_init( PACKAGE, VERSION,
				LIBGNOMEUI_MODULE, argc, argv,
				GNOME_PARAM_POPT_TABLE, popt_options,
				GNOME_PARAM_HUMAN_READABLE_NAME, _( "Neutrino" ),
				GNOME_PARAM_APP_DATADIR, DATADIR,
				NULL );

	g_object_get_property( G_OBJECT( program ),
				GNOME_PARAM_POPT_CONTEXT,
				g_value_init( &context_as_value, G_TYPE_POINTER ) );

	context = g_value_get_pointer( &context_as_value );

	g_value_unset( &context_as_value );

	gdk_threads_init();

	first_instance = neutrino_main_application_init();

	if( first_instance ) {

#ifdef HAVE_GNOME_VFS
		gnome_vfs_init();
#endif

		glade_gnome_init();

		neutrino_shell_new();

		g_idle_add( (GSourceFunc) neutrino_main_start, NULL );

		bonobo_main();
#ifdef HAVE_GNOME_VFS
		gnome_vfs_shutdown();
#endif
	}
	poptFreeContext( context );
	g_object_unref( program );
	
	return 0;
}

static gboolean neutrino_main_start( gpointer data )
{
	Neutrino_Application server;

	CORBA_exception_init( &corba_env );

	server = bonobo_activation_activate_from_id( "OAFIID:Neutrino",
			0, NULL, &corba_env );

	if( ! server ) {
		GtkWidget *dialog;

		dialog = gtk_message_dialog_new
			( NULL,
			  GTK_DIALOG_MODAL,
			  GTK_MESSAGE_ERROR,
			  GTK_BUTTONS_CLOSE,
			  _( "Neutrino can't be used now. "
			"Running the command \"bonobo-slay\" "
			"from the console may fix the problem. If not, "
			"you can try rebooting the computer or "
			"installing Neutrino again.\n\n"
			"Bonobo couldn't locate the neutrino.server"));
		gtk_dialog_run( GTK_DIALOG( dialog ) );
	} else if( quit_option ) {
		Neutrino_Application_quit( server, &corba_env );
	} else {
		Neutrino_Application_create_window( server, &corba_env );
	}

	if( first_instance ) {
		g_object_unref( G_OBJECT( neutrino_shell ) );
	}

	if( server ) {
		bonobo_object_release_unref( server, &corba_env );
	}

	CORBA_exception_free( &corba_env );

	gdk_notify_startup_complete();

	return FALSE;
}

static gboolean neutrino_main_application_init( void )
{
	CORBA_Object factory;

	factory = bonobo_activation_activate_from_id
		( NEUTRINO_FACTORY_OAFIID,
		  Bonobo_ACTIVATION_FLAG_EXISTING_ONLY,
		  NULL, NULL );

	if( ! factory ) {
		application_object = neutrino_application_new();
		return TRUE;
	} else {
		neutrino_main_start( NULL );
		return FALSE;
	}
}
