/*  Nomad:  nomad-jukebox.c
 *
 *  Copyright (C) 2002 David A Knight <david@ritter.demon.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <stdlib.h>

#include <libgnome/gnome-macros.h>

#include "nomad-jukebox.h"

#include "nomad-jukebox-job.h"

#include "nomad-util.h"

#include "nomadmarshal.h"

struct NomadTrack {
	guint id;
	gchar *artist;
	gchar *title;
	gchar *album;
	gchar *genre;
	gchar *length;
	gchar *year;
	gchar *size;
	gchar *codec;
	gchar *track;
	gchar *name;

	gboolean play_only;

	GMutex *lock;
};

enum {
	TIME_SIGNAL,
	USAGE_SIGNAL,
	SCANNING_TRACKS_SIGNAL,
	SCANNED_TRACKS_SIGNAL,
	SCANNING_PLAYLISTS_SIGNAL,
	PLAYLIST_ADD_SIGNAL,
	SCANNED_PLAYLISTS_SIGNAL,
	TRANSFER_SIGNAL,
	UPLOAD_COMPLETE_SIGNAL,
	DOWNLOAD_COMPLETE_SIGNAL,
	DOWNLOADED_TRACK_SIGNAL,
	PLAYLIST_TRACK_ERASED_SIGNAL,
	PLAYLIST_TRACKS_ERASED_SIGNAL,
	PLAYLIST_REMOVE_SIGNAL,
	PLAYLIST_RENAMED_SIGNAL,
	EAX_SIGNAL,
	PLAY_BEGIN_SIGNAL,
	PLAY_PROGRESS_SIGNAL,
	PLAY_FINISHED_SIGNAL,

	TRACK_ADD_SIGNAL,
	TRACK_REMOVE_SIGNAL,
	TRACK_CHANGED_SIGNAL,

	ERROR_SIGNAL,

	LAST_SIGNAL
};
static guint nomad_jukebox_signals[LAST_SIGNAL] = { 0 };


static void nomad_jukebox_refresh_id( NomadJukebox *jukebox );

static gint nomad_jukebox_xfer_cb( u_int64_t sent, u_int64_t total,
				      const gchar *buf, unsigned int len,
				      void *data );
static gpointer nomad_jukebox_transfer_thread( gpointer data );

static gchar *nomad_jukebox_filename_format( const gchar *dest, 
						const gchar *format,
						NomadTrack *track );

static void nomad_jukebox_class_init( NomadJukeboxClass *klass );
static void nomad_jukebox_instance_init( NomadJukebox *jukebox );
static void nomad_jukebox_finalize( GObject *object );
static void nomad_jukebox_set_prop( GObject *object, guint prop_id, 
				       const GValue *value, GParamSpec *spec );
static void nomad_jukebox_get_prop( GObject *object, guint prop_id, 
				       GValue *value, GParamSpec *spec );



static void nomad_jukebox_get_time_real( NomadJukebox *jukebox );

static void nomad_jukebox_build_tracklist_real( NomadJukebox *jukebox );
static void nomad_jukebox_build_playlist_real( NomadJukebox *jukebox );

static void nomad_jukebox_set_ownerstring_real( NomadJukebox *jukebox, 
						   const gchar *owner );

void nomad_jukebox_getusage_real( NomadJukebox *jukebox );

static void nomad_jukebox_upload_real( NomadJukebox *jukebox, 
					  GList *list );
static void nomad_jukebox_download_real( NomadJukebox *jukebox,
					    const gchar *dest,
					    const gchar *format,
					    GList *list );
static void nomad_jukebox_eax_set( NomadJukebox *jukebox, gint type,
				      gint8 value );

static void nomad_jukebox_delete_tracks_real( NomadJukebox *jukebox, 
						 GList *list );
static void
nomad_jukebox_delete_tracks_from_playlist_real( NomadJukebox *jukebox,
						   GList *list,
						   const gchar *name );

static void nomad_jukebox_set_metadata_real( NomadJukebox *jukebox, 
						NomadTrack *track );

static void nomad_jukebox_create_playlist_real( NomadJukebox *jukebox,
						   const gchar *name );
static void nomad_jukebox_delete_playlist_real( NomadJukebox *jukebox,
                                                   const gchar *name );
static void nomad_jukebox_rename_playlist_real( NomadJukebox *jukebox,
                                                   const gchar *orig,
                                                   const gchar *name );
static void
nomad_jukebox_add_tracks_to_playlist_real( NomadJukebox *jukebox,
                                              const gchar *name,
                                              GList *songidlist );

static void nomad_jukebox_get_eax_real( NomadJukebox *jukebox );

static void nomad_jukebox_play_track_real( NomadJukebox *jukebox, 
					      guint trackid );
static void nomad_jukebox_play_elapsed_real( NomadJukebox *jukebox );
static void nomad_jukebox_play_stop_real( NomadJukebox *jukebox );


static playlist_t *nomad_jukebox_fetch_playlist( NomadJukebox *jukebox,
						    const gchar *name );

static void nomad_jukebox_add_job( NomadJukebox *jukebox,
				      NomadJukeboxJobType type,
				      gpointer data,
				      gpointer data2,
				      gpointer data3,
				      guint priority );

NomadTrack* nomad_track_new( const gchar *artist, const gchar *title,
				   const gchar *album, const gchar *genre,
				   const gchar *length, const gchar *year,
				   const gchar *size, const gchar *codec,
				   const gchar *track, const gchar *name )
{
	NomadTrack *ntrack;

	ntrack = g_new0( NomadTrack, 1 );

	ntrack->lock = g_mutex_new();

	if( artist ) {
		ntrack->artist = g_strdup( artist );
	} else {
		ntrack->artist = g_strdup( "<Unknown>" );
	}

	if( title ) {
		ntrack->title = g_strdup( title );
	} else {
		ntrack->title = g_strdup( "<Unknown>" );
	}
	if( album ) {
		ntrack->album = g_strdup( album );
	} else {
		ntrack->album = g_strdup( "<Unknown>" );
	}
	if( genre ) {
		ntrack->genre = g_strdup( genre );
	} else {
		ntrack->genre = g_strdup( "<Unknown>" );
	}
	if( length ) {
		ntrack->length = g_strdup( length );
	} else {
		ntrack->length = g_strdup( "0:00" );
	}
	if( year ) {
		ntrack->year = g_strdup( year );
	} else {
		ntrack->year = g_strdup( "<Unknown>" );
	}
	if( size ) {
		ntrack->size = g_strdup( size );
	} else {
		ntrack->size = g_strdup( "0" );
	}
	if( codec ) {
		ntrack->codec = g_strdup( codec );
	} else {
		ntrack->codec = g_strdup( "1" );
	}
	if( track ) {
		ntrack->track = g_strdup( track );
	} else {
		ntrack->track = g_strdup( "<Unknown>" );
	}
	if( name ) {
		ntrack->name = g_strdup( name );
	} else {
		ntrack->name = g_strdup( "<Unknown>" );
	}

	return ntrack;
}

NomadTrack* nomad_track_new_with_id( guint id,
					   const gchar *artist,
					   const gchar *title,
					   const gchar *album,
					   const gchar *genre,
					   const gchar *length,
					   const gchar *year,
					   const gchar *size,
					   const gchar *codec,
					   const gchar *track, 
					   const gchar *name )
{
	NomadTrack *ntrack;

	ntrack = nomad_track_new( artist, title, album, genre, length,
				     year, size, codec, track, name );
	ntrack->id = id;

	return ntrack;
}

NomadTrack* nomad_track_copy( const NomadTrack *track )
{
	NomadTrack *ntrack;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ntrack = nomad_track_new_with_id( track->id,
					     track->artist,
					     track->title,
					     track->album,
					     track->genre,
					     track->length,
					     track->year,
					     track->size,
					     track->codec,
					     track->track,
					     track->name );

	g_mutex_unlock( track->lock );

	ntrack->play_only = track->play_only;

	return ntrack;
}

void nomad_track_assign( NomadTrack *track, const NomadTrack *from )
{
	g_mutex_lock( track->lock );
	g_mutex_lock( from->lock );

	if( track->artist ) {
		g_free( track->artist );
		track->artist = g_strdup( from->artist );
	}
	if( track->title ) {
		g_free( track->title );
		track->title = g_strdup( from->title );
	}
	if( track->album ) {
		g_free( track->album );
		track->album = g_strdup( from->album );
	}
	if( track->genre ) {
		g_free( track->genre );
		track->genre = g_strdup( from->genre );
	}
	if( track->length ) {
		g_free( track->length );
		track->length = g_strdup( from->length );
	}
	if( track->year ) {
		g_free( track->year );
		track->year = g_strdup( from->year );
	}
	if( track->size ) {
		g_free( track->size );
		track->size = g_strdup( from->size );
	}
	if( track->codec ) {
		g_free( track->codec );
		track->codec = g_strdup( from->codec );
	}
	if( track->track ) {
		g_free( track->track );
		track->track = g_strdup( from->track );
	}
	if( track->name ) {
		g_free( track->name );
		track->name = g_strdup( from->name );
	}

	g_mutex_unlock( from->lock );
	g_mutex_unlock( track->lock );
}

void nomad_track_free( NomadTrack *track )
{
	g_return_if_fail( track != NULL );

	if( track->artist ) {
		g_free( track->artist );
	} 
	if( track->title ) {
		g_free( track->title );
	}
	if( track->album ) {
		g_free( track->album );
	}
	if( track->genre ) {
		g_free( track->genre );
	} 
	if( track->length ) {
		g_free( track->length );
	}
	if( track->year ) {
		g_free( track->year );
	} 
	if( track->size ) {
		g_free( track->size );
	}
	if( track->codec ) {
		g_free( track->codec );
	} 
	if( track->track ) {
		g_free( track->track );
	} 
	if( track->name ) {
		g_free( track->name );
	}

	g_mutex_free( track->lock );

	g_free( track );
}

guint nomad_track_get_id( const NomadTrack *track )
{
	guint ret;

	g_return_val_if_fail( track != NULL, 0 );

	g_mutex_lock( track->lock );

	ret = track->id;

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_artist( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->artist );

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_title( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->title );

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_album( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->album );

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_genre( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->genre );

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_length( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->length );

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_year( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->year );

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_size( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->size );

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_codec( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->codec );

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_track( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->track );

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_name( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->name );

	g_mutex_unlock( track->lock );

	return ret;
}

gboolean nomad_track_get_play_only( const NomadTrack *track )
{
	g_return_val_if_fail( track != NULL, FALSE );

	return track->play_only;
}

void nomad_track_get( const NomadTrack *track,
			 gchar **artist, gchar **title,
			 gchar **album, gchar **genre, gchar **length,
			 gchar **year, gchar **size, gchar **codec,
			 gchar **ttrack, gchar **name, guint *id )
{
	g_return_if_fail( track != NULL );

	g_mutex_lock( track->lock );

	if( id ) {
		*id = track->id;
	}

	if( artist ) {
		*artist = g_strdup( track->artist );
	}
	if( title ) {
		*title = g_strdup( track->title );
	}
	if( album ) {
		*album = g_strdup( track->album );
	}
	if( genre ) {
		*genre = g_strdup( track->genre );
	}
	if( length ) {
		*length = g_strdup( track->length );
	}
	if( year ) {
		*year = g_strdup( track->year );
	}
	if( size ) {
		*size = g_strdup( track->size );
	}
	if( codec ) {
		*codec = g_strdup( track->codec );
	}
	if( ttrack ) {
		*ttrack = g_strdup( track->track );
	}
	if( name ) {
		*name = g_strdup( track->name );
	}

	g_mutex_unlock( track->lock );
}

NomadJukebox *nomad_jukebox_new( njb_t *njb )
{
	NomadJukebox *jukebox;

	jukebox = NOMAD_JUKEBOX( g_object_new( nomad_jukebox_get_type(),
						  NULL ) );

	jukebox->njb = njb;

	return jukebox;
}

void nomad_jukebox_lock( NomadJukebox *jukebox )
{
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_mutex_lock( jukebox->lock );
}

void nomad_jukebox_unlock( NomadJukebox *jukebox )
{
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_mutex_unlock( jukebox->lock );
}

gboolean nomad_jukebox_aquire( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), FALSE );
	g_return_val_if_fail( jukebox->njb != NULL, FALSE );

	jukebox->connected = ( NJB_Open( jukebox->njb ) != -1 && 
				NJB_Capture( jukebox->njb ) != -1 );

	if( jukebox->owner ) {
		g_free( jukebox->owner );
		jukebox->owner = NULL;
	}
	if( jukebox->firmware ) {
		g_free( jukebox->firmware );
		jukebox->firmware = NULL;
	}

	if( jukebox->connected ) {
		gchar *owner;

		owner = NJB_Get_Owner_String( jukebox->njb );
		if( ! owner ) {
			owner = g_strdup( "" );
		}
		jukebox->owner = owner;
		nomad_jukebox_refresh_id( jukebox );
		jukebox->firmware = g_strdup_printf( "%u.%u",
						     jukebox->njbid->fwMajor,
						     jukebox->njbid->fwMinor
						     );
	}


	return jukebox->connected;
}

void nomad_jukebox_release( NomadJukebox *jukebox )
{
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->njb != NULL );
	g_return_if_fail( jukebox->connected );
	
	if( NJB_Release( jukebox->njb ) != -1 ) {
		NJB_Close( jukebox->njb );
	} else {
		njb_error_dump( stdout );
	}

	if( jukebox->owner ) {
		g_free( jukebox->owner );
		jukebox->owner = NULL;
	}
	if( jukebox->firmware ) {
		g_free( jukebox->firmware );
		jukebox->firmware = NULL;
	}
}

const gchar *nomad_jukebox_get_idstring( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), NULL );
	g_return_val_if_fail( jukebox->njb != NULL, NULL );
	g_return_val_if_fail( jukebox->connected, NULL );

	return (const gchar*)jukebox->njb->idstring;
}

const gchar *nomad_jukebox_get_firmware( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), NULL );
	g_return_val_if_fail( jukebox->njb != NULL, NULL );
	g_return_val_if_fail( jukebox->connected, NULL );

	return jukebox->firmware;
}

const gchar *nomad_jukebox_get_prodname( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), NULL );
	g_return_val_if_fail( jukebox->njb != NULL, NULL );
	g_return_val_if_fail( jukebox->connected, NULL );


	return jukebox->njbid->productName;
}

guint nomad_jukebox_get_num_tracks( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return g_hash_table_size( jukebox->tracks );
}

guint nomad_jukebox_get_num_playlists( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return g_hash_table_size( jukebox->playlists );
}

guint nomad_jukebox_get_num_data_files( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return 0;
}



gboolean nomad_jukebox_get_power( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), FALSE );
	g_return_val_if_fail( jukebox->njb != NULL, FALSE );
	g_return_val_if_fail( jukebox->connected, FALSE );

	return ( jukebox->njbid->power > 0 );
}

void nomad_jukebox_get_time( NomadJukebox *jukebox )
{
	nomad_jukebox_add_job( jukebox, GET_TIME,
				  NULL, NULL, NULL, 0 );
}

static void nomad_jukebox_get_time_real( NomadJukebox *jukebox )
{
	njb_time_t *time;

	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->njb != NULL );
	g_return_if_fail( jukebox->connected );
	
	time = NJB_Get_Time( jukebox->njb );
	if( time ) {
		gchar *tstring;
		tstring = g_strdup_printf("%u-%.2u-%.2u %.2u:%.2u:%.2u",
					  time->year, time->month, time->day,
					  time->hours, time->minutes,
					  time->seconds);
		
		g_mutex_lock( jukebox->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ TIME_SIGNAL ], 0,
			       tstring );
		g_mutex_unlock( jukebox->job_queue_mutex );
		
		g_free( tstring );

	}
}

void nomad_jukebox_build_tracklist( NomadJukebox *jukebox )
{
	nomad_jukebox_add_job( jukebox, BUILD_TRACKLIST,
				  NULL, NULL, NULL, 0 );
}

static void nomad_jukebox_build_tracklist_real( NomadJukebox *jukebox )
{
	songid_t *songtag;
	gint total_tracks;

	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->njb != NULL );
	g_return_if_fail( jukebox->connected );

	/* emit signal to signify that tracks are about to be scanned */
	g_mutex_lock( jukebox->job_queue_mutex );
	g_signal_emit( G_OBJECT( jukebox ),
		       nomad_jukebox_signals[ SCANNING_TRACKS_SIGNAL ], 0 );
	g_mutex_unlock( jukebox->job_queue_mutex );

	NJB_Reset_Get_Track_Tag( jukebox->njb );
	total_tracks = 0;
	while( ( songtag = NJB_Get_Track_Tag( jukebox->njb ) ) ) {
		songid_frame_t *frame;
		NomadTrack *track;

		songid_reset_getframe( songtag );

		track = g_new0( NomadTrack, 1 );
		track->lock = g_mutex_new();

		while( ( frame = songid_getframe( songtag ) ) ) {
			gchar *label;
			gchar *data;

			label = g_strndup( frame->label, frame->labelsz );

			switch( frame->type ) {
			case ID_DATA_BIN:
				if( ! strcmp( label, FR_LENGTH ) ) {
					data = seconds_to_mmss( songid_frame_data32( frame ) );
				} else if( frame->datasz > 4 ) {
					/* 64 bit data, not handled yet by
					   libnjb */
					g_warning( "Found 64 bit data in songid tag frame.  Not Supported\n" );
					data = g_strdup( "" );
				} else {
					data = g_strdup_printf( "%lu",
								(unsigned long)songid_frame_data32( frame ) );
				}
				break;
			case ID_DATA_ASCII:
				if( frame->datasz ) {
					data = g_strdup( frame->data );
					if( ! data ) {
						data = g_strdup( "<Unknown Charset>" );
					}
				} else {
					data = g_strdup( "<Unknown>" );
				}
				break;
			default:
				g_warning( "Unknown frame type: %i\n",
					   frame->type );
				data = NULL;
				break;
			}

			if( ! strcmp( label, FR_ARTIST ) ) {
				track->artist = g_strstrip( data );
			} else if( ! strcmp( label, FR_TITLE ) ) {
				track->title = g_strstrip( data );
			} else if( ! strcmp( label, FR_ALBUM ) ) {
				track->album = g_strstrip( data );
			} else if( ! strcmp( label, FR_GENRE ) ) {
				track->genre = g_strstrip( data );
			} else if( ! strcmp( label, FR_LENGTH ) ) {
				track->length = g_strstrip( data );
			} else if( ! strcmp( label, FR_YEAR ) ) {
				track->year = g_strstrip( data );
			} else if( ! strcmp( label, FR_SIZE ) ) {
				track->size = g_strstrip( data );
			} else if( ! strcmp( label, FR_CODEC ) ) {
				track->codec = g_strstrip( data );
			} else if( ! strcmp( label, FR_TRACK ) ) {
				track->track = g_strstrip( data );
			} else if( ! strcmp( label, FR_FNAME ) ) {
				track->name = g_strstrip( data );
			} else if( ! strcmp( label, FR_PROTECTED ) ) {
				/* ignore this we don't care about it,
				   we could possibly set a column in the model
				   to indicate if a track is play only, and
				   then use it to give some indication in the
				   view */
				if( *data == '1' ) {
					track->play_only = TRUE;
				}
				g_free( data );
			} else if( data ) {
				/* unsupported frame tag */
				g_warning( "Unsupported label: %s -> %s\n",
					   label, data );
				g_free( data );
			}

			g_free( label );
		}
		if( ! track->artist ) {
			track->artist = g_strdup( "<Unknown>" );
		}
		if( ! track->title ) {
			track->title = g_strdup( "<Unknown>" );
		}
		if( ! track->album ) {
			track->album = g_strdup( "<Unknown>" );
		}
		if( ! track->genre ) {
			track->genre = g_strdup( "<Unknown>" );
		}
		if( ! track->length ) {
			track->length = g_strdup( "0:00" );
		}
		if( ! track->year ) {
			track->year = g_strdup( "<Unknown>" );
		}
		if( ! track->size ) {
			track->size = g_strdup( "0" );
		}
		if( ! track->codec ) {
			track->codec = g_strdup( "1" );
		}
		if( ! track->track ) {
			/* GNOMAD sets this to songtag->trid,
			   this isn't right IMO as it is just an id number
			   and not suitable for a track, it can give values
			   of hundreds of thousands, and there are no CDs with
			   that many tracks on */
			track->track = g_strdup( "Unknown" );
		}
		track->id = songtag->trid;

		/* insert NomadTrack into tracks hashtable */
		g_hash_table_insert( jukebox->tracks,
				     GUINT_TO_POINTER( songtag->trid ),
				     track );

		/* emit signal to signify that a track has been scanned */
		total_tracks ++;

		g_mutex_lock( jukebox->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ TRACK_ADD_SIGNAL ],
			       0, total_tracks,
			       track );
		g_mutex_unlock( jukebox->job_queue_mutex );

		songid_destroy( songtag );
	}
	/* emit signal to signify that tracks have been scanned */
	g_mutex_lock( jukebox->job_queue_mutex );
	g_signal_emit( G_OBJECT( jukebox ),
		       nomad_jukebox_signals[ SCANNED_TRACKS_SIGNAL ], 0,
		       total_tracks );
	g_mutex_unlock( jukebox->job_queue_mutex );
}

void nomad_jukebox_build_playlist( NomadJukebox *jukebox )
{
	nomad_jukebox_add_job( jukebox, BUILD_PLAYLIST,
				  NULL, NULL, NULL, 0 );
}

static void nomad_jukebox_build_playlist_real( NomadJukebox *jukebox )
{
	playlist_t *playlist;
        gint total_lists;
	
        g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
        g_return_if_fail( jukebox->njb != NULL );
        g_return_if_fail( jukebox->connected );
        g_return_if_fail( jukebox->tracks != NULL );
	
	g_mutex_lock( jukebox->job_queue_mutex );
        g_signal_emit( G_OBJECT( jukebox ),
                       nomad_jukebox_signals[ SCANNING_PLAYLISTS_SIGNAL ],
                       0 );
	g_mutex_unlock( jukebox->job_queue_mutex );

	NJB_Reset_Get_Playlist( jukebox->njb );
        total_lists = 0;
        while( ( playlist = NJB_Get_Playlist( jukebox->njb ) ) ) {
		GSList *tracks;
		gchar *name;
                playlist_track_t *track;  
                guint total_tracks;
	                                    
                playlist_reset_gettrack( playlist );
		for( tracks = NULL, total_tracks = 0;
		     ( track = playlist_gettrack( playlist ) ); ) {
                        NomadTrack *gdap_track;
			
                        gdap_track = g_hash_table_lookup( jukebox->tracks,
                                                          GUINT_TO_POINTER( track->trackid ) );
			if( gdap_track ) {
				tracks = g_slist_prepend( tracks, gdap_track );
				total_tracks ++;
                        } else {
                                g_warning( "Unknown track in playlist: %lu\n",
                                           (unsigned long)track->trackid );
                        }
                }
		if( tracks ) {
			tracks = g_slist_reverse( tracks );
		}
		/* we add the playlist id to the front */
		tracks = g_slist_prepend( tracks, 
					  GUINT_TO_POINTER( playlist->plid ) );

		/* add to playlists - key by name, data is a list,
		   first item in the list is the playlist id,
		   subsequent items are NomadTrack pointers */
		name = g_strdup( playlist->name );
		g_hash_table_insert( jukebox->playlists,
				     name,
				     tracks );
		g_hash_table_insert( jukebox->playlistids,
				     GUINT_TO_POINTER( playlist->plid ), 
				     name );
                                        
                /* emit signal to signify that a playlist has
                   been scanned */
		g_mutex_lock( jukebox->job_queue_mutex );
                g_signal_emit( G_OBJECT( jukebox ),
                               nomad_jukebox_signals[ PLAYLIST_ADD_SIGNAL],
                               0, total_tracks, playlist->name );
		g_mutex_unlock( jukebox->job_queue_mutex );

		total_lists ++;

		playlist_destroy( playlist );
        }
        /* emit signal to signify that tracks have been scanned */
	g_mutex_lock( jukebox->job_queue_mutex );
        g_signal_emit( G_OBJECT( jukebox ),
                       nomad_jukebox_signals[ SCANNED_PLAYLISTS_SIGNAL ], 0,
                       total_lists );
	g_mutex_unlock( jukebox->job_queue_mutex );

}

void nomad_jukebox_delete_tracks( NomadJukebox *jukebox, GList *list )
{
	/* erase tracks from playlists, we do this first,
	   as if something goes wrong it means the playlists
	   will be consistant, I.E. they won't contain non existant
	   tracks */
	nomad_jukebox_delete_tracks_from_playlist( jukebox, list, NULL );

	nomad_jukebox_add_job( jukebox, DELETE_TRACKS,
				  g_list_copy( list ), NULL, NULL, 0 );
}

void nomad_jukebox_delete_tracks_from_playlist( NomadJukebox *jukebox,
						   GList *list,
						   const gchar *name )
{
	gchar *nm;

	if( name ) {
		nm = g_strdup( name );
	} else {
		nm = NULL;
	}

	nomad_jukebox_add_job( jukebox, DELETE_TRACKS_FROM_PLAYLIST,
				  g_list_copy( list ), nm, NULL, 0 );
}

static void
nomad_jukebox_delete_tracks_from_playlist_real( NomadJukebox *jukebox,
						   GList *list,
						   const gchar *name )
{
        GList *tmplist;
        playlist_t *playlist;


        g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
        g_return_if_fail( jukebox->njb != NULL );
        g_return_if_fail( jukebox->connected );
        g_return_if_fail( list != NULL );
	
	NJB_Reset_Get_Playlist( jukebox->njb );
        while( ( playlist = NJB_Get_Playlist( jukebox->njb ) ) ) {
                playlist_track_t *track;
                gint pos = NJB_PL_START;
                gboolean changed = FALSE;
		GSList *tracks;
		GSList *copy;
		gchar *current;
		const gchar *pname;

                if( name && strcmp( name, playlist->name ) ) {
                        continue;
                }

		if( ! name ) {
			current = g_strdup( playlist->name );
		} else {
			current = g_strdup( name );
		}

		tracks = g_hash_table_lookup( jukebox->playlists, current );
		g_assert( tracks );
		copy = g_slist_copy( tracks->next );
		
		while( ( track = playlist_gettrack( playlist ) ) ) {
                        for( tmplist = list; tmplist;
                             tmplist = tmplist->next ) {
                                if( GPOINTER_TO_UINT(tmplist->data) ==
                                    track->trackid ) {
                                        NomadTrack *dtrack;
					
                                        dtrack = (NomadTrack*)g_hash_table_lookup( jukebox->tracks, tmplist->data );
					
                                        g_assert( dtrack );
	
					g_mutex_lock( jukebox->job_queue_mutex );
                                        g_signal_emit( G_OBJECT( jukebox ),
                                                       nomad_jukebox_signals[ PLAYLIST_TRACK_ERASED_SIGNAL ],
                                                       0,
                                                       current,
                                                       dtrack->artist,
                                                       dtrack->album,
                                                       dtrack->title );
					g_mutex_unlock( jukebox->job_queue_mutex );
                                        playlist_deltrack( playlist,
                                                           pos );
                                        changed = TRUE;

					copy = g_slist_remove( copy,
							       dtrack );
                                }
                        }
                        pos ++;
                }
		
		g_free( current );
		
		if( changed ) {
                        if( NJB_Update_Playlist( jukebox->njb, 
						 playlist ) != -1 ) {
				copy = g_slist_prepend( copy, GUINT_TO_POINTER( playlist->plid ) );
				pname = g_hash_table_lookup( jukebox->playlistids,
							     tracks->data );
				g_assert( pname );
				g_hash_table_remove( jukebox->playlistids,
						     tracks->data );
				g_hash_table_insert( jukebox->playlistids,
						     copy->data,
						     (gpointer)pname );
				g_hash_table_replace( jukebox->playlists,
						      (gpointer)pname,
						      copy );
				g_slist_free( tracks );

				g_mutex_lock( jukebox->job_queue_mutex );
				g_signal_emit( G_OBJECT( jukebox ),
					       nomad_jukebox_signals[ PLAYLIST_TRACKS_ERASED_SIGNAL ],
					       0,
					       pname );
				g_mutex_unlock( jukebox->job_queue_mutex );
			} else {
				gchar *errmsg;

				errmsg = g_strdup_printf( "Failed to delete tracks from %s", pname );

				g_mutex_lock( jukebox->job_queue_mutex );
				g_signal_emit( G_OBJECT( jukebox ),
					       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
					       errmsg );
				g_mutex_unlock( jukebox->job_queue_mutex );

				g_free( errmsg );
				g_slist_free( copy );
                        }
                } else {
			g_slist_free( copy );
		}
                playlist_destroy( playlist );
		
                if( changed && name ) {
                        break;
                }
        }
}

static void nomad_jukebox_delete_tracks_real( NomadJukebox *jukebox, 
						 GList *list )
{
	GList *tmplist;

	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->njb != NULL );
	g_return_if_fail( jukebox->connected );
	g_return_if_fail( list != NULL );

	/* erase the tracks */
	for( tmplist = list; tmplist; tmplist = tmplist->next ) {
		u_int32_t id;
		NomadTrack *track;

		id = GPOINTER_TO_UINT( tmplist->data );

		track = nomad_jukebox_request_track( jukebox, id );
		g_assert( track );

		if( NJB_Delete_Track( jukebox->njb, id ) != -1 ) {
			g_mutex_lock( jukebox->job_queue_mutex );
			g_signal_emit( G_OBJECT( jukebox ),
				       nomad_jukebox_signals[ TRACK_REMOVE_SIGNAL ],
				       0,
				       track );
			g_mutex_unlock( jukebox->job_queue_mutex );
			g_hash_table_remove( jukebox->tracks,
					     tmplist->data );
		} else {
			gchar *errmsg;
			gchar *trackname;

			trackname = nomad_track_get_title( track );
			errmsg = g_strdup_printf( "Failed to delete track %s",
						  trackname );
			g_mutex_lock( jukebox->job_queue_mutex );
			g_signal_emit( G_OBJECT( jukebox ),
				       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
				       errmsg );
			g_mutex_unlock( jukebox->job_queue_mutex );
			
			g_free( errmsg );
			g_free( trackname );
		}
		nomad_track_free( track );
	}
}

void nomad_jukebox_delete_files( NomadJukebox *jukebox, GList *list )
{
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->njb != NULL );
	g_return_if_fail( jukebox->connected );
	g_return_if_fail( list != NULL );

	g_warning( "Deleting files not yet supported\n" );
}

void nomad_jukebox_set_metadata( NomadJukebox *jukebox,
				    const NomadTrack *track )
{
	g_return_if_fail( track != NULL );

	nomad_jukebox_add_job( jukebox, SET_METADATA,
				  nomad_track_copy( track ),
				  NULL, NULL, 0 );
}

static void nomad_jukebox_set_metadata_real( NomadJukebox *jukebox, 
						NomadTrack *track )
{
	NomadTrack *old;
	u_int32_t tracknum;
	u_int32_t length;
	u_int32_t size;
	gchar *dummy;

	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->njb != NULL );
	g_return_if_fail( jukebox->connected );
	g_return_if_fail( track != NULL );

	old = g_hash_table_lookup( jukebox->tracks,
				   GUINT_TO_POINTER( track->id ) );
	g_assert( old );

	length = mmss_to_seconds( track->length );
	tracknum = strtoul( track->track, &dummy, 10 );
	size = strtoul( track->size, &dummy, 10 );
	
	if( NJB_Replace_Track_Tag( jukebox->njb, track->id, 
				   track->codec,
				   track->title,
				   track->album,
				   track->genre,
				   track->artist,
				   length, tracknum, size,
				   track->name,
				   track->year,
				   track->play_only ) != -1 ) {
		/* set new track data and emit signal */
		g_mutex_lock( jukebox->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ TRACK_CHANGED_SIGNAL ],
			       0,
			       old,
			       track );
		g_mutex_unlock( jukebox->job_queue_mutex );
		nomad_track_assign( old, track );
	} else {
		gchar *errmsg;
		gchar *trackname;

		trackname = nomad_track_get_title( track );
		errmsg = g_strdup_printf( "Failed to set track information for %s",
					  track->title );
		
		g_mutex_lock( jukebox->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
			       errmsg );
		g_mutex_unlock( jukebox->job_queue_mutex );
		
		g_free( errmsg );
		g_free( trackname );
	}
	nomad_track_free( track );
}

void nomad_jukebox_create_playlist( NomadJukebox *jukebox,
                                       const gchar *name )
{
	g_return_if_fail( name != NULL );

	nomad_jukebox_add_job( jukebox, CREATE_PLAYLIST,
				  g_strdup( name ), NULL, NULL, 0 );
}

static void nomad_jukebox_create_playlist_real( NomadJukebox *jukebox,
						   const gchar *name )
{
        playlist_t *playlist;
        guint plid;
	gboolean failed;

        g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
        g_return_if_fail( jukebox->njb != NULL );
        g_return_if_fail( jukebox->connected );
        g_return_if_fail( name != NULL );
	
        plid = 0;
        playlist = playlist_new();
	failed = FALSE;
	
        if( playlist ) {
		GSList *list;

		list = NULL;

                if( playlist_set_name( playlist, name ) != -1 ) {
                        if(NJB_Update_Playlist(jukebox->njb, playlist) != -1) {
				gchar *pname;
				
				list = g_slist_prepend( NULL, 
							GUINT_TO_POINTER( playlist->plid ) );
				pname = g_strdup( playlist->name );
				g_hash_table_insert( jukebox->playlists,
						     pname, list );
				g_hash_table_insert( jukebox->playlistids,
						     GUINT_TO_POINTER( playlist->plid ),
						     pname );
				g_mutex_lock( jukebox->job_queue_mutex );
				g_signal_emit( G_OBJECT( jukebox ),
                                               nomad_jukebox_signals[PLAYLIST_ADD_SIGNAL ], 0,
                                               0, playlist->name );
				g_mutex_unlock( jukebox->job_queue_mutex );
			} else {
                                /* failed to update playlist */
				failed = TRUE;
                        }
                } else {
                        /* failed to set the playlist name! */
			failed = TRUE;
                }
		if( failed ) {
			gchar *errmsg;
		
			errmsg = g_strdup_printf( "Failed to create playlist %s",
						  name );
		
			g_mutex_lock( jukebox->job_queue_mutex );
			g_signal_emit( G_OBJECT( jukebox ),
				       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
				       errmsg );
			g_mutex_unlock( jukebox->job_queue_mutex );
			
			g_free( errmsg );
		}
		playlist_destroy( playlist );
	}
}


void nomad_jukebox_delete_playlist( NomadJukebox *jukebox, 
				       const gchar *name )
{
	g_return_if_fail( name != NULL );

	nomad_jukebox_add_job( jukebox, DELETE_PLAYLIST,
				  g_strdup( name ), NULL, NULL, 0 );
}

static void nomad_jukebox_delete_playlist_real( NomadJukebox *jukebox,
                                                   const gchar *name )
{
	GSList *tracks;
	guint plid;

        g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
        g_return_if_fail( jukebox->njb != NULL );
        g_return_if_fail( jukebox->connected );
	g_return_if_fail( name != NULL );
	
	tracks = g_hash_table_lookup( jukebox->playlists, name );
	g_assert( tracks );
	plid = GPOINTER_TO_UINT( tracks->data );

        if( NJB_Delete_Playlist( jukebox->njb, plid ) != -1 ) {
		gchar *pname;
		
		pname = g_hash_table_lookup( jukebox->playlistids,
					     tracks->data );
		g_assert( pname );

		g_mutex_lock( jukebox->job_queue_mutex );
		g_signal_emit(G_OBJECT( jukebox ),
                              nomad_jukebox_signals[PLAYLIST_REMOVE_SIGNAL],
                              0, pname );
		g_mutex_unlock( jukebox->job_queue_mutex );
	
		g_hash_table_remove( jukebox->playlistids,
				     tracks->data );
		g_hash_table_remove( jukebox->playlists, pname );
		g_slist_free( tracks );

		g_free( pname );
        } else {
		gchar *errmsg;
		
		errmsg = g_strdup_printf( "Failed to delete playlist %s",
					  name );
		g_mutex_lock( jukebox->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
			       errmsg );
		g_mutex_unlock( jukebox->job_queue_mutex );
			
		g_free( errmsg );
        }
}
                  
void nomad_jukebox_rename_playlist( NomadJukebox *jukebox,
				       const gchar *orig,
                                       const gchar *name )
{
	g_return_if_fail( orig != NULL );
        g_return_if_fail( name != NULL );
	
	nomad_jukebox_add_job( jukebox, RENAME_PLAYLIST,
				  g_strdup( orig ), 
				  g_strdup( name ), NULL, 0 );
}

static void nomad_jukebox_rename_playlist_real( NomadJukebox *jukebox,
						   const gchar *orig,
                                                   const gchar *name )
{
        playlist_t *playlist;
	GSList *list;	

        g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
        g_return_if_fail( jukebox->njb != NULL );
        g_return_if_fail( jukebox->connected );
        g_return_if_fail( name != NULL );
	
	list = g_hash_table_lookup( jukebox->playlists, (gpointer)orig );
	g_assert( list );

	playlist = nomad_jukebox_fetch_playlist( jukebox, orig );
	g_assert( playlist );

	if( playlist_set_name( playlist, name ) != -1 ) {
		if( NJB_Update_Playlist( jukebox->njb,
					 playlist ) == -1 ) {
			gchar *errmsg;
		
			errmsg = g_strdup_printf( "Failed to rename playlist %s to %s",
						  orig, name );
		
			g_mutex_lock( jukebox->job_queue_mutex );
			g_signal_emit( G_OBJECT( jukebox ),
				       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
				       errmsg );
			g_mutex_unlock( jukebox->job_queue_mutex );
			
			g_free( errmsg );
		} else {
			gchar *pname;
			gchar *newname;

			pname = g_hash_table_lookup( jukebox->playlistids,
						     list->data );
			g_assert( pname );

			g_hash_table_remove( jukebox->playlistids,
					     list->data );

			list->data = GUINT_TO_POINTER( playlist->plid );

			newname = g_strdup( name );
			g_hash_table_insert( jukebox->playlistids,
					     list->data,
					     newname );

			g_hash_table_remove( jukebox->playlists,
					     (gpointer)orig );
			
			g_hash_table_insert( jukebox->playlists,
					     newname,
					     list );
			
			g_mutex_lock( jukebox->job_queue_mutex );
			g_signal_emit( G_OBJECT( jukebox ),
				       nomad_jukebox_signals[PLAYLIST_RENAMED_SIGNAL ],
				       0,
				       orig,
				       newname );
			g_mutex_unlock( jukebox->job_queue_mutex );

			g_free( pname );
		}
	}
}

void nomad_jukebox_add_tracks_to_playlist( NomadJukebox *jukebox,
                                              const gchar *name,
                                              GList *songidlist )
{
	g_return_if_fail( name != NULL );
	g_return_if_fail( songidlist != NULL );
	
	nomad_jukebox_add_job( jukebox, ADD_TO_PLAYLIST,
				  g_strdup( name ),
				  g_list_copy( songidlist ), NULL, 0 );
}

static void
nomad_jukebox_add_tracks_to_playlist_real( NomadJukebox *jukebox,
                                              const gchar *name,
                                              GList *songidlist )
{
        playlist_track_t *pl_track;
        playlist_t *playlist;
	guint playlistid;

	GList *list;
	GSList *tracks;

        g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
        g_return_if_fail( jukebox->njb != NULL );
        g_return_if_fail( jukebox->connected );
        g_return_if_fail( name != NULL );
        g_return_if_fail( songidlist != NULL );
	
	tracks = g_hash_table_lookup( jukebox->playlists, name );
	g_assert( tracks );
	playlistid = GPOINTER_TO_UINT( tracks->data );

	playlist = nomad_jukebox_fetch_playlist( jukebox, name );

	tracks = g_slist_reverse( tracks );


	for( list = songidlist; list; list = list->next ) {
		pl_track = playlist_track_new( GPOINTER_TO_UINT(list->data) );
		playlist_addtrack( playlist, pl_track, NJB_PL_END );
	}
	if( NJB_Update_Playlist( jukebox->njb, playlist ) != -1 ) {
		NomadTrack *track;
		
		for( list = songidlist; list; list = list->next ) {
			track = (NomadTrack*)g_hash_table_lookup( jukebox->tracks, list->data );
			g_assert( track );
			tracks = g_slist_prepend( tracks, track );
		}

		tracks = g_slist_reverse( tracks );
		tracks->data = GUINT_TO_POINTER( playlist->plid );

		name = g_hash_table_lookup( jukebox->playlistids, 
					    GUINT_TO_POINTER( playlistid ) );
		g_assert( name );
		g_hash_table_remove( jukebox->playlistids,
				     GUINT_TO_POINTER( playlistid ) );
		g_hash_table_insert( jukebox->playlistids, 
				     GUINT_TO_POINTER( playlist->plid ),
				     (gpointer)name );
		g_hash_table_replace( jukebox->playlists, (gpointer)name,
				      tracks );
	} else {
		gchar *errmsg;
		
		errmsg = g_strdup_printf( "Failed to add tracks to playlist %s",
					  name );
		
		g_mutex_lock( jukebox->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
			       errmsg );
		g_mutex_unlock( jukebox->job_queue_mutex );
		
		g_free( errmsg );
	}

	playlist_destroy( playlist );
}

const gchar *nomad_jukebox_get_ownerstring( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), NULL );
	g_return_val_if_fail( jukebox->njb != NULL, NULL );
	g_return_val_if_fail( jukebox->connected, NULL );

	return jukebox->owner;
}

void nomad_jukebox_set_ownerstring( NomadJukebox *jukebox,
				       const gchar *owner )
{
	g_return_if_fail( owner != NULL );

	nomad_jukebox_add_job( jukebox, SET_OWNER_STRING,
				  g_strdup( owner ), NULL, NULL, 0 );
}

static void nomad_jukebox_set_ownerstring_real( NomadJukebox *jukebox, 
						   const gchar *owner )
{
	gint len;

	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->njb != NULL );
	g_return_if_fail( jukebox->connected );
	g_return_if_fail( owner != NULL );

	if( jukebox->owner ) {
		g_free( jukebox->owner );
	}

	jukebox->owner = g_strdup( owner );

	len = strlen( jukebox->owner );
	if( len >= OWNER_STRING_LENGTH ) {
		gchar *temp;

		temp = g_strndup( jukebox->owner,
				  OWNER_STRING_LENGTH - 1 );
		g_free( jukebox->owner );
		jukebox->owner = temp;
	}

	if( NJB_Set_Owner_String( jukebox->njb, jukebox->owner ) == -1 ) {
		gchar *errmsg;
		
		errmsg = g_strdup_printf( "Failed to set owner string" );
		
		g_mutex_lock( jukebox->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
			       errmsg );
		g_mutex_unlock( jukebox->job_queue_mutex );
		
		g_free( errmsg );
	}
}

void nomad_jukebox_getusage( NomadJukebox *jukebox )
{
	nomad_jukebox_add_job( jukebox, GET_USAGE,
				  NULL, NULL, NULL, 0 );
}

void nomad_jukebox_getusage_real( NomadJukebox *jukebox )
{
	guint64 total;
	guint64 free;
	guint64 used;
	guint songs;
	guint playlists;
	guint datafiles;

	u_int64_t totalbytes;
	u_int64_t freebytes;
	
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->njb != NULL );
	g_return_if_fail( jukebox->connected );
	
	if(NJB_Get_Disk_Usage( jukebox->njb, &totalbytes, &freebytes ) != -1) {
		
		total = totalbytes;
		free = freebytes;
		used = totalbytes - freebytes;

		songs = -1;
		playlists = -1;
		datafiles = -1;
		
		g_mutex_lock( jukebox->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ USAGE_SIGNAL ], 0,
			       total, free, used );
		g_mutex_unlock( jukebox->job_queue_mutex );
	} else {
		gchar *errmsg;
		
		errmsg = g_strdup_printf( "Failed to get usage information" );
		g_mutex_lock( jukebox->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
			       errmsg );
		g_mutex_unlock( jukebox->job_queue_mutex );
		
		g_free( errmsg );
	}
}

void nomad_jukebox_get_eax( NomadJukebox *jukebox )
{
	nomad_jukebox_add_job( jukebox, GET_EAX,
				  NULL, NULL, NULL, 0 );
}

static void nomad_jukebox_get_eax_real( NomadJukebox *jukebox )
{
	eax_t *eax;

	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->njb != NULL );
	g_return_if_fail( jukebox->connected );

	eax = NJB_Get_EAX( jukebox->njb );

	g_mutex_lock( jukebox->job_queue_mutex );
	g_signal_emit( G_OBJECT( jukebox ),
		       nomad_jukebox_signals[ EAX_SIGNAL ], 0,
		       eax );
	g_mutex_unlock( jukebox->job_queue_mutex );

	/* we can destroy the settings now */
	eax_destroy( eax );
}

void nomad_jukebox_set_volume( NomadJukebox *jukebox, guint8 volume )
{
	nomad_jukebox_add_job( jukebox, SET_EAX,
				  GINT_TO_POINTER( NJB_SOUND_SET_VOLUME ),
				  GUINT_TO_POINTER( (guint)volume  ),
				  NULL, 0 );
}

void nomad_jukebox_set_muting( NomadJukebox *jukebox, 
				  gboolean mutingstatus )
{
	nomad_jukebox_add_job( jukebox, SET_EAX,
				  GINT_TO_POINTER( NJB_SOUND_SET_MUTING ),
				  GINT_TO_POINTER( (guint)mutingstatus  ),
				  NULL, 0 );
}

void nomad_jukebox_set_eq_active( NomadJukebox *jukebox, 
				     gboolean eqactive )
{
	nomad_jukebox_add_job( jukebox, SET_EAX,
				  GINT_TO_POINTER( NJB_SOUND_SET_EQSTATUS ),
				  GINT_TO_POINTER( (gint)eqactive  ),
				  NULL, 0 );
}

void nomad_jukebox_set_bass( NomadJukebox *jukebox, gint8 bass )
{
	nomad_jukebox_add_job( jukebox, SET_EAX,
				  GINT_TO_POINTER( NJB_SOUND_SET_BASS ),
				  GINT_TO_POINTER( (gint)bass ),
				  NULL, 0 );
}

void nomad_jukebox_set_midrange( NomadJukebox *jukebox, gint8 midrange )
{
	nomad_jukebox_add_job( jukebox, SET_EAX,
				  GINT_TO_POINTER( NJB_SOUND_SET_MIDRANGE ),
				  GINT_TO_POINTER( (gint)midrange ),
				  NULL, 0 );
}

void nomad_jukebox_set_treble( NomadJukebox *jukebox, gint8 treble )
{
	nomad_jukebox_add_job( jukebox, SET_EAX,
				  GINT_TO_POINTER( NJB_SOUND_SET_TREBLE ),
				  GINT_TO_POINTER( (gint)treble ),
				  NULL, 0 );
}

void nomad_jukebox_set_effamt( NomadJukebox *jukebox, gint8 effamt )
{
	nomad_jukebox_add_job( jukebox, SET_EAX,
				  GINT_TO_POINTER( NJB_SOUND_SET_EAXAMT ),
				  GINT_TO_POINTER( (gint)effamt ),
				  NULL, 0 );
}

void nomad_jukebox_set_midfreq( NomadJukebox *jukebox, gint8 freq )
{
	nomad_jukebox_add_job( jukebox, SET_EAX,
				  GINT_TO_POINTER( NJB_SOUND_SET_MIDFREQ ),
				  GINT_TO_POINTER( (gint)freq ),
				  NULL, 0 );
}

void nomad_jukebox_set_effect( NomadJukebox *jukebox, gint8 effect )
{
	nomad_jukebox_add_job( jukebox, SET_EAX,
				  GINT_TO_POINTER( NJB_SOUND_SET_EAX ),
				  GINT_TO_POINTER( (gint)effect ),
				  NULL, 0 );
}

void nomad_jukebox_set_headphone( NomadJukebox *jukebox, gint8 hpmode )
{
	nomad_jukebox_add_job( jukebox, SET_EAX,
				  GINT_TO_POINTER( NJB_SOUND_SET_HEADPHONE ),
				  GINT_TO_POINTER( (gint)hpmode ),
				  NULL, 0 );
}

void nomad_jukebox_set_rearmode( NomadJukebox *jukebox, gint8 rearmode )
{
	nomad_jukebox_add_job( jukebox, SET_EAX,
				  GINT_TO_POINTER( NJB_SOUND_SET_REAR ),
				  GINT_TO_POINTER( (gint)rearmode  ),
				  NULL, 0 );
}

static void nomad_jukebox_eax_set( NomadJukebox *jukebox, gint type,
				      gint8 value )
{
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->njb != NULL );
	g_return_if_fail( jukebox->connected );

	if( NJB_Adjust_Sound( jukebox->njb, type, value ) == -1 ) {
		gchar *errmsg;
		
		errmsg = g_strdup_printf( "Failed to adjust EAX settings" );
		
		g_mutex_lock( jukebox->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
			       errmsg );
		g_mutex_unlock( jukebox->job_queue_mutex );
		
		g_free( errmsg );
	}
}

void nomad_jukebox_upload( NomadJukebox *jukebox, GList *list )
{
	nomad_jukebox_add_job( jukebox, UPLOAD,
				  g_list_copy( list ),
				  NULL,
				  NULL, 0 );
}

static void nomad_jukebox_upload_real( NomadJukebox *jukebox, 
					  GList *list )
{
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->njb != NULL );
	g_return_if_fail( jukebox->connected );

	while( ! jukebox->abort && list ) {
		NomadTrack *track;
		guint32 tracknum;
		guint32 length;
		gchar *dummy;

		if( jukebox->transfer_file ) {
			g_free( jukebox->transfer_file );
			jukebox->transfer_file = NULL;
		}
		
		track = (NomadTrack*)list->data;

		if( track ) {
			jukebox->transfer_file = g_strdup( track->name );
			
			length = mmss_to_seconds( track->length );
			tracknum = strtoul( track->track, &dummy, 10 );

			if( NJB_Send_Track( jukebox->njb, 
					    jukebox->transfer_file,
					    track->codec,
					    track->title,
					    track->album,
					    track->genre,
					    track->artist,
					    length,
					    tracknum,
					    track->year, 0,
					    nomad_jukebox_xfer_cb,
					    jukebox, &track->id ) != -1 ) {
				guint total;
				/* insert a copy of the
				   NomadTrack into tracks hashtable */
				track = nomad_track_copy( track );
				g_hash_table_insert( jukebox->tracks,
						     GUINT_TO_POINTER( track->id ),
						     track );
				total = g_hash_table_size( jukebox->tracks );
				/* add to album / artist / genre stores */
				g_mutex_lock( jukebox->job_queue_mutex );
				g_signal_emit( G_OBJECT( jukebox ),
					       nomad_jukebox_signals[ TRACK_ADD_SIGNAL ],
					       0, total, track );
				g_mutex_unlock( jukebox->job_queue_mutex );
			} else {
				gchar *errmsg;
				
				errmsg = g_strdup_printf( "Failed to transfer %s to jukebox",
							  jukebox->transfer_file );
				g_warning( "%s", njb_error_geterror());	
				g_mutex_lock( jukebox->job_queue_mutex );
				g_signal_emit( G_OBJECT( jukebox ),
					       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
					       errmsg );
				g_mutex_unlock( jukebox->job_queue_mutex );
			
				g_free( errmsg );
			}
		}
		list = list->next;
	}
	/* signal uploading complete */
	g_mutex_lock( jukebox->job_queue_mutex );
	g_signal_emit( G_OBJECT( jukebox ),
		       nomad_jukebox_signals[ UPLOAD_COMPLETE_SIGNAL ], 0 );
	g_mutex_unlock( jukebox->job_queue_mutex );
	jukebox->abort = 0;
}


void nomad_jukebox_download( NomadJukebox *jukebox, 
				const gchar *dest,
				const gchar *format,
				GList *list )
{
	g_return_if_fail( dest != NULL );
	g_return_if_fail( format != NULL );
	g_return_if_fail( list != NULL );

	nomad_jukebox_add_job( jukebox, DOWNLOAD,
				  g_strdup( dest ),
				  g_strdup( format ),
				  g_list_copy( list ),
				  0 );
}

static void nomad_jukebox_download_real( NomadJukebox *jukebox,
					    const gchar *dest,
					    const gchar *format,
					    GList *list )
{
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->njb != NULL );
	g_return_if_fail( jukebox->connected );

	while( ! jukebox->abort && list ) {
		NomadTrack *daptrack;	
		gchar *dummy;
		u_int32_t size;
		gchar *path;
		
		daptrack = nomad_jukebox_request_track( jukebox, GPOINTER_TO_UINT( list->data ) );
		g_assert( daptrack );

		size = strtoul( daptrack->size, &dummy, 10 );
		
		if( jukebox->transfer_file ) {
			g_free( jukebox->transfer_file );
			jukebox->transfer_file = NULL;
		}
		path = nomad_jukebox_filename_format( dest,
							 format,
							 daptrack );
		jukebox->transfer_file = path;
		
		/* we possibly need to create the directory
		   for jukebox->transfer_file */
		path = g_dirname( path );
		if( nomad_mkdir_recursive( path ) && 
		    NJB_Get_Track( jukebox->njb, daptrack->id, size, 
				   jukebox->transfer_file,
				   nomad_jukebox_xfer_cb,
				   jukebox ) != -1 ) {
			/* got */
			g_mutex_lock( jukebox->job_queue_mutex );
			g_signal_emit( G_OBJECT( jukebox ),
				       nomad_jukebox_signals[ DOWNLOADED_TRACK_SIGNAL ],
				       0, jukebox->transfer_file,
				       daptrack );
			g_mutex_unlock( jukebox->job_queue_mutex );
		} else {
			gchar *errmsg;
			gchar *trackname;
			
			trackname = nomad_track_get_title( daptrack );

			errmsg = g_strdup_printf( "Failed to transfer %s from the jukebox",
						  trackname );
			g_warning( "%s\n", njb_error_geterror() );	
			g_mutex_lock( jukebox->job_queue_mutex );
			g_signal_emit( G_OBJECT( jukebox ),
				       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
				       errmsg );
			g_mutex_unlock( jukebox->job_queue_mutex );
			
			g_free( errmsg );
			g_free( trackname );
		}
		g_free( path );
		nomad_track_free( daptrack );

		list = list->next;
	}
	/* signal downloading complete */
	g_mutex_lock( jukebox->job_queue_mutex );
	g_signal_emit( G_OBJECT( jukebox ),
		       nomad_jukebox_signals[ DOWNLOAD_COMPLETE_SIGNAL ],
		       0 );
	g_mutex_unlock( jukebox->job_queue_mutex );
	jukebox->abort = 0;
}

void nomad_jukebox_play_track( NomadJukebox *jukebox, guint trackid )
{
	nomad_jukebox_add_job( jukebox, PLAY_TRACK,
				  GUINT_TO_POINTER( trackid ),
				  NULL,
				  NULL, 0 );
}

static void nomad_jukebox_play_track_real( NomadJukebox *jukebox, 
					      guint trackid )
{
	NomadTrack *track;

	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->njb != NULL );
	g_return_if_fail( jukebox->connected );

	track = g_hash_table_lookup( jukebox->tracks, GUINT_TO_POINTER( trackid ) );
	g_assert( track );

	NJB_Stop_Play( jukebox->njb );

	jukebox->playing = track;

	if( NJB_Play_Track( jukebox->njb, track->id ) != -1 ) {
		g_mutex_lock( jukebox->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ PLAY_BEGIN_SIGNAL ],
			       0, track );
		g_mutex_unlock( jukebox->job_queue_mutex );
	} else {
		gchar *errmsg;
		gchar *trackname;
		
		trackname = nomad_track_get_title( track );
		
		errmsg = g_strdup_printf( "Failed to play %s",
					  trackname );
		g_mutex_lock( jukebox->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
			       errmsg );
		g_mutex_unlock( jukebox->job_queue_mutex );
		
		g_free( errmsg );
		g_free( trackname );
	}
}

void nomad_jukebox_play_elapsed( NomadJukebox *jukebox )
{
	nomad_jukebox_add_job( jukebox, PLAY_ELAPSED,
				  NULL,
				  NULL,
				  NULL, 0 );
}

static void nomad_jukebox_play_elapsed_real( NomadJukebox *jukebox )
{
	guint16 elapsed;
	gint changed;

	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->njb != NULL );
	g_return_if_fail( jukebox->connected );

	if( NJB_Elapsed_Time( jukebox->njb, &elapsed, &changed ) != 0 ) {
		changed = 1;
		elapsed = 0;
	}
	g_mutex_lock( jukebox->job_queue_mutex );
	g_signal_emit( G_OBJECT( jukebox ),
		       nomad_jukebox_signals[ PLAY_PROGRESS_SIGNAL ],
		       0, elapsed, changed, jukebox->playing );
	g_mutex_unlock( jukebox->job_queue_mutex );
}

void nomad_jukebox_play_stop( NomadJukebox *jukebox )
{
	nomad_jukebox_add_job( jukebox, PLAY_STOP,
				  NULL,
				  NULL,
				  NULL, 0 );
}

static void nomad_jukebox_play_stop_real( NomadJukebox *jukebox )
{
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->njb != NULL );
	g_return_if_fail( jukebox->connected );

	NJB_Stop_Play( jukebox->njb );

	jukebox->playing = NULL;

	g_mutex_lock( jukebox->job_queue_mutex );
	g_signal_emit( G_OBJECT( jukebox ),
		       nomad_jukebox_signals[ PLAY_FINISHED_SIGNAL ],
		       0 );
	g_mutex_unlock( jukebox->job_queue_mutex );
}


void nomad_jukebox_abort( NomadJukebox *jukebox )
{
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );

	jukebox->abort = -1;
}

GHashTable* nomad_jukebox_request_tracks( NomadJukebox *jukebox )
{
	return jukebox->tracks;
}

GHashTable *nomad_jukebox_request_playlists( NomadJukebox *jukebox )
{
	return jukebox->playlists;
}

NomadTrack *nomad_jukebox_request_track( NomadJukebox *jukebox,
					       guint id )
{
	NomadTrack *track;
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), NULL );

	track = (NomadTrack*)
		g_hash_table_lookup( jukebox->tracks, 
				     GUINT_TO_POINTER( id ) );
	if( track ) {
		track = nomad_track_copy( track );
	}

	return track;
}

/* static stuff */
static void nomad_jukebox_refresh_id( NomadJukebox *jukebox )
{
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->njb != NULL );
	g_return_if_fail( jukebox->connected );

	if( jukebox->njbid ) {
		g_free( jukebox->njbid );
	}
	jukebox->njbid = NJB_Ping( jukebox->njb );
}

static gint nomad_jukebox_xfer_cb( u_int64_t sent, u_int64_t total,
				      const gchar *buf, unsigned int len,
				      void *data )
{
	NomadJukebox *jukebox;
	gint ret;

	jukebox = NOMAD_JUKEBOX( data );

	g_mutex_lock( jukebox->job_queue_mutex );
	g_signal_emit( G_OBJECT( jukebox ),
		       nomad_jukebox_signals[ TRANSFER_SIGNAL ], 0,
		       jukebox->transfer_file, FALSE,
		       sent, total );
	g_mutex_unlock( jukebox->job_queue_mutex );

	ret = jukebox->abort;

	return ret;
}


static gpointer nomad_jukebox_transfer_thread( gpointer data )
{
	NomadJukebox *jukebox;
	GQueue *jobs;
	gboolean terminate;

	jukebox = NOMAD_JUKEBOX( data );
	jobs = jukebox->job_queue;

	terminate = FALSE;

	while( ! terminate ) {
		NomadJukeboxJob *job;

		/* wait till we have something to do */
		sem_wait( &jukebox->transfer_semaphore );

		/* pull job off the queue */
		g_mutex_lock( jukebox->job_queue_mutex );

		job = (NomadJukeboxJob *)g_queue_pop_head( jobs );
	       
		g_mutex_unlock( jukebox->job_queue_mutex );

		/* hmm, shouldn't happen */
		if( ! job ) {
			continue;
		}
		
		nomad_jukebox_refresh_id( jukebox );

		/* process job */
		switch( job->type ) {
		case GET_TIME:
			nomad_jukebox_get_time_real( jukebox );
			break;
		case BUILD_TRACKLIST:
			nomad_jukebox_build_tracklist_real( jukebox );
			break;
	
		case SET_OWNER_STRING:
			nomad_jukebox_set_ownerstring_real( jukebox,
							       job->data );
			nomad_jukebox_getusage_real( jukebox );
			g_free( job->data );
		case GET_USAGE:
			nomad_jukebox_getusage_real( jukebox );
			break;
		case UPLOAD:
			nomad_jukebox_upload_real( jukebox, job->data );
			nomad_jukebox_getusage_real( jukebox );
			g_list_free( job->data );
			break;
		case DOWNLOAD:
			nomad_jukebox_download_real( jukebox, job->data,
							job->data2,
							job->data3 );
			g_free( job->data );
			g_free( job->data2 );
			g_list_free( job->data3 );
			break;
	
		case DELETE_TRACKS:
			nomad_jukebox_delete_tracks_real( jukebox,
							     job->data );
			nomad_jukebox_getusage_real( jukebox );
			g_list_free( job->data );
			break;
		case SET_METADATA:
			nomad_jukebox_set_metadata_real( jukebox,
							    job->data );
			/* job->data is either already free, or
			   it is in use */
			break;
		case GET_EAX:
			nomad_jukebox_get_eax_real( jukebox );
			break;
		case SET_EAX:
			nomad_jukebox_eax_set( jukebox, 
						  GPOINTER_TO_INT( job->data ),
						  GPOINTER_TO_INT( job->data2 ) );
			break;
		case PLAY_TRACK:
			nomad_jukebox_play_track_real( jukebox,
							  GPOINTER_TO_UINT( job->data ) );
			break;
		case PLAY_ELAPSED:
			nomad_jukebox_play_elapsed_real( jukebox );
			break;
		case PLAY_STOP:
			nomad_jukebox_play_stop_real( jukebox );
			break;
		case TERMINATE:
			terminate = TRUE;
			break;

		case BUILD_PLAYLIST:
			nomad_jukebox_build_playlist_real( jukebox );
			break;
		case DELETE_TRACKS_FROM_PLAYLIST:
                        nomad_jukebox_delete_tracks_from_playlist_real( jukebox,
									   job->data,
									   (const gchar*)job->data2 );
                        g_list_free( job->data );
			if( job->data2 ) {
				g_free( job->data2 );
			}
                        break;
                case CREATE_PLAYLIST:
                        nomad_jukebox_create_playlist_real( jukebox,
                                                               job->data );
                        nomad_jukebox_getusage_real( jukebox );
                        g_free( job->data );
                        break;
                case DELETE_PLAYLIST:
                        nomad_jukebox_delete_playlist_real( jukebox,
                                                               (const gchar*)job->data );
			g_free( job->data );
                        nomad_jukebox_getusage_real( jukebox );
                        break;
                case RENAME_PLAYLIST:
                        nomad_jukebox_rename_playlist_real( jukebox,
                                                               (const gchar*)job->data,
                                                               (const gchar*)job->data2 );
                        nomad_jukebox_getusage_real( jukebox );
			g_free( job->data );
                        g_free( job->data2 );
                        break;
                case ADD_TO_PLAYLIST:
                        nomad_jukebox_add_tracks_to_playlist_real( jukebox,
                                                                      (const gchar*)job->data,
                                                                      job->data2 );
                        nomad_jukebox_getusage_real( jukebox );
			g_free( job->data );
                        g_list_free( job->data2 );
                        break;
		default:
			if( job->type >= MAX_JOB_NUM ) {
				g_warning( "Invalid NomadJukeboxJobType: %i\n",
					   job->type );
			} else {
				g_warning( "Unimplemented NomadJukeboxJobType: %i\n",
					   job->type );
			}
			break;
		}
		g_free( job );
	}

	return NULL;
}

static gchar *nomad_jukebox_filename_format( const gchar *dest, 
						const gchar *format,
						NomadTrack *track )
{
	GString *string;
	const gchar *temp;
	gchar *ret;

	string = g_string_new( dest );
	g_string_append_c( string, '/' );

	for( temp = format; *temp; ++ temp ) {
		if( *temp == '%' ) {
			gchar *esc;

			temp ++;
			switch( *temp ) {
			case 'a': /* artist */
				esc = nomad_filename_escape(track->artist );
				g_string_append( string, esc );
				g_free( esc );
				break;
			case 't': /* title */
				esc = nomad_filename_escape( track->title );
				g_string_append( string, esc );
				g_free( esc );
				break;
			case 'b': /* album */
				esc = nomad_filename_escape( track->album );
				g_string_append( string, esc );
				g_free( esc );
				break;
			case 'g': /* genre */
				esc = nomad_filename_escape( track->genre );
				g_string_append( string, esc );
				g_free( esc );
				break;
			case 'y': /* year */
				esc = nomad_filename_escape( track->year );
				g_string_append( string, esc );
				g_free( esc );
				break;
			case 'n': /* track */		
				esc = nomad_filename_escape( track->track );
				g_string_append( string, esc );

				g_free( esc );
				break;
			default:
				temp --;
				break;
			}
		} else {
			g_string_append_c( string, *temp );
		}
	}

	/* append extension */
	if( ! strcmp( NJB_CODEC_MP3, track->codec ) ) {
		g_string_append( string, ".mp3" );
	} else if( ! strcmp( NJB_CODEC_WMA, track->codec ) ) {
		g_string_append( string, ".wma" );
	} else if( ! strcmp( NJB_CODEC_WAV, track->codec ) ) {
		g_string_append( string, ".wav" );
	}

	ret = string->str;

	g_string_free( string, FALSE );

	return ret;
}

static playlist_t *nomad_jukebox_fetch_playlist( NomadJukebox *jukebox,
						    const gchar *name )
{
	playlist_t *ret;

        g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), NULL );
        g_return_val_if_fail( jukebox->njb != NULL, NULL );
        g_return_val_if_fail( jukebox->connected, NULL );
        g_return_val_if_fail( name != NULL, NULL );

	NJB_Reset_Get_Playlist( jukebox->njb );
	for( ret = NULL; ! ret; ) {
		ret = NJB_Get_Playlist( jukebox->njb );
		if( ret ) {
			if( ! strcmp( name, ret->name ) ) {
				break;
			}
			playlist_destroy( ret );
			ret = NULL;
		}
	}

	return ret;
}

static void nomad_jukebox_add_job( NomadJukebox *jukebox,
				      NomadJukeboxJobType type,
				      gpointer data,
				      gpointer data2,
				      gpointer data3,
				      guint priority )
{
	NomadJukeboxJob *job;
	GList *ins;
	NomadJukeboxJob *j;

	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->njb != NULL );
	g_return_if_fail( jukebox->connected );

	job = g_new0( NomadJukeboxJob, 1 );
	job->type = type;
	job->data = data;
	job->data2 = data2;
	job->data3 = data3;
	job->priority = priority;

	g_mutex_lock( jukebox->job_queue_mutex );

	/* insert in priority order */
	ins = jukebox->job_queue->tail;
	while( ins ) {
		j = (NomadJukeboxJob*)ins->data;
		
		if( j->priority >= priority ) {
			/* insert here */
			GList *newitem;
			
			newitem = g_list_alloc();
			newitem->data = job;
			
			if( ins->next ) {
				ins->next->prev = newitem;
			} else {
				jukebox->job_queue->tail = newitem;
			}
			newitem->next = ins->next;
			ins->next = newitem;
			newitem->prev = ins;
			
			jukebox->job_queue->length ++;
			break;
		}
		ins = ins->prev;
	}
	
	if( ! ins ) {
		g_queue_push_head( jukebox->job_queue, job );
	}

	g_mutex_unlock( jukebox->job_queue_mutex );
	sem_post( &jukebox->transfer_semaphore );
}

/* G Object stuff */

GNOME_CLASS_BOILERPLATE( NomadJukebox,
			nomad_jukebox,
			GObject,
			G_TYPE_OBJECT );

static void
nomad_jukebox_class_init( NomadJukeboxClass *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS( klass );

	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = nomad_jukebox_finalize;
	object_class->get_property = nomad_jukebox_get_prop;
	object_class->set_property = nomad_jukebox_set_prop;

	nomad_jukebox_signals[ TIME_SIGNAL ] = 
		g_signal_new( "time",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       time ),
			      NULL, NULL,
			      nomad_marshal_VOID__STRING,
			      G_TYPE_NONE, 1,
			      G_TYPE_STRING );

	nomad_jukebox_signals[ USAGE_SIGNAL ] = 
		g_signal_new( "usage",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       usage ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT64_UINT64_UINT64,
			      G_TYPE_NONE, 3,
			      G_TYPE_UINT64,
			      G_TYPE_UINT64,
			      G_TYPE_UINT64 );

	nomad_jukebox_signals[ SCANNING_TRACKS_SIGNAL ] = 
		g_signal_new( "scanning_tracks",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       scanning_tracks ),
			      NULL, NULL,
			      nomad_marshal_VOID__VOID,
			      G_TYPE_NONE, 0 );

	nomad_jukebox_signals[ SCANNED_TRACKS_SIGNAL ] = 
		g_signal_new( "scanned_tracks",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       scanned_tracks ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT,
			      G_TYPE_NONE, 1,
			      G_TYPE_UINT );

	nomad_jukebox_signals[ SCANNING_PLAYLISTS_SIGNAL ] = 
		g_signal_new( "scanning_playlists",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       scanning_playlists ),
			      NULL, NULL,
			      nomad_marshal_VOID__VOID,
			      G_TYPE_NONE, 0 );

	nomad_jukebox_signals[ PLAYLIST_ADD_SIGNAL ] = 
		g_signal_new( "playlist_add",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       playlist_add ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_STRING,
			      G_TYPE_NONE, 2,
			      G_TYPE_UINT,
			      G_TYPE_STRING );

	nomad_jukebox_signals[ SCANNED_PLAYLISTS_SIGNAL ] = 
		g_signal_new( "scanned_playlists",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       scanned_playlists ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT,
			      G_TYPE_NONE, 1,
			      G_TYPE_UINT );

	nomad_jukebox_signals[ TRANSFER_SIGNAL ] = 
		g_signal_new( "transfer",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       transfer ),
			      NULL, NULL,
			      nomad_marshal_VOID__STRING_BOOLEAN_UINT64_UINT64,
			      G_TYPE_NONE, 4,
			      G_TYPE_STRING,
			      G_TYPE_BOOLEAN,
			      G_TYPE_UINT64,
			      G_TYPE_UINT64 );

	nomad_jukebox_signals[ UPLOAD_COMPLETE_SIGNAL ] = 
		g_signal_new( "upload_complete",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       upload_complete ),
			      NULL, NULL,
			      nomad_marshal_VOID__VOID,
			      G_TYPE_NONE, 0 );
	nomad_jukebox_signals[ DOWNLOAD_COMPLETE_SIGNAL ] = 
		g_signal_new( "download_complete",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       download_complete ),
			      NULL, NULL,
			      nomad_marshal_VOID__VOID,
			      G_TYPE_NONE, 0 );
	nomad_jukebox_signals[ DOWNLOADED_TRACK_SIGNAL ] = 
		g_signal_new( "downloaded_track",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       downloaded_track ),
			      NULL, NULL,
			      nomad_marshal_VOID__STRING_POINTER,
			      G_TYPE_NONE, 2,
			      G_TYPE_STRING,
			      G_TYPE_POINTER );

	nomad_jukebox_signals[ PLAYLIST_TRACK_ERASED_SIGNAL ] = 
		g_signal_new( "playlist_track_erased",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       playlist_track_erased ),
			      NULL, NULL,
			      nomad_marshal_VOID__STRING_STRING_STRING_STRING,
			      G_TYPE_NONE, 4,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING );
	nomad_jukebox_signals[ PLAYLIST_TRACKS_ERASED_SIGNAL ] = 
		g_signal_new( "playlist_tracks_erased",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       playlist_tracks_erased ),
			      NULL, NULL,
			      nomad_marshal_VOID__STRING,
			      G_TYPE_NONE, 1,
			      G_TYPE_STRING );
	nomad_jukebox_signals[ PLAYLIST_REMOVE_SIGNAL ] = 
		g_signal_new( "playlist_remove",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       playlist_remove ),
			      NULL, NULL,
			      nomad_marshal_VOID__STRING,
			      G_TYPE_NONE, 1,
			      G_TYPE_UINT );
	nomad_jukebox_signals[ PLAYLIST_RENAMED_SIGNAL ] = 
		g_signal_new( "playlist_renamed",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       playlist_renamed ),
			      NULL, NULL,
			      nomad_marshal_VOID__STRING_STRING,
			      G_TYPE_NONE, 2,
			      G_TYPE_STRING,
			      G_TYPE_STRING );
	nomad_jukebox_signals[ EAX_SIGNAL ] = 
		g_signal_new( "eax",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       eax ),
			      NULL, NULL,
			      nomad_marshal_VOID__POINTER,
			      G_TYPE_NONE, 1,
			      G_TYPE_POINTER );
	nomad_jukebox_signals[ PLAY_BEGIN_SIGNAL ] = 
		g_signal_new( "play_begin",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       play_begin ),
			      NULL, NULL,
			      nomad_marshal_VOID__POINTER,
			      G_TYPE_NONE, 1,
			      G_TYPE_POINTER );
	nomad_jukebox_signals[ PLAY_PROGRESS_SIGNAL ] = 
		g_signal_new( "play_progress",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       play_progress ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_BOOLEAN_POINTER,
			      G_TYPE_NONE, 3,
			      G_TYPE_UINT,
			      G_TYPE_BOOLEAN,
			      G_TYPE_POINTER );	
	nomad_jukebox_signals[ PLAY_FINISHED_SIGNAL ] = 
		g_signal_new( "play_finished",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       play_finished ),
			      NULL, NULL,
			      nomad_marshal_VOID__VOID,
			      G_TYPE_NONE, 0 );

	nomad_jukebox_signals[ TRACK_ADD_SIGNAL ] = 
		g_signal_new( "track_add",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       track_add ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_POINTER,
			      G_TYPE_NONE, 2,
			      G_TYPE_UINT,
			      G_TYPE_POINTER );
	nomad_jukebox_signals[ TRACK_REMOVE_SIGNAL ] = 
		g_signal_new( "track_remove",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       track_remove ),
			      NULL, NULL,
			      nomad_marshal_VOID__POINTER,
			      G_TYPE_NONE, 1,
			      G_TYPE_POINTER );	
	nomad_jukebox_signals[ TRACK_CHANGED_SIGNAL ] = 
		g_signal_new( "track_changed",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       track_changed ),
			      NULL, NULL,
			      nomad_marshal_VOID__POINTER_POINTER,
			      G_TYPE_NONE, 2,
			      G_TYPE_POINTER,
			      G_TYPE_POINTER );

	nomad_jukebox_signals[ ERROR_SIGNAL ] = 
		g_signal_new( "error",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       error ),
			      NULL, NULL,
			      nomad_marshal_VOID__STRING,
			      G_TYPE_NONE, 1,
			      G_TYPE_STRING );

}

static void
nomad_jukebox_instance_init( NomadJukebox *jukebox )
{
	jukebox->connected = FALSE;

	jukebox->njb = NULL;
	jukebox->njbid = NULL;

	jukebox->firmware = NULL;

	jukebox->owner = NULL;

	jukebox->tracks = g_hash_table_new_full( NULL, NULL,
			NULL, (GDestroyNotify)nomad_track_free );
	/* the key here is the same as that used in the next hashtable
	 * therefore there is nothing to free */
	jukebox->playlistids = g_hash_table_new( NULL, NULL );
	jukebox->playlists = g_hash_table_new_full( g_str_hash, 
			g_str_equal,
			(GDestroyNotify)g_free, 
			(GDestroyNotify)g_slist_free );
	
	jukebox->job_queue = g_queue_new();
	jukebox->job_queue_mutex = g_mutex_new();

	jukebox->lock = g_mutex_new();

	sem_init( &jukebox->transfer_semaphore, 0, 0 );
	jukebox->transfer_thread = 
		g_thread_create( nomad_jukebox_transfer_thread,
				 jukebox, TRUE, NULL );
}

static void
nomad_jukebox_set_prop( GObject *object, guint prop_id, 
			 const GValue *value, GParamSpec *spec )
{
	NomadJukebox *jukebox;

	jukebox = NOMAD_JUKEBOX( object );

	switch( prop_id ) {
	default:
		break;
	}
}

static void
nomad_jukebox_get_prop( GObject *object, guint prop_id, 
			 GValue *value, GParamSpec *spec )
{
	NomadJukebox *jukebox;

	jukebox = NOMAD_JUKEBOX( object );

	switch( prop_id ) {
	default:
		break;
	}
}

static void
nomad_jukebox_finalize( GObject *object )
{
	NomadJukebox *jukebox;
	NomadJukeboxJob *job;

	jukebox = NOMAD_JUKEBOX( object );

	job = g_new0( NomadJukeboxJob, 1 );
	job->type = TERMINATE;

	g_mutex_lock( jukebox->job_queue_mutex );
	g_queue_push_head( jukebox->job_queue, job );
	g_mutex_unlock( jukebox->job_queue_mutex );
	sem_post( &jukebox->transfer_semaphore );

	g_thread_join( jukebox->transfer_thread );

	g_mutex_free( jukebox->job_queue_mutex );
	g_mutex_free( jukebox->lock );
	
	g_hash_table_destroy( jukebox->tracks );
	g_hash_table_destroy( jukebox->playlistids );
	g_hash_table_destroy( jukebox->playlists );
	
	G_OBJECT_CLASS( parent_class )->finalize( object );
}


