#ifndef __SWITCH_H__
#define __SWITCH_H__

#ifdef __cplusplus
extern "C" {
#endif

#define __USE_ISOC99
#include <stdio.h>
#include <unistd.h>
#include <pthread.h>

extern int services /** This global variable contains a combination of 'EnumServices' values to store the actual capabilities of the system */ ;

/** EnumType represents the type of a network interface.
 */
enum EnumType {
  WIRELESS  /** means the interface is a Wifi card */ = 1,
  ETHERNET  /// means the interface is a basic interface
};

/** List of the services available on a computer
 *  This configuration is made at runtime
 */
enum EnumServices {
  SERVICES_SMTP /** This means the SMTP can be configured by Netswitch */  =  1,
  SERVICES_DHCP /** This means a DHCP client is available */ =  2,
  SERVICES_WIFI /** This means the tools managing WiFi cards are present */=  4,
  SERVICES_PPP  /** This means that pppd binary is present (VPN are ok) */ =  8,
  SERVICES_WPA  /** This means wpa_supplicant is present */ = 16,
  SERVICES_SERVICES /** Netswitch know how to manage the boot process */ = 32,
};

#define NETSWITCH_HAS_SMTP	(services & SERVICES_SMTP)
#define NETSWITCH_HAS_DHCP	(services & SERVICES_DHCP)
#define NETSWITCH_HAS_WIFI	(services & SERVICES_WIFI)
#define NETSWITCH_HAS_PPP	(services & SERVICES_PPP)
#define NETSWITCH_HAS_WPA	(services & SERVICES_WPA)
#define NETSWITCH_HAS_SERVICES	(services & SERVICES_SERVICES)

/// List of the modes availables for a wireless card
enum EnumMode {
  AUTO /** Automatic choice (by the driver, generally Managed) */ =0,
  MANAGED /** Try to connect an Access Point */,
  ADHOC /** Makes a point-to-point access, used by localnet */,
  MONITOR /** Passive mode do not work for Cisco cards */
};

/// List of encryptions modes used for wireless cards
enum KeyMode {
  OFF /** No encryption is done */ =0,
  OPEN /** We use WEP keys with 'open' mode */,
  RESTRICTED /** We use WEP keys with 'restricted' mode */,
  WPA /** We use WPA/WPA-PSK/LEAP... with wpa_supplicant (not yet implemented) */
};

/** Structure containing a profile name and letting us
 *  point to another element with the same structure.
 *  Netswitch use it making a closed linked chain.
 *  */
typedef struct profiles_list_t {
  char *name /** Name of the profile */;
  struct profiles_list_t *next /** Address of next element */;
} profiles_list;

/** Structure containing different information about an interface
 * */
typedef struct iface_info_t {
  int status /** Set to 1 if the interface is UP */;
  char *hwaddr /** It is a string containing the MAC address */;
  char *ipaddr /** A string containing IP address */;
  char *ipbroadcast /** A string containing the IP broadcast address */;
  char *ipgateway;
  char *ipnet /** A field containing the network mask */;
  char *wifiessid /** If applicable, contain the ESSID of a WiFi card */;
  int wifilink /** Contains the quality of the Wireless link if applicable */;
} iface_info;

/** Structure containing a Netswitch. This is tha base entity of the
 *  Library. It represents an interface and all its profiles. They are all
 *  represented in a linked chain
 * */
typedef struct netswitch_t {
  char *iface /** Name of the interface (ethX,wlanX,...) */;
  enum EnumType type /** Type of the interface (Wireless, Ethernet) */;
  profiles_list *profiles /** List of the profiles */;
  struct netswitch_t *next /** Link to next netswitch structure. */;
} netswitch;

/** This represents all the datas concerning a profile. All the fields that
 *  can be used are here. If you do not want to use something, please set
 *  it to NULL
 * */
typedef struct profile_t {
  char *ipaddr /** If not null, use a static IP. If null, then we will use a DHCCP */;
  char *hwaddr /** If you want to change your harware address */;
  int netmask /** Number of bits used to determine the network. For example: 24 for a class C network */;
  char *broadcast /** Broadcast address, if you use a subnet */;
  char *gateway /** Gateway to reach other networks */;
  char *domain /** Default domain where we should look for */;
  int rts;
  int frag;
  int freq /** Wifi parameter to determine the frequency of the communication */;
  char *essid /** Wifi parameter to set the SSID (id of a wifi network) */;
  char *nickname /** Wifi parameter to identify your machine on the network */;
  char *nwid;
  int sens /** Wifi sensitivity */;
  enum EnumMode mode /** Wifi mode (See the different modes available) */;
  int channel /** Wifi channel paramter */;
  int rate /** Wifi rate parameter */;
  int sendkey /** Set the default key (used to encrypt) when encryption is set to OPEN or RESTRICTED */;
  int peerdns /** When the DHCP client allows this (dhcpcd), local configuration is not affected by the DHCP client */;
  int metric /** Basic network parameter that let you precise the distance to the gateway */;
  char **keys /** NULL-ended array containing all the keys used to encrypt the wifi connection (limitations depends on memory ;), and your card) */;
  enum KeyMode key_mode /** Determine the wifi encryption (NONE, WEP, WPA). Note that WPA is not yet implemented */;
  char **nameservers /** NULL-ended array containing all the name servers */;
  char *smtp_host /** Defines the name of the SMTP smarthost */;
  char *smtp_domain /** Defines the name of the domain from which mails "seems" to come from (depends on the SMTP server used) */;
  int smtp_port /** Defines the port of the SMTP smarthost */;
  char *smtp_user /** If the smarthost needs an authentication, just give the username */;
  char *smtp_pass /** If the smarthost needs an authentication, just give the password */;
} profile;

/** The structure represent a wifi peer when you did a scan.
 *  Some information are stored here.
 * */
typedef struct netlist_t {
  char *bssid /** BSSID is a unique identifier for AP */;
  char *essid /** ESSID is a common name used to identify a network */;
  enum EnumMode mode /** Represents the mode of the peer (AP or AD-HOC) */;
  char *protocol /** Protocol represents the version of the Wireless protocol used */;
  int encryption /** If encryption is set to 1, then the connection with the peer is encrypted, else, the communication is a plain unsecured connection. */;
  int link /** Represents the quality of a signal (for old drivers, and in dBm) */;
  int quality /** Represents the quality of a signal (in percent) */;
  int rate /** Represents the rate of the communication with this peer */;
  int channel /** Channel is a way to identify the connection (by frequency) */;
  struct netlist_t *next /** All the elements are linked via the next field */;
} netlist;

/// List of the type of privilege acquirement method
enum CONN_TYPE {
  SU  /** We use su program, so the password needed is root */=0,
  SUDO /** We use sudo (for ubuntu compatibility), so user password is needed */
};

typedef void (netswitch_pwd)(char **pwd, enum CONN_TYPE type);
int netswitch_init(netswitch_pwd *p /*, netswitch_block *b*/);
netswitch *netswitch_new();
void netswitch_free(netswitch *n);
int netswitch_autoconnect(netswitch *n);
void netswitch_free_profiles(netswitch *n);
int test_switch();

int netswitch_number();
void netswitch_profiles_list(netswitch *n);
void netswitch_active_profiles(netswitch *n);
void netswitch_active_all(netswitch *n, char *profile);
void netswitch_active(netswitch *n, char *profile);
void netswitch_inactive_profile(netswitch *n, char *profile);
void netswitch_inactive(netswitch *n);
void netswitch_inactive_all_iface(netswitch *n);
void netswitch_delete(netswitch *n, char *profile);
int netswitch_install(netswitch *n, profile *p, char *name);
profile *netswitch_get(netswitch *n, char *profile);
profile *profile_new();
void profile_free(profile *p);

netlist *netswitch_scan();
void netlist_free(netlist *p);

iface_info *netswitch_info(netswitch *n);
void iface_info_free(iface_info *p);

/* Process management tools */

typedef struct process_t {
  int pid;
  int read_fd;
  int write_fd;
  //int lock;
  pthread_mutex_t *lock;
  FILE *read_stream;
  FILE *write_stream;
  void (*stop_func)();
} process;

int open_su_term(process *p, netswitch_pwd *getpwd);
int process_system(process *p, char *cmd);
FILE *process_read(process *p, char *cmd);

typedef void (callback)();
void check_event(callback *profiles);

#ifdef __cplusplus
}
#endif

#endif
