#ifndef FILE_LOCALHEAP
#define FILE_LOCALHEAP

/**************************************************************************/
/* File:   localheap.hpp                                                  */
/* Author: Joachim Schoeberl                                              */
/* Date:   19. Apr. 2000                                                  */
/**************************************************************************/


/**
   Exception on heap overflow.
   Thrown by allocation on LocalHeap.
*/
class LocalHeapOverflow : public Exception
{
public:
  LocalHeapOverflow ();
  virtual ~LocalHeapOverflow ();
};

/**
   Optimized memory handler.
   One block of data is organized as stack memory. 
   One can allocate memory out of it. This increases the stack pointer.
   With \Ref{CleanUp}, the pointer is reset to the beginning or to a
   specific position. 
 */
class LocalHeap
{
  char * data;
  char * p;
  unsigned int totsize;
  bool owner;

public:
  /// Allocate one block of size asize.
  LocalHeap (unsigned int asize);

  /// Use provided memory for the LocalHeap
  LocalHeap (char * adata, unsigned int asize) throw ()
  {
    totsize = asize;
    data = adata;
    p = data;
    owner = 0;
  }
  
  /// free memory
  ~LocalHeap ()
  {
    if (owner)
      delete [] data;
  }
  
  /// delete all memory on local heap
  void CleanUp() throw ()
  {
    p = data;
  }

  /// returns heap-pointer
  void * GetPointer () throw ()
  {
    return p;
  }

  /// deletes memory back to heap-pointer
  void CleanUp (void * addr) throw ()
  {
    p = (char*)addr;
  }

  /// allocates size bytes of memory from local heap
  void * Alloc (unsigned int size) throw (LocalHeapOverflow)
  {
    char * oldp = p;
    
    // 16 byte allignment
    size += (16 - size % 16);
    p += size;

    if ( (p - data) >= int(totsize) )
      ThrowException();

    return oldp;
  }

  /// allocates size objects of type T on local heap
  template <typename T>
  T * Alloc (unsigned int size) throw (LocalHeapOverflow)
  {
    char * oldp = p;
    size *= sizeof (T);

    // 16 byte allignment
    size += (16 - size % 16);
    p += size;

    if ( (p - data) >= int(totsize) )
      ThrowException();

    return reinterpret_cast<T*> (oldp);
  }


  ///
  void ThrowException() throw (LocalHeapOverflow);

  /// free memory (dummy function)
  void Free (void * data) throw () 
  {
    ;
  }

  /// available memory on LocalHeap
  int Available () const throw () { return (totsize - (p-data)); }
};


template <int S>
class LocalHeapMem : public LocalHeap
{
  char mem[S];
public:
  LocalHeapMem () throw () : LocalHeap (mem, S) { ; }
};



#endif
