/* pdu-generic.c
 
   A generic PDU builder.

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "pbuild-priv.h"

/*
 * Write a default value for a field based on the default information
 * provided for the field (defval_t pointer in the field_t structure)
 */
static void
write_field_default(const GNode *node, void *dest, const field_t *field)
{
    const GNode *parm, *pdu;
    struct payload *data;
    const field_t *referred_field;
    size_t len;
    uint32_t u32;
    long byte_offset;
    int bit_shift;
    uint32_t field_value;

    switch (field->defval->type) {
    case PDU_DEF_NUM:
	u32 = field->defval->_number;
	break;
    case PDU_DEF_RAND:
	while ( (u32 = rand() ) == 0);
	break;
    case PDU_DEF_PID:
	u32 = getpid();
	break;
    case PDU_DEF_LEN:
	if (!(parm = _pb_findnode(node, field->defval->_fname) ) )
	    /*
	     * This shouldn't happen; it means that the field references
	     * an inexistant field.
	     */
	    break;

	referred_field = ( (struct node_data *) parm->data)->_data_parm.field;

	switch (referred_field->type) {
	case PDU_FTYPE_DATA:
	    data = ( (struct node_data *) parm->data)->_data_parm.data;

	    if (referred_field->size != 0)
		/* Length of the data field is constrained */
		len = data->len > referred_field->size
		      ? referred_field->size : data->len;
	    else
		/* Unconstrained data field length */
		len = data->len;

	    u32 = len;
	    break;
	default:
	    /*
	     * No other field type is handled for now.
	     */
	    ;
	}
	break;
    case PDU_DEF_LEN_HDR:
	pdu = _pb_parent_pdu(node);
	u32 =	( (struct node_data *) pdu->data)->_data_pdu.hdr_len
	      + ( (struct node_data *) pdu->data)->_data_pdu.opts_len;
	break;
    case PDU_DEF_LEN_PDU:
	pdu = _pb_parent_pdu(node);
	u32 =	( (struct node_data *) pdu->data)->_data_pdu.hdr_len
	      + ( (struct node_data *) pdu->data)->_data_pdu.opts_len
	      + ( (struct node_data *) pdu->data)->_data_pdu.payload_len;
	break;
    case PDU_DEF_LEN_PAYLOAD:
	pdu = _pb_parent_pdu(node);
	u32 = ( (struct node_data *) pdu->data)->_data_pdu.payload_len;
	break;
    case PDU_DEF_LEN_TLV:
	u32 = pdu_generic_tlv_len(node);
	break;
    case PDU_DEF_LEN_TLV2:
	u32 = pdu_generic_tlv_len2(node);
	break;
    case PDU_DEF_LEN_TLV4:
	u32 = pdu_generic_tlv_len4(node);
	break;
    case PDU_DEF_IP:
    case PDU_DEF_IP6:
    case PDU_DEF_MAC:
	/* Nothing */
	break;
    }

    switch (field->type) {
    case PDU_FTYPE_UINT8:
	( (uint8_t *) dest)[field->offset] = u32;
	break;
    case PDU_FTYPE_UINT16:
	SSVAL(dest, field->offset, htons(u32) );
	break;
    case PDU_FTYPE_UINT32:
	SIVAL(dest, field->offset, htonl(u32) );
	break;
    case PDU_FTYPE_BIT:
	if (u32)
	    ( (uint8_t *) dest)[field->offset / 8] |= 1 << (7 - field->offset % 8);
	break;
    case PDU_FTYPE_BITFIELD:
	byte_offset = (field->offset / 32)*4; /* Note to self: (x/y)*z, where
						 x, y, and z are integers is
						 not the same thing as
						 x*z/y */
	bit_shift = 32 - field->size - field->offset % 32;
	field_value = ntohl(IVAL(dest, byte_offset) )
		      | ( (u32 & ( (1 << field->size) - 1) ) << bit_shift);
	SIVAL(dest, byte_offset, htonl(field_value) );
	break;
    case PDU_FTYPE_IP:
	SIVAL(dest, field->offset, field->defval->_ipaddr);
	break;
    case PDU_FTYPE_MACADDR:
	memcpy(dest + field->offset, field->defval->_macaddr, ETH_ADDR_LEN);
	break;
    case PDU_FTYPE_IP6ADDR:
	memcpy(dest + field->offset, &field->defval->_ip6addr, IP6_ADDR_LEN);
	break;
    case PDU_FTYPE_BRACED_ARGS:
    case PDU_FTYPE_DATA:
    case PDU_FTYPE_NUMTYPE:
    case PDU_FTYPE_BOOL:
    case PDU_FTYPE_UINT:
    case PDU_FTYPE_FIXEDP32:
    case PDU_FTYPE_FIXEDP64:
	/* Do nothing */
	break;
    }
}

/*
 * Write generic field "field", with data pointed to by "data" at
 * destination "dest".
 *
 * Input:
 *   dest: where to store the field.
 *   data: pointer to field data. Type of data depends on field type.
 *   field: pointer to structure defining the field.
 */
static void
write_field(void *dest, void *data, const field_t *field)
{
    size_t len;
    long byte_offset;
    int bit_shift;
    uint32_t field_value;

    switch (field->type) {
    case PDU_FTYPE_UINT8:
	( (uint8_t *) dest)[field->offset] = num_next(data);
	break;
    case PDU_FTYPE_UINT16:
	SSVAL(dest, field->offset, htons(num_next(data) ) );
	break;
    case PDU_FTYPE_UINT32:
	SIVAL(dest, field->offset, htonl(num_next(data) ) );
	break;
    case PDU_FTYPE_BIT:
	if (data)
	    ( (uint8_t *) dest)[field->offset / 8] |= 1 << (7 - field->offset % 8);
	break;
    case PDU_FTYPE_BITFIELD:
	byte_offset = (field->offset / 32)*4; /* Note to self: (x/y)*z, where
						 x, y, and z are integers is
						 not the same thing as
						 x*z/y */
	bit_shift = 32 - field->size - field->offset % 32;
	field_value = ntohl(IVAL(dest, byte_offset) )
		      | ( (num_next(data) & ( (1 << field->size) - 1) )
			 << bit_shift);
	SIVAL(dest, byte_offset, htonl(field_value) );
	break;
    case PDU_FTYPE_IP:
	SIVAL(dest, field->offset, ip_next(data) );
	break;
    case PDU_FTYPE_MACADDR:
	memcpy(dest + field->offset, data, ETH_ADDR_LEN);
	break;
    case PDU_FTYPE_DATA:
	if (field->size != 0)
	    /* Length of the data field is constrained */
	    len = ( (struct payload *) data)->len > field->size
		  ? field->size : ( (struct payload *) data)->len;
	else
	    /* Unconstrained data field length */
	    len = ( (struct payload *) data)->len;

	memcpy(dest + field->offset, ( (struct payload *) data)->data, len);
	break;
    case PDU_FTYPE_IP6ADDR:
	memcpy(dest + field->offset, data, IP6_ADDR_LEN);
	break;
    case PDU_FTYPE_FIXEDP32:
	SSVAL(dest, field->offset,
	      htons( ( (struct fixedp_num *) data)->int_part) );
	SSVAL(dest, field->offset + sizeof(uint16_t),
	      htons( ( (struct fixedp_num *) data)->frac_part) );
	break;
    case PDU_FTYPE_FIXEDP64:
	SIVAL(dest, field->offset,
	      htonl( ( (struct fixedp_num *) data)->int_part) );
	SIVAL(dest, field->offset + sizeof(uint32_t),
	      htonl( ( (struct fixedp_num *) data)->frac_part) );
	break;

    case PDU_FTYPE_BRACED_ARGS:
    case PDU_FTYPE_NUMTYPE:
    case PDU_FTYPE_BOOL:
    case PDU_FTYPE_UINT:
	/* Do nothing */
	break;
    }
}

/*
 * This is generic builder for PDUs.
 */
void
_pb_generic_build(const GNode *pdu, void *dest)
{
    const field_t *f;
    const GNode *parm;
    struct node_data *node_data;

    /*
     * We need to traverse the fields list associated with this node (which
     * represents a PDU, a PDU option, or simply a braced arguments list). For
     * this we must obtain the list of fields first.  This list is stored in
     * different places depending on whether this node represents a PDU or a
     * braced arguments list.
     */

    for (f = ( (struct node_data *) pdu->data)->type == PDU_NODE_PDU
	     ? ( (struct node_data *) pdu->data)->_data_pdu.template->fields
	     : ( (struct node_data *) pdu->data)->_data_braced.fields;
	 f->name;
	 f++) {
	if ( (parm = _pb_findnode(pdu, f->name) ) ) {
	    /*
	     * User-specified parameter; put on the wire the parameter's
	     * value.
	     */

	    node_data = parm->data;

	    if (node_data->type != PDU_NODE_PARM)
		/* Ignore nodes that are not PDU parameters */
		continue;

	    write_field(dest, node_data->_data_parm.data,
			node_data->_data_parm.field);
	} else {
	    /*
	     * User did not specify this parameter; use parameter's default
	     * (if there's one)
	     */

	    if (f->defval)
		write_field_default(pdu, dest, f);
	}
    }
}

/*
 * Calculates the length of a type 1 Type-Length-Value (TLV) definition,
 * i.e. "tlv(type = nn, value = 'some data')". Type 1 TLV uses one byte
 * to store the type and one byte to store the length, and the length
 * includes both the bytes for the type and the length.
 *
 * Input:
 *   - node: the node representing the TLV
 */
size_t
pdu_generic_tlv_len(const GNode *node)
{
    struct payload *data;
    size_t len;

    len = 1 /* type */ + 1 /* length */;

    if ( (data = _pb_pdata(node, "value") ) )
	len += data->len;

    return len;
}

/*
 * Calculates the length of a type 2 Type-Length-Value (TLV) definition,
 * i.e. "tlv(type = nn, value = 'some data')". Type 2 TLV uses two bytes
 * to store the type and two bytes to store the length, the length
 * includes both the bytes for the type and the length, and the length
 * is the total number of bytes TLV. Cisco DTP TLVs are an example of this
 * type of TLV.
 *
 * Input:
 *   - node: the node representing the TLV
 */
size_t
pdu_generic_tlv_len2(const GNode *node)
{
    struct payload *data;
    size_t len;

    len = 2 /* type */ + 2 /* length */;

    if ( (data = _pb_pdata(node, "value") ) )
	len += data->len;

    return len;
}

/*
 * Calculates the length of a type 4 Type-Length-Value (TLV) definition,
 * i.e. "tlv(type = nn, value = 'some data')". Type 4 TLV uses one byte
 * to store the type and one byte to store the length, the length
 * includes both the bytes for the type and the length, and the length
 * is the number of 32-bit words in the TLV. Cisco VTP VLAN info. TLVs
 * are an example of this type of TLV.
 *
 * Input:
 *   - node: the node representing the TLV
 */
size_t
pdu_generic_tlv_len4(const GNode *node)
{
    struct payload *data;
    size_t len;

    len = 1 /* type */ + 1 /* length */;

    if ( (data = _pb_pdata(node, "value") ) )
	len += data->len;

    return len/4; /* Convert number of bytes to number of 32-bit words */
}
