/* tcl_timeval.c
  
   The "timeval" Tcl command and the "timeval" Tcl object
   type are implemented here.

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"
#include "util-tcl.h"

/********************************************************************
 *		    The "timeval" Tcl object type                   *
 ********************************************************************/

/* Forward declarations */
static void update_string(Tcl_Obj *);
static void free_int_rep(Tcl_Obj *);
static int set_from_any(Tcl_Interp *interp, Tcl_Obj *objPtr);
static void dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr);

Tcl_ObjType tclTimevalType = {
    .name = "timeval",
    .freeIntRepProc = &free_int_rep,
    .dupIntRepProc = &dup_int_rep,
    .updateStringProc = &update_string,
    .setFromAnyProc = &set_from_any
};

static int
set_from_any(Tcl_Interp *interp, Tcl_Obj *objPtr _U_)
{
    if (interp)
	Tcl_SetResult(interp,
		      "timeval object internal representation can't be created "
		      "from string", TCL_VOLATILE);

    return TCL_ERROR;
}

static void
dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr)
{
    struct timeval *old, *new;

    old = srcPtr->internalRep.otherValuePtr;
    new = (struct timeval *) ckalloc(sizeof(struct timeval) );
    *new = *old;

    dupPtr->typePtr = &tclTimevalType;
    dupPtr->internalRep.otherValuePtr = new;
}

static void
update_string(Tcl_Obj *timeval)
{
    const char *s;
    struct timeval *t;
    size_t len;

    t = timeval->internalRep.otherValuePtr;

    s = ts2str(t);
    len = strlen(s);

    timeval->bytes = ckalloc(len + 1);
    if (!timeval->bytes)
	return;

    strlcpy(timeval->bytes, s, len + 1);

    timeval->length = len;
}

static void
free_int_rep(Tcl_Obj *obj)
{
    struct timeval *t;

    t = obj->internalRep.otherValuePtr;
    ckfree( (char *) t);
}

Tcl_Obj *
Tcl_NewTimevalObj(struct timeval *tv)
{
    Tcl_Obj *obj;
    struct timeval *t;

    obj = Tcl_NewObj();

    t = (struct timeval *) ckalloc(sizeof(struct timeval) );
    *t = *tv;

    obj->bytes = NULL;
    obj->typePtr = &tclTimevalType;
    obj->internalRep.otherValuePtr = t;

    return obj;
}

/********************************************************************
 *			       timeval				    *
 ********************************************************************/

static int
tcl_timeval_new(Tcl_Interp *interp, int argc, const char **argv)
{
    Tcl_Obj *tv_obj;
    struct timeval tv;

    if (argc != 1) {
	nexp_error(interp, "usage: %s", argv[0]);
	return TCL_ERROR;
    }

    gettimeofday(&tv, NULL);
    tv_obj = Tcl_NewTimevalObj(&tv);
    Tcl_SetObjResult(interp, tv_obj);

    return TCL_OK;
}

static int
tcl_timeval_delta(Tcl_Interp *interp, int argc, const char **argv)
{
    Tcl_Obj *tv_obj, *double_obj;
    struct timeval *tva, *tvb, delta_tv;
    double delta;

    if (argc != 3) {
	nexp_error(interp, "usage: %s <timeval var1> <timeval var2>", argv[0]);
	return TCL_ERROR;
    }

    tv_obj = Tcl_GetVar2Ex(interp, argv[1], NULL, TCL_LEAVE_ERR_MSG);
    if (!tv_obj)
	return TCL_ERROR;

    if (tv_obj->typePtr != &tclTimevalType) {
	nexp_error(interp, "\"%s\" is not a timeval object", argv[1]);
	return TCL_ERROR;
    }

    tva = tv_obj->internalRep.otherValuePtr;

    tv_obj = Tcl_GetVar2Ex(interp, argv[2], NULL, TCL_LEAVE_ERR_MSG);
    if (!tv_obj)
	return TCL_ERROR;

    if (tv_obj->typePtr != &tclTimevalType) {
	nexp_error(interp, "\"%s\" is not a timeval object", argv[2]);
	return TCL_ERROR;
    }

    tvb = tv_obj->internalRep.otherValuePtr;

    timersub(tva, tvb, &delta_tv);

    delta = delta_tv.tv_sec + delta_tv.tv_usec/1000000.0;

    double_obj = Tcl_NewDoubleObj(delta);
    Tcl_SetObjResult(interp, double_obj);

    return TCL_OK;
}

static int
NExp_TimevalCmd(ClientData clientData _U_, Tcl_Interp *interp, int argc,
		const char **argv)
{
    int retval;

    if (argc < 2) {
	nexp_error(interp, "\
usage: %s <new|delta> [command args ...]", argv[0]);
	return TCL_ERROR;
    }

    if (strstr("new", argv[1]) ) {
	argv[1] = "timeval new";
	retval = tcl_timeval_new(interp, argc - 1, &argv[1]);
    } else if (strstr("delta", argv[1]) ) {
	argv[1] = "timeval delta";
	retval = tcl_timeval_delta(interp, argc - 1, &argv[1]);
    } else {
	nexp_error(interp, "unknown %s sub-command \"%s\"", argv[0], argv[1]);
	return TCL_ERROR;
    }

    return retval;
}

static struct nexp_cmd_data cmd_data[] = {
    {"timeval", NULL, NExp_TimevalCmd, 0, 0},

    {NULL, NULL, NULL, 0, 0}
};

void
nexp_init_timeval_cmd(Tcl_Interp *interp)
{
    Tcl_RegisterObjType(&tclTimevalType);

    nexp_create_commands(interp, cmd_data);
}
