/* pdu-icmpv6.c
 
   PDU builder for ICMPv6 messages (RFC 2461 and 2462)

   Copyright (C) 2007, 2008, 2009 Eloy Paris, Darrin Miller

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
To-do:

Add the following options:

Option Name                             Type

Source Link-Layer Address                  1
Target Link-Layer Address                  2
Prefix Information                         3
Redirected Header                          4
MTU                                        5
*/

#include "pbuild-priv.h"
#include "pdu-icmpv6.h"

/*
 * Most ICMPv6 builders only need to calculate the ICMPv6 checksum in the
 * second pass. This generic second pass builder can be used for all these
 * ICMPv6 builders.
 */
static void
pdu_icmpv6_generic_pass2_builder(const GNode *pdu,
				 void *dest,
				 void *prev_pdu_hdr)
{
    struct ipv6_pseudo_hdr *ph;
    struct ip6_hdr *ip6;
    struct icmpv6_hdr *icmpv6;
    size_t hdr_len, opts_len, payload_len, total_length;
    uint16_t cksum;
    void *field;

    icmpv6 = dest;

    if ( (field = _pb_pdata(pdu, "cksum") ) )
	/*
	 * If we have a numspec for the checksum it means that the user
	 * wants to force a checksum.
	 */
	cksum = htons(num_next(field) );
    else {
	/*
	 * If we don't have a numspec for the checksum then we calculate
	 * the correct checksum.
	 */

	ip6 = prev_pdu_hdr;

	hdr_len = ( (struct node_data *) pdu->data)->_data_pdu.hdr_len;
	opts_len = ( (struct node_data *) pdu->data)->_data_pdu.opts_len;
	payload_len = ( (struct node_data *) pdu->data)->_data_pdu.payload_len;

	total_length = hdr_len + payload_len + opts_len; 

	ph = xmalloc(sizeof(struct ipv6_pseudo_hdr) + total_length);

	ph->saddr = ip6->ip6_src;
	ph->daddr = ip6->ip6_dst;
	ph->len = htonl(total_length);
	ph->zero1 = 0;
	ph->zero2 = 0;
	ph->nxt = IPPROTO_ICMPV6;

	/* Put the IPv6 Pseudo Header and the ICMPv6 PDU (dest) together */
	memcpy( (unsigned char *) ph + sizeof(struct ipv6_pseudo_hdr),
	       dest, total_length);

	/* Must be 0 before calculating cksum! */
	SSVAL(icmpv6, offsetof(struct icmpv6_hdr, sum), 0);

	/* Do a checksum on the new pseudo_hdr + ICMPv6 message */
	cksum = _pb_cksum(ph, sizeof(struct ipv6_pseudo_hdr) + total_length);

	free(ph);
    }

    SSVAL(icmpv6, offsetof(struct icmpv6_hdr, sum), cksum);
}

/*********************************************************************/
/************************* Raw ICMPv6 message ************************/
/*********************************************************************/

static void
pdu_icmpv6rawhdr_builder(const GNode *pdu, void *dest)
{
    struct icmpv6_hdr *icmpv6;

    icmpv6 = dest;

    icmpv6->type = num_next(_pb_pdata(pdu, "type") );
    icmpv6->code = num_next(_pb_pdata(pdu, "code") );
}

#if 0
static void
pdu_icmpv6rawhdr_dumper(pdu_t *p, const char *prefix)
{
    struct icmpv6hdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Type: %s\n", prefix, num_info(hdr_data->_raw.type) );
    printf("%s    Code: %s\n", prefix, num_info(hdr_data->_raw.code) );
    printf("%s    Checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

/*********************************************************************/
/************* ICMPv6 "destination unreachable" message **************/
/*********************************************************************/

static void
pdu_icmpv6unreachablehdr_builder(const GNode *pdu, void *dest)
{
    struct icmpv6unreachable_hdr *icmpv6;

    icmpv6 = dest;

    icmpv6->type = ICMPV6_DEST_UNREACH;
    icmpv6->code = num_next(_pb_pdata(pdu, "code") );
    icmpv6->unused = 0;
}

#if 0
static void
pdu_icmpv6unreachablehdr_dumper(pdu_t *p, const char *prefix)
{
    struct icmpv6hdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Code: %s\n", prefix, num_info(hdr_data->_unreachable.code) );
    printf("%s    Checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

/*********************************************************************/
/****************** ICMPv6 "packet too big" message ******************/
/*********************************************************************/

static void
pdu_icmpv6toobighdr_builder(const GNode *pdu, void *dest)
{
    struct icmpv6toobig_hdr *icmpv6;
    uint32_t mtu;

    icmpv6 = dest;

    icmpv6->type = ICMPV6_TOOBIG;
    icmpv6->code = num_next(_pb_pdata(pdu, "code") );

    mtu = htonl(num_next(_pb_pdata(pdu, "mtu") ) );
    SIVAL(icmpv6, offsetof(struct icmpv6toobig_hdr, mtu), mtu);
}

#if 0
static void
pdu_icmpv6toobighdr_dumper(pdu_t *p, const char *prefix)
{
    struct icmpv6hdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Code: %s\n", prefix, num_info(hdr_data->_toobig.code) );
    printf("%s    MTU: %s\n", prefix, num_info(hdr_data->_toobig.mtu) );
    printf("%s    Checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

/*********************************************************************/
/******************* ICMPv6 "time exceeded" message ******************/
/*********************************************************************/

static void
pdu_icmpv6timeexceedhdr_builder(const GNode *pdu, void *dest)
{
    struct icmpv6timeexceed_hdr *icmpv6;

    icmpv6 = dest;

    icmpv6->type = ICMPV6_TIME_EXCEEDED;
    icmpv6->code = num_next(_pb_pdata(pdu, "code") );
}

#if 0
static void
pdu_icmpv6timeexceedhdr_dumper(pdu_t *p, const char *prefix)
{
    struct icmpv6hdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Code: %s\n", prefix, num_info(hdr_data->_timeexceed.code) );
    printf("%s    Checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

/*********************************************************************/
/***************** ICMPv6 "parameter problem" message ****************/
/*********************************************************************/

static void
pdu_icmpv6parmproblemhdr_builder(const GNode *pdu, void *dest)
{
    struct icmpv6parmproblem_hdr *icmpv6;
    uint32_t pointer;

    icmpv6 = dest;

    icmpv6->type = ICMPV6_PARAMETERPROB;
    icmpv6->code = num_next(_pb_pdata(pdu, "code") );

    pointer = htonl(num_next(_pb_pdata(pdu, "pointer") ) );
    SIVAL(icmpv6, offsetof(struct icmpv6parmproblem_hdr, pointer), pointer);
}

#if 0
static void
pdu_icmpv6parmproblemhdr_dumper(pdu_t *p, const char *prefix)
{
    struct icmpv6hdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Code: %s\n", prefix, num_info(hdr_data->_parmproblem.code) );
    printf("%s    Pointer: %s\n", prefix,
	   num_info(hdr_data->_parmproblem.pointer) );
    printf("%s    Checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

/*********************************************************************/
/************************* ICMPV6 echo message ***********************/
/*********************************************************************/

static void
pdu_icmpv6echo_builder(const GNode *pdu, void *dest)
{
    struct icmpv6echo_hdr *icmpv6;
    uint16_t u16;
    void *field;

    icmpv6 = dest;

    icmpv6->type = ICMPV6_ECHO;
    icmpv6->code = 0;

    u16 = htons( (field = _pb_pdata(pdu, "id") )
		? num_next(field) : (uint32_t) getpid() );
    SSVAL(icmpv6, offsetof(struct icmpv6echo_hdr, id), u16);

    u16 = htons(num_next(_pb_pdata(pdu, "seq") ) );
    SSVAL(icmpv6, offsetof(struct icmpv6echo_hdr, seq), u16);
}

#if 0
static void
pdu_icmpv6echohdr_dumper(pdu_t *p, const char *prefix)
{
    struct icmpv6hdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    ID: %s\n", prefix, num_info(hdr_data->_echo.id) );
    printf("%s    Sequence: %s\n", prefix, num_info(hdr_data->_echo.seq) );
    printf("%s    Checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

/*********************************************************************/
/********************** ICMPv6 echo reply message ********************/
/*********************************************************************/

static void
pdu_icmpv6echoreply_builder(const GNode *pdu, void *dest)
{
    struct icmpv6echo_hdr *icmpv6;
    uint16_t u16;

    icmpv6 = dest;

    icmpv6->type = ICMPV6_ECHOREPLY;
    icmpv6->code = 0;

    u16 = htons(num_next(_pb_pdata(pdu, "id") ) );
    SSVAL(icmpv6, offsetof(struct icmpv6echo_hdr, id), u16);

    u16 = htons(num_next(_pb_pdata(pdu, "seq") ) );
    SSVAL(icmpv6, offsetof(struct icmpv6echo_hdr, seq), u16);
}

/*********************************************************************/
/************************* ICMPv6 redirect message *******************/
/*********************************************************************/

static void
pdu_icmpv6redirect_builder(const GNode *pdu, void *dest)
{
    struct icmpv6redirect_hdr *icmpv6;

    icmpv6 = dest;

    icmpv6->type = ICMPV6_REDIRECT;
    icmpv6->code = 0;

    /* Per RFC 2461 */
    SIVAL(icmpv6, offsetof(struct icmpv6redirect_hdr, reserved), 0);

    icmpv6->target = *(ip6_addr_t *) _pb_pdata(pdu, "target");
    icmpv6->dest = *(ip6_addr_t *) _pb_pdata(pdu, "dst");
}

#if 0
static void
pdu_icmpv6redirecthdr_dumper(pdu_t *p, const char *prefix)
{
    struct icmpv6hdr_options *hdr_data;
    char addr[INET6_ADDRSTRLEN];

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Target: %s\n", prefix,
	   inet_ntop(AF_INET6, &hdr_data->_redirect.target, addr,
		     INET6_ADDRSTRLEN) );
    printf("%s    Destination: %s\n", prefix,
	   inet_ntop(AF_INET6, &hdr_data->_redirect.dest, addr,
		     INET6_ADDRSTRLEN) );
    printf("%s    Checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

/*********************************************************************/
/*************** ICMPv6 Neighbor Solicitation  message ***************/
/*********************************************************************/

static void
pdu_nd_neighbor_solicit_builder(const GNode *pdu, void *dest)
{
    struct nd_neighbor_solicit_hdr *icmpv6;

    icmpv6 = dest;

    icmpv6->type = ND_NEIGHBOR_SOLICIT;
    icmpv6->code = 0;
    icmpv6->reserved = 0; /* Per RFC 2461 */
    icmpv6->target = *(ip6_addr_t *) _pb_pdata(pdu, "target");
}

#if 0
static void
pdu_nd_neighbor_solicithdr_dumper(pdu_t *p, const char *prefix)
{
    struct icmpv6hdr_options *hdr_data;
    char addr[INET6_ADDRSTRLEN];

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Target: %s\n", prefix,
	   inet_ntop(AF_INET6, &hdr_data->_nd_neighbor_solicit.target, addr,
		     INET6_ADDRSTRLEN) );
    printf("%s    Checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

/*********************************************************************/
/*************** ICMPv6 Neighbor Advertisement message ***************/
/*********************************************************************/

static void
pdu_nd_neighbor_advert_builder(const GNode *pdu, void *dest)
{
    struct nd_neighbor_advert_hdr *icmpv6;

    icmpv6 = dest;

    icmpv6->type = ND_NEIGHBOR_ADVERT;
    icmpv6->code = 0;
    icmpv6->r = (unsigned) _pb_pdata(pdu, "r");
    icmpv6->s = (unsigned) _pb_pdata(pdu, "s");
    icmpv6->o = (unsigned) _pb_pdata(pdu, "o");
    icmpv6->reserved1 = 0; /* Per RFC 2461 */
    icmpv6->reserved2 = 0; /* Per RFC 2461 */
    icmpv6->target = *(ip6_addr_t *) _pb_pdata(pdu, "target");
}

#if 0
static void
pdu_nd_neighbor_adverthdr_dumper(pdu_t *p, const char *prefix)
{
    struct icmpv6hdr_options *hdr_data;
    char addr[INET6_ADDRSTRLEN];

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Target: %s\n", prefix,
	   inet_ntop(AF_INET6, &hdr_data->_nd_neighbor_advert.target, addr,
		     INET6_ADDRSTRLEN) );
    printf("%s    Router flag: %s\n", prefix,
	   hdr_data->_nd_neighbor_advert.r ? "set" : "clear");
    printf("%s    Solicited flag: %s\n", prefix,
	   hdr_data->_nd_neighbor_advert.s ? "set" : "clear");
    printf("%s    Override flag: %s\n", prefix,
	   hdr_data->_nd_neighbor_advert.o ? "set" : "clear");
    printf("%s    Checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

/*********************************************************************/
/****************** ICMPv6 Router Solicit message ********************/
/*********************************************************************/

static void
pdu_nd_router_solicit_builder(const GNode *pdu _U_, void *dest)
{
    struct nd_router_solicit_hdr *icmpv6;

    icmpv6 = dest;

    icmpv6->type = ND_ROUTER_SOLICIT;
    icmpv6->code = 0;
}

#if 0
static void
pdu_nd_router_solicithdr_dumper(pdu_t *p, const char *prefix)
{
    struct icmpv6hdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

/*********************************************************************/
/*************** ICMPv6 Router Advertisement message *****************/
/*********************************************************************/

static void
pdu_nd_router_advert_builder(const GNode *pdu, void *dest)
{
    struct nd_router_advert_hdr *icmpv6;

    icmpv6 = dest;

    icmpv6->type = ND_ROUTER_ADVERT;
    icmpv6->code = 0;
    icmpv6->cur_hop_limit = num_next(_pb_pdata(pdu, "hop-limit") );
    icmpv6->m = (unsigned) _pb_pdata(pdu, "m");
    icmpv6->o = (unsigned) _pb_pdata(pdu, "o");
    icmpv6->h = (unsigned) _pb_pdata(pdu, "h");
    icmpv6->prf = num_next(_pb_pdata(pdu, "prf") );
    icmpv6->reserved = 0; /* Per RFC 2461, 3775, router selection draft */

    SSVAL(icmpv6, offsetof(struct nd_router_advert_hdr, router_lifetime),
	  num_next(_pb_pdata(pdu, "lifetime") ) );

    SIVAL(icmpv6, offsetof(struct nd_router_advert_hdr, reachable_time), 
	  num_next(_pb_pdata(pdu, "reachable-time") ) );

    SIVAL(icmpv6, offsetof(struct nd_router_advert_hdr, retrans_timer),
	  num_next(_pb_pdata(pdu, "retrans-time") ) );
}

#if 0
static void
pdu_nd_router_adverthdr_dumper(pdu_t *p, const char *prefix)
{
    struct icmpv6hdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
    printf("%s    Current Hop Limit: %u\n", prefix,
	   hdr_data->_nd_router_advert.cur_hop_limit);
    printf("%s    \"Managed address config.\" flag: %s\n", prefix,
	   hdr_data->_nd_router_advert.m ? "set" : "clear");
    printf("%s    \"Other stateful config.\" flag: %s\n", prefix,
	   hdr_data->_nd_router_advert.o ? "set" : "clear");
    printf("%s    Router Lifetime: %u\n", prefix,
	   hdr_data->_nd_router_advert.router_lifetime);
    printf("%s    Reachable Time: %u\n", prefix,
	   hdr_data->_nd_router_advert.reachable_time);
    printf("%s    Retransmission Timer: %u\n", prefix,
	   hdr_data->_nd_router_advert.retrans_timer);
}
#endif

/*********************************************************************/
/****************** ICMPv6 Router Renumber message *******************/
/*********************************************************************/

static void
pdu_icmpv6_router_renumber_builder(const GNode *pdu, void *dest)
{
    struct icmpv6_router_renumber_hdr *icmpv6;
    uint32_t u32;
    uint16_t u16;

    icmpv6 = dest;

    icmpv6->type = ICMPV6_ROUTER_RENUMBER;
    icmpv6->code = num_next(_pb_pdata(pdu, "code") );

    u32 = htonl(num_next(_pb_pdata(pdu, "seq") ) );
    SIVAL(icmpv6, offsetof(struct icmpv6_router_renumber_hdr, sequence_number),
	  u32);

    icmpv6->segment_number = num_next(_pb_pdata(pdu, "segment") );
    icmpv6->t = (unsigned) _pb_pdata(pdu, "t");
    icmpv6->r = (unsigned) _pb_pdata(pdu, "r");
    icmpv6->a = (unsigned) _pb_pdata(pdu, "a");
    icmpv6->s = (unsigned) _pb_pdata(pdu, "s");
    icmpv6->p = (unsigned) _pb_pdata(pdu, "p");
    icmpv6->flag_reserved = 0; /* Per RFC 2894 */

    u16 = htonl(num_next(_pb_pdata(pdu, "max-delay") ) );
    SSVAL(icmpv6, offsetof(struct icmpv6_router_renumber_hdr, max_delay), u16);

    /* Per RFC 2894 */
    SIVAL(icmpv6, offsetof(struct icmpv6_router_renumber_hdr, reserved), 0);
}

#if 0
static void
pdu_icmpv6_router_renumberhdr_dumper(pdu_t *p, const char *prefix)
{
    struct icmpv6hdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Code: %d\n", prefix, hdr_data->_router_renumber.code);
    printf("%s    Sequence number: %u\n", prefix,
	   hdr_data->_router_renumber.sequence_number);
    printf("%s    Segment number: %u\n", prefix,
	   hdr_data->_router_renumber.segment_number);
    printf("%s    \"Test command\" flag: %s\n", prefix,
	   hdr_data->_router_renumber.t ? "set" : "clear");
    printf("%s    \"Result requested\" flag: %s\n", prefix,
	   hdr_data->_router_renumber.r ? "set" : "clear");
    printf("%s    \"All interfaces\" flag: %s\n", prefix,
	   hdr_data->_router_renumber.a ? "set" : "clear");
    printf("%s    \"Site-specific\" flag: %s\n", prefix,
	   hdr_data->_router_renumber.s ? "set" : "clear");
    printf("%s    \"Processed previously\" flag: %s\n", prefix,
	   hdr_data->_router_renumber.p ? "set" : "clear");
    printf("%s    Maximum delay: %u\n", prefix,
	   hdr_data->_router_renumber.max_delay);
    printf("%s    Checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

/*********************************************************************/
/********************** ICMPv6 TLV PDU Option ************************/
/*********************************************************************/

/* This PDU option can be used by several of the ICMPv6 PDUs */

static void
icmpv6_tlv(const GNode *option, void *dest,
           void *curr_pdu_hdr _U_, void *prev_pdu _U_)
{
    struct payload *data;
    uint8_t *tlvptr;
    void *field;

    tlvptr = dest;

    /* TVL type */
    *tlvptr++ = num_next(_pb_pdata(option, "type") );

    data = _pb_pdata(option, "value");
    if (data) {
	/* TVL length */
	if ( (field = _pb_pdata(option, "length") ) )
	    /* User specified a fake length */
	    *tlvptr++ = num_next(field);
	else
	    /* Use real length -- see RFC 2461 to understand why
	     * (len + 2)/8 */
	    *tlvptr++ = (data->len + 2)/8;

	/* TLV data */
	memcpy(tlvptr, data->data, data->len);

	tlvptr += data->len;
    } else {
	/* TVL length */
	if ( (field = _pb_pdata(option, "length") ) )
	    /* User specified a fake length */
	    *tlvptr++ = num_next(field);
	else
	    *tlvptr++ = 0;
    }
}

static pdu_t pdu_icmpv6raw = {
    .name = "icmp6",
    .description = "Raw ICMPv6 message",
    .documented_in = "RFC 2463",
    .len = sizeof(struct icmpv6_hdr),
    .fields = (field_t []) {
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "type", .type = PDU_FTYPE_NUMTYPE},
	{.name = "code", .type = PDU_FTYPE_NUMTYPE},
	{.name = NULL}
    },
    .build = &pdu_icmpv6rawhdr_builder,
    .postbuild = &pdu_icmpv6_generic_pass2_builder,
};

static pdu_t pdu_icmpv6unreachable = {
    .name = "icmp6-unreachable",
    .description = "ICMPv6 destination unreachable",
    .documented_in = "RFC 2463",
    .len = sizeof(struct icmpv6unreachable_hdr),
    .fields = (field_t []) {
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "code", .type = PDU_FTYPE_NUMTYPE},
	{.name = NULL}
    },
    .build = &pdu_icmpv6unreachablehdr_builder,
    .postbuild = &pdu_icmpv6_generic_pass2_builder,
};

static pdu_t pdu_icmpv6toobig = {
    .name = "icmp6-toobig",
    .description = "ICMPv6 packet too big",
    .documented_in = "RFC 2463",
    .len = sizeof(struct icmpv6toobig_hdr),
    .fields = (field_t []) {
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "code", .type = PDU_FTYPE_NUMTYPE},
	{.name = "mtu", .type = PDU_FTYPE_NUMTYPE},
	{.name = NULL}
    },
    .build = &pdu_icmpv6toobighdr_builder,
    .postbuild = &pdu_icmpv6_generic_pass2_builder,
};

static pdu_t pdu_icmpv6timeexceed = {
    .name = "icmp6-timeexceed",
    .description = "ICMPv6 time exceeded",
    .documented_in = "RFC 2463",
    .len = sizeof(struct icmpv6timeexceed_hdr),
    .fields = (field_t []) {
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "code", .type = PDU_FTYPE_NUMTYPE},
	{.name = NULL}
    },
    .build = &pdu_icmpv6timeexceedhdr_builder,
    .postbuild = &pdu_icmpv6_generic_pass2_builder,
};

static pdu_t pdu_icmpv6parmproblem = {
    .name = "icmp6-parmproblem",
    .description = "ICMPv6 parameter problem",
    .documented_in = "RFC 2463",
    .len = sizeof(struct icmpv6parmproblem_hdr),
    .fields = (field_t []) {
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "code", .type = PDU_FTYPE_NUMTYPE},
	{.name = "pointer", .type = PDU_FTYPE_NUMTYPE},
	{.name = NULL}
    },
    .build = &pdu_icmpv6parmproblemhdr_builder,
    .postbuild = &pdu_icmpv6_generic_pass2_builder,
};

static pdu_t pdu_icmpv6echo = {
    .name = "icmp6-echo",
    .description = "ICMPv6 echo request",
    .documented_in = "RFC 2463",
    .len = sizeof(struct icmpv6echo_hdr),
    .fields = (field_t []) {
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "id", .type = PDU_FTYPE_NUMTYPE},
	{.name = "seq", .type = PDU_FTYPE_NUMTYPE},
	{.name = NULL}
    },
    .build = &pdu_icmpv6echo_builder,
    .postbuild = &pdu_icmpv6_generic_pass2_builder,
};

static pdu_t pdu_icmpv6echoreply = {
    .name = "icmp6-echoreply",
    .description = "ICMPv6 echo reply",
    .documented_in = "RFC 2463",
    .len = sizeof(struct icmpv6echo_hdr),
    .fields = (field_t []) {
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "id", .type = PDU_FTYPE_NUMTYPE},
	{.name = "seq", .type = PDU_FTYPE_NUMTYPE},
	{.name = NULL}
    },
    .build = &pdu_icmpv6echoreply_builder,
    .postbuild = &pdu_icmpv6_generic_pass2_builder,
};

static pdu_t pdu_icmpv6redirect = {
    .name = "icmp6-redirect",
    .description = "ICMPv6 redirect",
    .documented_in = "RFC 2461",
    .len = sizeof(struct icmpv6redirect_hdr),
    .fields = (field_t []) {
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "target", .type = PDU_FTYPE_IP6ADDR},
	{.name = "dst", .type = PDU_FTYPE_IP6ADDR},
	{.name = "options", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
	    {.name = "tlv", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
		{.name = "type", .type = PDU_FTYPE_NUMTYPE},
		{.name = "length", .type = PDU_FTYPE_NUMTYPE},
		{.name = "value", .type = PDU_FTYPE_DATA},
		{.name = NULL} }, .defval = NULL},
	    {.name = NULL} }, .defval = NULL},
	{.name = NULL}
    },
    .options_class = "icmp6",
    .build = &pdu_icmpv6redirect_builder,
    .postbuild = &pdu_icmpv6_generic_pass2_builder,
};

static pdu_t pdu_nd_neighbor_solicit = {
    .name = "icmp6-ns",
    .description = "ICMPv6 Neighbor Solicitation",
    .documented_in = "RFC 2461",
    .len = sizeof(struct nd_neighbor_solicit_hdr),
    .fields = (field_t []) {
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "target", .type = PDU_FTYPE_IP6ADDR},
	{.name = "options", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
	    {.name = "tlv", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
		{.name = "type", .type = PDU_FTYPE_NUMTYPE},
		{.name = "length", .type = PDU_FTYPE_NUMTYPE},
		{.name = "value", .type = PDU_FTYPE_DATA},
		{.name = NULL} }, .defval = NULL},
	    {.name = NULL} }, .defval = NULL},
	{.name = NULL}
    },
    .options_class = "icmp6",
    .build = &pdu_nd_neighbor_solicit_builder,
    .postbuild = &pdu_icmpv6_generic_pass2_builder,
};

static pdu_t pdu_nd_neighbor_advert = {
    .name = "icmp6-na",
    .description = "ICMPv6 Neighbor Advertisement",
    .documented_in = "RFC 2461",
    .len = sizeof(struct nd_neighbor_advert_hdr),
    .fields = (field_t []) {
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "r", .type = PDU_FTYPE_BIT},
	{.name = "s", .type = PDU_FTYPE_BIT},
	{.name = "o", .type = PDU_FTYPE_BIT},
	{.name = "target", .type = PDU_FTYPE_IP6ADDR},
	{.name = "options", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
	    {.name = "tlv", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
		{.name = "type", .type = PDU_FTYPE_NUMTYPE},
		{.name = "length", .type = PDU_FTYPE_NUMTYPE},
		{.name = "value", .type = PDU_FTYPE_DATA},
		{.name = NULL} }, .defval = NULL},
	    {.name = NULL} }, .defval = NULL},
	{.name = NULL}
    },
    .options_class = "icmp6",
    .build = &pdu_nd_neighbor_advert_builder,
    .postbuild = &pdu_icmpv6_generic_pass2_builder,
};

static pdu_t pdu_nd_router_solicit = {
    .name = "icmp6-rs",
    .description = "ICMPv6 Router Solicit",
    .documented_in = "RFC 2461",
    .len = sizeof(struct nd_router_solicit_hdr),
    .fields = (field_t []) {
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "options", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
	    {.name = "tlv", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
		{.name = "type", .type = PDU_FTYPE_NUMTYPE},
		{.name = "length", .type = PDU_FTYPE_NUMTYPE},
		{.name = "value", .type = PDU_FTYPE_DATA},
		{.name = NULL} }, .defval = NULL},
	    {.name = NULL} }, .defval = NULL},
	{.name = NULL}
    },
    .options_class = "icmp6",
    .build = &pdu_nd_router_solicit_builder,
    .postbuild = &pdu_icmpv6_generic_pass2_builder,
};

static pdu_t pdu_nd_router_advert = {
    .name = "icmp6-ra",
    .description = "ICMPv6 Router Advertisement",
    .documented_in = "RFC 2461",
    .len = sizeof(struct nd_router_advert_hdr),
    .fields = (field_t []) {
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "hop-limit", .type = PDU_FTYPE_NUMTYPE},
	{.name = "m", .type = PDU_FTYPE_BIT},
	{.name = "o", .type = PDU_FTYPE_BIT},
	{.name = "h", .type = PDU_FTYPE_BIT},
	{.name = "prf", .type = PDU_FTYPE_NUMTYPE},
	{.name = "lifetime", .type = PDU_FTYPE_NUMTYPE},
	{.name = "reachable-time", .type = PDU_FTYPE_NUMTYPE},
	{.name = "retrans-time", .type = PDU_FTYPE_NUMTYPE},
	{.name = "options", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
	    {.name = "tlv", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
		{.name = "type", .type = PDU_FTYPE_NUMTYPE},
		{.name = "length", .type = PDU_FTYPE_NUMTYPE},
		{.name = "value", .type = PDU_FTYPE_DATA},
		{.name = NULL} }, .defval = NULL},
	    {.name = NULL} }, .defval = NULL},
	{.name = NULL}
    },
    .options_class = "icmp6",
    .build = &pdu_nd_router_advert_builder,
    .postbuild = &pdu_icmpv6_generic_pass2_builder,
};

static pdu_t pdu_icmpv6_router_renumber = {
    .name = "icmp6-rn",
    .description = "ICMPv6 Router Renumbering",
    .documented_in = "RFC 2894",
    .len = sizeof(struct icmpv6_router_renumber_hdr),
    .fields = (field_t []) {
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "code", .type = PDU_FTYPE_NUMTYPE},
	{.name = "seq", .type = PDU_FTYPE_NUMTYPE},
	{.name = "segment", .type = PDU_FTYPE_NUMTYPE},
	{.name = "t", .type = PDU_FTYPE_BIT},
	{.name = "r", .type = PDU_FTYPE_BIT},
	{.name = "a", .type = PDU_FTYPE_BIT},
	{.name = "s", .type = PDU_FTYPE_BIT},
	{.name = "p", .type = PDU_FTYPE_BIT},
	{.name = "max-delay", .type = PDU_FTYPE_NUMTYPE},
	{.name = "options", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
	    {.name = "tlv", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
		{.name = "type", .type = PDU_FTYPE_NUMTYPE},
		{.name = "length", .type = PDU_FTYPE_NUMTYPE},
		{.name = "value", .type = PDU_FTYPE_DATA},
		{.data = NULL} }, .defval = NULL},
	    {.name = NULL} }, .defval = NULL},
	{.name = NULL}
    },
    .options_class = "icmp6",
    .build = &pdu_icmpv6_router_renumber_builder,
    .postbuild = &pdu_icmpv6_generic_pass2_builder,
};

static const pdu_option_t pdu_option_icmpv6tlv = {
    .name = "tlv",
    .description = "ICMPv6 type-length-value (TLV) options",
    .build = &icmpv6_tlv,
    .get_len = &pdu_generic_tlv_len
};

void
_pb_register_icmp6(void)
{
    _pb_register_protocol(&pdu_icmpv6raw);
    _pb_register_protocol(&pdu_icmpv6unreachable);
    _pb_register_protocol(&pdu_icmpv6toobig);
    _pb_register_protocol(&pdu_icmpv6timeexceed);
    _pb_register_protocol(&pdu_icmpv6parmproblem);
    _pb_register_protocol(&pdu_icmpv6echo);
    _pb_register_protocol(&pdu_icmpv6echoreply);
    _pb_register_protocol(&pdu_icmpv6redirect);
    _pb_register_protocol(&pdu_nd_neighbor_solicit);
    _pb_register_protocol(&pdu_nd_neighbor_advert);
    _pb_register_protocol(&pdu_nd_router_solicit);
    _pb_register_protocol(&pdu_nd_router_advert);
    _pb_register_protocol(&pdu_icmpv6_router_renumber);

    _pb_register_protocol_option("icmp6", &pdu_option_icmpv6tlv);
}
