(* $Id: http_client_aux.ml,v 1.1 1999/07/08 03:00:47 gerd Exp $
 * ----------------------------------------------------------------------
 * Auxiliary functions: this version uses the xstr package and is
 * multithreading-safe.
 *)

open Xstr_match
open Xstr_split

let digits = mkset "0-9";;
let hex_digits = mkset "0-9a-fA-F";;
let space_tab = mkset " \t";;
let no_slash_no_colon = mknegset "/:";;
let no_slash_no_colon_no_at = mknegset "/:@";;
let no_space = mknegset " ";;
let no_space_no_tab = mknegset " \t";;
let no_cr_no_lf = mknegset "\n\r";;


let ms which re s =
  try
    match_string re s
  with
    Failure e as f ->
      prerr_endline ("Failure in " ^ which);
      raise f
;;


let match_query query =
  let host = var "" in
  let port = var "80" in
  let path = var "" in
  let query_re =
    [ Literal "http://";
      Record (host,
	      [ Anychar_from no_slash_no_colon;
		Anystring_from no_slash_no_colon;
	      ]);
      Optional
	[ Literal ":";
	  Record (port,
		  [ Anychar_from digits;
		    Anystring_from digits;
		  ]);
	];
      Record (path,
	      [ Anystring_from no_space; ]);
    ] in
  if match_string (* ms "query" *) query_re query then 
    String.lowercase(found_string_of_var host),
    int_of_string(string_of_var port),
    found_string_of_var path
  else
    raise Not_found
;;


let match_status status_line =
  let version = var "" in
  let code = var "" in
  let message = var "" in
  let status_re =
    [ Record (version,
	      [ Anychar_from no_space_no_tab;
		Anystring_from no_space_no_tab;
	      ]);
      Anychar_from space_tab;
      Anystring_from space_tab;
      Record (code,
	      [ Anychar_from digits;
		Anychar_from digits;
		Anychar_from digits;
	      ]);
      Anychar_from space_tab;
      Anystring_from space_tab;
      Record (message,
	      [ Anystring_from no_cr_no_lf;
	      ]);
      Optional 
	[ Literal "\r"; ];
    ] in
  if match_string (* ms "status" *) status_re status_line then 
    found_string_of_var version,
    int_of_string(found_string_of_var code),
    found_string_of_var message
  else
    raise Not_found
;;
  

let match_header_line line =
  let name = var "" in
  let value = var "" in
  let line_re =
    [ Record(name,
	     [ Anychar_from no_space_no_tab;
	       Anystring_from no_space_no_tab;
	     ]);
      Anystring_from space_tab;
      Literal ":";
      Anystring_from space_tab;
      Record(value,
	     [ Anystring; ]);
    ] in
  if match_string (* ms "headerline" *) line_re line then
    String.lowercase(found_string_of_var name),
    found_string_of_var value
  else
    raise Not_found
;;


let match_hex hex =
  let hexnum = var "" in
  let hex_re =
    [ Record (hexnum,
	      [ Anychar_from hex_digits;
		Anystring_from hex_digits;
	      ]);
      Anystring;
    ] in
  if match_string (* ms "hex" *) hex_re hex then
    found_string_of_var hexnum
  else
    raise Not_found
;;


let match_ip ip =
  let ipnum = var "" in
  let ip_re =
    [ Record (ipnum,
	      [ Anychar_from digits;
		Anystring_from digits;
		Literal ".";
		Anychar_from digits;
		Anystring_from digits;
		Literal ".";
		Anychar_from digits;
		Anystring_from digits;
		Literal ".";
		Anychar_from digits;
		Anystring_from digits;
	      ]);
    ] in
  if match_string (* ms "ip" *) ip_re ip then
    ip
  else
    raise Not_found
;;


let match_http url =
  let user = var "" in
  let password = var "" in
  let host = var "" in
  let port = var "" in
  let path = var "" in
  let http_re =
    [ Literal "http://";
      Optional
	[ Record (user,
		  [ Anychar_from no_slash_no_colon_no_at;
		    Anystring_from no_slash_no_colon_no_at;
		  ]);
	  Optional
	    [ Literal ":";
	      Record (password,
		      [ Anychar_from no_slash_no_colon_no_at;
			Anystring_from no_slash_no_colon_no_at;
		      ]);
	    ];
	  Literal "@";
	];
      Record (host,
	      [ Anychar_from no_slash_no_colon_no_at;
		Anystring_from no_slash_no_colon_no_at;
	      ]);
      Optional
	[ Literal ":";
	  Record (port,
		  [ Anychar_from digits;
		    Anystring_from digits;
		  ]);
	];
      Optional
	[ Record (path,
		  [ Literal "/";
		    Anystring;
		  ]);
	];
    ] in
  if match_string (* ms "http" *) http_re url then begin
    (try Some(found_string_of_var user) with Not_found -> None),
    (try Some(found_string_of_var password) with Not_found -> None),
    found_string_of_var host,
    (try int_of_string(found_string_of_var port) with Not_found -> 80),
    (try found_string_of_var path with Not_found -> "")
  end
  else raise Not_found
;;
	  
		
let split_words_by_commas =
  split_string " \t\r\n" true true [ "," ];;


(* Note that ThreadUnix has a different interface (sic!) if the POSIX
 * thread implementation is used instead of the bytecode-only implementation.
 * The following construction works independent of this:
 *)

module Unx = 
  struct
    open Unix
    open ThreadUnix
    let execv = execv
    let execve = execve
    let execvp = execvp
    let wait = wait
    let waitpid = waitpid
    let system = system
    let read = read
    let write = write
    let select = select
    let pipe = pipe
    let open_process_in = open_process_in
    let open_process_out = open_process_out
    let open_process = open_process
    let sleep = sleep
    let socket = socket
    let socketpair = socketpair
    let accept = accept
    let connect = connect
    let recv = recv
    let recvfrom = recvfrom
    let send = send
    let sendto = sendto
    let open_connection = open_connection
    let establish_server = establish_server
  end
;;

module Mtx = Mutex;;  

(* ======================================================================
 * History:
 * 
 * $Log: http_client_aux.ml,v $
 * Revision 1.1  1999/07/08 03:00:47  gerd
 * 	Initial revision.
 *
 * 
 *)
