(* $Id: http_client.ml,v 1.5.2.2 2002/07/07 11:52:09 gerd Exp $
 * ----------------------------------------------------------------------
 *
 *)

(* Reference documents:
 * RFC 2068:      HTTP 1.1
 * RFC 2069:      Digest Authentication
 *)


(* TODO:
 * - Use extensible string buffers
 * - follow redirections of GET/HEAD for 301/302 responses
 * - Minor problem: if there is no user/password given for a realm, the client
 *   initiates a second request although this request is senseless.
 *)

open Unix

(* In an MT environment this overrides some of the Unix functions: *)
open Http_client_aux.Unx

open Netencoding

let listrm = List.remove_assoc;;
  (* This is Ocaml-2.02. In 2.01, use List.remove *)

exception Header_is_incomplete;;
exception Body_is_incomplete;;
exception Body_maybe_complete;;
exception Http_error of (int * string);;
exception Async_close;;         (* Connection closed between two requests *)
exception Broken_connection;;   (* Connection closed during request *)

type auth_method =
    Auth_basic of string                                (* realm *)
  | Auth_digest of (string * string * string * string)  (* realm, nonce,
							 * opaque, stale *)

type token =
    Word of string
  | Special of char             (* tspecial or CTL, but not double quotes *)
;;


type secret = unit


let rcv_increment = 65536;;
let timeout = 600.0;;

let better_unix_error f arg =
  try
    f arg
  with
    Unix_error (e,syscall,param) ->
      let error = error_message e in
      if param = "" then
	failwith error
      else
	failwith (param ^ ": " ^ error)


(* Parsing suitable for most header lines. See RFC 2068. *)

let parse_header_value v =
  let is_control c = let code = Char.code (c) in code < 32 or code = 127 in
  let is_special c = (c = '('  or c = ')' or c = '<' or c = '>' or
		      c = '@'  or c = ',' or c = ';' or c = ':' or
		      c = '\\' or c = '"' or c = '/' or c = '[' or
		      c = ']'  or c = '?' or c = '{' or c = '}' or
		      c = '\t' or c = ' ' or c = '=') in
  let l = String.length v in

  let rec parse_quoted k word =
    if k < l then begin
      let c = v.[k] in
      if c='"' then
	k, word
      else
	if c='\\' then begin
	  if k < l-1 then
	    parse_quoted (k+2) (word ^ String.make 1 (v.[k+1]))
	  else
	    failwith "http_client: cannot parse header line"
	end
	else
	  parse_quoted (k+1) (word ^ String.make 1 c)
    end
    else
      failwith "http_client: cannot parse header line"
  in

  let rec parse k word =
    if k < l then begin
      let c = v.[k] in
      if c <> '"' & (is_control c or is_special c) then begin
	if (c = ' ' or c = '\t') then
	  (if word <> "" then [Word word] else []) @
	  parse (k+1) ""
	else
	  (if word <> "" then [Word word] else []) @
	  [Special c] @
	  parse (k+1) ""
      end
      else
	if c = '"' then begin
	  let k', word' = parse_quoted (k+1) "" in
	    (if word <> "" then [Word word] else []) @
	    [Word word'] @
	    parse (k'+1) ""
	end
	else
	  parse (k+1) (word ^ String.make 1 c)
    end
    else []

  in
  let p = parse 0 "" in
  (*
    List.iter
    (function Word s -> prerr_endline ("Word " ^ s) |
              Special c -> prerr_endline ("Special " ^ String.make 1 c))
    p;
   *)
  p
;;


let hex_digits = [| '0'; '1'; '2'; '3'; '4'; '5'; '6'; '7';
		    '8'; '9'; 'a'; 'b'; 'c'; 'd'; 'e'; 'f' |];;

let encode_hex s =
  (* encode with lowercase hex digits *)
  let l = String.length s in
  let t = String.make (2*l) ' ' in
  for n = 0 to l - 1 do
    let x = Char.code s.[n] in
    t.[2*n]   <- hex_digits.( x lsr 4 );
    t.[2*n+1] <- hex_digits.( x land 15 );
  done;
  t
;;


type verbose =
    Verbose_status
  | Verbose_request_header
  | Verbose_response_header
  | Verbose_request_contents
  | Verbose_response_contents
;;


class virtual message =
  object (self)
    val mutable host = ""
    val mutable port = 80
    val mutable path = ""
    val mutable request = ""
    val mutable request_uri = ""
    val mutable request_method = ""
    val mutable header = []

    val mutable no_proxy = false

    val mutable body = ""

    val mutable received = ""
    val mutable received_header = []
    val mutable received_entity = 0     (* where entity begins in 'received' *)
    val mutable received_contents = ""  (* received contents (decoded) *)

    val mutable status_code = 0
    val mutable status_text = ""
    val mutable status_version = ""

    val mutable served = false

    method virtual prepare : bool -> unit

    method is_served = served
    method set_served (_:secret) = served <- true
    method set_unserved (_:secret) = served <- false

    method get_host() = host
    method get_port() = port

    method get_req_body() = body
    method get_request (_:secret) = request
    method get_req_header () = header
    method assoc_req_header name =
      List.assoc name header
    method set_req_header name0 value =
      let name = String.lowercase name0 in
      let rec set l =
	match l with
	  [] -> [name, value]
	|	(n,v) :: l' -> if name = n then (name,value)::l' else (n,v) :: set l'
      in
      header <- set header
    method get_req_uri () = request_uri
    method get_req_method () = request_method

    method get_resp_header () = received_header
    method assoc_resp_header name =
      List.assoc name received_header
    method get_resp_body () =
      (* raise Http_error if status is not 2xx *)
      let _,code,_ = self # dest_status() in
      if code >= 200 & code < 300 then
	received_contents
      else
	raise (Http_error(code, received_contents))

    method dump_header prefix h =
      List.iter
	(fun (n,v) ->
	  prerr_endline (prefix ^ n ^ ": " ^ v))
	h


    method no_proxy() =
      (* force not to use a proxy for this request *)
      no_proxy <- true

    method is_proxy_allowed() =
      not no_proxy


    method set_response (_:secret) response =
      received <- response;
      status_code <- 0


    method init_query (_:secret) query =
      try
	let h, pt, ph = Http_client_aux.match_query query in
	host <- h;
	port <- pt;
	if String.length ph >= 1 then
	  if ph.[0] <> '/' then
	    failwith "http_client: bad URL";
	path <- ph
      with
	Not_found ->
	  failwith "http_client: bad URL"


    method private set_request use_proxy use_asterisk req =
      self # set_req_header "host" (host ^ ":" ^ string_of_int port);
      request_method <- req;
      if use_proxy then
        (* absolute URI *)
	let query = "http://" ^ host ^ ":" ^ string_of_int port ^ path in
        request_uri <- query
      else begin
        request_uri <- if path = "" then
                          (if use_asterisk then "*" else "/")
		       else path
      end;
      request <- req ^ " " ^ request_uri ^ " HTTP/1.1";

    method dest_status () =
      (* analyzes "status" line in 'received' *)

      let rec search last_version last_code last_msg pos =
        (* Ignore 100 responses in front of other status lines *)
	try
	  let nl = String.index_from received  pos '\n' in

	  let status_line = String.sub received pos (nl-pos) in
	  let version, code, msg = Http_client_aux.match_status status_line in
	  if code = 100 then
	    search version code msg (nl+1)
	  else
	    version,code,msg
	      
	with
	  Not_found ->
	    if last_code <> -1 then
	      (last_version, last_code, last_msg)
	    else
	      failwith "http_client: bad status line in reply"

      in

      if status_code <> 0 then
        (status_version, status_code, status_text)
      else begin
        let version, code, text = search "" (-1) "" 0 in
	status_version <- version;
	status_code <- code;
	status_text <- text;
	(status_version, status_code, status_text)
      end


    method decode_header (_:secret) =
      (* Decode header in 'received' and store it in 'received_header'. *)
      (* Set also 'received_entity'. *)
      (* TODO: indicate error if bad header has been arrived *)

      try
        let header, length = self # decode_header_at () received 0 in
        received_header <- header;
        received_entity <- length
      with
        Header_is_incomplete ->
	  failwith "http_client: header is incomplete"


    method decode_header_at (_:secret) message position =
      (* This method is also used during interpretation of chunked encoding *)
      (* Raises Header_is_incomplete if "end of header" is missing *)

      let length = ref 0 in

      let decode_line line =
	try Http_client_aux.match_header_line line
	with
	  Not_found -> "", ""
      in

      let rec get_header pos prefix =
        (* next line: *)
	let nl =  try String.index_from message pos '\n'
	with Not_found -> raise Header_is_incomplete in
	let line =
	  if nl > pos & message.[ nl - 1 ] = '\r' then
	    String.sub message pos (nl-pos-1)
	  else
	    String.sub message pos (nl-pos)
	in

	if line = "" then begin
	  length := nl+1-position;
	  if prefix <> "" then [ decode_line prefix ] else []
	end
	else begin
	  if List.mem line.[0] [' '; '\t'] then
	    get_header (nl+1) (prefix ^ line)          (* continuation *)
	  else begin
	    let h = if prefix <> "" then [ decode_line prefix ] else [] in
	    h @ get_header (nl+1) line
	  end
	end
      in

      let lines = get_header position "" in

      List.flatten
	(List.map
	   (fun (n,v) -> if n = "" then [] else [n,v])
	   lines),
      !length


    method decode_body
	     (_:secret) verbose_response_header verbose_response_contents =
      (* Assumes that headers are already decoded.
       * Handles transfer encodings, currently only "chunked".
       * Note that this may add further entity headers.
       *)

      let rec chunks pos =
	(* get all chunks beginning at 'pos' *)
	let nl =
	  try
	    String.index_from received pos '\n'
	  with
	    Not_found -> failwith "http_client: cannot decode chunk"
	in
	let length_line = String.sub received pos (nl-pos) in
	let length =
	  try
	    let hexnum = Http_client_aux.match_hex length_line in
	    int_of_string ("0x" ^ hexnum)
	  with
	  | _ -> failwith "http_client: cannot decode chunk"
	in
	if length = 0 then begin
	  (* side-effect: decode any remaining footer *)
	  let footer, _ = self # decode_header_at () received (nl+1) in
	  if verbose_response_header then
	    self # dump_header "Footer " footer;
	  received_header <- received_header @ footer;
	  []
	end else
	  let k = nl + 1 + length in    (* here must be a CRLF *)
	  let nl' =
	    try
	      String.index_from received k '\n'
	    with
	      Not_found -> failwith "http_client: cannot decode chunk"
            | Invalid_argument _ -> failwith "http_client: cannot decode chunk"
	  in
	  String.sub received (nl+1) length :: chunks (nl'+1)
      in

      let transfer_encoding =
	String.lowercase
	  (try List.assoc "transfer-encoding" received_header
	   with Not_found -> "") in

      begin
	match transfer_encoding with
	  "chunked" ->
	    received_contents <- String.concat "" (chunks received_entity)
		(* Note: 'content-length' is ignored (see RFC 2068, 4.4) *)

	|	"" ->
	  (* If there is a "content-length" use this *)
	  (* NOTE: RFC 2068 demands to ignore "content-length" if a body
	   * is not allowed for that response.
	   * This is not yet implemented. TODO
	   *)
	    let length =
	      try
		int_of_string (List.assoc "content-length" received_header)
	      with
		Not_found -> String.length received - received_entity
	      |	_ -> failwith "http_client: bad Content-Length header field"
	    in
	    (* The followging error must be given due to RFC 2068, 4.4: *)
	    if length <> String.length received - received_entity then
	      failwith "http_client: message does not match Content-Length";
	    received_contents <-
	      String.sub received received_entity length
	|	_ ->
	    failwith ("http_client: unknown transfer encoding '" ^
		      transfer_encoding ^ "'")
      end;
      if verbose_response_contents then
	prerr_endline ("Response:\n" ^ received_contents)



    method body_is_complete (_:secret) header message body_pos =
      (* Only checks if the body is valid *)
      (* 'header':   already decoded header *)
      (* 'message':  the full message that has been received *)
      (* 'body_pos': where the body begins in 'message' *)
      (* Raises:
       * - Body_is_incomplete if the body is incomplete because of
       *   incomplete chunks or because of 'content-length'
       * - Body_maybe_complete for unframed messages (HTTP/1.0)
       * Otherwise, returns ()
       *)
      (* Note: This method does not depend on the object's state *)

      let rec chunks pos =
	(* check all chunks beginning at 'pos' *)
	let nl =
	  try
	    String.index_from message pos '\n'
	  with
	    Not_found -> raise Body_is_incomplete
	in
	let length_line = String.sub message pos (nl-pos) in
	let length =
	  try
	    let hexnum = Http_client_aux.match_hex length_line in
	    int_of_string ("0x" ^ hexnum)
	  with
	    _ -> failwith "http_client: cannot decode chunk"
	in
	if length = 0 then begin
	  let footer, footer_length =
	    try self # decode_header_at () message (nl+1)
	    with Header_is_incomplete -> raise Body_is_incomplete
	    (* TODO: it is not checked if there are extra characters
	     * after the footer
	     *)
          in
	  if footer_length < String.length message - nl - 1 then
	    failwith "http_client: response too long";
	  ()
	end else
	  let k = nl + 1 + length in    (* here must be a CRLF *)
	  let nl' =
	    try
	      String.index_from message k '\n'
	    with
	      Not_found -> raise Body_is_incomplete
	    | Invalid_argument _ -> raise Body_is_incomplete
	  in
	  chunks (nl'+1)
      in

      let transfer_encoding =
	String.lowercase
	  (try List.assoc "transfer-encoding" header
	   with Not_found -> "") in

      let beginning = body_pos in                  (* first position of body *)

      begin
	match transfer_encoding with
	  "chunked" ->
	    chunks beginning
	| "" ->
	  (* If there is a "content-length" use this *)
	    let length =
	      try
		int_of_string (List.assoc "content-length" header)
	      with
		Not_found -> raise Body_maybe_complete
	      |	_ -> failwith "http_client: bad Content-Length header field"
	    in
	    if length > String.length message - beginning then
	      raise Body_is_incomplete;
	    if length > String.length message - beginning then
	      failwith "http_client: response is too long"
	| _ ->
	    failwith ("http_client: unknown transfer encoding '" ^
		      transfer_encoding ^ "'")
      end;
      ()
  end
;;


class get the_query =
  object (self)
    inherit message

    val query = the_query

    initializer
      self # init_query () query

    method prepare use_proxy =
      self # set_request use_proxy false "GET"
  end
;;


class trace the_query max_hops =
  object (self)
    inherit message

    val query = the_query
    val hops = max_hops

    initializer
      self # init_query () query

    method prepare use_proxy =
      self # set_request use_proxy false "TRACE";
      self # set_req_header "max-forwards" (string_of_int hops)
  end
;;


class options the_query =
  object (self)
    inherit message

    val query = the_query

    initializer
      self # init_query () query

    method prepare use_proxy =
      self # set_request use_proxy true "OPTIONS"
  end
;;


class head the_query =
  object (self)
    inherit message

    val query = the_query

    initializer
      self # init_query () query

    method prepare use_proxy =
      self # set_request use_proxy false "HEAD"
  end
;;


class post the_query the_params =
  object (self)
    inherit message

    val query = the_query
    val params = the_params

    initializer
      self # init_query () query

    method prepare use_proxy =
      self # set_request use_proxy false "POST";
      self # set_req_header "content-type" "application/x-www-form-urlencoded";
      let s = List.map (fun (n,v) -> n ^ "=" ^ Cgi.encode v) params in
      body <- String.concat "&" s
  end
;;


class put the_query the_contents =
  object (self)
    inherit message

    val query = the_query
    val cont = the_contents

    initializer
      self # init_query () query

    method prepare use_proxy =
      self # set_request use_proxy false "PUT";
      body <- cont
  end
;;


class delete the_query =
  object (self)
    inherit message

    val query = the_query

    initializer
      self # init_query () query

    method prepare use_proxy =
      self # set_request use_proxy false "DELETE"
  end
;;


class basic_auth_method =
  object (self)
    val mutable userdb = []    (* pairs (realm, (user, password)) *)
    val mutable current_realm = ""
    val mutable current_host = ""
    val mutable current_port = -1

    method name = "basic"

    method set_realm realm user password =
      userdb <- (realm, (user,password)) ::
	          listrm realm userdb

    method get_credentials () =
      List.assoc current_realm userdb


    method www_authenticate (req : message) param_toks =
      (* Called on response codes 401 and 407 *)
      let p_alist = self # scan_params param_toks in
      let get n = try List.assoc n p_alist with Not_found -> "" in
      current_realm <- get "realm";
      current_host <- req # get_host();
      current_port <- req # get_port()

    method set_authorization (req : message) name_authorization =
      try
        (* Do not send the password to a host that is different from the host
         * which requested authorization the last time
	 *)
	if req # get_host() <> current_host or
	   req # get_port() <> current_port then raise Not_found;
	let (user,password) = self # get_credentials() in
	let basic_cookie = Base64.encode (user ^ ":" ^ password) in
	let cred = "Basic " ^ basic_cookie in
	req # set_req_header name_authorization cred
      with
	Not_found ->
	  ()

    method update (req : message) (param_toks:token list) =
      (* Called on any response code *)
      ()

    method private scan_params p =
      let rec scan p =
	match p with
	  [] ->
	    []
	| Word name :: Special '=' :: Word value :: p' ->
	    begin
	      match p' with
		Special ',' :: p'' ->
		  (name, value) :: scan p''
	      |	[] ->
		  [name,value]
	      |	_ ->
		  failwith "error scanning header parameters"
	    end
	| _ ->
	    failwith "error scanning header parameters"
      in
      scan p
  end
;;


class digest_auth_method =
  object (self)
    inherit basic_auth_method

    val mutable current_nonce = ""
    val mutable current_opaque = ""

    method name = "digest"

    method www_authenticate req param_toks =
      (* Called on response codes 401 and 407 *)
      let p_alist = self # scan_params param_toks in
      let get n = try List.assoc n p_alist with Not_found -> "" in
      let algorithm = get "algorithm" in
      if algorithm = "MD5" or algorithm = "" then begin
	current_realm <- get "realm";
	current_nonce <- get "nonce";
	current_opaque <- get "opaque";
        (* domain: not interpreted *)
        (* stale: not interpreted -- we just give always another chance *)
	current_host <- req # get_host();
	current_port <- req # get_port()
      end

(*
    method debug realm nonce opaque =
      current_realm <- realm;
      current_nonce <- nonce;
      current_opaque <- opaque
*)

    method set_authorization req name_authorization =
      try
        (* Do not send the password to a host that is different from the host
         * which requested authorization the last time
	 *)
	if req # get_host() <> current_host or
	   req # get_port() <> current_port then raise Not_found;
	let h s = encode_hex (Digest.substring s 0 (String.length s)) in
	let (user,password) = self # get_credentials() in
	let uri = req # get_req_uri() in
	let a1 = user ^ ":" ^ current_realm ^ ":" ^ password in
	let a2 = req # get_req_method() ^ ":" ^ uri in
	let response = h (h a1 ^ ":" ^ current_nonce ^ ":" ^ h a2) in

	let digest_response = "Digest username=\"" ^ user ^ "\", realm=\"" ^
	  current_realm ^ "\", nonce=\"" ^ current_nonce ^ "\", uri=\"" ^ uri ^
	  "\", response=\"" ^ response ^ "\", opaque=\"" ^ current_opaque ^
	  "\"" in
	req # set_req_header name_authorization digest_response
      with
	Not_found -> ()


    method update req param_toks =
      let p_alist = self # scan_params param_toks in
      let get n = try List.assoc n p_alist with Not_found -> "" in
      let nextnonce = get "nextnonce" in
      if nextnonce <> "" then begin
	current_nonce <- nextnonce
      end

  end
;;


class pipeline =
  object (self)
    val mutable verbose_status = false
    val mutable verbose_request_header = false
    val mutable verbose_response_header = false
    val mutable verbose_request_contents = false
    val mutable verbose_response_contents = false

    val mutable proxy = ""
    val mutable proxy_port = 80
    val mutable proxy_auth = false
    val mutable proxy_user = ""
    val mutable proxy_password = ""

    val mutable www_auth = []

    val mutable no_proxy_for = []

    val mutable pipeline = []

    val mutable current_host = ""
    val mutable current_port = -1
    val mutable current_peer = ""          (* host or proxy *)
    val mutable current_peer_port = -1

    val mutable current_auth = (None : basic_auth_method option)

    val mutable current_socket = stdin
      (* stdin: meaningless, only to have a default value *)

    val mutable current_np = false         (* np = no persistence *)

    val mutable no_persistence = false

    method add_authentication_method m =
      let name = m # name in
      www_auth <- (name,m) :: listrm name www_auth;

    method set_proxy the_proxy the_port =
      (* proxy="": disables proxy *)
      proxy       <- the_proxy;
      proxy_port  <- the_port;
      ()

    method set_proxy_auth user passwd =
      (* sets 'user' and 'password' if demanded by a proxy *)
      proxy_auth     <- user <> "";
      proxy_user     <- user;
      proxy_password <- passwd


    method avoid_proxy_for l =
      (* l: List of hosts or domains *)
      no_proxy_for <- l


    method avoid_persistent_connection () =
      (* Instructs the client not to use the HTTP/1.1 feature of
       * persistent connections.
       *)
      no_persistence <- true


    method reset () =
      (* deletes all pending requests; closes connection *)
      pipeline <- [];
      self # abort_connection()


    method add (request : message) =
      request # set_unserved();
      pipeline <- pipeline @ [request]


    method empty () =
      pipeline = []


    method pipeline = pipeline


    method run () =
      (* Runs through the requests in the pipeline. If a request can be
       * fulfilled, i.e. the server sends a response, the status of the
       * request is set and the request is removed from the pipeline.
       * If a request cannot be fulfilled (no response, bad response,
       * network error), an exception is raised and the request remains in
       * the pipeline (and is even the head of the pipeline).
       *
       * Exception Broken_connection:
       *  - The server has closed the connection before the full request
       *    could be sent. It is unclear if something happened or not.
       *    The application should figure out the current state and
       *    retry the request.
       *  - Also raised if only parts of the response have been received
       *    and the server closed the connection. This is the same problem.
       *    Note that this can only be detected if a "content-length" has
       *    been sent or "chunked encoding" was chosen. Should normally
       *    work for persistent connections.
       *  - NOT raised if the server forces a "broken pipe" (normally
       *    indicates a serious server problem). The intention of
       *    Broken_connection is that retrying the request will probably
       *    succeed.
       *)

      let do_request req =
	let continue = ref true in

	while !continue do
	  continue := false;
	  let use_proxy = self # check_proxy req in

	  req # prepare (proxy <> "" & use_proxy);
	  let body = req # get_req_body() in

	  if no_persistence then
	    req # set_req_header "connection" "close";
	    (* Note: current_np might not be defined *)

	  req # set_req_header "content-length" (string_of_int (String.length body));
	  (* 'content-length' must be set, see RFC 2068, 4.4 *)

	  self # send_and_receive_with_auth req;
	  req # decode_body () verbose_response_header verbose_response_contents;
	  req # set_served ();
	  let version,code,code_as_string = req # dest_status()  in
	  match code with
	    (301|302) ->
	      (* Follow redirections for GET and HEAD requests *)
	      begin
		let m = req # get_req_method() in
		if m = "GET" or m = "HEAD" then
		  try
		    let location = req # assoc_resp_header "location" in
		    req # init_query () location;
		    continue := true
		  with Not_found -> ()
	      end
	  | _ -> ()
	done
      in

      let rec do_requests reqlist =
	match reqlist with
	  [] ->
	    pipeline <- []
	| req::reqlist' ->
	    better_unix_error do_request req;
	    pipeline <- reqlist';
	    do_requests reqlist'
      in

      do_requests pipeline


    method private send_and_receive_with_auth req =
      (* Interprets proxy and www authentication and resends requests.
       * Includes decode_header.
       * Downgrades to HTTP/1.0 if appropriate.
       *)
      (* TODO: The server may send *multiple* challenges. Currently, only
       * the first challenge is used
       *)
      let proxy_auth_included = ref false in
      let www_auth_included = ref false in
      let continue = ref true in

      let header = req # get_req_header() in        (* sent header *)
      let body = req # get_req_body() in            (* sent body *)

      (* Perhaps we have several trials, so loop until successful: *)

      while !continue do
	(* set "authorization" header if needed *)
	begin match current_auth with
	  None -> ()
	| Some m -> m # set_authorization req "authorization"
	end;
	(* send request and receive response: *)
	self # send_and_receive req;
	(* decode header of response: *)
	req # decode_header();
	(* authentication method has the chance to interpret response: *)
	begin match current_auth with
	  None -> ()
	| Some m ->
	    try
	      let auth_info = req # assoc_resp_header "authentication-info" in
	      m # update req (parse_header_value auth_info)
	    with Not_found -> ()
	end;
	continue := false;
	let version,code,code_as_string = req # dest_status()  in
	if verbose_status then begin
	  prerr_endline ("Response code: " ^ string_of_int code);
	  prerr_endline ("Response text: " ^ code_as_string);
	  prerr_endline ("Response protocol: " ^ version);
	end;
	(* Downgrade to HTTP/1.0? *)
	(* Currently this only means not to use persistent connections. *)
	if version = "HTTP/1.0" then begin
	  current_np <- true;
	  self # close_connection();
	end;
	let rheader = req # get_resp_header() in
	if verbose_response_header then
	  req # dump_header "Response " rheader;
	(* Got a "connection: close" header? *)
	if (try List.assoc "connection" rheader with Not_found -> "") = "close"
	then begin
	  current_np <- true;
	  self # close_connection();
	end;
	(* Now find out if authentication is required *)
	match code with
	  407 -> (* proxy authentication required *)
	    if proxy_auth & not !proxy_auth_included then begin
	      let challenge =
		try
		  List.assoc "proxy-authenticate" rheader
		with
		  Not_found -> failwith "http_client: missing Proxy-Authenticate header field in response"
	      in
	      (* Note: We do not support the distinction of realms for
	       * proxy authorization.
	       *)
	      let basic_cookie = Base64.encode (proxy_user ^ ":" ^ proxy_password) in
	      let cred = "Basic " ^ basic_cookie in
	      req # set_req_header "proxy-authorization" cred;
	      proxy_auth_included := true;
	      continue := true
	    end
	| 401 -> (* Unauthorized *)
	    if not !www_auth_included then begin
	      let challenge =
		try
		  List.assoc "www-authenticate" rheader
		with
		  Not_found -> failwith "http_client: missing WWW-Authenticate header field in response"
	      in
	      let auth_method, params = self # scan_auth_header challenge in
	      try
		let auth = List.assoc auth_method www_auth in
		auth # www_authenticate req params;
		current_auth <- Some auth;
		www_auth_included := true;
		continue := true
	      with
		Not_found -> ()              (* ==> error message remains *)
	    end
	| _ -> ()
      done


    method private get_realm authenticate_header =
      (* Isolates the realm from a "www-authenticate" or "proxy-authenticate"
       * header line demanding basic authorization
       *)
      let p = parse_header_value authenticate_header in
      match p with
	[ Word basic; Word realm; Special '='; Word realm_value ]
	  when String.lowercase basic = "basic" &
	       String.lowercase realm = "realm"
	->
	  realm_value
      |	_
	->
	  failwith "http_client: unsupported authorization method"

    method private scan_auth_header h =
      let p = parse_header_value h in
      try
	begin match p with
	  Word m :: p' ->
	    String.lowercase m, p'
	|	_ ->
	    failwith "unsupported authentication method"
	end
      with
	Failure f ->
	  failwith ("http_client: problems with authentication: " ^ f)


    method private check_proxy req =
      let host = req # get_host() in
      let port = req # get_port() in

      (*
        prerr_endline ("CHECK PROXY: host = " ^ host);
        prerr_endline ("CHECK PROXY: no_proxy_for = " ^ String.concat "," no_proxy_for);
       *)

      (req # is_proxy_allowed()) &
      not
	(List.exists
	  (fun dom ->
	    if dom <> "" &
	       dom.[0] = '.' &
	       String.length host > String.length dom
	    then
	      let ld = String.length dom in
	      String.lowercase(String.sub host (String.length host - ld) ld)
		= String.lowercase dom
	    else
	      dom = host)
	  no_proxy_for)


    method private send_and_receive req =
      (* Sends a message using current_socket. If there is no current
       * socket, opens a new connection for 'req'. Waits until the full
       * response has been arrived. (i.e. works synchronously)
       *)

      let host = req # get_host() in
      let port = req # get_port() in

      let no_proxy = not (self # check_proxy req) in

      let request = req # get_request() in
      let header = req # get_req_header() in
      let body = req # get_req_body() in

      let crlf = "\r\n" in
      let rbuf = String.make rcv_increment ' ' in
      let sbuf = request ^ crlf ^
	String.concat "" (List.map (fun (n,v) -> n ^ ": " ^ v ^ crlf) header) ^
	crlf ^
	body
      in

      let sbuf_length = String.length sbuf in

      let still_sending = ref true in
      let request_sent = ref false in

      let received = ref "" in

      let is_message_complete msg =
        try
	  let header, header_length = req # decode_header_at () msg 0 in
	  req # body_is_complete () header msg header_length;
	  true
	with
	  Header_is_incomplete -> false
	| Body_is_incomplete   -> false
	| Body_maybe_complete  -> true
       in

      let can_message_continue msg =
        try
	  let header, header_length = req # decode_header_at () msg 0 in
	  req # body_is_complete () header msg header_length;
	  false
	with
	  Header_is_incomplete -> true
	| Body_is_incomplete   -> true
	| Body_maybe_complete  -> true
      in

      let rec s_a_r s sbuf_count =
        (* sends on socket s all in sbuf beginning at position sbuf_count;
	 * receives message at the same time and returns it in 'received'.
         *)
	let rl, wl, _ =
	  select 
	    [ s ] (if !still_sending then [ s ] else []) [] timeout in
	if rl = [] & wl = [] then begin
	  self # abort_connection();
	  failwith "http_client: Timeout"
	end;

	if rl <> [] then begin
	  (* Note: If the request has not yet been sent completely, there is
	   * something wrong
	   *)
	  let l = read s rbuf 0 rcv_increment in
	  if l = 0 then begin (* EOF *)
	    self # close_connection();
	    if not !request_sent & sbuf_count = 0 then
	      raise Async_close;
	    if not !request_sent then
	      raise Broken_connection;
	    if not (is_message_complete !received) then
	      raise Broken_connection;
	    ()
          end
	  else
	    let rbuf' = String.create l in
	    String.blit rbuf 0 rbuf' 0 l;
	    received := !received ^ rbuf';
	    if can_message_continue !received then begin
	      if !request_sent then
		s_a_r s sbuf_count
	      else begin
		(* If not !request_sent:
		 * This can only mean: We have seen the beginning of an
		 * error or informational message.
		 * TODO: Find out the latter
		 *)
		(* still_sending := false; -- only appropriate for errors *)
		s_a_r s sbuf_count
	      end
	    end
	    else
	      ()
		(* If not !request_sent:
		 * This can only mean: We have got an error message!
		 *)

	end
	else begin (* ==> wl <> [] *)
	  let len = write s sbuf 0 (sbuf_length - sbuf_count) in
	  let sbuf_count' = sbuf_count + len in

	  if sbuf_count' >= sbuf_length then begin
	    if current_np then
	      shutdown s SHUTDOWN_SEND;
	    request_sent := true;
	    still_sending := false
	  end;
	  s_a_r s sbuf_count'
	end
      in

      self # open_or_continue_connection host port no_proxy;

      if verbose_status then
        prerr_endline ("Request: " ^ request);
      if verbose_request_header then
	req # dump_header "Request " header;
      if verbose_request_contents & body <> "" then
	prerr_endline ("Request body:\n" ^ body);

      begin try
	s_a_r current_socket 0
      with
	Async_close ->
	  (* Retry any request, because the EOF has been seen before
	   * anything is sent
	   *)
	  self # abort_connection();
	  self # open_or_continue_connection host port no_proxy;
	  begin try
	    s_a_r current_socket 0
	  with
	    Async_close ->
	      (* Not good. Transform into Broken_connection *)
	      raise Broken_connection
	  | any ->
	      self # abort_connection();
	      raise any
	  end
      |	any ->
	  self # abort_connection();
	  raise any
      end;

      if current_np then
        self # close_connection();

      req # set_response () !received;

(*
prerr_endline("-------------------------");
prerr_endline !received;
prerr_endline("-------------------------");
*)
      ()



    method private open_or_continue_connection host port no_proxy =
      let peer, peer's_port =
	if proxy = "" or no_proxy then
	  host, port
	else
	  proxy, proxy_port
      in

      if peer <> current_peer or peer's_port <> current_peer_port then begin
	self # close_connection();

	let addr =
	  try
	    inet_addr_of_string (Http_client_aux.match_ip peer)
	  with
	    Not_found ->
	      try
		let h = gethostbyname peer in
		h.h_addr_list.(0)
	      with Not_found ->
		failwith ("http_client: host lookup failed for " ^ peer);
	in

	let s = socket PF_INET SOCK_STREAM 0 in
	( try 
	    connect s (ADDR_INET (addr, peer's_port));
	  with
	      e -> close s; raise e
	);

	current_host <- host;
	current_port <- port;
	current_socket <- s;
	current_peer <- peer;
	current_peer_port <- peer's_port;
	current_auth <- None;
	current_np <- no_persistence;

	if verbose_status then
	  prerr_endline ("Connected to " ^ peer ^ ":" ^ string_of_int peer's_port)
      end

    method close_connection() =
      if (current_peer <> "") then begin
        shutdown current_socket SHUTDOWN_SEND;
	close current_socket;
	current_host <- "";
	current_port <- -1;
	current_peer <- "";
	current_peer_port <- -1;
	current_auth <- None;
	if verbose_status then
	  prerr_endline ("Connection closed");
      end


    method abort_connection() =
      if (current_peer <> "") then begin
	close current_socket;
	current_host <- "";
	current_port <- -1;
	current_peer <- "";
	current_peer_port <- -1;
	current_auth <- None;
	if verbose_status then
	  prerr_endline ("Connection aborted");
      end


    method verbose level =
      verbose_status            <- List.mem Verbose_status level;
      verbose_request_header    <- List.mem Verbose_request_header level;
      verbose_response_header   <- List.mem Verbose_response_header level;
      verbose_request_contents  <- List.mem Verbose_request_contents level;
      verbose_response_contents <- List.mem Verbose_response_contents level

    method very_verbose () =
      self # verbose [ Verbose_status; Verbose_request_header;
		       Verbose_response_header;
		       Verbose_request_contents;
		       Verbose_response_contents ]
  end
;;


module Convenience =
  struct

    class simple_basic_auth_method f =
      object
	inherit basic_auth_method
	method get_credentials () = f current_realm
      end

    class simple_digest_auth_method f =
      object
	inherit digest_auth_method
	method get_credentials () = f current_realm
      end

    let http_url_decode url = Http_client_aux.match_http url

    let http_trials = ref 3
    let http_user = ref ""
    let http_password = ref ""

    let this_user = ref ""
    let this_password = ref ""

    let auth_basic =
      new simple_basic_auth_method
	(fun realm ->
	  if !this_user <> "" then
	    !this_user, !this_password
	  else
	    if !http_user <> "" then
	      !http_user, !http_password
	    else
	      raise Not_found)

    let auth_digest =
      new simple_digest_auth_method
	(fun realm ->
	  if !this_user <> "" then
	    !this_user, !this_password
	  else
	    if !http_user <> "" then
	      !http_user, !http_password
	    else
	      raise Not_found)


    let get_default_pipe() =

      let p = new pipeline in

      (* Is the environment variable "http_proxy" set? *)
      let http_proxy =
	try Sys.getenv "http_proxy" with Not_found -> "" in
      begin try
	let (user,password,host,port,path) = http_url_decode http_proxy in
	p # set_proxy (Cgi.decode host) port;
	match user with
	  Some user_s ->
	    begin match password with
	      Some password_s ->
		p # set_proxy_auth (Cgi.decode user_s) (Cgi.decode password_s)
	    | None -> ()
	    end
	| None -> ()
      with
	Not_found -> ()
      end;

      (* Is the environment variable "no_proxy" set? *)
      let no_proxy =
	try Sys.getenv "no_proxy" with Not_found -> "" in
      let no_proxy_list =
	Http_client_aux.split_words_by_commas no_proxy in
      p # avoid_proxy_for no_proxy_list;

      (* Add authentication methods: *)
      p # add_authentication_method auth_basic;
      p # add_authentication_method auth_digest;

      (* That's it: *)
      p


    let pipe = lazy (get_default_pipe())

    let mutex = Http_client_aux.Mtx.create()

    let request m trials =
      Http_client_aux.Mtx.lock mutex;
      try
	let p = Lazy.force pipe in
	if not (p # empty()) then
	  p # reset();
	p # add m;
	let rec next_trial todo e =
	  if todo > 0 then begin
	    try
	      p # run()
	    with
	    | Http_error(n,s) as e' ->
		if List.mem n [408; 413; 500; 502; 503; 504 ] then
		  next_trial (todo-1) e'
		else
		  raise e'
	    | e' -> next_trial (todo-1) e'
	  end
	  else
	    raise e
	in
	next_trial trials (Failure "bad number of http_trials");
	Http_client_aux.Mtx.unlock mutex;
      with
	any ->
	  Http_client_aux.Mtx.unlock mutex;
	  raise any

    let prepare_url url =
      Http_client_aux.Mtx.lock mutex;
      try
	this_user := "";
    	let (user,password,host,port,path) = http_url_decode url in
	begin match user with
	  Some user_s ->
	    this_user := Cgi.decode user_s;
	    this_password := "";
	    begin match password with
	      Some password_s ->
		this_password := Cgi.decode password_s
	    | None -> ()
	    end
	| None -> ()
	end;
	Http_client_aux.Mtx.unlock mutex;
	"http://" ^ host ^ ":" ^ string_of_int port ^ path
      with
	Not_found -> 
	  Http_client_aux.Mtx.unlock mutex;
	  url
      |	any -> 
	  Http_client_aux.Mtx.unlock mutex;
	  raise any

    let http_get_message url =
      let m = new get (prepare_url url) in
      request m !http_trials;
      m

    let http_get url = (http_get_message url) # get_resp_body()

    let http_head_message url =
      let m = new head (prepare_url url) in
      request m !http_trials;
      m

    let http_post_message url params =
      let m = new post (prepare_url url) params in
      request m 1;
      m

    let http_post url params = (http_post_message url params) # get_resp_body()

    let http_put_message url content =
      let m = new put (prepare_url url) content in
      request m !http_trials;
      m

    let http_put url content = (http_put_message url content) # get_resp_body()

    let http_delete_message url =
      let m = new delete (prepare_url url) in
      request m 1;
      m

    let http_delete url = (http_delete_message url) # get_resp_body()


    let http_verbose() =
      Http_client_aux.Mtx.lock mutex;
      try
	(Lazy.force pipe) # very_verbose();
	Http_client_aux.Mtx.unlock mutex;
      with
	any ->
	  Http_client_aux.Mtx.unlock mutex;
	  raise any
  end


(* ======================================================================
 * History:
 *
 * $Log: http_client.ml,v $
 * Revision 1.5.2.2  2002/07/07 11:52:09  gerd
 * 	Fix: Error on connect
 *
 * Revision 1.5.2.1  2001/12/15 16:15:48  gerd
 * 	Updates for O'Caml 3.04
 *
 * Revision 1.5  1999/07/08 02:59:47  gerd
 * 	Added support for multi-threaded applications. All regexps and
 * all system calls have been moved to the module Http_client_aux, for
 * which two implementations exist, one for MT apps, one for non-MT apps.
 * 	The Convenience module has automatic serialization.
 *
 * Revision 1.4  1999/06/10 19:26:15  gerd
 * 	Bugfix: The "basic" authentication method was  added
 * twice to the convenience pipeline, which is incorrect.  Now "basic" and
 * "digest" are added.
 * 	Bugfix: The End-of-file condition while receiving frameless
 * messages (no chunks, no content-length header) does not cause a
 * Broken_connection any longer. Instead, the message consists of all
 * what has been read till EOF.
 * 	Added: In verbose mode, the HTTP protocol version of the
 * server is printed, too.
 *
 * Revision 1.3  1999/06/10 00:12:53  gerd
 * 	Change: The HTTP response codes 301 and 302 ("moved permanently",
 * resp. "moved temporarily") are now interpreted by the 'pipeline' class.
 * (But 303 not (yet?) -- perhaps there should be a switch to turn this
 * feature on or off once it gets implemented...)
 * 	Bugfix: The decision whether a proxy should be used or not works
 * now. The weird function 'dest_query' has gone, it is substituted by
 * 'init_query' which can be called in the initializers of the method
 * subclasses. Thus the query is analyzed very early which is very useful
 * in order to decide if a proxy is needed to reach a certain host.
 * 	Added: There is now a Convenience module which has a simplified
 * interface. It is sufficient for many applications that only want to do
 * simple GET and POST operations.
 *
 * Revision 1.2  1999/06/09 19:41:32  gerd
 * 	Bugfix: Responses that neither have chunked encoding nor
 * are accompanied with a "content-length" header, are now read until
 * EOF. Formerly, the behaviour was that only the first network block
 * was read and the rest was discarded. This was especially a problem
 * with HTTP/1.0 servers and CGI output.
 *
 * Revision 1.1  1999/03/26 01:16:42  gerd
 * 	initial revision
 *
 *
 *)
