/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 * 
 * Contributor(s):
 * 
 * Portions Copyrighted 2008 Sun Microsystems, Inc.
 */

package org.netbeans.modules.parsing.spi;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.netbeans.api.annotations.common.CheckForNull;
import org.netbeans.modules.parsing.api.Source;
import org.netbeans.modules.parsing.impl.CurrentDocumentScheduler;
import org.netbeans.modules.parsing.impl.CursorSensitiveScheduler;
import org.netbeans.modules.parsing.impl.SchedulerAccessor;
import org.netbeans.modules.parsing.impl.SelectedNodesScheduler;
import org.netbeans.modules.parsing.impl.SourceAccessor;
import org.netbeans.modules.parsing.impl.SourceCache;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.loaders.DataObjectNotFoundException;

import org.openide.util.RequestProcessor;
import org.openide.util.RequestProcessor.Task;
import org.openide.util.WeakListeners;


/**
 * Scheduler defines when tasks should be started. Some {@link SchedulerTask}s (like syntax
 * coloring) are current document sensitive only. It means that such {@link SchedulerTask} 
 * is automatically scheduled when currently edited document is changed.
 * Other tasks may listen on different events. Implementation of Scheduler
 * just listens on various IDE events, and call one of schedule() methods
 * when something interesting happens. Implementation of Parsing API just finds
 * all {@link SchedulerTask}s registerred for this Scheduler and reschedules them.
 * Implementation of this class should be registerred in your manifest.xml file
 * in "Editors/your mime type" folder.
 * 
 * @author Jan Jancura
 */
public abstract class Scheduler {
    
    /**
     * Default reparse delay
     */
    public static final int DEFAULT_REPARSE_DELAY = 500;

    private static final Logger LOG = Logger.getLogger(Scheduler.class.getName());

    private final PropertyChangeListener listener = new PropertyChangeListener() {
        @Override
        public void propertyChange(PropertyChangeEvent evt) {
            if (DataObject.PROP_PRIMARY_FILE.equals(evt.getPropertyName())) {
                final DataObject dobj = (DataObject) evt.getSource();
                final Source newSource = Source.create(dobj.getPrimaryFile());
                if (newSource != null) {
                    LOG.log(
                        Level.FINE,
                        "Rescheduling {0} due to change of primary file.",  //NOI18N
                        dobj.getPrimaryFile());
                    schedule(newSource, new SchedulerEvent(newSource));
                }
            }
        }
    };
    
    /**
     * May be changed by unit test
     */
    int                     reparseDelay = DEFAULT_REPARSE_DELAY;

    //@GuardedBy("this")
    private Source          source;
    //@GuardedBy("this")
    private PropertyChangeListener wlistener;
    
    /**
     * This implementations of {@link Scheduler} reschedules all tasks when:
     * <ol>
     * <li>current document is changed (file opened, closed, editor tab switched), </li>
     * <li>text in the current document is changed, </li>
     * <li>cusor position is changed</li>
     * </ol>
     */
    public static final Class<? extends Scheduler>
                            CURSOR_SENSITIVE_TASK_SCHEDULER = CursorSensitiveScheduler.class;
    
    /**
     * This implementations of {@link Scheduler} reschedules all tasks when:
     * <ol>
     * <li>current document is changed (file opened, closed, editor tab switched), </li>
     * <li>text in the current document is changed</li>
     * </ol>
     */
    public static final Class<? extends Scheduler>
                            EDITOR_SENSITIVE_TASK_SCHEDULER = CurrentDocumentScheduler.class;
    
    /**
     * This implementations of {@link Scheduler} reschedules all tasks when
     * nodes selected in editor are changed.
     */
    public static final Class<? extends Scheduler>
                            SELECTED_NODES_SENSITIVE_TASK_SCHEDULER = SelectedNodesScheduler.class;

    /**
     * Reschedule all tasks registered for <code>this</code> Scheduler (see
     * {@link ParserResultTask#getSchedulerClass()}.
     */
    protected final synchronized void schedule (
        SchedulerEvent      event
    ) {
        if (source != null)
            schedule (source, event);
    }

    private RequestProcessor 
                            requestProcessor;
    private Task            task;
    
    /**
     * Reschedule all tasks registered for <code>this</code> Scheduler (see
     * {@link ParserResultTask#getSchedulerClass()}, and sets new {@link Source}s for them.
     * 
     * @param sources       A collection of {@link Source}s.
     */
    //tzezula: really unclear usages of sources field (synchronization, live cycle, may it be called twice with different set of sources?).
    //tzezula: should set CHANGE_EXPECTED flag on the sources.
    protected final synchronized void schedule (
        final Source        source,
        final SchedulerEvent event) {
        if (task != null) {
            task.cancel ();
        }
        task = null;
        if (requestProcessor == null) {
            requestProcessor = new RequestProcessor (
                    Scheduler.class.getName(),
                    1,
                    false,
                    false);
        }        
        if (this.source != source) {
            if (this.source != null) {
                final SourceCache cache = SourceAccessor.getINSTANCE().getCache(this.source);
                cache.unscheduleTasks(Scheduler.this.getClass());
                if (wlistener != null) {
                    final FileObject fo = this.source.getFileObject();
                    if (fo != null) {
                        try {
                            final DataObject dobj = DataObject.find(fo);
                            dobj.removePropertyChangeListener(wlistener);
                        } catch (DataObjectNotFoundException nfe) {
                            //No DataObject for file - ignore
                        }
                    }
                }
            }
            this.source = source;
            if (source != null) {
                final FileObject fo = source.getFileObject();
                if (fo != null) {
                    try {
                        final DataObject dobj = DataObject.find(fo);
                        wlistener = WeakListeners.propertyChange(listener, dobj);
                        dobj.addPropertyChangeListener(wlistener);
                    } catch (DataObjectNotFoundException ex) {
                        //No DataObject for file - ignore
                    }
                }
            }
        }
        if (source == null) {
            return;
        }
        task = requestProcessor.create (new Runnable () {
            @Override
            public void run () {
                SourceCache cache = SourceAccessor.getINSTANCE ().getCache (source);                
                SourceAccessor.getINSTANCE ().setSchedulerEvent (source, Scheduler.this, event);
                //S ystem.out.println ("\nSchedule tasks (" + Scheduler.this + "):");
                cache.scheduleTasks (Scheduler.this.getClass ());
            }
        });
        task.schedule (reparseDelay);
    }

    /**
     * Returns active {@link Source}.
     * The {@link Scheduler} subclasses should use this method to obtain the active
     * {@link Source} rather than caching the {@link Source} them self.
     * @return the {@link Source} currently handled by scheduler.
     * @since 1.69
     */
    @CheckForNull
    protected final synchronized Source getSource() {
        return this.source;
    }

    protected abstract SchedulerEvent createSchedulerEvent (SourceModificationEvent event);

    static {
        SchedulerAccessor.set (new Accessor ());
    }
    
    private static class Accessor extends SchedulerAccessor {

        @Override
        public SchedulerEvent createSchedulerEvent (Scheduler scheduler, SourceModificationEvent event) {
            return scheduler.createSchedulerEvent (event);
        }
    }
}






