/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 *
 * Contributor(s):
 *
 * Portions Copyrighted 2010 Sun Microsystems, Inc.
 */
package org.netbeans.modules.cnd.makeproject.ui;

import java.awt.Color;
import java.awt.EventQueue;
import java.awt.Image;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyVetoException;
import java.io.CharConversionException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.Action;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ProjectInformation;
import org.netbeans.api.project.ProjectUtils;
import org.netbeans.modules.cnd.api.project.BrokenIncludes;
import org.netbeans.modules.cnd.api.project.NativeProject;
import org.netbeans.modules.cnd.api.toolchain.CompilerSetManager;
import org.netbeans.modules.cnd.api.toolchain.ui.ToolsCacheManager;
import org.netbeans.modules.cnd.makeproject.MakeProject;
import org.netbeans.modules.cnd.makeproject.MakeProjectConfigurationProvider;
import org.netbeans.modules.cnd.makeproject.MakeProjectTypeImpl;
import org.netbeans.modules.cnd.makeproject.api.configurations.Configuration;
import org.netbeans.modules.cnd.makeproject.api.configurations.ConfigurationDescriptor.State;
import org.netbeans.modules.cnd.makeproject.api.configurations.Configurations;
import org.netbeans.modules.cnd.makeproject.api.configurations.Folder;
import org.netbeans.modules.cnd.makeproject.api.configurations.MakeConfiguration;
import org.netbeans.modules.cnd.makeproject.api.configurations.MakeConfigurationDescriptor;
import org.netbeans.modules.cnd.makeproject.api.configurations.MakefileConfiguration;
import org.netbeans.modules.cnd.utils.cache.CndFileUtils;
import org.netbeans.modules.nativeexecution.api.ExecutionEnvironment;
import org.netbeans.spi.project.ProjectConfigurationProvider;
import org.netbeans.spi.project.ui.ProjectProblemsProvider;
import org.netbeans.spi.project.ui.support.CommonProjectActions;
import org.openide.awt.Actions;
import org.openide.filesystems.FileObject;
import org.openide.nodes.Children;
import org.openide.nodes.Node;
import org.openide.util.Exceptions;
import org.openide.util.ImageUtilities;
import org.openide.util.Lookup;
import org.openide.util.LookupEvent;
import org.openide.util.LookupListener;
import org.openide.util.RequestProcessor;
import org.openide.util.SharedClassObject;
import org.openide.util.WeakListeners;
import org.openide.util.datatransfer.PasteType;
import org.openide.util.lookup.AbstractLookup;
import org.openide.util.lookup.InstanceContent;
import org.openide.xml.XMLUtil;

/**
 * Filter node contain additional features for the Make physical
 *
 * @author Alexander Simon
 */
final class MakeLogicalViewRootNode extends AnnotatedNode implements ChangeListener, LookupListener, PropertyChangeListener {


    private boolean brokenIncludes;
    private boolean brokenProject;
    private Folder folder;
    private final Lookup.Result<BrokenIncludes> brokenIncludesResult;
    private final MakeLogicalViewProvider provider;
    private final InstanceContent ic;
    private final RequestProcessor.Task stateChangedTask;
    private static final int WAIT_DELAY = 500;
    
    public MakeLogicalViewRootNode(Folder folder, MakeLogicalViewProvider provider, InstanceContent ic) {
        this(new ProjectRootChildren(folder, provider),folder, provider, ic);
    }
    
    private MakeLogicalViewRootNode(ProjectRootChildren children, Folder folder, MakeLogicalViewProvider provider, InstanceContent ic) {
        super(children, new AbstractLookup(ic), provider.getAnnotationRP());
        children.setMakeLogicalViewRootNode(MakeLogicalViewRootNode.this);
        this.ic = ic;
        this.folder = folder;
        this.provider = provider;        
//        setChildren(new ProjectRootChildren(folder, provider));
        setIconBaseWithExtension(MakeConfigurationDescriptor.ICON);
        setName(ProjectUtils.getInformation(provider.getProject()).getDisplayName());

        brokenIncludesResult = Lookup.getDefault().lookup(new Lookup.Template<>(BrokenIncludes.class));
        brokenIncludesResult.addLookupListener(MakeLogicalViewRootNode.this);
        resultChanged(null);

        brokenIncludes = hasBrokenIncludes(provider.getProject());
        // Handle annotations
        setForceAnnotation(true);
        if (folder != null) {
            updateAnnotationFiles();
        }
        ProjectInformation pi = provider.getProject().getLookup().lookup(ProjectInformation.class);
        pi.addPropertyChangeListener(WeakListeners.propertyChange(MakeLogicalViewRootNode.this, pi));
        ToolsCacheManager.addChangeListener(WeakListeners.change(MakeLogicalViewRootNode.this, null));
        if (gotMakeConfigurationDescriptor()) {
            MakeProjectConfigurationProvider confProvider = provider.getProject().getLookup().lookup(MakeProjectConfigurationProvider.class);
            if (confProvider != null){
                confProvider.addPropertyChangeListener(WeakListeners.propertyChange(MakeLogicalViewRootNode.this, confProvider));
            }
            
            
        }

        stateChangedTask = provider.getAnnotationRP().create(new StateChangeRunnableImpl(), true);
    }

    @Override
    public String getHtmlDisplayName() {
        String ret = getHtmlDisplayName2();
        ExecutionEnvironment env = provider.getProject().getFileSystemHost();
        if (env != null && env.isRemote()) {
            if (ret == null) {
                ret = getName();
            }
            ret = ret + " <font color=\"!controlShadow\">[" + env.getDisplayName() + "]"; // NOI18N
        }
        return ret;
    }
    
    private String getHtmlDisplayName2() {
        String ret = super.getHtmlDisplayName();
        if (brokenProject) {
            try {
                ret = XMLUtil.toElementContent(ret);
            } catch (CharConversionException ex) {
                return ret;
            }
            return "<font color=\"#"+Integer.toHexString(getErrorForeground().getRGB() & 0xffffff) +"\">" + ret + "</font>"; //NOI18N
        }
        return ret;
    }

    private static Color getErrorForeground() {
        Color result = UIManager.getDefaults().getColor("nb.errorForeground");  //NOI18N
        if (result == null) {
            result = Color.RED;
        }
        return result;
    }

    public void reInit(MakeConfigurationDescriptor configurationDescriptor) {
        Folder logicalFolders = configurationDescriptor.getLogicalFolders();
        if (folder != null) {
            ic.remove(folder);
        }
        folder = logicalFolders;
        ic.add(logicalFolders);
        setChildren(new LogicalViewChildren(folder, provider));
        MakeProjectConfigurationProvider confProvider = provider.getProject().getLookup().lookup(MakeProjectConfigurationProvider.class);
        if (confProvider != null) {
            confProvider.addPropertyChangeListener(WeakListeners.propertyChange(MakeLogicalViewRootNode.this, confProvider));
        }
        stateChanged(null);
    }
    
    void reInitWithRemovedPrivate() {
        final MakeConfigurationDescriptor mcd = provider.getMakeConfigurationDescriptor();
        Configuration[] confs = mcd.getConfs().toArray();
        for (int i = 0; i < confs.length; i++) {
            MakeConfiguration conf = (MakeConfiguration) confs[i];
            if (conf.getDevelopmentHost().isLocalhost()) {
                final int platform1 = CompilerSetManager.get(conf.getDevelopmentHost().getExecutionEnvironment()).getPlatform();
                final int platform2 = conf.getDevelopmentHost().getBuildPlatformConfiguration().getValue();
                if (platform1 != platform2) {
                    conf.getDevelopmentHost().getBuildPlatformConfiguration().setValue(platform1);
                }
            }
        }
        reInit(provider.getMakeConfigurationDescriptor());
    }
    
    void reInitWithUnsupportedVersion() {
        //checkVersion = false;
        reInit(provider.getMakeConfigurationDescriptor());
    }
    
    private void setRealProjectFolder(Folder folder) {
        assert folder != null;
        if (this.folder != null) {
            ic.remove(this.folder);
        }
        this.folder = folder;
        ic.add(folder);
        stateChanged(null);
        Runnable runnable = new Runnable() {
            @Override
            public void run() {
                try {
                    // to reinvalidate Run/Debug and other toolbar buttons, we use the workaround with selection
                    // remember selection
                    Node[] selectedNodes = ProjectTabBridge.getInstance().getExplorerManager().getSelectedNodes();
                    // clear
                    ProjectTabBridge.getInstance().getExplorerManager().setSelectedNodes(new Node[0]);
                    // restore
                    ProjectTabBridge.getInstance().getExplorerManager().setSelectedNodes(selectedNodes);
                } catch (PropertyVetoException ex) {
                    Exceptions.printStackTrace(ex);
                }
            }
        };
        if (SwingUtilities.isEventDispatchThread()) {
            runnable.run();
        } else {
            SwingUtilities.invokeLater(runnable);
        }
    }

    public Folder getFolder() {
        return folder;
    }

    private MakeProject getProject() {
        return provider.getProject();
    }

    private boolean gotMakeConfigurationDescriptor() {
        return provider.gotMakeConfigurationDescriptor();
    }

    MakeConfigurationDescriptor getMakeConfigurationDescriptor() {
        return provider.getMakeConfigurationDescriptor();
    }

    private void updateAnnotationFiles() {
        // Add project directory
        FileObject fo = getProject().getProjectDirectory();
        if (fo == null || !fo.isValid()) {
            // See IZ 125880
            Logger.getLogger("cnd.makeproject").log(Level.WARNING, "project.getProjectDirectory() == null - {0}", getProject());
        }
        if (!gotMakeConfigurationDescriptor()) {
            return;
        }
        // Add buildfolder from makefile projects to sources. See IZ 90190.
        MakeConfigurationDescriptor makeConfigurationDescriptor = getMakeConfigurationDescriptor();
        if (makeConfigurationDescriptor == null) {
            return;
        }
        Configurations confs = makeConfigurationDescriptor.getConfs();
        if (confs == null) {
            return;
        }
        Set<FileObject> set = new LinkedHashSet<>();
        for (Configuration conf : confs.toArray()) {
            MakeConfiguration makeConfiguration = (MakeConfiguration) conf;
            if (makeConfiguration.isMakefileConfiguration()) {
                MakefileConfiguration makefileConfiguration = makeConfiguration.getMakefileConfiguration();
                FileObject buildCommandFO = makefileConfiguration.getAbsBuildCommandFileObject();
                if (buildCommandFO != null && buildCommandFO.isValid()) {
                    try {
                        FileObject fileObject = CndFileUtils.getCanonicalFileObject(buildCommandFO);
                        if (fileObject != null /*paranoia*/ && fileObject.isValid()) {
                            set.add(fileObject);
                        }
                    } catch (IOException ioe) {
                        ioe.printStackTrace(System.err);
                    }
                }
            }
        }
        set.add(getProject().getProjectDirectory());
        setFiles(set);
        Folder aFolder = folder;
        if (aFolder != null) {
            List<Folder> allFolders = new ArrayList<>();
            allFolders.add(aFolder);
            allFolders.addAll(aFolder.getAllFolders(true));
            Iterator<Folder> iter = allFolders.iterator();
            while (iter.hasNext()) {
                iter.next().addChangeListener(this);
            }
        }
    }

    @Override
    public String getShortDescription() {
        return MakeLogicalViewProvider.getShortDescription(provider.getProject());
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        setName(ProjectUtils.getInformation(provider.getProject()).getDisplayName());
        String prop = evt.getPropertyName();
        if (ProjectInformation.PROP_DISPLAY_NAME.equals(prop)) {
            fireDisplayNameChange(null, null);
        } else if (ProjectInformation.PROP_NAME.equals(prop)) {
            fireNameChange(null, null);
        } else if (ProjectInformation.PROP_ICON.equals(prop)) {
            fireIconChange();
        } else if (ProjectConfigurationProvider.PROP_CONFIGURATIONS.equals(prop)) {
            stateChanged(null) ;
        }
    }

    private final class VisualUpdater implements Runnable {

        @Override
        public void run() {
            if (brokenProject) {
                MakeLogicalViewRootNode.this.setChildren(Children.LEAF);
            }
            fireIconChange();
            fireOpenedIconChange();
            fireDisplayNameChange(null, null);
        }
    }

    /*
     * Something in the folder has changed
     **/
    @Override
    public void stateChanged(ChangeEvent e) {
        if (stateChangedTask != null) {
            stateChangedTask.schedule(WAIT_DELAY);
        }
    }

    @Override
    public Object getValue(String valstring) {
        if (valstring == null) {
            return super.getValue(null);
        }
        if (valstring.equals("Folder")) // NOI18N
        {
            return folder;
        } else if (valstring.equals("Project")) // NOI18N
        {
            return getProject();
        } else if (valstring.equals("This")) // NOI18N
        {
            return this;
        }
        return super.getValue(valstring);
    }

    @Override
    public Image getIcon(int type) {
        ProjectInformation pi = provider.getProject().getLookup().lookup(ProjectInformation.class);
        return mergeBadge(annotateIcon(ImageUtilities.icon2Image(pi.getIcon()), type));
    }

    @Override
    public Image getOpenedIcon(int type) {
        return getIcon(type);
    }
    
    private Image mergeBadge(Image original) {
        ProjectProblemsProvider pp = getProject().getLookup().lookup(ProjectProblemsProvider.class);
        if (brokenProject) {
            return ImageUtilities.mergeImages(original, MakeLogicalViewProvider.brokenProjectBadge, 8, 0);
        }
        if (pp.getProblems().size()>0) {
            return ImageUtilities.mergeImages(original, MakeLogicalViewProvider.brokenProjectBadge, 8, 0);
        }
        if (brokenIncludes) {
            return ImageUtilities.mergeImages(original, MakeLogicalViewProvider.brokenIncludeBadge, 8, 0);
        }        
        return original;
    }

    @Override
    public Action[] getActions(boolean context) {
        List<Action> actions = new ArrayList<>();
        if (!gotMakeConfigurationDescriptor()) {
            actions.add(CommonProjectActions.closeProjectAction());
            return actions.toArray(new Action[actions.size()]);        
        }
        MakeConfigurationDescriptor descriptor = getMakeConfigurationDescriptor();

        // TODO: not clear if we need to call the following method at all
        // but we need to remove remembering the output to prevent memory leak;
        // I think it could be removed
        if (descriptor != null) {
            descriptor.getLogicalFolders();
        }

        MakeConfiguration active = (descriptor == null) ? null : descriptor.getActiveConfiguration();
        String projectType = MakeProjectTypeImpl.PROJECT_TYPE;
        Action[] projectActions = null;
        if (active != null && active.isCustomConfiguration()) {
            //TODO: fix it as all actions can use  HIDE_WHEN_DISABLE and be enabled in own context only
            projectActions = active.getProjectCustomizer().getActions(getProject(), Arrays.asList(CommonProjectActions.forType(projectType)));            
            projectType = active.getProjectCustomizer().getCustomizerId();                        
        }        
        projectActions = projectActions == null ? CommonProjectActions.forType(projectType) : projectActions;
        if (brokenProject) {
            actions.add(CommonProjectActions.closeProjectAction());
            Action resolveAction = Actions.forID("Project", "org.netbeans.modules.project.ui.problems.BrokenProjectActionFactory"); //NOI18N
            for(Action action : projectActions) {
                if (action == resolveAction) {
                    actions.add(action);
                }
            }
            return actions.toArray(new Action[actions.size()]);        
        }
        actions.addAll(Arrays.asList(projectActions));
        actions.add(null);
        return actions.toArray(new Action[actions.size()]);        
    }

    @Override
    public boolean canRename() {
        return false;
    }

    @Override
    public PasteType getDropType(Transferable transferable, int action, int index) {
        DataFlavor[] flavors = transferable.getTransferDataFlavors();
        for (int i = 0; i < flavors.length; i++) {
            if (flavors[i].getSubType().equals(MakeLogicalViewProvider.SUBTYPE)) {
                return super.getDropType(transferable, action, index);
            }
        }
        return null;
    }

    @Override
    protected void createPasteTypes(Transferable transferable, List<PasteType> list) {
        DataFlavor[] flavors = transferable.getTransferDataFlavors();
        for (int i = 0; i < flavors.length; i++) {
            if (flavors[i].getSubType().equals(MakeLogicalViewProvider.SUBTYPE)) {
                try {
                    ViewItemNode viewItemNode = (ViewItemNode) transferable.getTransferData(flavors[i]);
                    int type = new Integer(flavors[i].getParameter(MakeLogicalViewProvider.MASK));
                    list.add(new ViewItemPasteType(this.getFolder(), viewItemNode, type, provider));
                } catch (Exception e) {
                }
            }
        }
        super.createPasteTypes(transferable, list);
    }
   

    @Override
    public void resultChanged(LookupEvent ev) {
        for (BrokenIncludes elem : brokenIncludesResult.allInstances()) {
            elem.addChangeListener(this);
        }
    }

    private boolean hasBrokenIncludes(Project project) {
        BrokenIncludes biProvider = Lookup.getDefault().lookup(BrokenIncludes.class);
        if (biProvider != null) {
            NativeProject id = project.getLookup().lookup(NativeProject.class);
            if (id != null) {
                return biProvider.isBroken(id);
            }
        }
        return false;
    }

    private final static class ProjectRootChildren extends LogicalViewChildren {
        private MakeLogicalViewRootNode parent;
        private ProjectRootChildren(Folder folder, MakeLogicalViewProvider provider) {
            super(folder, provider);
        }

        @Override
        protected void onFolderChange(Folder folder) {
            assert parent != null;
            this.parent.setRealProjectFolder(folder);
        }
        
        private void setMakeLogicalViewRootNode(MakeLogicalViewRootNode parent) {
            assert this.parent == null;
            this.parent = parent;
        }

        @Override
        protected boolean isRoot() {
            return true;
        }
    }

    private final class StateChangeRunnableImpl implements Runnable {

        @Override
        public void run() {
            brokenIncludes = hasBrokenIncludes(getProject());
            if (provider.gotMakeConfigurationDescriptor()) {
                MakeConfigurationDescriptor makeConfigurationDescriptor = provider.getMakeConfigurationDescriptor();
                if (makeConfigurationDescriptor != null) {
                    brokenProject = makeConfigurationDescriptor.getState() == State.BROKEN;
                    if (makeConfigurationDescriptor.getConfs().size() == 0) {
                        brokenProject = true;
                    }
                    if (provider.isIncorectVersion()) {
                        brokenProject = true;
                    }
                }
            }
            updateAnnotationFiles();
            EventQueue.invokeLater(new VisualUpdater()); // IZ 151257
        }
    }
}
