/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.server.uihandler;

import java.io.IOException;
import java.io.InputStream;
import java.security.GeneralSecurityException;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.TreeMap;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;
import org.netbeans.lib.uihandler.PasswdEncryption;
import org.netbeans.modules.exceptions.bugs.BugReporter;
import org.netbeans.modules.exceptions.bugs.BugReporterFactory;
import org.netbeans.modules.exceptions.utils.PersistenceUtils;
import org.netbeans.server.componentsmatch.Component;
import org.netbeans.server.componentsmatch.Matcher;
import org.netbeans.modules.exceptions.entity.*;
import org.netbeans.modules.exceptions.utils.Components;
import org.netbeans.modules.exceptions.utils.HttpUtils;
import org.openide.util.RequestProcessor;
import org.openide.util.Task;
import static org.netbeans.server.uihandler.Utils.*;
/**
 *
 * @author Jindrich Sedek
 */

final class DbInsertion{
    private static final int MAXMESSAGELENGTH = 1024;
    private static final int MAXSUMMARY = 200;
    private static final int MAXCOMMENT = 1000;
    private static final Semaphore inserting = new Semaphore(1);
    private static final String GUEST_USER = "GUEST";// NOI18N
    private static final String UNKNOWN = "unknown";// NOI18N
    private static final String UNKNOWN_COMPONENT = "ide";// NOI18N
    private static final String UNKNOWN_SUB_COMPONENT = "code";// NOI18N
    private static PrivateKey privateKey;
    private static int IssueMax = 0;
    private static int StackTraceMax = 0;
    private static int MethodMax = 0;
    private static List<IssueZillaInsertionFilter> filters;
    
    private static final Logger logger = Logger.getLogger(DbInsertion.class.getName());
    private final LogRecord userMetaData;
    private final Throwable thrown;
    private final String userId;
    private final int sessionNumber;
    private int issueId;
    // parameter indexes
    private static final int os_idx = 0;
    private static final int vm_idx = 1;
    private static final int version_idx = 2;
    private static final int user_name_idx = 3;
    private static final int summary_idx = 4;
    private static final int comment_idx = 5;
    private static final int passwd_idx = 6;
    
    private final PersistenceUtils persUtil;
    private org.netbeans.modules.exceptions.entity.Exceptions duplicate;
    
    private static final RequestProcessor INSERT = new RequestProcessor("DbInsertion");
    private static volatile Task lastLogFile = Task.EMPTY;
    private static volatile Task lastIssue = Task.EMPTY;
    private static Long duplicatesNo;
    
    public DbInsertion(LogRecord _data, Throwable _thrown, String _userId, int _sessionNumber) {
        thrown = _thrown;
        userMetaData = _data;
        userId = _userId;
        sessionNumber = _sessionNumber;
        persUtil = Utils.getPersistenceUtils();
    }
    
    /** issueId is an ID of newly created issue if not duplicate
     *  if duplicate issueId is an ID of issue to be duplicate of
     *  issuezillaId is 0 if not assigned, otherwise an issuezilla Id
     * @return {int issueId, bool isDuplicate, int issuezillaId} or null if it's not an error report
     *
     */
    public Object[] start(){
        boolean dupl = false;
        int issuezillaId = 0;
        if ((userMetaData == null)){
            logger.warning("NO DATA TO INSERT");    // NOI18N
            return null;//not an error report
        }
        if (thrown != null){
            try{
                // wait till all new insertions are done, if it takes long time,
                // go through - don't block communication and face a possible mistake
                // in searching duplicates
                inserting.tryAcquire(10, TimeUnit.SECONDS);
                duplicate = Utils.checkIssue(thrown);
                issueId = getIssueNext();
                if (duplicate != null){
                    dupl = true;
                    inserting.release();// release for duplicates
                    if (duplicate.getIssuezillaid() != null) {
                        issuezillaId = duplicate.getIssuezillaid();
                    }
                }
                lastLogFile = INSERT.post(new Insert(dupl, this));
            }catch(Exception exception){
                logger.log(Level.SEVERE, "CHECKING FAILED", exception);
                inserting.release();//release semaphore on exception
            }
            if (dupl) return new Object[]{duplicate.getId(), dupl, issuezillaId};
            else return new Object[]{issueId, dupl, 0};
        }else{// only a log file insertion
            lastLogFile = INSERT.post(new Insert(false, this));
            return null;
        }
    }
    
    static void waitLogFileInsertFinished() {
        lastLogFile.waitFinished();
    }
    
    static void waitIssueInsertFinished(){
        lastIssue.waitFinished();
    }
    
    private static class Insert implements Runnable {
        private DbInsertion db;
        boolean dup;
        boolean firstRun = true;
        Logfile logfile = null;

        public Insert(boolean duplicate, DbInsertion db){
            dup = duplicate;
            this.db = db;
        }
        
        public void run() {
            if (firstRun){// insert just logFile
                logger.info("INSERTING LOGFILE INTO DB");   // NOI18N
                firstRun = false;
                logfile = db.insertLogFile();
                lastIssue = INSERT.post(this);// calls insertion of ISSUE INTO DB
            }else if (db.thrown != null){
                logger.fine("INSERTING INTO DB "+Integer.toString(db.issueId));   // NOI18N
                db.insertIssue(logfile);
                logger.fine("TRANSACTION COMMITED");  // NOI18N
                if (!dup){
                    inserting.release();// release for new issue
                }
                // clear data
                db = null;
            }
        }
    }
    
    private String[] getParams(){
        Object[] oParams = null;
        oParams = userMetaData.getParameters();
        String [] params = new String[oParams.length];
        for (int i = 0; i < oParams.length; i++) {
            params[i] = oParams[i].toString();
        }
        if (params[user_name_idx].length() == 0) params[user_name_idx] = GUEST_USER;
        if (params.length > 4){
            if (params[summary_idx].length() > MAXSUMMARY) params[summary_idx] = params[summary_idx].substring(0, MAXSUMMARY);
            if (params[comment_idx].length() > MAXCOMMENT) params[comment_idx] = params[comment_idx].substring(0, MAXCOMMENT);
        }
        return params;
    }
    
    private Logfile insertLogFile(){
        String buildNumber = null;
        ProductVersion productVersion = null;
        String[] params = getParams();
        
        if (params[version_idx] != null){
            productVersion =  getExistProductVersion(params[version_idx]);
            buildNumber = org.netbeans.modules.exceptions.utils.Utils.getBuildNumber(params[version_idx]);
            buildNumber = org.netbeans.modules.exceptions.utils.Utils.getCustomBuildFormat(buildNumber);
        } else {
            List<ProductVersion> list = persUtil.executeQuery("SELECT p FROM ProductVersion p WHERE p.productVersion is null", null);
            if (list.size() != 0) productVersion = list.get(0);
        }
        if (productVersion == null){
            productVersion = new ProductVersion();
            productVersion.setProductVersion(params[version_idx]);
            persUtil.persist(productVersion);
        }
        
        Logfile logFile = LogsManager.getDefault().getLogInfo(userId, sessionNumber, false);
        logFile.setProductVersionId(productVersion);
        if ((buildNumber!= null) && (buildNumber.length() > 0)) logFile.setBuildnumber(Long.valueOf(buildNumber));
        persUtil.merge(logFile);
        return logFile;
    }
    
    private Nbuser createUser(String userName){
        Nbuser nbuser = new Nbuser();
        nbuser.setName(userName);
        persUtil.persist(nbuser);
        return nbuser;
    }
    
    protected static Component getComponentFromComment(String comment) {
        int startIndex = comment.indexOf("[");
        int slashIndex = comment.indexOf("/");
        int endIndex = comment.indexOf("]");
        int newLineIndex = comment.indexOf("\n");
        if (newLineIndex == -1) newLineIndex = comment.length();
        if ((startIndex == 0)&&(startIndex < slashIndex)&&
            (slashIndex < endIndex)&&(endIndex < newLineIndex)){
            String comp = comment.substring(1, slashIndex);
            String subcomp = comment.substring(slashIndex+1, endIndex);
            
            Components realComps = Utils.getPersistenceUtils().getComponents();
            realComps.waitInitDone();
            if (realComps.getComponentsSet().contains(comp) && realComps.getSubcomponentsSet(comp).contains(subcomp)){
                return new Component(comp, subcomp, 0);
            }
        }
        return null;
    }

    private void insertIssue(Logfile logFile){
        org.netbeans.modules.exceptions.entity.Exceptions exc = new org.netbeans.modules.exceptions.entity.Exceptions();
        Nbuser nbuser;
        Comment comment;
        Hashcodes hashcode;
        
        // adding stacktrace
        Stacktrace stacktrace = addStackTrace(thrown);
        String[] params = getParams();
        String passwd = "";
        String userName = params[user_name_idx];
        if (params.length > passwd_idx) passwd = params[passwd_idx];
        if (!GUEST_USER.equals(userName)) {
            if (!HttpUtils.checkIssuezillaLogin(userName, passwd)) {//uncrypted password
                if (passwd.length() > 20) {
                    try {
                        passwd = PasswdEncryption.decrypt(passwd, privateKey);
                    } catch (IOException exception) {
                        logger.log(Level.SEVERE, "DECRIPTING PASSWORD", exception);
                    } catch (GeneralSecurityException exception) {
                        logger.log(Level.SEVERE, "DECRIPTING PASSWORD", exception);
                    }
                }else {
                     userName = GUEST_USER;
                }
            }
        }
        nbuser = getExistUser(userName);
        if (nbuser == null){//Adding NbUser into DB
            nbuser = createUser(userName);
        }
        // get component from comment
        Component comp = getComponentFromComment(params[comment_idx]);
        // use unknown component for STACK OVERFLOW ISSUE 112156
        if ((comp == null)&&(!thrown.getMessage().contains("StackOverflowError"))){// NOI18N
            comp = Matcher.getDefault().match(thrown.getStackTrace());
        }
        String component=UNKNOWN_COMPONENT, subComponent=UNKNOWN_SUB_COMPONENT;
        if (comp != null){
            component = comp.getComponent();
            subComponent = comp.getSubComponent();
        }
        exc.setId(issueId);
        exc.setSummary(params[summary_idx]);
        exc.setComponentAndSubcomponent(component, subComponent);
        exc.setStacktrace(stacktrace);
        exc.setLogfileId(logFile);
        exc.setLoggername(logFile.getFileName());
        exc.setVm(params[vm_idx]);
        exc.setProductversion(params[version_idx]);
        exc.setOperatingsystem(params[os_idx]);
        exc.setBuild(logFile.getBuildnumber());
        exc.setNbuserId(nbuser);
        if (duplicate != null) exc.setDuplicateof(duplicate);
        exc.setReportdate(new Date());
        persUtil.persist(exc);
        // adding comment
        comment = new Comment();
        comment.setCommentDate(new Date());
        comment.setExceptionId(exc);
        comment.setNbuserId(nbuser);
        comment.setComment(params[comment_idx]);
        persUtil.persist(comment);
        exc.setCommentCollection(Collections.singletonList(comment));
        persUtil.merge(exc);
        //add hashCode
        hashcode = new Hashcodes();
        hashcode.setCode(countHash(thrown.getStackTrace()));
        hashcode.setExceptions(exc);
        hashcode.setExceptionid(issueId);
        persUtil.persist(hashcode);
        if (GUEST_USER.equals(exc.getNbuserId().getName())){
            return;
        }
        BugReporter reporter = BugReporterFactory.getDefaultReporter();
        Properties props = new Properties();
        props.setProperty("username", userName); // NOI18N
        props.setProperty("password", passwd); // NOI18N
        for (IssueZillaInsertionFilter filter : getFilters()) {
            switch (filter.isInsertable(exc)) {
                case INSERT:
                    reporter.reportException(exc, props);
                    return;
                case CANCEL:
                    return;
                case ADDCOMMENT:
                    String addComment = filter.addComment(exc);
                    reporter.postTextComment(PersistenceUtils.getOriginal(exc).getIssuezillaid(), props, addComment);
                    return;
                default: // continue with next filter
            }
        }
    }
    
    private Stacktrace addStackTrace(Throwable thrown){
        Stacktrace stacktrace = new Stacktrace();
        Line line;
        LinePK linePK;
        Method method;
        Jarfile jarFile;
        
        int stackTraceId = getStackTraceNext();
        Stacktrace cause = null;
        
        if (thrown.getCause()!= null) cause = addStackTrace(thrown.getCause());//add recursively
        stacktrace.setId(stackTraceId);
        String message = thrown.getMessage();
        String stackClass = message;
        if (message!= null){
            if (message.length() > MAXMESSAGELENGTH) message = message.substring(0, MAXMESSAGELENGTH);
            stacktrace.setMessage(message);
            int index = message.indexOf(':');
            if (index != -1 ) stackClass = message.substring(0, index);
            stacktrace.setClass1(stackClass);
        }
        if (cause != null){
            stacktrace.setAnnotation(cause);
        }
        //add stack trace lines
        persUtil.persist(stacktrace);
        
        Integer lineNumber;
        String linesConcatenation = null;
        String methodName, jarFileName;
        Collection<Line> lineCollection = new ArrayList<Line>(thrown.getStackTrace().length);
        for (int i = 0; i < thrown.getStackTrace().length; i++) {
            
            StackTraceElement element = thrown.getStackTrace()[i];
            methodName = lineConcat(element);
            jarFileName = element.getFileName();
            if (linesConcatenation == null){
                linesConcatenation = methodName;
            }else{
                linesConcatenation = linesConcatenation.concat(methodName);
            }
            method = getExistingMethod(methodName);
            if (method == null){//adding method into table
                method = new Method();
                method.setId(getMethodNext());
                method.setName(methodName);
                persUtil.persist(method);
            }
            if (jarFileName == null){
                jarFileName = UNKNOWN;
            }
            jarFile = getExistingFile(jarFileName);
            if (jarFile == null){
                jarFile = new Jarfile();
                jarFile.setName(jarFileName);
                persUtil.persist(jarFile);
            }
            if (element.getLineNumber()>=0) lineNumber = element.getLineNumber();
            else lineNumber = 0;
            line = new Line();
            linePK = new LinePK();
            linePK.setLinenumber(lineNumber);
            linePK.setMethodId(method.getId());
            linePK.setStacktraceId(stackTraceId);
            linePK.setLineOrder(i);
            line.setStacktrace(stacktrace);
            line.setMethod(method);
            line.setLineHashcode(linesConcatenation.hashCode());
            lineCollection.add(line);
            line.setJarfileId(jarFile);
            line.setLinePK(linePK);
            line.setGeneratecomponent(0);
            persUtil.persist(line);
        }
        stacktrace.setLineCollection(lineCollection);
        // FIX-ME INSTEAD OF INSERT + MERGE SHOULD BE USED ONE TRANSACTION
        persUtil.merge(stacktrace);
        return stacktrace;
    }

    public static String lineConcat(StackTraceElement element){
        return element.getClassName().concat(".").concat(element.getMethodName());// NOI18N
    }

    public static PrivateKey loadPrivateKey(InputStream inputStream){
        if (privateKey == null){
            try{
                byte[] encodedKey = new byte[inputStream.available()];
                inputStream.read(encodedKey);
                PKCS8EncodedKeySpec privateKeySpec = new PKCS8EncodedKeySpec(encodedKey);
                KeyFactory kf = KeyFactory.getInstance("RSA");// NOI18N
                privateKey = kf.generatePrivate(privateKeySpec);
            } catch (IOException exc) {
                logger.log(Level.SEVERE, "GETTING PRIVATE KEY", exc);// NOI18N
            } catch (GeneralSecurityException exc) {
                logger.log(Level.SEVERE, "GETTING PRIVATE KEY", exc);// NOI18N
            }
        }
        return privateKey;
    }
    
    private Jarfile getExistingFile(String fileName){
        return (Jarfile) getExist("Jarfile.findByName", fileName);// NOI18N
    }
    
    private Method getExistingMethod(String methodName){
        return (Method) getExist("Method.findByName", methodName);// NOI18N
    }
    
    private Nbuser getExistUser(String userName){
        return (Nbuser) getExist("Nbuser.findByName", userName);// NOI18N
    }
    
    private ProductVersion getExistProductVersion(String versionName){
        return (ProductVersion) getExist("ProductVersion.findByProductVersion", versionName);// NOI18N
    }
    
    private Object getExist(String queryName, String subjectName){
        TreeMap<String, Object> params = new TreeMap<String, Object>();
        params.put("name", subjectName);// NOI18N
        List result = persUtil.executeNamedQuery(queryName, params);
        if (result.size()==0) return null;
        return result.get(0);
    }
    
    private synchronized Integer getMethodNext(){
        if (MethodMax == 0) MethodMax = maxId(Method.class);
        return ++MethodMax;
    }
    
    private synchronized Integer getStackTraceNext(){
        if (StackTraceMax == 0) StackTraceMax = maxId(Stacktrace.class);
        return ++StackTraceMax;
    }
    
    private synchronized Integer getIssueNext(){
        if (IssueMax == 0) IssueMax = maxId(org.netbeans.modules.exceptions.entity.Exceptions.class);
        return ++IssueMax;
    }

    private int maxId(Class entity){
        return persUtil.max(entity, "id");// NOI18N
    }

    private static Long getDuplicatesNo(){
            if (duplicatesNo==null){
                duplicatesNo = Utils.getVariable("duplicatesNo", Long.class);// NOI18N
            }
            return duplicatesNo;
    }
    
    private static List<IssueZillaInsertionFilter> getFilters(){
        if (filters == null){
            synchronized(DbInsertion.class){
                filters = new ArrayList<IssueZillaInsertionFilter>(6);
                filters.add(new IssueClosedFilter());// first priority
                filters.add(new ManyDuplicatesFromOneUserFilter());
                filters.add(new QEUserFilter());
                filters.add(new NullPointerExceptionFilter());
                filters.add(new DuplicatesNoFilter());
                filters.add(new ManyDuplicatesFilter());
            }
        }
        return filters;
    }
    
    protected static void setDuplicatesNo(Long duplicates){
        duplicatesNo = duplicates;
    }
    
    private interface IssueZillaInsertionFilter {
        public InsertionResult isInsertable(org.netbeans.modules.exceptions.entity.Exceptions exc);
        public String addComment(org.netbeans.modules.exceptions.entity.Exceptions exc);
    }
    private enum InsertionResult{
        INSERT, ADDCOMMENT, CANCEL, NONE
    }

    private static class DuplicatesNoFilter implements IssueZillaInsertionFilter{

        public InsertionResult isInsertable(org.netbeans.modules.exceptions.entity.Exceptions exc) {
            org.netbeans.modules.exceptions.entity.Exceptions dupl = exc.getDuplicateof();
            if ((dupl != null)&&(dupl.getDuplicatesFromDistinctUsers() > getDuplicatesNo())&&
                    (dupl.getDuplicatesFromDistinctUsers() < getDuplicatesNo() + 4)){
                return InsertionResult.INSERT;
            }
            return InsertionResult.NONE;
        }

        public String addComment(org.netbeans.modules.exceptions.entity.Exceptions exc){
            return null;
        }
    }
    
    private static class NullPointerExceptionFilter implements IssueZillaInsertionFilter{

        public InsertionResult isInsertable(org.netbeans.modules.exceptions.entity.Exceptions exc) {
            if (exc.getStacktrace().getClass1().contains("NullPointerException")) return InsertionResult.INSERT;// NOI18N
            return InsertionResult.NONE;
        }

        public String addComment(org.netbeans.modules.exceptions.entity.Exceptions exc){
            return null;
        }
    }
    
    private static class QEUserFilter implements IssueZillaInsertionFilter{

        public InsertionResult isInsertable(org.netbeans.modules.exceptions.entity.Exceptions exc) {
            Object object = exc.getNbuserId().getName();
            Map<String, Object> params = Collections.singletonMap("name", object);
            List<Directuser> directUsers = getPersistenceUtils().executeNamedQuery("Directuser.findByName", params);
            if (directUsers.size() > 0 ) return InsertionResult.INSERT;
            else return InsertionResult.NONE;
        }

        public String addComment(org.netbeans.modules.exceptions.entity.Exceptions exc){
            return null;
        }

    }
    
    private static class IssueClosedFilter implements IssueZillaInsertionFilter{

        public InsertionResult isInsertable(org.netbeans.modules.exceptions.entity.Exceptions exc) {
            Integer issueId = PersistenceUtils.getOriginal(exc).getIssuezillaid();
            if ((issueId != null)&&(getPersistenceUtils().issueIsFixed(issueId))) {
                // Issue is already closed so don't add more comments
                logger.log(Level.FINER, "Issue " + issueId + "is closed");
                return DbInsertion.InsertionResult.CANCEL;             
            }
            return DbInsertion.InsertionResult.NONE;
        }

        public String addComment(org.netbeans.modules.exceptions.entity.Exceptions exc){
            return null;
        }
    }
    
    private static class ManyDuplicatesFilter implements IssueZillaInsertionFilter{

        public InsertionResult isInsertable(org.netbeans.modules.exceptions.entity.Exceptions exc) {
            org.netbeans.modules.exceptions.entity.Exceptions original = PersistenceUtils.getOriginal(exc);
            if (original.getIssuezillaid() == null || original.getIssuezillaid() == 0) return DbInsertion.InsertionResult.NONE;
            int duplicatesCount = original.getDuplicates();
            if (duplicatesCount % getDuplicatesNo() == 0){
                return DbInsertion.InsertionResult.ADDCOMMENT;
            }
            return DbInsertion.InsertionResult.NONE;
        }
        
        public String addComment(org.netbeans.modules.exceptions.entity.Exceptions exc){
            return "THIS ISSUE HAS "+ getDuplicatesNo() +" MORE DUPLICATES"; // NOI18N
        }
    }
    
    private static class ManyDuplicatesFromOneUserFilter implements IssueZillaInsertionFilter{

        public InsertionResult isInsertable(org.netbeans.modules.exceptions.entity.Exceptions exc) {
            Nbuser user = exc.getNbuserId();
            int duplicates = PersistenceUtils.getOriginal(exc).getDuplicatesFromOneUser(user);
            if (duplicates > 2) return DbInsertion.InsertionResult.CANCEL;
            return DbInsertion.InsertionResult.NONE;
        }

        public String addComment(org.netbeans.modules.exceptions.entity.Exceptions exc){
            return null;
        }
    }

    
}
