/*
 * Copyright (c) 2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Compiler.Typedtree;
using Nemerle.Collections;
using Nemerle.Compiler.SolverMacros;

namespace Nemerle.Compiler 
{
  /** Represents a single possibility in the overloading resulution process.
      It is used in the TExpr.Overloaded node.  */
  public class OverloadPossibility : Located
  {
    /** Tell if this overload is still possible in the current solver.  */
    public StillPossible : bool
    {
      get {
        def solver = Passes.Solver;
        solver.PushState ();
        def was_error = Compile () is TExpr.Error;
        solver.PopState ();
        !was_error
      }
    }


    public SetGenericSpecifier (tyvars : list [TyVar]) : void
    {
      assert (generic_specifier == null || generic_specifier : object == tyvars);
      generic_specifier = tyvars;
    }


    CheckGenericSpecifier () : bool
    {
      if (generic_specifier == null) true
      else {
        def method_typarms =
          match (member) {
            | _ is TypeInfo // implicit value type ctor
            | meth is IMethod when meth.GetFunKind () is FunKind.Constructor =>
              from.args
            | _ =>
              method_typarms
          }
        method_typarms.Length == generic_specifier.Length &&
        List.ForAll2 (method_typarms, generic_specifier, 
                      fun (a, b) { a.Unify (b) })
      }
    }
    

    /** Enforce this particular overloading possibility in the current
        solver.  */
    public Compile () : TExpr
    {
      Util.locate (loc, {
        def is_ok = 
          CheckGenericSpecifier () &&
          (is_static || {
            def ti = member.DeclaringType;
            def ft = ti.GetFreshType ();
            expr.Type.Require (ft)
          });

        def expr =
          match (member) {
            | _ when !is_ok =>
              // XXX see when this happens and invent some better
              // error message
              ReportError (Passes.Solver.CurrentMessenger,
                           $ "  $(member) is no longer present in $(ty) "
                             "after constraining");
              TExpr.Error ()

            | ti is TypeInfo =>
              assert (ti.IsValueType);
              TExpr.ImplicitValueTypeCtor ()
              
            | fld is IField =>
              if (fld.IsStatic)
                if (IsConstantObject)
                  TExpr.ConstantObjectRef (from, fld)
                else if (fld.IsLiteral)
                  match (ConstantFolder.FieldValueAsLiteral (fld)) {
                    | None => TExpr.StaticRef (from, from, fld, [])
                    | Some (lit) => TExpr.Literal (fld.GetMemType (), lit)
                  }
                else
                  TExpr.StaticRef (from, fld, [])
              else
                TExpr.FieldMember (expr, fld)
                
            | prop is IProperty =>
              if (prop.IsStatic)
                TExpr.StaticPropertyRef (from, prop)
              else
                TExpr.PropertyMember (expr, prop)

            | meth is IMethod =>
              if (IsBaseCall)
                TExpr.Base (meth)
              else if (is_static)
                match (meth.BuiltinKind) {
                  | OpCode (ch, unch) =>
                    TExpr.OpCode (if (local_context.IsChecked) ch else unch)
                  | _ =>
                    TExpr.StaticRef (from, meth, method_typarms)
                }
              else
                TExpr.MethodRef (expr, meth, method_typarms, notvirtual = expr is TExpr.This && !(meth.Attributes %&& NemerleAttributes.Virtual))

            | ev is IEvent =>
              if (ev.IsStatic)
                TExpr.StaticEventRef (from, ev)
              else
                TExpr.EventMember (expr, ev)

            | _ =>
              Message.Warning ($ "evil member $(member)");
              assert (false)
          }
          
        expr.ty = ty;
        expr
      })
    }


    public Type : TyVar
    {
      get { ty }
    }


    public Member : IMember
    {
      get { member }
    }

    public override ToString () : string
    {
      member.ToString () +
      if (VarArgs) " [parms]" else ""
    }
    

    public this (typer : Typer, ty : TyVar, expr : TExpr, from : MType.Class, 
                 member : IMember, method_typarms : list [TyVar] = null)
    {
      if (member.IsStatic)
        is_static = true
      else
        match (member) {
          | meth is IMethod => 
            is_static =
              meth.GetFunKind () is FunKind.Constructor
          | _ => {}
        }

      /*assert ((expr == null) == is_static, 
              $ "expr/IsStatic $member $(member.IsStatic) $is_static");

      assert ((from == null) != is_static, 
              $ "from/IsStatic $member $(member.IsStatic) $is_static");*/

      this.expr = expr;
      this.from = from;
      this.member = member;
      this.solver = Passes.Solver;
      this.ty = ty;
      this.local_context = typer.GetLocals ();

      if (method_typarms == null)
        this.method_typarms = []
      else
        this.method_typarms = method_typarms;
    }


    [Nemerle.OverrideObjectEquals]
    public Equals (o : OverloadPossibility) : bool
    {
      member.Equals (o.member) &&
      this.expr : object == o.expr : object &&
      o.VarArgs == VarArgs
    }
    
  
    expr : TExpr;
    ty : TyVar;
    member : IMember;
    solver : Solver;
    is_static : bool;
    internal from : MType.Class;
    method_typarms : list [TyVar];
    local_context : LocalContext;
    mutable generic_specifier : list [TyVar];
    public mutable IsBaseCall : bool;
    public mutable VarArgs : bool;
    public mutable ExtensionMethodObject : TExpr;
    public mutable IsConstantObject : bool;
    public mutable UsedDefaultParms : bool;
    public mutable UsedLastTime : bool;

    mutable permutation_array : array [int];
    mutable formal_types : array [TyVar];
    mutable did_mambo_jumbo : bool;

    public ResetOverloadSelectionStuff () : void
    {
      permutation_array = null;
      formal_types = null;
      UsedDefaultParms = false;
      did_mambo_jumbo = false;
    }
    
    public PermutationArray : array [int]
    {
      set {
        permutation_array = value;
      }
    }

    public DidMamboJumbo : bool
    {
      get {
        _ = FormalTypes;
        did_mambo_jumbo
      }
    }

    public IsGeneric : bool
    {
      get {
        if ((from == null || from.args.IsEmpty) &&
            method_typarms.IsEmpty)
          false
        else
          true
      }
    }

    public FormalTypes : array [TyVar]
    {
      get {
        Util.cassert (formal_types != null || permutation_array != null);

        when (formal_types == null && ty.IsFixed) {
          did_mambo_jumbo = false;
          match (ty.FixedValue) {
            | MType.Fun (from, _) =>
              mutable formals =
                match (from.Fix ()) {
                  | MType.Tuple (lst) => lst
                  | MType.Void => []
                  | t => [t]
                }
              def res = array (permutation_array.Length);

              if (VarArgs) {
                def loop (acc, n, formals) {
                  if (n == 0) acc.Rev ()
                  else
                    match (formals) {
                      | [last : TyVar] =>
                        match (last.Fix ()) {
                          | MType.Array (t, 1) =>
                            loop (t :: acc, n - 1, formals)
                          | _ => Util.ice ()
                        }
                      | x :: xs =>
                        loop (x :: acc, n - 1, xs)
                      | [] => Util.ice ()
                    }
                }
                formals = loop ([], res.Length, formals)
              } else if (UsedDefaultParms) {
                // don't do anything
              } else {
                when (res.Length != formals.Length) {
                  did_mambo_jumbo = true;
                  match (res.Length) {
                    | 0 =>
                      formals = []
                    | 1 =>
                      formals = [MType.Tuple (formals)]
                    | n =>
                      // I hope this can only happen for formals = [object]
                      formals = List.Repeat (InternalType.Object, n)
                  }
                }
              }

              mutable pos = 0;
              foreach (t in formals) {
                when (permutation_array [pos] != -1)
                  res [permutation_array [pos]] = t;
                pos++;
              }

              formal_types = res;
            | _ => {}
          }
        }
        formal_types
      }

      // used for fake delegate constructors
      set {
        formal_types = value;
      }
    }

    static public Unique (overloads : list [OverloadPossibility]) : list [OverloadPossibility]
    {
      def ht = Hashtable ();
      mutable res = [];
      
      foreach (overload in overloads) {
        def id = 2 * overload.Member.GetHashCode () + if (overload.VarArgs) 1 else 0;
        if (ht.Contains (id)) {
          assert (overload.Equals (ht [id]))
        } else {
          ht [id] = overload;
          res = overload :: res;
        }
      }

      res
    }
    

    /** Filter out impossible overloads from the list.  If there are no 
        possible overloads and we are in the error reporting mode, present
        an appropriate error message.  */
    static public OnlyPossible (overloads : list [OverloadPossibility], expected : TyVar) 
                                          : list [OverloadPossibility]
    {
      assert (!overloads.IsEmpty);

      def solver = Passes.Solver;
      def res = List.RevFilter (overloads, 
                                fun (o : OverloadPossibility) { 
                                  solver.PushState ();
                                  def expr = o.Compile ();
                                  def was_error = 
                                    expr is TExpr.Error ||
                                    (expected != null && 
                                     !expected.Unify (expr.Type));
                                  solver.PopState ();
                                  !was_error
                                });
      
      when (res.IsEmpty) {
        def hd = List.Hd (overloads);
        def solver = Passes.Solver;
        def messenger = solver.CurrentMessenger;
        if (overloads.ForAll (fun (o) { !o.IsGeneric && o.generic_specifier != null }))
          ReportError (messenger,
                       $ "$(hd.member.DeclaringType).$(hd.member.Name) doesn't take generic parameters");
        else {
          ReportError (messenger,
                       $ "none of the overloads of $(hd.member.Name) is "
                         "possible:");
          foreach (overload in overloads) {
            solver.PushState ();
            messenger.NeedMessage = true;
            def res = overload.Compile ();
            unless (res is TExpr.Error) {
              if (expected.Unify (res.Type))
                assert (false)
              else
                ReportError (messenger, 
                             $ "  $(overload.Member) was expected to have "
                               "type $expected, while it has got $(res.Type)");
            }
            solver.PopState ();
          }
        }
      }
      
      res
    }

  }
}
