/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Compiler;
using Nemerle.Collections;
using Nemerle.Compiler.Parsetree;
using SCG = System.Collections.Generic;

namespace Nemerle.Compiler
{
  class PreParserException : System.Exception {
    public Location : Location;

    public this (loc : Location, msg : string) {
      base (msg);
      this.Location = loc;
    }
  }

  /** Transforms stream of tokens from given LexerBase to token tree
      with matched brackets.
   */
  public class PreParser
  {
    protected lexer : LexerBase;
    protected mutable last_token : Token = null;
    mutable Env : GlobalEnv;

    mutable finished : bool = false;
    
    /** Parent stream is the stack of processed token nodes,
        which are already assigned to be in currently build sequence.
        For example:
          a; b; c (); d e _we_are_here_
        'a, b, c()', are alredy known to be in parent sequence,
        while 'd e' are in current temporary sequence, which might
        get added to parent_stream if separator (e.g. ';') occurs
     */
    parent_stream : SCG.List [Token] = SCG.List (100);

    /** Currently builded stream of token nodes is an array of
        loose tokens, which have occured after last separator.
        It will probably form LooseGroup as an element of parent
        sequence or all elements will constitue parent
     */
    current_stream : SCG.List [Token] = SCG.List (50);

    internal static mutable doc_comments : Map [Location, string];

    public static Init () : void {
      if (Options.XmlDoc)
        doc_comments = NemerleMap ();
      else
        doc_comments = null;
    }
    
    public this (lex : LexerBase) {
      Env = GlobalEnv.Core;
      lex.Keywords = Env.Keywords;
      lexer = lex;
    }

    static reset_comment (tok : Token) : void {
      when (Options.XmlDoc) doc_comments = doc_comments.Replace (tok.Location, "");
    }
    static reset_comment (loc : Location) : void {
      when (Options.XmlDoc) doc_comments = doc_comments.Replace (loc, "");
    }

    /** Fetch next token (from one token buffer or lexer if it's empty */
    protected virtual get_token () : Token {
      if (last_token != null) {
        def result = last_token;
        last_token = null;
        result;
      }
      else {
        try {
          lexer.GetToken ()
        }
        catch {
          | e is LexerBase.Error =>
            Message.Error (lexer.Location, e.name + " " + e.StackTrace);
            get_token ()
        }
      }
    }

    /** Store token in our mini one token buffer */
    push_back (tok : Token) : void {
      assert (last_token == null);
      last_token = tok;
    }

    /** links Tokens from specified subarray to form a list and return its head */
    static make_list (tokens : SCG.List [Token], start : int) : Token
    {
      for (mutable i = tokens.Count - 2; i >= start; --i) 
        tokens [i].Next = tokens [i + 1];
      tokens [start]
    }

    
    public static Dump (tok : Token, ident : string) : string {
      def (open, close, sepstr, elements) =
        match (tok) {
          | Token.RoundGroup => ("(", ")", ", ", tok)
          | Token.BracesGroup => ("{\n" + ident, "}", ";\n" + ident, tok)
          | Token.SquareGroup => ("[", "]", ", ", tok)
          | Token.QuoteGroup  => ("<[\n", "]>", "; ", tok)
          | Token.LooseGroup  => ("", "", " ", tok)

          | _ => ("", tok.ToString (false), "", null)
        }
      def builder = System.Text.StringBuilder (open);
      when (elements != null)
        foreach (e is Token in elements) 
          _ = builder.Append (Dump (e, ident + "  ")).Append (sepstr);
      builder.Append (close).ToString ()
    }

    /** Closes both currently created LooseGroup and parent group.
        Returns list of tokens composing parent group */
    finish_parent (parent_begin : int, current_begin : int) : Token {
      finish_current (current_begin);
      def parent_group = 
        if (parent_begin == parent_stream.Count)
          null // case of `(` `)`
        else
          make_list (parent_stream, parent_begin);
      parent_stream.RemoveRange (parent_begin, parent_stream.Count - parent_begin);
      parent_group
    }

    /** Closes currently created LooseGroup and adds it at the end of the
        parent group. After that we are ready to make another LooseGroup.

        It is called mainly when separator token occurs.
     */   
    finish_current (current_begin : int) : void {
      unless (current_begin == current_stream.Count) {
        def loose_group = make_list (current_stream, current_begin);
        def loose = Token.LooseGroup (loose_group.Location, loose_group);
        parent_stream.Add (loose);
        current_stream.RemoveRange (current_begin, current_stream.Count - current_begin);
      }
    }

    /** Handle standard situations when new bracket group is beginning
        or there is erronous situation. Any non bracket token is
        appended to current LooseGroup.

        Throws PreParserException when there is unmatched end bracket.
     */
    handle_default_token (current_begin : int, tok : Token, braces_cut_current = true) : void {
      match (tok) {
        | Token.BeginBrace =>
          def brace_group = parse_brace_group (tok.Location);
          current_stream.Add (brace_group);
          when (braces_cut_current)
            finish_current (current_begin);

        | Token.BeginRound =>
          def round_group = parse_round_group (tok.Location);
          current_stream.Add (round_group);

        | Token.BeginSquare =>
          def square_group = parse_square_group (tok.Location);
          current_stream.Add (square_group);

        | Token.BeginQuote =>
          def quote_group = parse_quote_group (tok.Location);
          current_stream.Add (quote_group);

        | Token.EndRound | Token.EndSquare | Token.EndQuote | Token.EndBrace =>
          push_back (tok);
          throw PreParserException (tok.Location, $"unexpected closing bracket `$(tok)'");

        | Token.EndOfFile =>
          throw PreParserException (tok.Location, "unexpected end of file");

        | Token.Comment (comment) when Options.XmlDoc =>
          doc_comments = doc_comments.Replace (tok.Location, comment);

        | Token.Comment => ()
          
        | _ => current_stream.Add (tok);
      }
    }
    
    parse_brace_group (loc : Location) : Token.BracesGroup
    {
      def parent_begin = parent_stream.Count;
      def current_begin = current_stream.Count;

      reset_comment (loc);

      def loop () {
        def tok = get_token ();
        match (tok) {
          // finish entire brace group
          | Token.EndBrace =>
            reset_comment (tok);
            def brace_group = finish_parent (parent_begin, current_begin);
            Token.BracesGroup (loc + tok.Location, brace_group);

          // finish current loose group
          | Token.Semicolon => 
            reset_comment (tok);
            finish_current (current_begin); loop ()

          | Token.EndOfFile when parent_begin == 0 =>
            def brace_group = finish_parent (parent_begin, current_begin);
            finished = true;
            Token.BracesGroup (loc + tok.Location, brace_group);
            
          | _ => handle_default_token (current_begin, tok); loop ()
        }
      }
      try { loop () }
      catch { e is PreParserException =>
        Message.Error (loc, "when parsing this `{' brace group");
        Message.Error (e.Location, e.Message);
        def group = finish_parent (parent_begin, current_begin);
        Token.BracesGroup (loc, group);
      }
    }

    parse_round_group (loc : Location) : Token.RoundGroup
    {
      def parent_begin = parent_stream.Count;
      def current_begin = current_stream.Count;

      def loop () {
        def tok = get_token ();
        match (tok) {
          // finish entire round group
          | Token.EndRound =>
            def round_group = finish_parent (parent_begin, current_begin);
            Token.RoundGroup (loc + tok.Location, round_group);

          // finish current loose group
          | Token.Comma =>
            finish_current (current_begin);
            loop ()

          | _ => handle_default_token (current_begin, tok, false); loop ()
        }
      }
      try { loop () }
      catch { e is PreParserException =>
        Message.Error (loc, "when parsing this `(' brace group");
        Message.Error (e.Location, e.Message);
        def group = finish_parent (parent_begin, current_begin);
        Token.RoundGroup (loc, group);
      }
    }

    parse_square_group (loc : Location) : Token.SquareGroup
    {
      def parent_begin = parent_stream.Count;
      def current_begin = current_stream.Count;

      def loop () {
        def tok = get_token ();
        match (tok) {
          // finish entire brace group
          | Token.EndSquare =>
            def group = finish_parent (parent_begin, current_begin);
            Token.SquareGroup (loc + tok.Location, group);

          // finish current loose group
          | Token.Comma => finish_current (current_begin); loop ()

          | _ => handle_default_token (current_begin, tok, false); loop ()
        }
      }
      try { loop () }
      catch { e is PreParserException =>
        Message.Error (loc, "when parsing this `[' brace group");
        Message.Error (e.Location, e.Message);
        def group = finish_parent (parent_begin, current_begin);
        Token.SquareGroup (loc, group);
      }
    }

    parse_quote_group (loc : Location) : Token.QuoteGroup
    {
      def parent_begin = parent_stream.Count;
      def current_begin = current_stream.Count;

      def loop () {
        def tok = get_token ();
        match (tok) {
          // finish entire brace group
          | Token.EndQuote =>
            def group = finish_parent (parent_begin, current_begin);
            Token.QuoteGroup (loc + tok.Location, group);

          // finish current loose group
          | Token.Semicolon => finish_current (current_begin); loop ()

          | _ => handle_default_token (current_begin, tok); loop ()
        }
      }
      try { loop () }
      catch { e is PreParserException =>
        Message.Error (loc, "when parsing this `<[' brace group");
        Message.Error (e.Location, e.Message);
        def group = finish_parent (parent_begin, current_begin);
        Token.QuoteGroup (loc, group);
      }
    }

    ParseTopLevelImpl () : Token
    {
      def parent_begin = parent_stream.Count;
      def current_begin = current_stream.Count;

      def get_qid () {
        match (get_token ()) {
          | Token.Identifier (x) =>
            match (get_token ()) {
              | Token.Operator (".") => x :: get_qid ()
              | t => push_back (t); [x]
            }
          | t => Message.Error (t.Location, "expected qualified identifier"); []
        }
      }
      
      def loop () {
        def tok = get_token ();
        match (tok) {
          | Token.Keyword ("using") =>
            finish_current (current_begin);
            
            def id = get_qid ();
            match (get_token ()) {
              | Token.Semicolon as st =>
                def loc = tok.Location + st.Location;
                Env = Env.AddOpenNamespace (id, loc);
                lexer.Keywords = Env.Keywords;

                def using_tok = Token.Using (loc, Env);
                current_stream.Add (using_tok);
                
              | Token.Operator ("=") =>
                def id' = get_qid ();

                def st = get_token ();
                match (st) {
                  | Token.Semicolon => ()
                  | _ => Message.Error (st.Location, "expecting `;' after using alias")
                }

                match (id) {
                  | [name] => Env = Env.AddNamespaceAlias (name, id', tok.Location);
                  | _ => Message.Error (tok.Location, "using alias must be simple name without dots")
                }
             
                def using_tok = Token.Using (tok.Location + st.Location, Env);
                current_stream.Add (using_tok);
                
              | x => Message.Error (x.Location, "expecting `;' or `='")
            }
            finish_current (current_begin);
            loop ()

          | Token.Keyword ("namespace") =>
            finish_current (current_begin);
            
            def id = get_qid ();
            match (get_token ()) {
              | Token.BeginBrace as br =>
                def loc = tok.Location + br.Location;
                def oldenv = Env;
                Env = Env.EnterIntoNamespace (id);
                lexer.Keywords = Env.Keywords;
                
                def decls = ParseTopLevelImpl ();
                def namespace_tok = Token.Namespace (loc, Env, decls);                

                Env = oldenv;
                lexer.Keywords = Env.Keywords;

                current_stream.Add (namespace_tok);                  

              | x => Message.Error (x.Location, "expecting `{' opening namespace scope")
            }
            finish_current (current_begin);            
            loop ()
          
          // finish entire brace group
          | Token.EndBrace =>
            reset_comment (tok);
            finish_parent (parent_begin, current_begin);

          // finish current loose group
          | Token.Semicolon => finish_current (current_begin); loop ()

          | Token.EndOfFile when parent_begin == 0 =>
            def brace_group = finish_parent (parent_begin, current_begin);
            finished = true;
            brace_group;
            
          | _ => handle_default_token (current_begin, tok); loop ()
        }
      }
      try { loop () }
      catch { e is PreParserException =>
        Message.Error (e.Location, e.Message);
        finish_parent (parent_begin, current_begin);
      }
    }

    indention_based_copy () : PreParser
    {
      def copy = PreParserIndent (lexer);
      copy.last_token = last_token;
      copy.Env = Env;
      copy.finished = finished;
      copy
    }

    public ParseTopLevel () : Token.BracesGroup {
      try {
        def stream = ParseTopLevelImpl ();
        lexer.Dispose (); // make lexer release file handle
        Token.BracesGroup (stream)
      } catch {
        | _ is LexerBase.PragmaIndent =>
          indention_based_copy ().ParseTopLevel ()
      }
    }
    [Nemerle.Assertions.Ensures (value != null)]
    public PreParse () : Token.BracesGroup {
      try {
        def top = parse_brace_group (Location.Default);
        unless (finished)
          Message.Error (lexer.Location, "expected end of file, encountered closing brace");
        top
      } catch {
        | _ is LexerBase.PragmaIndent =>
          indention_based_copy ().PreParse ()
      }
    }
  }
}
