/*
 * Copyright (c) 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.IO;
using Nemerle.Collections;

using Nemerle.Compiler.Typedtree;

using SR = System.Reflection;

namespace Nemerle.Compiler {

public module SystemType 
{
  public mutable Array : System.Type;
  public mutable Boolean : System.Type;
  public mutable Byte : System.Type;
  public mutable Char : System.Type;
  public mutable Decimal : System.Type;
  public mutable Double : System.Type;
  public mutable Enum : System.Type;
  public mutable FlagsAttribute : System.Type;
  public mutable Int16 : System.Type;
  public mutable Int32 : System.Type;
  public mutable Int64 : System.Type;
  public mutable IntPtr : System.Type;
  public mutable Delegate : System.Type;
  public mutable MulticastDelegate : System.Type;
  public mutable Object : System.Type;
  public mutable Reflection_AssemblyConfigurationAttribute : System.Type;
  public mutable Reflection_FieldInfo : System.Type;
  public mutable Reflection_PropertyInfo : System.Type;
  public mutable Reflection_DefaultMemberAttribute : System.Type;
  public mutable Runtime_CompilerServices_IsVolatile : System.Type;
  public mutable DebuggableAttribute : System.Type;
  public mutable DebuggableAttribute_DebuggingModes : System.Type;
  public mutable CompilationRelaxationsAttribute : System.Type;
  public mutable SByte : System.Type;
  public mutable Single : System.Type;
  public mutable String : System.Type;
  public mutable Type : System.Type;
  public mutable UInt16 : System.Type;
  public mutable UInt32 : System.Type;
  public mutable UInt64 : System.Type;
  public mutable Void : System.Type;
  public mutable ParamArrayAttribute : System.Type;

  // set in LibrariesLoader upon first possiblity
  public mutable ExtensionAttribute : System.Type;
  public mutable SQ_ExtensionAttribute : System.Type;

  public mutable Decimal_ctors : Hashtable [string, SR.ConstructorInfo];
  public mutable Type_GetTypeFromHandle : SR.MethodInfo;
  public mutable AssemblyBuilder_EmbedResourceFile : SR.MethodInfo; 
  public mutable String_opEquality : SR.MethodInfo;
  public mutable String_opInequality : SR.MethodInfo;
  public mutable Decimal_opEquality : SR.MethodInfo;
  public mutable String_Concat : SR.MethodInfo;
  public mutable ObjectCtor : SR.ConstructorInfo;
  public mutable Delegate_Combine : SR.MethodInfo;
  public mutable Delegate_Remove : SR.MethodInfo;

  public NemerleAttribute : System.Type {
    mutable nemerle_attribute : System.Type;
    
    get {
      when (nemerle_attribute == null)
        InternalType.InitNemerleTypes ();

      nemerle_attribute
    }

    internal set {
      nemerle_attribute = value;
    }
  }
  
  public NullMatchException : System.Type
  {
    get { InternalType.NullMatchException_tc.SystemType }
  }
  
  public ContainsMacroAttribute : System.Type
  {
    get { InternalType.ContainsMacroAttribute_tc.SystemType }
  }
  
  public TypeAttribute : System.Type
  {
    get { InternalType.TypeAttribute_tc.SystemType }
  }
  
  public VariantAttribute : System.Type
  {
    get { InternalType.VariantAttribute_tc.SystemType }
  }
  
  public VariantOptionAttribute : System.Type
  {
    get { InternalType.VariantOptionAttribute_tc.SystemType }
  }

  public VolatileModifier : System.Type
  {
    get { InternalType.VolatileModifier_tc.SystemType }
  }
  
  public ImmutableAttribute : System.Type
  {
    get { InternalType.ImmutableAttribute_tc.SystemType }
  }
  
  public ConstantVariantOptionAttribute : System.Type
  {
    get { InternalType.ConstantVariantOptionAttribute_tc.SystemType }
  }
  
  public TypeAliasAttribute : System.Type
  {
    get { InternalType.TypeAliasAttribute_tc.SystemType }
  }

  public ExtensionPatternEncodingAttribute : System.Type
  {
    get { InternalType.ExtensionPatternEncodingAttribute_tc.SystemType }
  }
  
  /**
   * Reflects a type using NamespaceTree
   */
  public Reflect (type_name : string) : System.Type
  {
    match (NamespaceTree.LookupSystemType (type_name)) {
      | Some (t) => t
      | _ => Util.ice ("cannot reflect `" + type_name + "'")
    }
  }

  internal Init () : void
  {
    Array = Reflect ("System.Array");
    Boolean = Reflect ("System.Boolean");
    Byte = Reflect ("System.Byte");
    Char = Reflect ("System.Char");
    Decimal = Reflect ("System.Decimal");
    Double = Reflect ("System.Double");
    Enum = Reflect ("System.Enum");
    FlagsAttribute = Reflect ("System.FlagsAttribute");
    Int16 = Reflect ("System.Int16");
    Int32 = Reflect ("System.Int32");
    Int64 = Reflect ("System.Int64");
    IntPtr = Reflect ("System.IntPtr");
    Delegate = Reflect ("System.Delegate");
    MulticastDelegate = Reflect ("System.MulticastDelegate");
    Object = Reflect ("System.Object");
    Reflection_FieldInfo = Reflect ("System.Reflection.FieldInfo");
    Reflection_PropertyInfo = Reflect ("System.Reflection.PropertyInfo");
    Reflection_AssemblyConfigurationAttribute = Reflect ("System.Reflection.AssemblyConfigurationAttribute");
    Runtime_CompilerServices_IsVolatile = Reflect ("System.Runtime.CompilerServices.IsVolatile");
    DebuggableAttribute = Reflect ("System.Diagnostics.DebuggableAttribute");
    DebuggableAttribute_DebuggingModes = Reflect ("System.Diagnostics.DebuggableAttribute.DebuggingModes");
    CompilationRelaxationsAttribute = Reflect ("System.Runtime.CompilerServices.CompilationRelaxationsAttribute");
    SByte = Reflect ("System.SByte");
    Single = Reflect ("System.Single");
    String = Reflect ("System.String");
    SystemType.Type = Reflect ("System.Type");
    UInt16 = Reflect ("System.UInt16");
    UInt32 = Reflect ("System.UInt32");
    UInt64 = Reflect ("System.UInt64");
    Void = Reflect ("System.Void");
    ParamArrayAttribute = Reflect ("System.ParamArrayAttribute");
    Reflection_DefaultMemberAttribute = Reflect ("System.Reflection.DefaultMemberAttribute");

    {
      Decimal_ctors = Hashtable ();
      def decimal_ctors = SystemType.Decimal.GetConstructors ();

      foreach (decimal_ctor : SR.ConstructorInfo in decimal_ctors) {
        def parameters = decimal_ctor.GetParameters ();

        when (parameters.Length == 1) {
          def parameter_ty = parameters [0].ParameterType;
          Decimal_ctors [parameter_ty.Name] = decimal_ctor
        }
      }

      assert (Decimal_ctors.Count >=  7)
    }
    
    Type_GetTypeFromHandle = SystemType.Type.GetMethod ("GetTypeFromHandle");
    AssemblyBuilder_EmbedResourceFile = Reflect ("System.Reflection.Emit.AssemblyBuilder")
      .GetMethod ("EmbedResourceFile", SR.BindingFlags.Instance %| SR.BindingFlags.Public
                   %| BindingFlags.NonPublic, null, SR.CallingConventions.Any,
                   array [String, String], null);
    String_opEquality = SystemType.String.GetMethod ("op_Equality");
    String_opInequality = SystemType.String.GetMethod ("op_Inequality");                     
    String_Concat = String.GetMethod ("Concat", array [String, String]);
    Decimal_opEquality = SystemType.Decimal.GetMethod ("op_Equality");
    ObjectCtor = Object.GetConstructor (System.Type.EmptyTypes);
    assert (ObjectCtor != null);

    Delegate_Combine = Delegate.GetMethod ("Combine", array [Delegate, Delegate]);
    Delegate_Remove = Delegate.GetMethod ("Remove", array [Delegate, Delegate]);
  }
}

public class TupleType 
{
  internal tycon : TypeInfo;
  internal fields : array [IField];
  internal ctor : IMethod;

  public GetField (pos : int) : IField
  {
    fields [pos]
  }

  public Ctor : IMethod
  {
    get { ctor }
  }

  public TyCon : TypeInfo
  {
    get { tycon }
  }

  public static Make (ty : TyVar) : MType.Class
  {
    match (ty.Fix ()) {
      | Tuple (types) =>
        def inst = InternalType.GetTupleType (types.Length);
        MType.Class (inst.tycon, types)
      | _ => Util.ice ()
    }
  }

  get_field (pos : int) : IField
  {
    match (tycon.LookupMember (sprintf ("field%d", pos - 1))) {
      | [x] =>
        match (x.GetKind ()) {
          | MemberKind.Field (f) => f
          | _ => assert (false)
        }
      | _ => assert (false)
    }
  }

  static name = ["Nemerle", "Builtins", "Tuple"] : list [string]; 

  public static IsTupleMember (ty : IMember) : bool
  {
    def dt = ty.DeclaringType;
    def typarms = dt.TyparmsCount;
    typarms > 1 && dt.Equals (InternalType.GetTupleType (typarms).tycon)
  }
  
  internal this (size : int)
  {
    tycon = NamespaceTree.LookupInternalType (name, size);
    fields = array (size + 1);
    for (mutable i = 1; i <= size; ++i)
      fields [i] = get_field (i);
    ctor = tycon.LookupMember (".ctor").Head :> IMethod;
  }
}

public class FunctionType {
  internal tycon : TypeInfo;
  internal void_tycon : TypeInfo;
  internal apply : IMethod;
  internal apply_void : IMethod;

  public ApplyMethod : IMethod
  {
    get { apply }
  }

  public ApplyVoidMethod : IMethod
  {
    get { apply_void }
  }

  public TyCon : TypeInfo
  {
    get { tycon }
  }

  public VoidTyCon : TypeInfo
  {
    get { void_tycon }
  }

  public static Make (ty : TyVar) : MType.Class
  {
    match (ty.Fix ().FunReturnTypeAndParms ()) {
      | Some ((parms, ret)) =>
        def inst = InternalType.GetFunctionType (parms.Length);
        if (ret.Fix () is MType.Void)
          MType.Class (inst.VoidTyCon, parms)
        else
          MType.Class (inst.TyCon, parms + [ret])
      | None => Util.ice ()
    }
  }

  static function_name = ["Nemerle", "Builtins", "Function"] : list [string];
  static function_void_name = ["Nemerle", "Builtins", "FunctionVoid"] : list [string];
  
  internal this (size : int)
  {
     tycon = NamespaceTree.LookupInternalType (function_name, size + 1);
     foreach (meth :> IMethod in tycon.LookupMember ("apply")) {
       when (meth.GetParameters ().Length == size) {
         assert (apply == null);
         apply = meth
       }
     };
     assert (apply != null);

     void_tycon = NamespaceTree.LookupInternalType (function_void_name, size);
     apply_void = void_tycon.LookupMember ("apply_void").Head :> IMethod;
  }
}
  

public module InternalType
{
  public mutable Array_tc : TypeInfo;
  public mutable Attribute_tc : TypeInfo;
  public mutable Boolean_tc : TypeInfo;
  public mutable Byte_tc : TypeInfo;
  public mutable Char_tc : TypeInfo;
  public mutable Decimal_tc : TypeInfo;
  public mutable Delegate_tc : TypeInfo;
  public mutable Double_tc : TypeInfo;
  public mutable Enum_tc : TypeInfo;
  public mutable Exception_tc : TypeInfo;
  public mutable Int16_tc : TypeInfo;
  public mutable Int32_tc : TypeInfo;
  public mutable Int64_tc : TypeInfo;
  public mutable Object_tc : TypeInfo;
  public mutable SByte_tc : TypeInfo;
  public mutable Single_tc : TypeInfo;
  public mutable String_tc : TypeInfo;
  public mutable Type_tc : TypeInfo;
  public mutable UInt16_tc : TypeInfo;
  public mutable UInt32_tc : TypeInfo;
  public mutable UInt64_tc : TypeInfo;
  public mutable ValueType_tc : TypeInfo;
  public mutable MatchFailureException_tc : TypeInfo;
  public mutable NullMatchException_tc : TypeInfo;
  public mutable ContainsMacroAttribute_tc : TypeInfo;
  public mutable TypeAttribute_tc : TypeInfo;
  public mutable VariantAttribute_tc : TypeInfo;
  public mutable ImmutableAttribute_tc : TypeInfo;
  public mutable ExtensionAttribute_tc : TypeInfo;
  public mutable TypeAliasAttribute_tc : TypeInfo;
  public mutable VariantOptionAttribute_tc : TypeInfo;
  public mutable VolatileModifier_tc : TypeInfo;
  public mutable ConstantVariantOptionAttribute_tc : TypeInfo;
  public mutable ExtensionPatternEncodingAttribute_tc : TypeInfo;
  public mutable FlagsAttribute_tc : TypeInfo;
  public mutable ParamArrayAttribute_tc : TypeInfo;
  public mutable AssemblyVersionAttribute_tc : TypeInfo;
  public mutable AssemblyKeyFileAttribute_tc : TypeInfo;
  public mutable AssemblyCultureAttribute_tc : TypeInfo;  
  public mutable Nemerle_list_tc : TypeInfo;
  public mutable IEnumerable_tc : TypeInfo;
  public mutable IEnumerator_tc : TypeInfo;
  public mutable Generic_IEnumerable_tc : TypeInfo;
  public mutable Generic_IEnumerator_tc : TypeInfo;
  public mutable Generic_IList_tc : TypeInfo;
  public mutable Generic_ICollection_tc : TypeInfo;
  public mutable Generic_Nullable_tc : TypeInfo;
  public mutable DllImport_tc : TypeInfo;
  public mutable Serializable_tc : TypeInfo;    

  mutable function_types : array [FunctionType];
  mutable tuple_types : array [TupleType];
  mutable array_types : array [TypeInfo];

  public mutable Void : MType.Void;
  public mutable Array : MType.Class;
  public mutable Attribute : MType.Class;
  public mutable Boolean : MType.Class;
  public mutable Byte : MType.Class;
  public mutable Char : MType.Class;
  public mutable Decimal : MType.Class;
  public mutable Delegate : MType.Class;
  public mutable Double : MType.Class;
  public mutable Enum : MType.Class;
  public mutable Exception : MType.Class;
  public mutable Int16 : MType.Class;
  public mutable Int32 : MType.Class;
  public mutable Int64 : MType.Class;
  public mutable Object : MType.Class;
  public mutable SByte : MType.Class;
  public mutable Single : MType.Class;
  public mutable String : MType.Class;
  public mutable Type : MType.Class;
  public mutable UInt16 : MType.Class;
  public mutable UInt32 : MType.Class;
  public mutable UInt64 : MType.Class;
  public mutable ValueType : MType.Class;
  public mutable MatchFailureException : MType.Class;
  public mutable IObjectReference : MType.Class;  

  public mutable Delegate_Combine : IMethod;
  public mutable Delegate_Remove : IMethod;
  public mutable String_Concat : IMethod;

  public IntegralTypes : array [MType.Class]
  {
    //[Nemerle.Memoize (InvalidValue = null)]
    get {
      array [InternalType.Int32, InternalType.SByte, InternalType.Byte,
             InternalType.Int16, InternalType.UInt16, InternalType.UInt32,
             InternalType.Int64, InternalType.UInt64]
    }
  }
  
  public NewMatchFailureException_ctor : IMethod
  {
    //[Nemerle.Memoize (InvalidValue = null)]    
    get {
      get_single_method (MatchFailureException_tc, ".ctor");
    }
  }

  public MatchFailureException_ctor : SR.ConstructorInfo
  {
    get { NewMatchFailureException_ctor.GetConstructorInfo () }
  }
  
  public String_opEquality : IMethod
  {
    //[Nemerle.Memoize (InvalidValue = null)]        
    get {
      get_single_method (String_tc, "op_Equality");
    }
  }

  public Decimal_opEquality : IMethod
  {
    //[Nemerle.Memoize (InvalidValue = null)]            
    get {
      get_single_method (Decimal_tc, "op_Equality");
    }
  }

  get_single_method (tc : TypeInfo, name : string) : IMethod
  {
    match (tc.LookupMember (name)) {
      | [x] => x :> IMethod
      | _ => Util.ice ()
    }
  }

  public GetFunctionType (len : int) : FunctionType
  {
    when (function_types.Length <= len)
      Message.FatalError ($ "function types only up to $(function_types.Length) "
                            "parameters are supported, sorry (got $len)");
    when (function_types [len] == null)
      function_types [len] = FunctionType (len);

    function_types [len]
  }

  public GetTupleType (len : int) : TupleType
  {
    when (tuple_types.Length <= len)
      Message.FatalError ($ "tuple types only up to $(tuple_types.Length) "
                            "parameters are supported, sorry (got $len)");
    when (tuple_types [len] == null)
      tuple_types [len] = TupleType (len);

    tuple_types [len]
  }

  public GetArrayType (dims : int) : TypeInfo
  {
    when (array_types [dims] == null)
      array_types [dims] = lookup ($ "Nemerle.Builtins.Array$dims");
    array_types [dims]
  }


  lookup (type_name : string) : TypeInfo
  {
    match (NamespaceTree.LookupExactType (type_name)) {
      | Some (t) => t
      | None => Util.ice ("internal type " + type_name + " not found")
    }
  }

  lookup (type_name : string, args_count : int) : TypeInfo
  {
    match (NamespaceTree.LookupExactType (type_name, args_count)) {
      | Some (t) => t
      | None => Util.ice ("internal type " + type_name + " not found")
    }
  }
    
  internal InitSystemTypes () : void
  {
    // ordering is important here
    Boolean_tc = lookup ("System.Boolean"); Boolean = MType.Class (Boolean_tc, []);
    Int32_tc = lookup ("System.Int32"); Int32 = MType.Class (Int32_tc, []);

    LibraryReference.add_buildins = true;
    (Boolean_tc :> LibraryReference.NetType).AddBuiltins ();
    (Int32_tc :> LibraryReference.NetType).AddBuiltins ();
    
    // and here not
    Void = MType.Void ();
    Array_tc = lookup ("System.Array"); Array = MType.Class (Array_tc, []);
    Attribute_tc = lookup ("System.Attribute"); Attribute = MType.Class (Attribute_tc, []);
    Byte_tc = lookup ("System.Byte"); Byte = MType.Class (Byte_tc, []);
    Char_tc = lookup ("System.Char"); Char = MType.Class (Char_tc, []);
    Decimal_tc = lookup ("System.Decimal"); Decimal = MType.Class (Decimal_tc, []);
    Delegate_tc = lookup ("System.Delegate"); Delegate = MType.Class (Delegate_tc, []);
    Double_tc = lookup ("System.Double"); Double = MType.Class (Double_tc, []);
    Enum_tc = lookup ("System.Enum"); Enum = MType.Class (Enum_tc, []);
    Exception_tc = lookup ("System.Exception"); Exception = MType.Class (Exception_tc, []);
    Int16_tc = lookup ("System.Int16"); Int16 = MType.Class (Int16_tc, []);
    Int64_tc = lookup ("System.Int64"); Int64 = MType.Class (Int64_tc, []);
    Object_tc = lookup ("System.Object"); Object = MType.Class (Object_tc, []);
    (Object_tc :> LibraryReference.NetType).AddBuiltins ();
    SByte_tc = lookup ("System.SByte"); SByte = MType.Class (SByte_tc, []);
    Single_tc = lookup ("System.Single"); Single = MType.Class (Single_tc, []);
    String_tc = lookup ("System.String"); String = MType.Class (String_tc, []);
    Type_tc = lookup ("System.Type"); InternalType.Type = MType.Class (Type_tc, []);
    UInt16_tc = lookup ("System.UInt16"); UInt16 = MType.Class (UInt16_tc, []);
    UInt32_tc = lookup ("System.UInt32"); UInt32 = MType.Class (UInt32_tc, []);
    UInt64_tc = lookup ("System.UInt64"); UInt64 = MType.Class (UInt64_tc, []);
    ValueType_tc = lookup ("System.ValueType"); ValueType = MType.Class (ValueType_tc, []);
    IEnumerable_tc = lookup ("System.Collections.IEnumerable");
    IEnumerator_tc = lookup ("System.Collections.IEnumerator");
    Generic_IEnumerable_tc = lookup ("System.Collections.Generic.IEnumerable");
    Generic_IEnumerator_tc = lookup ("System.Collections.Generic.IEnumerator");
    Generic_IList_tc = lookup ("System.Collections.Generic.IList");
    Generic_ICollection_tc = lookup ("System.Collections.Generic.ICollection");
    Generic_Nullable_tc = lookup ("System.Nullable", 1);
    DllImport_tc = lookup ("System.Runtime.InteropServices.DllImportAttribute");
    Serializable_tc = lookup ("System.SerializableAttribute");    
    IObjectReference = MType.Class (lookup ("System.Runtime.Serialization.IObjectReference"), []);
    
    ParamArrayAttribute_tc = lookup ("System.ParamArrayAttribute");
    FlagsAttribute_tc = lookup ("System.FlagsAttribute");
    AssemblyVersionAttribute_tc = lookup ("System.Reflection.AssemblyVersionAttribute");
    AssemblyKeyFileAttribute_tc = lookup ("System.Reflection.AssemblyKeyFileAttribute");
    AssemblyCultureAttribute_tc = lookup ("System.Reflection.AssemblyCultureAttribute");

    def is_right (mem : IMember) {
      match (mem) {
        | meth is IMethod =>
          def parms = meth.GetParameters ();
          meth.IsStatic &&
          parms.Length == 2 &&
          ! parms.Head.ty.Equals (Object)
        | _ => false
      }
    }

    def single (tc : TypeInfo, name) {
      match (List.Filter (tc.LookupMember (name), is_right)) {
        | [s] => s :> IMethod
        | _ => assert (false)
      }
    }

    Delegate_Combine = single (Delegate_tc, "Combine");
    Delegate_Remove = single (Delegate_tc, "Remove");
    String_Concat = single (String_tc, "Concat");
    
    function_types = array (20);
    tuple_types = array (20);
    array_types = array (20);

    InternalType.MatchFailureException_tc = null; // cleanup
  }

  // to be called after scan_globals (think about compiling nemerle.dll)
  internal InitNemerleTypes () : void
  {
    // prevent multiple execution
    when (InternalType.MatchFailureException_tc == null) { 
      SystemType.NemerleAttribute = SystemType.Reflect ("Nemerle.Internal.NemerleAttribute");    
      InternalType.MatchFailureException_tc = lookup ("Nemerle.Core.MatchFailureException"); 
      InternalType.MatchFailureException = MType.Class (InternalType.MatchFailureException_tc, []);

      InternalType.NullMatchException_tc = lookup ("Nemerle.Core.NullMatchException"); 
      InternalType.ContainsMacroAttribute_tc = lookup ("Nemerle.Internal.ContainsMacroAttribute"); 
      InternalType.TypeAttribute_tc = lookup ("Nemerle.Internal.TypeAttribute"); 
      InternalType.VariantAttribute_tc = lookup ("Nemerle.Internal.VariantAttribute"); 
      InternalType.TypeAliasAttribute_tc = lookup ("Nemerle.Internal.TypeAliasAttribute"); 
      InternalType.VariantOptionAttribute_tc = lookup ("Nemerle.Internal.VariantOptionAttribute");
      InternalType.VolatileModifier_tc = lookup ("Nemerle.Internal.VolatileModifier");
      InternalType.ImmutableAttribute_tc = lookup ("Nemerle.Internal.ImmutableAttribute");
      InternalType.ExtensionAttribute_tc = lookup ("Nemerle.Internal.ExtensionAttribute");
      InternalType.ConstantVariantOptionAttribute_tc = lookup ("Nemerle.Internal.ConstantVariantOptionAttribute");
      InternalType.ExtensionPatternEncodingAttribute_tc = lookup ("Nemerle.Internal.ExtensionPatternEncodingAttribute");
      
      InternalType.Nemerle_list_tc = lookup ("Nemerle.Core.list");
    }
  }
}

} // end ns
