using Nemerle.Collections.List;
using Nemerle.Compiler.Parsetree;
using Nemerle.Text;
using Nemerle.Utility.NString;
using SC = System.Console;
using System;
using System.IO;
using System.Text;
using System.Text.RegularExpressions.Regex;
using Interp = Nemerle.Evaluation.Interpreter.Internals;
  
[assembly: System.Reflection.AssemblyTitle("Nemerle Interactive Shell")]
[assembly: System.Reflection.AssemblyDescription("Nemerle (http://nemerle.org) Interactive Shell")]
[assembly: System.Reflection.AssemblyCompany("University of Wroclaw")]
[assembly: System.Reflection.AssemblyProduct("Nemerle Interactive Shell")]
[assembly: System.Reflection.AssemblyCopyright("Copyright @ University of Wroclaw 2005")]

[assembly: System.Reflection.AssemblyKeyFile("../../misc/keys/Nemerle.Compiler.snk")]
[assembly: System.Reflection.AssemblyVersion("0.3.2")]
  
  
namespace Nemerle.Evaluation.Interpreter
{  
  module MainClass
  {
    public Main (args : array [string]) : void
    {
      // Check readline.n for details.
      HacksFramework.Run ();

      def profile = Path.Combine (Environment.GetFolderPath (Environment.
                    SpecialFolder.Personal), ".nemerlish_profile");

      // create nemerlish_profile file if it does not exists
      unless (File.Exists (profile)) {
        def stream = File.Create (profile);
		def sw = StreamWriter (stream);
	    sw.Write ("/* Open the namespace holding nemerlish-related properties. */\n");
		sw.Write ("using Nemerle.Evaluation.Interpreter.Internals;\n");
	    sw.Close ();
	    stream.Close ();
	  }
		
      def histfile = Path.Combine (Environment.GetFolderPath (Environment.
                     SpecialFolder.Personal), ".nemerlish_history"); 

      def ce = Console.Error;

      def print_welcome () : void {
        SC.WriteLine ("Welcome to Nemerle interpreter " +
                      $"(using ncc $(Interp.nccversion)).\n");
        SC.WriteLine ("Please enter expressions appended with \";;\".");
        SC.WriteLine ("Type \"Help;;\" for more info.\n");
      }
      print_welcome ();
  
      // Read code for evaluation from a file.
      def fromfile (filename) {
        def sr = File.OpenText (filename);
        def readfile (acc) {
          def l = sr.ReadLine ();
          if (l != null)
            if (acc == "")
              readfile (l)
            else
              readfile (acc + " " + l)
           else
             acc
        }

        def res = readfile ("");
        sr.Close ();
        res
      } 

      // Read the contents of all files specified after "-f" at the 
      // command-line. FIXME: Warn about problems, instead of ignoring
      // invalid input silently.
      def initialcode = if (args.Length > 1)
                          match (FromArray (args)) {
                            | "-f" :: filenames =>
                              FoldRight (profile :: filenames, "", 
                                        fun (file, acc) {
                                          try {
                                            acc + fromfile (file)
                                          }
                                          catch {
                                            | _ => acc
                                          }
                                        })
                            | _ => "" 
                          }
                        else 
                          try {
                            fromfile (profile)
                          }
                          catch {
                            | _ => ""
                          }

      // We need to add the application directory to the search path, so 
      // the assemblies in nemish.exe directory can be loaded.
      def this_dir = Nemerle.Compiler.LibraryReferenceManager.
                     DirectoryOfCodebase (Interp.interpassembly);
      Nemerle.Compiler.LibraryReferenceManager.AddSearchDirectory (this_dir);

      def e = Nemerle.Evaluation.Evaluator ();
      Evaluator.refr = [Interp.interpassembly];
     
      // Evaluate code "c" and print the results.
      def printev (c) {
        try { 
          def (nmnvts) = e.Eval (c);
          Iter (nmnvts, fun (x) {
            // (is_new, is_mutable, name, value, type).
            def (newvar, mut, name, val, ty) = x;
            when (newvar) 
              match ((mut, val, ty)) {
                | (false, null, _) =>
                  SC.WriteLine ($"def $name : $ty")
                | (_, null, _) =>
                  SC.WriteLine ($"mutable $name : $ty")
                | (_, _, <[ $_ -> $_ ]>) =>
                  SC.WriteLine ($"def $name : $ty")
                | (false, _, _) => 
                  try {
                    SC.WriteLine ($"def $name = $val : $ty")
                  } 
                  catch {
                    | _ => 
                      ce.WriteLine ($"Warning: omitting the value of $name, "
                                    "because it contains null(s).");
                      SC.WriteLine ($"def $name : $ty")
                  }
                | _ => 
                  try {
                    SC.WriteLine ($"mutable $name = $val : $ty")
                  }
                  catch {
                    | _ => 
                      ce.WriteLine ($"Warning: omitting the value of $name, "
                                    "because it contains null(s).");
                      SC.WriteLine ($"mutable $name : $ty")
                  }
            }
          })
        } 
        catch {
          | e is Exception =>
            unless (Nemerle.Compiler.Message.SeenError) {
              ce.WriteLine (e.Message + e.StackTrace);
              ce.WriteLine (e.ToString ())
            }
        } 
      }

      def rl = NativeReadLine.GetInstance ();
      def usehist = HistoryFile.RecreateHistory (rl, histfile);

      // The main loop. 
      def readinput (code, line) {
        def readandloop (c) {
          readinput (c, rl.ReadLine (Interp.Prompt))
        }

        match (line) {
          | null => ()
          | _ => 
            rl.AddHistory (line);

            when (usehist)
              try {
                HistoryFile.AddHistory (histfile, line)
              }
              catch {
                | e => ce.WriteLine ("Warning: " + e.Message)
              }

            regexp match (line) {
              | @"^\s*$" => readandloop (code)
              | _ =>  { 
                def l = line.TrimEnd ();
                if (l.EndsWith (";;")) {
                  def m = code + l.TrimEnd (array [';',' ','\t']);
                  regexp match (m) {
                    // Don't evaluate lines containing only whitespace and ";".
                    | @"^[\s;]*$" => Interp.Prompt = Interp.PS1; 
                                     readandloop ("")
                    // Fish out using statements, open apropriate namespaces
                    // and/or set namespace aliases, cut the statement out of
                    // the code and evaluate what remained.
                    | @"(?<beg>.*(([;}]|(\*/))\s*|^\s*))using\s+(?<use>[a-zA-Z0-9]+((\.[a-zA-Z0-9]+)*(\s*=\s*[a-zA-Z0-9]+)*)*)[\s;]*(?<end>.*)$" => 
                      def newns = match (Split (Replace (use, @"\s", ""), array ['='])) {
                        | [h] => (None (), Split (h, array ['.']))
                        | [h, t] => (Some (h), Split (t, array ['.']))
                        | _ => assert (false)
                      } 
                      Evaluator.ns = newns :: Evaluator.ns;
                      def c = beg + end;
                      regexp match (c) {
                        | @"[\s;]*" => Interp.Prompt = Interp.PS1; 
                                       readandloop ("")
                        // Use readinput instead of printev (c), so other 
                        // using statements can be evaluated in the same way.
                        | _ => readinput (c, ";;")
                      }
                    // Run external commands in a sub-shell.
                    | @"^\s*!(?<input>.*)$" => 
                      def p = Diagnostics.Process ();
                      regexp match (input) {
                        | @"\s*" => 
                          if (Environment.
                             GetEnvironmentVariable ("SHELL") != "")
                            p.StartInfo.FileName = "$SHELL"
                          else
                            p.StartInfo.FileName = "cmd.exe"
                        | _ => p.StartInfo.FileName = input
                      }
                      p.StartInfo.UseShellExecute = true;
                      _ = p.Start();    
                      p.WaitForExit()
                    | _ => printev (m)
                  }
                  Interp.Prompt = Interp.PS1;
                  readandloop ("")
                }
                else { 
                  Interp.Prompt = Interp.PS2;
                  readandloop (code + " " + line)
                }
              }
            }
        }
      }

      if (initialcode != "") {
        SC.WriteLine ("Please wait while evaluating the config file..");
        readinput (initialcode, ";;")
      } 
      else
        readinput ("", rl.ReadLine (Interp.Prompt))
    }
  }

  module HistoryFile {
    public AddHistory (filename: string, line : string) : void {
      def sw = File.AppendText (filename);
      sw.WriteLine (line);
      sw.Close ();
    }

    public RecreateHistory (rl : IReadLine, filename : string) 
    : bool {
      def ce = Console.Error;
      try {
        def sr = File.OpenText (filename);
        def loop (i, prev) {
          def l = sr.ReadLine ();
          when (l != null) {
            when (l != prev) 
              rl.AddHistory (l);
            loop (i + 1, l)
          }
        }

        loop (0, "");
        sr.Close ();
        true
      } 
      catch {
        | _ is FileNotFoundException => 
          try {
            _ = AddHistory (filename, ""); 
            true
          }
          catch {
            | e => ce.WriteLine ("Warning: " + e.Message); 
                   false
          }
        | e => ce.WriteLine ("Warning: " + e.Message); 
               false
      }
    }
  }
} 
