using Nemerle.Collections;
using Nemerle.Compiler;
using Nemerle.Compiler.Parsetree;
using Nemerle.Text;
using Nemerle.Utility;

[assembly: System.Reflection.AssemblyTitle("Nemerle Evaluation Library")]
[assembly: System.Reflection.AssemblyDescription("Nemerle (http://nemerle.org) Runtime evaluation library")]
[assembly: System.Reflection.AssemblyCompany("University of Wroclaw")]
[assembly: System.Reflection.AssemblyProduct("Nemerle Evaluation Library")]
[assembly: System.Reflection.AssemblyCopyright("Copyright @ University of Wroclaw 2005")]

[assembly: System.Reflection.AssemblyKeyFile("../../misc/keys/Nemerle.Compiler.snk")]
[assembly: System.Reflection.AssemblyVersion("0.3.2")]

namespace Nemerle.Evaluation
{
  /**
   * Allows dynamic evaluation of code, with persistent computation 
   * history (retaining computed variables, functions, etc.) between
   * subsequent calls to Evaluator ().Eval ().
   *
   * <seealso cref="Nemerle.Compiler.NemerleCodeProvider">
   *   NemerleCodeProvider
   * </seealso>
   */
  public class Evaluator
  {
    // Evaluated code.
    static mutable code : string;
    // Whether the code compiled successfully.
    mutable static compiled : bool;
    // [(is_mutable, name, type)]
    static mutable locals : list [bool * string * PExpr] = [];
    // The new values returned after evaluation. This 
    // was made a field because it's in a try block.
    mutable static newvals : list [object] = [];
    // A list of namespace aliases and namespaces that are to be opened.
    // [(Some (shortname), longname) or (None (), longname)]
    public static mutable ns : list [option [string] * list [string]] = [];
    // A list of assemblies to reference.
    public static mutable refr : list [string] = [];
    // [value]
    public static mutable vals : list [object] = [];

    public this () {
      Message.InitOutput (System.Console.Out);         
  
      Options.CompileToMemory = true;
      Options.IgnoreConfusion = true;
      Options.ProgressBar = false;
      Options.Sources = [""];
      // unused variable
      WarningOptions.Disable (168);   
      // ignore computed value
      WarningOptions.Disable (10005); 
  
      Passes.LexingPipeline = fun (_) { null };
      Passes.ParsingPipeline = fun (_) { [null] };
      Passes.ScanningPipeline = DoTheStuff;
    }
  
    /**
     * Evaluates supplied code in memory and returns computation results.
     * Persistent computation history is maintained between subsequent calls.
     * 
     * <param name="code">
     *   The code to evaluate.
     * </param>
     * <returns>
     *   A sorted list of computed variables and functions with additional 
     *   descriptive information. The tuple members from the list are:
     *   #1 - true if the variable was introduced during the last Eval,
     *        false if it was onthe list before.
     *   #2 - true if the variable is mutable, false if not.
     *   #3 - name of the variable.
     *   #4 - value of the variable.
     *   #5 - type of the variable.
     *
     *   The last computed value - the return value of the evaluated code
     *   is returned as a special variable named "it".
     * </returns>
     */
    public Eval (code: string) : list [bool * bool * string * object * PExpr] {
      Evaluator.code = code;
      
      // Link ourselves.
      Options.ReferencedLibraries = System.Reflection.Assembly.GetAssembly 
                                    (this.GetType ()).Location :: 
                                    Evaluator.refr;
      
      // Locals and vals from previous calls to Eval.
      def prevlocals = Evaluator.locals;
      def prevvals = Evaluator.vals;
  
      try {
        Passes.Run ();

        def ass = Passes.GeneratedAssembly;
        def meth = ass.GetTypes () [0].GetMethod ("Run");
        // And here are the new values (along with the old ones).
        newvals = meth.Invoke (null, null) :> list [object];
        Evaluator.compiled = true
      }
      catch {
        | e => match (e) {
                 | _ is AssertionException
                 | _ is Recovery 
                 | _ is System.ApplicationException => 
                  if (e.Source == "Nemerle.Compiler" || 
                     e.Source == "Nemerle.Evaluation") {
                    Evaluator.locals = prevlocals;
                    Evaluator.newvals = List.Rev (prevvals);
                    Evaluator.compiled = false
                  }
                  else throw;

                 | _ => throw
               }
      }
  
      // Check which variables are new to this call and set the first field
      // in the return tuple - true for variable "it", new ones and those 
      // whose values and/or type has changed; false - for the rest.
      def loop (l1, l2, l3, l4, acc) {
        match ((l1, l2, l3, l4)) {
          | ([], [], [], []) => acc
          | ((mut, name, ty) :: t1, [], h3 :: t3, []) => 
            loop (t1, [], t3, [], (true, mut, name, h3, ty) :: acc)
          | ((mut, name, ty) :: t1, (prevmut, prevname, prevty) :: t2, 
            h3 :: t3, h4 :: t4) =>
            if (name == prevname)
              if ((name == "it" && Evaluator.compiled == true) ||
                 (h3 != null && !(h3.Equals (h4))) ||
                 (h3 == null && h4 != null) ||                
                 mut != prevmut)
                loop (t1, t2, t3, t4, (true, mut, name, h3, ty) :: acc)
              else
                loop (t1, t2, t3, t4, (false, mut, name, h3, ty) :: acc)
            else
              loop (t1, (prevmut, prevname, prevty) :: t2, t3, h4 :: t4,
                   (true, mut, name, h3, ty) :: acc)
          // This should match only when no code has been successfully
          // evaluated yet and we have _no_ variables at all.
          | _ => acc
        }
      }
  
      Evaluator.vals = List.Rev (newvals);
  
      loop (Evaluator.locals, prevlocals, newvals, List.Rev (prevvals), [])
    }
  
    internal static DoTheStuff (_tdecl : Parsetree.TopDeclaration) : void {
      MacroRegistry.register_macro (StagedMacro ());
      def n = (None (), ["Nemerle", "Core"]) :: 
              (None (), ["Nemerle", "Core", "option"]) :: Evaluator.ns;
      // Open namespaces and set aliases.
      def env = List.FoldLeft (n, GlobalEnv.Empty, fun (x, acc : GlobalEnv) { 
                  def (x1, x2) = x;
                  match (x1) {
                    | None => acc.AddOpenNamespace (x2, Location.Default)
                    | Some (sname) => acc.AddNamespaceAlias (sname, x2, 
                                                             Location.Default)
                  }
                });
  
      MacroColorizer.PushNewColor (-1, env);
  
      // Set the class in which we're going to put the evaluated code.
      def cname = Macros.NewSymbol ();
      def tb = GlobalEnv.Core.Define (<[ decl: public class $(cname : name) {
                                               } ]>);
      def body = MainParser.ParseExpr (env, Evaluator.code); 
      match (body) {
        | <[ ]> => throw System.ApplicationException ("Nothing to parse.")
        | _ => ()
      }
      
      // If the code ends with an assignment, append `()'.
      // Put a call to stagedmacro at the end of the evaluated code
      // with the last expression moved to the stagedmacro argument.
      def whole = match (body) {
        | <[ {..$seq } ]> => 
          def (beg, last) = List.DivideLast (seq);
          def last = 
            match (last) {
              | <[ def $_ = $_ ]> 
              | <[ mutable $_ = $_ ]> 
              | <[ def .. $_ ]> =>
                <[ $last; $("stagedmacro" : usesite) (()) ]>
              | _ => <[ $("stagedmacro" : usesite) ($last) ]>
            }
          <[ {..$ (beg + [last]) } ]>
        | _ => <[ $("stagedmacro" : usesite) ($body) ]>
      }
  
      // Recreate variables defined in previous calls to Eval.
      def inits = List.FoldLeft (Evaluator.locals, [], fun (x, acc) {
        def (mut, name, ty) = x;
        match ((mut, ty)) {
          | (false, <[ System.Object ]>) => 
            <[ def $(name : usesite) = List.Hd (Evaluator.vals) : $ty ]> :: 
            <[ Evaluator.vals = List.Tl (Evaluator.vals) ]> :: acc
          | (false, _) => 
            <[ def $(name : usesite) = List.Hd (Evaluator.vals) :> $ty ]> ::
            <[ Evaluator.vals = List.Tl (Evaluator.vals) ]> :: acc
          | (_, <[ System.Object ]>) => 
            <[ mutable $(name : usesite) = 
            List.Hd (Evaluator.vals) : $ty ]> ::
            <[ Evaluator.vals = List.Tl (Evaluator.vals) ]> :: acc; 
          | _ => 
            <[ mutable $(name : usesite) = 
            List.Hd (Evaluator.vals) :> $ty ]> ::
            <[ Evaluator.vals = List.Tl (Evaluator.vals) ]> :: acc; 
        }
      });
      
      def w = <[ {.. $(inits + [whole]) } ]>;
  
//      PrettyPrint.PrintExpr (None (), w);
      
      tb.Define (<[ decl: public static Run () : list [System.Object] {
                          $w } ]>);
      tb.Compile ();
  
      MacroColorizer.PopColor ();
    }
  
    internal class StagedMacro : IMacro 
    {
      public Run (ctx : Typer, val : list [SyntaxElement]) : PExpr {
        match (val) {
          // Fish out variables/functions and store them in our fields.
          | [SyntaxElement.Expression (expr)] =>
            Evaluator.locals = [];
            def l = ctx.GetLocals ();
            def values = <[ $expr : System.Object ]> :: 
                         l.Fold ([], fun (n : Name, loc : LocalValue, acc) {
              regexp match (n.ToString ()) {
                | @"^it$" => acc
                | _ => 
                  Evaluator.locals = (loc.IsMutable, loc.Name, 
                                      PrettyPrint.TyVarToParseTree 
                                      (loc.Type)) :: Evaluator.locals;
                  <[ ($(n : name) : System.Object) ]> :: acc
              }
            });
            Evaluator.locals = (false, "it", PrettyPrint.TyVarToParseTree 
                               (ctx.TypeExpr (expr).Type)) :: Evaluator.locals;
            <[ [..$values] ]>
          | _ => Util.ice ()
        }
      }
  
      // This below is only to satisfy the interface requirements.
      public CallTransform (l : list [PExpr]) : list [SyntaxElement]
      { List.Map (l, SyntaxElement.Expression) }
      public GetName () : string { "stagedmacro" }
      public GetNamespace () : string { "" }
      public IsInherited : bool { get { false } }
      public IsTailRecursionTransparent : bool { get { false } }
      public Keywords : list [string] { get { [] } }
      public SyntaxExtension () : GrammarElement * (list [SyntaxElement] -> 
                                                    list [SyntaxElement])
      { (null, fun (_) { [] }) }
      public Usage : Nemerle.MacroUsageAttribute { get { null } }
    }
  }
}
