/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;
using Nemerle.Utility;

using Nemerle.Compiler;
using Nemerle.Compiler.Parsetree;
using Nemerle.Compiler.SolverMacros;

namespace Nemerle.Compiler
{
  public class TyVarEnv
  {
    tyvars : NemerleMap [Name, StaticTyVar];
    solver : Solver;
    messenger : Messenger;
    
    private this (tv : NemerleMap [Name, StaticTyVar])
    {
      tyvars = tv;
      solver = Passes.Solver;
      messenger = solver.CurrentMessenger;
    }

    public this ()
    {
      tyvars = NemerleMap ();
      solver = Passes.Solver;
      messenger = solver.CurrentMessenger;
    }

    public IsEmpty : bool
    {
      get { tyvars.IsEmpty }
    }
    
    public MonoBind (env : GlobalEnv,
                     curtc : TypeBuilder,
                     t : PExpr,
                     check_parms : bool) : MType
    {
      def t = Bind (env, curtc, t, 
                    allow_tyvars = false, 
                    check_parms = check_parms);
      t.Fix ()
    }
    
    /** Perform typing of Parsetree type to Typedtree type, looking up
        type constructors in given global environment (with accessibility
        information implied by given current TypeInfo) and type variables
        in current instance.

        If [check_parms] is true, then we check if type substituted for
        a given type variable conforms to all of its constraints. It
        should be true in general, but is false when we don't know the
        subtyping relations yet (during scanning of global types).
     */
    public Bind (env : GlobalEnv,
                 curtc : TypeBuilder,
                 t : PExpr,
                 allow_tyvars : bool,
                 check_parms : bool) : TyVar
    {
      def f (t) {
        f2 (t, false)
      } and f2 (t, allow_ref) {
        match (t) {
          | <[ ref $t ]> when allow_ref =>
            MType.Ref (f (t))
          
          | <[ out $t ]> when allow_ref =>
            MType.Out (f (t))
          
          | <[ ref $_ ]>
          | <[ out $_ ]> =>
            ReportError (messenger, "nested ref/out type found");
            InternalType.Void

          | <[ array [$t] ]> =>
            MType.Array (f (t), 1)
            
          | <[ array [$(rank : int), $t] ]> =>
            MType.Array (f (t), rank : int)

          | <[ $x -> $y ]> =>
            MType.Fun (f2 (x, true), f (y))

          | <[ @* (.. $args) ]> =>
            MType.Tuple (List.Map (args, fun (t) { f2 (t, allow_ref) }))
            
          | PExpr.Void => InternalType.Void

          | PExpr.Member
          | PExpr.Ref =>
            type_class (t, [])

          | PExpr.Indexer (t, args) =>
            when (args.IsEmpty) {
              ReportError (messenger, $"$t[] is not a valid type, use just $t");
              when (messenger.NeedMessage)
                Message.HintOnce ("if you had array type on mind, its syntax is `array [SomeType]'");
            }
              
            type_class (t, args)

          | PExpr.TypedType (body) =>
            body
            
          | PExpr.Wildcard =>
            if (allow_tyvars)
              Solver.FreshTyVar ()
            else {
              ReportError (messenger, "type inference not allowed here");
              InternalType.Object
            }

          | PExpr.Spliced =>
            Util.ice ("Spliced type survived to typying.")

          | PExpr.Array =>
            ReportError (messenger, 
                         $ "array type must take form `array [T]' or "
                           "`array [rank, T]', not $t");
            InternalType.Void
            
          | PExpr.Ellipsis =>
            Util.ice ("Type arguments list survived to typying.")
            
          | x =>
            ReportError (messenger, $ "$x is not a legal type expression");
            InternalType.Void
        }
      } and type_class (name, args : list [_]) : TyVar {
        match (Util.QidOfExpr (name)) {
          | Some ((idl, nm)) =>
            match (tyvars.Find (nm)) {
              | Some (tv) =>
                when (!args.IsEmpty)
                  ReportError (messenger,
                               $ "type variable `$(nm.Id)' supplied with "
                                 "arguments");

                MType.TyVarRef (tv)
              | None =>
                def env = nm.GetEnv (env);
                assert (env != null);
                def ti = env.GetType (idl, curtc);
                def args = List.Map (args, f);
                
                ti.HasBeenUsed = true;

                match (ti.GetTydecl ()) {
                  | Typedtree.TypeDeclaration.Alias (t) =>
                    def subst = ti.MakeSubst (args);
                    subst.Apply (t)

                  | _ =>
                    when (check_parms)
                      _ = ti.MakeSubst (args);
                    MType.Class (ti, args)
                }
            }
          | None =>
            ReportError (messenger, 
                         $ "expected qualified identifier in type, not $name");
            InternalType.Void
        }
      }

      f (t)
    }
    
    
    public AddTyparms (env : GlobalEnv, tp : Typarms,
                       curtc : TypeBuilder,
                       check_parms : bool) : TyVarEnv * list [StaticTyVar]
    {
      def name_of_tv (tv) {
        def (name, color) = tv;
        Name (name, color, null)
      }
      def loop (tv : string * int, acc) {
        def (map, the_list) = acc;
        def tv_obj = StaticTyVar (Pair.First (tv));
        ((map : NemerleMap [Name, StaticTyVar]).Replace (name_of_tv (tv), tv_obj),
         tv_obj :: the_list)
      }
      def (m, l) = List.FoldLeft (tp.tyvars, (this.tyvars, []), loop);
      def tenv = TyVarEnv (m);

      def constraints = Hashtable ();
      
      def get (id) {
        if (constraints.Contains (id))
          constraints [id]
        else (SpecialConstraint.None, [])
      }

      // bind constraints to what they really are
      foreach (c in tp.constraints) {
        match (m.Find (name_of_tv (c.tyvar))) {
          | Some (tv) =>
            def cons = get (tv.id);
            mutable special = Pair.First (cons);
            mutable subtype = Pair.Second (cons);
            
            match (c.ty) {
              | <[ @class ]> => special |= SpecialConstraint.Class;
              | <[ @struct ]> => special |= SpecialConstraint.Struct;
              | <[ @new ]> => special |= SpecialConstraint.New;                                
              | _ =>
                subtype = tenv.MonoBind (env, curtc, c.ty, check_parms) :: subtype;
            }
            constraints [tv.id] = (special, subtype);
            
          | None =>
            Message.Error ("unbound type variable `" + Pair.First (c.tyvar) + "' in constraint")
        }
      }  
      
      foreach (tv : StaticTyVar in l) {
        // FIXME: check Intersection invariants and flag error
        // to the user, otherwise we'll get an ICE
        tv.SetConstraints (get (tv.id)); 
      }
      
      (tenv, List.Rev (l))
    }
  }
}
