/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* Transformations of quoted expressions into syntax trees
 */

using Nemerle.Collections;
using Nemerle.Utility;
using Nemerle.IO;

using Nemerle.Compiler.Parsetree;
using TT = Nemerle.Compiler.Typedtree;
using System.Text;

namespace Nemerle.Compiler {

public module Macros
{
  /// PUBLIC API FOR USAGE IN MACROS

  /** generates new unique symbol, which can be safely used
      as name of quoted variables, functions, etc.
   */
  public NewSymbol () : Name
  {
    NewSymbol ("")
  }
  
  /** generates new unique symbol, which can be safely used
      as name of quoted variables, functions, etc.
      The [root] parameter can be used to specify a string to be 
      part of the name, mostly for debugging purposes.
   */
  public NewSymbol (root : string) : Name
  {
    // passing global context here makes little sense, but as name is unique here,
    // we can pass any to do not introduce nulls into Name
    def ctx = 
      if (MacroColorizer.UseContext == null) GlobalEnv.Core
      else MacroColorizer.UseContext;
    Name (Util.tmpname (root), MacroColorizer.Color, ctx)
  }

  /** controlled hygiene breaking - generates symbol, which
      placed in generated code would bind to variables
      at macro-use site
   */
  public UseSiteSymbol (id : string) : Name
  {
    Name (id, MacroColorizer.UseColor, MacroColorizer.UseContext)
  }

  /** Checks if given expression describes name of a type, like
      [System.Console] (it is a class)
   */
  public IsTypeName (e : PExpr) : bool
  {
    def collect_member (obj : PExpr, acc) {
      match (obj) {
        // name in expressions has always one element in name
        | <[ $(n : name) ]> =>
          def env = n.context;
          match (env.LookupType (n.Id :: acc)) {
            | Some => true
            | _ => false
          }
        | <[ $head.$(id : dyn) ]> => collect_member (head, id :: acc)
        | _ => false
      }
    };
    collect_member (e, [])
  }

  /** Checks if given expression describes name of a type, like
      [System.Console] (it is a class)
   */
  public GetIfIsType (env : GlobalEnv, e : PExpr) : option [TypeInfo]
  {
    def collect_member (obj : PExpr, acc) {
      match (obj) {
        // name in expressions has always one element in name
        | <[ $(n : name) ]> =>
          def env = n.GetEnv (env);
          env.LookupType (n.Id :: acc)
        | <[ $head.$(id : dyn) ]> => collect_member (head, id :: acc)
        | _ => None ()
      }
    };
    collect_member (e, [])
  }

  public DefaultValueOfType (ty : MType) : PExpr
  {
    match (ty) {
      | MType.Class (tc, _) when tc.IsValueType =>
        def ty = Util.ExprOfQid (tc.FullName);
        <[ $ty () ]>

      | MType.Void => <[ () ]>

      | _ => <[ null ]>
    }
  }

  public GetImplicitCTXName () : Name { MacroClasses.implicit_ctx_name }
  
  /** Lifts given list of expressions to syntax tree of list containing
      elements, whose syntax trees are defined by those expressions.
      (expressions on the list are not lifted)
   */
  internal Lift (l : list[PExpr]) : PExpr
  {
    | x :: xs => <[ $x :: $(Lift (xs)) ]>
    | [] => <[ list.Nil () ]>
  }

  /** Lifts given list to syntax tree of this list, applying
      supplied function to each of its elements. The function
      is used here to lift elements of list, so we can build
      entire syntax trees from them.
   */
  public Lift['a] (l : list['a], f : 'a -> PExpr) : PExpr
  {
    <[ [.. $(List.Map (l, f)) ] ]>
  }

  public Lift['a] (o : option ['a], f : 'a -> PExpr) : PExpr
  {
    match (o) {
      | Some (v) => <[ Some ($(f (v))) ]>
      | _ => <[ None () ]>
    }
  }
  
  /** Creates syntax tree of given number (expression building it) */
  public Lift (x : int) : PExpr { <[ $(x : int) ]> }

  public Lift (x : string) : PExpr { <[ $(x : string) ]> }


  // QUOTATION HANDLING SECTION
  public TypedChoose (e : Typedtree.TExpr) : PExpr {
    PExpr.Typed (e)
  }

  public TypedChoose (e : MType) : PExpr {
    PExpr.TypedType (e)
  }

  public TypedChoose (e : TyVar) : PExpr {
    PExpr.TypedType (e)
  }

  /** Lifts up [Splicable]. [Splicable.Expression] is substituted by its
      content and appropriate [Splicable.Name] expression is created.
   */
  quoted_sstring (st : Splicable) : PExpr
  {
    match (st) {
      | Splicable.Name (name) =>
        assert (name.context != null);
        def c = name.context.GetMacroContext ();
        <[ Splicable.Name (Name.NameInCurrentColor ($(name.Id : string),
                                                    $("_N_MacroContexts" : dyn).Get ($(c : int)))) ]>

      | Splicable.Expression (PExpr.TypeEnforcement (e,  <[ $(ty : name) ]> )) =>        
        match (ty.Id) {
          | "name"  => <[ Splicable.Name ($e) ]>
          | "dyn" => <[ Splicable.Name (Name ($e, -1, MacroColorizer.UseContext)) ]>
          | "usesite" => <[ Splicable.Name (Name ($e, MacroColorizer.UseColor,
                                                  MacroColorizer.UseContext)) ]>
          | x => Message.FatalError ("unsupported splicing type `" + x + "' in splicable expression")
        }

      | Splicable.Expression (e) => e
    }
  }

  /** Lifts up [Name] giving it supplied context number. */
  quoted_name (n : Name, context : int) : PExpr
  {
    <[ Name.NameInCurrentColor ($(n.Id : string),
                                $("_N_MacroContexts" : dyn).Get ($(context : int))) ]>
  }

  public QuotedMatchCase (cas : MatchCase) : PExpr
  {
    def guards = cas.patterns;
    def expr = cas.body;

    match (guards) {
      | [PExpr.Ellipsis (args)] =>
        <[ MatchCase ($(quoted_expr (args)), $(quoted_expr (expr))) ]>
        
      | _ =>
        <[ MatchCase ($(Lift (guards, quoted_expr)), $(quoted_expr (expr))) ]>
    }
  }

  public quoted_fparam (p : Fun_parm) : PExpr
  {
    match (p) {
      | <[ parameter: $name : $ty ]> =>
        <[ Fun_parm (name = $(quoted_sstring (name)),
                     ty = $(quoted_expr (ty)),
                     modifiers = Modifiers (mods = NemerleAttributes.None, custom_attrs = [])) ]>

      | <[ parameter: params $name : $ty ]> =>
        def qattr = quoted_expr (<[ System.ParamArrayAttribute ]>);
        <[ Fun_parm (name = $(quoted_sstring (name)),
                     ty = $(quoted_expr (ty)),
                     modifiers = Modifiers (mods = NemerleAttributes.None, custom_attrs = [$qattr])) ]>

      | <[ parameter: $name : $ty = $expr ]> =>
        def e = quoted_expr (expr);
        def qattr =
          quoted_expr (<[ System.ComponentModel.DefaultValueAttribute ($e) ]>);
        <[ Fun_parm (name = $(quoted_sstring (name)),
                     ty = $(quoted_expr (ty)),
                     modifiers = Modifiers (mods = NemerleAttributes.None, custom_attrs = [$qattr])) ]>
 
      // FIXME: currently quoted_attributes returns thing, which cannot be pattern matched                     
      | <[ parameter: ..$attrs $name : $ty ]> =>
        <[ Fun_parm (name = $(quoted_sstring (name)),
                     ty = $(quoted_expr (ty)),
                     modifiers = $(quoted_attributes (attrs))) ]>
    }
  }

  quoted_tparms (tyvars : list [string * int], 
                 constraints : list [Constraint]) : PExpr 
  {
    def quoted_tvar (name, color) {
      <[ ( $(name : string), $(color : int) ) ]>
    }
    def quoted_constr (c : Constraint) {
      def (name, color) = c.tyvar;
      <[ Constraint ($(name : string), $(color : int), $(quoted_expr (c.ty))) ]>
    }
    match (constraints) {
      | [Constraint where (("", _), PExpr.Tuple ([PExpr.Wildcard, PExpr.Ellipsis (PExpr.Spliced (e))]))] =>
        <[ Typarms ($(Lift (tyvars, quoted_tvar)), $e) ]>

      | [Constraint where (("", _), PExpr.Tuple ([PExpr.Ellipsis (PExpr.Spliced (e1)), PExpr.Ellipsis (PExpr.Spliced (e2))]))] =>
        <[ Typarms ($e1, $e2) ]>

      | Constraint where (("", _), PExpr.Tuple ([PExpr.Ellipsis (PExpr.Spliced (e)), PExpr.Void])) :: where_cts =>
        <[ Typarms ($e, $(Lift (where_cts, quoted_constr))) ]>

      | _ =>
        <[ Typarms ($(Lift (tyvars, quoted_tvar)),
                    $(Lift (constraints, quoted_constr))) ]>
    }
  }

  make_quoted_funheader (parms : list[Fun_parm], qtparms : PExpr,
                         ty : PExpr, qname : PExpr) : PExpr
  {
    def qparms =
      match (parms) {
        | [Fun_parm where (name = Splicable.Name, ty = PExpr.Void,
            modifiers = Modifiers where (_, [PExpr.Ellipsis (e)], _))] => quoted_expr (e)
            
        | _ => Lift (parms, quoted_fparam)
      };
    def qtype = quoted_expr (ty);
    <[ Fun_header ($qtparms, $qname, $qtype, $qparms) ]>
  } 
  
  make_quoted_fundecl (parms : list[Fun_parm], qtparms : PExpr,
                       ty : PExpr, qname : PExpr, body : PExpr) : PExpr
  {
    def qheader = make_quoted_funheader (parms, qtparms, ty, qname);
    def qbody = quoted_expr (body);
    <[ Function_decl ($qheader, $qbody) ]>
  }
  
  quoted_attributes (attrs : Modifiers) : PExpr
  {
    match (attrs) {
      | Modifiers where (_, [PExpr.Ellipsis (e)], _) => quoted_expr (e)
      | _ =>
        <[ Modifiers (($((attrs.mods :> int) : int) :> NemerleAttributes),
                      $(Lift (attrs.custom_attrs, quoted_expr))) ]>
    };
  }

  internal quoted_member (mem : ClassMember) : PExpr {
    def qnm = quoted_sstring (mem.name);
    def qattrs = quoted_attributes (mem.modifiers);
    match (mem) {
      | ClassMember.TypeDeclaration (td) =>
        <[ ClassMember.TypeDeclaration (name = $qnm, modifiers = $qattrs,
                   td = $(quoted_tydecl (td))) ]>
        
      | ClassMember.Field (t) =>
//      | <[ decl: ..$_ $_ : $t; ]> => // field
        <[ ClassMember.Field (name = $qnm, modifiers = $qattrs,
                              ty = $(quoted_expr (t))) ]>

      // Example
      //   <[ decl: ..$attrs $n < ..$tparms> (..$fparms) : $t where ..$cts
      //              implements ..$impl $body ]> 
      | ClassMember.Function ( header = Fun_header where ( typarms = Typarms where (tparms, cts),
                       ret_type = t, parms = fparms),
                       kind = kd, body = bd) =>
        def quoted_funkind (x) {
          | FunKind.Method (impl) =>
            match (impl) {
              | [PExpr.Ellipsis (e)] => <[ FunKind.Method ($(quoted_expr (e))) ]>
              | _ => <[ FunKind.Method ($(Lift (impl, quoted_expr))) ]>
            }
          // quotation of IMethod? noooo!
          | FunKind.BoundMethod => <[ FunKind.BoundMethod ([]) ]>
          | FunKind.Constructor => <[ FunKind.Constructor () ]>
          | FunKind.StaticConstructor => <[ FunKind.StaticConstructor () ]>
          | FunKind.Function => <[ FunKind.Function () ]>
        };

        def qtparms = quoted_tparms (tparms, cts);
        def qhd = make_quoted_funheader (fparms, qtparms, t, qnm);
        
        <[ ClassMember.Function (name = $qnm,
                       modifiers = $qattrs,
                       header = $qhd,
                       kind = $(quoted_funkind (kd)),
                       body = $(quoted_funbody (bd))) ]>
        
      | ClassMember.EnumOption (val) =>
        def qval = Lift (val, quoted_expr);
        <[ ClassMember.EnumOption (name = $qnm, modifiers = $qattrs, value = $qval) ]>
        
      | ClassMember.Event ( ty = t, add = a, remove = r) =>
        <[ ClassMember.Event (name = $qnm, modifiers = $qattrs, ty = $(quoted_expr (t)),
                    add = $(quoted_member (a)),
                    remove = $(quoted_member (r))) ]>
        
      | ClassMember.Property (ty = t, prop_ty = p, dims = fps, set = s, get = g) =>
        def qfps = Lift (fps, quoted_fparam);
        <[ ClassMember.Property (name = $qnm, modifiers = $qattrs,
                       ty = $(quoted_expr (t)),
                       prop_ty = $(quoted_expr (p)),
                       set = $(Lift (s, quoted_member)),
                       get = $(Lift (g, quoted_member)), dims = $qfps) ]>
    }
  }
  
  internal quoted_tydecl (td : TopDeclaration) : PExpr {
    def qn = quoted_sstring (td.name);
    def qattr = quoted_attributes (td.modifiers);
    
    match (td) {
      | TopDeclaration.Class ( typarms = tprms, t_extends = extend, decls = members) =>
        def qtparms = quoted_tparms (tprms.tyvars, tprms.constraints);
        def qexten = Lift (extend, quoted_expr);        
        def qmems = Lift (members, quoted_member);
                          
        <[ TopDeclaration.Class (name = $qn, modifiers = $qattr, t_extends = $qexten,
                     typarms = $qtparms, decls = $qmems) ]>

      | TopDeclaration.Alias ( typarms = tprms, ty = t) =>
        def qtparms = quoted_tparms (tprms.tyvars, tprms.constraints);                
        <[ TopDeclaration.Alias (name = $qn, modifiers = $qattr,
                     typarms = $qtparms, ty = $(quoted_expr (t))) ]>
        
      | TopDeclaration.Interface (typarms = tprms, t_extends = extend, methods = members) =>
        def qtparms = quoted_tparms (tprms.tyvars, tprms.constraints);
        def qexten = Lift (extend, quoted_expr);        
        def qmems = Lift (members, quoted_member);
                          
        <[ TopDeclaration.Interface (name = $qn, modifiers = $qattr, t_extends = $qexten,
                     typarms = $qtparms, methods = $qmems) ]>
                     
      | TopDeclaration.Variant (typarms = tprms, t_extends = extend, decls = members) =>
        def qtparms = quoted_tparms (tprms.tyvars, tprms.constraints);
        def qexten = Lift (extend, quoted_expr);          
        def qmems = Lift (members, quoted_member);
                          
        <[ TopDeclaration.Variant (name = $qn, modifiers = $qattr, t_extends = $qexten,
                       typarms = $qtparms, decls = $qmems) ]>

      | TopDeclaration.VariantOption (decls = members) =>
        def qmems = Lift (members, quoted_member);
                          
        <[ TopDeclaration.VariantOption (name = $qn, modifiers = $qattr, decls = $qmems) ]>

      | _ =>
        Util.ice ("this quotation is not supported yet")
    }
  }

  quoted_funbody (x : FunBody) : PExpr
  {
    match (x) {
      | FunBody.Parsed (expr) =>
        <[ FunBody.Parsed ($(quoted_expr (expr))) ]>
      | FunBody.Typed (expr) => <[ FunBody.Parsed ($(expr : typed)) ]>
      | FunBody.Abstract => <[ FunBody.Abstract () ]>
      | FunBody.ILed => <[ FunBody.ILed () ]>
    }
  }

  quoted_literal (lit : Literal) : PExpr
  {
    | Literal.Void => <[ Literal.Void () ]>
    | Literal.Null => <[ Literal.Null () ]>
    | Literal.String (val) => <[ Literal.String ($(val : string)) ]>
    | Literal.Float (val) => <[ Literal.Float ($(val : float)) ]>
    | Literal.Double (val) => <[ Literal.Double ($(val : double)) ]>
    | Literal.Decimal (val) => <[ Literal.Decimal ($(val : decimal)) ]>
    | Literal.Integer (val, is_negative, _) => 
      <[ Literal.Integer ($(val : ulong), $(is_negative : bool), null).WithProperType () ]>
    | Literal.Bool (val) => <[ Literal.Bool ($(val : bool)) ]>
    | Literal.Char (val) => <[ Literal.Char ($(val : char)) ]>
    | Literal.Enum (l, _) => <[ Literal.Enum ($(quoted_literal (l)), null) ]>
  }
  
  public quoted_fundecl (d : Function_decl) : PExpr
  {
    def <[ fundecl: $name [ ..$typarms] (..$args) : $ty
                    where ..$tyconstrs $body ]> = d;
    def qtparms = quoted_tparms (typarms, tyconstrs);
    def qname = quoted_sstring (name);
    make_quoted_fundecl (args, qtparms, ty, qname, body)
  }

  /** Creates parse tree of (expression which builds) given typed type. */
  public quoted_ttype (t : PExpr) : PExpr
  {
    def constructor (tycon, args) {
      def tyco = 
        match (Util.QidOfExpr (tycon)) {
          | Some ((s, _)) => Lift (s, Lift)
          | _ => Message.FatalError ("type constructor must be qualified id");
        }
      def findtyco = <[
        match (Macros.ImplicitCTX ().Env.LookupType ($tyco)) {
          | Some (x) => x
          | None =>
            Message.FatalError ("unbound type " + $tyco.ToString ("."))
        }
      ]>;
      match (args) {
        | [PExpr.Ellipsis(ar)] => 
          <[ MType.Class ($findtyco, $(quoted_ttype (ar))) ]>
        | _ =>
          <[ MType.Class ($findtyco, $(Lift (args, quoted_ttype))) ]>
      }
    }

    match (t) {
      | <[ $tycon [ .. $args ] ]> => constructor (tycon, args)

      | <[ $(_ : name) ]>
      | <[ $_.$_ ]> => constructor (t, [])

      | <[ ref $ty ]> =>
        <[ MType.Ref ($(quoted_ttype (ty))) ]>
  
      | <[ out $ty ]> =>
        <[ MType.Out ($(quoted_ttype (ty))) ]>
  
      | <[ $from -> $to ]> =>
        <[ MType.Fun ($(quoted_ttype (from)), $(quoted_ttype (to))) ]>

      | <[ void ]> => <[ MType.Void () ]>

      | <[ @* (..$args) ]> =>
        def x = Lift (args, quoted_ttype);
        <[ MType.Tuple ($x) ]>

      | <[ array [ $r, $ty] ]> =>
        <[ MType.Array ($(quoted_ttype (ty)), $(quoted_expr (r))) ]>

      // rest of constructs must be in not quoted form, because they define
      // internal data structures
      | PExpr.Spliced (PExpr.TypeEnforcement (val, <[ $(ty : name) ]>)) =>
        match (ty.Id) {
          | "name" => <[ MType.TyVarRef ($val) ]>

          // it doesn't make much sense here, as it is the same as <[ $v ]>,
          // but we put it here for consistency
          | "typed" => val
          | x =>
            Message.FatalError ("unsupported type of spliced special token `" + x + "' in typed type")
        }

      | PExpr.Spliced (e) => e

      | PExpr.Wildcard => <[ Passes.Solver.FreshTyVar () ]>

      | PExpr.Ellipsis (ar) =>
        <[ MType.Tuple ($(quoted_ttype (ar))) ]>

      | PExpr.Typed => Util.ice ("You've got beer from me for generating such a code...");

      | _ => Util.ice ("quoted code not supported: " + PrettyPrint.SprintExpr (None (), t));
    } 
  }

  public mutable in_pattern : bool = false;
  
  public quoted_expr (expr : PExpr) : PExpr 
  {
    match (expr) {
      | <[ $(id : name) ]> =>
        assert (id.context != null, id.Id);
        <[ PExpr.Ref ($(quoted_name (id, id.context.GetMacroContext ()))) ]>
        
      | <[ $obj . $mem ]> => 
        <[ PExpr.Member ($(quoted_expr (obj)), $(quoted_sstring (mem))) ]>

      | <[ $func (.. $parms) ]> =>
        match (parms) {
          | [ PExpr.Ellipsis (args) ] =>
            <[ PExpr.Call ($(quoted_expr (func)), $(quoted_expr (args))) ]>
          | _ =>
            <[ PExpr.Call ($(quoted_expr (func)), $(Lift (parms, quoted_expr))) ]>
        }

      | <[ $func .[..$parms] ]> =>
        match (parms) {
          | [ PExpr.Ellipsis (args) ] =>
            <[ PExpr.GenericSpecifier ($(quoted_expr (func)), $(quoted_expr (args))) ]>
          | _ =>
            <[ PExpr.GenericSpecifier ($(quoted_expr (func)), $(Lift (parms, quoted_expr))) ]>
        }

      | <[ $target = $source ]> =>
        <[ PExpr.Assign ($(quoted_expr (target)), $(quoted_expr (source))) ]>

      | <[ def $name = $val ]> =>        
        <[ PExpr.Define ($(quoted_expr (name)), $(quoted_expr (val))) ]>

      | <[ mutable $name = $val ]> =>
        <[ PExpr.DefMutable ($(quoted_expr (name)), $(quoted_expr (val))) ]>

        
      | <[ def .. $funs ]> =>
        match (funs) {
          | [Function_decl where (_, PExpr.Ellipsis (args))] =>
            <[ PExpr.DefFunctions ($(quoted_expr (args))) ]>
          | _ =>
            <[ PExpr.DefFunctions ($(Lift (funs, quoted_fundecl))) ]>
        };

      | <[ fun [ ..$typarms] (..$args) : $ty where .. $tyconstrs $body ]> =>
        def qtparms = quoted_tparms (typarms, tyconstrs);
        def qname = <[ Splicable.Name (Name ("")) ]>;
        // lift function declaration from lambda expression
        def fdecl = make_quoted_fundecl (args, qtparms, ty, qname, body);
        // return syntax tree of lifted lambda
        <[ PExpr.Lambda ($fdecl) ]>

      | <[ match ($expr) {.. $cases } ]> =>
        match (cases) {
          | [cas] when cas.patterns is [] =>
            match (cas.body) {
              | PExpr.Ellipsis (e) =>
                <[ PExpr.Match ($(quoted_expr (expr)), $(quoted_expr (e))) ]>
              | _ =>
                Util.ice ("parser generated strange match_case")
            }
          | _ => <[ PExpr.Match ($(quoted_expr (expr)),
                                 $(Lift (cases, QuotedMatchCase))) ]>
        }

      | <[ throw $exc ]> =>
        if (exc == null)
          <[ PExpr.Throw (null) ]> // throw;
        else        
          <[ PExpr.Throw ($(quoted_expr (exc))) ]>

      | <[ ref $e ]> =>
        <[ PExpr.ParmByRef ($(quoted_expr (e))) ]>
        
      | <[ out $e ]> =>
        <[ PExpr.ParmOut ($(quoted_expr (e))) ]>

      | <[ try $body catch { $exn is $exn_ty => $handler } ]> => 
        def qbody = quoted_expr (body);
        <[ PExpr.TryWith ($qbody, $(quoted_sstring (exn)),
                       $(quoted_expr (exn_ty)), 
                       $(quoted_expr (handler))) ]>

      | <[ try $body finally $handler ]> =>
        assert (body != null);
        assert (handler != null);
        <[ PExpr.TryFinally ($(quoted_expr (body)), 
                          $(quoted_expr (handler))) ]>

      | PExpr.Literal (lit) => <[ PExpr.Literal ($(quoted_literal (lit))) ]>

      | <[ this ]> => <[ PExpr.This () ]>

      | <[ base ]> => <[ PExpr.Base () ]>

      | <[ typeof ($t) ]> => <[ PExpr.Typeof ($(quoted_expr (t))) ]>

      | <[ $expr : $ty ]> =>
        <[ PExpr.TypeEnforcement ($(quoted_expr (expr)), $(quoted_expr (ty))) ]>

      | <[ $expr :> $ty ]> =>
        <[ PExpr.TypeConversion ($(quoted_expr (expr)), $(quoted_expr (ty))) ]>

      | <[ {.. $seq } ]> =>
        match (seq) {
          | [PExpr.Ellipsis (seq)] =>
            <[ PExpr.Sequence ($(quoted_expr (seq))) ]>
          | _ =>
            <[ PExpr.Sequence ($(Lift (seq, quoted_expr))) ]>
        }

      | <[ (.. $args) ]> =>
        match (args) {
          | [PExpr.Ellipsis (args)] =>
            <[ PExpr.Tuple ($(quoted_expr (args))) ]>
          | _ =>
            <[ PExpr.Tuple ($(Lift (args, quoted_expr))) ]>
        }

      | <[ array .[ $rank ] $value ]> =>
        <[ PExpr.Array ($(quoted_expr (rank)), $(quoted_expr (value))) ]>

      | <[ array (.. $sizes) ]> =>
        match (sizes) {
          | [PExpr.Ellipsis (args)] =>
            <[ PExpr.EmptyArray ($(quoted_expr (args))) ]>
          | _ =>
            <[ PExpr.EmptyArray ($(Lift (sizes, quoted_expr))) ]>
        }

      | <[ $obj [.. $args] ]> =>  
        match (args) {
          | [PExpr.Ellipsis (args)] =>
            <[ PExpr.Indexer ($(quoted_expr (obj)), $(quoted_expr (args))) ]>
          | _ =>
            <[ PExpr.Indexer ($(quoted_expr (obj)), $(Lift (args, quoted_expr))) ]>
        }

      | <[ _ ]>  => <[ PExpr.Wildcard () ]>

      | <[ void ]> => <[ PExpr.Void () ]>

      | <[ $pat as $name ]> => 
        <[ PExpr.As ($(quoted_expr (pat)), $(quoted_sstring (name))) ]>

      | <[ $e1 is $e2 ]> =>
        <[ PExpr.Is ($(quoted_expr (e1)), $(quoted_expr (e2))) ]>

      | <[ $e1 where $e2 ]> =>
        <[ PExpr.Where ($(quoted_expr (e1)), $(quoted_expr (e2))) ]>

      | PExpr.ListLiteral (elems) =>
        match (elems) {
          | [PExpr.Ellipsis (args)] =>
            <[ PExpr.ListLiteral ($(quoted_expr (args))) ]>
          | _ =>
            <[ PExpr.ListLiteral ($(Lift (elems, quoted_expr))) ]>
        }
        
      // rest of constructs must be in not quoted form, because they define
      // internal data structures
      | PExpr.MacroCall (name, namespc, parms) =>
        def quoted_syntax (s) {
          | SyntaxElement.Expression (body) =>
            <[ SyntaxElement.Expression ($(quoted_expr (body))) ]>

          | SyntaxElement.MatchCase (body) =>
            <[ SyntaxElement.MatchCase ($(QuotedMatchCase (body))) ]>
            
          | SyntaxElement.Function (body) =>
            <[ SyntaxElement.Function ($(quoted_fundecl (body))) ]>
            
          | SyntaxElement.Parameter (body) =>
            <[ SyntaxElement.Parameter ($(quoted_fparam (body))) ]>
            
          | SyntaxElement.ClassMember (body) =>
            <[ SyntaxElement.ClassMember ($(quoted_member (body))) ]>
            
          | SyntaxElement.TType (body) =>
            <[ SyntaxElement.TType ($(quoted_ttype (body))) ]>

          | SyntaxElement.RawToken
          | SyntaxElement.TypeBuilder
          | SyntaxElement.MethodBuilder
          | SyntaxElement.FieldBuilder
          | SyntaxElement.PropertyBuilder
          | SyntaxElement.EventBuilder
          | SyntaxElement.ParameterBuilder =>
            Util.ice ("syntax elements shouldn't appear in quotations")
        };
        assert (name.context != null);
        <[ PExpr.MacroCall ($(quoted_name (name, name.context.GetMacroContext ())),
                            NamespaceTree.ExactPath ($(Lift (namespc.Name, Lift))),
                            $(Lift (parms, quoted_syntax))) ]>

      | PExpr.Error => <[ PExpr.Error () ]>
                        
      | PExpr.Spliced (PExpr.TypeEnforcement (val, <[ $(ty : name) ]>)) =>
        match (ty.Id) {
          | "name" => <[ PExpr.Ref ($val) ]>
          | "usesite" => <[ PExpr.Ref (Name ($val, 
                                             MacroColorizer.UseColor, 
                                             MacroColorizer.UseContext)) ]>
          | "dyn" => <[ PExpr.Ref (Name ($val, -1, null)) ]>
          | "byte" => <[ PExpr.Literal (Literal.FromByte ($val)) ]>
          | "sbyte" => <[ PExpr.Literal (Literal.FromSByte ($val)) ]>
          | "short" => <[ PExpr.Literal (Literal.FromShort ($val)) ]>
          | "ushort" => <[ PExpr.Literal (Literal.FromUShort ($val)) ]>
          | "int" => <[ PExpr.Literal (Literal.FromInt ($val)) ]>
          | "uint" => <[ PExpr.Literal (Literal.FromUInt ($val)) ]>
          | "long" => <[ PExpr.Literal (Literal.FromLong ($val)) ]>
          | "ulong" => <[ PExpr.Literal (Literal.FromULong ($val)) ]>
          | "string" => <[ PExpr.Literal (Literal.String ($val)) ]>
          | "bool" => <[ PExpr.Literal (Literal.Bool ($val)) ]>
          | "char" => <[ PExpr.Literal (Literal.Char ($val)) ]>
          | "float" => <[ PExpr.Literal (Literal.Float ($val)) ]>
          | "double" => <[ PExpr.Literal (Literal.Double ($val)) ]>
          | "decimal" => <[ PExpr.Literal (Literal.Decimal ($val)) ]>

          | "typed" => <[ Macros.TypedChoose ($val) ]>
          | x =>
            Message.FatalError ("unsupported type `" + x + "' of spliced literal in expression")
        }

      | PExpr.Spliced (e) when !in_pattern => e

      | PExpr.Spliced => expr

      | PExpr.Lambda => Util.ice ("this kind of quoted fun () ... is not supported")
      | PExpr.Quoted => 
        Message.FatalError ("compound of several <[ ... ]> macro scopes is"
                             " not allowed");
      | PExpr.Typed | PExpr.TypedPattern | PExpr.TypedType => 
        Util.ice ("You've got beer from me for generating such a code (and me to)...");
      | PExpr.Ellipsis =>
        Message.FatalError (expr.Location, "List of expression parameters outside of quoted sequence:"
                             " use <[ { .. $x } ]> pattern")
    }
  } // end quoted_expr

  /** transforms given expression (which is supposed to be generated from
      quoted expression) into pattern 
   */    
  public patternize_quotation (exp : PExpr) : PExpr 
  {
    | PExpr.Ref => exp

    | <[ $obj . $mem ]> => <[ $(patternize_quotation (obj)) . $mem ]>

    // Literal.Integer (2, true, null).WithProperType ()
    // (it is created only when quoting plain numeric literals
    | <[ Literal.$_ ($v, $n, $_) . $_ () ]> =>
      <[ Literal.Integer ($v, $n, _) ]>
      
    | <[ $constr (.. $pars) ]> =>
      def (con, name) = Option.UnSome (Util.qidl_of_expr (constr));
      def last = List.Last (con);
      match (name.context.LookupType (con)) {        
        | Some (tcon) =>
          def convert_params (pars : list [PExpr], mems : list[IField], acc) {
            match ((pars, mems)) {
              | ([], []) => List.Rev (acc)

              | (<[ $(n : name) = $expr ]> :: xs, ms) =>
                convert_params (xs, ms, <[ $(n : name) = $(patternize_quotation (expr)) ]> :: acc)

              | (expr :: xs, m :: ms) =>
                convert_params (xs, ms, <[ $(m.Name : dyn) = $(patternize_quotation (expr)) ]> :: acc)

              | ([], _ :: _) => convert_params ([], [], acc)

              | (_ :: _, []) =>
                Message.FatalError ("number of supplied parameters is too large")
            }
          };

          def unalias (tcon : TypeInfo) {
            def flds = tcon.GetFields (BindingFlags.DeclaredOnly %|
                                       BindingFlags.Public %|
                                       BindingFlags.NonPublic %|
                                       BindingFlags.Instance);
          
            match (tcon.GetTydecl ()) {
              | Typedtree.TypeDeclaration.VariantOption =>
                def pars =
                  if (last.EndsWith ("ClassMember.Function"))
                    List.Tail (pars)
                  else
                    pars;
                PExpr.Call (constr, convert_params (pars, flds, []))

              | Typedtree.TypeDeclaration.Class  =>
                // we lose informations about constructor here, but as it's 
                // compiler internal computation we can ignore it
                def name_expr = Util.ExprOfQid (tcon.FullName);
                if (last.EndsWith ("Name")) 
                  <[ $name_expr where ( idl = $(patternize_quotation (List.Head (pars))) ) ]>
                else
                  PExpr.Where (name_expr,
                               PExpr.Tuple (convert_params (pars, flds, [])))

              | Typedtree.TypeDeclaration.Alias (MType.Class (tc, _)) => unalias (tc)

              | _ =>
                Util.ice ("expression generated from quotation has neither variant"
                          " nor class constructor")
            }
          }

          unalias (tcon)

        | None when last.EndsWith ("NameInCurrentColor") =>
          <[ Name where ( idl = $(patternize_quotation (List.Head (pars))) ) ]>

        | None when last.EndsWith ("ExactPath") => PExpr.Wildcard ()

        | None when last.StartsWith ("From") =>
          <[ Literal.Integer ($(last.Replace ("From", "As") : dyn)
                              = Some ($(patternize_quotation (List.Head (pars))))) ]>
          
        | None =>
          Util.ice ("expression generated from quotation has broken constructor")
      }

    | PExpr.ListLiteral (elems) =>
      PExpr.ListLiteral (List.Map (elems, patternize_quotation))
      
    | <[ (..$elems) ]> => <[ (..$(List.Map (elems, patternize_quotation))) ]>

    | PExpr.Literal 
    | PExpr.Wildcard => exp

    | PExpr.Spliced (e) => e

    | _ =>
      Message.Debug (exp.ToString ()); 
      Util.ice ("Bad constructed quoted expression in pattern matching")
  }

  public TraverseExpr (ctx : option[Typer], expr : PExpr, in_pattern : bool,
                       call : bool * bool * PExpr -> PExpr) : PExpr
  {
    Util.locate (expr.loc, {
      def expr = 
        match (ctx) {
          | Some (c) => MacroRegistry.expand_macro (c, expr)
          | _ => expr
        };

      def trav_funparms (fps) {
        def go_funparm (p : Fun_parm) {
          | <[ parameter: $n : $t = $_e ]> =>
            <[ parameter: $n : $t = $(traverse (_e)) ]>
          | _ => p
        };
        List.Map (fps, go_funparm)
      };
      def traverse (x) { TraverseExpr (ctx, x, in_pattern, call) };
      def traversep (x) { TraverseExpr (ctx, x, true, call) };      

      _ = call (in_pattern, false, expr);
 
      def recursed =
        match (expr) {
          | <[ $(_ : name) ]> => expr

          | <[ $obj . $mem ]> => 
            <[ $(traverse (obj)) . $mem ]>

          | <[ $func (.. $parms) ]> =>
            def parms = List.Map (parms, traverse);
            <[ $(traverse (func)) (..$parms) ]>

          | <[ $func .[.. $parms] ]> =>
            def parms = List.Map (parms, traverse);
            <[ $(traverse (func)) .[..$parms] ]>
            
          | <[ $target = $source ]> =>
            <[ $(traverse (target)) = $(traverse (source)) ]>

          | <[ def $n = $val ]> =>
            <[ def $(traversep (n)) = $(traverse (val)) ]>

          | <[ mutable $n = $val ]> => <[ mutable $n = $(traverse (val)) ]>

          | <[ match ($mexpr) {.. $cases } ]> =>
            def go_case (c : MatchCase) {
              
              def go_guard (g : PExpr) {
                | <[ $pat when $e ]> => <[ $(traversep (pat)) when $(traverse (e)) ]>
                | _ => traversep (g)
              };
              def <[ case: | ..$guards => $exp ]> = c;
              def guards = List.Map (guards, go_guard);
              <[ case: | ..$guards => $(traverse (exp)) ]>
            };

            def cases = List.Map (cases, go_case);
            <[ match ($(traverse (mexpr))) {.. $cases } ]>

          | <[ throw $exc ]> =>
            <[ throw $(traverse (exc)) ]>

          | <[ try $body catch { $exn is $exn_ty => $handler } ]> => 
            <[ try $(traverse (body)) catch { 
                 $exn is $exn_ty => $(traverse (handler))
               } ]>

          | <[ try $body finally $handler ]> =>
            <[ try $(traverse (body)) finally $(traverse (handler)) ]>

          | PExpr.Literal => expr

          | <[ this ]> => expr

          | <[ base ]> => expr

          | <[ typeof ($_) ]> => expr

          | <[ $expr :> $ty ]> => <[ $(traverse (expr)) :> $ty ]>

          | <[ $expr : $ty ]> => <[ $(traverse (expr)) : $ty ]>

          | <[ {.. $seq } ]> => <[ { ..$(List.Map (seq, traverse)) } ]>

          | <[ (.. $args) ]> => <[ ( ..$(List.Map (args, traverse)) ) ]>

          | <[ ref $e ]> => <[ ref $(traverse (e)) ]>

          | <[ out $e ]> => <[ out $(traverse (e)) ]>

          | <[ array (..$args) ]> =>
            <[ array ( ..$(List.Map (args, traverse)) ) ]>

          | <[ array $args ]> => <[ array $(traverse (args)) ]>

          | <[ array .[ $rank ] $args ]> =>
            <[ array .[ $(traverse (rank))] $(traverse (args)) ]>

          | <[ $obj [.. $args] ]> =>
            <[ $(traverse (obj)) [ ..$(List.Map (args, traverse)) ] ]>

          | <[ fun [ ..$tparms] (..$args) where ..$tconstrs $body ]> =>
            def args = trav_funparms (args);
            <[ fun [ ..$tparms] (..$args) where ..$tconstrs $(traverse (body)) ]>

          | <[ def ..$funs ]> =>
            def go_fun (f : Function_decl) {
              | <[ fundecl: $n [ ..$tparms] (..$args)
                   where .. $tconstrs $body ]> =>
                def args = trav_funparms (args);
                <[ fundecl: $n [ ..$tparms] (..$args)
                   where .. $tconstrs $(traverse (body)) ]>
              | _ => f
            };
            <[ def ..$(List.Map (funs, go_fun)) ]>

          | <[ $pat as $nm ]> => PExpr.As (traverse (pat), nm) 
          
          | <[ $nm where $pat ]> => PExpr.Where (traverse (nm), traverse (pat)) 
          
          | <[ $e1 is $e2 ]> => PExpr.Is (traverse (e1), traverse (e2))

          | PExpr.ListLiteral (elems) =>
            PExpr.ListLiteral (List.Map (elems, traverse))

          | PExpr.Error | PExpr.Wildcard | PExpr.Void => expr
            
          | PExpr.MacroCall (x, namespc, parms) =>
            def go_parm (y) {
              | SyntaxElement.Expression (e) =>
                SyntaxElement.Expression (traverse (e))
              | _ => y
            };
            PExpr.MacroCall (x, namespc, List.Map (parms, go_parm))

          | PExpr.Spliced (e) => PExpr.Spliced (traverse (e))

          | PExpr.Ellipsis (e) => PExpr.Ellipsis (traverse (e))

          | PExpr.Quoted (quot) =>
            def inner =
            match (quot) {
              | SyntaxElement.Expression (body) => SyntaxElement.Expression (traverse (body))
              | SyntaxElement.MatchCase 
              | SyntaxElement.Function 
              | SyntaxElement.Parameter 
              | SyntaxElement.TType 

              | SyntaxElement.ClassMember 
              | SyntaxElement.TypeBuilder 
              | SyntaxElement.FieldBuilder 
              | SyntaxElement.MethodBuilder 
              | SyntaxElement.PropertyBuilder 
              | SyntaxElement.EventBuilder
              | SyntaxElement.RawToken                
              | SyntaxElement.ParameterBuilder => quot
            }
            PExpr.Quoted (inner)

          | PExpr.Typed | PExpr.TypedPattern | PExpr.TypedType => expr

          | PExpr.Lambda => Util.ice ("Bad construction of PExpr.Lambda")
        };
      call (in_pattern, true, recursed)
    })
  }

  public RecursiveRename (tc : TypeBuilder, expr : PExpr, from : Name, to : Name) : PExpr
  {
    def rename_expr (_, is_post, e) {
      if (is_post) 
        match (e) {
          | <[ $(n : name) ]> when n.Equals (from) => 
            <[ $(to : name) ]>

          | <[ mutable $(n : name) = $val ]> when n.Equals (from) => 
            <[ mutable $(to : name) = $val ]>

          | <[ this.$(n : name) ]> when n.Equals (from) =>
            <[ this.$(to : name) ]>

          | <[ $obj.$(n : name) ]> when n.Equals (from) =>
            match (GetIfIsType (tc.GlobalEnv, obj)) {
              | Some (t) when t.Equals (tc) =>
                <[ $obj.$(to : name) ]>
              | _ => e
            }

          | PExpr.TryWith (body, Splicable.Name (exn), exn_ty, handler) 
            when exn.Equals (from) =>
            PExpr.TryWith (body, Splicable.Name (to), exn_ty, handler)
          | _ => e    
        }
      else e
    }
    Util.locate (expr.loc, {
      TraverseExpr (None (), expr, false, rename_expr)
    });
  }
}
} // end ns
