/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;
using Nemerle.Utility;
using Nemerle.IO;
using Nemerle.Compiler.Parsetree;

using Nemerle.Text;

namespace Nemerle.Compiler
{
  public module MacroRegistry
  {
    extensions : Hashtable [NamespaceTree.Node, list [SyntaxDefinition]];

    this () {
      extensions = Hashtable (50);
      Init ();
    }

    internal Init () : void
    {
      extensions.Clear ();
    }

    internal GetSyntaxExtensions (name_space : NamespaceTree.Node) : list [SyntaxDefinition]
    {
      match (extensions [name_space]) {
        | null => []
        | definition => definition
      }
    }

    public AddOperatorDefinition (info : OperatorDefinition) : void
    {
      def nsnode = info.MacroNamespace;
      match (extensions.Get (nsnode)) {
        | Some (l) => extensions.Set (nsnode, info :: l)
        | None => extensions.Set (nsnode, [info])
      }
    }
    
    public register_macro (m : IMacro) : void {
      def ns = m.GetNamespace ();
      def nsnode =
        if (ns == "") NamespaceTree.ExactPath (["Nemerle", "Core"]);
        else NamespaceTree.ExactPath (NString.Split (ns, '.'));

      def child = [m.GetName ()];
      
      match (nsnode.LookupMacro (child)) {
        | Some => Message.Error ("macro `" + ns + m.GetName () + "' already defined")
        | None =>
          def macro_ns = nsnode.Path (child);
          
          // store syntax extension in MacroRegistry
          match (m.SyntaxExtension ()) {
            | (null, _) => ()
            | (tree, permute) =>
              def key = tree.ToString (); // this is the first token of extension
              def syntax_def = SyntaxDefinition (key, m.Keywords,
                                                 macro_ns, tree.Next, permute);

              // store extensions for this namespace
              match (extensions.Get (nsnode)) {
                | Some (l) => extensions.Set (nsnode, syntax_def :: l)
                | None => extensions.Set (nsnode, [syntax_def])
              }
          };
          NamespaceTree.AddMacro (macro_ns, m);
      }
    }

    internal expand_macro (ctx : Typer, expr : PExpr) : PExpr {
      match (expr) {
        | PExpr.Call (name, args) =>
          match (Util.qidl_of_expr (name)) {
            | Some ((namepath, name)) =>
              match (name.GetEnv (ctx.Env).LookupMacro (namepath)) {
                | Some (x) =>
                  MacroColorizer.PushNewColor (name.color, name.GetEnv (ctx.Env));
                  def expanded = Util.locate (expr.loc, x.Run (ctx, x.CallTransform (args)));
                  MacroColorizer.PopColor ();
                  expand_macro (ctx, expanded);
                | None => expr
              }
            | None => expr
          }

        | PExpr.MacroCall (name, namespc, parms) =>
          match (namespc.Value) {
            | NamespaceTree.TypeInfoCache.MacroCall (m) =>
              MacroColorizer.PushNewColor (name.color, name.GetEnv (ctx.Env));
              def expanded = Util.locate (expr.loc, m.Run (ctx, parms));
              MacroColorizer.PopColor ();        
              expand_macro (ctx, expanded)
            | _ =>
              Util.ice ("failed to resolve macro name `" + namespc.Name.ToString (".") + "'")
          }

        | _ => expr
      }
    }

    internal lookup_macro (env : GlobalEnv, expr : PExpr, suff : string)
    : option [Name * IMacro * list [SyntaxElement]]
    {
      match (expr) {
        | <[ $(_ : name) ]>
        | <[ $_ . $_ ]> => lookup_macro (env, <[ $expr () ]>, suff)
          
        | <[ $name ( .. $parms ) ]> =>
            match (Util.QidOfExpr (name)) {
              | Some ((id, name)) =>
                def ctx = name.GetEnv (env);
                def id =
                  match (List.Rev (id)) {
                    | x :: xs => List.Rev (x + suff :: xs)
                    | [] => assert (false)
                  };
                match (ctx.LookupMacro (id)) {
                  | Some (m) => Some ((name, m, m.CallTransform (parms)))
                  | None => None ()
                };
              | None => None ()
            }

        | PExpr.MacroCall (name, ns, parms) =>
          match (ns.Value) {
            | NamespaceTree.TypeInfoCache.MacroCall (m) =>
              if (m.GetName ().EndsWith (suff))
                Some ((name, m, parms))
              else
                None ()
              
            | _ =>
              Util.ice ("failed to resolve macro name `" + ns.Name.ToString (".") + "'")
          }
            
        | _ => None ()
      }
    }
  }
} // end ns
