/*
 * Copyright (c) 2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Compiler;
using Nemerle.Compiler.Typedtree;

namespace Nemerle.Compiler
{
  class BuiltinMethod : IMethod
  {
    id : int;
    header : Fun_header;
    declaring : TypeInfo;
    special_kind : BuiltinMethodKind;

    internal this (parent : TypeInfo, 
                   name : string, 
                   on_type : TyVar, 
                   kind : BuiltinMethodKind)
    {
      id = Util.next_id ();
      declaring = parent;
      def fp (name, ty) {
        Fun_parm (name = name,
                  ty = ty, 
                  color = MacroColorizer.Color,
                  modifiers = Modifiers.Empty,
                  kind = ParmKind.Normal)
      }
      
      header = 
        Fun_header (
           loc = parent.Location,
           name = name,
           ret_type = on_type,
           parms = [fp ("left", on_type), fp ("right", on_type)],
           typarms = [],
           tenv = TyVarEnv ());

      special_kind = kind;
    }
    
    public GetKind () : MemberKind
    {
      MemberKind.Method (this)
    }
    
    public GetMemType () : MType
    {
      MType.ConstructFunctionType (GetHeader ())
    }

    public Location : Location
    {
      get { Nemerle.Compiler.Location.Default }
    }

    public GetId () : int
    {
      id
    }
    
    public GetHandle () : System.Reflection.MemberInfo
    {
      assert (false)
    }
    
    public CanAccess (_ : TypeInfo) : bool
    {
      true
    }
    
    public GetFreshType () : MType
    {
      GetMemType ()
    }
    
    public GetHeader () : Fun_header
    {
      header
    }

    public GetParameters () : list [Fun_parm]
    {
      header.parms
    }

    public ReturnType : TyVar
    {
      get {
        header.ret_type
      }
    }
    
    public GetFunKind () : FunKind
    {
      FunKind.Function ()
    }

    public GetMethodBase () : System.Reflection.MethodBase
    {
      assert (false)
    }
    
    public GetMethodInfo () : System.Reflection.MethodInfo
    {
      assert (false)
    }
    
    public GetConstructorInfo () : System.Reflection.ConstructorInfo
    {
      assert (false)
    }
    
    public IsVarArgs : bool
    { 
      get { false }
    }
    
    public IsFinal : bool
    {
      get { true }
    }

    public DeclaringType : TypeInfo
    {
      get { declaring }
    }
    
    public Name : string
    {
      get { header.name }
    }

    public MemberType : MemberTypes
    {
      get { MemberTypes.Method }
    }
    
    public IsStatic : bool
    { 
      get { true } 
    }
    
    public HasBeenUsed : bool
    { 
      get { true }
      set { ignore (value); }
    }

    public Attributes : NemerleAttributes {
      get { 
        NemerleAttributes.Static | 
        NemerleAttributes.Public
      }
    }
    
    public GetModifiers () : Modifiers
    {
      assert (false)
    }
    
    public OverloadName : string
    {
      get { assert (false) }
    }
      
    public BuiltinKind : BuiltinMethodKind
    {
      get { special_kind }
    }

    public override ToString () : string
    {
      NemerleMember.DescribeMember (this)
    }
  }
}
