/*
 * Copyright (c) 2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;
using Nemerle.Utility;

using Nemerle.Compiler;
using Nemerle.Compiler.Typedtree;
using Nemerle.Compiler.SolverMacros;

namespace Nemerle.Compiler
{
  class Typer4
  {
    current_fun : Fun_header;
    current_type : TypeBuilder;
    messenger : Messenger;
    the_method : NemerleMethod;

    #region Entry points
    public this (meth : NemerleMethod)
    {
      current_fun = meth.GetHeader ();
      the_method = meth;
      messenger = Passes.Solver.CurrentMessenger;
      current_type = meth.DeclaringType :> TypeBuilder;
    }


    public Run () : void
    {
      Util.locate (current_fun.loc, {
        match (current_fun.body) {
          | FunBody.Typed (body) =>
            // Message.Debug ("start ush");
            def body' = UnShare (body);
            // Message.Debug ("start T4");
            def body' = Walk (body');
            def body' =
              if (NeedBoxing (current_fun.ret_type.Fix (), body'))
                Box (current_fun.ret_type.Fix (), body')
              else body';

            goto_targets.Clear ();
            _ = Throws (body', allow_try = true);

            current_fun.body = FunBody.Typed (body');
            when (Options.ShouldDump (current_fun))
              Message.Debug ($ "after T4: $the_method [$(current_fun.name)] "
                               ":\n$(body')\n");
          | _ => assert (false)
        }
      })
    }


    static NeedBoxing (target_type : MType, expr : TExpr) : bool
    {
      def src = expr.Type.Fix ();

      ((src.IsValueType || src is MType.Void) 
       && !target_type.IsValueType)
      // emit conversions for array types, becasue there is an error for 
      // converting a value type array to non-value type array i ILEmitter
      || (target_type is MType.Array && 
          NeedConversionRec (target_type, src))
    }
    

    static Box (target_type : MType, expr : TExpr) : TExpr
    {
      Typer3.CheckedConversion (expr, target_type)
    }

    static Convert (target_type : TyVar, expr : TExpr) : TExpr
    {
      Typer3.CheckedConversion (expr, target_type)
    }

    static MaybeBoxToObject (expr : TExpr) : TExpr
    {
      if (NeedBoxing (InternalType.Object, expr))
        Box (InternalType.Object, Walk (expr))
      else
        Walk (expr)
    }


    static IsGeneric (t : MType) : bool
    {
      match (t) {
        | MType.TyVarRef => true
        | MType.Class (tc, []) => tc.Equals (InternalType.Object_tc)
        | MType.Array (t, _) => IsGeneric (t.Fix ())
        | _ => false
      }
    }

    static NeedConversionRec (real_type : MType, expr_type : MType) : bool
    {
      (IsGeneric (real_type) && ! IsGeneric (expr_type)) ||
      match (real_type) {
        | MType.Array (real, _) =>
          match (expr_type) {
            | MType.Array (expr, _) =>
              NeedConversionRec (real.Fix (), expr.Fix ())
            | _ => Util.ice ()
          }
        | _ => false
      }
    }

    static NeedConversion (real_type : MType, _expr : TExpr) : bool
    {
      IsGeneric (real_type)
      // Message.Debug ($"maybe convert $real_type to $(expr.Type) (for $expr)");
      // NeedConversionRec (real_type, expr.Type.Fix ())
    }
    #endregion


    #region Throw handling
    static goto_targets : Hashtable [int, object] = Hashtable ();

    static IsJumpTarget (expr : TExpr) : bool
    {
      def res =
        match (expr) {
          | Label (id, _) =>
            goto_targets.Contains (id);
          | Sequence (e, _) =>
            IsJumpTarget (e)
          | _ => false
        }
      expr.JumpTarget = res;
      res
    }

    static NoThrowPlease (e : TExpr) : void
    {
      when (Throws (e))
        Message.Error (e.loc, "`throw' is not allowed here");
    }

    static Throws (expr : TExpr, allow_try : bool = false) : bool
    {
      //Message.Debug ($ "throws: $(expr.GetType()) $(expr.GetHashCode())");
      def res = 
        match (expr) {
          | Throw (e) =>
            unless (e == null)
              NoThrowPlease (e);
            true

          | Goto (id) =>
            goto_targets [id] = null;
            true

          | Sequence (e1, e2)
          | DefValIn (_, e1, e2) =>
            if (Throws (e1, allow_try)) {
              if (IsJumpTarget (e2))
                Throws (e2, allow_try)
              else true
            } else Throws (e2, allow_try)

          | If (cond, e1, e2) =>
            NoThrowPlease (cond);
            def t1 = Throws (e1, allow_try);
            def t2 = Throws (e2, allow_try);
            t1 && t2

          | Assign (e1, e2) =>
            when (Throws (e1))
              Message.Error (expr.loc, "`throw' in assignment target");

            if (e1 is TExpr.LocalRef)
              if (Throws (e2, allow_try)) {
                Message.Warning (expr.loc, "bad style: each branch in this block "
                                 "ends with a break");
                true
              } else false
            else
              if (Throws (e2)) {
                Message.Error (expr.loc, "`throw' in assignment source");
                true
              } else false

          | TypeConversion (e, _, _)
          | Label (_, e) =>
            Throws (e, allow_try)

          | Switch (idx, defl, opts) =>
            NoThrowPlease (idx);
            // avoid closure
            mutable all = true;
            foreach ((_, e) in opts)
              all = Throws (e, allow_try) && all;
            match (defl) {
              | Some (e) => Throws (e) && all
              | None => all // ???
            }

          | PropertyMember
          | StaticPropertyRef
          | EventMember
          | StaticEventRef
          | ConstantObjectRef
          | Delayed
          | Error
          | DefFunctionsIn
          | Match
          | SelfTailCall =>
            Util.cassert (Message.SeenError);
            false

          | FieldMember (e, _)
          | MethodRef (e, _, _)
          | TupleIndexer (e, _, _)
          | HasType (e, _)
          | NotNull (e) =>
            NoThrowPlease (e);
            false
          
          | ArrayIndexer (obj, args) =>
            NoThrowPlease (obj);
            args.Iter (NoThrowPlease);
            false
            
          | MultipleAssign (assigns) =>
            foreach ((_, e) in assigns)
              NoThrowPlease (e);
            false

          | Array (args, dimensions) =>
            args.Iter (NoThrowPlease);
            dimensions.Iter (NoThrowPlease);
            false
            
          | Call (func, parms, _) =>
            NoThrowPlease (func);
            foreach (parm in parms) NoThrowPlease (parm.expr);
            false
            
          | Tuple (exprs) =>
            exprs.Iter (NoThrowPlease);
            false

          | TryWith (body, _, handler)
          | TryFinally (body, handler) =>
            when (! allow_try)
              // use ice here?
              Message.Error (expr.loc, 
                             "try-blocks cannot be used inside expressions, "
                             "this message shouldn't happen though");
            _ = Throws (body, true);
            _ = Throws (handler, true);
            false

          | StaticRef
          | LocalRef
          | ImplicitValueTypeCtor
          | Literal
          | This
          | Base
          | TypeOf
          | OpCode
          | MethodAddress
          | DefaultValue =>
            false
        }

      expr.Throws = res;
      res
    }
    #endregion


    #region Unsharing and throw handling
    static UnShare (expr : TExpr) : TExpr
    {
      expr.Walk (DoUnShare)
    }
    
    static DoUnShare (expr : TExpr) : TExpr
    {
      // Message.Debug ($"unshare $expr");
      if (expr.Visited) {
        def expr = expr.Copy ();
        Util.cassert (!expr.Visited);
        UnShare (expr)
      } else {
        expr.Visited = true;
        null
      }
    }
    #endregion


    #region Top level stuff
    static Walk (expr : TExpr) : TExpr
    {
      expr.Walk (DoWalk)
    }


    static DoWalk (expr : TExpr) : TExpr
    {
      // Message.Debug ($ "dowalk: $(expr.GetType()) $(expr.Type)");
      def res =
        match (expr) {
          | FieldMember (obj, fld) when ! expr.IsAssigned =>
            def obj = Walk (obj);
              
            // use address here for better performance
            when (! obj.NeedAddress && fld.DeclaringType.IsValueType)
              obj.NeedAddress = true;
              
            if (NeedConversion (fld.GetMemType (), expr))
              Convert (expr.Type,
                       TExpr.FieldMember (fld.GetMemType (), obj, fld))
            else
              TExpr.FieldMember (obj, fld)

          | TupleIndexer (obj, pos, len) when NeedConversion (InternalType.Object, expr) =>
            Convert (expr.Type,
                     TExpr.TupleIndexer (InternalType.Object,
                                         Walk (obj), pos, len));

          | Assign (e1, e2) =>
            e1.IsAssigned = true;

            def real_type =
              match (e1) {
                | FieldMember (_, fld)
                | StaticRef (fld is IField) => fld.GetMemType ()
                | LocalRef (decl) => decl.Type.Fix ()
                | ArrayIndexer
                | This => e1.Type.Fix ()
                | _ => Util.ice ($ "wrong assignment target $e1")
              }
            def e1 = Walk (e1);
            def e2 = Walk (e2);
            
            if (e1.Type.Fix ().IsValueType && ! e1.IsAddressable)
              Message.Error ("this expression is not a proper lvalue: "
                             "cannot load value type address");
            else
              e1.NeedAddress = true;

            if (NeedBoxing (real_type, e2))
              TExpr.Assign (InternalType.Void, e1, Box (real_type, e2))
            else 
              TExpr.Assign (InternalType.Void, e1, e2)

          
          | Call (ImplicitValueTypeCtor, [], _) => null
          
          // I hope opcodes don't need boxing ...
          | Call (OpCode, _, _) => null
          
          | Call (origfunc, parms, is_tail) =>
            mutable func = Walk (origfunc);
            def meth =
              match (func) {
                | MethodRef (obj, meth, nonvirt) =>
                
                  // we would kinda like address here
                  when (obj.Type.Fix ().IsValueType) {
                    def methty = meth.DeclaringType;
                    if (methty.IsValueType)
                      obj.NeedAddress = true;
                    else {
                      // but maybe we should employ boxing
                      def obj = Box (methty.GetMemType (), obj);
                      func = TExpr.MethodRef (func.Type, obj, meth, nonvirt);
                    }
                  }
                  
                  meth
                  
                | Base (meth)
                | StaticRef (meth is IMethod) => meth
                | _ => Util.ice ($ "invalid thing called $func")
              }

            unless (parms.IsEmpty) {
              if (meth.DeclaringType.IsDelegate && meth.Name == ".ctor") {
                def parm = parms.Head;
                when (NeedBoxing (InternalType.Object, parm.expr))
                  parm.expr = Box (InternalType.Object, parm.expr);
              } else {
                mutable formals = meth.GetParameters ();
                
                Util.cassert (formals.Length == parms.Length,
                              $ "call to $meth $parms $formals");
                
                foreach (parm in parms) {
                  match (formals) { 
                    | f :: fs =>
                      def t = f.ty.Fix ();
                      parm.expr = Walk (parm.expr);
                      if (parm.kind == ParmKind.Normal) {
                        when (NeedBoxing (t, parm.expr))
                          parm.expr = Box (t, parm.expr);
                      } else {
                        if (parm.expr.IsAddressable)
                          parm.expr.NeedAddress = true;
                        else
                          Message.Error ($ "non-addressable expression passed "
                                           "as a ref/out parameter `$(f.name)'");
                      }
                      formals = fs;
                    | [] => Util.ice ();
                  }
                }
              }
            }

            if (NeedConversion (meth.ReturnType.Fix (), expr))
              Convert (expr.Type,
                       TExpr.Call (meth.ReturnType.Fix (), func, parms, false))
            else
              TExpr.Call (func, parms, is_tail)

          | Tuple (exprs) =>
            TExpr.Tuple (exprs.Map (MaybeBoxToObject))

          | Array (parms, dimensions) =>
            def ty =
              match (expr.Type.Fix ()) {
                | MType.Array (t, _) => t.Fix ()
                | _ => Util.ice ()
              }
            mutable res = [];
            foreach (parm in parms) {
              def parm = Walk (parm);
              def parm =
                if (NeedBoxing (ty, parm))
                  Box (ty, parm)
                else parm;
              res = parm :: res;
            }
            TExpr.Array (res.Rev (), dimensions)

          | DefFunctionsIn
          | Match
          | SelfTailCall =>
            Util.cassert (Message.SeenError);
            null
          
          | _ => null
        }
      //Message.Debug ($"do walk: $expr -> $res");
      res
    }
    #endregion
  }
}
