/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;
using Nemerle.Utility;

using Nemerle.Compiler;
using Nemerle.Compiler.Typedtree;
using Nemerle.Compiler.SolverMacros;

namespace Nemerle.Compiler
{
  class Typer3
  {
    // this can be Fun_header of a function expanded to a loop:
    current_fun : Fun_header;
    // and this not:
    mutable real_current_fun : Fun_header;
    mutable current_closure : LocalValue;
    mutable local_function_type : TypeBuilder;
    current_type : TypeBuilder;
    messenger : Messenger;
    mutable static_proxies : Hashtable [int, IField];
    mutable closure_fields : Hashtable [int, IField];
    mutable start_label : int;
    the_method : NemerleMethod;
    mutable redirects : Hashtable [int, LocalValue];

    #region Entry points
    public this (ty : TypeBuilder, meth : NemerleMethod)
    {
      this (null, ty, meth);
    }


    this (parent : Typer3, ty : TypeBuilder, meth : NemerleMethod)
    {
      this (parent, ty, meth.GetHeader ());
      the_method = meth;
    }


    this (parent : Typer3, ty : TypeBuilder, fn : Fun_header)
    {
      current_fun = fn;
      current_type = ty;
      real_current_fun = current_fun;

      if (parent != null) {
        static_proxies = parent.static_proxies;
        redirects = parent.redirects;
      } else {
        static_proxies = Hashtable ();
        redirects = Hashtable ();
      }

      messenger = Passes.Solver.CurrentMessenger;
    }


    public Run () : void
    {
      Util.locate (current_fun.loc, {
        //Message.Debug ($"T3::run: $(current_fun.name)");
        start_label = Util.next_id ();

        def initializers = PrepareProlog ();

        match (current_fun.body) {
          | FunBody.Typed (body) =>
            // FIXME?: Typed3 
            def body = BuildRevSequence (Walk (body) :: initializers);
            def expr = TExpr.Label (body.Type, start_label, body);
            current_fun.body = FunBody.Typed (expr);
            when (Options.ShouldDump (current_fun))
              Message.Debug ($ "after T3: $current_type.$(current_fun.name) "
                               "-> $(current_fun.ret_type) : $body\n");

            // run this for toplevel methods
            when (the_method != null) {
              def t4 = Typer4 (the_method);
              t4.Run ();
            }
            
          | _ => assert (false)
        }
      })
    }
    #endregion


    #region Utilities
    // expects reversed list
    internal static BuildRevSequence (exprs : list [TExpr]) : TExpr
    {
      match (exprs) {
        | [] =>
          TExpr.Literal (InternalType.Void, Literal.Void ())
        | x :: xs =>
          def loop (acc, l) {
            match (l) {
              | [] => acc
              | TExpr.DefValIn (name, val, null) :: xs =>
                loop (TExpr.DefValIn (acc.Type, name, val, acc), xs)
              | x :: xs =>
                loop (TExpr.Sequence (acc.Type, x, acc), xs)
            }
          }
          loop (x, xs)
      }
    }


    static SingleMemberLookup (tb : TypeInfo, name : string) : IMember
    {
      match (tb.LookupMember (name)) {
        | [mem] => mem
        | _ => assert (false)
      }
    }


    IsTopLevelFun : bool
    {
      get { real_current_fun.decl == null }
    }


    /** Just a shorthand for TExpr.LocalRef.  */
    static PlainRef (decl : LocalValue) : TExpr
    {
      assert (decl != null);
      TExpr.LocalRef (decl.Type, decl)
    }


    static StaticRef (mem : IMember) : TExpr
    {
      assert (mem is IMethod || mem is IField);
      TExpr.StaticRef (mem.GetMemType (), mem)
    }

   
    static internal CheckedConversion (expr : TExpr, target_type : TyVar) : TExpr
    {
      TExpr.TypeConversion (target_type, expr, target_type,
                            ConversionKind.IL (true));
    }


    /** Given a call with [parms] to [fh] return a list of parameters
        that should be passed along with any initialization code.  */
    TupleParms (fh : Fun_header, parms : list [Parm]) 
               : list [Parm] * list [TExpr]
    {
      if (parms.Length == 1 && fh.parms.Length > 1) {
        def tupled = List.Hd (parms).expr;
        def cache =
          LocalValue (current_fun, Util.tmpname ("tupl_cache"),
                      tupled.Type, LocalValue.Kind.Plain (),
                      is_mutable = false);
        cache.Register ();
        cache.UseFrom (current_fun);

        def len = fh.parms.Length;
        mutable pos = -1;
        def parms = fh.parms.Map (fun (fp) {
          def ty =
            if (fp.decl == null) fp.ty
            else fp.decl.Type;
          pos++;
          Parm (TExpr.TupleIndexer (ty, PlainRef (cache), pos, len))
        });
         
        (parms, 
         [TExpr.DefValIn (cache, tupled, null)])

      } else if (parms.Length > 1 && fh.parms.Length == 1) {
        def types = parms.Map (fun (fp) { fp.expr.Type });
        def exprs = parms.Map (fun (fp) { fp.expr });
        def parm = TExpr.Tuple (MType.Tuple (types), exprs);
        ([Parm (parm)], [])

      } else {
        Util.cassert (parms.Length == fh.parms.Length,
                      $ "parms length mismatch, $(fh.name) "
                        "$parms $(fh.parms)");
        (parms, [])
      }
    }


    WithCached (e : TExpr, f : TExpr -> TExpr) : TExpr
    {
      def needs_cache =
        match (e) {
          | TExpr.LocalRef             
          | TExpr.StaticRef            
          | TExpr.ConstantObjectRef    
          | TExpr.Literal              
          | TExpr.This => false
          | _ => true
        }

      if (needs_cache) {
        def cache =
          LocalValue (current_fun, Util.tmpname ("cache"),
                      e.Type, LocalValue.Kind.Plain (),
                      is_mutable = false);
        cache.Register ();
        cache.UseFrom (current_fun);
        def body = f (PlainRef (cache));
        TExpr.DefValIn (body.Type, cache, e, body)
      } else f (e)
    }
    #endregion


    #region Local reference through closures
    /** Return code referencing closure of [hd].  */
    ClosureRef (hd : Fun_header) : TExpr
    {
      if (hd.id == real_current_fun.id) {
        PlainRef (current_closure)
      } else {
        Util.cassert (closure_fields != null);
        Util.cassert (closure_fields.Contains (hd.id), $ "no clo for $(hd.name)");
        def field = closure_fields [hd.id];
        TExpr.FieldMember (field.GetMemType (),
                           TExpr.This (local_function_type.GetMemType ()),
                           field)
      }
    }


    LocalRef (decl : LocalValue, for_store : bool = false) : TExpr
    {
      // Message.Debug ($"local ref $decl $(decl.Id) $for_store");
      def decl =
        if (redirects.Contains (decl.Id))
          redirects [decl.Id]
        else decl;
      def is_this = decl.ValKind is LocalValue.Kind.ClosurisedThisPointer;

      def res =
        if (decl.InClosure)
          if (!for_store && is_this && IsTopLevelFun)
            TExpr.This ()
          else {
            assert (decl.ClosureField != null);
            TExpr.FieldMember (decl.ClosureField.GetMemType (), 
                               ClosureRef (decl.DefinedIn), decl.ClosureField)
          }
        else if (is_this)
          TExpr.This ()
        else
         PlainRef (decl);

      when (res.ty == null)
        res.ty = decl.Type;
      res
    }
    #endregion


    #region Proxies
    static AddTupledMethod (tb : TypeBuilder, parm_cnt : int) : void
    {
      if (parm_cnt == 0)
        tb.Define (<[ decl:
          public apply (_ : object) : object {
            apply ()
          } ]>)
      else if (parm_cnt == 1) {}
      else {
        def objects = List.Repeat (<[ object ]>, parm_cnt);
        tb.Define (<[ decl:
          public apply (o : object) : object
          {
            apply (o :> @* (.. $objects))
          }
        ]>);
      }
    }
    

    EmitStaticProxy (meth : IMethod) : TExpr
    {
      unless (static_proxies.Contains (meth.GetId ())) {
        def parms = meth.GetHeader ().parms;
        def parm_cnt = parms.Length;
        def fnty = InternalType.GetFunctionType (parm_cnt).InternalType;
        
        def name = Macros.NewSymbol ("static_proxy");
        def parm_names =
          parms.Map (fun (_) { Macros.NewSymbol ("sp_parm") });
        def formal_parms =
          parm_names.Map (fun (name) { <[ parameter: $(name : name) : object ]> });
        def parm_refs =
          List.Map2 (parm_names, parms, fun (name, parm) {
            <[ $(name : name) :> $(parm.ty : typed) ]> 
          });

        def meth_name =
          if (meth.GetFunKind () is FunKind.Constructor)
            meth.DeclaringType.FullName
          else
            meth.DeclaringType.FullName + "." + meth.Name;
          
        def builder =
          current_type.DefineNestedType (<[ decl:
            private sealed class $(name : name) : $(fnty : typed)
            {
              public static single_instance : $(name : name);

              private this ()
              {
              }
              
              static this ()
              {
                single_instance = $(name : name) ();
              }

              public apply (.. $formal_parms) : object
              {
                $(Util.ExprOfQid (meth_name)) (.. $parm_refs)
              }
            }
          ]>);

        AddTupledMethod (builder, parm_cnt);
        
        builder.MarkWithSpecialName ();
        builder.Compile ();

        static_proxies [meth.GetId ()] =
          SingleMemberLookup (builder, "single_instance") :> IField;
      }

      def field = static_proxies [meth.GetId ()];
      StaticRef (field)
    }
    

    EmitDelegateProxy (expr : TExpr) : TExpr * IMethod
    {
      def decl =
        match (expr.Type.Fix ()) {
          | MType.Fun (from, ret_type) as ty =>
            def parms =
              from.Fix ().GetFunctionArguments ().Map (fun (ty) {
                def name = Macros.NewSymbol ("parm");
                (<[ parameter: $(name : name) : $(ty : typed) ]>,
                 <[ $(name : name) ]>)
              });
            def (parms, parm_refs) = List.Split (parms);
            
            <[ decl: 
              private sealed class $(Macros.NewSymbol ("delegate_proxy") : name)
              {
                funptr : $(ty : typed);
                public this (fp : $(ty : typed))
                {
                  funptr = fp;
                }

                public InvokeDelegate (.. $parms) : $(ret_type : typed)
                {
                  funptr (.. $parm_refs)
                }
              }
            ]>

          | _ => assert (false)
        }

      def tb = current_type.DefineNestedType (decl);
      tb.MarkWithSpecialName ();
      tb.Compile ();

      def ctor = SingleMemberLookup (tb, ".ctor");

      def ctor_call =
        TExpr.Call (tb.GetMemType (), StaticRef (ctor), [Parm (expr)], false);

      (ctor_call, SingleMemberLookup (tb, "InvokeDelegate") :> IMethod)
    }
    #endregion


    #region Function prolog
    PrepareClosure () : list [TExpr]
    {
      //Message.Debug ($"closure for $(current_fun.name) $(current_fun.closure_vars)");
      if (current_fun.closure_vars.IsEmpty) []
      else {
        def clo_type = current_type.DefineNestedType (<[ decl:
          private sealed class $(Macros.NewSymbol ("closure") : name)
          {
            internal this () {}
          }
        ]>);
        current_fun.closure_type = clo_type;
        def closure_val =
          LocalValue (current_fun, "_N_closure", clo_type.GetMemType (), 
                      LocalValue.Kind.Plain (), is_mutable = false);
        closure_val.Register ();
        closure_val.UseFrom (current_fun);
        current_closure = closure_val;
        foreach (decl in current_fun.closure_vars) {
          def ptdecl = <[ decl: 
            internal mutable 
              $(Macros.NewSymbol (decl.Name) : name) : $(decl.Type : typed);
          ]>;
          def fld = Option.UnSome (clo_type.DefineAndReturn (ptdecl));
          decl.ClosureField = fld :> IField;
          decl.ClosureField.HasBeenAssigned = true;
        }
        def ctor = SingleMemberLookup (clo_type, ".ctor");
        def ctor_call = 
          TExpr.Call (clo_type.GetMemType (), StaticRef (ctor), [], false);
                      
        clo_type.MarkWithSpecialName ();
        clo_type.HasBeenUsed = true;
        clo_type.Compile ();

        [TExpr.DefValIn (InternalType.Void, closure_val, ctor_call, null)]
      }
    }


    LoadParameters () : list [TExpr]
    {
      mutable initializers = [];
      
      foreach (fp in current_fun.parms) {
        def parm = fp.decl;
        assert (parm != null);

        def parmtype = parm.Type.Fix ();

        def need_cast =
          fp.ty.Fix ().IsSystemObject && ! parmtype.IsSystemObject;

        when (need_cast)
          parm.SetObjectType (); // modify its type

        // Message.Debug ($"handle fp $parm $(parm.Id) $(parm.Type)");
        if (parm.InClosure) {
          def expr =
            if (need_cast)
              CheckedConversion (PlainRef (parm), parmtype)
            else
              PlainRef (parm);
          def a = TExpr.Assign (InternalType.Void,
                                LocalRef (parm, for_store = true), 
                                expr);
          
          initializers = a :: initializers
        // this should be dead code with generics
        } else if (need_cast) {
          def local = LocalValue (current_fun, parm.Name, parmtype, 
                                  LocalValue.Kind.Plain (), is_mutable = parm.IsMutable);
          local.Register ();
          local.UseFrom (current_fun);
          // Message.Debug ($ "redirect $parm $(parm.Id) --> $local $(local.Id)");
          redirects [parm.Id] = local;
          def val =
            CheckedConversion (PlainRef (parm), parmtype);
          def a = TExpr.DefValIn (InternalType.Void, local, val, null);
          initializers = a :: initializers
        } else {}
      }

      initializers
    }


    GetBaseCall () : option [TExpr]
    {
      // put base () / this () call before storing 'this' in closure inside constructor
      if (current_fun.name == ".ctor" && !current_type.IsValueType)
        match (current_fun.body) {
          | FunBody.Typed (TExpr.Sequence (TExpr.Call as basecall, rest)) =>
            current_fun.body = FunBody.Typed (rest);
            Some (Walk (basecall))

          | FunBody.Typed (TExpr.Call as basecall) =>
            current_fun.body = FunBody.Typed (BuildRevSequence ([]));
            Some (Walk (basecall))

          | FunBody.Typed (body) =>
            Message.Warning (body.loc, body.ToString ());
            assert (false)

          | _ => assert (false)
        }
      else None ()
    }
    

    // the result is reversed!
    PrepareProlog () : list [TExpr]
    {
      // assigments of parameters and 'this' to closure fields
      // interleaved with base (..) call in constructor
      mutable initializers = [];

      // we build the initialization stuff of method:
      // 1. method's closure ctor()
      // 2. store parameters into closure
      // 3. base (..) / this (..) for constructors
      // 4. store 'this' in closure

      initializers = PrepareClosure ();

      initializers = LoadParameters () + initializers;

      match (GetBaseCall ()) {
        | Some (call) =>
          initializers = call :: initializers;
        | None => {}
      }

      // store 'this' into closure object
      foreach (d in current_fun.closure_vars) 
        when (d.InClosure && 
              d.ValKind is LocalValue.Kind.ClosurisedThisPointer) {
          def ini =
            TExpr.Assign (InternalType.Void,
                          LocalRef (d, for_store = true),
                          TExpr.This (current_type.GetMemType ()));
          initializers = ini :: initializers;
        }

      initializers
    }
    #endregion


    #region Local function generation
    static ParentsWithClosures (h : Fun_header) : list [Fun_header]
    {
      def loop (fh : Fun_header, acc) {
        match (fh.closure_vars) {
          | [] => acc
          | vars => 
            if (vars.Exists (fun (var) {
                // FIXME: doesn't work, because function have to be removed from external closures
                //var.id != h.decl.id && // call to our function shouldn't be closurised
                (h :: h.children_funs).Exists (fun (child) { 
                  var.UsedIn.Contains (child) 
                })
              })) 
              fh :: acc
            else
              acc
        }
      }
      List.FoldLeft (h.GetParents (), [], loop);
    }


    static PrepareForEmission (meth : NemerleMethod, fn : Fun_header) : void
    {
      def new_header = meth.fun_header;
      meth.fun_header = fn;

      if (new_header.parms.Length > 1 && fn.parms.Length == 1) {
        match (fn.body) {
          | FunBody.Typed (body) =>
            def vals = new_header.parms.Map (fun (parm : Fun_parm) {
              def local = LocalValue (fn, parm.name, parm.ty, 
                                      LocalValue.Kind.Plain (), is_mutable = false);
              local.Register ();
              local.UseFrom (fn);
              parm.decl = local;
              local
            });
            def parm = fn.parms.Head;
            def expr =
              TExpr.DefValIn (body.Type,
                              parm.decl, 
                              TExpr.Tuple (parm.ty, vals.Map (PlainRef)),
                              body);
            fn.body = FunBody.Typed (expr);
            fn.parms = new_header.parms;
          | _ => assert (false)
        }
      } else {
        List.Iter2 (fn.parms, new_header.parms, fun (orig, copy : Fun_parm) {
          copy.decl = orig.decl;
        });
        fn.parms = new_header.parms;
      }

      fn.ret_type = new_header.ret_type;
    }


    EmitStaticLocalFunction (fn : Fun_header) : void
    {
      def parms = fn.parms.Map (fun (fp) {
        <[ parameter: $(fp.decl.Name : dyn) : $(fp.decl.Type : typed) ]>
      });
        
      def meth =
        Option.UnSome (current_type.DefineAndReturn (<[ decl:
          private static $(Macros.NewSymbol (fn.name) : name) (.. $parms) 
                    : $(fn.ret_type : typed)
          {
          }
        ]>)) :> NemerleMethod;

      PrepareForEmission (meth, fn);
      fn.static_method = meth;

      def child = Typer3 (this, current_type, meth);
      child.Run ();
    }


    EmitFunctionalValue (fn : Fun_header, closures : list [Fun_header]) : TExpr
    {
      // we cannot use fn.parms.Length here, because if the argument is a tuple
      // we will want more arguments
      def parm_types =
        match (MType.ConstructFunctionType (fn)) {
          | MType.Fun (from, _) =>
            from.Fix ().GetFunctionArguments ()
          | _ => assert (false)
        }
      def parm_cnt = parm_types.Length;
      def fnty = InternalType.GetFunctionType (parm_cnt).InternalType;
      
      def formal_parms =
        parm_types.Map (fun (_) { <[ parameter: 
          $(Macros.NewSymbol ("fp") : name) : object
        ]> });

      def builder = current_type.DefineNestedType (<[ decl:
        private sealed class $(Macros.NewSymbol ("lambda") : name) : $(fnty : typed)
        {
          public apply (.. $formal_parms) : object
          {
          }
        }
      ]>, false);
      builder.DisableImplicitConstructor ();
      builder.FixupDefinedClass ();

      def clo_fields = Hashtable ();

      def (parms, assigns) =
        List.Split (closures.Map (fun (hd) {
          def name = Macros.NewSymbol (hd.name + "_clo");
          Util.cassert (hd.closure_type != null, $ "null closure for $(hd.name)");
          def clo_type = hd.closure_type.GetMemType ();
          def field =
            Option.UnSome (builder.DefineAndReturn (<[ decl: 
              $(name : name) : $(clo_type : typed) 
            ]>)) :> IField;
          clo_fields [hd.id] = field;
          (<[ parameter: $(name : name) : $(clo_type : typed) ]>,
           <[ this . $(name : name) = $(name : name) ]>)
        }));

      def ctor =
        builder.DefineAndReturn (<[ decl:
          public this (.. $parms)
          {
            { .. $assigns }
          }
        ]>);
      def ctor = Option.UnSome (ctor);
      
      def the_method = SingleMemberLookup (builder, "apply") :> NemerleMethod;
      AddTupledMethod (builder, parm_cnt);

      PrepareForEmission (the_method, fn);

      def child = Typer3 (this, current_type, the_method);
      child.local_function_type = builder;
      child.closure_fields = clo_fields;
      child.Run ();

      builder.MarkWithSpecialName ();
      builder.Compile ();

      def ctor_parms = closures.Map (fun (hd) { Parm (ClosureRef (hd)) });
      TExpr.Call (fnty, StaticRef (ctor), ctor_parms, false)
    }

    
    HandleLocalFunction (fn : Fun_header, is_single : bool) : list [TExpr]
    {
      def closures = ParentsWithClosures (fn);

      //Message.Debug ($"hlf: $(fn.name) $(fn.usage) $closures");
      match (fn.usage) {
        | FunctionUsage.UsedJustOnce => 
          // handled in EmitLoop
          []

        | FunctionUsage.NotUsed =>
          // obvious
          []

        | FunctionUsage.Used when is_single && closures.IsEmpty =>
          EmitStaticLocalFunction (fn);
          []

        | FunctionUsage.Used
        | FunctionUsage.UsedAsFirstClass =>
          def fval = EmitFunctionalValue (fn, closures);
          assert (fn.decl != null);
          [TExpr.DefValIn (fn.decl, fval, null)] 
      }
    }
    #endregion
    

    #region Language constructs
    EmitLoop (hd : Fun_header, parms : list [Parm]) : TExpr
    {
      def child = Typer3 (this, current_type, hd);
      // the types are the same
      child.real_current_fun = real_current_fun;
      child.local_function_type = local_function_type;
      // the child can reuse our closure
      child.current_closure = current_closure;
      child.closure_fields = closure_fields;
      child.Run ();

      def body =
        match (hd.body) {
          | FunBody.Typed (x) => x
          | _ => assert (false);
        }

      // need to pass parameters
      def (parms, ini) = TupleParms (hd, parms);
      def assigns = List.RevMap2 (parms, hd.parms, fun (actual, formal) {
        assert (actual.kind == ParmKind.Normal);
        TExpr.DefValIn (formal.decl, actual.expr, null)
      });

      BuildRevSequence (body :: (assigns + ini));
    }


    EmitDelegateCtor (ctor : IMethod, parm : TExpr) : TExpr
    {
      mutable is_virt = false;
      
      def (obj, meth) =
        match (parm) {
          | TExpr.StaticRef (meth is IMethod) =>
            (TExpr.Literal (InternalType.Object, Literal.Null ()),
             meth)
             
          | TExpr.MethodRef (obj, meth, nonvirt) =>
            is_virt = !nonvirt;
            (Walk (obj), meth)
            
          | _ =>
            EmitDelegateProxy (Walk (parm))
        }

      TExpr.Call (ctor.DeclaringType.GetMemType (),
                  StaticRef (ctor),
                  [Parm (obj), Parm (TExpr.MethodAddress (meth, is_virt))], false)
    }


    EmitCall (ret_type : TyVar, func : TExpr, parms : list [Parm], is_tail : bool) : TExpr
    {
      foreach (p in parms)
        p.expr = Walk (p.expr);

      def just_call (meth, func) {
        def (parms, ini) = TupleParms (meth.GetHeader (), parms);
        def call = TExpr.Call (ret_type, func, parms, is_tail);
        BuildRevSequence (call :: ini)
      }

      def plain_call () {
        def meth = InternalType.GetFunctionType (parms.Length).ApplyMethod;
        just_call (meth, TExpr.MethodRef (func.ty, Walk (func), meth, false))
      }

      match (func) {
        | TExpr.MethodRef (obj, meth, notvirt) =>
          just_call (meth,
                     TExpr.MethodRef (func.ty, Walk (obj), meth, notvirt))
          
        | TExpr.ConstantObjectRef => Util.ice ()
        
        | TExpr.Base (meth)
        | TExpr.StaticRef (meth is IMethod) =>
          just_call (meth, func)
          
        | TExpr.OpCode => TExpr.Call (func, parms, false)

        | TExpr.LocalRef (LocalValue where 
                          (ValKind = LocalValue.Kind.Function (hd, _))) =>
          match (hd.usage) {
            | FunctionUsage.UsedJustOnce =>
              EmitLoop (hd, parms)

            | _ =>
              if (hd.static_method != null)
                just_call (hd.static_method,
                           StaticRef (hd.static_method))
              else
                plain_call ()
          }

        | _ => plain_call ()
      }
    }
    #endregion

    
    #region Matching compilation
    CompileMatch (m : TExpr.Match) : TExpr
    {
      mutable vals = Set ();
      
      foreach (case in m.cases)
        foreach ((pat, _) in case.patterns)
          _ = pat.Walk (fun (_) {
            | Pattern.As (_, decl) =>
              vals = vals.Replace (decl);
              null
            | _ => null
          });

      def expr = 
        WithCached (m.expr, fun (e) {
          MatchingCompiler.Run (m.Type, e, m.cases)
        });

      def expr =
        vals.Fold (expr, fun (decl, expr) {
          if (decl.InClosure)
            expr
          else {
            // Message.Debug ($"store $decl $(decl.Id)");
            decl.UseFrom (current_fun);
            TExpr.DefValIn (expr.Type, decl, 
                            TExpr.DefaultValue (decl.Type), expr)
          }
        });

      Walk (expr)
    }
    #endregion


    #region Top level stuff
    Walk (expr : TExpr) : TExpr
    {
      expr.Walk (DoWalk)
    }


    DoWalk (expr : TExpr) : TExpr
    {
      // Message.Debug ($ "dowalk: $(expr.GetType()) $(expr)");
      match (expr) {
        | TExpr.LocalRef (decl) =>
          LocalRef (decl)
          
        | TExpr.StaticRef (_ is IField) =>
          null

        | TExpr.StaticRef (meth is IMethod) =>
          EmitStaticProxy (meth)

        // we do not support any other staticrefs here
        // everything should be handled by Typer2 already
        | TExpr.StaticRef => assert (false)
          
        | TExpr.DefFunctionsIn (funs, body) =>
          def res =
            match (funs) {
              | [single] =>
                HandleLocalFunction (single, is_single = true);
              | _ =>
                List.Flatten (funs.Map (fun (fn) {
                  HandleLocalFunction (fn, is_single = false)
                }))
            }
          Walk (BuildRevSequence (body :: res))

        | TExpr.DefValIn (decl, val, body) when decl.InClosure =>
          if (val is TExpr.DefaultValue)
            Walk (body)
          else {
            def assign =
              TExpr.Assign (InternalType.Void, LocalRef (decl), Walk (val));
            TExpr.Sequence (assign, Walk (body))
          }

        // handled by Typer2.LambdaTransform
        | TExpr.MethodRef => assert (false)

        | TExpr.Call (TExpr.StaticRef (m is IMethod), [parm], _)
          when
            m.GetFunKind () is FunKind.Constructor && 
            m.DeclaringType.IsDelegate =>
          EmitDelegateCtor (m, parm.expr)

        | TExpr.Call (func, parms, is_tail) =>
          EmitCall (expr.Type, func, parms, is_tail)

        | TExpr.SelfTailCall (parms) =>
          def (parms, ini) = TupleParms (current_fun, parms);
          def assigns =
            List.Map2 (parms, current_fun.parms, fun (parm, fp) {
              assert (parm.kind == ParmKind.Normal);
              def ty = fp.ty;
              def conv = CheckedConversion (Walk (parm.expr), ty);
              (fp.decl, conv)
            });

          def beg =
            if (assigns.IsEmpty)
              ini
            else
              TExpr.MultipleAssign (InternalType.Void, assigns) :: ini;
              
          def goto = TExpr.Goto (InternalType.Void, start_label);
          
          BuildRevSequence (goto :: beg)

        | TExpr.ConstantObjectRef (mem) =>
          // FIXME this doesn't seem to be the right place for such a message
          Message.Warning ("using a constant object reference directly");
          Message.Warning ("  you probably have meant to write `" +
                           mem.DeclaringType.FullName + " ()'");
          def meth = SingleMemberLookup (mem.DeclaringType, 
                                         "_N_constant_object_generator");
          Walk (StaticRef (meth))

        | TExpr.Match as m =>
          CompileMatch (m)

        // we cannot handle closurised values as placeholders for exceptions
        // so we use a fresh variable for exception and assign it using
        // regular DefValIn
        | TExpr.TryWith (body, orig, handler) =>
          if (orig.InClosure) {
            def val =
              LocalValue (current_fun, orig.Name,
                          orig.Type,
                          LocalValue.Kind.ExceptionValue (),
                          is_mutable = false);
            val.Register ();
            val.UseFrom (current_fun);
            def handler =
              TExpr.DefValIn (handler.Type, orig, PlainRef (val), handler);
            TExpr.TryWith (Walk (body), val, Walk (handler))
          } else null


        // optimize ifs
        | If (cond, e1, e2) => 
          match (Walk (cond)) {
            | Literal (Bool (lit)) => Walk (if (lit) e1 else e2)
            
            | Sequence (prep, Literal (Bool (lit))) =>
              def e = Walk (if (lit) e1 else e2);
              TExpr.Sequence (e.Type, prep, e)
              
            | cond =>
              match ((Walk (e1), Walk (e2))) {
                | (Literal (Bool (true)), Literal (Bool (false))) =>
                  cond
                | (e1, e2) =>
                  TExpr.If (cond, e1, e2)
              }
          }
          
        | _ => null
      }
    }
    #endregion
  }
}
