/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;
using Nemerle.Utility;

using Nemerle.Compiler;
using Nemerle.Compiler.Typedtree;
using Nemerle.Compiler.SolverMacros;

namespace Nemerle.Compiler
{
  class ImplementsWrapperMaker
  {
    the_method : NemerleMethod;
    current_type : TypeBuilder;

    public this (meth : NemerleMethod)
    {
      the_method = meth;
      current_type = meth.DeclaringType :> TypeBuilder;
    }
    

    public Run () : void
    {
      // Message.Debug ($"check for impl wrap $the_method");
      // now we will create special wrappers implementing interface methods
      // when our methods are co/contra-variant (and .NET doesn't allow it automatically)
      match (the_method.GetFunKind ()) {
        | FunKind.BoundMethod (impl_list) =>
          def is_correct_impl (impl_meth : IMethod) {
            if (impl_meth.GetMemType ().Equals (the_method.GetMemType ())) 
              true
            else {
              Util.locate (the_method.Location, MakeImplWrapper (impl_meth));
              false
            }
          }
          // we filter out interface methods, which we have implemented by means of
          // wrappers, the rest are ok to implement them directly
          def thesame = impl_list.Filter (is_correct_impl);
          the_method.SetFunKind (FunKind.BoundMethod (thesame));
          // Message.Debug ($"impl: $impl_list --> $thesame");
        | _ => ()
      }
    }


    MakeImplWrapper (impl_meth : IMethod) : void
    {
      // Message.Debug ($"make impl wrap $the_method IMPL $impl_meth");
      def hd = impl_meth.GetHeader ();
      def parms = hd.parms;

      Util.cassert (the_method.Name.IndexOf ('.') == -1,
                    $ "impl loop, $the_method IMPL $impl_meth, "
                      "$(the_method.GetMemType()) != $(impl_meth.GetMemType())");
      
      def parm_types_names =
        parms.Map (fun (parm) { (parm.ty, Macros.NewSymbol ("sp_parm")) });
      def formal_parms =
        parm_types_names.Map (fun (ty, name) { 
          <[ parameter: $(name : name) : $(ty : typed) ]> 
        });
      def name = impl_meth.DeclaringType.FullName + "." + impl_meth.Name;

      def meth =
        current_type.bind_and_add_member (<[ decl:
          private $(name : dyn) (.. $formal_parms) : $(hd.ret_type : typed)
          {
          }
        ]>) :> NemerleMethod;
      // Message.Debug ($"added $meth");
      meth.SetFunKind (FunKind.BoundMethod ([impl_meth]));
      meth.Attributes |= NemerleAttributes.SpecialName | NemerleAttributes.Sealed;
      meth.HasBeenUsed = true;

      def current_fun = meth.GetHeader ();
      def refs =
        current_fun.parms.Map (fun (parm : Fun_parm) {
          def val =
            LocalValue (current_fun, parm.name, parm.ty,
                        LocalValue.Kind.FunParm (ParmKind.Normal),
                        is_mutable = false);
          val.Register ();
          val.UseFrom (current_fun);
          parm.decl = val;
          TExpr.LocalRef (val.Type, val)
        });

      def casted_refs = List.Map2 (refs, the_method.GetParameters (),
                                   fun (e, fp) {
                                     Parm (Typer3.CheckedConversion (e, fp.ty))
                                   });
      def methref =
        TExpr.MethodRef (the_method.GetFreshType (),
                           TExpr.This (current_type.GetMemType ()),
                           the_method, false);
      def call = TExpr.Call (the_method.ReturnType, methref, casted_refs, false);
      def body = Typer3.CheckedConversion (call, current_fun.ret_type);
      
      meth.GetHeader ().body = FunBody.Typed (body);

      def t4 = Typer4 (meth);
      t4.Run ();

      Passes.Emitter.MaybeCompile (current_type, meth);
    }
  }
}
