/*
 * Copyright (c) 2003-2005 University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

namespace Nemerle.Compiler
{
  using Nemerle.Collections;
  using Nemerle.Utility;

  using System.IO;
  using System.Reflection;
  using System.Reflection.Emit;
  using System.Diagnostics.SymbolStore;

  using Nemerle.Compiler.Typedtree;

  /* ----------------------------------------------------------------------- */
  /* -- CODE GENERATOR CLASS ----------------------------------------------- */
  /* ----------------------------------------------------------------------- */

  /**
   * Generates the IL code for a method or a constructor
   */
  class ILEmitter
  {
    private _module_builder : ModuleBuilder;
    private _ilg : ILGenerator;    

    private _this_is_value_type : bool;
    private _type_of_this : System.Type;

    private _method_name : string;
    private _method_is_static : bool;
    private _fun_header : Typedtree.Fun_header;
    private mutable _method_start_label : Label;
    private mutable _debug_doc : ISymbolDocumentWriter;        

    private _labels : Hashtable [int, Label] = Hashtable ();
    private _label_usage : Hashtable [int, TExpr] = Hashtable ();

    /**
     * Creates and executes a code generator for a method
     */
    internal this (method_builder : MethodBuilder, module_builder : ModuleBuilder, 
                   fun_header : Typedtree.Fun_header)
    {
      _fun_header = fun_header;

      _ilg = method_builder.GetILGenerator ();
      _module_builder = module_builder;

      _method_name = method_builder.DeclaringType.FullName + "::" + method_builder.Name;
      _method_is_static = method_builder.IsStatic;
      _this_is_value_type = method_builder.DeclaringType.IsValueType;
      _type_of_this = method_builder.DeclaringType;
    }


    /**
     * Creates and executes a code generator for a method
     */
    internal this (constructor_builder : ConstructorBuilder,
                   module_builder : ModuleBuilder, fun_header : Typedtree.Fun_header)
    {
      _fun_header = fun_header;

      _ilg = constructor_builder.GetILGenerator ();
      _module_builder = module_builder;

      _method_name = constructor_builder.DeclaringType.FullName + "::"
                      + constructor_builder.Name;
      _method_is_static = constructor_builder.IsStatic;
      _this_is_value_type = constructor_builder.DeclaringType.IsValueType;
      _type_of_this = constructor_builder.DeclaringType;
    }

    public Run () : void
    {
      _method_start_label = _ilg.DefineLabel ();
      _ilg.MarkLabel (_method_start_label);

      match (_fun_header.body) {
        | FunBody.Typed (body) =>
          emit (body);
          unless (body.Throws)
            _ilg.Emit (OpCodes.Ret);
          // let GC take it
          _fun_header.body = FunBody.ILed ();
        | _ => Util.ice ()
      }
      
      _label_usage.Iter (fun (id, v : TExpr) {
        when (v != null)
          Message.Error (v.loc, 
                         $ "non local goto (block return?) detected (l$id)");
      })
    }

    private is_always_true (expr : TExpr) : bool
    {
      | Literal (Literal.Bool (true)) => true

      | If (cond_expr, then_expr, else_expr) =>
        def cond_always_true = is_always_true (cond_expr);
        def cond_always_false = is_always_false (cond_expr);
        def then_always_true = is_always_true (then_expr);
        def else_always_true = is_always_true (else_expr);
        
        cond_always_true && then_always_true ||
        cond_always_false && else_always_true
        // unsafe, cond can have side effects
        // then_always_true && else_always_true

      | _ => false
    }

    public SetDocument (loc : Location) : void
    {
      if (loc != Location.Default)
        _debug_doc = _module_builder.DefineDocument (Path.GetFullPath (loc.File), 
          SymDocumentType.Text, SymLanguageType.ILAssembly, SymLanguageVendor.Microsoft);
      else
        Util.ice ("missing location");
    }

    private is_always_false (expr : TExpr) : bool
    {
      | Literal (Bool (false)) => true

      | If (cond_expr, then_expr, else_expr) =>
        def cond_always_true = is_always_true (cond_expr);
        def cond_always_false = is_always_false (cond_expr);
        def then_always_false = is_always_false (then_expr);
        def else_always_false = is_always_false (else_expr);

        cond_always_true && then_always_false ||
        cond_always_false && else_always_false
        // unsafe, cond can have side effects
        // then_always_false && else_always_false
        
      | _ => false
    }

    /** Marks the point in source file for currently emmited opcode */
    Mark (loc : Location) : void
    {
      when (Options.EmitDebug && loc.Line != 0) {
        Util.cassert (loc.Line <= loc.EndLine, "spoiled location " + loc.File + " " + loc.ToString ());
        _ilg.MarkSequencePoint (_debug_doc, loc.Line, loc.Column, loc.EndLine, loc.EndColumn);
      }
    }


    /** Emit volatile prefix if needed. */
    private maybe_volatile (expr : TExpr) : void
    {
      | StaticRef (f is IField)
      | FieldMember (_, f) =>
        when (f.IsVolatile)
          _ilg.Emit (OpCodes.Volatile)
      | _ => ()
    }


    /** Box the value of type [t] if needed.  */
    need_reference (t : System.Type) : void
    {
      when (t.IsValueType)
        _ilg.Emit (OpCodes.Box, t);
    }


    static is_void (ty : TyVar) : bool
    {
      ty.Fix () is MType.Void
    }


    static is_ctor (meth : IMethod) : bool
    {
      meth.GetFunKind () is FunKind.Constructor
    }


    store_local (decl : LocalValue) : void
    {
      if (decl.IsParm) {
        Util.cassert (! decl.IsByRefParm);
        store_argument (decl.ParmIndex)
      } else
        // simply skip assignment here
        unless (is_void (decl.Type))
          _ilg.Emit (OpCodes.Stloc, decl.LocalSlot);
    }


    emit_parms (parms : list [Parm]) : void
    {
      foreach (parm in parms)
        emit (parm.expr);
    }

    static skipped (expr : TExpr) : void
    {
      unless (expr is TExpr.Goto) {
        Message.Warning (expr.loc,
                         "this expression has been skipped in code "
                         "generation due to throw usage");
        Message.Warning (expr.loc, expr.ToString ());
      }
    }

    static no_cast_needed (src_type : System.Type, target_type : System.Type) : bool
    {
      src_type.Equals (target_type) ||
      (! src_type.IsValueType &&
       ! target_type.IsValueType &&
       target_type.IsAssignableFrom (src_type))
    }


    static make_ref_type (t : System.Type) : System.Type
    {
#if _NET_2_0
      t.MakeByRefType ()
#else
      def ty = t.Module.GetType (t.FullName + "&");
      assert (ty != null);
      ty
#endif
    }

    /**
     * Emits an expression.
     */
    private emit (expr : TExpr) : void
    {
      // Message.Debug (expr.loc, "emit: " + expr.ToString ());
      match (expr) {

        /* -- SEQUENCING --------------------------------------------------- */

        /* emits a sequence of expressions */
        | Sequence (e1, e2) =>
          emit (e1);
          if (!e1.Throws || e2.JumpTarget)
            emit (e2);
          else
            skipped (e2);

        | DefValIn (_, let_val, body) when let_val.Throws && ! body.JumpTarget =>
          emit (let_val);
          skipped (body);
        
        /* emits a local value definition */
        | DefValIn (decl, let_val, let_in) =>
          def newscope = 
            match (let_val) {
              | DefaultValue => 
                if (is_void (decl.Type))
                  false
                else {
                  _ = declare_val_local_slot (decl);
                  true
                }

              | ImplicitValueTypeCtor (t) =>
                def slot = declare_val_local_slot (decl);
                Mark (decl.Location);
                _ilg.Emit (OpCodes.Ldloca, slot);
                _ilg.Emit (OpCodes.Initobj, t.GetSystemType ());
                true
                              
              | _ =>
                emit (let_val);
                if (is_void (decl.Type)) 
                  false
                else if (!decl.EverUsed) {
                  _ilg.Emit (OpCodes.Pop);
                  false
                } else {
                  def local_slot = declare_val_local_slot (decl);
                  Mark (decl.Location);
                  unless (let_val.Throws)
                    _ilg.Emit (OpCodes.Stloc, local_slot);
                  true
                }
            }
          emit (let_in);
          when (Options.EmitDebug && newscope) _ilg.EndScope ();


        /* -- CONDITIONAL CONSTRUCTIONS ------------------------------------ */

        /* emits the if/then/else construction */
        | If (cond_expr, then_expr, else_expr) =>
          def emit_branch (expr : TExpr, else_label : Label) {
            match (expr) {
              | Call (OpCode (opcode), parms, _) =>
                emit_parms (parms);

                /* NOTE: the opcodes are swapped so that the 'then' expression
                         can be emitted before the 'else' expression */
                def opcode =
                  match (opcode) {
                    | "==.ref"
                    | "==" => OpCodes.Bne_Un
                    | "!=.ref"
                    | "!=" => OpCodes.Beq
                    | "<.s" | "<.f" => OpCodes.Bge
                    | "<.u" => OpCodes.Bge_Un
                    | ">.s" | ">.f" => OpCodes.Ble
                    | ">.u" => OpCodes.Ble_Un
                    | "<=.s" | "<=.f" => OpCodes.Bgt
                    | "<=.u" => OpCodes.Bgt_Un
                    | ">=.s" | ">=.f" => OpCodes.Blt
                    | ">=.u" => OpCodes.Blt_Un
                    | "bool.!" => OpCodes.Brtrue
                    | _ => Util.ice ("invalid opcode '" + opcode +
                                     "' in optimized TExpr.If pattern")
                  };
                Mark (expr.loc);
                _ilg.Emit (opcode, else_label);

              | HasType (val, ty) =>
                emit (val);
                Mark (expr.loc);
                need_reference (val.SystemType);
                _ilg.Emit (OpCodes.Isinst, ty.GetSystemType ());
                _ilg.Emit (OpCodes.Brfalse, else_label);

              | If (new_cond, new_then, new_else) =>
                def my_label_else = _ilg.DefineLabel ();
                def my_label_fi = _ilg.DefineLabel ();
                emit_branch (new_cond, my_label_else);
                emit_branch (new_then, else_label);
                unless (new_then.Throws)
                  _ilg.Emit (OpCodes.Br, my_label_fi);
                _ilg.MarkLabel (my_label_else);                  
                emit_branch (new_else, else_label);              
                _ilg.MarkLabel (my_label_fi);                                                 

              | e when is_always_true (e) => ()
              | e when is_always_false (e) =>
                Mark (e.loc);
                _ilg.Emit (OpCodes.Br, else_label);                

              | _ =>
                //Message.Debug (e.loc, pretty_print (e));
                // most of other expressions here are TExpr.Ref
                emit (expr);
                Mark (cond_expr.loc);
                _ilg.Emit (OpCodes.Brfalse, else_label);                
            }
          }

          def label_condition_else = _ilg.DefineLabel ();
          def label_condition_fi = _ilg.DefineLabel ();
          emit_branch (cond_expr, label_condition_else);

          emit (then_expr);
          unless (then_expr.Throws)
            _ilg.Emit (OpCodes.Br, label_condition_fi);

          _ilg.MarkLabel (label_condition_else);
          emit (else_expr);

          _ilg.MarkLabel (label_condition_fi);
        
        /* emit a type check construction */
        | HasType (expr, ty) =>
          emit (expr);
          def label_condition_not_null = _ilg.DefineLabel ();
          def label_condition_has_type = _ilg.DefineLabel ();
          def label_condition_continue = _ilg.DefineLabel ();

          _ilg.Emit (OpCodes.Dup);
          _ilg.Emit (OpCodes.Brtrue_S, label_condition_not_null);

          _ilg.ThrowException (SystemType.NullMatchException);

          _ilg.MarkLabel (label_condition_not_null);

          need_reference (expr.SystemType);
          _ilg.Emit (OpCodes.Isinst, ty.GetSystemType ());
          _ilg.Emit (OpCodes.Brtrue_S, label_condition_has_type);
          _ilg.Emit (OpCodes.Ldc_I4_0);
          _ilg.Emit (OpCodes.Br_S, label_condition_continue);
          _ilg.MarkLabel (label_condition_has_type);
          _ilg.Emit (OpCodes.Ldc_I4_1);
          _ilg.MarkLabel (label_condition_continue);


        /* emit a nullness check construction */
        | NotNull (val) =>
          emit (val);
          def label_condition_not_null = _ilg.DefineLabel ();

          Mark (expr.loc);
          _ilg.Emit (OpCodes.Brtrue_S, label_condition_not_null);

          _ilg.ThrowException (SystemType.NullMatchException);

          _ilg.MarkLabel (label_condition_not_null);


        /* load runtime representation of given type */
        | TypeOf (t) =>
          Mark (expr.loc);
          _ilg.Emit (OpCodes.Ldtoken, t.GetSystemType ());
          _ilg.Emit (OpCodes.Call, SystemType.Type_GetTypeFromHandle);


        /* emits the TExpr.Switch instruction */
        | Switch (indexing_expr, default_expr, cases) =>
          match (cases) {
            | [] => Message.Warning ("empty TExpr.Switch instruction");
            | _ =>
              emit (indexing_expr);
              emit_switch (default_expr, cases)
          }


        /* insert label so TExpr.Goto can work */
        | Label (id, expr) =>
          if (_labels.Contains (id))
            _label_usage [id] = null;
          else
            _labels [id] = _ilg.DefineLabel ();
          _ilg.MarkLabel (_labels [id]);
          emit (expr)


        | Goto (id) =>
          if (_labels.Contains (id)) {
            _label_usage [id] = null;
          } else {
            _label_usage [id] = expr;
            _labels [id] = _ilg.DefineLabel ();
          }
          
          Mark (expr.loc);
          _ilg.Emit (OpCodes.Br, _labels [id]);


        /* -- TYPE CONVERSIONS --------------------------------------------- */

        /* do not perform casts between same types */
        | TypeConversion (expr, cast_to_type, _) 
          when no_cast_needed (expr.SystemType, cast_to_type.GetSystemType ()) =>
          emit (expr)

        // when expr throws we shouldn't emit anything
        | TypeConversion (expr, _, _) when expr.Throws =>
          emit (expr)

        | TypeConversion (ignored, ty, _) when is_void (ty) =>
          // a little trickery, so emit can be tail called sometimes
          if (is_void (ignored.Type))
            emit (ignored)
          else {
            emit (ignored);
            Mark (expr.loc);
            _ilg.Emit (OpCodes.Pop);
          }

        /* to nonvalue type */
        | TypeConversion (expr, cast_to_type, _) when !cast_to_type.Fix ().IsValueType =>
          emit (expr);
          if (is_void (expr.Type))
            /* special case casts from void */
            _ilg.Emit (OpCodes.Ldnull);
          else if (expr.SystemType.IsValueType)
            /* box value types casted to non value types */          
            _ilg.Emit (OpCodes.Box, expr.SystemType);
          else {
            /* upcast / downcast non-value types */
            def cast_to_type = cast_to_type.GetSystemType ();
            Mark (expr.loc);
            when (expr.SystemType.IsArray && cast_to_type.IsArray &&
                  expr.SystemType.GetElementType ().IsValueType !=
                  cast_to_type.GetElementType ().IsValueType)
              Message.Error (expr.loc, $ "attempting to cast a value type array "
                                         "$(expr.SystemType) to non-value type array "
                                         "$(cast_to_type) which cannot succeed");
            _ilg.Emit (OpCodes.Castclass, cast_to_type);
          }

        /* unbox value types or perform value type conversion */
        | TypeConversion (val, cast_to_type, kind) =>
          def is_checked = ! (kind is IL (false));
          emit (val);
          Mark (expr.loc);
          def type_of_expr = val.SystemType;
          def cast_to_type = cast_to_type.GetSystemType ();

          if (type_of_expr.IsValueType) {
            /* perform conversion of value types */
            emit_value_type_conversion (expr.loc, type_of_expr, cast_to_type, is_checked)
          }
          else if (SystemType.Object.Equals (type_of_expr) || type_of_expr.IsInterface
                   || (SystemType.Enum.Equals (type_of_expr) && cast_to_type.IsEnum)) {
            /* unboxing returns a pointer to a value */
            _ilg.Emit (OpCodes.Unbox, cast_to_type);

            maybe_volatile (val);

            emit_ldind_for_value_type (cast_to_type)
          }
          else {
            Util.ice ($ "$_method_name: failed to convert non-value "
                        "type $type_of_expr to a value type "
                        "$cast_to_type");
          }


        /* -- REFERENCES --------------------------------------------------- */

        /* load the value of a local variable or a method parameter */
        | LocalRef (decl) =>
          unless (is_void (decl.Type))
            emit_ce_ref (decl, get_address_for_value_types = expr.NeedAddress)

        /* load the value of a field */
        | FieldMember (base_object, field) =>
          def result_will_be_address = expr.NeedAddress;
          
          if (result_will_be_address) 
            _ = emit_and_convert_to_address (base_object);
          else
            emit (base_object);
          
          maybe_volatile (expr);

          if (result_will_be_address)
            _ilg.Emit (OpCodes.Ldflda, field.GetFieldInfo ());
          else
            _ilg.Emit (OpCodes.Ldfld, field.GetFieldInfo ());

        /* load the value of a static field */
        | StaticRef (f is IField) =>
          def field_info = f.GetFieldInfo ();

          assert (field_info.IsStatic, "GlobalRef to a non-static field");

          when (field_info.IsLiteral)
            // should not happen
            Message.Warning ("loading literal as field `" + field_info.ToString () + "'");

          maybe_volatile (expr);
          if (expr.NeedAddress)
            _ilg.Emit (OpCodes.Ldsflda, field_info);
          else
            _ilg.Emit (OpCodes.Ldsfld, field_info);

        /* load an array element */
        | ArrayIndexer (array_obj, [index]) =>
          emit (array_obj);
          emit (index);
          def element_type = array_obj.SystemType.GetElementType ();
          assert (element_type != null, "non-array in TExpr.ArrayIndexer");

          Mark (expr.loc);
          if (expr.NeedAddress)
            _ilg.Emit (OpCodes.Ldelema, element_type)
          else
            emit_array_load_opcode (element_type);
          

        | ArrayIndexer (array_obj, indexes) =>
          emit (array_obj);
          List.Iter (indexes, emit);
          Mark (expr.loc);
          def method =
            if (expr.NeedAddress) array_addr_method (array_obj)
            else array_get_method (array_obj);
          emit_method_call (expr, true, method);


        /* -- ASSIGNMENTS -------------------------------------------------- */

        | Assign (LocalRef, val) when val.Throws
        | Assign (LocalRef (local_var), val) when is_void (local_var.Type) =>
          emit (val);
          
        
        /* special case these assignments for performance */
        | Assign (LocalRef (decl), ImplicitValueTypeCtor (t)) =>
          Mark (expr.loc);
          emit_ce_ref (decl, get_address_for_value_types = true);
          _ilg.Emit (OpCodes.Initobj, t.GetSystemType ());

     
        /* assignment to ref/out parameter */
        | Assign (LocalRef (local_var), val) when local_var.IsByRefParm =>
          emit_ldarg (local_var.ParmIndex);
          emit (val);
          Mark (expr.loc);
          def ty = val.SystemType;
          if (ty.IsValueType)
            _ilg.Emit (OpCodes.Stobj, ty)
          else
            _ilg.Emit (OpCodes.Stind_Ref);


        /* assignment to a local variable */
        | Assign (LocalRef (local_var), val) =>
          emit (val);
          Mark (expr.loc); 
          unless (val.Throws)
            store_local (local_var);


        /* assignment to a field */
        | Assign (FieldMember (base_object, field) as target, val) =>
          emit (base_object);
          emit (val);
          Mark (expr.loc);
          maybe_volatile (target);
          _ilg.Emit (OpCodes.Stfld, field.GetFieldInfo ());


        /* assignment to a static field */
        | Assign (StaticRef (f is IField) as target, val) =>
          emit (val);
          Mark (expr.loc);
          maybe_volatile (target);
          _ilg.Emit (OpCodes.Stsfld, f.GetFieldInfo ());


        /* assignment to an array element */
        | Assign (ArrayIndexer (array_obj, [index]), val) =>
          def type_of_val = val.SystemType;

          emit (array_obj);
          emit (index);
          Mark (expr.loc);
          when (type_of_val.IsValueType && !type_of_val.IsPrimitive)
            _ilg.Emit (OpCodes.Ldelema, type_of_val);

          emit (val);
          emit_array_store_opcode (type_of_val);

        | Assign (ArrayIndexer (array_obj, indexes), val) =>
          emit (array_obj);
          List.Iter (indexes, emit);
          emit (val);
          Mark (expr.loc);
          def method = array_set_method (array_obj);
          emit_method_call (expr, true, method);

        | Assign (This, val) =>
          Util.cassert (_this_is_value_type,
                        "we can assign to 'this' only in valuetypes");
          _ilg.Emit (OpCodes.Ldarg_0);          
          emit (val);
          _ilg.Emit (OpCodes.Stobj, _type_of_this)
          
        | MultipleAssign (assigns) =>
          def emit_store (local_var : LocalValue, _) {
            Mark (local_var.Location);
            store_local (local_var);
          }
          foreach ((_, e) in assigns) emit (e);
          List.Iter (List.Rev (assigns), emit_store);


        /* -- CALLS -------------------------------------------------------- */

        /* call the base constructor */
        | Call (Base (base_ctor), ctor_params, _) =>
          _ilg.Emit (OpCodes.Ldarg_0);
          emit_parms (ctor_params);
          Mark (expr.loc);          
          _ilg.Emit (OpCodes.Call, base_ctor.GetConstructorInfo ());

        /* create a new object */
        | Call (StaticRef (meth is IMethod), ctor_params, _) when is_ctor (meth) =>
          emit_parms (ctor_params);
          Mark (expr.loc);
          _ilg.Emit (OpCodes.Newobj, meth.GetConstructorInfo ());


        /* emit a call to an instance method, basing on the 'this' pointer for value types */
        | Call (MethodRef (This, method, _), method_params, _)
                                                      when _this_is_value_type =>
          _ilg.Emit (OpCodes.Ldarg_0);
          def method = method.GetMethodInfo ();
          
          // it actually IS the proper way to do it --
          // if this is our method and we do not need to box
          unless (method.DeclaringType.Equals (_type_of_this)) {
            _ilg.Emit (OpCodes.Ldobj, _type_of_this);
            _ilg.Emit (OpCodes.Box, _type_of_this);
          }
          emit_parms (method_params);
          Mark (expr.loc);
          emit_method_call (expr, true, method)


        /* emit a call to an instance method */
        | Call (MethodRef (base_object, method, notvirt), method_params, _) =>
          def is_value_type = emit_and_convert_to_address (base_object);

          emit_parms (method_params);
          Mark (expr.loc);
          emit_method_call (expr, is_value_type || notvirt, method.GetMethodInfo ())


        /* emit a call to a static method */
        | Call (StaticRef (mi is IMethod), method_parms, _) =>
          def method_info = mi.GetMethodInfo ();

          Util.cassert (method_info.IsStatic, 
                        $ "Call TExpr.GlobalRef to a non-static field: $(method_info.Name)");

          emit_parms (method_parms);
          Mark (expr.loc);          
          emit_method_call (expr, true, method_info)


        /* emit an operator */
        | Call (OpCode (opcode), parms, _) =>
          // FIXME: seperate out
          emit_parms (parms);
          Mark (expr.loc);
          match (opcode) {
            | "+.s" =>  _ilg.Emit (OpCodes.Add_Ovf)
            | "+.u" => _ilg.Emit (OpCodes.Add_Ovf_Un)
            | "+.f" =>  _ilg.Emit (OpCodes.Add)

            | "<<.s" | "<<.f" | "<<.u" =>  _ilg.Emit (OpCodes.Shl)
            | ">>.s" | ">>.f" =>  _ilg.Emit (OpCodes.Shr)
            | ">>.u" =>  _ilg.Emit (OpCodes.Shr_Un)

            | "unary.+.s" | "unary.+.u" | "unary.+.f" => ()

            | "-.s" =>  _ilg.Emit (OpCodes.Sub_Ovf)
            | "-.u" => _ilg.Emit (OpCodes.Sub_Ovf_Un)
            | "-.f" =>  _ilg.Emit (OpCodes.Sub)

            | "unary.-.s" =>
              _ilg.Emit (OpCodes.Ldc_I4_M1);
              emit_value_type_conversion (expr.loc, SystemType.Int32,
                                          parms.Head.expr.SystemType, true);
              _ilg.Emit (OpCodes.Mul_Ovf)
            | "unary.-.f" =>
              _ilg.Emit (OpCodes.Neg)

            | "unary.~.u"
            | "unary.~.s" =>
              _ilg.Emit (OpCodes.Not);

            | "++.i4.f" => _ilg.Emit (OpCodes.Ldc_I4_1); _ilg.Emit (OpCodes.Add);
            | "--.i4.f" => _ilg.Emit (OpCodes.Ldc_I4_1); _ilg.Emit (OpCodes.Sub);
            | "++.i4.s" => _ilg.Emit (OpCodes.Ldc_I4_1); _ilg.Emit (OpCodes.Add_Ovf);
            | "--.i4.s" => _ilg.Emit (OpCodes.Ldc_I4_1); _ilg.Emit (OpCodes.Sub_Ovf);
            | "++.i4.u" => _ilg.Emit (OpCodes.Ldc_I4_1); _ilg.Emit (OpCodes.Add_Ovf_Un);
            | "--.i4.u" => _ilg.Emit (OpCodes.Ldc_I4_1); _ilg.Emit (OpCodes.Sub_Ovf_Un);
            | "++.i8.f" => _ilg.Emit (OpCodes.Ldc_I4_1); _ilg.Emit (OpCodes.Conv_I8); _ilg.Emit (OpCodes.Add);
            | "--.i8.f" => _ilg.Emit (OpCodes.Ldc_I4_1); _ilg.Emit (OpCodes.Conv_I8); _ilg.Emit (OpCodes.Sub);
            | "++.i8.s" => _ilg.Emit (OpCodes.Ldc_I4_1); _ilg.Emit (OpCodes.Conv_I8); _ilg.Emit (OpCodes.Add_Ovf);
            | "--.i8.s" => _ilg.Emit (OpCodes.Ldc_I4_1); _ilg.Emit (OpCodes.Conv_I8); _ilg.Emit (OpCodes.Sub_Ovf);
            | "++.i8.u" => _ilg.Emit (OpCodes.Ldc_I4_1); _ilg.Emit (OpCodes.Conv_I8); _ilg.Emit (OpCodes.Add_Ovf_Un);
            | "--.i8.u" => _ilg.Emit (OpCodes.Ldc_I4_1); _ilg.Emit (OpCodes.Conv_I8); _ilg.Emit (OpCodes.Sub_Ovf_Un);
            | "++.r4.f" => _ilg.Emit (OpCodes.Ldc_I4_1); _ilg.Emit (OpCodes.Conv_R4); _ilg.Emit (OpCodes.Add);
            | "--.r4.f" => _ilg.Emit (OpCodes.Ldc_I4_1); _ilg.Emit (OpCodes.Conv_R4); _ilg.Emit (OpCodes.Sub);
            | "++.r8.f" => _ilg.Emit (OpCodes.Ldc_I4_1); _ilg.Emit (OpCodes.Conv_R8); _ilg.Emit (OpCodes.Add);
            | "--.r8.f" => _ilg.Emit (OpCodes.Ldc_I4_1); _ilg.Emit (OpCodes.Conv_R8); _ilg.Emit (OpCodes.Sub);

            | "unary.~.f" =>
              def type_of_parm = parms.Head.expr.SystemType;
              when (type_of_parm.FullName == "System.Single" ||
                    type_of_parm.FullName == "System.Double" ||
                    type_of_parm.FullName == "System.Decimal")
                Util.ice ("float, double or decimal as an argument for binary negation in CGIL");
              _ilg.Emit (OpCodes.Not);

            | "*.s" =>  _ilg.Emit (OpCodes.Mul_Ovf)
            | "*.u" => _ilg.Emit (OpCodes.Mul_Ovf_Un)
            | "*.f" =>  _ilg.Emit (OpCodes.Mul)

            | "/.s" | "/.f" =>  _ilg.Emit (OpCodes.Div)
            | "/.u" => _ilg.Emit (OpCodes.Div_Un)
            | "%.s" | "%.f" =>  _ilg.Emit (OpCodes.Rem)
            | "%.u" => _ilg.Emit (OpCodes.Rem_Un)

            | "|.s" | "|.u" | "|.f" => _ilg.Emit (OpCodes.Or);
            | "&.s" | "&.u" | "&.f" => _ilg.Emit (OpCodes.And);
            | "^.s" | "^.u" | "^.f" => _ilg.Emit (OpCodes.Xor);
            | "bool.!" =>
              _ilg.Emit (OpCodes.Ldc_I4_0);
              _ilg.Emit (OpCodes.Ceq)
            | "==.ref" | "==" =>
              _ilg.Emit (OpCodes.Ceq);
            | "!=.ref" | "!=" =>
              _ilg.Emit (OpCodes.Ceq);
              _ilg.Emit (OpCodes.Ldc_I4_0);
              _ilg.Emit (OpCodes.Ceq);
            | "<.s" | "<.f" => emit_compare (OpCodes.Blt_S)
            | "<.u" => emit_compare (OpCodes.Blt_Un_S)
            | ">.s" | ">.f" => emit_compare (OpCodes.Bgt_S)
            | ">.u" => emit_compare (OpCodes.Bgt_Un_S)
            | "<=.s" | "<=.f" => emit_compare (OpCodes.Ble_S)
            | "<=.u" => emit_compare (OpCodes.Ble_Un_S)
            | ">=.s" | ">=.f" => emit_compare (OpCodes.Bge_S)
            | ">=.u" => emit_compare (OpCodes.Bge_Un_S)
            | _ =>
              Util.ice (_method_name + ": unmatched TExpr.Opcode (" + opcode + ")")
          };

        /* -- EXCEPTION HANDLING ------------------------------------------- */

        /* throw an exception */
        | Throw (null) =>
          Mark (expr.loc);
          _ilg.Emit (OpCodes.Rethrow);
        
        | Throw (exc) =>
          emit (exc);
          Mark (expr.loc);
          _ilg.Emit (OpCodes.Throw);


        /* emit the try/with construction */
        | TryWith (try_expr, catch_val, catch_expr) =>
          def type_of_try_expr = expr.SystemType;

          def ignore_try_with_result = is_void (try_expr.Type);
          mutable try_result = null;

          unless (ignore_try_with_result)
            try_result = _ilg.DeclareLocal (type_of_try_expr);

          _ = _ilg.BeginExceptionBlock ();

          emit (try_expr);

          unless (try_expr.Throws || ignore_try_with_result)
            _ilg.Emit (OpCodes.Stloc, try_result);

          _ilg.BeginCatchBlock (catch_val.Type.GetSystemType ());

          def catch_val_local_slot = declare_val_local_slot (catch_val);
          _ilg.Emit (OpCodes.Stloc, catch_val_local_slot);

          emit (catch_expr);
          // close scope opened by declare_val_local_slot
          when (Options.EmitDebug) _ilg.EndScope ();

          unless (catch_expr.Throws || ignore_try_with_result)
            _ilg.Emit (OpCodes.Stloc, try_result);

          _ilg.EndExceptionBlock ();

          unless ((catch_expr.Throws && try_expr.Throws) || ignore_try_with_result)
            _ilg.Emit (OpCodes.Ldloc, try_result);


        | TryFinally (try_expr, finally_expr) =>
          def type_of_try_expr = try_expr.SystemType;

          def _ = _ilg.BeginExceptionBlock ();

          emit (try_expr);

          def ignore_try_result = try_expr.Throws || is_void (try_expr.Type);
          mutable try_result = null;

          unless (ignore_try_result) {
            try_result = _ilg.DeclareLocal (type_of_try_expr);
            _ilg.Emit (OpCodes.Stloc, try_result)
          };

          _ilg.BeginFinallyBlock ();

          emit (finally_expr);

          _ilg.EndExceptionBlock ();
          unless (ignore_try_result)
            _ilg.Emit (OpCodes.Ldloc, try_result);


        /* -- TUPLES -------------------------------------------------------- */
        | Tuple (vals) =>
          foreach (v in vals) emit (v);
          def tuple_ty = InternalType.GetTupleType (vals.Length);
          _ilg.Emit (OpCodes.Newobj, tuple_ty.Ctor.GetConstructorInfo ());
          

        | TupleIndexer (obj, pos, len) =>
          def tt = InternalType.GetTupleType (len);
          when (tt.SystemType.IsValueType && !obj.NeedAddress)
            obj.NeedAddress = true;
          emit (obj);
          _ilg.Emit (OpCodes.Ldfld, tt.GetField (pos + 1).GetFieldInfo ());


        /* -- MISC --------------------------------------------------------- */

        /* loads the 'this' pointer on the evaluation stack */
        | This =>
          Util.cassert (!_method_is_static, "This is a static method");
          _ilg.Emit (OpCodes.Ldarg_0);
          
          // when we are in value type, this contains address of current
          // value type - so if value is expected instead of address, we must
          // create a copy
          when (_this_is_value_type && !expr.NeedAddress)
            _ilg.Emit (OpCodes.Ldobj, expr.SystemType);


        /* loads a literal on the evaluation stack */
        | Literal (l) => 
          Util.cassert (l != null);
          emit_literal (l);


        /* loads address of given method */
        | MethodAddress (meth, is_virt) =>
          Mark (expr.loc);
          def meth = meth.GetMethodInfo ();
          if (is_virt && meth.IsVirtual) {
            // ldvirtftn expects also an object reference
            // this is always part of delegate ctor, therefore
            // we can just dup the object reference
            _ilg.Emit (OpCodes.Dup);
            _ilg.Emit (OpCodes.Ldvirtftn, meth);
          } else
            _ilg.Emit (OpCodes.Ldftn, meth);


        /* creates object of value type using implicit ctor */
        | ImplicitValueTypeCtor (t) =>
          Mark (expr.loc);
          when (Options.EmitDebug) _ilg.BeginScope ();
          def t = t.GetSystemType ();
          def local_slot = _ilg.DeclareLocal (t);
          _ilg.Emit (OpCodes.Ldloca, local_slot);
          _ilg.Emit (OpCodes.Initobj, t);
          _ilg.Emit (OpCodes.Ldloc, local_slot);
          when (Options.EmitDebug) _ilg.EndScope ();


        /* creates a new array, given a list of initializers */
        | Array (initializers, [size]) =>
          def element_type =
            match (expr.Type.Fix ()) {
              | MType.Array (t, _) => t.GetSystemType ()
              | _ => Util.ice ()
            }
            
          emit (size);
          Mark (expr.loc);
          _ilg.Emit (OpCodes.Newarr, element_type);

          def load_elements (index : int, elements : list [TExpr]) {
            match (elements) {
              | [] => ()
              | element :: rest =>
                _ilg.Emit (OpCodes.Dup);
                _ilg.Emit (OpCodes.Ldc_I4, index);

                when (element_type.IsValueType && !element_type.IsPrimitive)
                  _ilg.Emit (OpCodes.Ldelema, element_type);

                emit (element);
                emit_array_store_opcode (element_type);
                load_elements (index + 1, rest)
            }
          };
          load_elements (0, initializers);

        | Array (initializers, dimensions) =>
          mutable size = 0;
          foreach (element in dimensions) {
            ++size;
            emit (element);
          }

          def arg_types = array (size);
          for (mutable i = 0; i < size; ++i)
            arg_types [i] = SystemType.Int32;
          def method =
            _module_builder.GetArrayMethod (expr.SystemType, ".ctor", 
                                            CallingConventions.HasThis,
                                            null, arg_types);
          Mark (expr.loc);                                                       
          _ilg.Emit (OpCodes.Newobj, method);

          unless (initializers.IsEmpty) {
            def method = array_set_method (expr.SystemType);

            def dimensions_array = array (size);
            mutable i = 0;
            foreach (TExpr.Literal (Literal.Integer (it, _, _)) in dimensions) {
              dimensions_array [i] = it;
              ++i;
            };
            Util.cassert (i == size,
                          "dimensions must be literals when initializes are supplied");

            def load_elements (index : array [ulong], elements : list [TExpr]) {
              match (elements) {
                | [] => ()
                | element :: rest =>
                  _ilg.Emit (OpCodes.Dup);                  
                  for (mutable j = 0; j < index.Length; ++j)
                    emit_literal (Literal.Integer ((index [j] :> int) :> ulong, false, InternalType.Int32));
                  emit (element);
                  def next (index : array [ulong])
                  {
                    ++index [index.Length - 1];
                    for (mutable j = index.Length - 1; index [j] == dimensions_array [j] && j > 0; --j)
                    {
                      index [j] = 0;
                      ++index [j - 1];
                    }
                    index
                  }
                  emit_method_call (expr, true, method);
                  load_elements (next (index), rest)
              }
            }
            load_elements (array (dimensions_array.Length), initializers);
          }

        | _ =>
          Message.Warning ($ "FIXME: unmatched: $expr");
      }
    }

    private array_set_method (t : System.Type) : MethodInfo
    {
        def rank = t.GetArrayRank ();

        def args = array (rank + 1);
        for (mutable i = 0; i < rank; ++i)
        {
            args [i] = SystemType.Int32;
        }
        args [rank] = t.GetElementType ();

        _module_builder.GetArrayMethod (t, "Set",
                                        CallingConventions.HasThis %| CallingConventions.Standard,
                                        SystemType.Void, args);
    }


    /*
     * Emits a local reference instruction
     */
    private emit_ce_ref (decl : LocalValue, get_address_for_value_types : bool) : void
    {
      if (decl.IsParm) {
        if (get_address_for_value_types && ! decl.IsByRefParm)
          emit_ldarga (decl.ParmIndex)
        else {
          emit_ldarg (decl.ParmIndex);
          when (! get_address_for_value_types && decl.IsByRefParm) {
            def ty = decl.Type.Fix ().GetSystemType ();
            if (ty.IsValueType)
               emit_ldind_for_value_type (ty)
            else
              _ilg.Emit (OpCodes.Ldind_Ref);
          }
        }
      } else {
        if (get_address_for_value_types)
          _ilg.Emit (OpCodes.Ldloca, decl.LocalSlot)
        else
          _ilg.Emit (OpCodes.Ldloc, decl.LocalSlot)
      }
    }


    private array_set_method (arr : TExpr) : MethodInfo
    {
        array_set_method (arr.SystemType);
    }

    private array_get_method (arr : TExpr) : MethodInfo
    {
        def t = arr.SystemType;
        def rank = t.GetArrayRank ();

        def args = array (rank);
        for (mutable i = 0; i < rank; ++i)
        {
            args [i] = SystemType.Int32;
        }

        _module_builder.GetArrayMethod (t, "Get",
                                        CallingConventions.HasThis %| CallingConventions.Standard,
                                        t.GetElementType (), args);
    }

    private array_addr_method (arr : TExpr) : MethodInfo
    {
        def t = arr.SystemType;
        def rank = t.GetArrayRank ();

        def args = array (rank);
        for (mutable i = 0; i < rank; ++i)
        {
            args [i] = SystemType.Int32;
        }

        _module_builder.GetArrayMethod (t, "Address",
                                        CallingConventions.HasThis | CallingConventions.Standard,
                                        make_ref_type (t.GetElementType ()), args);
    }

    /**
     * Emits the TExpr.Switch instruction
     *
     * Asserts that the first of the case indices is zero. Quietly
     * asserts that the indexer is verified to be in a valid range -- i.e.
     * that the rescaling substraction does not overflow and that we never
     * generate too long switch instructions -- assuring this should belong
     * to a higher level logic.
     */
    private emit_switch (default_expr : option [TExpr], 
                         cases : list [int * TExpr]) : void
    {
      def label_default = _ilg.DefineLabel ();
      def label_next_instruction = _ilg.DefineLabel ();

      // step zero: calculate the default jump target
      def default_jump_target =
        match (default_expr) {
          | Some => label_default
          | None => label_next_instruction
        };

      // step one: sort the cases increasingly
      def compare_cases (l : int * TExpr, r : int * TExpr) : int {
        Pair.First (l).CompareTo (r [0])
      };

      def cases = List.Sort (cases, compare_cases);
      Util.cassert (cases.Head [0] == 0);

      // step three: generate the jump labels for the switch cases
      mutable reversed_labels = [];

      def make_labels (index : int, cases : list [int * TExpr])
                                        : list [Label * TExpr] {
        Util.cassert (index >= 0);
        match (cases) {
          | (case_index, case_expr) :: rest =>
            Util.cassert (index <= case_index);

            // insert jumps to the default expression where appropriate
            if (case_index == index) {
              def case_label = _ilg.DefineLabel ();
              reversed_labels = case_label :: reversed_labels;

              (case_label, case_expr) :: make_labels (index + 1, rest)
            }
            else {
              reversed_labels = default_jump_target :: reversed_labels;
              make_labels (index + 1, cases)
            }
          | [] => []
        }
      };

      def labeled_cases = make_labels (0, cases);
      def labels = List.Rev (reversed_labels);

      def labels_array = label_list_to_array (labels);

      // step four: emit the switch instruction and the default expression
      _ilg.Emit (OpCodes.Switch, labels_array);
      _ilg.MarkLabel (label_default);

      // emit the default expression (if none present jump to the next instruction)
      match (default_expr) {
        | Some (default_expr) =>
          emit (default_expr);
          unless (default_expr.Throws)
            _ilg.Emit (OpCodes.Br, label_next_instruction);
        | None =>
          _ilg.Emit (OpCodes.Br, label_next_instruction);
      };

      // step five: emit the switch cases
      def walk_cases (cases : list [Label * TExpr]) : void {
        | (label, expr) :: rest =>
          _ilg.MarkLabel (label);
          emit (expr);
          unless (expr.Throws)
            _ilg.Emit (OpCodes.Br, label_next_instruction);
          walk_cases (rest)
        | [] => ()
      };

      walk_cases (labeled_cases);

      // step six: mark the next instruction after the switch
      _ilg.MarkLabel (label_next_instruction);
    }


    /**
     * Emits an LDARG instruction
     */
    private emit_ldarg (index : int) : void
    {
      | 0 => _ilg.Emit (OpCodes.Ldarg_0)
      | 1 => _ilg.Emit (OpCodes.Ldarg_1)
      | 2 => _ilg.Emit (OpCodes.Ldarg_2)
      | 3 => _ilg.Emit (OpCodes.Ldarg_3)
      | n when n <= 255 => _ilg.Emit (OpCodes.Ldarg_S, (index :> System.Byte))
      | _ => _ilg.Emit (OpCodes.Ldarg, index)
    }


    private store_argument (index : int) : void
    {
      if (index < 256)
        _ilg.Emit (OpCodes.Starg_S, index :> byte)
      else
        _ilg.Emit (OpCodes.Starg, index);
    }

    
    /**
     * Emits an LDARGA instruction
     */
    private emit_ldarga (index : int) : void
    {
      | n when n <= 255 => _ilg.Emit (OpCodes.Ldarga_S, (index :> System.Byte))
      | _ => _ilg.Emit (OpCodes.Ldarga, index)
    }


    /**
     * Performs value type conversion form l to r
     */
    private emit_value_type_conversion (loc : Location,
                                        l : System.Type, r : System.Type,
                                        is_checked : bool) : void
    {
      def is_unsigned (t : System.Type) {
        match (t.Name [0]) {
          // UInt32, UInt64, UInt16
          | 'U'
          // Byte
          | 'B'
          // Char
          | 'C'   => true
          | _ => false
        }
      };
      def fail () {
        Message.Warning (loc, 
                         $ "failed to emit value type conversion from $l to $r")
      };

      match (()) {
        | _ when r.Equals (l) => ()
        | _ when l.IsPrimitive || l.IsEnum =>
          if (r.IsPrimitive || r.IsEnum) {
            def rname =
              // FIXME: use true underalying type
              if (r.IsEnum) "Int32" else r.Name;
            def opcode =
              if (is_checked) {
                match (rname) {
                  | "Int64" when is_unsigned (l) => OpCodes.Conv_Ovf_I8_Un
                  | "Int64" => OpCodes.Conv_Ovf_I8
                  | "Int32" when is_unsigned (l) => OpCodes.Conv_Ovf_I4_Un
                  | "Int32" => OpCodes.Conv_Ovf_I4
                  | "Int16" when is_unsigned (l) => OpCodes.Conv_Ovf_I2_Un
                  | "Int16" => OpCodes.Conv_Ovf_I2
                  | "SByte" when is_unsigned (l) => OpCodes.Conv_Ovf_I1_Un
                  | "SByte" => OpCodes.Conv_Ovf_I1
                  | "UInt64" when is_unsigned (l) => OpCodes.Conv_Ovf_U8_Un
                  | "UInt64" => OpCodes.Conv_Ovf_U8
                  | "UInt32" when is_unsigned (l) => OpCodes.Conv_Ovf_U4_Un
                  | "UInt32" => OpCodes.Conv_Ovf_U4
                  | "UInt16" when is_unsigned (l) => OpCodes.Conv_Ovf_U2_Un
                  | "UInt16" => OpCodes.Conv_Ovf_U2
                  | "Boolean" when is_unsigned (l) => OpCodes.Conv_Ovf_U1_Un
                  | "Boolean" => OpCodes.Conv_Ovf_U1
                  | "Byte" when is_unsigned (l) => OpCodes.Conv_Ovf_U1_Un
                  | "Byte" => OpCodes.Conv_Ovf_U1
                  | "Char" when is_unsigned (l) => OpCodes.Conv_Ovf_U2_Un
                  | "Char" => OpCodes.Conv_Ovf_U2

                  | "Single" when is_unsigned (l) => OpCodes.Conv_R_Un
                  | "Single" => OpCodes.Conv_R4
                  // there is no OpCodes.Conv_R8_Un
                  | "Double" when !is_unsigned (l) => OpCodes.Conv_R8

                  | _ => fail (); OpCodes.Nop
                }
              }
              else {
                match (rname) {
                  | "Int64" => OpCodes.Conv_I8
                  | "Int32" => OpCodes.Conv_I4
                  | "Int16" => OpCodes.Conv_I2
                  | "SByte" => OpCodes.Conv_I1
                  | "UInt64" => OpCodes.Conv_U8
                  | "UInt32" => OpCodes.Conv_U4
                  | "UInt16" => OpCodes.Conv_U2
                  | "Byte" => OpCodes.Conv_U1
                  | "Boolean" => OpCodes.Conv_U1
                  | "Char" => OpCodes.Conv_U2

                  | "Single" when is_unsigned (l) => OpCodes.Conv_R_Un
                  | "Single" => OpCodes.Conv_R4
                  | "Double" => OpCodes.Conv_R8

                  | _ => fail (); OpCodes.Nop
                }
              }
            _ilg.Emit (opcode)
          }
          else if (r.Equals (SystemType.Decimal)) {
            // FIXME: use true underalying type
            def lname = if (l.IsEnum) "Int32" else l.Name;
            match (lname) {
              | "Int32" | "UInt32" | "Single" | "Int64" | "UInt64"
              | "Double" =>
                _ilg.Emit (OpCodes.Newobj, Option.UnSome (SystemType.Decimal_ctors.Get (lname)))
              | _ => fail ()
            }
          }
          else fail ()

        | _ => fail ()
      }
    }


    /**
     * Emits a method call.
     *
     * If it is a tail call (which is checked based on [expr]), then we emit
     * tail modifier.
     */
    private emit_method_call (expr : TExpr,
                              base_is_value_type_or_static : bool,
                              method : MethodInfo) : void
    {
      def is_tail =
        match (expr) {
          | Call (_, _, true) when Options.GeneralTailCallOpt =>
            _ilg.Emit (OpCodes.Tailcall);
            true
          | _ => false
        };

      if (base_is_value_type_or_static)
        _ilg.Emit (OpCodes.Call, method)
      else
        _ilg.Emit (OpCodes.Callvirt, method);

      when (is_tail)
        _ilg.Emit (OpCodes.Ret);
    }

    /* Emits code of given expression and converts its value to adress
       (if it is a value type) or leave it as reference (non value type).
       This is needed in various places, where we need a real reference
       (address in case of value types) to access its methods or fields.

       Returns value indicating if base_object was really a value type
    */
    private emit_and_convert_to_address (base_object : TExpr) : bool 
    {
      emit (base_object);
      def is_value_type = base_object.SystemType.IsValueType;
      when (is_value_type && ! base_object.NeedAddress) {
        def local_slot = _ilg.DeclareLocal (base_object.SystemType);
        _ilg.Emit (OpCodes.Stloc, local_slot);
        _ilg.Emit (OpCodes.Ldloca, local_slot);
      };
      is_value_type
    }


    /**
     * Declares a local variable
     */
    private declare_val_local_slot (val : LocalValue) : LocalBuilder
    {
      when (Options.EmitDebug) _ilg.BeginScope ();              
      def local_slot = _ilg.DeclareLocal (val.Type.GetSystemType ());
      when (Options.EmitDebug) local_slot.SetLocalSymInfo (val.Name);
      val.LocalSlot = local_slot;
      local_slot
    }

    /**
     * Emits a STELEM instruction for a given type
     *
     * NOTE: there is no Stelem_U* in System.Reflection.Emit.OpCodes.
     */
    private emit_array_store_opcode (ty : System.Type) : void
    {
      assert (!SystemType.Void.Equals (ty), "tried to store a System.Void in an array");

      if (ty.IsPrimitive) {
        def opcode =
          match (ty.Name) {
            | "SByte" | "Boolean"
            | "Byte"                     => OpCodes.Stelem_I1
            | "Char" | "Int16"
            | "UInt16"                   => OpCodes.Stelem_I2
            | "Int32" | "UInt32"         => OpCodes.Stelem_I4
            | "Int64" | "UInt64"         => OpCodes.Stelem_I8
            | "Single"                   => OpCodes.Stelem_R4
            | "Double"                   => OpCodes.Stelem_R8
            | _ => Util.ice ("System.Type IsPrimitive has lied")
          }
          _ilg.Emit (opcode);
        }
        else if (ty.IsValueType) _ilg.Emit (OpCodes.Stobj, ty)
        else _ilg.Emit (OpCodes.Stelem_Ref);
    }


    /**
     * Emits a LDELEM instruction for a given type
     */
    private emit_array_load_opcode (ty : System.Type) : void
    {
      assert (!SystemType.Void.Equals (ty), "tried to load a System.Void from an array");

      if (ty.IsValueType && !ty.IsPrimitive) {
        _ilg.Emit (OpCodes.Ldelema, ty);
        _ilg.Emit (OpCodes.Ldobj, ty);
      } 
      else {
        def opcode =
          if (ty.IsPrimitive)
            match (ty.Name) {
              | "SByte" | "Boolean" => OpCodes.Ldelem_I1
              | "Byte"              => OpCodes.Ldelem_U1
              | "Char" | "Int16"    => OpCodes.Ldelem_I2
              | "UInt16"            => OpCodes.Ldelem_U2
              | "Int32"             => OpCodes.Ldelem_I4
              | "UInt32"            => OpCodes.Ldelem_U4
              | "Int64" | "UInt64"  => OpCodes.Ldelem_I8
              | "Single"            => OpCodes.Ldelem_R4
              | "Double"            => OpCodes.Ldelem_R8
              | _ => Util.ice ("System.Type.IsPrimitive has lied")
            }
          else
            OpCodes.Ldelem_Ref;

        _ilg.Emit (opcode)
      }
    }


    /**
     * Emits a LDIND instruction for a given value type
     */
    private emit_ldind_for_value_type (ty : System.Type) : void
    {
      if (ty.IsEnum || !ty.IsPrimitive)
        _ilg.Emit (OpCodes.Ldobj, ty)
      else {
        def opcode =
          match (ty.Name) {
            | "SByte" | "Boolean"     => OpCodes.Ldind_I1
            | "Byte"                  => OpCodes.Ldind_U1
            | "Char" | "Int16"        => OpCodes.Ldind_I2
            | "UInt16"                => OpCodes.Ldind_U2
            | "Int32"                 => OpCodes.Ldind_I4
            | "UInt32"                => OpCodes.Ldind_U4
            | "Int64"                 => OpCodes.Ldind_I8
            | "UInt64"                => OpCodes.Ldind_I8
            | "Single"                => OpCodes.Ldind_R4
            | "Double"                => OpCodes.Ldind_R8
            | _ =>
              Util.ice (_method_name + ": not a recognized value type: " + ty.FullName);
          }

        _ilg.Emit (opcode)
      }
    }


    /**
     * Emits a literal
     */
    private emit_literal (l : Literal) : void
    {
      def emit_i4 (x) {
        | -1 => _ilg.Emit (OpCodes.Ldc_I4_M1)
        | 0  => _ilg.Emit (OpCodes.Ldc_I4_0)
        | 1  => _ilg.Emit (OpCodes.Ldc_I4_1)
        | 2  => _ilg.Emit (OpCodes.Ldc_I4_2)
        | 3  => _ilg.Emit (OpCodes.Ldc_I4_3)
        | 4  => _ilg.Emit (OpCodes.Ldc_I4_4)
        | 5  => _ilg.Emit (OpCodes.Ldc_I4_5)
        | 6  => _ilg.Emit (OpCodes.Ldc_I4_6)
        | 7  => _ilg.Emit (OpCodes.Ldc_I4_7)
        | 8  => _ilg.Emit (OpCodes.Ldc_I4_8)
        | k when (k >= -128 && k <= 127) =>
          _ilg.Emit (OpCodes.Ldc_I4_S, k :> sbyte)
        | k => _ilg.Emit (OpCodes.Ldc_I4, k)
      }

      match (l) {
        | Literal.Void         => ()
        | Literal.Null         => _ilg.Emit (OpCodes.Ldnull)
        | Literal.Bool (true)  => _ilg.Emit (OpCodes.Ldc_I4_1)
        | Literal.Bool (false) => _ilg.Emit (OpCodes.Ldc_I4_0)

        | Literal.Char (k)     => emit_i4 (k :> int)

        | Literal.Integer (k, is_neg, t) =>
          assert (t != null);
          if (t.Equals (InternalType.UInt64)) {
            assert (!is_neg);
            unchecked (_ilg.Emit (OpCodes.Ldc_I8, k :> long));
            // i don't think this is needed here
            _ilg.Emit (OpCodes.Conv_U8)
          } else if (t.Equals (InternalType.Int64)) {
            if (is_neg)
              if (k == 0x8000000000000000UL)
                _ilg.Emit (OpCodes.Ldc_I8, System.Int64.MinValue);
              else
                _ilg.Emit (OpCodes.Ldc_I8, - (k :> long));
            else
              unchecked (_ilg.Emit (OpCodes.Ldc_I8, k :> long));
          } else if (t.Equals (InternalType.UInt32)) {
            assert (!is_neg);
            unchecked (emit_i4 (k :> int))
          } else
            if (k <= System.Int32.MaxValue)
              emit_i4 (if (is_neg) - (k :> int) else (k :> int))
            else if (k == (- (System.Int32.MinValue : long)) :> ulong && is_neg)
              emit_i4 (System.Int32.MinValue)
            else {
              Message.Warning ($ "cannot emit: $l : $t");
              assert (false)
            }
          
        | Literal.String (s)   =>
          assert (s != null); _ilg.Emit (OpCodes.Ldstr, s)
          
        | Literal.Float (f)    => _ilg.Emit (OpCodes.Ldc_R4, f)
        | Literal.Double (f)   => _ilg.Emit (OpCodes.Ldc_R8, f)
        | Literal.Decimal (m)  =>
          def decimal_bits = System.Decimal.GetBits (m);

          _ilg.Emit (OpCodes.Ldc_I4_4);

          _ilg.Emit (OpCodes.Newarr, SystemType.Int32);

          when (decimal_bits [0] != 0)
            _ilg.Emit (OpCodes.Dup);
          when (decimal_bits [1] != 0)
            _ilg.Emit (OpCodes.Dup);
          when (decimal_bits [2] != 0)
            _ilg.Emit (OpCodes.Dup);
          when (decimal_bits [3] != 0)
            _ilg.Emit (OpCodes.Dup);

          when (decimal_bits [0] != 0) {
            _ilg.Emit (OpCodes.Ldc_I4_0);
            _ilg.Emit (OpCodes.Ldc_I4, decimal_bits [0]);
            _ilg.Emit (OpCodes.Stelem_I4)
          }
          when (decimal_bits [1] != 0) {
            _ilg.Emit (OpCodes.Ldc_I4_1);
            _ilg.Emit (OpCodes.Ldc_I4, decimal_bits [1]);
            _ilg.Emit (OpCodes.Stelem_I4)
          }
          when (decimal_bits [2] != 0) {
            _ilg.Emit (OpCodes.Ldc_I4_2);
            _ilg.Emit (OpCodes.Ldc_I4, decimal_bits [2]);
            _ilg.Emit (OpCodes.Stelem_I4)
          }
          when (decimal_bits [3] != 0) {
            _ilg.Emit (OpCodes.Ldc_I4_3);
            _ilg.Emit (OpCodes.Ldc_I4, decimal_bits [3]);
            _ilg.Emit (OpCodes.Stelem_I4)
          }

          _ilg.Emit (OpCodes.Newobj, Option.UnSome (SystemType.Decimal_ctors.Get ("Int32[]")))

        | Literal.Enum (l, _) => emit_literal (l)
      }
    }


    /**
     * Emits a comparison operation
     */
    private emit_compare (opcode : OpCode) : void
    {
      def label_condition_met = _ilg.DefineLabel ();
      def label_condition_not_met = _ilg.DefineLabel ();

      _ilg.Emit (opcode, label_condition_met);
      _ilg.Emit (OpCodes.Ldc_I4_0);
      _ilg.Emit (OpCodes.Br, label_condition_not_met);
      _ilg.MarkLabel (label_condition_met);
      _ilg.Emit (OpCodes.Ldc_I4_1);
      _ilg.MarkLabel (label_condition_not_met)
    }

    /**
     * Converts a list [Label] into a array [Label]
     * FIXME: this should work now?
     */
    private static label_list_to_array (x : list [Label]) : array [Label]
    {
      def result = array (List.Length (x));

      def loop (index : int, x : list [Label]) : void {
        match (x) {
          | head :: tail =>
            result [index] = head;
            loop (index + 1, tail)
          | [] => ()
        }
      };

      loop (0, x);

      result
    }

  } /* end of the class */

} /* end of the namespace */
