/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

namespace Nemerle.Compiler 
{
  using Nemerle.Collections;
  using Nemerle.Utility;
  using Nemerle.IO;
  using System.IO;
  using System.Text.RegularExpressions;

  using Nemerle.Compiler.Typedtree;

  using SR = System.Reflection;
  using SRE = System.Reflection.Emit;
  using PT = Nemerle.Compiler.Parsetree;  

  /**
   * The library reference exception is typically thrown
   * when unable to load an assembly file.
   */
  public class LibraryReferenceException : System.Exception 
  {
  }


  /**
   * Manager for the referenced libraries
   */
  public module LibraryReferenceManager
  {
    /* -- CONSTRUCTORS ----------------------------------------------------- */
  
    /**
     * Static constructor. Always load the Framework core library.
     */
    this () {
      namespace_nodes = Hashtable (300);

      def assembly_dir (t) {
        DirectoryOfCodebase (t.Assembly.Location);
      };
      
      // include current directory and directories where 
      // mscorlib.dll, System.dll and Nemerle.dll reside
      _lib_path = [".",
                    assembly_dir (typeof (Nemerle.Core.AssertionException)),
                    assembly_dir (typeof (System.Text.RegularExpressions.Match)),
                    assembly_dir (typeof (LibraryReferenceManager)),                    
                    assembly_dir (typeof (System.Object))];

      _loaded_assemblies_by_name = Hashtable (20);
      _assemblies_loaded_by_hand = Hashtable (20);
    }


    /* -- PUBLIC METHODS --------------------------------------------------- */

    public Init () : void
    {
      namespace_nodes.Clear ();
      _loaded_assemblies_by_name.Clear ();
      _assemblies_loaded_by_hand.Clear ();
    }

  
    /**
     * Stores the referenced library for later lookup by the namespaces. Also
     * add transitive closure of libraries needed by given assembly.
     */
    public AddLibrary (name : string) : void 
    {
      def assembly =
        match (name) {
          | "mscorlib" when Options.UseLoadedCorlib =>
            typeof (System.Object).Assembly
          | "System" when Options.UseLoadedCorlib =>
            typeof (System.Text.RegularExpressions.Match).Assembly
          | name => LookupAssembly (name)
        };
      if (assembly == null) {
        Message.Error ("cannot find assembly `" + name + "'");
        throw AssemblyFindException ()
      }
      else if (_loaded_assemblies_by_name.Contains (assembly.FullName)) {
        when (_assemblies_loaded_by_hand.Contains (assembly.FullName))
          Message.Warning ("assembly `" + name + "' already loaded");
        _assemblies_loaded_by_hand [assembly.FullName] = null;
      } else {
        _assemblies_loaded_by_hand [assembly.FullName] = null;
        AddAssembly (assembly)
      }
    }

    public AddAssembly (assembly : SR.Assembly) : void
    {
      unless (_loaded_assemblies_by_name.Contains (assembly.FullName)) {
        _ = LibraryReference (assembly);
        _loaded_assemblies_by_name.Set (assembly.FullName, assembly);
        
        when (Options.GreedyReferences) {
          def refs = assembly.GetReferencedAssemblies ();
          foreach (name : SR.AssemblyName in refs) {
            def asm =
              try {
                SR.Assembly.Load (name)
              } catch { _ is FileNotFoundException => 
                try {
                  SR.Assembly.LoadFrom (name.CodeBase)
                } catch { _ is System.Exception => 
                  LookupAssembly (name.Name)
                }
              };
            when (asm != null)
              AddAssembly (asm)
          }
        }
      }
    }
      
    public AddSearchDirectory (name : string) : void
    {
      _lib_path = name :: _lib_path;
    }
    
    /**
     * Remove assembly file name from URL returned by Assembly.CodeBase.
     */
    public DirectoryOfCodebase (mutable path : string) : string
    {
      path = Path.GetDirectoryName (path);

      // hack for mono returning location in the GAC
      def mono_gac = path.IndexOf ("mono/gac");
      path =
        if (mono_gac != -1)
          if (path.LastIndexOf ("Nemerle") > mono_gac)
            path.Substring (0, mono_gac + 5) + "nemerle/"
          else
            if (path.LastIndexOf ("/2.0.") > mono_gac)
              path.Substring (0, mono_gac + 5) + "2.0/"
            else
              path.Substring (0, mono_gac + 5) + "1.0/"
        else path;

      path
    }

    /* -- PRIVATE METHODS -------------------------------------------------- */

    private LookupAssembly (name : string) : SR.Assembly
    {
      def assembly_by_name (path : string) {
        try {
          def ext = Path.GetExtension (path);
          def path =
            match (ext.ToLower ()) {
              | ".dll" | ".exe" => path
              | _ => path + ".dll" 
            }
          SR.Assembly.LoadFrom (path)
        } catch {
          | _ is FileNotFoundException => null
          | _ is System.ArgumentException => null
        };
      };

      match (name) {
        | x when x.IndexOf ('/') != -1 || x.IndexOf ('\\') != -1 =>
          assembly_by_name (x)

        | x when x.IndexOf (',') != -1 =>
          try {
            SR.Assembly.Load (x)
          }
          catch {
            | _ is FileNotFoundException => null
            | _ is System.BadImageFormatException =>
              Message.Error ("passed strong name is invalid or assembly was not found");
              null
          }

        | _ =>
          def lookup (libs) {
            | lib :: libs =>
              def ret = assembly_by_name (Path.Combine (lib, name));
              if (ret == null) lookup (libs)
              else ret
            | [] => null
          };

          lookup (_lib_path);
      }
    }


    private load_macro (lib : SR.Assembly, macro_name : string) : void
    {
      def macro_type = lib.GetType (macro_name.Replace ("&", "\\&"));
      if (macro_type == null) {
        Message.Warning ($"failed to lookup macro `$(macro_name)' in assembly $(lib.FullName)");
      } 
      else {
        def ctor = macro_type.GetConstructor (System.Type.EmptyTypes);
        if (ctor == null) {
          Message.Warning ("failed to lookup constructor in macro `" + macro_name + 
                           "' from assembly " + lib.FullName);
        } 
        else {
          def macro_object = ctor.Invoke (null);
          if (macro_object == null) {
            Message.Warning ("failed to construct instance of macro `" + macro_name + 
                             "' from assembly " + lib.FullName);
          } 
          else {
            def casted_macro =
              try {
                (macro_object :> IMacro)
              } catch {
                | _ is System.InvalidCastException =>
                  Message.Warning ("instance of macro `" + macro_name + 
                                   "' from assembly " + lib.FullName + 
                                   " does not implement IMacro interface");
                  null
              };
              when (casted_macro != null) {
                def ns = casted_macro.GetNamespace ();
                when (namespace_nodes.Get (ns).IsNone)
                  _ = cache_namespace_node (ns);

                MacroRegistry.register_macro (casted_macro);
              }
          }
        }
      }
    }

    public LoadMacrosFrom (lib : SR.Assembly) : void
    {
//      assert (lib != null);
//      assert (_loaded_assemblies_by_name != null);
      unless (_loaded_assemblies_by_name.Contains (lib.FullName)) {
        foreach (attribute in lib.GetCustomAttributes (typeof (Nemerle.Internal.ContainsMacroAttribute), false))
        {
          def meth = attribute.GetType ().GetMethod ("GetName");
          def macro_name = (meth.Invoke (attribute, null) :> string);
          load_macro (lib, macro_name)
        }
        foreach (attribute :> Nemerle.Internal.OperatorAttribute
                 in lib.GetCustomAttributes (typeof (Nemerle.Internal.OperatorAttribute), false))
        {
          def node = NamespaceTree.ExactPath (NString.Split (attribute.env, '.'));
          def od = OperatorDefinition (node, attribute.name, attribute.left,
                                       attribute.right, attribute.IsUnary);
          MacroRegistry.AddOperatorDefinition (od);
        }
      }
    }
    
    
    public LoadMacrosFrom (name : string) : void
    {
      def assembly = LookupAssembly (name);
      if (assembly == null)
        Message.Error ("cannot load assembly with macros `" + name + "'")
      else
        LoadMacrosFrom (assembly);
    }
      
    
    /* -- PRIVATE VARIABLES ------------------------------------------------ */
    
    /**
     * List of directories we look for assemblies in.
     */
    private mutable _lib_path : list [string];

    private _loaded_assemblies_by_name : Hashtable [string, SR.Assembly];
    
    private _assemblies_loaded_by_hand : Hashtable [string, object];

    private namespace_nodes : Hashtable [string, NamespaceTree.Node];


    /* -- TYPE CACHE ----- */

    [Record]
    public class ExternalType {
      internal system_type : System.Type;
      internal library : LibraryReference;
      internal mutable tycon : TypeInfo;
      internal mutable message : string;
    }

    internal LoadTypesFrom (lib : LibraryReference) : void
    {
      def assembly = lib.GetAssembly ();
      def types = assembly.GetExportedTypes ();
      def codebasens = NamespaceTree.ExactPath ([assembly.Location]);

      mutable i = 0;
      for (; i < types.Length; ++i) {
        def t = types[i];
#if _GENERICS        
        def name = Regex.Replace (t.FullName.Replace ('+', '.'), "`.", "");
#else
        def name = t.FullName.Replace ('+', '.');
#endif        
//        Message.Debug (name);
        def e = ExternalType (t, lib, null, null);
        def notcached = NamespaceTree.TypeInfoCache.NotLoaded (e);

        def (ns_node, path_to_type) = framework_nesting (t);

        def mainnode = ns_node.Path (path_to_type);
        def fullnode = codebasens.Path (name);

        // check if we have met such type before
        def ambiguous_other = 
          match ((mainnode.Value, fullnode.Value))
          {
            | (NamespaceTree.TypeInfoCache.NotLoaded (x), _)
            | (_, NamespaceTree.TypeInfoCache.NotLoaded (x)) => x
            | _ => null
          };

        when (ambiguous_other != null) {
          // such type was there somewhere, so create verbose message
          // for the case of ambiguous use
          assert (ambiguous_other.tycon == null);

          e.message = 
            "[" + assembly.FullName + "], " + 
            (if (ambiguous_other.message == null)
               "[" + ambiguous_other.library.GetAssembly ().FullName + "]"
             else 
               ambiguous_other.message);
        };
        
        // assign wrappers for future loading of typecons
        mainnode.Value = notcached;
        fullnode.Value = notcached;
      }
    }

    internal CacheTypeInfo (t : System.Type, tc : TypeInfo) : void
    {
      def codebase = NamespaceTree.ExactPath ([t.Assembly.Location]); 
      def (ns_node, path_to_type) = framework_nesting (t);
      def ns = codebase.Path (ns_node.Name).Path (path_to_type);
      match (ns.Value) {
        | NamespaceTree.TypeInfoCache.NotLoaded (e) =>
          e.tycon = tc;
          ns.Value = NamespaceTree.TypeInfoCache.Cached (tc)
          
        | NamespaceTree.TypeInfoCache.No =>
          ns.Value = NamespaceTree.TypeInfoCache.Cached (tc)
          
        | NamespaceTree.TypeInfoCache.NamespaceReference
        | NamespaceTree.TypeInfoCache.MacroCall => 
          Util.ice ("wanted to cache in wrong place - value " + t.FullName)
        | NamespaceTree.TypeInfoCache.Cached =>
          Util.ice ("wanted to cache cached value " + t.FullName)
      }
    }

    internal GetInternalType (lib : LibraryReference, t : System.Type) : TypeInfo
    {
      def codebase = NamespaceTree.ExactPath ([t.Assembly.Location]);
      def (ns_node, path_to_type) = framework_nesting (t);
      def ns = codebase.Path (ns_node.Name).Path (path_to_type);
      match (ns.Value) {
        | NamespaceTree.TypeInfoCache.Cached (tc) => tc
        | NamespaceTree.TypeInfoCache.No =>
          def tc = lib.ConstructTypeInfo (t, ns);
          ns.Value = NamespaceTree.TypeInfoCache.Cached (tc);
          tc

        | _ =>
          Util.ice ("not loaded internal type... " + t.Assembly.CodeBase
                    + ":" + t.FullName)
      }
    }

    internal cache_namespace_node (namespc : string) : NamespaceTree.Node
    {
      def nd = NamespaceTree.ExactPath (NString.Split (namespc, array ['.']));
      when (nd.Value is NamespaceTree.TypeInfoCache.No)
        nd.Value = NamespaceTree.TypeInfoCache.NamespaceReference ();
      namespace_nodes.Add (namespc, nd);
      nd
    }

    internal framework_nesting (t : System.Type)
    : NamespaceTree.Node * list [string]
    {
      def loop (t : System.Type, acc) {
        def dt = t.DeclaringType;
        if (dt != null)
          loop (dt, dt.Name :: acc)
        else {
          def namespc = if (t.Namespace != null) t.Namespace else "";
          match (namespace_nodes.Get (namespc)) {
            | Some (nd) => (nd, acc)
            | None =>
              (cache_namespace_node (namespc), acc)
          }
        }
      }
      // workaround mono bug #63768
      if (t.IsPointer) {
        def find_name (acc, t : System.Type) {
          if (t.IsPointer)
            find_name (acc + "*", t.GetElementType ())
          else
            loop (t, [t.Name + acc])
        }
        find_name ("", t)
      } else
#if _GENERICS      
        loop (t, [Regex.Replace (t.Name, "`.", "")])
#else
        loop (t, [t.Name])
#endif
    }
  }


  /**
   * This class stores information extracted from a referenced external library.
   */
  public class LibraryReference
  {
    /**
     * The assembly associated with this object
     */
    private mutable _library : SR.Assembly;

    /**
     * The location of this library
     */
    private mutable _location : Location;
    
    /**
     * If set to true, the current assembly declares itself 
     * to be generated by the Nemerle compiler.
     */
    internal mutable _is_generated_by_nemerle : bool;

    private solver : Solver;
    

    // a little hack to prevent AddBuildins() calling in NetType
    internal static mutable add_buildins : bool = false;
    
    /**
     * Load an assembly. Extracts and processes the custom attributes array.
     */
    public this (assembly : SR.Assembly) 
    {
      _library = assembly;
      _location = Location (Location.GetFileIndex (assembly.Location), 0, 0);
      solver = Passes.Solver;

      // Scans the assembly custom attributes looking for something interesting...
      foreach (x :> SR.AssemblyConfigurationAttribute in
               _library.GetCustomAttributes (typeof (SR.AssemblyConfigurationAttribute), false))
        _is_generated_by_nemerle = x.Configuration == "ContainsNemerleTypes";

      LibraryReferenceManager.LoadTypesFrom (this);
      LibraryReferenceManager.LoadMacrosFrom (_library);
    }


    /**
     * Turns a Framework type into something edible by Nemerle's type system
     */
    internal TypeOfType (_tenv : SystemMap [int, StaticTyVar],
                         framework_type : System.Type) : MType
    {
      //Message.Debug (framework_type.ToString ());
      if (framework_type.Equals (SystemType.Void))
        InternalType.Void
      else
        if (framework_type.IsArray)
          MType.Array (TypeOfType (_tenv, framework_type.GetElementType ()),
                       framework_type.GetArrayRank ())
        else 
          if (framework_type.IsByRef) {
            Message.Error ($ "ref type referenced $framework_type");
            TypeOfType (_tenv, framework_type.GetElementType ())
          }
#if _NET_2_0            
          else if (framework_type.IsGenericParameter) {
#if _GENERICS
            assert (_tenv != null, framework_type.ToString ());
            def tv = Option.UnSome (_tenv.Find (framework_type.MetadataToken));
            MType.TyVarRef (tv)
#else
            InternalType.Void
#endif
          }
#endif                      
          else {
            def tc = TypeInfoOfType (framework_type);
#if _GENERICS
            if (framework_type.ContainsGenericParameters)
              MType.Class (tc, List.MapFromArray (framework_type.GetGenericArguments (),
                                                  fun (t) { TypeOfType (_tenv, t) }))
            else
              MType.Class (tc, [])
#else
            MType.Class (tc, [])
#endif            
          }
    } 

    private TypeInfoOfType (framework_type : System.Type) : TypeInfo
    {
      def (ns_node, path_to_type) =
        LibraryReferenceManager.framework_nesting (framework_type);

      match (ns_node.LookupType (path_to_type)) {
        | Some (tc) => tc
          // protected external types are not fetched automatically from external assembly
          // so sometimes we must load them by hand
        | None => LibraryReferenceManager.GetInternalType (this, framework_type)
      }
    } 

    internal GetAssembly () : SR.Assembly
    {
      this._library
    }
    
    /**
     * Construct TypeInfo object from given type in current assembly.
     */
    internal ConstructTypeInfo (reflected_type : System.Type,
                                ns_node : NamespaceTree.Node) : TypeInfo
    {
      if (
#if _GENERICS
          reflected_type.IsGenericTypeDefinition ||
          _is_generated_by_nemerle &&
          has_attribute ("Nemerle.Internal.TypeAttribute",
                         reflected_type.GetCustomAttributes (false))
//          reflected_type.IsDefined (typeof (Nemerle.Internal.TypeAttribute), false)
#else
          has_attribute ("Nemerle.Internal.TypeAttribute",
                         reflected_type.GetCustomAttributes (false))
//          reflected_type.IsDefined (typeof (Nemerle.Internal.NemerleAttribute), true)
#endif
       )
        (NetTypeInfo (this, reflected_type, ns_node) : TypeInfo)
      else
        (NetType (this, reflected_type, ns_node) : TypeInfo)
    }

    /**
     * Looks for type named [name] that is internal to current assembly. Used
     * by type attribute decoder.
     */
    internal LookupInternalType (name : string) : option [TypeInfo]
    {
    //  Message.Debug ($"looking `$(name)'");
      def st = _library.GetType (name);
      if (st == null)
        None ()
      else
        Some (LibraryReferenceManager.GetInternalType (this, st))
    }

    private MethodOfMethodInfo (self : IMember, meth : SR.MethodInfo) : MethodInfo
    {
      if (meth == null)
        null
      else
        MethodInfo ((self.DeclaringType :> NetType).tenv, this, meth)
    }



    /* -----------------------------------------------------------------
     * .NET types
     * ----------------------------------------------------------------- 
     */
    internal class NetType : TypeInfo
    {
      protected library : LibraryReference;
      protected attributes : NemerleAttributes;

      internal tenv : SystemMap [int, StaticTyVar] = SystemMap ();
      protected direct_supertypes : list [MType.Class];
      protected supertypes : list [MType.Class];
      protected supertype_map : SystemMap [int, MType.Class] = SystemMap ();

      
      /** member_list is used to collect all members of given type
         When we are queried for member "foo", we look for it, and if
         it was never looked up before, we add it to the member_list.
         When we are queried for the "" member (which means to list all
         the members), we call system_type.GetMembers () and skip
         members already in the member_list (they are marked in
         queried_members). This avoids creating duplicate members,
         which is crucial.  */
      mutable member_list : list [IMember];
      queried_members : Hashtable [string, object];
      mutable implicit_conversion_initialized : bool;

      /** members is kind of cache, but it is also used (in connection with
         member_list) to make sure each S.R.MemberInfo has only one 
         N.C.IMember */
      protected members : Hashtable [string, list [IMember]];

      public this (lib : LibraryReference, h : System.Type, ns_node : NamespaceTree.Node)
      {
        base (ns_node);
        
        // first cache ourself to avoid loops
        LibraryReferenceManager.CacheTypeInfo (h, this);
        library = lib;
        system_type = h;
        members = Hashtable (2);
        queried_members = Hashtable (2);
        member_list = [];

        when (system_type.DeclaringType != null)
          tenv = (lib.TypeInfoOfType (system_type.DeclaringType) :> NetType).tenv;

        // init typarms and self_type
        (typarms, tenv, direct_supertypes, supertypes) =
          if (library._is_generated_by_nemerle) {
            def type_attrs = system_type.GetCustomAttributes (false);
            def attr_str = get_encoded_type (type_attrs);         

            if (has_attribute ("Nemerle.Internal.TypeAttribute",
                               type_attrs))
              TyCodec.DecodeTypeBuilder (library, tenv, attr_str)
            else
              TyCodec.ReflectTypeBuilder (library, tenv, system_type);
          }
          else
            TyCodec.ReflectTypeBuilder (library, tenv, system_type);

        foreach (t in supertypes)
          supertype_map = supertype_map.Add (t.tycon.GetId (), t);

        fullname = system_type.FullName.Replace ('+', '.');
          
        def parms = List.Map (typarms, MType.TyVarRef);
        self_type = MType.Class (this, parms);
        
        // FIXME: get rid of this!
        when (IsDelegate) {
          def pref = "problem with delegate type `" + FullName + "': ";
          match (LookupMember (".ctor")) {
            | [ctor] =>
              def meth = (ctor :> IMethod);
              def fh = meth.GetHeader ();
              match (meth.GetParameters ()) {
                | [_, p2] =>
                  match (LookupMember ("Invoke")) {
                    | [inv] => 
                      def ty = (inv :> IMethod).GetMemType ();
                      fh.parms = [p2];
                      p2.ty = ty
                    | invs =>
                      Message.Warning (pref + sprintf ("invoke count %d",
                                                       List.Length (invs)))
                  }
                | parms =>
                  Message.Warning (pref + sprintf ("ctor has %d parms",
                                                   List.Length (parms)))
              }
            | ctors => Message.Warning (pref + sprintf ("%d constructors",
                                                        List.Length (ctors)))
          }
        }
        attributes = NemerleAttributes.None;

        when (system_type.IsAbstract) attributes |= NemerleAttributes.Abstract;
        when (system_type.IsSealed)   attributes |= NemerleAttributes.Sealed;
        when (system_type.IsValueType) attributes |= NemerleAttributes.Struct;

        if (system_type.DeclaringType == null) {
          if (system_type.IsPublic) attributes |= NemerleAttributes.Public
          else attributes |= NemerleAttributes.Internal;
        } else {
          when (system_type.IsNestedPrivate)  attributes |= NemerleAttributes.Private;
          when (system_type.IsNestedPublic)   attributes |= NemerleAttributes.Public;
          when (system_type.IsNestedFamily)   attributes |= NemerleAttributes.Protected;
          when (system_type.IsNestedAssembly) attributes |= NemerleAttributes.Internal;
        }

        def attrs = system_type.GetCustomAttributes (
                        SystemType.Reflection_DefaultMemberAttribute, true);
        default_indexer =
          if (attrs.Length > 0) {
            def dma = attrs [0] :> System.Reflection.DefaultMemberAttribute;
            Some (dma.MemberName)
          } else None ();
          
        when (LibraryReference.add_buildins) AddBuiltins ();
//        else Message.Debug ($"omiititng $FullName");
      }

      #region Special numeric overloads
      variant NumericKind
      {
        | Signed
        | Unsigned
        | Float
        | Char
      }

      static numeric_types : Hashtable [string, NumericKind];
      
      static this ()
      {
        numeric_types = Hashtable ();
        numeric_types.Add ("SByte", NumericKind.Signed ());
        numeric_types.Add ("Int16", NumericKind.Signed ());
        numeric_types.Add ("Int32", NumericKind.Signed ());
        numeric_types.Add ("Int64", NumericKind.Signed ());
        numeric_types.Add ("Byte", NumericKind.Unsigned ());
        numeric_types.Add ("UInt16", NumericKind.Unsigned ());
        numeric_types.Add ("UInt32", NumericKind.Unsigned ());
        numeric_types.Add ("UInt64", NumericKind.Unsigned ());
        numeric_types.Add ("Single", NumericKind.Float ());
        numeric_types.Add ("Double", NumericKind.Float ());
        numeric_types.Add ("Char", NumericKind.Char ()); 
      }

     
      FixupEquality (name : string) : void
      {
        FixupCompare (name);
        #if 0
        match (special_members [name]) {
          | [elem : IMethod] =>
            def tv = StaticTyVar ("'a");
            tv.Constraints = [];
            def hd = elem.GetHeader ();
            hd.typarms = [tv];
            match (hd.parms) {
              | [parm1, parm2] =>
                parm1.ty = MType.TyVarRef (tv);
                parm2.ty = MType.TyVarRef (tv);
              | _ => assert (false)
            }
          | _ => assert (false)
        }
        #endif
      }

      FixupShift (name : string) : void
      {
        match (special_members [name]) {
          | [elem is IMethod] =>
            match (elem.GetParameters ()) {
              | [_, parm2] =>
                assert (InternalType.Int32 != null);
                parm2.ty = InternalType.Int32;
              | _ => assert (false)
            }
          | _ => assert (false)
        }
      }

      AddConversions () : void
      {
        mutable is_numeric = true;
        def name = system_type.Name;
        
        def targets =
          if (system_type.Namespace != "System") {
            is_numeric = false;
            []
          } else
            match (name) {
              | "SByte" => ["Int16", "Int32", "Int64", "Single", "Double"]
              | "Int16" => ["Int32", "Int64", "Single", "Double"]
              | "Int32" => ["Int64", "Single", "Double"]
              | "Int64" => ["Single", "Double"]
              
              | "Byte" => 
                ["Int16", "Int32", "Int64", "UInt16", "UInt32", "UInt64", 
                 "Single", "Double"]
              | "UInt16" => 
                ["Int32", "Int64", "UInt32", "UInt64", "Single", "Double"]
              | "UInt32" => ["Int64", "UInt64", "Single", "Double"]
              | "UInt64" => ["Single", "Double"]
              
              | "Single" => ["Double"]
              | "Double" => []
              | "Char" => []
              | _ =>
                is_numeric = false;
                []
            }

        when (is_numeric) {
          foreach (target in targets) {
            def t = NamespaceTree.LookupInternalType (["System", target]);
            AddConversion ("op_Implicit", t);
          }

          // add remaining as explicit
          def numeric_types =
            ["SByte", "Int16", "Int32", "Int64", "Byte", "UInt16", "UInt32", 
             "UInt64", "Single", "Double", "Char"];

          foreach (target in numeric_types) {
            when (name != target && ! List.Contains (targets, target)) {
              def t = NamespaceTree.LookupInternalType (["System", target]);
              AddConversion ("op_Explicit", t);
            }
          }
        }
      }

      internal AddBuiltins () : void
      {
        InitBuiltinMembers ();
        when (system_type.DeclaringType == null &&
              system_type.Namespace == "System") {
          def name = system_type.Name;
          if (name == "String") {
            // == and != are surprisingly properly overloaded
            // and this is done in LookupMemberImpl, because otherwise
            // we would use uninitialized InternalType.String_Concat
            // AddBuiltin ("op_Addition", InternalType.String_Concat);
          } else if (name == "Boolean") {
            AddBuiltin ("op_LogicalNot", "bool.!");
            MakeSingleParm ("op_LogicalNot");
            AddBuiltin ("op_Equality", "==");
            AddBuiltin ("op_Inequality", "!=");
            // FixupEquality not needed here -- return type is already correct
            // and InternalType.Boolean is here null
          } else if (name == "Object") {
            AddBuiltin ("op_Equality", "==.ref");
            AddBuiltin ("op_Inequality", "!=.ref");
            FixupEquality ("op_Equality");
            FixupEquality ("op_Inequality");
          } else if (numeric_types.Contains (name)) {
            def kind = numeric_types [name];
            def suff =
              match (kind) {
                | NumericKind.Float => ".f"
                | NumericKind.Char
                | NumericKind.Signed => ".s"
                | NumericKind.Unsigned => ".u"
              }
            def is_small_type =
              name == "SByte" || name == "Byte" ||
              name == "Int16" || name == "UInt16";

            unless (kind is NumericKind.Char) {
              // checked/unchecked:
              AddBuiltin ("op_Addition", "+" + suff, "+.f");
              AddBuiltin ("op_Subtraction", "-" + suff, "-.f");
              AddBuiltin ("op_Multiply", "*" + suff, "*.f");
              
              // no checked version:
              AddBuiltin ("op_Division", "/" + suff);
              AddBuiltin ("op_Modulus", "%" + suff);

              // unary
              AddBuiltin ("op_UnaryPlus", "unary.+" + suff);
              MakeSingleParm ("op_UnaryPlus");

              when (is_small_type) {
                ForceIntType ("op_Addition");
                ForceIntType ("op_Subtraction");
                ForceIntType ("op_Multiply");
                ForceIntType ("op_Division");
                ForceIntType ("op_Modulus");
                ForceIntType ("op_UnaryPlus");
              }

              def one_size =
                if (name == "Int64" || name == "UInt64")
                  "i8"
                else if (name == "Float")
                  "r4"
                else if (name == "Double")
                  "r8"
                else
                  "i4";

              AddBuiltin ("op_Increment", "++." + one_size + suff, 
                                          "++." + one_size + ".f");
              AddBuiltin ("op_Decrement", "--." + one_size + suff, 
                                          "--." + one_size + ".f");
              MakeSingleParm ("op_Increment");
              MakeSingleParm ("op_Decrement");
            }

            when (kind is NumericKind.Signed ||
                  kind is NumericKind.Unsigned) {
              AddBuiltin ("op_BitwiseAnd", "&" + suff);
              AddBuiltin ("op_BitwiseOr", "|" + suff);
              AddBuiltin ("op_ExclusiveOr", "^" + suff);
              
              AddBuiltin ("op_LeftShift", "<<" + suff);
              AddBuiltin ("op_RightShift", ">>" + suff);
              when (Name != "Int32") {
                FixupShift ("op_LeftShift");
                FixupShift ("op_RightShift");
              }
              when (is_small_type) {
                ForceIntType ("op_LeftShift");
                ForceIntType ("op_RightShift");
                ForceIntType ("op_BitwiseAnd");
                ForceIntType ("op_BitwiseOr");
                ForceIntType ("op_ExclusiveOr");
              }
              
              AddBuiltin ("op_UnaryBitwiseNot", "unary.~" + suff);
              MakeSingleParm ("op_UnaryBitwiseNot");
              when (is_small_type)
                ForceIntType ("op_UnaryBitwiseNot");
            }

            unless (kind is NumericKind.Unsigned || 
                    kind is NumericKind.Char) {
              AddBuiltin ("op_UnaryMinus", "unary.-" + suff, "unary.-.f");
              MakeSingleParm ("op_UnaryMinus");
              when (is_small_type)
                ForceIntType ("op_UnaryMinus");
            }

            // comparisions
            AddBuiltin ("op_Equality", "==");
            AddBuiltin ("op_Inequality", "!=");
            AddBuiltin ("op_LessThan", "<" + suff);
            AddBuiltin ("op_LessThanOrEqual", "<=" + suff);
            AddBuiltin ("op_GreaterThan", ">" + suff);
            AddBuiltin ("op_GreaterThanOrEqual", ">=" + suff);

            FixupCompare ("op_Equality");
            FixupCompare ("op_Inequality");
            FixupCompare ("op_LessThan");
            FixupCompare ("op_LessThanOrEqual");
            FixupCompare ("op_GreaterThan");
            FixupCompare ("op_GreaterThanOrEqual");
          } else {}
        }
      }
      #endregion

      /** Checks if underlying .NET type is value type */
      public override IsValueType : bool {
        get { system_type.IsValueType }
      }

      public override IsInterface : bool {
        get { system_type.IsInterface }
      }
      
      public override MemberType : MemberTypes
      { get { system_type.MemberType } }

      public override Name : string
      { get { system_type.Name } }
      
      internal override GetLibraryReference () : LibraryReference
      { library }

      public override IsSealed : bool
      { get { system_type.IsSealed } }
      
      public override IsDelegate : bool
      {
        get { system_type.IsSubclassOf (SystemType.MulticastDelegate) }
      }

      public override GetConstantObject () : IField
      {
        null
      }

      public override DeclaringType : TypeInfo {
        get {
          if (system_type.DeclaringType == null)
            null
          else
            library.TypeInfoOfType (system_type.DeclaringType)
        }
      }
      
      public override Location : Location
      {
        get { library._location }
      }

      public override Attributes : NemerleAttributes
      {
        get { attributes }
      }

      public override GetModifiers () : Modifiers
      {
        Util.ice ("GetModifiers not supported on external entities")
      }

      public override AttributeTargets : System.AttributeTargets
      {
        get {
          def usage = system_type.GetCustomAttributes (typeof (System.AttributeUsageAttribute), false);
          if (usage.Length > 0) {
            assert (usage.Length == 1);
            (usage[0] :> System.AttributeUsageAttribute).ValidOn
          }
          else
            System.AttributeTargets.All
        }
      }
      
      public override HasAttribute (attribute : TypeInfo) : bool
      {
        def t = attribute.GetSystemType ();
        system_type.IsDefined (t, false)
      }
      
      public override Accessibility : Accessibility
      {
        get {
          match (system_type.Attributes %& SR.TypeAttributes.VisibilityMask) {
            | SR.TypeAttributes.NestedAssembly => Accessibility.Internal
            | SR.TypeAttributes.NestedFamANDAssem => Accessibility.ProtectedAndInternal
            | SR.TypeAttributes.NestedFamily => Accessibility.Protected
            | SR.TypeAttributes.NestedFamORAssem => Accessibility.ProtectedOrInternal
            | SR.TypeAttributes.Public
            | SR.TypeAttributes.NestedPublic => Accessibility.Public
            | SR.TypeAttributes.NestedPrivate              
            | SR.TypeAttributes.NotPublic => Accessibility.Private
            | _ => Util.ice ("system type returned invalid attributes")
          }
        }
      }

      public override IsExternallyAccessible : bool
      {
        // if it wasn't, we wouldn't have reflected it?
        get { true }
      }
      
      public override HasBeenUsed : bool
      {
        get { true }
        set { ignore (value) }
      }
      
      public override GetTydecl () : TypeDeclaration
      {
        if (system_type.IsInterface)
          TypeDeclaration.Interface ()
        else if (system_type.IsEnum)
          TypeDeclaration.Enum ()
        else
          TypeDeclaration.Class ()
      }

      public override GetSuperTypes () : list [MType.Class]
      {
        supertypes
      }

      public override GetDirectSuperTypes () : list [MType.Class]
      {
        direct_supertypes
      }
      
      public override SuperType (tc : TypeInfo) : option [list [MType]]
      {
        if (tc : object == this)
          Some (Solver.FixedValues (self_type.args))
        else if (tc : object == InternalType.Object_tc) Some ([])
        else
          match (supertype_map.Find (tc.GetId ())) {
            | Some (ti) => Some (Solver.FixedValues (ti.args))
            | None => None ()
          }
      }

      internal override SubtypingSubst (tc : TypeInfo) : Subst
      {
        if (this.Equals (tc))
          Subst () // current type, no substitution
        else
          tc.MakeSubst1 (Option.UnSome (SuperType (tc)));
      }

      public override BaseType : TypeInfo
      {
        get {
          if (system_type.BaseType == null) null
          else library.TypeInfoOfType (system_type.BaseType)
        }
      }
      
      public override SuperClass () : option [TypeInfo]
      {
        if (system_type.BaseType == null)
          None ()
        else                                    
          Some (library.TypeInfoOfType (system_type.BaseType))
      }

      public override GetMembers () : list [IMember]
      {
        LookupMember ("")
      }
      
      public override GetMembers (bindingAttr : BindingFlags) : list [IMember]
      {
        def check (x : IMember) {
          if (bindingAttr %&& BindingFlags.DeclaredOnly)
            this.Equals (x.DeclaringType) &&
            TypeBuilder.constrain_member (x, bindingAttr)
          else
            TypeBuilder.constrain_member (x, bindingAttr)
        };
        List.RevFilter (GetMembers (), check)
      }

      public override GetFields (bindingAttr : BindingFlags) : list [IField] 
      {
        List.FoldLeft (GetMembers (bindingAttr), [], fun (x : IMember, acc) {
          if (TypeBuilder.constrain_member (x, bindingAttr))
            match (x) {
              | f is IField => f :: acc
              | _ => acc
            }
          else acc
        })
      }

      public override GetConstructors (bindingAttr : BindingFlags) : list [IMethod] 
      {
        List.FoldLeft (GetMembers (bindingAttr), [], fun (x : IMember, acc) {
          if (TypeBuilder.constrain_member (x, bindingAttr))
            match (x.GetKind ()) {
              | MemberKind.Method (m) =>
                match (m.GetFunKind ()) {
                  | FunKind.Constructor | FunKind.StaticConstructor => m :: acc
                  | _ => acc
                }
              | _ => acc
            }
          else acc
        })
      }
      
      private static is_internal (m : SR.MemberInfo) : bool
      {
        def x = m.MemberType;
        if (x %&& (MemberTypes.Constructor | MemberTypes.Method)) 
        {
          def meth = m :> SR.MethodBase;
          meth.IsPrivate || meth.IsAssembly || meth.IsFamilyAndAssembly
        } 
        else if (x %&& MemberTypes.Field) 
        {
          def field = m :> SR.FieldInfo;
          field.IsPrivate || field.IsAssembly || field.IsFamilyAndAssembly
        } 
        else if (x %&& MemberTypes.Property) 
        {
          def prop = m :> SR.PropertyInfo;
          def accessors = prop.GetAccessors (true);
          def propMeth = NArray.Fold (accessors, null, fun (x : SR.MethodInfo, acc) { 
            if (x == null || x.IsPrivate || x.IsAssembly || x.IsFamilyAndAssembly) acc 
            else x
          });
          propMeth == null
        }
        else if (x %&& MemberTypes.NestedType) 
        {
          def ty = m :> System.Type;
          ty.IsNestedAssembly || ty.IsNestedPrivate || ty.IsNestedFamANDAssem
        }
        else if (x %&& MemberTypes.Event) 
        {
          // we lookup nonpublic method here, because it can be protected and we want to
          // access it from derived class
          // (I just hope it won't trigger any authorization exceptions)
          def evMeth = (m :> SR.EventInfo).GetAddMethod (true);
          evMeth == null || evMeth.IsPrivate || evMeth.IsAssembly || evMeth.IsFamilyAndAssembly
        } 
        else true
      }

      protected virtual imember_of_memberinfo (m : SR.MemberInfo) : IMember
      {
        match (m.MemberType) {
          | MemberTypes.Constructor 
          | MemberTypes.Method => MethodInfo (tenv, library, m :> SR.MethodBase) : IMember
          | MemberTypes.Field => FieldInfo (library, m :> SR.FieldInfo)
          | MemberTypes.Property => NetProperty (null, library, m :> SR.PropertyInfo)
          | MemberTypes.Event => NetEvent (library, m :> SR.EventInfo)
          | MemberTypes.NestedType => library.TypeInfoOfType (m :> System.Type)
          | _ => null
        }
      }

      private collect_members (name : string) : void
      {
        assert (! queried_members.Contains (name));
        queried_members [name] = null;

        def fields_to_shadow = Hashtable ();
        
        def store_by_name (m : IMember) {
          members [m.Name] =
            if (members.Contains (m.Name))
              m :: members [m.Name]
            else
              [m];
        }
        
        def add_member (m : SR.MemberInfo, acc) {
          // Message.Debug ($"adding $m from $this");
          
          if (!library.TypeInfoOfType (m.DeclaringType).Equals (this)) {
            acc
          } else if (is_internal (m) ||
                     (name == "" && queried_members.Contains (m.Name))) {
            acc
          } else {
            //Message.Debug ($ "add_member: $m to $(this) [$(GetId ())]");
            match (imember_of_memberinfo (m)) {
              | null => acc
              | r =>
                //Message.Debug ($ "on: $(r.OverloadName)");
                fields_to_shadow [r.OverloadName] = null;
                store_by_name (r);
                r :: acc
            }
          }
        }
        
        def flags = SR.BindingFlags.Public     |
                    SR.BindingFlags.NonPublic  |
                    SR.BindingFlags.Static     |
                    SR.BindingFlags.Instance   |
                    SR.BindingFlags.DeclaredOnly;

        def member_array = 
          if (name == "")
            system_type.GetMembers (flags) 
          else 
            system_type.GetMember (name, flags);

        def collected_now = NArray.Fold (member_array, [], add_member);

        def our_fields =
          if (name == "") {
            foreach (m : IMember in member_list)
              fields_to_shadow [m.OverloadName] = null;
            member_list + collected_now
          } else {
            member_list = collected_now + member_list;
            collected_now
          }

        def fields_from_parents =
          match (SuperClass ()) {
            | Some (sc) => sc.LookupMember (name)
            | None =>
              // make sure each and every member gets its place just once
              def uniq (lst) {
                def ht = Hashtable ();
                List.RevFilter (lst, fun (e : IMember) { 
                  if (ht.Contains (e.GetId ())) false
                  else { ht [e.GetId ()] = null; true }
                })
              }
              
              if (system_type.IsInterface)
                // collect members from inherited interfaces
                uniq (
                  NArray.Fold (system_type.GetInterfaces (), [], 
                  fun (st, acc) {
                    library.TypeInfoOfType (st).LookupMember (name) + acc
                  }) +
                  // and from System.Object
                  InternalType.Object_tc.LookupMember (name))
              else []
          };

        def fields_from_parents = 
          List.RevFilter (fields_from_parents, 
                       fun (mem : IMember) {
                         !(mem is BuiltinMethod) && // FIXME: reorganize builtins
                         mem.Name != ".ctor" &&
                         mem.Name != ".cctor" &&
                         ! fields_to_shadow.Contains (mem.OverloadName)
                       });

        if (name == "") {
          foreach (mem : IMember in fields_from_parents)
            unless (queried_members.Contains (mem.Name))
              store_by_name (mem);
          members [""] = List.RevAppend (our_fields, fields_from_parents);
        } else
          List.Iter (fields_from_parents, store_by_name);
      }

      public override LookupMemberImpl (name : string) : list [IMember] 
      {
        if (this.Equals (InternalType.String_tc) && 
            name == "op_Addition") {
          assert (InternalType.String_Concat != null);
          AddBuiltin ("op_Addition", InternalType.String_Concat);
          LookupMember (name)
        } else if (!implicit_conversion_initialized && 
                   (name == "op_Implicit" || name == "op_Explicit")) {
          implicit_conversion_initialized = true;
          AddConversions ();
          LookupMember (name)
        } else {
          match (members.Get (name)) {
            | Some (r) => r
            | None =>
              collect_members (name);
              if (members.Contains (name))
                members [name]
              else {
                members [name] = [];
                []
              }
          }
        }
      }

      public override CanAccess (source : TypeInfo) : bool
      {
        system_type.IsPublic || system_type.IsNestedPublic ||
          (system_type.DeclaringType != null &&
           Option.IsSome (source.SuperType (DeclaringType)) &&
           (system_type.IsNestedFamily || system_type.IsNestedFamORAssem))
      }

      public override UnderlyingType : TypeInfo
      {
        get {
          assert (system_type.IsEnum);
          library.TypeInfoOfType (system_type.GetField ("value__").FieldType)
        }
      }
    }
    
    private class FieldInfo : IField
    {
      protected handle : SR.FieldInfo;
      protected library : LibraryReference;
      protected id : int;
      protected mutable tt_type : MType;
      protected mutable attributes : NemerleAttributes;

      public this (lib : LibraryReference, h : SR.FieldInfo)
      {
        id = Util.next_id ();
        library = lib;
        handle = h;
        tt_type = null;
        set_attributes ();
      }

      public this (tenv : SystemMap [int, StaticTyVar], lib : LibraryReference,
                   h : SR.FieldInfo)
      {
        id = Util.next_id ();
        library = lib;
        handle = h;
       
        def type_attrs = h.GetCustomAttributes (false);
        def attr_str = get_encoded_type (type_attrs); 
        tt_type = TyCodec.DecodeType (lib, tenv, attr_str);

        set_attributes ();
      }

      set_attributes () : void
      {
        attributes = NemerleAttributes.None;
        when (!handle.IsInitOnly) attributes |= NemerleAttributes.Mutable;
        when (handle.IsStatic)    attributes |= NemerleAttributes.Static;
       
        when (handle.IsPrivate)  attributes |= NemerleAttributes.Private;
        when (handle.IsPublic)   attributes |= NemerleAttributes.Public;
        when (handle.IsFamily)   attributes |= NemerleAttributes.Protected;
        when (handle.IsAssembly) attributes |= NemerleAttributes.Internal;
        when (handle.IsFamilyAndAssembly) attributes |=
          NemerleAttributes.Internal %| NemerleAttributes.Protected;
      }

      public DeclaringType : TypeInfo
      {
        get {
          library.TypeInfoOfType (handle.DeclaringType)
        }
      }
      
      public Name : string
      {
        get { handle.Name }
      }

      public override ToString () : string
      {
        NemerleMember.DescribeMember (this)
      }

      public OverloadName : string
      {
        get { Name }
      }

      public MemberType : MemberTypes
      {
        get { MemberTypes.Field }
      }

      public GetFieldInfo () : SR.FieldInfo
      {
        handle
      }
      
      public GetHandle () : SR.MemberInfo
      {
        handle
      }

      public GetMemType () : MType
      {
        when (tt_type == null)
          tt_type = library.TypeOfType (null, handle.FieldType);
        tt_type
      }
      
      [Nemerle.OverrideObjectEquals]
      public Equals (o : IMember) : bool
      {
        //Message.Debug ($ "$(this).Equals($o)");
        GetId () == o.GetId ()
      }

      public GetKind () : MemberKind
      {
        MemberKind.Field (this)
      }

      public Location : Location
      {
        get { library._location }
      }

      public Attributes : NemerleAttributes
      {
        get { attributes }
      }

      public GetModifiers () : Modifiers
      {
        Util.ice ("GetModifiers not supported on external entities")
      }

      public IsStatic : bool
      {
        get { handle.IsStatic }
      }

      public HasBeenUsed : bool
      {
        get { true }
        set { ignore (value) }
      }

      public HasBeenAssigned : bool
      {
        get { true }
        set { ignore (value) }
      }
      
      public IsLiteral : bool
      {
        get { handle.IsLiteral }
      }

      public GetValue () : Literal
      {
        assert (IsLiteral);
        def fval = handle.GetValue (null);
        def lit = 
          match (fval) {
            | val is ulong => Literal.Integer (val, false, null).WithProperType ()
            | val is double => Literal.Double (val)
            | val is float => Literal.Float (val)
            | val is string => Literal.String (val)
            | val is char => Literal.Char (val)
            | val is bool => Literal.Bool (val)
            | val =>
              def val = System.Convert.ToInt64 (val);
              if (val == long.MinValue)
                Literal.Integer (0x8000000000000000UL, true, InternalType.Int64)
              else
                Literal.Integer (System.Math.Abs (val) :> ulong, val < 0, null).WithProperType ()
          }
        if (handle.FieldType.IsEnum)
          Literal.Enum (lit :> Literal.Integer, library.TypeInfoOfType (handle.DeclaringType))
        else
          lit
      }

      public GetId () : int
      {
        id
      }

      public IsMutable : bool
      {
        get { !handle.IsInitOnly }
      }

      public IsVolatile : bool
      {
        get
        {
          def modreqs = handle.GetCustomAttributes (false);

          NArray.Exists (modreqs,
                         fun (x : System.Object) {
                           (x is System.Runtime.CompilerServices.IsVolatile) ||
                           (x is Nemerle.Internal.VolatileModifier)
                         })
        }
      }
      
      public CanAccess (source : TypeInfo) : bool
      {
        handle.IsPublic || 
          (Option.IsSome (source.SuperType (DeclaringType)) &&
           (handle.IsFamily || handle.IsFamilyOrAssembly))
      }
    }

    class NetEvent : IEvent
    {
      protected handle : SR.EventInfo;
      protected library : LibraryReference;
      protected id : int;
      protected is_static : bool;
      protected adder : MethodInfo;
      protected remover : MethodInfo;
      protected mutable mem_type : MType;

      public this (lib : LibraryReference, h : SR.EventInfo)
      {
        id = Util.next_id ();
        library = lib;
        handle = h;
        
        def add_method = handle.GetAddMethod (true);
        def remove_method = handle.GetRemoveMethod (true);

        is_static =
          (add_method != null && add_method.IsStatic) ||
          (remove_method != null && remove_method.IsStatic);

        adder = library.MethodOfMethodInfo (this, add_method);
        remover = library.MethodOfMethodInfo (this, remove_method);

        assert (adder != null);
        assert (remover != null);

        match (adder.GetMemType ()) {
          | MType.Fun (t, _) => mem_type = t.FixedValue
          | _ => assert (false)
        }
      }

      public DeclaringType : TypeInfo
      {
        get {
          library.TypeInfoOfType (handle.DeclaringType)
        }
      }
      
      public Name : string
      {
        get { handle.Name }
      }

      public OverloadName : string
      {
        get { Name }
      }

      public override ToString () : string
      {
        NemerleMember.DescribeMember (this)
      }

      public MemberType : MemberTypes
      {
        get { MemberTypes.Event }
      }

      public GetAdder () : IMethod
      {
        adder
      }

      public GetRemover () : IMethod
      {
        remover
      }

      public GetEventInfo () : SR.EventInfo
      {
        handle
      }
      
      public GetHandle () : SR.MemberInfo
      {
        handle
      }

      public GetMemType () : MType
      {
        mem_type
      }

      [Nemerle.OverrideObjectEquals]
      public Equals (o : IMember) : bool
      {
        GetId () == o.GetId ()
      }

      public GetKind () : MemberKind
      {
        MemberKind.Event (this)
      }

      public Location : Location
      {
        get { library._location }
      }

      public Attributes : NemerleAttributes
      {
        get { adder.Attributes }
      }

      public GetModifiers () : Modifiers
      {
        Util.ice ("GetModifiers not supported on external entities")
      }

      public IsStatic : bool
      {
        get { is_static }
      }

      public HasBeenUsed : bool
      {
        get { true }
        set { ignore (value) }
      }
      
      public GetId () : int
      {
        id
      }

      public CanAccess (_source : TypeInfo) : bool
      {
        // FIXME: this is broken... we need to check method attributes
        true
      }
    }
    
    class NetProperty : IProperty
    {
      protected handle : SR.PropertyInfo;
      protected library : LibraryReference;
      protected id : int;
      protected is_static : bool;
      protected getter : MethodInfo;
      protected setter : MethodInfo;
      protected any_method : MethodInfo;
      protected mutable mem_type : MType;
      protected overload_name : string;

      public this (tenv : SystemMap [int, StaticTyVar], lib : LibraryReference, h : SR.PropertyInfo)
      {
        id = Util.next_id ();
        library = lib;
        handle = h;
        
        def get_method = handle.GetGetMethod (true);
        def set_method = handle.GetSetMethod (true);

        getter = library.MethodOfMethodInfo (this, get_method);
        setter = library.MethodOfMethodInfo (this, set_method);

        if (getter == null)
          any_method = setter;
        else
          any_method = getter;

        assert (any_method != null);

        is_static = any_method.Attributes %&& NemerleAttributes.Static;
        
        def ret_type = library.TypeOfType (tenv, handle.PropertyType);

        def (args, ret_type) =
          if (getter != null)
            match (getter.GetMemType ()) {
              | MType.Fun (t, r) =>
                (t.FixedValue.GetFunctionArguments (), r.FixedValue)
              | _ => assert (false)
            }
          else if (setter != null)
            match (setter.GetMemType ()) {
              | MType.Fun (t, _) =>
                def (args, r) = List.DivideLast (t.FixedValue.GetFunctionArguments ());
                (args, r)
              | _ => assert (false)
            }
          else
            ([], ret_type);

        if (args.IsEmpty)
          mem_type = ret_type
        else
          mem_type = MType.ConstructFunctionType (args, ret_type);
          
        overload_name =
          if (args.IsEmpty) Name
          else $ "$(this.Name)[$args]";
      }

      public DeclaringType : TypeInfo
      {
        get {
          library.TypeInfoOfType (handle.DeclaringType)
        }
      }
      
      public Name : string
      {
        get { handle.Name }
      }

      public OverloadName : string
      {
        get { overload_name }
      }

      public MemberType : MemberTypes
      {
        get { MemberTypes.Property }
      }

      public override ToString () : string
      {
        NemerleMember.DescribeMember (this)
      }

      public GetGetter () : IMethod
      {
        getter
      }

      public GetSetter () : IMethod
      {
        setter
      }

      public GetPropertyInfo () : SR.PropertyInfo
      {
        handle
      }
      
      public GetHandle () : SR.MemberInfo
      {
        handle
      }

      public GetMemType () : MType
      {
        mem_type
      }

      [Nemerle.OverrideObjectEquals]
      public Equals (o : IMember) : bool
      {
        GetId () == o.GetId ()
      }

      public GetKind () : MemberKind
      {
        MemberKind.Property (this)
      }

      public Location : Location
      {
        get { library._location }
      }

      public Attributes : NemerleAttributes
      {
        get { any_method.Attributes }
      }

      public GetModifiers () : Modifiers
      {
        Util.ice ("GetModifiers not supported on external entities")
      }

      public IsStatic : bool
      {
        get { is_static }
      }
      
      public HasBeenUsed : bool
      {
        get { true }
        set { ignore (value) }
      }

      public GetId () : int
      {
        id
      }

      public IsMutable : bool
      {
        get { handle.CanWrite }
      }

      public CanAccess (source : TypeInfo) : bool
      {
        any_method.CanAccess (source)
      }

      public IsIndexer : bool
      {
        get
        {
          def index_parms = handle.GetIndexParameters ();

          index_parms.Length > 0
        }
      }
    }
    
    class MethodInfo : IMethod
    {
      protected handle : SR.MethodBase;
      protected library : LibraryReference;
      protected id : int;
      protected fun_header : Fun_header;
      protected attributes : NemerleAttributes;
      protected is_var_args : bool;
      protected overload_name : string;
      
      public this (tenv : SystemMap [int, StaticTyVar], lib : LibraryReference, h : SR.MethodBase)
      {
        library = lib;
        handle = h;
        def mkparm (p : SR.ParameterInfo) {
          def (parmkind, ty) =
            if (p.ParameterType.IsByRef) {
              def ty = library.TypeOfType (tenv, p.ParameterType.GetElementType ());
              if (p.IsOut)
                (ParmKind.Out, MType.Out (ty))
              else
                (ParmKind.Ref, MType.Ref (ty))
            } else
              (ParmKind.Normal, library.TypeOfType (tenv, p.ParameterType));
          def fp = Fun_parm (
            loc = lib._location,
            name = p.Name,
            color = 0,
            ty = ty,
            kind = parmkind,
            modifiers = Modifiers.Empty // FIXME?
          );
          def deflt = p.DefaultValue;
          when (deflt != System.DBNull.Value)
            fp.default_value = Some (Literal.FromObject (deflt));
          fp
        };
        def ret_type =
          if (handle.Name == ".ctor" || handle.Name == ".cctor")
            InternalType.Void
          else
            library.TypeOfType (tenv, (handle :> SR.MethodInfo).ReturnType);
        
        def parms = handle.GetParameters ();

        when (parms.Length > 0) {
          def attrs = parms [parms.Length - 1]
                .GetCustomAttributes (SystemType.ParamArrayAttribute, false);
          is_var_args = attrs.Length > 0;
        }

        fun_header =
          Fun_header (
            loc = library._location,
            name = Name,
            ret_type = ret_type,
            parms = List.MapFromArray (parms, mkparm),
            typarms = [],
            tenv = null
          );

        def parm_types = List.Map (fun_header.parms, 
                                   fun (p : Fun_parm) { p.ty });
        overload_name = $ "$(this.Name)($parm_types)";

        id = fun_header.id;

        attributes = NemerleAttributes.None;
        when (handle.IsAbstract) attributes |= NemerleAttributes.Abstract;
        when (handle.IsVirtual)  attributes |= NemerleAttributes.Virtual;
        when (handle.IsFinal)    attributes |= NemerleAttributes.Sealed;
        when (handle.IsStatic)   attributes |= NemerleAttributes.Static;
       
        when (handle.IsPrivate)  attributes |= NemerleAttributes.Private;
        when (handle.IsPublic)   attributes |= NemerleAttributes.Public;
        when (handle.IsFamily || handle.IsFamilyOrAssembly) attributes |= NemerleAttributes.Protected;

        when (tenv != null) {
          def type_attrs = h.GetCustomAttributes (false);
          def attr_str = get_encoded_type (type_attrs); 
          when (attr_str != null)
            TyCodec.FixupFunctionHeader (lib, tenv, fun_header, attr_str);
        }
      }
      
      public DeclaringType : TypeInfo
      {
        get {
          library.TypeInfoOfType (handle.DeclaringType)
        }
      }
      
      public Name : string
      {
        get { handle.Name }
      }

      public MemberType : MemberTypes
      {
         get { handle.MemberType }
      }

      public OverloadName : string
      {
        get { overload_name }
      }

      public override ToString () : string
      {
        NemerleMember.DescribeMember (this)
      }

      public IsVarArgs : bool
      {
        get { is_var_args }
      }

      public GetMethodBase () : SR.MethodBase
      {
        handle
      }
      
      public GetHandle () : SR.MemberInfo
      {
        handle
      }

      public GetMemType () : MType
      {
        MType.ConstructFunctionType (GetHeader ())
      }

      [Nemerle.OverrideObjectEquals]
      public Equals (o : IMember) : bool
      {
        GetId () == o.GetId ()
      }

      public GetKind () : MemberKind
      {
        MemberKind.Method (this)
      }

      public Location : Location
      {
        get { library._location }
      }

      public Attributes : NemerleAttributes
      {
        get { attributes }
      }

      public GetModifiers () : Modifiers
      {
        Util.ice ("GetModifiers not supported on external entities")
      }

      public IsFinal : bool
      { get { handle.IsFinal } }
      
      public IsStatic : bool
      {
        get { handle.IsStatic }
      }
      
      public HasBeenUsed : bool
      {
        get { true }
        set { ignore (value) }
      }

      public GetId () : int
      {
        id
      }

      public GetFreshType () : MType
      {
        if (fun_header.typarms.IsEmpty)
          GetMemType ()
        else {
          def subst = Subst.Fresh (fun_header.typarms);
          subst.MonoApply (GetMemType ())
        }
      }

      public GetFunKind () : FunKind
      {
        if (handle.IsConstructor)
          if (handle.IsStatic)
            FunKind.StaticConstructor ()
          else
            FunKind.Constructor ()
        else
          if (handle.IsStatic)
            FunKind.Function ()
          else
            FunKind.BoundMethod ([])
      }

      public GetConstructorInfo () : SR.ConstructorInfo
      {
        assert (handle.IsConstructor);
        (handle :> SR.ConstructorInfo)
      }

      public GetMethodInfo () : SR.MethodInfo
      {
        assert (!handle.IsConstructor);
        (handle :> SR.MethodInfo)
      }

      public GetHeader () : Fun_header
      {
        fun_header
      }

      public GetParameters () : list [Fun_parm]
      {
        fun_header.parms
      }

      public ReturnType : TyVar
      {
        get {
          fun_header.ret_type
        }
      }
            
      public CanAccess (source : TypeInfo) : bool
      {
        handle.IsPublic || 
          (Option.IsSome (source.SuperType (DeclaringType)) &&
           (handle.IsFamily || handle.IsFamilyOrAssembly))
      }
      
      public BuiltinKind : BuiltinMethodKind
      {
        get { BuiltinMethodKind.NotBuiltin () }
      }
    }
   
    /* -----------------------------------------------------------------
     * External Nemerle types
     * ----------------------------------------------------------------- 
     */
     
    private static get_string_attribute (attr_name : string, meth_name : string,
                                         attributes : array [object]) : string
    {
      def loop (i) {
        if (i >= attributes.Length) null
        else {
          def attr = attributes [i];
          def attr_ty = attr.GetType ();
          if (attr_ty.FullName == attr_name) {
            def meth = attr_ty.GetMethod (meth_name);
            (meth.Invoke (attr, null) :> string)
          } else loop (i + 1)
        }
      };
      loop (0)
    }

    private static get_encoded_type (attributes : array [object]) : string
    {
      get_string_attribute ("Nemerle.Internal.TypeAttribute", "GetEncodedType", attributes)
    }

    private static get_variant_options (attributes : array [object]) : list [string]
    {
      def str = get_string_attribute ("Nemerle.Internal.VariantAttribute", 
                                      "GetVariantOptions", attributes);
      NString.Split (str, array [','])
    }

    private static has_attribute (name : string, attributes : array [object]) : bool
    {
      def loop (i) {
        i < attributes.Length &&
        ({
          def attr = attributes [i];
          def attr_ty = attr.GetType ();
          if (attr_ty.FullName == name) true
          else loop (i + 1)
        })
      };
      loop (0)
    }
    
    private class NetTypeInfo : NetType
    {
      tydecl : TypeDeclaration;
      constant_object : IField;

      public this (lib : LibraryReference, h : System.Type, ns_node : NamespaceTree.Node)
      {
        base (lib, h, ns_node);

        def type_attrs = system_type.GetCustomAttributes (false);        
        
        if (has_attribute ("Nemerle.Internal.VariantAttribute", type_attrs)) {
          def names = get_variant_options (type_attrs);
          def get_opt (name : string) {
            match (NamespaceTree.LookupExactType (NString.Split (name, array ['.', '+']))) {
              | Some (tc) => tc
              | None =>
                Message.Debug (GlobalEnv.Core.ToString ());
                Util.ice ("cannot find variant option named " + name)
                
            }
          };
          tydecl = TypeDeclaration.Variant (List.Map (names, get_opt))
        } 
        else if (has_attribute ("Nemerle.Internal.VariantOptionAttribute", type_attrs)) {
          tydecl = TypeDeclaration.VariantOption ()
        } 
        else if (has_attribute ("Nemerle.Internal.ConstantVariantOptionAttribute",
                                type_attrs)) {
          tydecl = TypeDeclaration.VariantOption ();
          match (LookupMember ("_N_constant_object")) {
            | [fld] => constant_object = fld :> IField
            | _ => Util.ice ("cannot find _N_constant_object")
          }
        } 
        else if (has_attribute ("Nemerle.Internal.TypeAliasAttribute", type_attrs)) {
          def type_string = 
            get_string_attribute ("Nemerle.Internal.TypeAliasAttribute",
                                  "GetAliasedType", type_attrs);
          tydecl = TypeDeclaration.Alias (TyCodec.DecodeType (lib, tenv, type_string))
        } 
        else if (system_type.IsInterface)
          tydecl = TypeDeclaration.Interface ()
        else if (system_type.IsEnum)
          tydecl = TypeDeclaration.Enum ()
        else
          tydecl = TypeDeclaration.Class ();
      }

      public override GetTydecl () : TypeDeclaration
      {
        tydecl
      }

      public override GetConstantObject () : IField
      {
        constant_object
      }

      protected override imember_of_memberinfo (m : SR.MemberInfo) : IMember
      {
        def is_nem (x : SR.MemberInfo) {
#if _GENERICS
          true ||
#endif
          has_attribute ("Nemerle.Internal.TypeAttribute", x.GetCustomAttributes (false))
        };
        match (m.MemberType) {
          | MemberTypes.Constructor | MemberTypes.Method => 
            MethodInfo (tenv, library, m :> SR.MethodBase) : IMember

          | MemberTypes.Field => 
            if (is_nem (m))
              FieldInfo (tenv, library, m :> SR.FieldInfo)
            else
              FieldInfo (library, m :> SR.FieldInfo)

          | MemberTypes.Property => NetProperty (tenv, library, m :> SR.PropertyInfo)
          | MemberTypes.Event => NetEvent (library, m :> SR.EventInfo)
          | MemberTypes.NestedType => library.TypeInfoOfType (m :> System.Type)
          | _ => 
            Util.ice ("unknown object in nemerle type " + m.Name + " in " + FullName)
        }
      }
    }
  }
} /* namespace */
