//
// NemerleCodeGenerator.n
//
// Author:
//  Atsushi Enomoto <atsushi@ximian.com>
//  Kamil Skalski <nazgul@nemerle.org>
//
// This derived part of the source code is published as public domain.
//
// Original: CSharpCodeGenerator.cs
//
// Original Author:
//   Daniel Stodden (stodden@in.tum.de)
//
// (C) 2002 Ximian, Inc.
//

//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
using System;
using System.CodeDom;
using System.CodeDom.Compiler;
using System.IO;
using System.Reflection;
using System.Collections;

namespace Nemerle.Compiler
{
  public class NemerleCodeGenerator : CodeGenerator
  {
    mutable dont_write_semicolon : bool = false;
    mutable id : int = 0;

    private new_name () : string {
      ++id;
      "_N" + id.ToString() + "_";
    }

    public this () { }

    protected override NullToken : string
    {
      get { "null" }
    }

    protected override GenerateArrayCreateExpression (expression : CodeArrayCreateExpression) : void
    {
      def output = Output;

      output.Write( "array " );

      def initializers = expression.Initializers;
      mutable createType = expression.CreateType;

      if (initializers.Count > 0) {
        output.WriteLine ("[");
        ++Indent;
        OutputExpressionList ( initializers, true );
        --Indent;
        output.Write ("]");
      }
      else {
        mutable arrayType = createType.ArrayElementType;
        while ( arrayType != null ) {
          createType = arrayType;
          arrayType = arrayType.ArrayElementType;
        }

        output.Write( '(' );

        def size = expression.SizeExpression;
        if ( size != null )
          GenerateExpression( size );
        else
          output.Write( expression.Size );

        output.Write( ')' );
      }
      output.WriteLine ();
    }
    
    protected override GenerateBaseReferenceExpression (_ : CodeBaseReferenceExpression) : void
    {
      Output.Write ("base");
    }
    
    protected override GenerateCastExpression (expression : CodeCastExpression) : void
    {
      def output = Output;
      output.Write( "(" );
      GenerateExpression( expression.Expression );
      output.Write( " :> " );
      OutputType( expression.TargetType );
      output.Write( ")" );
    }


    protected override GenerateCompileUnitStart (_compileUnit : CodeCompileUnit) : void
    {
      GenerateComment( CodeComment( "------------------------------------------------------------------------------" ) );
      GenerateComment( CodeComment( " [autogenerated]" ) );
      GenerateComment( CodeComment( "     This code was generated by a tool." ) );
      GenerateComment( CodeComment( "     Runtime Version: " +  System.Environment.Version.ToString () ) );
      GenerateComment( CodeComment( "" ) );
      GenerateComment( CodeComment( "     Changes to this file may cause incorrect behavior and will be lost if " ) );
      GenerateComment( CodeComment( "     the code is regenerated." ) );
      GenerateComment( CodeComment( " </autogenerated>" ) );
      GenerateComment( CodeComment( "------------------------------------------------------------------------------" ) );
      Output.WriteLine();
    }

    protected override GenerateDelegateCreateExpression (expression : CodeDelegateCreateExpression) : void
    {
      def output = Output;

      // we need to explicitly create delegate, because sometimes
      // one generate code and use delegate object, instead
      // our cool functional values
      OutputType(expression.DelegateType);
      output.Write( '(' );
                        
      def targetObject = expression.TargetObject;
      when ( targetObject != null ) {
        GenerateExpression( targetObject );
        Output.Write( '.' );
      }
      output.Write( GetSafeName (expression.MethodName) );
      output.Write( ')' );
    }

    protected override GenerateFieldReferenceExpression (expression : CodeFieldReferenceExpression) : void
    {
      def targetObject = expression.TargetObject;
      when (targetObject != null) {
        GenerateExpression (targetObject);
        Output.Write ('.');
      }
      Output.Write (GetSafeName (expression.FieldName));
    }
    
    protected override GenerateArgumentReferenceExpression (expression : CodeArgumentReferenceExpression) : void
    {
      Output.Write (GetSafeName (expression.ParameterName));
    }

    protected override GenerateVariableReferenceExpression (expression : CodeVariableReferenceExpression) : void
    {
      Output.Write (GetSafeName (expression.VariableName));
    }
      
    protected override GenerateIndexerExpression (expression : CodeIndexerExpression) : void
    {
      def output = Output;

      GenerateExpression (expression.TargetObject);
      output.Write ( '[' );
      OutputExpressionList( expression.Indices );
      output.Write( ']' );
    }
    
    protected override GenerateArrayIndexerExpression (expression : CodeArrayIndexerExpression) : void
    {
      def output = Output;

      GenerateExpression (expression.TargetObject);
      output.Write ('[');
      OutputExpressionList (expression.Indices);
      output.Write (']');
    }
    
    protected override GenerateSnippetExpression (expression : CodeSnippetExpression) : void
    {
      Output.Write (expression.Value);
    }
    
    protected override GenerateMethodInvokeExpression (expression : CodeMethodInvokeExpression) : void
    {
      def output = Output;

      GenerateMethodReferenceExpression (expression.Method);

      output.Write ('(');
      OutputExpressionList (expression.Parameters);
      output.Write (')');
    }

    protected override GenerateMethodReferenceExpression (expression : CodeMethodReferenceExpression) : void
    {
      when (expression.TargetObject != null)
      {
        GenerateExpression (expression.TargetObject);
        Output.Write ('.');
      };
      Output.Write (GetSafeName (expression.MethodName));
    }

    protected override GenerateEventReferenceExpression (expression : CodeEventReferenceExpression) : void
    {
      GenerateExpression (expression.TargetObject);
      Output.Write ('.');
      Output.Write (GetSafeName (expression.EventName));
    }

    protected override GenerateDelegateInvokeExpression (expression : CodeDelegateInvokeExpression) : void
    {
      GenerateExpression (expression.TargetObject);
      Output.Write ('(');
      OutputExpressionList (expression.Parameters);
      Output.Write (')');
    }
    
    protected override GenerateObjectCreateExpression (expression : CodeObjectCreateExpression) : void
    {
      OutputType (expression.CreateType);
      Output.Write ('(');
      OutputExpressionList (expression.Parameters);
      Output.Write (')');
    }

    protected override GeneratePropertyReferenceExpression (expression : CodePropertyReferenceExpression) : void
    {
      def targetObject = expression.TargetObject;
      when (targetObject != null) {
        GenerateExpression (targetObject);
        Output.Write ('.');
      }
      Output.Write (GetSafeName (expression.PropertyName));
    }

    protected override GeneratePropertySetValueReferenceExpression (_ : CodePropertySetValueReferenceExpression) : void
    {
      Output.Write ( "value" );
    }

    protected override GenerateThisReferenceExpression (_ : CodeThisReferenceExpression) : void
    {
      Output.Write ("this");
    }

    // FIXME: Is it allowed? It looks likely to be confused with
    // CodeMethodReturnStatement...
    protected override GenerateExpressionStatement (statement : CodeExpressionStatement) : void
    {
      GenerateExpression (statement.Expression);
      unless (dont_write_semicolon)
        Output.WriteLine (';');
    }

    protected override GenerateIterationStatement (statement : CodeIterationStatement) : void
    {
      def output = Output;

      output.Write( "for (" );
      dont_write_semicolon = true;
      GenerateStatement( statement.InitStatement );
      output.Write( "; " );
      GenerateExpression( statement.TestExpression );
      output.Write( "; " );
      GenerateStatement( statement.IncrementStatement );
      output.Write( ") " );
      dont_write_semicolon = false;
      output.WriteLine('{');
      ++Indent;
      GenerateStatements(statement.Statements);
      --Indent;
      output.WriteLine('}');
    }

        protected new GenerateStatements(c : CodeStatementCollection) : void
        {
          foreach (x is CodeStatement in c) GenerateStatement(x);
          when (c.Count > 0 && (c[c.Count - 1] is CodeVariableDeclarationStatement))
            Output.WriteLine("();");
        }

        protected override GenerateThrowExceptionStatement (statement : CodeThrowExceptionStatement) : void
    {
      Output.Write ("throw");
      when (statement.ToThrow != null) {
        Output.Write (' ');
        GenerateExpression (statement.ToThrow);
      }
      unless (dont_write_semicolon)
        Output.WriteLine (";");
    }

    protected override GenerateComment (comment : CodeComment) : void
    {
      def output = Output;
      def lines = comment.Text.Split ('\n');
      mutable first = true;
      foreach (line : string in lines) {
        if ( comment.DocComment )
          output.Write ("///");
        else
          output.Write ("//");
        when (first) {
          output.Write (' ');
          first = false;
        }
        output.WriteLine (line);
          }
    }

    // FIXME: How should I represent return statement?
    protected override GenerateMethodReturnStatement (statement : CodeMethodReturnStatement) : void
    {
      def output = Output;

      when (statement.Expression != null)
        GenerateExpression (statement.Expression);
        
      unless (dont_write_semicolon)
        output.WriteLine (";");
    }

    protected override GenerateConditionStatement (statement : CodeConditionStatement) : void
    {
      def output = Output;

      def falses = statement.FalseStatements;

      if (falses.Count > 0)
        output.Write( "if (" );
      else
        output.Write( "when (" );

      GenerateExpression (statement.Condition);

      output.WriteLine( ") {" );
      ++Indent;
      GenerateStatements (statement.TrueStatements);
      --Indent;

      when (falses.Count > 0) {
        output.Write ('}');
        if (this.Options.ElseOnClosing)
          output.Write(' ');
        else
          output.WriteLine ();
        output.WriteLine ("else {");
        ++Indent;
        GenerateStatements (falses);
        --Indent;
      }
      output.WriteLine( '}' );
    }

    protected override GenerateTryCatchFinallyStatement (statement : CodeTryCatchFinallyStatement) : void
    {
      def output = Output;
      def options = this.Options;

      output.WriteLine ("try {");
      ++Indent;
      GenerateStatements (statement.TryStatements);
      --Indent;
      output.Write ('}');

      when (statement.CatchClauses.Count > 0) {
        output.WriteLine ( " catch {" );
                ++Indent;
        foreach (clause is CodeCatchClause in statement.CatchClauses) {
          output.Write ("| ");
          OutputTypeNamePair (clause.CatchExceptionType, GetSafeName (clause.LocalName));
          output.WriteLine (" =>");
          ++Indent;
          GenerateStatements (clause.Statements);
          --Indent;
        }
                --Indent;
                output.Write("}");
            }

      def finallies = statement.FinallyStatements;
      when ( finallies.Count > 0 ) {
        if ( options.ElseOnClosing )
          output.Write( ' ' );
        else
          output.WriteLine();
        output.WriteLine( "finally {" );
        ++Indent;
        GenerateStatements( finallies );
        --Indent;
        output.WriteLine( '}' );
      }

      output.WriteLine();
    }

    protected override GenerateAssignStatement (statement : CodeAssignStatement) : void
    {
      def output = Output;
      GenerateExpression (statement.Left);
      output.Write (" = ");
      GenerateExpression (statement.Right);
      unless (dont_write_semicolon)
        output.WriteLine(';');
    }

    protected override GenerateAttachEventStatement (statement : CodeAttachEventStatement) : void
    {
      def output = Output;

      GenerateEventReferenceExpression (statement.Event);
      output.Write (" += ");
      GenerateExpression (statement.Listener);
      unless (dont_write_semicolon)
        output.WriteLine (';');
    }

    protected override GenerateRemoveEventStatement (statement : CodeRemoveEventStatement) : void
    {
      def output = Output;
      GenerateEventReferenceExpression (statement.Event);
      Output.Write (" -= ");
      GenerateExpression (statement.Listener);
      unless (dont_write_semicolon)
        output.WriteLine (';');
    }

    protected override GenerateGotoStatement (_ : CodeGotoStatement) : void
    {
      throw NotSupportedException ("Goto statement is not supported in Nemerle language.");
    }
    
    protected override GenerateLabeledStatement (_ : CodeLabeledStatement) : void
    {
      throw NotSupportedException ("Labeled statement is not supported in Nemerle language.");
    }

    protected override GenerateVariableDeclarationStatement (statement : CodeVariableDeclarationStatement) : void
    {
      def output = Output;

      output.Write ("mutable ");

      output.Write (GetSafeName (statement.Name));

      def initExpression = statement.InitExpression;
      output.Write(" = ");
      if (initExpression != null) {
        GenerateExpression( initExpression );
        match (initExpression) {
          | x is CodePrimitiveExpression =>
            when (x.Value == null)
            {
              Output.Write(" : ");
              OutputType(statement.Type);
            }
          | _ => ()
        }
      }
      else {
        /// FIXME: we should check if it is value type and emit null or () ctor
        if (statement.Type.ArrayElementType != null) {
          output.Write(NullToken);
          output.Write(" : ");
          OutputType (statement.Type);
        }
        else {
          output.Write (statement.Type.BaseType);
          output.Write (" ()");
        }
      }
      unless (dont_write_semicolon)
        output.WriteLine( ';' );
    }

    protected override GenerateLinePragmaStart (linePragma : CodeLinePragma) : void
    {
      Output.WriteLine ();
      Output.Write ("#line ");
      Output.Write (linePragma.LineNumber);
      Output.Write (" \"");
      Output.Write (linePragma.FileName);
      Output.Write ("\"");
      Output.WriteLine ();
    }

    protected override GenerateLinePragmaEnd (_ : CodeLinePragma) : void
    {
      Output.WriteLine ();
      Output.WriteLine ("#line default");
    }

    protected override GenerateEvent (eventRef : CodeMemberEvent, _declaration : CodeTypeDeclaration) : void
    {
      when (eventRef.CustomAttributes.Count > 0)
        OutputAttributeDeclarations(eventRef.CustomAttributes);

      OutputMemberAccessModifier (eventRef.Attributes);
      OutputMemberScopeModifier (eventRef.Attributes %| MemberAttributes.Final); // Don't output "virtual"
      Output.Write ("event ");
      OutputTypeNamePair (eventRef.Type, GetSafeName (eventRef.Name));
      Output.WriteLine (';');
    }

    protected override GenerateField (field : CodeMemberField) : void
    {
      def output = Output;

      def attributes = field.Attributes;
      if (IsCurrentEnum) {
        output.Write("| ");
        Output.Write(field.Name);
      }
      else
      {
        when (field.CustomAttributes.Count > 0)
          OutputAttributeDeclarations(field.CustomAttributes);

        OutputMemberAccessModifier(attributes);
        OutputFieldScopeModifier(attributes);
        OutputTypeNamePair(field.Type, GetSafeName(field.Name));
      }

      def initExpression = field.InitExpression;
      when ( initExpression != null ) {
        output.Write ( " = " );
        GenerateExpression ( initExpression );
      }

      if (!IsCurrentEnum)
        output.WriteLine(';');
      else
        output.WriteLine("");
    }

    protected override OutputFieldScopeModifier (attributes : MemberAttributes) : void
    {
      when (attributes %& MemberAttributes.VTableMask == MemberAttributes.New)
        Output.Write("new ");
      when ((attributes %&& MemberAttributes.Static) ||
            (attributes %&& MemberAttributes.Const))
        Output.Write("static ");
//      when (! (attributes %&& MemberAttributes.Const))
        Output.Write("mutable ");
    }

    protected override GenerateSnippetMember (member : CodeSnippetTypeMember) : void
    {
      Output.Write (member.Text);
    }
    
    protected override GenerateEntryPointMethod (method : CodeEntryPointMethod, declaration : CodeTypeDeclaration) : void
    {
      method.Name = "Main";
      GenerateMethod (method, declaration);
    }

    protected override GenerateMethod (method : CodeMemberMethod, declaration : CodeTypeDeclaration) : void
    {
      def output = Output;

      when (method.CustomAttributes.Count > 0)
        OutputAttributeDeclarations (method.CustomAttributes);

      when (method.ReturnTypeCustomAttributes.Count > 0)
        OutputAttributeDeclarations (method.ReturnTypeCustomAttributes);

      def attributes = method.Attributes;

      when (method.PrivateImplementationType == null && !declaration.IsInterface)
        OutputMemberAccessModifier (attributes);

      when (!declaration.IsInterface)
        OutputMemberScopeModifier (attributes);

            // we have to generate fresh name for method explicitly implementing
            // interface, so there will be no collisions with other such methods
            def privateType = method.PrivateImplementationType;
            if (privateType != null)
            {
                output.Write(new_name());
                output.Write(GetSafeName(method.Name));
            }
            else
                output.Write(GetSafeName(method.Name));

            output.Write (' ');

      output.Write ('(');
      OutputParameters (method.Parameters);
      output.Write (')');

      output.Write (" : ");
      OutputType (method.ReturnType);

      if ( (attributes %& MemberAttributes.ScopeMask) == MemberAttributes.Abstract 
           || declaration.IsInterface)
        output.WriteLine ( ';' );
      else {
                when (privateType != null)
                {
                    output.Write(" implements ");
                    OutputType(privateType);
                    output.Write (".");
              output.Write (GetSafeName (method.Name));
                }

                output.WriteLine ( " {" );
        ++Indent;
        GenerateStatements (method.Statements);
        --Indent;
        output.WriteLine ( '}' );
      }
    }

    protected override GenerateProperty (property : CodeMemberProperty, declaration : CodeTypeDeclaration) : void
    {
      def output = Output;

      when (property.CustomAttributes.Count > 0)
        OutputAttributeDeclarations( property.CustomAttributes );

      def attributes = property.Attributes;
      OutputMemberAccessModifier( attributes );
      OutputMemberScopeModifier( attributes );


      if (property.Name == "Item")
      {
        // indexer
                          output.Write (GetSafeName (property.Name));
                          output.Write (' ');
                          output.Write("[");
                          OutputParameters(property.Parameters);
                          output.Write("]");
                          output.Write (" : ");
                          OutputType (property.Type);                          
      }
      else
      {
                     OutputTypeNamePair( property.Type, GetSafeName (property.Name));
      }
      output.WriteLine (" {");
      ++Indent;

      if (declaration.IsInterface)
      {
        when (property.HasGet) output.WriteLine("get; ");
        when (property.HasSet) output.WriteLine("set; ");
      }
      else
      {
        when (property.HasGet)
        {
          output.WriteLine ("get {");
          ++Indent;

          GenerateStatements (property.GetStatements);

          --Indent;
          output.WriteLine ("}");
        }

       when (property.HasSet)
        {
          output.WriteLine ("set {");
          ++Indent;

          GenerateStatements (property.SetStatements);

          --Indent;
          output.WriteLine ("}");
        }
      }

      --Indent;
      output.WriteLine ("}");
    }

    protected override GenerateConstructor(constructor : CodeConstructor, _declaration : CodeTypeDeclaration) : void
    {
      when (constructor.CustomAttributes.Count > 0)
        OutputAttributeDeclarations(constructor.CustomAttributes);

            OutputMemberAccessModifier (constructor.Attributes);
      Output.Write ("this (");
      OutputParameters (constructor.Parameters);
      Output.Write (") ");
      Output.WriteLine ("{");
      ++Indent;
      when (constructor.ChainedConstructorArgs.Count > 0)
      {
        Output.Write("this (");
        mutable first = true;
        foreach (ex is CodeExpression in constructor.ChainedConstructorArgs)
        {
          when (!first)
            Output.Write(", ");
          first = false;
          GenerateExpression(ex);
        }
        
        Output.Write(");");
      };
       when (constructor.BaseConstructorArgs.Count > 0)
      {
        Output.Write("base (");
        mutable first = true;
        foreach (ex is CodeExpression in constructor.BaseConstructorArgs)
        {
          when (!first)
            Output.Write(", ");
          first = false;
          GenerateExpression(ex);
        }
        
        Output.WriteLine (");");
      };
      GenerateStatements (constructor.Statements);
      --Indent;
      Output.WriteLine ('}');
    }
    
    protected override GenerateTypeConstructor (constructor : CodeTypeConstructor) : void
    {
      Output.WriteLine ("static this () {");
      ++Indent;
      GenerateStatements (constructor.Statements);
      --Indent;
      Output.WriteLine ('}');
    }

    protected override GenerateTypeStart (declaration : CodeTypeDeclaration) : void
    {
      def output = Output;

      when (declaration.CustomAttributes.Count > 0)
        OutputAttributeDeclarations ( declaration.CustomAttributes );

      def attributes = declaration.TypeAttributes;
      OutputTypeAttributes (attributes,
                declaration.IsStruct,
                declaration.IsEnum );

      output.Write (GetSafeName (declaration.Name));

      output.Write( ' ' );
      
      def enumerator = declaration.BaseTypes.GetEnumerator();
      when ( enumerator.MoveNext() ) {
        mutable ty = (enumerator.Current :> CodeTypeReference);
      
        output.Write( ": " );
        OutputType( ty );
        
        while ( enumerator.MoveNext() ) {
          ty = enumerator.Current :> CodeTypeReference;
        
          output.Write( ", " );
          OutputType( ty );
        }

        output.Write( ' ' );
      }
      if (declaration is CodeTypeDelegate)
        output.Write ( "(" );
      else
        output.WriteLine ( "{" );
      ++Indent;
    }

    protected override GenerateTypeEnd (declaration : CodeTypeDeclaration) : void
    {
      --Indent;
      if (declaration is CodeTypeDelegate) {
        Output.Write (") : ");
        def ty = (declaration :> CodeTypeDelegate).ReturnType;
        if (ty != null)
          OutputType (ty);
        else
          Output.Write ("void");
        Output.WriteLine (";");
      }
      else
        Output.WriteLine ("}");
    }

    protected override GenerateNamespaceStart (ns : CodeNamespace) : void
    {
      def output = Output;
      
      def name = ns.Name;
      when ( name != null && name != "" ) {
        output.Write( "namespace " );
        output.Write( GetSafeName (name) );
        output.WriteLine( " {" );
        ++Indent;
      }
    }

    protected override GenerateNamespaceEnd (ns : CodeNamespace) : void
    {
      def name = ns.Name;
      when ( name != null && name != "" ) {
        --Indent;
        Output.WriteLine( "}" );
      }
    }

    protected override GenerateNamespaceImport (import : CodeNamespaceImport) : void
    {
      def output = Output;

      output.Write( "using " );
      output.Write( GetSafeName (import.Namespace) );
      output.WriteLine( ';' );
    }

    protected override GenerateAttributeDeclarationsStart (attributes : CodeAttributeDeclarationCollection) : void
    {
      Output.Write( '[' );

      match (CurrentMember) {
        | met is CodeMemberMethod when
          met.ReturnTypeCustomAttributes == (attributes : object) =>
          Output.Write ("return: ");

          def enumerator = attributes.GetEnumerator();
          while (enumerator.MoveNext())
          {
            def attribute = (enumerator.Current :> CodeAttributeDeclaration);
            attribute.Name = attribute.Name.Replace('+', '.');
          }
        | _ => ()
      }
    }

    protected override GenerateAttributeDeclarationsEnd (_ : CodeAttributeDeclarationCollection) : void
    {
      Output.WriteLine( ']' );
    }

    protected override OutputType (ty : CodeTypeReference) : void
    {
      Output.Write (GetTypeOutput(ty));
    }

    protected override OutputTypeNamePair (ty : CodeTypeReference, name : string) : void
    {
      Output.Write (GetSafeName (name));
      Output.Write (" : ");
      Output.Write (GetTypeOutput (ty));
    }

    protected override QuoteSnippetString (value : string) : string
    {
      // FIXME: this is weird, but works.
      mutable output = value.Replace ("\\", "\\\\");
      output = output.Replace ("\"", "\\\"");
      output = output.Replace ("\t", "\\t");
      output = output.Replace ("\r", "\\r");
      output = output.Replace ("\n", "\\n");

      "\"" + output + "\"";
    }

    protected override GenerateParameterDeclarationExpression (e : CodeParameterDeclarationExpression) : void
    {
      when (e.CustomAttributes != null && e.CustomAttributes.Count > 0)
        OutputAttributeDeclarations (e.CustomAttributes);
      Output.Write (GetSafeName (e.Name));
      Output.Write (" : ");
      OutputDirection(e.Direction);
      OutputType (e.Type);
    }

    protected override GenerateTypeOfExpression (e : CodeTypeOfExpression) : void
    {
      Output.Write ("typeof(");
      OutputType (e.Type);
      Output.Write (")");
    }

    protected override OutputOperator (op : CodeBinaryOperatorType) : void
    {
      | CodeBinaryOperatorType.BitwiseAnd =>
        Output.Write(" %& ");
              
      | CodeBinaryOperatorType.BitwiseOr =>
        Output.Write(" %| ");
                
      | _ =>
        base.OutputOperator(op);
    }

    /* 
     * ICodeGenerator
     */
    protected override CreateEscapedIdentifier (value : string) : string
    {
      when (value == null)
        throw NullReferenceException ("Argument identifier is null.");
      GetSafeName (value);
    }

    protected override CreateValidIdentifier (value : string) : string
    {
      when (value == null)
        throw NullReferenceException ();

      when (keywordsTable == null)
        FillKeywordTable ();

      if (keywordsTable.Contains (value) || typesTable.Contains (value))
        "_" + value;
      else
        value;
    }

    protected override GetTypeOutput (ty : CodeTypeReference) : string
    {
      def arrayType = ty.ArrayElementType;
      def output =
        if ( arrayType != null ) {
          "array <" +
          (if (ty.ArrayRank > 1) { ty.ArrayRank.ToString () + ", " } else "") +
          GetTypeOutput (arrayType) + ">";
        }
        else
          match (ty.BaseType.ToLower(System.Globalization.CultureInfo.InvariantCulture))
          {
            | "system.decimal" => "decimal";
            | "system.double" => "double";
            | "system.single" => "float";
            | "system.byte" => "byte";
            | "system.sbyte" => "sbyte";
            | "system.int32" => "int";
            | "system.uint32" => "uint";
            | "system.int64" => "long";
            | "system.uint64" => "ulong";
            | "system.int16" => "short";
            | "system.uint16" => "ushort";
            | "system.boolean" => "bool";
            | "system.char" => "char";
            | "system.string" => "string";
            | "system.object" => "object";
            | "system.void" => "void";
            | _ => GetSafeTypeName (ty.BaseType);
          };

      output.Replace ("+", ".");
    }

    protected override IsValidIdentifier (identifier : string) : bool
    {
      when (keywordsTable == null)
        FillKeywordTable ();

      !keywordsTable.Contains (identifier) && !typesTable.Contains (identifier);
    }

    protected override Supports (supports : GeneratorSupport) : bool
    {
      | GeneratorSupport.Win32Resources => false;
      | GeneratorSupport.GotoStatements => false;
      | _ => true;
    }

    static GetSafeName (id : string) : string
    {
      when (keywordsTable == null) {
        FillKeywordTable ();
      }
      if (keywordsTable.Contains (id) || typesTable.Contains (id))
        "@" + id;
      else
        id;
    }

    static GetSafeTypeName (id : string) : string
    {
      when (keywordsTable == null) {
        FillKeywordTable ();
      }
      if (keywordsTable.Contains (id))
        "@" + id;
      else
        id;
    }

    static FillKeywordTable () : void
    {
      keywordsTable = Hashtable ();
      foreach (keyword : string in keywords) keywordsTable.Add (keyword,keyword);
      typesTable = Hashtable ();
      foreach (ty : string in types) typesTable.Add (ty,ty);
    }

    static mutable typesTable : Hashtable;
    static types : array [ string] = array [
      "object","bool","byte","float","uint","char","ulong","ushort",
      "decimal","int","sbyte","short","double","long","string","void"
    ];

    static mutable keywordsTable : Hashtable;
    static keywords : array [string] = array [
      "abstract", "and", "array", "as", "base", "catch", "class", "def", "delegate", "do", "else", "enum", "event", "extends", "extern", "false", "finally", "for", "foreach", "fun", "if", "implements", "in", "interface", "internal", "lock", "macro", "match", "module", "mutable", "namespace", "new", "null", "out", "override", "params", "private", "protected", "public", "ref", "sealed", "static", "struct", "syntax", "this", "throw", "true", "try", "type", "typeof", "unless", "using", "variant", "virtual", "void", "when", "where", "while", "assert", "abort", "ignore"
    ];
  }
}
