/*
 * Copyright (c) 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;
using Nemerle.Utility;
using System.Text;
 
namespace Nemerle.Compiler 
{
  public module Options
  {
    public mutable OutputFileName : string;
    public mutable XmlDoc : bool;
    public mutable DumpTypedTree : bool;
    public mutable DumpNamedMethod : string;
    public mutable TargetIsLibrary : bool;
    public mutable TargetIsWinexe : bool;
    public mutable IgnoreConfusion : bool;
    public mutable ThrowOnError : bool;
    public mutable GeneralTailCallOpt : bool;
    public mutable ProgressBar : bool;
    public mutable ColorMessages : bool;
    public mutable UseLoadedCorlib : bool;
    public mutable DoNotLoadMacros : bool;
    public mutable DoNotLoadStdlib : bool;
    public mutable EmitDebug : bool;
    public mutable CompileToMemory : bool;
    public mutable EarlyExit : bool;
    public mutable GreedyReferences : bool;

    public mutable LinkedResources : list [string];
    public mutable EmbeddedResources : list [string];
    public mutable ReferencedLibraries : list [string];
    public mutable Sources : list [string];

    internal Validate () : void {
      when (System.IO.Path.GetExtension (OutputFileName) == "")
        OutputFileName +=
          (if (Options.TargetIsLibrary) ".dll" else ".exe");
    }

    this ()
    {
      Clear ();
    }

    public Clear () : void
    {
      OutputFileName = "out.exe";
      XmlDoc = false;
      DumpTypedTree = false;
      TargetIsLibrary = false;
      TargetIsWinexe = false;
      IgnoreConfusion = false;
      ThrowOnError = false;
      GeneralTailCallOpt = false;
      ProgressBar = false;
      ColorMessages = true;
      UseLoadedCorlib = false;
      DoNotLoadMacros = false;
      DoNotLoadStdlib = false;
      EmitDebug = false;
      CompileToMemory = false;    
      EarlyExit = false;    
      GreedyReferences = true;
      DumpNamedMethod = "";

      LinkedResources = [];
      EmbeddedResources = [];
      ReferencedLibraries = [];
      Sources = [];
    }

    public ShouldDump (fn : Typedtree.Fun_header) : bool
    {
      DumpTypedTree &&
      (DumpNamedMethod == "" || DumpNamedMethod == fn.name)
    }

    public GetCommonOptions () : list [Getopt.CliOption]
    {
      def split_opt (s) {
        if (s == null)
          []
        else {
          def split = NString.Split (s, array [' ', '\t', '\n', '\r']);
          def split = List.Map (split, fun (x : string) { x.Trim () });
          List.Filter (split, fun (x) { x != "" })
        }
      }
      
      def execute_pkgconfig (opt : string) {
        def pkg = System.Diagnostics.Process ();
        pkg.StartInfo.FileName = "pkg-config"; 
        pkg.StartInfo.Arguments = ("--libs " + opt);
        pkg.StartInfo.RedirectStandardOutput = true;
        pkg.StartInfo.UseShellExecute = false;
        mutable result = "";
        try {
          ignore (pkg.Start ());
          result = pkg.StandardOutput.ReadLine ();
          if (pkg.WaitForExit (5000))
            pkg.Close ();
          else {
            pkg.Kill ();
            throw System.Exception ("operation timeouted")
          }
        }
        catch { 
          | e => Message.Warning ($"pkg-config execution failed: $(e.Message)")
        };
        match (split_opt (result)) {
          | [] => ["-r", opt]
          | x => x
        }
      }
      
      def execute_fromfile (s : string) {
        try {
          mutable line = "";
          mutable args = [];

          // Create an instance of StreamReader to read from a file.
          // The using statement also closes the StreamReader.
          using (sr = System.IO.StreamReader(s)) {
            def sb = StringBuilder ();
            
            // process each line
            while (line != null) {
              def t = line.Length;

              // iterate through line parsing "" and '' quoted strings
              for (mutable i = 0; i < t; i++)
              {
                mutable c = line [i];
                match (c) {
                  | '"' | '\'' =>
                    i++;
                    def end = c;

                    def loop () {
                      when (i < t) {
                        // if it is not the end of quotation, proceed
                        unless (line [i] == end) {
                          _ = sb.Append (line [i]);
                          i++;
                          loop ()
                        }
                      } 
                    }
                    loop ();
                
                  | ' ' =>
                    // whitespace not inside quotation
                    when (sb.Length > 0){
                      args = sb.ToString () :: args;
                      sb.Length = 0;
                    }
                
                  | _ => ignore (sb.Append (c));
                }
              }
              // if something was read, store it
              when (sb.Length > 0){
                args = sb.ToString () :: args;
                sb.Length = 0;
              }
              line = sr.ReadLine ();
            }
            // return result
            List.Rev (args)
          }
        } 
        catch {
          | _ =>
            Message.Error ("cannot read response file `" + s + "'");
            []
        }  
      }

      def set_target (_) {
        | "winexe"
        | "wexe"
        | "win" =>
          Options.TargetIsLibrary = false;
          Options.TargetIsWinexe = true;
        | "lib"
        | "library"
        | "dll" =>
          Options.TargetIsLibrary = true;
        | "exe"
        | "console" =>
          Options.TargetIsLibrary = false;
          Options.TargetIsWinexe = false;
        | x =>
          Getopt.Error ($ "invalid target `$(x)'");
          System.Environment.Exit (1);
      }

      [
        Getopt.CliOption.String (name = "-out",
                       aliases = ["-o"],
                       help = "Output file name",
                       handler = fun (s) { Options.OutputFileName = s }),

        Getopt.CliOption.String (name = "-target", 
                     aliases = ["-t"],
                     help = "Specifies the target (exe, library, winexe)",
                     handler = set_target),

        Getopt.CliOption.String (name = "-reference",
                       aliases = ["-r", "-ref"],
                       help = "Link specified assembly",
                       handler = fun (s) { Options.ReferencedLibraries = 
                         s :: Options.ReferencedLibraries
                       }),

        Getopt.CliOption.String (name = "-library-path",
                       aliases = ["-lib", "-L"],
                       help = "Add specified directory to library search path",
                       handler = LibraryReferenceManager.AddSearchDirectory),

        Getopt.CliOption.String (name = "-define",
                       aliases = ["-d", "-def"],
                       help = "Define preprocessor symbol for conditional compilation",
                       handler = fun (x) { LexerFile.command_defines.Set (x, true) }),

        Getopt.CliOption.String (name = "-doc", 
                     aliases = [], 
                     help = "Output XML documentation of program's class hierarchy",
                     handler = fun (x) { 
                       Options.XmlDoc = true;
                       XmlDoc.OutputFileName = x;
                       LexerFile.store_comments = true;
                     }),

        Getopt.CliOption.String (name = "-resource",
                       aliases = ["-res"],
                       help = "Embed resource file to output",
                       handler = fun (x) {
                         Options.EmbeddedResources = x :: Options.EmbeddedResources;
                       }),

        Getopt.CliOption.String (name = "-linkresource",
                       aliases = ["-linkres"],
                       help = "Link resource file from output assembly",
                       handler = fun (x) {
                         Options.LinkedResources = x :: Options.LinkedResources;
                       }),

        Getopt.CliOption.Flag (name = "-debug", 
                     aliases = ["-g"],
                     help = "Enable debug symbols generation",
                     handler = fun () { Options.EmitDebug = true; }),
                       
        Getopt.CliOption.SubstitutionString (name = "-pkg-config",
                       aliases = ["-pkg", "-p"],
                       help = "Link to assemblies listed by pkg-config run on"
                              " given string",
                       substitute = execute_pkgconfig),

        Getopt.CliOption.SubstitutionString (name = "-from-file",
                       aliases = ["@"],
                       help = "Read command line options from given file", 
                       substitute = execute_fromfile),

        Getopt.CliOption.String (name = "-warn",
                                 aliases = ["-W"],
                                 help = "Specify warning level", 
                                 handler = fun (x) {
                                   WarningOptions.Level =
                                     match (x) {
                                       | "1" => 1 | "2" => 2 | "3" => 3 | "4" => 4
                                       | _ =>
                                         Message.Error (x + " is not a valid warning level (must be 0-4)");
                                         -1
                                     }
                                 }),

        Getopt.CliOption.String (name = "-nowarn",
                                 aliases = [],
                                 help = "Suppress Specified Warnings", 
                                 handler = fun (x : string) {
                                   foreach (str in x.Split (',')) {
                                     try {
                                       def num = int.Parse (str);
                                       WarningOptions.Disable (num);
                                     }
                                     catch {
                                       | _ =>
                                         Message.Error (str + " is not a valid warning number format")
                                     }
                                   }
                                 }),

        Getopt.CliOption.String (name = "-dowarn",
                                 aliases = [],
                                 help = "Enable Specified Warnings", 
                                 handler = fun (x : string) {
                                   foreach (str in x.Split (',')) {
                                     try {
                                       def num = int.Parse (str);
                                       WarningOptions.Enable (num);
                                     }
                                     catch {
                                       | _ =>
                                         Message.Error (str + " is not a valid warning number format")
                                     }
                                   }
                                 }),

        Getopt.CliOption.Boolean (name = "-greedy-references", 
                                  aliases = ["-greedy"],
                                  help = "Recursive loading references of used assemblies",
                                  handler = fun (val) { Options.GreedyReferences = val; }),

        Getopt.CliOption.Flag (name = "-general-tail-call-opt",
                     aliases = ["-Ot"],
                     help = "Enable general tail call optimization (programs are slower on MS.NET, but faster on Mono)",
                     handler = fun () { Options.GeneralTailCallOpt = true }),
                                 
        Getopt.CliOption.Flag (name = "-no-stdmacros",
                               aliases = ["-nostdmacros"],
                     help = "Do not load standard macros",
                     handler = fun () { Options.DoNotLoadMacros = true }),
        Getopt.CliOption.Flag (name = "-no-stdlib",
                               aliases = ["-nostdlib"],
                     help = "Do not load Nemerle.dll",
                     handler = fun () { Options.DoNotLoadStdlib = true }),
        Getopt.CliOption.Flag (name = "-use-loaded-corlib",
                     help = "Use already loaded mscorlib.dll and System.dll",
                     handler = fun () { Options.UseLoadedCorlib = true }),
                          
        Getopt.CliOption.Flag (name = "-ignore-confusion",
                     help = "Output stack trace even when seen errors",
                     handler = fun () { Options.IgnoreConfusion = true }),
        Getopt.CliOption.Flag (name = "-throw-on-error",
                     help = "Output stack trace on first error",
                     handler = fun () { 
                       Options.ThrowOnError = true; 
                       Options.IgnoreConfusion = true; 
                     }),
        Getopt.CliOption.Flag (name = "-early-exit",
                     help = "Exit just after first method with an error",
                     handler = fun () { 
                       Options.EarlyExit = true; 
                     }),
        Getopt.CliOption.Flag (name = "-dump-typed-tree", 
                     aliases = ["-dt"],
                     help = "Pretty prints the typed tree on stdout",
                     handler = fun () {
                       Options.DumpTypedTree = true; 
                     }),
                  
        Getopt.CliOption.String (name = "-dump-typed-method", 
                     aliases = ["-dm"],
                     help = "Pretty prints the named typed tree on stdout",
                     handler = fun (s : string) {
                       Options.DumpTypedTree = true; 
                       Options.DumpNamedMethod = s;
                     }),
                  
        Getopt.CliOption.Flag (name = "-boolean-constant-matching-opt",
                     aliases = ["-Obcm"],
                     help = "NOHELP",
                     handler = fun () {
                       Message.Warning ($"This command line option (-Obcm) has beed removed, it is"
                                        " enabled by default")
                     }),
        Getopt.CliOption.Flag (name = "-ordinal-constant-matching-opt",
                     aliases = ["-Oocm"],
                     help = "NOHELP",
                     handler = fun () {
                       Message.Warning ($"This command line option (-Oocm) has been removed - "
                                        "it didn't work correctly. If you would like to contribute"
                                        " code for optimized integer constants matching, please contact us.")
                     }),
        Getopt.CliOption.Flag (name = "-string-constant-matching-opt",
                     aliases = ["-Oscm"],
                     help = "NOHELP",
                     handler = fun () {
                       Message.Warning ($"This command line option (-Oscm) has been removed - "
                                        "it didn't work correctly. If you would like to contribute"
                                        " code for optimized string matching, please contact us.")
                     }),
      
        Getopt.CliOption.Flag (name = "-target-library",
                     aliases = ["-tdll"],
                     help = "NOHELP",
                     handler = fun () { Options.TargetIsLibrary = true; }),
        Getopt.CliOption.Flag (name = "-target-exe",
                     aliases = ["-texe"],
                     help = "NOHELP",
                     handler = fun () { Options.TargetIsLibrary = false; }),
                     
        Getopt.CliOption.Flag (name = "-nologo",
                               help = "Suppress compiler copyright message",
                               handler = fun () { Options.ProgressBar = false; }),

        Getopt.CliOption.Flag (name = "-no-color",
                               help = "Disable ANSI coloring of error/warning/hint messages",
                               handler = fun () { Options.ColorMessages = false }),

        Getopt.CliOption.Flag (name = "-pedantic-lexer",
                               help = "Enable some pedantic checks for illegal characters"
                               " in input stream (default at warning level 5)",
                               handler = fun () { WarningOptions.Enable (10002); }),

        Getopt.CliOption.Boolean (name = "-progress-bar", 
                                  aliases = ["-bar"],
                                  help = "Enable / disable progress bar for compilation",
                                  handler = fun (val) { Options.ProgressBar = val; }),
                               
        Getopt.CliOption.Flag (name = "-no-progress-bar",
                               aliases = ["-q"],
                               help = "NOHELP",
                               handler = fun () { Options.ProgressBar = false; }),
                               
        Getopt.CliOption.Flag (name = "-warn-help",
                               help = "Display help about available warnings.",
                               handler = fun () {
                                 System.Console.Write (
                                   "Following warning options are available:\n" +
                                   WarningOptions.Help);
                                 System.Environment.Exit (0);
                               }),
      ]
    }
  }

  /** Module used to enumerate and filter warnings emitted by compiler
   */
  public module WarningOptions
  {
    public Level : int
    {
      mutable cur_level : int;

      get { cur_level }
      set {
        when (value >= 0 && value <= 4) {
          cur_level = value;
          currently_enabled = Hashtable ();
          for (mutable lev = 0; lev <= value; lev++) {
            foreach (x in levels [lev])
              unless (disabled.Contains (x))
                currently_enabled.Add (x, null);
          }
          enabled.Iter (fun (x, _) {
            unless (currently_enabled.Contains (x))
              currently_enabled.Add (x, null)
          });
        }
      }
    }

    mutable currently_enabled : Hashtable [int, object];
    mutable enabled : Hashtable [int, object] = Hashtable ();
    mutable disabled : Hashtable [int, object] = Hashtable ();

    this () { Level = 4; }
    
    /** Gives information if warning with given number should be emited by
        compiler.

        It depends on currently set warning level and enabled / disabled
        particular warnings.
     */
    public IsEnabled (nr : int) : bool
    {
      currently_enabled.Contains (nr)
    }

    public Enable (nr : int) : void
    {
      if (disabled.Contains (nr))
        Message.Warning ($"warning N$nr is already explicitly disabled, thus it cannot be enabled");
      else {
        unless (currently_enabled.Contains (nr))
          currently_enabled.Add (nr, null);
        unless (enabled.Contains (nr))
          enabled.Add (nr, null)
      }
    }

    public Disable (nr : int) : void
    {
      if (enabled.Contains (nr))
        Message.Warning ($"warning N$nr is already explicitly enabled, thus it cannot be disabled");
      else {
        when (currently_enabled.Contains (nr))
          currently_enabled.Remove (nr);
        unless (disabled.Contains (nr))
          disabled.Add (nr, null)
      }        
    }
    
    levels : array [array [int]] = array [
      array [], // level 0
      array [183, 184, 602, 626, 5000], // level 1

      // level 2
      array [108, 114, 162, 251, 252, 253, 618], 

      // level 3
      array [67, 105, 168, 169, 219], 

      // level 4
      array [28, 78, 109, 628, 649,
             10001, 10003, 10005],

      // level 5
      array [10002, 10004, 10006]  
    ];


    public Help : string
    {
      get {
        def warnings = [
          (28, "'function declaration' has the wrong signature to be an entry point"),
          (105, "The using directive for 'namespace' appeared previously in this namespace"),
          (114, "'f1' hides inherited member 'f2'. To make the current method override "
                 "that implementation, add the override keyword.  Otherwise add the new keyword."),
          (168, "The variable 'var' is declared but never used"),
          (649, "Field 'field' is never assigned to, and will always have its default value 'value"),
          (10001, "Cast is unnecessary"),
          (10002, "Enable some pedantic checks for illegal characters in input stream"),
          (10003, "Other global unused member warnings"),
          (10004, "warnings about usage of bit operations on enums without correct attribute"),
          (10005, "warnings about ignoring computed values"),
          (10006, "`this' is unused, consider making method static")
        ];

        def sb = StringBuilder ();
        
        foreach ((no, str) in warnings) {
          def loop (lev) {
            if (lev >= levels.Length) ""
            else {
              mutable found = false;
              foreach (warn in levels [lev])
                when (warn == no) found = true;
              if (found) $ "[enabled by -warn:$lev]"
              else loop (lev + 1)
            }
          }
          def lev = loop (0);
          _ = sb.Append ($ "$no\t$str $lev\n");
        }

        sb.ToString ()
      }
    }


    /* warnings reference:
       from C#
    
      0067 - The event 'event' is never used. An event was declared but never
             used in the class in which it was declared.
      0078 - The 'l' suffix is easily confused with the digit '1' -- use 'L' for clarity
      0108 - The keyword new is required on 'member1' because it hides inherited member 'member2'
      0109 - The member 'member' does not hide an inherited member. The new keyword is not required
      0162 - Unreachable code detected
      0169 - The private field 'class member' is never used
      0183 - The given expression is always of the provided ('type') type
      0184 - The given expression is never of the provided ('type') type
      0219 - The variable 'variable' is assigned but never used
      0251 - Indexing an array with a negative index (array indices always start at zero)
      0252 - Possible unintended reference comparison; to get a value comparison, cast
             the left hand side to type 'type'
      0253 - Possible unintended reference comparison; to get a value comparison, cast
             the right hand side to type 'type'
      0602 - The feature 'old_feature' is deprecated. Please use 'new_feature' instead
      0618 - A class member was marked with the Obsolete attribute, such that a warning
             will be issued when the class member is referenced.
      0626 - Method, operator, or accessor 'method' is marked external and has no
             attributes on it. Consider adding a DllImport attribute to specify the external implementation
      0628 - 'member' : new protected member declared in sealed class
      ...
      5000 - Unknown compiler option '/option'
    */
  }
}
