/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

namespace Nemerle.Collections
{
  /**
   * A stack
   */
  public class Stack ['a] : ICollection ['a]
  {
    /* -- PUBLIC CONSTRUCTORS ----------------------------------------------- */
    
    public this ()
    {
      m_stack = []
    }


    /* -- PUBLIC PROPERTIES ------------------------------------------------- */

    /**
     * Returns `true' iff the stack is empty.
     */
    public IsEmpty : bool
    {
      get { m_stack is [] }
    }


    /**
     * Returns the height of the stack.
     */
    public Count : int
    {
      get { List.Length (m_stack) }
    }


    /**
     * An alias for `Count'.
     */
    public Length : int
    {
      get { Count }
    }

    
    /**
     * An alias for `Count'.
     */
    public Height : int
    {
      get { Count }
    }



    /* -- PUBLIC METHODS ---------------------------------------------------- */

    /**
     * Pushes an object to the top of the stack
     */
    public Push (x : 'a) : void
    {
      m_stack = x :: m_stack;
    }

    
    /**
     * Alias for Push.
     */
    public Add (x : 'a) : void
    {
      Push (x)
    }

    
    /**
     * Pops an object off the stack top.
     */
    public Pop () : 'a
    {
      match (m_stack) {
        | x :: xs =>
          m_stack = xs;
          x
        | _ =>
          throw EmptyCollection ("Nemerle.Collections.Stack")
      }
    }

    
    /**
     * Peeks at the object on the top of the stack.
     */
    public Peek () : 'a
    {
      match (m_stack) {
        | x :: _ => x
        | _ =>
          throw EmptyCollection ("Nemerle.Collections.Stack")
      }
    }

    
    /**
     * When read -- peeks at the object on the top of the stack. When
     * written -- replaces the topmost element with specified value (there
     * has to be one).
     */
    public Top : 'a
    {
      get {
        Peek ()
      }
      set {
        _ = Pop ();
        Push (value)
      }
    }

    
    /**
     * Same as Peek, but does not throw an exception
     * -- instead it returns an optional result.
     */
    public First () : option ['a] 
    { 
      match (m_stack) {
        | x :: _ => Some (x)
        | _ => None ()
      }
    }

    
    /**
     * Clears the contents of the stack.
     */
    public Clear () : void
    {
      m_stack = []
    }


    /**
     * Creates a shallow copy of this stack
     */
    public Clone () : Stack ['a]
    {
      def s = Stack ();
      s.m_stack = List.Copy (m_stack);
      s
    }


    /**
     * See List.Iter.
     */
    public Iter (f : ('a -> void)) : void
    {
      List.Iter (m_stack, f)
    }

    
    /**
     * See List.Map.
     */
    public Map ['b] (f : 'a -> 'b) : ICollection ['b]
    {
      def s = Stack ();
      s.m_stack = List.Map (m_stack, f);
      s
    }


    /**
     * See List.Filter.
     */
    public Filter (f : 'a -> bool) : void
    {
      m_stack = List.Filter (m_stack, f)
    }


    /**
     * See List.ForAll.
     */
    public ForAll (f : 'a -> bool) : bool
    {
      List.ForAll (m_stack, f)
    }


    /**
     * See List.Exists.
     */
    public Exists (f : 'a -> bool) : bool
    {
      List.Exists (m_stack, f)
    }


    /**
     * See List.Member.
     */
    public Contains (x : 'a) : bool
    {
      List.Member (m_stack, x)
    }


    /**
     * See List.Remove.
     */
    public Remove (x : 'a) : void
    {
      m_stack = List.Remove (m_stack, x)
    }


    /**
     * See List.FoldLeft.
     */
    public Fold ['b] (f : 'a * 'b -> 'b, x : 'b) : 'b
    {
      List.FoldLeft (m_stack, x, f)
    }


    /**
     * See List.Partition.
     */
    public Partition (f : 'a -> bool) : (ICollection ['a] * ICollection ['a])
    {
      def sat = Stack ();
      def nonsat = Stack ();
      def (s, n) = List.Partition (m_stack, f);
      sat.m_stack = s;
      nonsat.m_stack = n;
      (sat, nonsat)
    }


    /**
     * Returns the enumerator for this stack's contents,
     * top of the stack first.
     */
    public GetEnumerator () : IEnumerator ['a]
    {
      ListEnumerator (m_stack)
    }


    /* -- PRIVATE FIELDS ---------------------------------------------------- */
    
    private mutable m_stack : list ['a];
  }
}
