/*
 * Copyright (c) 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

namespace Nemerle.Collections {
  using Nemerle.Assertions;

  /**
   *  Doubly linked mutable list.
   *
   *  Insert and Remove operations on this list require constant time irrespective of whether it is
   *  a single item or another LinkedList object, that is added. 
   */
  public class LinkedList ['a] : ICollection ['a]
  {
    // Node class - not visible outside. Client code should use Enumerator for
    // list operations.
    [Record]
    variant Node ['a]
    {
      | Body {
        mutable prev : Node ['a];
        mutable next : Node ['a];
        mutable data : 'a;
      }
      | Head { // head guard - convenient for Enumerator
        mutable first : Node ['a];
      }
      | Tail { // tail guard - as above
        mutable last : Node ['a];
      }

        [Nemerle.OverrideObjectEquals]
        public Equals (obj : Node ['a]) : bool
        {
          System.Object.ReferenceEquals (this, obj);
        }

        public Next : Node ['a]
        {
          get {
            match (this) {
              | Node.Head (first) => first
              | Node.Body (_, next, _) => next
              | Node.Tail => null
            }
          }
          set {
            match (this) {
              | (Node.Head) as hd => hd.first = value
              | (Node.Body) as bd => bd.next = value
              | Node.Tail => assert (false)
            }
          }
        }

        public Prev : Node ['a]
        {
          get {
            match (this) {
              | Node.Head => null
              | Node.Body (prev, _, _) => prev
              | Node.Tail (last) => last
            }
          }
          set {
            match (this) {
              | Node.Head => assert (false)
              | (Node.Body) as bd => bd.prev = value
              | (Node.Tail) as tl => tl.last = value
            }
          }
        }

        public IsBody : bool
        {
          get {
            match (this) {
              | Node.Body => true
              | _ => false
            }
          }
        }

    }

    // beginning and end of the list for fast access to those
    mutable begin : Node.Head ['a];
    mutable end : Node.Tail ['a];

    /** Constructor initiliasing object with contents of a Nemerle list. */
    public this (some_list : list ['a])
    {
      // Note, that empty list contains two nodes - the guards for Enumerator
      // In such case the first node is the tail guard, and the last node is
      // the head guard.
      begin = Node.Head (null);
      end = Node.Tail (begin);
      begin.first = end;
      
      unless (some_list == null)
        foreach (item in some_list) 
          Insert (end, item);
    }

    /** Default constructor creating empty sequence. */
    public this ()
    {
      begin = Node.Head (null);
      end = Node.Tail (begin);
      begin.first = end;
    }

    concat_helper (separator : string, sb : System.Text.StringBuilder) : void
    {
      unless (IsEmpty) {
        def e = Enumerator (this);
        _ = e.MoveNext ();
        _ = sb.Append (e.Current);
        while (e.MoveNext ()) {
          _ = sb.Append (separator);
          _ = sb.Append (e.Current);
        }
      }
    }

    /** Returns string representing contents of the list. */
    public override ToString () : string
    {
      def sb = System.Text.StringBuilder ("[");
      concat_helper (", ", sb);
      sb.Append ("]").ToString ();
    }

    /** Constructs string out of list contents using given argument as a separator.
      * <param name="separator">String to use a separator - it will be put between each
      *  two items of the list.</param>
      */
    public ToString (separator : string) : string
    {
      def sb = System.Text.StringBuilder ();
      concat_helper (separator, sb);
      sb.ToString ();
    }        

    /** Compares two lists item by item using Equals method of contained objects. */
    [Nemerle.OverrideObjectEquals]
    public Equals (another_list : LinkedList ['a]) : bool
    {
      def e = Enumerator (this);
      def f = Enumerator (another_list);

      def compare () : bool {
        def rete = e.MoveNext ();
        def retf = f.MoveNext ();
        if (rete != retf)
          false;
        else
          if (rete == true) // there is something to compare
            if (e.Current.Equals (f.Current))
              compare ();
            else
              false;
          else // everything has been compared
            true;
      }

      compare ();      
    }

    /** Reverses elements of the list in place. Complexity is O(n). */
    public Reverse () : void
    {
      unless (IsEmpty) {
        mutable first = null;
        mutable last = null;

        def reverse_node (node : Node ['a]) : void {
          | (Node.Head) as hd => 
            first = hd.first;
            reverse_node (hd.first);
          | (Node.Body) as bd =>
            bd.prev <-> bd.next;
            reverse_node (bd.prev);
          | (Node.Tail) as tl => 
            last = tl.last;
        }

        reverse_node (begin);

        // Exchange head with tail
        end.last = first;
        begin.first = last;
        // Now - inform border nodes about change
        first.Next = end;
        last.Prev = begin;
      } // non-empty
    }

    // Inserts item d after position pos. If pos is head, inserts at the beginning.
    // In case pos is tail, inserts item at the end (so not really after pos..)
    static Insert ['a] ([NotNull] pos : Node ['a], d : 'a) : void
    {
      match (pos) {
        | (Node.Tail) as tl => // insert at the end of the list
          def node = Node.Body (tl.last, pos, d);
          tl.last.Next = node;
          tl.last = node;
        | _ => // anything else
          def node = Node.Body (pos, pos.Next, d);
          pos.Next.Prev = node;
          pos.Next = node;
      }
    }

    // Insert list l after position pos
    static Insert ['a] ([NotNull] pos : Node ['a], l : LinkedList ['a]) : void
    {
      unless (l.IsEmpty) {
        match (pos) {
          | (Node.Tail) as tl => // another way of adding at the end
            tl.last.Next = l.begin.first;
            l.begin.first.Prev = tl.last;
            l.end.last.Next = tl;
            tl.last = l.end.last;
          | _ => // anything else (there is some place after current position)
            pos.Next.Prev = l.end.last;
            l.end.last.Next = pos.Next;
            pos.Next = l.begin.first;
            pos.Next.Prev = pos;
        }

        l.Clear();             
      } // l is not empty
        
    } // Insert list   

    /** Adds item at the beginning of the list. */
    public Prepend (item : 'a) : void
    {
      Insert (begin, item);
    }

    /** Add given list at the beginning. The source will be cleared. */
    public Prepend ([NotNull] l : LinkedList ['a]) : void
    {
      Insert (begin, l);
    }

    /** Append item to the list. */
    public Append (item : 'a) : void
    {
      Insert (end, item);
    }

    /** Append another list to an end. The source list will be cleared. */
    public Append ([NotNull] l : LinkedList ['a]) : void
    {
      Insert (end, l);
    }

    // Removes given position. It should not be any of the guards.
    static Remove ['a] ([NotNull] pos : Node ['a]) : void
    {
      | Node.Head =>
        throw System.ArgumentException ("Unable to remove head of the list!");
      | Node.Tail =>
        throw System.ArgumentException ("Unable to remove tail of the list!");
      | (Node.Body) as body =>
        body.prev.Next = body.next;
        body.next.Prev = body.prev;
    }

    /**
     *  Enumerator for access to the elements of the LinkedList object.
     *
     *  This enumerator is compatible with any other .Net enumerator, but it also provides
     *  functionality to modify the list.
     */
    public class Enumerator ['a] : IEnumerator ['a]
    {
      mutable current : Node ['a]; // current position
      mutable coll : LinkedList ['a]; // corresponding list

      /** Constructs Enumerator object out of LinkedList object.
       
        * Enumerator is set to position immediately before the first element, as could be
        * expected.
        */
      public this (l : LinkedList ['a]) {
        coll = l;
        current = coll.begin; // place the cursor before the sequence
      }

      /** Get or Set value at current position in the list. */
      public Current : 'a 
      {
        get { // get current value
          match (current) {
            | (Node.Body) as body =>
              body.data;
            | _ => throw System.ArgumentException ("Unable to read outside the list!");
          }
        }
        set { // write at current position
          match (current) {
            | (Node.Body) as body =>
              body.data = value;
            | _ => throw System.ArgumentException ("Unable to write outside the list!");
          }
        }
      }

      /** Move to the next position in the list. Returns false, if there is no valid next position. */
      public MoveNext () : bool 
      {
        current = current.Next;
        current.IsBody;
      }

      /** Same as <see cref="MoveNext"/> but moves backwards. */
      public MovePrev () : bool
      {
        current = current.Prev;
        current.IsBody;
      }

      /** Reset position of the enumerator to the beginning of the list. */
      public Reset () : void
      {
        current = coll.begin;
      }

      /** Reset position of the enumerator to the end of the list. */
      public ResetEnd () : void
      {
        current = coll.end;
      }

      /** Inserts item after current position or at the end, if enumerator points past the end.
       *
       *  <param name="d">Item to be added.</param>
       */
      public Insert (d : 'a) : void
      {
        Insert (current, d);
      }

      /** Same as <see cref="Insert (d : 'a)"/>, but adds whole list. */
      public Insert ([NotNull] l : LinkedList ['a]) : void
      {
        Insert (current, l);
      }

      /** Removes element from the list and moves forward.
       *
       *  After removing element it behaves as if <see cref="MoveNext"/> was called.
       *  <exception cref="System.ArgumentException">Thrown upon attempt to remove element
       *  while being at invalid position.</exception>
       */
      public Remove () : bool
      {
        match (current) {
          | Node.Body => // valid position
            def to_delete = current;
            def retval = MoveNext();
            Remove (to_delete);
            retval;
          | _ => // invalid
            throw System.ArgumentException ("Trying to remove inproper item!");
        }
      }

      /** Returns underlying <see cref="LinkedList"/> object. */
      public Collection : LinkedList ['a]
      {
        get { coll; }
      }

    } // Enumerator

    /** Returns Enumerator for access to the list. */
    public GetEnumerator () : Enumerator ['a]
    {
      Enumerator (this);
    }

    // Now the ICollection['a] interface implementation

    /** Returns number of elements held by the list. Complexity is O(n). */
    public Count : int {
      get {
        mutable lp = 0;
        def count_them (pos : Node ['a]) : void {
          | Node.Head (first) => count_them (first);
          | (Node.Body) as body => lp += 1; count_them (body.next);
          | Node.Tail => ();
        }

        count_them (begin);
        lp;
      }
    }

    /** Returns true, if the list is empty. */
    public IsEmpty : bool
    {
      get {
        begin.first.Equals(end);
      }
    }

    /** Adds item at the beginning of the list. */
    public Add (item : 'a) : void
    {
      Prepend (item);
    }

    /** Removes all occurrences of given value in the list.
     *
     *  <param name="item">Value to be removed.</param>
     */
    public Remove (item : 'a) : void
    {
      def e = Enumerator (this);

      while (e.MoveNext ())
        when (e.Current.Equals (item)) {
          _ = e.Remove ();
          _ = e.MovePrev ();
        }        
    }

    /** Clears the list. */
    public Clear() : void
    {
      begin.first = end;
      end.last = begin;
    }

    /** Returns first element of the list as an option. */
    public First () : option ['a]
    {
      match (begin.first) {
        | Node.Tail => None ()
        | (Node.Body) as bd => Some (bd.data)
        | Node.Head => assert (false)
      }
    }

    /** Returns true, if the list contains supplied value. */
    public Contains (item : 'a) : bool
    {
      def e = Enumerator (this);

      def member () : bool {
        if (e.MoveNext ())
          if (e.Current.Equals (item))
            true;
          else
            member ();
        else
          false;
      }

      member ();
    }

    /** Returns shallow copy of the list. */
    public Clone () : LinkedList ['a]
    {
      def l = LinkedList ();
      foreach (item in this)
        l.Append (item);
      l;
    }

    public Fold ['b] (f : 'a * 'b -> 'b, x : 'b) : 'b
    {
      mutable retval = x;
      foreach (item in this)
        retval = f (item, retval);
      retval;
    }

    public Map ['b] (f : 'a -> 'b) : LinkedList ['b]
    {
      def l = LinkedList();
      foreach (item in this)
        l.Append (f (item));
      l;
    }

    public Iter (f : 'a -> void) : void
    {
      foreach (item in this)
        f (item);
    }

    public ForAll (f : 'a -> bool) : bool
    {
      def e = Enumerator (this);
      def check () : bool {
        if (e.MoveNext())
          if (f (e.Current))
            check ();
          else
            false;
        else
          true;
      }
      check ();
    }

    public Exists (f : 'a -> bool) : bool
    {
      def e = Enumerator (this);
      def check () : bool {
        if (e.MoveNext())
          if (f (e.Current))
            true;
          else
            check ();
        else
          false;
      }
      check ();
    }

    public Filter (f : 'a -> bool) : void
    {
      def e = Enumerator (this);
      def filter () : void {
        if (f (e.Current))
          when (e.MoveNext ())
            filter ();
        else
          when (e.Remove ())
            filter();
      }

      when (e.MoveNext ())
        filter();
    }

    public Partition (f : 'a -> bool) : LinkedList ['a] * LinkedList ['a]
    {
      def does = LinkedList ();
      def donot = LinkedList ();
      foreach (item in this)
        if (f (item))
          does.Append (item);
        else
          donot.Append (item);
      (does, donot);
    }

  } // LinkedList
} // namespace 
